/* back.c */

/*
Copyright (c) 1999 Alban Hertroys
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:
1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.
3. The name of the author may not be used to endorse or promote products
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include "wmdate.h"
#include "geometry.h"

#define ATTRIBUTE_CLOSENESS 40000
#define VERSION "0.5"

int testDate = 1;
Bool doTest = False;
Bool drawn  = False;
Bool shaped = False;

/* window initialisation */
Window	root, win[2];
GC			copy_gc;
GC			clip_gc;

int activeWin;  /* The window that will be drawn in (dockapp or tile) */

char *geometry;
unsigned int borderwidth;

int screen;
int x_fd;

XSizeHints a_sizehints;
XWMHints   a_wmhints;
Pixel	   fore_pix, back_pix;

/* pixmap stuff */

time_t  actualtime;
int     weekday, day, month;

Pixmap back, disp, mask;
Pixmap tile;

XpmAttributes attributes;

Atom _XA_GNUSTEP_WM_FUNC;
Atom deleteWin;

char* progName;

/* Local functions */
void    repaint ();
void    drawAll ();
void    drawDate ();
void    calcDate ();
void	readArgs (int argc, char *argv[]);
void	ButtonPressEvent (XButtonEvent *x_ev);

/*************************************************************************/

int main (int argc, char *argv[])
{
  Bool finished;
  int old_day;

/*
 * Initialization of X-display
 */
  char *display_name = NULL;
  char *wname = NAME;

  XGCValues gcv;
  unsigned long gcm;
  XEvent event;
  XTextProperty name;
  XClassHint classhint;

  readArgs(argc, argv);

/*
 * Try to open display
 */
  if ( !( dpy = XOpenDisplay (display_name)))
  {
    fprintf (stderr, "Can't open display !!!");
    exit (1);
  }
  screen = DefaultScreen (dpy);
  root = RootWindow (dpy, screen);
  x_fd = XConnectionNumber (dpy);

/*
 * Set atoms for unshaped window
 */
 _XA_GNUSTEP_WM_FUNC = XInternAtom(dpy, "_GNUSTEP_WM_FUNCTION", False);
 deleteWin = XInternAtom(dpy, "WM_DELETE_WINDOW", False);

/*
 * Create a GC
 */
  gcm = GCForeground | GCBackground | GCGraphicsExposures;
  gcv.graphics_exposures = False;
  copy_gc = XCreateGC (dpy, root, gcm, &gcv);
  clip_gc = XCreateGC (dpy, root, gcm, &gcv);

/*
 * Create and set up windows
 */
  if (shaped)
    a_sizehints.flags = PPosition | PMinSize | PMaxSize;
  else
    a_sizehints.flags = PMinSize | PMaxSize;

  a_sizehints.x = TLEFT;
  a_sizehints.y = TTOP;
  a_sizehints.min_width = TWIDTH;
  a_sizehints.min_height = THEIGHT;
  a_sizehints.max_width = TWIDTH;
  a_sizehints.max_height = THEIGHT;

  win[0] = XCreateSimpleWindow (dpy, root, a_sizehints.x, a_sizehints.y, TWIDTH, THEIGHT, 0, fore_pix, back_pix);
  win[1] = XCreateSimpleWindow (dpy, root, a_sizehints.x, a_sizehints.y, TWIDTH, THEIGHT, 0, fore_pix, back_pix);

  XSetWMNormalHints (dpy, win[0], &a_sizehints);

  a_wmhints.window_group = win[0];

  if (shaped) {
    a_wmhints.initial_state = WithdrawnState;
    a_wmhints.icon_window = win[1];
    a_wmhints.flags = StateHint|IconWindowHint|WindowGroupHint;
    activeWin = 1;
  } else {
    a_wmhints.initial_state = NormalState;
    a_wmhints.flags = StateHint|WindowGroupHint;
    activeWin = 0;
  }

  XSetWMHints(dpy, win[0], &a_wmhints); /* This is ALWAYS win[0] */

  classhint.res_name = NAME;
  classhint.res_class = CLASS;
  XSetClassHint (dpy, win[0], &classhint);

  if (XStringListToTextProperty(&wname, 1, &name) ==0) {
    fprintf(stderr, "Can't allocate window name\n");
    exit(-1);
  }

  XSetWMName (dpy, win[0], &name);
  XSetWMProtocols (dpy, win[0], &deleteWin, 1);

  XSetCommand (dpy, win[0], argv, argc);
  XStoreName (dpy, win[0], NAME);
  XSetIconName (dpy, win[0], NAME);


/*
 * Convert xpm's to usable ximages
 */
  attributes.valuemask = XpmExactColors|XpmCloseness;
  attributes.exactColors = False;
  attributes.closeness = ATTRIBUTE_CLOSENESS;

  XpmCreatePixmapFromData (dpy, root, back_xpm, &back, &mask, &attributes);
  if (!shaped)
    XpmCreatePixmapFromData (dpy, root, tile_xpm, &tile, NULL, &attributes);

  disp = XCreatePixmap(dpy, root, TWIDTH, THEIGHT, DefaultDepth(dpy,DefaultScreen(dpy)));

  fore_pix = GetColor ("black");
  back_pix = GetColor ("white");

  //drawAll();


  XMapWindow (dpy, win[0]);


/*
 * ---------------------------------------
 *|    M A I N   L O O P - Starts here    |
 * ---------------------------------------------------------------------------
 */

  calcDate ();
  drawAll ();

  XSelectInput (dpy, win[activeWin], ExposureMask | ButtonPressMask);

  old_day = 0;  /* So old_day and day will initially be different */

  finished = False;
  while (!finished)
  {
    while (XPending (dpy))
    {
      XNextEvent (dpy, &event);
      switch (event.type)
      {
        case Expose:            /* If window has been "not visible" */
			 repaint();
			 drawn = True;
          XFlush (dpy);
          break;

        case DestroyNotify:     /* When X wants it to close   -> does this "work" ? */
          finished = True;
          break;

        case ClientMessage:
          if (event.xclient.data.l[0] == deleteWin)
            finished = True;
          break;

        case ButtonPress:	/* A mouse-button was pressed */
	       if (doTest)
             ButtonPressEvent (&event.xbutton);
          break;

        default:
          break;
      }
    }

    if (drawn && !doTest && !finished)
      sleep (1);         /* sleep a second */

    calcDate (weekday, day, month);
    if (day != old_day)
    {
      drawDate (weekday, day, month);
      repaint ();
      old_day = day;
    }
  }

  XFreeGC (dpy, copy_gc);
  XFreeGC (dpy, clip_gc);
  XFreePixmap (dpy, disp);
  XFreePixmap (dpy, mask);
  if (!shaped)
    XFreePixmap (dpy, tile);
  XDestroyWindow (dpy, win[1]);
  XDestroyWindow (dpy, win[0]);
  XCloseDisplay (dpy);

  return 0;
}



Pixel GetColor(char *name)
{
  XColor color;
  XWindowAttributes attributes;

  XGetWindowAttributes(dpy, root, &attributes);
  color.pixel = 0;
   if (!XParseColor (dpy, attributes.colormap, name, &color))
     {
       fprintf (stderr, "Can not parse %s", name);
     }
   else if(!XAllocColor (dpy, attributes.colormap, &color))
     {
       fprintf (stderr, "Can not allocate %s",name);
     }
  return color.pixel;
}



void repaint()
{
   XEvent xev;
   XCopyArea (dpy, disp, win[activeWin], copy_gc, TLEFT, TTOP, TWIDTH, THEIGHT, TLEFT, TTOP);
   while(XCheckTypedEvent(dpy, Expose, &xev));
}


void drawAll ()
{
if (shaped)
  XShapeCombineMask (dpy, win[activeWin], ShapeBounding, TLEFT, TTOP, mask, ShapeSet);
else
  XCopyArea (dpy, tile, disp, copy_gc, 0, 0, TWIDTH, THEIGHT, 0, 0);

 drawDate (weekday, day, month);
}


void drawBack ()
{
  XSetClipMask (dpy, clip_gc, mask);
  XSetClipOrigin (dpy, clip_gc, 0, 0);
  XCopyArea (dpy, back, disp, clip_gc, TLEFT, TTOP, TWIDTH, THEIGHT, TLEFT, TTOP);
  XSetClipMask (dpy, clip_gc, None);
}

void drawDate ()
{
  int day10, day1;
  int offset10, offset1;
  int dummy;
  int width, width10, width1;

  if (!doTest)
     dummy = day;
  else
     dummy = testDate;

  day10 = dummy / 10;
  day1  = dummy % 10;

  width10 = D_WID;
  width1  = D_WID;  

  if (day10 == 0) width10 = 0;
  if (day10 == 1) width10 = D_ONE;
  if (day1 == 1)  width1  = D_ONE;
  width = (width1 + width10) / 2;
  offset10 = 32 - width;
  offset1  = offset10 + width10;

  /* Clear the area */
  drawBack();

  /* Write day of week */
  XCopyArea (dpy, back, disp, copy_gc, TW_LFT, W_HEI * weekday, W_WID, W_HEI, PW_LFT, PW_TOP);

  /* Write day of month */
  if (day > 9)
  {
    XCopyArea (dpy, back, disp, copy_gc, D_WID * day10 + (D_WID - width10), TD_TOP, 
                                      width10, D_HEI, 
                                      offset10, PD_TOP);
    XCopyArea (dpy, back, disp, copy_gc, D_WID * day1 + (D_WID - width1), TD_TOP, 
                                      width1, D_HEI, 
                                      offset1, PD_TOP);
  }
  else
    XCopyArea (dpy, back, disp, copy_gc, D_WID *day + (D_WID - width1), TD_TOP, 
	       width1, D_HEI, 32-width, PD_TOP);

  /* Write month */
  XCopyArea (dpy, back, disp, copy_gc, TM_LFT, M_HEI*month, M_WID, M_HEI, PM_LFT, PM_TOP);
}


void calcDate ()
{
  static struct tm *clk;

  actualtime = time(0);
  clk = localtime (&actualtime);

  weekday = clk->tm_wday;
  day     = clk->tm_mday;
  month   = clk->tm_mon;
}


void ButtonPressEvent (XButtonEvent *x_ev)
{
   testDate++;
   if (testDate > 31) testDate = 1;

   drawDate();
   repaint ();
}


void readArgs (int argc, char *argv[])
{
   int i;

   for (i=1; i<argc;i++)
   {
      if (strcmp(argv[i], "-h") == 0 || strcmp(argv[i],"-help")==0 || 
	  strcmp(argv[i],"--help")==0)
      {
	 fprintf(stderr, "\nwmDate - version %s.\n", VERSION);
	 fprintf(stderr, "A utility to show today's date.\n\n");
	 fprintf(stderr, "Usage: %s [options]\n", argv[0]);
	 fprintf(stderr, "\t-h , -help , --help\tdisplay this help screen.\n");
    fprintf(stderr, "\t-s , --shaped\t\tshaped window.\n");
	 fprintf(stderr, "\t-t , --test\t\tgo to test mode.\n");
	 fprintf(stderr, "\t-v , --version\t\tdisplay version.\n\n");
	 exit(0);
      }
      if (strcmp(argv[i], "-s") == 0 || strcmp(argv[i], "--shaped") == 0) shaped = True;
      if (strcmp(argv[i], "-t") == 0 || strcmp(argv[i], "--test") == 0) doTest = True;
      if (strcmp(argv[i], "-v") == 0 || strcmp(argv[i], "--version") == 0) 
      {
	 fprintf(stderr, "\n%s version %s\n\n", argv[0], VERSION);
	 exit(0);
      }

   }
}
