/* textview.cc
 * This file belongs to Worker, a filemanager for UNIX/X11.
 * Copyright (C) 2005 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not^, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/* $Id: textview.cc,v 1.6 2005/12/05 06:39:51 ralf Exp $ */

#include "textview.h"
#include "acontainer.h"

TVCallBack::TVCallBack( TextView *_tv )
{
  tv = _tv;
}

TVCallBack::~TVCallBack()
{
}

void TVCallBack::run( GUIElement *elem, int value )
{
  if ( tv != NULL ) tv->gui_callback( elem, value );
}

void TVCallBack::run( GUIElement *elem, void *p )
{
}

TextView::TextView( AGUIX *parent,
		    int x,
		    int y,
		    int width,
		    int height,
		    int bg,
		    std::string title,
		    TextStorage &_ts ) : AWindow( parent, x, y, width, height, bg, title ), ts( _ts ), tvcb( this )
{
  hbar = vbar = NULL;
  cont = NULL;
  font = NULL;
  last_w = last_h = last_cont_width = -1;
  line_wrap = false;
  _display_focus = false;
  
  setCanHandleFocus();
  setAcceptFocus( true );
  
  gettimeofday( &_lastwheel, NULL );
}

TextView::~TextView()
{
  destroy();
}

void TextView::doCreateStuff()
{
  AWindow::doCreateStuff();
  if ( _created == false ) return;

  hbar = new Slider( _aguix, 0, 0, 30, 12, false, 0 );
  hbar->setCallbackHandler( &tvcb );
  hbar->setAcceptFocus( false );
  vbar = new Slider( _aguix, 0, 0, 12, 30, true, 1 );
  vbar->setCallbackHandler( &tvcb );
  vbar->setAcceptFocus( false );
  
  createContainer();
  updateBars();
  return;
}

void TextView::createContainer()
{
  int bw;

  if ( isCreated() == false ) return;
  
  if ( cont != NULL ) {
    setContainer( NULL );
    delete cont;
  }
  cont = new AContainer( this, 2, ( line_wrap == true ) ? 1 : 2 );
  cont->setMinSpace( 0 );
  cont->setMaxSpace( 0 );

  bw = ( _display_focus == true ) ? 2 : 1;
  cont->setBorderWidth( bw );
  setContainer( cont, true );
  
  if ( vbar != NULL ) {
    cont->add( vbar, 1, 0, AContainer::CINCH );
    cont->setMinHeight( 30, 1, 0 );
  }
  if ( hbar != NULL ) {
    if ( line_wrap == false ) {
      cont->add( hbar, 0, 1, AContainer::CINCW );
      cont->setMinWidth( 10, 0, 1 );
      hbar->show();
    } else {
      hbar->hide();
    }
  }
  cont->setMinWidth( 2, 0, 0 );
  cont->setMinHeight( 2, 0, 0 );
  
  cont->resize( _w, _h );
  cont->rearrange();
}

void TextView::boxRedraw()
{
  prepareBG();
  _aguix->ClearWin( win );

  _aguix->setFG( 0, 2 );
  _aguix->DrawLine( win, 0, 0, _w - 1, 0 );
  _aguix->DrawLine( win, 0, 0, 0, _h - 1 );
  _aguix->setFG( 0, 1 );
  _aguix->DrawLine( win, 0, _h - 1, _w - 1, _h - 1 );
  _aguix->DrawLine( win, _w - 1, _h - 1, _w - 1, 1 );
  
  if ( getDisplayFocus() == true ) {
    if ( getHasFocus() == true ) {
      _aguix->setFG( 0, 2 );
    } else {
      _aguix->setFG( 0, 1 );
    }
    _aguix->DrawLine( win, 1, _h - 2, _w - 2, _h - 2 );
    _aguix->DrawLine( win, _w - 2, _h - 2, _w - 2, 2 );
    
    if ( getHasFocus() == true ) {
      _aguix->setFG( 0, 1 );
    } else {
      _aguix->setFG( 0, 2 );
    }
    _aguix->DrawLine( win, 1, 1, _w - 2, 1 );
    _aguix->DrawLine( win, 1, 1, 1, _h - 2 );
  }
}

bool TextView::handleMessage(XEvent *E,Message *msg)
{
  struct timeval t2;
  int dt, scrollspeed;
  
  if ( isCreated() == false ) return false;
  
  if ( ( msg->type == Expose ) &&
       ( msg->window == win ) ) {
    redraw();
  } else if ( msg->type == KeyPress ) {
    if ( ( getAcceptFocus() == true ) && ( getHasFocus() == true ) ) {
      if ( isVisible() == true ) {
	if ( isTopParent( msg->window ) == true ) {
	  // we have the focus so let's react to some keys
	  // lets call an extra handler so it can be overwritten
	  switch ( msg->key ) {
	    case XK_Up:
	      if ( vbar != NULL ) {
		vbar->setOffset( vbar->getOffset() - 1 );
		redraw();
	      }
	      break;
	    case XK_Left:
	      if ( hbar != NULL ) {
		hbar->setOffset( hbar->getOffset() - getHScrollStep() );
		redraw();
	      }
	      break;
	    case XK_Down:
	      if ( vbar != NULL ) {
		vbar->setOffset( vbar->getOffset() + 1 );
		redraw();
	      }
	      break;
	    case XK_Right:
	      if ( hbar != NULL ) {
		hbar->setOffset( hbar->getOffset() + getHScrollStep() );
		redraw();
	      }
	      break;
	    case XK_Home:
	      if ( vbar != NULL ) {
		vbar->setOffset( 0 );
		redraw();
	      }
	      break;
	    case XK_End:
	      if ( vbar != NULL ) {
		vbar->setOffset( vbar->getMaxLen() );
		redraw();
	      }
	      break;
	    case XK_Prior:
	      if ( vbar != NULL ) {
		vbar->setOffset( vbar->getOffset() - vbar->getMaxDisplay() + 1 );
		redraw();
	      }
	      break;
	    case XK_Next:
	      if ( vbar != NULL ) {
		vbar->setOffset( vbar->getOffset() + vbar->getMaxDisplay() - 1 );
		redraw();
	      }
	      break;
	    case XK_w:
	      setLineWrap( ( getLineWrap() == true ) ? false : true );
	      break;
	    default:
	      break;
	  }
	}
      }
    }
  } else if ( ( msg->type == ButtonPress ) && ( msg->window == win ) ) {
    if ( ( msg->button == Button4 ) || ( msg->button == Button5 ) ) {
      gettimeofday( &t2, NULL );
      dt = ldiffgtod_m( &t2, &_lastwheel );
      
      if ( dt < 200 ) scrollspeed = 5;
      else if ( dt < 400 ) scrollspeed = 2;
      else scrollspeed = 1;
      
      if ( msg->button == Button4 ) {
	if ( vbar != NULL ) {
	  int old_offset = vbar->getOffset();
	  vbar->setOffset( vbar->getOffset() - scrollspeed );
	  if ( vbar->getOffset() != old_offset )
	    redraw();
	}
      } else if ( msg->button == Button5 ) {
	if ( vbar != NULL ) {
	  int old_offset = vbar->getOffset();
	  vbar->setOffset( vbar->getOffset() + scrollspeed );
	  if ( vbar->getOffset() != old_offset )
	    redraw();
	}
      }
      
      _lastwheel = t2;
    } else if ( msg->button == Button1 ) {
      applyFocus( this );
    }
  }

  return AWindow::handleMessage( E, msg );
}

void TextView::doDestroyStuff()
{
  AWindow::doDestroyStuff();
  delete cont;
  cont = NULL;
  setContainer( NULL );
  delete hbar;
  delete vbar;
  hbar = vbar = NULL;
}

void TextView::gui_callback( GUIElement *elem, int value )
{
  if ( ( elem == vbar ) || ( elem == hbar ) ) {
    redraw();
  }
}

void TextView::redraw()
{
  int tx, ty, elementHeight, lines, width;
  GC usegc;
  XRectangle clip_rect;
  int bw;

  if ( isCreated() == false ) return;

  boxRedraw();

  bw = ( _display_focus == true ) ? 3 : 2;

  tx = ty = bw;

  if ( font == NULL ) usegc = 0;
  else usegc = font->getGC();

  if ( font == NULL ) elementHeight = _aguix->getCharHeight();
  else elementHeight = font->getCharHeight();

  getLinesWidth( lines, width );

  if ( ( _w != last_w ) || ( _h != last_h ) || ( width != last_cont_width ) ) {
    if ( line_wrap == true ) {
      ts.setLineLimit( width );
    } else {
      ts.setLineLimit( -1 );
    }
    updateBars();
    last_w = _w;
    last_h = _h;
    last_cont_width = width;
  }

  if ( font == NULL )
    _aguix->setFG( 1 );
  else
    _aguix->setFG( font->getGC(), 1 );

  clip_rect.x = tx;
  clip_rect.y = 0;
  clip_rect.width = width;
  clip_rect.height = getHeight();
  
  _aguix->SetClipRectangles( font, 0, 0, &clip_rect, 1, Unsorted );

  for ( int i = 0; i < lines; i++ ) {
    std::string s1;
    // width is a good upper limit for the line length
    ts.getLine( vbar->getOffset() + i, 0, width, s1 );
    if ( font == NULL )
      _aguix->DrawText( win, s1.c_str(), tx - hbar->getOffset(), i * elementHeight + ty );
    else
      _aguix->DrawText( win, font, s1.c_str(), tx - hbar->getOffset(), i * elementHeight + ty );
  }
  _aguix->UnsetClipMask( font );
}

int TextView::setFont( char *fontname )
{
  font = _aguix->getFont( fontname );
  last_w = last_h = last_cont_width = -1;
  if ( font == NULL ) return -1;
  return 0;
}

void TextView::updateBars()
{
  int lines, width;
  
  if ( isCreated() == false ) return;

  getLinesWidth( lines, width );
  
  vbar->setMaxDisplay( lines );
  hbar->setMaxDisplay( width );
  vbar->setMaxLen( ts.getNrOfLines() );
  hbar->setMaxLen( ts.getMaxLineWidth() );
}

void TextView::setLineWrap( bool nv )
{
  line_wrap = nv;
  last_w = last_h = last_cont_width = -1; // force textstorage rebuild
  createContainer();
  redraw();
}

bool TextView::getLineWrap() const
{
  return line_wrap;
}

void TextView::getLinesWidth( int &lines, int &width ) const
{
  int tw ,th, elementHeight;

  if ( cont != NULL ) {
    tw = cont->getWidth( 0, 0 ) - 2;
    if ( tw < 0 ) tw = 0;
    th = cont->getHeight( 0, 0 ) - 2;
    if ( th < 0 ) th = 0;
  } else {
    tw = th = 0;
  }

  if ( font == NULL ) elementHeight = _aguix->getCharHeight();
  else elementHeight = font->getCharHeight();

  lines = th / elementHeight;
  width = tw;
}

void TextView::setDisplayFocus( bool nv )
{
  _display_focus = nv;
  redraw();
}

bool TextView::getDisplayFocus() const
{
  return _display_focus;
}

void TextView::maximizeX()
{
  int l;
  int bw;

  if ( line_wrap == true ) return;
  
  bw = ( _display_focus == true ) ? 2 : 1;

  l = ts.getMaxLineWidth() + 5;
  l += 2 * bw + 2 + ( vbar != NULL ) ? vbar->getWidth() : 0;

  if ( l >= _aguix->getRootWindowWidth() )
    l = _aguix->getRootWindowWidth();

  resize( l, getHeight() );
}

void TextView::maximizeYLines( int max_lines )
{
  int l, elementHeight;
  int bw;

  if ( font == NULL ) elementHeight = _aguix->getCharHeight();
  else elementHeight = font->getCharHeight();
  
  bw = ( _display_focus == true ) ? 2 : 1;

  l = ts.getNrOfLines();
  if ( l > max_lines ) l = max_lines;
  l = ( l + 1 ) * elementHeight;
  l += 2 * bw + 2 + ( ( hbar != NULL ) && ( line_wrap == false ) ) ? hbar->getHeight() : 0;

  if ( l >= _aguix->getRootWindowHeight() )
    l = _aguix->getRootWindowHeight();

  resize( getWidth(), l );
}

void TextView::prepareBG( bool force )
{
  if ( isCreated() == false ) return;

  _aguix->SetWindowBG( win, getBG() );
}

int TextView::getHScrollStep() const
{
  //TODO some better value
  return 8;
}
