/*
 *   Copyright (C) 2002-2004 by Jonathan Naylor G4KLX
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "FSK441Preferences.h"
#include "FSK441Frame.h"
#include "FSK441App.h"
#include "FSK441Defs.h"

#include <wx/tglbtn.h>

const int BORDER_SIZE = 5;

enum {
	Menu_File_Open = 100,
	Menu_File_Saving,
	Menu_File_Delete,
	Menu_File_Quit,
	Menu_Edit_Preferences,
	Menu_Help_About
};

BEGIN_DECLARE_EVENT_TYPES()
	DECLARE_EVENT_TYPE(LEVEL_EVENT,   2651)
	DECLARE_EVENT_TYPE(MESSAGE_EVENT, 2652)
	DECLARE_EVENT_TYPE(AUDIO_EVENT,   2653)
        DECLARE_EVENT_TYPE(ERROR_EVENT,   2654)
END_DECLARE_EVENT_TYPES()

DEFINE_EVENT_TYPE(LEVEL_EVENT)
DEFINE_EVENT_TYPE(MESSAGE_EVENT)
DEFINE_EVENT_TYPE(AUDIO_EVENT)
DEFINE_EVENT_TYPE(ERROR_EVENT)

BEGIN_EVENT_TABLE(CFSK441Frame, wxFrame)
	EVT_MENU(Menu_File_Open, CFSK441Frame::onOpen)
	EVT_MENU(Menu_File_Saving, CFSK441Frame::onSaving)
	EVT_MENU(Menu_File_Delete, CFSK441Frame::onDelete)
	EVT_MENU(Menu_File_Quit, CFSK441Frame::onQuit)
	EVT_MENU(Menu_Edit_Preferences, CFSK441Frame::onPreferences)
	EVT_MENU(Menu_Help_About, CFSK441Frame::onAbout)

	EVT_TOGGLEBUTTON(Button_Input_Record, CFSK441Frame::onRecord)
	EVT_CHOICE(Choice_TX_Order, CFSK441Frame::onTXOrder)
	EVT_CHECKBOX(Button_TX_Enable, CFSK441Frame::onTXEnable)
	EVT_BUTTON(Button_Remote, CFSK441Frame::onUpdate)

	EVT_TEXT(Text_Messages_Begin + 0, CFSK441Frame::onText)
	EVT_TEXT(Text_Messages_Begin + 1, CFSK441Frame::onText)
	EVT_TEXT(Text_Messages_Begin + 2, CFSK441Frame::onText)
	EVT_TEXT(Text_Messages_Begin + 3, CFSK441Frame::onText)
	EVT_TEXT(Text_Messages_Begin + 4, CFSK441Frame::onText)
	EVT_TEXT(Text_Messages_Begin + 5, CFSK441Frame::onText)
	EVT_TEXT(Text_Messages_Begin + 6, CFSK441Frame::onText)
	EVT_TEXT(Text_Messages_Begin + 7, CFSK441Frame::onText)
	EVT_TEXT(Text_Messages_Begin + 8, CFSK441Frame::onText)
	EVT_TEXT(Text_Messages_Begin + 9, CFSK441Frame::onText)

	EVT_RADIOBUTTON(Button_Messages_Begin + 0, CFSK441Frame::onChoice)
	EVT_RADIOBUTTON(Button_Messages_Begin + 1, CFSK441Frame::onChoice)
	EVT_RADIOBUTTON(Button_Messages_Begin + 2, CFSK441Frame::onChoice)
	EVT_RADIOBUTTON(Button_Messages_Begin + 3, CFSK441Frame::onChoice)
	EVT_RADIOBUTTON(Button_Messages_Begin + 4, CFSK441Frame::onChoice)
	EVT_RADIOBUTTON(Button_Messages_Begin + 5, CFSK441Frame::onChoice)
	EVT_RADIOBUTTON(Button_Messages_Begin + 6, CFSK441Frame::onChoice)
	EVT_RADIOBUTTON(Button_Messages_Begin + 7, CFSK441Frame::onChoice)
	EVT_RADIOBUTTON(Button_Messages_Begin + 8, CFSK441Frame::onChoice)
	EVT_RADIOBUTTON(Button_Messages_Begin + 9, CFSK441Frame::onChoice)

	EVT_CUSTOM(LEVEL_EVENT,   -1, CFSK441Frame::onLevels)
	EVT_CUSTOM(MESSAGE_EVENT, -1, CFSK441Frame::onMessage)
	EVT_CUSTOM(AUDIO_EVENT,   -1, CFSK441Frame::onAudio)
	EVT_CUSTOM(ERROR_EVENT,   -1, CFSK441Frame::onError)
END_EVENT_TABLE()

CFSK441Frame::CFSK441Frame() :
wxFrame(NULL, -1, title),
m_firstDisplay(NULL),
m_secondDisplay(NULL),
m_fileDisplay(NULL),
m_record(NULL),
m_transmit(NULL),
m_remote(NULL),
m_clock(NULL),
m_message(NULL),
m_messageNo(-1)
{
	SetMenuBar(createMenuBar());

	// The top level sizer, graph, list box and then controls
	wxBoxSizer* mainSizer = new wxBoxSizer(wxVERTICAL);

	// Spectrum graphs
	wxNotebook* noteBook = new wxNotebook(this, -1);
	wxNotebookSizer* noteSizer = new wxNotebookSizer(noteBook);

	m_firstDisplay = new CFSK441Display(noteBook);
	noteBook->AddPage(m_firstDisplay, wxT("First"), true);

	m_secondDisplay = new CFSK441Display(noteBook);
	noteBook->AddPage(m_secondDisplay, wxT("Second"), true);

	m_fileDisplay = new CFSK441Display(noteBook);
	noteBook->AddPage(m_fileDisplay, wxT("WAV Files"), false);

	noteBook->SetSelection(0);

	mainSizer->Add(noteSizer);

	// The three input sections, left two are control, right is messages
	wxBoxSizer* inputSizer = new wxBoxSizer(wxHORIZONTAL);

	// Handle the recording and transmit control boxes
	wxBoxSizer* leftSizer = new wxBoxSizer(wxVERTICAL);

	m_record = new CRecordControl(this);
	leftSizer->Add(m_record);

	bool txFirst = ::wxGetApp().getTXFirst();
	bool txEnable = ::wxGetApp().getTXEnable();
	m_transmit = new CTransmitControl(this, -1, txFirst, txEnable);
	leftSizer->Add(m_transmit);

	inputSizer->Add(leftSizer);

	wxBoxSizer* middleSizer = new wxBoxSizer(wxVERTICAL);

	m_remote = new CRemoteControl(this);
	middleSizer->Add(m_remote);

	m_clock = new CClockPanel(this, -1, wxDefaultPosition, wxSize(145, 80));
	middleSizer->Add(m_clock);

	inputSizer->Add(middleSizer);

	wxString* messages = new wxString[FSK441_MAX_MESSAGES];
	for (int i = 0; i < FSK441_MAX_MESSAGES; i++)
		messages[i] = ::wxGetApp().getReplacedMessage(i);
	m_message = new CMessageControl(this, -1, messages, FSK441_MAX_MESSAGES, FSK441_MAX_MESSAGE_LENGTH);
	inputSizer->Add(m_message);
	delete[] messages;

	m_messageNo = 0;
	wxString message = m_message->getMessage(m_messageNo);
	::wxGetApp().sendMessage(message);

	mainSizer->Add(inputSizer);

	SetSizer(mainSizer);

	mainSizer->SetSizeHints(this);
}

CFSK441Frame::~CFSK441Frame()
{
}

wxMenuBar* CFSK441Frame::createMenuBar()
{
	wxMenu* fileMenu = new wxMenu();
	fileMenu->Append(Menu_File_Open, wxT("&Open sound file\tCtrl-O"));
	fileMenu->AppendSeparator();
	fileMenu->AppendCheckItem(Menu_File_Saving, wxT("&Save received text\tCtrl-A"));
	fileMenu->Append(Menu_File_Delete, wxT("&Remove received text\tCtrl-R"));
	fileMenu->AppendSeparator();
	fileMenu->Append(Menu_File_Quit, wxT("E&xit\tAlt-X"));

	wxMenu* editMenu = new wxMenu();
	editMenu->Append(Menu_Edit_Preferences, wxT("&Preferences...\tCtrl-P"));

	wxMenu* helpMenu = new wxMenu();
	helpMenu->Append(Menu_Help_About, wxT("&About FSK441\tF1"));

	wxMenuBar* menuBar = new wxMenuBar();
	menuBar->Append(fileMenu, wxT("&File"));
	menuBar->Append(editMenu, wxT("&Edit"));
	menuBar->Append(helpMenu, wxT("&Help"));

	return menuBar;
}

void CFSK441Frame::showLevels(CFSK441Levels* levels, EWho who)
{
	wxASSERT(levels != NULL);

	CFSK441LevelEvent event(levels, who, LEVEL_EVENT);

	AddPendingEvent(event);
}

void CFSK441Frame::showMessage(CFSK441Message* message, EWho who)
{
	wxASSERT(message != NULL);

	CFSK441MessageEvent event(message, who, MESSAGE_EVENT);

	AddPendingEvent(event);
}

void CFSK441Frame::showAudio(double audio, EWho who)
{
	CLevelEvent event(audio, who, AUDIO_EVENT);

	AddPendingEvent(event);
}

void CFSK441Frame::error(const wxString& text)
{
	CErrorEvent event(text, ERROR_EVENT);

	AddPendingEvent(event);
}

void CFSK441Frame::onOpen(const wxCommandEvent& event)
{
	wxString pathAudio, pathText;
	wxGetApp().getPaths(pathAudio, pathText);

	wxFileDialog dialog(this, wxT("Open sound file"), pathAudio, wxEmptyString, wxT("WAV files (*.wav;*.WAV)|*.wav;*.WAV"));

	if (dialog.ShowModal() != wxID_OK)
		return;

	wxString fileName = dialog.GetPath();

	wxGetApp().readFile(fileName);
}

void CFSK441Frame::onSaving(const wxCommandEvent& event)
{
	bool saving = event.IsChecked();

	::wxGetApp().setSaving(saving);
}

void CFSK441Frame::onDelete(const wxCommandEvent& event)
{
	::wxGetApp().deleteMessages();
}

void CFSK441Frame::onQuit(const wxCommandEvent& event)
{
	Close(true);
}

void CFSK441Frame::onPreferences(const wxCommandEvent& event)
{
	CFSK441Preferences dialog(this);

	if (dialog.ShowModal() != wxID_OK)
		return;

	for (int i = 0; i < FSK441_MAX_MESSAGES; i++)
		m_message->setMessage(i, ::wxGetApp().getReplacedMessage(i));
}

void CFSK441Frame::onAbout(const wxCommandEvent& event)
{
	::wxMessageBox(title + wxT("\n\nWritten by Jonathan Naylor  G4KLX\n\nLots of advice from Tomi Manninen  OH2BNS"), wxT("About FSK441"), wxICON_INFORMATION);
}

void CFSK441Frame::onRecord(const wxCommandEvent& event)
{
	bool recording = event.IsChecked();

	::wxGetApp().setRecording(recording);
}

void CFSK441Frame::onTXOrder(const wxCommandEvent& event)
{
	bool txFirst = int(event.GetClientData()) == Transmit_First;

	::wxGetApp().setTXFirst(txFirst);
}

void CFSK441Frame::onTXEnable(const wxCommandEvent& event)
{
	bool txEnabled = event.IsChecked();

	::wxGetApp().setTXEnable(txEnabled);
}

void CFSK441Frame::onUpdate(const wxCommandEvent& event)
{
        wxASSERT(m_remote != NULL);
	wxASSERT(m_message != NULL);

	wxString callsign = m_remote->getCallsign();
	wxString report   = m_remote->getReport();

	::wxGetApp().setRemote(callsign, report);

	for (int i = 0; i < FSK441_MAX_MESSAGES; i++)
		m_message->setMessage(i, ::wxGetApp().getReplacedMessage(i));
}

void CFSK441Frame::onText(const wxCommandEvent& event)
{
	int messageNo = event.GetId() - Text_Messages_Begin;

	wxASSERT(messageNo >= 0 && messageNo < FSK441_MAX_MESSAGES);

	if (messageNo == m_messageNo) {
		wxString message = m_message->getMessage(m_messageNo);
		::wxGetApp().sendMessage(message);
	}
}

void CFSK441Frame::onChoice(const wxCommandEvent& event)
{
	m_messageNo = event.GetId() - Button_Messages_Begin;

	wxASSERT(m_messageNo >= 0 && m_messageNo < FSK441_MAX_MESSAGES);

	wxString message = m_message->getMessage(m_messageNo);

	::wxGetApp().sendMessage(message);
}

void CFSK441Frame::onLevels(const CFSK441LevelEvent& event)
{
	CFSK441Levels* levels = event.getLevels();
	EWho who              = event.getWho();

	wxASSERT(levels != NULL);

	switch (who) {
		case RX_FIRST:
			wxASSERT(m_firstDisplay != NULL);
			m_firstDisplay->showLevels(levels);
			break;
		case RX_SECOND:
			wxASSERT(m_secondDisplay != NULL);
			m_secondDisplay->showLevels(levels);
			break;
		case RX_FILE:
			wxASSERT(m_fileDisplay != NULL);
			m_fileDisplay->showLevels(levels);
			break;
	}

	delete levels;
}

void CFSK441Frame::onMessage(const CFSK441MessageEvent& event)
{
	CFSK441Message* message = event.getMessage();
	EWho who                = event.getWho();

	wxASSERT(message != NULL);

	switch (who) {
		case RX_FIRST:
			wxASSERT(m_firstDisplay != NULL);
			m_firstDisplay->showMessage(message);
			break;
		case RX_SECOND:
			wxASSERT(m_secondDisplay != NULL);
			m_secondDisplay->showMessage(message);
			break;
		case RX_FILE:
			wxASSERT(m_fileDisplay != NULL);
			m_fileDisplay->showMessage(message);
			break;
	}
}

void CFSK441Frame::onAudio(const CLevelEvent& event)
{
	double audio = event.getAudio();
	EWho who     = event.getWho();

	switch (who) {
		case RX_FIRST:
			wxASSERT(m_firstDisplay != NULL);
			m_firstDisplay->showAudio(audio);
			break;
		case RX_SECOND:
			wxASSERT(m_secondDisplay != NULL);
			m_secondDisplay->showAudio(audio);
			break;
		case RX_FILE:
			wxASSERT(m_fileDisplay != NULL);
			m_fileDisplay->showAudio(audio);
			break;
	}
}

void CFSK441Frame::onError(const CErrorEvent& event)
{
	wxString text = event.getText();

	wxMessageDialog dialog(this, text, wxT("FSK441 Error"), wxICON_ERROR);

	dialog.ShowModal();
}
