/*
 *   Copyright (C) 2002-2004 by Jonathan Naylor G4KLX
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <cmath>
using namespace std;

#include "common/SoundFile.h"
#include "common/Average.h"
#include "common/FFT.h"
#include "common/Exception.h"
#include "common/Hamming.h"
#include "common/Inline.h"

#include "AudioData.h"

int main(int argc, char **argv)
{
	if (argc < 6) {
		::fprintf(stderr, "Usage: AudioData <fft length> <start bin> <end bin> <sample rate> <filename>\n");
		return 1;
	}

	int     fftLength = ::atoi(argv[1]);
	int      startBin = ::atoi(argv[2]);
	int        endBin = ::atoi(argv[3]);
	int    sampleRate = ::atoi(argv[4]);
	wxString fileName = wxString(argv[5]);

	try {
		CAudioData data(fftLength, startBin, endBin, sampleRate, fileName);
		data.run();
	}
	catch (CException& ex) {
		fprintf(stderr, "AudioData: %s\n", ex.getMessage().c_str());
		return 1;
	}

	return 0;
}

CAudioData::CAudioData(int fftLength, int startBin, int endBin, int sampleRate, const wxString& fileName) :
m_fftLength(fftLength),
m_startBin(startBin),
m_endBin(endBin),
m_sampleRate(sampleRate),
m_fileName(fileName)
{
}

CAudioData::~CAudioData()
{
}

void CAudioData::run()
{
	CAverage amplitude;

	CFFT fft(m_fftLength);
	CHamming hamming(m_fftLength);

	complex<double>* bins = new complex<double>[m_fftLength];

	CSoundFile file(m_fileName, m_sampleRate);

	file.openRead();

	double* in = new double[30 * m_sampleRate];
	double* out = new double[m_fftLength];

	int step = int(double(m_fftLength) / 2.0 + 0.5);

	int n = 0;

	for (;;) {
		int len = m_fftLength;
		file.read(in + n, len);

		if (len <= 0)
			break;

		for (int i = 0; i < len; i += step) {
			amplitude.addValue(in[n + i] * in[n + i]);

			if (n >= m_fftLength) {
				int offset = n + i - m_fftLength;

				hamming.filter(in + offset, out);
				fft.process(out, bins);

				::fprintf(stderr, "%05d: ", offset);

				for (int j = m_startBin; j <= m_endBin; j++) {
					double  ampl = ::CABS(bins[j]);
					double phase = ::CPHASE(bins[j]) / M_PI;

					::fprintf(stderr, "%.3f/%.3f  ", ampl, phase);
				}

				::fprintf(stderr, "\n");
			}
		}

		n += len;
	}

	file.close();

	::fprintf(stderr, "Audio Amplitude: %d samples, min/ave/max: %.2f/%.2f/%.2f\n",
		amplitude.getCount(),
		::sqrt(amplitude.getMinimum()),
		::sqrt(amplitude.getAverage()),
		::sqrt(amplitude.getMaximum()));

	delete[] bins;
	delete[] in;
	delete[] out;
}
