/*
 *	xeno_color.h:
 *		XenoColor			Deep color format
 *		XenoImage			Remappable image representation
 *		XenoPixmap/Mask/	Pixmap abstraction, for rendering
 *		XenoImageBuffer		Deep image buffer, for working in
 */

#ifndef XENO_COLOR_H
#define XENO_COLOR_H

#include "xeno_theme.h"

/*
 *	XenoColor.
 *		Deep color format. May change color space in the future (as it has in the past)
 *		so use appropriate access functions.
 */

#define XENO_COLOR(col)		((XenoColor *)col)

typedef struct {
	gfloat	v[3];
} XenoColor;

extern const XenoColor xeno_color_black;
extern const XenoColor xeno_color_white;

void	xeno_color_to_gdk	(const XenoColor *color, GdkColor *dst);
void	xeno_color_init		(XenoColor *color, gfloat red, gfloat green, gfloat blue);

#define xeno_color_from_gdk(xeno_dst, gdk_src) \
	{ \
		const static gfloat f = 1.0/65535.0; \
		xeno_color_init (xeno_dst, f * (gdk_src)->red, f * (gdk_src)->green, f * (gdk_src)->blue); \
	}

#define xeno_color_blend(a, b, k, dst) \
	{ \
		(dst)->v[0] = LERP((a)->v[0], (b)->v[0], (k)); \
		(dst)->v[1] = LERP((a)->v[1], (b)->v[1], (k)); \
		(dst)->v[2] = LERP((a)->v[2], (b)->v[2], (k)); \
	}

#define xeno_color_add(a, b, dst) \
	{ \
		(dst)->v[0] = (b)->v[0] + (a)->v[0]; \
		(dst)->v[1] = (b)->v[1] + (a)->v[1]; \
		(dst)->v[2] = (b)->v[2] + (a)->v[2]; \
	}

#define xeno_color_combine(a, b, k, dst) \
	{ \
		(dst)->v[0] = (k) * (b)->v[0] + (a)->v[0]; \
		(dst)->v[1] = (k) * (b)->v[1] + (a)->v[1]; \
		(dst)->v[2] = (k) * (b)->v[2] + (a)->v[2]; \
	}

#define xeno_color_flip(a, b, c, d) \
	{ \
		(d)->v[0] = (a)->v[0] - ((b)->v[0] - (c)->v[0]); \
		(d)->v[1] = (a)->v[1] - ((b)->v[1] - (c)->v[1]); \
		(d)->v[2] = (a)->v[2] - ((b)->v[2] - (c)->v[2]); \
	}

void	xeno_color_shade	(const XenoColor	*src,
							 gfloat				k,
							 XenoColor			*dst);

#if 0
void	xeno_color_gamma	(const XenoColor	*src,
							 XenoColor			*dst);
void	xeno_color_linear	(const XenoColor	*src,
							 XenoColor			*dst);
#else
#define xeno_color_gamma(src, dst)		(*(dst) = *(src))
#define	xeno_color_linear(src, dst)		(*(dst) = *(src))
#endif


extern const guchar xeno_dither_table[16][16];

#define xeno_dither(x, y)	(xeno_dither_table[x & 0x0f][y & 0x0f] * (1.0/255.0))

void	xeno_color_dither	(const XenoColor	*src,
							 const GdkVisual	*visual,
							 gint				x,
							 gint				y,
							 GdkColor			*dst);

void	xeno_color_from_pixmap	(XenoColor	*color,
								 GdkPixmap	*pixmap);
								

typedef struct {
	gfloat	v[4];
} XenoTexel;

#define xeno_texel_init(dst, red, green, blue, alfa) \
	{ \
		XenoTexel *d = (dst); \
		xeno_color_init(XENO_COLOR(d), red, green, blue)); \
		d->v[3] = alfa; \
	}

#define xeno_texel_clear(dst) \
	{ \
		XenoTexel *d = (dst); \
		(d)->v[0] = 0.0F; \
		(d)->v[1] = 0.0F; \
		(d)->v[2] = 0.0F; \
		(d)->v[3] = 0.0F; \
	}


/*
 *	XenoPixmap:
 *		Abstraction, makes it easier to change implementation if need be.
 */

typedef struct _XenoPixmap		XenoPixmap;		/* opaque */
typedef struct _XenoPixmapMask	XenoPixmapMask;	/* opaque */

void	xeno_pixmap_unref		(XenoPixmap		*pixmap);
void	xeno_pixmap_get_size	(XenoPixmap		*pixmap,
								 gint			*width_p,
								 gint			*height_p);

void	xeno_pixmap_mask_unref	(XenoPixmapMask	*mask);

void	xeno_draw_pixmap		(GdkDrawable	*drawable,
								 GdkGC			*gc,
								 GdkRectangle	*area,
								 XenoPixmap		*pixmap,
								 XenoPixmapMask	*mask,
								 gint			xsrc,
								 gint			ysrc,
								 gint			xdest,
								 gint			ydest,
								 gint			width,
								 gint			height);

GdkPixmap *	xeno_pixmap_get_gdk_pixmap (XenoPixmap *pixmap); /* remember to unref afterwards */


/*
 *	XenoImageBuffer:
 *		Because GdkRGB doesnt cut it.
 */

typedef struct {
	XenoTexel	*data;
	
	guint16		width;
	guint16		height;
} XenoImageBuffer;

/* dynamic allocation */
XenoImageBuffer	* xeno_image_buffer_new			(guint16			width,
												 guint16	 		height);
void			  xeno_image_buffer_destroy		(XenoImageBuffer	*buffer);

/* static allocation */
void			  xeno_image_buffer_init		(XenoImageBuffer	*buffer,
												 guint16			width,
												 guint16			height);
void			  xeno_image_buffer_finalize	(XenoImageBuffer	*buffer);

/* convert into format for drawing */
XenoPixmap *	  xeno_image_buffer_render		(const XenoImageBuffer	*buffer,
												 const XenoColor		*bg_color);

XenoPixmapMask *  xeno_image_buffer_render_mask	(const XenoImageBuffer	*buffer);


/*
 *	XenoImage:
 *		A description of how to compose images out of style colors.
 */

typedef struct {
	const guchar			*alpha;
	const guchar			*bitmap;
	guchar					x, y, width, height;
	guchar					pen, modifier;
} XenoImageLayer;

typedef struct {
	guchar					pen, modifier;
} XenoImagePen;

typedef struct {
	const XenoImageLayer	*layers;
	guchar					n_layers;
	guchar					width, height;
} XenoImage;

typedef void (* XenoPenCallback)(guint pen, guint modifier, XenoColor *color, gpointer user_data);

void	xeno_image_render	(const XenoImage	*image,
							 XenoImageBuffer	*image_buffer,
							 guint16			x,
							 guint16			y,
							 XenoPenCallback	pen_callback,
							 gpointer			user_data);


/* Misc */

#endif /* XENO_COLOR_H */

