/*
 * Copyright (C) 2002-5 Edscott Wilson Garcia
 * EMail: edscott@imp.mx
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif
#ifdef HAVE_SYS_WAIT_H
#include <sys/wait.h>
#endif
#ifdef HAVE_SYS_PARAM_H
#include <sys/param.h>
#endif
#ifdef HAVE_SYS_UCRED_H
#include <sys/ucred.h>
#endif
#ifdef HAVE_SYS_MOUNT_H
#include <sys/mount.h>
#endif

#include <stdio.h>
#include <errno.h>
#ifdef HAVE_FSTAB_H
#include <fstab.h>
#endif
#include <limits.h>
#ifdef HAVE_MEMORY_H
#include <memory.h>
#endif
#include <stdlib.h>
#include <signal.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>

#include "constants.h"
#include "types.h"

#include "primary.h"
#include "gui.h"
#include "actions_lib.h"

/* this should be first 2 lines after headers: */
G_MODULE_EXPORT
LIBXFFM_MODULE

static
gchar * 
get_desktop_string(const gchar *string, const gchar *file);

G_MODULE_EXPORT
const gchar * 
g_module_check_init(GModule *module){
#ifdef ENABLE_NLS
    bindtextdomain (GETTEXT_PACKAGE,PACKAGE_LOCALE_DIR);
#ifdef HAVE_BIND_TEXTDOMAIN_CODESET
    bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
#endif
    TRACE("domain=%s", GETTEXT_PACKAGE);
#endif
    return NULL;
}


/****************   void  functions  ***********************/

G_MODULE_EXPORT
const gchar *
module_name(void){
    return "xffm_apps";
}
G_MODULE_EXPORT
const gchar *
submodule_dir(void){
    return "plugins";
}

G_MODULE_EXPORT
void *
is_root_module(void){
    return GINT_TO_POINTER(1);
}

G_MODULE_EXPORT
void *
module_init(void){
    return NULL;
}

G_MODULE_EXPORT
void *
exec_name(void){
    return "xffm-applications";
}

G_MODULE_EXPORT
void *
plugin_info(void){
    return _("The APPLICATIONS plugin allows you to locate, run, modify and copy applications for which a .desktop file has been installed in the default system location.");
}

void *entry_tip(void *p){
    record_entry_t *en=p;
    static gchar *tip=NULL;
    gchar *comment;
    gchar *base;
    if (!en || !en->path ) return NULL;
    comment=get_desktop_string("Comment",en->path);
    if (!comment) return NULL;
    g_free(tip);
    base=g_path_get_basename(en->path);
    tip=g_strdup_printf("%s\n(%s)\n",my_utf_string(comment),base);
/*    tip=g_strdup_printf("%s:%s",comment,path_info(p));*/
    g_free(comment);
    g_free(base);
    return (tip);
}

/*******************  natural functions **********************/

G_MODULE_EXPORT
void *
get_dnd_path(void *p)
{
    static gchar *desktop=NULL;
    if (!desktop) {
	desktop=g_strdup_printf("%s%c%s%c%s",
		PACKAGE_DATA_DIR,G_DIR_SEPARATOR,
		"applications",G_DIR_SEPARATOR,
		"Xffm-applications.desktop");
    }
    return (void *)desktop;
}

G_MODULE_EXPORT
void *
module_icon_id(void * p){
    return "xffm-applications.png";
}

G_MODULE_EXPORT
void *
module_label(void * p){
    return _("Applications");
}

G_MODULE_EXPORT
void *
is_selectable(void *p){
    record_entry_t *en=(record_entry_t *)p;
    if (g_file_test(en->path, G_FILE_TEST_EXISTS))
	return (void *)"Yes";
    
    return NULL;
}


G_MODULE_EXPORT
void *
valid_drop_site(void *p){
    return NULL;
}


/*******************  rational functions **********************/
// FIXME: this static function is repeated in gui/deskview/deskview_desktop.i,
// so it could be move to secondary_lib...
static
gchar * 
get_desktop_string(const gchar *key, const gchar *file){
	GKeyFile *key_file;
	GError *error=NULL;
	gchar *value;
	key_file=g_key_file_new ();
	if (!g_key_file_load_from_file (key_file, file, 0, &error)){
		g_warning("%s (%s)",error->message,file);
		g_error_free(error);
		return NULL;
	}
	value=g_key_file_get_locale_string    (key_file,"Desktop Entry",key,NULL,&error);
	if (error){
		g_warning("%s (%s)",error->message,file);
		g_error_free(error);
	}
	g_key_file_free (key_file);
	return value;
}

static
int 
get_desktop_bool(const gchar *key, const gchar *file){
	GKeyFile *key_file;
	GError *error=NULL;
	gboolean value=FALSE;
	key_file=g_key_file_new ();
	if (!g_key_file_load_from_file (key_file, file, 0, &error)){
		g_warning("%s (%s)",error->message,file);
		g_error_free(error);
		return FALSE;
	}
	value=g_key_file_get_boolean(key_file,"Desktop Entry",key,&error);
	if (error){
		g_warning("%s (%s)",error->message,file);
		g_error_free(error);
	}
	g_key_file_free (key_file);
	return value;
}



static 
xfdir_t 
*private_get_xfdir(const gchar *appdir){
    static xfdir_t *xfdir_p;
    int i;
    record_entry_t *en;

    xfdir_p =(xfdir_t *)malloc(sizeof(xfdir_t));
    
    if (!g_file_test(appdir,G_FILE_TEST_EXISTS)) {
	g_warning("%s does not exist",appdir);
	xfdir_p->pathc=0;
	return xfdir_p;
    }
    
    en=stat_entry(appdir,0);
    SET_ROOT_TYPE(en->type);
    get_xfdir_local(xfdir_p, en, "\\.desktop$");
    /* set icon and exec */
    for (i=0; i<xfdir_p->pathc; i++){
        gchar *name;
	if (g_file_test(xfdir_p->gl[i].en->path,G_FILE_TEST_IS_DIR)) continue;
	name=get_desktop_string("Name", xfdir_p->gl[i].en->path);
        if (name) {
	    g_free(xfdir_p->gl[i].pathv);
	    xfdir_p->gl[i].pathv=name;
	}
	xfdir_p->gl[i].en->exec=get_desktop_string("Exec", xfdir_p->gl[i].en->path);
	xfdir_p->gl[i].en->icon=get_desktop_string("Icon", xfdir_p->gl[i].en->path);
	xfdir_p->gl[i].en->terminal=get_desktop_bool("Terminal", xfdir_p->gl[i].en->path);
	if (xfdir_p->gl[i].en->icon && !ICON_check4icon_path(xfdir_p->gl[i].en->icon)) 
	{
	    g_warning("icon not found: %s",xfdir_p->gl[i].en->icon);
	    xfdir_p->gl[i].en->icon=g_strdup("xffm/default");
	} 
    }
    return(xfdir_p);
}

void *
get_xfdir(void *p, void *q){
    static xfdir_t xfdir;
    xfdir_t *xfdir_p[]={NULL,NULL};
    int length[]={0,0};
    /*record_entry_t *en=(record_entry_t *)p;*/
    xfdir.pathc=0;
#ifdef HAVE__USR_SHARE_APPLICATIONS
    xfdir_p[0] = private_get_xfdir("/usr/share/applications");
    length[0] = (xfdir_p[0] && xfdir_p[0]->pathc)?xfdir_p[0]->pathc:0;
#endif 
#ifdef HAVE__USR_LOCAL_SHARE_APPLICATIONS
    xfdir_p[1] = private_get_xfdir("/usr/local/share/applications");
    length[1] = ((xfdir_p[1] && xfdir_p[1]->pathc)?xfdir_p[1]->pathc:0);
#endif
    
    xfdir.pathc = length[0] +  length[1];
    if (!xfdir.pathc) {
	if (xfdir_p[0]) free(xfdir_p[0]);
	if (xfdir_p[1]) free(xfdir_p[1]);
	return (&xfdir.pathc);
    }
    
    xfdir.gl = (dir_t *)malloc(xfdir.pathc * sizeof(dir_t));
    /* concatenate now */
    if (xfdir_p[0] && length[0]){
	memcpy(xfdir.gl,xfdir_p[0]->gl,sizeof(dir_t)*length[0]);
	free(xfdir_p[0]);
    }
    if (xfdir_p[1] && length[1]){
	memcpy(xfdir.gl+length[0],xfdir_p[1]->gl,sizeof(dir_t)*length[1]);
	free(xfdir_p[1]);
    }
    return(&xfdir);
}


