// *************************************************************************
// * Xgsm - mobile phone manager
// *
// * File:    xgsm_device.h
// *
// * Purpose: Device implementation
// *
// * Author:  Peter Hofmann (software@pxh.de)
// *
// * Created: 18.6.2000
// *************************************************************************

#ifndef XGSM_DEVICE_H
#define XGSM_DEVICE_H

#include <gtk--/frame.h>
#include <sigc++/signal_system.h>
#include <gsmlib/gsm_util.h>
#include <gsmlib/gsm_error.h>
#include <gsmlib/gsm_me_ta.h>
#include <gsmlib/gsm_sorted_phonebook.h>
#include <gsmlib/gsm_sorted_sms_store.h>
#include <gsmlib/gsm_sms.h>
#include <pthread.h>
#include <string>

// constants
#define PHONEBOOK_FILE_EXTENSION ".pb"
#define SMSSTORE_FILE_EXTENSION  ".sms"


namespace Xgsm
{

  // forward declarations
  class Request;
  typedef gsmlib::Ref<Request> RequestRef;
  class Response;
  typedef gsmlib::Ref<Response> ResponseRef;
  class Device;
  typedef gsmlib::Ref<Device> DeviceRef;
  class OpenResponse;

  // Device data source
  enum DeviceDataSource {SMSFile, PhonebookFile, MeTaDevice};

  // entries for phonebooks and SMS stores

  class Entry : public gsmlib::RefBase
  {
    // phonebook entry
    gsmlib::Ref<gsmlib::PhonebookEntryBase> _phonebookEntry;
    gsmlib::SortedPhonebookIterator _phonebookIterator;

    // SMS store entry
    gsmlib::Ref<gsmlib::SMSStoreEntry> _smsStoreEntry;
    gsmlib::SortedSMSStoreIterator _smsStoreIterator;

  public:
    // constructors are called by the device worker threads
    Entry(gsmlib::SortedPhonebookIterator entryIterator);
    Entry(gsmlib::SortedSMSStoreIterator entryIterator);
    Entry(gsmlib::PhonebookEntryBase &phonebookEntry);
    Entry(gsmlib::SMSStoreEntry &smsStoreEntry);

    // phonebook entry section
    string pbText() const {return _phonebookEntry->text();}
    string pbTelephone() const {return _phonebookEntry->telephone();}
    int pbIndex() const {return _phonebookEntry->index();}
    gsmlib::SortedPhonebookIterator phonebookIterator() const
      {return _phonebookIterator;}
    void setPbText(string text);
    void setPbTelephone(string telephone);
    operator gsmlib::PhonebookEntryBase() {return _phonebookEntry();}

    // SMS store entry section
    string smsText() const;
    string smsTelephone() const;
    string smsDate() const;     // date in string format
    string smsType() const;     // type as single character ("D", "S", "R")
    int smsIndex() const {return _smsStoreEntry->index();}
    gsmlib::SortedSMSStoreIterator smsStoreIterator() const
      { return _smsStoreIterator;}
    void setSMSMessage(string telephone, string message,
                       gsmlib::SMSMessage::MessageType type =
                       gsmlib::SMSMessage::SMS_SUBMIT);
    operator gsmlib::SMSStoreEntry() {return _smsStoreEntry();}
  };

  class EntryList : public vector<Entry>, public gsmlib::RefBase
  {
  };

  typedef gsmlib::Ref<EntryList> EntryListRef;

  // *** callbacks

  // callback and signal for request completion
  // the callbacks gets passed the response
  typedef SigC::Slot1<void, ResponseRef> CompletionCallback;
  typedef SigC::Signal1<void, ResponseRef> CompletionSignal;

  // callback for device close
  enum EventType {CloseEvent, IdleEvent, BusyEvent};

  typedef SigC::Slot1<void, EventType> EventCallback;

  // *** requests and response structures

  // base classes

  class Request : public gsmlib::RefBase
  {
    int _requestId;
    static int _nextRequestId;
    CompletionSignal _signal;
    SigC::Connection _completionConnection;

  public:
    Request(const CompletionCallback &callback);

    // return ID of this request
    int requestId() const {return _requestId;}

    // return completion signal
    CompletionSignal &signal() {return _signal;}

    // return connection to completion signal
    SigC::Connection &connection() {return _completionConnection;}

    virtual ~Request() {}
  };

  class Response : public gsmlib::RefBase
  {
    RequestRef _request;        // corresponding request object

  public:
    Response(RequestRef &request) : _request(request) {}

    // return ID of corresponding request
    int requestId() const {return _request->requestId();}

    // return corresponding request
    RequestRef request() {return _request;}

    virtual ~Response() {}
  };

  // open device or file

  class OpenRequest : public Request
  {
    string _deviceName;
    string _baudRate;
    string _initString;
    bool _swHandshake;
    string _fileName;
    DeviceDataSource _dataSource;
    EventCallback _eventCallback;
    SigC::Connection _eventConnection;

  public:
    // open request for devices
    // may be called several times to get more info about the device
    OpenRequest(string deviceName, string baudRate,
                string initString, bool swHandshake,
                const EventCallback &eventCallback,
                const CompletionCallback &callback) :
      Request(callback),
      _deviceName(deviceName), _baudRate(baudRate),
      _initString(initString), _swHandshake(swHandshake),
      _dataSource(MeTaDevice),
      _eventCallback(eventCallback) {}

    // open request for files
    // this can be called with an empty filename to find out the
    // names of the files in the ~/.xgsm directory
    OpenRequest(string fileName,
                DeviceDataSource dataSource,
                const EventCallback &eventCallback,
                const CompletionCallback &callback) :
      Request(callback),
      _fileName(fileName),
      _dataSource(dataSource),
      _eventCallback(eventCallback) {}

    virtual ~OpenRequest() {}

    friend class Device;
    friend class OpenResponse;
  };

  class OpenResponse : public Response
  {
    SigC::Connection _eventConnection;
    DeviceRef _device;
    vector<string> _phonebookNames, _SMSStoreNames;

  public:
    OpenResponse(RequestRef &request, SigC::Connection eventConnection,
                 DeviceRef device) :
      Response(request), _eventConnection(eventConnection), _device(device) {}

    // return device that was just opened
    DeviceRef getDevice() {return _device;}

    // return close callback connection
    SigC::Connection eventConnection() {return _eventConnection;}

    // return phonebook and SMS store names
    vector<string> phonebookNames() {return _phonebookNames;}
    vector<string> SMSStoreNames() {return _SMSStoreNames;}

    virtual ~OpenResponse() {}

    friend class Device;
  };

  // get device information
  class DeviceInfoRequest : public Request
  {
  public:
    // information parameters
    enum DeviceInfo {MeInfo =                     1 << 0, // must be first!
                     OperatorInfo =               1 << 1,
                     CurrentOperatorInfo =        1 << 2,
                     // FacilityLockStateInfo =      1 << 3, // Not
                     // FacilityLockCapabilityInfo = 1 << 4, // implemented
                     // PasswordInfo =               1 << 5,
                     CLIPInfo =                   1 << 3,
                     CallForwardingInfo =         1 << 4,
                     BatteryInfo =                1 << 5,
                     BitErrorInfo =               1 << 6,
                     SCAInfo =                    1 << 7,
                     CharSetInfo =                1 << 8,
                     SignalInfo =                 1 << 9}; // must be last!

  private:
    DeviceInfo _deviceInfo;     // bit mask of device information requests
    vector<gsmlib::ForwardReason> _forwardReasons;

  public:
    DeviceInfoRequest(const DeviceInfo deviceInfo,
                      const CompletionCallback &callback) :
      Request(callback), _deviceInfo(deviceInfo) {}

    DeviceInfoRequest(const DeviceInfo deviceInfo,
                      vector<gsmlib::ForwardReason> &forwardReasons,
                      const CompletionCallback &callback) :
      Request(callback),
      _deviceInfo(deviceInfo), _forwardReasons(forwardReasons) {}

    // info functions for use of DeviceInfoResponse
    DeviceInfo deviceInfo() const {return _deviceInfo;}
    vector<gsmlib::ForwardReason> forwardReasons() const
      {return _forwardReasons;}
  };

  class DeviceInfoResponse : public Response
  {
  public:
    struct ForwardInfoTriple
    {
      gsmlib::ForwardInfo _voice, _fax, _data;
    };

  private:
    gsmlib::MEInfo _meInfo;                       // MeInfo
    vector<gsmlib::OPInfo> _operatorInfo;         // OperatorInfo
    gsmlib::OPInfo _currentOperatorInfo;          // CurrentOperatorInfo
    bool _clipInfo;                               // CLIPInfo
    vector <ForwardInfoTriple> _forwardingInfo;   // CallForwardingInfo
    int _batteryInfo;                             // BatteryInfo
    int _bitErrorInfo;                            // BitErrorInfo
    string _serviceCentreInfo;                    // SCAInfo
    vector<string> _charSetInfo;                  // CharSetInfo
    int _signalStrengthInfo;                      // SignalInfo
    // information actually collected
    DeviceInfoRequest::DeviceInfo _deviceInfo;

  public:
    DeviceInfoResponse(RequestRef request) :
      Response(request), _deviceInfo((DeviceInfoRequest::DeviceInfo)0) {}

    // the following information is only valid if the corresponding
    // deviceInfo bit was set in the request
    // FacilityLockStateInfo, FacilityLockCapabilityInfo, PasswordInfo
    // are not implemented
    gsmlib::MEInfo getMeInfo();                   // MeInfo
    vector<gsmlib::OPInfo> getAvailableOPInfo();  // OperatorInfo
    gsmlib::OPInfo getCurrentOPInfo();            // CurrentOperatorInfo
    bool getNetworkCLIP();                        // CLIPInfo
                                                  // CallForwardingInfo
    ForwardInfoTriple getCallForwardInfo(gsmlib::ForwardReason reason);
    int getBatteryChargeStatus();                 // BatteryInfo
    int getBitErrorRate();                        // BitErrorInfo
    string getServiceCentreAddress();             // SCAInfo
    vector<string> getSupportedCharSets();        // CharSetInfo
    int getSignalStrength();                      // SignalInfo

    friend class Device;
  };

  // special error response

  class ErrorResponse : public Response
  {
    gsmlib::GsmException _exception;
    string _sourceName;
    SigC::Connection _eventConnection;

  public:
    ErrorResponse(RequestRef request, gsmlib::GsmException exception,
                  string sourceName) :
      Response(request), _exception(exception), _sourceName(sourceName) {}

    gsmlib::GsmException exception() {return _exception;}
    string sourceName() const {return _sourceName;}

    virtual ~ErrorResponse() {}
  };

  // delete entries from phonebook

  class DeletePbEntriesRequest : public Request
  {
    string _phonebookName;
    vector<gsmlib::SortedPhonebookIterator> _entryIterators;

  public:
    DeletePbEntriesRequest(string phonebookName,
                           const CompletionCallback &callback) :
      Request(callback), _phonebookName(phonebookName) {}

    string phonebookName() {return _phonebookName;}
    vector<gsmlib::SortedPhonebookIterator> &entryIterators()
      {return _entryIterators;}

    // add entry to be deleted
    void addEntry(Entry &entry)
      {_entryIterators.push_back(entry.phonebookIterator());}
  };

  class DeletePbEntriesResponse : public Response
  {
  public:
    DeletePbEntriesResponse(RequestRef request) :
      Response(request) {}
  };

  // update entry in phonebook

  class UpdatePbEntryRequest : public Request
  {
    string _phonebookName;
    Entry _entry;

  public:
    UpdatePbEntryRequest(string phonebookName,
                         Entry &entry,
                         const CompletionCallback &callback) :
      Request(callback), _phonebookName(phonebookName), _entry(entry) {}

    string phonebookName() {return _phonebookName;}
    Entry &entry() {return _entry;}
  };

  class UpdatePbEntryResponse : public Response
  {
  public:
    UpdatePbEntryResponse(RequestRef request) :
      Response(request) {}
  };

  // add entry to phonebook

  class AddPbEntryRequest : public Request
  {
  public:
    enum Operation {Add, Synchronize};

  private:
    Operation _operation;
    string _phonebookName;
    EntryListRef _newEntries;

  public:
    AddPbEntryRequest(Operation operation, string phonebookName,
                      EntryListRef &newEntries,
                      const CompletionCallback &callback) :
      Request(callback), _operation(operation), _phonebookName(phonebookName),
      _newEntries(newEntries) {}

    string phonebookName() const {return _phonebookName;}
    EntryListRef newEntries() {return _newEntries;}
    Operation operation() {return _operation;}
  };

  class AddPbEntryResponse : public Response
  {
  public:
    AddPbEntryResponse(RequestRef request) :
      Response(request) {}
  };

  // get all entries from phonebook

  class GetPbEntriesRequest : public Request
  {
    string _phonebookName;
    gsmlib::SortOrder _sortOrder;

  public:
    GetPbEntriesRequest(string phonebookName, gsmlib::SortOrder sortOrder,
                        const CompletionCallback &callback) :
      Request(callback), _phonebookName(phonebookName),
      _sortOrder(sortOrder) {}

    string phonebookName() const {return _phonebookName;}
    gsmlib::SortOrder sortOrder() const {return _sortOrder;}
  };

  class GetPbEntriesResponse : public Response
  {
    EntryListRef _entries;
    unsigned int _maxTextLength, _maxTelephoneLength;

  public:
    GetPbEntriesResponse(RequestRef request, EntryListRef &entries,
                         unsigned int maxTextLength,
                         unsigned int maxTelephoneLength) :
      Response(request), _entries(entries),
      _maxTextLength(maxTextLength), _maxTelephoneLength(maxTelephoneLength) {}

    // return list of entries
    EntryListRef entries() {return _entries;}

    // return maximum sizes for text and telephone fields
    unsigned int maxTextLength() const {return _maxTextLength;}
    unsigned int maxTelephoneLength() const {return _maxTelephoneLength;}
  };

  // get all entries from SMS store

  class GetSMSEntriesRequest : public Request
  {
    string _SMSStoreName;
    gsmlib::SortOrder _sortOrder;

  public:
    GetSMSEntriesRequest(string SMSStoreName, gsmlib::SortOrder sortOrder,
                        const CompletionCallback &callback) :
      Request(callback), _SMSStoreName(SMSStoreName),
      _sortOrder(sortOrder) {}

    string SMSStoreName() const {return _SMSStoreName;}
    gsmlib::SortOrder sortOrder() const {return _sortOrder;}
  };

  class GetSMSEntriesResponse : public Response
  {
    EntryListRef _entries;

  public:
    GetSMSEntriesResponse(RequestRef request, EntryListRef &entries) :
      Response(request), _entries(entries) {}

    // return list of entries
    EntryListRef entries() {return _entries;}
  };

  // delete entries from SMS store

  class DeleteSmsEntriesRequest : public Request
  {
    string _smsStoreName;
    vector<gsmlib::SortedSMSStoreIterator> _entryIterators;

  public:
    DeleteSmsEntriesRequest(string smsStoreName,
                            const CompletionCallback &callback) :
      Request(callback), _smsStoreName(smsStoreName) {}

    string smsStoreName() {return _smsStoreName;}
    vector<gsmlib::SortedSMSStoreIterator> &entryIterators()
      {return _entryIterators;}

    // add entry to be deleted
    void addEntry(Entry &entry)
      {_entryIterators.push_back(entry.smsStoreIterator());}
  };

  class DeleteSmsEntriesResponse : public Response
  {
  public:
    DeleteSmsEntriesResponse(RequestRef request) :
      Response(request) {}
  };

  // add entry to SMS store

  class AddSmsEntryRequest : public Request
  {
  public:
    enum Operation {Add, Synchronize, Backup};

  private:
    Operation _operation;
    string _smsStoreName;
    EntryListRef _newEntries;

  public:
    AddSmsEntryRequest(Operation operation, string smsStoreName,
                       EntryListRef &newEntries,
                       const CompletionCallback &callback) :
      Request(callback), _operation(operation), _smsStoreName(smsStoreName),
      _newEntries(newEntries) {}

    string smsStoreName() const {return _smsStoreName;}
    EntryListRef newEntries() {return _newEntries;}
    Operation operation() {return _operation;}
  };

  class AddSmsEntryResponse : public Response
  {
  public:
    AddSmsEntryResponse(RequestRef request) :
      Response(request) {}
  };

  // send SMS

  class SendSMSRequest : public Request
  {
    gsmlib::Ref<gsmlib::SMSSubmitMessage> _submitSMS;

  public:
    SendSMSRequest(gsmlib::Ref<gsmlib::SMSSubmitMessage> submitSMS,
                   const CompletionCallback &callback) :
      Request(callback), _submitSMS(submitSMS) {}

    gsmlib::Ref<gsmlib::SMSSubmitMessage> submitSMS() {return _submitSMS;}
  };

  class SendSMSResponse : public Response
  {
  public:
    SendSMSResponse(RequestRef request) :
      Response(request) {}
  };

  // class to handle interrupting gsmlib
  class GsmlibInterrupt : public gsmlib::InterruptBase
  {
    static pthread_mutex_t _interruptedMtx;
    static pthread_key_t _interruptedKey;

  public:
    GsmlibInterrupt();

    // inherited, return true if current thread's Device::_interrupted == true
    bool interrupted();

    // register current thread's Device::_interrupted (called by worker thread)
    void registerInterrupted(bool &interrupted);

    // interrupt (called by GUI thread)
    void interrupt(bool &interrupted);

    friend class Device;        // may use the mutex for Device::_interrupted
  };

  // class to handle gsmlib progress reports

  class GsmlibProgress : public gsmlib::ProgressBase
  {
    struct ProgressInfo
    {
      int *_progressId;
      int _total, _part;
      ProgressInfo(int *progressId) :
        _progressId(progressId), _total(-1), _part(-1) {}
    };

    static pthread_key_t _progressKey;

  public:
    GsmlibProgress();

    // inherited, report progress
    void reportProgress(int part, int total);

    // register progress id
    void registerProgressId(int &progressId);

    // unregister
    void unregisterProgressId();

    friend class Device;        // may use the mutex for Device::_interrupted
  };

  // device abstraction

  class Device : public gsmlib::RefBase, public SigC::Object
  {
  private:
    // --- GUI thread
    string _name;               // name of this device
    SigC::Signal1<void, EventType> _eventSignal; // signal for device events
    SigC::Connection _idle;     // connection to the idle() function
    bool _busy;                 // true if busy at the moment
    vector<string> _phonebookNames; // "cache" - in case
    vector<string> _SMSStoreNames; // of multiple OpenRequests

    // variables for communication with the worker thread
    RequestRef _request;        // currently processed request
    ResponseRef _response;      // response for the _request
    pthread_mutex_t _requestMtx; // protects _request and _response
    pthread_cond_t _newRequestCond; // signals the workers thread a new req
    pthread_t _workerThread;    // thread that accesses the device
    bool _interrupted;          // set to true by interrupt() call
    bool _terminate;            // set to true if thread should terminate

    // idle function, periodically checks for worker thread activity
    // (such as available responses)
    gint idle();

    // check whether file exists, create if not existing in XGSM_DEFAULT_DIR
    // return absolute filename or "" if not found
    string fileExists(string filename, string extension);

    // find out phonebook names (.pb/.sms files existing in xgsm default dir)
    vector<string> getStoreNames(string extension);

    // --- worker thread
    // worker thread methods, these run in the context of _workerThread
    static void *runDevice(Device *device);
    void run();

    // return phonebook given the name
    gsmlib::SortedPhonebookRef getPhonebook(string phonebookName);

    // return SMS store given the name
    gsmlib::SortedSMSStoreRef getSmsStore(string smsStoreName);

    // open ME/TA if not already done
    void openMeTa();

    // worker thread variables (may be used if worker is not busy())
    string _sourceName;         // filename or device name
    DeviceDataSource _dataSource;
    gsmlib::SortedPhonebookRef _filePhonebook;
    gsmlib::SortedSMSStoreRef _fileSMSStore;
    map<string, gsmlib::SortedPhonebookRef> _metaPhonebooks;
    map<string, gsmlib::SortedSMSStoreRef> _metaSMSStores;
    gsmlib::SortedSMSStoreRef _smsStore;
    gsmlib::Ref<gsmlib::MeTa> _meta;

    // constructor
    Device(string name);

    // --- global state
    static GsmlibInterrupt *_gsmlibInterruptObject;
    static GsmlibProgress *_gsmlibProgressObject;

    // --- global device registry interface
  private:
    static vector<DeviceRef> _openDevices;

    // remove device from list of open devices
    static void unregisterDevice(Device *device);

  public:
    // return true if no devices are open
    static bool noBusyDevices();

    // get device by name (create if not existing if create == true)
    static DeviceRef getDevice(string name, bool create = true);

    // --- public device interface
    // request an action from the device, returns request ID
    // returns connection to the completion handler
    SigC::Connection request(RequestRef req);

    // close the device if only two references are left
    void checkClose();

    // return the device name
    string name() const {return _name;}

    // interrupt device activity
    void interrupt() {_gsmlibInterruptObject->interrupt(_interrupted);}

    // return true if device is busy handling request
    bool busy() const {return _busy;}

    ~Device();
  };

};

#endif // XGSM_DEVICE_H
