// *************************************************************************
// * Xgsm - mobile phone manager
// *
// * File:    xgsm_dialogs.cc
// *
// * Purpose: Various dialogs
// *
// * Author:  Peter Hofmann (software@pxh.de)
// *
// * Created: 12.7.2000
// *************************************************************************

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "xgsm_dialogs.h"
#include "xgsm_util.h"
#include "xgsm_pref.h"
#include "xgsm_main.h"
#include <typeinfo>
extern "C" {
#include "interface.h"
#include "support.h"
}

#include <gtk--/fileselection.h>
#include <gtk--/main.h>
#include <gtk--/menu.h>
#include <gtk--/optionmenu.h>
#include <gnome--/dialog.h>
#include <gnome--/file-entry.h>
#include <gtk--/colorselection.h>
#include <libgnome/gnome-i18n.h>
#include <sigc++/signal_system.h>
#include <gsmlib/gsm_util.h>
#include <termios.h>
#include <limits.h>
#include <iostream>

using namespace SigC;
using namespace std;
using namespace Xgsm;

// file selection box

static Gtk::FileSelection *fileSelectionBox = NULL;
static bool fileSelectionOkPressed, fileSelectionCancelPressed;
static bool fileSelectionDestroyed;

static void onFileSelectionOkPressed()
{
  fileSelectionOkPressed = true;
}

static void onFileSelectionCancelPressed()
{
  fileSelectionCancelPressed = true;
}

static void onFileSelectionDestroyed()
{
  fileSelectionDestroyed = true;
}

string Xgsm::runFileSelectionDialog(string title)
{
  string result;

  // create file selection box if not existing
  if (fileSelectionBox == NULL)
  {
    fileSelectionBox = new Gtk::FileSelection();
    fileSelectionBox->realize();
    fileSelectionBox->get_ok_button()->
      clicked.connect(slot(&onFileSelectionOkPressed));
    fileSelectionBox->get_cancel_button()->
      clicked.connect(slot(&onFileSelectionCancelPressed));
    fileSelectionBox->destroy.connect(slot(&onFileSelectionDestroyed));
    fileSelectionBox->set_modal(true);
    fileSelectionBox->show_all();
  }
  else
  {
    fileSelectionBox->set_modal(true);
    gtk_widget_map((GtkWidget*)fileSelectionBox->gtkobj());
  }

  fileSelectionBox->set_title(title);
  
  while (true)
  {
    fileSelectionOkPressed = false;
    fileSelectionCancelPressed = false;
    fileSelectionDestroyed = false;

    while (! (fileSelectionOkPressed || fileSelectionCancelPressed ||
              fileSelectionDestroyed))
      Gtk::Main::iteration();

    if (fileSelectionOkPressed)
    {
      result = fileSelectionBox->get_filename();
      if (! isFile(result))
        Gnome::Dialogs::error(
          gsmlib::stringPrintf(_("Error: %s is not a regular file"),
                               result.c_str()))->run_and_close();
      else
        break;
      result = "";
    }
    else
      break;
  }
  gtk_widget_unmap((GtkWidget*)fileSelectionBox->gtkobj());
  fileSelectionBox->set_modal(false);
  return result;
}

// device selection box

static Gnome::Dialog *deviceSelectionBox;
static Gnome::FileEntry *deviceFileEntry;
static Gnome::Entry *deviceInitStringEntry;
static Gtk::CheckButton *deviceSoftwareHandshakeButton;
static DynamicOptionMenu deviceBaudRateMenu;
static string deviceBaudRate;
static string deviceOldInitString;
static string deviceOldName;

static void onDeviceBaudRateSet()
{
  deviceBaudRate = deviceBaudRateMenu.getActiveChoice();
}

string Xgsm::runDeviceSelectionDialog(string title,
                                      string &baudRate,
                                      string &initString,
                                      bool &swHandshake)
{
  string result;

  // create file selection box if not existing
  if (deviceSelectionBox == NULL)
  {
    deviceSelectionBox = Gtk::wrap((GnomeDialog*)create_devdialog());

    LOOKUP_WIDGET(deviceFileEntry, "devdialog_phone_device",
                  GnomeFileEntry, GNOME_IS_FILE_ENTRY, deviceSelectionBox);
    ((Gtk::Entry*)deviceFileEntry->gtk_entry())->
      set_text(config.getPhoneDevice());
    ((Gnome::Entry*)deviceFileEntry->gnome_entry())->load_history();
    deviceOldName = config.getPhoneDevice();

    Gtk::OptionMenu *om;
    LOOKUP_WIDGET(om, "devdialog_baud_rate", GtkOptionMenu,
                  GTK_IS_OPTION_MENU, deviceSelectionBox);
    deviceBaudRateMenu = DynamicOptionMenu(om);
    if (deviceBaudRate == "")
      deviceBaudRate = config.getBaudRate();
    deviceBaudRateMenu = DynamicOptionMenu(om);
    deviceBaudRateMenu.append("300", deviceBaudRate,
                              slot(&onDeviceBaudRateSet));
    deviceBaudRateMenu.append("600", deviceBaudRate,
                              slot(&onDeviceBaudRateSet));
    deviceBaudRateMenu.append("1200", deviceBaudRate,
                              slot(&onDeviceBaudRateSet));
    deviceBaudRateMenu.append("2400", deviceBaudRate,
                              slot(&onDeviceBaudRateSet));
    deviceBaudRateMenu.append("4800", deviceBaudRate,
                              slot(&onDeviceBaudRateSet));
    deviceBaudRateMenu.append("9600", deviceBaudRate,
                              slot(&onDeviceBaudRateSet));
    deviceBaudRateMenu.append("19200", deviceBaudRate,
                              slot(&onDeviceBaudRateSet));
    deviceBaudRateMenu.append("38400", deviceBaudRate,
                              slot(&onDeviceBaudRateSet));
#ifdef B57600
    deviceBaudRateMenu.append("57600", deviceBaudRate,
                              slot(&onDeviceBaudRateSet));
#endif
#ifdef B115200
    deviceBaudRateMenu.append("115200", deviceBaudRate,
                              slot(&onDeviceBaudRateSet));
#endif
#ifdef B230400
    deviceBaudRateMenu.append("230400", deviceBaudRate,
                              slot(&onDeviceBaudRateSet));
#endif
#ifdef B460800
    deviceBaudRateMenu.append("460800", deviceBaudRate,
                              slot(&onDeviceBaudRateSet));
#endif
    
    LOOKUP_WIDGET(deviceInitStringEntry, "devdialog_init_string_entry",
                  GnomeEntry, GNOME_IS_ENTRY, deviceSelectionBox);
    ((Gtk::Entry*)deviceInitStringEntry->gtk_entry())->
      set_text(config.getInitString());
    deviceInitStringEntry->load_history();
    deviceOldInitString = config.getInitString();

    LOOKUP_WIDGET(deviceSoftwareHandshakeButton,
                  "devdialog_software_handshake",
                  GtkCheckButton, GTK_IS_CHECK_BUTTON, deviceSelectionBox);
    deviceSoftwareHandshakeButton->set_active(config.getSoftwareHandshake());
  }
  deviceSelectionBox->show_all();
  deviceSelectionBox->set_title(title);

  bool finished = false;
  while (! finished)
  {
    int i;
    switch (i = deviceSelectionBox->run())
    {
    case -1:
      DEBUG_START(1);
      cout << "deviceSelectionBox destroyed" << endl;
      DEBUG_END;
      deviceSelectionBox = NULL;
      result = "";
      finished = true;
      break;
    case 0:                     // OK button
      DEBUG_START(1);
      cout << "deviceSelectionBox OK" << endl;
      DEBUG_END;
      result = deviceFileEntry->get_full_path(false);
      if (! isDevice(result))
        Gnome::Dialogs::error(
          gsmlib::stringPrintf(_("Error: %s is not a character device"),
                               result.c_str()))->run_and_close();
      else
      {
        // collect out parameters

        // start with baudrate
        baudRate = deviceBaudRate;

        // save new device name in history if actually different
        if (result != deviceOldName)
          ((Gnome::Entry*)deviceFileEntry->gnome_entry())->
            append_history(TRUE, result);
        else
          deviceOldName = result;

        ((Gnome::Entry*)deviceFileEntry->gnome_entry())->save_history();

        // save new init string in history if actually different
        initString = ((Gtk::Entry*)deviceInitStringEntry->gtk_entry())->
          get_text();

        if (initString != deviceOldInitString)
          deviceInitStringEntry->append_history(TRUE, initString);
        else
          deviceOldInitString = initString;

        deviceInitStringEntry->save_history();

        // get SW handshake
        swHandshake = deviceSoftwareHandshakeButton->get_active();

        finished = true;
      }
      break;
    default:                    // Cancel button
      result = "";
      finished = true;
      break;
    }
  }
  if (deviceSelectionBox != NULL)
    deviceSelectionBox->hide_all();

  return result;
}

// color selection dialog

static bool colorSelectionOkPressed, colorSelectionCancelPressed;
static bool colorSelectionDestroyed, colorSelectionHelpPressed;

static void onColorSelectionOkPressed()
{
  colorSelectionOkPressed = true;
}

static void onColorSelectionCancelPressed()
{
  colorSelectionCancelPressed = true;
}

static void onColorSelectionHelpPressed()
{
  colorSelectionCancelPressed = true;
}

static void onColorSelectionDestroyed()
{
  colorSelectionDestroyed = true;
}

string Xgsm::runColorSelectionDialog(string title, string oldColor)
{
  string result = oldColor;
  
  // connect handlers and setup dialog
  Gtk::ColorSelectionDialog *dialog = new Gtk::ColorSelectionDialog(title);
  dialog->realize();
  dialog->get_ok_button()->
    clicked.connect(slot(&onColorSelectionOkPressed));
  dialog->get_cancel_button()->
    clicked.connect(slot(&onColorSelectionCancelPressed));
  dialog->get_help_button()->
    clicked.connect(slot(&onColorSelectionHelpPressed));
  dialog->destroy.connect(slot(&onColorSelectionDestroyed));
  dialog->set_modal(true);
  
  // set old color
  Gdk_Color oc(oldColor);
  gdouble values[4] = {oc.red_p(), oc.green_p(), oc.blue_p(), 1.0};
  dialog->get_colorsel()->set_color(values);

  dialog->show_all();
  
  while (true)
  {
    colorSelectionOkPressed = false;
    colorSelectionCancelPressed = false;
    colorSelectionHelpPressed = false;
    colorSelectionDestroyed = false;

    while (! (colorSelectionOkPressed || colorSelectionCancelPressed ||
              colorSelectionDestroyed || colorSelectionHelpPressed))
      Gtk::Main::iteration();

    if (colorSelectionOkPressed)
    {
      gdouble values[4];
      dialog->get_colorsel()->get_color(values);
      Gdk_Color resultColor;
      resultColor.set_rgb_p(values[0], values[1], values[2]);
      result = colorToString(resultColor);
      break;
    }
    else if (colorSelectionHelpPressed)
    {
      // FIXME
    }
    else
      break;
  }
  delete dialog;

  return result;
}

// DeviceHelper members

gint DeviceHelper::idle()
{
  DeviceRef dev = Device::getDevice(_sourceName);
  if (dev->busy())
    return TRUE;

  _completionConnection = dev->request(_openRequest);
  _waitingForOpen = false;
  _idle.disconnect();
  return FALSE;
}

DeviceHelper::DeviceHelper(string messageText) :
  _waitingForOpen(false), _destroyed(false), _messageText(messageText)
{
}

void DeviceHelper::open(RequestRef openRequest, bool printBusyMessage)
{
  DeviceRef dev = Device::getDevice(_sourceName);
  if (dev->busy())
  {
    _waitingForOpen = true;
    _openRequest = openRequest;
    _idle = Gtk::Main::timeout.connect(slot(this, &DeviceHelper::idle), 10);
    if (printBusyMessage)
      mainWindow->message(
        gsmlib::stringPrintf(_("Waiting for busy device or file %s"),
                             _sourceName.c_str()));
  }
  else
    _completionConnection = dev->request(openRequest);
}

void DeviceHelper::openDefaultDevice()
{
  _sourceName = config.getPhoneDevice();
  if (_sourceName != "")
    open(new OpenRequest(config.getPhoneDevice(),
                         config.getBaudRate(),
                         config.getInitString(),
                         config.getSoftwareHandshake(),
                         slot(this, &DeviceHelper::onDeviceEvent),
                         slot(this, &DeviceHelper::onOpenDone)), true);
}

void DeviceHelper::onDeviceEvent(EventType event)
{
  if (_destroyed) return;
  switch (event)
  {
  case CloseEvent:
    onClose();
    break;

  case IdleEvent:
    sensitive(true);
    break;

  case BusyEvent:
    sensitive(false);
    break;

  default:
    assert(0);
    break;
  }
}

void DeviceHelper::onOpenDone(ResponseRef response)
{
  if (typeid(response()) == typeid(ErrorResponse))
  {
    // handle error response
    Gnome::Dialogs::error
      (gsmlib::stringPrintf(_("Error opening %s\n("), _sourceName.c_str()) +
       ((ErrorResponse&)response()).exception().what() + ")")->run_and_close();
    onClose();
  }
  else
  {
    assert(typeid(response()) == typeid(OpenResponse));
    _dev = ((OpenResponse&)response()).getDevice();
    _eventConnection = ((OpenResponse&)response()).eventConnection();
    sensitive(true);
  }
}

DeviceHelper::~DeviceHelper()
{
  closeDevice();
}

void DeviceHelper::closeDevice()
{
  // disconnect handlers
  _eventConnection.disconnect();
  _completionConnection.disconnect();
  _idle.disconnect();
  _dev = NULL;
  sensitive(false);
}

// ToplevelHelper members

vector<Gtk::Widget*> ToplevelHelper::_openDialogs;

ToplevelHelper::ToplevelHelper(Gtk::Widget *dialog) :
  _dialog(dialog)
{
  cout << "registerDialog" << hex << dialog << dec << endl;

  // avoid to enter twice
  for (vector<Gtk::Widget*>::iterator i = _openDialogs.begin();
       i != _openDialogs.end(); ++i)
    if (*i == dialog)
      return;

  _openDialogs.push_back(dialog);
}

ToplevelHelper::~ToplevelHelper()
{
  for (vector<Gtk::Widget*>::iterator i = _openDialogs.begin();
       i != _openDialogs.end(); ++i)
    if (*i == _dialog)
    {
      _openDialogs.erase(i);
      return;
    }
}

void ToplevelHelper::closeAllDialogs()
{
  // work on a copy to be on the safe side
  vector<Gtk::Widget*> openDialogs = _openDialogs;
  for (vector<Gtk::Widget*>::iterator i = openDialogs.begin();
       i != openDialogs.end(); ++i)
    (*i)->destroy.emit();
}
