/*
    $Id: init.c,v 1.16 2002/05/24 05:51:59 belyi Exp $

    xkeysw - window bound/multi code keyboard switch
    Copyright (C) 1999  Dima Barsky, Igor Belyi

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <string.h>
#include <X11/Xlib.h>
#include <unistd.h>
#include "parse.h"
#include "xwork.h"
#include "init.h"

#define STD_CONF_FILE ".xkeyswrc"
#define DATADIR_PATH "XKEYSW_DATAPATH"

static int InitializeSwitchTable();
static char* getFullName(char* filename);
static FILE* GetConfigFile(char* filename);
static void DestroyCodeRecords();

int switchTableSize = 0;
codeRecord_t* switchTable = NULL;
codeRecord_t* defaultCode = NULL;

unsigned int modNameToMask(char* modifier)
{
    int index;
    struct modifiers_t {
	char* name;
	unsigned int mask;
    } modifiers [] = {
	{"Shift", ShiftMask},
	{"Lock", LockMask},
	{"Control", ControlMask},
	{"Mod1", Mod1Mask},
	{"Mod2", Mod2Mask},
	{"Mod3", Mod3Mask},
	{"Mod4", Mod4Mask},
	{"Mod5", Mod5Mask},
	{"Button1", Button1Mask},
	{"Button2", Button2Mask},
	{"Button3", Button3Mask},
	{"Button4", Button4Mask},
	{"Button5", Button5Mask}
    };

    for(index=sizeof(modifiers)/sizeof(struct modifiers_t)-1; index>=0; index--)
	if(strcmp(modifiers[index].name, modifier) == 0)
	    return modifiers[index].mask;
    return -1;
}

static void DestroyCodeRecords()
{
    codeRecord_t* ptr;

    if(!switchTable) return;

    for(ptr=switchTable; switchTableSize; switchTableSize--, ptr++) {
	if(ptr->prefix) free(ptr->prefix);
	if(ptr->keymap.keysyms) XFree(ptr->keymap.keysyms);
	if(ptr->switchKeys) free(ptr->switchKeys);
    }

    free(switchTable);
    switchTable = NULL;

}

static char* getFullName(char* filename) {
    char *paths[4];
    static char *file = NULL;
    int i;

    if(filename == NULL || *filename == '\0')
	return NULL;

    if(strchr(filename, '/'))
	return filename;

    paths[0] = getenv(DATADIR_PATH);
    paths[1] = getenv("HOME");
    paths[2] = PKGDATADIR;

    for(i=0; i<3; i++)
	if(paths[i] != NULL) {
	    file = malloc(strlen(paths[i]) + strlen(filename) + 2);
	    if(file == NULL) {
		fprintf(stderr, "Cannot allocat string for full path name.\n");
		return NULL;
	    }
	    strcpy(file, paths[i]);
	    strcat(file, "/");
	    strcat(file, filename);
	    
	    if(!access(file, R_OK))
		return file;

	    free(file);
	}

    return filename;
}

/* Finds out the complete path to the config file and opens it */
static FILE* GetConfigFile(char* filename)
{
    FILE *cfd = NULL;
    
    if(filename == NULL || *filename == '\0')
	filename = getFullName(STD_CONF_FILE);

    if(filename == NULL) {
	fprintf(stderr, "Cannot find configuration file.\n");
	return NULL;
    }

    if((cfd = fopen(filename, "r")) == NULL) {
	fprintf(stderr, "Cannot open file %s.\n", filename);
    }

    return cfd;
}

/*
 * Initializes switchTable from ParsedConfFile.
 * This function should be called when parsing and testing of values are made.
 */
static int InitializeSwitchTable()
{
    parsedCodeRecord_t *currCode;
    parsedSwitchKey_t *parsedKey;
    codeRecord_t *currRecord;
    switchKey_t *currKey;

    for(switchTableSize=0, currCode=ParsedConfFile;
	currCode;
	switchTableSize++, currCode=currCode->next);

    switchTable = (codeRecord_t*)calloc(switchTableSize, sizeof(codeRecord_t));

    if(switchTable == NULL) {
	fprintf(stderr, "Cannot allocate switchTable.\n");
	return 1;
    }

    for(currRecord=switchTable, currCode=ParsedConfFile;
	currCode;
	currRecord++, currCode=currCode->next) {
	
	currRecord->prefix = currCode->prefix;
	currCode->prefix = NULL;

	/*
	 * Set KeyMap to the original value to make sure that XMM file
	 * does not have problems with resolution of keysyms altered by
	 * prefious iterations.
	 * Set it to shorten KeyMap if XMM file is present. This may
	 * help with a problem of KeySyms assigned to more than one
	 * KeyCode in the original keyboard layout.
	 */
	if (currCode->xmmfile) {
	    SetKeyMap(originalKeyMap2);
 	
	    /* Set keymap according to XMM file */
	    if(SetXMMKeyMap(getFullName(currCode->xmmfile))) {
		fprintf(stderr, "'%s' layout setup failed.\n", currCode->name);
		return 1;
	    }
	}
	else
	    SetKeyMap(originalKeyMap);

	for(parsedKey=currCode->switchKeys, currRecord->switchKeysSize=0;
	    parsedKey;
	    parsedKey=parsedKey->next, currRecord->switchKeysSize++);

	currRecord->switchKeys=(switchKey_t*)calloc(currRecord->switchKeysSize,
						    sizeof(switchKey_t));

	if(currRecord->switchKeys == NULL) {
	    fprintf(stderr, "Cannot allocate switchKeys for record %d\n",
		    currRecord - switchTable);
	    return 1;
	}

	/* Read Keycodes now while we have keymap for correct layout */
	for(parsedKey=currCode->switchKeys, currKey=currRecord->switchKeys;
	    parsedKey;
	    parsedKey=parsedKey->next, currKey++) {

	    currKey->switchCode = GetKeyCode(parsedKey->keyName);
	    if (currKey->switchCode == 0) {
		fprintf(stderr, "Error: Undefined keysym '%s' "
			"(switching from '%s' code to '%s')\n",
			parsedKey->keyName, currCode->name,
			parsedKey->code->name);
		return 1;
	    }
	    currKey->modifiers = parsedKey->modifiers;
	    if(SetNewKeySyms(parsedKey->keyName, parsedKey->newName))
		return 1;

	    currKey->code = switchTable + parsedKey->code->index;
	}

	/* Store current keymap here after adjustment of switch keys. */
	currRecord->keymap = GetKeyMap();
    }

    /* Remember the defaultCode and set its keymap before continue */
    defaultCode = switchTable + DefaultRecord->index;
    SetKeyMap(defaultCode->keymap);

    return 0;
    
}

/* Reads config file and initializes internal structure for codes switching */
int Initialize(char* filename)
{
    FILE* cfd = NULL;

    if((cfd = GetConfigFile(filename)) == NULL) {
	return 1;
    }

    if(ParseConfigFile(cfd)) {
	fclose(cfd);
	freeParsedRecords();
	return 1;
    }

    fclose(cfd);
    cfd = NULL;

    if(AdjustParsedRecords()) {
	freeParsedRecords();
	return 1;
    }

    if (InitializeSwitchTable()) {
	SetKeyMap(originalKeyMap);
	freeParsedRecords();
	DestroyCodeRecords();
	return 1;
    }

    freeParsedRecords();

    return 0;

}
