/* XML Im-/Exporter: Copyright 2002-2004, Oliver Zeigermann (oliver@zeigermann.de)
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modification, are 
 * permitted provided that the following conditions are met:
 * 
 * - Redistributions of source code must retain the above copyright notice, this list of
 *   conditions and the following disclaimer.
 * - Redistributions in binary form must reproduce the above copyright notice, this list
 *   of conditions and the following disclaimer in the documentation and/or other materials
 *   provided with the distribution.
 * - Neither the name of the Oliver Zeigermann nor the names of its contributors may
 *   be used to endorse or promote products derived from this software without specific
 *   prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND 
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, ORTORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 */ 
package de.zeigermann.xml;

import java.io.*;

/**
 * Adds XML export functionality to the underlying output stream. Formatting and
 * encoding is done as straight forward as possible. <br>
 * Everything you know better than this class must be done by you, e.g. you will
 * have to tell <code>XMLOutputStreamWriter</code> where you wish to have
 * newlines.In effect, no unexpected so called
 * <em>intelligent</em> behavior is to be feared. Another effect is high speed.
 * <br>
 * <br>
 * <em>Caution</em>: Do not forget to call {@link #flush} at the end of your
 * exporting process as otherwise no data might be written.
 *
 * <em>Warning</em>: When using two byte encoding (e.g. UTF-16) underlying
 * OutputStream can
 * not savely be brought to string. Do <em>not</em> use
 * {@link ByteArrayOutputStream} with two byte encoding, as XML declaration
 * will be in single byte encoding (according to XML spec) and the rest will be
 * in double byte totally confusing ByteArrayOutputStream encoding to string.
 * <b>If you want to have string output use {@link XMLWriter} filtering 
 * {@link StringWriter} or for convenience {@link XMLStringWriter}.</b>
 *
 * @author <a href="mailto:oliver@zeigermann.de">Olli Z.</a>
 */
public class XMLOutputStreamWriter extends XMLWriter {
    /** Name of UTF-8 encoding  */
    public static String ENCODING_UTF_8 = "UTF-8";
    /** Name of UTF-16 encoding */
    public static String ENCODING_UTF_16 = "UTF-16";
    /** Name of ISO-8859-1 encoding */
    public static String ENCODING_ISO_8859_1 = "ISO-8859-1";

    /** Name of standard encoding */
    public static String ENCODING_STANDARD = ENCODING_UTF_8;
    /** Alias for ISO-8859-1 encoding */
    public static String ENCODING_ISO_LATIN1 = ENCODING_ISO_8859_1;

    protected OutputStream os;
    protected String encodingName;

    /** Creates a new output stream writer for XML export.
     * @param out the underlying output stream the XML is exported to
     * @param encodingName name of the encoding used to write XML as well as
     * for the XML declataration (e.g. UTF-8, ISO-8859-1, ...)
     */
    public XMLOutputStreamWriter(OutputStream os, String encodingName) throws UnsupportedEncodingException {
        super(new OutputStreamWriter(os, encodingName));
        this.encodingName = encodingName;
        this.os = os;
    }

    /** Creates a new output stream writer for XML export. Standard encoding
     * will be used as found in {@link #ENCODING_STANDARD}, which usually is
     * UTF-8.
     * @param out the underlying output stream the XML is exported to
     * @see #XMLOutputStreamWriter(OutputStream, String)
     */
    public XMLOutputStreamWriter(OutputStream os) throws UnsupportedEncodingException {
        this(os, ENCODING_STANDARD);
    }

    /** Gets the name of the encoding as it would be inserted into the
     * XML declaration. {@link OutputStreamWriter#getEncoding} may return something less verbose.
     * @see OutputStreamWriter#getEncoding
     */
    public String getEncodingName() {
        return encodingName;
    }

    /** Writes XML delcaration using version 1.0 and encoding specified in
     * constructor.
     * <em>Caution</em>: As XML declaration must be in plain text (no UNICODE)
     * it will not be passed to writer, but directly to stream!
     */
    public void writeXMLDeclaration() throws IOException {
        String xmlDecl = "<?xml version=\"1.0\" encoding=\"" + getEncodingName() + "\"?>\n";
        byte[] xmlDeclBytes = xmlDecl.getBytes("US-ASCII");

        // flush to ensure correct sequence
        flush();
        os.write(xmlDeclBytes);
    }

}
