/*
 * Copyright (C) 2003 the xmms-kde team
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
#include "configdialog.h"
#include "configdialog.moc"




XmmsKdeConfigDialog::XmmsKdeConfigDialog(QWidget *parent, 
					 const QStringList themeList, 
					 KStandardDirs *dataDirs,
                                         int currentScrollMode, 
					 KGlobalAccel *globalAccel,
					 QString currentTheme,
					 PlayerInterface::Player currentPlayer,
					 bool hideXmms,
					 bool readTags,
					 bool underscores,
					 OSDFrame *osd,
					 XmmsKdeDB *datab,
					 XmmsKdeDBQuery *querydb)
  : KDialogBase (IconList, i18n("Options"), Ok|Cancel, Ok, parent, "configdialog", false, false) {

  scrollMode  = currentScrollMode;
  tlist       = themeList;
  accel       = globalAccel;

  nameList = new QStringList();

  for (unsigned int index = 0; index < tlist.count(); index ++) 
    nameList->append(QFileInfo(tlist[index]).fileName());

  dDirs = dataDirs;

  noScreen = new QPixmap(readFile(dDirs->findResource("data", "xmms-kde/noscreen.tgz"), "noscreen.png"));

  setupThemesPage(currentTheme);
  setupPlayerPage(currentPlayer, hideXmms, readTags,
		  underscores);
  setupAccelPage();
  setupOSDPage(osd);

  setupDBPage(datab, querydb);

  setFixedSize(400, 280);

  selectedPlayer = currentPlayer;
}




XmmsKdeConfigDialog::~XmmsKdeConfigDialog() {

  if (themeListBox) delete themeListBox;
  if (imageLabel) delete imageLabel;
  if (nameList) delete nameList;
  if (noScreen) delete noScreen;
}


/*
 * setup the Theme configuration page
 */
void XmmsKdeConfigDialog::setupThemesPage(QString currentTheme) {
  
  QPixmap themes(readFile(dDirs->findResource("data", "xmms-kde/icons.tgz"), "themes.png"));

  QWidget *page = addPage(i18n("Themes"), 0, themes);
  
  QVBoxLayout *vbox = new QVBoxLayout(page);
  QHBox *box = new QHBox(page);
  vbox->addWidget(box);
  
  themeListBox = new QListBox(box);
  themeListBox->insertStringList(*nameList);
  themeListBox->setFixedSize(200, 120);

  connect(themeListBox, SIGNAL(highlighted(int)), this, SLOT(showTheme(int)));

  imageLabel = new QLabel(box);

  imageLabel->setFixedSize(140, 100);
  imageLabel->setAlignment(Qt::AlignCenter);

  // set scroll mode
  QButtonGroup *scrollGroup = new QButtonGroup(4, Qt::Horizontal, page);
  vbox->addWidget(scrollGroup);
  QRadioButton *btn1 = new QRadioButton(i18n("Loop"), scrollGroup); // 0
  QRadioButton *btn2 = new QRadioButton(i18n("Ping pong"), scrollGroup); // 1
  QRadioButton *btn3 = new QRadioButton(i18n("Sinus"), scrollGroup); // 2
  QRadioButton *btn4 = new QRadioButton(i18n("Loop once"), scrollGroup); // 3
  scrollGroup->setButton(scrollMode);
  scrollGroup->setTitle(i18n("Title scroll mode"));
  connect(scrollGroup, SIGNAL(clicked(int)), this, 
	  SLOT(scrollModeChanged(int)));
  
  themeListBox->setCurrentItem(themeListBox->findItem(QFileInfo(currentTheme).fileName()));  

  
}

void XmmsKdeConfigDialog::setupAccelPage() {

  QPixmap shortcuts(readFile(dDirs->findResource("data", "xmms-kde/icons.tgz"), "shortcuts.png"));

  QWidget *page = addPage(i18n("Shortcuts"), 0, shortcuts);
  
  QVBoxLayout *vbox = new QVBoxLayout(page);
  accelBox = new QCheckBox(i18n("enable Shortcuts"), page);
  vbox->addWidget(accelBox);

  QPushButton *accelButton = new QPushButton(i18n("Change Shortcuts"), page);
  vbox->addWidget(accelButton);

  connect(accelButton, SIGNAL(clicked()), this, SLOT(accelButtonClicked()));

  if (accel->isEnabled()) accelBox->setChecked(true);
  //else accelBox->setChecked(false);
}

void XmmsKdeConfigDialog::setupOSDPage(OSDFrame *osd) {
  
  QPixmap osdPixmap(readFile(dDirs->findResource("data", "xmms-kde/icons.tgz"), "osd.png"));

  QWidget *page = addPage(i18n("OSD"), 0, osdPixmap);
  QVBoxLayout *pagebox = new QVBoxLayout(page);
  
  QWidget *w = osd->getConfigurationWidget(page);

  pagebox->addWidget(w);
}


void XmmsKdeConfigDialog::setupDBPage(XmmsKdeDB *db, XmmsKdeDBQuery *query) {

  QPixmap dbPix(readFile(dDirs->findResource("data", "xmms-kde/icons.tgz"), "db.png"));

  QWidget *page = addPage(i18n("Database"), 0, dbPix);

  QVBoxLayout *pagebox = new QVBoxLayout(page);

  QTabWidget *dbWidget = new QTabWidget(page);  
  dbWidget->addTab(db->getConfigurationWidget(page), i18n("Database"));
  dbWidget->addTab(query->getConfigurationWidget(page), i18n("Query window"));

  pagebox->addWidget(dbWidget);
}


void XmmsKdeConfigDialog::setupPlayerPage(PlayerInterface::Player currentPlayer,
					  bool hideXmms,
					  bool readTags,
					  bool underscores) {

  QPixmap playerIcon(readFile(dDirs->findResource("data", "xmms-kde/icons.tgz"), "players.png"));
  QPixmap okIcon(readFile(dDirs->findResource("data", "xmms-kde/icons.tgz"), "ok.png"));
  QPixmap koIcon(readFile(dDirs->findResource("data", "xmms-kde/icons.tgz"), "ko.png"));

  QWidget *page = addPage(i18n("Player"), 0, playerIcon);
  
  QVBoxLayout *vbox = new QVBoxLayout(page);
  vbox->setSpacing(10);

  playerGroup = new QButtonGroup(3, Qt::Horizontal, page);

  QRadioButton *xmmsButton = new QRadioButton("XMMS", playerGroup);
  QRadioButton *noatunButton = new QRadioButton("Noatun", playerGroup);
  QRadioButton *smpegButton = new QRadioButton("SMPEG", playerGroup);
  QRadioButton *amarokButton = new QRadioButton("amaroK", playerGroup);

  playerGroup->setButton(currentPlayer);
  playerGroup->setTitle(i18n("Player"));

  connect(playerGroup, SIGNAL(clicked(int)), 
	  this, SLOT(playerChanged(int)));

  /*
   * build the xmms configuration widget
   */
  QTabWidget *confWidget = new QTabWidget(page);
  
  QWidget *xmmsWidget = new QWidget(confWidget);  
  QVBoxLayout *xvbox = new QVBoxLayout(xmmsWidget);
  minimizeXmmsPlayer = new QCheckBox(i18n("completely hide XMMS"),
				     xmmsWidget);

  xvbox->addWidget(minimizeXmmsPlayer);

  minimizeXmmsPlayer->setChecked(hideXmms);

  confWidget->addTab(xmmsWidget, "XMMS");

  /*
   * build the noatun configuration widget
   */
  QWidget *noatunWidget = new QWidget(confWidget);  
  xvbox = new QVBoxLayout(noatunWidget);

  confWidget->addTab(noatunWidget, "Noatun");

  /*
   * build the smpeg configuration widget
   */
  QWidget *smpegWidget = new QWidget(confWidget);  
  xvbox = new QVBoxLayout(smpegWidget);
  readID3Tags = new QCheckBox(i18n("read ID3 tags when files are dropped"),
			      smpegWidget);
  xvbox->addWidget(readID3Tags);
  readID3Tags->setChecked(readTags);

  convertUnderscores = new QCheckBox(i18n("convert underscores to spaces"),
				     smpegWidget);
  xvbox->addWidget(convertUnderscores);
  convertUnderscores->setChecked(underscores);

  confWidget->addTab(smpegWidget, "SMPEG");

  vbox->addWidget(playerGroup);
  vbox->addWidget(confWidget);

#ifndef HAVE_XMMS
  xmmsButton->setEnabled(false);
#endif
#ifndef HAVE_NOATUN
  noatunButton->setEnabled(false);
#endif
#ifndef HAVE_SMPEG
  smpegButton->setEnabled(false);
#endif
#ifndef HAVE_AMAROK
  amarokButton->setEnabled(false);
#endif

}



void XmmsKdeConfigDialog::showTheme(int index) {
  /*
   * load the main pixmap from the selected theme
   */
  loadTheme(tlist[index]);
}


/*
 * load a theme from a file
 *
 */
void XmmsKdeConfigDialog::loadTheme(QString themeName) {
  
  qDebug("xmms-kde: loading theme :" + themeName);

  /*
   * load theme
   */
  KSimpleConfig* themeConfig = new KSimpleConfig(themeName);
  
  themeConfig->setGroup("BITMAPS");
  QString file = themeConfig->readEntry("file");

  if (file.isNull()) {
    qDebug("xmms-kde: error loading theme");

    return;
  }

  QString tarFile = dDirs->findResource("data", "xmms-kde/" + file);
  if (tarFile.isNull()) {
    qDebug("xmms-kde: error loading theme");
    return;
  }
  
  delete themeConfig;

  qDebug("xmms-kde: trying to load skin: '" + tarFile + "'");
  QByteArray array = readFile(tarFile, "screenshot.png");

  if (array.isNull()) {
      imageLabel->setPixmap(*noScreen);
  } else {
    QPixmap *pix = new QPixmap(array);
    imageLabel->setPixmap(*pix);
  }

  theme = themeName;
}


/*
 * load a file from a tar.gz archive
 *
 */
QByteArray readFile(QString tarName, QString fileName) {

  KTarGz tar(tarName);

  if (!tar.open(IO_ReadOnly)) {
    qDebug("xmms-kde: didn't find file " + tarName);
    return 0;
  }

  const KTarDirectory* dir = tar.directory();

  const KTarEntry* e = dir->entry(fileName);
  if (e == NULL) {
    qDebug("xmms-kde: didn't find file " + fileName + " in file " + tarName);
    return 0;
  }
  
  ASSERT( e && e->isFile() );
  const KTarFile* f = (KTarFile*)e;
  
  QByteArray arr(f->data());

  tar.close();
  return arr;
}


void XmmsKdeConfigDialog::scrollModeChanged(int btn) {

  scrollMode = btn;
}

QString XmmsKdeConfigDialog::getSelectedTheme() {

  return theme;
}

PlayerInterface::Player XmmsKdeConfigDialog::getSelectedPlayer() {

  return selectedPlayer;
}

bool XmmsKdeConfigDialog::enableAccels() {

  return accelBox->isChecked();
}

int XmmsKdeConfigDialog::getScrollMode() {

  return scrollMode;
}

void XmmsKdeConfigDialog::accelButtonClicked() {

  KKeyDialog::configure(accel);
}


void XmmsKdeConfigDialog::playerChanged(int index) {
  
  selectedPlayer = (PlayerInterface::Player) index;
}

