/*
 * Copyright (C) 2003 the xmms-kde team
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
#include "xmms-kde.h"
#include "xmms-kde.moc"

extern "C" {
  KPanelApplet* init(QWidget *parent, const QString& configFile) {
    KGlobal::locale()->insertCatalogue("xmms-kde");

    return new XmmsKde(configFile, KPanelApplet::Normal,
		       KPanelApplet::About | KPanelApplet::Help | 
		       KPanelApplet::Preferences, parent, "XmmsKde");
  }
}



/*
 * constructor
 *
 */
XmmsKde::XmmsKde(const QString& configFile, Type type, int actions,
                             QWidget *parent, const char *name)
  : KPanelApplet(configFile, type, actions, parent, name), QToolTip(this), inVolumeChange(false), inSeekChange(false) {

  pausePressed = false;
  nextPressed  = false;
  backPressed  = false;
  playPressed  = false;
  stopPressed  = false;
  loopedOnce   = false;

  aboutData = NULL;
  playList = NULL;

  seekTime = 5000;
    
  setAcceptDrops(true);

  qDebug("xmms-kde: Welcome to xmms-kde\n");

  QString homeDir = QString(QDir::home().path());

  dataDirs = KGlobal::dirs();

  mainConfig = config();
  mainConfig->setGroup("THEMES");

  QString defaultTheme = mainConfig->readEntry("default");

  /*
   * if the resource couldn't be loaded, insert default values
   */
  if (defaultTheme.isEmpty()) {
    qDebug("xmms-kde: couldn't read config file");    

    mainConfig->writeEntry("default", "horizontalblue.rc");
    
    mainConfig->sync();

    defaultTheme = mainConfig->readEntry("default");

    qDebug("xmms-kde: default: " + defaultTheme);
  }

  QString resourceDir = dataDirs->findResourceDir("data", "xmms-kde/xmms-kderc") + "xmms-kde/";

  dataDirs->addPrefix(homeDir + "/.xmms-kde");

  /*
   * create the list of themes
   */  
  themeMenu = new QPopupMenu(this);
  themeList = new QStringList();

  QDir themeDir = QDir(resourceDir, "*.rc");

  unsigned int i = 0;   

  for (; i < themeDir.count(); i++) {
    themeMenu->insertItem(themeDir[i], i);
    themeList->append(resourceDir + themeDir[i]);
  }

  /*
   * search for themes in the home directory
   */
  themeDir = QDir(homeDir + "/.xmms-kde/", "*.rc");
  
  if (themeDir.exists()) {
    for (unsigned int k = 0; k < themeDir.count(); k++) {
      themeMenu->insertItem(themeDir[k], i + k);
      themeList->append(homeDir + "/.xmms-kde/" + themeDir[k]);
      qDebug("xmms-kde: theme: %s", (homeDir + "/.xmms-kde/" + themeDir[k]).latin1());
    }
  }

  mainConfig->setGroup("TITLE");

  titleScrollMode = mainConfig->readNumEntry("scrollmode", 1);
  titleScrollSpeed= mainConfig->readNumEntry("scrollspeed", 1);
  titleDefaultScrollSpeed= mainConfig->readNumEntry("scrollspeed", 1);
  displayElapsedTime = mainConfig->readNumEntry("displayelapsedtime", 1);
  
  title = "xmms-kde";

  mainConfig->setGroup("PLAYER");
  PlayerInterface::Player currentPlayer = 
    (PlayerInterface::Player) mainConfig->readNumEntry("player", 
						       PlayerInterface::XMMS);

  smpegPlayListPos = QPoint(0, 0);
  smpegPlayListSize = QSize(240, 240);
  
  mainConfig->setGroup("SMPEG");
  smpegPlayList = mainConfig->readListEntry("recent");
  smpegPlayListPos = 
    mainConfig->readPointEntry("playlistposition", &smpegPlayListPos);
  smpegPlayListSize = 
    mainConfig->readSizeEntry("playlistsize", &smpegPlayListSize);
  readID3Tags = mainConfig->readBoolEntry("readallid3tags", false);
  convertUnderscores = mainConfig->readBoolEntry("convertunderscores", false);
  smpegRepeat = mainConfig->readBoolEntry("repeat", false);
  smpegShuffle = mainConfig->readBoolEntry("shuffle", false);

  QPixmap *dbIcon = new QPixmap(readFile(dataDirs->findResource("data", "xmms-kde/icons.tgz"), "db2.png"));

  db = NULL;
  db = new XmmsKdeDB(mainConfig, dbIcon);

  scrollTimer = new QTimer(this);
  connect(scrollTimer, SIGNAL(timeout()), SLOT(scroll()));

  receiveTimer = new QTimer(this);
  connect(receiveTimer, SIGNAL(timeout()), SLOT(receive()));

  QString defTheme =dataDirs->findResource("data", "xmms-kde/" + defaultTheme);

  // if theme doesn't exist, try horizontalblue.rc  
  if (defTheme.isNull())
    defTheme = dataDirs->findResource("data", "xmms-kde/horizontalblue.rc");

  loadTheme(defTheme);

  titleXPosition     = theme.titleClipRect.right();
  titleSinArg        = 0;

  /*
   * read and set the global accelerator keys
   */
  globalAccel = new KGlobalAccel(this);
  
  globalAccel->insert("xmms-kde next", i18n("next song"), 
		      i18n("Jump to next song in playlist"),
		      CTRL+SHIFT+Key_S,
		      CTRL+SHIFT+Key_S,
		      (QObject *) this, 
		      SLOT(next()));
  
  globalAccel->insert("xmms-kde back", i18n("previous song"),
		      i18n("Jump to previous song in playlist"),
		      CTRL+SHIFT+Key_A,
		      CTRL+SHIFT+Key_A,
		      (QObject *) this, 
		      SLOT(previous()));

  globalAccel->insert("xmms-kde seek forward", i18n("seek forward"),
		      i18n("Seek forward in the current song"),
		      CTRL+SHIFT+Key_V,
		      CTRL+SHIFT+Key_V,
		      (QObject *) this,
		      SLOT(seekForward()));

  globalAccel->insert("xmms-kde seek backward", i18n("seek backward"),
		      i18n("Seek backward in the current song"),
		      CTRL+SHIFT+Key_C,
		      CTRL+SHIFT+Key_C,
		      (QObject *) this,
		      SLOT(seekBackward()));

  globalAccel->insert("xmms-kde play", i18n("play"), 
		      i18n("Start playing or restart the current song"),
		      CTRL+SHIFT+Key_Q, 
		      CTRL+SHIFT+Key_Q, 
		      (QObject *) this, 
		      SLOT(play()));
  
  globalAccel->insert("xmms-kde stop", i18n("stop"),
		      i18n("Stop the current song"),
		      CTRL+SHIFT+Key_W,
		      CTRL+SHIFT+Key_W,
		      (QObject *) this,
		      SLOT(stop()));
  
  globalAccel->insert("xmms-kde pause", i18n("pause"),
		      i18n("Pause the current song"),
		      CTRL+SHIFT+Key_E, 
		      CTRL+SHIFT+Key_E, 
		      (QObject *) this, 
		      SLOT(pause()));

  globalAccel->insert("xmms-kde play/pause", i18n("play / pause"),
		      i18n("Play or pause the current song"),
		      CTRL+SHIFT+Key_R, 
		      CTRL+SHIFT+Key_R, 
		      (QObject *) this, 
		      SLOT(playpause()));

  globalAccel->insert("xmms-kde jump to file", i18n("jump to file"),
		      i18n("Jump to a specific file in the playlist"),
		      CTRL+SHIFT+Key_J, 
		      CTRL+SHIFT+Key_J, 
		      (QObject *) this, 
		      SLOT(jumpToFile()));
  
  globalAccel->insert("xmms-kde volume plus", i18n("increase volume"), 
		      i18n("Increase the volume"),
		      CTRL+SHIFT+Key_X,
		      CTRL+SHIFT+Key_X,
		      (QObject *) this, 
		      SLOT(volp()));

  globalAccel->insert("xmms-kde volume minus", i18n("decrease volume"), 
		      i18n("Decrease the volume"),
		      CTRL+SHIFT+Key_Y,
		      CTRL+SHIFT+Key_Y,
		      (QObject *) this, 
		      SLOT(volm()));
		      
  globalAccel->insert("xmms-kde show track", i18n("Show On-Screen-Display"),
  		      i18n("Show the current track on the display"),
		      CTRL+SHIFT+Key_T,
		      CTRL+SHIFT+Key_T,
		      (QObject *) this,
		      SLOT(showOSD()));

  globalAccel->insert("xmms-kde popup database query", i18n("show database window"), 
		      i18n("Show the database window"),
		      CTRL+SHIFT+Key_D,
		      CTRL+SHIFT+Key_D,
		      (QObject *) this, 
		      SLOT(showDB()));

  globalAccel->insert("xmms-kde update database", i18n("update database"), 
		      i18n("Update the database"),
		      CTRL+SHIFT+Key_U,
		      CTRL+SHIFT+Key_U,
		      (QObject *) db, 
		      SLOT(updateDatabase()));

  globalAccel->insert("xmms-kde toggle shuffle", i18n("toggle shuffle"), 
		      i18n("Toggle Shuffle"),
		      CTRL+SHIFT+Key_F,
		      CTRL+SHIFT+Key_F,
		      (QObject *) this, 
		      SLOT(toggleShuffle()));

  globalAccel->insert("xmms-kde toggle repeat", i18n("toggle repeat"), 
		      i18n("Toggle Repeat"),
		      CTRL+SHIFT+Key_G,
		      CTRL+SHIFT+Key_G,
		      (QObject *) this, 
		      SLOT(toggleRepeat()));

  globalAccel->insert("xmms-kde show lyrics", i18n("show lyrics"), 
		      i18n("Show Lyrics"),
		      CTRL+SHIFT+Key_L,
		      CTRL+SHIFT+Key_L,
		      (QObject *) this, 
		      SLOT(showLyrics()));

  globalAccel->readSettings(mainConfig);

  globalAccel->updateConnections();

  mainConfig->setGroup("ACCEL");
  
  if (mainConfig->readBoolEntry("enable", 0)) {
    globalAccel->setEnabled(true);
  } else {
    globalAccel->setEnabled(false);
  }
  
  mainConfig->setGroup("XMMS");
  hideXmms = mainConfig->readBoolEntry("hide", false);

  mainConfig->setGroup("MPLAYER");
  mplayerBorderless = mainConfig->readBoolEntry("noborder", false);
  mplayerScreen = mainConfig->readBoolEntry("showscreen", false);

  player = NULL;

  playListIcon = new QPixmap(readFile(dataDirs->findResource("data", "xmms-kde/icons.tgz"), "playlist.png"));

  if (currentPlayer == PlayerInterface::XMMS)
#ifdef HAVE_XMMS
    player = new XMMSPlayer(hideXmms);
#endif

  if (currentPlayer == PlayerInterface::NOATUN) player = new NoatunPlayer();
#ifdef HAVE_SMPEG
  if (currentPlayer == PlayerInterface::SMPEG) 
    player = new SMPEGPlayer(smpegPlayList, smpegPlayListPos, 
			     smpegPlayListSize, readID3Tags,
			     convertUnderscores, smpegRepeat, smpegShuffle,
			     playListIcon);
#endif

#ifdef HAVE_AMAROK
  if (currentPlayer == PlayerInterface::AMAROK) player = new AmarokPlayer();
#endif

#ifdef HAVE_MPLAYER
  if (currentPlayer == PlayerInterface::MPLAYER) player = new MPlayer();
#endif

#ifdef HAVE_NOATUN
  if (!player) player = new NoatunPlayer();
#endif

  if (!player) player = new PlayerInterface();
  
  qDebug("xmms-kde: player: %d", player->is());

  osd = new OSDFrame(mainConfig);

  query = new XmmsKdeDBQuery(db, player, playListIcon, mainConfig);

  if (query->getPopup() == 0) // kicker startup
    query->popup();

  title = "xmms-kde";

  if (!aboutData) {
    aboutData = new KAboutData("XmmsKde", "XmmsKde", 
			       "3.1", "MP3 Player Applet",
			       KAboutData::License_GPL_V2, "(c) 2002, Flo Niebling", 0, "http://xmms-kde.sourceforge.net", "tranqlzer@users.sourceforge.net");
    
    aboutData->addAuthor("Flo Niebling", 0, "tranqlzer@users.sourceforge.net");
    aboutData->addAuthor("Rajendra R Patil", "mplayer support",
			 "rajendra_p123@yahoo.com");

  }

  lyrics = new SongLyrics(mainConfig);
}


/*
 * destructor
 * 
 */
XmmsKde::~XmmsKde() {

  scrollTimer->stop();
  receiveTimer->stop();
  
  saveConfig();

  if (themeList) 
    delete themeList;  

  if (themeMenu) 
    delete themeMenu;

  if (player) 
    delete player;

  if (query)
    delete query;

  if (db)
    delete db;

  if (osd)
    delete osd;

  qDebug("xmms-kde: quit");
}

/*
 * Starts the timers if @param start is true
 */
void XmmsKde::setTimers(bool start) {
  if (start) {
    receiveTimer->start(200);
    scrollTimer->start(40);
  } else {
    scrollTimer->stop();
    receiveTimer->stop();
  }
}


/*
 * tooltips
 * 
 */
void XmmsKde::maybeTip(const QPoint& p) {

  if (!running) tip(rect(), i18n("Click to launch Player"));

  if(!rect().contains(p)) return;

  if      (theme.rectPlay.contains(p))  tip(theme.rectPlay, i18n("Play"));
  else if (theme.rectPause.contains(p)) tip(theme.rectPause, i18n("Pause"));
  else if (theme.rectStop.contains(p))  tip(theme.rectStop, i18n("Stop"));
  else if (theme.rectSkip.contains(p))  tip(theme.rectSkip, i18n("Skip"));
  else if (theme.rectBack.contains(p))  tip(theme.rectBack, i18n("Back"));
  else if (theme.titleClipRect.contains(p)) tip(theme.titleClipRect, title);
  
  if ((theme.rectShuffle != NULL) && theme.rectShuffle->contains(p)) 
    tip(*theme.rectShuffle, i18n("Shuffle"));
  if ((theme.rectRepeat != NULL) && theme.rectRepeat->contains(p)) 
    tip(*theme.rectRepeat, i18n("Repeat"));
}



/*
 * show an about box
 * 
 */
void XmmsKde::about() {

  KAboutApplication aboutDialog(aboutData);
  aboutDialog.exec();
}


/*
 * show a help box
 * 
 */
void XmmsKde::help() {

  kapp->invokeHelp("", "xmms-kde");
  //KMessageBox::information(0, "Controls:\n - buttons on the bottom (left mousebutton): back, play, pause, stop, skip\n - time (left mousebutton): seek in current song\n - volume (left mousebutton): change volume\n - middle mousebutton: open fileselector\n\n - any button if mp3 player is not running: start player\n\n - right mousebutton: menu");
}


/*
 * preferences dialog
 * 
 */
void XmmsKde::preferences() {

  XmmsKdeConfigDialog *dialog = new XmmsKdeConfigDialog(this, *themeList, 
							dataDirs, 
							titleScrollMode,
							globalAccel,
							theme.themeName,
							player->is(),
							hideXmms,
							readID3Tags,
							convertUnderscores,
							osd,
							db,
							query);
  
  if (dialog->exec()) {

    PlayerInterface::Player newPlayer = dialog->getSelectedPlayer();

    if (!player || newPlayer != player->is()) {
      if (player) player->exitPlayer();
      delete player;
            
      if (newPlayer == PlayerInterface::XMMS) 
#ifdef HAVE_XMMS
	player = new XMMSPlayer(hideXmms);
#else
      qDebug("xmms-kde: don't have xmms");
#endif
#ifdef HAVE_NOATUN
      if (newPlayer == PlayerInterface::NOATUN) player = new NoatunPlayer();
#else
      qDebug("xmms-kde: don't have noatun");
#endif
#ifdef HAVE_SMPEG
      if (newPlayer == PlayerInterface::SMPEG) 
	player = new SMPEGPlayer(smpegPlayList, smpegPlayListPos, 
				 smpegPlayListSize, readID3Tags,
				 convertUnderscores, smpegRepeat, smpegShuffle,
				 playListIcon);
#else
      qDebug("xmms-kde: don't have smpeg");
#endif

#ifdef HAVE_AMAROK
      if (newPlayer == PlayerInterface::AMAROK) player = new AmarokPlayer();
#endif

#ifdef HAVE_MPLAYER
      if (newPlayer == PlayerInterface::MPLAYER) player = new MPlayer();
#endif

      if (!player) player = new PlayerInterface();

      if (query)
	query->setPlayer(player);
    }

    titleScrollMode = dialog->getScrollMode();

    // Since the scroll mode may have changed, we will reset
    // the scroll speed. This is needed since ping-pong may have
    // reveresed the scroll direction.
    titleScrollSpeed = titleDefaultScrollSpeed;

    if (!dialog->getSelectedTheme().isNull())
      loadTheme(dialog->getSelectedTheme());

    hideXmms = dialog->getHideXmms();
    
    readID3Tags = dialog->getReadID3Tags();
    convertUnderscores = dialog->getConvertUnderscores();

    if (player) {
#ifdef HAVE_XMMS
      if (player->is() == PlayerInterface::XMMS)
	((XMMSPlayer *) player)->minimize(hideXmms);
#endif
    }

    if (db) {

      if (!db->getSync())
	db->updateDatabase();
    }

    globalAccel->setEnabled(dialog->enableAccels());    
    setTimers(false);
    titleXPosition = theme.titleClipRect.right();
    setTimers(true);
    saveConfig();
  }
  delete dialog;

  osd->stopDemo();
}


/*
 * set size of applet
 * 
 */
int XmmsKde::widthForHeight(int) const {
  return theme.myWidth;
}


/*
 * set size of applet
 * 
 */
int XmmsKde::heightForWidth(int) const {
  return theme.myHeight;
}


void XmmsKde::resizeEvent(QResizeEvent*) {
}

void XmmsKde::paintEvent(QPaintEvent*) {
}



/*
 * repaint xmms-kde
 *
 */
void XmmsKde::paint() {
  
  QPixmap pmt(theme.myWidth, theme.myHeight);
  pmt.fill(this, 0, 0);

  QPainter pt;
  pt.begin(&pmt);

  pt.setClipping(false);

  /*
   * draw the main pixmap onto the offscreenpixmap
   */
  pt.drawPixmap(0, 0, *theme.mainPixmap);
  
  pt.setFont(theme.mainFont);
  QFontMetrics fm = pt.fontMetrics();
  
    /*
     * Draw the selected buttons.
     */
    if (pausePressed && theme.pausePixmap)
      pt.drawPixmap(theme.rectPause.x(), theme.rectPause.y(), *theme.pausePixmap);
    if (stopPressed && theme.stopPixmap)
      pt.drawPixmap(theme.rectStop.x(), theme.rectStop.y(), *theme.stopPixmap);
    if (nextPressed && theme.nextPixmap)
      pt.drawPixmap(theme.rectSkip.x(), theme.rectSkip.y(), *theme.nextPixmap);
    if (backPressed && theme.pausePixmap)
      pt.drawPixmap(theme.rectBack.x(), theme.rectBack.y(), *theme.backPixmap);
    if (playPressed && theme.playPixmap )
      pt.drawPixmap(theme.rectPlay.x(), theme.rectPlay.y(), *theme.playPixmap);
    
    
    /*
     * Draw the selected repeat / shuffle buttons if avaliable
     */
    if (repeatOn && theme.repeatOnPixmap && theme.rectRepeat) {
      pt.drawPixmap(theme.rectRepeat->x(), theme.rectRepeat->y(), 
		    *theme.repeatOnPixmap);
    }
    
    if (shuffleOn && theme.shuffleOnPixmap && theme.rectShuffle) {
      pt.drawPixmap(theme.rectShuffle->x(), theme.rectShuffle->y(), 
		    *theme.shuffleOnPixmap );
    }
    
    /*
     * draw the title
     */
    pt.setPen(theme.titleColor);
    pt.setFont(theme.titleFont);
    fm = pt.fontMetrics();
    QFontInfo fi = pt.fontInfo();
    if (theme.titleYPos > -1) {
      pt.setClipping(true);
      pt.setClipRect(theme.titleClipRect);
      titleWidth = fm.boundingRect(title).width();
      pt.drawText(titleXPosition, theme.titleYPos + fm.ascent(), title);
      pt.setClipping(false);
    }
    
    /*
     * draw the volumebar
     */
    pt.drawPixmap((int) theme.rectVolume.x() + 
		  (volume * theme.rectVolume.width() / 100) - 5, 
		  theme.rectVolume.y(), 
		  *theme.volumePixmap);

    /*
     * draw seek related things
     */
    int xv = 0;
    if (inSeekChange) {
      xv = (int) (((float) currentSeekTime) / ((float) trackTime) * 
		  theme.rectSeek.width()) + theme.rectSeek.x();
    } else {
      xv = (int) (((float) currentTime) / ((float) trackTime) * 
		  theme.rectSeek.width()) + theme.rectSeek.x();
    }
    
    pt.drawPixmap(xv - 3, theme.rectSeek.y(), *theme.seekPixmap);

    /*
     * draw elapsed time
     */
    if (theme.timePlayedPos.y() > -1) {
      pt.setPen(theme.timeColor); // set the selected pen.
      pt.setFont(theme.timeFont);
      fi = pt.fontInfo();
      fm = pt.fontMetrics();
      if (displayElapsedTime) {
	char pTime[5];
	sprintf(pTime, "%02d:%02d", 
		(int) (currentTime / 60000),
		(int) (currentTime / 1000) % 60);
	
	pt.drawText(theme.timePlayedPos.x(), theme.timePlayedPos.y() + fm.ascent(), pTime);
      } else {
	int remainingTime = trackTime - currentTime;
	char pTime[6];
	sprintf(pTime, "-%02d:%02d", 
		(int) (remainingTime / 60000),
		(int) (remainingTime / 1000) % 60);
	
	pt.drawText(theme.timePlayedPos.x(), theme.timePlayedPos.y() + fm.ascent(), pTime);
      }
    }

    /*
     * draw total time
     */
    if (theme.timeTotalPos.y() > -1) {
      char totalTime[5];
      sprintf(totalTime, "%02d:%02d", 
	      (int) (trackTime / 60000),
	      (int) (trackTime / 1000) % 60);
      
      pt.drawText( theme.timeTotalPos.x(), theme.timeTotalPos.y() + fm.ascent(), totalTime);
    }      

  pt.end();

  /*
   * draw the offscreen buffer
   */
  QPainter hvpainter(this);
  hvpainter.drawPixmap(0, 0, pmt);
}



/*
 * scroll the title, calls paint()
 * 
 */
void XmmsKde::scroll() {

  if (theme.titleYPos > -1) {
    titleXPosition -= titleScrollSpeed;
      
    if (titleWidth < (theme.titleClipRect.width() - 5)) {
      /*
       * don't scroll if title too small
       */
      titleXPosition = (int) (theme.titleClipRect.width() / 2 - titleWidth / 2);
    } else {
      
      if (titleScrollMode == 0) {
	/*
	 * normal scrolling
	 */
	if (titleScrollSpeed > 0) {
	  if (titleXPosition < -titleWidth) titleXPosition = theme.titleClipRect.right();
	  
	} else {
	  if (titleXPosition > theme.titleClipRect.right())
	    titleXPosition = -titleWidth;
	}
      } else if (titleScrollMode == 1) {
	/*
	 * ping pong scrolling
	 */
	if (titleScrollSpeed > 0) {
	  if (titleXPosition < -titleWidth + theme.titleClipRect.right() - 15)
	    titleScrollSpeed = -titleScrollSpeed;
	} else {
	  if (titleXPosition > theme.titleClipRect.left()+15) 
	    titleScrollSpeed = -titleScrollSpeed;
	} 
      } else if (titleScrollMode == 2) {
	/*
	 * sinus-scrolling
	 */
	int scrolllength = titleWidth - theme.titleClipRect.width() + 30;
	
	titleSinArg += M_PI / 3 * ((double) titleScrollSpeed) / scrolllength;
	if (titleSinArg > 2 * M_PI) titleSinArg = 0;
	
	titleXPosition = theme.titleClipRect.left() + 
	  (int) rint(15 - scrolllength * (sin(titleSinArg)*sin(titleSinArg)));
      } else if (titleScrollMode == 3) {
        /*
	 * loop once scrolling
	 *
	 * Added by Mike Pilone.
	 * Works just like normal loop, but after the text scrolls through
	 * one time, it will scroll to the left edge and stop.
	 */
	if (loopedOnce && (titleXPosition <= 2)) {
	   titleXPosition = 2;
	} else if (titleXPosition < -titleWidth) {
	   titleXPosition = theme.titleClipRect.right();
	   loopedOnce = true;
	} else if (titleXPosition > theme.titleClipRect.right()) {
	    titleXPosition = -titleWidth;
	}
      }        
    }    
  }
  paint();
}


/*
 * get values from player
 *
 */
void XmmsKde::receive() {

  QString newTitle;

  if (player) {
    volume = player->getVolume();
    currentTime = player->getCurrentTime();
    trackTime = player->getTrackTime();
    newTitle = player->getTitle();
    repeatOn = player->getRepeat();
    shuffleOn = player->getShuffle();

    running = player->isPlayerRunning();
  } else {
    newTitle = title;
  }

  // if title has changed, display osd and lyrics
  if (newTitle && title.compare(newTitle) && newTitle.compare("xmms-kde")) {
    if (osd) {
      osd->startSequence(newTitle);
      titleXPosition     = theme.titleClipRect.right();
      loopedOnce = false;
    }

    if (lyrics && lyrics->isVisible()) {
      QString lartist = newTitle.section(" - ", 0, 0);
      QString ltitle = newTitle.section(" - ", 1, 10);
      
      qDebug("search: [%s] [%s]", lartist.latin1(), ltitle.latin1());

      lyrics->request(lartist, ltitle);
    }
  }

  title = newTitle;

  if (!running) {
    currentTime = 0;
    trackTime = 0;
    volume = 0;
    title = "xmms-kde";
  }
}


/*
 * Show the on-screen-display
 *
 */
void XmmsKde::showOSD( ) {

  if (!osd) 
    return;
  
  osd->startSequence(title);
  titleXPosition = theme.titleClipRect.right();
  loopedOnce = false;
}


/*
 * mouse handling
 *
 */
void XmmsKde::mousePressEvent(QMouseEvent *event) {

  /*
   * if there is no active player process, fork one
   */  
  if (event->button() == LeftButton && player && !player->isPlayerRunning()) {
    player->startPlayer();

    if (query->getPopup() == 1) // player startup
      query->popup();
    return;
  }

  /*
   * left mouse button: remote control buttons
   */
  if (event->button() == LeftButton) {
    
    if ((event->y() >= theme.rectBack.y()) && 
        (event->y() <= (theme.rectBack.y() + theme.rectBack.height())) &&
        (event->x() >= theme.rectBack.x()) &&
        (event->x() <= (theme.rectBack.x() + theme.rectBack.width()))) {

      backPressed = true;
    }

    if ((event->y() >= theme.rectPlay.y()) && 
	(event->y() <= (theme.rectPlay.y() + theme.rectPlay.height())) &&
	(event->x() >= theme.rectPlay.x()) &&
	(event->x() <= (theme.rectPlay.x() + theme.rectPlay.width()))) {
      
      playPressed = true;
    }
    
    if ((event->y() >= theme.rectPause.y()) && 
	(event->y() <= (theme.rectPause.y() + theme.rectPause.height())) &&
	(event->x() >= theme.rectPause.x()) &&
	(event->x() <= (theme.rectPause.x() + theme.rectPause.width()))) {
      
      pausePressed = true;
    }
    
    if ((event->y() >= theme.rectStop.y()) && 
	(event->y() <= (theme.rectStop.y() + theme.rectStop.height())) &&
	(event->x() >= theme.rectStop.x()) &&
	(event->x() <= (theme.rectStop.x() + theme.rectStop.width()))) {

      stopPressed = true;
    }
        
    if ((event->y() >= theme.rectSkip.y()) && 
	(event->y() <= (theme.rectSkip.y() + theme.rectSkip.height())) &&
	(event->x() >= theme.rectSkip.x()) &&
	(event->x() <= (theme.rectSkip.x() + theme.rectSkip.width()))) {
      
      nextPressed = true;
    }
        
    if ((event->y() >= theme.rectVolume.y()) && 
	(event->y() <= (theme.rectVolume.y() + theme.rectVolume.height())) &&
	(event->x() >= theme.rectVolume.x()) &&
	(event->x() <= (theme.rectVolume.x() + theme.rectVolume.width()))) {

      int volume = (int) ((event->x() - theme.rectVolume.x()) / 
			  (float) theme.rectVolume.width() * 100);
      if (player) player->setVolume(volume);
      inVolumeChange = true;
    }

    if ((event->y() >= theme.rectSeek.y()) && 
	(event->y() <= (theme.rectSeek.y() + theme.rectSeek.height())) &&
	(event->x() >= theme.rectSeek.x()) &&
	(event->x() <= (theme.rectSeek.x() + theme.rectSeek.width()))) {

      currentSeekTime = (int) ((event->x() - theme.rectSeek.x()) / 
			       (float) theme.rectSeek.width() * trackTime);
      inSeekChange = true;
    }

    // shuffle
    if (theme.rectShuffle && (event->y() >= theme.rectShuffle->y()) && 
	(event->y() <= (theme.rectShuffle->y() + theme.rectShuffle->height()))
	&& (event->x() >= theme.rectShuffle->x()) &&
	(event->x() <= (theme.rectShuffle->x() + theme.rectShuffle->width()))){
      toggleShuffle();
    }
    
    // repeat
    if (theme.rectRepeat && (event->y() >= theme.rectRepeat->y()) && 
	(event->y() <= (theme.rectRepeat->y() + theme.rectRepeat->height())) &&
	(event->x() >= theme.rectRepeat->x()) &&
	(event->x() <= (theme.rectRepeat->x() + theme.rectRepeat->width()))) {
      toggleRepeat();
    }
  }
  
  /*
   * right mouse button: 
   *   - switch between displaying elapsed time / displaying time to play
   *   - switch between title scrolltype
   *
  else if (event->button() == 2) {

    if ((event->y() >= theme.timePlayedPos.y()) && 
	(event->y() <= (theme.timePlayedPos.y() + 10)) &&
	(event->x() >= theme.timePlayedPos.x()) &&
	(event->x() <= (theme.timePlayedPos.x() + 24))) 

      displayElapsedTime = 1 - displayElapsedTime;
    
    if ((event->y() >= theme.titleYPos) && 
	     (event->y() <= theme.titleYPos + 10)) {

      titleScrollMode++;
      if (titleScrollMode > 2) titleScrollMode = 0;     
    }

    if ((event->y() >= theme.rectStop.y()) && 
	(event->y() <= (theme.rectStop.y() + theme.rectStop.height())) &&
	(event->x() >= theme.rectStop.x()) &&
	(event->x() <= (theme.rectStop.x() + theme.rectStop.width())))

      player->exitPlayer();
  }
  */
  
  /*
   * right mouse button:
   *   - show popup menu
   */
  else if (event->button() == RightButton) {

    QPopupMenu *menu = new QPopupMenu(this);

    menu->insertItem(i18n("Jump to file"), this, SLOT(jumpToFile()));
    menu->insertItem(i18n("Play"), this, SLOT(play()));
    menu->insertItem(i18n("Previous"), this, SLOT(previous()));
    menu->insertItem(i18n("Next"), this, SLOT(next()));
    menu->insertItem(i18n("Pause"), this, SLOT(pause()));
    menu->insertItem(i18n("Stop"), this, SLOT(stop()));
    menu->insertItem(i18n("Exit Player"), this, SLOT(exitPlayer()));
    menu->insertSeparator();

    if (db) {
      menu->insertItem(i18n("Database Query Window"), this, SLOT(showDB()));
      menu->insertItem(i18n("Update Database"), db, SLOT(updateDatabase()));
    }
    
    if (lyrics) {
      menu->insertItem(i18n("Show Lyrics"), this, SLOT(showLyrics()));
    }

    menu->insertSeparator();

    menu->insertItem(i18n("Help"), this, SLOT(help()));
    menu->insertItem(i18n("Preferences"), this, SLOT(preferences()));
    menu->popup(event->globalPos());    
  }


  /*
   * middle mouse button: open fileselector
   */
  else if (event->button() == MidButton) {
    if (player) player->openFileSelector();
  }
}

/*
 * Stop neat control mode.
 */
void XmmsKde::mouseReleaseEvent(QMouseEvent *event) {
  if (inSeekChange) {
    player->setCurrentTime(currentSeekTime);
  }

  if (event->button() == 1) {
    
    if ((event->y() >= theme.rectBack.y()) && 
	(event->y() <= (theme.rectBack.y() + theme.rectBack.height())) &&
	(event->x() >= theme.rectBack.x()) &&
	(event->x() <= (theme.rectBack.x() + theme.rectBack.width())) && backPressed )
      previous();

    if ((event->y() >= theme.rectPlay.y()) && 
	(event->y() <= (theme.rectPlay.y() + theme.rectPlay.height())) &&
	(event->x() >= theme.rectPlay.x()) &&
	(event->x() <= (theme.rectPlay.x() + theme.rectPlay.width())) && playPressed )
      play();
    
    if ((event->y() >= theme.rectPause.y()) && 
	(event->y() <= (theme.rectPause.y() + theme.rectPause.height())) &&
	(event->x() >= theme.rectPause.x()) &&
	(event->x() <= (theme.rectPause.x() + theme.rectPause.width())) && pausePressed)
      pause();
    
    if ((event->y() >= theme.rectStop.y()) && 
	(event->y() <= (theme.rectStop.y() + theme.rectStop.height())) &&
	(event->x() >= theme.rectStop.x()) &&
	(event->x() <= (theme.rectStop.x() + theme.rectStop.width())) && stopPressed)
      stop();
    
    if ((event->y() >= theme.rectSkip.y()) && 
	(event->y() <= (theme.rectSkip.y() + theme.rectSkip.height())) &&
	(event->x() >= theme.rectSkip.x()) &&
	(event->x() <= (theme.rectSkip.x() + theme.rectSkip.width())) && nextPressed)
      next();
    
  }
  
  inSeekChange = false;
  inVolumeChange = false;
  pausePressed = false;
  nextPressed = false;
  backPressed = false;
  playPressed = false;
  stopPressed = false;
}

/*
 * Set volume if in volumeevent.
 */
void XmmsKde::mouseMoveEvent(QMouseEvent *event) {
  
  if (inVolumeChange) {
    int volume = (int) ((event->x() - theme.rectVolume.x()) / 
			(float) theme.rectVolume.width() * 100);
    player->setVolume(volume);
  }

  if (inSeekChange) {
    currentSeekTime = (int) ((event->x() - theme.rectSeek.x()) / 
			     (float) theme.rectSeek.width() * trackTime);
    if (currentSeekTime < theme.rectSeek.x())
      currentSeekTime = 0;
    else if ((currentSeekTime / trackTime) == 1)
      currentSeekTime = trackTime;
  }
  paint();
}

void XmmsKde::wheelEvent( QWheelEvent *event )
{
    if (event->delta() > 0)
        volp();
    else
        volm();
}


/*
 * saves the current configuration to disk
 *
 */
void XmmsKde::saveConfig() {

  if (mainConfig) {
    mainConfig->setGroup("THEMES");
    mainConfig->writeEntry("default", QFileInfo(theme.themeName).fileName());

    mainConfig->setGroup("TITLE");
    mainConfig->writeEntry("scrollmode" , titleScrollMode);
    mainConfig->writeEntry("scrollSpeed", 1);
    mainConfig->writeEntry("displayelapsedtime", displayElapsedTime);
    
    mainConfig->setGroup("PLAYER");
    if (player) mainConfig->writeEntry("player", player->is());

    if (osd)
      osd->writeConfig();

    if (db) {
      db->writeConfig();
    }

    if (query)
      query->writeConfig();

    if (lyrics)
      lyrics->writeConfig();

    if (globalAccel) {
      mainConfig->setGroup("ACCEL");
      mainConfig->writeEntry("enable", globalAccel->isEnabled());
    }

    mainConfig->setGroup("XMMS");
    mainConfig->writeEntry("hide", hideXmms);
    
#ifdef HAVE_SMPEG
    if (player && player->is() == PlayerInterface::SMPEG) {
      smpegPlayList = ((SMPEGPlayer *) player)->getPlayList();
      
      mainConfig->setGroup("SMPEG");
      mainConfig->writeEntry("recent", smpegPlayList, ",", false);
      
      mainConfig->writeEntry("playlistposition", ((SMPEGPlayer *) player)->getPlayListPos());
      mainConfig->writeEntry("playlistsize", ((SMPEGPlayer *) player)->getPlayListSize());
      mainConfig->writeEntry("readallid3tags", readID3Tags);
      mainConfig->writeEntry("convertunderscores", convertUnderscores);
      mainConfig->writeEntry("repeat", ((SMPEGPlayer *) player)->getRepeat());
      mainConfig->writeEntry("shuffle", ((SMPEGPlayer *) player)->getShuffle());
    }
#endif
    
    mainConfig->setGroup("MPLAYER");
    mainConfig->writeEntry("noborder", mplayerBorderless);
    mainConfig->writeEntry("showscreen", mplayerScreen);
 
    mainConfig->setGroup("Global Keys");

    globalAccel->updateConnections();    
    globalAccel->writeSettings(mainConfig);
   
    mainConfig->sync();
  }
}


/*
 * drag and drop support:
 * dragged object enters xmms-kde
 * 
 */
void XmmsKde::dragEnterEvent(QDragEnterEvent* event) {

  event->accept(QTextDrag::canDecode(event));
}



/*
 * drag and drop support:
 * dragged object is dropped onto xmms-kde
 * 
 */
void XmmsKde::dropEvent(QDropEvent* event) {

  if (!player)
    return;

  QString text;

  if (QTextDrag::decode(event, text)) {  

    /*
     * test if player is currently playing
     */
    bool playing = player->isPlaying();

    /*
     * clear the playlist
     */
    player->playlistClear();

    /*
     * add the file that was dropped
     */
    player->playlistAdd(text);

    /*
     * if player was playing before, restart playing
     */
    if (playing) player->play();
  }
}

void XmmsKde::jumpto(int pos) {

  player->jumpto(pos);

   if (!player->isPlaying())
    player->play();  
}

void XmmsKde::previous() {
  player->previous();
}

void XmmsKde::play() {
  player->play();
}

void XmmsKde::playpause() {

  if (player->isPlaying())
    player->pause();
  else
    player->play();
}

void XmmsKde::pause() {
  player->pause();
}

void XmmsKde::stop() {
  player->stop();
}

void XmmsKde::next() {
  player->next();
}

void XmmsKde::volp() {
  volume += 5; if (volume > 100) volume = 100;
  player->setVolume(volume);
}

void XmmsKde::volm() {
  volume -= 5; if (volume < 0) volume = 0;
  player->setVolume(volume);
}

void XmmsKde::exitPlayer() {
  player->exitPlayer();
}

void XmmsKde::seekForward() {

  int time;

  if ((currentTime + seekTime) <= trackTime) {
    time = currentTime + seekTime;
    player->setCurrentTime(time);
  } else {
    player->next();
  }
}

void XmmsKde::seekBackward() {
  
  int time;

  if (currentTime >= seekTime) {
    time = currentTime - seekTime;
  } else {
    time = 0;
  }

  player->setCurrentTime(time);
}


void XmmsKde::showDB() {
  query->popup();
}

void XmmsKde::toggleShuffle() {
  player->toggleShuffle();
}
void XmmsKde::toggleRepeat() {
  player->toggleRepeat();
}


 
/*
 * load a theme from a file
 *
 */
void XmmsKde::loadTheme(QString themeName) {

  setTimers(false);
  theme.loadTheme(themeName, dataDirs);
  
  setTimers(true);
  QSize size(theme.myWidth, theme.myHeight);
  resize(size);
}



void XmmsKde::jumpToFile() {

  if (!player)
    return;

  if (playList)
    delete playList;

  playList = new PlayList();
  QStringList *list = new QStringList();

  playList->resize(400, 300);

  QRect rect = QApplication::desktop()->geometry();
  playList->move(rect.center() - playList->rect().center());

  int length = player->getPlaylistLength();

  for (int index = 0; index < length; index ++)
    list->append(player->getPlaylistTitle(index));

  playList->setContents(list);
  playList->setSelected(player->getPlaylistPos(), true);
  playList->centerCurrentItem();

  connect((QObject *) playList, SIGNAL(sigSelectionChanged(int)),
	  (QObject *) this, SLOT(jumpto(int)));

  KWin::setOnDesktop(playList->winId(), KWin::currentDesktop());

  playList->show();
}

void XmmsKde::showLyrics() {

  if (lyrics) {
    if (!lyrics->isVisible()) {
      QString lartist = title.section(" - ", 0, 0);
      QString ltitle = title.section(" - ", 1, 10);
    
      qDebug("search: [%s] [%s]\n", lartist.latin1(), ltitle.latin1());
    
      lyrics->request(lartist, ltitle);
    }

    lyrics->show();
  }
}


PlayList::PlayList() 
  : KDialogBase (0, 0, false, i18n("Jump to file"), Ok | Cancel, Ok) {

  contents = 0;
  setFocusProxy(0);

  QVBox *box = new QVBox(this);
  
  line = new QLineEdit(box);
  playListBox = new QListBox(box);

  setMainWidget(box);

  connect((QObject *) playListBox, SIGNAL(returnPressed(QListBoxItem *)),
	  (QObject *) this, SLOT(selectionChanged()));

  connect((QObject *) line, SIGNAL(returnPressed()),
	  (QObject *) this, SLOT(selectionChanged()));

  connect((QObject *) line, SIGNAL(textChanged(const QString &)),
	  (QObject *) this, SLOT(textChanged(const QString &)));

  connect((QObject *) playListBox, SIGNAL(doubleClicked(QListBoxItem *)),
	  (QObject *) this, SLOT(selectionChanged()));

  connect((QObject *) this, SIGNAL(okClicked()),
	  (QObject *) this, SLOT(selectionChanged()));

  connect((QObject *) this, SIGNAL(cancelClicked()),
	  (QObject *) this, SLOT(cancel()));


  setCaption(i18n("Jump to file"));
  line->setFocus();

  //KWin::setIcons(winId(), *icon, *icon);
}

PlayList::~PlayList() {
}

void PlayList::setContents(QStringList *list) {

  playListBox->clear();

  playListBox->insertStringList(*list);

  if (contents)
    delete contents;

  contents = list;
}

int PlayList::getSelected() {
  return playListBox->currentItem();
}

int PlayList::setSelected(int pos, bool status) {
  playListBox->setSelected(pos, status);
  return pos;
}

void PlayList::clear() {
  playListBox->clear();
}


void PlayList::centerCurrentItem() {
  playListBox->centerCurrentItem();
}

void PlayList::selectionChanged() {
  
  // find the currently selected item in our contents list
  QString current(playListBox->currentText());
  int index = 0;

  for (QStringList::Iterator it = contents->begin(); it != contents->end(); ++it ) {
    
    if (!current.compare(*it))
      break;
    
    index ++;
  }
      
  emit sigSelectionChanged(index);
  hide();
}

void PlayList::textChanged(const QString & s) {

  QStringList::Iterator it;

  QStringList exlist = QStringList::split(" ", s);

  /*
  if (exlist.size() == 0)
    return;
  */

    // build a number of regular expressions from the input string
  QRegExp **e = new QRegExp*[exlist.size()];

  int index = 0;
  for (it = exlist.begin(); it != exlist.end(); ++it )
    e[index++] = new QRegExp(*it, false);

  playListBox->clear();

  for (it = contents->begin(); it != contents->end(); ++it ) {

    bool test = true;
    for (int i = 0; i < index; i++)
      if (e[i]->search(*it) < 0)
	test = false;
    
    if (test)
      playListBox->insertItem(*it);
  }

  delete[] e;
}

void PlayList::keyPressEvent(QKeyEvent *e) {

  if (e->key() == Qt::Key_Escape) {
    e->accept();
    hide();
  } else if (e->key() == Qt::Key_Up) {
    int s = playListBox->currentItem();
    playListBox->setFocus();
    playListBox->setSelected(s - 1, true);
  } else if (e->key() == Qt::Key_Down) {
    int s = playListBox->currentItem();
    playListBox->setFocus();
    playListBox->setSelected(s + 1, true);
    e->ignore();
  } else {
    e->ignore();
  }
}

void PlayList::cancel() {

  hide();
}

void PlayList::focusOutEvent(QFocusEvent *e) {

  hide();
}

/********************* THEME CLASS  *************************/
XmmsKdeTheme::XmmsKdeTheme() {
  mainFont = QFont("helvetica", 10);
  mainPixmap= 0;
  seekPixmap = 0;
  volumePixmap = 0;
  backPixmap= 0;
  nextPixmap = 0;
  pausePixmap = 0;
  stopPixmap = 0;
  playPixmap = 0;
  shuffleOnPixmap = 0;
  repeatOnPixmap = 0;
  
  rectShuffle = 0;
  rectRepeat = 0;
}


XmmsKdeTheme::~XmmsKdeTheme() {
  deleteOld();
}

void XmmsKdeTheme::deleteOld() {

  if (mainPixmap) delete mainPixmap;
  if (seekPixmap) delete seekPixmap;
  if (volumePixmap) delete volumePixmap;
  if (backPixmap) delete backPixmap;
  if (nextPixmap) delete nextPixmap;
  if (pausePixmap) delete pausePixmap;
  if (stopPixmap) delete stopPixmap;
  if (playPixmap) delete playPixmap;
  if (repeatOnPixmap) delete repeatOnPixmap;
  if (shuffleOnPixmap) delete shuffleOnPixmap;
  if (rectShuffle) delete rectShuffle;
  if (rectRepeat) delete rectRepeat;

  mainPixmap = 0;
  seekPixmap = 0;
  volumePixmap = 0;
  backPixmap= 0;
  nextPixmap = 0;
  pausePixmap = 0;
  stopPixmap = 0;
  playPixmap = 0;
  shuffleOnPixmap = 0;
  repeatOnPixmap = 0;
  rectShuffle = 0;
  rectRepeat = 0;
}

int XmmsKdeTheme::loadTheme(QString name, KStandardDirs *dataDirs) {

  themeName = name;
  qDebug("xmms-kde: loading theme :" + themeName);
  /*
   * load theme
   */
  KSimpleConfig* themeConfig = new KSimpleConfig(themeName);
  
  themeConfig->setGroup("BITMAPS");
  QString file = dataDirs->findResource("data", "xmms-kde/" + themeConfig->readEntry("file"));
  if (file.isNull()) {
    file = dataDirs->findResource("data", themeConfig->readEntry("file"));
    if (file.isNull()) {
      qDebug("xmms-kde: error loading theme");
      return 0;
    }
  }

  deleteOld();

  QString mainBitmapFile   = themeConfig->readEntry("main");
  QString volumeBitmapFile = themeConfig->readEntry("volume");
  QString seekBitmapFile   = themeConfig->readEntry("seek");

  QString backBitmapFile = themeConfig->readEntry("back");
  QString nextBitmapFile = themeConfig->readEntry("next");
  QString pauseBitmapFile = themeConfig->readEntry("pause");
  QString playBitmapFile = themeConfig->readEntry("play");
  QString stopBitmapFile = themeConfig->readEntry("stop");

  QString shuffleOnBitmapFile = themeConfig->readEntry("shuffleon");
  QString repeatOnBitmapFile = themeConfig->readEntry("repeaton");
  
  themeConfig->setGroup("SIZE");    
  myWidth  = themeConfig->readNumEntry("width");
  myHeight = themeConfig->readNumEntry("height");
  
  themeConfig->setGroup("CONTROL");    
  rectBack  = themeConfig->readRectEntry("back");    
  rectPlay  = themeConfig->readRectEntry("play");
  rectPause = themeConfig->readRectEntry("pause");
  rectStop  = themeConfig->readRectEntry("stop");
  rectSkip  = themeConfig->readRectEntry("skip");
    
  themeConfig->setGroup("VOLUME");
  rectVolume = themeConfig->readRectEntry("area");
  
  themeConfig->setGroup("SEEK");
  rectSeek = themeConfig->readRectEntry("area");

  QString defaultFont = QString("helvetica");
  int defaultFontSize = 10;

  QColor defaultColor = QColor(200, 200, 200);

  themeConfig->setGroup("TITLE");
  titleYPos = themeConfig->readNumEntry("ypos");

  titleFont = QFont(themeConfig->readEntry("font", defaultFont));
  int size = themeConfig->readNumEntry("fontsize", defaultFontSize);
  titleFont.setPixelSize(size);

  titleColor = themeConfig->readColorEntry("color", &defaultColor);
  titleClipRect = themeConfig->readRectEntry( "clip" );
  if (titleClipRect.isNull())
    qDebug("xmms-kde: error while reading clip rect");
  
  themeConfig->setGroup("TIME");
  timePlayedPos = themeConfig->readPointEntry("played");
  timeTotalPos  = themeConfig->readPointEntry("total");
  timeColor = themeConfig->readColorEntry("color", &defaultColor);

  timeFont = QFont(themeConfig->readEntry("font", defaultFont));
  size = themeConfig->readNumEntry("fontsize", defaultFontSize);
  timeFont.setPixelSize(size);
  
  themeConfig->setGroup("TOGGLES");
  rectShuffle = new QRect( themeConfig->readRectEntry("shuffle") );
  rectRepeat = new QRect( themeConfig->readRectEntry("repeat") );
  if(rectShuffle->isNull()) {
    delete rectShuffle; rectShuffle = 0;
  }
  if (rectRepeat->isNull()) {
    delete rectRepeat; rectRepeat = 0;
  }
  delete themeConfig;

  loadBasicSkin(file, mainBitmapFile, volumeBitmapFile, seekBitmapFile);
  loadExtendedSkin(file, backBitmapFile, nextBitmapFile, pauseBitmapFile,
		   stopBitmapFile, playBitmapFile, shuffleOnBitmapFile,
		   repeatOnBitmapFile);

  return 1;
}


/*!
  Loads the target skin from disk.
  Returns true if it succeeded, false otherwise.
 */
bool XmmsKdeTheme::loadBasicSkin(QString file, QString mainBitmapFile, 
				 QString volumeBitmapFile, 
				 QString seekBitmapFile) {

  bool skinloadfailed = false;

  // main.bmp
  mainPixmap = new QPixmap(readFile(file, mainBitmapFile)); 
  if (!mainPixmap->isNull()) {
    qDebug("xmms-kde: Found " + mainBitmapFile);
  } else {
    skinloadfailed = TRUE;
    qDebug("xmms-kde: Didn't find " + mainBitmapFile);
  }

  // volume.bmp
  volumePixmap = new QPixmap(readFile(file, volumeBitmapFile));
  if (!volumePixmap->isNull()) {
    qDebug("xmms-kde: Found " + volumeBitmapFile);
  } else {
    skinloadfailed = TRUE;
    qDebug("xmms-kde: Didn't find " + volumeBitmapFile);
  }

  // seek.bmp
  seekPixmap = new QPixmap(readFile(file, seekBitmapFile));
  if (!seekPixmap->isNull()) {
    qDebug("xmms-kde: Found " + seekBitmapFile);
  } else {
    skinloadfailed = TRUE;
    qDebug("xmms-kde: Didn't find " + seekBitmapFile);
  }
 
  if (skinloadfailed) {
    qDebug("xmms-kde: Skin could not be loaded!\n");
  }
  return skinloadfailed;
}

bool XmmsKdeTheme::loadExtendedSkin(QString file, 
				    QString &backBitmapFile, 
				    QString &nextBitmapFile,
				    QString &pauseBitmapFile,
				    QString &stopBitmapFile,
				    QString &playBitmapFile,
				    QString &shuffleBitmapFile,
				    QString &repeatBitmapFile) {

  if (!backBitmapFile.isNull()) {
    backPixmap = new QPixmap(readFile(file, backBitmapFile));
    if (backPixmap->isNull()) {
      delete backPixmap;
      backPixmap = 0;
    }
  }
  
  if (!nextBitmapFile.isNull()) {
    nextPixmap = new QPixmap(readFile(file, nextBitmapFile));
    if (nextPixmap->isNull()) {
      delete nextPixmap;
      nextPixmap = 0;
    }
  }
  
  if(!pauseBitmapFile.isNull()) {
    pausePixmap = new QPixmap(readFile(file, pauseBitmapFile));
    if (pausePixmap->isNull()) {
      delete pausePixmap;
      pausePixmap = 0;
    }
  }
  
  if (!stopBitmapFile.isNull()) {
    stopPixmap = new QPixmap(readFile(file, stopBitmapFile));
    if (stopPixmap->isNull()) {
      delete stopPixmap;
      stopPixmap = 0;
    }
  }
  
  if (!playBitmapFile.isNull()) {
    playPixmap = new QPixmap(readFile(file, playBitmapFile));
    if (playPixmap->isNull()) {
      delete playPixmap;
      playPixmap = 0;
    }
  }
 
  if(!shuffleBitmapFile.isNull()) {
    shuffleOnPixmap = new QPixmap(readFile(file, shuffleBitmapFile));
    if (shuffleOnPixmap->isNull()) {
      delete shuffleOnPixmap;
      shuffleOnPixmap = 0;
    }
  }
  
  if (!repeatBitmapFile.isNull()) {
    repeatOnPixmap = new QPixmap(readFile(file, repeatBitmapFile));
    if (repeatOnPixmap->isNull()) {
      delete repeatOnPixmap;
      repeatOnPixmap = 0;
    }
  }

  return true;
}
