
#ifndef __GENERAL_FILE_SYS_H__
#define __GENERAL_FILE_SYS_H__

//#define EG_DEBUG 	1

#ifdef EG_DEBUG
#pragma message( "Warning: EG_DEBUG debug is on" );
#endif

#include "UtilStr.h"
#include "ErrorInfo.h"
class XLongList;
class FileSysIDList;

/*   A general file system is something that stores an retrieves data with associated
titles.  "File objects" are *any* entity in a file system, including objects that store
data (files), objects can contain other file objects (folders), and references to other
objects (aliases).
     The term "file" is used loosely here because a GeneralFileSys can apply to a hard
disk (as we most commonly see it), it could apply to a single file where each "file"
is a sub part of that file (ex, the "resource fork" on Macs), or it could even apply
to a wide area network where the root level's "folders" are sub-networks, the next
sub-folder are really computers and so on.  The goal here is to make a base class
for structure.
     Sub classes of GeneralFileSys implement GeneralFileSys's functions to
a single file in an OS (ResourceFileSys), to a folder in the OS where subsequent IO
and catalog operations use the OS file system (FolderFileSys), or to RAM for
use as a flexible data structure (MemFileSys).
     A big feature of all GeneralFileSys instances is that all file object IDs are
persistently unique.  That is, no two file objects will *ever* have the same ID, even
after an item is deleted.  You can store an object's ID permanently and always
use it to fetch the data object.
*/


typedef 	unsigned long 		FileObj;
typedef		unsigned long		FileResult;

// ======  Special FileObj values
#define		FILE_SYS_ROOT_ID			1
#define		FILE_SYS_DUPE_ID			2			/*	Used internally to denote identical names  */
#define		FILE_SYS_INVALID_ID			0

// ======  Possible FileResult values
#define 	FILE_SYS_SUCCESS			-1
#define 	FILE_SYS_FAIL				0

// ======  File Object flags.  Every file object can have the following flags set.
//  Use Create() and GetInfo() to load flags and use SetInfo() to store flags.
#define		FILE_SYS_DATA				0x0001
#define		FILE_SYS_FOLDER				0x0002
#define		FILE_SYS_READ_ONLY			0x0004		/*	Can't be written to	*/
#define		FILE_SYS_INVISIBLE			0x0008
#define		FILE_SYS_ALIAS  			0x0010
#define		FILE_SYS_LOCKED				0x0020    	/*	Can't be deleted, renamed, or moved */

#define		FILE_SYS_USER_FLAGS			0xFF00
#define		FILE_SYS_USER_FLAG1			0x0100
#define		FILE_SYS_USER_FLAG2			0x0200
#define		FILE_SYS_USER_FLAG3			0x0400
#define		FILE_SYS_USER_FLAG4			0x0800
#define		FILE_SYS_USER_FLAG5			0x1000
#define		FILE_SYS_USER_FLAG6			0x2000
#define		FILE_SYS_USER_FLAG7			0x4000
#define		FILE_SYS_USER_FLAG8			0x8000


#define 	FILE_SYS_USER_ACCESSABLE_FLAGS		( FILE_SYS_USER_FLAGS | FILE_SYS_INVISIBLE | FILE_SYS_LOCKED | FILE_SYS_READ_ONLY )

// ======  For use for Create()
#define 	FILE_SYS_REPLACE			0x010000	/* Tells Create() to overrwrite anything with the given name */

// ======  Error codes
#define 	FILE_SYS_FILE_NEEDED		-54
#define		FILE_SYS_FOLDER_NEEDED		-55
#define		FILE_SYS_FNF			-56
#define		FILE_SYS_AMBIGUOUS_NAME		-57
#define		FILE_SYS_IN_USE			-58
#define		FILE_SYS_ALREADY_EXISTS		-59
#define		FILE_SYS_NO_ROOT_PARENT		-60	/* The root has no parent, dumbass. */
#define		FILE_SYS_ITEM_LOCKED		-61	/* Item is locked */
#define		FILE_SYS_UNSUPPORTED_FCN	-62	/* File system does not support this fcn	*/
#define		FILE_SYS_PARAM_NOT_FOUND	-63	/* The GetInfo/SetInfo param wasn't found */
#define 	FILE_SYS_ROOT_LOCKED		-64	/* Can't move, rename, or generally screw with the root */
#define		FILE_SYS_NAME_TOO_LONG		-66
#define 	FILE_SYS_ILLEGAL_CHARS		-67
#define		FILE_SYS_BAD_ASSIGN_SPEC	-68
#define		FILE_SYS_WRITE_ERR		-69
#define		FILE_SYS_OPEN_ERR		-70
#define 	FILE_SYS_CREATE_ERR		-71
#define 	FILE_SYS_FULL_ERR		-72
#define		FILE_SYS_NO_WRITE_PERM		-73
#define 	FILE_SYS_VOL_LOCKED		-74
#define		FILE_SYS_PATH_NOT_FOUND		-75
#define 	FILE_SYS_TOME_LOAD_ERR		-76
#define 	FILE_SYS_TOME_WRITE_ERR		-77
#define 	FILE_SYS_READ_ERR		-78
#define		FILE_SYS_ACCESS_DENIED		-79

/*
#define		FILE_SYS_SORT_BY_NAME_CI	55
#define		FILE_SYS_SORT_BY_NAME_CS	56
#define		FILE_SYS_SORT_BY_DATE	57
#define		FILE_SYS_SORT_BY_SIZE		58
*/

class GeneralFileSys {


	public:

		// If inNamesUnique is true, all items in the same folder have to have names and must be distinct
		GeneralFileSys( bool inNamesUnique );
		virtual ~GeneralFileSys();

		// NOTE: For any FileObj, you may use FILE_SYS_ROOT_ID as the ID

		// Obtains a list of sub file objects within a given file object
		// Note:  ioList is not cleared before starting, so remove its contents if you need to...
		virtual FileResult			Catalog( FileObj inID, XLongList& ioList ) = 0;
		//virtual FileResult			Catalog( FileObj inID, XLongList& ioList, long inSort );

		// Returns a file ID for a file residing in a given parent and with a given name.
		// If two or more items with the given name are in parent folder (inNamesUnique would have to have
		//     been false), the FILE_SYS_AMBIGUOUS_NAME error will result, *but* one of the item's ID will be returned in outID.
		virtual FileResult			GetItemID( FileObj inParentID, const UtilStr& inName, 	FileObj& outID ) = 0;
		FileResult					GetItemID( FileObj inParentID, const char* inName, 		FileObj& outID );

		// Get and set info about a file object
		//		'size' accesses how many bytes the data is
		//		'subs' returns how many items are inside the object
		//		'flag' accesses all the flags associated with the object
		//		'ext'  returns the 1-4 chars right of the lastmost '.' in the name in uppercase (ex, '.JPG', 'MPEG', '.C')
		//			   If the name contains no extension, then 0 is returned
		virtual long				GetInfo( FileObj inID, long inParamID, FileResult* outResult = NULL ) = 0;
		virtual FileResult			SetInfo( FileObj inID, long inParamID, long inParam ) = 0;
		bool						Exists( FileObj inID )														{ return GetInfo( inID, MCC4_TO_INT("flag") );								}

		//	Sets the title of the given object.
		virtual	FileResult			SetName( FileObj inID, const UtilStr& inName ) = 0;

		// 	Gets the title of the given object
		virtual FileResult			GetName( FileObj inID, UtilStr& outName );

		// NULL indicates a failure to retrieve the name.
		// GetName() should only be used immediately after the call and not used again.  In other words, if you
		// want to keep the name around, make a copy of the data in the UtilStr* returned.
		virtual const UtilStr*		GetName( FileObj inID ) = 0;

		// Atomically fetches data associated with a file ID
		// Returns 0 on failure, non-zero otherwise
		virtual FileResult			Read( FileObj inID, UtilStr& outBuf ) = 0;

		// Atomically stores data associated with a file ID
		// Returns 0 on failure, non-zero otherwise.
		virtual FileResult			WriteData( FileObj inID, void* inData, long inNumBytes ) = 0;
		FileResult					Write( FileObj inID, UtilStr& inBuf ) 										{ return WriteData( inID, inBuf.getCStr(), inBuf.length() );	}

		// Creates a new object inside the given parent.  If:
		//     - a name is provided, and
		//     - the FILE_SYS_REPLACE flag is on, and
		//     - names need to be unique, and
		//     - there's already an item in the given parent with an identical name,
		//    then the pre-existing item will be replaced.
		// If you don't pass a name and names need to be unique,
		//    then a unique name is chosen for you (ex, Untitled Folder, Unititled 4).
		// Legal flags: FILE_SYS_FOLDER, FILE_SYS_DATA, FILE_SYS_REPLACE
		// Turn the FILE_SYS_REPLACE bit on in inFlags to overwrite a preexisting item with the given name.
		// If Create() fails because of a FILE_SYS_ALREADY_EXISTS error, the ID of the existing item is returned in outID, otherwise FILE_SYS_INVALID_ID is returned on an error.
		virtual FileResult			Create( FileObj inParent, const UtilStr& inName, FileObj& outID, long inFlags ) = 0;
		virtual FileResult			Create( FileObj inParent, FileObj& outID, long inFlags );

		// Deletes a file object from this file system
		virtual FileResult			Delete( FileObj inID ) = 0;

		// If this item is a folder, everything inside it is deleted
		virtual FileResult			DeleteContents( FileObj inID ) = 0;

		// Creates an alias to a file object and places it in the original's folder. If unique names are required, the alias
		// will have an "Alias" appended to it's name (and possibly a number if that name already exists)
		//virtual FileResult			CreateAlias( FileObj inOriginal, FileObj& outAlias );

		// Moves a file object to a new parent directory.
		// If inReplace is true, an object in the dest directory with a matching name will be deleted
		virtual FileResult			Move( FileObj inID, FileObj inNewParent, bool inReplace = true ) = 0;

		// Returns the parent of the given file object.  Note, if the object is an alias, the alias' parent
		// is returned, not the parent of the object the alias references.
		virtual FileResult			GetParent( FileObj inID, FileObj& outParent ) = 0;

		// Returns a complete textual pathname of the given file object with a given character separating each name
		// NOTE:  If duplicate names are allowed, it's possible the pathname will be ambiguous.  This will show up if
		// you use parse the pathname and make calls to GetItemID() in a folder where there's dupe file objects...
		// NOTE:  Even if the item is a folder, no separator char will be added onto outPathName.
		virtual FileResult			GetPathname( FileObj inID, UtilStr& outPathName, char inSepChar );

		// Any GeneralFileSys fcn that fails will have associated error info.
		virtual long				GetLastError();
		void						GetLastError( UtilStr& outMsg );

		// Use this to accociate a message w/ the most recent error
		void						SetErrorMsg( const char* inMsg );

	protected:

		// Util fcn that stores error info
		virtual void				ThrowErr( long inErr, FileObj inID );
		virtual void				ThrowErr( long inErr, FileObj inID, const char* inMsg );
		virtual void				ThrowErr( long inErr, long inOSErr, FileObj inID, const char* inMsg );
		virtual void				ThrowErr( GeneralFileSys& inFileSys );

		FileResult					ThrowUnimp();

		virtual const char*			GetErrMsg( long inErrID, long inOSErrID );

		//bool						mCaseSensitiveNames;
		bool						mUniqueNames;
		UtilStr						mTempName;

		ErrorInfo					mLastErr;

};

#endif // __GENERAL_FILE_SYS_H__
