/*  XMMS STATUS PLUGIN - Status Docklet Plugin for XMMS
 *  Copyright (C) 2000,2001,2002  Ian Campbell.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public Licensse as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <gtk/gtk.h>

#include <xmms/configfile.h>

#include "xmms-status-plugin.h"

#define STATUS_DOCKLET_IMAGE_DELAY_MIN 0.0
#define STATUS_DOCKLET_IMAGE_DELAY_MAX 5000.0

#define STATUS_DOCKLET_BALLOON_DELAY_MIN 0.0
#define STATUS_DOCKLET_BALLOON_DELAY_MAX 10.0

#define STATUS_DOCKLET_CONFIG_SECTION "status_docklet"

StatusDockletConfig status_docklet_config;

static void status_docklet_cancel_cb(GtkWidget *widget, gpointer data);
static void status_docklet_ok_cb(GtkWidget *widget, gpointer data);

static void status_docklet_browse_cb(GtkWidget *widget, GtkWidget *entry);
static void status_docklet_browse_ok_callback(GtkWidget *widget, GtkWidget *entry);
static void status_docklet_browse_cancel_callback(GtkWidget *widget, GtkWidget *entry);

static void status_docklet_create_button_pane(GtkWidget *notebook, StatusDockletButtonModifier modifier);

static GtkWidget *button_action[STATUS_DOCKLET_BUTTON_MODIFIER_LAST][STATUS_DOCKLET_BUTTON_LAST];
static GtkWidget *playing_image_entry;
static GtkWidget *paused_image_entry;
static GtkWidget *stopped_image_entry;

static GtkObject *playing_image_adjustment;
static GtkObject *paused_image_adjustment;
static GtkObject *stopped_image_adjustment;

static GtkObject *balloon_delay_adjustment;

static GtkWidget *freedesktop_support_check_button;

static GtkWidget *config_window;

static gchar *button_action_text[] = {
	N_ ("None"),
	N_ ("Toggle All Windows"),
	N_ ("Play/Pause"),
	N_ ("Stop"),
	N_ ("Volume +5%"),
	N_ ("Volume -5%"),
	N_ ("Next Track"),
	N_ ("Previous Track"),
	N_ ("Popup Menu"),
	N_ ("Add Files"),
	NULL
};

static gchar *button_modifier_text [] = {
	N_ ("None"),
	N_ ("Shift"),
	N_ ("Control"),
	N_ ("Alt"),
	NULL
};

void status_docklet_save_config(void)
{
	ConfigFile *cfgfile;
	GtkWidget *active;

	gint button, mod;
	gchar *key;

	cfgfile = xmms_cfg_open_default_file();

	for (mod=0; mod<STATUS_DOCKLET_BUTTON_MODIFIER_LAST; mod++) {
		for (button=0; button<STATUS_DOCKLET_BUTTON_LAST; button++) {
			active = gtk_menu_get_active(GTK_MENU(button_action[mod][button]));
			status_docklet_config.button_actions[mod][button] = GPOINTER_TO_INT(gtk_object_get_data( GTK_OBJECT(active), "cmd"));
		}
	}

	g_free(status_docklet_config.playing_image);
	status_docklet_config.playing_image = g_strdup(gtk_entry_get_text(GTK_ENTRY(playing_image_entry)));

	g_free(status_docklet_config.paused_image);
	status_docklet_config.paused_image = g_strdup(gtk_entry_get_text(GTK_ENTRY(paused_image_entry)));

	g_free(status_docklet_config.stopped_image);
	status_docklet_config.stopped_image = g_strdup(gtk_entry_get_text(GTK_ENTRY(stopped_image_entry)));

	status_docklet_config.playing_image_delay = (gint)GTK_ADJUSTMENT(playing_image_adjustment)->value;
	status_docklet_config.paused_image_delay = (gint)GTK_ADJUSTMENT(paused_image_adjustment)->value;
	status_docklet_config.stopped_image_delay = (gint)GTK_ADJUSTMENT(stopped_image_adjustment)->value;

	status_docklet_config.balloon_delay = (gint)(GTK_ADJUSTMENT(balloon_delay_adjustment)->value);

	for (mod=0; mod<STATUS_DOCKLET_BUTTON_MODIFIER_LAST; mod++) {
		for (button=0; button<STATUS_DOCKLET_BUTTON_LAST; button++) {
			key = g_strdup_printf("button_action_%s%s%d",
					      (mod==STATUS_DOCKLET_BUTTON_MODIFIER_NONE) ? "" : button_modifier_text[mod],
					      (mod==STATUS_DOCKLET_BUTTON_MODIFIER_NONE) ? "" : "_",
					      button+1);
			xmms_cfg_write_int(cfgfile, STATUS_DOCKLET_CONFIG_SECTION, 
					   key, status_docklet_config.button_actions[mod][button]);
			g_free(key);
		}
	}
	
	status_docklet_config.freedesktop_docklet_support = gtk_toggle_button_get_active
		(GTK_TOGGLE_BUTTON(freedesktop_support_check_button));

	if ( status_docklet_config.playing_image )
		xmms_cfg_write_string(cfgfile, STATUS_DOCKLET_CONFIG_SECTION, 
				      "playing_image", status_docklet_config.playing_image);
	if ( status_docklet_config.paused_image )
		xmms_cfg_write_string(cfgfile, STATUS_DOCKLET_CONFIG_SECTION, 
				      "paused_image", status_docklet_config.paused_image);
	if ( status_docklet_config.stopped_image )
		xmms_cfg_write_string(cfgfile, STATUS_DOCKLET_CONFIG_SECTION, 
				      "stopped_image", status_docklet_config.stopped_image);

	xmms_cfg_write_int(cfgfile, STATUS_DOCKLET_CONFIG_SECTION, 
			   "playing_image_delay", status_docklet_config.playing_image_delay);
	xmms_cfg_write_int(cfgfile, STATUS_DOCKLET_CONFIG_SECTION, 
			   "paused_image_delay", status_docklet_config.paused_image_delay);
	xmms_cfg_write_int(cfgfile, STATUS_DOCKLET_CONFIG_SECTION, 
			   "stopped_image_delay", status_docklet_config.stopped_image_delay);

	xmms_cfg_write_int(cfgfile, STATUS_DOCKLET_CONFIG_SECTION,
			   "balloon_delay", status_docklet_config.balloon_delay);

	xmms_cfg_write_boolean(cfgfile, STATUS_DOCKLET_CONFIG_SECTION, 
			       "freedesktop_system_tray", status_docklet_config.freedesktop_docklet_support);

	xmms_cfg_write_default_file(cfgfile);
	
	xmms_cfg_free(cfgfile);

	status_docklet_load_images();
}

void status_docklet_load_config(void)
{
	ConfigFile *cfgfile;
	gchar *tmp;
	gint mod, button;
	gint delay;

	/* Setup defaults */
	status_docklet_config.button_actions[STATUS_DOCKLET_BUTTON_MODIFIER_NONE][0] = STATUS_DOCKLET_BUTTON_ACTION_TOGGLE_WINDOWS;
	status_docklet_config.button_actions[STATUS_DOCKLET_BUTTON_MODIFIER_NONE][1] = STATUS_DOCKLET_BUTTON_ACTION_FORWARD;
	status_docklet_config.button_actions[STATUS_DOCKLET_BUTTON_MODIFIER_NONE][2] = STATUS_DOCKLET_BUTTON_ACTION_MENU;
	/* and the rest ... */
	for(button = 3; button < STATUS_DOCKLET_BUTTON_LAST; button++)
		status_docklet_config.button_actions
			[STATUS_DOCKLET_BUTTON_MODIFIER_NONE][button] = 
			STATUS_DOCKLET_BUTTON_ACTION_NONE;

	for (mod=1; mod<STATUS_DOCKLET_BUTTON_MODIFIER_LAST; mod++)
		for (button=0; button<STATUS_DOCKLET_BUTTON_LAST; button++)
			status_docklet_config.button_actions[mod][button] = 
				STATUS_DOCKLET_BUTTON_ACTION_NONE;
			     
	status_docklet_config.playing_image = 
		g_strdup(XMMS_DATA_DIR "/status_docklet/stave-anim.xpm");
	status_docklet_config.paused_image = 
		g_strdup(XMMS_DATA_DIR "/status_docklet/rest.xpm");
	status_docklet_config.stopped_image = g_strdup("");

	status_docklet_config.playing_image_delay = 250;
	status_docklet_config.paused_image_delay = 250;
	status_docklet_config.stopped_image_delay = 250;

	status_docklet_config.freedesktop_docklet_support = TRUE;
	status_docklet_config.balloon_delay = 2;

	cfgfile = xmms_cfg_open_default_file();
	if (cfgfile) {
		gchar *key;
		/* If these fail then the value won't be updated, so
                   the above defaults prevail. */
		for (mod=0; mod<STATUS_DOCKLET_BUTTON_MODIFIER_LAST; mod++) {
			for (button=0; button<STATUS_DOCKLET_BUTTON_LAST; button++) {
				key = g_strdup_printf("button_action_%s%s%d",
						      (mod==STATUS_DOCKLET_BUTTON_MODIFIER_NONE) ? "" : button_modifier_text[mod],
						      (mod==STATUS_DOCKLET_BUTTON_MODIFIER_NONE) ? "" : "_",
						      button+1);
				xmms_cfg_read_int(cfgfile, STATUS_DOCKLET_CONFIG_SECTION, 
						  key, (gint*)&status_docklet_config.button_actions[mod][button]);
				g_free(key);
			}
		}
	
		if ( xmms_cfg_read_string (cfgfile, STATUS_DOCKLET_CONFIG_SECTION, "playing_image", &tmp) ) {
			g_free (status_docklet_config.playing_image);
			status_docklet_config.playing_image = tmp;
			tmp = NULL;
		}
		if ( xmms_cfg_read_string (cfgfile, STATUS_DOCKLET_CONFIG_SECTION, "paused_image", &tmp) ) {
			g_free (status_docklet_config.paused_image);
			status_docklet_config.paused_image = tmp;
			tmp = NULL;
		}

		if ( xmms_cfg_read_string (cfgfile, STATUS_DOCKLET_CONFIG_SECTION, "stopped_image", &tmp) ) {
			g_free (status_docklet_config.stopped_image);
			status_docklet_config.stopped_image = tmp;
			tmp = NULL;
		}

		if ( xmms_cfg_read_int (cfgfile, STATUS_DOCKLET_CONFIG_SECTION, "playing_image_delay", 
					&delay) &&
		     delay > STATUS_DOCKLET_IMAGE_DELAY_MIN && 
		     delay < STATUS_DOCKLET_IMAGE_DELAY_MAX )
			status_docklet_config.playing_image_delay = delay;

		if ( xmms_cfg_read_int (cfgfile, STATUS_DOCKLET_CONFIG_SECTION, "paused_image_delay", 
					&delay) &&
		     delay > STATUS_DOCKLET_IMAGE_DELAY_MIN && 
		     delay < STATUS_DOCKLET_IMAGE_DELAY_MAX )
			status_docklet_config.paused_image_delay = delay;

		if ( xmms_cfg_read_int (cfgfile, STATUS_DOCKLET_CONFIG_SECTION, "paused_image_delay", 
				   &delay) &&
		     delay > STATUS_DOCKLET_IMAGE_DELAY_MIN && 
		     delay < STATUS_DOCKLET_IMAGE_DELAY_MAX )
			status_docklet_config.stopped_image_delay = delay;
		
		if ( xmms_cfg_read_int(cfgfile, STATUS_DOCKLET_CONFIG_SECTION, "balloon_delay", 
				  &delay) &&
		     delay > STATUS_DOCKLET_BALLOON_DELAY_MIN && 
		     delay < STATUS_DOCKLET_BALLOON_DELAY_MAX )
			status_docklet_config.balloon_delay = delay;

		xmms_cfg_free(cfgfile);
	}


	status_docklet_load_images();

	status_docklet_config.configured = TRUE;
}

static void status_docklet_freedesktop_support_toggled(GtkWidget *toggle, GtkWidget *spin)
{
	gboolean active = gtk_toggle_button_get_active
		(GTK_TOGGLE_BUTTON(toggle));
	gtk_widget_set_sensitive( spin, active );
}

static void status_docklet_cancel_cb(GtkWidget *widget, gpointer data)
{
	gtk_widget_destroy(config_window);
}

static void status_docklet_ok_cb(GtkWidget *widget, gpointer data)
{
	status_docklet_save_config();
	gtk_widget_destroy(config_window);
}

static void status_docklet_create_button_pane(GtkWidget *notebook, StatusDockletButtonModifier modifier)
{
	GtkWidget *table;
	GtkWidget *option, *menu_item, *label;
	gint button, action;
	
	table = gtk_table_new ( STATUS_DOCKLET_BUTTON_LAST, 2, FALSE );
	gtk_container_set_border_width(GTK_CONTAINER(table), 5);
	gtk_table_set_row_spacings(GTK_TABLE(table), 5);
	gtk_table_set_col_spacings(GTK_TABLE(table), 2);
	gtk_widget_show(table);

	/* ------------------------- */

	for ( button = 0; button < STATUS_DOCKLET_BUTTON_LAST; button++ ) {
		button_action[modifier][button] = gtk_menu_new ();
		for ( action = 0 ; action < STATUS_DOCKLET_BUTTON_ACTION_LAST ; action++ ) {
			menu_item = gtk_menu_item_new_with_label ( _( button_action_text[action] ) );	
				
			gtk_object_set_data(GTK_OBJECT(menu_item), "cmd", GINT_TO_POINTER(action));
			gtk_object_set_data(GTK_OBJECT(menu_item), "mod", GINT_TO_POINTER(modifier));
			gtk_widget_show(menu_item);
			gtk_menu_append(GTK_MENU(button_action[modifier][button]), menu_item);
		}
		gtk_widget_show(button_action[modifier][button]);
	}

	for(button = 0; button < STATUS_DOCKLET_BUTTON_LAST; button++ ) {
		gchar *lbl;
		lbl = g_strdup_printf(_ ("Button %d"), button+1);
		label = gtk_label_new ( lbl );
		g_free(lbl);

		gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);

		gtk_table_attach(GTK_TABLE(table), label, 0, 1, 
				 button, button+1, 
				 GTK_SHRINK, GTK_SHRINK, 3, 3);
		gtk_widget_show(label);
		
		option = gtk_option_menu_new();
		gtk_option_menu_set_menu (GTK_OPTION_MENU(option), 
					  button_action[modifier][button]);
		gtk_option_menu_set_history(GTK_OPTION_MENU(option), 
					    status_docklet_config.button_actions[modifier][button]);
		
		gtk_table_attach(GTK_TABLE(table), option, 1, 2, button, button+1, GTK_EXPAND|GTK_FILL, GTK_SHRINK, 3, 3);
		gtk_widget_show(option);
	}

	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), table, gtk_label_new( _( button_modifier_text[modifier] ) ) );
}

void status_docklet_configure(void)
{

	if ( !status_docklet_config.configured )
		status_docklet_load_config();

	if ( !config_window ) {
		GtkWidget *vbox, *notebook, *box, *frame, *label;
		GtkWidget *button_notebook, *button, *spin, *sep;
		GtkWidget *table;
		gint mod;

		config_window = gtk_window_new (GTK_WINDOW_DIALOG);
		gtk_signal_connect(GTK_OBJECT(config_window), "destroy", 
				   GTK_SIGNAL_FUNC(gtk_widget_destroyed), &config_window);
		gtk_window_set_title(GTK_WINDOW(config_window), _("XMMS Status Docklet Configuration"));
		gtk_container_border_width(GTK_CONTAINER(config_window), 5);

		vbox = gtk_vbox_new ( FALSE, 3 );
		gtk_container_add (GTK_CONTAINER(config_window), vbox);
		gtk_widget_show(vbox);

		notebook = gtk_notebook_new ();
		gtk_box_pack_start ( GTK_BOX(vbox), notebook, TRUE, TRUE, 0);
		gtk_widget_show(notebook);

		frame = gtk_frame_new ( NULL );
		gtk_widget_show(frame);

		button_notebook = gtk_notebook_new ();
		gtk_container_add ( GTK_CONTAINER(frame), button_notebook );
		gtk_widget_show(button_notebook);
		
		for (mod=0; mod<STATUS_DOCKLET_BUTTON_MODIFIER_LAST; mod++)
			status_docklet_create_button_pane(button_notebook, mod);

		gtk_notebook_append_page(GTK_NOTEBOOK(notebook), frame, gtk_label_new(_("Button Action")));

		/* ------------------------- */
		/* Images Page */

		frame = gtk_frame_new ( NULL );
		gtk_widget_show(frame);

		table = gtk_table_new ( 0, 0, FALSE );	
		gtk_container_set_border_width(GTK_CONTAINER(table), 5);
		gtk_container_add ( GTK_CONTAINER(frame), table );
		gtk_table_set_row_spacings(GTK_TABLE(table), 5);
		gtk_table_set_col_spacings(GTK_TABLE(table), 5);
		gtk_widget_show(table);

		/* ------------------------- */

		label = gtk_label_new ( _("Playing") );
		gtk_misc_set_alignment ( GTK_MISC (label), 0.0, 0.5);
		gtk_table_attach(GTK_TABLE(table), label, 0, 3, 0, 1, GTK_EXPAND|GTK_FILL, GTK_SHRINK, 2, 1);
		gtk_widget_show(label);

		label = gtk_label_new ( _("Filename") );
		gtk_misc_set_alignment ( GTK_MISC (label), 0.0, 0.5);
		gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, GTK_SHRINK|GTK_FILL, GTK_SHRINK, 2, 1);
		gtk_widget_show (label);

		playing_image_entry = gtk_entry_new();
		if (status_docklet_config.playing_image != NULL)
			gtk_entry_set_text(GTK_ENTRY(playing_image_entry), status_docklet_config.playing_image);
		gtk_table_attach(GTK_TABLE(table), playing_image_entry, 1, 2, 1, 2, GTK_EXPAND|GTK_FILL, GTK_SHRINK, 0, 1);
		gtk_widget_show(playing_image_entry);

		button = gtk_button_new_with_label ( _("Browse") );
		gtk_signal_connect ( GTK_OBJECT(button), "clicked", status_docklet_browse_cb, playing_image_entry);		
		gtk_table_attach(GTK_TABLE(table), button, 2, 3, 1, 2, GTK_SHRINK|GTK_FILL, GTK_SHRINK, 0, 1);
		gtk_widget_show(button);

		label = gtk_label_new ( _("Frame Delay") );
		gtk_misc_set_alignment ( GTK_MISC (label), 0.0, 0.5);
		gtk_table_attach ( GTK_TABLE(table), label, 0, 1, 2, 3, GTK_SHRINK|GTK_FILL, GTK_SHRINK, 2, 1);
		gtk_widget_show (label);

		playing_image_adjustment = gtk_adjustment_new(status_docklet_config.playing_image_delay, 
							      STATUS_DOCKLET_IMAGE_DELAY_MIN, 
							      STATUS_DOCKLET_IMAGE_DELAY_MAX, 
							      10.0, 100.0, 100.0);
		spin = gtk_spin_button_new( GTK_ADJUSTMENT(playing_image_adjustment), 10, 0);
		gtk_table_attach ( GTK_TABLE(table), spin, 1, 2, 2, 3, GTK_EXPAND|GTK_FILL, GTK_SHRINK, 0, 1);
		gtk_widget_show(spin);
		
		sep = gtk_hseparator_new();
		gtk_table_attach ( GTK_TABLE(table), sep, 0, 3, 3, 4, GTK_EXPAND|GTK_FILL, GTK_SHRINK, 0, 2);
		gtk_widget_show(sep);

		/* ------------------------- */

		label = gtk_label_new ( _("Paused") );
		gtk_misc_set_alignment ( GTK_MISC (label), 0.0, 0.5);
		gtk_table_attach(GTK_TABLE(table), label, 0, 3, 4, 5, GTK_EXPAND|GTK_FILL, GTK_SHRINK, 2, 1);
		gtk_widget_show(label);

		label = gtk_label_new ( _("Filename") );
		gtk_misc_set_alignment ( GTK_MISC (label), 0.0, 0.5);
		gtk_table_attach(GTK_TABLE(table), label, 0, 1, 5, 6, GTK_SHRINK|GTK_FILL, GTK_SHRINK, 2, 1);
		gtk_widget_show (label);

		paused_image_entry = gtk_entry_new();
		if (status_docklet_config.paused_image != NULL)
			gtk_entry_set_text(GTK_ENTRY(paused_image_entry), status_docklet_config.paused_image);
		gtk_table_attach(GTK_TABLE(table), paused_image_entry, 1, 2, 5, 6, GTK_EXPAND|GTK_FILL, GTK_SHRINK, 0, 1);
		gtk_widget_show(paused_image_entry);

		button = gtk_button_new_with_label ( _("Browse") );
		gtk_signal_connect ( GTK_OBJECT(button), "clicked", status_docklet_browse_cb, paused_image_entry);
		gtk_table_attach(GTK_TABLE(table), button, 2, 3, 5, 6, GTK_SHRINK|GTK_FILL, GTK_SHRINK, 0, 1);
		gtk_widget_show(button);

		label = gtk_label_new ( _("Frame Delay") );
		gtk_misc_set_alignment ( GTK_MISC (label), 0.0, 0.5);
		gtk_table_attach ( GTK_TABLE(table), label, 0, 1, 7, 8, GTK_SHRINK|GTK_FILL, GTK_SHRINK, 2, 1);
		gtk_widget_show (label);

		paused_image_adjustment = gtk_adjustment_new(status_docklet_config.paused_image_delay, 
							     STATUS_DOCKLET_IMAGE_DELAY_MIN, 
							     STATUS_DOCKLET_IMAGE_DELAY_MAX, 
							     10.0, 100.0, 100.0);
		spin = gtk_spin_button_new( GTK_ADJUSTMENT(paused_image_adjustment), 10, 0);
		gtk_table_attach ( GTK_TABLE(table), spin, 1, 2, 7, 8, GTK_EXPAND|GTK_FILL, GTK_SHRINK, 0, 1);
		gtk_widget_show(spin);
		
		sep = gtk_hseparator_new();
		gtk_table_attach ( GTK_TABLE(table), sep, 0, 3, 8, 9, GTK_EXPAND|GTK_FILL, GTK_SHRINK, 0, 2);
		gtk_widget_show(sep);

		/* ------------------------- */

		label = gtk_label_new ( _("Stopped") );
		gtk_misc_set_alignment ( GTK_MISC (label), 0.0, 0.5);
		gtk_table_attach(GTK_TABLE(table), label, 0, 3, 9, 10, GTK_EXPAND|GTK_FILL, GTK_SHRINK, 2, 1);
		gtk_widget_show(label);

		label = gtk_label_new ( _("Filename") );
		gtk_misc_set_alignment ( GTK_MISC (label), 0.0, 0.5);
		gtk_table_attach(GTK_TABLE(table), label, 0, 1, 10, 11, GTK_SHRINK|GTK_FILL, GTK_SHRINK, 2, 1);
		gtk_widget_show (label);

		stopped_image_entry = gtk_entry_new();
		if (status_docklet_config.stopped_image != NULL)
			gtk_entry_set_text(GTK_ENTRY(stopped_image_entry), status_docklet_config.stopped_image);
		gtk_table_attach(GTK_TABLE(table), stopped_image_entry, 1, 2, 10, 11, GTK_EXPAND|GTK_FILL, GTK_SHRINK, 0, 1);
		gtk_widget_show(stopped_image_entry);

		button = gtk_button_new_with_label ( _("Browse") );
		gtk_signal_connect ( GTK_OBJECT(button), "clicked", status_docklet_browse_cb, stopped_image_entry);
		gtk_table_attach(GTK_TABLE(table), button, 2, 3, 10, 11, GTK_SHRINK|GTK_FILL, GTK_SHRINK, 0, 1);
		gtk_widget_show(button);

		label = gtk_label_new ( _("Frame Delay") );
		gtk_misc_set_alignment ( GTK_MISC (label), 0.0, 0.5);
		gtk_table_attach ( GTK_TABLE(table), label, 0, 1, 12, 13, GTK_SHRINK|GTK_FILL, GTK_SHRINK, 2, 1);
		gtk_widget_show (label);

		stopped_image_adjustment = gtk_adjustment_new(status_docklet_config.stopped_image_delay, 
							      STATUS_DOCKLET_IMAGE_DELAY_MIN, 
							      STATUS_DOCKLET_IMAGE_DELAY_MAX, 
							      10.0, 100.0, 100.0);
		spin = gtk_spin_button_new( GTK_ADJUSTMENT(stopped_image_adjustment), 10, 0);
		gtk_table_attach ( GTK_TABLE(table), spin, 1, 2, 12, 13, GTK_EXPAND|GTK_FILL, GTK_SHRINK, 0, 1);
		gtk_widget_show(spin);
		
		/* ------------------------- */

		gtk_notebook_append_page(GTK_NOTEBOOK(notebook), frame, gtk_label_new(_("Docklet Images")));

		/* ------------------------- */
		/* Freedesktop.org Support Page */

		frame = gtk_frame_new ( NULL );
		gtk_widget_show(frame);
		
		table = gtk_table_new ( 0, 0, FALSE );	
		gtk_container_set_border_width(GTK_CONTAINER(table), 5);
		gtk_container_add ( GTK_CONTAINER(frame), table );
		gtk_table_set_row_spacings(GTK_TABLE(table), 5);
		gtk_table_set_col_spacings(GTK_TABLE(table), 5);
		gtk_widget_show(table);

		freedesktop_support_check_button 
			= gtk_check_button_new_with_label(_("Support freedesktop.org System Tray\n" 
							    "Specification. (Requires restart)"));
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(freedesktop_support_check_button),
					     status_docklet_config.freedesktop_docklet_support);

		gtk_table_attach ( GTK_TABLE(table), freedesktop_support_check_button, 0, 2, 0, 1, 
				   GTK_SHRINK|GTK_FILL, GTK_SHRINK, 2, 1);
		gtk_widget_show(freedesktop_support_check_button);

		label = gtk_label_new ( _("Balloon Timeout (seconds)") );
		gtk_misc_set_alignment ( GTK_MISC (label), 0.0, 0.5);
		gtk_table_attach ( GTK_TABLE(table), label, 0, 1, 1, 2, GTK_SHRINK|GTK_FILL, GTK_SHRINK, 2, 1);
		gtk_widget_show (label);

		balloon_delay_adjustment = gtk_adjustment_new(status_docklet_config.balloon_delay, 
							      STATUS_DOCKLET_BALLOON_DELAY_MIN, 
							      STATUS_DOCKLET_BALLOON_DELAY_MAX, 
							      1.0, 1.0, 1.0);
		spin = gtk_spin_button_new( GTK_ADJUSTMENT(balloon_delay_adjustment), 10, 0);
		gtk_widget_set_sensitive( GTK_WIDGET(spin), status_docklet_config.freedesktop_docklet_support);
		
		gtk_signal_connect( GTK_OBJECT(freedesktop_support_check_button), "toggled",
				    (GtkSignalFunc)status_docklet_freedesktop_support_toggled, spin);				    
		gtk_table_attach ( GTK_TABLE(table), spin, 1, 2, 1, 2, GTK_EXPAND|GTK_FILL, GTK_SHRINK, 0, 1);
		gtk_widget_show(spin);

		/* ------------------------- */
		
		gtk_notebook_append_page(GTK_NOTEBOOK(notebook), frame, gtk_label_new(_("System Tray Protocol")));
		
		/* ------------------------- */
		
		box = gtk_hbutton_box_new();
		gtk_button_box_set_layout(GTK_BUTTON_BOX(box), GTK_BUTTONBOX_END);
		gtk_button_box_set_spacing(GTK_BUTTON_BOX(box), 5);
		gtk_box_pack_start(GTK_BOX(vbox), box, FALSE, FALSE, 0);
		gtk_widget_show(box);

		button = gtk_button_new_with_label(_("Ok"));
		gtk_signal_connect_object(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(status_docklet_ok_cb), NULL);
		GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
		gtk_box_pack_start(GTK_BOX(box), button, TRUE, TRUE, 0);
		gtk_widget_grab_default(button);
		gtk_widget_show(button);

		button = gtk_button_new_with_label(_("Cancel"));
		gtk_signal_connect_object(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(status_docklet_cancel_cb), NULL);
		GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
		gtk_box_pack_start(GTK_BOX(box), button, TRUE, TRUE, 0);
		gtk_widget_show(button);

		/* ------------------------- */

   		gtk_window_set_policy(GTK_WINDOW(config_window), FALSE, FALSE, FALSE);
		gtk_widget_show(config_window);
		
	}

	
}

static void status_docklet_browse_cb(GtkWidget *widget, GtkWidget *entry)
{
	GtkWidget *file_sel;
	gchar *file_name;

	file_sel = gtk_file_selection_new ( _("Select an Image to Load"));

	file_name = gtk_entry_get_text ( GTK_ENTRY(entry) );

	if ( file_name && file_name[0] != '\0' )
		gtk_file_selection_set_filename ( GTK_FILE_SELECTION(file_sel), file_name);
	else
		gtk_file_selection_set_filename ( GTK_FILE_SELECTION(file_sel), XMMS_DATA_DIR "/status_docklet/" );
		
	gtk_file_selection_hide_fileop_buttons ( GTK_FILE_SELECTION(file_sel) );
	gtk_window_set_modal ( GTK_WINDOW(file_sel), TRUE);
	gtk_window_set_transient_for ( GTK_WINDOW(file_sel), GTK_WINDOW(config_window));

	gtk_object_set_data ( GTK_OBJECT(file_sel), "file-entry", (gpointer)entry );

	gtk_signal_connect ( GTK_OBJECT (GTK_FILE_SELECTION(file_sel)->ok_button), "clicked", 
			     status_docklet_browse_ok_callback, (gpointer)file_sel);
	gtk_signal_connect ( GTK_OBJECT (GTK_FILE_SELECTION(file_sel)->cancel_button), "clicked", 
			     status_docklet_browse_cancel_callback, (gpointer)file_sel);

	gtk_widget_show ( file_sel );
}

static void status_docklet_browse_ok_callback(GtkWidget *widget, GtkWidget *file_sel)
{
	GtkWidget *file_entry;
	gchar *file_name;

	file_entry = GTK_WIDGET( gtk_object_get_data ( GTK_OBJECT(file_sel), "file-entry" ) );

	file_name = gtk_file_selection_get_filename ( GTK_FILE_SELECTION (file_sel) );

	gtk_entry_set_text( GTK_ENTRY(file_entry), file_name );

	gtk_object_destroy( GTK_OBJECT(file_sel) );

}

static void status_docklet_browse_cancel_callback(GtkWidget *widget, GtkWidget *file_sel)
{
	gtk_object_destroy( GTK_OBJECT(file_sel) );
}

