/*
# X-BASED DIAL
#
#  Dial.c
#
###
#
#  Copyright (c) 1994 - 2006	David Albert Bagley, bagleyd@tux.org
#
#                   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "useful",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

/* Methods file for Dial */

#include <math.h>
#ifndef M_PI
#define M_PI 3.14159265358979323846
#endif
#include "DialP.h"

#define RT_ANGLE 90.0
#define ST_ANGLE 180.0
#define RADIANS(x) (M_PI*(x)/ST_ANGLE)
#define DEGREES(x) ((x)/M_PI*ST_ANGLE)

static void InitializeDial(Widget request, Widget renew);
static void ExposeDial(Widget renew, XEvent *event, Region region);
static void ResizeDial(DialWidget w);
static void DestroyDial(Widget old);
static Boolean SetValuesDial(Widget current, Widget request, Widget renew);
static void SelectDial(DialWidget w, XEvent *event, char **args, int n_args);
static void CheckDial(DialWidget w);

static char defaultTranslationsDial[] =
	"<Btn1Down>: select()\n\
	 <Btn1Motion>: select()";
static XtActionsRec actionsListDial[] =
{
	{(char *) "select", (XtActionProc) SelectDial}
};
static XtResource resourcesDial[] =
{
	{XtNwidth, XtCWidth, XtRDimension, sizeof(Dimension),
	 XtOffset(DialWidget, core.width),
	 XtRString, (caddr_t) "100"},
	{XtNheight, XtCHeight, XtRDimension, sizeof(Dimension),
	 XtOffset(DialWidget, core.height),
	 XtRString, (caddr_t) "100"},
	{XtNforeground, XtCForeground, XtRPixel, sizeof(Pixel),
	 XtOffset(DialWidget, dial.foreground),
	 XtRString, (caddr_t) "Black"},
	{XtNindicatorColor, XtCColor, XtRPixel, sizeof(Pixel),
	 XtOffset(DialWidget, dial.indicatorColor),
	 XtRString, (caddr_t) "Blue"},
	{XtNmarkers, XtCMarkers, XtRInt, sizeof(int),
	 XtOffset(DialWidget, dial.markers),
	 XtRString, (caddr_t) "17"},
	{XtNangle, XtCAngle, XtRInt, sizeof(int),
	 XtOffset(DialWidget, dial.angle),
	 XtRString, (caddr_t) "270"},
	{XtNminimum, XtCMinimum, XtRInt, sizeof(int),
	 XtOffset(DialWidget, dial.minimum),
	 XtRString, (caddr_t) "0"},
	{XtNmaximum, XtCMaximum, XtRInt, sizeof(int),
	 XtOffset(DialWidget, dial.maximum),
	 XtRString, (caddr_t) "100"},
	{XtNmarkerLength, XtCPosition, XtRPosition, sizeof(Position),
	 XtOffset(DialWidget, dial.markerLength),
	 XtRString, (caddr_t) "5"},
	{XtNval, XtCVal, XtRInt, sizeof(int),
	 XtOffset(DialWidget, dial.val), XtRString, (caddr_t) "0"},
	{XtNversionOnly, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(DialWidget, dial.versionOnly),
	 XtRString, (caddr_t) "FALSE"},
	{XtNselectCallback, XtCCallback, XtRCallback, sizeof(caddr_t),
	 XtOffset(DialWidget, dial.select),
	 XtRCallback, (caddr_t) NULL}
};

DialClassRec dialClassRec =
{
	{
		(WidgetClass) & widgetClassRec,		/* superclass */
		(char *) "Dial",	/* class name */
		sizeof(DialRec),	/* widget size */
		NULL,		/* class initialize */
		NULL,		/* class part initialize */
		FALSE,		/* class inited */
		(XtInitProc) InitializeDial,	/* initialize */
		NULL,		/* initialize hook */
		XtInheritRealize,	/* realize */
		actionsListDial,	/* actions */
		XtNumber(actionsListDial),	/* num actions */
		resourcesDial,		/* resources */
		XtNumber(resourcesDial),	/* num resources */
		NULLQUARK,	/* xrm class */
		TRUE,		/* compress motion */
		TRUE,		/* compress exposure */
		TRUE,		/* compress enterleave */
		TRUE,		/* visible interest */
		(XtWidgetProc) DestroyDial,	/* destroy */
		(XtWidgetProc) ResizeDial,	/* resize */
		(XtExposeProc) ExposeDial,	/* expose */
		(XtSetValuesFunc) SetValuesDial,	/* set values */
		NULL,		/* set values hook */
		XtInheritSetValuesAlmost,	/* set values almost */
		NULL,		/* get values hook */
		NULL,		/* accept focus */
		XtVersion,	/* version */
		NULL,		/* callback private */
		defaultTranslationsDial,	/* tm table */
		NULL,		/* query geometry */
		NULL,		/* display accelerator */
		NULL		/* extension */
	},
	{
		0		/* ignore */
	}
};

WidgetClass dialWidgetClass = (WidgetClass) &dialClassRec;

static void
intCat(char ** string, const char * var1, const int var2)
{
	if (!(*string = (char *) malloc(strlen(var1) + 21)))
		XtError("Not enough memory, exiting.");
	(void) sprintf(*string, "%s%d", var1, var2);
}

static void
stringCat(char ** string, const char * var1, const char * var2)
{
	if (!(*string = (char *) malloc(strlen(var1) + strlen(var2) + 1)))
		XtError("Not enough memory, exiting.");
	(void) sprintf(*string, "%s%s", var1, var2);
}

static void InitializeDial(Widget request, Widget renew)
{
	DialWidget w = (DialWidget) renew;
	XGCValues values;
	XtGCMask valueMask;

	CheckDial(w);
	valueMask = GCForeground | GCBackground;
	values.foreground = w->dial.foreground;
	values.background = w->core.background_pixel;
	w->dial.dialGC = XtGetGC(renew, valueMask, &values);
	values.foreground = w->dial.indicatorColor;
	w->dial.indicatorGC = XtGetGC(renew, valueMask, &values);
	valueMask = GCForeground | GCBackground;
	values.foreground = w->core.background_pixel;
	values.background = w->dial.indicatorColor;
	w->dial.inverseGC = XtGetGC(renew, valueMask, &values);

	ResizeDial(w);
}

static void DestroyDial(Widget old)
{
	DialWidget w = (DialWidget) old;

	XtReleaseGC(old, w->dial.indicatorGC);
	XtReleaseGC(old, w->dial.dialGC);
	XtReleaseGC(old, w->dial.inverseGC);
	XtRemoveCallbacks(old, XtNselectCallback, w->dial.select);
}

static void ResizeDial(DialWidget w)
{
	double angle, cosine, sine, increment;
	int i;
	XSegment *ptr;

	ptr = w->dial.segments;
	w->dial.center.x = w->core.width / 2;
	w->dial.center.y = w->core.height / 2;
	increment = RADIANS(w->dial.angle) / (float) (w->dial.markers - 1);
	w->dial.outerDiam = MIN(w->core.width, w->core.height) / 2;
	w->dial.innerDiam = w->dial.outerDiam - w->dial.markerLength;
	angle = RADIANS(w->dial.angle / 2 - w->dial.angle);
	for (i = 0; i < w->dial.markers; i++) {
		cosine = cos(angle);
		sine = sin(angle);
		ptr->x1 = (short int) (w->dial.center.x +
			w->dial.outerDiam * sine);
		ptr->y1 = (short int) (w->dial.center.y -
			w->dial.outerDiam * cosine);
		ptr->x2 = (short int) (w->dial.center.x +
			w->dial.innerDiam * sine);
		ptr++->y2 = (short int) (w->dial.center.y -
			w->dial.innerDiam * cosine);
		angle += increment;
	}
	calculate_indicator_pos(w);
}

static void ExposeDial(Widget renew, XEvent *event, Region region)
{
	DialWidget w = (DialWidget) renew;

	if (w->core.visible) {
		XDrawSegments(XtDisplay(w), XtWindow(w),
			w->dial.dialGC, w->dial.segments, w->dial.markers);
		XDrawLine(XtDisplay(w), XtWindow(w), w->dial.indicatorGC,
			w->dial.center.x, w->dial.center.y,
			w->dial.indicator.x, w->dial.indicator.y);
	}
}

static Boolean SetValuesDial (Widget current, Widget request, Widget renew)
{
	DialWidget c = (DialWidget) current, w = (DialWidget) renew;
	XGCValues values;
	XtGCMask valueMask;
	Boolean redraw = FALSE;
	Boolean redraw_indicator = FALSE;

	CheckDial(w);
	if (w->dial.indicatorColor != c->dial.indicatorColor ||
			w->core.background_pixel != c->core.background_pixel) {
		valueMask = GCForeground | GCBackground;
		values.foreground = w->dial.indicatorColor;
		values.background = w->core.background_pixel;
		XtReleaseGC(renew, w->dial.indicatorGC);
		w->dial.indicatorGC = XtGetGC(renew, valueMask, &values);
		values.foreground = w->core.background_pixel;
		values.background = w->dial.indicatorColor;
		XtReleaseGC(renew, w->dial.inverseGC);
		w->dial.inverseGC = XtGetGC(renew, valueMask, &values);
		redraw_indicator = TRUE;
	}
	if (w->dial.foreground != c->dial.foreground) {
		valueMask = GCForeground | GCBackground;
		values.foreground = w->dial.foreground;
		values.background = w->core.background_pixel;
		XtReleaseGC(renew, w->dial.dialGC);
		w->dial.dialGC = XtGetGC(renew, valueMask, &values);
		redraw = TRUE;
	}
	if (w->dial.val != c->dial.val ||
			w->dial.minimum != c->dial.minimum ||
			w->dial.maximum != c->dial.maximum) {
		calculate_indicator_pos(w);
		redraw_indicator = TRUE;
	}
	if (redraw_indicator && !redraw && XtIsRealized(renew) &&
			renew->core.visible) {
		XDrawLine(XtDisplay(c), XtWindow(c), c->dial.inverseGC,
			c->dial.center.x, c->dial.center.y,
			c->dial.indicator.x, c->dial.indicator.y);
		XDrawLine(XtDisplay(w), XtWindow(w), w->dial.indicatorGC,
			w->dial.center.x, w->dial.center.y,
			w->dial.indicator.x, w->dial.indicator.y);
	}
	return (redraw);
}

static void SelectDial(DialWidget w, XEvent *event, char **args, int n_args)
{
	Position pos;
	double angle;
	float avg;
	dialCallbackStruct cb;
	int x, y;

	pos = w->dial.val;
	if (event->type == ButtonPress || event->type == MotionNotify) {
		x = (event->xbutton.y - w->dial.center.y);
		y = (event->xbutton.x - w->dial.center.x);
		if (x != 0 || y != 0) {
			angle = atan2((double) y, (double) -x);
		} else
			angle = 0;
		avg = (w->dial.maximum + w->dial.minimum) / 2.0;
		pos = (short int) (avg + angle * avg /
			RADIANS(w->dial.angle / 2));
		if (pos < w->dial.minimum)
			 pos = w->dial.minimum;
		if (pos > w->dial.maximum)
			 pos = w->dial.maximum;
#if DEBUG
		(void) printf("angle %g, pos %d\n", angle, pos);
#endif
	}
	cb.reason = DIAL_SELECTED;
	cb.event = event;
	cb.val = pos;
	XtCallCallbacks((Widget) w, XtNselectCallback, &cb);
}

static void CheckDial(DialWidget w)
{
	char *buf1 = NULL, *buf2 = NULL;

	if (w->dial.minimum >= w->dial.maximum) {
		intCat(&buf1, "Maximum (", w->dial.maximum);
		stringCat(&buf2, buf1, ") must be greater than minimum (");
		free(buf1);
		intCat(&buf1, buf2, w->dial.minimum);
		free(buf2);
		stringCat(&buf2, buf1, "), minimum defaulting to ");
		free(buf1);
				w->dial.minimum = w->dial.maximum - 1;
		intCat(&buf1, buf2, w->dial.minimum);
		free(buf2);
		XtWarning(buf1);
		free(buf1);
	}
	if (w->dial.val < w->dial.minimum && w->dial.val > w->dial.maximum) {
		intCat(&buf1, "Dial value out of bounds, use ",
			w->dial.minimum);
		stringCat(&buf2, buf1, "..");
		free(buf1);
		intCat(&buf1, buf2, w->dial.maximum);
		free(buf2);
		stringCat(&buf2, buf1, ", defaulting to ");
		free(buf1);
		intCat(&buf1, buf2, w->dial.minimum);
		free(buf2);
		XtWarning(buf1);
		free(buf1);
		w->dial.val = w->dial.minimum;
	}
	if (w->dial.markers < 2 && w->dial.markers > MAXSEGMENTS / 2) {
		w->dial.markers = MAXSEGMENTS / 2;
		intCat(&buf1, "Dial value out of bounds, use 2..",
			MAXSEGMENTS / 2);
		stringCat(&buf2, buf1, ", defaulting to ");
		free(buf1);
		intCat(&buf1, buf2, w->dial.markers);
		free(buf2);
		XtWarning(buf1);
		free(buf1);
	}
}

void calculate_indicator_pos(DialWidget w)
{
	double angle;
	float avg;
	Position indicatorLength;

	indicatorLength = w->dial.outerDiam - w->dial.markerLength - 2;
	avg = (w->dial.maximum + w->dial.minimum) / 2.0;
	angle = w->dial.val * RADIANS(w->dial.angle / 2) / avg +
		 RADIANS(ST_ANGLE - RT_ANGLE - w->dial.angle / 2);
#if DEBUG
	(void) printf("angle %g, avg %g\n", angle, avg);
#endif
	w->dial.indicator.x = (short int) (w->dial.center.x -
		indicatorLength * cos(angle));
	w->dial.indicator.y = (short int) (w->dial.center.y -
		indicatorLength * sin(angle));
}
