/*
 * Copyright 2006-2007 Luc Verhaegen.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

/*
 * This file handles everything to do with the Host Bridge.
 * Things like FB sizing, FB memory typing and direct access are handled here.
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "via_driver.h"
#include "via_id.h"

/*
 * Get the northbridge type.
 */
void
ViaHostIdentify(ScrnInfoPtr pScrn)
{
    struct {
        CARD16 ID;
        CARD8 Host;
        char *Name;
    } Hosts[] = {
        {0x3123, VIA_HOST_CLE266, "CLE266"},
        {0x3205, VIA_HOST_KM400 , "KM400/KN400"},
        {0x0296, VIA_HOST_P4M800, "P4M800"},
        {0x0204, VIA_HOST_K8M800, "K8M800/K8N800"},
        {0xFFFF, 0x00, NULL}
    };
    VIAPtr pVia = VIAPTR(pScrn);
    PCITAG Tag = pciTag(0, 0, 0);
    CARD16 ID;
    int i;

    VIAFUNC(pScrn->scrnIndex);

    ID = pciReadWord(Tag, 0x02);

    for (i = 0; Hosts[i].Name; i++)
        if (Hosts[i].ID == ID) {
            pVia->Host = Hosts[i].Host;
            pVia->HostRev = pciReadByte(Tag, 0xF6);

            xf86DrvMsg(pScrn->scrnIndex, X_PROBED,
                       "Found %s HostBridge (rev. 0x%02X).\n",
                       Hosts[i].Name, pVia->HostRev);
            return;
        }

    xf86DrvMsg(pScrn->scrnIndex, X_WARNING,
               "%s: Unable to identify HostBridge\n", __func__);
    pVia->Host = VIA_HOST_UNKNOWN;
}

/*
 * Talk to the host bridge directly to get RAM timing for bandwidth checking.
 *
 */
static CARD8
CLE266RAMTypeGet(ScrnInfoPtr pScrn)
{
    PCITAG Tag = pciTag(0, 0, 0);
    CARD8 Reg54, Reg60, Reg69, RegE3;
    int freq = 0;

    VIAFUNC(pScrn->scrnIndex);

    Reg54 = pciReadByte(Tag, 0x54);
    Reg69 = pciReadByte(Tag, 0x69);
    switch (Reg54 >> 6) { /* FSB frequency */
    case 0x01: /* 100Mhz */
        switch (Reg69 >> 6) {
        case 0x00:
            freq = 100;
            break;
        case 0x01:
            freq = 133;
            break;
        case 0x02:
            freq = 66;
            break;
        default: /* Reserved */
            xf86DrvMsg(pScrn->scrnIndex, X_WARNING,
                       "%s: Illegal FSB/DRAM frequency.\n", __func__);
            return VIA_MEM_NONE;
        }
        break;
    case 0x02: /* 133Mhz */
    case 0x03:
        switch (Reg69 >> 6) {
        case 0x00:
            freq = 133;
            break;
        case 0x02:
            freq = 100;
            break;
        default: /* Reserved or excessive 166Mhz */
            xf86DrvMsg(pScrn->scrnIndex, X_WARNING,
                       "%s: Illegal FSB/DRAM frequency.\n", __func__);
            return VIA_MEM_NONE;
        }
        break;
    default: /* Reserved */
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING,
                   "%s: Illegal FSB frequency.\n", __func__);
        break;
    }

    Reg60 = pciReadByte(Tag, 0x60);

    RegE3 = pciReadByte(Tag, 0xE3);
    if (RegE3 & 0x02) /* FB is on banks 2/3 */
        Reg60 >>= 2;

    switch (Reg60 & 0x03) { /* Memory type */
    case 0x00: /* SDR */
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING,
                   "%s: SDR memory is not handled.\n", __func__);
        return VIA_MEM_NONE;
    case 0x02: /* DDR */
        switch (freq) {
        case 100:
            return VIA_MEM_DDR200;
        case 133:
            return VIA_MEM_DDR266;
        default:
            xf86DrvMsg(pScrn->scrnIndex, X_WARNING,
                       "%s: Unhandled RAM frequency: %dMhz.\n", __func__, freq);
            return VIA_MEM_NONE;
        }
    default:
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING,
                   "%s: Illegal RAM type.\n", __func__);
        return VIA_MEM_NONE;
    }
}

/*
 * Values come from the BIOS. The FSB value was easy to distinguish.
 * How the FSB->RAM relation works out is not that clear, especially
 * not on KM400A.
 */
static CARD8
KM400RAMTypeGet(ScrnInfoPtr pScrn)
{
    VIAPtr pVia = VIAPTR(pScrn);
    PCITAG Tag = pciTag(0, 0, 0);
    CARD8 FSB, RAM;

    VIAFUNC(pScrn->scrnIndex);

    FSB = pciReadByte(Tag, 0x54) >> 6; /* FSB frequency */
    RAM = pciReadByte(Tag, 0x69) >> 6; /* FSB vs DRAM Clock */

    if (pVia->HostRev < 0x80) { /* KM400 */
        switch (FSB) { /* FSB Clock */
        case 0x00:
            switch (RAM) {
            case 0x00:
                return VIA_MEM_DDR200;
            case 0x01:
                return VIA_MEM_DDR266;
            case 0x02:
                return VIA_MEM_DDR400;
            case 0x03:
                return VIA_MEM_DDR333;
            }
            break;
        case 0x01:
            switch (RAM) {
            case 0x00:
                return VIA_MEM_DDR266;
            case 0x01:
                return VIA_MEM_DDR333;
            case 0x02:
                return VIA_MEM_DDR400;
            }
            break;
        case 0x02:
        case 0x03: /* No 200Mhz FSB on KM400 */
            switch (RAM) {
            case 0x00:
                return VIA_MEM_DDR333;
            case 0x02:
                return VIA_MEM_DDR400;
            case 0x03:
                return VIA_MEM_DDR266;
            }
            break;
        }
    } else { /* KM400A */
        if (pciReadByte(Tag, 0x67) & 0x80) /* Beats me */
            RAM |= 0x04;

        switch (FSB) {
        case 0x00:
            switch (RAM) {
            case 0x00:
                return VIA_MEM_DDR200;
            case 0x01:
                return VIA_MEM_DDR266;
            case 0x03:
                return VIA_MEM_DDR333;
            case 0x07:
                return VIA_MEM_DDR400;
            }
            break;
        case 0x01:
            switch (RAM) {
            case 0x00:
                return VIA_MEM_DDR266;
            case 0x01:
                return VIA_MEM_DDR333;
            case 0x03:
                return VIA_MEM_DDR400;
            }
            break;
        case 0x02:
            switch (RAM) {
            case 0x00:
                return VIA_MEM_DDR400;
            case 0x04:
                return VIA_MEM_DDR333;
            case 0x06:
                return VIA_MEM_DDR266;
            }
            break;
        case 0x03:
            switch (RAM) {
            case 0x00:
                return VIA_MEM_DDR333;
            case 0x01:
                return VIA_MEM_DDR400;
            case 0x04:
                return VIA_MEM_DDR266;
            }
            break;
        }
    }

    xf86DrvMsg(pScrn->scrnIndex, X_WARNING,
               "%s: Illegal RAM type: FSB %1d, RAM, %1d\n", __func__, FSB, RAM);
    return VIA_MEM_NONE;
}

/*
 * TEST ME
 *
 * Values are again from the BIOS, but here it was easy to see how things
 * added up.
 *
 */
static CARD8
P4M800RAMTypeGet(ScrnInfoPtr pScrn)
{
    CARD8 FSB, FSBtoRAM;
    int Freq;

    /* 0x4296 */
    FSB = pciReadByte(pciTag(0, 0, 4), 0xF3); /* VIA scratch area */
    switch(FSB >> 5) {
    case 0:
        Freq = 3; /* x33Mhz  = 100Mhz */
        break;
    case 1:
        Freq = 4; /* 133Mhz */
        break;
    case 3:
        Freq = 5; /* 166Mhz */
        break;
    case 2:
        Freq = 6; /* 200Mhz */
        break;
    case 4:
        Freq = 7; /* 233Mhz */
        break;
    default:
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "%s: Unhandled FSB: %d\n",
                   __func__, FSB);
        return VIA_MEM_NONE;
    }

    /* FSB to RAM timing from 0x3296 */
    FSBtoRAM = pciReadByte(pciTag(0, 0, 3), 0x68);
    FSBtoRAM &= 0x0F;

    /* This link was straightforward this time */
    if (FSBtoRAM & 0x02)
        Freq -= FSBtoRAM >> 2;
    else {
        Freq += FSBtoRAM >> 2;
        if (FSBtoRAM & 0x01)
            Freq++;
    }

    switch (Freq) {
    case 0x03:
        return VIA_MEM_DDR200;
    case 0x04:
        return VIA_MEM_DDR266;
    case 0x05:
        return VIA_MEM_DDR333;
    case 0x06:
        return VIA_MEM_DDR400;
    default:
        break;
    }

    xf86DrvMsg(pScrn->scrnIndex, X_WARNING,
               "%s: Illegal RAM type: FSB %1d, FSBtoRAM, %1d\n",
               __func__, FSB, FSBtoRAM);
    return VIA_MEM_NONE;
}

#ifdef UNUSED
/*
 * TEST ME
 *
 */
static CARD8
P4M800ProRAMType(ScrnInfoPtr pScrn)
{
    CARD8 tmp;

    VIAFUNC(pScrn->scrnIndex);

    tmp = pciReadByte(pciTag(0, 0, 3), 0x90) & 0x07;
    switch(tmp) {
    case 0x00:
        return VIA_MEM_DDR200;
    case 0x01:
        return VIA_MEM_DDR266;
    case 0x02:
        return VIA_MEM_DDR333;
    case 0x03:
        return VIA_MEM_DDR400;
#if 0
    /* Implement DDR2 first. */
    case 0x04:
        return VIA_MEM_DDR2_400
    case 0x05:
        return VIA_MEM_DDR2_533
#endif
    default:
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING,
                   "%s: Unhandled DRAM MemClk: 0x%02X.\n", __func__, tmp);
        return VIA_MEM_NONE;
    }
}
#endif /* UNUSED */

/*
 * Simply poke the K8 DRAM controller.
 */
static CARD8
AMDK8RAMTypeGet(ScrnInfoPtr pScrn)
{
    PCITAG Tag;
    CARD8 tmp;

    VIAFUNC(pScrn->scrnIndex);

    /* AMD K8 DRAM Controller */
    Tag = pciFindFirst(0x11021022, 0xFFFFFFFF);
    if (Tag == PCI_NOT_FOUND) {
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING,
                   "%s: Unable to find AMD K8 DRAM Controller.\n", __func__);
        return VIA_MEM_NONE;
    }

    tmp = (pciReadByte(Tag, 0x96) >> 4) & 0x07;
    switch(tmp) {
    case 0x00:
        return VIA_MEM_DDR200;
    case 0x02:
        return VIA_MEM_DDR266;
    case 0x05:
        return VIA_MEM_DDR333;
    case 0x07:
        return VIA_MEM_DDR400;
    default:
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING,
                   "%s: Unhandled DRAM MemClk: 0x%02X.\n", __func__, tmp);
        return VIA_MEM_NONE;
    }
}

/*
 *
 */
static char *
ViaMemoryInfoString(CARD8 MemType)
{
    struct {
        CARD8 Index;
        char *Info;
    } MemoryInfo[VIA_MEM_END] = {
        { VIA_MEM_DDR200, "100Mhz DDR - PC1600"},
        { VIA_MEM_DDR266, "133Mhz DDR - PC2100"},
        { VIA_MEM_DDR333, "166Mhz DDR - PC2700"},
        { VIA_MEM_DDR400, "200Mhz DDR - PC3200"},
    };
    int i;

    for (i = 0; i < VIA_MEM_END; i++)
        if (MemoryInfo[i].Index == MemType)
            return MemoryInfo[i].Info;

    return "Unknown";
}

/*
 *
 */
Bool
ViaFBInit(ScrnInfoPtr pScrn)
{
    VIAPtr pVia = VIAPTR(pScrn);
    CARD8 tmp;

    VIAFUNC(pScrn->scrnIndex);

    switch (pVia->Host) {
    case VIA_HOST_CLE266:
        tmp = pciReadByte(pciTag(0, 0, 0), 0xE1);
        pScrn->videoRam = (1 << ((tmp & 0x70) >> 4)) * 1024;
        pVia->MemType = CLE266RAMTypeGet(pScrn);
        break;
    case VIA_HOST_KM400:
        tmp = pciReadByte(pciTag(0, 0, 0), 0xE1);
        pScrn->videoRam = (1 << ((tmp & 0x70) >> 4)) * 1024;
        pVia->MemType = KM400RAMTypeGet(pScrn);
        break;
    case VIA_HOST_P4M800:
        tmp = pciReadByte(pciTag(0, 0, 3), 0xA1);
        pScrn->videoRam = (1 << ((tmp & 0x70) >> 4)) * 1024;
        pVia->MemType = P4M800RAMTypeGet(pScrn);
        break;
    case VIA_HOST_K8M800:
        tmp = pciReadByte(pciTag(0, 0, 3), 0xA1);
        pScrn->videoRam = (1 << ((tmp & 0x70) >> 4)) * 1024;
        pVia->MemType = AMDK8RAMTypeGet(pScrn);
        break;
    default:
        xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
                   "%s: Unhandled HostBridge.\n", __func__);
        return FALSE;
    }

    if (pVia->MemType == VIA_MEM_NONE)
        pVia->MemType = pVia->Scratch->MemType;

    /* should be hostbridge specific and should be extended */
    if (pVia->MemType == VIA_MEM_DDR200)
        pVia->Bandwidth = 394000000; /* 1280x1024@75Hz */
    else
        pVia->Bandwidth = 461000000; /* 1600x1200@60Hz */

    pVia->videoRambytes = pScrn->videoRam << 10;
    xf86DrvMsg(pScrn->scrnIndex, X_PROBED, "Using %dkB of RAM (%s)\n",
               pScrn->videoRam, ViaMemoryInfoString(pVia->MemType));
    return TRUE;
}

/*
 * On recent AMD CPUs the RAM sits off the CPU. Unichromes are all IGPs, so
 * they use a part of this RAM for their framebuffer. This makes chipsets for
 * AMD cpus extremely slow and very latency prone.
 *
 * The Int15 0xE820 call to find out the RAM size (as used by the linux kernel)
 * simply returns (RAM Size - FB Size). If the linux kernel talked to the AMD
 * CPU directly, then it would wrongly map the FB as well.
 *
 * If we assume that the FB always sits at the high end of the RAM (like VIAs
 * code now does), we can map this directly, and have direct CPU access to the
 * FB.
 *
 * So now we can now shift in and out of system RAM at lighting speed, faster
 * than on all other IGPs. But then, this doesn't change the latency issues for
 * the graphics engines themselves.
 *
 * VIA's own RAM Controllers can intercept FB accesses from CPU, saving us
 * quite some latency and on-chip bandwidth. So on devices that are not using
 * and AMD K8, we also get some advantage here.
 */
void
ViaFBBaseGet(ScrnInfoPtr pScrn)
{
    VIAPtr pVia = VIAPTR(pScrn);
    CARD16 tmp;

    VIAFUNC(pScrn->scrnIndex);

    /* PCI BAR */
    pVia->FrameBufferBase = pVia->PciInfo->memBase[0];
    pVia->FBDirectCPUAccess = FALSE;

    /* See if we have some magic */
    switch (pVia->Host) {
    case VIA_HOST_CLE266:
    case VIA_HOST_KM400:
        tmp = pciReadWord(pciTag(0, 0, 0), 0xE0);
        if ((tmp & 0x0001) && (tmp & 0x0FFE)) {
            pVia->FBDirectCPUAccess = TRUE;
            pVia->FrameBufferBase = (tmp & 0xFFE) << 20;
        }
        break;
    case VIA_HOST_K8M800:
        pVia->FrameBufferBase = pciReadByte(pciTag(0, 0, 3), 0x47) << 24;
        pVia->FrameBufferBase -= (pScrn->videoRam * 1024);
        pVia->FBDirectCPUAccess = TRUE;
        break;
    case VIA_HOST_P4M800:
        tmp = pciReadWord(pciTag(0, 0, 3), 0xA0);
        if ((tmp & 0x0001) && (tmp & 0x0FFE)) {
            pVia->FBDirectCPUAccess = TRUE;
            pVia->FrameBufferBase = (tmp & 0xFFE) << 20;
        }
        break;
    default:
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING,
                   "%s: Unhandled HostBridge.\n", __func__);
        break;
    }
}
