#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <strings.h>

#include "xsupconfig.h"
#include "xsupconfcheck.h"

/******************************************************************
 *
 * Do anything necessary to set up the library.
 *
 ******************************************************************/
int xsupconfcheck_init()
{
  return 0;
}

/*****************************************************************
 *
 * Add an error to the list.  This function should only be used inside this
 * library!!
 *
 *****************************************************************/
void xsupconfcheck_add_log(cerrs **errors, char *line, ...)
{
  va_list ap;
  cerrs *cur;
  char errorline[2048];

  if (errors == NULL)
    {
      *errors = NULL;

      *errors = (cerrs *)malloc(sizeof(cerrs));
      if (!(*errors)) return;

      cur = *errors;
    } else {
      cur = *errors;

      while (cur->next) cur = cur->next;
      
      cur->next = (cerrs *)malloc(sizeof(cerrs));
      if (!cur->next) return;

      cur = cur->next;
    }

  // At this point, cur should point to some memory space that needs to be
  // populated.
  va_start(ap, line);
  
  vsnprintf((char *)&errorline, 2048, line, ap);

  cur->line = strdup(errorline);
  cur->next = NULL;
}

/*****************************************************************
 *
 * Check the global settings to verify that the needed settings are in place.
 *
 *****************************************************************/
int xsupconfcheck_check_globals(struct config_globals *globals, 
				cerrs **errors)
{
  int retval = 0;

  // In the globals, we only need the default_net and allowed_nets to be set.
  if (!globals->default_net)
    {
      xsupconfcheck_add_log(errors, "No 'default_netname' is configured.\n");
      retval = -1;
    }

  if (!globals->allowed_nets)
    {
      xsupconfcheck_add_log(errors, "No 'network_list' values are defined!\n");
      retval = -1;
    }
  
  return retval;
}

/*****************************************************************
 *
 * Verify that we have valid keys set for the static WEP method.
 *
 *****************************************************************/
int xsupconfcheck_check_static_wep_method(struct config_static_wep *wep,
					  char *netname, cerrs **errors)
{
  int retval = 0;
  int len;

  if ((wep->tx_key < 1) || (wep->tx_key > 4))
    {
      xsupconfcheck_add_log(errors, "In network clause '%s', the static WEP"
			    " key index is invalid.\n");
      retval = -1;
    } else {
      // We can't check for the validity of a key, if the index is wrong,
      // since we mostly care about the key that is defined by the index value.
      if (!wep->key[wep->tx_key])
	{
	  xsupconfcheck_add_log(errors, "In network clause '%s', the static "
				"key pointed to by the 'tx_key' setting is "
				"empty.\n", netname);
	} else {
	  // Make sure the length of the key is valid.  It should be either
	  // 10, or 26 characters.  (The key is represented by a string of
	  // hex digits.
	  len = strlen(wep->key[wep->tx_key]);

	  if ((len != 26) && (len != 10))
	    {
	      xsupconfcheck_add_log(errors, "In network clause '%s', the "
				    "static key pointed to by the tx_key (%d)"
				    "setting has an invalid length.\n",
				    netname, wep->tx_key);
	      retval = -1;
	    }
	}
    }
      
  return retval;
}

/*****************************************************************
 *
 * Verify that we have valid keys set for the initial WEP method.
 *
 *****************************************************************/
int xsupconfcheck_check_initial_wep_method(struct config_static_wep *wep,
					   char *netname, cerrs **errors)
{
  int retval = 0;
  int len;

  if ((wep->tx_key < 1) || (wep->tx_key > 4))
    {
      xsupconfcheck_add_log(errors, "In network clause '%s', the initial WEP"
			    " key index is invalid.\n");
      retval = -1;
    } else {
      // We can't check for the validity of a key, if the index is wrong,
      // since we mostly care about the key that is defined by the index value.
      if (!wep->key[wep->tx_key])
	{
	  xsupconfcheck_add_log(errors, "In network clause '%s', the initial "
				"WEP key pointed to by the 'tx_key' setting is"
				" empty.\n", netname);
	} else {
	  // Make sure the length of the key is valid.  It should be either
	  // 10, or 26 characters.  (The key is represented by a string of
	  // hex digits.
	  len = strlen(wep->key[wep->tx_key]);

	  if ((len != 26) && (len != 10))
	    {
	      xsupconfcheck_add_log(errors, "In network clause '%s', the "
				    "initial WEP key pointed to by the tx_key "
				    "(%d) setting has an invalid length.\n",
				    netname, wep->tx_key);
	      retval = -1;
	    }
	}
    }
      
  return retval;
}

/*****************************************************************
 *
 * Verify that there are valid settings for WPA_PSK.
 *
 *****************************************************************/
int xsupconfcheck_check_wpa_psk(struct config_wpa_psk *psk, char *netname,
				cerrs **errors)
{
  int retval = 0;

  if ((!psk->key) && (!psk->hex_key))
    {
      xsupconfcheck_add_log(errors, "WPA-PSK needs either a 'key', or "
			    "'hex_key' in network clause '%s'.\n", netname);
      retval = -1;
    }

  return retval;
}

/*****************************************************************
 *
 * Check the configuration options for EAP-MD5.
 *
 *****************************************************************/
int xsupconfcheck_check_eap_md5(struct config_eap_md5 *md5, char *netname, 
				cerrs **errors)
{
  int retval = 0;

  // For EAP-MD5, we don't need anything defined.  The only thing that matters
  // is that we have a password, and if we don't currently have one, we will
  // ask a GUI to provide one for us.

  return retval;
}

/*****************************************************************
 *
 * Check the configuration options for EAP-OTP.
 *
 *****************************************************************/
int xsupconfcheck_check_eap_otp(struct config_eap_otp *otp, char *netname, 
				cerrs **errors)
{
  int retval = 0;

  // For EAP-OTP, we don't need anything defined.  The only thing that matters
  // is that we have a password, and if we don't currently have one, we will
  // ask a GUI to provide one for us.

  return retval;
}

/*****************************************************************
 *
 * Check the configuration options for EAP-GTC.
 *
 *****************************************************************/
int xsupconfcheck_check_eap_gtc(struct config_eap_otp *gtc, char *netname, 
				cerrs **errors)
{
  int retval = 0;

  // For EAP-GTC, we don't need anything defined.  The only thing that matters
  // is that we have a password, and if we don't currently have one, we will
  // ask a GUI to provide one for us.

  return retval;
}

/*****************************************************************
 *
 * Verify that the configuration information we have for EAP-TLS contains 
 * enough information to complete a configuration.
 *
 *****************************************************************/
int xsupconfcheck_check_eap_tls(struct config_eap_tls *tls, char *netname,
				cerrs **errors)
{
  int retval = 0;

  if (!tls->user_cert)
    {
      xsupconfcheck_add_log(errors, "In network clause '%s', EAP-TLS requires"
			    " a 'user_cert' option.\n", netname);
      retval = -1;
    }

  if ((!tls->root_cert) && (!tls->root_dir))
    {
      xsupconfcheck_add_log(errors, "In network clause '%s', EAP-TLS requires"
			    " either a 'root_cert' or 'root_dir' setting.",
			    netname);
      retval = -1;
    }

  if (!tls->user_key)
    {
      xsupconfcheck_add_log(errors, "In network clause '%s', EAP-TLS requires"
			    " a 'user_key' setting.  (It may be the same"
			    " file as the 'user_cert'.)\n", netname);
      retval = -1;
    }

  return retval;
}

/*****************************************************************
 *
 * Check the configuration options for LEAP.
 *
 *****************************************************************/
int xsupconfcheck_check_eap_leap(struct config_eap_leap *leap, char *netname, 
				 cerrs **errors)
{
  int retval = 0;

  // For LEAP, we don't need anything defined.  The only thing that matters
  // is that we have a password, and if we don't currently have one, we will
  // ask a GUI to provide one for us.

  return retval;
}

/*****************************************************************
 *
 * Check the configuration options for EAP-MSCHAPv2.
 *
 *****************************************************************/
int xsupconfcheck_check_eap_mschapv2(struct config_eap_mschapv2 *mscv2, 
				     char *netname, cerrs **errors)
{
  int retval = 0;

  // For EAP-MSCHAPv2, we don't need anything defined.  The only thing that 
  // matters is that we have a password, and if we don't currently have one, 
  // will ask a GUI to provide one for us.

  return retval;
}

/*****************************************************************
 *
 * Check the configuration options for EAP-SIM.
 *
 *****************************************************************/
int xsupconfcheck_check_eap_sim(struct config_eap_sim *sim, char *netname, 
				cerrs **errors)
{
  int retval = 0;

  // For EAP-SIM, we don't need anything defined.  The only thing that matters
  // is that we have a password, and if we don't currently have one, we will
  // ask a GUI to provide one for us.

  return retval;
}

/*****************************************************************
 *
 * Check the configuration options for EAP-AKA.
 *
 *****************************************************************/
int xsupconfcheck_check_eap_aka(struct config_eap_aka *aka, char *netname, 
				cerrs **errors)
{
  int retval = 0;
  
  // For EAP-AKA, we don't need anything defined.  The only thing that matters
  // is that we have a password, and if we don't currently have one, we will
  // ask a GUI to provide one for us.
  
  return retval;
}

/********************************************************************
 *
 * Check the PEAP phase 2 methods.
 *
 ********************************************************************/
int xsupconfcheck_check_peap_phase2(struct config_eap_method *phase2,
				    char *netname, cerrs **errors)
{
  int retval = 0;
  struct config_eap_method *cur;
  
  cur = phase2;

  while (cur)
    {
      switch (cur->method_num)
	{
	case EAP_TYPE_MSCHAPV2:
	  // No settings that need to be checked.
	  break;
	  
	case EAP_TYPE_MD5:
	  // No settings that need to be checked.
	  break;
	  
	case EAP_TYPE_SIM:
	  // No settings that need to be checked.
	  break;
	  
	case EAP_TYPE_GTC:
	  // No settings that need to be checked.
	  break;
	  
	case EAP_TYPE_OTP:
	  // No settings that need to be checked.
	  break;
	  
	default:
	  xsupconfcheck_add_log(errors, "Invalid PEAP phase 2 method in "
				"network clause '%s'.\n", netname);
	  retval = -1;
	  break;
	}
      
      cur = cur->next;
    }

  return retval;
}

/*****************************************************************
 *
 * Check the validity of the PEAP configuration.  This will also check
 * all of the phase 2 methods that are contained in it.
 *
 *****************************************************************/
int xsupconfcheck_check_eap_peap(struct config_eap_peap *peap, char *netname,
				 cerrs **errors)
{
  int retval = 0;

  if ((!peap->root_cert) && (!peap->root_dir))
    {
      xsupconfcheck_add_log(errors, "PEAP requires a valid setting for either"
			    " 'root_cert' or 'root_dir' in clause for network"
			    " '%s'.\n", netname);
      retval = -1;
    }

  if (peap->flags == 0)
    {
      xsupconfcheck_add_log(errors, "PEAP requires a valid setting for "
			    "'allow_types' in the clause for network '%s'.\n",
			    netname);
    }

  if (xsupconfcheck_check_peap_phase2(peap->phase2, netname, errors) != 0)
    retval = -1;

  return retval;
}

/**********************************************************************
 *
 * Check that all of our phase 2 data is valid.
 *
 **********************************************************************/
int xsupconfcheck_check_ttls_phase2(struct config_ttls_phase2 *p2type,
				    char *netname, cerrs **errors)
{
  int retval = 0;

  // Right now, there is nothing to check.  The currently supported phase 2
  // methods only require a username and password.  If the username is empty,
  // then we will copy up the phase 1 username.  If the password is empty,
  // we will ask a GUI for one.

  return retval;
}

/*****************************************************************
 *
 * Check the EAP-TTLS configuration.  This will also check the inner methods.
 *
 *****************************************************************/
int xsupconfcheck_check_eap_ttls(struct config_eap_ttls *ttls, char *netname,
				 cerrs **errors)
{
  int retval = 0;

  if ((!ttls->root_cert) && (!ttls->root_dir))
    {
      xsupconfcheck_add_log(errors, "EAP-TTLS requires a setting for either"
			    " 'root_cert' or 'root_dir' in network clause "
			    " '%s'.\n", netname);
      retval = -1;
    }

  if ((ttls->phase2_type < TTLS_PHASE2_PAP) || (ttls->phase2_type > TTLS_PHASE2_MSCHAPV2))
    {
      xsupconfcheck_add_log(errors, "EAP-TTLS requires a valid setting for the"
			    " 'phase2_type' option in network clause '%s'.\n",
			    netname);
      retval = -1;
    }

  if (!ttls->phase2)
    {
      xsupconfcheck_add_log(errors, "EAP-TTLS requires an inner authentication"
			    " method in network clause '%s'.\n", netname);
      retval = -1;
    } else {
      if (xsupconfcheck_check_ttls_phase2(ttls->phase2, netname, 
					  errors) != 0)	retval = -1;
    }

  return retval;
}

/*****************************************************************
 *
 * Loop through all of the EAP methods that are defined in a network clause
 * and verify that they have valid configuration data.
 *
 *****************************************************************/
int xsupconfcheck_check_eap_methods(struct config_eap_method *method,
				    char *netname, cerrs **errors)
{
  struct config_eap_method *cur;
  int retval = 0;

  cur = method;

  while (cur)
    {
      switch (cur->method_num)
	{
	case STATIC_WEP_METHOD:
	  if (xsupconfcheck_check_static_wep_method(cur->method_data, netname,
						    errors) != 0) retval = -1;
	  break;

	case WPA_PSK:
	  if (xsupconfcheck_check_wpa_psk(cur->method_data, netname,
					  errors) != 0) retval = -1;
	  break;

	case EAP_TYPE_MD5:
	  if (xsupconfcheck_check_eap_md5(cur->method_data, netname, 
					  errors) != 0) retval = -1;
	  break;

	case EAP_TYPE_OTP:
	  if (xsupconfcheck_check_eap_otp(cur->method_data, netname,
					  errors) != 0) retval = -1;
	  break;

	case EAP_TYPE_GTC:
	  if (xsupconfcheck_check_eap_gtc(cur->method_data, netname,
					  errors) != 0) retval = -1;
	  break;

	case EAP_TYPE_TLS:
	  if (xsupconfcheck_check_eap_tls(cur->method_data, netname,
					  errors) != 0) retval = -1;
	  break;

	case EAP_TYPE_LEAP:
	  if (xsupconfcheck_check_eap_leap(cur->method_data, netname,
					   errors) != 0) retval = -1;
	  break;

	case EAP_TYPE_SIM:
	  if (xsupconfcheck_check_eap_sim(cur->method_data, netname,
					  errors) != 0) retval = -1;
	  break;

	case EAP_TYPE_TTLS:
	  if (xsupconfcheck_check_eap_ttls(cur->method_data, netname,
					   errors) != 0) retval = -1;
	  break;

	case EAP_TYPE_AKA:
	  if (xsupconfcheck_check_eap_aka(cur->method_data, netname,
					  errors) != 0) retval = -1;
	  break;

	case EAP_TYPE_PEAP:
	  if (xsupconfcheck_check_eap_peap(cur->method_data, netname,
					   errors) != 0) retval = -1;
	  break;

	case EAP_TYPE_MSCHAPV2:
	  if (xsupconfcheck_check_eap_mschapv2(cur->method_data, netname,
					       errors) != 0) retval = -1;
	  break;

	default:
	  xsupconfcheck_add_log(errors, "Unknown EAP type %d in clause for "
				"network '%s'.\n", cur->method_num, netname);
	  retval = -1;
	  break;
	}
    }
  return retval;
}

/*****************************************************************
 *
 * Check all of the network clauses.  This will also result in all of the
 * EAP clauses being checked.
 *
 *****************************************************************/
int xsupconfcheck_check_networks(struct config_network *network, 
				 cerrs **errors)
{
  int retval = 0;
  struct config_network *cur;
  int clause = 1;

  cur = network;

  while (cur)
    {
      if (!cur->name)
	{
	  xsupconfcheck_add_log(errors, "Network clause %d is missing a name!",
				clause);
	  retval = -1;
	}

      if (cur->flags == 0)
	{
	  xsupconfcheck_add_log(errors, "Network '%s' has no allowed EAP"
				" methods defined!\n", cur->name);
	  retval = -1;
	}

      if (!cur->identity)
	{
	  xsupconfcheck_add_log(errors, "Network '%s' doesn't have an "
				"'identity' value set.\n", cur->name);
	  retval = -1;
	}

      if (cur->initial_wep)
	{
	  if (xsupconfcheck_check_initial_wep_method(cur->initial_wep, 
						     cur->name,
						     errors) != 0) retval = -1;
	}

      if (xsupconfcheck_check_eap_methods(cur->methods, cur->name, errors)
	  != 0) retval = -1;

      clause++;
      cur = cur->next;
    }

  return retval;
}

/*****************************************************************
 *
 * Check an entire config.  If there are errors, a linked list of strings
 * are returned, and the return value !=0.
 *
 *****************************************************************/
int xsupconfcheck_check_config(struct config_data *confdata, cerrs **errors)
{
  int retval = 0;

  retval = xsupconfcheck_check_globals(confdata->globals, errors);

  return retval;
}

/******************************************************************
 *
 * Return the first logline that is in the queue.  Once the line is returned,
 * the node that contains it will be destroyed.
 *
 ******************************************************************/
char *xsupconfcheck_pop_logline(cerrs **errors)
{
  cerrs *cur;
  char *line;

  if (!errors) return NULL;

  cur = (*errors);

  if (!cur)
    {
      free(errors);
      errors = NULL;
      return NULL;
    }

  *errors = cur->next;

  line = cur->line;
  free(cur);

  return line;
}

/*****************************************************************
 *
 * Clean up the string list that may be leftover.  (Assuming it isn't NULL
 * already.)  And do anything else that needs to be done to quit cleanly.
 *
 *****************************************************************/
int xsupconfcheck_deinit(cerrs **errors)
{
  cerrs *cur, *cur2;

  if (errors)
    {
      cur = (*errors);

      while (cur)
	{
	  if (cur->line) free(cur->line);
	  cur->line = NULL;
	  
	  cur2 = cur->next;

	  free(cur);
	  cur = cur2;
	}

      free(errors);
    }

  return 0;
}
