/*******************************************************************
 * File: cardif_linux_rtnetlink.c
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: cardif_linux_rtnetlink.c,v 1.88 2006/03/02 04:52:41 chessing Exp $
 * $Date: 2006/03/02 04:52:41 $
 * $Log: cardif_linux_rtnetlink.c,v $
 * Revision 1.88  2006/03/02 04:52:41  chessing
 * Added the ability to load a config based on the hints that are included in an EAP request identity message.  Should consider having a way to override listening to these messages.
 *
 * Revision 1.87  2006/02/23 22:26:53  chessing
 * Fix for bug id #1415020.  'Building Xsupplicant 1.2.3 Fails on FC4'.
 *
 * Revision 1.86  2006/01/24 04:42:26  chessing
 * A few more fixes to WPA code, along with a fix to the scan reaping code.
 *
 * Revision 1.85  2006/01/23 07:13:50  chessing
 * A couple of fixups to the WPA and WPA2 code.
 *
 * Revision 1.84  2006/01/23 05:28:37  chessing
 * Fixed a few settings that were causing errors with IOCTLs on some cards.  Updated WPA2 code to properly process group key packets. We now record quality, signal level, and noise level from scan results, so that we can start to make better decisions on which AP to associate to.
 *
 * Revision 1.83  2006/01/20 06:53:48  chessing
 * Added support for large scan results.
 *
 * Revision 1.82  2006/01/12 17:54:03  chessing
 * WPA almost works again on cards that use the GENIE setting.  Added a small fix to allow Xsupplicant to build on Open SuSE 10.
 *
 * Revision 1.81  2006/01/07 03:12:58  chessing
 * Small update to fix a problem where we were processing wireless events twice.
 *
 * Revision 1.80  2006/01/03 04:17:43  chessing
 * Added a library to check that the configuration information in the configuration structure should be enough to complete an authentication.
 *
 * Revision 1.79  2005/12/25 04:54:11  chessing
 * Fixup to code that added wext support needed for using ndiswrapper.
 *
 * Revision 1.78  2005/12/24 04:51:51  chessing
 * Removed an unneeded file.  Updated scanning code to parse IWEVGENIE events correctly, and make use of them.
 *
 * Revision 1.77  2005/12/24 02:39:48  chessing
 * Fixes to autoconf script to correctly identify the number of arguments that iw_extract_event_stream() takes, along with checking iwlib.h compiles correctly.
 *
 * Revision 1.76  2005/12/18 07:44:12  chessing
 * Added the ability to associate using IWAUTH options instead of setting a full IE.  This allows NDISwrapper to work with vanilla wireless extensions.  For some reason, we can't parse IWEVASSOCREQIE/IWEVASSOCRESPIE since we see the length as 0, even though iwevent sees the correct IE information. :-/  Need to figure out why.
 *
 * Revision 1.75  2005/12/03 22:18:51  chessing
 * Added an include to fix some problems when compiling with WE19.
 *
 * Revision 1.74  2005/11/30 00:42:29  chessing
 * Cleaned up some warnings during the build process.
 *
 * Revision 1.73  2005/11/22 19:49:42  chessing
 * When the desired interface isn't available when Xsupplicant starts, Xsupplicant will wait for it to show up before starting to authenticate.  In this case, the interface isn't 'available' when an ifconfig doesn't show that the interface exists.  It has *NOTHING* to do with the up/down state of the interface.  This is mostly useful for situations where a cardbus wireless card is plugged in before linux is booted.
 *
 * Revision 1.72  2005/11/16 02:18:38  chessing
 * Updates to the config writing library.  The config-parser tool has been modified to be able to parse a config, and rewrite it.
 *
 * Revision 1.71  2005/11/14 16:41:05  chessing
 * Fix to cardif_linux_rtnetlink code to work when *REALLY* old versions of WE are used that don't define IW_CUSTOM_MAX.
 *
 * Revision 1.70  2005/11/12 23:01:12  chessing
 * Fix for problems with compiling when older versions of iwlib are used.  Added the missing Makefile.in to the lib directory so that we don't have to run automake before compiling.
 *
 * Revision 1.69  2005/10/17 03:56:54  chessing
 * Updates to the libxsupconfig library.  It no longer relies on other source from the main tree, so it can be used safely in other code with problems.
 *
 * Revision 1.68  2005/10/17 00:15:55  chessing
 * Moved the config parsing routines out to a new library called libxsupconfig.a, so that it will be easier to create third-party tools that can easily parse a config.
 *
 * Revision 1.67  2005/10/16 04:31:30  chessing
 * Changed gui_interface.c/h to be a library called libxsupgui, and moved it to a new lib directory.
 *
 * Revision 1.66  2005/10/14 02:26:18  shaftoe
 * - cleanup gcc 4 warnings
 * - (re)add support for a pid in the form of /var/run/xsupplicant.<iface>.pid
 *
 * -- Eric Evans <eevans@sym-link.com>
 *
 * Revision 1.65  2005/10/13 18:57:28  chessing
 * Removed some leftover debug strings.  Cleaned up some compiler warnings. Fixed an association bug in the madwifi driver.
 *
 * Revision 1.64  2005/10/13 18:46:47  chessing
 * Fixed to the Madwifi driver to allow it to do dynamic WEP again.  Fixed up the wext driver to behave correctly again. ;)
 *
 * Revision 1.63  2005/09/26 02:04:07  shaftoe
 * Yet another round of log output cleanup.
 *
 * -- Eric Evans <eevans@sym-link.com>
 *
 * Revision 1.62  2005/09/23 02:31:33  shaftoe
 * Log output cleanup for the benefit of syslog.
 *
 * -- Eric Evans <eevans@sym-link.com>
 *
 * Revision 1.61  2005/09/19 21:45:46  chessing
 * Fix for the madwifi driver when it connects to certain APs to do WEP.  (Currently the only known one with this problem is the Trapeze APs when they are running MSS 4.x+.)
 *
 * Revision 1.60  2005/09/14 03:34:54  chessing
 * Small cosmetic changes.  Default association mode is now auto instead of manual. Fixes for bug IDs #1290449 & #1290323.
 *
 * Revision 1.59  2005/09/14 02:50:44  chessing
 * Major updates.  Auto association now works.  Started to rewrite the rtnetlink pieces using iwlib from the wireless tools to avoid compatibility issues.  As a result, getting the WPA and RSN IEs via the IWEVGENIE event no longer works for some reason, but the old style using IWEVCUSTOM events should still work like a champ.
 *
 * Revision 1.58  2005/09/08 16:27:02  chessing
 * Some small updates to the new state machine code.  First attempt at an auto association mode.  (It mostly works. ;)
 *
 * Revision 1.57  2005/09/05 01:00:37  chessing
 * Major overhaul to most of the state machines in Xsupplicant.  Also added additional error messages to the TLS functions to try to debug the one of the problems reported on the list.  Basic testing shows this new code to be more stable than previous code, but it needs more testing.
 *
 * Revision 1.56  2005/08/25 03:34:05  chessing
 * Removed a bunch of functions from config.c that could be handled better in other ways.
 *
 * Revision 1.55  2005/08/25 02:20:20  chessing
 * Some cleanup in xsup_debug.c, added the ability to wait for an interface to come up if it is down when Xsupplicant is started.  Roughed in the ability to choose between having Xsupplicant auto associate you, or allow you to set the ssid manually.  The stale key timer can now be set in the config file.  The association timeout can be set in the config file, and will also be used once the functionality is in place to attempt to guess the key settings needed for association, as well as the functionality to auto associate.
 *
 * Revision 1.54  2005/08/20 19:06:54  chessing
 * Patch from Carsten Grohmann to fix a few things in xsup_get_state.c.  Also added the ability to define an empty network clause, that will set the card in to encryption disabled mode.  From there, anything short of changing the SSID will be ignored by Xsupplicant.
 *
 * Revision 1.53  2005/08/18 03:19:04  chessing
 * Added the ability to define an SSID with static WEP keys.  When we switch to a network that has this type of configuration we will set the keys, and stop the various association timers.
 *
 * Revision 1.52  2005/08/09 01:39:15  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/

#ifdef LINUX_FRAMER

// Use kernel headers
#define HEADERS_KERNEL

#include <unistd.h>
#include <sys/socket.h>
#include <linux/compiler.h>
#include <linux/wireless.h>
#include <iwlib.h>
#include <sys/ioctl.h>
#include <linux/netlink.h>
#include <linux/rtnetlink.h>
#include <linux/if_packet.h>
#include <math.h>

#include "profile.h"
#include "xsupconfig.h"
#include "xsup_debug.h"
#include "xsup_err.h"
#include "wireless_sm.h"
#include "cardif_linux.h"
#include "statemachine.h"
#include "../cardif.h"
#include "wpa.h"
#include "wpa2.h"
#include "eap.h"
#include "eap_types/mschapv2/mschapv2.h"
#include "config_ssid.h"
#include "eapol_key_type254.h"
#include "timer.h"
#include "mic.h"
#include "cardif_linux_rtnetlink.h"

// Define this, so the compiler doesn't complain.
extern char *if_indextoname(unsigned int, char *);

#define INT_DEL    0
#define INT_NEW    1

static int rtnl_sock=-1;
static struct sockaddr_nl rtnl_data;

#ifndef IWEVCUSTOM
#warning IWEVCUSTOM is not defined!  We will define it, and try to continue!
#define IWEVCUSTOM   0x8C02
#endif

#ifndef IW_CUSTOM_MAX
#warning IW_CUSTOM_MAX is not defined!  You should upgrade to a more current version of wireless extensions.  We will attempt to define it ourselves, but the results may not be good.
#define IW_CUSTOM_MAX  256  // In bytes, matches current WE versions. (11/14/05)
#endif

// Forward def to avoid compiler warnings.
void cardif_linux_rtnetlink_process_token(struct interface_data *ctx,
					  struct iw_event *iwe,
					  struct wireless_state *state);

/********************************************************
 *
 * Do whatever is needed to establish a netlink socket so that we can
 * catch events, and take action.
 *
 ********************************************************/
void cardif_linux_rtnetlink_init()
{
  rtnl_sock = socket(PF_NETLINK, SOCK_RAW, NETLINK_ROUTE);
  
  if (rtnl_sock < 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't establish an rtnetlink socket!  Some functionality won't be available!\n");
      return;
    }

  bzero((char *)&rtnl_data, sizeof(rtnl_data));

  rtnl_data.nl_family = AF_NETLINK;
  rtnl_data.nl_groups = RTMGRP_LINK;

  if (bind(rtnl_sock, (struct sockaddr *)&rtnl_data, sizeof(rtnl_data)) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't bind netlink socket!  Some functionality won't be available!\n");
      close(rtnl_sock);
      return;
    }
}

/********************************************************
 *
 * Do whatever is needed to shutdown the netlink socket that we set up.
 *
 ********************************************************/
void cardif_linux_rtnetlink_cleanup()
{
  debug_printf(DEBUG_INT, "Called cardif_linux_rtnetlink_cleanup()!\n");

  // Close the rtnetlink socket.
  close(rtnl_sock);
}

/*********************************************************************
 *
 * Get the wireless extensions version that this driver was built with.
 *
 *********************************************************************/
int cardif_linux_rtnetlink_get_we_ver(struct interface_data *intdata)
{
  struct iwreq iwr;
  struct iw_range *range;
  char buffer[sizeof(iwrange)*2];
  int sock;

  bzero(buffer, (sizeof(iwrange)*2));

  iwr.u.data.pointer = (caddr_t) buffer;
  iwr.u.data.length = sizeof(buffer);
  iwr.u.data.flags = 0;

  sock = cardif_get_socket(intdata);
  if (iw_get_ext(sock, intdata->intName, SIOCGIWRANGE, &iwr) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Error with ioctl SIOCGIWRANGE (Error: %s)\n",
		   strerror(errno));
      return -1;
    }

  range = (struct iw_range *)buffer;

  return range->we_version_compiled;
}

/********************************************************
 *
 * Testing replacement for code taken from wpa_supplicant.
 *
 ********************************************************/
void cardif_linux_rtnetlink_reap(struct interface_data *intdata, 
				 char *data, int len)
{
  struct stream_descr stream;
  struct iw_event iwe;
  int retval;

  iw_init_event_stream(&stream, data, len);
  do
    {
#ifdef NEW_IWLIB
      retval = iw_extract_event_stream(&stream, &iwe, 
				       cardif_linux_rtnetlink_get_we_ver(intdata));
#else
      retval = iw_extract_event_stream(&stream, &iwe);
#endif

      if (retval == 1)  // Then we got something
	{
	  cardif_linux_rtnetlink_process_token(intdata, &iwe, NULL);
	} else {
	  switch (retval)
	    {
	    case 0:
	      // No error.
	      break;

	    case -1:debug_printf(DEBUG_NORMAL, "Invalid event!\n");
	      break;

	    case 2:debug_printf(DEBUG_NORMAL, "Unknown event found. Skipping."
				"(Event %04X) (Not supported in wireless "
				"extensions %d?)\n", iwe.cmd, WIRELESS_EXT);
	      break;

	    case -2:debug_printf(DEBUG_NORMAL, "Invalid event data. Skipping."
				 "\n");
	      break;

	    default:debug_printf(DEBUG_NORMAL, "Unknown result code from "
				 "iw_extract_event_stream(). (Result was %d)"
				 "\n", retval);
	      break;
	    }
	}
    }
  while (retval > 0);
}

/*******************************************************
 *
 * Check to see if we have data in the returned scan buffer, even if 
 * we didn't get a scan complete event.  (Some cards may not send the
 * scan complete event.)
 *
 *******************************************************/
u_char cardif_linux_rtnetlink_check_nets(struct interface_data *idata)
{
  struct lin_sock_data *sockData;
  struct iwreq iwr;
  char *buffer;
  uint16_t buf_size = 8192;

  if (!TEST_FLAG(idata->flags, SCANNING)) 
    {
      return XENONE;
    }

  debug_printf(DEBUG_INT, "Checking for returned SSID information....\n");

  sockData = idata->sockData;

  buffer = (char *)malloc(buf_size);
  if (!buffer)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't allocate memory for scan buffer!"
		   "\n");
      return XEMALLOC;
    }

  iwr.u.data.pointer = buffer;
  iwr.u.data.flags = 0;
  iwr.u.data.length = buf_size;

  strcpy(iwr.ifr_name, idata->intName);
  
  if (ioctl(sockData->sockInt, SIOCGIWSCAN, &iwr) < 0)
    {
      if (errno == E2BIG)
	{
	  // Our return results are too big for our default buffer.  So,
	  // allocate more and try again!
	  
	  free(buffer);
	  buf_size *= 2;

	  buffer = (char *)malloc(buf_size);
	  if (!buffer)
	    {
	      debug_printf(DEBUG_NORMAL, "Couldn't allocate memory for scan "
			   "buffer!\n");
	      return XEMALLOC;
	    }

	  iwr.u.data.pointer = buffer;
	  iwr.u.data.length = buf_size;

	  while (ioctl(sockData->sockInt, SIOCGIWSCAN, &iwr) < 0)
	    {
	      free(buffer);
	      
	      if (buf_size > 60000)
		{
		  debug_printf(DEBUG_NORMAL, "Buffer size to allocate has "
			       "become unreasonable!  (If you really have "
			       "that many SSIDs, you won't get much data "
			       "across the network anyway!)\n");
		  return -1;
		}

	      buf_size *= 2;

	      buffer = (char *)malloc(buf_size);
	      if (!buffer)
		{
		  debug_printf(DEBUG_NORMAL, "Couldn't allocate memory for scan buffer!\n");
		  return XEMALLOC;
		}

	      iwr.u.data.pointer = buffer;
	      iwr.u.data.length = buf_size;
	    }
	}
      else
	{
	  if (errno == EAGAIN)
	    {
	      debug_printf(DEBUG_INT, "No data available! (%s)\n", 
			   strerror(errno));
	      return XENONE;
	    }
	  else
	    {
	      debug_printf(DEBUG_NORMAL, "Error with scan results!\n");
	      debug_printf(DEBUG_NORMAL, "Error was : %s\n", strerror(errno));
	      UNSET_FLAG(idata->flags, SCANNING);
	      return -1;
	    }
	}
    }

  // Then harvest the data.
  debug_printf(DEBUG_INT, "Reaping data. (Size : %d)\n", iwr.u.data.length);
  debug_hex_dump(DEBUG_INT, (u_char *)buffer, iwr.u.data.length);
  cardif_linux_rtnetlink_reap(idata, (char *)buffer, iwr.u.data.length);

  UNSET_FLAG(idata->flags, SCANNING);

  // Clean up after ourselves.
  free(buffer);
  buffer = NULL;

  return XDATA;
}

/***********************************************************
 *
 * Check the MAC that we were given.  If it is all 0s, 4s, or Fs then the
 * event is a disassociation.  If it isn't then it is an association.
 *
 ***********************************************************/
int cardif_linux_rtnetlink_validate(struct interface_data *idata, u_char *mac)
{
  char newmac[6];

  // Is it a disassociation?
  memset(newmac, 0x00, 6);
  if (memcmp(newmac, mac, 6) == 0)
    {
      return FALSE;
    }

  memset(newmac, 0x44, 6);
  if (memcmp(newmac, mac, 6) == 0)
    {
      return FALSE;
    }

  memset(newmac, 0xff, 6);
  if (memcmp(newmac, mac, 6) == 0)
    {
      return FALSE;
    }
  
  // Otherwise, it was an association
  //Now to check to see if it was roaming ww
  /*
  if (memcmp(mac, idata->dest_mac, 6) != 0)
    {
      debug_printf(DEBUG_INT, "Roaming to new access point on the same SSID.\n");
      // We don't need to check the keying problem timer here, since if we
      // roamed to a different AP we can assume that the key should change.
      // So, checking for the keying problem would result in a false positive.
      statemachine_reinit(idata);
      eap_clear_active_method(network_data->activemethod);
      if (ws) ws->associated = FALSE;
      return FALSE;
    }
  */

  return TRUE;
}

/**********************************************************************
 *
 * Process a SIOCGIWAP event.
 *
 **********************************************************************/
void cardif_linux_rtnetlink_process_SIOCGIWAP(struct interface_data *idata,
					      struct iw_event *iwe,
					      struct wireless_state *ws)
{
  char mac[6];
  int assoc;
  struct config_globals *globals;

  memcpy(mac, iwe->u.ap_addr.sa_data, 6);
  debug_printf(DEBUG_INT, "AP MAC : ");
  debug_hex_printf(DEBUG_INT, (u_char *)mac, 6);

  if (!ws)
    {
      config_ssid_add_bssid(mac);
    } else {
      globals = config_get_globals();
      
      if (globals == NULL)
	{
	  debug_printf(DEBUG_NORMAL, "No valid configuration global data in "
		       "%s!\n", __FUNCTION__);
	  return;
	}

      assoc = cardif_linux_rtnetlink_validate(idata, (u_char *)&mac);
      if (assoc)
	{
	  // We have changed to associated mode.  Populate the destination
	  // MAC with the BSSID, as long as we are in auto mode.
	  
	  ws->associated = TRUE;
	  
	  if (globals->destination == DEST_AUTO)
	    memcmp(idata->dest_mac, mac, 6);
	  
	} else {
	  ws->associated = FALSE;
	}
      debug_printf(DEBUG_INT, "(%s)  ws->associated == %d\n",
		   __FUNCTION__, ws->associated);
    }
}

/**********************************************************************
 *
 * Process a SIOCGIWESSID event.
 *
 **********************************************************************/
void cardif_linux_rtnetlink_process_SIOCGIWESSID(struct interface_data *ctx,
						 struct iw_event *iwe,
						 struct wireless_state *ws)
{
  char essid[IW_ESSID_MAX_SIZE+1];

  bzero(essid, IW_ESSID_MAX_SIZE+1);
  
  memcpy(essid, iwe->u.essid.pointer, iwe->u.essid.length);
  essid[iwe->u.essid.length] = '\0';

  if (!ws)
    {
      debug_printf(DEBUG_INT, "ESSID : %s\n", essid);
      
      config_ssid_add_ssid_name(essid);
    } else {
      debug_printf(DEBUG_NORMAL, "Got a get SSID event!? "
		   "Notify your wireless driver maintainer.\n");
    }
}

/*****************************************************************
 *
 * Process an SIOCSIWESSID.
 *
 *****************************************************************/
void cardif_linux_rtnetlink_process_SIOCSIWESSID(struct interface_data *ctx,
						 struct iw_event *iwe,
						 struct wireless_state *ws)
{
  char essid[IW_ESSID_MAX_SIZE+1];
  struct config_network *network_data;

  bzero(essid, IW_ESSID_MAX_SIZE+1);
  
  memcpy(essid, iwe->u.essid.pointer, iwe->u.essid.length);
  essid[iwe->u.essid.length] = '\0';

  if (!ws)
    {
      debug_printf(DEBUG_NORMAL, "Got an SSID set request from a scan!? "
		   "Notify your driver maintainer!\n");
    } else {
      network_data = config_get_network_config();
      
      if (network_data == NULL)
	{
	  debug_printf(DEBUG_CONFIG, "Invalid network configuration "
		       "structure! (%s:%d)\n", __FUNCTION__, __LINE__);
	}
      
      debug_printf(DEBUG_INT, "ESSID set .. name : %s\n", essid);
      
      if (strcmp(essid, ctx->cur_essid) != 0)
	{
	  if (config_ssid_ssid_known(essid) != TRUE)
	    {
	      // We only want to set this to TRUE if we don't already know
	      // something about the SSID we connected to.
	      ws->ssid_change = TRUE;
	    }
	  
	  if (config_build(ctx, essid) == FALSE)
	    {
	      debug_printf(DEBUG_NORMAL, "Couldn't build a valid configuration"
			   " for ESSID %s!\n", essid);
	    }
	  
	  // We changed ssids, so record the new one.
	  if (ctx->cur_essid != NULL)
	    {
	      free(ctx->cur_essid);
	      
	      ctx->cur_essid = (char *)malloc(strlen(essid)+1);
	      if (ctx->cur_essid != NULL)
		{
		  strcpy(ctx->cur_essid, essid);
		}
	    }
	  
	  if (network_data)
	    eap_clear_active_method(network_data->activemethod);
	}
    }
}

/**********************************************************************
 *
 * Scan through whatever was returned by the IWEVGENIE event, and pull
 * out any interesting IEs.
 *
 **********************************************************************/
void cardif_linux_rtnetlink_parse_ies(struct interface_data *ctx,
				      uint8_t *iedata, int ielen)
{
  int i = 0;

  while (i < ielen)
    {
      if (iedata[i] == WPA_EID)
	{
	  if (wpa_parse_ie(&iedata[i]) > 0)
	    {
	      // We have a valid IE, save it.
	      ctx->wpa_ie = (char *)malloc(iedata[i+1]+2);
	      if (!ctx->wpa_ie)
		{
		  debug_printf(DEBUG_NORMAL, "Couldn't allocate memory to "
			       "store WPA IE!\n");
		  return;
		}
	      config_ssid_update_abilities(WPA_IE);
	      config_ssid_add_wpa_ie((u_char *)&iedata[i], iedata[i+1]+2);
	    }
	}

      if (iedata[i] == WPA2_EID)
	{
	  if (wpa2_parse_ie(&iedata[i]) > 0)
	    {
	      // We have a valid IE, save it.
	      ctx->rsn_ie = (char *)malloc(iedata[i+1]+2);
	      if (!ctx->rsn_ie)
		{
		  debug_printf(DEBUG_NORMAL, "Couldn't allocate memory to "
			       "store RSN IE!\n");
		  return;
		}
	      config_ssid_update_abilities(RSN_IE);
	      config_ssid_add_rsn_ie((u_char *)&iedata[i], iedata[i+1]+2);
	    }
	}
      i += (iedata[i+1]+2);
    }
}

/**********************************************************************
 *
 * Process an IWEVGENIE event.
 *
 **********************************************************************/
void cardif_linux_rtnetlink_process_IWEVGENIE(struct interface_data *ctx,
					      struct iw_event *iwe)
{
  debug_printf(DEBUG_INT, "IWEVGENIE (%d)\n", iwe->u.data.length);
  debug_printf_nl(DEBUG_INT, "IE : ");
  debug_hex_printf(DEBUG_INT, iwe->u.data.pointer, iwe->u.data.length);
  cardif_linux_rtnetlink_parse_ies(ctx, iwe->u.data.pointer, 
				   iwe->u.data.length);
}

/**********************************************************************
 *
 * Process an IWEVCUSTOM event.
 *
 **********************************************************************/
void cardif_linux_rtnetlink_process_IWEVCUSTOM(struct interface_data *ctx,
					       struct iw_event *iwe,
					       struct wireless_state *ws)
{
  char custom[IW_CUSTOM_MAX+1];
  char temp[IW_CUSTOM_MAX+1];

  if (!ws)
    {
      strncpy(custom, iwe->u.data.pointer, iwe->u.data.length);
      
      debug_printf(DEBUG_INT, "IWEVCUSTOM : %s\n", custom);
      
      if (strncmp("wpa_ie=", custom, 7) == 0)
	{
	  config_ssid_update_abilities(WPA_IE);
	  debug_printf(DEBUG_INT, "AP appears to support WPA!\n");
	  
	  process_hex(&custom[7], (iwe->len -7), temp);
	  wpa_parse_ie(temp);
	  
	  config_ssid_add_wpa_ie((u_char *)temp, ((iwe->u.data.length - 7)/2));
	}
      
      if (strncmp("rsn_ie=", custom, 7) == 0)
	{
	  config_ssid_update_abilities(RSN_IE);
	  debug_printf(DEBUG_INT, "AP appears to support WPA2/802.11i!\n");
	  
	  process_hex(&custom[7], (iwe->len -7), temp);
	  wpa2_parse_ie(temp);
	  
	  config_ssid_add_rsn_ie((u_char *)temp, ((iwe->u.data.length - 7)/2));
	}
    } else {
      strncpy(custom, iwe->u.data.pointer, iwe->u.data.length);

      bzero(temp, IW_CUSTOM_MAX+1);
      
      memcpy(temp, custom, iwe->u.data.length);
      temp[iwe->u.data.length] = '\0';
      debug_printf(DEBUG_INT, "Custom Data : \n");
      debug_hex_dump(DEBUG_INT, (u_char *)temp, iwe->u.data.length);
      cardif_linux_rtnetlink_check_custom(ctx, temp);
    }
}

/**********************************************************************
 *
 * Process an SIOCGIWSCAN event.
 *
 **********************************************************************/
void cardif_linux_rtnetlink_process_SIOCGIWSCAN(struct interface_data *ctx,
						struct wireless_state *state)
{
  if (!state)
    {
      debug_printf(DEBUG_NORMAL, "Got an SIOCGIWSCAN in a scan result!? "
		   "Contact your wireless card driver maintainer!\n");
    } else {
      debug_printf(DEBUG_INT, "Wireless scan complete!\n");
      cardif_linux_rtnetlink_check_nets(ctx);
    }
}

/**********************************************************************
 *
 * Process an association request IE, if one is returned.
 *
 **********************************************************************/
void cardif_linux_rtnetlink_process_IWEVASSOCREQIE(struct interface_data *ctx, 
						   struct iw_event *iwe)
{
  if (iwe->u.data.length <= 0)
    {
      debug_printf(DEBUG_NORMAL, "Got an IWEVASSOCREQIE, but it contains "
		   "no data?!\n");
      return;
    }

  debug_printf(DEBUG_INT, "IWEVASSOCREQIE returned : \n");
  debug_hex_printf(DEBUG_INT, iwe->u.data.pointer, iwe->u.data.length);
  cardif_linux_rtnetlink_parse_ies(ctx, iwe->u.data.pointer, 
				   iwe->u.data.length);
}

/**********************************************************************
 *
 * Process a response IE if one is returned.
 *
 **********************************************************************/
void cardif_linux_rtnetlink_process_IWEVASSOCRESPIE(struct interface_data *ctx,
						    struct iw_event *iwe)
{
  if (iwe->u.data.length <= 0)
    {
      debug_printf(DEBUG_NORMAL, "Got an IWEVASSOCRESPIE, but it contains "
		   "no data?!\n");
      return;
    }
  debug_printf(DEBUG_INT, "IWEVASSOCRESPIE returned : \n");
  debug_hex_printf(DEBUG_INT, iwe->u.data.pointer, iwe->u.data.length);
  cardif_linux_rtnetlink_parse_ies(ctx, iwe->u.data.pointer,
				   iwe->u.data.length);
}

void cardif_linux_rtnetlink_process_IWEVMICHAELMICFAILURE(struct interface_data *ctx,
							  struct iw_event *iwe)
{
#ifdef IW_MICFAILURE_KEY_ID
  struct iw_michaelmicfailure *mic;

  // TODO : Double check this implementation.

  if (iwe->u.data.length <= 0)
    {
      debug_printf(DEBUG_NORMAL, "Got a MIC failure, but it contained no "
		   "data!  Ignoring! (This may be dangerous.)\n");
      return;
    }

  mic = iwe->u.data.pointer;

  debug_printf(DEBUG_INT, "MIC failure :\n");
  debug_printf(DEBUG_INT, "  Key ID   : %d\n", mic->flags & IW_MICFAILURE_KEY_ID);
  
  if (mic->flags & IW_MICFAILURE_GROUP)
    {
      debug_printf(DEBUG_INT, "  Key Type : Group\n");
      debug_printf(DEBUG_NORMAL, "MIC failure on group key!\n");
      eapol_key_type254_request_new_key(ctx, 0);
      ctx->send_size = 0;
    } 

  if (mic->flags & IW_MICFAILURE_PAIRWISE)
    {
      debug_printf(DEBUG_INT, "  Key Type : Pairwise\n");
      debug_printf(DEBUG_NORMAL, "MIC failure on pairwise key!\n");
      eapol_key_type254_request_new_key(ctx, 1);
      ctx->send_size = 0;
    }

  if (mic->flags & IW_MICFAILURE_STAKEY)
    {
      debug_printf(DEBUG_INT, "  STAKEY\n");
    }

  // Some wireless cards may also return a count.  But we maintain our own
  // internal counter, so it isn't relevant.
  
  ctx->statemachine->MICfailures++;
  debug_printf(DEBUG_NORMAL, "MIC failure #%d!\n",
	       ctx->statemachine->MICfailures);
      
  if (ctx->statemachine->MICfailures >= 2)
    {
      // The WPA/802.11i standard requires we assert countermeasures 
      // for 60 seconds.
      if (timer_check_existing(COUNTERMEASURE_TIMER))
	{
	  debug_printf(DEBUG_NORMAL, "For some reason, we already have "
		       "a countermeasure timer in the queue!  Resetting "
		       "the timer!\n");
	  timer_reset_timer_count(COUNTERMEASURE_TIMER, 
				  MIC_COUNTERMEASURE_TIMEOUT);
	} else {
	  debug_printf(DEBUG_NORMAL, "Enabling MIC countermeasures!\n");
	  timer_add_timer(COUNTERMEASURE_TIMER, MIC_COUNTERMEASURE_TIMEOUT,
			  NULL, &mic_disable_countermeasures);
	}
      cardif_countermeasures(ctx, TRUE);
    }
#else
  debug_printf(DEBUG_NORMAL, "MIC failure support is not enabled in the "
	       "version of wireless extensions on your platform.  You should"
	       " consider upgrading to a more current version!\n");
#endif
}

/**********************************************************************
 *
 * Given a wireless event, process it.  If *state is NULL, then the event
 * is the result of a requested scan, so it needs to be added to the
 * SSID list.  If *state is not NULL, then this is an event generated by
 * the wireless interface.
 *
 **********************************************************************/
void cardif_linux_rtnetlink_process_token(struct interface_data *ctx,
					  struct iw_event *iwe,
					  struct wireless_state *state)
{
  switch (iwe->cmd)
    {
    case SIOCGIWAP:
      cardif_linux_rtnetlink_process_SIOCGIWAP(ctx, iwe, state);
      break;
      
    case SIOCGIWFREQ:
      // Don't care.
      break;
      
    case SIOCGIWMODE:
      // Don't care.
      break;
      
    case SIOCGIWESSID:
      cardif_linux_rtnetlink_process_SIOCGIWESSID(ctx, iwe, state);
      break;

    case SIOCSIWESSID:
      cardif_linux_rtnetlink_process_SIOCSIWESSID(ctx, iwe, state);
      break;
      
    case SIOCGIWNAME:
      // Don't care.
      break;

    case SIOCGIWSCAN:
      cardif_linux_rtnetlink_process_SIOCGIWSCAN(ctx, state);
      break;

#ifdef IWEVTXDROP
      // This is mostly for the gee-whiz factor.
    case IWEVTXDROP:
      debug_printf(DEBUG_INT, "Wireless TX Drop\n");
      break;
#endif
      
#if WIRELESS_EXT > 17
    case IWEVASSOCREQIE:
      debug_printf(DEBUG_INT, "IWEVASSOCREQIE\n");
      debug_hex_printf(DEBUG_INT, iwe->u.data.pointer, iwe->u.data.length);
      cardif_linux_rtnetlink_process_IWEVASSOCREQIE(ctx, iwe);
      break;
      
    case IWEVASSOCRESPIE:
      debug_printf(DEBUG_INT, "IWEVASSOCRESPIE\n");
      cardif_linux_rtnetlink_process_IWEVASSOCRESPIE(ctx, iwe);
      break;
      
    case IWEVGENIE:
      if (!state)
	{
	  cardif_linux_rtnetlink_process_IWEVGENIE(ctx, iwe);
	}
      break;

    case IWEVMICHAELMICFAILURE:
      debug_printf(DEBUG_INT, "MIC Failure!\n");
      cardif_linux_rtnetlink_process_IWEVMICHAELMICFAILURE(ctx, iwe);
      break;
      
    case IWEVPMKIDCAND:
      debug_printf(DEBUG_INT, "IWEVPMKIDCAND\n");
      break;
#endif
    case SIOCGIWENCODE:
      if (!state)
	{
	  if (!(iwe->u.data.flags & IW_ENCODE_DISABLED))
	    {
	      config_ssid_update_abilities(ENC);
	    }
	}
      break;
      
    case SIOCGIWRATE:
      // Don't care.
      break;
      
    case IWEVQUAL:
      debug_printf(DEBUG_INT, "Quality : %d  Signal : %d   Noise : %d\n",
		   iwe->u.qual.qual, (iwe->u.qual.level - 0x100), 
		   (iwe->u.qual.noise - 0x100));
      config_ssid_add_qual(iwe->u.qual.qual, (iwe->u.qual.level - 0x100), 
			   (iwe->u.qual.noise - 0x100));
      break;
      
    case IWEVCUSTOM:
      cardif_linux_rtnetlink_process_IWEVCUSTOM(ctx, iwe, state);
      break;

    case SIOCSIWENCODE:
      debug_printf(DEBUG_INT, "Encryption key set\n");
      break;

    default:
      debug_printf(DEBUG_INT, "Unknown event (%04X)  (Unknown in wireless "
		   "extensions %d?)\n", iwe->cmd, WIRELESS_EXT);
    }
}

/*******************************************************
 *
 * Check to see if we have any events, and act on them if we do.
 *
 *******************************************************/
u_char cardif_linux_rtnetlink_check_event(struct interface_data *idata,
					  struct wireless_state *ws)
{
  int rtnl_data_size, remain, length;
  char buf[8192];
  struct nlmsghdr *nlhead;

  // Grab the next message off the rtnetlink socket.
  rtnl_data_size = sizeof(rtnl_data);
  if (rtnl_sock < 0)
    {
      debug_printf(DEBUG_NORMAL, "RTnetlink socket not available!\n");
      return XENOSOCK;
    }

  remain = recvfrom(rtnl_sock, buf, sizeof(buf), MSG_DONTWAIT,
		    (struct sockaddr *)&rtnl_data, 
		    (u_int *) &rtnl_data_size);
  if (remain >= 0)
    {
      // We need a pointer to the buffer to work with.
      nlhead = (struct nlmsghdr *)buf;

      // There may be more than one message in the packet.  So, loop through!
      while (remain >= sizeof(struct nlmsghdr))
	{
	  // Make sure we have enough data for a real message.
	  if ((nlhead->nlmsg_len > remain) || 
	      ((nlhead->nlmsg_len - sizeof(struct nlmsghdr)) < 0))
	    {
	      debug_printf(DEBUG_NORMAL, "Invalid netlink message!\n");
	      break;
	    }
	  
	  // See what kind of message it is.
	  switch (nlhead->nlmsg_type)
	    {
	    case RTM_NEWLINK:
	      debug_printf(DEBUG_INT, "Got an RTM_NEWLINK!\n");
	      cardif_linux_rtnetlink_do_link(idata, nlhead,
					     nlhead->nlmsg_len, INT_NEW, ws);
	      break;
	      
	    case RTM_DELLINK:
	      debug_printf(DEBUG_INT, "Got an RTM_DELLINK!\n");
	      cardif_linux_rtnetlink_do_link(idata, nlhead, 
					     nlhead->nlmsg_len, INT_DEL, ws);
	      break;
	    }

	  // Find the aligned length of the message, so we can skip
	  // to the next message.
	  length = NLMSG_ALIGN(nlhead->nlmsg_len);

	  remain -= length;
	  
	  nlhead = (struct nlmsghdr *) ((char *)nlhead + length);
	}

      // If we have anything left, then there may be a problem.  So, report
      // the we may have a problem.
      if (remain > 0)
	{
	  debug_printf(DEBUG_NORMAL, "Extra bytes at the end of the netlink message.\n");
	}
    }

  // There may be situations where the wireless card driver can scan,
  // but doesn't provide an event when the scan has completed.  So,
  // we need to check if we have data, even if we don't have an event.
  return cardif_linux_rtnetlink_check_nets(idata);
}

/************************************************************
 *
 *  We got an RTM_NEWLINK or RTM_DELLINK message, so process it, and 
 *  decide how to proceed.
 *
 ************************************************************/
void cardif_linux_rtnetlink_do_link(struct interface_data *intdata,
				    struct nlmsghdr *msg, int len, int type,
				    struct wireless_state *ws)
{
  struct ifinfomsg *ifinfo;
  int nlmsg_len, rtalen, rtlen;
  struct rtattr *rtattr;

  if (len < sizeof(struct ifinfomsg))
    {
      debug_printf(DEBUG_NORMAL, "Netlink message too short!\n");
      return;
    }

  // Get the actual message from the block.
  ifinfo = NLMSG_DATA(msg);

  // Find out how big the message is.
  nlmsg_len = NLMSG_ALIGN(sizeof(struct ifinfomsg));

  if ((msg->nlmsg_len - nlmsg_len) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Message inside newlink isn't valid!\n");
      return;
    }

  rtattr = (struct rtattr *)(((char *)ifinfo) + nlmsg_len);

  rtalen = RTA_ALIGN(sizeof(struct rtattr));
  
  // Validate the attribute we have, and determine if it is for wireless,
  // or wired.
  while (RTA_OK(rtattr, (msg->nlmsg_len - nlmsg_len)))
    {
      switch (rtattr->rta_type)
	{
	case IFLA_IFNAME:
	  // This is a non-wireless event. (Ignore it.)
	  break;

	case IFLA_WIRELESS:
	  // This is a wireless event.
	  cardif_linux_rtnetlink_ifla_wireless(intdata, ifinfo->ifi_index,
					       ((char *) rtattr)+rtalen,
					       rtattr->rta_len - rtalen, ws);
	  break;
	}

      rtlen = msg->nlmsg_len - nlmsg_len;

      // Get the next attribute
      rtattr = RTA_NEXT(rtattr, rtlen);
    }
}

/***********************************************************
 *
 * Check the string that identifies the custom event that we got. And
 * act on it.
 *
 ***********************************************************/
void cardif_linux_rtnetlink_check_custom(struct interface_data *intdata,
					 char *str)
{
  if (strncmp(str, "MLME-MICHAELMICFAILURE.indication", 33) == 0)
    {
      intdata->statemachine->MICfailures++;
      debug_printf(DEBUG_NORMAL, "MIC failure #%d!\n",
		   intdata->statemachine->MICfailures);
      
      if (strstr(str, " unicast ") != NULL)
	{
	  // The attempted attack was probably against the unicast key.
	  debug_printf(DEBUG_NORMAL, "MIC failure on unicast key!\n");
	  eapol_key_type254_request_new_key(intdata, 1);
	  intdata->send_size = 0;
	} else {
	  // The attempted attack was probably against the group key.
	  debug_printf(DEBUG_NORMAL, "MIC failure on group key!\n");
	  eapol_key_type254_request_new_key(intdata, 0);
	  intdata->send_size = 0;
	}

      if (intdata->statemachine->MICfailures >= 2)
	{
	  // The WPA/802.11i standard requires we assert countermeasures 
	  // for 60 seconds.
	  if (timer_check_existing(COUNTERMEASURE_TIMER))
	    {
	      debug_printf(DEBUG_NORMAL, "For some reason, we already have "
			   "a countermeasure timer in the queue!  Resetting "
			   "the timer!\n");
	      timer_reset_timer_count(COUNTERMEASURE_TIMER, 
				      MIC_COUNTERMEASURE_TIMEOUT);
	    } else {
	      debug_printf(DEBUG_NORMAL, "Enabling MIC countermeasures!\n");
	      timer_add_timer(COUNTERMEASURE_TIMER, MIC_COUNTERMEASURE_TIMEOUT,
			      NULL, &mic_disable_countermeasures);
	    }
	  cardif_countermeasures(intdata, TRUE);
	}
    }
}

/***********************************************************
 *
 * Check to see if we have become disassociated before the rekey_prob_timer
 * (found in profile.h) reaches 0.  If we have, it may indicate that we
 * have a card driver that resets the card on a key set.  This should only be
 * a problem with WEP and older card drivers.
 *
 ***********************************************************/
void cardif_linux_rtnetlink_check_key_prob(struct interface_data *idata)
{
  struct config_globals *globals;

  globals = config_get_globals();

  if (globals == NULL)
    {
      debug_printf(DEBUG_NORMAL, "No valid configuration globals available in"
		   " %s!\n", __FUNCTION__);
      return;
    }

  if (timer_check_existing(REKEY_PROB_TIMER))
    {
      if (!TEST_FLAG(globals->flags, CONFIG_GLOBALS_NO_FRIENDLY_WARNINGS))
	{
	  debug_printf(DEBUG_NORMAL, "** WARNING! ** You were disassocated "
		       "within a short time of setting a key!\nThis usually "
		       "means there is a problem with the card driver.\n"
		       "Please e-mail the developers for more information on "
		       "what this means. Be sure to include the type of card, "
		       "driver in use, and driver version number.\n");
	}
    }
}

/***********************************************************
 *
 * Process wireless events.
 *
 ***********************************************************/
void cardif_linux_rtnetlink_ifla_wireless(struct interface_data *idata, 
					  int ifindex, char *data, int len,
					  struct wireless_state *ws)
{
  struct iw_event iwe;
  struct config_network *network_data;
  struct config_globals *globals;
  struct stream_descr stream;
  int ret = 0;

  network_data = config_get_network_config();
  
  if (network_data == NULL)
    {
      // If we are in manual mode, and we just switched from an nonexistant
      // SSID to something else, we will get this "error".  It should
      // generally be considered a warning.
      debug_printf(DEBUG_CONFIG, "<warning> Invalid network configuration "
		   "structure! (%s:%d)\n", __FUNCTION__, __LINE__);
    }
 
  if ((data == NULL) || (len == 0))
    {
      debug_printf(DEBUG_NORMAL, "No data available in event!\n");
      return;
    }

  globals = config_get_globals();
  
  if (globals == NULL)
    {
      debug_printf(DEBUG_NORMAL, "No valid configuration global data in %s!\n",
		   __FUNCTION__);
      return;
    }

  iw_init_event_stream(&stream, data, len);
  do
    {
#ifdef NEW_IWLIB
      ret = iw_extract_event_stream(&stream, &iwe, 
				    cardif_linux_rtnetlink_get_we_ver(idata));
#else
      ret = iw_extract_event_stream(&stream, &iwe);
#endif

      if (ret <= 0) return;

      if ((ifindex != idata->intIndex) &&
	  (!(idata->flags & ONEDOWN)))
	{
	  debug_printf(DEBUG_INT, "Got a wireless event!  Interface index is %d, "
		       "we are using index %d!\n", ifindex, idata->intIndex);
	  debug_printf(DEBUG_INT, "Ignoring!\n");
	  return;
	}
 
      if (idata->flags & ONEDOWN)
	{
	  if ((ifindex < (idata->intIndex - 1)) || (ifindex > (idata->intIndex)))
	    {
	      debug_printf(DEBUG_INT, "Got a wireless event! Interface index is "
			   "%d, we are using indexes %d & %d!\n", ifindex,
			   idata->intIndex-1, idata->intIndex);
	      return;
	    }
	}

      // Process the event.
      cardif_linux_rtnetlink_process_token(idata, &iwe, ws);
    } while (ret > 0);
}

#endif
