/*******************************************************************
 * EAP One Time Password (OTP/GTC) implementation.
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * File: eapotp.c
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: eapotp.c,v 1.15 2005/10/17 03:56:54 chessing Exp $
 * $Date: 2005/10/17 03:56:54 $
 * $Log: eapotp.c,v $
 * Revision 1.15  2005/10/17 03:56:54  chessing
 * Updates to the libxsupconfig library.  It no longer relies on other source from the main tree, so it can be used safely in other code with problems.
 *
 * Revision 1.14  2005/10/14 02:26:18  shaftoe
 * - cleanup gcc 4 warnings
 * - (re)add support for a pid in the form of /var/run/xsupplicant.<iface>.pid
 *
 * -- Eric Evans <eevans@sym-link.com>
 *
 * Revision 1.13  2005/08/09 01:39:16  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/

#include <openssl/ssl.h>
#include <string.h>
#include <strings.h>

#include "xsupconfig.h"
#include "profile.h"
#include "eap.h"
#include "xsup_debug.h"
#include "xsup_err.h"
#include "frame_structs.h"
#include "eapotp.h"


/*****************************************************
 *
 * Setup to handle OTP EAP requests
 *
 *****************************************************/
int eapotp_setup(struct generic_eap_data *thisint)
{
  if (!thisint)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface struct passed to eapotp_setup()!\n");
      return XEMALLOC;
    }

  thisint->eap_data = (int *)malloc(sizeof(int));
  if (thisint->eap_data == NULL) return XEMALLOC;

  // Do anything special that might be needed for this EAP type to work.
  debug_printf(DEBUG_EVERYTHING, "Initalized EAP-OTP!\n");

  return XENONE;
}


/*****************************************************
 *
 * Process OTP EAP Requests
 *
 *
 ******************************************************/
int eapotp_process(struct generic_eap_data *thisint, u_char *dataoffs, 
		   int insize, u_char *outframe, int *outsize)
{
  char *otp_chal;
  char resp[512];
  struct config_eap_otp *userdata;

  debug_printf(DEBUG_EVERYTHING, "(EAP-OTP) Processing.\n");

  if ((!thisint) || (!dataoffs) || (!outframe))
    {
      debug_printf(DEBUG_NORMAL, "Invalid paramaters passed to eapotp_process()!\n");
      return XEMALLOC;
    }

  if (!outsize)
    {
      debug_printf(DEBUG_NORMAL, "Invalid pointer to out size in eapotp_process()!\n");
      return XEMALLOC;
    }
  
  debug_printf(DEBUG_AUTHTYPES, "OTP/GTC packet dump : \n");
  debug_hex_printf(DEBUG_AUTHTYPES, dataoffs, insize);

  *outsize = 0;

  userdata = thisint->eap_conf_data;

  if (!userdata)
    {
      debug_printf(DEBUG_NORMAL, "Invalid configuration data in eapotp_process()!\n");
      return XENOUSERDATA;
    }

  if (thisint->tempPwd == NULL) 
    {
      otp_chal = (char *)malloc(insize+1);
      if (otp_chal == NULL)
	{
	  debug_printf(DEBUG_NORMAL, "Couldn't allocate memory for OTP/GTC challenge!\n");
	  *outsize = 0;
	  return 0;
	}

      bzero(otp_chal, insize+1);

      memcpy(otp_chal, dataoffs, insize);
      debug_printf(DEBUG_NORMAL, "Challenge : %s\n",otp_chal);

      // We need a password.
      thisint->need_password = 1;
      thisint->eaptype = strdup("EAP-OTP/GTC");
      thisint->eapchallenge = otp_chal;

      *outsize = 0;
      return XENONE;
    }

  // Make sure we have something to process...
  if (dataoffs == NULL) return XENONE;


  /*  debug_printf(DEBUG_NORMAL, "Response : ");
      gets(&resp); */

  strcpy((char *) outframe, resp);
  *outsize = strlen(resp);

  return *outsize;
}

/*******************************************************
 *
 * Return any keying material that we may have.
 *
 *******************************************************/
int eapotp_get_keys(struct interface_data *thisint)
{
  return -1;   // No keys to return;
}

/*******************************************************
 *
 * Clean up after ourselves.  This will get called when we get a packet that
 * needs to be processed requests a different EAP type.  It will also be 
 * called on termination of the program.
 *
 *******************************************************/
int eapotp_cleanup(struct generic_eap_data *thisint)
{
  // Clean up after ourselves.
  debug_printf(DEBUG_AUTHTYPES, "(EAP-OTP) Cleaning up.\n");
  return XENONE;
}

