/*******************************************************************
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * File: ipc_callout.c
 *
 * Authors: Chris.Hessing@utah.edu, Terry.Simons@utah.edu
 *
 * $Id: ipc_callout.c,v 1.27 2005/10/17 03:56:53 chessing Exp $
 * $Date: 2005/10/17 03:56:53 $
 * $Log: ipc_callout.c,v $
 * Revision 1.27  2005/10/17 03:56:53  chessing
 * Updates to the libxsupconfig library.  It no longer relies on other source from the main tree, so it can be used safely in other code with problems.
 *
 * Revision 1.26  2005/08/09 01:39:14  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/
#include <netinet/in.h>
#include <strings.h>
#include <string.h>
#include <stdlib.h>

#include "profile.h"
#include "xsupconfig.h"
#include "xsup_debug.h"
#include "xsup_ipc.h"
#include "ipc_callout.h"

/*******************************************************************
 *
 * Fill in the next command record with the authentication state of the
 * selected interface.
 *
 *******************************************************************/
void ipc_callout_auth_state(struct interface_data *thisint, int *bufptr,
			    char *buffer, int bufsize, char *resbuf, 
			    int *resbufptr)
{
  struct ipc_cmd *cmd;
  int intidx;

  if ((!thisint) || (!bufptr) || (!buffer) || (!resbuf) || (!resbufptr) ||
      (!thisint->statemachine))
    {
      debug_printf(DEBUG_NORMAL, "Invalid data passed in to ipc_callout_auth_state()!\n");
      return;
    }

  cmd = (struct ipc_cmd *)&buffer[*bufptr];

  if (cmd->version != IPC_VERSION_NUM)
    {
      debug_printf(DEBUG_NORMAL, "Invalid IPC version request!  (Version %d)"
		   "\n", cmd->version);
      return;
    }

  intidx = cmd->int_idx;

  if (cmd->attribute != AUTH_STATE) 
    {
      debug_printf(DEBUG_NORMAL, "Incorrect call to ipc_callout_auth_state!"
		   "\n");
      *bufptr = bufsize;
      return;
    }

  if (cmd->getset != IPC_GET)
    {
      debug_printf(DEBUG_NORMAL, "Attempt to SET authentication state!? "
		   "Skipping!\n");
      *bufptr += (sizeof(struct ipc_cmd)+cmd->len);
      return;
    }

  if (cmd->len != 0)
    {
      debug_printf(DEBUG_NORMAL, "Invalid length!  This will be the last"
		   " request we answer! (In this packet.)\n");
      *bufptr = bufsize;
      return;
    }
  *bufptr+=sizeof(struct ipc_cmd);

  // Build the actual answer.
  cmd = (struct ipc_cmd *)&resbuf[*resbufptr];
  cmd->version = IPC_VERSION_NUM;
  cmd->int_idx = intidx;
  cmd->attribute = AUTH_STATE;
  cmd->getset = IPC_RESPONSE;
  cmd->len = 1;
  *resbufptr+=sizeof(struct ipc_cmd);

  resbuf[*resbufptr] = thisint->statemachine->curState;
  *resbufptr+=1;
}

/*****************************************************************
 *
 * Get or set config values.
 *
 *****************************************************************/
void ipc_callout_process_conf(struct interface_data *thisint, int *bufptr,
			      char *buffer, int bufsize, char *resbuf, 
			      int *resbufptr)
{
  // XXXX Finish this at some point!
  struct ipc_cmd *cmd;

  if ((!thisint) || (!bufptr) || (!buffer) || (!resbuf) || (!resbufptr))
    {
      debug_printf(DEBUG_NORMAL, "Invalid data passed in to ipc_callout_process_conf()!\n");
      return;
    }

  cmd = (struct ipc_cmd *)&buffer[*bufptr];

  if (cmd->version != IPC_VERSION_NUM)
    {
      debug_printf(DEBUG_NORMAL, "Invalid IPC control version requested! ("
		   "Version : %d)\n", cmd->version);
      return;
    }

  if (cmd->attribute != CONFIG) 
    {
      debug_printf(DEBUG_NORMAL, "Incorrect call to ipc_callout_process_conf!\n");
    }

  *bufptr += sizeof(struct ipc_cmd);

  debug_printf(DEBUG_NORMAL, "Configuration value modification is not "
	       "implemented yet!\n");
}

/******************************************************************
 *
 * Build a message to be sent.  This should be used *ONLY* as a call
 * internal to the ipc_callout.c file!
 *
 ******************************************************************/
void ipc_callout_build_msg(struct interface_data *thisint, int *bufptr,
			   char *buffer, int bufsize, int msgtype, 
			   char *message)
{
  struct ipc_cmd *cmd;

  // XXX Double check this.

  if ((!thisint) || (!bufptr) || (!buffer) || (!message))
    {
      debug_printf(DEBUG_NORMAL, "Invalid data passed in to ipc_callout_build_msg()!\n");
      return;
    }

  cmd = (struct ipc_cmd *)&buffer[*bufptr];

  cmd->version = IPC_VERSION_NUM;
  //  cmd->int_idx = intidx;
  cmd->attribute = msgtype;
  cmd->len = strlen(message);

  *bufptr += sizeof(struct ipc_cmd);

  strcpy((char *)&buffer[*bufptr], message);

  *bufptr += strlen(message);
}

/****************************************************************
 *
 * Send an error message to a client.
 *
 ****************************************************************/
void ipc_callout_send_error(struct interface_data *thisint, int *bufptr,
			    char *buffer, int bufsize, char *message)
{
  if ((!thisint) || (!bufptr) || (!buffer) || (!message))
    {
      debug_printf(DEBUG_NORMAL, "Invalid data passed in to ipc_callout_send_error()!\n");
      return;
    }

  ipc_callout_build_msg(thisint, bufptr, buffer, bufsize, ERROR_MSG, message);
}

/****************************************************************
 *
 * Send a notification to the client.
 *
 ****************************************************************/
void ipc_callout_send_notify(struct interface_data *thisint, int *bufptr,
			     char *buffer, int bufsize, char *message)
{
  if ((!thisint) || (!bufptr) || (!buffer) || (!message))
    {
      debug_printf(DEBUG_NORMAL, "Invalid data passed in to ipc_callout_send_notify()!\n");
      return;
    }

  ipc_callout_build_msg(thisint, bufptr, buffer, bufsize, NOTIFY, message);
}

/****************************************************************
 *
 * Get or set the profile we are using.
 *
 ****************************************************************/
void ipc_callout_getset_profile(struct interface_data *thisint, int *bufptr,
				char *buffer, int bufsize, char *resbuf, 
				int *resbufptr)
{
  //  struct ipc_cmd *cmd;

  if ((!thisint) || (!bufptr) || (!buffer) || (!resbuf) || (!resbufptr))
    {
      debug_printf(DEBUG_NORMAL, "Invalid data passed in to ipc_callout_getset_profile()!\n");
      return;
    }

  debug_printf(DEBUG_NORMAL, "Get/Set Profile Not Implemented!\n");
}

/***************************************************************
 *
 * Set a temporary password.  This password will be used by the first EAP
 * method that needs it.  Once it has been used, the EAP method should
 * bzero, and free the memory, in order to keep the password from sitting
 * in memory too long.
 *
 ***************************************************************/
void ipc_callout_set_password(struct interface_data *thisint, int *bufptr,
			      char *buffer, int bufsize, char *resbuf, 
			      int *resbufptr)
{
  struct ipc_cmd *cmd;
  int intidx;

  if ((!thisint) || (!bufptr) || (!buffer) || (!resbuf) || (!resbufptr))
    {
      debug_printf(DEBUG_NORMAL, "Invalid data passed in to ipc_callout_set_password()!\n");
      return;
    }

  cmd = (struct ipc_cmd *)&buffer[*bufptr];

  if (cmd->version != IPC_VERSION_NUM)
    {
      debug_printf(DEBUG_NORMAL, "Invalid IPC control version requested! ("
		   "Version : %d)\n", cmd->version);
      return;
    }

  intidx = cmd->int_idx;

  if (cmd->attribute != TEMPPASSWORD)
    {
      debug_printf(DEBUG_NORMAL, "Invalid call to %s!\n", __FUNCTION__);
      *bufptr = bufsize;
      return;
    }

  if (cmd->getset != IPC_SET)
    {
      debug_printf(DEBUG_NORMAL, "Cannot GET a temp password value!\n");
      *bufptr = bufsize;
      return;
    }

  // If we already have a temp password, we need to get rid of it.
  if (thisint->tempPassword != NULL)
    {
      free(thisint->tempPassword);
      thisint->tempPassword = NULL;
    }
  
  thisint->tempPassword = (char *)malloc(cmd->len+1);
  if (thisint->tempPassword == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't allocate memory for temporary password!\n");
      return;
    }

  bzero(thisint->tempPassword, cmd->len+1);
  *bufptr += sizeof(struct ipc_cmd);
  strcpy(thisint->tempPassword, (char *)&buffer[*bufptr]);

  *bufptr += strlen(thisint->tempPassword);

  cmd = (struct ipc_cmd *)&resbuf[*resbufptr];

  cmd->version = IPC_VERSION_NUM;
  cmd->int_idx = intidx;
  cmd->attribute = TEMPPASSWORD;
  cmd->getset = IPC_RESPONSE;
  cmd->len = 1;
  
  *resbufptr += sizeof(struct ipc_cmd);

  resbuf[*resbufptr] = ACK;
  *resbufptr += 1;
}

/***********************************************************************
 *
 * Ask any attached clients for a password.  In this message, we will
 * also pass the EAP type that is requesting the password, and any
 * challenge string that the EAP type may need.
 *
 ***********************************************************************/
void ipc_callout_request_password(int intidx, int *bufptr, char *buffer, 
				  int bufsize, char *eapname, char *challenge)
{
  struct ipc_cmd *cmd;

  if ((!bufptr) || (!buffer) || (!eapname))
    {
      debug_printf(DEBUG_NORMAL, "Invalid data passed in to ipc_callout_request_password()!\n");
      return;
    }

  cmd = (struct ipc_cmd *)&buffer[*bufptr];

  cmd->version = IPC_VERSION_NUM;
  cmd->int_idx = intidx;
  cmd->attribute = PASSWORD;
  cmd->getset = IPC_GET;
  if (challenge != NULL)
    {
      cmd->len = strlen(eapname)+strlen(challenge)+2;  // The string, with a NULL.
    } else {
      cmd->len = strlen(eapname)+1;
    }

  *bufptr += sizeof(struct ipc_cmd);

  bzero((char *)&buffer[*bufptr],cmd->len);
  strcpy((char *)&buffer[*bufptr], eapname);
  *bufptr += (strlen(eapname)+1);

  if (challenge != NULL)
    {
      strcpy((char *)&buffer[*bufptr], challenge);
      *bufptr += (strlen(challenge)+1);
    }
}
