/*******************************************************************
 *
 * File: wireless_sm.c
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: wireless_sm.c,v 1.13 2006/01/24 04:42:26 chessing Exp $
 * $Date: 2006/01/24 04:42:26 $
 * $Log: wireless_sm.c,v $
 * Revision 1.13  2006/01/24 04:42:26  chessing
 * A few more fixes to WPA code, along with a fix to the scan reaping code.
 *
 * Revision 1.12  2006/01/23 20:33:47  chessing
 * Added support for the replay counters in WPA/WPA2.  BSSID is now selected based on the signal quality information returned during the scan.  (Assuming signal quality information is returned during a scan. ;)
 *
 * Revision 1.11  2005/11/10 04:56:54  chessing
 * Added patch from Ben Gardner to add support for setting a specific WEP key prior to attempting to associte.  (With a few slight modifications by me to make it fit in the current CVS code, and get it supported in config-parse.)  Added patch from Pekka Savola to fix some header ordering issues, and a potential buffer overflow.
 *
 * Revision 1.10  2005/10/17 03:56:54  chessing
 * Updates to the libxsupconfig library.  It no longer relies on other source from the main tree, so it can be used safely in other code with problems.
 *
 * Revision 1.9  2005/10/14 02:26:17  shaftoe
 * - cleanup gcc 4 warnings
 * - (re)add support for a pid in the form of /var/run/xsupplicant.<iface>.pid
 *
 * -- Eric Evans <eevans@sym-link.com>
 *
 * Revision 1.8  2005/10/13 18:57:28  chessing
 * Removed some leftover debug strings.  Cleaned up some compiler warnings. Fixed an association bug in the madwifi driver.
 *
 * Revision 1.7  2005/10/13 18:46:47  chessing
 * Fixed to the Madwifi driver to allow it to do dynamic WEP again.  Fixed up the wext driver to behave correctly again. ;)
 *
 * Revision 1.6  2005/09/19 21:45:46  chessing
 * Fix for the madwifi driver when it connects to certain APs to do WEP.  (Currently the only known one with this problem is the Trapeze APs when they are running MSS 4.x+.)
 *
 * Revision 1.5  2005/09/17 18:38:17  chessing
 * Some updates to make the madwifi driver work with dynamic WEP again.
 *
 * Revision 1.4  2005/09/14 03:34:54  chessing
 * Small cosmetic changes.  Default association mode is now auto instead of manual. Fixes for bug IDs #1290449 & #1290323.
 *
 * Revision 1.3  2005/09/14 02:50:44  chessing
 * Major updates.  Auto association now works.  Started to rewrite the rtnetlink pieces using iwlib from the wireless tools to avoid compatibility issues.  As a result, getting the WPA and RSN IEs via the IWEVGENIE event no longer works for some reason, but the old style using IWEVCUSTOM events should still work like a champ.
 *
 * Revision 1.2  2005/09/08 16:27:01  chessing
 * Some small updates to the new state machine code.  First attempt at an auto association mode.  (It mostly works. ;)
 *
 * Revision 1.1  2005/09/05 01:00:34  chessing
 * Major overhaul to most of the state machines in Xsupplicant.  Also added additional error messages to the TLS functions to try to debug the one of the problems reported on the list.  Basic testing shows this new code to be more stable than previous code, but it needs more testing.
 *
 *
 *******************************************************************/

#include <stdlib.h>
#include <unistd.h>
#include <inttypes.h>
#include <string.h>

#include "profile.h"
#include "xsupconfig.h"
#include "xsup_debug.h"
#include "xsup_err.h"
#include "wireless_sm.h"
#include "eapol.h"
#include "event_core.h"
#include "cardif/cardif.h"
#include "config_ssid.h"
#include "cardif/core.h"
#include "timer.h"
#include "statemachine.h"
#include "eap.h"
#include "cardif/linux/cardif_linux_wext.h"

struct wireless_state *wireless_sm = NULL;

void set_static_wep_keys(struct interface_data *ctx,
                         struct config_static_wep *wepdata);

// This is in mschapv2.c.  We need it for the static WEP piece.
extern void process_hex(char *, int, char *);


/********************************************************************
 *
 * Initialize our wireless statemachine.  We should always start in
 * ACTIVE_SCAN mode, since we won't know anything about the SSID we
 * are attempting to connect to.
 *
 ********************************************************************/
void wireless_sm_init(int zeros_on_roam, struct interface_data *ctx)
{
   struct config_network *network_data;

  debug_printf(DEBUG_STATE, "Init wireless state machine.\n");
  wireless_sm = (struct wireless_state *)malloc(sizeof(struct wireless_state));
  if (!wireless_sm)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't allocate memory for the wireless"
		   " state machine!\n");
      _exit(1);
    }

  bzero(wireless_sm, sizeof(struct wireless_state));
  
  if (cardif_get_if_state(ctx) == FALSE)
    {
      // Our interface is currently down.  So go to PORT_DOWN mode.
      debug_printf(DEBUG_NORMAL, "PORT_DOWN from %s!\n", __FUNCTION__);
      wireless_sm_change_state(PORT_DOWN, ctx);
    } else {
      wireless_sm_change_state(ACTIVE_SCAN, ctx);
    }

  if (zeros_on_roam == TRUE) 
    {
      SET_FLAG(wireless_sm->flags, ZEROSONROAM);
    } else {
      UNSET_FLAG(wireless_sm->flags, ZEROSONROAM);
    }

  // Probably don't need to do this here, but do it anyway for good 
  // measure, unless we determine it causes a problem for some cards.
    network_data = config_get_network_config();
    if ((network_data == NULL) || (network_data->initial_wep == NULL))
    {
       cardif_clear_keys(ctx);
    }
    else
    {
       set_static_wep_keys(ctx, network_data->initial_wep);
    }
}


/**
 * Sets the WEP keys according to the config_static_wep structure.
 * This is used with the initial_wep setting and with static_wep.
 *
 * @todo It may be a good idea to clear the keys not being set
 *
 * @param ctx     The interface data
 * @param wepdata WEP keys
 */
void set_static_wep_keys(struct interface_data *ctx,
                         struct config_static_wep *wepdata)
{
   int keyidx, klen, t;
   char key[26];

   for (keyidx = 1; keyidx < 5; keyidx++)
   {
      if (wepdata->key[keyidx] != NULL)
      {
         klen = strlen((char *)wepdata->key[keyidx]);
         if (((klen/2) == 5) || ((klen/2) == 13))
         {
            // We have a valid length key.  So, convert it, and set it.
            process_hex((char *)wepdata->key[keyidx], klen, key);

            // Calculate the proper key index.
            t = keyidx-1;

            if (keyidx == wepdata->tx_key)
            {
               debug_printf(DEBUG_INT, "%s: Setting TX key! [%d]\n",
                            __func__, wepdata->tx_key);
               t |= 0x80;
               cardif_set_wep_key(ctx, (u_char *)key, (klen/2), t);
            }
            else
            {
               cardif_set_wep_key(ctx, (u_char *)key, (klen/2), t);
            }
         }
      }
   }
}


/********************************************************************
 *
 * Deinit our wireless statemachine.  Currently does nothing, but is 
 * included to maintain the init/active/deinit model.
 *
 ********************************************************************/
void wireless_sm_deinit()
{
  if (wireless_sm)
    {
      free(wireless_sm);
      wireless_sm = NULL;
    }
}

/********************************************************************
 *
 * Print out the text value for the state that we have been passed.
 *
 ********************************************************************/
void wireless_sm_disp_state(int debug_level, int showstate)
{
  switch (showstate)
    {
    case UNASSOCIATED:
      debug_printf_nl(debug_level, "UNASSOCIATED");
      break;

    case ASSOCIATED:
      debug_printf_nl(debug_level, "ASSOCIATED");
      break;

    case ACTIVE_SCAN:
      debug_printf_nl(debug_level, "ACTIVE_SCAN");
      break;

    case ASSOCIATING:
      debug_printf_nl(debug_level, "ASSOCIATING");
      break;

    case ASSOCIATION_TIMEOUT_S:
      debug_printf_nl(debug_level, "ASSOCIATION_TIMEOUT");
      break;

    case STATIC_ASSOCIATION:
      debug_printf_nl(debug_level, "STATIC_ASSOCIATION");
      break;

    case PORT_DOWN:
      debug_printf_nl(debug_level, "PORT_DOWN");
      break;

    case NO_ENC_ASSOCIATION:
      debug_printf_nl(debug_level, "NO_ENC_ASSOCIATION");
      break;

    default:
      debug_printf_nl(DEBUG_INT, "UNKNOWN!!!!!");
      break;
    }
}

/********************************************************************
 *
 * Display information about our state change.
 *
 ********************************************************************/
void wireless_sm_disp_state_change(int debug_level, int newstate)
{
  wireless_sm_disp_state(debug_level, wireless_sm->state);
  debug_printf_nl(debug_level, " -> ");
  wireless_sm_disp_state(debug_level, newstate);
  debug_printf_nl(debug_level, "\n");
}

/********************************************************************
 *
 * Change to unassociated state.
 *
 ********************************************************************/
void wireless_sm_change_to_unassociated(struct interface_data *ctx)
{
  struct config_network *network_data;

  wireless_sm->associated = 2;   // Change to something that will keep us from
                                 // triggering this change over and over.

  network_data = config_get_network_config();

  if (wireless_sm->state != ASSOCIATED)
    {
      // Only change to unassociated state if we are already in associated
      // state.
      return;
    }

  // We are allowed to switch from any state to unassociated state.
  wireless_sm_disp_state_change(DEBUG_STATE, UNASSOCIATED);

  eap_clear_active_method(network_data->activemethod);

  // Update our state variables to indicate what state we are in now.
  wireless_sm->state = UNASSOCIATED;

  // Then, switch to active scan state to see if we can associate again.
  wireless_sm_change_state(ACTIVE_SCAN, ctx);
}

/********************************************************************
 *
 * Change to ACTIVE_SCAN state.
 *
 ********************************************************************/
void wireless_sm_change_to_active_scan(struct interface_data *ctx)
{
  //  u_char abil;

  // Make sure that last state we were in has a valid transition to this
  // state.
  wireless_sm_disp_state_change(DEBUG_STATE, ACTIVE_SCAN);

  /*
    // XXX This code is broken!
  if ((wireless_sm->state == ASSOCIATING) ||
      (wireless_sm->state == ASSOCIATED))
    {
      abil = config_ssid_get_ssid_abilities();

      // We only flip the zeros on roam bit if we are doing WEP.
      if ((!(abil & WPA_IE)) && (!(abil & RSN_IE)))
	{
	  // We switched to active scan from associating, so flip the
	  // zeros on roam bit.
	  if (TEST_FLAG(wireless_sm->flags, ZEROSONROAM))
	    {
	      debug_printf(DEBUG_INT, "Next association will attempt to use no"
			   " encryption.\n");
	      UNSET_FLAG(wireless_sm->flags, ZEROSONROAM);
	    } else {
	      debug_printf(DEBUG_INT, "Next association will attempt to use all"
			   " zeros.\n");
	      SET_FLAG(wireless_sm->flags, ZEROSONROAM);
	    }
	}
    }
  */

  // Update our state variables to indicate what state we are in now.
  wireless_sm->state = ACTIVE_SCAN;
  wireless_sm->initialize = FALSE;
  wireless_sm->ssid_change = FALSE;

  // Do any setup that is needed to enter the new state.
  statemachine_reinit(ctx);
  cardif_do_wireless_scan(ctx, FALSE);
}

/********************************************************************
 *
 * Change to ASSOCIATED state.
 *
 ********************************************************************/
void wireless_sm_change_to_associated(struct interface_data *ctx)
{

  wireless_sm->associated = 2;        // Change to something that will keep
                                      // us from triggering this state over
                                      // and over.

  // Make sure that last state we were in has a valid transition to this
  // state.
  wireless_sm_disp_state_change(DEBUG_STATE, ASSOCIATED);

  // We are associated, so clear the timer.
  timer_cancel(ASSOCIATION_TIMER);

  // Update our state variables to indicate what state we are in now.
  wireless_sm->state = ASSOCIATED;
}


/********************************************************************
 *
 * We have a static key for this SSID. So set it.
 *
 ********************************************************************/
void wireless_sm_change_to_static_association(struct interface_data *ctx)
{
  struct config_network *netdata;
  struct config_static_wep *wepdata;

  wireless_sm_disp_state_change(DEBUG_STATE, STATIC_ASSOCIATION);

  debug_printf(DEBUG_INT, "We appear to be using static WEP.\n");

  netdata = config_get_network_config();
  if (!netdata)
    {
      debug_printf(DEBUG_NORMAL, "No valid network config! (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      return;
    }

  if (!netdata->methods)
    {
      debug_printf(DEBUG_NORMAL, "No valid method config! (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      return;
    }

  wireless_sm->state = STATIC_ASSOCIATION;

  wepdata = (struct config_static_wep *)netdata->methods->method_data;

  set_static_wep_keys(ctx, wepdata);
}

/********************************************************************
 *
 * We have no key, so turn off encryption.
 *
 ********************************************************************/
void wireless_sm_change_to_no_enc_association(struct interface_data *ctx)
{
  wireless_sm_disp_state_change(DEBUG_STATE, NO_ENC_ASSOCIATION);

  debug_printf(DEBUG_INT, "We have no configuration, clearing encryption.\n");
  if (ctx) cardif_enc_disable(ctx);
  wireless_sm->state = NO_ENC_ASSOCIATION;
}

/********************************************************************
 *
 * If we get a timeout trying to associate.
 *
 ********************************************************************/
void wireless_sm_association_timeout(struct interface_data *ctx)
{
  // Cancel our timer, so we don't keep running the timeout function.
  timer_cancel(ASSOCIATION_TIMER);

  // Our association timer expired, so change to ASSOCIATION_TIMEOUT_S
  // state.
  wireless_sm_change_state(ASSOCIATION_TIMEOUT_S, ctx);
}

/********************************************************************
 *
 * Change state to ASSOCIATING.
 *
 ********************************************************************/
void wireless_sm_change_to_associating(struct interface_data *ctx)
{
  struct config_network *netdata;
  struct config_globals *globals;

  // Make sure that last state we were in has a valid transition to this
  // state.
  wireless_sm_disp_state_change(DEBUG_STATE, ASSOCIATING);

  netdata = config_get_network_config();

  if (!netdata)
    {
      debug_printf(DEBUG_INT, "No valid network data!! (%s)\n",
		   __FUNCTION__);
      return;
    }

  // Clear the replay counter.
  bzero(ctx->statemachine->replay_counter, 8);

  /*
  if ((wireless_sm->state != ASSOCIATING) && 
      (wireless_sm->state != ACTIVE_SCAN))
    {
      // We have attempted to switch to associating mode from an invalid
      // state.
      debug_printf(DEBUG_NORMAL, "Attempted to change to associating state "
		   "from an invalid state.  Forcing a reinit of wireless sm "
		   "in an attempt to recover.\n");
      wireless_sm->initialize = TRUE;
      return;
      }*/

  wireless_sm->state = ASSOCIATING;

  globals = config_get_globals();
  if (globals)
    {
      if (timer_check_existing(ASSOCIATION_TIMER))
	{
	  timer_reset_timer_count(ASSOCIATION_TIMER, globals->assoc_timeout);
	} else {
	  timer_add_timer(ASSOCIATION_TIMER, globals->assoc_timeout, NULL,
			  &wireless_sm_association_timeout);
	}
    } else {
      debug_printf(DEBUG_NORMAL, "Couldn't read global variable information!"
		   " Association timer will not function correctly!\n");
    }

  statemachine_reinit(ctx);

  if (!(netdata->initial_wep == NULL))
    {
      set_static_wep_keys(ctx, netdata->initial_wep);
    }
				  
  if (config_ssid_using_wep())
    {
      // Do any setup that is needed to enter the new state.
      cardif_wep_associate(ctx, (wireless_sm->flags & ZEROSONROAM));
      debug_printf(DEBUG_NORMAL, "Listed SSID is %s\n", ctx->cur_essid);
    }  else {
      cardif_reassociate(ctx, 1);
    }

  /*
  if (cardif_GetSSID(ctx, temp_ssid) != XENONE)
    {
      cardif_reassociate(ctx, 1);
    } else {
      if (strcmp(temp_ssid, ctx->cur_essid) != 0)
	{
	  cardif_reassociate(ctx, 1);
	}
    }
  */
}

/********************************************************************
 *
 * If we timed out trying to associate, then do this, and switch to
 * active scan mode.
 *
 ********************************************************************/
void wireless_sm_change_to_association_timeout(struct interface_data *ctx)
{
  u_char abilities;

  // Make sure that last state we were in has a valid transition to this
  // state.
  wireless_sm_disp_state_change(DEBUG_STATE, ASSOCIATION_TIMEOUT_S);

  abilities = config_ssid_get_ssid_abilities();

  // If we are doing WPA or RSN, then we shouldn't mess with the zeros on
  // roam flag.
  if ((!(abilities & WPA_IE)) && (!(abilities & RSN_IE)))
    {
      if (TEST_FLAG(wireless_sm->flags, ZEROSONROAM))
	{
	  UNSET_FLAG(wireless_sm->flags, ZEROSONROAM);
	} else {
	  SET_FLAG(wireless_sm->flags, ZEROSONROAM);
	}
    }

  wireless_sm->state = ASSOCIATION_TIMEOUT_S;
  wireless_sm_change_state(ACTIVE_SCAN, ctx);
}

/********************************************************************
 *
 * If the port is down, change to the PORT_DOWN state.
 *
 ********************************************************************/
void wireless_sm_change_to_port_down(struct interface_data *ctx)
{
  // Make sure that last state we were in has a valid transition to this
  // state.
  wireless_sm_disp_state_change(DEBUG_STATE, PORT_DOWN);  
  wireless_sm->state = PORT_DOWN;
  statemachine_reinit(ctx);
}

/********************************************************************
 *
 * Check for global events that would signal a state change.
 *
 ********************************************************************/
void wireless_sm_check_globals(struct interface_data *ctx)
{
  struct config_network *netdata;

  netdata = config_get_network_config();

  if (!netdata)
    {
      debug_printf(DEBUG_INT, "No valid network data!! (%s)\n",
		   __FUNCTION__);
      return;
    } else {
      if ((netdata->methods == NULL) &&
	  (wireless_sm->state != NO_ENC_ASSOCIATION))
	{
	  ctx->statemachine->portControl = FORCE_AUTHORIZED;
	  statemachine_change_state(ctx, S_FORCE_AUTH);
	  wireless_sm_change_state(NO_ENC_ASSOCIATION, ctx);
	  return;
	}
      
      if ((netdata->methods != NULL) &&
	  (netdata->methods->method_num == STATIC_WEP_METHOD) &&
	  (wireless_sm->state != STATIC_ASSOCIATION))
	{
	  ctx->statemachine->portControl = FORCE_AUTHORIZED;
	  statemachine_change_state(ctx, S_FORCE_AUTH);
	  wireless_sm_change_state(STATIC_ASSOCIATION, ctx);
	  return;
	}

    }

  if ((wireless_sm->port_active == TRUE) && (!cardif_get_if_state(ctx)))
    {
      // Our interface is down.
      wireless_sm->port_active = FALSE;
      wireless_sm_change_state(PORT_DOWN, ctx);
    }

  if ((wireless_sm->initialize) || (wireless_sm->ssid_change))
    {
      debug_printf(DEBUG_INT, "Initialize : %d    SSID Change : %d\n",
		   wireless_sm->initialize, wireless_sm->ssid_change);
      //      for (i=0;i<30;i++) sleep(1);
      wireless_sm_change_state(ACTIVE_SCAN, ctx);
    }

  if (wireless_sm->associated == TRUE)
    {
      //      wireless_sm_change_state(ASSOCIATED, ctx);
    }

  if (wireless_sm->associated == FALSE)
    {
      wireless_sm_change_state(UNASSOCIATED, ctx);
    }

}

/********************************************************************
 *
 * Change state from where we are, to a new state.  The individual state
 * change handlers *MUST* make sure they are changing from a valid state
 * in to the new state.
 *
 ********************************************************************/
void wireless_sm_change_state(int newstate, struct interface_data *ctx)
{
  // If we aren't wireless, then ignore state change requests.
  if (!TEST_FLAG(ctx->flags, IS_WIRELESS)) return;

  switch (newstate)
    {
    case UNASSOCIATED:
      wireless_sm_change_to_unassociated(ctx);
      break;

    case ASSOCIATED:
      wireless_sm_change_to_associated(ctx);
      break;

    case ACTIVE_SCAN:
      wireless_sm_change_to_active_scan(ctx);
      break;

    case ASSOCIATING:
      wireless_sm_change_to_associating(ctx);
      break;

    case ASSOCIATION_TIMEOUT_S:
      wireless_sm_change_to_association_timeout(ctx);
      break;

    case STATIC_ASSOCIATION:
      wireless_sm_change_to_static_association(ctx);
      break;

    case PORT_DOWN:
      wireless_sm_change_to_port_down(ctx);
      break;

    case NO_ENC_ASSOCIATION:
      wireless_sm_change_to_no_enc_association(ctx);
      break;
    }
}

/*********************************************************************
 *
 * Handle an event while in port down state.
 *
 *********************************************************************/
void wireless_sm_do_port_down(struct interface_data *ctx)
{
  wireless_sm->port_active = cardif_get_if_state(ctx);

  if (wireless_sm->port_active == FALSE)
    {
      // Interface is still down. Sleep before checking again.
      sleep(1);
    } else {
      wireless_sm_change_state(ACTIVE_SCAN, ctx);
    }
}

/*********************************************************************
 *
 * Handle an event while we are in unassociated mode.
 *
 *********************************************************************/
void wireless_sm_do_unassociated(struct interface_data *ctx)
{
  struct config_globals *globals;

  globals = config_get_globals();
  if (!globals)
    {
      debug_printf(DEBUG_NORMAL, "Invalid configuration information "
		   "structure.\n");
      _exit(1);
    }

  if (TEST_FLAG(globals->flags, CONFIG_GLOBALS_ASSOC_AUTO))
    {
      // We are set to auto associate.  So, switch to active scan mode.
      wireless_sm_change_state(ACTIVE_SCAN, ctx);
    } else {
      // Otherwise, we have nothing to do, so take a short nap.  (To avoid
      // hammering the processor.

      if (wireless_sm->associated)
	{
	  wireless_sm_change_state(ASSOCIATED, ctx);
	} else {
	  usleep(500000);
	}
    }
}

/*********************************************************************
 *
 * Handle an event while we are in associated mode.
 *
 *********************************************************************/
void wireless_sm_do_associated(struct interface_data *ctx)
{
  // If we are associated, then go ahead and run the EAPOL state machine.
  event_core(ctx);
}

/*********************************************************************
 *
 * Handle an event while we are in active scan mode.
 *
 *********************************************************************/
void wireless_sm_do_active_scan(struct interface_data *ctx)
{
  char *newssid;

  if (!TEST_FLAG(ctx->flags, SCANNING))
    {
      config_ssid_dump();

      // We aren't scanning, so see if we have a valid SSID we can attach
      // to.
      newssid = config_ssid_get_desired_ssid(ctx);
      if (newssid != NULL)
	{
	  debug_printf(DEBUG_STATE, "Switching to Associating mode to"
		       " connect to %s.\n", newssid);

	  // Clear out the SSID we are currently connected to.
	  if (ctx->cur_essid != NULL)
	    {
	      free(ctx->cur_essid);
	    }

	  ctx->cur_essid = strdup(newssid);

	  config_build(ctx, ctx->cur_essid);

	  wireless_sm_change_state(ASSOCIATING, ctx);
	  return;
	} else {
	  // If we didn't find anything, sleep for a few seconds before we try
	  // again. 
	  timer_sleep(30);

	  cardif_do_wireless_scan(ctx, FALSE);
	}
    } else {
      // Otherwise, start scanning again.
      cardif_do_wireless_scan(ctx, FALSE);
    }
}

/*********************************************************************
 *
 * Handle an event while we are in associating mode.
 *
 *********************************************************************/
void wireless_sm_do_associating(struct interface_data *ctx)
{
  // There may be situations where we are already associated when we get
  // to this point.  So, check.
  if (cardif_check_associated(ctx) == IS_ASSOCIATED)
    {
      wireless_sm_change_state(ASSOCIATED, ctx);
      return;
    }
  usleep(500000);
}

/*********************************************************************
 *
 * Handle events while we are in static association mode.
 *
 *********************************************************************/
void wireless_sm_do_static_association(struct interface_data *ctx)
{
  usleep(500000);
}

/*********************************************************************
 *
 * Handle events while we are in no encryption association mode.
 *
 *********************************************************************/
void wireless_sm_do_no_enc_association(struct interface_data *ctx)
{
  usleep(500000);
}

/*********************************************************************
 *
 * Actually call routines that act on the state that we are currently
 * in.
 *
 *********************************************************************/
void wireless_sm_do_state(struct interface_data *ctx)
{
  if (!wireless_sm)
    {
      debug_printf(DEBUG_NORMAL, "No valid wireless state machine context!\n");

      // We can't continue!
      _exit(2);
      return;
    }

  ext_event_core(ctx, wireless_sm);

  wireless_sm_check_globals(ctx);

  switch (wireless_sm->state)
    {
    case UNASSOCIATED:
      wireless_sm_do_unassociated(ctx);
      break;

    case ASSOCIATED:
      wireless_sm_do_associated(ctx);
      break;

    case ACTIVE_SCAN:
      wireless_sm_do_active_scan(ctx);
      break;

    case ASSOCIATING:
      wireless_sm_do_associating(ctx);
      break;

    case ASSOCIATION_TIMEOUT_S:
      // The association timeout state simply changes the configuration
      // needed to attempt to associate, and then changes to ACTIVE_SCAN
      // state.
      break;

    case STATIC_ASSOCIATION:
      wireless_sm_do_static_association(ctx);
      break;

    case PORT_DOWN:
      wireless_sm_do_port_down(ctx);
      break;

    case NO_ENC_ASSOCIATION:
      wireless_sm_do_no_enc_association(ctx);
      break;

    default:
      debug_printf(DEBUG_NORMAL, "Unknown state %d!\n", wireless_sm->state);
      break;
    }

  if (ctx->tick) timer_tick(ctx);
}
