/*******************************************************************
 * File: wpa2.c
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: wpa2.c,v 1.17 2006/01/23 05:28:37 chessing Exp $
 * $Date: 2006/01/23 05:28:37 $
 * $Log: wpa2.c,v $
 * Revision 1.17  2006/01/23 05:28:37  chessing
 * Fixed a few settings that were causing errors with IOCTLs on some cards.  Updated WPA2 code to properly process group key packets. We now record quality, signal level, and noise level from scan results, so that we can start to make better decisions on which AP to associate to.
 *
 * Revision 1.16  2006/01/20 06:53:48  chessing
 * Added support for large scan results.
 *
 * Revision 1.15  2006/01/19 23:01:43  chessing
 * Added a CHANGELOG file.  Fixed a couple of things in wpa.c
 *
 * Revision 1.14  2006/01/19 22:23:36  chessing
 * Added the ability to associate to an AP without having wpa_group_cipher, or wpa_pairwise_cipher defined in the config file.  If they are defined, then the value in the config file will be used no matter what.  Also, changed a piece in xsup_driver.c so that we print the capabilities that the cards reports on startup.  (This should help debug some problems that are almost certain to pop up with this new code. ;)
 *
 * Revision 1.13  2006/01/19 05:37:04  chessing
 * WPA2 is working correctly.  Added the ability to query the card to gather encryption/authentication capabilities.  1.2.3 is now ready to go.
 *
 * Revision 1.12  2005/12/24 04:51:51  chessing
 * Removed an unneeded file.  Updated scanning code to parse IWEVGENIE events correctly, and make use of them.
 *
 * Revision 1.11  2005/10/17 03:56:54  chessing
 * Updates to the libxsupconfig library.  It no longer relies on other source from the main tree, so it can be used safely in other code with problems.
 *
 * Revision 1.10  2005/08/25 03:34:05  chessing
 * Removed a bunch of functions from config.c that could be handled better in other ways.
 *
 * Revision 1.9  2005/08/09 01:39:14  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/

#include <string.h>
#include <inttypes.h>
#include <openssl/hmac.h>
#include <openssl/aes.h>

#include "profile.h"
#include "xsupconfig.h"
#include "xsup_debug.h"
#include "wpa.h"
#include "wpa_common.h"
#include "wpa2.h"
#include "config_ssid.h"
#include "cardif/cardif.h"

const char wpa2oui[3] = {0x00, 0x0f, 0xac};
uint16_t wpa2_ver = 1;

/********************************************************************
 *
 * Determine the pairwise crypto type to use.  If something is set in the
 * config file, we will use that no matter what the card reports as it's
 * capabilitites.  Otherwise, we will look at what the AP sent us in it's
 * IE, and attempt to match it.
 *
 ********************************************************************/
uint8_t wpa2_get_pairwise_crypt(struct interface_data *ctx)
{
  struct config_network *network_data;
  uint8_t available_pair = 0, rsn_ie_len = 0;
  uint8_t *pairptr;
  uint8_t *rsn_ie, i;
  uint16_t *ciphers;

  if (!ctx)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface data in %s!\n",
		   __FUNCTION__);
      return -1;
    }

  network_data = config_get_network_config();
  
  if (network_data == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid network configuration struct! "
		   "(%s:%d)\n", __FUNCTION__, __LINE__);
      return -1;
    }

  config_ssid_get_rsn_ie(&rsn_ie, &rsn_ie_len);

  if (!rsn_ie)
    {
      debug_printf(DEBUG_NORMAL, "This SSID didn't return an RSN IE in our "
		   "scan!  We won't be able to connect!\n");
    }

  if ((rsn_ie_len <= 0) && (network_data->wpa_pairwise_crypt == 0))
    {
      debug_printf(DEBUG_NORMAL, "Insufficient information to build WPA2 "
		   "IE.  Please set 'wpa_pairwise_crypt' value in your network"
		   " clause for this network.\n");
      return -1;
    }

  // If the user has manually set a crypto type, then use it.
  if (network_data->wpa_pairwise_crypt != 0) 
    return network_data->wpa_pairwise_crypt;

  // Otherwise, see what the card has told us we can support, and compare it
  // to what the AP claims to support.
  ciphers = (uint16_t *)&rsn_ie[8];

  debug_printf(DEBUG_INT, "There are %d pairwise cipher(s) in this IE.\n",
	       *ciphers);
  pairptr = (uint8_t *)&rsn_ie[10];

  for (i=0;i<(*ciphers);i++)
    {
      debug_hex_printf(DEBUG_NORMAL, pairptr, 4);

      if (memcmp(pairptr, &wpa2oui, 3) != 0)
	{
	  debug_printf(DEBUG_NORMAL, "One of this AP's pairwise key settings "
		       "seems to be proprietary.  Skipping.\n");
	  pairptr += 4;  // Skip this.
	}
      else 
	{
	  pairptr += 3;
	  
	  if ((*pairptr) == CIPHER_WEP40) available_pair |= DOES_WEP40;
	  if ((*pairptr) == CIPHER_WEP104) available_pair |= DOES_WEP104;
	  if ((*pairptr) == CIPHER_TKIP) available_pair |= DOES_TKIP;
	  if ((*pairptr) == CIPHER_CCMP) available_pair |= DOES_CCMP;

	  pairptr++;
	}
    }

  // We want to test for cipher types from the best down to the worst, so
  // that we will select the best cipher possible.
  if (available_pair & DOES_CCMP)
    {
      if (ctx->enc_capa & DOES_CCMP)
	{
	  return CIPHER_CCMP;
	} 
      else 
	{
	  debug_printf(DEBUG_NORMAL, "The AP has requested that we use CCMP"
		       " for the pairwise cipher.  But, your card reports that"
		       " it doesn't support CCMP.  If you are sure that your"
		       " card supports CCMP, you should add "
		       "'wpa_pairwise_cipher = CCMP' to your network clause for"
		       " this network.\n");
	  return -1;
	}
    }

 if (available_pair & DOES_TKIP) 
    {
      if (ctx->enc_capa & DOES_TKIP)
	{
	  return CIPHER_TKIP;
	}
      else
	{
	  debug_printf(DEBUG_NORMAL, "The AP has requested that we use TKIP"
		       " for the pairwise cipher.  But, your card reports that"
		       " it doesn't support TKIP.  If you are sure that your"
		       " card supports TKIP, you should add "
		       "'wpa_pairwise_cipher = TKIP' to your network clause for"
		       " this network.\n");
	  return -1;
	}
    }

  if (available_pair & DOES_WEP104)
    {
      if (ctx->enc_capa & DOES_WEP104)
	{
	  return CIPHER_WEP104;
	}
      else
	{
	  debug_printf(DEBUG_NORMAL, "The AP has requested that we use WEP104"
		       " for the pairwise cipher.  But, your card reports that"
		       " it doesn't support WEP104.  If you are sure that your"
		       " card supports WEP104, you should add "
		       "'wpa_pairwise_cipher = WEP104' to your network clause for"
		       " this network.\n");
	  return -1;
	}
    }

 if (available_pair & DOES_WEP40)
   {
     if (ctx->enc_capa & DOES_WEP40)
       {
	 return CIPHER_WEP40;
       }
     else
       {
	 debug_printf(DEBUG_NORMAL, "The AP has requested that we use WEP40"
		      " for the pairwise cipher.  But, your card reports that"
		      " it doesn't support WEP40.  If you are sure that your"
		      " card supports WEP40, you should add "
		      "'wpa_pairwise_cipher = WEP40' to your network clause for"
		      " this network.\n");
	 return -1;
       }
   }

  // If we get here, then the AP has requested a cipher type we don't 
  // understand.
  debug_printf(DEBUG_NORMAL, "The AP has requested a cipher type that we "
	       "don't understand.\n");
  return -1;
}

/***************************************************************************
 *
 * Determine the proper group cipher to use. If something was set in the
 * configuration file, then we will use that.  Otherwise we will attempt to 
 * figure out we should used based on what the AP tells us in the IE, and
 * the capabilities the card reports.
 *
 ***************************************************************************/
uint8_t wpa2_get_group_crypt(struct interface_data *ctx)
{
  struct config_network *network_data;
  uint8_t desired_group = -1, rsn_ie_len = 0;
  uint8_t *grpptr;
  uint8_t *rsn_ie;

  if (!ctx)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface data in %s!\n",
		   __FUNCTION__);
      return -1;
    }

  network_data = config_get_network_config();
  
  if (network_data == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid network configuration struct! "
		   "(%s:%d)\n", __FUNCTION__, __LINE__);
      return -1;
    }

  config_ssid_get_rsn_ie(&rsn_ie, &rsn_ie_len);

  if (!rsn_ie)
    {
      debug_printf(DEBUG_NORMAL, "This SSID didn't return an RSN IE in our "
		   "scan!  We won't be able to connect!\n");
    }

  if ((rsn_ie_len <= 0) && (network_data->wpa_group_crypt == 0))
    {
      debug_printf(DEBUG_NORMAL, "Insufficient information to build WPA2 "
		   "IE.  Please set 'wpa_group_crypt' value in your network"
		   " clause for this network.\n");
      return -1;
    }

  // If the user has manually set a crypto type, then use it.
  if (network_data->wpa_group_crypt != 0) 
    return network_data->wpa_group_crypt;

  // Otherwise, see what the card has told us we can support, and compare it
  // to what the AP claims to support.
  grpptr = (uint8_t *)&rsn_ie[4];
  if (memcmp(grpptr, &wpa2oui, 3) != 0)
    {
      debug_printf(DEBUG_NORMAL, "AP's group key setting seems to be "
		   "proprietary.  This is unsupported.\n");
      return -1;
    }

  // Get the key type that is desired.
  desired_group = grpptr[3];

  if (desired_group == CIPHER_WEP40)
    {
      if (ctx->enc_capa & DOES_WEP40)
	{
	  return CIPHER_WEP40;
	}
      else
	{
	  debug_printf(DEBUG_NORMAL, "The AP has requested that we use WEP40"
		       " for the group cipher.  But, your card reports that"
		       " it doesn't support WEP40.  If you are sure that your"
		       " card supports WEP40, you should add "
		       "'wpa_group_cipher = WEP40' to your network clause for"
		       " this network.\n");
	  return -1;
	}
    }

  if (desired_group == CIPHER_WEP104)
    {
      if (ctx->enc_capa & DOES_WEP104)
	{
	  return CIPHER_WEP104;
	}
      else
	{
	  debug_printf(DEBUG_NORMAL, "The AP has requested that we use WEP104"
		       " for the group cipher.  But, your card reports that"
		       " it doesn't support WEP104.  If you are sure that your"
		       " card supports WEP104, you should add "
		       "'wpa_group_cipher = WEP104' to your network clause for"
		       " this network.\n");
	  return -1;
	}
    }

  if (desired_group == CIPHER_TKIP) 
    {
      if (ctx->enc_capa & DOES_TKIP)
	{
	  return CIPHER_TKIP;
	}
      else
	{
	  debug_printf(DEBUG_NORMAL, "The AP has requested that we use TKIP"
		       " for the group cipher.  But, your card reports that"
		       " it doesn't support TKIP.  If you are sure that your"
		       " card supports TKIP, you should add "
		       "'wpa_group_cipher = TKIP' to your network clause for"
		       " this network.\n");
	  return -1;
	}
    }

  if (desired_group == CIPHER_CCMP)
    {
      if (ctx->enc_capa & DOES_CCMP)
	{
	  return CIPHER_CCMP;
	} 
      else 
	{
	  debug_printf(DEBUG_NORMAL, "The AP has requested that we use CCMP"
		       " for the group cipher.  But, your card reports that"
		       " it doesn't support CCMP.  If you are sure that your"
		       " card supports CCMP, you should add "
		       "'wpa_group_cipher = CCMP' to your network clause for"
		       " this network.\n");
	  return -1;
	}
    }

  // If the desired group cipher is set to 0, then it means that we should
  // use the same cipher as the pairwise cipher.
  if (desired_group == 0)
    {
      return wpa2_get_pairwise_crypt(ctx);
    }

  // If we get here, then the AP has requested a cipher type we don't 
  // understand.
  debug_printf(DEBUG_NORMAL, "The AP has requested a cipher type that we "
	       "don't understand.  Type %d.\n", desired_group);
  return -1;
}

/**************************************************************************
 *
 * Generate the IE needed to associate correctly to a WPA2 network.
 *
 **************************************************************************/
void wpa2_gen_ie(struct interface_data *thisint, char *iedata, int *ielen)
{
  struct config_network *network_data;
  struct config_globals *globals;

  if (thisint == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface data in %s!\n",
		   __FUNCTION__);
      return;
    }

  if (iedata == NULL)
    {
      debug_printf(DEBUG_NORMAL, "No space to put the IE! (%s)\n",
		   __FUNCTION__);
      return;
    }

  if (ielen == NULL)
    {
      debug_printf(DEBUG_NORMAL, "No space to put the IE length! (%s)\n",
		   __FUNCTION__);
      return;
    }

  network_data = config_get_network_config();
  
  if (network_data == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid network configuration struct! "
		   "(%s:%d)\n", __FUNCTION__, __LINE__);
      return;
    }

  globals = config_get_globals();

  if (globals == NULL)
    {
      debug_printf(DEBUG_NORMAL, "No valid configuration globals available in"
		   " %s!\n", __FUNCTION__);
      return;
    }

  // The first byte is the Element ID for WPA2, which is 0x30.
  iedata[0] = WPA2_EID;

  // A simple IE with capabilities should be 20 bytes long.
  iedata[1] = 20;

  // Set the version #
  iedata[2] = 0x01;
  iedata[3] = 0x00;

  // The group key cipher suite.
  memcpy(&iedata[4], wpa2oui, 3);

  if ((iedata[7] = wpa2_get_group_crypt(thisint)) == -1)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't build WPA2 IE!  (Error getting "
		   "group cipher information.\n");
      *ielen = 0;
      return;
    }

  debug_printf(DEBUG_CONFIG, "Using Group Cipher Suite : ");
  wpa_print_cipher_suite(DEBUG_CONFIG, iedata[7]);

  // We can only support 1 pairwise cipher suite!
  iedata[8] = 0x01;
  iedata[9] = 0x00;

  // The pairwise cipher suite.
  memcpy(&iedata[10], wpa2oui, 3);
  
  if ((iedata[13] = wpa2_get_pairwise_crypt(thisint)) == -1)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't build WPA2 IE!  (Error getting "
		   "pairwise cipher information.)\n");
      *ielen = 0;
      return;
    }

  debug_printf(DEBUG_CONFIG, "Using Pairwise Cipher Suite : ");
  wpa_print_cipher_suite(DEBUG_CONFIG, iedata[13]);

  if ((network_data->wpa_group_crypt == CIPHER_TKIP) &&
      ((network_data->wpa_pairwise_crypt == CIPHER_WRAP) ||
       (network_data->wpa_pairwise_crypt == CIPHER_CCMP)))
    {
      if (!TEST_FLAG(globals->flags, CONFIG_GLOBALS_NO_FRIENDLY_WARNINGS))
	{
	  debug_printf(DEBUG_NORMAL, "WARNING : Group cipher is TKIP and "
		       "pairwise cipher is using AES.  Many wireless cards "
		       "have problems with this combination!\n");
	}
    }

  if (((network_data->wpa_group_crypt == CIPHER_WEP40) ||
       (network_data->wpa_group_crypt == CIPHER_WEP104)) &&
      ((network_data->wpa_pairwise_crypt == CIPHER_WRAP) ||
       (network_data->wpa_pairwise_crypt == CIPHER_CCMP)))
    {
      if (!TEST_FLAG(globals->flags, CONFIG_GLOBALS_NO_FRIENDLY_WARNINGS))
	{
	  debug_printf(DEBUG_NORMAL, "WARNING : Group cipher is WEP and "
		       "pairwise cipher is using AES.  Many wireless cards "
		       "have problems with this combination!\n");
	}
    }

  // For the authenticated key management suite, we can also only have 1.
  iedata[14] = 0x01;
  iedata[15] = 0x00;

  // The authenticated key management suite.
  memcpy(&iedata[16], wpa2oui, 3);

  if (network_data->methods->method_num == WPA_PSK)
    {
      iedata[19] = 2;  // PSK
      debug_printf(DEBUG_CONFIG, "Using PSK.\n");
    } else {
      iedata[19] = 1;  // 802.1X
      debug_printf(DEBUG_CONFIG, "Using 802.1X\n");
    }

  // We don't support capabilities yet.
  iedata[20] = 0x00;
  iedata[21] = 0x00;

  *ielen = 22;
}
 
void wpa2_gen_ie_caps(struct interface_data *thisint, char *iedata)
{
  wpa_gen_ie(thisint, iedata);

  iedata[1] = 24;
  iedata[24] = 0x00;
  iedata[25] = 0x00;
} 
  
/*************************************************************************
 *
 * Parse an information element.  Returns IE length  if it is a valid IE, 
 * and -1 if it isn't.
 *
 *************************************************************************/
int wpa2_parse_ie(char *iedata)
{
  struct wpa2_ie_struct *ie_struct;
  char suite_id[4];
  int i, ieptr;
  uint16_t value16;

  ie_struct = (struct wpa2_ie_struct *)iedata;

  if (ie_struct->wpaid != WPA2_EID)
    {
      debug_printf(DEBUG_NORMAL, "IE is not a valid WPA2/802.11i IE! (Invalid vendor value!)\n");
      return -1;
    }

  debug_printf(DEBUG_INT, "--- WPA2/802.11i Data ---\n");

  debug_printf(DEBUG_INT, "WPA2/802.11i Version : %d\n", 
	       ie_struct->rsn_ver);

  if (ie_struct->wpalen <= 2)
    {
      debug_printf(DEBUG_NORMAL, "Short IE.  Should assume TKIP/TKIP for "
		   "ciphers!\n");
      return ie_struct->wpalen;
    }

  debug_printf(DEBUG_INT, "Group Key Cipher Suite : ");
  wpa_print_cipher_suite(DEBUG_CONFIG, ie_struct->group_cipher[3]);

  if (ie_struct->wpalen <= 6)
    {
      debug_printf(DEBUG_NORMAL, "Short IE.  Should assume TKIP for "
		   "pairwise cipher.\n");
      return ie_struct->wpalen;
    }

  debug_printf(DEBUG_INT, "Pairwise Key Cipher Suite Count : %d\n",
	       ie_struct->pk_suite_cnt);

  ieptr = sizeof(struct wpa2_ie_struct);

  for (i=0;i<ie_struct->pk_suite_cnt;i++)
    {
      if (ie_struct->wpalen < (ieptr-2) + 4)
	{
	  debug_printf(DEBUG_NORMAL, "Invalid IE!  The length specified by the"
		       " IE isn't long enough to cover the number of "
		       "pairwise ciphers the IE claims it lists!\n");
	  return -1;
	}

      debug_printf(DEBUG_INT, "Cipher Suite : ");
      memcpy((char *)&suite_id, (char *)&iedata[ieptr], 4);

      if (memcmp(suite_id, wpa2oui, 3) != 0)
	{
	  debug_printf(DEBUG_NORMAL, "Proprietary\n");
	} else {
	  wpa_print_cipher_suite(DEBUG_CONFIG, suite_id[3]);
	}

      ieptr+=4;
    }

  if (ie_struct->wpalen < (ieptr - 2) + 2)
    {
      debug_printf(DEBUG_NORMAL, "Short IE.  Should assume an AKM of EAP!\n");
      return ie_struct->wpalen;
    }

  memcpy((char *)&value16, (char *)&iedata[ieptr], 2);
  ieptr+=2;
  debug_printf(DEBUG_INT, "Authenticated Key Management Suite Count : %d\n",
	       value16);

  for (i=0;i<value16;i++)
    {
      if (ie_struct->wpalen < (ieptr-2) + 4)
	{
	  debug_printf(DEBUG_NORMAL, "Invalid IE!  The length claimed by the "
		       "IE isn't long enough to cover the number of "
		       "Authenticated Key Management suites it claims!\n");
	  return -1;
	}

      debug_printf(DEBUG_INT, "Authentication Suite : ");
      memcpy((char *)&suite_id, (char *)&iedata[ieptr], 4);

      if (memcmp(suite_id, wpa2oui, 3) != 0)
	{
	  debug_printf(DEBUG_INT, "Proprietary\n");
	} else {
	  wpa_print_auth_suite(DEBUG_CONFIG, suite_id[3]);
	}
      ieptr+=4;
    }

  if ((ieptr-2) < ie_struct->wpalen)
    {
      memcpy((char *)&value16, (char *)&iedata[ieptr], 2);
      debug_printf(DEBUG_INT, "RSN Capabilities : %04X\n\n", value16);
    }

  return ie_struct->wpalen;
}

