/* dhcp.h
 * Definitions for Dynamic Host Configuration Protocol
 *
 * $Id: dhcp.h,v 1.7 2005/08/02 17:43:40 t0mac Exp $ 
 *
 * Yersinia
 * By David Barroso <tomac@wasahero.org> and Alfredo Andres <slay@wasahero.org>
 * Copyright 2005 Alfredo Andres and David Barroso
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#ifndef __DHCP_H__
#define __DHCP_H__

#include <libnet.h>

#include "terminal-defs.h"
#include "interfaces.h"

#define DHCP_HW_TYPE_E10MB 0x01
#define DHCP_HW_LEN_E10MB  0x06

#define DHCP_CLIENT_PORT 68
#define DHCP_SERVER_PORT 67

#define DHCP_MAX_OPTIONS 100

#define DHCP_TOTAL_FIELDS 23

#define DHCP_SMAC        0
#define DHCP_DMAC        1
#define DHCP_SIP         2
#define DHCP_DIP         3
#define DHCP_SPORT       4
#define DHCP_DPORT       5
#define DHCP_OP          6
#define DHCP_HTYPE       7
#define DHCP_HLEN        8
#define DHCP_HOPS        9
#define DHCP_XID         10
#define DHCP_SECS        11
#define DHCP_FLAGS       12
#define DHCP_CIADDR      13
#define DHCP_YIADDR      14
#define DHCP_SIADDR      15
#define DHCP_GIADDR      16
#define DHCP_CHADDR      17
#define DHCP_TLV         18

#define DHCP_SNAME       20
#define DHCP_FILE        21
#define DHCP_OPTIONS     22
#define DHCP_MSG         23

/* Default values */
#define DHCP_DFL_OPCODE    LIBNET_DHCP_REQUEST
#define DHCP_DFL_HW_TYPE   DHCP_HW_TYPE_E10MB
#define DHCP_DFL_HW_LEN    DHCP_HW_LEN_E10MB
#define DHCP_DFL_MSG       LIBNET_DHCP_MSGDISCOVER
#define DHCP_DFL_HOPS      0
#define DHCP_DFL_SECS      0
#define DHCP_DFL_FLAGS     0x8000

static const struct tuple_type_desc dhcp_type_desc[] = {
    { LIBNET_DHCP_PAD,             "PAD" },           
    { LIBNET_DHCP_SUBNETMASK,      "SUBNETMASK" },     
    { LIBNET_DHCP_TIMEOFFSET,      "TIMEOFFSET" },     
    { LIBNET_DHCP_ROUTER,          "ROUTER" },         
    { LIBNET_DHCP_TIMESERVER,      "TIMESERVER" },    
    { LIBNET_DHCP_NAMESERVER,      "NAMESERVER" },     
    { LIBNET_DHCP_DNS,             "DNS" },            
    { LIBNET_DHCP_LOGSERV,         "LOGSERV" },        
    { LIBNET_DHCP_COOKIESERV,      "COOKIESERV" },     
    { LIBNET_DHCP_LPRSERV,         "LPRSERV" },        
    { LIBNET_DHCP_IMPSERV,         "IMPSERV" },        
    { LIBNET_DHCP_RESSERV,         "RESSERV" },        
    { LIBNET_DHCP_HOSTNAME,        "HOSTNAME" },       
    { LIBNET_DHCP_BOOTFILESIZE,    "BOOTFILESIZE" },   
    { LIBNET_DHCP_DUMPFILE,        "DUMPFILE" },       
    { LIBNET_DHCP_DOMAINNAME,      "DOMAINNAME" },     
    { LIBNET_DHCP_SWAPSERV,        "SWAPSERV" },       
    { LIBNET_DHCP_ROOTPATH,        "ROOTPATH" },       
    { LIBNET_DHCP_EXTENPATH,       "EXTENPATH" },      
    { LIBNET_DHCP_IPFORWARD,       "IPFORWARD" },      
    { LIBNET_DHCP_SRCROUTE,        "SRCROUTE" },       
    { LIBNET_DHCP_POLICYFILTER,    "POLICYFILTER" },   
    { LIBNET_DHCP_MAXASMSIZE,      "MAXASMSIZE" },     
    { LIBNET_DHCP_IPTTL,           "IPTTL" },          
    { LIBNET_DHCP_MTUTIMEOUT,      "MTUTIMEOUT" },     
    { LIBNET_DHCP_MTUTABLE,        "MTUTABLE" },       
    { LIBNET_DHCP_MTUSIZE,         "MTUSIZE" },        
    { LIBNET_DHCP_LOCALSUBNETS,    "LOCALSUBNETS" },   
    { LIBNET_DHCP_BROADCASTADDR,   "BROADCASTADDR" },  
    { LIBNET_DHCP_DOMASKDISCOV,    "DOMASKDISCOV" },   
    { LIBNET_DHCP_MASKSUPPLY,      "MASKSUPPLY" },     
    { LIBNET_DHCP_DOROUTEDISC,     "DOROUTEDISC" },    
    { LIBNET_DHCP_ROUTERSOLICIT,   "ROUTERSOLICIT" },  
    { LIBNET_DHCP_STATICROUTE,     "STATICROUTE" },    
    { LIBNET_DHCP_TRAILERENCAP,    "TRAILERENCAP" },   
    { LIBNET_DHCP_ARPTIMEOUT,      "ARPTIMEOUT" },     
    { LIBNET_DHCP_ETHERENCAP,      "ETHERENCAP" },    
    { LIBNET_DHCP_TCPTTL,          "TCPTTL" },        
    { LIBNET_DHCP_TCPKEEPALIVE,    "TCPKEEPALIVE" },  
    { LIBNET_DHCP_TCPALIVEGARBAGE, "TCPALIVEGARBAGE" },
    { LIBNET_DHCP_NISDOMAIN,       "NISDOMAIN" },      
    { LIBNET_DHCP_NISSERVERS,      "NISSERVERS" },    
    { LIBNET_DHCP_NISTIMESERV,     "NISTIMESERV" },   
    { LIBNET_DHCP_VENDSPECIFIC,    "VENDSPECIFIC" },  
    { LIBNET_DHCP_NBNS,            "NBNS" },          
    { LIBNET_DHCP_NBDD,            "NBDD" },          
    { LIBNET_DHCP_NBTCPIP,         "NBTCPIP" },       
    { LIBNET_DHCP_NBTCPSCOPE,      "NBTCPSCOPE" },    
    { LIBNET_DHCP_XFONT,           "XFONT" },         
    { LIBNET_DHCP_XDISPLAYMGR,     "XDISPLAYMGR" },   
    { LIBNET_DHCP_DISCOVERADDR,    "DISCOVERADDR" },  
    { LIBNET_DHCP_LEASETIME,       "LEASETIME" },     
    { LIBNET_DHCP_OPTIONOVERLOAD,  "OPTIONOVERLOAD" },
    { LIBNET_DHCP_MESSAGETYPE,     "MESSAGETYPE" },   
    { LIBNET_DHCP_SERVIDENT,       "SERVIDENT" },     
    { LIBNET_DHCP_PARAMREQUEST,    "PARAMREQUEST" },  
    { LIBNET_DHCP_MESSAGE,         "MESSAGE" },       
    { LIBNET_DHCP_MAXMSGSIZE,      "MAXMSGSIZE" },    
    { LIBNET_DHCP_RENEWTIME,       "RENEWTIME" },     
    { LIBNET_DHCP_REBINDTIME,      "REBINDTIME" },    
    { LIBNET_DHCP_CLASSSID,        "CLASSSID" },      
    { LIBNET_DHCP_CLIENTID,        "CLIENTID" },      
    { LIBNET_DHCP_NISPLUSDOMAIN,   "NISPLUSDOMAIN" }, 
    { LIBNET_DHCP_NISPLUSSERVERS,  "NISPLUSSERVERS" }, 
    { LIBNET_DHCP_MOBILEIPAGENT,   "MOBILEIPAGENT" }, 
    { LIBNET_DHCP_SMTPSERVER,      "SMTPSERVER" },    
    { LIBNET_DHCP_POP3SERVER,      "POP3SERVER" },    
    { LIBNET_DHCP_NNTPSERVER,      "NNTPSERVER" },    
    { LIBNET_DHCP_WWWSERVER,       "WWWSERVER" },     
    { LIBNET_DHCP_FINGERSERVER,    "FINGERSERVER" },  
    { LIBNET_DHCP_IRCSERVER,       "IRCSERVER" },     
    { LIBNET_DHCP_STSERVER,        "STSERVER" },      
    { LIBNET_DHCP_STDASERVER,      "STDASERVER" },    
    { LIBNET_DHCP_END,             "END" },
    { 0, NULL }
};

static struct attack_param dhcp_tlv[] = {
    { NULL, "SUBNETMASK",               4, 0,   0, FIELD_IP, 15, NULL },
    { NULL, "ROUTER",                   4, 0,   0, FIELD_IP, 15, NULL },
    { NULL, "DNS",                      4, 0,   0, FIELD_IP, 15, NULL },
    { NULL, "HOSTNAME",                 15, 0,  0, FIELD_STR, 15, NULL },
    { NULL, "DOMAINNAME",               15, 0,  0, FIELD_STR, 15, NULL },
    { NULL, "DISCOVERADDR",             4, 0,   0, FIELD_IP, 15, NULL },
    { NULL, "LEASETIME",                4, 0,   0xFFFFFFFF, FIELD_HEX, 8, NULL },
    { NULL, "MESSAGETYPE",              1, 0,   0xFF, FIELD_HEX, 2, NULL },
    { NULL, "SERVIDENT",                4, 0,   0, FIELD_IP, 15, NULL },
    { NULL, "MESSAGE",                 15, 0,   0, FIELD_STR, 15, NULL },
    { NULL, "RENEWTIME",                4, 0,   0xFFFFFFFF, FIELD_HEX, 8, NULL },
    { NULL, "REBINDTIME",               4, 0,   0xFFFFFFFF, FIELD_HEX, 8, NULL },
    { NULL, "CLASSID",                 15, 0,  0, FIELD_STR, 15, NULL },
    { NULL, "END",                      3, 0,  0, FIELD_STR, 3, NULL },
};

static const struct tuple_type_desc dhcp_opcode[] = {
    { LIBNET_DHCP_REQUEST, "(REQUEST)" },
    { LIBNET_DHCP_REPLY, "(REPLY)" },
    { 0, NULL }
};

static const struct tuple_type_desc dhcp_message[] = {
    { LIBNET_DHCP_MSGDISCOVER, "(DISCOVER)" },
    { LIBNET_DHCP_MSGOFFER,    "(OFFER)" },
    { LIBNET_DHCP_MSGREQUEST,  "(REQUEST)" },
    { LIBNET_DHCP_MSGDECLINE,  "(DECLINE)" },
    { LIBNET_DHCP_MSGACK,      "(ACK)" },     
    { LIBNET_DHCP_MSGNACK,     "(NACK)" },
    { LIBNET_DHCP_MSGRELEASE,  "(RELEASE)" },
    { LIBNET_DHCP_MSGINFORM,   "(INFORM)" },
    { 0,                       NULL }
};

static const struct tuple_type_desc dhcp_htype[] = {
    { DHCP_HW_TYPE_E10MB, "(E10MB)" },
    { 0,                  NULL }
};

static const struct tuple_type_desc dhcp_port[] = {
    { DHCP_CLIENT_PORT, "(CLIENT)" },
    { DHCP_SERVER_PORT, "(SERVER)" },
    { 0, NULL}
};

static struct proto_features dhcp_features[] = {
    { F_UDP_PORT, DHCP_CLIENT_PORT},
    { F_UDP_PORT, DHCP_SERVER_PORT},
    { -1, 0 }
};

/* DHCP stuff */
struct dhcp_data { /* DHCP and Ethernet fields*/
    u_int8_t op;
    u_int8_t htype;
    u_int8_t hlen;
    u_int8_t hops;
    u_int32_t xid;
    u_int16_t secs;
    u_int16_t flags;
    u_int32_t ciaddr;
    u_int32_t yiaddr;
    u_int32_t siaddr;
    u_int32_t giaddr;
    u_int8_t chaddr[ETHER_ADDR_LEN];
    char *sname; /* max 64 bytes */
    char *file; /* max 128 bytes */
    /* specific options */
    u_int8_t options[MAX_TLV*MAX_VALUE_LENGTH];
    u_int8_t options_len;
    /* UDP Data */
    u_int16_t sport;
    u_int16_t dport;
    /* IP Data */
    u_int32_t sip;
    u_int32_t dip;
    /* Ethernet Data */
    u_int8_t mac_source[ETHER_ADDR_LEN];
    u_int8_t mac_dest[ETHER_ADDR_LEN];
};

static const struct tuple_type_desc dhcp_tlv_desc[] = {
    { 0,              NULL }
};

static struct position_fields dhcp_fields[] = {
   { DHCP_SMAC,   "Source MAC",      1,  1, 12, 17, 0, 0, FIELD_MAC, FIELD_FIRST, NULL},
   { DHCP_DMAC,   "Destination MAC", 1, 46, 62, 17, 0, 0, FIELD_MAC, FIELD_NORMAL, NULL},
   { DHCP_SIP,    "SIP",             2,  1,  5, 15, 1, 2, FIELD_IP, FIELD_NORMAL, NULL},
   { DHCP_DIP,    "DIP",             2, 22, 26, 15, 1, 18, FIELD_IP, FIELD_NORMAL, NULL},
   { DHCP_SPORT,  "SPort",           2, 43, 49, 5 , 0, 0, FIELD_DEC, FIELD_NORMAL, dhcp_port},
   { DHCP_DPORT,  "DPort",           2, 56, 62, 5 , 0, 0, FIELD_DEC, FIELD_NORMAL, dhcp_port},
   { DHCP_OP,     "Op",              3,  1,  4, 2 , 0, 0, FIELD_HEX, FIELD_NORMAL, dhcp_opcode},
   { DHCP_HTYPE,  "Htype",           3,  7, 13, 2 , 0, 0, FIELD_HEX, FIELD_NORMAL, dhcp_htype},
   { DHCP_HLEN,   "Hlen",            3, 16, 21, 2 , 0, 0, FIELD_HEX, FIELD_NORMAL, NULL},
   { DHCP_HOPS,   "Hops",            3, 24, 29, 2 , 0, 0, FIELD_HEX, FIELD_NORMAL, NULL},
   { DHCP_XID,    "Xid",             3, 32, 36, 8 , 0, 0, FIELD_HEX, FIELD_NORMAL, NULL},
   { DHCP_SECS,   "Secs",            3, 45, 50, 4 , 0, 0, FIELD_HEX, FIELD_NORMAL, NULL},
   { DHCP_FLAGS,  "Flags",           3, 55, 61, 4 , 0, 0, FIELD_HEX, FIELD_NORMAL, NULL},
   { DHCP_CIADDR, "CI",              4,  1,  4, 15, 0, 0, FIELD_IP, FIELD_NORMAL, NULL},
   { DHCP_YIADDR, "YI",              4, 21, 24, 15, 0, 0, FIELD_IP, FIELD_NORMAL, NULL},
   { DHCP_SIADDR, "SI",              4, 41, 44, 15, 0, 0, FIELD_IP, FIELD_NORMAL, NULL},
   { DHCP_GIADDR, "GI",              4, 61, 64, 15, 0, 0, FIELD_IP, FIELD_NORMAL, NULL},
   { DHCP_CHADDR, "CH",              5,  1,  4, 17, 0, 0, FIELD_MAC, FIELD_NORMAL, NULL},
   { DHCP_TLV,    "TLV",             5, 23, 23, MAX_TLV*MAX_VALUE_LENGTH, 0, 0, FIELD_TLV, FIELD_LAST,  dhcp_tlv_desc},
   /* TLV values that appear in the main window */
   { DHCP_MSG,    "MESSAGETYPE",     0,  0,  0,  0, 1, 40,FIELD_TLV, FIELD_NORMAL, dhcp_message},
   { 0,           NULL,              0,  0,  0, 0 , 0, 0, FIELD_NONE, FIELD_NORMAL, NULL}
};

#define DHCP_ATTACK_SEND_RAW           0
#define DHCP_ATTACK_SEND_DISCOVER      1
#define DHCP_ATTACK_DOS_SEND_DISCOVER  2
#define DHCP_ATTACK_SEND_OFFER         3
#define DHCP_ATTACK_SEND_REQUEST       4
#define DHCP_ATTACK_SEND_DECLINE       5 
#define DHCP_ATTACK_SEND_INFORM        6
#define DHCP_ATTACK_ROGUE_SERVER       7
#define DHCP_ATTACK_DOS_SEND_RELEASE   8

void   dhcp_th_send_raw(void *);
void   dhcp_th_send_raw_exit(struct attacks *);
void   dhcp_th_send_discover(void *);
void   dhcp_th_send_discover_exit(struct attacks *);
void   dhcp_th_send_inform(void *);
void   dhcp_th_send_inform_exit(struct attacks *);
void   dhcp_th_send_offer(void *);
void   dhcp_th_send_offer_exit(struct attacks *);
void   dhcp_th_send_request(void *);
void   dhcp_th_send_request_exit(struct attacks *);
void   dhcp_th_send_decline(void *);
void   dhcp_th_send_decline_exit(struct attacks *);
void   dhcp_th_dos_send_discover(void *);
void   dhcp_th_dos_send_discover_exit(struct attacks *);
void   dhcp_th_rogue_server(void *);
void   dhcp_th_rogue_server_exit(struct attacks *);
void   dhcp_th_dos_send_release(void *);
void   dhcp_th_dos_send_release_exit(struct attacks *);


#define DHCP_ROGUE_SERVER     0
#define DHCP_ROGUE_START_IP   1 
#define DHCP_ROGUE_END_IP     2 
#define DHCP_ROGUE_LEASE      3
#define DHCP_ROGUE_RENEW      4
#define DHCP_ROGUE_SUBNET     5
#define DHCP_ROGUE_ROUTER     6
#define DHCP_ROGUE_DNS        7
#define DHCP_ROGUE_DOMAIN     8

static struct attack_param dhcp_rogue_server_params[] = {
    { NULL, "Server ID",               4, 0,    0, FIELD_IP,  15, NULL },
    { NULL, "Start IP",                4, 0,    0, FIELD_IP,  15, NULL },
    { NULL, "End IP",                  4, 0,    0, FIELD_IP,  15, NULL },
    { NULL, "Lease Time (secs)",       4, 0, 0xFFFFFFFF, FIELD_HEX,  8, NULL },
    { NULL, "Renew Time (secs)",       4, 0, 0xFFFFFFFF, FIELD_HEX,  8, NULL },
    { NULL, "Subnet Mask",             4, 0,    0, FIELD_IP,  15, NULL },
    { NULL, "Router",                  4, 0,    0, FIELD_IP,  15, NULL },
    { NULL, "DNS Server",              4, 0,    0, FIELD_IP,  15, NULL },
    { NULL, "Domain",                  15, 0,   0, FIELD_STR, 15, NULL }
};

#define DHCP_DOS_SEND_RELEASE_SERVER     0
#define DHCP_DOS_SEND_RELEASE_START_IP   1 
#define DHCP_DOS_SEND_RELEASE_END_IP     2 

static struct attack_param dhcp_dos_send_release_params[] = {
    { NULL, "Server ID",               4, 0,    0, FIELD_IP,  15, NULL },
    { NULL, "Start IP",                4, 0,    0, FIELD_IP,  15, NULL },
    { NULL, "End IP",                  4, 0,    0, FIELD_IP,  15, NULL }
};


static struct attack dhcp_attack[] = {
  { DHCP_ATTACK_SEND_RAW,          "sending RAW packet",        NONDOS, SINGLE,    dhcp_th_send_raw,      NULL, 0 },
//  { DHCP_ATTACK_SEND_DISCOVER,     "sending DISCOVER packet",   NONDOS, dhcp_th_send_discover, NULL, 0 },
  { DHCP_ATTACK_DOS_SEND_DISCOVER, "sending DISCOVER packet",   DOS,    CONTINOUS, dhcp_th_dos_send_discover,NULL, 0 },
//  { DHCP_ATTACK_SEND_OFFER,        "sending OFFER packet",      NONDOS, dhcp_th_send_offer,    NULL, 0 },
//  { DHCP_ATTACK_SEND_REQUEST,      "sending REQUEST packet",    NONDOS, dhcp_th_send_request,  NULL, 0 },
//  { DHCP_ATTACK_SEND_DECLINE,      "sending DECLINE packet",    NONDOS, dhcp_th_send_decline,  NULL, 0 },
//  { DHCP_ATTACK_SEND_INFORM,       "sending INFORM packet",     NONDOS, dhcp_th_send_inform,   NULL, 0 },
  { DHCP_ATTACK_ROGUE_SERVER,      "creating DHCP rogue server",NONDOS, CONTINOUS, dhcp_th_rogue_server,  dhcp_rogue_server_params,
                                                SIZE_ARRAY(dhcp_rogue_server_params) },
  { DHCP_ATTACK_DOS_SEND_RELEASE,  "sending RELEASE packet",    DOS,    CONTINOUS, dhcp_th_dos_send_release, dhcp_dos_send_release_params,
                                                SIZE_ARRAY(dhcp_dos_send_release_params) },
  { 0,                   NULL,              0,      0,    NULL, NULL, 0     }
};


void   dhcp_register(void);
int8_t dhcp_send_discover(struct attacks *);
int8_t dhcp_send_inform(struct attacks *);
int8_t dhcp_send_offer(struct attacks *);
int8_t dhcp_send_request(struct attacks *);
int8_t dhcp_send_release(struct attacks *, u_int32_t, u_int32_t, u_int8_t *, u_int8_t *);
int8_t dhcp_send_decline(struct attacks *);
int8_t dhcp_send_packet(struct attacks *);
int8_t dhcp_learn_offer(struct attacks *);
int8_t dhcp_load_values(struct pcap_data *, void *);
char **dhcp_get_printable_packet(struct pcap_data *);
char **dhcp_get_printable_store(struct term_node *);
char *dhcp_get_info(u_int8_t, char *);
int8_t dhcp_update_field(int8_t, struct term_node *, void *);
char  *dhcp_get_type_info(u_int16_t);
int8_t dhcp_init_attribs(struct term_node *);
int8_t dhcp_update_data(int8_t, int8_t, int8_t, struct term_node *);
int8_t dhcp_edit_tlv(struct term_node *, u_int8_t, u_int8_t, u_int16_t, u_int8_t *);
void   dhcp_help(void);
int8_t dhcp_parser( struct term_node *, int8_t, char **, void *);
int8_t dhcp_send_arp_request(struct attacks *, u_int32_t);
int8_t dhcp_learn_mac(struct attacks *, u_int32_t, u_int8_t *);

#ifndef HAVE_USLEEP
extern int usleep(unsigned long);
#endif

extern void   thread_libnet_error( char *, libnet_t *);
extern int8_t vrfy_bridge_id( char *, u_int8_t * );
extern int8_t parser_get_formated_inet_address(u_int32_t, char *, u_int16_t);
extern int8_t thread_create(pthread_t *, void *, void *);
extern void   write_log( u_int16_t mode, char *msg, ... );
extern int8_t attack_th_exit(struct attacks *);
extern void   attack_gen_mac(u_int8_t *);
extern int8_t interfaces_get_packet(u_int8_t *, struct pcap_pkthdr *, u_int8_t *, u_int16_t, time_t);
extern int8_t parser_vrfy_mac(char *, u_int8_t *);
extern int8_t parser_get_inet_aton(char *, struct in_addr *);

extern int8_t parser_command2index(register const struct attack *, register int8_t);
extern struct interface_data interfaces[MAX_INTERFACES];
extern struct terminals *terms;

extern int8_t bin_data[];
#endif
/* vim:set tabstop=4:set expandtab:set shiftwidth=4:set textwidth=78: */
