#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "../include/Y2/Y.h"
#include "../include/Y2/Ylib.h"
#include "../include/Y2/Ymixercodes.h"

#include "../include/string.h"


static void print_help(void);


#define MIN(a,b)        ((a) < (b) ? (a) : (b))
#define MAX(a,b)        ((a) > (b) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define ABSOLUTE(x)     (((x) < 0) ? ((x) * -1) : (x))


/*
 *	Prints help message.
 */
static void print_help(void)
{
	printf("\
Usage: ymixer [mixer] [value1] [value2] [value3] [value4] [options]\n\
\n\
    [mixer] specifies the mixer channel device name, running without\n\
    any arguments prints a list of available mixer names.\n\
\n\
    [value#] specify the percent values (from 0.0%%, 100.0%%) to be set.\n\
    If no [value#] is specified then the current value is printed.\n\
\n\
    [options] can be any of the following:\n\
\n\
        --recorder <address:port>    Specify which Y server to connect to.\n\
\n\
    Return values:\n\
\n\
        0       Success.\n\
        1       General error.\n\
        2       Unable to connect to the Y server.\n\
        3       Systems error.\n\
\n"
	);
}


int main(int argc, char *argv[])
{
	int i, n;
	char *strptr;
	const char *con_arg = NULL;

	YConnection *con = NULL;

	int mixer_code = 0;
	int mixer_num = 0;

	Boolean got_value[YMixerValues];
	Coefficient val[YMixerValues];
	char *mixer_name[] = YMixerConicalNames;


	memset(got_value, False, YMixerValues * sizeof(Boolean));


	/* Parse arguments. */
	for(i = 1; i < argc; i++)
	{
	    if(argv[i] == NULL)
		continue;

	    /* Help. */
	    if(strcasepfx(argv[i], "--h") ||
               strcasepfx(argv[i], "-h") ||
               !strcmp(argv[i], "?")
	    )
	    {
		print_help();
		return(0);
	    }
            /* Connect address. */
            else if(strcasepfx(argv[i], "--rec") || 
                    strcasepfx(argv[i], "-rec")
            )
            {
                i++;
                if(i < argc)
                {
                    con_arg = argv[i];
                }
                else
                {
                    fprintf(stderr,
                        "%s: Requires argument.\n",
                        argv[i - 1]
                    );
                }
            }
	    /* Mixer code and index number, check if this is
	     * the first argument.
	     */
	    else if(i == 1)
	    {
		strptr = argv[i];

		/* Go through mixer names list. */
		for(n = 0; n < YTotalMixers; n++)
		{
		    /* Mixer names match? */
		    if(!strcmp(strptr, mixer_name[n]))
		    {
			/* Got match, but remember to add base offset
			 * to get correct mixer code, the mixer code
			 * is different from the mixer number.
			 */
			mixer_code = n + YMixerCodeBaseOffset;
			mixer_num = n;
		    }
		}
	    }
	    /* Subsequent arguments are mixer channel values. */
            /* Value 1. */
            else if(i == 2)
            {
		val[0] = atof(argv[i]);
		got_value[0] = True;
            }
            /* Value 2. */
            else if(i == 3)
            {
                val[1] = atof(argv[i]);
		got_value[1] = True;
            }
            /* Value 3. */
            else if(i == 4)
            {
                val[2] = atof(argv[i]);
                got_value[2] = True;
            }
            /* Value 4. */
            else if(i == 5)
            {
                val[3] = atof(argv[i]);
                got_value[3] = True;
            }
	}

	/* Connect to Y server. */
	con = YOpenConnection(
	    NULL,		/* No start argument. */
	    con_arg
	);
	if(con == NULL)
	{
	    fprintf(stderr, "Unable to connect to the Y server");
	    if(con_arg == NULL)
		con_arg = getenv("RECORDER");
	    if(con_arg == NULL)
		fprintf(stderr, ".\n");
	    else
		fprintf(stderr, ": %s\n", con_arg);
	    return(2);
	}

	/* Check if we should get or set mixer values. If no mixer
	 * values were given on the command line then got_values
	 * will be false.
	 */
	if(got_value[0])
	{
	    /* We have values, so set them. */
	    if(YSetMixerChannelQuad(
		con,
		mixer_code,
		(((YMixerValues > 0) && got_value[0]) ? val[0] : 0.0) / 100.0,
                (((YMixerValues > 1) && got_value[1]) ? val[1] : val[0]) / 100.0,
                (((YMixerValues > 2) && got_value[2]) ? val[2] : val[0]) / 100.0,
                (((YMixerValues > 3) && got_value[3]) ? val[3] : val[0]) / 100.0
	    ))
		fprintf(
		    stderr,
		    "%s channel: Unable to set values.\n",
		    mixer_name[mixer_num]
		);
	}
	else
	{
	    /* No values specified, so print mixer channel value(s). */

	    if(mixer_code == 0)
	    {
		/* Mixer code is 0, that implies that we print all mixer
		 * channel values.
		 */

		/* Go through mixer numbers. */
		for(i = 0; i < YTotalMixers; i++)
		{
		    /* Calculate mixer code, remember i is mixer
		     * number. To get the mixer code we need to add
		     * YMixerCodeBaseOffset to the mixer number.
		     */
		    mixer_code = i + YMixerCodeBaseOffset;

		    /* Set mixer index number. */
		    mixer_num = i;

		    /* Get mixer values for this mixer code. */
		    if(YGetMixerChannelQuad(
			con, mixer_code,
			&val[0], &val[1], &val[2], &val[3]
		    ))
			printf(
			    "%s: Unable to obtain values.\n",
			    mixer_name[mixer_num]
			);
		    else
			printf(
			    "%s: %.1f%% %.1f%% %.1f%% %.1f%%\n",
			    mixer_name[mixer_num],
			    (float)(val[0] * 100), (float)(val[1] * 100),
			    (float)(val[2] * 100), (float)(val[3] * 100)
			);
		}
	    }
	    else
	    {
		/* Print specific mixer channel. */

		/* Sanitize mixer index number. */
		if(mixer_num >= YTotalMixers)
		    mixer_num = YTotalMixers - 1;
		if(mixer_num < 0)
		    mixer_num = 0;

                /* Get mixer values. */
                if(YGetMixerChannelQuad(
                    con, mixer_code,
                    &val[0], &val[1], &val[2], &val[3]
                ))
                    printf(
                        "%s: Unable to obtain values.\n",
                        mixer_name[mixer_num]
                    );
                else
                    printf(
                        "%s: %.1f%% %.1f%% %.1f%% %.1f%%\n",
                        mixer_name[mixer_num],
			(float)(val[0] * 100), (float)(val[1] * 100),
			(float)(val[2] * 100), (float)(val[3] * 100)
                    );
	    }
	}

	/* Disconnect from Y server. */
	YCloseConnection(con, False);
	con = NULL;

	return(0);
}
