#ifndef ZORP_PROXY_TRANSFER2_H_INCLUDED
#define ZORP_PROXY_TRANSFER2_H_INCLUDED

#include <zorp/zorp.h>
#include <zorp/zobject.h>
#include <zorp/proxy.h>
#include <zorp/poll.h>
#include <zorp/log.h>

typedef enum
{
  ZT2_RESULT_FINISHED = 0,    /* success */
  ZT2_RESULT_SUSPENDED = 1,   /* temporary suspend */
  ZT2_RESULT_FAILED = 2,      /* general error, continuation is possible */
  ZT2_RESULT_ABORTED = 3,     /* error, state is unknown, proxy should abort */
} ZTransfer2Result;

#define ZT2F_COMPLETE_COPY        0x0001
#define ZT2F_PROXY_STREAMS_POLLED 0x0002
#define ZT2F_SUSPEND_NOOP	  0x0004

/* transfer status constants */

#define ZT2S_FINISHED  0x0001
#define ZT2S_SUSPENDED 0x0002

#define ZT2S_FAILED    0x0004
#define ZT2S_TIMEDOUT  0x0008
#define ZT2S_ABORTED   0x0010

#define ZT2S_STARTED       0x0020
#define ZT2S_COPYING_TAIL  0x0040
#define ZT2S_EOF_SOURCE    0x0100
#define ZT2S_EOF_DEST      0x0200

#define ZT2S_SOFT_EOF_SOURCE 0x0400
#define ZT2S_SOFT_EOF_DEST   0x0800

#define ZT2S_EOF_BITS (ZT2S_EOF_SOURCE | ZT2S_EOF_DEST | ZT2S_SOFT_EOF_SOURCE | ZT2S_SOFT_EOF_DEST)

#define ZT2E_STACKED     0x02

#define ZT2E_SOURCE      0
#define ZT2E_DEST        1
#define ZT2E_DOWN_SOURCE ZT2E_SOURCE | ZT2E_STACKED
#define ZT2E_DOWN_DEST   ZT2E_DEST | ZT2E_STACKED
#define ZT2E_MAX         3

typedef struct _ZTransfer2Buffer ZTransfer2Buffer;
typedef struct _ZTransfer2 ZTransfer2;

struct _ZTransfer2Buffer
{
  gchar *buf;
  gsize size;
  gsize ifs, ofs, end;
};

struct _ZTransfer2 
{
  ZObject super;
  ZProxy *owner;
  ZPoll *poll;
  ZTransfer2Buffer buffers[2];
  ZStream *endpoints[2];
  ZStreamContext transfer_contexts[2];
  ZStreamContext proxy_contexts[2];
  gsize buffer_size;
  glong timeout, progress_interval;
  guint32 flags;
  
  ZStackedProxy *stacked;
  GSource *timeout_source;
  GSource *progress_source;

  /* internal state */
  guint32 status;
  gint suspend_reason;
};

typedef struct _ZTransfer2Funcs
{
  ZObjectFuncs super;
  GIOStatus (*src_read)(ZTransfer2 *self, ZStream *s, gchar *buf, gsize count, gsize *bytes_read, GError **error);
  GIOStatus (*dst_write)(ZTransfer2 *self, ZStream *s, const gchar *buf, gsize count, gsize *bytes_written, GError **error);
  GIOStatus (*src_shutdown)(ZTransfer2 *self, ZStream *s, GError **error);
  GIOStatus (*dst_shutdown)(ZTransfer2 *self, ZStream *s, GError **error);
  void (*stack_proxy)(ZTransfer2 *self);
  gboolean (*setup)(ZTransfer2 *self);
  ZTransfer2Result (*run)(ZTransfer2 *self);
  gboolean (*progress)(ZTransfer2 *self);
} ZTransfer2Funcs;

extern ZClass ZTransfer2__class;


gboolean z_transfer2_start(ZTransfer2 *self);
void z_transfer2_suspend(ZTransfer2 *self, gint suspend_reason);
gboolean z_transfer2_cancel(ZTransfer2 *self);
void z_transfer2_enable_progress(ZTransfer2 *elf, glong progress_interval);
gboolean z_transfer2_simple_run(ZTransfer2 *self);


ZTransfer2 *
z_transfer2_new(ZClass *class, 
                ZProxy *owner, ZPoll *poll, 
                ZStream *source, ZStream *dest, 
                gsize buffer_size, 
                glong timeout, 
                guint32 flags);

void z_transfer2_free_method(ZObject *s);


static inline gint
z_transfer2_get_suspend_reason(ZTransfer2 *self)
{
  return self->suspend_reason;
}

static inline void
z_transfer2_update_status(ZTransfer2 *self, guint32 status_bit, gint enable)
{
  guint32 old_mask = self->status & ZT2S_EOF_BITS;
  if (enable)
    self->status |= status_bit;
  else
    self->status &= ~status_bit;
  
  /*LOG
    This message reports that the data-transfer to or from some endpoint is closed.
   */
  z_proxy_log(self->owner, CORE_DEBUG, 7, "Eofmask is updated; old_mask='%04x', eof_mask='%04x'", old_mask, self->status & ZT2S_EOF_BITS);
}

static inline guint32
z_transfer2_get_status(ZTransfer2 *self, guint32 status_bit)
{
  return !!(self->status & status_bit);
}

static inline void
z_transfer2_set_stacked_proxy(ZTransfer2 *self, ZStackedProxy *stacked)
{
  g_assert(!z_transfer2_get_status(self, ZT2S_STARTED));

  if (self->stacked)
    z_stacked_proxy_destroy(self->stacked);
  self->stacked = stacked;
}

/* helper functions for virtual methods */
static inline GIOStatus
z_transfer2_src_read(ZTransfer2 *self, ZStream *s, gchar *buf, gsize count, gsize *bytes_read, GError **err)
{
  return Z_FUNCS(self, ZTransfer2)->src_read(self, s, buf, count, bytes_read, err);
}

static inline GIOStatus
z_transfer2_dst_write(ZTransfer2 *self, ZStream *s, gchar *buf, gsize count, gsize *bytes_written, GError **err)
{
  return Z_FUNCS(self, ZTransfer2)->dst_write(self, s, buf, count, bytes_written, err);
}

static inline GIOStatus
z_transfer2_src_shutdown(ZTransfer2 *self, ZStream *s, GError **err)
{
  if (Z_FUNCS(self, ZTransfer2)->src_shutdown)
    return Z_FUNCS(self, ZTransfer2)->src_shutdown(self, s, err);
  else
    return G_IO_STATUS_NORMAL;
}

static inline GIOStatus
z_transfer2_dst_shutdown(ZTransfer2 *self, ZStream *s, GError **err)
{
  if (Z_FUNCS(self, ZTransfer2)->dst_shutdown)
    return Z_FUNCS(self, ZTransfer2)->dst_shutdown(self, s, err);
  else
    return G_IO_STATUS_NORMAL;
}
  
static inline void
z_transfer2_stack_proxy(ZTransfer2 *self)
{
  if (Z_FUNCS(self, ZTransfer2)->stack_proxy)
    Z_FUNCS(self, ZTransfer2)->stack_proxy(self);
}

static inline ZTransfer2Result
z_transfer2_run(ZTransfer2 *self)
{
  return Z_FUNCS(self, ZTransfer2)->run(self);
}

static inline gboolean
z_transfer2_setup(ZTransfer2 *self)
{
  if (Z_FUNCS(self, ZTransfer2)->setup)
    return Z_FUNCS(self, ZTransfer2)->setup(self);
  else
    return TRUE;
}

static inline gboolean
z_transfer2_progress(ZTransfer2 *self)
{
  if (Z_FUNCS(self, ZTransfer2)->progress)
    return Z_FUNCS(self, ZTransfer2)->progress(self);
  else
    return TRUE;
}

#endif
