############################################################################
##
## Copyright (c) 2000, 2001, 2002, 2003, 2004, 2005 BalaBit IT Ltd, Budapest, Hungary
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; if not, write to the Free Software
## Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
##
##
## $Id: Dispatch.py,v 1.29 2004/07/20 17:36:36 bazsi Exp $
##
## Author  : Bazsi
## Auditor : 
## Last audited version: 1.15
## Notes:
##
############################################################################

from Zorp import *
from Session import MasterSession
from Cache import ShiftCache
from traceback import print_exc
from string import atoi
import Zorp
import types, sys

listen_hook = None
unlisten_hook = None
zone_dispatcher_shift_threshold = 1000

ZD_PRI_LISTEN = 100
ZD_PRI_NORMAL = 0
ZD_PRI_RELATED = -100

class AbstractDispatch:
	def __init__(self, session_id, protocol, bindto, kw=None):
		self.session_id = session_id
		self.dispatches = []
		prio = getKeywordArg(kw, 'prio', ZD_PRI_LISTEN)
		if kw == None:
			kw = {}
		if type(bindto) == types.TupleType or type(bindto) == types.ListType:
			for b in bindto:
				self.dispatches.append(Dispatch(self.session_id, protocol, b, prio, self.accepted, kw))
		else:
			self.dispatches.append(Dispatch(self.session_id, protocol, bindto, prio, self.accepted, kw))
			
		Globals.dispatches.append(self)

	def accepted(self):
		"""Function called when a connection is established.
		
		This function is called when a connection is established. 
		It does nothing here, it should be overridden by descendant
		classes.
		
		Arguments
		
		  self -- this instance
		"""
		return Z_REJECT

	def destroy(self):
		"""Stops the listener on the given port

		Calls the destroy method of the low-level object

		Arguments
		
		  self -- this instance
		  
		"""
		for d in self.dispatches:
			d.destroy()


class Dispatcher(AbstractDispatch):
	"""Class encapsulating a Listener, which starts services for established connections.
	
	This class is the starting point of Zorp services. It listens on the
	given port, and when a connection is accepted it starts a session
	and the given service.
                   
	Attributes
	
	  listen --       A Zorp.Listen instance
	  
	  service --      the service to be started
	  
	  bindto --       bind address
	  
	  local --        local address where the listener is bound

	  protocol --     the protocol we were bound to
	"""

	def __init__(self, protocol, bindto, service, kw=None):
		"""Constructor to initialize a Listen instance
		
		Creates the instance, sets the initial attributes, and
		starts the listener

		Arguments

		  self -- this instance
		  
		  bindto --  the address to bind to
		  
		  service -- the service name to start
		  
		  transparent -- TRUE if this is a listener of a 
		                 transparent service, specifying this is
		                 not mandatory but performs additional checks

		"""
		try:
			if service != None:
				self.service = Globals.services[service]
			else:
				self.service = None
		except KeyError:
			raise ServiceException, "Service %s not found" % (service,)
		self.bindto = bindto
		self.transparent = getKeywordArg(kw, 'transparent', FALSE)
		self.protocol = protocol
	        AbstractDispatch.__init__(self, Zorp.firewall_name, protocol, bindto, kw=kw)
	        
	def accepted(self, stream, client_address, client_local, client_listen):
		"""Callback to inform the python layer about incoming connections.
		
		This callback is called by the core when a connection is 
		accepted. Its primary function is to check access control
		(whether the client is permitted to connect to this port),
		and to spawn a new session to handle the connection.
		
		Exceptions raised due to policy violations are handled here.

		Arguments

		  self   --  this instance
		  
		  stream --  the stream of the connection to the client

		  client_address --  the address of the client

		  client_local -- client local address (contains the original destination if transparent)

		  client_listen -- the address where the listener was bound to
		  
		Returns
		
		  TRUE if the connection is accepted
		"""
		if stream == None:
			return None
		session = None
		try:
			session = MasterSession(self.session_id)
			session.setProtocol(self.protocol)
			stream.name = session.session_id
			session.setClient(stream, client_address)
			session.client_local = client_local
			session.client_listen = client_listen
			try:
				if stream.fd != -1:
					session.client_tos = Zorp.getPeerToS(stream.fd)
					Zorp.setOurToS(stream.fd, session.client_tos)
			except IOError:
				pass
			
			service = self.getService(session)
			if not service:
				raise DACException, "No applicable service found"
			session.setService(service)
			
			service.router.routeConnection(session)

			if self.transparent and client_local.ip == client_listen.ip and client_local.port == client_listen.port:
				## LOG ##
				# This message indicates that
				# Listener/Receiver was configured to be
				# accept transparent connections, but it was
				# connected directly.  Configure it either
				# non-transparent or deny direct access to
				# it and set up the appropriate TPROXY rule.
				# @see: Listener
				# @see: Receiver
				##
 				log(session.session_id, CORE_ERROR, 1, "Transparent listener connected directly, dropping connection; local='%s', client_local='%s'", (session.client_listen, session.client_local))
			elif session.isClientPermitted() == Z_ACCEPT:
				## LOG ##
				# This message indicates that a new connection is accepted.
				##
 				log(session.session_id, CORE_DEBUG, 8, "Connection accepted; client_address='%s'", (client_address,))
 				sys.exc_clear()
				return session.service.startInstance(session)
			raise DACException, "This service was not permitted outbound"
		except ZoneException, s:
			## LOG ##
			# This message indicates that no appropriate zone was found for the client address.
			# @see: Zone
			##
 			log(session.session_id, CORE_POLICY, 1, "Zone not found; info='%s'", (s,))
		except DACException, s:
			## LOG ##
			# This message indicates that an DAC policy violation occurred.
			# It is likely that the new connection was not permitted as an outbound_service in the given zone.
			# @see: Zone
			##
 			log(session.session_id, CORE_POLICY, 1, "DAC policy violation; info='%s'", (s,))
		except MACException, s:
			## LOG ##
			# This message indicates that a MAC policy violation occurred.
			##
 			log(session.session_id, CORE_POLICY, 1, "MAC policy violation; info='%s'", (s,))
		except AuthException, s:
			## LOG ##
			# This message indicates that an authentication failure occurred.
			# @see: Auth
			##
 			log(session.session_id, CORE_POLICY, 1, "Authentication failure; info='%s'", (s,))
		except LimitException, s:
			## LOG ##
			# This message indicates that the maximum number of concurrent instance number is reached.
			# Try increase the Service "max_instances" attribute.
			# @see: Service.Service
			##
 			log(session.session_id, CORE_POLICY, 1, "Connection over permitted limits; info='%s'", (s,))
		except LicenseException, s:
			## LOG ##
			# This message indicates that the licensed number of IP address limit is reached, and no new IP address is allowed or an unlicensed component is used.
			# Check your license's "Licensed-Hosts" and "Licensed-Options" options.
			##
			log(session.session_id, CORE_POLICY, 1, "Attempt to use an unlicensed component, or number of licensed hosts exceeded; info='%s'", (s,))
		except:
			print_exc()
			
		if session != None: 
			session.destroy()

		return None

	def getService(self, session):
		"""Returns the service associated with the listener

		Returns the service to start.

		Arguments

		  self    -- this instance
		  
		  session -- session reference
		"""
		return self.service


class ZoneDispatcher(Dispatcher):
	"""Class to listen on the selected address, and start a service based on the client's zone.
	
	This class is similar to a simple Dispatcher, but instead of
	starting a fixed service, it chooses one based on the client
	zone.
	
	It takes a mapping of services indexed by a zone name, with
	an exception of the '*' service, which matches anything.
	
	Attributes
	
	  services -- services mapping indexed by zone name
	"""
	
	def __init__(self, protocol, bindto, services, kw=None):
		"""Constructor to initialize a ZoneDispatcher instance.
		
		This constructor initializes a ZoneDispatcher instance and sets
		its initial attributes based on arguments.
		
		Arguments
		 
		  self -- this instance
		  
		  bindto -- bind to this address
		  
		  services -- a mapping between zone names and services

		  follow_parent -- whether to follow the administrative hieararchy when finding the correct service

		"""
		self.follow_parent = getKeywordArg(kw, 'follow_parent', FALSE)
		Dispatcher.__init__(self, protocol, bindto, None, kw=kw)
		self.services = services
		self.cache = ShiftCache('sdispatch(%s)' % bindto, zone_dispatcher_shift_threshold)
	
	def getService(self, session):
		"""Virtual function which returns the service to be ran
		
		This function is called by our base class to find out the
		service to be used for the current session. It uses the
		client zone name to decide which service to use.
		
		Arguments
		
		  self -- this instance
		  
		  session -- session we are starting
		  
		"""

		cache_ndx = session.client_zone.getName()

		try:
			cached = self.cache.lookup(cache_ndx)
			if not cached:
				## LOG ##
				# This message indicates that no applicable service was found for this client zone in the services cache.
				# It is likely that there is no applicable service configured in this ZoneListener/Receiver at all.
				# Check your ZoneListener/Receiver service configuration.
				# @see: Listener.ZoneListener
				# @see: Receiver.ZoneReceiver
				##
				log(None, CORE_POLICY, 2, "No applicable service found for this client zone (cached); bindto='%s', client_zone='%s'", (self.bindto, session.client_zone))
			return cached
		except KeyError:
			pass

		src_hierarchy = {}
		if self.follow_parent:
			z = session.client_zone
			level = 0
			while z:
				src_hierarchy[z.getName()] = level
				z = z.admin_parent
				level = level + 1
			src_hierarchy['*'] = level
			max_level = level + 1
		else:
			src_hierarchy[session.client_zone.getName()] = 0
			src_hierarchy['*'] = 1
			max_level = 10

		best = None
		for spec in self.services.keys():
			try:
				src_level = src_hierarchy[spec]
			except KeyError:
				src_level = max_level
				
			if not best or 							\
			   (best_src_level > src_level):
				best = self.services[spec]
				best_src_level = src_level

		s = None
		if best_src_level < max_level:
			try:
				s = Globals.services[best]
			except KeyError:
				log(None, CORE_POLICY, 2, "No such service; service='%s'", (best))

		else:
			## LOG ##
			# This message indicates that no applicable service was found for this client zone. 
			# Check your ZoneListener/Receiver service configuration.
			# @see: Listener.ZoneListener
			# @see: Receiver.ZoneReceiver
			##
			log(None, CORE_POLICY, 2, "No applicable service found for this client zone; bindto='%s', client_zone='%s'", (self.bindto, session.client_zone))

		self.cache.store(cache_ndx, s)
		return s

class CSZoneDispatcher(Dispatcher):
	"""Class to listen on the selected address, and start a service based on the client's and the original server zone.
	
	This class is similar to a simple Dispatcher, but instead of
	starting a fixed service, it chooses one based on the client
	and the destined server zone.
	
	It takes a mapping of services indexed by a client and the server
	zone name, with an exception of the '*' zone, which matches
	anything.

	NOTE: the server zone might change during proxy and NAT processing,
	therefore the server zone used here only matches the real
	destination if those phases leave the server address intact.
	
	Attributes
	
	  services -- services mapping indexed by zone names
	"""
	
	def __init__(self, protocol, bindto, services, kw=None):
		"""Constructor to initialize a ZoneDispatcher instance.
		
		This constructor initializes a ZoneDispatcher instance and sets
		its initial attributes based on arguments.
		
		Arguments
		 
		  self -- this instance
		  
		  bindto -- bind to this address
		  
		  services -- a mapping between zone names and services
		  
		  follow_parent -- whether to follow the administrative hieararchy when finding the correct service
		"""
		self.follow_parent = getKeywordArg(kw, 'follow_parent', FALSE)
		Dispatcher.__init__(self, protocol, bindto, None, kw=kw)
		self.services = services
		self.cache = ShiftCache('csdispatch(%s)' % self.bindto, zone_dispatcher_shift_threshold)
	
	def getService(self, session):  
		"""Virtual function which returns the service to be ran

		This function is called by our base class to find out the  
		service to be used for the current session. It uses the
		client and the server zone name to decide which service to
		use.

		Arguments

		  self -- this instance

		  session -- session we are starting

		"""
		from Zone import root_zone
		dest_zone = root_zone.findZone(session.client_local)
		
		cache_ndx = (session.client_zone.getName(), dest_zone.getName())

		try:
			cached = self.cache.lookup(cache_ndx)
			if not cached:
				## LOG ##
				# This message indicates that no applicable service was found for this client zone in the services cache.
				# It is likely that there is no applicable service configured in this CSZoneListener/Receiver at all.
				# Check your CSZoneListener/Receiver service configuration.
				# @see: Listener.CSZoneListener
				# @see: Receiver.CSZoneReceiver
				##
				log(None, CORE_POLICY, 2, "No applicable service found for this client & server zone (cached); bindto='%s', client_zone='%s', server_zone='%s'", (self.bindto, session.client_zone, dest_zone))
			return cached
		except KeyError:
			pass

		src_hierarchy = {}
		dst_hierarchy = {}
		if self.follow_parent:
			z = session.client_zone
			level = 0
			while z:
				src_hierarchy[z.getName()] = level
				z = z.admin_parent
				level = level + 1
			src_hierarchy['*'] = level
			max_level = level + 1
			z = dest_zone
			level = 0
			while z:
				dst_hierarchy[z.getName()] = level
				z = z.admin_parent
				level = level + 1
			dst_hierarchy['*'] = level
			max_level = max(max_level, level + 1)
		else:
			src_hierarchy[session.client_zone.getName()] = 0
			src_hierarchy['*'] = 1
			dst_hierarchy[dest_zone.getName()] = 0
			dst_hierarchy['*'] = 1
			max_level = 10

		best = None
		for spec in self.services.keys():
			try:
				src_level = src_hierarchy[spec[0]]
				dst_level = dst_hierarchy[spec[1]]
			except KeyError:
				src_level = max_level
				dst_level = max_level
				
			if not best or 							\
			   (best_src_level > src_level) or				\
			   (best_src_level == src_level and best_dst_level > dst_level):
				best = self.services[spec]
				best_src_level = src_level
				best_dst_level = dst_level
				
		s = None
		if best_src_level < max_level and best_dst_level < max_level:
			try:
				s = Globals.services[best]
			except KeyError:
				log(None, CORE_POLICY, 2, "No such service; service='%s'", (best))
		else:
			## LOG ##
			# This message indicates that no applicable service was found for this client zone. 
			# Check your CSZoneListener/Receiver service configuration.
			# @see: Listener.CSZoneListener
			# @see: Receiver.CSZoneReceiver
			##
			log(None, CORE_POLICY, 2, "No applicable service found for this client & server zone; bindto='%s', client_zone='%s', server_zone='%s'", (self.bindto, session.client_zone, dest_zone))
		self.cache.store(cache_ndx, s)
		return s

def purgeDispatches():
	for i in Globals.dispatches:
		i.destroy()
	del Globals.dispatches

