/* vi: ts=8 sw=8
 *
 * TI USB Serial Driver
 *
 * Copyright (C) 2004 Texas Instruments
 *
 * This driver is based on the Linux usbserial driver, which is
 * Copyright (C) 1999 - 2002 Greg Kroah-Hartman (greg@kroah.com)
 * Copyright (C) 2000 Peter Berger (pberger@brimson.com)
 * Copyright (C) 2000 Al Borchers (alborchers@steinerpoint.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * For questions or problems with this driver, contact Texas Instruments
 * technical support, or Al Borchers <alborchers@steinerpoint.com>, or
 * Peter Berger <pberger@brimson.com>.
 */

#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/signal.h>
#include <linux/errno.h>
#include <linux/poll.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/fcntl.h>
#include <linux/tty.h>
#include <linux/tty_driver.h>
#include <linux/tty_flip.h>
#include <linux/module.h>
#include <linux/spinlock.h>
#include <linux/list.h>
#include <linux/smp_lock.h>
#include <linux/usb.h>
#include <linux/ctype.h>

extern int debug;

#include "usb-serial.h"
#include "ti_usb_serial.h"


/* Defines */

#define DRIVER_VERSION "v0.9"
#define DRIVER_DESC "TI USB Serial Driver"

#define TI_SERIAL_MAJOR		0	/* dynamically allocated */
#define TI_SERIAL_MINOR_START	0	/* can't easily be changed */
#define TI_SERIAL_DEVICE_COUNT	255


/* Function Declarations */

static int  ti_serial_open(struct tty_struct *tty, struct file *file);
static void ti_serial_close(struct tty_struct *tty, struct file *file);
static void __ti_serial_close(struct usb_serial_port *port,
	struct file *file);
static int  ti_serial_write(struct tty_struct *tty, int from_user,
	const unsigned char *buf, int count);
static int  ti_serial_write_room(struct tty_struct *tty);
static int  ti_serial_chars_in_buffer(struct tty_struct *tty);
static void ti_serial_throttle(struct tty_struct *tty);
static void ti_serial_unthrottle(struct tty_struct *tty);
static int  ti_serial_ioctl(struct tty_struct *tty, struct file *file,
	unsigned int cmd, unsigned long arg);
static void ti_serial_set_termios(struct tty_struct *tty,
	struct termios *old);
static void ti_serial_break(struct tty_struct *tty, int break_state);
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,4,19)
static int ti_serial_read_proc(char *page, char **start, off_t off, \
	int count, int *eof, void *data);
#endif

static void *ti_usb_serial_probe(struct usb_device *dev, unsigned int ifnum,
			       const struct usb_device_id *id);
static void ti_usb_serial_disconnect(struct usb_device *dev, void *ptr);

static void ti_port_softint(void *private);
static int ti_match_serial_number(char *rsv, struct usb_device *dev);
static int ti_match_vendor_product(char *rsv, struct usb_device *dev);
static int ti_match_device_path(char *rsv, struct usb_device *dev);
static int ti_match(char *rsv, struct usb_device *dev, int first);
static struct usb_serial *ti_get_serial_minors(int num_ports, int *minor,
	struct usb_device *dev);
static void ti_put_serial_minors(struct usb_serial *serial);


/* Global Data */

static int ti_serial_refcount;
static struct tty_driver ti_serial_tty_driver;
static struct tty_struct *ti_serial_tty[TI_SERIAL_DEVICE_COUNT];
static struct termios *ti_serial_termios[TI_SERIAL_DEVICE_COUNT];
static struct termios *ti_serial_termios_locked[TI_SERIAL_DEVICE_COUNT];
static struct usb_serial *ti_serial_table[TI_SERIAL_DEVICE_COUNT];
static char *reserve_ports[TI_SERIAL_DEVICE_COUNT];

static int major = TI_SERIAL_MAJOR;

static LIST_HEAD(ti_usb_serial_driver_list);


/* Module Parameters */

MODULE_PARM(major, "i");
MODULE_PARM_DESC(major, "Major number, 0 for dynamic, default is dynamic");

#define usb_serial_cat_str_num_str(a,b,c) _usb_serial_cat_str_num_str(a,b,c)
#define _usb_serial_cat_str_num_str(a,b,c) a #b c
MODULE_PARM(reserve_ports, usb_serial_cat_str_num_str("1-",TI_SERIAL_DEVICE_COUNT,"s"));
MODULE_PARM_DESC(reserve_ports, usb_serial_cat_str_num_str("Reserve tty devices for these serial numbers, vendor/product ids, or device paths, 1-",TI_SERIAL_DEVICE_COUNT," strings"));


/* Structs */

static struct tty_driver ti_serial_tty_driver = {
	.magic =		TTY_DRIVER_MAGIC,
	.driver_name =		"ti_usb",
#ifndef CONFIG_DEVFS_FS
	.name =			"ttyTIUSB",
#else
	.name =			"usb/ti/%d",
#endif
	.major =		TI_SERIAL_MAJOR,
	.minor_start =		TI_SERIAL_MINOR_START,
	.num =			TI_SERIAL_DEVICE_COUNT,
	.type =			TTY_DRIVER_TYPE_SERIAL,
	.subtype =		SERIAL_TYPE_NORMAL,
	.flags =		TTY_DRIVER_REAL_RAW | TTY_DRIVER_NO_DEVFS,

	.refcount =		&ti_serial_refcount,
	.table =		ti_serial_tty,
	.termios =		ti_serial_termios,
	.termios_locked =	ti_serial_termios_locked,

	.open =			ti_serial_open,
	.close =		ti_serial_close,
	.write =		ti_serial_write,
	.write_room =		ti_serial_write_room,
	.ioctl =		ti_serial_ioctl,
	.set_termios =		ti_serial_set_termios,
	.throttle =		ti_serial_throttle,
	.unthrottle =		ti_serial_unthrottle,
	.break_ctl =		ti_serial_break,
	.chars_in_buffer =	ti_serial_chars_in_buffer,
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,4,19)
	.read_proc =		ti_serial_read_proc,
#endif
};

static struct usb_driver ti_usb_serial_driver = {
	.name =		"ti_usb",
	.probe =	ti_usb_serial_probe,
	.disconnect =	ti_usb_serial_disconnect,
	.id_table =	NULL, 			/* check all devices */
};


/* Functions */

extern int __init ti_usb_serial_init(void)
{
	int i;
	int result;

	/* Initalize our global data */
	for (i = 0; i < TI_SERIAL_DEVICE_COUNT; ++i) {
		ti_serial_table[i] = NULL;
	}

	/* register the tty driver */
	ti_serial_tty_driver.init_termios = tty_std_termios;
	ti_serial_tty_driver.init_termios.c_cflag = B9600 | CS8 | CREAD | HUPCL | CLOCAL;
	ti_serial_tty_driver.major = major;
	if (tty_register_driver(&ti_serial_tty_driver) < 0) {
		err("%s - failed to register tty driver", __FUNCTION__);
		return -1;
	}

	/* register the USB driver */
	result = usb_register(&ti_usb_serial_driver);
	if (result < 0) {
		tty_unregister_driver(&ti_serial_tty_driver);
		err("%s - usb_register failed for the usb-serial driver, error %d", __FUNCTION__, result);
		return -1;
	}

	/* info(DRIVER_DESC " " DRIVER_VERSION); */

	return 0;
}


extern void __exit ti_usb_serial_exit(void)
{
	usb_deregister(&ti_usb_serial_driver);
	tty_unregister_driver(&ti_serial_tty_driver);
}


int ti_usb_serial_register(struct usb_serial_device_type *new_device)
{
	/* Add this device to our list of devices */
	list_add(&new_device->driver_list, &ti_usb_serial_driver_list);

	info("TI USB Serial support registered for %s", new_device->name);

	usb_scan_devices();

	return 0;
}


void ti_usb_serial_deregister(struct usb_serial_device_type *device)
{
	struct usb_serial *serial;
	int i;

	info("TI USB Serial deregistering driver %s", device->name);

	/* clear out the ti_serial_table */
	for(i = 0; i < TI_SERIAL_DEVICE_COUNT; ++i) {
		serial = ti_serial_table[i];
		if ((serial != NULL) && (serial->type == device)) {
			usb_driver_release_interface(&ti_usb_serial_driver,
				serial->interface);
			ti_usb_serial_disconnect(NULL, serial);
		}
	}

	list_del(&device->driver_list);
}


static int ti_serial_open(struct tty_struct *tty, struct file *file)
{
	struct usb_serial *serial;
	struct usb_serial_port *port;
	unsigned int portNumber;
	int retval = 0;
	
	dbg("%s", __FUNCTION__);

	/* get the serial object associated with this tty pointer */
	serial = ti_serial_table[MINOR(tty->device)];

	if (serial_paranoia_check(serial, __FUNCTION__)) {
		tty->driver_data = NULL;
		return -ENODEV;
	}

	/* set up the port structure, link the tty driver and port */
	portNumber = MINOR(tty->device) - serial->minor;
	port = &serial->port[portNumber];
	tty->driver_data = port;

	down(&port->sem);
	port->tty = tty;
	 
	/* lock this module before we call it */
	if (serial->type->owner)
		__MOD_INC_USE_COUNT(serial->type->owner);

	++port->open_count;
	if (port->open_count == 1) {
		/* only call the device specific open if this 
		 * is the first time the port is opened */
		if (serial->type->open)
			retval = serial->type->open(port, file);
	}

	if (retval) {
		port->open_count = 0;
		if (serial->type->owner)
			__MOD_DEC_USE_COUNT(serial->type->owner);
	}

	up(&port->sem);
	return retval;
}


static void __ti_serial_close(struct usb_serial_port *port, struct file *file)
{
	if (!port->open_count) {
		dbg ("%s - port not opened", __FUNCTION__);
		return;
	}

	--port->open_count;
	if (port->open_count <= 0) {
		/* only call the device specific close if this 
		 * port is being closed by the last owner */
		if (port->serial->type->close)
			port->serial->type->close(port, file);
		port->open_count = 0;
		if (port->tty) {
			port->tty->driver_data = NULL;
			port->tty = NULL;
		}
	}

	if (port->serial->type->owner)
		__MOD_DEC_USE_COUNT(port->serial->type->owner);
}


static void ti_serial_close(struct tty_struct *tty, struct file *file)
{
	struct usb_serial_port *port = (struct usb_serial_port *)tty->driver_data;
	struct usb_serial *serial = get_usb_serial(port, __FUNCTION__);

	if (!serial)
		return;

	down(&port->sem);

	dbg("%s - port %d", __FUNCTION__, port->number);

	/* if disconnect beat us to the punch here, there's nothing to do */
	if (tty->driver_data) {
		__ti_serial_close(port, file);
	}

	up(&port->sem);
}


static int ti_serial_write(struct tty_struct *tty, int from_user,
	const unsigned char *buf, int count)
{
	struct usb_serial_port *port = (struct usb_serial_port *)tty->driver_data;
	struct usb_serial *serial = get_usb_serial(port, __FUNCTION__);
	int retval = -EINVAL;

	if (!serial)
		return -ENODEV;

	dbg("%s - port %d, %d byte(s)", __FUNCTION__, port->number, count);

	if (!port->open_count) {
		dbg("%s - port not opened", __FUNCTION__);
		return -ENODEV;
	}

	/* pass on to the driver specific version of this function if it is available */
	if (serial->type->write)
		retval = serial->type->write(port, from_user, buf, count);

	return retval;
}


static int ti_serial_write_room(struct tty_struct *tty) 
{
	struct usb_serial_port *port = (struct usb_serial_port *)tty->driver_data;
	struct usb_serial *serial = get_usb_serial(port, __FUNCTION__);
	int retval = -EINVAL;

	if (!serial)
		return -ENODEV;

	dbg("%s - port %d", __FUNCTION__, port->number);

	if (!port->open_count) {
		dbg("%s - port not open", __FUNCTION__);
		return -ENODEV;
	}

	/* pass on to the driver specific version of this function if it is available */
	if (serial->type->write_room)
		retval = serial->type->write_room(port);

	return retval;
}


static int ti_serial_chars_in_buffer(struct tty_struct *tty) 
{
	struct usb_serial_port *port = (struct usb_serial_port *)tty->driver_data;
	struct usb_serial *serial = get_usb_serial(port, __FUNCTION__);
	int retval = -EINVAL;

	if (!serial)
		return -ENODEV;

	dbg("%s = port %d", __FUNCTION__, port->number);

	if (!port->open_count) {
		dbg("%s - port not open", __FUNCTION__);
		return -ENODEV;
	}

	/* pass on to the driver specific version of this function if it is available */
	if (serial->type->chars_in_buffer)
		retval = serial->type->chars_in_buffer(port);

	return retval;
}


static void ti_serial_throttle(struct tty_struct *tty)
{
	struct usb_serial_port *port = (struct usb_serial_port *)tty->driver_data;
	struct usb_serial *serial = get_usb_serial(port, __FUNCTION__);

	if (!serial)
		return;

	dbg("%s - port %d", __FUNCTION__, port->number);

	if (!port->open_count) {
		dbg ("%s - port not open", __FUNCTION__);
		return;
	}

	/* pass on to the driver specific version of this function */
	if (serial->type->throttle)
		serial->type->throttle(port);
}


static void ti_serial_unthrottle(struct tty_struct *tty)
{
	struct usb_serial_port *port = (struct usb_serial_port *)tty->driver_data;
	struct usb_serial *serial = get_usb_serial(port, __FUNCTION__);

	if (!serial)
		return;

	dbg("%s - port %d", __FUNCTION__, port->number);

	if (!port->open_count) {
		dbg("%s - port not open", __FUNCTION__);
		return;
	}

	/* pass on to the driver specific version of this function */
	if (serial->type->unthrottle)
		serial->type->unthrottle(port);
}


static int ti_serial_ioctl(struct tty_struct *tty, struct file *file,
	unsigned int cmd, unsigned long arg)
{
	struct usb_serial_port *port = (struct usb_serial_port *)tty->driver_data;
	struct usb_serial *serial = get_usb_serial(port, __FUNCTION__);
	int retval = -ENODEV;

	if (!serial)
		return -ENODEV;

	dbg("%s - port %d, cmd 0x%.4x", __FUNCTION__, port->number, cmd);

	if (!port->open_count) {
		dbg ("%s - port not open", __FUNCTION__);
		return -ENODEV;
	}

	/* pass on to the driver specific version of this function if it is available */
	if (serial->type->ioctl)
		retval = serial->type->ioctl(port, file, cmd, arg);
	else
		retval = -ENOIOCTLCMD;

	return retval;
}


static void ti_serial_set_termios(struct tty_struct *tty,
	struct termios *old)
{
	struct usb_serial_port *port = (struct usb_serial_port *)tty->driver_data;
	struct usb_serial *serial = get_usb_serial(port, __FUNCTION__);

	if (!serial)
		return;

	dbg("%s - port %d", __FUNCTION__, port->number);

	if (!port->open_count) {
		dbg("%s - port not open", __FUNCTION__);
		return;
	}

	/* pass on to the driver specific version of this function if it is available */
	if (serial->type->set_termios)
		serial->type->set_termios(port, old);
}


static void ti_serial_break(struct tty_struct *tty, int break_state)
{
	struct usb_serial_port *port = (struct usb_serial_port *)tty->driver_data;
	struct usb_serial *serial = get_usb_serial(port, __FUNCTION__);

	if (!serial)
		return;

	dbg("%s - port %d", __FUNCTION__, port->number);

	if (!port->open_count) {
		dbg("%s - port not open", __FUNCTION__);
		return;
	}

	/* pass on to the driver specific version of this function if it is available */
	if (serial->type->break_ctl)
		serial->type->break_ctl(port, break_state);
}


#if LINUX_VERSION_CODE > KERNEL_VERSION(2,4,19)
static int ti_serial_read_proc(char *page, char **start, off_t off,
	int count, int *eof, void *data)
{
	struct usb_serial *serial;
	int length = 0;
	int i;
	off_t begin = 0;
	char tmp[40];

	dbg("%s", __FUNCTION__);
	length += sprintf (page, "tiusbserinfo:1.0 driver:%s\n", DRIVER_VERSION);
	for (i = 0; i < TI_SERIAL_DEVICE_COUNT && length < PAGE_SIZE; ++i) {
		serial = ti_serial_table[i];
		if (serial == NULL)
			continue;

		length += sprintf (page+length, "%d:", i);
		if (serial->type->owner)
			length += sprintf (page+length, " module:%s", serial->type->owner->name);
		length += sprintf (page+length, " name:\"%s\"", serial->type->name);
		length += sprintf (page+length, " vendor:%04x product:%04x", serial->vendor, serial->product);
		length += sprintf (page+length, " num_ports:%d", serial->num_ports);
		length += sprintf (page+length, " port:%d", i - serial->minor + 1);

		usb_make_path(serial->dev, tmp, sizeof(tmp));
		length += sprintf (page+length, " path:%s", tmp);
			
		length += sprintf (page+length, "\n");
		if ((length + begin) > (off + count))
			goto done;
		if ((length + begin) < off) {
			begin += length;
			length = 0;
		}
	}
	*eof = 1;
done:
	if (off >= (length + begin))
		return 0;
	*start = page + (off-begin);
	return ((count < begin+length-off) ? count : begin+length-off);
}
#endif


static void *ti_usb_serial_probe(struct usb_device *dev, unsigned int ifnum,
       const struct usb_device_id *id)
{
	int i, found, minor, max_endpoints;
	struct usb_serial *serial = NULL;
	struct usb_serial_port *port;
	struct usb_interface *interface;
	struct usb_serial_device_type *type = NULL;
	struct list_head *tmp;
	char sn[256];

	/* loop through our list of known serial converters, and see if this
	   device matches. */
	found = 0;
	interface = &dev->actconfig->interface[ifnum];
	list_for_each(tmp, &ti_usb_serial_driver_list) {
		type = list_entry(tmp, struct usb_serial_device_type, driver_list);
		if (usb_match_id(dev, interface, type->id_table)) {
			dbg("%s - descriptor matches", __FUNCTION__);
			found = 1;
			break;
		}
	}
	if (!found) {
		/* no match */
		dbg("%s - none matched", __FUNCTION__);
		return NULL;
	}

	/* found */
	if (dev->descriptor.iSerialNumber == 0)
  		strcpy( sn, "NA" );
	else
  		usb_string(dev, dev->descriptor.iSerialNumber, sn, 256);
	info("detected %s, vendor 0x%04X, product 0x%04X, serial number '%s'", type->name, dev->descriptor.idVendor, dev->descriptor.idProduct, sn);

	/* get serial device */
	serial = ti_get_serial_minors(type->num_ports, &minor, dev);
	if (serial == NULL) {
		err("%s - no more free serial devices", __FUNCTION__);
		return NULL;
	}

	/* set up serial device */
	serial->dev = dev;
	serial->type = type;
	serial->interface = interface;
	serial->minor = minor;
	serial->num_ports = type->num_ports;
	serial->vendor = dev->descriptor.idVendor;
	serial->product = dev->descriptor.idProduct;

	if (ti_usb_serial_alloc_urbs(serial))
		goto probe_error;

	/* initialize some parts of the port structures */
	/* we don't use num_ports here cauz some devices have more endpoint pairs than ports */
	max_endpoints = max(serial->num_bulk_in, serial->num_bulk_out);
	max_endpoints = max(max_endpoints, (int)serial->num_interrupt_in);
	max_endpoints = max(max_endpoints, (int)serial->num_ports);
	dbg("%s - setting up %d port structures for this device", __FUNCTION__, max_endpoints);
	for (i = 0; i < max_endpoints; ++i) {
		port = &serial->port[i];
		port->number = i + serial->minor;
		port->serial = serial;
		port->magic = USB_SERIAL_PORT_MAGIC;
		port->tqueue.routine = ti_port_softint;
		port->tqueue.data = port;
		init_MUTEX (&port->sem);
	}

	/* if this device type has a startup function, call it */
	if (type->startup) {
		i = type->startup(serial);
		if (i < 0)
			goto probe_error;
		if (i > 0)
			return serial;
	}

	/* initialize the devfs nodes for this device and let the user know what ports we are bound to */
	for (i = 0; i < serial->num_ports; ++i) {
		tty_register_devfs(&ti_serial_tty_driver, 0, serial->port[i].number);
		info("%s now attached to ttyTIUSB%d (or usb/ti/%d for devfs)", 
		     type->name, serial->port[i].number, serial->port[i].number);
	}

	return serial; /* success */

probe_error:
	/* free the urbs */
	ti_usb_serial_free_urbs(serial, 0);

	/* return the minor range that this device had */
	ti_put_serial_minors(serial);

	/* free up any memory that we allocated */
	kfree(serial);
	return NULL;
}


static void ti_usb_serial_disconnect(struct usb_device *dev, void *ptr)
{
	struct usb_serial *serial = (struct usb_serial *)ptr;
	struct usb_serial_port *port;
	int i;

	dbg ("%s", __FUNCTION__);
	if (serial) {
		/* fail all future close/read/write/ioctl/etc calls */
		for (i = 0; i < serial->num_ports; ++i) {
			port = &serial->port[i];
			down(&port->sem);
			if (port->tty != NULL)
				while (port->open_count > 0)
					__ti_serial_close(port, NULL);
			up(&port->sem);
		}

		serial->dev = NULL;
		if (serial->type->shutdown)
			serial->type->shutdown(serial);

		for (i = 0; i < serial->num_ports; ++i)
			serial->port[i].open_count = 0;

		ti_usb_serial_free_urbs(serial, 1);

		for (i = 0; i < serial->num_ports; ++i) {
			tty_unregister_devfs(&ti_serial_tty_driver, serial->port[i].number);
			info("%s now disconnected from ttyTIUSB%d", serial->type->name, serial->port[i].number);
		}

		/* return the minor range that this device had */
		ti_put_serial_minors(serial);

		/* free up any memory that we allocated */
		kfree(serial);

	} else {
		info("device disconnected");
	}

}


int ti_usb_serial_alloc_urbs(struct usb_serial *serial)
{
	int i, buffer_size;
	struct usb_device *dev = serial->dev;
	struct usb_serial_port *port;
	struct usb_interface_descriptor *iface_desc;
	struct usb_endpoint_descriptor *endpoint;


	serial->num_bulk_in = 0;
	serial->num_bulk_out = 0;
	serial->num_interrupt_in = 0;

	/* check out the endpoints */
	iface_desc = serial->interface->altsetting;
	for (i = 0; i < iface_desc->bNumEndpoints; ++i) {
		endpoint = &iface_desc->endpoint[i];

		/* bulk in endpoint */
		if ((endpoint->bEndpointAddress & 0x80) &&
		    ((endpoint->bmAttributes & 3) == 0x02)) {
			port = &serial->port[(int)serial->num_bulk_in];
			port->read_urb = usb_alloc_urb(0);
			if (!port->read_urb) {
				err("%s - no free urbs available", __FUNCTION__);
				return -ENOMEM;
			}
			buffer_size = endpoint->wMaxPacketSize;
			port->bulk_in_endpointAddress = endpoint->bEndpointAddress;
			port->bulk_in_buffer = kmalloc(buffer_size, GFP_KERNEL);
			if (!port->bulk_in_buffer) {
				usb_free_urb(port->read_urb);
				err("%s - couldn't allocate bulk_in_buffer",
					__FUNCTION__);
				return -ENOMEM;
			}
			usb_fill_bulk_urb(port->read_urb, dev,
			  usb_rcvbulkpipe(dev, endpoint->bEndpointAddress),
			  port->bulk_in_buffer, buffer_size,
			  serial->type->read_bulk_callback,
			  port);
			++serial->num_bulk_in;
		}

		/* bulk out endpoint */
		if (((endpoint->bEndpointAddress & 0x80) == 0x00) &&
		    ((endpoint->bmAttributes & 3) == 0x02)) {
			port = &serial->port[(int)serial->num_bulk_out];
			port->write_urb = usb_alloc_urb(0);
			if (!port->write_urb) {
				err("%s - no free urbs available", __FUNCTION__);
				return -ENOMEM;
			}
			buffer_size = endpoint->wMaxPacketSize;
			port->bulk_out_size = buffer_size;
			port->bulk_out_endpointAddress = endpoint->bEndpointAddress;
			port->bulk_out_buffer = kmalloc(buffer_size, GFP_KERNEL);
			if (!port->bulk_out_buffer) {
				usb_free_urb(port->write_urb);
				err("%s - couldn't allocate bulk_out_buffer",
					__FUNCTION__);
				return -ENOMEM;
			}
			usb_fill_bulk_urb(port->write_urb, dev,
			  usb_sndbulkpipe(dev, endpoint->bEndpointAddress),
			  port->bulk_out_buffer, buffer_size, 
			  serial->type->write_bulk_callback,
			  port);
			++serial->num_bulk_out;
		}
		
		if ((endpoint->bEndpointAddress & 0x80) &&
		    ((endpoint->bmAttributes & 3) == 0x03)) {
			port = &serial->port[(int)serial->num_interrupt_in];
			port->interrupt_in_urb = usb_alloc_urb(0);
			if (!port->interrupt_in_urb) {
				err("%s - no free urbs available", __FUNCTION__);
				return -ENOMEM;
			}
			buffer_size = endpoint->wMaxPacketSize;
			port->interrupt_in_endpointAddress = endpoint->bEndpointAddress;
			port->interrupt_in_buffer = kmalloc(buffer_size, GFP_KERNEL);
			if (!port->interrupt_in_buffer) {
				usb_free_urb(port->interrupt_in_urb);
				err("%s - couldn't allocate interrupt_in_buffer",
					__FUNCTION__);
				return -ENOMEM;
			}
			usb_fill_int_urb(port->interrupt_in_urb, dev, 
			  usb_rcvintpipe(dev, endpoint->bEndpointAddress),
			  port->interrupt_in_buffer, buffer_size, 
			  serial->type->read_int_callback,
			  port, endpoint->bInterval);
			++serial->num_interrupt_in;
		}
	}

	dbg("%s - found %d bulk in, %d bulk out, and %d interrupt in endpoints", __FUNCTION__, serial->num_bulk_in, serial->num_bulk_out, serial->num_interrupt_in);

	return 0;
}


void ti_usb_serial_free_urbs(struct usb_serial *serial, int unlink)
{
	int i;
	struct usb_serial_port *port;


	for (i = 0; i < serial->num_bulk_in; ++i) {
		port = &serial->port[i];
		if (port->read_urb) {
			if (unlink)
				usb_unlink_urb(port->read_urb);
			usb_free_urb(port->read_urb);
		}
		if (port->bulk_in_buffer)
			kfree(port->bulk_in_buffer);
	}
	for (i = 0; i < serial->num_bulk_out; ++i) {
		port = &serial->port[i];
		if (port->write_urb) {
			if (unlink)
				usb_unlink_urb(port->write_urb);
			usb_free_urb(port->write_urb);
		}
		if (port->bulk_out_buffer)
			kfree(port->bulk_out_buffer);
	}
	for (i = 0; i < serial->num_interrupt_in; ++i) {
		port = &serial->port[i];
		if (port->interrupt_in_urb) {
			if (unlink)
				usb_unlink_urb(port->interrupt_in_urb);
			usb_free_urb(port->interrupt_in_urb);
		}
		if (port->interrupt_in_buffer)
			kfree(port->interrupt_in_buffer);
	}
}


static void ti_port_softint(void *private)
{
	struct usb_serial_port *port = (struct usb_serial_port *)private;
	struct usb_serial *serial = get_usb_serial (port, __FUNCTION__);
	struct tty_struct *tty;

	dbg("%s - port %d", __FUNCTION__, port->number);
	
	if (!serial)
		return;

	tty = port->tty;
	if (!tty)
		return;

	if ((tty->flags & (1 << TTY_DO_WRITE_WAKEUP)) && tty->ldisc.write_wakeup) {
		dbg("%s - write wakeup call.", __FUNCTION__);
		(tty->ldisc.write_wakeup)(tty);
	}

	wake_up_interruptible(&tty->write_wait);
}


static int ti_match_serial_number(char *rsv, struct usb_device *dev)
{
	int len_sn;
	int len_rsv = strlen(rsv);
	char sn[256];

	if (dev->descriptor.iSerialNumber == 0)
		return 0;

	usb_string(dev, dev->descriptor.iSerialNumber, sn, 256);
	len_sn = strlen(sn);

	if (*rsv == '*')
		return (strncmp(sn+len_sn-len_rsv+1, rsv+1, len_rsv-1) == 0);
	else if (len_rsv > 0 && *(rsv+len_rsv-1) == '*')
		return (strncmp(sn, rsv, len_rsv-1) == 0);
	else
		return (strcmp(sn, rsv) == 0);
}


static int ti_match_vendor_product(char *rsv, struct usb_device *dev)
{
	char *p = rsv + 1;

	if (*rsv == '\0'
	|| (*rsv != '*' && simple_strtol(rsv,&p,16) != dev->descriptor.idVendor))
		return 0;

	if (*p != ':' && *p != '/')
		return 0;
	++p;

	if (*p == '\0'
	|| (*p != '*' && simple_strtol(p,NULL,16) != dev->descriptor.idProduct))
		return 0;

	return 1;
}


static int ti_match_device_path(char *rsv, struct usb_device *dev)
{
	int port;
	char *port_str = rsv + strlen(rsv);
	struct usb_device *pdev, *cdev;

	cdev = dev;
	pdev = dev->parent;
	while (pdev) {

		if (port_str <= rsv)
			return 0;

		for (port=0; port < pdev->maxchild; port++)
			if (pdev->children[port] == cdev)
				break;
		++port;		/* change 0 based port numbers to 1 based */

		while (port_str>rsv && *(port_str-1) != '/')
			--port_str;

		if (*port_str != '*' && simple_strtol(port_str,NULL,10) != port)
			return 0;

		--port_str;

		cdev = pdev;
		pdev = pdev->parent;
	}

	while (port_str>rsv && *(port_str-1) != '/')
		--port_str;

	if (*port_str != '*' && simple_strtol(port_str,NULL,10) != dev->bus->busnum)
		return 0;

	if (port_str == rsv || (port_str == rsv+1 && *rsv == '/'))
		return 1;

	return 0;
}


/* return 1 if the device matches the reserve port entry, 0 otherwise */
/* a NULL reserve port entry always matches */
static int ti_match(char *rsv, struct usb_device *dev, int first)
{
	if (rsv == NULL)
		return 1;	/* unspecified, always matches */

	switch (toupper(*rsv)) {
		case 'S': return ti_match_serial_number(rsv+1,dev);
		case 'V': return ti_match_vendor_product(rsv+1,dev);
		case 'D': return ti_match_device_path(rsv+1,dev);
		case '^': return !first;
		case '-': return 0;
		case '*': return 1;
		default: return 0;
	}
}


static struct usb_serial *ti_get_serial_minors(int num_ports, int *minor,
	struct usb_device *dev)
{
	struct usb_serial *serial = NULL;
	int i, j;
	int good_spot;

	dbg("%s %d", __FUNCTION__, num_ports);

	if (num_ports > TI_SERIAL_DEVICE_COUNT)
		return NULL;

	*minor = 0;
	for (i = 0; i < TI_SERIAL_DEVICE_COUNT - num_ports + 1; ++i) {
		if (ti_serial_table[i] || !ti_match(reserve_ports[i],dev,1))
			continue;

		good_spot = 1;
		for (j = 1; j < num_ports; ++j)
			if (ti_serial_table[i+j]
			|| !ti_match(reserve_ports[i+j],dev,0))
				good_spot = 0;
		if (good_spot == 0)
			continue;

		if (!(serial = kmalloc(sizeof(struct usb_serial), GFP_KERNEL))) {
			err("%s - Out of memory", __FUNCTION__);
			return NULL;
		}
		memset(serial, 0, sizeof(struct usb_serial));
		serial->magic = USB_SERIAL_MAGIC;
		*minor = i;
		dbg("%s - minor base = %d", __FUNCTION__, *minor);
		for (j = *minor; j < *minor + num_ports; ++j)
			ti_serial_table[j] = serial;
		return serial;
	}
	return NULL;
}


static void ti_put_serial_minors(struct usb_serial *serial)
{
	int i;

	dbg("%s", __FUNCTION__);

	if (serial == NULL)
		return;

	for (i = 0; i < serial->num_ports; ++i) {
		ti_serial_table[serial->minor + i] = NULL;
	}

	return;
}
