/* dp83815.c -- National DP83815 PCI Ethernet Controller Driver */

static const char *version =
    "dp83815.c:v1.51 National DP83815 Fast Ethernet Driver\n";

/*    
1. INTRODUCTION
---------------

The DP83815 is a PCI bus based single chip 10/100 Mbps ethernet
controller targeted for PC mother boards, adapter cards, and embedded
systems. It fully implements the V2.2 33MHz PCI bus interface for host
communications with power management support. Packet descriptors and
data are transferred via bus-mastering, reducing the burden on the
host CPU. The DP83815 can support full duplex 10/100 Mbps transmission
and reception.

The DP83815 driver is developed for use on Linux v2.0.36. It can be
configured either as a static driver into the kernel or as a loadable
module. The driver is capable of 10/100 Mbps Full/Half duplex mode
operation via an internal PHY. Detailed features are given below.

2.1 Chip Features

o The chip is bus-master capable and transfers packet descriptors
and data using DMA with burst sizes upto 128 words.

o It supports 10 Mbps Ethernet and 100 Mbps Fast Ethernet via
an internal PHY and emerging 1-2 Mbps home networking
solutions via external PHY.

o Flexible Rx packet filtration including: perfect match, broadcast, 
multicast/unicast hash table, deep packet pattern matching.

o Internal Tx and Rx data FIFOs of 2KB each

o 802.3u Auto-Negotiation - advertised features configurable via
EEPROM

o Full duplex support for 10 and 100Mb/s data rates


2.2 Driver Features

2.2.1 Supported Features

o 10 Mbps full duplex and half duplex
o 100 Mbps full duplex and half duplex
o Auto Negotiation
o Broadcast and perfect match transmit and receive
o ARP transmit and receive
o Promiscuous mode
o Multicast support
o Loadable Kernel Module
o PCI bus probe for auto-configuration when configured as a static
  driver or a loadable kernel module
o Endian, and CPU architecture neutral
o Driver now supports insmod params.

2.2.2 Unsupported Features

o ACPI
o Multiple ethernet addresses 
o Wake on LAN and OnNow support for PC98/99, and other power-management
  features.
o Magic packets with SecureON, VLAN
o Programming EEPROM/Flash
o Old Linux kernels 1.xx.xx
o Newer MP kernels 2.2.xx 
o Remote boot           


3. DRIVER OPERATION
-------------------

Buffer management used in this version of the driver sets up the
transmit and receive buffer descriptors in a ring. The rings, shared
between the driver and the device, are located in memory buffers 
allocated in the kernel. The device accesses these buffers via DMA.

Each device will be allocated two similar queues of descriptors, one
for transmit and the other for receive. These queues are linked
together into a ring. For the default size of the transmit rings is
10, while the size of the receive ring is 30. These values can be
changed via macros in the driver source file.

The driver implements no copy transmit and receive operation. All
receive descriptors are created with no associated buffers, but the
descriptor uses the sk_buff passed to it via the transmit routine. The
sk_buff is held by the driver until the next transmit operation that
follows the actual transmit of the packet.

Each receive descriptor is initialized, at creation time, with an
sk_buff. Each sk_buff has a data buffer that can hold a max ethernet
packet. When a packet is suceesfully received, the skb is directly
sent to the stack with no copy, and it is replaced with a newly
allocated sk_buff. But if the packet less than 128 bytes
was received, the data is instead copied into a new skb. 

The driver makes of appropriate macros to adjust for endian-ness of
the host CPU, PCI bus, Ethernet, and the DP device. It also performs
appropriate translations of addresses for device use. These builtin
features makes the driver extremely portable to run on Linux systems
with non-Intel CPUs with minimum changes.


4. FILES IN THIS RELEASE
------------------------

The following files are included in a tar ball, dp83815.tar.gz

README.Linux   - This file
dp83815.c      - Driver source file
dp83815.h      - Driver header file
dp83815.o      - Driver object module
Makefile       - Rules to compile and install the driver.


5. MODIFICATION HISTORY
-----------------------

version 1.51:

o fixes for short cable attenuation.

version 1.50:

o merged dp83815.c driver, to support link aggregration
o added function dp83815_rx_control function for fail over only support.

version 1.40:

o upgraded the driver to support linux kernel version 2.4.x.
o fixed duplex support.
o added bytes count support to ethernet MIB.
o fixed MAC address read routine to read from eeprom.
o implemented support for load-time parameters.

version 1.30:

o Driver release supporting Redhat Linux ver 5.2 and 
  linux kernel version 2.0.36.
 
6. INSTALLATION
---------------

6.1 Kernel Requirements:

This version of the driver supports Linux kernel version 2.4.x and
has been tested on Red Hat 7.0 distribution.

6.2 Installation Procedure:

To install the driver, login as root, and extract the files into an 
appropriate directory.

     # mkdir <path>/dp83815 
     # cp dp83815.c dp83815.h makefile  <path>/dp83815/
     # cd <path>/dp83815/
     # make

To install the current driver object module in the filesystem run,

     # make install

You'll need to edit the following files to configure the driver as a
valid ethernet device in the system

  o /etc/hosts
  o /etc/sysconfig/network-scripts/ifcfg-eth?
  o /etc/conf.modules

6.3 Load Parameters:

This version of the driver supports to specify parameters via insmod
while loading the driver dynamically in to the running
kernel. Parameter are specified as list of string inside double quotes
seprated by commas.  This does not apply to "debug" parameter which is
a integer value. For example

     fullduplex="1,0,1,0,0,0" debug=7

Using these parameters, except "debug", maximum MAX_DEV dp83815
devices can be configured to user specified configuartion. If there
more than MAX_DEV dp83815 devices first MAX_DEV devices will be
configured using these parameters if they are defined otherwise they
will use defaults, and other devices beyond MAX_DEV will be configured
for autonegotiation advertising for 10/100 Mb/s and full/half duplex.

Following is the list parameters supported:

  o fullduplex: This parameter can be assigned values 0, 1, or 2.

    0: Device in half duplex mode. 
    1: Device in full duplex mode.
    2: full/half duplex (used during autonegotiation to advertise 
       that device is capable of doing both full and half duplex 
       mode of operation.)

  If no value is defined for full-duplex parameter, in case of
  autonegotiation, driver will use full/half duplex capability to advertise,
  otherwise it will put device in half duplex mode. If no autonegotiation is
  being done, then value of 2 for fullduplex will put the device in half
  duplex mode.

  o speed100: This parameter can be assigned value 0, 1, or 2.

    0: Device in 10 Mb/s mode.
    1: Device in 100 Mb/s mode.
    2: Device either in 10 or 100 Mb/s mode. (used during auto 
       negotiation to advertise the capability of the device.)
  
  If no value is defined for this parameter, in case of
  autonegotiation device capability of 10/100 Mb/s will be advertised,
  otherwise in case of no autonegotiation it will put device in 100
  Mb/s mode. If no autonegoatiation is done then value of 2 for
  speed100 will put the device in 100 Mb/s mode.

  o autoneg: This parameter can be assigned 0 or 1.
    
    0: no autonegotiation.
    1: do autonegotiation.
      
  If no value is defined for this parameter, driver will do the
  autonegotiation for the device capability defined by fullduplex and
  speed100 parameter.
  
  Autonegotiation can be done for the following capabilities:
  a) 10 Mb/s full/half duplex.
  b) 10/100 Mb/s half duplex.
  c) 100 Mb/s half duplex.
  d) 10/100 Mbs/s hfull/half duplex.

 o txdesc: This parameter specify number of desriptor used for
           tranmit, it can be assigned any interger value. By default
           driver uses 32 descriptors for transmit.

 o rxdesc: This parameter specify number of receive desriptor, and can
           be assigned any interger value. By default driver uses 30
           desriptors.
 
 o debug: This parameter can be assigned any or sum of the following values.
 
   1:   debug enabled at device probe level.
   2:   debug enabled at device open level.
   4:   debug enabled at device close level.
   8:   debug enabled at device ioctl level.
   16:  debug enabled at device transmit level.
   32:  debug enabled at device receive level.
   64:  debug enabled at device rxfilter setup level.
   128: debug enabled at device autonegotiation level.
   256: debug enabled at device interrupt level.
   
   By defualt none of the debug level will be enabled.

 
The driver can be dynamically loaded and unloaded from a running kernel 
using insmod/rmmod.

	# insmod dp83815
        # rmmod dp83815

Load time parameters can be specified while  running insmod. The example below
shows parameters for   3   dp83815   cards.

  - Card 0 is configured with full-duplex/100/no-autoneg/64 txDesc/32 rxDesc

  - Card 1 is configured with half-duplex/10/no-autoneg/32 txdesc/default
  rxdesc

  - Card 2 is configured with full-or-half-duplex/10-or-100/autoneg/default
  tx-and-rx desc

  - Driver debug level is set to 6 (open & close debug level enabled).


	# insmod dp83815 fullduplex="1,0,2" speed100="1,0,2" \
          autoneg="0,0,1" txdesc="64,32" rxdec="32" debug=6

To install and test the driver, shutdown the network stack, install
the driver, and re-start the startup the stack again.

	# sh /etc/rc.d/init.d/network stop
	# insmod dp83815
	# sh /etc/rc.d/init.d/network start
	

7. FUNCTIONALITY TESTING
------------------------

When the driver is loaded into the system via `insmod' it probes the
PCI bus to locate all DP83815 devices, and creates control structures
for each. The driver logs a couple of messages available in
`/var/log/messages' for each device with information about its PCI
geographic location, IRQ, IO address, and some basic debug information 
(addresses of some important structures).

All the devices on the PCI bus can be listed by,

# cat /proc/pci

IRQ and IO address information from this can be correlated with the 
information displayed by the driver in `/var/log/messages'.

When the TCP/IP stack is initialized, it opens all configured ethernet 
devices, and initializes them for use. At this time, the driver will
perform autonegotiation and log information about the link status.
The driver can then be tested by running ping, telnet, ftp, NFS etc.


8. KNOWN DRIVER PROBLEMS
------------------------

o  Since the device requires that the receive buffer be aligned on a
4-byte (word) boundary, it is not possible to align the IP header
on a word boundary unless the received packet is copied. Since this
version of the driver support a no-copy receive operation, the IP
header will not be aligned. This is not a problem for Intel CPUs,
but will cause exceptions on RISC CPUs (PowerPC, Alpha, ...). While
porting the driver to these platforms, the no-copy receive can be
turned off, but setting the DP_RX_COPY_THRESHOLD to ETH_MAX_PKT_SIZE,
whereby forcing all receive buffers to be copied.

The driver ensures that the IP headers in a copied buffer is aligned
on a word-boundary.


9. DOCUMENTATION REFERENCES
---------------------------

1. National DP83815 10/100 Mb/s Integrated PCI Ethernet
   Media Access Controller and Physical Layer, 1998.

2. Linux Device Drivers, Alessandro Rubini, 1998.

10. BUG REPORTS
--------------

Bug reports and enhancement requests should be directed to
National. Please include the driver version number, hardware
information, and Linux kernel version number along with the bug
report.

mailto: support@nsc.com

*/

/* Generic Kernel Module/Driver Headers */

#include <linux/module.h>
#include <linux/kernel.h>

#include <linux/pci.h>

#include <linux/ioport.h>
#include <asm/io.h>         /* IO stuff */
#include <asm/byteorder.h>
#include <linux/delay.h>

/* Ethernet Driver Specific Headers */

#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/skbuff.h>
#include <linux/malloc.h>
#include <linux/if_ether.h>

#include "dp83815.h"

#ifdef LINK_AGGR
#include "adapter.h"
#endif

/* Macros */

/*
 * SWAP_BUS_TO_CPU_XX and SWAP_CPU_TO_BUS_XX macros swap 16 and 32 bit values
 * between the PCI bus' little endian byte-order and the CPU's native
 * byte-order.
 */

#ifdef __BIG_ENDIAN
#define BUS_TO_CPU_SWAP_16(X)    swap_16(X)
#define BUS_TO_CPU_SWAP_32(X)    swap_32(X)
#define CPU_TO_BUS_SWAP_16(X)    swap_16(X)
#define CPU_TO_BUS_SWAP_32(X)    swap_32(X)
#define CPU_TO_NET_SWAP_16(X)    ((u16)(X))
#define CPU_TO_NET_SWAP_32(X)    ((u32)(X))
#define NET_TO_CPU_SWAP_16(X)    ((u16)(X))
#define NET_TO_CPU_SWAP_32(X)    ((u32)(X))
#else
#define BUS_TO_CPU_SWAP_32(X)    ((u32)(X))
#define BUS_TO_CPU_SWAP_16(X)    ((u16)(X))
#define CPU_TO_BUS_SWAP_32(X)    ((u32)(X))
#define CPU_TO_BUS_SWAP_16(X)    ((u16)(X))
#define CPU_TO_NET_SWAP_16(X)    swap_16(X)
#define CPU_TO_NET_SWAP_32(X)    swap_32(X)
#define NET_TO_CPU_SWAP_16(X)    swap_16(X)
#define NET_TO_CPU_SWAP_32(X)    swap_32(X)
#endif

/* Macros translate addresses between PCI bus and CPU */
#define BUS_TO_CPU_ADDR_XLATE(X)  BUS_TO_CPU_SWAP_32(bus_to_virt(X))
#define CPU_TO_BUS_ADDR_XLATE(X)  CPU_TO_BUS_SWAP_32(virt_to_bus(X))

/* Macros to read/write 32/16 bit data to/from DP83815 device registers. */
#define DP_REG32_WRITE(reg, val)  io_write_32 (iobase+(reg),            \
                                               CPU_TO_BUS_SWAP_32(val))
#define DP_REG32_READ(reg)        BUS_TO_CPU_SWAP_32(io_read_32 (iobase+(reg)))
#define DP_REG16_WRITE(reg, val)  io_write_16 (iobase+(reg),            \
                                               CPU_TO_BUS_SWAP_16(val))
#define DP_REG16_READ(reg)        BUS_TO_CPU_SWAP_16(io_read_16 (iobase+(reg)))
#define DP_REG32_SET(reg, val)    DP_REG32_WRITE(reg,DP_REG32_READ(reg)|(val))
#define DP_REG32_CLR(reg, val)    DP_REG32_WRITE(reg,DP_REG32_READ(reg)&~(val))
#define DP_REG16_SET(reg, val)    DP_REG16_WRITE(reg,DP_REG16_READ(reg)|(val))
#define DP_REG16_CLR(reg, val)    DP_REG16_WRITE(reg,DP_REG16_READ(reg)&~(val))

/* Debug Macros */
#define DP_DEBUG_PROBE            0x00000001
#define DP_DEBUG_OPEN             0x00000002
#define DP_DEBUG_CLOSE            0x00000004
#define DP_DEBUG_IOCTL            0x00000008
#define DP_DEBUG_TX               0x00000010
#define DP_DEBUG_RX               0x00000020
#define DP_DEBUG_MC               0x00000040
#define DP_DEBUG_ANEG             0x00000080
#define DP_DEBUG_INTR             0x00000100
#define DP_DEBUG_LOAD             0x00000200
#define DP_DEBUG_UNLOAD           0x00000400

#define DEBUG

#ifdef DEBUG
u32     dp_debug_level=0;
#define DP_DEBUG(level, X)  if (level & dp_debug_level) printk X
#else
#define DP_DEBUG(level, X)
#endif

/* data types */
typedef u32    status;                /* return status */
typedef volatile u8 * virt_addr;      /* CPU virtual address */
typedef volatile u8 * bus_addr;       /* BUS physical address */
typedef u8     bool;

#define OK        0                   /* status: OK */
#define ERROR    -1                   /* status: ERROR */
#define TRUE      1
#define FALSE     0

#ifdef LINK_AGGR
#define DP_RX_DISABLE               0 /* disable rx engine */
#define DP_RX_ENABLE                1 /* enable rx engine */
#endif

/* Default Driver Parameters */
#define DP_DEFAULT_TXQ_SIZE         32
#define DP_DEFAULT_RXQ_SIZE         30
#define DP_RX_COPY_THRESHOLD        128 /* upper limit for rx packet copy */
#define DP_POLYNOMIAL               0x04C11DB6

/* Alignment and packet sizes */
#define ETH_CRC_LEN                 4
#define ETH_MAX_PKT_SIZE            (ETH_FRAME_LEN + ETH_CRC_LEN)
#define DP_ALIGN                    4     /* word alignment */

/* Driver private descriptor macros */
#define DP_DESC_SKBPTR              0x0c  /* SKB pointer offset */
#define DP_QUEUE_ELE_SIZE           (DP_DESC_SIZE + 4)
#define DP_QUEUE_ELE_NEXT(q)                                            \
        q->cur_desc_addr = DP_QUEUE_ELE_NEXT_GET(q, q->cur_desc_addr)
#define DP_QUEUE_ELE_NEXT_GET(q, desc_addr)                                    \
        ((desc_addr) == (q)->last_desc_addr) ? (q)->first_desc_addr :        \
                                           (desc_addr) + DP_QUEUE_ELE_SIZE

/* Macros to get/set the values of the descriptor fields */
#define DP_DESC_LNK_GET(ptr)     *(u32 *)((virt_addr)ptr + DP_DESC_LNK)
#define DP_DESC_CMDSTS_GET(ptr)  *(u32 *)((virt_addr)ptr + DP_DESC_CMDSTS)
#define DP_DESC_BUFPTR_GET(ptr)  *(u32 *)((virt_addr)ptr + DP_DESC_BUFPTR)
#define DP_DESC_SKBPTR_GET(ptr)  *(u32 *)((virt_addr)ptr + DP_DESC_SKBPTR)

#define DP_DESC_LNK_SET(ptr, val)      DP_DESC_LNK_GET(ptr) = (val)
#define DP_DESC_CMDSTS_SET(ptr, val)   DP_DESC_CMDSTS_GET(ptr) = (val)
#define DP_DESC_BUFPTR_SET(ptr,val)    DP_DESC_BUFPTR_GET(ptr) = (val)
#define DP_DESC_SKBPTR_SET(ptr,val)    DP_DESC_SKBPTR_GET(ptr) = (val)

/*
 * Macros to get/set the values of descriptor fields with
 * appropriate address and byte order translations
 */
#define DP_DESC_LNK_XLATE_GET(p)    BUS_TO_CPU_ADDR_XLATE(DP_DESC_LNK_GET(p))
#define DP_DESC_CMDSTS_XLATE_GET(p)    BUS_TO_CPU_SWAP_32(DP_DESC_CMDSTS_GET(p))
#define DP_DESC_BUFPTR_XLATE_GET(p) BUS_TO_CPU_ADDR_XLATE(DP_DESC_BUFPTR_GET(p))

#define DP_DESC_LNK_XLATE_SET(p, v)                                    \
        DP_DESC_LNK_SET(p, CPU_TO_BUS_ADDR_XLATE(v))
#define DP_DESC_CMDSTS_XLATE_SET(p, v)                                 \
        DP_DESC_CMDSTS_SET(p, CPU_TO_BUS_SWAP_32(v))
#define DP_DESC_BUFPTR_XLATE_SET(p,v)                                  \
        DP_DESC_BUFPTR_SET(p, CPU_TO_BUS_ADDR_XLATE(v))


/* Descriptor queue */
struct dp83815_queue {
    virt_addr  first_desc_addr;  /* descriptor array address */
    virt_addr  last_desc_addr;   /* last descriptor address */
    virt_addr  cur_desc_addr;    /* current descriptor address */
    virt_addr  read_desc_addr;   /* current reclaim desc address */
    virt_addr  qbuf;             /* allocated queue buffer */
    u16        count;            /* number of elements */
};

/* Queue types -- qtype */
#define DP_QUEUE_TYPE_TX    1    /* Transmit queue */
#define DP_QUEUE_TYPE_RX    2    /* Receive queue */

/* Device private data */
struct dp83815_priv {
    struct net_device *        next;        /* Next dp83815device */
    struct dp83815_queue       tx_queue;    /* Transmit Descriptor Queue */
    struct dp83815_queue       rx_queue;    /* Receive Descriptor Queue */
    struct net_device_stats    stats;       /* MIB data */
    unsigned int               fullduplex;  /* half or full duplex */
    unsigned int               speed100;    /* 10/100 MBps */
    unsigned int               tx_desc;     /* number of transmit descriptor */
    unsigned int               rx_desc;     /* number of receive descriptor */
    unsigned int               autoneg;
#ifdef  LINK_AGGR
    struct NsmInfo             NsmCtxt;
#endif
};

static struct net_device *    dp83815_dev_list;        /* List of dp83815devices */

/* Linux Network Driver interface routines */
extern int dp83815_probe (struct net_device *dev);
static int dp83815_open (struct net_device *dev);
static int dp83815_close (struct net_device *dev);
static int dp83815_start_xmit (struct sk_buff *skb, struct net_device *dev);
static void dp83815_set_multicast_list (struct net_device *dev);
static int dp83815_ioctl (struct net_device *dev, struct ifreq *rq, int cmd);
static void dp83815_interrupt (int irq, void *dev_id, struct pt_regs *regs);
static struct net_device_stats *dp83815_get_stats (struct net_device *dev);

/* Support Functions */
/*static u16 swap_16 (u16 us);*/
/*static u32 swap_32 (u32 ui);*/
static u32 io_read_32 (u16 io_port);
static void io_write_32 (u16 io_port, u32 data);
static u16 io_read_16 (u16 io_port);
static void io_write_16 (u16 io_port, u16 data);

/* Driver Private Routines */
#ifdef LINK_AGGR
static void dp83815_mac_address_set (AdapterContext *pAdp, char *mac_addr);
static status dp83815_dev_reset (AdapterContext *pAdp);
static UCHAR  dp83815_send_packet (AdapterContext *pAdp, PktInfo *pPkt, int priority);
#else
static void dp83815_mac_address_set (u32 iobase, char *mac_addr);
static status dp83815_dev_reset (u32 iobase);
static int dp83815_send_packet (struct sk_buff *skb, struct net_device *dev);
#endif
static void dp83815_mac_address_get (u32 iobase, char *mac_addr);
static status dp83815_queue_create (struct dp83815_queue *q, int count, int qtype);
static status dp83815_queue_delete (struct dp83815_queue *q);
static virt_addr dp83815_tx_desc_get (struct net_device *dev);
static virt_addr dp83815_rx_desc_get (struct net_device *dev);
static status dp83815_phy_setup (struct net_device *dev);
static int dp83815_crc (char * mc_addr);
static void dp83815_tx_skb_reclaim_irq (struct net_device *dev, virt_addr desc_addr);
static void dp83815_eeprom_delay (u32 iobase);
static u16  dp83815_eeprom_read (u32 iobase, int addr);

/* LACP specific routines */
#ifdef LINK_AGGR
static void dp83815_rx_control (AdapterContext *pAdp, int flag);
static void dp83815_clear_mca_tbl (AdapterContext *pAdapter);
static unsigned char dp83815_multicast_delete (AdapterContext* pAdp, MultiList *pMulti);
static unsigned char dp83815_multicast_add (AdapterContext *pAdp, MultiList *pMulti, UCHAR  ReloadFlag);
static void dp83815_IndicateLinkStatus (AdapterContext *pAdapter);
static unsigned char dp83815_setphy (AdapterContext *pAdapter, UINT PhysFlags, UINT SetSpeed);
static unsigned char dp83815_rxfilter (AdapterContext *pAdapter, UINT rxFilterFlag);
#endif

/* Driver Debug Routines */
#ifdef DEBUG
#if 0 /* not used */
static void dp83815_regs_info (u16 iobase);
#endif /* not used */
#endif

/* default driver parameters */
#define MAX_DEV   10   /* number of devices could be more */
unsigned int fullduplex[MAX_DEV]  = {-1, -1, -1, -1, -1, -1, -1, -1, -1, -1};
unsigned int speed100[MAX_DEV]    = {-1, -1, -1, -1, -1, -1, -1, -1, -1, -1};
unsigned int txdesc[MAX_DEV]      = {-1, -1, -1, -1, -1, -1, -1, -1, -1, -1};
unsigned int rxdesc[MAX_DEV]      = {-1, -1, -1, -1, -1, -1, -1, -1, -1, -1};
unsigned int autoneg[MAX_DEV]     = {-1, -1, -1, -1, -1, -1, -1, -1, -1, -1};
unsigned int debug                = 0;

/* Module description and parameters */

#ifdef MODULE
MODULE_DESCRIPTION("National Semiconductor DP83815 10/100 Ethernet Driver");
MODULE_PARM(fullduplex, "1-" __MODULE_STRING(MAX_DEV) "i");
MODULE_PARM(speed100,   "1-" __MODULE_STRING(MAX_DEV) "i");
MODULE_PARM(autoneg,    "1-" __MODULE_STRING(MAX_DEV) "i");
MODULE_PARM(txdesc,     "1-" __MODULE_STRING(MAX_DEV) "i");
MODULE_PARM(rxdesc,     "1-" __MODULE_STRING(MAX_DEV) "i");
MODULE_PARM(debug, "i");
#endif

/*
 * dp83815_probe - enumerate the PCI bus for instances of DP83815
 *
 * This routine enumerates DP83815 ethernet devices on the PCI bus, and
 * registers each DP83815 device found.
 */
int
dp83815_probe (struct net_device *dev)
{
    int  dev_count;
    u8   bus;
    u8   func;
    u8   irq;
    u32  iobase;
    u32  version;
    struct dp83815_priv *dev_priv;
#ifdef LINK_AGGR
    NsmContext *pNsm = NULL;
    AdapterContext *pAdp = NULL;
#endif

    if (! pcibios_present())
        return -ENODEV;        /* No such device */

    dev_count=0;
    while (pcibios_find_device (PCI_VENDOR_ID_NS, PCI_VENDOR_ID_NS_83815,
                                dev_count, &bus, &func)
           == PCIBIOS_SUCCESSFUL) {

        /* Allocate, name the device and add to ethernet device list */
        dev = init_etherdev(dev, sizeof (struct dp83815_priv));

        if (dev == NULL) {
            printk (KERN_INFO "%s: Failed to create device struct\n",
                    DP_DRV_NAME);
            break;
        }

#ifdef LINK_AGGR
        pNsm = &((struct dp83815_priv *)(dev->priv))->NsmCtxt;
        pNsm->dev = dev;
        pAdp = &pNsm->AdpCtxt;
        pAdp->pNsmContext = (void *)pNsm;
        pAdp->MacStats.txOkCount = 0;
#endif
        
        /* Read PCI Configuration Registers */
        pcibios_read_config_byte (bus, func, PCI_INTERRUPT_LINE, &irq);
        pcibios_read_config_dword (bus, func, PCI_BASE_ADDRESS_0, &iobase);
        iobase &= PCI_BASE_ADDRESS_IO_MASK;

#ifdef LINK_AGGR
        pAdp->RegAddr = (UCHAR *)iobase;
#endif        

        /* set the parameters for the driver */
        dev_priv = (struct dp83815_priv *)dev->priv;
        if (dev_count < MAX_DEV) {
        
            if (autoneg[dev_count] != 0) {

	        dev_priv->autoneg = 1;
                
                /* set the speed and duplex mode*/
		if (speed100[dev_count] == 1) {

		    dev_priv->speed100 = 1;
                    dev_priv->fullduplex = 2;
		}
		else if (speed100[dev_count] == 0) {

		    dev_priv->speed100 = 0; 
                    dev_priv->fullduplex = 2;
		}
		else {

		    dev_priv->speed100 = 2;
                    if (fullduplex[dev_count] == 0)
                        dev_priv->fullduplex = 0;
                    else
                        dev_priv->fullduplex = 2;                                                 
		}
            } else {
	        dev_priv->autoneg = 0;
                
                /* set the speed */
                if (speed100[dev_count] == 0)
		    dev_priv->speed100 = 0;
                else
                    dev_priv->speed100 = 1;
            
                /* set the duplex mode */
                if (fullduplex[dev_count] == 1)
		    dev_priv->fullduplex = 1;
                else
		    dev_priv->fullduplex = 0;
	    }

            dev_priv->tx_desc = ((txdesc[dev_count] == -1) ?
                                 DP_DEFAULT_TXQ_SIZE : txdesc[dev_count]);  
            dev_priv->rx_desc = ((rxdesc[dev_count] == -1) ?
                                 DP_DEFAULT_RXQ_SIZE : rxdesc[dev_count]);
	}
        else {

	    dev_priv->autoneg = 1;
	    dev_priv->fullduplex = 2;
	    dev_priv->speed100 = 2;
            dev_priv->tx_desc = DP_DEFAULT_TXQ_SIZE;
            dev_priv->rx_desc = DP_DEFAULT_RXQ_SIZE;
	}
        
#ifdef DEBUG
	dp_debug_level = debug;         
#endif        

        /* Put the device in a quiescent state */
#ifdef LINK_AGGR
        if (dp83815_dev_reset (pAdp) != OK)
#else            
        if (dp83815_dev_reset (iobase) != OK)
#endif            
        {

            printk (KERN_INFO "%s: Device Reset failed.\n",
                    DP_DRV_NAME);
            kfree (dev); 
            continue;             /* Try the next device */
        }

        /* Get the ethernet address */
        dp83815_mac_address_get (iobase, dev->dev_addr);


        /* initialize the device data */
        dev->base_addr          = iobase;
        dev->irq                = irq;
        dev->open               = &dp83815_open;
        dev->stop               = &dp83815_close;
        dev->get_stats          = &dp83815_get_stats;
        dev->do_ioctl           = &dp83815_ioctl;
        dev->hard_start_xmit    = &dp83815_start_xmit;
        dev->set_multicast_list = &dp83815_set_multicast_list;
        
        ether_setup (dev);        /* initialize generic fields */

        /* reserve IO region */
        request_region (iobase, DP83815_PCI_IO_SIZE, dev->name);
        
        /* Display board info */
        version = DP_REG32_READ(DP_SRR);
        printk (KERN_INFO "%s: bus=%d func=%d io=0x%x irq=%d ver=%d.%d\n",
                dev->name, bus, func, iobase, irq,
                (version & DP_SRR_MAJ) >> DP_SRR_MAJ_SHF,
                (version & DP_SRR_MIN));
        
        printk (KERN_INFO "%s: ethernet addr=%02x:%02x:%02x:%02x:%02x:%02x\n",
                dev->name,
                dev->dev_addr[0], dev->dev_addr[1], dev->dev_addr[2],
                dev->dev_addr[3], dev->dev_addr[4], dev->dev_addr[5]);
        
        /* Chain the device */
        ((struct dp83815_priv *)(dev->priv))->next = dp83815_dev_list;
        dp83815_dev_list = dev;

#ifdef DEBUG
        printk (KERN_INFO "%s: DebugInfo: dev=0x%x priv=0x%x "
                "&dp_debug_level=0x%x\n",
                dev->name, (u32)dev, (u32)dev->priv, (u32)&dp_debug_level);
#endif

        /* Update Counters */
        dev_count++;
#ifdef LINK_AGGR
        memcpy (pAdp->PermMacAddr, dev->dev_addr, 6);
        memcpy (pAdp->CurrMacAddr, dev->dev_addr, 6);

        /* intialize call back routines for LACP */
        pAdp->FuncStruct.HsmSendPkt = dp83815_send_packet;
        pAdp->FuncStruct.HsmSendPkts = NULL;
        pAdp->FuncStruct.DrvSend = NULL;            
        pAdp->FuncStruct.HsmMulticastAdd = dp83815_multicast_add;
        pAdp->FuncStruct.HsmMulticastDelete = dp83815_multicast_delete;
        pAdp->FuncStruct.HsmSetMacAddress = dp83815_mac_address_set;
        pAdp->FuncStruct.HsmRxFilter = dp83815_rxfilter;
        pAdp->FuncStruct.HsmSetPhyCapabilities = dp83815_setphy;
        pAdp->FuncStruct.NsmSetPktStatus = NULL;
        pAdp->FuncStruct.NsmIndicateLinkStatus = dp83815_IndicateLinkStatus;
        pAdp->FuncStruct.HsmRxControl = dp83815_rx_control;
        
        /* register ethernet adapter with LACP */
        LacpRegister (pAdp);
#endif
        
        dev=NULL;
    }
    return dev_count ? OK : -ENODEV;
}

/* dp83815_open - open and initialize a device */
static int
dp83815_open (struct net_device *dev)
{
    u32     iobase = dev->base_addr;
    struct dp83815_priv* priv   = dev->priv;
    u32     txcfg, rxcfg, phy_status;
#ifdef LINK_AGGR
    NsmContext     *pNsm = &(priv->NsmCtxt);
    AdapterContext *pAdp = &(pNsm->AdpCtxt);
#endif

    /* Reset the device -- paranoia! */
#ifdef LINK_AGGR    
    if (dp83815_dev_reset (pAdp) != OK)
#else        
    if (dp83815_dev_reset (iobase) != OK)
#endif        
        return -EAGAIN;
    
    /* Allocate Tx and Rx queues */
    if (dp83815_queue_create (&priv->tx_queue, priv->tx_desc,
                            DP_QUEUE_TYPE_TX) != OK) {
        printk (KERN_INFO "%s: Failed to create tx queue\n", dev->name);
        return -EAGAIN;
    }
    
    if (dp83815_queue_create (&priv->rx_queue, priv->rx_desc,
                            DP_QUEUE_TYPE_RX) != OK) {
        dp83815_queue_delete (&priv->tx_queue);
        printk (KERN_INFO "%s: Failed to create rx queue\n", dev->name);
        return -EAGAIN;
    }

    DP_DEBUG (DP_DEBUG_OPEN,
              (KERN_INFO "%s: tx queue size %d rx queue size %d \n",
               dev->name, priv->tx_desc, priv->rx_desc));    

    /* Install the Tx and Rx queues on the device */
    DP_REG32_WRITE (DP_TXDP, virt_to_bus(priv->tx_queue.first_desc_addr));
    DP_REG32_WRITE (DP_RXDP, virt_to_bus(priv->rx_queue.first_desc_addr));
    
    DP_DEBUG (DP_DEBUG_OPEN,
              (KERN_INFO "dp: setting TXDP=0x%x RXDP=0x%x\n",
               (u32)virt_to_bus (priv->tx_queue.first_desc_addr),
               (u32)virt_to_bus (priv->rx_queue.first_desc_addr)));

    /* Install interrupt vector */
    if (request_irq (dev->irq, &dp83815_interrupt, SA_SHIRQ,
                     dev->name, dev) != OK) {
        dp83815_queue_delete (&priv->tx_queue);
        dp83815_queue_delete (&priv->rx_queue);
        return -EAGAIN;
    }
    
    /* Setup phy capabilities */
    if (dp83815_phy_setup (dev) != OK) {
        printk (KERN_INFO "%s: Warning PHY setup did not complete. 
                           Check cable.\n",    dev->name);
    }
    
    phy_status = DP_REG16_READ (DP_PHYSTS);

    /* Setup transmit control */
    txcfg =  DP_TXCFG_DRTH_SET(0x30) | 
             DP_TXCFG_FLTH_SET(0x10) |
             DP_TXCFG_MXDMA_32       |
             DP_TXCFG_ATP;

    /* if full duplex set the CSI and HBI in TXCFG */
    if (phy_status & DP_PHYSTS_FDX)
        txcfg |= DP_TXCFG_CSI | DP_TXCFG_HBI;


    DP_REG32_WRITE (DP_TXCFG, txcfg);
    
    DP_DEBUG (DP_DEBUG_OPEN,
              (KERN_INFO "%s: TXCFG set to 0x%x\n", dev->name, txcfg));
    

    /* Setup receive control */
    rxcfg = DP_RXCFG_DRTH_SET(0x08) |
            DP_RXCFG_MXDMA_32;  

    /* if full duplex set the ATX in RXCFG */
    if (phy_status & DP_PHYSTS_FDX)
        rxcfg |= DP_RXCFG_ATX;


    DP_REG32_WRITE (DP_RXCFG, rxcfg);
    
    DP_DEBUG (DP_DEBUG_OPEN,
              (KERN_INFO "%s: RXCFG set to 0x%x\n", dev->name, rxcfg));
    
    /* Setup the ethernet address */
#ifdef LINK_AGGR    
    dp83815_mac_address_set (pAdp, dev->dev_addr);
#else    
    dp83815_mac_address_set (iobase, dev->dev_addr);
#endif    
    
    /* Receive perfect match and broadcast packets */
    DP_REG32_WRITE (DP_RFCR, 0);
    DP_REG32_WRITE (DP_RFCR, (DP_RFCR_AAB  | /* all broadcast pkts */
                              DP_RFCR_APM  | /* perfect match pkts */
                              DP_RFCR_RFEN));

    DP_DEBUG (DP_DEBUG_OPEN,
              (KERN_INFO "dp: RFCR set to 0x%x\n",
               DP_RFCR_RFEN | DP_RFCR_APM | DP_RFCR_AAB));
    
    /* Turn on device interrupts */
    DP_REG32_WRITE (DP_IMR, (DP_IMR_RXOK   | DP_IMR_TXOK   |
                             DP_IMR_TXERR  | DP_IMR_TXURN  |
                             DP_IMR_MIB    | DP_IMR_TXDESC |
                             DP_IMR_RTABT  | DP_IMR_RMABT  |
                             DP_IMR_SSERR  | DP_IMR_PHY));

    DP_REG32_WRITE (DP_IER, DP_IER_IE);

    /* Enable Tx/Rx */
    DP_REG32_WRITE (DP_CR, DP_CR_TXE | DP_CR_RXE);
    
    /* Increment module reference count */
    MOD_INC_USE_COUNT;

#ifdef LINK_AGGR
    pAdp->lacprxFilterFlag = 0;
    pAdp->AdapterStatus |= ADAPTER_OPEN;
#endif
    
    return OK;
}
 
/* dp83815_close - close a device, and reclaim resources */
static int
dp83815_close (struct net_device *dev)
{
    u32  iobase = dev->base_addr;
    struct dp83815_priv* priv   = dev->priv;
    
#ifdef LINK_AGGR
    NsmContext     *pNsm = &(priv->NsmCtxt);
    AdapterContext *pAdp = &(pNsm->AdpCtxt);
#endif
    
    /* Stop the Tx/Rx */
    /* XXX Do we need to do this ??? */
    DP_REG32_WRITE (DP_CR, (DP_CR_TXD | DP_CR_RXD)); 
    
    /* Reset the device */
#ifdef LINK_AGGR
    dp83815_dev_reset (pAdp);
#else    
    dp83815_dev_reset (iobase);
#endif    

    /* Uninstall the interrupt vector */
    free_irq (dev->irq, dev);
            
    /* Free the Tx and Rx queues */
    dp83815_queue_delete (&priv->tx_queue);
    dp83815_queue_delete (&priv->rx_queue);

    /* Decrement module reference count */
    MOD_DEC_USE_COUNT;
    
    return OK;
}

/*
 * dp83815_start_xmit - transmit an ethernet packet.
 *
 * This routine writes to a tx descriptor, sets the ownership bit of the
 * CMDSTS, and signals the chip
 */
static int
dp83815_start_xmit (struct sk_buff *skb, struct net_device *dev)
{
#ifdef LINK_AGGR
    int CableStatus;
    PktInfo     Pkt;
    NsmContext     *pNsm = &((struct dp83815_priv *)(dev->priv))->NsmCtxt;
    AdapterContext *pAdp = &(pNsm->AdpCtxt);
    int Priority;
    unsigned char status;
    
    Pkt.FragArray[0].pFrag =
        (UCHAR *)virt_to_phys (skb->data);
    
    Pkt.FragArray[0].FragLen = skb->len;

    Pkt.pOSCtrlBlk = skb;

    Pkt.NumFrags = 1;

    Pkt.pDLHeader = (void *)(skb->data);

    Priority = 0;

    if ((pAdp->nFlags) & GEC_FLAG)
        status = AggrSendPacket(pAdp, &Pkt, Priority, &CableStatus);
    else
        status = dp83815_send_packet(pAdp, &Pkt, Priority);
    
    if (status == FAILURE)
    {
        /* just for debug */
        printk (KERN_INFO "%s : error sending packets \n",dev->name);
        return ERROR;
    }
    
#else
    dp83815_send_packet(skb, dev);
#endif
    return OK;
}

#ifdef LINK_AGGR
static UCHAR
dp83815_send_packet (AdapterContext *pAdp, PktInfo *pPkt, int Priority)
#else    
static int
dp83815_send_packet (struct sk_buff *skb, struct net_device *dev)
#endif    
{
#ifdef LINK_AGGR
    struct net_device *dev = ((NsmContext *)(pAdp->pNsmContext))->dev;
    struct sk_buff *skb = pPkt->pOSCtrlBlk;
#endif
    
    u32        iobase = dev->base_addr;
    u32        cmdsts;
    virt_addr  tx_desc;
    struct net_device_stats *stats_p;
    int count = 50 ;

    if (skb->len > ETH_MAX_PKT_SIZE)
        return ERROR;

    tx_desc = dp83815_tx_desc_get(dev);
    while (tx_desc == NULL && --count) {
        DP_REG32_SET (DP_CR, DP_CR_TXE);
        DP_REG32_SET (DP_IMR, DP_IMR_TXOK | DP_IMR_TXIDLE);
        tx_desc = dp83815_tx_desc_get (dev);
    }

    stats_p = &((struct dp83815_priv *)(dev->priv))->stats;

    /* Update tx_desc to point to SKB data, set CMDSTS, and signal the chip */
    if (tx_desc != NULL) {

        /* update statistics of the previous transmit */
        cmdsts = DP_DESC_CMDSTS_XLATE_GET (tx_desc);
        if (cmdsts & DP_DESC_CMDSTS_TX_ERRORS) {

            stats_p->tx_errors++;
            
            /* Update individual counters */

            stats_p->collisions += DP_DESC_CMDSTS_TX_COLLISIONS_GET(cmdsts);

            if (cmdsts & DP_DESC_CMDSTS_TX_TXA) /* tx aborted */
                stats_p->tx_packets--;
            
            if (cmdsts & DP_DESC_CMDSTS_TX_TFU) /* fifo errors */
                stats_p->tx_fifo_errors++;

            if (cmdsts & DP_DESC_CMDSTS_TX_CRS) /* lost carrier */
                stats_p->tx_carrier_errors++;

            if (cmdsts & DP_DESC_CMDSTS_TX_OWC) /* out of window collisions */
                stats_p->tx_window_errors++;
        } else {
#ifdef LINK_AGGR
          pAdp->MacStats.txOkCount++;
#endif        
	  stats_p->tx_bytes += (cmdsts & DP_DESC_CMDSTS_SIZE) - ETH_CRC_LEN; /* number bytes transmitted */ 
	}

        /* clean up the descriptor */
        DP_DESC_CMDSTS_XLATE_SET (tx_desc, 0);
        /* Update the descriptor */
        DP_DESC_CMDSTS_XLATE_SET (tx_desc, DP_DESC_CMDSTS_OWN|
                                  DP_DESC_CMDSTS_INTR|skb->len);
        DP_DESC_BUFPTR_XLATE_SET (tx_desc, skb->data);
        DP_DESC_SKBPTR_SET (tx_desc, (u32)skb);
        
        dev->trans_start = jiffies;
        DP_REG32_SET (DP_CR, DP_CR_TXE);
        DP_DEBUG (DP_DEBUG_TX, 
                  (KERN_INFO "%s Tx: tx_desc=0x%x ", dev->name, (u32)tx_desc));
        stats_p->tx_packets++;
    } else {
        stats_p->tx_dropped++;
        dev_kfree_skb (skb);
    }

    return OK;
}
    

/*
 * dp83815_start_receive - receive the data from a Rx Queue descriptor
 *
 * This routine receives the data from Rx queue as long as it gets a valid
 * rx_descriptor and resets the descriptor's CMDSTS field back to the Buffer
 * size, and updates the BUFPTR and SKBPTR fields to the newly allocated SKB.
 */
static int
dp83815_start_receive (struct net_device *dev)
{
    u32         cmdsts;
    int         len;
    bool        do_copy;
    virt_addr   rx_desc;
    struct sk_buff *cur_skb;
    struct sk_buff *new_skb;
    struct sk_buff *rx_skb;
    struct net_device_stats *stats_p;

#if LINK_AGGR
    NsmContext  *pNsm = &((struct dp83815_priv *)(dev->priv))->NsmCtxt;
    AdapterContext *pAdp = &pNsm->AdpCtxt;
    AdapterContext *pAggrCtxt = pAdp->pAggregatorContext;
    NsmContext *pAggrNsmContext = (NsmContext *)(pAggrCtxt->pNsmContext);
#endif
    
    stats_p = &((struct dp83815_priv *)(dev->priv))->stats;
    
    for (rx_desc = dp83815_rx_desc_get(dev);
         (rx_desc != NULL);
         rx_desc = dp83815_rx_desc_get(dev)) {
        
        DP_DEBUG (DP_DEBUG_RX, 
                  (KERN_INFO "%s Rx: rx_desc=0x%x, CMDSTS = 0x%x",
                   dev->name, (u32)rx_desc, DP_DESC_CMDSTS_XLATE_GET(rx_desc)));

        cmdsts = DP_DESC_CMDSTS_XLATE_GET (rx_desc);

        /* Send the packet to the stack if no errors */
        if ((cmdsts & DP_DESC_CMDSTS_RX_ERRORS) == 0) {

            len = (cmdsts & DP_DESC_CMDSTS_SIZE) - ETH_CRC_LEN;

            /*
             * Allocate a new SKB
             * small data packets less than DP_RX_COPY_THRESHOLD are copied
             * into the new SKB, other allocate one to replace the current SKB.
             */
            if (len < DP_RX_COPY_THRESHOLD) {
                do_copy = TRUE;
                new_skb = alloc_skb (len + 2, GFP_ATOMIC);
            } else {
                do_copy = FALSE;
                new_skb = alloc_skb (ETH_MAX_PKT_SIZE, GFP_ATOMIC);
            }
        
            if (new_skb) {
                cur_skb = (struct sk_buff *) DP_DESC_SKBPTR_GET (rx_desc);

                if (do_copy) {
                    /* Copy data from current SKB and send the new SKB up */
                    rx_skb = new_skb;
                    skb_reserve (rx_skb, 2);
                    memcpy (skb_put(rx_skb, len), cur_skb->data, len);
                } else {
                    /* Replace the the current SKB with the new SKB */
                    rx_skb = cur_skb;
                    DP_DESC_BUFPTR_XLATE_SET (rx_desc, new_skb->data);
                    DP_DESC_SKBPTR_SET (rx_desc, (u32) new_skb);
                    (void) skb_put(rx_skb, len);
                }

                /* update the SKB and set it up */
#ifdef LINK_AGGR
                if (pAdp->nFlags & GEC_FLAG)
                    rx_skb->dev = pAggrNsmContext->dev_aggr;
                else
                    rx_skb->dev = pNsm->dev;
#else                
                rx_skb->dev = dev;
#endif                                

                rx_skb->protocol = eth_type_trans (rx_skb, dev);

                netif_rx (rx_skb);

                dev->last_rx = jiffies;
                stats_p->rx_packets++;
                stats_p->rx_bytes += len;
            } else 
                stats_p->rx_dropped++; /* no resources */

            if (cmdsts & DP_DESC_CMDSTS_RX_DEST_MC)
                stats_p->multicast++;
        }
        else {
            stats_p->rx_errors++; /* bad packet */

            /* Update individual counters */
            if (cmdsts & (DP_DESC_CMDSTS_RX_RUNT |
                          DP_DESC_CMDSTS_RX_LONG))
                stats_p->rx_length_errors++;

            if (cmdsts & DP_DESC_CMDSTS_RX_CRCE)
                stats_p->rx_crc_errors++;

            if (cmdsts & DP_DESC_CMDSTS_RX_FAE)
                stats_p->rx_frame_errors++;

            if (cmdsts & DP_DESC_CMDSTS_RX_RXO)
                stats_p->rx_fifo_errors++;
        }

        /* Cleanup the descriptor and make available for reception */
        DP_DESC_CMDSTS_XLATE_SET (rx_desc, ETH_MAX_PKT_SIZE);
    }
    return OK;
}

/* dp83815_get_stats - get current device statistics */
static struct net_device_stats *
dp83815_get_stats (struct net_device *dev)
{
    return &((struct dp83815_priv *)(dev->priv))->stats;
}

/* dp83815_set_multicast_list - sets multicast, & promiscuous mode */
static void
dp83815_set_multicast_list (struct net_device *dev)
{
    u32      rfcr_flags;
#ifndef LINK_AGGR    
    u32      iobase = dev->base_addr;
    u16   hash_table[32];
    int   i;
#endif    
    struct dev_mc_list * mc_list;

#ifdef LINK_AGGR
    MultiList *pMultiList = NULL;
    MultiList *pMulti, *pMultiNext;
    NsmContext *pNsm = &((struct dp83815_priv *)(dev->priv))->NsmCtxt;
    AdapterContext *pAdp = &(pNsm->AdpCtxt);
#endif
    
    /* default RFCR mode */
#ifdef LINK_AGGR
    rfcr_flags = ACCEPT_PERFECT_MATCH_ON | ACCEPT_ALL_BROADCAST_ON;
#else    
    rfcr_flags = DP_RFCR_APM | DP_RFCR_AAB | DP_RFCR_RFEN;
#endif
    
    /* Setup promiscuous mode */
    if (dev->flags & IFF_PROMISC) {
        DP_DEBUG (DP_DEBUG_OPEN,
                  (KERN_INFO "IFF_PROMISC\n"));
#ifdef LINK_AGGR
        rfcr_flags = PROMISCUOUS_ON;
#else                
        rfcr_flags = (DP_RFCR_AAU | DP_RFCR_AAM | DP_RFCR_AAB |
                      DP_RFCR_RFEN);
#endif        
    } else if (dev->flags & IFF_ALLMULTI) {

        /* Receive all multicast packets */
        DP_DEBUG (DP_DEBUG_OPEN,
                  (KERN_INFO "IFF_ALLMULTI\n"));
#ifdef LINK_AGGR
        rfcr_flags |= ACCEPT_ALL_MULTICAST_ON;
#else                
        rfcr_flags |= DP_RFCR_AAM;
#endif        
    } else {
#ifdef LINK_AGGR
        mc_list = dev->mc_list;
        pMultiList = NULL;
        
        /* Get the multicast address list from the mc_list and allocate a
           MultiList structure and put the multicast addresses into the
           MultiList structure  */
        
        while(mc_list)
        {
            pMulti = (MultiList *)kmalloc(sizeof(MultiList), GFP_KERNEL);
            if(pMulti == NULL)
            {
                if(pMultiList)
                {
                    pMulti = pMultiList;
                    while(pMulti)
                    {
                        pMultiNext = pMulti->Next;
                        kfree(pMulti);
                        pMulti = pMultiNext;
                    }
                }
                return;
            }
            memcpy( pMulti->MulticastAddr, mc_list->dmi_addr, 6); 
            pMulti->Next = pMultiList;
            pMultiList = pMulti;
            mc_list = mc_list->next;                    
        }
        
        /* Call the HSM function to add the list of multicast addresses to the
           filter */
        dp83815_multicast_add(pAdp, pMultiList, TRUE);    
        
        /* Free up the list of MultiList structures */
        pMulti = pMultiList;
        while(pMulti)
        {
            pMultiNext = pMulti->Next;
            kfree(pMulti);
            pMulti = pMultiNext;
        }
#else        
        /* Setup to receive programmed multicast packets */
        memset (hash_table, 0, 32);
        for (i=0, mc_list=dev->mc_list; mc_list && i < dev->mc_count;
             i++, mc_list = mc_list->next) {
            DP_DEBUG (DP_DEBUG_OPEN,
                      (KERN_INFO "mc_addr=%p\n", mc_list->dmi_addr));
            set_bit (dp83815_crc((char *)mc_list->dmi_addr) & 0x1ff, hash_table);
        }
        
        /* install the hash table */
        for (i=0; i<32; i++) {
            DP_REG32_WRITE (DP_RFCR, DP_RFCR_RFADDR_FMEM_LO + i*2);
            DP_REG32_WRITE (DP_RFDR, (u32) hash_table[i]);
        }
        
        rfcr_flags |= DP_RFCR_MHEN;
#endif        
    }

#ifdef LINK_AGGR
    dp83815_rxfilter (pAdp, rfcr_flags);
#else    
    DP_REG32_WRITE (DP_RFCR, 0);
    DP_REG32_WRITE (DP_RFCR, rfcr_flags);
#endif
    DP_DEBUG (DP_DEBUG_OPEN,
              (KERN_INFO "%s : MC Setup RFCR flags=0x%x\n",dev->name, rfcr_flags));
    
    return;
}

/* dp83815_crc - computer CRC for hash table entries */
static int
dp83815_crc (char * mc_addr)
{
    u32    crc;
    u8    cur_byte;
    u8    msb;
    u8    byte, bit;

    crc = ~0;
    for (byte=0; byte<6; byte++) {
        cur_byte = *mc_addr++;
        for (bit=0; bit<8; bit++) {
            msb = crc >> 31;
            crc <<= 1;
            if (msb ^ (cur_byte & 1)) {
                crc ^= DP_POLYNOMIAL;
                crc |= 1;
            }
            cur_byte >>= 1;
        }
    }
    crc >>= 23;

    return (crc);
}

/* dp83815_interrupt - handle driver specific ioctls */
static int
dp83815_ioctl (struct net_device *dev, struct ifreq *rq, int cmd)
{
#if LINK_AGGR
    void *data = &(rq->ifr_data);
    NsmContext *pNsm = &((struct dp83815_priv *)(dev->priv))->NsmCtxt;
    AdapterContext *pAdp = &(pNsm->AdpCtxt);

    switch (cmd)
    {
    case SIOCSIFADDR:
        memcpy ((void *)dev->dev_addr, data, 6);
        memcpy ((void *)pAdp->CurrMacAddr, data, 6);
        dp83815_mac_address_set (pAdp, pAdp->CurrMacAddr);
        break;
    case SIOCDEVPRIVATE:
        break;
    default:
        return -EOPNOTSUPP;
    }
    return OK;
#else
    return -EOPNOTSUPP;
#endif    
}

/* 02/15/2000 */
static u32 PreviousLinkStatus=0;
/* dp83815_interrupt - handle the interrupts */
static void
dp83815_interrupt (int irq, void *dev_id, struct pt_regs *regs)
{
    struct net_device * dev = dev_id;
    u16  iobase = dev->base_addr;
    u32  reg_isr;
    u32  CurrentLinkStatus;
	u16  tmpVal;
	u32  version;

#ifdef TEST_RX
    static int rx_cnt = 0;
    u16 phy_status;    
#endif    
#ifdef LINK_AGGR
    u16 phy_status;
    NsmContext     *pNsm = &((struct dp83815_priv *)(dev->priv))->NsmCtxt;
    AdapterContext *pAdp =  &pNsm->AdpCtxt;
#endif
    
    CurrentLinkStatus= DP_REG32_READ(DP_PHYSTS);
#define HAL_PHY_SPEED10   0x00000002
    if(PreviousLinkStatus!=CurrentLinkStatus)
    {
        if(CurrentLinkStatus & HAL_PHY_SPEED10)
        {
            DP_REG32_WRITE(DP_PHY_PAGE,0x0001);
            DP_REG32_WRITE(DP_PHY_EXTCFG,0x0000);
            DP_REG32_WRITE(DP_PHY_PAGE,0x0000);
        }
        else
        {
            DP_REG32_WRITE(DP_PHY_PAGE,0x0001);
            DP_REG32_WRITE(DP_PHY_EXTCFG,0x0010);
            DP_REG32_WRITE(DP_PHY_PAGE,0x0000);
	/* IP_0522 begin */
	
	version = DP_REG32_READ(DP_SRR);

	if (  ((version & DP_SRR_MAJ) == 0x0300) || ((version & DP_SRR_MAJ) == 0x0400) )
	{
    DP_REG32_WRITE (DP_PHY_PAGE, DP_PHY_PAGE_VAL);
	tmpVal = DP_REG32_READ ( DP_PHY_DSPCFG);
	tmpVal &= 0x0FFF;
    DP_REG32_WRITE (DP_PHY_DSPCFG, tmpVal | 0x1000);  /* Allow coef visibility */
	udelay (100);
	tmpVal = DP_REG32_READ ( DP_PHY_TDATA );

	if ( !(tmpVal & 0x0080) || ( ( tmpVal > 0x00D8 ) && (tmpVal <= 0x00FF) ) )
	{
    DP_REG32_WRITE (DP_PHY_TDATA, 0x00E8);
	tmpVal = DP_REG32_READ ( DP_PHY_DSPCFG);
    DP_REG32_WRITE (DP_PHY_DSPCFG, tmpVal | 0x0020);  /* Adjust and lock coefficient */
	}
    DP_REG32_WRITE (DP_PHY_PAGE, (u16) 0x0000); 
	}
/* IP_0522 end */

            
        }
        PreviousLinkStatus=CurrentLinkStatus;
    }
    
    reg_isr = DP_REG32_READ (DP_ISR);
    
    DP_DEBUG (DP_DEBUG_IOCTL,
              (KERN_INFO "%s: intr_status=0x%x\n", dev->name,
               reg_isr));

    if (reg_isr & DP_ISR_RXOK)
        dp83815_start_receive (dev);    

    
    if (reg_isr & (DP_ISR_TXOK | DP_ISR_TXERR | DP_ISR_TXURN | DP_ISR_TXIDLE |
                   DP_ISR_TXDESC))
        dp83815_tx_skb_reclaim_irq (dev, NULL);


    if (reg_isr & DP_ISR_PHY)
    {
#ifdef LINK_AGGR
        phy_status = DP_REG16_READ (DP_PHYSTS);
        
        if ((phy_status & DP_PHYSTS_LNK_VALID) &&
            !(pAdp->AdapterStatus & LINK_UP))
        {
            DP_DEBUG (DP_DEBUG_IOCTL,
                      (KERN_INFO " %s: is up \n", dev->name));
            pAdp->AdapterStatus |= LINK_UP;
        }
        if (!(phy_status & DP_PHYSTS_LNK_VALID) &&
            (pAdp->AdapterStatus & LINK_UP))
        {
            DP_DEBUG (DP_DEBUG_IOCTL,
                      (KERN_INFO "%s: is down \n", dev->name));
            pAdp->AdapterStatus &= ~LINK_UP;            
        }

        if (pAdp->nFlags)
            LacpAdapterStatusCheck(pAdp);
#else    
        DP_DEBUG (DP_DEBUG_IOCTL,
                   (KERN_INFO " %s: Link status changed \n", dev->name));
#endif        
    }
}

/* dp83815_mac_address_set - set the ethernet address */
static void
#ifdef LINK_AGGR    
dp83815_mac_address_set (AdapterContext *pAdp, char* mac_addr)
#else    
dp83815_mac_address_set (u32 iobase, char *mac_addr)
#endif    
{
    u16 *    mac_addr_ptr;
    int i;
#ifdef LINK_AGGR
    u32 iobase = (u32) pAdp->RegAddr;
#endif
    
    for (i=0, mac_addr_ptr = (u16 *)mac_addr; i<3; i++, mac_addr_ptr++) {
        DP_REG32_WRITE (DP_RFCR, DP_RFCR_RFADDR_PMATCH1 + i*2);
        DP_REG32_WRITE (DP_RFDR, CPU_TO_BUS_SWAP_16(*mac_addr_ptr));
    }
#ifdef LINK_AGGR
    if (pAdp->nFlags)
        LacpSetMacAddress(pAdp, mac_addr);
#endif    
}

#define EEPROM_READ  (0x6 << 6)

static void
dp83815_eeprom_delay (u32 iobase)
{
    DP_REG32_READ(DP_MEAR);
}

static u16
dp83815_eeprom_read (u32 iobase, int addr)
{
    int cmd = addr | EEPROM_READ;
    int i;
    u16 val = 0;

    DP_REG32_WRITE (DP_MEAR, DP_MEAR_EESEL);

    for (i = 10; i>=0; i--)
    {
        if ( cmd & (1 << i))
            DP_REG32_WRITE (DP_MEAR, DP_MEAR_EESEL | DP_MEAR_EEDI);
        else
            DP_REG32_WRITE (DP_MEAR, DP_MEAR_EESEL);
        dp83815_eeprom_delay (iobase);
        DP_REG32_WRITE (DP_MEAR, DP_MEAR_EECLK |
                        DP_REG32_READ(DP_MEAR));
        dp83815_eeprom_delay (iobase);
    }
    
    DP_REG32_WRITE (DP_MEAR, DP_MEAR_EESEL);
    dp83815_eeprom_delay (iobase);

    for (i = 0; i < 16; i++) {
        DP_REG32_WRITE (DP_MEAR, DP_MEAR_EECLK | DP_MEAR_EESEL);
        dp83815_eeprom_delay (iobase);
        if (DP_REG32_READ (DP_MEAR) & DP_MEAR_EEDO)
            val |= 1 << i;
        DP_REG32_WRITE (DP_MEAR, DP_MEAR_EESEL);
        dp83815_eeprom_delay (iobase);
    }
    DP_REG32_WRITE (DP_MEAR, DP_MEAR_EESEL);   
    DP_REG32_WRITE (DP_MEAR, 0);
    return val;
}

/* dp83815_mac_address_get - get the ethernet address */
static void
dp83815_mac_address_get (u32 iobase, char *mac_addr)
{
    u16 old_val, new_val;
    int  i;
    
    old_val = dp83815_eeprom_read (iobase, 6);
    for (i = 0; i < 3; i++)
    {
        new_val = dp83815_eeprom_read (iobase, i + 7);
        mac_addr[i*2] = (new_val << 1) + (old_val >> 15);
        mac_addr[(i*2) + 1] = new_val >> 7;
        old_val = new_val;
    }
    /* If address is not set, set up a default one */
    /* XXX needed only for pre-release versions of the hardware */
    if ((mac_addr[0] == 0) && (mac_addr[1] == 0) &&
	(mac_addr[2] == 0) && (mac_addr[3] == 0) &&
	(mac_addr[4] == 0) && (mac_addr[5] == 0)) {
         u32   random = jiffies;
         u8 *  ptr = (u8 *) &random;

	 mac_addr[0] = 0x08; /* National's Ethernet ID 0 */
	 mac_addr[1] = 0x00; /* National's Ethernet ID 1 */
	 mac_addr[2] = 0x17; /* National's Ethernet ID 2 */
	 mac_addr[3] = *ptr++;
	 mac_addr[4] = *ptr++;
	 mac_addr[5] = *ptr;
    }

}

/* dp83815_dev_reset - soft reset DP83815 */
static status
#ifdef LINK_AGGR
dp83815_dev_reset (AdapterContext *pAdp)
#else    
dp83815_dev_reset (u32 iobase)
#endif    
{
    int  timeout = 100;     /* XXX cleanup: use macro */
#ifdef LINK_AGGR
    u32 iobase = (u32) pAdp->RegAddr;
    pAdp->MacStats.txOkCount = 0;
#endif
    
    DP_REG32_WRITE (DP_CR, DP_CR_RST);
    while (timeout--) {
        if (DP_REG32_READ (DP_ISR) == 0x03008000) {
            return (OK);
        }
        udelay (5);
    }
    return ERROR;
}

/*
 * dp83815_queue_create - create a circular queue of descriptors
 *
 * This routine allocates a descriptor buffer array aligned  on a word
 * boundary, initializes and links the array to make a circular queue.
 */
static status
dp83815_queue_create (struct dp83815_queue *q, int count, int qtype)
{
    virt_addr     desc_addr;
    int           i;
    struct sk_buff *skb;
    
    /* allocate the desc buffer array */
    q->qbuf = (virt_addr) kmalloc (DP_QUEUE_ELE_SIZE * count + DP_ALIGN,
                                   GFP_DMA);
    if (q->qbuf == (virt_addr) NULL)
        return ERROR;

    memset ((char *)q->qbuf, 0, DP_QUEUE_ELE_SIZE * count + DP_ALIGN);

    /* Adjust alignment and Initialize queue data */
    q->cur_desc_addr =
        q->first_desc_addr =
        (virt_addr)(((u32)q->qbuf + DP_ALIGN) & ~(DP_ALIGN - 1));
    q->last_desc_addr = q->first_desc_addr + ((count -1) * DP_QUEUE_ELE_SIZE);
    q->count = count;
    q->read_desc_addr = q->cur_desc_addr;

    /* Initialize each buffer descriptor, and link them into circular queue */
    for (i=0, desc_addr=q->first_desc_addr; i<count;
         i++, desc_addr+=DP_QUEUE_ELE_SIZE) {
        DP_DESC_LNK_XLATE_SET (desc_addr, desc_addr + DP_QUEUE_ELE_SIZE);

        /* Update the size, BUFPTR, and SKBPTR fields for RX descriptors */
        if (qtype == DP_QUEUE_TYPE_RX) {
            skb = alloc_skb (ETH_MAX_PKT_SIZE, GFP_ATOMIC);
            if (skb == NULL) {
                dp83815_queue_delete (q);
                return (ERROR);
            }
            DP_DESC_CMDSTS_XLATE_SET (desc_addr, ETH_MAX_PKT_SIZE);
            DP_DESC_BUFPTR_XLATE_SET (desc_addr, skb->data);
            DP_DESC_SKBPTR_SET (desc_addr, (u32) skb);
        }
    }
    /* Make the queue circular */
    DP_DESC_LNK_XLATE_SET (q->last_desc_addr, q->first_desc_addr);

    return OK;
}

/* dp83815_queue_delete - frees an allocated descriptor queue */
static status
dp83815_queue_delete (struct dp83815_queue *q)
{
    int       i;
    virt_addr desc_addr;
    struct sk_buff *skb;
        
    /* Free all SKBs in the queue */
    for (i=0, desc_addr=q->first_desc_addr;
         (i < q->count);
         i++, desc_addr += DP_QUEUE_ELE_SIZE) {
        skb = (struct sk_buff *) DP_DESC_SKBPTR_GET (desc_addr);
        if (skb != NULL)
            dev_kfree_skb (skb);
    }

    /* Free the queue buffer */
    kfree ((char *)q->qbuf);
    
    return (OK);
}

/*
 * dp83815_tx_desc_get - get a valid transmit descriptor
 *
 * This routine returns the current descriptor from the tx_queue if driver is 
 * the owner, else returns NULL
 */
static virt_addr
dp83815_tx_desc_get (struct net_device *dev)
{
    struct dp83815_queue *   q;
    virt_addr  desc_addr = NULL;

    q = &((struct dp83815_priv *)(dev->priv))->tx_queue;

    /* Check if we own the descriptor */
    if  (((DP_DESC_CMDSTS_XLATE_GET (q->cur_desc_addr) & DP_DESC_CMDSTS_OWN)
          == 0) &&
         (DP_DESC_SKBPTR_GET(q->cur_desc_addr) == (u32)NULL)) {
        desc_addr = q->cur_desc_addr;
        DP_QUEUE_ELE_NEXT (q);                 /* Move to the next element */
    }

    return desc_addr;
}

/* dp83815_tx_skb_reclaim_irq - reclaim SKBs in transmitted descriptors */
static void
dp83815_tx_skb_reclaim_irq (struct net_device *dev, virt_addr desc_addr)
{
    struct sk_buff *       skb;
    struct dp83815_queue * q;

    /* Reclaim buffers from all descriptors we own. */
    q = &((struct dp83815_priv *)(dev->priv))->tx_queue;
    while (((DP_DESC_CMDSTS_XLATE_GET(q->read_desc_addr) & DP_DESC_CMDSTS_OWN) == 0) &&
           ((skb=(struct sk_buff *) DP_DESC_SKBPTR_GET(q->read_desc_addr)) != NULL))
    {
        dev_kfree_skb_irq (skb);
        DP_DESC_SKBPTR_SET (q->read_desc_addr, 0);
        q->read_desc_addr = DP_QUEUE_ELE_NEXT_GET (q, q->read_desc_addr);
    }
}

/*
 * dp83815_rx_desc_get - get a valid receive descriptor
 *
 * This routine returns the current descriptor from the rx_queue if driver is 
 * the owner, else returns NULL
 */
static virt_addr
dp83815_rx_desc_get (struct net_device *dev)
{
    struct dp83815_queue *   q;
    virt_addr                desc_addr = NULL;

    q = &((struct dp83815_priv *)(dev->priv))->rx_queue;

    /* Check if we own the descriptor */
    if  (DP_DESC_CMDSTS_XLATE_GET (q->cur_desc_addr) & DP_DESC_CMDSTS_OWN) {
        desc_addr = q->cur_desc_addr;
        DP_QUEUE_ELE_NEXT(q);                 /* Move to the next element */
    }

    return desc_addr;
}

/* dp83815_phy_setup - reset and setup the PHY device */
static status
dp83815_phy_setup (struct net_device *dev)
{
    u32  iobase = dev->base_addr;
    u32  dp_cfg_val;
    u16  phy_status;
    u16  timeout;
	u16  tmpVal;
	u32  version;

    char strbuf[80];
    struct dp83815_priv *dev_priv = (struct dp83815_priv *)dev->priv;
#ifdef LINK_AGGR
    NsmContext     *pNsm = &((struct dp83815_priv *)(dev->priv))->NsmCtxt;
    AdapterContext *pAdp =  &pNsm->AdpCtxt;
#endif
    if (dev_priv->autoneg == 1)
    {
        dp_cfg_val =  (DP_CFG_PESEL           |     /* parity error detect */
                       DP_CFG_PAUSE_ADV       |     /* pause capable */
                       DP_CFG_PINT_ACEN       |     /* phy intr auto clear */
                       0x00040000);                 /* phy config */

        if ((dev_priv->speed100 == 2) && (dev_priv->fullduplex == 2))
	{
            dp_cfg_val |= DP_CFG_ANEG_SEL_ALL_XD;     /* negotiate 10/100 full/half */
            sprintf (strbuf, "auto negotiate for 10/100 full/half duplex\n");
	}
        if ((dev_priv->speed100 == 2) && (dev_priv->fullduplex == 0))
	{
            dp_cfg_val |= DP_CFG_ANEG_SEL_ALL_HD;     /* negotiate 10/100 half duplex */
            sprintf (strbuf, "auto negotiate for 10/100 half duplex\n");
	}
        if ((dev_priv->speed100 == 0) && (dev_priv->fullduplex == 2))
	{
            dp_cfg_val |= DP_CFG_ANEG_SEL_10_XD;     /* negotiate 10 half/full duplex */
            sprintf (strbuf, "auto negotiate 10 half/full duplex\n");
	}
        if ((dev_priv->speed100 == 1) && (dev_priv->fullduplex == 2))
	{
            dp_cfg_val |= DP_CFG_ANEG_SEL_100_XD;     /* negotiate 100 half/full duplex */
            sprintf (strbuf, "auto negotiate 100 half/full duplex\n");
	}

	DP_DEBUG (DP_DEBUG_OPEN, (KERN_INFO "%s: %s \n",dev->name, strbuf));    

	DP_REG32_WRITE (DP_CFG, dp_cfg_val | DP_CFG_PHY_RST);
	udelay (500);

	DP_REG32_WRITE (DP_CFG, dp_cfg_val);
	for (timeout=10000; timeout; timeout--) {
	    if (DP_REG32_READ (DP_CFG) & DP_CFG_ANEG_DN)
                break;
	    udelay (500);
	}

	if (timeout == 0) {
	    printk (KERN_INFO "Phy Autonegotiate Failed, please check the cable\n");
	}
    } 
    else {
        dp_cfg_val =  (DP_CFG_PESEL           |     /* parity error detect */
                       DP_CFG_PAUSE_ADV       |     /* pause capable */
                       DP_CFG_PINT_ACEN       |     /* phy intr auto clear */
                       0x00040000);                 /* phy config */
    
        if ((dev_priv->fullduplex == 1) && (dev_priv->speed100 == 1))
	{
	    dp_cfg_val |= DP_CFG_ANEG_SEL_100_FD;   /* force 100 & full duplex */
            sprintf (strbuf, "force 100 & full duplex\n");
	}
        else if ((dev_priv->fullduplex != 1) && (dev_priv->speed100 == 1))
        {
	    dp_cfg_val |= DP_CFG_ANEG_SEL_100_HD;   /* force 100 & half duplex */
            sprintf (strbuf, "force 100 & half duplex\n");
	}
	else if ((dev_priv->fullduplex != 1) && (dev_priv->speed100 != 1))
	{
            dp_cfg_val |= DP_CFG_ANEG_SEL_10_HD;    /* force 10 & half duplex */
            sprintf (strbuf, "force 10 & half duplex\n");
        }
        else if ((dev_priv->fullduplex == 1) && (dev_priv->speed100 != 1))
	{
            dp_cfg_val |= DP_CFG_ANEG_SEL_10_FD;    /* force 10 & full duplex */
            sprintf (strbuf, "force 10 & full duplex\n");
	}

	DP_DEBUG (DP_DEBUG_OPEN, (KERN_INFO "%s: %s",dev->name, strbuf));    

	DP_REG32_WRITE (DP_CFG, DP_CFG_PHY_RST);
        udelay (1000);	

	DP_REG32_WRITE (DP_CFG, dp_cfg_val);     
    }

    /**** enable the phy interrupt ****/
    DP_REG16_WRITE (DP_MICR, DP_MICR_PHY_INT);
    
#if 0
    DP_REG32_WRITE (DP_PHY_PAGE, DP_PHY_PAGE_VAL);
    DP_REG32_WRITE (DP_PHYCR, DP_PHYCR_PMDCSR_VAL);
    DP_REG32_WRITE (DP_PHY_TDATA, DP_PHY_TDATA_VAL);
    DP_REG32_WRITE (DP_PHY_DSPCFG, DP_PHY_DSPCFG_VAL);
    DP_REG32_WRITE (DP_PHY_SDCFG, DP_PHY_SDCFG_VAL);
    DP_REG32_WRITE (DP_PHY_PAGE, (u16) 0x0000);
#endif
	/*IP_0521*/
	version = DP_REG32_READ(DP_SRR);

	if ((version & DP_SRR_MAJ) == 0x0300)	/* For Rev C devices the DP_SRR is 3xxh */
	{
    DP_REG32_WRITE (DP_PHY_PAGE, DP_PHY_PAGE_VAL);
    DP_REG32_WRITE (DP_PHYCR, DP_PHYCR_PMDCSR_VAL);
    DP_REG32_WRITE (DP_PHY_TDATA, DP_PHY_TDATA_VAL);
    DP_REG32_WRITE (DP_PHY_DSPCFG, DP_PHY_DSPCFG_VAL);
    DP_REG32_WRITE (DP_PHY_SDCFG, DP_PHY_SDCFG_VAL);
    DP_REG32_WRITE (DP_PHY_PAGE, (u16) 0x0000); /* from fa311.c driver */
	}
	else if ( ((version & DP_SRR_MAJ) == 0x0400) || ((version & DP_SRR_MAJ) == 0x0500) )	
		/* For Rev D devices the DP_SRR is 4xxh */
		/* For Rev A dp83816 device DP-SRR is 5xxh */
	{
    DP_REG32_WRITE (DP_PHY_PAGE, DP_PHY_PAGE_VAL);
    DP_REG32_WRITE (DP_PHYCR, DP_PHYCR_PMDCSR_VAL);
    DP_REG32_WRITE (DP_PHY_PAGE, (u16) 0x0000); 
	}


                        
    phy_status = DP_REG16_READ (DP_PHYSTS);
    printk (KERN_INFO "%s: speed=%d duplex=%s link=%s\n",
            dev->name,
            (phy_status & DP_PHYSTS_SPEED_10) ? 10 : 100,
            (phy_status & DP_PHYSTS_FDX)? "full" : "half",
            (phy_status & DP_PHYSTS_LNK_VALID) ? "up" : "down");

#ifdef LINK_AGGR
    if (phy_status & DP_PHYSTS_SPEED_10)
        pAdp->MediaSpeed = SPEED10;
    else
        pAdp->MediaSpeed = SPEED100;
    if (phy_status & DP_PHYSTS_FDX)
        pAdp->DuplexMode = FULL_DUPLEX;
    else
        pAdp->DuplexMode = HALF_DUPLEX;
    if (phy_status & DP_PHYSTS_LNK_VALID)
        pAdp->AdapterStatus = LINK_UP;
    else
        pAdp->AdapterStatus &= ~LINK_UP;
#endif

    // Add by Bruce to do the PHY coefficient check
    if( !(phy_status & DP_PHYSTS_SPEED_10)){
	    DP_REG32_WRITE(DP_PHY_PAGE, DP_PHY_PAGE_VAL);
	    tmpVal = DP_REG32_READ(DP_PHY_DSPCFG);
	    tmpVal &= 0xFFF;
	    DP_REG32_WRITE(DP_PHY_DSPCFG,(tmpVal | 0x1000));
	    udelay(100);
	    tmpVal = DP_REG32_READ(DP_PHY_TDATA);
	    tmpVal &= 0x00FF;
	    if( !(tmpVal &0x0080) || ((tmpVal >=0x00D8)&&(tmpVal <= 0x00FF))){
		    DP_REG32_WRITE(DP_PHY_TDATA,0x00E8);
		    tmpVal = DP_REG32_READ(DP_PHY_DSPCFG);
		    DP_REG32_WRITE(DP_PHY_DSPCFG,(tmpVal | 0x0020));
	    }
    	    DP_REG32_WRITE(DP_PHY_PAGE,0);
    }

    
    return (OK);
}

#if 0
/* swap_16 - swap a 16 bit value between little endian & big endian byteorder */
static u16 swap_16 (u16 us)
{
    return ((us << 8) & 0xff00) | ((us >> 8) & 0x00ff);
}
#endif

#if 0 /* Not used */
/* swap_32 - swap a 32 bit value between little endian & big endian byteorder */
static u32 swap_32 (u32 ui)
{
    return ((swap_16 (ui & 0x0000ffff) << 16) | swap_16 (ui >> 16));
}
#endif

/* io_read_16 - read the 16 bit value stored at given io address */
static u16 io_read_16 (u16 io_port)
{
    return inw (io_port);
}

/* io_write_16 - write the 16 bit data at given io address */
static void io_write_16 (u16 io_port, u16 data)
{
    outw (data, io_port);
}

/* io_read_32 - read the 32 bit value stored at given io address*/
static u32 io_read_32 (u16 io_port)
{
    return inl (io_port);
}

/* io_write_32 - write the 32 bit data at given io address */
static void io_write_32 (u16 io_port, u32 data)
{
    outl (data, io_port);
}

#ifdef DEBUG
#if 0 /* not used */
/* dp83815_desc_info - print info on descriptors
 *
 * This routine displays values of the elements of dp83815_queue;
 * if option is -1, prints info on all the descs and their field vals;
 * and in all other cases prints info on the  specified descriptor.
 */

static void dp83815_desc_info (struct dp83815_queue * q, int option)
{
    virt_addr    desc_addr;
    u32          i;

    /* Display info header */
    printk ( KERN_DEBUG "dp83815_desc_info(): q=0x%x, "
             "first_desc_addr=0x%x, last_desc_addr=0x%x cur_desc_addr=0x%x\n",
             (u32)q,  (u32)q->first_desc_addr,
             (u32)q->last_desc_addr, (u32)q->cur_desc_addr);

    /* Print requested element info */
    switch (option) {
    case -1:                                 /* all elements */
        desc_addr = q->first_desc_addr;
        for (i=0; i<q->count; i++) {
            printk (KERN_DEBUG "desc_addr=0x%x link=0x%x "
                    "cmdsts=0x%x bufptr=0x%x\n",
                    (u32)desc_addr, DP_DESC_LNK_GET (desc_addr),
                    DP_DESC_CMDSTS_GET (desc_addr),
                    DP_DESC_BUFPTR_GET (desc_addr));
            desc_addr += DP_QUEUE_ELE_SIZE;
        }
        break;
        
    default:                                 /* a specific element */
        if (option > q->count) {
            printk (KERN_DEBUG "Can't print info for ele=%d when count=%d\n",
                    option, q->count);
        }
        else {
            desc_addr = q->first_desc_addr + (DP_QUEUE_ELE_SIZE * (option - 1));
            printk (KERN_DEBUG "ele=%d desc_addr=0x%x link=0x%x "
                    "cmdsts=0x%x bufptr=0x%x\n",
                    option, (u32)desc_addr, DP_DESC_LNK_GET (desc_addr),
                    DP_DESC_CMDSTS_GET (desc_addr),
                    DP_DESC_BUFPTR_GET (desc_addr));
        }
    }
}


/* dp83815_regs_info - prints values of registers */
static void dp83815_regs_info (u16 iobase)
{
    printk (KERN_INFO "dp(0x%x): CFG=0x%x IMR=0x%x IER=0x%x\n",
            iobase, DP_REG32_READ (DP_CFG), DP_REG32_READ (DP_IMR),
            DP_REG32_READ (DP_IER));
    printk (KERN_INFO " ++ TXDP=0x%x TXCFG=0x%x RXDP=0x%x RXCFG=0x%x\n",
            DP_REG32_READ (DP_TXDP), DP_REG32_READ (DP_TXCFG),
            DP_REG32_READ (DP_RXDP), DP_REG32_READ (DP_RXCFG));
}
#endif /* not used */
#endif

#ifdef MODULE

/*
 * init_module - Initialize the driver module
 *
 * This routine is called to initialize the driver when the module in
 * installed into a running kernel via insmod.
 *
 */

int __init
dp83815_init_module (void)
{
    printk (KERN_INFO "%s", version);
    return dp83815_probe (NULL);
}

/*
 * cleanup_module - Deinstall the driver module.
 *
 * This routine is called when removing the driver module from a running
 * kernel via rmmod.
 *
 */

void __exit
dp83815_cleanup_module (void)
{
    struct net_device *cur_dev;

#ifdef LINK_AGGR
    NsmContext *pNsm;
    AdapterContext *pAdp;
#endif

    cur_dev=dp83815_dev_list;
    while (cur_dev) {
        dp83815_dev_list = ((struct dp83815_priv *)(cur_dev->priv))->next;

#ifdef LINK_AGGR
        pNsm = &((struct dp83815_priv *)(cur_dev->priv))->NsmCtxt;
        pAdp = &(pNsm->AdpCtxt);

        if (pAdp->nFlags)
            LacpUnInitialize (pAdp);
#endif
        
        unregister_netdev (cur_dev);
        release_region (cur_dev->base_addr, DP83815_PCI_IO_SIZE);
        kfree (cur_dev);
        cur_dev = dp83815_dev_list;
    }
}


module_init (dp83815_init_module);
module_exit (dp83815_cleanup_module);
#endif /* MODULE */

#ifdef LINK_AGGR

static void
dp83815_rx_control (AdapterContext *pAdp, int flag)
{
    unsigned int cr_val;
    int timeout = 50000;
    struct net_device *dev = ((NsmContext *)(pAdp->pNsmContext))->dev;
    u32 iobase = (u32) pAdp->RegAddr;
    
    cr_val = DP_REG32_READ (DP_CR);

    if (flag == DP_RX_DISABLE)
    {
        cr_val |= (DP_CR_RXD & ~DP_CR_RXE);
        DP_REG32_WRITE (DP_CR, cr_val);
        cr_val |= (DP_CR_RXR & ~DP_CR_RXE);
        DP_REG32_WRITE (DP_CR, cr_val);

    }

    if (flag == DP_RX_ENABLE)
    {
        cr_val |= (DP_CR_RXR & ~DP_CR_RXE);
        DP_REG32_WRITE (DP_CR, cr_val);

        while (timeout--) {
            if (DP_REG32_READ (DP_ISR) & 0x01000000)
                break;
            udelay (100);
        }

        dp83815_start_receive (dev);
        
        cr_val = DP_REG32_READ (DP_CR);

        cr_val |= (DP_CR_RXE & ~DP_CR_RXR);
        DP_REG32_WRITE (DP_CR, cr_val);
    }
}

static unsigned char
dp83815_rxfilter (AdapterContext *pAdapter, UINT rxFilterFlag)
{
    u32 iobase = (u32) pAdapter->RegAddr;
    unsigned int rfcr_flag = 0;

    rfcr_flag |= DP_RFCR_RFEN;

    if (rxFilterFlag & PROMISCUOUS_ON)
        rfcr_flag |= (DP_RFCR_AAU | DP_RFCR_AAM | DP_RFCR_AAB);        

    if (rxFilterFlag & PROMISCUOUS_OFF)
        rfcr_flag &= ~(DP_RFCR_AAU | DP_RFCR_AAM);                

    if (rxFilterFlag & ACCEPT_ALL_MULTICAST_ON)
        rfcr_flag |= DP_RFCR_AAM;        

    if (rxFilterFlag & ACCEPT_ALL_UNICAST_ON)
        rfcr_flag |= DP_RFCR_AAU;

    if (rxFilterFlag & ACCEPT_ALL_BROADCAST_ON)
        rfcr_flag |= DP_RFCR_AAB;

    if (rxFilterFlag & MULTICAST_HASH_ENABLE)
        rfcr_flag |= DP_RFCR_MHEN;

    if (rxFilterFlag & ACCEPT_PERFECT_MATCH_ON)
        rfcr_flag |= DP_RFCR_APM;

    if (rxFilterFlag & ACCEPT_ALL_MULTICAST_OFF)
        rfcr_flag &= ~DP_RFCR_AAM;

    DP_REG32_WRITE (DP_RFCR, rfcr_flag);    

	pAdapter->lacprxFilterFlag = rxFilterFlag;

	if (pAdapter->nFlags)
		LacpSetRxFilter(pAdapter, rxFilterFlag);

    return OK;
}

static unsigned char
dp83815_setphy (AdapterContext *pAdapter, UINT PhysFlags, UINT SetSpeed)
{
    return OK;
}

static void dp83815_IndicateLinkStatus (AdapterContext *pAdapter)
{
    return;
}

static unsigned char
dp83815_multicast_add (AdapterContext *pAdp, MultiList *pMulti, UCHAR  ReloadFlag)
{

    unsigned int crc;
    unsigned short wordindex;
    unsigned short bitindex;
    unsigned short hashbit;
    mca_hash_t *hashEntry, *prevEntry;
    int found = FALSE;
    int numaddr = 0;
    int cnt = 0;
    u32 iobase = (u32) pAdp->RegAddr;
    MultiList *tempMulti = pMulti;

    if (ReloadFlag == TRUE)
    {
        dp83815_clear_mca_tbl (pAdp);
    }
    
    while (pMulti)
    {
        
        crc = dp83815_crc(pMulti->MulticastAddr);    
        wordindex = ( crc >> 4 ) & 0x0000001f;
        bitindex = ( crc & 0xfL );
        hashbit = ( 1 << ( crc & 0x0000000f ));
        hashEntry = pAdp->mca_tbl[wordindex][bitindex];

        while ( hashEntry != NULL ) {

            if(strncmp(hashEntry->MulticastAddr, pMulti->MulticastAddr, 6) == 0)
            {
                found = TRUE;
                break;
            }
            prevEntry = hashEntry;
            hashEntry = hashEntry->next;
        }
        
        if (found == FALSE)
        {
            hashEntry = (mca_hash_t *)kmalloc(sizeof(mca_hash_t), GFP_KERNEL);
            if (hashEntry == NULL)
            {
#ifdef LINK_AGGR
                if (numaddr && pAdp->nFlags)
                    LacpMulticastAdd(pAdp, tempMulti, numaddr, ReloadFlag);
#endif
                return FAILURE;
            }
            memcpy(hashEntry->MulticastAddr, pMulti->MulticastAddr, 6);
            hashEntry->next = pAdp->mca_tbl[wordindex][bitindex];
            pAdp->mca_tbl[wordindex][bitindex] = hashEntry;

            cnt++;     /* keep track of count */

            DP_REG32_WRITE (DP_RFCR, DP_RFCR_RFADDR_FMEM_LO + wordindex*2);
            DP_REG32_WRITE (DP_RFDR, (u32) hashbit);     
        }
#ifdef LINK_AGGR
        numaddr++;
#endif        
        pMulti = pMulti->Next;
    }

        
    if (cnt != 0)
        DP_REG32_WRITE (DP_RFCR, DP_RFCR_MHEN | DP_RFCR_RFEN | DP_RFCR_APM | DP_RFCR_AAB);
    else
        DP_REG32_WRITE (DP_RFCR, DP_RFCR_RFEN);        
        
#ifdef LINK_AGGR
    /* 
     * Invoke LACP callback which adds the multicast addresses to the rest 
     * of the members in aggregation if this is an aggregator port
     */
    if ((numaddr || ReloadFlag) && pAdp->nFlags)
        LacpMulticastAdd(pAdp, tempMulti, numaddr, ReloadFlag);
#endif        
        
    /*     release the lock */
    return SUCCESS;            
}

static unsigned char
dp83815_multicast_delete (AdapterContext* pAdp, MultiList *pMulti)
{

    unsigned short hashbit, wordindex, bitindex;
    mca_hash_t *hashEntry;
    MultiList *tempMulti = pMulti;
    u32 hashValue;
    mca_hash_t *prevEntry = NULL;    
    u32 iobase = (u32) pAdp->RegAddr;
    unsigned char found = FALSE;
    unsigned int crc;
    
    while (pMulti)
    {
        crc = dp83815_crc(pMulti->MulticastAddr);    
        wordindex = ( crc >> 4 ) & 0x0000001f;
        bitindex = ( crc & 0xfL );
        hashbit = ( 1 << ( crc & 0x0000000f ));
        hashEntry = pAdp->mca_tbl[wordindex][bitindex];

        while ( hashEntry != NULL ) {

            if(strncmp(hashEntry->MulticastAddr, pMulti->MulticastAddr, 6) == 0)
            {
                found = TRUE;
                break;
            }
            prevEntry = hashEntry;
            hashEntry = hashEntry->next;
        }
        
        if (found == TRUE)
        {
            if (pAdp->mca_tbl[wordindex][bitindex] == hashEntry)
                pAdp->mca_tbl[wordindex][bitindex] = hashEntry->next;
            else
                prevEntry->next  = hashEntry->next;

            kfree (hashEntry);

            if (pAdp->mca_tbl[wordindex][bitindex] == NULL)
            {
                DP_REG32_WRITE (DP_RFCR, 0x100 + (wordindex * 2));
                hashValue = DP_REG32_READ (DP_RFCR);
                hashValue &= ~hashbit;
                DP_REG32_WRITE (DP_RFDR, hashValue);
            }
        }
        else
            return FAILURE;
        pMulti = pMulti->Next;
    }
    if (tempMulti  && pAdp->nFlags)
        LacpMulticastDelete(pAdp, tempMulti);


    return SUCCESS;    
}

static void
dp83815_clear_mca_tbl(AdapterContext *pAdapter)
{
    INT i, j;
    mca_hash_t *start, *next;
    u32   iobase = (u32) pAdapter->RegAddr;

    for ( i = 0; i < 128; ++i) {
        
        for ( j = 0; j < 16; ++j ) {
            start = pAdapter->mca_tbl[i][j];
            while ( start ) {
                next = start->next;
                /* The following  lines added To fix HCT failure in NDIS5 */
                DP_REG32_WRITE (DP_RFCR, 0x100 + (i * 2));
                /*All 32 bits*/
                DP_REG32_WRITE (DP_RFDR, 0x0);
                /* End */
                kfree(start);
                start = next;
            }
            pAdapter->mca_tbl[i][j] = NULL;
        }

    }


}

#endif


/*
 * Local variables:
 *  compile-command: "gcc -D__KERNEL__ -DMODULE -Wall -Wstrict-prototypes -O -m486 -c dp83815.c"
 *  version-control: t
 *  kept-new-versions: 5
 * End:
 */
