  //     RADIO BRIDGE PACKET DECODER v1.5
// (c)2022 Radio Bridge USA by John Sheldon and Timothy Throop
////////////////////////////////////////////
// NOTE: This decoder is not officially supported and is provided "as-is" as a developer template.
// Multitech / Radio Bridge make no guarantees about use of this decoder in a production environment.
////////////////////////////////////////////

// SUPPORTED SENSORS:
//   RBS301 Series
//   RBS304 Series
//   RBS305 Series
//   RBS306 Series (except RBS306-VSHB & RBS306-CMPS)

// VERSION 1.2 NOTES:
//   Changed output to JSON
//   Various bug fixes for decodes
//   Compatible with TTNv3
//   Bug fix for thermocouple temperature to 2 decimal places

// VERSION 1.3 NOTES:
//  Removed obsolete code and comments
//  Added RBS301-TEMP-INT Sensor

// VERSION 1.4 NOTES:
//  Fixed Reset Event function
// Added comments to clarify the various temperature probe types
// Cleaned up RESET event

// VERSION 1.4.1 NOTES:
//  Add Gen 3 Sensors

// VERSION 1.5 NOTES:
// Added Australia Sensor models in the notes. No functional changes to the decoder.

// If using the sensors listed below, sensor definition files for those sensors must be included with the distribution of this file 
// and must reside in the same directory as the radio_bridge_decoder.js file.


// New sensor definition files include:

// RBS3010xx22BN00
// RBS3010NA22BN00 - temp/humidity/water probe: reports events 0D,08
// RBS3010EU22BN00 - UK/EU version of above
// RBS3010AU22BN00 - Australia version of above

// RBS3010xx0ABN00 
// RBS3010NA0ABN00 - water probe: reports events 08
// RBS3010EU0ABN00  - UK/EU version of above
// RBS3010AU0ABN00  - Australia version of above

// RBS3010xx22BN08
// RBS3010NA22BN08- temp/humidity/1m water rope: reports events 0D,08
// RBS3010EU22BN08 - UK/EU version of above
// RBS3010AU22BN08 - Australia version of above

// RBS3010xx19BN00
// RBS3010NA19BN00 - Indoor probeless temp sensor: reports event 19
// RBS3010EU19BN00 - UK/EU version of above
// RBS3010AU19BN00 - Australia version of above
// The contents of the sensor definition file for this sensor is the same as the gen 2 version of this sensor (RBS301-INT-TEMP)

// RBS3010xx05BN00
// RBS3010NA05BN00 - Indoor external probe temp sensor: reports event 09
// RBS3010EU05BN00 - UK/EU version of above
// RBS3010AU05BN00 - Australia version of above
// The contents of the sensor definition file for this sensor is the same as the gen 2 version of this sensor (RBS301-EXT-TEMP)

// RBS3010xx0EBN00 
// RBS3010NA0EBN00 - Indoor probeless temp/humidity sensor: reports event 0D
// RBS3010EU0EBN00 - UK/EU version of above
// RBS3010AU0EBN00 - Australia version of above
// The contents of the sensor definition file for this sensor is the same as the RBS306-ATH


// RBS3010xx08BN00
// RBS3010NA08BN00 - Indoor movement sensor: reports event 0E
// RBS3010EU08BN00 - UK/EU version of above
// RBS3010AU08BN00 - Australia version of above
// The contents of the sensor definition file for this sensor is the same as the RBS306-ATH

// RBS3010NA09BN00 - Indoor tilt sensor: reports event 0A
// RBS3010EU09BN00 - UK/EU version of above
// RBS3010AU09BN00 - Australia version of above
// The contents of the sensor definition file for this sensor is the same as the RBS301-TILT

// RBS3010xx01BN00
// RBS3010NA01BN00 - DWS: reports event 03
// RBS3010EU01BN00 - UK/EU version of above
// RBS3010AU01BN00 - Australia version of above
// The contents of the sensor definition file for this sensor is the same as the RBS301-DWS

// RBS3010xx03BN00
// RBS3010NA03BN00 - Contact Sensor: reports event 07
// RBS3010EU03BN00 - UK/EU version of above
// RBS3010AU03BN00 - Australia version of above
// The contents of the sensor definition file for this sensor is the same as the RBS301-CON

// RBS3010xx0ABN08
// RBS3010NA0ABN08 - 1m Water Rope: reports event 08
// RBS3010EU0ABN08 - UK/EU version of above
// RBS3010UA0ABN08 - Australia version of above
// The contents of the sensor definition file for this sensor is the same as the RBS301-WR1M

// RBS3010xx0ABN0B
// RBS3010NA0ABN0B - 5m Water Rope: reports event 08
// RBS3010EU0ABN0B - UK/EU version of above
// RBS3010UA0ABN0B - Australia version of above
// The contents of the sensor definition file for this sensor is the same as the RBS301-WR5M

// RBS3010xx0ABN09 - 10m Water Rope: reports event 08
// RBS3010NA0ABN09 - 10m Water Rope: reports event 08
// RBS3010EU0ABN09 - UK/EU version of above
// RBS3010UA0ABN09 - Australia version of above
// The contents of the sensor definition file for this sensor is the same as the RBS301-WR10M


// General defines used in decode

//  Common Events
var RESET_EVENT               = "00";
var SUPERVISORY_EVENT         = "01";
var TAMPER_EVENT              = "02";
var LINK_QUALITY_EVENT        = "FB";
var DOWNLINK_ACK_EVENT        = "FF";

//  Device-Specific Events
var DOOR_WINDOW_EVENT         = "03";
var PUSH_BUTTON_EVENT         = "06";
var CONTACT_EVENT             = "07";
var WATER_EVENT               = "08";
var RBS301_EXT_TEMP_EVENT     = "09";
//the above event is also known as a thermistor temp event
var TILT_EVENT                = "0A";
var ATH_EVENT                 = "0D";
var ABM_EVENT                 = "0E";
var TILT_HP_EVENT             = "0F";
var ULTRASONIC_EVENT          = "10";
var SENSOR420MA_EVENT         = "11";
var THERMOCOUPLE_EVENT        = "13";
var VOLTMETER_EVENT           = "14";
var RBS301_INT_TEMP_EVENT     = "19";
//the above event is also known as a CMOS temp event
var VIBRATION_HB_EVENT        = "1A";

var decoded = {};

// function called by MultiTech
function decodeUplink(input) {
        return {
                data: Generic_Decoder(input.bytes, input.fPort),
        warnings: [],
        errors: []
    };
}

// The generic decode function called by the above network server specific callbacks
function Generic_Decoder(bytes , port) {
    // data structure which contains decoded messages
    var decode = { data: { Event: "UNDEFINED" }};

    // The first byte contains the protocol version (upper nibble) and packet counter (lower nibble)
    var ProtocolVersion = (bytes[0] >> 4) & 0x0f;
    var PacketCounter = bytes[0] & 0x0f;

    // the event type is defined in the second byte
    var PayloadType = Hex(bytes[1]);

    // the rest of the message decode is dependent on the type of event


    switch (PayloadType) {

        // ==================    RESET EVENT    ====================
        case RESET_EVENT:
           // This will decode the hardware and firmware versions for most sensors.
           // third byte is device type, convert to hex format for case statement

            var EventType = Hex(bytes[2]);
      
            // the hardware version has the major version in the upper nibble, and the minor version in the lower nibble
            var HardwareVersionStr = ((bytes[3] >> 4) & 0x0f) + "." + (bytes[3] & 0x0f);
            var HardwareVersion = bytes[3];
            // the firmware version has two different formats depending on the most significant bit
            var FirmwareFormat = (bytes[4] >> 7) & 0x01;
            // FirmwareFormat of 0 is old format, 1 is new format
            // old format is has two sections x.y
            // new format has three sections x.y.z
            var FirmwareVersionStr = ""
            if (FirmwareFormat === 0)
                FirmwareVersionStr = bytes[4] + "." + bytes[5];
            else
                FirmwareVersionStr = ((bytes[4] >> 2) & 0x1F) + "." + ((bytes[4] & 0x03) + ((bytes[5] >> 5) & 0x07)) + "." + (bytes[5] & 0x1F);

            FirmwareVersion = (bytes[4]  * 256) + bytes[5]

            decode = {
                protocolVersion: ProtocolVersion,
                packetCounter: PacketCounter,
                deviceType: bytes[2],
                firmwareVersion: FirmwareVersionStr,
                hardwareVersion: HardwareVersionStr
            };
            break;
        // ================   SUPERVISORY EVENT   ==================
        case SUPERVISORY_EVENT:
            // note that the sensor state in the supervisory message is being depreciated, so those are not decoded here
            // battery voltage is in the format x.y volts where x is upper nibble and y is lower nibble
            var BatteryLevel = ((bytes[4] >> 4) & 0x0f) + "." + (bytes[4] & 0x0f);
            BatteryLevel = parseFloat(BatteryLevel);
            // the accumulation count is a 16-bit value
            var AccumulationCount = (bytes[9] * 256) + bytes[10];
            // decode bits for error code byte
            var TamperSinceLastReset = (bytes[2] >> 4) & 0x01;
            var CurrentTamperState = (bytes[2] >> 3) & 0x01;
            var ErrorWithLastDownlink = (bytes[2] >> 2) & 0x01;
            var BatteryLow = (bytes[2] >> 1) & 0x01;
            var RadioCommError = bytes[2] & 0x01;

            decode = {
                protocolVersion: ProtocolVersion,
                packetCounter: PacketCounter,
                accumulationCount: AccumulationCount,
                tamperSinceLastReset: TamperSinceLastReset,
                tamperState: CurrentTamperState,
                errorWithLastDownlink: ErrorWithLastDownlink,
                batteryLow: BatteryLow,
                radioCommError: RadioCommError,
                batteryLevel: BatteryLevel
            };
            break;
        // ==================   TAMPER EVENT    ====================
        case TAMPER_EVENT:
            var EventType = bytes[2];
            // tamper state is 0 for open, 1 for closed
            decode = {
                protocolVersion: ProtocolVersion,
                packetCounter: PacketCounter,
                tamperState : EventType
            };
            break;

        // ==================   LINK QUALITY EVENT    ====================
        case LINK_QUALITY_EVENT:
            var CurrentSubBand = bytes[2];
            var RSSILastDownlink = (-256 + bytes[3]); // RSSI is always negative
            var SNRLastDownlink = bytes[4];

            decode = {
                protocolVersion: ProtocolVersion,
                packetCounter: PacketCounter,
                lastDownlinkRSSI: RSSILastDownlink,
                lastDownlinkSNR: SNRLastDownlink,
                currentSubBand: CurrentSubBand
            };
            break;
        // ================  DOOR/WINDOW EVENT  ====================
        case DOOR_WINDOW_EVENT:
            var EventType = bytes[2];
            // 0 is closed, 1 is open
            decode = {
                protocolVersion: ProtocolVersion,
                packetCounter: PacketCounter,
                sensorState: bytes[2]
            };
            break;

        // ===============  PUSH BUTTON EVENT   ===================
        case PUSH_BUTTON_EVENT:
            var EventType = bytes[2];
            var SensorState = bytes[3];
            decode = {
                protocolVersion: ProtocolVersion,
                packetCounter: PacketCounter
              };
            switch (EventType) {
                // 01 and 02 used on two button
                case 0x01:
                    decode.button1Event = SensorState;
                    break;
                case 0x02:
                    decode.button2Event = SensorState;
                    break;
                // 03 is single button
                case 0x03:
                    decode.buttonEvent = SensorState;
                    break;
                // 12 when both buttons pressed on two button
                case 0x12:
                    decode.buttonBothEvent = SensorState;
                    break;
            }
            break;
        // =================   CONTACT EVENT   =====================
        case CONTACT_EVENT:
            var EventType = bytes[2];
            // if state byte is 0 then shorted, if 1 then opened
            decode = {
              protocolVersion: ProtocolVersion,
              packetCounter: PacketCounter,
              contactEvent: EventType,
            };
            break;
        // ===================  WATER EVENT  =======================
        case WATER_EVENT:
            var SensorState = bytes[2];
            var WaterRelative = bytes[3];
            decode = {
              protocolVersion: ProtocolVersion,
              packetCounter: PacketCounter,
              waterPresent: !SensorState,
              waterRelative: WaterRelative,
            };
            break;

        // ================== RBS301 EXTERNAL PROBE TEMPERATURE EVENT ====================
        case RBS301_EXT_TEMP_EVENT:
            var EventType = bytes[2];
            // current temperature reading
            var Temperature = Convert(bytes[3], 0);
            // relative temp measurement for use with an alternative calibration table
            var TempRelative = Convert(bytes[4], 0);

            decode = {
                protocolVersion: ProtocolVersion,
                packetCounter: PacketCounter,
                temperatureEvent: EventType,
                currentTemperature: Temperature,
                relativeMeasurement: TempRelative
            };
            break;

        // ====================  TILT EVENT  =======================
        case TILT_EVENT:
            var EventType = bytes[2];
            console.log("raw event type is " + EventType)
            var TiltAngle = bytes[3];
            decode = {
                protocolVersion: ProtocolVersion,
                packetCounter: PacketCounter,
                tiltEvent: EventType,
                tiltAngle: TiltAngle,
            };
            break;
        //==================== RBS301_INT_TEMP_EVENT ============================
        case RBS301_INT_TEMP_EVENT:
           var EventType = bytes[2];
           //var Temperature = Convert(bytes[3], 0)
           var Temperature = Convert((bytes[3]) + ((bytes[4] >> 4) / 10), 1);
           var TempRelative = Convert(bytes[4], 0);
           decode = {
                protocolVersion: ProtocolVersion,
                packetCounter: PacketCounter,
                temperatureEvent: EventType,
                currentTemperature: Temperature,
                relativeMeasurement: TempRelative,
            };
        break;
        // =============  AIR TEMP & HUMIDITY EVENT  ===============
        case ATH_EVENT:
            var EventType = bytes[2];
            // integer and fractional values between two bytes
            var Temperature = Convert((bytes[3]) + ((bytes[4] >> 4) / 10), 1);
            // integer and fractional values between two bytes
            var Humidity = parseFloat(+(bytes[5] + ((bytes[6]>>4) / 10)).toFixed(1));

            decode = {
                protocolVersion: ProtocolVersion,
                packetCounter: PacketCounter,
                tempHumidityEvent: EventType,
                temperature: Temperature,
                humidity: Humidity
            };
            break;

        // ============  ACCELERATION MOVEMENT EVENT  ==============
        case ABM_EVENT:
            var EventType = bytes[2];
            decode = {
                protocolVersion: ProtocolVersion,
                packetCounter: PacketCounter,
                movementEvent: EventType,
                movementPresent: !EventType,
            };
            break;

        // =============  HIGH-PRECISION TILT EVENT  ===============
        case TILT_HP_EVENT:
            var EventType = bytes[2];
            // integer and fractional values between two bytes
            var TiltHPAngle = +(bytes[3] + (bytes[4] / 10)).toFixed(1);
            var Temperature = Convert(bytes[5], 0);
            decode = {
                protocolVersion: ProtocolVersion,
                packetCounter: PacketCounter,
                tiltEvent: EventType,
                tiltAngle: TiltHPAngle,
                temperature: Temperature
            };
            break;

        // ===============  ULTRASONIC LEVEL EVENT  ================
        case ULTRASONIC_EVENT:
            var EventType = bytes[2];
            // distance is calculated across 16-bits
            var Distance = ((bytes[3] * 256) + bytes[4]);

            decode = {
                protocolVersion: ProtocolVersion,
                packetCounter: PacketCounter,
                ultrasonicEvent: EventType,
                distance: Distance
            };
            break;

        // ================  4-20mA ANALOG EVENT  ==================
        case SENSOR420MA_EVENT:
            var EventType = bytes[2];
            // calculatec across 16-bits, convert from units of 10uA to mA
            var Analog420mA = ((bytes[3] * 256) + bytes[4]) / 100;

            decode = {
                protocolVersion: ProtocolVersion,
                packetCounter: PacketCounter,
                a420mAEvent: EventType,
                a420mAMeasurement: Analog420mA
            };
            break;

        // =================  THERMOCOUPLE EVENT  ==================
        case THERMOCOUPLE_EVENT:
            var EventType = bytes[2];
            // decode is across 16-bits
            var Temperature = +(((bytes[3] * 256) + bytes[4]) / 16).toFixed(2);
            var Faults = bytes[5];
            // decode each bit in the fault byte
            var FaultColdOutsideRange = (Faults >> 7) & 0x01;
            var FaultHotOutsideRange = (Faults >> 6) & 0x01;
            var FaultColdAboveThresh = (Faults >> 5) & 0x01;
            var FaultColdBelowThresh = (Faults >> 4) & 0x01;
            var FaultTCTooHigh = (Faults >> 3) & 0x01;
            var FaultTCTooLow = (Faults >> 2) & 0x01;
            var FaultVoltageOutsideRange = (Faults >> 1) & 0x01;
            var FaultOpenCircuit = Faults & 0x01;
            // Decode faults
            if (FaultColdOutsideRange)    FaultCOR = true; else FaultCOR = false;
            if (FaultHotOutsideRange)     FaultHOR = true; else FaultHOR = false;
            if (FaultColdAboveThresh)     FaultCAT = true; else FaultCAT = false;
            if (FaultColdBelowThresh)     FaultCBT = true; else FaultCBT = false;
            if (FaultTCTooHigh)           FaultTCH = true; else FaultTCH = false;
            if (FaultTCTooLow)            FaultTCL = true; else FaultTCL = false;
            if (FaultVoltageOutsideRange) FaultVOR = true; else FaultVOR = false;
            if (FaultOpenCircuit)         FaultOPC = true; else FaultOPC = false;

            decode = {
                protocolVersion: ProtocolVersion,
                packetCounter: PacketCounter,
                thermocoupleEvent: EventType,
                thermocoupleTemperature: Temperature,
                faultColdOutsideRange: FaultCOR,
                faultHotOutsideRange: FaultHOR,
                faultColdAboveThresh: FaultCAT,
                faultColdBelowThresh: FaultCBT,
                faultTCTooHigh: FaultTCH,
                faultTCTooLow: FaultTCL,
                faultVoltageOutsideRange: FaultVOR,
                faultOpenCircuit: FaultOPC
            };
            break;

        // ================  VOLTMETER EVENT  ==================
        case VOLTMETER_EVENT:
            var EventType = bytes[2];

            // voltage is measured across 16-bits, convert from units of 10mV to V
            var Voltage = ((bytes[3] * 256) + bytes[4]) / 100;

            decode = {
                protocolVersion: ProtocolVersion,
                packetCounter: PacketCounter,
                voltageEvent: EventType,
                voltage: Voltage
            };
            break;

        // ==================   DOWNLINK EVENT  ====================
        case DOWNLINK_ACK_EVENT:
            var EventType = bytes[2];
            decode = {
                protocolVersion: ProtocolVersion,
                packetCounter: PacketCounter,
                downlinkMessageAck: EventType,
            };
            break;

        // end of EventType Case
    }

// return decoded object
  return decode;
}

function Hex(decimal) {
    var decimal = ('0' + decimal.toString(16).toUpperCase()).slice(-2);
    return decimal;
}

function Convert(number, mode) {
    var result = 0;
    switch (mode) {
        // for EXT-TEMP and INT_TEMP
        case 0: if (number > 127) { result = number - 256 } else { result = number };break
        //for ATH temp
        case 1: if (number > 127) { result = -+(number - 128).toFixed(1) } else { result = +number.toFixed(1) };break
    }
    return result;
}

