#!/bin/bash
# This script filters the gpspipe -w
# (json) output from GPSD.  The GPS fix
# is taken from the mode parameter of
# the GPSD TPV object in the json output.
# Values are:
#
# 0  Unseen
# 1  No fix
# 2  2D
# 3  3D
#
# Fix file creation for GPS.
# If GPS reaches expected
# fix level, file GSP_FIXFILE is created.
# This is to know when to set the system
# and hardware clock, and when 
# it is safe to start ntp.
# DEBUG=1 to set debug
NAME=gpsd_ubx_fixed
shopt -s expand_aliases
alias LG="logger -t ${NAME}"
if [[ -r /etc/default/gpsd ]] ; then
  . /etc/default/gpsd
else
  echo "Must configure GPSD requirements"
  exit 1
fi

function rm_gps_file
{
  if [[ -n "${GPS_FIXFILE}" ]] && [[ -f ${GPS_FIXFILE} ]] ; then
    rm -f "${GPS_FIXFILE}"
  fi
}
function kill_it
{
    cmd=$1
    pid=$2
    sig=$3
    ((DEBUG)) && LG -p user.info "terminating $cmd(${pid}) with SIG${sig}"
    kill -SIG${sig} ${pid} >/dev/null 2>&1
    result=$?                                                                                                                                                                                      
    ((DEBUG)) && LG -p user.info "kill -SIG${sig} ${pid} result is $result"  
}        

# Most exits are errors, so remove the GPS fix file.
trap rm_gps_file EXIT

if ! [[ -x /usr/bin/gpspipe ]] ; then
  LG -p user.err "Please install gpspipe"
  exit 0
fi

# Terminate gpspipe after 10 seconds if it is our child
# gpspipe has issues with terminals and stalling on terminal
# read of the console.
(
  sleep 10
  # We use BASHPID because our parent, $$, may have already exited, and
  # then ps will not work.  BASHPID is the current subshell.
  ppid=$BASHPID
  pgid=$(ps --no-heading -o pgid -p $ppid)
  
  if ((${#pgid} == 0)) ; then
    LG -p user.err "Could not find a pgid for $ppid"
    ps -fjp $ppid | LG -p user.error
  fi

  # pgrep does not work during boot.
  pgrpmemb=$(ps --no-heading -o pgid,pid,comm -e | grep '^ *'"${pgid}"' *' | sed -r 's/^ *[0-9]+ *//')
  gpspipepid=$(echo "${pgrpmemb}" | egrep '[[:space:]]gpspipe$' | sed 's/[[:space:]].*//')

  ((DEBUG)) && LG -p user.info "$pgid is pgid $gpspipepid is gpspipepid"
  # Find the gpspipepid that is our grandchild
  ((didkill=0))
  
  # for loop is in case gpspipepid has children.
  for p in ${gpspipepid} ; do
    if ((${#p})) ; then
        kill_it gpspipe $p TERM
        ((didkill=p))
        break
    fi
  done
  if ((didkill == 0)) ; then
    exit 0
  fi
  # Do a sigkill to be sure.
  sleep 2
  kill_it gpspipe $didkill KILL
) &

fix=""
status=""

# egrep in busybox has a horrible buffering issue.
# echo My shell pid is $$
# echo Place data fix, status
fix="$(gpspipe -w | grep -m1 '{"class":"TPV","device":"/dev/gps0",.*"mode":'  | sed -e 's/.*"mode"://' -e 's/[},].*//')"

if ((${#fix} == 0)) ; then
  LG -p user.err "No data from GPS"
  exit 1
fi


# echo "status is $status.  fixOK is $fixOK"


# echo "fix is $fix.   Is it in $GPSFIX?"
# Test the GPS fixOK
for x in $GPSFIX ; do
  # echo test $x with $fix
  if [[ $x == $fix ]] ; then
    LG -p user.info "GPS has fix $fix found in list GPSFIX: $GPSFIX"
    echo $x >"${GPS_FIXFILE}"
    GPS_FIXFILE=""
    exit 0
  fi
done

LG -p user.info "GPS fix is bad: $fix and should be one of: $GPSFIX"
# Start later
exit 1
