#!/bin/bash
# Radio ID loop count
TRY_COUNT=30

CONFIG_PATH=/var/run/config

#system capabilities
CAPABILITY_ADC="false"
CAPABILITY_BLUETOOTH="false"
CAPABILITY_DIN="false"
CAPABILITY_DOUT="false"
#CAPABILITY_EXTERNAL_SERIAL_PORT="false"
#CAPABILITY_FLEXIBLE_SERIAL_PORT="false"
CAPABILITY_GPS="false"
CAPABILITY_WIFI="false"
CAPABILITY_CELL="false"
CAPABILITY_LORA="false"
CAPABILITY_GPIO="false"
CAPABILITY_SERIAL="false"
CAPABILITY_RS232="false"
CAPABILITY_RS422="false"
CAPABILITY_RS485="false"
CAPABILITY_NODE_RED="false"
CAPABILITY_LORA_NETWORK_SERVER="false"


# accessory cards constans
TEMP_AC_PATH="/var/volatile/tmp"
TEMP_AC_JSON="$TEMP_AC_PATH/ac_data.json"
# Number of accessory card places - size of array
AC_CARD_COUNT=4

function usage {
	echo 'get-eeprom-device-info [-t TRY_COUNT]'
        echo '  Where TRY_COUNT is the number of half'
        echo '  second waits for the cellular radio'
        echo '  to be ready for identification'
        exit 1
}

# Reads accessory cards data
# WARNING: Make sure function called after CAPABILITIY_* variables is set with initial values.
ReadAcData() {
  # Doing the same as mts-io-sysfs, but with one step, without unnesesary parsing.
  MTS_IO_DIR="/sys/devices/platform/mts-io"
  AC_PRFIX="ap"
  TEMP_AC_FILE_PREFIX="/card"
  TEMP_AC_PREFIX="$TEMP_AC_PATH$TEMP_AC_FILE_PREFIX"
  # Fill property data for each accessory card
  rm -f $TEMP_AC_PREFIX*
  for f in `find $MTS_IO_DIR/$AC_PRFIX* -follow -maxdepth 1 -type f -follow -maxdepth 1 -type f 2>/dev/null`;
  do
    FILENAME=${f##*$MTS_IO_DIR/$AC_PRFIX}
    case $FILENAME in
      modalias )
        ;;
      power )
        ;;
      subsystem* )
        ;;
      uevent )
        ;;
      * )
        AC_NUMBER=$(echo "$FILENAME" | cut -d "/" -f1)
        AC_PROP=$(echo "$FILENAME" | cut -d "/" -f2)
        # Do camel casing
        # Because {sed -r '{s/-([a-z])/\U\1/g;}'} did not work for Not GNU sed 4.0 use awk + tr
        # Removing next 2 lines gives about 50% function speed-up
        AC_PROP=$( echo $AC_PROP | awk -F"-" '{for(j=1;j<=NF;j++){$j=toupper(substr($j,1,1))substr($j,2)}}1' | sed -r '{s/ //g;}' )
        AC_PROP=`echo ${AC_PROP:0:1} | tr '[A-Z]' '[a-z]'`${AC_PROP:1}
        #Output to temp file
        AC_FILE="$TEMP_AC_PREFIX$AC_NUMBER"
        printf "\"$AC_PROP\" : \"%s\",\n" `cat $f` >> $AC_FILE
        ;;
    esac
  done
  # Compose accessory cards data to array
  echo "\"accessoryCards\" : [ " > $TEMP_AC_JSON
  for ((I=1; I<=$AC_CARD_COUNT; ++I)) ;
  do
    AC_FILE="$TEMP_AC_PREFIX$I"
    if [ -f "$AC_FILE" ];then
      sed -i '$s/,$//' $AC_FILE # remove last comma
      echo "{" >> $TEMP_AC_JSON
      cat $AC_FILE >> $TEMP_AC_JSON
      echo "}," >> $TEMP_AC_JSON

      # Map product-id with capabilities.
      if $(grep -q '"productId" : "MTAC-MFSER-DTE"' $AC_FILE) || $(grep -q '"productId" : "MTAC-MFSER-DCE"' $AC_FILE); then
        CAPABILITY_SERIAL="true"
        CAPABILITY_RS232="true"
        CAPABILITY_RS422="true"
        CAPABILITY_RS485="true"
      elif $(grep -q '"productId" : "MTAC-GPIOB"' $AC_FILE) || $(grep -q '"productId" : "MTAC-GPIOI"' $AC_FILE); then
        CAPABILITY_GPIO="true"
        CAPABILITY_ADC="true"
        CAPABILITY_DIN="true"
        CAPABILITY_DOUT="true"
      elif $(grep -q '"productId" : "MTAC-LORA-.*"' $AC_FILE); then
        CAPABILITY_LORA="true"
      fi
    else
      echo "null," >> $TEMP_AC_JSON
    fi
  done
  sed -i '$s/,$//' $TEMP_AC_JSON # remove last comma
  echo "]," >> $TEMP_AC_JSON
}

#check if LORA is present onboard a device.
#Lora capability from an MTAC card is determined by ReadAcData()
isNativeLoraPresent() {
  LORA_ID_PREFIX="MTCAP-LORA-"
  LORA_ID=$(mts-io-sysfs show lora/product-id 2> /dev/null)
  if [[ $LORA_ID == $LORA_ID_PREFIX* ]]; then
    return 0
  else
    return 1
  fi
}

TEMP=$(getopt -o t: -n 'get-eeprom-device-info' -- "$@")
eval set -- "$TEMP"
# extract options and their arguments into variables.
while true ; do
    case "$1" in
        -t)
            case "$2" in
                "") usage ;;
                *) TRY_COUNT=$2 ; shift 2 ;;
            esac ;;
        --) shift ; break ;;
        *) echo "usage" ;;
    esac
done

# Read device information from eeprom device

mts-id-eeprom --in-file /sys/bus/i2c/devices/i2c-0/0-0056/eeprom > ${CONFIG_PATH}/device_info

DEVICE_CAPA=${CONFIG_PATH}/device_capa
DEVICE_INFO_JSON=${CONFIG_PATH}/device_info.json

# Create json data files
echo "{" > $DEVICE_CAPA
echo "{" > ${CONFIG_PATH}/device_info.json


# Read each line and parse the device data
while read line
do

if [[ $line =~ ^vendor-id:\ \"(.*)\" ]]; then
    echo ${BASH_REMATCH[1]} > ${CONFIG_PATH}/vendor_id
    echo \"vendorId\": \"${BASH_REMATCH[1]}\", >> $DEVICE_INFO_JSON
elif [[ $line =~ ^product-id:\ \"(.*)\" ]]; then
    echo ${BASH_REMATCH[1]} > ${CONFIG_PATH}/device_type
    echo \"productId\": \"${BASH_REMATCH[1]}\", >> $DEVICE_INFO_JSON
elif [[ $line =~ ^device-id:\ \"(.*)\" ]]; then
    echo ${BASH_REMATCH[1]} > ${CONFIG_PATH}/device_id
    echo \"deviceId\": \"${BASH_REMATCH[1]}\", >> $DEVICE_INFO_JSON
elif [[ $line =~ ^hw-version:\ \"(.*)\" ]]; then
    echo ${BASH_REMATCH[1]} > ${CONFIG_PATH}/hw_version
    echo \"hardwareVersion\": \"${BASH_REMATCH[1]}\", >> $DEVICE_INFO_JSON
elif [[ $line =~ ^mac-addr:\ \"(.*)\" ]]; then
    echo ${BASH_REMATCH[1]} > ${CONFIG_PATH}/mac_addr
    echo \"macAddress\": \"${BASH_REMATCH[1]}\", >> $DEVICE_INFO_JSON
elif [[ $line =~ ^mac-bluetooth:\ \"(.*)\" ]]; then
    echo ${BASH_REMATCH[1]} > ${CONFIG_PATH}/mac_bluetooth
    echo \"macBluetooth\": \"${BASH_REMATCH[1]}\", >> $DEVICE_INFO_JSON
elif [[ $line =~ ^mac-wifi:\ \"(.*)\" ]]; then
    echo ${BASH_REMATCH[1]} > ${CONFIG_PATH}/mac_wifi
    echo \"macWifi\": \"${BASH_REMATCH[1]}\", >> $DEVICE_INFO_JSON
elif [[ $line =~ ^uuid:\ \"(.*)\" ]]; then
    echo ${BASH_REMATCH[1]} > ${CONFIG_PATH}/uuid
    echo \"uuid\": \"${BASH_REMATCH[1]}\", >> $DEVICE_INFO_JSON
elif [[ $line =~ ^imei:\ \"(.*)\" ]]; then
    echo ${BASH_REMATCH[1]} > ${CONFIG_PATH}/modem_imei
    echo \"imei\": \"${BASH_REMATCH[1]}\", >> $DEVICE_INFO_JSON
elif [[ $line =~ capa-gps:\ (.*) ]]; then
    CAPABILITY_GPS=$(echo ${BASH_REMATCH[1]})
elif [[ $line =~ capa-din:\ (.*) ]]; then
    CAPABILITY_DIN=$(echo ${BASH_REMATCH[1]})
elif [[ $line =~ capa-dout:\ (.*) ]]; then
    CAPABILITY_DOUT=$(echo ${BASH_REMATCH[1]})
elif [[ $line =~ capa-adc:\ (.*) ]]; then
    CAPABILITY_ADC=$(echo ${BASH_REMATCH[1]})
elif [[ $line =~ capa-wifi:\ (.*) ]]; then
    CAPABILITY_WIFI=$(echo ${BASH_REMATCH[1]})
elif [[ $line =~ capa-bluetooth:\ (.*) ]]; then
    CAPABILITY_BLUETOOTH=$(echo ${BASH_REMATCH[1]})
elif [[ $line =~ capa:\ \"(.*)\" ]]; then
    echo \"mask\": \"${BASH_REMATCH[1]}\", >> $DEVICE_CAPA
fi

done <${CONFIG_PATH}/device_info

HASRADIO=$(cat /sys/devices/platform/mts-io/has-radio)                               
if ((HASRADIO == 1)) ; then
  echo "Finding cellular module..."
  for ((i=0; i < TRY_COUNT; i++))
    do
      # Check cellular module
      echo "Try #"$i
      if [[ -L "/dev/modem_at0" && -L "/dev/modem_at1" ]]; then
        CAPABILITY_CELL="true"
        break
      fi
      if ((TRY_COUNT > 1)) ; then
        sleep 0.5
      fi
    done
fi

if isNativeLoraPresent; then
  CAPABILITY_LORA="true"
fi

#
# Check if NODE-RED and Lora Network Server packages are installed
#
OPKG_FILE=/tmp/opkg_inst.tmp
opkg list-installed > $OPKG_FILE

cat $OPKG_FILE | grep node-red > /dev/null
[ $? -eq 0 ] && CAPABILITY_NODE_RED="true"

cat $OPKG_FILE | grep lora-network-server > /dev/null
[ $? -eq 0 ] && CAPABILITY_LORA_NETWORK_SERVER="true"

rm -f $OPKG_FILE


# Read firmware info from /etc/issue
cat /etc/issue |
(
while read line
do

if [[ $line =~ Version:\ (.*) ]]; then
    echo \"firmware\": \"${BASH_REMATCH[1]}\", >> $DEVICE_INFO_JSON
elif [[ $line =~ Date:\ (.*) ]]; then
    echo \"firmwareDate\": \"${BASH_REMATCH[1]}\", >> $DEVICE_INFO_JSON
fi

done
)
# Add accessory cards
echo "Reading accessory cards data"
ReadAcData
echo "Adding accessory cards data"
cat $TEMP_AC_JSON >> $DEVICE_INFO_JSON

# Complete the json files
echo \"adc\": $CAPABILITY_ADC, >> $DEVICE_CAPA
echo \"bluetooth\": $CAPABILITY_BLUETOOTH, >> $DEVICE_CAPA
echo \"din\": $CAPABILITY_DIN, >> $DEVICE_CAPA
echo \"dout\": $CAPABILITY_DOUT, >> $DEVICE_CAPA
echo \"gps\": $CAPABILITY_GPS, >> $DEVICE_CAPA
echo \"wifi\": $CAPABILITY_WIFI, >> $DEVICE_CAPA
echo \"cell\": $CAPABILITY_CELL, >> $DEVICE_CAPA
echo \"lora\": $CAPABILITY_LORA, >> $DEVICE_CAPA
echo \"gpio\": $CAPABILITY_GPIO, >> $DEVICE_CAPA
echo \"serial\": $CAPABILITY_SERIAL, >> $DEVICE_CAPA
echo \"rs232\": $CAPABILITY_RS232, >> $DEVICE_CAPA
echo \"rs422\": $CAPABILITY_RS422, >> $DEVICE_CAPA
echo \"rs485\": $CAPABILITY_RS485, >> $DEVICE_CAPA
echo \"nodeRed\": $CAPABILITY_NODE_RED, >> $DEVICE_CAPA
echo \"loraNetworkServer\": $CAPABILITY_LORA_NETWORK_SERVER >> $DEVICE_CAPA
echo "}" >> $DEVICE_CAPA
echo \"capabilities\": >> $DEVICE_INFO_JSON
cat $DEVICE_CAPA >> $DEVICE_INFO_JSON
echo "}" >> $DEVICE_INFO_JSON

# Clean up temp files
rm $DEVICE_CAPA
rm ${CONFIG_PATH}/device_info

#echo device_info.json ----------
#cat /var/run/config/device_info.json
#echo ---------------------------
