#!/bin/sh

# Copyright (c) 2008, 2010, Oracle.
# Copyright (c) 2009-2011 Monty Program Ab
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Library General Public
# License as published by the Free Software Foundation; version 2
# of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Library General Public License for more details.
#
# You should have received a copy of the GNU Library General Public
# License along with this library; if not, write to the Free
# Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
# MA 02111-1307, USA

die()
{
  echo "ERROR: $@"; exit 1;
}

get_key_value()
{
  echo "$1" | sed 's/^--[a-zA-Z_-]*=//'
}

developer_usage()
{
cat <<EOF
  This script can be used by developers of MariaDB wanting to try out
  early versions before binary versions are available, anyone needing
  a version with a special patch included that needs to be built from
  source code, or anyone else wanting to exercise full control over
  the build process.

  This help text is targeted towards those that want to debug and test
  source code releases. If you have downloaded a source code release
  and simply want to build a usable binary, you should read the
  --sysadmin-help instead.

  Three environment variables can be used to define the default behaviour:

  MYSQL_DEVELOPER
    Defining this variable is similar to setting the --developer flag
  MYSQL_DEVELOPER_PACKAGE=package
    Defining this variable is similar to setting the --package=* 
    variable
  MYSQL_DEVELOPER_DEBUG
    Defining this variable sets the --with-debug flag

  Options used with this script always override any default behaviour. 

  More information for developers can be found in --help, 
  --sysadmin-help, and --extended-help.

  The most common usage for developers is to set the three variables 
  mentioned previously and then simply to run the script without any 
  additional parameters, using them only when needing to change some 
  things like not requiring debug build. If some of these environment 
  variables have already been set, you can use the corresponding options 
  to unset them and reverse their effects.
EOF
}

sysadmin_usage()
{
cat <<EOF

  It is assumed that you are building on a computer which is of the 
  same type as that on which you intend to run the program.

  The simplest possible way to run this script is to allow it to use the 
  built-in defaults everywhere, invoking it simply as (from top-level
  MariaDB source directory):

  shell> BUILD/build_mccge.sh

  This performs the following operations:
    1) Detects the operating system. Currently, Linux, FreeBSD, Solaris 
      8/9/10/11, and Mac OS X are supported by this script.
    2) Detect the type of CPU being used. Currently supported processors 
      are: x86 for all supported operating systems, Itanium for Linux 
      with GCC, and x86 + SPARC for Solaris using the Forte compiler and
      finally x86 on Linux using the Intel compiler.
    3) Invokes the GCC compiler.
    4) Builds a set of of binaries ; for more information about these,
       see --extended-help.
    5) Default compiler is always gcc.
  
  The default version assumes that you have a source code tarball from 
  which you are building, and thus autoconf and automake do not need to 
  be run. If you have downloaded a launchpad tree then you should read 
  --developer-help.

  If your building on a Solaris SPARC machine and you want to compile
  using SunStudio you must set 
  --compiler=forte; if you want to build using the Intel compiler on 
  Linux, you need to set --compiler=icc.

  A synonym for forte is SunStudio, so one can also use
  --compiler=SunStudio.

  If you want to make sure that a 64-bit version is built then you 
  should add the flag --64. This is always set on Solaris machines and 
  when check-cpu is able to discover that a 64-bit CPU is being used. If 
  you want to ensure that a 32-bit binary is produced, use --32 instead.

  If you need the binaries to be installed in a different location from 
  /usr/local/mysql, then you should set --prefix to point to where you
  want the binaries installed.

  Using a data directory other than the default (PREFIX/data) can be 
  done when starting the server, or by invoking this script with 
  the --datadir option.

  If you want your binaries stripped of surplus debug or other 
  information, use the --strip option.

  If you want debug information in the binary (for example, to be
  able to send gdb core dumps to support), then you should add the 
  flag --with-debug; if you want a production build with only debugging 
  information in the binary then use --debug.
EOF
}

usage()
{
cat <<EOF

Usage: $0 [options]
  --help                  Show this help message.
  --sysadmin-help         Show help for system administrators wishing 
                          to build MariaDB
  --developer-help        Show help for developers trying to build MariaDB
  --with-help             Show extended help on --with-xxx options to 
                          configure
  --extended-help         Show extended help message
  --without-debug         Build non-debug version
  --use-comment           Set the comment in the build
  --with-fast-mutexes     Use try/retry method of acquiring mutex
  --with-debug            Build debug version
  --with-debug=full       Build with full debug.
  --with-link-time-optimizer
                          Link time optimizations enabled (Requires GCC 4.5
                          if GCC used), available for icc as well. This flag
                          is only considered if also fast is set.
  --configure-only        Stop after running configure.
  --use-autotools         Start by running autoconf, automake,.. tools
  --no-autotools          Start from configure
  --print-only            Print commands that the script will execute, 
                          but do not actually execute
  --prefix=path           Build with prefix 'path'
  --datadir=path          Build with data directory set to non-standard 
                          'path'
  --debug                 Build normal version, but add debug 
                          information to binary
  --developer             Use extensions that most MariaDB developers use
  --no-developer          Do not use extensions that most developers of 
                          MariaDB use
  --compiler=[gcc|icc|forte|SunStudio]        Select compiler
  --cpu=[x86|x86_64|sparc|itanium]            Select CPU type
                          x86 => x86 and 32-bit binary
                          x86_64 => x86 and 64 bit binary
  --warning-mode=[extra|pedantic|normal|no]   Set warning mode level
  --warnings              Set warning mode to normal
  --32                    Build a 32-bit binary even if CPU is 64-bit
  --64                    Build a 64-bit binary even if not sure a
                          64-bit CPU is being used
  --package=[pro|classic] Select package to build
  --parallelism=number    Define parallelism in make
  --strip                 Strip binaries
  --error-inject          Enable error injection into MariaDB Server and 
                          data nodes
  --valgrind              Build with valgrind
  --fast                  Optimise for CPU architecture built on
  --static-linking        Statically link system libraries into binaries
  --use-tcmalloc          Link with tcmalloc instead of standard malloc (Linux only)
  --with-flags *          Pass extra --with-xxx options to configure
EOF
  if test "x$1" != "x" ; then
    echo "Failure Message: $1"
  fi
}

extended_usage()
{
  cat <<EOF

  Extended help text for this script:
  ----------------------------------- 

  This script is intended to make it easier to build
  performance-optimised MariaDB versions from source on these
  platforms/compilers: Linux/x86 (32-bit and 64-bit) (either using gcc
  or icc), Linux Itanium, Solaris 8,9,10 and 11 x86 and SPARC using
  gcc or SunStudio and MacOSX/x86/gcc.

  The script automatically detects CPU type and operating system; The
  default compiler is always gcc.

  To build on other platforms you can use the --print-only option on a
  supported platform and edit the output for a proper set of commands on
  the specific platform you are using.

  Using the --package option, it is also possible to build a "classic"
  version of MariaDB having only the MyISAM storage engine, a "Pro"
  package including all storage engines and other features except MariaDB
  Cluster, and an "extended" package including these features plus MariaDB
  Cluster (this is the default if the --developer option is used).

  Different MariaDB storage engines are included in the build, depending
  on which --package option is used. The comment and version strong
  suffix are also set according to the package selected.

  --package=pro
    storage engines:
      ARCHIVE, BLACKHOLE, CSV, FEDERATED, INNODB, MYISAM
      (All storage engines except NDB)
    comment: Pro versions
    version string suffix: [none]

  --package=classic
    storage engines: CSV, MYISAM
    comment: Version without InnoDB and Maria
    version string suffix: [none]

  All packages except Classic include support for user-defined
  partitioning.

  If --with-debug is used, an additional "-debug" is appended to the
  version string.

  --with-debug[=full]
    This option will ensure that the version is built with debug
    information enabled; the optimisation level is decreased to -O.

  --developer
    This option changes a number of things to make the version built
    more appropriate to the debugging and testing needs of developers. 
    It changes the default package to "extended". It also changes the 
    default warning mode from "none" to "normal", which allows an 
    extensive list of warnings to be generated.

  --error-inject
    This flag is used only when the --developer option is also used, and
    enables error injection in both the MySQL Server and in MySQL
    Cluster data nodes.

  The following is a list of the default configure options used for all
  packages:

  --prefix: /usr/local/mysql (can be overridden)

  --libexecdir: <prefix>/bin (can be overridden)

  --localstatedir: <prefix>/data, unless --datadir is used, in which
    case it defaults to <datadir>/data (can be overridden by setting
    --localstatedir explicitly).

  --enable-local-infile: Enable use of the LOAD DATA FROM  LOCAL INFILE
    command (cannot be overridden).

  --enable-thread-safe-client: Enable the multi-threaded mysql client
    library (cannot be overridden).

  --with-big-tables: Enable use of tables with more than 4G rows (cannot
    be overridden).

  --with-extra-charsets=all: Enable use of all character sets supported
    by MySQL (cannot be overridden).

  --with-ssl: Enable use of yaSSL library included in the MySQL source
    if possible (GCC and same CC and CXX).
    (cannot be overridden).

  --with-pic: Build all binaries using position independent assembler
    to avoid problems with dynamic linkers (cannot be overridden).

  --without-example-engine: Ensure that the example engine isn't built,
    it cannot do any useful things, it's merely intended as documentation.
    (cannot be overridden)

  --with-csv-storage-engine: Ensure that the CSV storage engine is
    included in all builds. Since CSV is required for log tables in
    MySQL 5.1, this option cannot be overridden.

    (Note that MyISAM support is always built into the MySQL Server; the
    server *cannot* be built without MyISAM.)

  --with-mysqld-ldflags=-static
  --with-client-ldflags=-static
    Ensures that binaries for, respectively, the MySQL server and client
    are built with static libraries except for the system libraries,
    which use dynamically loaded libraries provided by the operating
    system. Building with --developer sets these to all-static instead, 
    to build everything statically.

  In addition there are some configure options that are specific to
  Linux operating systems:

  --enable-assembler
    Include assembler code optimisations for a number of mostly string
    methods. Used for x86 processors only.

  Neither of the preceding options can be disabled.

  MySQL Cluster Carrier Grade edition also adds the following options
  (also used by the extended package):

  --with-ndbcluster
    Include the NDB Cluster storage engine, its kernel, management
    server, and client, as well as support for the NDB and MGM APIs.

  --without-ndb-debug
    Do not include specific NDB debug code, not even in debug versions
    (cannot be overridden).

  Package-specific options:
  -------------------------
  --with-innodb
    Specifically included in the "pro" package.

  --with-comment
    Sets the comment for the MySQL version, by package, as described
    above.

  --with-server-suffix
    Sets the server suffix on the MySQL version, by package, as
    described above.

  Other options used:
  -------------------
  --with-readline
    Use the GPL readline library for command editing functions.

  --with-libedit
    Use the BSD licensed library for command editing functions.

  --with-zlib-dir=bundled
    Use the zlib package bundled with MySQL.

  --with-mysqld-libs=-lmtmalloc
    Used on Solaris to ensure that the proper malloc library is used.
    Investigations have shown mtmalloc to be the best choice on Solaris,
    also umem has good performance on Solaris but better debugging
    capabilities.

  Compiler options:
  -----------------

    This section describes the compiler options for each of the different
    platforms supported by this script.

    The --fast option adds -mtune=cpu_arg to the C/C++ flags (provides
    support for Nocona, K8, and other processors).

    Use of the --debug option adds -g to the C/C++ flags.

    In all cases it is possible to override the definition of CC and CXX
    by calling the script as follows:
    CC="/usr/local/bin/gcc" CXX="/usr/local/bin/gcc" BUILD/build_mccge.sh

  FreeBSD/x86/gcc
  ---------------
    No flags are used. Instead, configure determines the proper flags to 
    use.

  Linux/x86+Itanium/gcc
  -------------
    For debug builds -O is used and otherwise -O3 is used.Discovery of a
    Nocona or Core 2 Duo CPU causes a 64-bit binary to be built;
    otherwise, the binary is 32-bit. To build a 64-bit binary, -m64 is
    added to the C/C++ flags. (To build a 32-bit binary on a 64-bit CPU,
    use the --32 option as described previously.)

  Linux/x86+Itanium/icc
  -------------
    Flags used:
    CC  = icc -static-libgcc -static-intel
    C++ = icpc -static-libgcc -static-intel
    C/C++ flags = -mp -restrict

    On Itanium we also add -no-ftz and to CC and C++ flags.

    Note that if the user of this script sets CC or CXX explicitly then
    also -static-libgcc and -static-intel needs to be set in the CC and
    CXX.

    The non-debug versions also add the following:
      C/C++ flags += -O3 unroll2 -ip

    The fast version adds (if --with-link-time-optimizer is used):
      C/C++ flags += -ipo

    On discovery of a Core 2 Duo architecture while using icc, -xT is also 
    added to the C/C++ flags; this provides optimisations specific to Core 
    2 Duo. This is added only when the --fast flag is set.

  Solaris/x86/gcc
  ---------------
    All builds on Solaris are by default 64-bit, so -m64 is always used in
    the C/C++ flags. LDFLAGS is set to -m64 -O/-O2/-O3. If for
    some reason a 32-bit Solaris is used it is necessary to add the flag
    --32 to the script invocation. Due to bugs in compiling with -O3 on
    Solaris only -O2 is used by default, when --fast flag is used -O3 will
    be used instead.

    Sets -m64 (default) or -m32 (if specifically set) in LDFLAGS and
    C/C++ flags.

  Solaris/Sparc/Forte
  -------------------
    Uses cc as CC and CC as CXX
    Note that SunStudio uses different binaries for C and C++ compilers.

    Set -m64 (default) or -m32 (if specifically set) in ASFLAGS,
    LDFLAGS and C/C++ flags.

    Sets ASFLAGS=LDFLAGS=compiler flags=xarch=sparc, so that we compile
    Sparc v9 binaries, also -mt is set in all those since we're always
    building a multithreaded program.

    C flags   = -xstrconst    This flag is set only on SPARC
    C++ flags = -noex

    Set the following C/C++ flags:
    -fsimple=1
    -ftrap=%none
    -nofstore          This flag is set only on x86
    -xbuiltin=%all
    -xlibmil
    -xlibmopt

    Set the C++ flag:
    -noex
    -features=no%except    This flag is set only on x86

    When compiling with fast we set (-ipo only used if we have
    set --with-link-time-optimizer):
    C/C++ flags: -xtarget=native -xunroll=3 -xipo
    LDFLAGS: -xipo

    When not compiling with fast we always set -xtarget=generic

    When compiling with fast on SPARC we also set:
    C/C++ flags: -xbinopt=prepare
    LDFLAGS: -xbinopt=prepare

    When compiling with fast on x86 we also set:
    C/C++ flags: -xregs=frameptr
    When not compiling with fast we set on x86
    C/C++ flags: -xregs=no%frameptr

    On SPARC we set
    ASFLAGS = LDFLAGS = C/C++ flags = -xarch=sparc

    The optimisation level is
    -xO         Debug builds
    -xO2        Production build on SPARC
    -xO3        Production build on x86
    -xO4        Fast builds on SPARC/x86

  MacOSX/x86/gcc
  --------------
    C/C++ flags include -fno-common -arch i386.
    When 64-bits builds then i386 is replaced by x86_64.

    Non-debug versions also add -Os -felide-constructors, where "-Os"
    means the build is space-optimised as long as the space optimisations
    do not negatively affect performance. Debug versions use -O.
  
    Mac OS X builds will always be 32-bit by default, when --64 is added
    the build will be 64 bit instead. Thus the flag --m64 is added only
    when specifically given as an option.
EOF
}

with_usage()
{
  cat <<EOF

  To obtain extended help on the --with-* options available, run this
  script with --configure-only to create a configuration file. Then
  issue the command ./configure --help to get an extensive list of
  possible configure options.

  The remainder of this text focuses on those options which are useful
  in building binaries for MySQL Cluster Carrier Grade Edition.

  --with-ndb-sci=/opt/DIS
    Used to build a MySQL Cluster Carrier Grade Edition that can use the
    SCI Transporter. The Dolphin SCI installation must be completed
    first (see 
    http://dev.mysql.com/doc/refman/5.1/en/mysql-cluster-interconnects.html
    for more information).

  --with-ndb-test
    Compile the MySQL Cluster test programs.

  --with-ndb-port=PORT
    Change the default port for the MySQL Cluster management server.

  --with-ndb-port-base=PORT
    Change the default port base for MySQL Cluster data nodes.

  --without-query-cache
    Build the MySQL Server without the query cache, which is often not
    of value in MySQL Cluster applications.

  --with-atomic-ops=rwlocks|smp|up
                          Implement atomic operations using pthread
                          rwlocks or atomic CPU instructions for
                          multi-processor (default) or single-processor
                          configurations.

  --without-geometry      Do not build geometry-related portions of the
                          MySQL Server. Seldom used in MySQL Cluster
                          applications.

  --with-ndb-cc-flags=FLAGS
    This option can be used to build MySQL Cluster with error injection
    on the data nodes. It can be used to pass special options to
    programs in the NDB kernel for special test builds.
    The option for enabling data node error injection is -DERROR_INSERT.
EOF
}

parse_package()
{
  case "$package" in
    classic )
      package="classic"
      ;;
    pro )
      package="pro"
      ;;
    extended )
      package="extended"
      ;;
    *)
      echo "Unknown package '$package'"
      exit 1
      ;;
  esac
}

parse_warning_mode()
{
  case "$warning_mode" in
    pedantic )
      warning_mode="pedantic"
      ;;
    extra_warnings | extra-warnings | extra )
      warning_mode="extra"
      ;;
    no )
      warning_mode=
      ;;
    normal )
      warning_mode="normal"
      ;;
    *)
      echo "Unknown warning mode '$warning_mode'"
      exit 1
      ;;
  esac
}

#
# We currently only support x86, Itanium and UltraSparc processors.
#
parse_cpu_type()
{
  case "$cpu_type" in
    x86 )
      cpu_type="x86"
      if test "x$m64" = "x" ; then
        m64="no"
      fi
      ;;
    x86_64 )
      cpu_type="x86"
      if test "x$m64" = "x" ; then
        m64="yes"
      fi
      ;;
    itanium )
      cpu_type="itanium"
      ;;
    sparc )
      cpu_type="sparc"
      ;;
    * )
      echo "Unknown CPU type $cpu_type"
      exit 1
      ;;
  esac
  return
}

#
# We currently only support gcc, icc and Forte.
#
parse_compiler()
{
  case "$compiler" in
    gcc )
      compiler="gcc"
      ;;
    icc )
      compiler="icc"
      ;;
    forte | SunStudio | sunstudio )
      compiler="forte"
      ;;
    *)
      echo "Unknown compiler '$compiler'"
      exit 1
      ;;
  esac
}

parse_options()
{
  while test $# -gt 0
  do
    case "$1" in
    --prefix=*)
      prefix=`get_key_value "$1"`
      ;;
    --datadir=*)
      datadir=`get_key_value "$1"`
      ;;
    --with-link-time-optimizer)
      with_link_time_optimizer="yes"
      ;;
    --with-debug=full)
      full_debug="=full"
      with_debug_flag="yes"
      fast_flag="no"
      ;;
    --without-debug)
      with_debug_flag="no"
      if test "x$fast_flag" != "xyes" ; then
        fast_flag="generic"
      fi
      ;;
    --use-comment)
      without_comment="no"
      ;;
    --with-fast-mutexes)
      with_fast_mutexes="yes"
      ;;
    --use-tcmalloc)
      use_tcmalloc="yes"
      ;;
    --with-debug)
      with_debug_flag="yes"
      fast_flag="no"
      ;;
    --debug)
      compile_debug_flag="yes"
      ;;
    --no-developer)
      developer_flag="no"
      ;;
    --developer)
      developer_flag="yes"
      ;;
    --compiler=*)
      compiler=`get_key_value "$1"`
      parse_compiler
      ;;
    --cpu=*)
      cpu_type=`get_key_value "$1"`
      parse_cpu_type
      ;;
    --warning-mode=*)
      warning_mode=`get_key_value "$1"`
      parse_warning_mode
      ;;
    --warnings)
      warning_mode="normal"
      ;;
    --32)
      if test "x$explicit_size_set" != "x" ; then
        echo "Cannot set both --32 and --64"
        exit 1
      fi
      explicit_size_set="yes"
      m64="no"
      ;;
    --64)
      if test "x$explicit_size_set" != "x" ; then
        echo "Cannot set both --32 and --64"
        exit 1
      fi
      explicit_size_set="yes"
      m64="yes"
      ;;
    --package=*)
      package=`get_key_value "$1"`
      parse_package
      ;;
    --parallelism=*)
      parallelism=`get_key_value "$1"`
      ;;
    --use-autotools)
      use_autotools="yes"
      ;;
    --no-autotools)
      use_autotools="no"
      ;;
    --configure-only)
      just_configure="yes"
      ;;
    --print-only|--just-print)
      just_print="yes"
      ;;
    --static-linking)
      static_linking_flag="yes"
      ;;
    --strip)
      strip_flag="yes"
      ;;
    --error-inject)
      error_inject_flag="yes"
      ;;
    --valgrind)
      valgrind="yes"
      ;;
    --fast)
      fast_flag="yes"
      ;;
    --with-flags)
      shift
      break
      ;;
    --with-help)
      with_usage
      exit 0
      ;;
    --sysadmin-help)
      sysadmin_usage
      exit 0
      ;;
    --developer-help)
      developer_usage
      exit 0
      ;;
    --extended-help)
      extended_usage
      exit 0
      ;;
    --help)
      usage
      exit 0
      ;;
    *)
      echo "Unknown option '$1'"
      exit 1
      ;;
    esac
    shift
  done
  for flag in $@
  do
    with_flags="$with_flags $flag"
  done
}

#
# We currently only support Linux, FreeBSD/OpenBSD, Mac OS X and Solaris
#
check_os()
{
  case "`uname -s`" in
    Linux)
      os="linux"
      ;;
    FreeBSD|OpenBSD)
      os="bsd"
      ;;
    Darwin)
      os="MacOSX"
      ;;
    SunOS)
      os="Solaris"
      ;;
    *)
      os="Unknown"
      ;;
  esac

}

set_cpu_base()
{
  if test "x$cpu_type" = "x" ; then
    if test "x$cpu_arg" = "x" ; then
      usage "CPU type not discovered, cannot proceed"
      exit 1
    fi
    case "$cpu_arg" in
      core2 | nocona | prescott | pentium* | i*86 )
        # Intel CPU
        cpu_base_type="x86"
        ;;
      athlon* | opteron* | k6 | k8 )
        # AMD CPU
        cpu_base_type="x86"
        ;;
      sparc )
        cpu_base_type="sparc"
        ;;
      itanium )
        cpu_base_type="itanium"
        ;;
      * )
        usage "CPU type $cpu_arg not handled by this script"
        exit 1
        ;;
    esac
  else
    cpu_base_type="$cpu_type"
    check_cpu_cflags=""
  fi
  if test "x$os" = "xMacOSX" ; then
    if test "x$m64" = "x" ; then
      m64="no"
    fi
  elif test "x$os" = "xSolaris" ; then
    if test "x$m64" = "x" ; then
      m64="yes"
    fi
  elif test "x$m64" = "x" ; then
    if test "x$cpu_arg" = "xnocona" || test "x$cpu_arg" = "xcore2" || \
       test "x$cpu_arg" = "xathlon64" || test "x$cpu_arg" = "xopteron" ; then
      m64="yes"
    else
      m64="no"
    fi
  fi
  echo "Discovered CPU of type $cpu_base_type ($cpu_arg) on $os"
  if test "x$m64" = "xyes" ; then
    echo "Will compile 64-bit binaries"
  else
    echo "Will compile 32-bit binaries"
  fi
  return 0
}

#
# Add to the variable commands with the configure command
# 
init_configure_commands()
{
  cflags="$c_warnings $base_cflags $compiler_flags"
  cxxflags="$cxx_warnings $base_cxxflags $compiler_flags"
  configure="./configure $base_configs $with_flags"

  env_flags="CC=\"$CC\" CFLAGS=\"$cflags\" CXX=\"$CXX\" CXXFLAGS=\"$cxxflags\""
  if test "x$LDFLAGS" != "x" ; then
    env_flags="$env_flags LDFLAGS=\"$LDFLAGS\""
  fi
  if test "x$ASFLAGS" != "x" ; then
    env_flags="$env_flags ASFLAGS=\"$ASFLAGS\""
  fi
  commands="$commands
    $env_flags $configure"
} 

#
# Initialise the variable commands with the commands needed to generate
# the configure script.
#
init_auto_commands()
{
  set_libtoolize_version
  commands="\
  $make -k maintainer-clean || true
  /bin/rm -rf */.deps/*.P configure config.cache
  /bin/rm -rf  storage/*/configure storage/*/config.cache autom4te.cache
  /bin/rm -rf storage/*/autom4te.cache;"
#
# --add-missing instructs automake to install missing auxiliary files
# and --force to overwrite them if they already exist
#
  commands="$commands
  aclocal || die \"Can't execute aclocal\"
  autoheader || die \"Can't execute autoheader\"
  $LIBTOOLIZE --automake --copy --force || die \"Can't execute libtoolize\"
  automake --add-missing --copy --force || die \"Can't execute automake\"
  autoconf || die \"Can't execute autoconf\""
}

#
# Add to the variable commands the make command and possibly also
# strip commands
#
add_make_commands()
{
  AM_MAKEFLAGS="-j $parallelism"
  commands="$commands
  $make $AM_MAKEFLAGS"

  if test "x$strip_flag" = "xyes" ; then
    commands="$commands
    mkdir -p tmp
    nm --numeric-sort sql/mysqld  > tmp/mysqld.sym
    objdump -d sql/mysqld > tmp/mysqld.S
    strip sql/mysqld
    strip storage/ndb/src/kernel/ndbd
    strip storage/ndb/src/mgmsrv/ndb_mgmd
    strip storage/ndb/src/mgmclient/ndb_mgm"
  fi
}

#
# Set make version, but only gmake is supported :)
#
set_make_version()
{
  if gmake --version > /dev/null 2>&1
  then
    make=gmake
  else
    make=make
  fi
  if test "x`$make --version | grep GNU`" = "x" ; then
    die "Only gmake is supported"
  fi
}

#
# Find a libtoolize binary, both libtoolize and glibtoolize are
# ok, use first found.
#
set_libtoolize_version()
{
  LIBTOOLIZE=not_found
  save_ifs="$IFS"; IFS=':'
  for dir in $PATH
  do
    if test -x $dir/libtoolize
    then
      LIBTOOLIZE=libtoolize
      echo "Found libtoolize in $dir"
      break
    fi
    if test -x $dir/glibtoolize
    then
      LIBTOOLIZE=glibtoolize
      echo "Found glibtoolize in $dir"
      break
    fi
  done
  IFS="$save_ifs"
  if test "x$LIBTOOLIZE" = "xnot_found" ; then
    die "Found no libtoolize version, quitting here"
  fi
  return
}

#
# If ccache (a compiler cache which reduces build time)
# (http://samba.org/ccache) is installed, use it.
# We use 'grep' and hope that 'grep' works as expected
# (returns 0 if finds lines)
# We do not use ccache when gcov is used. Also only when
# gcc is used.
#
set_ccache_usage()
{
  if test "x$compiler" = "xgcc" ; then
    if ccache -V > /dev/null 2>&1 && test "$USING_GCOV" != "1"
    then
      echo "$CC" | grep "ccache" > /dev/null || CC="ccache $CC"
      echo "$CXX" | grep "ccache" > /dev/null || CXX="ccache $CXX"
    fi
  fi
}

#
# Set flags for various build configurations.
# Used in -valgrind builds
#
set_valgrind_flags()
{
  if test "x$valgrind_flag" = "xyes" ; then
    loc_valgrind_flags="-USAFEMALLOC -UFORCE_INIT_OF_VARS -DHAVE_valgrind "
    loc_valgrind_flags="$loc_valgrind_flags -DMYSQL_SERVER_SUFFIX=-valgrind-max"
    compiler_flags="$compiler_flags $loc_valgrind_flags"
    with_flags="$with_flags --with-valgrind"
  fi
}

#
# Set up warnings; default is to use no warnings, but if warning_mode
# is used a lot of warning flags are set up. These flags are valid only
# for gcc, so for other compilers we ignore the warning_mode.
#
set_warning_flags()
{
  if test "x$developer_flag" = "xyes" && test "x$warning_mode" = "x" ; then
    warning_mode="normal"
  fi
  if test "x$compiler" = "xgcc" ; then
    if test "x$warning_mode" = "normal" || test "x$warning_mode" = "extra" ; then
# Both C and C++ warnings
      warnings="$warnings -Wimplicit -Wreturn-type -Wswitch -Wtrigraphs"
      warnings="$warnings -Wcomment -W"
      warnings="$warnings -Wchar-subscripts -Wformat -Wparentheses -Wsign-compare"
      warnings="$warnings -Wwrite-strings -Wunused-function -Wunused-label"
      warnings="$warnings -Wunused-value -Wunused-variable -Wno-uninitialized"

      if test "x$warning_mode" = "extra" ; then
        warnings="$warnings -Wshadow"
      fi
# C warnings
      c_warnings="$warnings -Wunused-parameter"
# C++ warnings
      cxx_warnings="$warnings -Woverloaded-virtual -Wsign-promo -Wreorder"
      cxx_warnings="$warnings -Wctor-dtor-privacy -Wnon-virtual-dtor"
# Added unless --with-debug=full
      if test "x$full_debug" = "x" ; then
        compiler_flags="$compiler_flags -Wuninitialized"
      fi
    elif test "x$warning_mode" = "xpedantic" ; then
      warnings="-W -Wall -ansi -pedantic -Wno-long-long -D_POSIX_SOURCE"
      c_warnings="$warnings"
      cxx_warnings="$warnings -std=c++98"
# Reset CPU flags (-mtune), they don't work in -pedantic mode
     check_cpu_cflags=""
    fi
  fi
}

#
# Used in -debug builds
#
set_with_debug_flags()
{
  if test "x$with_debug_flag" = "xyes" ; then
    if test "x$developer_flag" = "xyes" ; then
      loc_debug_flags="-DUNIV_MUST_NOT_INLINE -DEXTRA_DEBUG"
      loc_debug_flags="$loc_debug_flags -Wuninitialized -DFORCE_INIT_OF_VARS"
      loc_debug_flags="$loc_debug_flags -DSAFEMALLOC -DPEDANTIC_SAFEMALLOC"
      compiler_flags="$compiler_flags $loc_debug_flags"
    fi
  fi
}

#
# Flag for optimizing builds for developers.
#
set_no_omit_frame_pointer_for_developers()
{
  if test "x$fast_flag" != "xno" ; then
    if test "x$developer_flag" = "xyes" && test "x$compiler" = "xgcc" ; then
# Be as fast as we can be without losing our ability to backtrace.
      compiler_flags="$compiler_flags -fno-omit-frame-pointer"
    fi
  fi
}

#
# Add -g to all builds that requested debug information in build
#
set_debug_flag()
{
  if test "x$compile_debug_flags" = "xyes" ; then
    compiler_flags="$compiler_flags -g"
  fi
}

#
# We compile in SSL support if we can, this isn't possible if CXX
# and CC aren't the same and we're not using GCC.
# 
set_ssl()
{
  if test "x$compiler" = "xgcc" && \
     test "x$CC" = "x$CXX" ; then
    base_configs="$base_configs --with-ssl"
  fi
}

#
# Base options used by all packages
#
# SSL library to use. --with-ssl selects the bundled yaSSL
# implementation of SSL. To use openSSL, you must point out the location
# of the openSSL headers and libs on your system.
# For example: --with-ssl=/usr
#
set_base_configs()
{
  base_configs="$base_configs --prefix=$prefix"
  base_configs="$base_configs --libexecdir=$prefix/bin"
  base_configs="$base_configs --with-zlib-dir=bundled"
  if test "x$datadir" = "x" ; then
    base_configs="$base_configs --localstatedir=$prefix/data"
  else
    base_configs="$base_configs --localstatedir=$datadir"
  fi
  if test "x$with_debug_flag" = "xyes" ; then
    base_configs="$base_configs --with-debug$full_debug"
  fi
  base_configs="$base_configs --enable-local-infile"
  base_configs="$base_configs --enable-thread-safe-client"
  base_configs="$base_configs --with-big-tables"
  base_configs="$base_configs --with-extra-charsets=all"
  if test "x$with_fast_mutexes" = "xyes" ; then
    base_configs="$base_configs --with-fast-mutexes"
  fi
  base_configs="$base_configs --with-pic"
  base_configs="$base_configs --with-csv-storage-engine"
  base_configs="$base_configs --with-perfschema"
  base_configs="$base_configs --with-libevent"
}

#
# Add all standard engines and partitioning
#
set_max_engines()
{
  engine_configs="--with-plugins=max --with-plugin-maria --with-maria-tmp-tables"
  engine_configs="$engine_configs --without-plugin-innodb_plugin"
  base_configs="$base_configs $engine_configs"
}

set_ndb_engine()
{
  base_configs="$base_configs --with-ndbcluster"
  base_configs="$base_configs --without-ndb-debug"
}

set_pro_package()
{
  if test "x$without_comment" != "xyes" ; then
    base_configs="$base_configs --with-comment=\"Pro $version_text built from source\""
  fi
  if test "x$with_debug_flag" = "xyes" ; then
    base_configs="$base_configs --with-server-suffix=\"-debug\""
  fi
}

set_classic_package()
{
  if test "x$without_comment" != "xyes" ; then
    base_configs="$base_configs --with-comment=\"Classic $version_text built from source\""
  fi
  if test "x$with_debug_flag" = "xyes" ; then
    base_configs="$base_configs --with-server-suffix=\"-debug\""
  fi
  base_configs="$base_configs --without-example-storage-engine"
}

#
# Special handling of readline; use readline from the MySQL
# distribution if building a GPL version, otherwise use libedit.
#
set_readline_package()
{
  if test -d "$path/../cmd-line-utils/readline" && test "x$gpl" = "xyes" ; then
    base_configs="$base_configs --with-readline"
  elif test -d "$path/../cmd-line-utils/libedit" ; then
    base_configs="$base_configs --with-libedit"
  fi
}

#
# If fast flag set by user we also add architecture as discovered to 
# compiler flags to make binary optimised for architecture at hand. 
# We use this feature on gcc compilers.
#
set_gcc_special_options()
{
  if test "x$fast_flag" = "xyes" && test "x$compiler" = "xgcc" ; then
    compiler_flags="$compiler_flags $check_cpu_cflags"
  fi
}

set_cc_and_cxx_for_gcc()
{
  if test "x$CC" = "x" ; then
    CC="gcc -static-libgcc -fno-exceptions"
  fi
  if test "x$CXX" = "x" ; then
    CXX="gcc -static-libgcc -fno-exceptions"
  fi
}

set_cc_and_cxx_for_icc()
{
  if test "x$CC" = "x" ; then
    CC="icc -static-intel -static-libgcc"
  fi
  if test "x$CXX" = "x" ; then
    CXX="icpc -static-intel -static-libgcc"
  fi
}

set_cc_and_cxx_for_forte()
{
  if test "x$CC" = "x" ; then
    CC="cc"
  fi
  if test "x$CXX" = "x" ; then
    CXX="CC"
  fi
}

#
# If we discover a Core 2 Duo architecture and we have enabled the fast
# flag, we enable a compile especially optimised for Core 2 Duo. This
# feature is currently available on Intel's icc compiler only.
#
set_icc_special_options()
{
  if test "x$fast_flag" = "xyes" && test "x$cpu_arg" = "xcore2" && \
     test "x$compiler" = "xicc" ; then
    compiler_flags="$compiler_flags -xT"
  fi
}

#
# FreeBSD Section
#
set_bsd_configs()
{
  if test "x$cpu_base_type" != "xx86" ; then
    usage "Only x86 CPUs supported for FreeBSD"
    exit 1
  fi
  if test "x$compiler" != "xgcc" ; then
    usage "Only gcc supported for FreeBSD"
    exit 1
  fi
  base_configs="$base_configs --enable-assembler"
  if test "x$fast_flag" != "xno" ; then
    compiler_flags="$compiler_flags -O3"
  else
    compiler_flags="$compiler_flags -O0"
  fi
  set_cc_and_cxx_for_gcc
}

check_64_bits()
{
  echo "Checking for 32/64-bits compilation"
  echo "int main() { return 0; }" > temp_test.c
  if test "x$m64" = "xyes" ; then
    cmd="$CC $compile_flags -m64 temp_test.c"
    if ! $cmd 2>1 ; then
      m64="no"
      echo "Changing to 32-bits since 64-bits didn't work"
    else
      echo "Will use 64-bits"
    fi
  else
    cmd="$CC $compile_flags -m32 temp_test.c"
    if ! $cmd 2>1 ; then
      m64="yes"
      echo "Changing to 64-bits since 32-bits didn't work"
    else
      echo "Will use 32-bits"
    fi
  fi
  rm temp_test.c
}

#
# Get GCC version
#
get_gcc_version()
{
  # check if compiler is gcc and dump its version
  cc_verno=`$cc -dumpversion 2>/dev/null`
  if test "x$?" = "x0" ; then
    set -- `echo $cc_verno | tr '.' ' '`
    cc_ver="GCC"
    cc_major=$1
    cc_minor=$2
    cc_patch=$3
    gcc_version=`expr $cc_major '*' 100 '+' $cc_minor`
  fi
}

#
# Link Time Optimizer in GCC (LTO) uses a parameter -flto
# which was added to GCC 4.5, if --with-link-time-optimizer
# is set then use this feature
#
check_for_link_time_optimizer()
{
  get_gcc_version
  if test "$gcc_version" -ge 405 && \
     test "x$with_link_time_optimizer" = "xyes" ; then
    compiler_flags="$compiler_flags -flto"
    LDFLAGS="$LDFLAGS -flto"
  fi
}
#
# Linux Section
#
set_linux_configs()
{
  if test "x$cpu_base_type" != "xx86" && \
     test "x$cpu_base_type" != "xitanium" ; then
    usage "Only x86 and Itanium CPUs supported for Linux"
    exit 1
  fi
  if test "x$use_tcmalloc" = "xyes" ; then
    base_configs="$base_configs --with-mysqld-libs=-ltcmalloc_minimal"
  fi
  if test "x$cpu_base_type" = "xx86" ; then
    base_configs="$base_configs --enable-assembler"
  fi
  if test "x$compiler" = "xgcc" ; then
    set_cc_and_cxx_for_gcc
    if test "x$fast_flag" != "xno" ; then
      if test "x$fast_flag" = "xyes" ; then
        compiler_flags="$compiler_flags -O3"
        check_for_link_time_optimizer
      else
        compiler_flags="$compiler_flags -O2"
      fi
    else
      compiler_flags="$compiler_flags -O0"
    fi
    check_64_bits
    if test "x$m64" = "xyes" ; then
      compiler_flags="$compiler_flags -m64"
    else
      compiler_flags="$compiler_flags -m32"
    fi
# configure will set proper compiler flags for gcc on Linux
  elif test "x$compiler" = "xicc" ; then
    compiler_flags="$compiler_flags -mp -restrict"
    set_cc_and_cxx_for_icc
    if test "x$cpu_base_type" = "xitanium" ; then
      compiler_flags="$compiler_flags -no-ftz"
    fi
    if test "x$fast_flag" != "xno" ; then
      compiler_flags="$compiler_flags -O3 -unroll2 -ip"
      if test "x$fast_flag" = "xyes" && \
         test "x$with_link_time_optimizer" = "xyes" ; then
        compiler_flags="$compiler_flags -ipo"
        LDFLAGS="$LDFLAGS -ipo"
      fi
    fi
  else
    usage "Only gcc and icc compilers supported for Linux"
    exit 1
  fi
}

#
# Solaris Section
#
set_solaris_configs()
{
# Use mtmalloc as malloc, see Tim Cook blog
# For information on optimal compiler settings, see article at
# http://developers.sun.com/solaris/articles/mysql_perf_tune.html
# by Luojia Chen at Sun.
  base_configs="$base_configs --with-named-curses=-lcurses"
  case "`uname -a`" in
    *5.8* | *5.9* )
      ;;

    *5.10* | *5.11*)
      base_configs="$base_configs --with-mysqld-libs=-lmtmalloc"
      ;;
    *)
      usage "Only versions 8,9, 10 and 11 supported for Solaris"
      exit 1
  esac
  if test "x$cpu_base_type" != "xx86" && \
     test "x$cpu_base_type" != "xsparc" ; then
    usage "Only x86 and Sparc CPUs supported for Solaris"
    exit 1
  fi
  if test "x$compiler" != "xgcc" && \
     test "x$compiler" != "xforte" ; then
    usage "Only gcc and Forte compilers supported for Solaris"
    exit 1
  fi
  if test "x$m64" = "xyes" ; then
    compiler_flags="$compiler_flags -m64"
    LDFLAGS="-m64"
    ASFLAGS="$ASFLAGS -m64"
  else
    compiler_flags="$compiler_flags -m32"
    LDFLAGS="-m32"
    ASFLAGS="$ASFLAGS -m32"
  fi
  if test "x$compiler" = "xgcc" ; then
    set_cc_and_cxx_for_gcc
    if test "x$cpu_base_type" != "xx86" ; then
      usage "gcc currently not supported for Solaris on SPARC"
      exit 1
    fi
    if test "x$fast_flag" = "xyes" ; then
      LDFLAGS="$LDFLAGS -O3"
      compiler_flags="$compiler_flags -O3"
      check_for_link_time_optimizer
    else
      if test "x$fast_flag" = "xgeneric" ; then
        LDFLAGS="$LDFLAGS -O2"
        compiler_flags="$compiler_flags -O2"
      else
        LDFLAGS="$LDFLAGS -O0"
        compiler_flags="$compiler_flags -O0"
      fi
    fi
  else
#Using Forte compiler (SunStudio)
    set_cc_and_cxx_for_forte
    compiler_flags="$compiler_flags -mt"
    LDFLAGS="$LDFLAGS -mt"
    compiler_flags="$compiler_flags -fsimple=1"
    compiler_flags="$compiler_flags -ftrap=%none"
    compiler_flags="$compiler_flags -xbuiltin=%all"
    compiler_flags="$compiler_flags -xlibmil"
    compiler_flags="$compiler_flags -xlibmopt"
    if test "x$fast_flag" = "xyes" ; then
      compiler_flags="$compiler_flags -xtarget=native"
      compiler_flags="$compiler_flags -xunroll=3"
      if test "x$with_link_time_optimizer" = "xyes" ; then
        compiler_flags="$compiler_flags -xipo"
        LDFLAGS="$LDFLAGS -xipo"
      fi
    else
      compiler_flags="$compiler_flags -xtarget=generic"
    fi
    if test "x$cpu_base_type" = "xx86" ; then
      compiler_flags="$compiler_flags -nofstore"
      base_cxx_flags="$base_cxx_flags -features=no%except"
      if test "x$fast_flag" = "xyes" ; then
        compiler_flags="$compiler_flags -xregs=frameptr"
        compiler_flags="$compiler_flags -xO4"
      else
        compiler_flags="$compiler_flags -xregs=no%frameptr"
        if test "x$fast_flag" = "xgeneric" ; then
          compiler_flags="$compiler_flags -xO2"
        else
          compiler_flags="$compiler_flags -xO0"
        fi
      fi
    else
#Using SPARC cpu with SunStudio (Forte) compiler
      ASFLAGS="$ASFLAGS -xarch=sparc"
      LDFLAGS="$LDFLAGS -xarch=sparc"
      base_cxxflags="$base_cxxflags -noex"
      base_cflags="$base_cflags -xstrconst"
      compiler_flags="$compiler_flags -xarch=sparc"
      if test "x$fast_flag" = "xyes" ; then
        compiler_flags="$compiler_flags -xbinopt=prepare"
        LDFLAGS="$LDFLAGS -xbinopt=prepare"
        compiler_flags="$compiler_flags -xO4"
      elif test "x$fast_flag" = "xgeneric" ; then
        compiler_flags="$compiler_flags -xO3"
      else
        compiler_flags="$compiler_flags -xO0"
      fi
    fi
  fi
}

#
# Mac OS X Section
#
set_macosx_configs()
{
  if test "x$cpu_base_type" != "xx86" || test "x$compiler" != "xgcc" ; then
    usage "Only gcc/x86 supported for Mac OS X"
    exit 1
  fi
#
# Optimize for space as long as it doesn't affect performance, use some
# optimisations also when not in fast mode.
#
  base_cxxflags="$base_cxxflags -felide-constructors"
  compiler_flags="$compiler_flags -fno-common"
  if test "x$m64" = "xyes" ; then
    compiler_flags="$compiler_flags -m64"
    compiler_flags="$compiler_flags -arch x86_64"
  else
    compiler_flags="$compiler_flags -m32"
    compiler_flags="$compiler_flags -arch i386"
  fi
  if test "x$fast_flag" != "xno" ; then
    compiler_flags="$compiler_flags -Os"
  else
    compiler_flags="$compiler_flags -O0"
  fi
  set_cc_and_cxx_for_gcc
}

#
# Use static linking for own modules and dynamic linking for system
# modules unless specifically requested to do everything statically.
# Should normally not be used; static_linking_flag kept in case someone 
# really needs it. Available only if developer flag is also set.
#
set_static_link_configs()
{
  if test "x$static_linking_flag" = "xyes" && test "x$developer_flag" = "xyes" ; then
    loc_static_link="--with-mysqld-ldflags=\"-all-static\""
    loc_static_link="$loc_static_link --with-client-ldflags=\"-all-static\""
  else
    loc_static_link="--with-mysqld-ldflags=\"-static\""
    loc_static_link="$loc_static_link --with-client-ldflags=\"-static\""
  fi
  base_configs="$base_configs $loc_static_link"
}

#
# Enable error injection in MySQL Server (for developer build only -
# extra check for developer flag required).
#
set_error_inject_configs()
{
  if test "x$error_inject_flag" = "xyes" && test "x$developer_flag" = "xyes" ; then
    base_configs="$base_configs --with-error-inject"
    if test "x$package" = "xndb" || test "x$package" = "xextended" ; then
      base_configs="$base_configs --with-ndb-ccflags='-DERROR_INSERT'"
    fi
  fi
}

set_default_package()
{
  if test "x$package" = "x" ; then
      package="extended"
  fi
}

set_autotool_flags()
{
  if test "x$use_autotools" = "x" ; then
    if test "x$developer_flag" = "xno" ; then
      use_autotools="no"
    else
      use_autotools="yes"
    fi
  fi
}

set_defaults_based_on_environment()
{
  if test ! -z "$MYSQL_DEVELOPER" ; then
    developer_flag="yes"
  fi
  if test ! -z "$MYSQL_DEVELOPER_DEBUG" ; then
    with_debug_flag="yes"
    fast_flag="no"
  fi
  if test ! -z "$MYSQL_DEVELOPER_PACKAGE" ; then
    package="$MYSQL_DEVELOPER_PACKAGE"
    parse_package
  fi
}

########################################################################

if test ! -f sql/mysqld.cc ; then
  die "You must run this script from the MySQL top-level directory"
fi

cpu_type=
package=
prefix="/usr/local/mysql"
parallelism="8"
fast_flag="generic"
compiler="gcc"
gpl="yes"
version_text=
developer_flag="no"
just_configure=
full_debug=
warning_mode=
with_flags=
error_inject_flag=
with_debug_flag=
compile_debug_flag=
strip_flag=
valgrind_flag=
static_linking_flag=
compiler_flags=
os=
cpu_base_type=
warnings=
c_warnings=
cflags=
base_cflags=
cxx_warnings=
base_cxxflags=
base_configs=
debug_flags=
cxxflags=
m64=
explicit_size_set=
datadir=
commands=
use_autotools=
engine_configs=
ASFLAGS=
LDFLAGS=
use_tcmalloc=
without_comment="yes"
with_fast_mutexes=
with_link_time_optimizer=
gcc_version="0"

set_defaults_based_on_environment

parse_options "$@"

set_autotool_flags
set_default_package

set -e

#
# Check for the CPU and set up CPU specific flags. We may reset them
# later.
# This call sets the cpu_arg and check_cpu_args parameters
#
path=`dirname $0`
if test "x$compiler" = "xgcc" ; then
  compiler=
fi
. "$path/check-cpu"
if test "x$compiler" = "x" ; then
  compiler="gcc"
fi
check_os
set_cpu_base
if test "x$?" = "x1" ; then
  exit 1
fi

#
# Set up c_warnings and cxx_warnings; add to compiler_flags.
# Possibly reset check_cpu_flags.
#
set_warning_flags

#
# Add to compiler_flags.
#
set_valgrind_flags
set_with_debug_flags
set_no_omit_frame_pointer_for_developers
set_debug_flag
set_gcc_special_options
set_icc_special_options

#
# Definitions of various packages possible to compile. The default is to
# build a source variant including all storage engines except InnoDB.
#
set_base_configs
  version_text="GPL version"
if test "x$package" = "xpro" ; then
  set_max_engines
  set_pro_package
elif test "x$package" = "xclassic" ; then
  set_classic_package
else
  die "No supported package was used, internal error"
fi
set_readline_package
set_static_link_configs
set_error_inject_configs

#
# This section handles flags for specific combinations of compilers,
# operating systems, and processors.
#

if test "x$os" = "xlinux" ; then
  set_linux_configs
elif test "x$os" = "xSolaris" ; then
  set_solaris_configs
elif test "x$os" = "xMacOSX" ; then
  set_macosx_configs
elif test "x$os" = "xbsd" ; then
  set_bsd_configs
else
  die "Operating system not supported by this script"
fi
set_ssl
#
# Final step before setting up commands is to set up proper make and
# proper libtoolize versions, and to determine whether to use ccache.
#
set_make_version
set_ccache_usage

#
# Set up commands variable from variables prepared for base 
# configurations, compiler flags, and warnings flags.
# 
if test "x$use_autotools" = "xyes" ; then
  init_auto_commands
fi
init_configure_commands

if test "x$just_configure" != "xyes" ; then
  add_make_commands
fi

#
# The commands variable now contains the entire command to be run for
# the build; we either execute it, or merely print it out.
#
if test "x$just_print" = "xyes" ; then
  echo "$commands"
else
  eval "set -x; $commands"
fi
