/*
 *  linux/arch/arm/mm/fault-armv.c
 *
 *  Copyright (C) 1995  Linus Torvalds
 *  Modifications for ARM processor (c) 1995-2003 Russell King
 *  Modifications for FASS (C) 2001-2003 Adam Wiggins and John Zaitseff
 *
 * FASS Notes:
 *   - The Modified Virtual Address (MVA) is changed into a Virtual Address (VA
 *     with ARM PID relocation removed) before being forwarded to the
 *     do_*_fault() handlers. Only the FASS handlers do_(cpd/domain)_fault()
 *     are aware of the ARM PID relocaiton.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/ptrace.h>
#include <linux/mm.h>
#include <linux/bitops.h>
#include <linux/init.h>

#include <asm/pgalloc.h>
#include <asm/pgtable.h>
#include <asm/proc/cpd.h>
#include <asm/proc/pid.h>

#include "fault.h"

/*
 * Some section permission faults need to be handled gracefully.
 * They can happen due to a __{get,put}_user during an oops.
 */
static int
do_sect_fault(unsigned long addr, unsigned int fsr, struct pt_regs *regs)
{
	struct task_struct *tsk = current;
	do_bad_area(tsk, tsk->active_mm, addr, fsr, regs);
	return 0;
}

/*
 * This abort handler always returns "fault".
 */
static int
do_bad(unsigned long addr, unsigned int fsr, struct pt_regs *regs)
{
	return 1;
}

#ifdef CONFIG_ARM_FASS
/* awiggins
 * do_cpd_fault: Handle Caching Page Directory 'invalid' entry miss (ie CPD
 *   entry is invalid) or a conflict miss (ie another process is using the CPD
 *   entry).
 * Reason:
 *   (1) CPD entry is just empty, ie not allocate to any process.
 *   (2) 'current' processes mm context has no section entry (ie no mapping).
 *   (3) CPD entry is tagged with another processes domain.
 * Action:
 *   (1) Load the section entry from the pg_dir of 'current' process.
 *   (2) Call do_translation_fault() and add entry if any mapping is granted,
 *       else the kernel will raise a seg fault.
 *   (3) Same as (1)/(2) depending on if 'current' process has a mapping in it's
 *       pg_dir or not.
 */
static int
do_cpd_fault(unsigned long v_addr, unsigned int fsr, struct pt_regs *regs)
{
	int fault;
	unsigned long mv_addr = va_to_mva(v_addr, current->mm);
	pmd_t pmd = *pmd_offset(pgd_offset(current->mm, v_addr), v_addr);

//	printk("** do_cpd_fault: **\n");

	/* 
	 * if there is a valid entry in the pg_dir of 'current' load
	 * it into CPD
	 */
	if (!pmd_none(pmd)) {
		goto success;
	} 

	/* Else take the page fault. and try again */
	if ((fault = do_translation_fault(v_addr, fsr, regs)))
		return fault;

	pmd = *pmd_offset(pgd_offset(current->mm, v_addr), v_addr);

	if (pmd_none(pmd))
		return fault;

success:
	cpd_load(pmd, mv_addr);

	return 0; /* Success */

}

/* awiggins
 * do_domain_fault: Handle domain faults, these are generated by conflict 
 *   misses or threads touching their currently allocated domain for the first
 *   time.
 * Reason:
 *   (1) CPD entry is tagged with the wrong domain.
 *   (2) A domain is allocated but it hasn't been activated.
 * Action:
 *   (1) Call do_cpd_fault() to handle the conflict miss.
 *   (2) Activate allocated domain in the CPU DACR and return.
 */
static int
do_domain_fault(unsigned long v_addr, unsigned int fsr, struct pt_regs *regs)
{
	int domain = cpd_get_active_domain(current->mm, v_addr);


	if (domain && !domain_active(get_dacr(), domain)) { 
		update_dacr(domain); /* modified */
     
		return 0; /* Success */
	}

//	printk("** do_domain_fault: called from %p **\n", __builtin_return_address(0));
	return do_cpd_fault(v_addr, fsr, regs);
}

/* do_domain_fixup: This is required to fixup page translation faults on a
 *   shared domain.
 */
static int
do_domain_fixup(unsigned long v_addr, unsigned int fsr, struct pt_regs *regs)
{
	unsigned long mv_addr = va_to_mva(v_addr, current->mm); /* Modified VA */
	pmd_t* cpd_p = pmd_offset(pgd_offset_k(mv_addr), mv_addr); 
	pmd_t pmd = *pmd_offset(pgd_offset(current->mm, v_addr), v_addr);

	/* This needs to be cleaned up a little */
	if ((pmd_val(*cpd_p) >> 10) != (pmd_val(pmd) >> 10)) {
		//printk("** do_domain_fixup: **\n");
		pmd_val(pmd) |= PMD_DOMAIN(pmd_domain(*cpd_p)); /* Tag CPD entry */
		cpu_set_pmd(cpd_p, pmd);
	}
  
	return do_page_fault(v_addr, fsr, regs);

}
#else
/*
 * Map functions for non-fass use.
 */
#define	do_cpd_fault	do_translation_fault
#define	do_domain_fixup	do_page_fault
#define	do_domain_fault	do_bad
#endif /* CONFIG_ARM_FASS */

static struct fsr_info {
	int	(*fn)(unsigned long addr, unsigned int fsr, struct pt_regs *regs);
	int	sig;
	const char *name;
} fsr_info[] = {
	{ do_bad,		SIGSEGV, "vector exception"		   },
	{ do_bad,		SIGILL,	 "alignment exception"		   },
	{ do_bad,		SIGKILL, "terminal exception"		   },
	{ do_bad,		SIGILL,	 "alignment exception"		   },
	{ do_bad,		SIGBUS,	 "external abort on linefetch"	   },
	{ do_cpd_fault,		SIGSEGV, "section translation fault"	   },
	{ do_bad,		SIGBUS,	 "external abort on linefetch"	   },
	{ do_domain_fixup,	SIGSEGV, "page translation fault"	   },
	{ do_bad,		SIGBUS,	 "external abort on non-linefetch" },
	{ do_domain_fault,	SIGSEGV, "section domain fault"		   },
	{ do_bad,		SIGBUS,	 "external abort on non-linefetch" },
	{ do_domain_fault,	SIGSEGV, "page domain fault"		   },
	{ do_bad,		SIGBUS,	 "external abort on translation"   },
	{ do_sect_fault,	SIGSEGV, "section permission fault"	   },
	{ do_bad,		SIGBUS,	 "external abort on translation"   },
	{ do_page_fault,	SIGSEGV, "page permission fault"	   },
};


void __init
hook_fault_code(int nr, int (*fn)(unsigned long, unsigned int, struct pt_regs *),
		int sig, const char *name)
{
	if (nr >= 0 && nr < ARRAY_SIZE(fsr_info)) {
		fsr_info[nr].fn   = fn;
		fsr_info[nr].sig  = sig;
		fsr_info[nr].name = name;
	}
}

/*
 * Dispatch a data abort to the relevant handler.
 */
/* awiggins
 *
 * do_DataAbort: Modified to check for a translation fault on a page
 *	 that should be classed as a domain fault on a
 *	 page. Translation faults have a higher priority then domain
 *	 fault. What can happen is  that the CPD entry a process
 *	 touches is not its own and the 2nd  level page table has no
 *	 entry for the faulting address, so you get a translation
 *	 fault on a page even though you're not accessing your own
 *	 pg_dir entry because the domain tag will be ignored.  In this
 *	 case we need to flag the fault as a domain fault on a page
 *	 to produce the correct behaviour of replacing the CPD entry
 *	 to one owned by the current process.
 * Notes:
 *   - For data aborts the faulting address is supplied by the ARM 
 *     Fault Address Register (FAR) as it is a Modified Virtual
 *     Address (MVA), NOT the Virtual Address (VA) as seen by the core
 *     and the programs.
 */

asmlinkage void
do_DataAbort(unsigned long mv_addr, unsigned int fsr, struct pt_regs *regs)
{
#ifdef CONFIG_ARM_FASS
	unsigned long addr = mva_to_va(mv_addr, current->mm); /* Unmodified VA */
	pmd_t cpd = *pmd_offset(pgd_offset_k(mv_addr), mv_addr);
#else
	unsigned long addr = mv_addr;
#endif
	const struct fsr_info *inf = fsr_info + (fsr & 15);

#ifdef CONFIG_KGDB
	if(kgdb_fault_expected)
		kgdb_handle_bus_error();
#endif

#ifdef CONFIG_ARM_FASS
	/* 
	 * The following code does the translation fault on a page to
	 * domain fault on a page check/conversion. Could be cleaned
	 * up probably
	 */
	if ((pmd_domain(cpd) >= DOMAIN_START) &&
	    !domain_active(get_dacr(), pmd_domain(cpd)) &&
	    ((fsr & 15) == 7)) { /* 7 is translation fault on page */
		inf = fsr_info + 9;   /* 9 is domain fault on page */
	}


#if 0
	switch (fsr & 0xf) {
	case 0xB:
	case 0x9:
		if ((((fsr & 0xF0) >> 4) != pmd_domain(cpd)) 
		    && (get_dacr() != 0x15)) {
			printk("** do_DataAbort: VA 0x%lx MVA 0x%lx fsr 0x%x %d dacr 0x%lx **\n",
			       addr, mv_addr, fsr & 0xF, (fsr & 0xF0) >> 4, get_dacr());
			dump_cpd(1);
		}
		break;

	default:
		break;
	}
#endif
#endif /* CONFIG_ARM_FASS */

	if (!inf->fn(addr, fsr, regs))
		return;

	printk(KERN_ALERT "Unhandled fault: %s (0x%03x) at va 0x%08lx mva 0x%08lx\n",
	       inf->name, fsr, addr, mv_addr);
	force_sig(inf->sig, current);
	show_pte(current->mm, addr);

#ifdef CONFIG_KGDB
	if(!user_mode(regs))
		do_kgdb(regs, inf->sig);
#endif

	die_if_kernel("Oops", regs, 0);
}

/* awiggins
 * do_prefetchAbort: Modified to check if the reason for the abort and call
 *   one of the following functions:
 *   - do_cpd_fault():         Fault due to invalid CPD entry.
 *   - do_domain_fault():      Fault due to inactive domain.
 *   - do_translation_fault(): All other cases.
 * Notes:
 *   - For prefetch aborts the faulting address is calculated by the kernel not
 *     supplied by the ARM Fault Address Register (FAR) so it is a Virtual
 *     Address (VA), NOT a Modified Virtual Address (MVA).
 */
asmlinkage void
do_PrefetchAbort(unsigned long addr, struct pt_regs *regs)
{
#ifdef CONFIG_ARM_FASS
	unsigned long mv_addr = va_to_mva(addr, current->mm); /* MVA */
	pmd_t cpd = *pmd_offset(pgd_offset_k(mv_addr), mv_addr);

	if (pmd_none(cpd)) {
		do_cpd_fault(addr, 0, regs);
	}
	else if(!domain_active(get_dacr(), pmd_domain(cpd))) {
		do_domain_fault(addr, 0, regs);
	}
	else {
		do_translation_fault(addr, 0, regs);
	}
#else
	do_translation_fault(addr, 0, regs);
#endif
}

/*
 * We take the easy way out of this problem - we make the
 * PTE uncacheable.  However, we leave the write buffer on.
 */
static int adjust_pte(struct vm_area_struct *vma, unsigned long address)
{
	pgd_t *pgd;
	pmd_t *pmd;
	pte_t *pte, entry;
	int ret = 0;

	pgd = pgd_offset(vma->vm_mm, address);
	if (pgd_none(*pgd))
		goto no_pgd;
	if (pgd_bad(*pgd))
		goto bad_pgd;

	pmd = pmd_offset(pgd, address);
	if (pmd_none(*pmd))
		goto no_pmd;
	if (pmd_bad(*pmd))
		goto bad_pmd;

	pte = pte_offset(pmd, address);
	entry = *pte;

	/*
	 * If this page isn't present, or is already setup to
	 * fault (ie, is old), we can safely ignore any issues.
         *
         * FASS: Marked as unbufferable too otherwise we could miss an update
         *   that is still in the write-buffer.
	 */
#ifdef CONFIG_ARM_FASS
#define	BUFFERABLE	L_PTE_BUFFERABLE
#else
#define	BUFFERABLE	(0)
#endif
	if (pte_present(entry) && pte_val(entry) &
            (L_PTE_CACHEABLE | BUFFERABLE)) {
		flush_cache_page(vma, address);
		pte_val(entry) &= ~(L_PTE_CACHEABLE | BUFFERABLE);
		set_pte(pte, entry);
		flush_tlb_page(vma, address);
		ret = 1;
	}
	return ret;

bad_pgd:
	pgd_ERROR(*pgd);
	pgd_clear(pgd);
no_pgd:
	return 0;

bad_pmd:
	pmd_ERROR(*pmd);
	pmd_clear(pmd);
no_pmd:
	return 0;
}

/* awiggins: Not using this as I'm too lazy to properly patch things for now */
static void
make_coherent(struct vm_area_struct *vma, unsigned long addr, struct page *page, int dirty)
{
	struct vm_area_struct *mpnt;
	struct mm_struct *mm = vma->vm_mm;
	unsigned long pgoff = (addr - vma->vm_start) >> PAGE_SHIFT;
	int aliases = 0;

	/*
	 * If we have any shared mappings that are in the same mm
	 * space, then we need to handle them specially to maintain
	 * cache coherency.
	 */
	for (mpnt = page->mapping->i_mmap_shared; mpnt;
	     mpnt = mpnt->vm_next_share) {
		unsigned long off;

		/*
		 * If this VMA is not in our MM, we can ignore it.
		 * Note that we intentionally don't mask out the VMA
		 * that we are fixing up.
		 */
		if (mpnt->vm_mm != mm || mpnt == vma)
			continue;

		/*
		 * If the page isn't in this VMA, we can also ignore it.
		 */
		if (pgoff < mpnt->vm_pgoff)
			continue;

		off = pgoff - mpnt->vm_pgoff;
		if (off >= (mpnt->vm_end - mpnt->vm_start) >> PAGE_SHIFT)
			continue;

		off = mpnt->vm_start + (off << PAGE_SHIFT);

		/*
		 * Ok, it is within mpnt.  Fix it up.
		 */
		aliases += adjust_pte(mpnt, off);
	}
	if (aliases)
		adjust_pte(vma, addr);
	else if (dirty)
		flush_cache_page(vma, addr);
}

#ifdef CONFIG_ARM_FASS
/* writable_pte: Tests if a page is writable. */
static int
writable_pte(struct vm_area_struct *vma, unsigned long address)
{
	pgd_t *pgd;
	pmd_t *pmd;
	pte_t *pte, entry;

	pgd = pgd_offset(vma->vm_mm, address);
	if (pgd_none(*pgd)) {
		return 0;
	}

	pmd = pmd_offset(pgd, address);
	if (pmd_none(*pmd)) {
		return 0;
	}

	pte = pte_offset(pmd, address);
	entry = *pte;

	//printk("** writable_pte: entry = 0x%x **\n", entry);

	if (pte_present(entry)) {
		//printk("** writable_pte: entry 0x%x writable %x **\n",
		//	   entry, pte_write(entry));

		return pte_write(entry);
	}
	return 0;
}
#endif /* CONFIG_ARM_FASS */

/*
 * Take care of architecture specific things when placing a new PTE into
 * a page table, or changing an existing PTE.  Basically, there are three
 * things that we need to take care of:
 *
 *  1. If PG_dcache_dirty is set for the page, we need to ensure
 *     that any cache entries for the kernels virtual memory
 *     range are written back to the page.
 *  2. If we have multiple shared mappings of the same space in
 *     an object, we need to deal with the cache synonym issues.
 *  3. If the page is part of a shared vma, here we check if vm_shared_data has
 *     a region_struct allocated. If not a suitable one is set.
 *
 * Note that the page_table_lock will be held.
 *
 * FASS Notes:
 *   - Since we do not flush the cache on every context switch shared
 *     mappings that map to different virtual addresses cause synonyms in the
 *     cache, these synonyms independent of whether the shared mapping is within
 *     the same or between different address-spaces. There are two possible
 *     solutions:
 * 
 *     1) Mark the shared mappings as uncachable and unbufferable. This reduces
 *        the performance of every access of a shared mapping, infact this is
 *        only needed if the mappings map to different virtual addresses. This
 *        is how ARM Linux currently handles shared mapping, ie synonyms, within
 *        the same address-space.
 *
 *     2) Only one shared mapping is enabled at a time. This can infact be
 *        extented to only one set of mappings that map the shared physical
 *        frame to the same virtual page are enabled at a time. The performance
 *        lost here is a page fault every time a disabled mapping is re-enabled
 *        and the synonym at different addresses are disabled. This solution is
 *        also independed with regard to the whether the synonym is within the
 *        same address-space or between address-spaces. An additional problem
 *        with this solution is any shared mappings within the PID relocated
 *        area, ie the first 32MB will not flag as synonyms when it will be if
 *        the sharing address-spaces have different PID's. This could be checked
 *        for easily enough though.
 *
 *     I want to benchmark the performance of both solutions but for now I'll
 *     use 1) since it is ARM Linux's original solution. I'm also going to add
 *     a check to only mark the pages un(cacheable/bufferable) iff at least one
 *     or more mappings are writable. Otherwise the mappings will always be
 *     consistent.
 *  - The checks to use the same region are technically incorrect, if we have a
 *    process sharing a mapping read-only and another one sharing it read-write
 *    both will have read-write access to it. Checks needed to be added here to
 *    make sure both the access rights and the size of the shared mapping are
 *    the same. If they a new region must be allocated.
 */
void update_mmu_cache(struct vm_area_struct *vma, unsigned long addr, pte_t pte)
{
	unsigned long pfn = pte_pfn(pte);
	struct page *page;
#ifdef CONFIG_ARM_FASS
	struct vm_area_struct *mpnt;
	struct mm_struct *mm;
	unsigned long pgoff;
	int synonyms, writable;

	if (!pfn_valid(pfn))
		return;
	page = pfn_to_page(pfn);

	if (!page->mapping)
		return;

	/* region_struct updates */
	if ((vma->vm_flags & VM_SHARED) && !vma->vm_sharing_data) {

		/* Check for existing region to use */
		for (mpnt = page->mapping->i_mmap_shared; mpnt;
		     mpnt = mpnt->vm_next_share) {

			/* If we have a shared vma of the same access type */
			if (mpnt->vm_sharing_data &&
			    (vma->vm_start == mpnt->vm_start) &&
			    (vma->vm_pgoff == mpnt->vm_pgoff)) {

				vma->vm_sharing_data = mpnt->vm_sharing_data;
				((struct region_struct*)vma->vm_sharing_data)->mm_count++;

				break;
			}
		}

		/* First vma of the shared set */
		if (!vma->vm_sharing_data) {
			vma->vm_sharing_data = cpd_allocate_region();
		}
	}

	{
		int dirty = test_and_clear_bit(PG_dcache_dirty, &page->flags);

		if (dirty) {
			unsigned long kvirt = (unsigned long)page_address(page);
			cpu_cache_clean_invalidate_range(kvirt, kvirt + PAGE_SIZE, 0);
		}
	}
	mm = vma->vm_mm;
	pgoff = (addr - vma->vm_start) >> PAGE_SHIFT;

	writable = writable_pte(vma, addr);
	synonyms = 0;

	/* Need to first check for writable synonyms */
	for (mpnt = page->mapping->i_mmap_shared; mpnt && ~writable;
	    mpnt = mpnt->vm_next_share) {
		unsigned long off;

		/* If the shared mapping has the same region it isn't a synonym */
		if (vma->vm_sharing_data &&
		    (vma->vm_sharing_data == mpnt->vm_sharing_data))
			continue;

		/*
		 * If the page isn't in this VMA, we can also ignore it.
		 */
		if (pgoff < mpnt->vm_pgoff)
			continue;
	  
		off = pgoff - mpnt->vm_pgoff;
		if (off >= (mpnt->vm_end - mpnt->vm_start) >> PAGE_SHIFT)
			continue;

		writable = writable_pte(mpnt, mpnt->vm_start + (off << PAGE_SHIFT));
		synonyms++;
	}

	/* If there are synonyms or they are all read-only do nothing */ 
	if (!synonyms || !writable) {
		return;
	}

	printk("** update_mmu_cache: Mark writable synonym uncachable **\n");

	/* Mark all synonyms as uncachable */
	for (mpnt = page->mapping->i_mmap_shared; mpnt;
	     mpnt = mpnt->vm_next_share) {
		unsigned long off;
	  
		/*
		 * If the page isn't in this VMA, we can also ignore it.
		 */
		if (pgoff < mpnt->vm_pgoff)
			continue;
	  
		off = pgoff - mpnt->vm_pgoff;
		if (off >= (mpnt->vm_end - mpnt->vm_start) >> PAGE_SHIFT)
			continue;
	  
		/*
		 * Ok, it is within mpnt.  Fix it up.
		 */
		adjust_pte(mpnt, mpnt->vm_start + (off << PAGE_SHIFT));
	}
	adjust_pte(vma, addr);
#else
	if (!pfn_valid(pfn))
		return;
	page = pfn_to_page(pfn);
	if (page->mapping) {
		int dirty = test_and_clear_bit(PG_dcache_dirty, &page->flags);

		if (dirty) {
			unsigned long kvirt = (unsigned long)page_address(page);
			cpu_cache_clean_invalidate_range(kvirt, kvirt + PAGE_SIZE, 0);
		}

		make_coherent(vma, addr, page, dirty);
	}
#endif /* CONFIG_ARM_FASS */
}
