///
///	@file 	genDepend.cpp 
/// @brief 	Generate C, C++ Include Dependencies
///
///	usage:  genDepend [-q] [-o objDir] [-I includes] files...
///
////////////////////////////////////////////////////////////////////////////////
//
//	Copyright (c) Mbedthis Software LLC, 2003-2004. All Rights Reserved.
//	The latest version of this code is available at http://www.mbedthis.com
//
//	This software is open source; you can redistribute it and/or modify it 
//	under the terms of the GNU General Public License as published by the 
//	Free Software Foundation; either version 2 of the License, or (at your 
//	option) any later version.
//
//	This program is distributed WITHOUT ANY WARRANTY; without even the 
//	implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
//	See the GNU General Public License for more details at:
//	http://www.mbedthis.com/downloads/gplLicense.html
//	
//	This General Public License does NOT permit incorporating this software 
//	into proprietary programs. If you are unable to comply with the GPL, a 
//	commercial license for this software and support services are available
//	from Mbedthis Software at http://www.mbedthis.com
//
////////////////////////////////// Includes ////////////////////////////////////

#include	<ctype.h>
#include	<signal.h>
#include	<stdio.h>
#include	<stdlib.h>
#include	<string.h>
#include	<time.h>
#if WIN || _WIN32
	#include	<direct.h>
	#include	<io.h>
	#include	<windows.h>
#else
	#include	<libgen.h>
	#include	<unistd.h>
#endif

#include	"getopt.h"

////////////////////////////////// Defines /////////////////////////////////////

#define MAX_INC		128						// Max include directories
#define MAX_DEPEND	4096					// Max dependencies
#ifndef R_OK
#define R_OK		4
#endif

/////////////////////////////////// Locals /////////////////////////////////////

static char		*dependencies[MAX_DEPEND];
static int		finished;
static FILE		*fp;	
static char		*includeDir[MAX_INC];
static int 		numDependencies;
static int 		numIncludeDir;
static char		*program;
static char		*objDir;
static int		quiet;

///////////////////////////// Forward Declarations /////////////////////////////

static char 	*mprGetBaseName(char *name);
static int 		mprGetDirName(char *buf, int bufsize, char *path);
static void		openSignals();
static int 		findDependencies(FILE *fp, char *fname);
static int 		depSort(const void *p1, const void *p2);
static char		*mapExtension(char *path);
static void		catchInterrupt(int signo);
static char		*mapDelimiters(char *s);

///////////////////////////////////// Code /////////////////////////////////////

int main(int argc, char *argv[])
{
	time_t		now;
	struct tm	*t;
	char		path[2048], cwd[2048], dirName[2048], parent[2048];
	char		*ext, *previous;
	int 		errors, c, j, i;

	errors = 0;
	program = mprGetBaseName(argv[0]);
	quiet = 0;
	objDir = "$(BLD_OBJ_DIR)";

	while ((c = getopt(argc, argv, "?I:o:q")) != EOF) {
		switch(c) {
		case 'I':
			if (numIncludeDir >= MAX_INC) {
				fprintf(stderr, "Too many include directories\n");
				exit(1);
			}
			includeDir[numIncludeDir++] = strdup(optarg);
			break;
	
		case 'o':
			objDir = optarg;
			break;

		case 'q':
			quiet++;
			break;

		case '?':
			errors++;
			break;
		}
	}
	if (errors) {
		fprintf(stderr, 
			"%s: usage: [-q] [-o objDir] [-I includeDir] files...\n", program);
		exit(2);
	}

	includeDir[numIncludeDir++] = strdup(".");
#if !WIN && !_WIN32
	includeDir[numIncludeDir++] = strdup("/usr/include");
#endif

	openSignals();
	if ((fp = fopen("make.dep", "w")) == 0) {
		fprintf(stderr, "Cant open make.dep\n");
		exit(255);
	}

	now = time(0);
	t = localtime(&now);
	fprintf(fp, "#\n#	make.dep -- Makefile dependencies. Generated by genDepend.\n#\n\n");

	getcwd(cwd, sizeof(cwd));
	mapDelimiters(cwd);

	fprintf(fp, "all: compile\n\n");

#if NEW || 1
	fprintf(fp, "BLD_TOP := ");
	strcpy(dirName, cwd);
	for (i = 0; *dirName; i++) {
		sprintf(path, "%s/make.rules", dirName);
		if (access(path, R_OK) == 0) {
			break;
		}
		mprGetDirName(parent, sizeof(parent), dirName);
		strcpy(dirName, parent);
		if (i > 0) {
			fprintf(fp, "/..");
		} else {
			fprintf(fp, "..");
		}
	}
	if (i == 0) {
		fprintf(fp, ".");
	}
	fprintf(fp, "\n\n");
#else
	fprintf(fp, "#\n#	Find make.rules in the top level directory.\n#	");
	fprintf(fp, "This trick enables the Makefile to be position "
				"independent.\n#\n");
	fprintf(fp, "parents := . .. ../.. ../../.. ../../../.. ../../../../..\n");
	fprintf(fp, 
		"base := $(foreach dir,$(parents),$(wildcard $(dir)/make.rules))\n");
	fprintf(fp, "BLD_TOP := $(subst /make.rules,,$(word 1,$(base)))\n\n");
#endif

	fprintf(fp, "#\n#	Read the build configuration settings and make"
				"variable definitions.\n#\n"); 
	fprintf(fp, "include $(BLD_TOP)/config.make\n\n");

	fprintf(fp, "FILES =");
	for (i = optind; i < argc; i++) {
		if (access(argv[i], R_OK) != 0) {
			continue;
		}
		strncpy(path, argv[i], sizeof(path));
		mapExtension(path);
		fprintf(fp, " \\\n\t%s/%s", objDir, mprGetBaseName(path));
	}
	fprintf(fp, "\n");

	for (i = optind; !finished && i < argc; i++) {
		if (*argv[i] == '*') {
			continue;
		}
		strcpy(path, argv[i]);
		ext = mapExtension(path);
		fprintf(fp, "\n%s/%s: ", objDir, mprGetBaseName(path));

		numDependencies = 0;
		findDependencies(fp, argv[i]);
		qsort(dependencies, numDependencies, sizeof(char*), depSort);

		previous = "";
		for (j = 0; j < numDependencies; j++) {
			if (strcmp(previous, dependencies[j]) != 0) {
				fprintf(fp, " \\\n\t%s", dependencies[j]);
			}
			previous = dependencies[j];
		}
		for (j = 0; j < numDependencies; j++) {
			free(dependencies[j]);
		}
		fprintf(fp, "\n");
	}

	fprintf(fp, "\n#\n#	Read the Makefile rules\n#\n");
	fprintf(fp, "include $(BLD_TOP)/make.rules\n");
	fprintf(fp, "include $(BLD_TOP)/make.os\n");
	fclose(fp);

	return 0;
}


////////////////////////////////////////////////////////////////////////////////
//
//	Do the C and C++ dependencies
//

static int findDependencies(FILE *fp, char *fname)
{
	FILE	*ifp;
	char	path[2048], buf[8192], dirName[2048];
	char	*cp, *ep;
	int		line, i, j;

	if ((ifp = fopen(fname, "r")) == 0) {
		if (!quiet) {
			fprintf(stderr, "Cant open %s\n", fname);
		}
		return -1;
	}

	for (line = 0; ! feof(ifp); line++) {
		if (fgets(buf, sizeof(buf), ifp) == 0)
			break;
		cp = buf;
		if (*cp++ != '#') {
			continue;
		}
		while (*cp == '\t' || *cp == ' ') {
			cp++;
		}
		if (*cp != 'i' || strncmp(cp, "include", 7)) {
			continue;
		}
		cp += 7;

		while (*cp == '\t' || *cp == ' ' || *cp == '\"') {
			cp++;
		}
		//
		//	Skip system headers
		//
		if (*cp == '<') {
			continue;
		}

		ep = cp;
		while (isalnum(*ep) || *ep == '_' || *ep == '.' || *ep == '/' || 
				*ep == '-') {
			ep++;
		}
		*ep = '\0';

		strcpy(buf, cp);
		if (buf[0] == '/' || (buf[0] == '.' && buf[1] == '.')) {
			if (access(buf, R_OK) < 0) {
				if (!quiet) {
					fprintf(stderr, "Cant find include %s\n", buf);
				}
				continue;
			}

		} else {
			//
			//	First search relative to the including file
			//
			mprGetDirName(dirName, sizeof(dirName), fname);
			if (*dirName) {
				sprintf(path, "%s/%s", dirName, buf);
			} else {
				strcpy(path, dirName);
			}
			if (access(path, R_OK) < 0) {
				for (j = 0; j < numIncludeDir; j++) {
					sprintf(path, "%s/%s", includeDir[j], buf);
					if (access(path, R_OK) == 0) {
						break;
					}
				}
				if (j == numIncludeDir) {
					if (!quiet) {
						fprintf(stderr, "Cant find include %s in %s at %d\n", 
							buf, fname, line);
					}
					continue;
				}
			}
		}

		if (numDependencies >= MAX_DEPEND) {
			fprintf(stderr, "Too many dependencies\n");
		} else {
			for (i = 0; i < numDependencies; i++) {
				if (strcmp(path, dependencies[i]) == 0)
					break;
			}
			if (i == numDependencies) {
				dependencies[numDependencies++] = strdup(path);
				findDependencies(fp, path);
			}
		}
	}
	fclose(ifp);

	return 0;
}

////////////////////////////////////////////////////////////////////////////////

static int depSort(const void *p1, const void *p2)
{
	char	*s1, *s2;

	s1 = *(char**) p1;
	s2 = *(char**) p2;

	return strcmp(s1, s2);
}

////////////////////////////////////////////////////////////////////////////////

static char *mapDelimiters(char *s)
{
	char	*p;

	if (s == 0) {
		return 0;
	}
	for (p = s; *p; p++) {
		if (*p == '\\') {
			*p = '/';
		} else if (*p == ':') {
			s = &p[1];
		}
	}
	return s;
}

////////////////////////////////////////////////////////////////////////////////

static char *mapExtension(char *path)
{
	static char	ext[16];
	char		*cp;
	char		*object;

	object = "$(BLD_OBJ)";
	if ((cp = strrchr(path, '.'))) {
		strcpy(ext, cp);
		if (strcmp(cp, ".c") == 0) {
			strcpy(cp, object);
		} else if (strcmp(cp, ".cpp") == 0) {
			strcpy(cp, object);
		}
	}
	return ext;
}

////////////////////////////////////////////////////////////////////////////////
//
//	Return the last portion of a pathname
//

static char *mprGetBaseName(char *name)
{
	char *cp;

	cp = strrchr(name, '/');

	if (cp == 0) {
		cp = strrchr(name, '\\');
		if (cp == 0) {
			return name;
		}
	} 
	if (cp == name) {
		if (cp[1] == '\0') {
			return name;
		}
	} else {
		if (cp[1] == '\0') {
			return "";
		}
	}
	return &cp[1];
}

////////////////////////////////////////////////////////////////////////////////
//
//	Return the directory portion of a pathname into the users buffer.
//

int mprGetDirName(char *buf, int bufsize, char *path)
{
	char	*cp;
	int		dlen;

	cp = strrchr(path, '/');
	if (cp == 0) {
#if WIN
		cp = strrchr(path, '\\');
		if (cp == 0)
#endif
		{
			buf[0] = '\0';
			return 0;
		}
	}

	if (cp == path && cp[1] == '\0') {
		strcpy(buf, ".");
		return 0;
	}

	dlen = cp - path;
	if (dlen < bufsize) {
		if (dlen == 0) {
			dlen++;
		}
		memcpy(buf, path, dlen);
		buf[dlen] = '\0';
		return 0;
	}
	return -1;
}

////////////////////////////////////////////////////////////////////////////////
//
//	Initialize signals
//

static void openSignals() 
{
#if !WIN && !_WIN32
	struct sigaction	act;

	act.sa_flags = 0;
	sigemptyset(&act.sa_mask);

	act.sa_handler = catchInterrupt;
	sigaction(SIGINT, &act, 0);
#endif
}

////////////////////////////////////////////////////////////////////////////////

static void catchInterrupt(int signo)
{
	finished++;
}


//
// Local variables:
// tab-width: 4
// c-basic-offset: 4
// End:
// vim:tw=78
// vim600: sw=4 ts=4 fdm=marker
// vim<600: sw=4 ts=4
//
