///
///	@file 	httpComp.cpp
/// @brief 	Compile files and web pages and documents into C++ source.
///
///	Usage: httpComp -p prefix -r romName filelist >webrom.c
///
///	Prefix is a string to be removed from the front of all file names.
///	RomName is the name of data structure to hold the compiled files.
///
////////////////////////////////////////////////////////////////////////////////
//
//	Copyright (c) Mbedthis Software LLC, 2003-2004. All Rights Reserved.
//	Portions Copyright (c) GoAhead Software Inc., 1995-2000.
//		The bin2C routine was adapted from the GoAhead WebServer.
//	The latest version of this code is available at http://www.mbedthis.com
//
//	This software is open source; you can redistribute it and/or modify it 
//	under the terms of the GNU General Public License as published by the 
//	Free Software Foundation; either version 2 of the License, or (at your 
//	option) any later version.
//
//	This program is distributed WITHOUT ANY WARRANTY; without even the 
//	implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
//	See the GNU General Public License for more details at:
//	http://www.mbedthis.com/downloads/gplLicense.html
//	
//	This General Public License does NOT permit incorporating this software 
//	into proprietary programs. If you are unable to comply with the GPL, a 
//	commercial license for this software and support services are available
//	from Mbedthis Software at http://www.mbedthis.com
//
////////////////////////////////// Includes ////////////////////////////////////

#include	"http/http.h"

///////////////////////////// Forward Declarations /////////////////////////////

static void printUsage();
static int	binToC(char *files, char *romName, char *prefix);

//////////////////////////////////// Code //////////////////////////////////////
//
//	Main program
// 

int main(int argc, char *argv[])
{
	char	*programName, *prefix, *romName, *argp;
	int		c, errflg, nextArg;

	programName = mprGetBaseName(argv[0]);
	errflg = 0;
	prefix = "";
	romName = "defaultRomFiles";

	MprCmdLine cmdLine(argc, argv, "p:r:");
	while ((c = cmdLine.next(&argp)) != EOF) {
		switch(c) {
		case 'p':
			prefix = argp;
			break;

		case 'r':
			romName = argp;
			break;

		default:
			errflg++;
			break;
		}
	}

	nextArg = cmdLine.firstArg();
	if ((nextArg + 1) > argc) {
		errflg++;
	}

	if (errflg) {
		printUsage();
		exit(2);
	}	

	if (binToC(argv[nextArg], romName, prefix) < 0) {
		return -1;
	}
	return 0;
}

////////////////////////////////////////////////////////////////////////////////
//
//	Output usage message
 

static void printUsage()
{
	mprFprintf(MPR_STDERR, 
		"usage: httpComp -p prefix -r romName filelist >output.cpp\n");
}

////////////////////////////////////////////////////////////////////////////////

/// 
///	Encode the files as C++ code
///

static int binToC(char *files, char *romName, char *prefix)
{
	struct stat		finfo;
	FILE			*fp;
	time_t			now;
	char			fileBuf[MPR_MAX_FNAME], buf[512], timeBuf[80];
	char			*cp, *sl, *p;
	int				j, i, len, fd, nFile;

	if ((fp = fopen(files, "r")) == NULL) {
		mprFprintf(MPR_STDERR, "Can't open file list %s\n", files);
		return MPR_ERR_CANT_OPEN;
	}

	time(&now);
	mprPrintf("//\n//	romFiles.cpp -- Compiled Files\n//\n");
	mprCtime(&now, timeBuf, sizeof(timeBuf));
	timeBuf[strlen(timeBuf) - 1] = '\0';
	mprPrintf("//	GENERATED by httpComp on %s \n//\n", timeBuf);
	mprPrintf("#include \"http/http.h\"\n\n");
#if UNUSED
	mprPrintf("#if BLD_FEATURE_NAMESPACES\n");
	mprPrintf("using namespace mbedthis::mpr;\n");
	mprPrintf("using namespace mbedthis::http;\n");
	mprPrintf("#endif\n\n");
#endif
	mprPrintf("#if BLD_FEATURE_ROMFS\n");

	//
	//	Open each input file and compile
	//
	nFile = 0;
	while (fgets(fileBuf, sizeof(fileBuf), fp) != NULL) {
		if ((p = strchr(fileBuf, '\n')) || (p = strchr(fileBuf, '\r'))) {
			*p = '\0';
		}
		if (*fileBuf == '\0') {
			continue;
		}
		if (stat(fileBuf, &finfo) == 0 && finfo.st_mode & S_IFDIR) {
			continue;
		} 
		if ((fd = open(fileBuf, O_RDONLY | O_BINARY)) < 0) {
			mprFprintf(MPR_STDERR, "Can't open file %s\n", fileBuf);
			return -1;
		}
		mprPrintf("static uchar _file_%d[] = {\n", nFile);

		while ((len = read(fd, buf, sizeof(buf))) > 0) {
			p = buf;
			for (i = 0; i < len; ) {
				mprPrintf("    ");
				for (j = 0; p < &buf[len] && j < 16; j++, p++) {
					mprPrintf("%3d,", (uchar) *p);
				}
				i += j;
				mprPrintf("\n");
			}
		}
		mprPrintf("    0 };\n\n");

		close(fd);
		nFile++;
	}
	fclose(fp);

	//
	//	Now output the page index
	// 
	mprPrintf("MaRomInode %s[] = {\n", romName);

	if ((fp = fopen(files, "r" MPR_TEXT)) == NULL) {
		mprFprintf(MPR_STDERR, "Can't open file list %s\n", files);
		return -1;
	}
	nFile = 0;
	while (fgets(fileBuf, sizeof(fileBuf), fp) != NULL) {
		if ((p = strchr(fileBuf, '\n')) || (p = strchr(fileBuf, '\r'))) {
			*p = '\0';
		}
		if (*fileBuf == '\0') {
			continue;
		}
		//
		//	Replace the prefix with a leading "/"
		// 
		if (strncmp(fileBuf, prefix, strlen(prefix)) == 0) {
			cp = &fileBuf[strlen(prefix)];
		} else {
			cp = fileBuf;
		}
		while((sl = strchr(fileBuf, '\\')) != NULL) {
			*sl = '/';
		}
		if (*cp == '/') {
			cp++;
		}

		if (*cp == '.' && cp[1] == '\0') {
			cp++;
		}
		if (stat(fileBuf, &finfo) == 0 && finfo.st_mode & S_IFDIR) {
			mprPrintf("    { \"/%s\", 0, 0, 0 },\n", cp);
			continue;
		}
		mprPrintf("    { \"/%s\", _file_%d, %d, %d },\n", cp, nFile, 
			finfo.st_size, nFile);
		nFile++;
	}
	fclose(fp); 
	
	mprPrintf("    { 0, 0, 0 },\n");
	mprPrintf("};\n");
	mprPrintf("#endif // BLD_FEATURE_ROMFS\n");

	fclose(fp);
	return 0;
}

////////////////////////////////////////////////////////////////////////////////

//
// Local variables:
// tab-width: 4
// c-basic-offset: 4
// End:
// vim:tw=78
// vim600: sw=4 ts=4 fdm=marker
// vim<600: sw=4 ts=4
//
