#include "ucnhash.h"

/*
 * The hash is produced using the algorithm described in
 * "Optimal algorithms for minimal perfect hashing",
 * G. Havas, B.S. Majewski.  Available as a technical report
 * from the CS department, University of Queensland
 * (ftp://ftp.cs.uq.oz.au/).
 *
 * Generated using a heavily tweaked version of Andrew Kuchling's
 * perfect_hash.py: 
 * http://starship.python.net/crew/amk/python/code/perfect-hash.html
 *
 * Generated on: Fri Jul 14 08:00:58 2000
 */

#define k_cHashElements 18836
#define k_cchMaxKey  83
#define k_cKeys  10538


staticforward const unsigned short G[k_cHashElements]; 
staticforward const _Py_UnicodeCharacterName aucn[k_cKeys];   

static long f1(const char *key, unsigned int cch)
{
    register int len;
    register unsigned char *p;
    register unsigned long x;

    len = cch;
    p = (unsigned char *) key;
    x = 0x64fc2234;
    while (--len >= 0)
    {   
        /* (1000003 * x) ^ toupper(*(p++)) 
         * translated to handle > 32 bit longs 
         */
        x = (0xf4243 * x);
        x = x & 0xFFFFFFFF;
        x = x ^ toupper(*(p++));
    }
    x ^= cch + 10;
    if (x == 0xFFFFFFFF)
        x = 0xfffffffe;
    if (x & 0x80000000) 
    {
        /* Emulate 32-bit signed (2's complement) modulo operation */
        x = (~x & 0xFFFFFFFF) + 1;
        x %= k_cHashElements;
        if (x != 0)
        {
            x = x + (~k_cHashElements & 0xFFFFFFFF) + 1;
            x = (~x & 0xFFFFFFFF) + 1;
        }
    }
    else
        x %= k_cHashElements;
    return x;
}

    
static long f2(const char *key, unsigned int cch)
{
    register int len;
    register unsigned char *p;
    register unsigned long x;

    len = cch;
    p = (unsigned char *) key;
    x = 0x8db7d737;
    while (--len >= 0)
    {   
        /* (1000003 * x) ^ toupper(*(p++)) 
         * translated to handle > 32 bit longs 
         */
        x = (0xf4243 * x);
        x = x & 0xFFFFFFFF;
        x = x ^ toupper(*(p++));
    }
    x ^= cch + 10;
    if (x == 0xFFFFFFFF)
        x = 0xfffffffe;
    if (x & 0x80000000) 
    {
        /* Emulate 32-bit signed (2's complement) modulo operation */
        x = (~x & 0xFFFFFFFF) + 1;
        x %= k_cHashElements;
        if (x != 0)
        {
            x = x + (~k_cHashElements & 0xFFFFFFFF) + 1;
            x = (~x & 0xFFFFFFFF) + 1;
        }
    }
    else
        x %= k_cHashElements;
    return x;
}

    
static unsigned long hash(const char *key, unsigned int cch)
{
    return ((unsigned long)(G[ f1(key, cch) ]) + (unsigned long)(G[ f2(key, cch) ]) ) % k_cHashElements;
}

const void *getValue(unsigned long iKey)
{
    return &aucn[iKey];
}

/* Helper for adding objects to dictionaries. Check for errors with
   PyErr_Occurred() */
static 
void insobj(PyObject *dict,
     char *name,
     PyObject *v)
{
    PyDict_SetItemString(dict, name, v);
    Py_XDECREF(v);
}

static const _Py_UCNHashAPI hashAPI = 
{
    k_cKeys,
    k_cchMaxKey,
    &hash,
    &getValue,
};

static  
PyMethodDef Module_methods[] =
{   
    {NULL, NULL},
};

static char *Module_docstring = "ucnhash hash function module";

/* Error reporting for module init functions */

#define Py_ReportModuleInitError(modname) {			\
    PyObject *exc_type, *exc_value, *exc_tb;			\
    PyObject *str_type, *str_value;				\
								\
    /* Fetch error objects and convert them to strings */	\
    PyErr_Fetch(&exc_type, &exc_value, &exc_tb);		\
    if (exc_type && exc_value) {				\
	    str_type = PyObject_Str(exc_type);			\
	    str_value = PyObject_Str(exc_value);			\
    }								\
    else {							\
	   str_type = NULL;					\
	   str_value = NULL;					\
    }								\
    /* Try to format a more informative error message using the	\
       original error */					\
    if (str_type && str_value &&				\
	    PyString_Check(str_type) && PyString_Check(str_value))	\
	    PyErr_Format(						\
   		    PyExc_ImportError,				\
		    "initialization of module "modname" failed "	\
		    "(%s:%s)",					\
		PyString_AS_STRING(str_type),			\
		PyString_AS_STRING(str_value));			\
    else							\
	    PyErr_SetString(					\
		    PyExc_ImportError,				\
		    "initialization of module "modname" failed");	\
    Py_XDECREF(str_type);					\
    Py_XDECREF(str_value);					\
    Py_XDECREF(exc_type);					\
    Py_XDECREF(exc_value);					\
    Py_XDECREF(exc_tb);						\
}


/* Create PyMethodObjects and register them in the module's dict */
DL_EXPORT(void) 
initucnhash(void)
{
    PyObject *module, *moddict;
    /* Create module */
    module = Py_InitModule4("ucnhash", /* Module name */
             Module_methods, /* Method list */
             Module_docstring, /* Module doc-string */
             (PyObject *)NULL, /* always pass this as *self */
             PYTHON_API_VERSION); /* API Version */
    if (module == NULL)
        goto onError;
    /* Add some constants to the module's dict */
    moddict = PyModule_GetDict(module);
    if (moddict == NULL)
        goto onError;

    /* Export C API */
    insobj(
        moddict,
        "ucnhashAPI",
        PyCObject_FromVoidPtr((void *)&hashAPI, NULL));
    
onError:
    /* Check for errors and report them */
    if (PyErr_Occurred())
        Py_ReportModuleInitError("ucnhash");
    return;
}

static const unsigned short G[] = 
{ 0, 0, 0, 0, 0, 0, 0, 11953, 0, 0, 0, 9525, 0, 0, 0, 0, 
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 8379, 0, 0, 0, 0, 
 0, 0, 0, 0, 3426, 0, 0, 0, 0, 0, 0, 0, 15561, 0, 0, 0, 
 9861, 4304, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2876, 
 0, 0, 0, 0, 0, 0, 14935, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3610, 0, 0, 0, 0, 
 11753, 0, 0, 0, 0, 0, 0, 0, 0, 5516, 0, 12757, 0, 0, 0, 
 0, 0, 0, 0, 6111, 10127, 0, 0, 0, 0, 12107, 0, 0, 0, 0, 
 0, 0, 0, 0, 0, 1112, 16812, 7868, 0, 4523, 8156, 0, 0, 0, 
 0, 0, 0, 0, 8333, 10969, 0, 0, 0, 0, 3166, 0, 0, 0, 16562, 
 0, 0, 0, 14539, 0, 0, 0, 11038, 9829, 9342, 2598, 0, 0, 0, 
 0, 3920, 0, 0, 0, 0, 0, 0, 0, 0, 3292, 0, 0, 4859, 0, 11347, 
 4380, 0, 7279, 7152, 0, 0, 0, 0, 8923, 0, 0, 12282, 0, 6341, 
 0, 0, 0, 0, 0, 0, 5170, 0, 9366, 2765, 15341, 0, 0, 0, 0, 
 0, 0, 0, 0, 0, 0, 3646, 9501, 7605, 0, 0, 0, 0, 0, 0, 0, 
 0, 0, 0, 0, 0, 0, 0, 6980, 0, 6035, 11135, 0, 0, 0, 0, 0, 
 0, 0, 0, 0, 15770, 0, 0, 4362, 0, 0, 0, 0, 0, 0, 0, 0, 
 0, 0, 0, 0, 8371, 3425, 5911, 14745, 248, 0, 0, 0, 0, 0, 
 0, 0, 0, 877, 0, 0, 0, 0, 0, 0, 8592, 0, 0, 0, 0, 0, 0, 
 18225, 16638, 0, 0, 6933, 644, 0, 3481, 0, 0, 0, 820, 0, 0, 
 4178, 0, 0, 0, 0, 4463, 17116, 5252, 0, 0, 12656, 0, 7208, 
 16280, 17140, 2486, 0, 16076, 0, 0, 0, 0, 0, 0, 13341, 0, 
 889, 500, 0, 0, 422, 0, 7971, 0, 0, 0, 10624, 0, 17241, 0, 
 0, 0, 4910, 0, 0, 0, 0, 0, 0, 0, 0, 17764, 0, 687, 0, 0, 
 3623, 12593, 15985, 0, 8813, 0, 0, 7118, 0, 0, 0, 0, 0, 0, 
 0, 2864, 0, 0, 0, 0, 0, 12314, 0, 10396, 0, 13564, 0, 0, 
 13957, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5421, 0, 0, 3295, 
 0, 0, 10087, 10726, 0, 8836, 1294, 0, 0, 0, 0, 3223, 0, 0, 
 2619, 0, 0, 0, 0, 0, 0, 10328, 0, 16532, 0, 0, 0, 2104, 0, 
 4117, 0, 6193, 16507, 0, 0, 0, 0, 0, 0, 1306, 4632, 6720, 
 0, 0, 0, 0, 0, 0, 3534, 0, 0, 6831, 0, 0, 0, 0, 0, 0, 16363, 
 0, 0, 0, 0, 11894, 0, 16253, 0, 7203, 0, 0, 777, 4049, 0, 
 0, 0, 0, 0, 8599, 0, 8588, 0, 0, 18560, 0, 0, 0, 0, 8590, 
 0, 0, 0, 0, 0, 13580, 0, 11458, 0, 10873, 5454, 0, 0, 0, 
 0, 0, 0, 0, 0, 11314, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
 0, 18649, 0, 0, 0, 0, 0, 6946, 0, 0, 0, 11988, 0, 18260, 
 0, 0, 0, 8917, 0, 10821, 0, 0, 0, 0, 0, 8174, 12682, 13253, 
 201, 0, 0, 0, 0, 0, 0, 18267, 13005, 0, 0, 8121, 9311, 0, 
 5362, 8118, 0, 0, 0, 8119, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
 0, 0, 8123, 0, 12260, 2050, 0, 11882, 0, 0, 0, 17886, 3476, 
 0, 0, 7135, 0, 0, 9416, 0, 0, 4644, 16253, 0, 1902, 0, 0, 
 0, 15756, 0, 6484, 0, 0, 108, 844, 6921, 17728, 6424, 0, 0, 
 0, 7717, 0, 0, 0, 381, 0, 0, 0, 6452, 2227, 0, 0, 0, 0, 
 7028, 0, 0, 0, 6997, 4164, 0, 0, 0, 0, 9262, 0, 4528, 6195, 
 0, 4063, 0, 932, 2889, 0, 0, 0, 8041, 0, 0, 15598, 0, 0, 
 0, 0, 0, 0, 0, 0, 11375, 0, 0, 13072, 0, 16495, 0, 0, 0, 
 17233, 18773, 7319, 17372, 0, 0, 0, 0, 0, 0, 11654, 7165, 0, 
 0, 0, 0, 15293, 17430, 5464, 0, 0, 18062, 0, 0, 0, 10590, 
 0, 0, 0, 0, 0, 8406, 0, 11680, 0, 5350, 0, 0, 0, 0, 0, 1977, 
 0, 0, 0, 0, 12278, 18014, 18086, 8793, 0, 6302, 12721, 2469, 
 0, 18320, 18508, 0, 2873, 0, 3936, 1333, 7212, 0, 0, 14781, 
 0, 0, 0, 0, 0, 0, 5281, 0, 9605, 0, 43, 0, 10238, 0, 0, 
 15219, 11138, 0, 0, 0, 1697, 3380, 0, 2257, 0, 18481, 0, 0, 
 0, 0, 0, 0, 2229, 0, 0, 16743, 11151, 0, 0, 17764, 0, 0, 
 0, 1162, 0, 0, 0, 0, 0, 15353, 5671, 12216, 0, 0, 7584, 0, 
 0, 653, 0, 8468, 0, 0, 3293, 10551, 18807, 0, 0, 0, 0, 0, 
 9624, 0, 0, 0, 0, 5696, 0, 0, 0, 0, 0, 0, 4684, 0, 0, 14619, 
 0, 3397, 2814, 0, 9194, 0, 0, 0, 4513, 0, 13821, 0, 0, 14675, 
 0, 3572, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2103, 0, 0, 0, 0, 0, 
 0, 0, 0, 0, 0, 0, 0, 0, 2301, 0, 0, 0, 0, 0, 0, 0, 0, 
 0, 0, 0, 0, 13235, 0, 6802, 17294, 0, 0, 5403, 0, 0, 0, 0, 
 0, 9985, 10010, 0, 0, 0, 0, 0, 4204, 0, 0, 0, 0, 0, 4527, 
 0, 7721, 2112, 7131, 0, 0, 10425, 13913, 0, 0, 0, 4594, 6445, 
 0, 11256, 0, 0, 0, 11062, 0, 16060, 14342, 0, 17353, 0, 14359, 
 0, 0, 0, 0, 0, 12763, 0, 0, 13908, 1111, 0, 0, 772, 0, 1461, 
 0, 0, 0, 0, 13384, 0, 9391, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
 0, 0, 6318, 1907, 0, 0, 0, 0, 0, 0, 0, 9845, 17007, 0, 0, 
 0, 0, 0, 0, 60, 0, 11072, 0, 0, 0, 378, 0, 8598, 0, 9581, 
 451, 0, 0, 16608, 0, 0, 0, 15462, 18347, 0, 0, 18040, 0, 0, 
 0, 0, 1976, 0, 0, 0, 0, 0, 0, 2566, 0, 0, 0, 0, 0, 11895, 
 0, 3535, 0, 0, 7840, 0, 0, 0, 0, 0, 0, 816, 2429, 0, 0, 
 0, 2441, 0, 0, 0, 2433, 0, 0, 0, 0, 0, 2443, 15565, 0, 0, 
 0, 13106, 10201, 7908, 5076, 0, 623, 0, 4118, 0, 0, 0, 0, 
 0, 0, 0, 7390, 0, 18120, 0, 0, 6697, 0, 0, 9979, 7667, 0, 
 17494, 0, 0, 0, 17370, 3835, 0, 0, 1241, 13040, 0, 0, 16051, 
 17879, 0, 0, 0, 0, 2152, 0, 1273, 7124, 11242, 4690, 17351, 
 0, 0, 8447, 15632, 2250, 1787, 0, 1281, 8442, 0, 0, 18032, 
 8443, 10621, 0, 12894, 0, 0, 12359, 0, 0, 2387, 0, 15846, 17372, 
 0, 10852, 12932, 0, 4105, 0, 8432, 0, 17891, 8334, 12053, 0, 
 3144, 15499, 0, 0, 0, 0, 1532, 0, 5470, 0, 7580, 6434, 0, 
 0, 1344, 3376, 0, 0, 2184, 12892, 0, 0, 0, 0, 3656, 16799, 
 499, 1645, 10069, 0, 5807, 0, 719, 0, 0, 0, 0, 0, 7240, 3794, 
 0, 0, 16343, 0, 0, 2304, 1944, 2144, 0, 6113, 13315, 2098, 
 9359, 12600, 7242, 0, 0, 7886, 1127, 0, 0, 11230, 13039, 0, 
 7244, 0, 3057, 0, 6949, 0, 5853, 0, 0, 0, 0, 0, 0, 0, 0, 
 0, 14986, 0, 1869, 0, 0, 0, 0, 0, 0, 18066, 0, 0, 0, 0, 
 0, 16842, 0, 0, 16997, 8458, 1750, 0, 0, 0, 0, 211, 0, 0, 
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 7417, 0, 0, 0, 0, 0, 0, 
 6291, 0, 10449, 0, 0, 835, 0, 0, 1225, 0, 8351, 0, 7212, 4835, 
 12503, 0, 0, 0, 4291, 0, 0, 12876, 0, 0, 0, 8086, 0, 0, 6953, 
 0, 0, 0, 0, 0, 6726, 0, 0, 0, 12951, 0, 0, 11324, 0, 12148, 
 0, 0, 0, 0, 0, 0, 0, 9612, 0, 17062, 0, 0, 15443, 6462, 0, 
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 13876, 0, 0, 0, 0, 0, 
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 16879, 0, 1949, 13842, 0, 
 0, 0, 0, 0, 0, 13967, 10546, 16609, 0, 9339, 9340, 11539, 9953, 
 5854, 9336, 8693, 2405, 0, 0, 0, 0, 9343, 9344, 6426, 0, 0, 
 0, 0, 2952, 4583, 0, 0, 0, 0, 0, 0, 0, 0, 0, 14748, 14545, 
 0, 0, 9803, 3720, 17206, 0, 0, 0, 0, 0, 0, 3029, 0, 0, 0, 
 0, 12759, 0, 0, 2663, 0, 0, 3617, 5349, 17556, 3622, 9434, 
 0, 0, 6249, 6408, 0, 0, 0, 16756, 0, 0, 11979, 1634, 14292, 
 5846, 0, 688, 0, 0, 0, 0, 450, 1177, 0, 0, 5614, 0, 10046, 
 0, 0, 0, 0, 0, 13501, 0, 2952, 0, 4825, 0, 0, 0, 0, 4791, 
 0, 10136, 1577, 0, 5100, 12453, 0, 151, 0, 0, 0, 4885, 9349, 
 9350, 0, 4883, 9345, 4094, 11067, 16437, 0, 0, 9401, 0, 5282, 
 1674, 0, 0, 0, 3251, 0, 0, 0, 10504, 777, 0, 0, 0, 0, 0, 
 5561, 16257, 0, 0, 0, 1990, 6401, 0, 2318, 2359, 1904, 5628, 
 0, 0, 389, 0, 0, 12362, 0, 0, 9992, 4144, 0, 4560, 0, 413, 
 0, 6405, 0, 9781, 0, 16889, 0, 4401, 0, 3643, 9420, 11305, 
 0, 0, 0, 0, 18529, 6319, 0, 0, 0, 2401, 0, 14441, 0, 0, 4581, 
 0, 0, 0, 17869, 6443, 0, 15350, 9213, 4457, 12653, 0, 496, 
 0, 3363, 12875, 12168, 0, 0, 0, 9014, 4546, 9106, 0, 0, 15047, 
 0, 0, 0, 6262, 0, 0, 14589, 0, 4576, 5352, 0, 8404, 0, 0, 
 0, 0, 0, 0, 6168, 4783, 0, 3264, 6106, 9469, 0, 0, 0, 0, 
 11017, 7225, 1192, 17507, 8441, 18305, 0, 8991, 10682, 1244, 
 0, 11489, 0, 0, 4028, 0, 0, 8344, 3201, 11961, 0, 0, 0, 0, 
 0, 0, 0, 11775, 0, 0, 0, 0, 0, 2991, 0, 0, 3212, 11780, 0, 
 0, 5921, 2984, 0, 0, 0, 10593, 0, 0, 0, 8141, 0, 0, 0, 0, 
 0, 6443, 0, 0, 0, 0, 0, 0, 9471, 0, 0, 0, 0, 0, 13470, 0, 
 18195, 0, 12739, 49, 0, 4523, 0, 12729, 5404, 0, 7128, 0, 0, 
 0, 2041, 2890, 15275, 13863, 0, 0, 0, 0, 7064, 6644, 7980, 
 12220, 0, 0, 0, 18129, 0, 18637, 0, 0, 9559, 0, 14026, 0, 
 0, 17886, 9608, 0, 0, 0, 0, 11667, 0, 0, 1438, 1316, 0, 0, 
 0, 0, 0, 0, 0, 7576, 0, 0, 0, 0, 0, 0, 0, 0, 0, 14930, 
 0, 9905, 1321, 0, 0, 11266, 0, 6686, 378, 5296, 0, 0, 3163, 
 17057, 0, 1655, 0, 1010, 0, 0, 0, 9652, 0, 0, 11662, 0, 12991, 
 0, 0, 11708, 0, 725, 8698, 0, 0, 9610, 0, 10282, 10289, 7833, 
 0, 0, 18076, 0, 0, 3921, 4770, 3472, 1683, 14225, 18715, 17607, 
 7333, 3923, 0, 0, 12921, 10777, 5030, 0, 2170, 0, 6253, 0, 
 0, 0, 0, 0, 7066, 0, 14297, 0, 12024, 0, 0, 0, 17278, 0, 
 0, 0, 372, 2573, 0, 0, 7410, 17801, 0, 0, 0, 0, 0, 0, 7750, 
 0, 3067, 0, 6384, 3534, 0, 18470, 17129, 0, 0, 6559, 11415, 
 15597, 11744, 0, 3109, 0, 16547, 6325, 17456, 16933, 0, 9139, 
 5124, 13848, 0, 0, 16141, 0, 3079, 0, 0, 0, 0, 0, 10056, 0, 
 5945, 0, 0, 2263, 0, 0, 5047, 0, 0, 0, 7419, 16237, 10189, 
 0, 0, 0, 0, 16924, 0, 10528, 15866, 0, 9022, 0, 0, 11385, 
 0, 0, 15499, 0, 15745, 13379, 0, 16367, 5370, 2838, 6074, 0, 
 0, 0, 0, 12845, 9694, 16351, 11022, 16294, 0, 14353, 15189, 
 17818, 0, 7046, 0, 1258, 0, 5062, 0, 10497, 1786, 0, 10124, 
 0, 0, 18212, 0, 0, 0, 0, 0, 0, 13449, 0, 0, 0, 0, 0, 0, 
 11820, 9949, 0, 0, 0, 0, 0, 0, 8632, 7575, 0, 0, 0, 3370, 
 0, 0, 0, 0, 0, 1259, 0, 15632, 0, 17263, 0, 8949, 0, 0, 4452, 
 0, 13770, 0, 0, 14760, 0, 0, 0, 0, 6385, 0, 4943, 16143, 3295, 
 4690, 0, 0, 0, 0, 0, 4101, 0, 0, 11292, 0, 0, 0, 0, 11519, 
 0, 0, 0, 0, 6396, 8285, 0, 0, 0, 0, 15557, 5366, 0, 0, 2852, 
 12752, 0, 0, 0, 0, 0, 0, 0, 1620, 0, 11532, 16380, 0, 14800, 
 7192, 0, 11043, 2454, 0, 0, 7609, 0, 2846, 0, 0, 0, 0, 0, 
 0, 0, 0, 0, 0, 0, 10230, 0, 2261, 0, 0, 0, 0, 0, 0, 0, 
 2694, 0, 0, 4938, 1707, 0, 18806, 0, 2830, 9293, 0, 2253, 16647, 
 18517, 17467, 0, 4160, 6353, 352, 0, 18174, 0, 7257, 2649, 647, 
 0, 0, 0, 0, 0, 10537, 0, 0, 0, 0, 18148, 0, 0, 0, 0, 0, 
 0, 0, 9132, 8656, 0, 4332, 0, 0, 1079, 0, 0, 0, 0, 7608, 
 0, 0, 0, 5347, 3194, 0, 0, 0, 0, 0, 95, 0, 0, 5678, 0, 0, 
 11875, 0, 12956, 0, 0, 0, 0, 1226, 0, 736, 0, 0, 0, 0, 2186, 
 1029, 0, 6620, 0, 16839, 0, 2579, 0, 0, 0, 17848, 0, 0, 16472, 
 18396, 0, 5079, 6202, 0, 0, 16593, 6750, 0, 0, 2129, 2086, 
 0, 0, 0, 0, 0, 17466, 0, 18670, 0, 0, 0, 0, 0, 0, 3573, 
 0, 0, 0, 0, 0, 0, 0, 0, 2775, 0, 9691, 0, 0, 0, 12695, 0, 
 6411, 921, 0, 6511, 17117, 0, 0, 0, 0, 10151, 0, 3683, 0, 
 0, 15913, 0, 0, 9659, 0, 0, 0, 0, 14816, 16088, 620, 0, 0, 
 0, 0, 0, 0, 9340, 0, 0, 0, 1263, 0, 0, 8685, 0, 15154, 5536, 
 0, 0, 17445, 0, 1148, 0, 0, 9758, 11222, 0, 0, 0, 0, 0, 14407, 
 0, 0, 0, 0, 4075, 0, 0, 0, 8897, 4538, 11702, 1226, 18622, 
 2500, 0, 0, 0, 2499, 13886, 0, 0, 0, 0, 0, 13815, 13862, 0, 
 0, 12183, 0, 0, 0, 0, 0, 3663, 0, 0, 0, 4677, 0, 1291, 0, 
 0, 2472, 0, 3051, 5862, 7695, 0, 0, 4170, 0, 16578, 0, 17616, 
 1705, 0, 0, 8656, 1611, 0, 0, 0, 0, 3020, 4828, 16444, 7538, 
 0, 4399, 18246, 0, 0, 0, 0, 205, 13049, 0, 0, 2167, 0, 11741, 
 0, 0, 0, 8334, 3153, 0, 0, 6130, 10731, 15006, 13378, 16295, 
 0, 12350, 4871, 0, 461, 0, 16175, 597, 16278, 6236, 0, 0, 3054, 
 0, 2851, 0, 9420, 10180, 302, 9699, 0, 3101, 1622, 0, 1062, 
 0, 0, 0, 0, 0, 0, 0, 0, 0, 17197, 0, 0, 11311, 9700, 0, 
 5857, 0, 0, 2701, 0, 0, 10490, 0, 0, 10007, 10080, 0, 0, 0, 
 0, 9712, 13474, 0, 10093, 7178, 0, 0, 0, 0, 0, 0, 4218, 17671, 
 0, 0, 0, 7677, 0, 0, 0, 17385, 0, 10641, 0, 0, 262, 0, 0, 
 0, 18834, 16729, 0, 5869, 0, 0, 0, 0, 0, 0, 0, 8902, 0, 15571, 
 15543, 0, 0, 18092, 1305, 4025, 3162, 17699, 0, 0, 0, 0, 0, 
 5798, 0, 0, 12874, 13159, 0, 0, 0, 11548, 0, 0, 0, 6058, 0, 
 0, 0, 0, 0, 0, 14747, 0, 0, 0, 0, 0, 17557, 1512, 0, 12733, 
 0, 0, 0, 16872, 4602, 18244, 17596, 1532, 7268, 0, 0, 2614, 
 7055, 0, 0, 0, 0, 0, 7027, 0, 0, 0, 0, 0, 17635, 0, 7239, 
 14742, 5205, 748, 0, 0, 0, 18797, 0, 0, 0, 5074, 0, 0, 0, 
 0, 0, 0, 13715, 0, 18540, 10712, 0, 0, 0, 0, 0, 0, 0, 0, 
 0, 18068, 0, 0, 0, 0, 0, 0, 0, 2178, 7006, 0, 6430, 0, 0, 
 3762, 18021, 0, 5716, 0, 16535, 16226, 0, 3277, 0, 0, 5256, 
 10896, 0, 17774, 0, 0, 0, 6315, 0, 0, 4673, 0, 0, 15217, 18645, 
 0, 0, 0, 4064, 13527, 0, 0, 16956, 0, 4628, 0, 0, 0, 12879, 
 0, 6010, 0, 7124, 0, 14804, 0, 0, 0, 0, 0, 14123, 0, 0, 16706, 
 0, 5227, 0, 0, 16006, 0, 5219, 16782, 0, 1453, 13717, 0, 12269, 
 0, 17363, 1344, 10396, 3069, 0, 14914, 0, 5519, 0, 2130, 15400, 
 0, 0, 16329, 8461, 17950, 8801, 0, 0, 0, 10634, 15059, 0, 0, 
 0, 0, 4241, 0, 5467, 0, 0, 0, 0, 13845, 11492, 4914, 15843, 
 0, 0, 2853, 15699, 0, 0, 0, 0, 0, 0, 4771, 0, 14396, 6561, 
 0, 0, 0, 0, 0, 0, 0, 8553, 0, 0, 0, 0, 0, 0, 582, 17811, 
 0, 10303, 0, 0, 9804, 986, 13785, 8859, 0, 8857, 9351, 0, 8734, 
 2926, 1371, 13714, 0, 16722, 0, 0, 0, 0, 0, 0, 17950, 6216, 
 0, 4073, 0, 0, 0, 0, 0, 0, 0, 3084, 5567, 16082, 0, 0, 6207, 
 0, 0, 3160, 0, 0, 0, 0, 0, 10381, 0, 0, 0, 0, 0, 13174, 
 8994, 7419, 0, 0, 13972, 18502, 13530, 0, 0, 0, 0, 0, 0, 18302, 
 3113, 0, 0, 1730, 0, 578, 0, 11071, 0, 10404, 18786, 0, 9062, 
 12129, 0, 0, 9569, 18726, 0, 0, 2610, 0, 5450, 0, 0, 13137, 
 0, 5558, 0, 0, 0, 0, 2907, 0, 0, 0, 1460, 16623, 0, 2775, 
 0, 9115, 7738, 7735, 0, 0, 0, 0, 0, 0, 0, 928, 0, 0, 13386, 
 0, 13164, 0, 0, 0, 14503, 0, 0, 0, 0, 0, 4253, 6025, 0, 0, 
 0, 0, 17025, 15422, 7083, 0, 0, 0, 7110, 0, 0, 0, 14021, 0, 
 2249, 0, 18136, 0, 202, 10671, 0, 1395, 2784, 0, 1877, 0, 10617, 
 4951, 3906, 17273, 0, 17759, 11539, 18372, 11879, 2496, 0, 5591, 
 8753, 3367, 8751, 0, 0, 4340, 8748, 0, 0, 0, 8749, 11019, 0, 
 0, 0, 0, 0, 0, 0, 9085, 12947, 342, 0, 0, 3002, 0, 0, 0, 
 0, 0, 3165, 0, 0, 9269, 0, 0, 0, 17119, 0, 0, 0, 13173, 4407, 
 4537, 0, 18543, 15076, 0, 958, 17501, 0, 9122, 0, 10652, 8318, 
 0, 8798, 3451, 0, 6506, 0, 17649, 0, 3876, 0, 0, 0, 0, 0, 
 0, 16293, 5323, 5039, 9347, 0, 0, 0, 0, 0, 0, 5477, 0, 5924, 
 7631, 0, 3365, 0, 4343, 17018, 0, 4466, 0, 16876, 0, 0, 410, 
 13960, 0, 0, 0, 0, 0, 0, 0, 12440, 0, 0, 0, 3019, 6856, 18151, 
 0, 0, 10885, 7937, 8786, 0, 1996, 16101, 0, 0, 0, 15484, 0, 
 0, 0, 0, 1632, 18212, 0, 0, 3945, 0, 0, 0, 4906, 0, 0, 5095, 
 9679, 6222, 0, 0, 0, 7124, 1407, 0, 388, 2610, 0, 0, 0, 588, 
 9151, 0, 0, 0, 0, 0, 0, 11972, 2311, 0, 0, 0, 7589, 0, 5850, 
 13706, 0, 7010, 3506, 0, 781, 0, 0, 14042, 0, 0, 0, 0, 0, 
 0, 0, 0, 1597, 0, 3781, 0, 0, 0, 11608, 16707, 0, 0, 1704, 
 16612, 0, 0, 0, 430, 18228, 0, 2748, 2364, 0, 0, 0, 0, 0, 
 0, 0, 0, 0, 0, 4030, 0, 0, 0, 6181, 0, 0, 0, 7818, 2481, 
 15713, 10428, 490, 2033, 266, 0, 13669, 11221, 0, 16014, 0, 
 11693, 14715, 0, 0, 7457, 9772, 0, 16022, 11821, 17187, 3188, 
 0, 0, 1432, 6846, 9563, 46, 15662, 13437, 0, 11839, 0, 0, 5036, 
 2441, 0, 13385, 0, 13371, 18564, 0, 12407, 13396, 0, 4094, 0, 
 0, 0, 259, 0, 8940, 10910, 14261, 0, 11202, 0, 3276, 0, 14885, 
 17669, 12805, 0, 9238, 1389, 7974, 0, 1946, 0, 3028, 13761, 
 14958, 0, 0, 6676, 11714, 4342, 0, 0, 6787, 0, 15695, 8876, 
 3947, 2567, 2008, 17553, 4936, 7614, 12409, 17437, 10281, 10435, 
 0, 3453, 0, 982, 2163, 7905, 0, 0, 1947, 14513, 0, 0, 1542, 
 16556, 10012, 0, 0, 0, 15677, 0, 0, 0, 4036, 0, 5813, 0, 6463, 
 0, 0, 0, 0, 0, 9512, 14194, 18828, 0, 0, 0, 8315, 8603, 0, 
 11277, 16506, 4391, 0, 0, 2658, 0, 0, 10330, 12098, 0, 3578, 
 17153, 13952, 0, 11160, 1390, 0, 0, 6846, 0, 0, 0, 0, 0, 4770, 
 9602, 9289, 6785, 7952, 4671, 0, 16670, 0, 0, 16050, 0, 0, 
 0, 0, 2356, 8167, 0, 10609, 0, 7675, 0, 0, 0, 0, 0, 0, 0, 
 18073, 9823, 18165, 0, 7858, 1628, 12431, 0, 5165, 6870, 0, 
 0, 16425, 6767, 0, 6255, 5696, 0, 0, 4958, 0, 0, 6724, 6590, 
 0, 1307, 0, 0, 0, 18771, 1804, 15490, 0, 0, 0, 13173, 0, 13847, 
 0, 0, 0, 4244, 0, 13395, 0, 11601, 0, 0, 11122, 18635, 0, 
 18447, 0, 12813, 0, 4264, 6455, 0, 8741, 0, 12648, 0, 13076, 
 0, 0, 9642, 0, 2040, 16773, 15176, 0, 2588, 0, 0, 0, 0, 0, 
 0, 1080, 2729, 0, 0, 7421, 0, 0, 18068, 0, 7382, 11692, 0, 
 16529, 0, 0, 0, 17205, 9578, 0, 13833, 7462, 0, 0, 10253, 0, 
 0, 0, 18785, 0, 4463, 0, 0, 13955, 0, 5710, 1983, 0, 186, 
 5210, 0, 0, 0, 14919, 1518, 15990, 0, 5983, 14724, 0, 0, 0, 
 17814, 0, 0, 0, 0, 0, 6862, 13803, 14553, 0, 0, 0, 300, 6338, 
 3995, 0, 8376, 0, 3730, 0, 5888, 0, 8371, 8595, 8525, 0, 8372, 
 0, 0, 14834, 8520, 0, 0, 15218, 0, 2897, 5402, 0, 0, 15147, 
 7509, 0, 11808, 8363, 521, 8349, 5367, 0, 11669, 0, 2602, 0, 
 14844, 0, 12942, 0, 0, 0, 0, 0, 14905, 0, 0, 17223, 0, 0, 
 15329, 1444, 0, 0, 4629, 0, 0, 0, 0, 4976, 5864, 7345, 0, 
 0, 0, 8198, 0, 10505, 10015, 0, 8999, 0, 0, 7941, 3461, 10097, 
 0, 0, 13525, 0, 0, 0, 0, 0, 0, 5846, 0, 0, 10449, 0, 9671, 
 13729, 16937, 0, 0, 585, 0, 10935, 0, 9530, 16545, 6383, 1693, 
 0, 0, 0, 0, 14936, 0, 0, 0, 0, 883, 0, 0, 1120, 5173, 0, 
 8529, 9929, 0, 500, 15759, 6241, 9892, 0, 3742, 0, 17265, 0, 
 16295, 2925, 2881, 7865, 0, 6837, 0, 13752, 8444, 0, 7542, 8094, 
 0, 1493, 0, 769, 10880, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
 0, 3960, 3416, 18179, 0, 0, 0, 0, 0, 5094, 0, 0, 0, 15181, 
 17908, 7083, 0, 16247, 0, 11150, 0, 18792, 0, 0, 0, 3620, 15284, 
 16721, 10098, 0, 0, 9152, 0, 0, 0, 0, 0, 12558, 0, 17634, 
 0, 0, 2712, 0, 0, 1955, 0, 0, 0, 0, 13704, 0, 0, 0, 9836, 
 0, 0, 13010, 0, 0, 4761, 0, 0, 6152, 0, 0, 4172, 8737, 8958, 
 13101, 13710, 12795, 14398, 0, 0, 0, 9556, 0, 0, 0, 0, 0, 
 1192, 9199, 0, 0, 0, 3406, 6452, 2277, 15040, 17914, 0, 0, 
 7629, 0, 0, 12282, 1690, 13060, 2742, 0, 0, 14316, 9060, 4390, 
 0, 7653, 0, 0, 0, 16064, 10120, 16191, 0, 0, 0, 1855, 3466, 
 7052, 0, 4987, 8702, 9841, 3037, 0, 3576, 18470, 0, 10986, 0, 
 0, 0, 18064, 0, 5267, 11284, 12900, 0, 0, 0, 17130, 0, 2403, 
 8892, 0, 3899, 8345, 12380, 0, 0, 0, 926, 0, 18394, 0, 0, 
 2240, 0, 0, 13854, 0, 1666, 0, 4444, 9260, 754, 14101, 0, 14065, 
 12330, 0, 0, 13039, 9804, 409, 2555, 0, 0, 0, 0, 0, 2590, 
 15217, 0, 17997, 2643, 0, 0, 6810, 0, 0, 0, 0, 3733, 462, 
 0, 0, 0, 12528, 15551, 14695, 0, 2854, 9893, 0, 10189, 0, 0, 
 0, 0, 0, 17507, 0, 13638, 6451, 5146, 0, 14243, 2326, 1232, 
 0, 8631, 0, 0, 0, 4749, 0, 0, 12927, 0, 0, 0, 0, 4285, 3319, 
 2231, 0, 0, 0, 0, 9902, 16330, 0, 4790, 0, 0, 2626, 426, 0, 
 0, 515, 0, 12349, 0, 0, 9988, 0, 0, 0, 5565, 0, 0, 16071, 
 1135, 6392, 11401, 0, 13132, 2198, 10188, 500, 0, 8422, 1868, 
 12010, 0, 1791, 878, 18493, 2905, 502, 11628, 0, 0, 4710, 0, 
 0, 0, 5761, 5482, 0, 0, 0, 12765, 4342, 15055, 0, 6355, 0, 
 9867, 0, 11601, 0, 0, 1281, 0, 0, 0, 0, 0, 0, 0, 10791, 6532, 
 0, 0, 0, 6308, 3904, 0, 1082, 15350, 0, 0, 3190, 0, 8038, 
 148, 0, 0, 17836, 0, 6800, 0, 8820, 0, 0, 18557, 791, 5003, 
 0, 0, 1159, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 4902, 0, 3974, 
 3067, 8583, 13036, 4761, 0, 0, 17830, 8240, 0, 0, 0, 0, 0, 
 0, 0, 6336, 0, 626, 0, 0, 0, 0, 0, 13618, 12974, 0, 0, 0, 
 0, 0, 8292, 13037, 9039, 0, 0, 0, 2418, 15953, 0, 0, 8735, 
 0, 0, 0, 3017, 0, 1295, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 17979, 
 3866, 11271, 15803, 0, 5746, 10975, 0, 0, 6706, 0, 10206, 0, 
 15615, 0, 0, 0, 0, 0, 7586, 0, 0, 0, 0, 4388, 0, 0, 4594, 
 0, 0, 3598, 0, 0, 3987, 11023, 3303, 7705, 5680, 17456, 2310, 
 9775, 3096, 6202, 9481, 8471, 0, 0, 12028, 8470, 0, 8475, 0, 
 16429, 0, 0, 12115, 0, 17853, 18365, 16066, 0, 0, 14538, 0, 
 0, 0, 17816, 14018, 0, 11937, 18366, 13271, 4856, 8747, 0, 0, 
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 14373, 9819, 0, 0, 
 0, 0, 0, 0, 0, 0, 0, 0, 0, 2264, 276, 9780, 0, 8017, 0, 
 0, 0, 3721, 0, 9976, 16722, 0, 0, 0, 0, 13949, 3815, 0, 0, 
 0, 6834, 0, 0, 17801, 0, 817, 0, 0, 0, 7443, 18181, 0, 0, 
 18602, 0, 0, 0, 0, 15725, 0, 4403, 0, 0, 12289, 0, 0, 3958, 
 1180, 0, 12065, 3071, 0, 0, 9605, 0, 14084, 1198, 0, 6082, 
 4678, 9838, 670, 0, 8576, 0, 807, 9603, 12056, 6102, 14243, 
 0, 8574, 0, 0, 0, 0, 0, 5634, 0, 17556, 828, 8406, 0, 0, 
 531, 1996, 0, 0, 0, 0, 0, 2472, 0, 0, 0, 0, 1103, 1504, 7975, 
 0, 11270, 13503, 0, 1375, 0, 0, 4824, 11409, 4974, 0, 7974, 
 0, 0, 1926, 0, 7773, 0, 0, 11650, 9117, 18582, 0, 0, 0, 4245, 
 18289, 0, 0, 0, 3635, 0, 0, 0, 0, 0, 2861, 0, 165, 0, 0, 
 13865, 18491, 0, 0, 0, 16115, 3595, 0, 4720, 0, 0, 5446, 0, 
 0, 8739, 15673, 0, 9612, 474, 0, 8594, 0, 15104, 0, 16213, 
 0, 12003, 0, 7641, 1128, 16831, 1025, 0, 0, 0, 14838, 0, 0, 
 0, 11991, 2027, 0, 0, 11879, 0, 0, 17151, 0, 17527, 13168, 
 7006, 12918, 6971, 15435, 0, 5222, 17120, 0, 0, 0, 1391, 6080, 
 2559, 0, 6352, 373, 0, 0, 0, 4375, 12582, 0, 512, 0, 0, 5503, 
 12157, 0, 14865, 16650, 0, 0, 12367, 0, 0, 7275, 0, 18588, 
 0, 5068, 7522, 0, 7914, 17939, 4776, 8741, 0, 0, 0, 0, 3631, 
 6381, 0, 11346, 0, 0, 0, 16806, 0, 4051, 0, 3926, 0, 0, 0, 
 0, 7472, 0, 6884, 1560, 0, 504, 0, 0, 0, 0, 0, 13333, 0, 
 0, 0, 0, 0, 0, 14126, 0, 3168, 12200, 3832, 9936, 0, 8897, 
 0, 16149, 0, 0, 2682, 2784, 17909, 0, 0, 0, 0, 5586, 9157, 
 0, 5608, 0, 362, 0, 0, 0, 0, 14409, 15279, 13622, 0, 4210, 
 0, 0, 3176, 9091, 3552, 10199, 0, 9063, 0, 0, 0, 0, 5875, 
 0, 0, 5124, 959, 1067, 0, 0, 17210, 7041, 266, 0, 5802, 7027, 
 12368, 12448, 0, 2719, 0, 0, 0, 17544, 0, 0, 3479, 0, 0, 3480, 
 0, 0, 0, 1273, 13168, 7114, 9989, 12190, 7707, 1744, 15577, 
 0, 12188, 0, 2480, 5763, 3474, 339, 0, 0, 5178, 0, 3478, 0, 
 3477, 3917, 0, 6193, 0, 12647, 14655, 0, 14672, 0, 0, 0, 1912, 
 9064, 4250, 0, 0, 4151, 0, 0, 1733, 10657, 5898, 18296, 0, 
 0, 12594, 8270, 3009, 1661, 13555, 0, 8567, 0, 0, 0, 16697, 
 0, 0, 3781, 11445, 1985, 0, 1121, 11531, 2208, 9923, 0, 0, 
 0, 2436, 0, 0, 17372, 0, 0, 0, 9519, 0, 12214, 0, 3068, 16947, 
 0, 11471, 13392, 15836, 2886, 84, 5787, 11502, 5461, 13705, 0, 
 0, 0, 0, 0, 0, 11807, 0, 1536, 0, 0, 0, 0, 11115, 5753, 10534, 
 0, 1770, 4306, 234, 0, 0, 0, 16029, 10478, 2448, 18526, 0, 
 16452, 0, 121, 0, 0, 0, 749, 0, 0, 0, 2645, 3787, 0, 0, 0, 
 0, 0, 0, 9823, 0, 0, 1939, 3604, 0, 6136, 1207, 16203, 18021, 
 11940, 18455, 0, 0, 0, 8371, 0, 13831, 0, 12932, 6253, 5426, 
 0, 4240, 8839, 0, 0, 10582, 5879, 3132, 0, 17580, 0, 18360, 
 7092, 8052, 6008, 7497, 0, 8450, 10279, 0, 6299, 5061, 0, 18092, 
 3051, 8860, 6994, 0, 0, 0, 0, 0, 0, 0, 0, 0, 15212, 0, 0, 
 11415, 18223, 0, 0, 0, 0, 7392, 1882, 3641, 8172, 15480, 974, 
 9613, 6857, 3604, 0, 10662, 12711, 7237, 0, 10863, 2077, 533, 
 17182, 0, 8196, 0, 0, 11891, 18017, 6980, 0, 0, 14546, 0, 3083, 
 0, 0, 4490, 0, 0, 4253, 4939, 0, 0, 0, 0, 3894, 0, 4132, 
 0, 0, 0, 0, 17320, 0, 0, 0, 0, 6754, 0, 13479, 8968, 15269, 
 18524, 760, 11251, 5072, 15143, 6460, 15375, 5458, 6120, 17438, 
 15266, 0, 13087, 3698, 16782, 0, 0, 0, 7394, 0, 0, 0, 0, 15257, 
 0, 18710, 1956, 5242, 7743, 5035, 15405, 0, 0, 14874, 7700, 
 8811, 0, 0, 3611, 7838, 9488, 10138, 3621, 0, 3617, 0, 2289, 
 6068, 0, 0, 3613, 0, 0, 2450, 6093, 0, 10079, 0, 11249, 1012, 
 8177, 10181, 0, 1485, 8137, 3000, 15358, 5533, 4516, 12619, 0, 
 14174, 4419, 0, 0, 0, 0, 18816, 0, 8342, 18442, 12703, 4357, 
 12203, 10678, 12155, 0, 7461, 5977, 1920, 8870, 14419, 0, 16176, 
 0, 8028, 0, 5357, 5337, 0, 0, 12368, 0, 0, 0, 14754, 14600, 
 11469, 5293, 0, 4394, 0, 0, 0, 4874, 0, 7496, 15441, 0, 0, 
 517, 6712, 2344, 1353, 0, 7388, 0, 16165, 0, 6312, 4962, 1601, 
 8880, 0, 562, 0, 7338, 14007, 8627, 17826, 0, 0, 3947, 14769, 
 0, 0, 8110, 1736, 16317, 0, 0, 0, 13400, 0, 0, 0, 9584, 9244, 
 7270, 15576, 1979, 0, 0, 0, 11756, 0, 7851, 5182, 0, 6087, 
 0, 9507, 0, 0, 338, 13159, 2123, 0, 835, 0, 0, 3230, 0, 10930, 
 0, 0, 1573, 0, 0, 0, 2118, 0, 13649, 5580, 0, 13123, 13374, 
 0, 0, 3282, 0, 13709, 3581, 5602, 0, 0, 9355, 11455, 18228, 
 0, 13988, 10095, 0, 0, 4641, 0, 595, 0, 11503, 0, 12475, 0, 
 11931, 0, 8484, 0, 0, 4963, 3713, 17542, 1450, 267, 6875, 0, 
 0, 0, 17823, 13502, 0, 0, 0, 5776, 0, 18022, 0, 0, 11728, 
 0, 0, 0, 1796, 13258, 9092, 0, 14734, 10013, 0, 4248, 6952, 
 0, 14755, 992, 0, 2305, 2498, 0, 18371, 0, 0, 0, 14394, 0, 
 0, 0, 5410, 0, 11926, 0, 16139, 964, 2982, 17280, 16813, 0, 
 15805, 0, 8169, 0, 5922, 15034, 1432, 13050, 0, 0, 2222, 0, 
 13309, 15123, 10063, 0, 7771, 0, 14241, 0, 13387, 0, 0, 5793, 
 0, 4826, 0, 0, 0, 0, 0, 0, 0, 18237, 0, 0, 0, 5656, 0, 4231, 
 8223, 718, 0, 0, 8454, 0, 0, 2775, 0, 15756, 0, 0, 0, 0, 
 0, 1966, 0, 14523, 0, 8380, 6143, 0, 0, 0, 6683, 13731, 0, 
 10097, 0, 8628, 0, 0, 0, 0, 0, 0, 9762, 7439, 0, 1227, 7360, 
 5544, 4486, 0, 3677, 0, 13224, 0, 0, 13798, 17794, 0, 8353, 
 0, 12287, 16253, 0, 0, 4482, 0, 6734, 73, 0, 1111, 0, 6271, 
 11872, 0, 10032, 9916, 2194, 0, 1066, 14067, 2406, 0, 5462, 
 17380, 3738, 9727, 5457, 12719, 17721, 0, 17339, 18057, 0, 14495, 
 11761, 0, 0, 0, 0, 0, 0, 3338, 10825, 14806, 0, 953, 6078, 
 0, 5880, 0, 0, 2255, 3081, 18136, 0, 0, 4951, 0, 17255, 0, 
 0, 10416, 1621, 0, 12, 0, 0, 0, 0, 0, 0, 0, 0, 12443, 0, 
 0, 0, 0, 10321, 0, 6825, 10261, 0, 13785, 0, 0, 17534, 4375, 
 12580, 0, 13516, 7409, 0, 1959, 0, 0, 11970, 10132, 0, 0, 0, 
 0, 17380, 7402, 0, 12035, 11745, 2600, 15699, 18483, 0, 13456, 
 4498, 9044, 0, 0, 3623, 16057, 0, 0, 0, 11245, 10246, 12813, 
 0, 0, 0, 0, 1605, 5051, 0, 0, 0, 9059, 1222, 1346, 0, 0, 
 0, 0, 0, 9487, 2450, 8956, 0, 13659, 0, 7479, 0, 0, 0, 16859, 
 0, 5560, 2266, 2342, 0, 4502, 0, 0, 15466, 0, 9725, 0, 0, 
 0, 0, 15821, 17276, 7046, 18768, 0, 18174, 0, 4748, 6954, 0, 
 7768, 0, 13650, 0, 7703, 0, 0, 0, 3512, 8667, 0, 0, 0, 17071, 
 4936, 0, 1442, 9508, 0, 2728, 7395, 0, 5816, 0, 0, 4369, 0, 
 14738, 2960, 0, 16188, 8239, 0, 8641, 17423, 15568, 2768, 0, 
 3527, 1371, 14878, 1867, 8249, 0, 12427, 0, 6848, 0, 0, 4563, 
 8251, 4267, 8253, 126, 6967, 9428, 0, 0, 7515, 0, 0, 0, 11699, 
 1546, 15346, 0, 4741, 6992, 0, 0, 12531, 0, 15700, 4116, 8430, 
 0, 0, 0, 6371, 7563, 17018, 17026, 0, 15042, 1853, 7984, 1490, 
 6248, 0, 500, 2327, 0, 6365, 0, 5680, 7642, 0, 2005, 13150, 
 0, 0, 18482, 9892, 0, 0, 0, 0, 1878, 7210, 0, 13290, 15593, 
 7792, 0, 0, 0, 4211, 0, 463, 0, 0, 0, 0, 7770, 16029, 3495, 
 0, 0, 0, 17625, 16033, 8255, 4131, 15929, 5876, 14013, 16733, 
 0, 18605, 6414, 15692, 0, 15702, 3998, 0, 0, 1207, 17869, 17497, 
 0, 243, 16153, 16061, 0, 11752, 2074, 18370, 0, 0, 0, 0, 0, 
 0, 0, 15552, 4212, 13798, 0, 2981, 0, 0, 0, 13882, 12004, 18615, 
 0, 1631, 7933, 0, 0, 14138, 0, 8950, 0, 18087, 211, 6860, 0, 
 0, 0, 0, 0, 366, 2256, 0, 4303, 7573, 0, 0, 17711, 0, 1709, 
 18211, 0, 17589, 0, 11848, 0, 9575, 0, 5590, 12737, 0, 0, 0, 
 0, 0, 2945, 0, 11256, 1104, 0, 0, 0, 0, 10095, 18634, 11403, 
 468, 17486, 8596, 17184, 6882, 18524, 7218, 2327, 0, 17156, 16243, 
 14482, 8643, 0, 0, 0, 5286, 16596, 0, 9803, 6843, 1519, 0, 
 18788, 0, 15317, 17671, 2427, 0, 16595, 6578, 1097, 3961, 3277, 
 0, 0, 0, 0, 3412, 3305, 18224, 0, 0, 0, 17171, 0, 0, 0, 0, 
 0, 0, 10246, 17055, 11375, 0, 0, 17704, 4678, 2544, 70, 3514, 
 15719, 0, 68, 3515, 0, 3516, 0, 0, 0, 7617, 5671, 8549, 42, 
 7803, 0, 3517, 8847, 0, 3122, 0, 0, 11171, 0, 0, 0, 0, 0, 
 0, 0, 0, 17399, 0, 3217, 0, 0, 0, 3971, 11036, 0, 11346, 0, 
 7334, 0, 0, 0, 5649, 2091, 737, 618, 0, 3715, 0, 8931, 4732, 
 156, 0, 17577, 7873, 2103, 12228, 0, 0, 4598, 0, 14614, 0, 
 2461, 0, 5375, 0, 14444, 0, 4583, 0, 0, 9329, 0, 2395, 386, 
 14515, 0, 17859, 0, 874, 9338, 0, 13416, 16035, 0, 0, 17625, 
 0, 0, 12935, 17368, 0, 0, 0, 0, 15712, 0, 0, 8369, 2116, 0, 
 13492, 8811, 1650, 4837, 1421, 15256, 0, 0, 0, 11235, 0, 0, 
 0, 16474, 5521, 9032, 0, 17704, 0, 0, 12281, 5242, 0, 0, 3428, 
 0, 0, 0, 0, 648, 0, 14880, 9313, 5307, 1941, 5627, 0, 0, 5210, 
 2214, 467, 13167, 4256, 9919, 0, 0, 2363, 0, 0, 0, 0, 4591, 
 5534, 6755, 8272, 0, 0, 0, 3971, 4770, 0, 0, 3526, 3253, 0, 
 0, 18570, 0, 0, 9526, 0, 0, 0, 16348, 0, 1547, 13801, 0, 8763, 
 0, 0, 10188, 0, 248, 6820, 1181, 16234, 5715, 0, 0, 16565, 
 13795, 16384, 1775, 4314, 14736, 13321, 9336, 17296, 0, 0, 13839, 
 0, 4224, 0, 0, 0, 0, 17350, 8887, 3639, 8607, 0, 0, 0, 0, 
 0, 0, 17368, 6352, 13589, 0, 1170, 0, 0, 7713, 0, 2019, 0, 
 0, 0, 0, 107, 0, 0, 0, 0, 9811, 0, 0, 0, 0, 7409, 0, 0, 
 0, 0, 2412, 950, 0, 8681, 4838, 0, 0, 10467, 9965, 4111, 0, 
 2146, 4776, 6679, 11517, 12788, 0, 0, 0, 12149, 3475, 0, 0, 
 0, 10194, 14353, 0, 6424, 0, 5471, 12941, 4864, 11572, 139, 
 0, 15767, 13607, 0, 0, 0, 7704, 4929, 6040, 6096, 7333, 15247, 
 0, 10523, 4803, 0, 0, 929, 0, 10894, 0, 4200, 0, 0, 0, 2791, 
 1848, 0, 6328, 14169, 3688, 249, 0, 0, 0, 0, 0, 0, 0, 3593, 
 771, 0, 4203, 14152, 0, 0, 15025, 12950, 5214, 6220, 0, 16386, 
 17394, 0, 3299, 7509, 175, 7506, 0, 17902, 17396, 0, 0, 7856, 
 0, 0, 9936, 3488, 0, 5836, 11156, 0, 0, 0, 16991, 7534, 804, 
 0, 0, 2041, 14587, 7531, 0, 0, 0, 0, 0, 0, 0, 3073, 913, 
 0, 3447, 0, 0, 10013, 3499, 0, 12812, 6191, 0, 9643, 0, 5986, 
 0, 584, 0, 0, 0, 3923, 0, 8419, 17569, 3629, 0, 0, 1306, 8185, 
 18675, 0, 0, 0, 3252, 0, 0, 3882, 9419, 5456, 7025, 0, 8391, 
 0, 0, 0, 0, 7966, 0, 1126, 0, 2685, 1361, 16154, 17954, 0, 
 7492, 17157, 0, 0, 0, 0, 5400, 0, 10257, 3857, 0, 0, 0, 0, 
 10751, 17757, 0, 8324, 16457, 0, 0, 0, 0, 6828, 17441, 15849, 
 0, 0, 15249, 3288, 6521, 0, 8060, 1465, 0, 6521, 8054, 3379, 
 0, 7042, 0, 6641, 4754, 4989, 0, 3679, 0, 0, 8032, 0, 8414, 
 8754, 8050, 1671, 0, 14582, 8040, 0, 0, 0, 0, 0, 0, 0, 0, 
 14201, 0, 1997, 0, 0, 9299, 0, 2641, 0, 0, 0, 18476, 0, 6887, 
 10445, 0, 0, 9825, 16333, 0, 0, 6715, 3720, 0, 0, 0, 12587, 
 6781, 3496, 4395, 0, 8998, 2075, 0, 10447, 0, 0, 0, 0, 15836, 
 0, 285, 0, 2698, 476, 6989, 3885, 0, 18689, 0, 3876, 0, 8995, 
 10081, 0, 1522, 18274, 15774, 455, 13442, 8993, 0, 6649, 0, 
 0, 0, 11000, 15934, 0, 0, 0, 17058, 9928, 10157, 13842, 18390, 
 10424, 0, 0, 0, 70, 0, 0, 4497, 0, 0, 0, 0, 0, 16732, 2822, 
 17964, 0, 2780, 0, 4413, 3044, 357, 18284, 5491, 1464, 15942, 
 0, 0, 0, 0, 5868, 0, 0, 0, 0, 139, 0, 0, 0, 12259, 0, 3636, 
 2438, 0, 6964, 0, 5588, 9497, 12682, 0, 17212, 0, 18782, 793, 
 0, 0, 0, 0, 13749, 0, 0, 519, 0, 2257, 0, 1382, 0, 8709, 
 0, 0, 8609, 13555, 6927, 0, 11998, 7387, 5982, 7792, 7809, 1137, 
 1167, 0, 0, 472, 0, 4597, 15701, 17390, 9733, 13502, 10718, 
 17899, 12594, 4154, 616, 13056, 10427, 4451, 8797, 18499, 7010, 
 12248, 5236, 11810, 15095, 7494, 0, 1118, 0, 17469, 0, 14206, 
 0, 17001, 0, 0, 11719, 1462, 1254, 0, 0, 0, 0, 7425, 0, 5599, 
 1555, 3383, 0, 11112, 0, 2575, 0, 0, 0, 0, 0, 7853, 0, 1297, 
 14697, 10743, 8936, 0, 111, 0, 0, 0, 0, 123, 6435, 2840, 13761, 
 0, 16145, 0, 0, 5240, 0, 2913, 5374, 878, 0, 0, 5752, 0, 5227, 
 0, 0, 0, 3230, 0, 2156, 755, 0, 0, 7807, 529, 18714, 14176, 
 0, 7014, 1110, 0, 15065, 7836, 0, 5382, 6374, 0, 5434, 1448, 
 0, 7536, 3386, 2271, 0, 0, 0, 16098, 14408, 0, 2039, 0, 231, 
 0, 7644, 4643, 3793, 101, 5207, 101, 0, 1422, 13324, 4789, 18185, 
 0, 1346, 9176, 16018, 15072, 18066, 16257, 15093, 1153, 13785, 
 7265, 0, 1182, 5881, 0, 6360, 7263, 0, 0, 7043, 12500, 0, 0, 
 8489, 99, 0, 14742, 13221, 2003, 3433, 0, 8487, 7261, 0, 18432, 
 0, 4495, 7341, 7326, 16864, 5623, 2390, 7359, 7353, 4628, 7327, 
 7323, 7342, 7319, 7338, 8666, 7340, 0, 0, 8680, 0, 7999, 8383, 
 75, 0, 3196, 0, 6972, 1001, 9525, 5086, 8689, 7650, 0, 0, 16646, 
 0, 16133, 16540, 0, 433, 2056, 2217, 0, 7265, 0, 4989, 9298, 
 17232, 6409, 5614, 7169, 0, 0, 2108, 11856, 17264, 0, 1028, 
 0, 11084, 6207, 0, 13773, 3646, 0, 16840, 0, 0, 14361, 1059, 
 0, 0, 0, 0, 6037, 0, 1938, 0, 0, 0, 0, 6947, 15527, 9454, 
 3370, 622, 17022, 16167, 3670, 4852, 8293, 465, 3371, 2752, 7062, 
 8183, 1928, 0, 0, 0, 13571, 14413, 0, 0, 4194, 6107, 0, 13856, 
 16864, 16300, 6024, 8033, 0, 1564, 0, 0, 0, 9943, 10091, 13766, 
 17750, 11949, 5324, 1708, 0, 0, 0, 0, 0, 9794, 10127, 0, 4, 
 0, 6193, 0, 0, 17938, 14759, 408, 0, 0, 0, 7109, 0, 0, 0, 
 0, 0, 262, 18755, 11089, 3122, 15060, 0, 6441, 0, 3060, 0, 
 0, 11183, 0, 17940, 0, 9584, 0, 1678, 7149, 0, 0, 9945, 0, 
 7142, 1147, 0, 0, 2934, 10443, 6796, 0, 12607, 1097, 18548, 
 0, 1534, 0, 0, 0, 0, 14948, 15912, 10039, 0, 0, 5117, 12866, 
 17615, 0, 12429, 10087, 14894, 0, 0, 0, 4920, 0, 12508, 0, 
 12281, 13752, 0, 2833, 7897, 0, 3194, 18062, 15712, 0, 0, 4990, 
 1124, 1196, 0, 0, 18086, 11010, 0, 0, 15526, 913, 16295, 4525, 
 6504, 7105, 0, 0, 0, 18249, 13321, 1249, 10315, 5380, 0, 0, 
 16833, 0, 9615, 0, 0, 0, 14352, 0, 1067, 0, 14587, 0, 0, 2903, 
 13802, 15820, 3218, 13552, 0, 9590, 16186, 1459, 0, 10596, 0, 
 0, 15786, 6863, 0, 536, 0, 0, 2581, 3734, 8281, 0, 984, 0, 
 0, 10873, 0, 6548, 7634, 0, 6260, 0, 9202, 0, 16982, 0, 15458, 
 16680, 6558, 0, 11748, 0, 18591, 14079, 14991, 0, 0, 2227, 0, 
 5462, 0, 0, 0, 0, 0, 6576, 0, 0, 9063, 17969, 9649, 0, 18194, 
 1572, 590, 0, 0, 2740, 0, 0, 0, 0, 1798, 0, 2887, 4713, 0, 
 1904, 8200, 428, 0, 0, 13683, 15766, 3447, 2554, 0, 0, 6170, 
 12783, 12867, 0, 0, 0, 7095, 11702, 15834, 0, 164, 0, 0, 8122, 
 17150, 854, 3472, 0, 290, 0, 5840, 13453, 3471, 4847, 0, 0, 
 8730, 4160, 13470, 0, 0, 4735, 3469, 4248, 0, 0, 3468, 0, 1718, 
 0, 6174, 967, 5024, 13427, 0, 9920, 3314, 16773, 0, 107, 4065, 
 7991, 0, 8275, 0, 17810, 18262, 14730, 16062, 0, 0, 5725, 5166, 
 1213, 12865, 0, 0, 256, 0, 0, 0, 0, 10603, 0, 962, 15408, 
 1100, 14891, 0, 0, 15301, 8145, 10482, 0, 0, 4454, 0, 7886, 
 4248, 14299, 0, 0, 16822, 9173, 0, 17198, 6197, 13603, 0, 0, 
 0, 0, 394, 420, 0, 0, 0, 9690, 8558, 0, 0, 0, 18351, 4464, 
 79, 510, 0, 918, 6349, 3989, 5957, 0, 0, 0, 0, 925, 3246, 
 8137, 2468, 18, 0, 0, 397, 17803, 7833, 16675, 8443, 0, 16968, 
 0, 6752, 16222, 0, 3617, 10547, 17284, 0, 0, 0, 0, 0, 0, 0, 
 3930, 0, 1355, 0, 0, 12335, 9548, 16378, 1275, 1470, 17096, 
 0, 0, 0, 0, 4694, 0, 7631, 0, 12071, 0, 395, 6329, 0, 1552, 
 0, 10435, 0, 0, 14339, 5292, 13362, 1027, 4757, 758, 18805, 
 0, 4961, 0, 12957, 13005, 0, 0, 4164, 5002, 9247, 0, 11450, 
 10311, 0, 0, 784, 0, 12090, 6960, 0, 11140, 0, 0, 0, 11805, 
 4013, 15973, 5022, 0, 0, 5314, 16340, 10207, 0, 0, 695, 1315, 
 10165, 0, 0, 0, 6803, 3853, 0, 14025, 0, 0, 10715, 7939, 0, 
 12105, 0, 730, 4196, 4092, 0, 0, 11176, 14355, 0, 11364, 0, 
 0, 0, 3204, 0, 10189, 0, 1568, 6026, 10, 0, 2893, 0, 0, 0, 
 8375, 18725, 1393, 0, 0, 463, 0, 0, 17333, 0, 0, 2022, 0, 
 0, 0, 7041, 14306, 7041, 0, 0, 0, 0, 0, 6292, 12033, 0, 0, 
 8823, 10727, 0, 0, 3939, 1395, 3426, 0, 9455, 17548, 0, 10244, 
 0, 0, 0, 0, 0, 0, 0, 0, 2458, 0, 18055, 18244, 3687, 17387, 
 0, 16183, 0, 18180, 4265, 8016, 12865, 8802, 2640, 0, 16151, 
 0, 0, 4634, 0, 16149, 0, 15139, 0, 0, 0, 0, 7324, 0, 0, 13545, 
 4105, 0, 15363, 0, 437, 2492, 0, 0, 7084, 15450, 2014, 0, 13810, 
 0, 0, 0, 16130, 0, 7038, 782, 1583, 0, 17411, 0, 5075, 9900, 
 0, 7116, 2171, 0, 0, 5316, 1591, 0, 14764, 0, 4058, 0, 0, 
 702, 2072, 0, 5526, 0, 4057, 6688, 18427, 0, 11765, 0, 0, 0, 
 0, 0, 0, 0, 0, 0, 17050, 9622, 17927, 0, 10121, 0, 5043, 0, 
 4870, 0, 0, 0, 0, 0, 2634, 0, 0, 0, 10931, 0, 7362, 0, 9912, 
 1459, 0, 0, 8192, 3326, 0, 9427, 0, 14172, 0, 14880, 4600, 
 10425, 0, 17994, 14477, 5716, 3039, 9649, 0, 4616, 0, 17908, 
 0, 0, 4248, 8510, 5303, 3231, 0, 0, 17953, 0, 2351, 9395, 0, 
 2130, 3038, 3030, 3515, 8215, 4638, 0, 8769, 6022, 6535, 6567, 
 8834, 1036, 1202, 2324, 6631, 6013, 8924, 1097, 4535, 5718, 2833, 
 13618, 10376, 0, 16554, 5637, 12239, 0, 3944, 11539, 2167, 0, 
 0, 0, 3962, 0, 0, 0, 5247, 8871, 2519, 5710, 2565, 0, 3961, 
 0, 6861, 0, 0, 11859, 1748, 0, 0, 0, 8080, 0, 1846, 0, 0, 
 2461, 0, 12844, 4225, 18623, 2044, 9310, 7090, 7252, 8872, 9306, 
 11599, 12007, 10833, 15387, 0, 11474, 377, 5887, 4062, 3247, 
 2877, 6782, 13092, 6195, 17061, 9396, 1337, 9313, 5060, 3111, 
 0, 0, 0, 0, 16387, 0, 0, 0, 0, 0, 0, 0, 0, 12412, 18320, 
 0, 0, 5872, 7638, 0, 0, 0, 11381, 15352, 4580, 0, 3517, 10902, 
 0, 8954, 0, 7471, 0, 0, 2824, 0, 0, 1952, 0, 0, 17829, 13818, 
 17885, 0, 8746, 8070, 3343, 0, 12692, 4655, 10374, 11954, 7629, 
 1311, 11905, 0, 7296, 0, 7292, 0, 2452, 9887, 11276, 9384, 0, 
 0, 8511, 0, 17211, 87, 3654, 0, 18689, 0, 0, 0, 4212, 0, 10070, 
 13506, 0, 0, 0, 0, 6935, 18716, 3956, 6331, 12330, 0, 10237, 
 0, 4635, 15059, 0, 0, 6812, 0, 10431, 0, 10699, 9176, 10433, 
 8255, 3246, 0, 14974, 2082, 6388, 0, 0, 0, 0, 954, 5121, 6257, 
 3667, 0, 7769, 0, 9787, 8877, 6193, 4823, 0, 6458, 11349, 15534, 
 0, 756, 9798, 10335, 0, 0, 0, 4845, 11438, 0, 7436, 4151, 718, 
 0, 3738, 6053, 0, 7712, 3845, 17128, 0, 2999, 1784, 0, 0, 0, 
 1906, 4922, 10077, 0, 17641, 15011, 13249, 18113, 6666, 0, 0, 
 16467, 12965, 16249, 3990, 15813, 2218, 9564, 433, 4778, 6089, 
 0, 0, 0, 0, 10025, 0, 6821, 0, 6065, 0, 0, 0, 0, 0, 0, 0, 
 4988, 0, 0, 2552, 0, 6696, 2904, 1895, 12667, 0, 3341, 3471, 
 0, 6416, 18059, 8200, 12995, 17718, 13989, 0, 6399, 0, 0, 0, 
 5885, 12235, 13292, 0, 8841, 3740, 14817, 0, 0, 8644, 2985, 
 10404, 2112, 6367, 11690, 14153, 0, 0, 0, 13505, 8947, 0, 18816, 
 0, 0, 0, 0, 0, 0, 3694, 8691, 0, 17323, 4892, 0, 0, 7411, 
 2090, 0, 11279, 9443, 0, 316, 0, 0, 1104, 2113, 333, 7503, 
 9005, 0, 0, 1670, 0, 3783, 0, 0, 8615, 0, 0, 0, 5728, 0, 
 9747, 3626, 9382, 0, 5430, 0, 0, 17194, 16828, 10218, 2624, 
 0, 0, 6909, 0, 11200, 0, 1930, 0, 0, 0, 10657, 10984, 15505, 
 0, 18156, 0, 12287, 0, 4334, 6041, 5078, 0, 18315, 12365, 0, 
 14355, 18357, 0, 10027, 6115, 0, 10373, 874, 7495, 6728, 2738, 
 13126, 10134, 13113, 8434, 2266, 0, 0, 0, 0, 3284, 0, 0, 6262, 
 0, 0, 0, 11724, 9982, 0, 5933, 0, 0, 1577, 0, 0, 0, 0, 6091, 
 6850, 5251, 0, 7261, 0, 11105, 0, 12841, 0, 5104, 1197, 17123, 
 8658, 15679, 17558, 0, 0, 0, 4738, 10669, 0, 0, 3758, 5088, 
 0, 3359, 0, 0, 0, 12536, 13902, 0, 0, 14834, 5875, 0, 0, 0, 
 1894, 3793, 15810, 2574, 16204, 0, 4418, 2141, 7759, 8966, 7712, 
 1735, 0, 4036, 0, 6560, 0, 4039, 0, 3745, 6485, 0, 8135, 0, 
 0, 7909, 602, 0, 544, 11614, 10456, 6659, 471, 2773, 11171, 
 0, 0, 1750, 9516, 0, 0, 11532, 0, 14981, 0, 0, 0, 1959, 2302, 
 0, 0, 8957, 0, 17307, 0, 13956, 1031, 0, 9855, 0, 13992, 10835, 
 0, 2618, 99, 10979, 0, 0, 10121, 8671, 9536, 5140, 0, 1883, 
 0, 15211, 9677, 4163, 0, 0, 0, 0, 0, 6792, 0, 975, 0, 12893, 
 0, 0, 0, 0, 4242, 0, 0, 0, 10428, 5978, 0, 0, 0, 10083, 0, 
 12829, 6827, 1432, 2000, 13431, 13518, 1916, 7959, 3485, 0, 0, 
 0, 0, 7304, 0, 3736, 0, 0, 0, 0, 0, 4611, 6681, 14108, 9031, 
 1621, 0, 12667, 1080, 4520, 0, 0, 17273, 6967, 7576, 10890, 
 1982, 9476, 0, 14351, 2600, 0, 7677, 7219, 0, 0, 1980, 5552, 
 9695, 0, 18532, 0, 602, 0, 5565, 5697, 0, 1248, 2650, 1029, 
 0, 9680, 1977, 11312, 13887, 14824, 9515, 0, 3729, 15864, 0, 
 12790, 0, 0, 9856, 0, 7282, 13852, 0, 11735, 3792, 0, 9801, 
 4276, 4719, 6891, 7285, 0, 7276, 0, 4649, 8290, 7279, 0, 0, 
 0, 9532, 11445, 13520, 0, 1376, 0, 0, 9713, 0, 0, 2436, 17790, 
 4425, 0, 0, 2528, 1982, 0, 0, 0, 3744, 0, 12357, 0, 18133, 
 0, 3920, 0, 10811, 10135, 0, 0, 0, 15089, 3506, 0, 0, 3307, 
 0, 0, 0, 3302, 0, 9849, 11850, 8066, 0, 3308, 10784, 18743, 
 14761, 13345, 4926, 7464, 11694, 14282, 0, 14007, 6879, 5084, 
 2462, 10839, 17887, 15085, 10122, 0, 3823, 9546, 0, 0, 0, 16619, 
 0, 0, 13875, 9012, 0, 1849, 13023, 17712, 4307, 1764, 0, 4345, 
 1797, 0, 2972, 11978, 17114, 8656, 15611, 3584, 0, 10036, 350, 
 0, 4389, 15147, 7568, 9254, 2825, 0, 13734, 17892, 0, 15950, 
 0, 0, 3807, 0, 0, 0, 5472, 0, 2034, 5655, 1932, 0, 10477, 
 761, 4949, 208, 18116, 0, 0, 0, 0, 0, 5973, 0, 15208, 10967, 
 0, 0, 0, 567, 1092, 0, 0, 6457, 0, 0, 14791, 0, 1325, 0, 
 18548, 1540, 5238, 0, 0, 6568, 7892, 8876, 0, 14384, 0, 0, 
 0, 18741, 16349, 0, 0, 0, 0, 0, 2212, 0, 0, 10500, 3954, 0, 
 0, 10072, 6880, 0, 18010, 6527, 15953, 11728, 8127, 0, 1141, 
 3844, 0, 12611, 10067, 0, 13310, 438, 2386, 0, 0, 7602, 274, 
 0, 0, 17697, 14683, 13962, 0, 11420, 0, 0, 0, 0, 7646, 0, 
 0, 0, 0, 0, 0, 4021, 0, 7060, 0, 2888, 0, 1675, 0, 0, 919, 
 2833, 1255, 0, 6122, 1668, 0, 1476, 18647, 6949, 1461, 12234, 
 0, 12991, 15147, 0, 0, 7887, 0, 0, 8526, 11182, 0, 6176, 3233, 
 14014, 0, 18660, 0, 2835, 0, 0, 0, 2251, 0, 0, 193, 0, 0, 
 3133, 0, 0, 0, 2114, 1340, 0, 0, 15792, 4800, 16202, 0, 0, 
 3199, 0, 4542, 2941, 5357, 6241, 3800, 13171, 4757, 9768, 0, 
 18140, 0, 0, 2739, 0, 9602, 12207, 10605, 0, 0, 8284, 10203, 
 5006, 0, 0, 10594, 10376, 6810, 1950, 0, 8353, 7222, 5074, 10103, 
 13327, 0, 17241, 1853, 18745, 8739, 0, 0, 0, 16451, 0, 0, 15877, 
 9719, 2663, 2760, 14511, 0, 7664, 0, 14781, 0, 17402, 1355, 
 0, 1339, 8250, 0, 2177, 0, 15793, 5573, 9100, 0, 8014, 2520, 
 0, 6442, 8235, 0, 3826, 0, 16107, 9738, 3927, 0, 4552, 2839, 
 0, 509, 0, 3259, 11082, 797, 16749, 12202, 5825, 0, 537, 0, 
 0, 18754, 0, 0, 9102, 0, 17937, 0, 0, 0, 8141, 6359, 0, 0, 
 17167, 5775, 17668, 3758, 0, 18123, 0, 0, 2711, 8130, 0, 0, 
 0, 8129, 0, 0, 14825, 6191, 14332, 0, 0, 0, 0, 15283, 1034, 
 0, 0, 3359, 0, 0, 0, 0, 0, 4654, 15131, 15782, 1607, 0, 17116, 
 0, 0, 14239, 2037, 0, 0, 0, 5191, 8911, 1228, 11329, 0, 6269, 
 13374, 6449, 13313, 8913, 15911, 14811, 6042, 3117, 16969, 0, 
 3279, 1245, 2472, 0, 18238, 0, 17616, 5235, 0, 16834, 0, 7047, 
 3731, 1405, 0, 15270, 17207, 8633, 4094, 16720, 2778, 340, 6458, 
 4749, 0, 17754, 11489, 13331, 15165, 11293, 0, 0, 3314, 0, 8969, 
 0, 0, 0, 4391, 14732, 1107, 4231, 0, 7713, 4065, 0, 0, 0, 
 0, 4655, 15347, 0, 13599, 18570, 0, 0, 2496, 303, 0, 2269, 
 16811, 1121, 2809, 13686, 0, 16047, 0, 0, 7945, 3968, 14074, 
 3970, 0, 8807, 13301, 0, 16232, 4668, 0, 15732, 18312, 17219, 
 11039, 8075, 0, 5620, 6122, 8656, 5799, 2974, 0, 0, 9871, 0, 
 0, 7950, 408, 977, 9021, 4157, 0, 0, 2403, 0, 0, 10736, 0, 
 0, 741, 9066, 124, 0, 11068, 0, 0, 0, 0, 15880, 2156, 0, 0, 
 1999, 1559, 8009, 0, 0, 0, 0, 10048, 0, 0, 0, 2098, 0, 0, 
 3316, 3706, 0, 15618, 7537, 2376, 5260, 0, 0, 7990, 16441, 455, 
 7925, 7968, 0, 2459, 6173, 0, 6904, 15374, 0, 16889, 5853, 0, 
 4719, 5550, 11724, 7216, 0, 7214, 0, 12604, 14147, 12079, 0, 
 18738, 7643, 17055, 0, 4650, 17645, 0, 0, 0, 4064, 10444, 1210, 
 0, 0, 0, 4326, 0, 0, 11108, 0, 15355, 16635, 0, 1889, 0, 0, 
 0, 0, 2616, 0, 8459, 0, 6616, 0, 15043, 0, 0, 13019, 0, 7203, 
 0, 0, 239, 4262, 2694, 6253, 2268, 1995, 0, 0, 0, 0, 2747, 
 1350, 17408, 0, 0, 919, 0, 7188, 0, 4546, 0, 6009, 0, 0, 5688, 
 473, 15153, 11072, 0, 13650, 0, 0, 4531, 1834, 0, 6662, 0, 
 7185, 6953, 14561, 0, 0, 8989, 10162, 15069, 0, 17769, 0, 7643, 
 0, 0, 0, 3244, 5675, 0, 0, 0, 10997, 8836, 4530, 0, 0, 7437, 
 0, 4091, 10028, 3861, 0, 4902, 0, 12944, 0, 4089, 0, 16799, 
 0, 8131, 0, 2147, 8041, 15560, 9658, 0, 0, 297, 5626, 0, 0, 
 0, 7831, 17927, 0, 14009, 822, 4942, 0, 0, 808, 4378, 0, 0, 
 9038, 5339, 0, 9824, 7873, 7879, 5563, 0, 0, 6789, 11975, 1652, 
 0, 17746, 9462, 18337, 18758, 0, 2347, 9463, 0, 0, 0, 1244, 
 13361, 2076, 0, 6685, 496, 4059, 0, 0, 0, 7147, 5233, 0, 0, 
 0, 0, 32, 14575, 0, 0, 0, 1751, 7627, 0, 17591, 0, 6180, 6594, 
 5142, 7631, 0, 9093, 3127, 18791, 9568, 4221, 0, 0, 0, 4319, 
 0, 0, 0, 15900, 6120, 15489, 0, 0, 0, 1266, 1399, 1291, 0, 
 1292, 0, 0, 0, 3811, 15024, 819, 0, 0, 0, 18107, 9906, 0, 
 13501, 0, 6804, 0, 16998, 15731, 0, 16618, 9461, 0, 10298, 1946, 
 16254, 3857, 4383, 0, 14471, 0, 9913, 0, 17232, 0, 14081, 9868, 
 9526, 0, 0, 1947, 0, 4245, 14066, 14716, 0, 0, 11605, 13445, 
 9875, 0, 7608, 11810, 329, 12364, 3034, 3410, 0, 11201, 10094, 
 14901, 8370, 0, 723, 0, 0, 5798, 17532, 0, 0, 0, 0, 16713, 
 0, 0, 0, 985, 0, 0, 2725, 14085, 0, 8176, 13816, 0, 0, 15443, 
 18756, 12581, 0, 0, 0, 16320, 2144, 11382, 0, 3981, 0, 16261, 
 15189, 0, 1153, 6926, 2124, 16641, 0, 326, 0, 0, 0, 8681, 0, 
 0, 2454, 12589, 0, 18700, 0, 15912, 0, 7575, 0, 7255, 12733, 
 0, 0, 4958, 15518, 3546, 5065, 4229, 16885, 897, 9614, 12864, 
 5865, 14656, 9298, 13592, 2896, 15610, 0, 2041, 3591, 0, 0, 
 14166, 10679, 0, 0, 0, 0, 3142, 0, 4410, 0, 0, 7254, 0, 0, 
 10368, 15877, 0, 0, 12713, 0, 15718, 1594, 5891, 0, 0, 1323, 
 18289, 0, 0, 0, 0, 0, 0, 0, 0, 0, 7891, 9375, 0, 0, 11306, 
 0, 7086, 0, 14030, 9367, 15587, 4853, 3839, 12849, 6374, 1236, 
 733, 7404, 0, 0, 0, 3311, 11847, 0, 1177, 6669, 0, 0, 0, 4526, 
 6348, 16159, 11138, 14819, 2326, 0, 4278, 0, 4958, 7918, 908, 
 10979, 8485, 6535, 607, 1219, 0, 0, 13021, 0, 0, 0, 0, 6956, 
 6638, 0, 3838, 12756, 0, 18495, 9292, 1897, 9294, 9293, 9288, 
 11341, 5592, 15038, 0, 18773, 0, 0, 0, 0, 0, 0, 466, 14422, 
 0, 0, 907, 16164, 0, 0, 0, 0, 0, 2035, 0, 0, 0, 15600, 0, 
 0, 0, 0, 15110, 10601, 14580, 5381, 0, 10751, 0, 0, 0, 0, 
 4163, 10001, 4790, 752, 0, 0, 0, 0, 14123, 0, 0, 0, 14413, 
 15097, 1916, 12683, 16060, 17536, 16014, 5023, 1966, 9776, 5047, 
 0, 0, 5917, 246, 0, 0, 13423, 18367, 0, 0, 0, 0, 7955, 0, 
 2940, 0, 0, 0, 12653, 17781, 423, 773, 0, 0, 9211, 0, 0, 0, 
 0, 10121, 0, 0, 18832, 14483, 1597, 0, 0, 0, 0, 3306, 5186, 
 1711, 0, 15925, 16399, 5321, 0, 0, 0, 14582, 0, 0, 0, 1215, 
 0, 15745, 0, 112, 10160, 0, 0, 0, 4851, 3932, 0, 2165, 0, 
 7984, 0, 17327, 0, 0, 0, 1051, 16812, 779, 0, 4476, 8639, 0, 
 15892, 0, 1643, 16919, 0, 5652, 0, 5388, 0, 10005, 1144, 2721, 
 5442, 148, 0, 14505, 0, 3118, 16386, 0, 7875, 6363, 18510, 9778, 
 17861, 7660, 3411, 7639, 4015, 13972, 0, 1061, 8848, 5216, 0, 
 10736, 0, 696, 0, 4492, 2243, 18211, 4758, 0, 1979, 0, 4121, 
 8788, 3353, 0, 11067, 0, 0, 0, 3349, 6642, 13983, 18569, 0, 
 14338, 0, 1063, 0, 12369, 0, 0, 4463, 11797, 0, 0, 10158, 0, 
 956, 0, 0, 0, 3190, 760, 1574, 12021, 12153, 8295, 7381, 0, 
 0, 0, 0, 11542, 0, 8190, 10519, 125, 0, 5959, 15060, 0, 0, 
 509, 2581, 729, 0, 4204, 0, 0, 0, 2329, 0, 0, 16549, 0, 2100, 
 2470, 12539, 2428, 0, 0, 9560, 0, 17504, 0, 575, 0, 2675, 0, 
 1002, 5047, 16485, 0, 3818, 14740, 13570, 0, 0, 0, 947, 0, 
 10522, 0, 653, 0, 0, 1888, 15816, 27, 0, 0, 647, 115, 16342, 
 10023, 12989, 0, 11818, 0, 6814, 0, 3088, 249, 0, 10735, 3879, 
 17943, 0, 12226, 12731, 5805, 10876, 0, 0, 8400, 12540, 6413, 
 0, 0, 0, 5593, 0, 11316, 0, 13108, 12396, 0, 12928, 0, 0, 
 12405, 10318, 0, 13831, 7777, 3675, 0, 16371, 9121, 8726, 0, 
 4074, 16097, 0, 0, 0, 6277, 18148, 0, 0, 2294, 0, 0, 0, 5397, 
 2329, 9758, 0, 1820, 17779, 0, 0, 11173, 15352, 0, 0, 1832, 
 2372, 15702, 18319, 0, 0, 5705, 3715, 0, 0, 0, 0, 0, 0, 17599, 
 1021, 0, 6579, 6068, 0, 7133, 17174, 0, 0, 9760, 0, 10022, 
 0, 18273, 0, 0, 13728, 12674, 705, 0, 5117, 15399, 0, 2278, 
 0, 0, 0, 17197, 0, 7165, 14993, 12963, 0, 10144, 12651, 0, 
 9218, 0, 0, 4550, 0, 0, 0, 13778, 9850, 2323, 0, 35, 9634, 
 0, 5248, 0, 0, 0, 11291, 2598, 0, 1752, 0, 2026, 0, 0, 17264, 
 132, 14126, 0, 2740, 0, 0, 7166, 0, 0, 0, 6076, 12622, 7173, 
 0, 2459, 0, 0, 14329, 14715, 0, 18704, 8820, 0, 0, 936, 16632, 
 0, 0, 1336, 3753, 14227, 1826, 5524, 6027, 18174, 3560, 9037, 
 0, 0, 0, 8357, 0, 0, 7022, 0, 5934, 0, 0, 0, 880, 0, 0, 
 2026, 17858, 11910, 16697, 0, 1497, 0, 17862, 17577, 3052, 0, 
 0, 12759, 1104, 1762, 6799, 9971, 18101, 7907, 830, 3835, 15799, 
 0, 17224, 0, 0, 0, 17560, 6285, 0, 0, 17509, 6704, 14494, 3868, 
 10442, 0, 4318, 0, 0, 6543, 0, 0, 0, 1722, 0, 0, 7310, 10759, 
 0, 0, 13300, 0, 0, 5042, 3973, 12753, 14076, 11015, 0, 8246, 
 2113, 0, 0, 7221, 18794, 8350, 0, 17977, 0, 9193, 10911, 0, 
 430, 0, 9799, 3947, 1196, 10130, 0, 15015, 5405, 17513, 0, 0, 
 0, 4152, 467, 18785, 0, 10540, 8420, 16532, 5649, 4153, 0, 8682, 
 0, 7739, 0, 0, 0, 0, 7555, 5531, 0, 8686, 9505, 0, 15836, 
 0, 17751, 0, 3421, 6853, 12765, 18699, 16757, 18125, 17680, 3821, 
 3423, 0, 3940, 15767, 720, 3424, 15141, 0, 0, 0, 0, 0, 0, 
 0, 0, 0, 0, 450, 0, 0, 13675, 0, 0, 6336, 2425, 5895, 176, 
 0, 0, 0, 5754, 7772, 0, 14365, 0, 13540, 13978, 13317, 11876, 
 2365, 17737, 3189, 5363, 0, 2575, 6132, 0, 0, 0, 3743, 0, 14887, 
 8775, 3425, 6298, 0, 0, 0, 0, 2831, 3974, 0, 0, 7865, 0, 8312, 
 0, 9072, 16598, 1684, 11412, 9169, 12968, 0, 16272, 4069, 9288, 
 6295, 14967, 3422, 6344, 10730, 0, 6134, 0, 12821, 0, 11659, 
 0, 0, 0, 0, 0, 1233, 0, 0, 0, 0, 0, 0, 0, 17662, 9472, 0, 
 0, 11361, 12183, 0, 0, 3488, 16467, 766, 0, 17140, 0, 0, 0, 
 5732, 7352, 0, 0, 5805, 0, 0, 0, 0, 0, 1131, 0, 0, 0, 13604, 
 5518, 15811, 0, 0, 0, 10741, 3003, 0, 3693, 13618, 8720, 2267, 
 3802, 0, 5776, 12368, 3876, 0, 2055, 2833, 0, 44, 5351, 2911, 
 5340, 0, 0, 1181, 93, 9939, 1692, 6217, 6208, 410, 978, 0, 
 16156, 3519, 136, 0, 0, 13891, 7681, 0, 1180, 14264, 18102, 
 0, 7763, 4411, 2375, 11729, 4076, 3034, 18779, 14345, 11193, 
 0, 7691, 0, 0, 15929, 525, 2736, 14313, 0, 14458, 0, 7334, 
 2611, 7691, 0, 6224, 7957, 701, 6535, 3933, 6539, 2998, 11416, 
 11392, 4994, 1666, 0, 3040, 18095, 2591, 15996, 0, 342, 6536, 
 2309, 10832, 7633, 11769, 18170, 733, 17352, 13839, 0, 13212, 
 0, 0, 17252, 8887, 6518, 6550, 0, 0, 1949, 5389, 4121, 0, 0, 
 0, 6614, 9206, 6988, 2579, 0, 10098, 15772, 18080, 0, 12464, 
 0, 0, 9739, 0, 0, 0, 0, 0, 17689, 14803, 8657, 0, 16580, 0, 
 15633, 12410, 0, 18552, 0, 0, 8653, 0, 17295, 13131, 5685, 10779, 
 0, 4358, 10278, 9296, 1832, 1113, 0, 0, 0, 5492, 18561, 0, 
 0, 0, 16714, 3879, 0, 0, 15681, 7864, 14383, 7878, 1069, 14989, 
 10852, 8067, 1356, 0, 8063, 0, 449, 14109, 6942, 0, 12490, 0, 
 8065, 1821, 7612, 0, 9431, 16080, 0, 1414, 0, 0, 4004, 0, 0, 
 0, 5035, 0, 15631, 14390, 1497, 4620, 0, 6830, 12952, 0, 11332, 
 0, 9692, 3599, 14303, 3233, 17165, 7267, 538, 0, 7363, 0, 14098, 
 631, 14700, 13171, 1891, 11351, 0, 9249, 7572, 16807, 0, 3084, 
 10505, 1870, 0, 0, 0, 7263, 11639, 2104, 3620, 0, 8206, 10744, 
 3610, 5278, 1817, 2231, 13089, 0, 0, 5766, 0, 8347, 1672, 3155, 
 0, 0, 2107, 0, 0, 12471, 0, 10719, 0, 9722, 0, 15020, 0, 11930, 
 7620, 3084, 0, 273, 0, 13310, 0, 516, 5767, 4387, 0, 0, 2669, 
 0, 5196, 0, 15006, 14805, 4304, 0, 0, 11533, 0, 4815, 4481, 
 10984, 0, 0, 1720, 0, 0, 0, 1764, 17325, 2445, 821, 0, 0, 
 0, 0, 0, 0, 0, 5884, 0, 13451, 0, 1950, 18148, 2604, 14830, 
 0, 15006, 1495, 18835, 17478, 14820, 0, 0, 0, 6663, 0, 0, 0, 
 0, 0, 0, 5121, 0, 0, 18475, 0, 0, 2748, 0, 10345, 0, 15839, 
 0, 0, 18005, 0, 9970, 0, 0, 0, 0, 1003, 0, 0, 12471, 0, 7218, 
 0, 104, 14927, 2902, 6065, 18738, 8723, 0, 10330, 7675, 10061, 
 6486, 4067, 9800, 0, 7377, 4986, 1812, 0, 0, 5422, 8946, 4892, 
 0, 8269, 0, 14829, 0, 0, 0, 0, 0, 0, 4468, 5251, 0, 15991, 
 2742, 0, 0, 13344, 801, 0, 0, 0, 0, 9835, 4824, 14983, 4826, 
 8119, 8604, 17858, 14753, 8175, 8605, 5084, 0, 0, 15857, 0, 
 6187, 0, 0, 2628, 3299, 18583, 17735, 0, 0, 4491, 797, 0, 0, 
 637, 0, 6736, 0, 0, 13106, 2070, 2073, 0, 6285, 0, 14247, 11346, 
 0, 0, 8430, 0, 17358, 15670, 18830, 15001, 10091, 18408, 5261, 
 18419, 13916, 12825, 14121, 938, 0, 1125, 8281, 0, 676, 0, 5163, 
 6730, 0, 0, 244, 0, 518, 4912, 1835, 0, 0, 0, 10185, 1904, 
 2746, 0, 0, 10498, 10430, 11694, 8605, 3916, 11588, 12269, 13754, 
 15372, 10069, 0, 0, 4190, 3700, 1735, 2420, 0, 757, 1626, 0, 
 0, 0, 13250, 0, 3942, 5583, 883, 4878, 0, 0, 0, 5363, 7143, 
 0, 3977, 0, 0, 0, 0, 0, 2358, 1788, 0, 0, 0, 0, 2498, 0, 
 0, 0, 8056, 0, 0, 0, 0, 0, 7507, 0, 6245, 0, 7296, 0, 7571, 
 4139, 8241, 9410, 681, 11216, 6696, 0, 0, 0, 5725, 7240, 0, 
 14061, 0, 2099, 7514, 0, 0, 0, 0, 2488, 10331, 0, 11774, 5293, 
 4460, 8654, 9752, 1476, 0, 9657, 12716, 7661, 6612, 8153, 0, 
 4251, 0, 18617, 0, 13224, 0, 8650, 2608, 5060, 2856, 0, 5175, 
 3625, 1677, 3467, 2948, 15995, 8944, 1014, 11796, 5717, 0, 4868, 
 0, 18069, 0, 8200, 0, 0, 0, 12693, 0, 0, 5628, 0, 0, 0, 7622, 
 0, 0, 0, 0, 6801, 7764, 0, 0, 0, 0, 6515, 18252, 4379, 17614, 
 0, 748, 14191, 13266, 10295, 4891, 5098, 0, 7809, 0, 4318, 0, 
 0, 9112, 5097, 3964, 1868, 3342, 0, 1759, 0, 4481, 16037, 16928, 
 9820, 0, 9044, 12347, 9766, 0, 2879, 0, 10103, 0, 2686, 0, 
 16603, 0, 0, 0, 5489, 0, 0, 0, 0, 0, 0, 2621, 0, 0, 11793, 
 16788, 10048, 7498, 4315, 0, 753, 0, 0, 0, 4629, 0, 0, 5496, 
 10588, 11611, 0, 12991, 1997, 9374, 1015, 18011, 10118, 14106, 
 0, 4125, 916, 0, 0, 1213, 7847, 0, 1577, 3628, 15485, 0, 0, 
 2340, 9169, 16869, 16494, 3257, 13184, 7607, 3914, 1055, 0, 9529, 
 0, 14904, 0, 6430, 0, 4861, 0, 0, 0, 0, 2653, 8110, 14975, 
 970, 0, 16998, 16535, 0, 0, 0, 6228, 0, 0, 10341, 1949, 4275, 
 7521, 0, 0, 3451, 5261, 0, 0, 0, 0, 9757, 3227, 0, 2886, 6477, 
 0, 0, 6518, 4535, 0, 0, 10218, 7704, 15711, 2775, 0, 14895, 
 0, 0, 6334, 0, 8771, 6396, 0, 14696, 0, 3122, 0, 0, 1124, 
 0, 15761, 3687, 0, 0, 15880, 352, 11444, 0, 0, 0, 14005, 0, 
 0, 0, 0, 6969, 7581, 671, 15022, 0, 9094, 0, 8323, 15042, 3105, 
 15149, 0, 0, 0, 0, 0, 0, 0, 0, 17416, 0, 0, 16337, 10132, 
 13056, 0, 7509, 467, 16991, 0, 13177, 0, 0, 0, 0, 11899, 1912, 
 0, 0, 0, 0, 3021, 1400, 0, 16532, 0, 13727, 0, 13056, 0, 15040, 
 9123, 0, 8943, 4485, 565, 0, 2869, 0, 2653, 2639, 15091, 0, 
 7299, 6432, 13213, 0, 7000, 0, 10258, 0, 10539, 0, 13679, 0, 
 4853, 0, 815, 0, 0, 5601, 2294, 0, 14647, 7094, 2004, 11307, 
 11756, 18666, 1604, 13965, 5150, 0, 14406, 0, 0, 17753, 0, 7709, 
 14523, 10275, 10299, 0, 2969, 7500, 0, 0, 15324, 10487, 0, 16431, 
 0, 4444, 16166, 850, 4920, 0, 7285, 17012, 15360, 0, 16749, 
 0, 1087, 0, 4976, 2468, 5827, 10719, 0, 0, 704, 13801, 17143, 
 500, 0, 8814, 0, 4162, 7897, 3415, 0, 6710, 4755, 10249, 0, 
 0, 0, 4466, 1108, 0, 13114, 0, 0, 10394, 3524, 0, 0, 2766, 
 0, 12047, 2370, 15818, 4798, 16451, 0, 3460, 2325, 5256, 0, 
 6244, 1121, 0, 0, 1242, 15573, 0, 0, 3133, 9390, 0, 4327, 8994, 
 3273, 0, 0, 12495, 16498, 0, 6034, 5677, 15985, 0, 0, 18282, 
 9509, 6998, 0, 11681, 17319, 17697, 3746, 559, 0, 9025, 0, 15692, 
 3551, 0, 5856, 0, 0, 0, 0, 10469, 9508, 0, 0, 8116, 0, 14382, 
 14503, 13325, 17995, 0, 14073, 14270, 10297, 0, 2398, 4780, 9064, 
 0, 0, 1509, 0, 14359, 13497, 8319, 16524, 0, 13024, 0, 0, 0, 
 0, 0, 0, 4333, 16584, 0, 14946, 17620, 9267, 17972, 8069, 0, 
 17861, 10593, 3864, 13089, 7926, 11521, 0, 0, 0, 17458, 7898, 
 4542, 0, 2921, 0, 6081, 0, 2030, 7118, 0, 0, 0, 16903, 4938, 
 1059, 13941, 12019, 10634, 11445, 0, 15757, 0, 4643, 6359, 17133, 
 4233, 3043, 10508, 12078, 13098, 0, 6759, 4485, 8053, 16218, 
 14776, 14114, 14551, 3899, 6072, 10359, 6894, 0, 13505, 6689, 
 9263, 6765, 0, 12747, 3573, 7742, 15543, 0, 4318, 16437, 4297, 
 0, 4987, 7869, 6626, 0, 0, 5102, 10142, 6562, 6530, 0, 18245, 
 0, 0, 8355, 0, 9762, 0, 0, 12304, 10962, 0, 7258, 0, 4530, 
 10103, 5972, 16337, 8216, 8015, 0, 0, 7515, 0, 0, 9542, 16476, 
 9787, 431, 0, 0, 0, 0, 6101, 0, 0, 0, 5533, 8557, 0, 0, 379, 
 0, 3094, 0, 956, 17443, 12141, 0, 0, 8556, 957, 12200, 18808, 
 10927, 18646, 0, 0, 16853, 0, 2521, 4245, 0, 0, 2545, 2187, 
 2381, 239, 6354, 13558, 2335, 9598, 7631, 1221, 0, 0, 16608, 
 0, 16232, 0, 0, 0, 0, 9752, 17915, 2670, 3749, 18768, 11213, 
 0, 13633, 18097, 0, 0, 0, 0, 0, 14464, 0, 6471, 8416, 5535, 
 4819, 0, 2396, 0, 0, 0, 0, 0, 0, 0, 11471, 7432, 10437, 7483, 
 241, 0, 7362, 7188, 0, 0, 6841, 985, 0, 18632, 0, 0, 0, 0, 
 2187, 7793, 0, 9791, 0, 0, 0, 0, 0, 5641, 0, 0, 17147, 428, 
 3928, 673, 4896, 0, 4856, 8304, 0, 7399, 0, 11683, 5576, 5699, 
 1695, 7116, 0, 0, 5178, 0, 18155, 0, 8310, 0, 3659, 18671, 
 9837, 0, 8724, 0, 15743, 0, 6809, 2448, 9494, 6627, 14502, 9708, 
 0, 995, 0, 4795, 7417, 8048, 0, 0, 4024, 5898, 0, 18439, 3551, 
 16288, 0, 7134, 17877, 260, 138, 5522, 13867, 3162, 6996, 0, 
 4584, 3167, 5217, 4088, 6831, 3164, 17188, 7385, 966, 5122, 0, 
 5128, 12618, 8, 0, 0, 3964, 0, 5160, 0, 0, 0, 8627, 0, 5208, 
 3466, 0, 8495, 9712, 2906, 0, 3744, 0, 209, 4027, 0, 0, 0, 
 17307, 11319, 0, 3820, 0, 6405, 5287, 0, 1431, 266, 0, 0, 7326, 
 0, 0, 9899, 0, 0, 0, 8478, 0, 99, 0, 0, 0, 0, 10619, 3896, 
 4197, 0, 0, 0, 270, 0, 6529, 6561, 0, 0, 0, 0, 0, 14393, 
 0, 0, 0, 0, 4402, 4447, 0, 0, 6101, 0, 9759, 13433, 1851, 
 0, 0, 466, 6831, 0, 14247, 0, 13740, 0, 2021, 6774, 0, 0, 
 1989, 0, 10132, 0, 0, 0, 15821, 0, 15153, 5942, 18617, 2678, 
 4687, 934, 3317, 0, 1550, 0, 13239, 6196, 17109, 0, 4414, 71, 
 0, 9796, 0, 0, 15711, 0, 17959, 0, 0, 6745, 5688, 0, 0, 0, 
 0, 17880, 7558, 3564, 5599, 17178, 0, 3575, 0, 3570, 4293, 14766, 
 1358, 15992, 0, 3566, 0, 0, 11605, 0, 9785, 923, 0, 10611, 
 0, 15667, 10271, 18043, 4287, 7185, 0, 8503, 14679, 7409, 13298, 
 0, 15835, 5844, 16782, 18615, 2623, 0, 4001, 0, 7314, 0, 4162, 
 5254, 13215, 0, 0, 0, 17971, 0, 639, 11345, 0, 0, 13385, 0, 
 0, 237, 0, 0, 16731, 0, 0, 0, 0, 9832, 6639, 12588, 5081, 
 8473, 0, 8834, 0, 0, 0, 0, 4559, 6575, 0, 8395, 547, 0, 5543, 
 13179, 12838, 17222, 0, 4921, 5157, 9277, 7297, 9275, 17043, 
 9273, 5919, 483, 15949, 0, 1140, 0, 18389, 1028, 7037, 4787, 
 13884, 0, 0, 17471, 4068, 0, 7098, 7788, 7162, 0, 2532, 0, 
 0, 9401, 0, 2842, 2583, 6136, 12122, 0, 15483, 3018, 487, 0, 
 0, 6832, 9656, 299, 7490, 648, 421, 0, 15929, 4239, 8283, 9284, 
 14549, 13378, 5463, 0, 4628, 5509, 14007, 7491, 10237, 9597, 
 8656, 0, 8386, 3077, 6219, 0, 15799, 0, 0, 7980, 4363, 15495, 
 10665, 8635, 11528, 0, 0, 1360, 0, 0, 3246, 6054, 0, 8389, 
 2861, 11987, 4132, 11808, 2657, 431, 11381, 18018, 7514, 16929, 
 0, 0, 2294, 0, 9957, 16521, 0, 1048, 12520, 0, 0, 9481, 6360, 
 0, 5698, 13548, 0, 0, 1345, 0, 0, 6372, 5970, 0, 0, 0, 0, 
 12373, 7908, 16397, 0, 14586, 0, 3416, 8054, 18675, 0, 13576, 
 0, 7508, 12512, 3414, 0, 0, 0, 17672, 0, 0, 9847, 0, 0, 0, 
 0, 0, 451, 6691, 0, 0, 1250, 0, 18653, 1658, 8057, 3690, 10186, 
 10126, 12883, 0, 15781, 0, 15931, 9115, 0, 0, 10733, 2732, 0, 
 0, 0, 17991, 1688, 4918, 0, 7861, 8494, 0, 0, 0, 0, 1218, 
 0, 7240, 0, 5436, 0, 7664, 3801, 1403, 5071, 16416, 12064, 13266, 
 0, 0, 0, 0, 7657, 2428, 15953, 0, 10284, 0, 1519, 7910, 8441, 
 0, 11628, 15271, 3643, 0, 0, 549, 1305, 0, 800, 0, 6888, 6017, 
 0, 0, 0, 0, 15307, 0, 7561, 17356, 0, 16995, 8958, 1851, 6751, 
 0, 0, 6922, 0, 0, 9620, 0, 1289, 8262, 0, 0, 4752, 7387, 7281, 
 0, 747, 0, 13717, 5682, 0, 8763, 0, 0, 16482, 8770, 0, 3139, 
 0, 15435, 4848, 12710, 0, 0, 0, 15236, 0, 18140, 18715, 472, 
 0, 0, 11155, 9873, 4871, 5399, 0, 8016, 793, 15126, 0, 3679, 
 0, 3991, 7663, 18659, 0, 311, 9484, 0, 0, 16743, 5811, 15442, 
 6177, 5370, 6035, 7672, 2186, 2211, 0, 6774, 11907, 987, 0, 
 8613, 16938, 5808, 0, 0, 4849, 9265, 0, 16532, 17182, 0, 16451, 
 16028, 6754, 0, 0, 1851, 0, 15339, 12901, 0, 0, 0, 10293, 0, 
 0, 0, 9813, 16370, 0, 13928, 2123, 0, 0, 0, 0, 0, 6793, 10050, 
 581, 0, 18671, 474, 14784, 0, 0, 14346, 0, 0, 5886, 8134, 9854, 
 11628, 5779, 556, 8454, 0, 17341, 11157, 0, 0, 7670, 0, 737, 
 13958, 15174, 0, 11874, 3078, 0, 0, 18815, 17754, 4341, 0, 16499, 
 0, 12371, 0, 0, 173, 17447, 0, 0, 0, 0, 0, 0, 0, 0, 3326, 
 16370, 866, 0, 0, 11757, 0, 18263, 0, 4257, 0, 0, 0, 3884, 
 1017, 2829, 14752, 0, 0, 2115, 0, 0, 8420, 15790, 6695, 8334, 
 18634, 661, 4707, 0, 14813, 0, 13421, 0, 0, 0, 18675, 3747, 
 5905, 7896, 15811, 0, 8395, 16710, 5131, 1038, 15768, 12243, 
 10568, 3078, 8057, 0, 2285, 0, 0, 0, 449, 0, 12707, 8374, 12180, 
 659, 0, 0, 8163, 17632, 6607, 6543, 12998, 15343, 6487, 6483, 
 0, 0, 7748, 0, 0, 0, 7960, 0, 0, 978, 7131, 0, 3579, 18116, 
 7431, 7913, 0, 0, 13856, 9520, 0, 4417, 11904, 0, 4806, 5356, 
 4540, 6267, 11288, 6409, 1460, 8395, 6588, 11401, 0, 8398, 4490, 
 5682, 17211, 4746, 11493, 0, 7232, 16639, 17832, 4363, 14664, 
 12436, 0, 14230, 8245, 4110, 0, 0, 0, 2952, 3616, 9973, 0, 
 2479, 0, 16681, 7240, 5148, 0, 0, 0, 0, 828, 11679, 16587, 
 5731, 6755, 0, 5051, 3325, 0, 0, 8368, 0, 0, 3865, 16128, 3264, 
 13910, 0, 11322, 0, 16670, 2177, 542, 0, 0, 0, 0, 13184, 3348, 
 15400, 2660, 182, 0, 6404, 1405, 0, 0, 9513, 8847, 0, 403, 
 4351, 9683, 0, 0, 12810, 14642, 0, 16583, 1100, 0, 0, 1943, 
 0, 0, 6195, 34, 4605, 0, 141, 0, 0, 0, 0, 0, 1272, 0, 0, 
 0, 0, 0, 8165, 0, 3785, 0, 0, 13747, 3218, 5309, 0, 9644, 
 282, 8530, 16696, 0, 0, 7556, 6521, 12288, 0, 0, 6161, 0, 4564, 
 16631, 0, 6146, 0, 17158, 6059, 14818, 12248, 0, 5326, 11067, 
 7149, 11951, 2401, 10365, 0, 11380, 7153, 6472, 8254, 0, 12605, 
 2335, 7147, 0, 0, 5559, 3988, 0, 405, 2201, 6767, 9724, 12401, 
 14946, 3978, 17710, 5808, 0, 16380, 6433, 0, 12323, 9604, 0, 
 14129, 0, 0, 5601, 0, 7441, 11456, 3880, 0, 0, 0, 1125, 10164, 
 16666, 14891, 4461, 7500, 5226, 11733, 12478, 1816, 0, 18390, 
 6620, 99, 6524, 6508, 5366, 767, 14056, 7211, 0, 0, 7475, 0, 
 4876, 17857, 0, 2012, 6570, 14342, 7151, 5004, 0, 8690, 0, 12778, 
 2755, 9734, 0, 5980, 0, 6952, 3113, 76, 5840, 9180, 4174, 0, 
 4296, 4632, 2083, 0, 0, 8872, 366, 0, 0, 4692, 7014, 0, 0, 
 229, 16713, 11572, 0, 12630, 0, 9551, 0, 9316, 209, 3913, 3983, 
 8581, 0, 2420, 14036, 17824, 2717, 476, 687, 17404, 12837, 0, 
 0, 0, 17175, 0, 4649, 0, 4532, 3688, 9695, 0, 0, 0, 0, 0, 
 12963, 5398, 8725, 0, 5183, 0, 4751, 9686, 2524, 0, 0, 10373, 
 2979, 6938, 15682, 0, 0, 0, 0, 1063, 0, 15225, 1807, 10536, 
 0, 6525, 0, 4234, 0, 0, 16454, 2506, 10524, 12817, 6815, 0, 
 132, 8652, 3220, 0, 0, 18136, 16101, 7361, 0, 3984, 8540, 3672, 
 2767, 5722, 6845, 15074, 0, 8495, 0, 15685, 12199, 7526, 0, 
 2221, 2626, 1027, 0, 0, 5067, 0, 0, 6658, 7409, 226, 1738, 
 8364, 4218, 0, 9625, 0, 10710, 11346, 3279, 2481, 17009, 4044, 
 642, 8849, 0, 10087, 4930, 8157, 0, 0, 0, 12701, 0, 8853, 0, 
 8848, 5236, 0, 2520, 5497, 13043, 0, 0, 2306, 0, 0, 8839, 0, 
 7080, 1271, 0, 2735, 9855, 0, 0, 0, 0, 0, 0, 2600, 0, 0, 
 12498, 9774, 3269, 0, 12132, 11341, 0, 8685, 14682, 3817, 16576, 
 7328, 498, 3289, 3703, 18011, 1231, 11289, 4731, 12066, 0, 11797, 
 8321, 8752, 4821, 246, 16427, 0, 12302, 15541, 0, 15236, 820, 
 15228, 0, 0, 0, 13303, 13478, 16543, 6399, 8322, 0, 0, 5587, 
 4542, 0, 0, 2868, 0, 0, 10367, 11831, 5573, 3123, 0, 6734, 
 8030, 7011, 0, 16090, 0, 54, 0, 1616, 9402, 9401, 10096, 9399, 
 16035, 18639, 9404, 6664, 0, 18718, 9408, 17638, 0, 0, 2029, 
 9905, 0, 11825, 3384, 14625, 3841, 5457, 0, 14854, 10467, 3874, 
 0, 14534, 13772, 0, 0, 5129, 5113, 15126, 1053, 13810, 0, 0, 
 8232, 14544, 17160, 7140, 2804, 2046, 0, 0, 9417, 14558, 4560, 
 2157, 8879, 11, 7470, 8375, 4484, 1767, 806, 7142, 1276, 0, 
 5848, 1431, 10322, 2666, 1533, 296, 16089, 13731, 8865, 0, 0, 
 18174, 488, 2955, 8862, 6932, 0, 12252, 10157, 17004, 0, 6290, 
 0, 2565, 0, 8621, 1352, 872, 6882, 1491, 10469, 16856, 6532, 
 0, 0, 0, 3437, 13048, 12032, 6246, 0, 0, 0, 10018, 0, 0, 0, 
 6957, 0, 0, 0, 8892, 7461, 2600, 17869, 10216, 0, 2192, 1060, 
 9412, 16989, 5071, 4645, 0, 6470, 0, 0, 15176, 9631, 5530, 8125, 
 0, 0, 17765, 0, 0, 9503, 18375, 0, 0, 3501, 0, 16429, 0, 18788, 
 13553, 10461, 0, 0, 13666, 3842, 0, 0, 5134, 14074, 3721, 13916, 
 15070, 4646, 18675, 0, 0, 10824, 839, 0, 12650, 1809, 17130, 
 2220, 0, 0, 0, 0, 9534, 5867, 1323, 14160, 17788, 0, 0, 13565, 
 6520, 4006, 2540, 0, 0, 7341, 163, 0, 0, 4589, 0, 0, 12909, 
 18489, 10123, 11312, 0, 0, 0, 1270, 12203, 0, 0, 4770, 0, 0, 
 3775, 5591, 13496, 203, 1603, 9043, 13686, 0, 2498, 1, 2859, 
 0, 8504, 10372, 8762, 0, 0, 0, 0, 0, 18219, 13588, 6138, 9212, 
 17021, 12087, 8246, 0, 17913, 8708, 587, 0, 1302, 9424, 11406, 
 0, 1896, 18658, 7084, 14700, 3067, 9177, 7034, 7231, 6872, 11543, 
 6554, 1245, 11480, 0, 9854, 1941, 15420, 15188, 2273, 17686, 
 10417, 9445, 1375, 11658, 179, 0, 0, 11370, 9799, 2962, 15641, 
 1010, 0, 0, 11602, 0, 3005, 17752, 0, 2015, 0, 7558, 0, 10267, 
 0, 4174, 3571, 4918, 0, 15922, 0, 18827, 0, 7806, 17795, 883, 
 1274, 4157, 0, 0, 0, 9281, 18505, 3660, 0, 0, 2553, 4452, 10059, 
 4159, 18113, 0, 17548, 5356, 6424, 5396, 0, 1788, 0, 14134, 
 0, 2241, 2066, 6131, 9189, 1942, 9187, 1995, 16791, 1387, 4954, 
 11767, 9636, 0, 1692, 8930, 14658, 2538, 17908, 3362, 5534, 13721, 
 14157, 13498, 12055, 91, 18054, 0, 7412, 3428, 0, 0, 0, 15426, 
 3143, 6294, 7944, 16688, 1208, 13219, 3197, 0, 5786, 10910, 5317, 
 0, 8781, 4890, 0, 0, 10827, 10519, 13633, 15928, 0, 9910, 0, 
 0, 1922, 0, 15207, 0, 4320, 5712, 5568, 0, 0, 0, 14172, 4281, 
 0, 0, 10785, 0, 9609, 13249, 0, 0, 15998, 312, 0, 0, 0, 3536, 
 1447, 4443, 0, 2501, 7102, 0, 891, 2267, 0, 10322, 8548, 0, 
 10149, 2199, 1127, 1119, 9175, 0, 0, 0, 10189, 0, 2582, 10263, 
 9525, 4436, 15999, 3208, 1924, 5064, 7876, 9409, 15820, 0, 0, 
 8087, 0, 0, 0, 0, 0, 1785, 5172, 7282, 2034, 11770, 203, 0, 
 6507, 16129, 0, 0, 0, 16411, 9494, 0, 18682, 7329, 10860, 18594, 
 0, 0, 10588, 0, 0, 2159, 15564, 2217, 5009, 0, 7083, 0, 0, 
 2938, 0, 0, 0, 16897, 842, 5702, 2109, 0, 1853, 5508, 5388, 
 4660, 0, 9732, 14860, 2943, 3609, 16787, 0, 0, 0, 4130, 18087, 
 1594, 14682, 10587, 0, 10209, 3487, 5145, 0, 0, 15386, 7493, 
 3844, 1563, 3489, 10863, 5279, 390, 6734, 9038, 14137, 2367, 
 0, 16489, 6192, 0, 0, 0, 0, 3051, 16644, 2223, 0, 16606, 0, 
 0, 1908, 6188, 0, 0, 10545, 1500, 13866, 862, 0, 0, 12758, 
 6426, 0, 18830, 15600, 5632, 110, 876, 17480, 5730, 0, 0, 0, 
 0, 11190, 0, 3114, 881, 17045, 16348, 0, 4762, 4560, 0, 0, 
 0, 14154, 0, 843, 4305, 0, 0, 8482, 1912, 0, 0, 2235, 2550, 
 7937, 3176, 7426, 3479, 7086, 0, 0, 8693, 2953, 2724, 11000, 
 1231, 445, 0, 0, 5663, 3974, 0, 0, 699, 16499, 0, 16647, 0, 
 10792, 14128, 0, 16200, 0, 317, 14458, 0, 999, 5740, 0, 0, 
 1523, 11618, 17219, 6143, 4676, 0, 0, 9405, 6448, 0, 0, 2344, 
 0, 0, 10261, 11566, 0, 9732, 0, 4706, 10976, 16506, 0, 0, 4422, 
 0, 6617, 0, 4679, 0, 10833, 0, 5685, 0, 16292, 6047, 6937, 
 0, 0, 0, 15963, 7538, 2581, 0, 8991, 13302, 0, 8781, 0, 0, 
 13306, 0, 2832, 0, 17931, 0, 3243, 10522, 0, 0, 9918, 0, 4092, 
 11321, 0, 2854, 10629, 8325, 4462, 0, 13771, 5407, 864, 3261, 
 0, 16641, 5517, 18820, 6698, 0, 0, 15457, 4097, 757, 0, 0, 
 0, 2060, 16575, 14724, 4096, 0, 0, 3291, 0, 0, 0, 890, 4171, 
 0, 0, 4273, 0, 10700, 0, 12082, 7139, 2670, 0, 8048, 697, 7372, 
 3137, 0, 18217, 17534, 9730, 0, 0, 0, 0, 0, 11586, 0, 0, 927, 
 0, 1202, 0, 0, 0, 11684, 0, 16816, 0, 899, 0, 0, 0, 1508, 
 7580, 0, 0, 1545, 0, 1650, 15208, 5563, 0, 6404, 470, 13243, 
 14236, 0, 1920, 10713, 0, 0, 0, 0, 0, 16267, 1049, 2989, 6237, 
 11400, 17346, 0, 0, 0, 0, 0, 0, 7791, 0, 3958, 3714, 3093, 
 0, 1902, 0, 0, 1998, 191, 0, 0, 0, 18688, 6560, 0, 15788, 
 0, 0, 11954, 14655, 15930, 1357, 7917, 3722, 1815, 0, 0, 0, 
 0, 641, 17446, 5843, 6717, 0, 6899, 4986, 0, 4535, 292, 4221, 
 0, 0, 17133, 5005, 16513, 0, 14611, 0, 9991, 6655, 9244, 0, 
 3019, 961, 7529, 0, 17823, 12873, 16020, 0, 17080, 0, 1105, 
 0, 0, 0, 3151, 1063, 18343, 1530, 14920, 0, 13995, 0, 0, 948, 
 14008, 0, 1740, 0, 0, 16175, 8502, 0, 0, 2064, 2653, 0, 0, 
 10471, 7802, 0, 8812, 3971, 15901, 5218, 192, 4656, 7603, 0, 
 8745, 3657, 3144, 0, 18663, 0, 18050, 1478, 0, 5323, 16423, 
 0, 3123, 0, 8917, 0, 1250, 4384, 8920, 4594, 8890, 0, 0, 0, 
 15298, 0, 9190, 2942, 15132, 0, 2979, 10265, 0, 0, 0, 0, 0, 
 0, 0, 18179, 0, 0, 0, 0, 0, 4804, 8036, 0, 2096, 6839, 5945, 
 11439, 5943, 18337, 5949, 5948, 5447, 5057, 8713, 5952, 3465, 
 7646, 5957, 5956, 5955, 5954, 5961, 5960, 707, 7678, 8521, 17592, 
 5963, 12142, 8549, 1790, 5967, 5966, 1811, 10528, 0, 7948, 0, 
 6936, 0, 0, 0, 2483, 9416, 16951, 11791, 12319, 2907, 6703, 
 0, 16558, 0, 2506, 5275, 4313, 0, 4630, 7781, 8057, 0, 8055, 
 2830, 0, 0, 3683, 5539, 15852, 0, 9818, 15849, 0, 3974, 12981, 
 10268, 3777, 5862, 15038, 0, 0, 0, 13648, 18062, 0, 6188, 0, 
 4246, 13546, 4233, 0, 5144, 0, 6380, 4737, 2895, 0, 16048, 5700, 
 7494, 0, 8628, 0, 5324, 17254, 10216, 5645, 3598, 12851, 7148, 
 13418, 7154, 0, 6478, 8094, 233, 373, 6898, 13778, 15584, 573, 
 17617, 0, 11618, 4650, 11888, 13466, 0, 9299, 8089, 5816, 0, 
 6617, 5570, 5658, 560, 447, 15578, 8261, 6382, 0, 15183, 6378, 
 0, 3155, 652, 8385, 0, 1235, 221, 0, 16369, 7088, 3973, 0, 
 0, 0, 0, 3344, 4071, 0, 692, 0, 0, 1999, 3536, 11195, 17090, 
 0, 3133, 1150, 881, 7380, 7755, 18421, 5176, 0, 0, 11035, 0, 
 0, 7886, 0, 1308, 0, 0, 14941, 2632, 0, 0, 0, 0, 10110, 0, 
 0, 0, 10646, 2869, 1278, 2171, 4892, 13030, 2978, 3216, 12495, 
 305, 75, 76, 14915, 3212, 79, 1419, 1048, 17711, 83, 2859, 2768, 
 18679, 87, 554, 16851, 7174, 0, 7477, 6497, 6513, 6545, 7049, 
 0, 0, 0, 0, 13668, 6206, 0, 6333, 5665, 8323, 5126, 8613, 0, 
 7005, 13794, 0, 1307, 4686, 18203, 11004, 5035, 0, 0, 0, 8789, 
 16381, 5521, 0, 1639, 0, 0, 0, 0, 0, 1492, 152, 0, 114, 0, 
 1665, 0, 0, 0, 6811, 0, 0, 0, 16132, 17973, 0, 5277, 0, 0, 
 100, 10571, 13803, 6540, 10081, 0, 7795, 13241, 13247, 10535, 
 4597, 0, 0, 1505, 1947, 8505, 12013, 4606, 2874, 6564, 0, 626, 
 15244, 11038, 0, 18562, 3066, 0, 6564, 921, 12388, 2081, 16723, 
 9044, 12105, 3127, 0, 16085, 9143, 0, 0, 12963, 1404, 12739, 
 0, 0, 0, 0, 15633, 0, 0, 5429, 2642, 0, 0, 0, 1157, 0, 0, 
 0, 0, 0, 16499, 0, 10187, 16804, 0, 0, 8749, 0, 15886, 0, 
 0, 0, 0, 2554, 4020, 0, 4363, 0, 0, 5555, 0, 0, 16155, 0, 
 6741, 120, 8864, 0, 0, 0, 0, 3993, 15549, 0, 8866, 0, 18815, 
 8514, 0, 0, 4393, 0, 1478, 0, 4658, 8516, 6415, 0, 5162, 8513, 
 0, 2913, 0, 0, 0, 12499, 0, 0, 0, 7283, 2763, 2507, 6575, 
 0, 2750, 0, 5416, 14270, 0, 0, 4670, 7552, 0, 0, 17003, 7936, 
 3822, 15759, 10180, 1974, 431, 10521, 18594, 16127, 15889, 0, 
 2766, 17904, 6245, 2964, 4965, 1983, 9045, 0, 0, 7655, 0, 7955, 
 8905, 6464, 3912, 8312, 7603, 0, 0, 5312, 0, 15880, 1105, 0, 
 3883, 7656, 8745, 17730, 18157, 7654, 6286, 8420, 0, 0, 6995, 
 9073, 4497, 3721, 0, 11572, 2899, 2703, 4438, 0, 6676, 0, 0, 
 17017, 0, 0, 14101, 15560, 4368, 4056, 18397, 8042, 9758, 8100, 
 2034, 17678, 14775, 1357, 12765, 7803, 7718, 0, 14623, 0, 0, 
 0, 12004, 0, 11566, 0, 0, 0, 2480, 5788, 0, 6932, 0, 0, 10218, 
 0, 13758, 7647, 0, 0, 4722, 0, 0, 0, 9498, 17952, 0, 8462, 
 2181, 7583, 17934, 5993, 1716, 13380, 17588, 0, 0, 0, 3068, 
 8463, 0, 5654, 6796, 0, 1333, 0, 0, 0, 0, 7041, 0, 2392, 1630, 
 0, 0, 2866, 3769, 0, 0, 2744, 0, 1245, 8549, 2302, 12403, 16853, 
 0, 0, 0, 0, 4281, 0, 0, 10237, 9566, 0, 9315, 2243, 14886, 
 0, 8498, 15715, 17280, 9981, 8706, 5425, 4177, 2932, 10344, 2866, 
 0, 11335, 4179, 1251, 4181, 9249, 0, 8739, 16481, 0, 16848, 
 0, 14897, 16151, 7850, 13386, 0, 18144, 2177, 0, 0, 2298, 0, 
 0, 6039, 0, 8580, 0, 5803, 0, 8577, 7833, 9637, 0, 276, 0, 
 0, 7723, 9839, 7776, 16522, 18142, 9147, 0, 331, 0, 1406, 6690, 
 0, 4225, 18675, 0, 7397, 9056, 0, 0, 0, 6561, 0, 15880, 13119, 
 4655, 0, 17698, 0, 6284, 7349, 7714, 10004, 7345, 15574, 4995, 
 7376, 17497, 2872, 1280, 2935, 0, 2935, 5408, 0, 0, 10657, 0, 
 5183, 0, 13480, 8850, 0, 270, 167, 0, 0, 12748, 7352, 16071, 
 0, 15814, 4007, 10616, 0, 6699, 0, 3550, 4791, 8414, 4074, 7635, 
 0, 0, 0, 6054, 0, 2300, 18609, 0, 0, 10528, 9483, 0, 4335, 
 18114, 0, 10569, 0, 0, 0, 0, 16553, 3728, 0, 0, 13712, 13313, 
 0, 7940, 0, 18092, 0, 6058, 6877, 2021, 0, 0, 13186, 9669, 
 0, 0, 14595, 12266, 1404, 18311, 111, 17088, 0, 168, 0, 17374, 
 0, 0, 0, 8736, 1664, 4130, 13038, 8735, 4321, 8066, 9889, 17394, 
 2127, 9684, 18457, 0, 1318, 12208, 0, 18302, 5413, 9684, 15926, 
 0, 0, 8941, 1561, 11571, 9617, 0, 0, 11802, 1633, 0, 6811, 
 0, 6792, 0, 0, 12819, 0, 0, 15224, 0, 0, 0, 0, 1701, 0, 0, 
 146, 1543, 0, 0, 0, 0, 0, 18688, 0, 15794, 18792, 9963, 0, 
 5946, 3806, 15086, 0, 0, 8014, 9717, 6415, 0, 805, 16829, 9696, 
 18737, 967, 295, 0, 0, 463, 0, 3264, 15132, 10489, 7855, 1209, 
 8001, 14889, 11617, 0, 12942, 8905, 0, 18042, 19, 0, 9973, 13466, 
 9810, 8841, 12923, 0, 1831, 8840, 0, 16803, 0, 8843, 5183, 77, 
 0, 0, 0, 0, 12783, 0, 0, 0, 4712, 10218, 0, 11427, 11669, 
 0, 0, 4352, 0, 13084, 0, 189, 2944, 8829, 501, 0, 9567, 0, 
 2633, 0, 0, 0, 0, 0, 6347, 0, 3272, 0, 6909, 0, 3018, 13918, 
 6495, 15642, 1981, 0, 12509, 4546, 2162, 13283, 0, 2200, 230, 
 2008, 0, 8528, 4404, 8544, 0, 15140, 0, 0, 16899, 13871, 0, 
 5530, 8877, 3722, 0, 0, 4493, 0, 11956, 0, 3955, 0, 17177, 
 0, 0, 8652, 3459, 16946, 0, 0, 0, 0, 4335, 8990, 1810, 0, 
 0, 3740, 12889, 15285, 12468, 12270, 0, 4080, 4639, 12685, 9735, 
 2352, 0, 16196, 0, 6544, 18616, 10409, 0, 0, 1847, 0, 0, 0, 
 14193, 17629, 0, 16056, 5311, 0, 0, 0, 0, 1486, 0, 11533, 7219, 
 0, 0, 13816, 6225, 4314, 0, 11830, 9514, 814, 1538, 18619, 5343, 
 0, 0, 5821, 0, 7850, 17866, 6748, 0, 4705, 6612, 16350, 0, 
 0, 14654, 16143, 10747, 635, 0, 0, 0, 9978, 4934, 6959, 0, 
 0, 0, 0, 0, 0, 15243, 1879, 0, 0, 16618, 0, 0, 11976, 0, 
 0, 11914, 4937, 10060, 0, 0, 818, 9918, 0, 0, 0, 0, 1479, 
 8270, 1397, 0, 4209, 0, 0, 0, 10357, 6745, 10359, 10360, 10361, 
 4269, 7913, 10364, 9353, 17953, 10367, 10849, 1989, 0, 6205, 
 0, 5302, 13820, 7343, 2207, 15007, 10346, 16564, 5832, 10349, 
 10350, 10351, 5933, 2011, 10354, 10375, 10356, 7059, 6465, 16408, 
 2393, 653, 9022, 0, 0, 11712, 13635, 16081, 3133, 317, 6022, 
 11739, 5700, 2442, 15565, 110, 331, 14815, 7948, 0, 0, 4768, 
 1660, 0, 0, 6079, 8978, 1586, 0, 0, 7949, 15653, 6485, 0, 10185, 
 12433, 2862, 9580, 3458, 0, 16510, 0, 270, 0, 6744, 0, 395, 
 0, 1531, 0, 11565, 0, 16125, 5793, 0, 10485, 10408, 5201, 13816, 
 18142, 18266, 0, 0, 3022, 1886, 325, 0, 18827, 22, 11916, 16685, 
 1325, 0, 6316, 0, 0, 0, 0, 10026, 7316, 0, 0, 4838, 0, 1911, 
 5674, 3867, 0, 0, 0, 0, 7913, 1834, 1170, 1279, 1773, 4948, 
 7463, 12086, 15698, 11212, 0, 8805, 4569, 11226, 0, 4122, 16763, 
 0, 7835, 4126, 9553, 4627, 0, 3633, 0, 4123, 6297, 13251, 0, 
 0, 3209, 4124, 0, 7313, 709, 2917, 0, 1806, 9867, 12892, 8411, 
 0, 5537, 13751, 0, 6990, 7051, 14463, 0, 2078, 11793, 0, 0, 
 0, 0, 88, 12192, 0, 6313, 3456, 840, 10254, 947, 0, 8334, 3128, 
 2933, 17690, 9815, 0, 0, 5836, 4032, 0, 6587, 6343, 0, 0, 1049, 
 5922, 17998, 0, 1468, 10137, 9488, 6629, 0, 3561, 0, 14781, 
 0, 0, 13078, 4994, 1866, 4667, 9813, 9461, 0, 110, 0, 11605, 
 2868, 0, 0, 897, 0, 7787, 10283, 3046, 18257, 0, 5535, 12591, 
 2459, 0, 0, 0, 0, 0, 242, 4264, 3588, 6541, 6104, 3680, 0, 
 1637, 1050, 7002, 0, 0, 18054, 14082, 0, 8221, 358, 4435, 0, 
 4373, 12935, 0, 1239, 8223, 1828, 7955, 2131, 16437, 0, 0, 11468, 
 0, 13169, 0, 0, 0, 1807, 17430, 8453, 0, 0, 0, 16193, 4912, 
 12975, 519, 11943, 0, 0, 7882, 5182, 0, 3091, 6675, 3945, 0, 
 0, 28, 12625, 5099, 3944, 11218, 15752, 2917, 18153, 0, 2105, 
 8481, 6290, 0, 0, 0, 0, 4811, 14856, 0, 0, 0, 7884, 7447, 
 4431, 8838, 5678, 11896, 0, 0, 9475, 0, 8087, 3079, 7602, 6845, 
 8691, 3889, 3496, 0, 17274, 2352, 4875, 0, 13700, 6964, 0, 5557, 
 18048, 0, 15865, 0, 353, 0, 6263, 1453, 15989, 4583, 3678, 0, 
 8239, 2075, 18825, 0, 17598, 10161, 4379, 9723, 8820, 12671, 
 10481, 0, 0, 0, 0, 8324, 0, 0, 0, 0, 4851, 14264, 0, 12220, 
 17351, 17581, 11323, 0, 1035, 13330, 559, 0, 0, 9071, 6615, 
 0, 9424, 9072, 6826, 6503, 6720, 14265, 16234, 0, 0, 7427, 16584, 
 0, 5461, 0, 9755, 12278, 8236, 0, 0, 15139, 8343, 1132, 10454, 
 0, 6398, 0, 0, 15735, 0, 12011, 5571, 0, 12003, 6924, 10787, 
 901, 0, 656, 4225, 13925, 0, 0, 0, 0, 0, 0, 0, 5506, 12282, 
 9693, 1640, 0, 13291, 3851, 3505, 0, 5424, 15494, 14146, 0, 
 16017, 0, 0, 4709, 5956, 12083, 0, 2830, 4368, 10038, 0, 8034, 
 5626, 0, 0, 0, 0, 2135, 2665, 6380, 15442, 0, 0, 9899, 4505, 
 1533, 3334, 0, 8402, 8580, 8412, 8198, 0, 0, 15492, 4216, 8400, 
 0, 10762, 0, 0, 0, 0, 6640, 0, 0, 4616, 12376, 16011, 0, 8964, 
 9003, 4297, 0, 13221, 9004, 13499, 6132, 7823, 0, 0, 461, 8965, 
 12957, 0, 0, 1246, 0, 0, 7577, 10940, 7898, 5777, 15072, 11783, 
 1371, 0, 12138, 0, 13688, 0, 4562, 8513, 0, 6991, 0, 1007, 
 11063, 417, 6792, 0, 4837, 0, 7761, 458, 1808, 6145, 3161, 7915, 
 16138, 12107, 31, 18430, 0, 4547, 0, 6561, 6210, 3818, 6944, 
 852, 6665, 0, 10502, 0, 0, 0, 5777, 974, 0, 10174, 0, 16600, 
 7316, 1430, 0, 0, 15700, 0, 141, 18128, 777, 0, 3936, 18690, 
 1076, 0, 10715, 14019, 3180, 18149, 9850, 7970, 6244, 17641, 
 11961, 15495, 0, 9130, 0, 0, 10168, 5273, 0, 3558, 12635, 939, 
 7857, 14596, 15875, 8870, 12089, 8935, 12870, 0, 10427, 0, 4985, 
 11887, 4959, 10471, 0, 0, 0, 0, 0, 0, 0, 2365, 0, 2577, 1861, 
 4550, 5860, 0, 12564, 518, 10429, 4418, 3871, 3744, 3177, 358, 
 17731, 3115, 8500, 0, 3532, 6064, 3374, 3748, 17010, 5186, 0, 
 12800, 8492, 0, 3233, 17245, 0, 0, 16227, 8743, 14055, 0, 0, 
 0, 2136, 10240, 3256, 15193, 0, 5038, 5800, 5406, 0, 9124, 3254, 
 0, 0, 10380, 0, 2833, 1788, 12330, 13599, 7690, 0, 11762, 12476, 
 0, 16993, 7689, 9635, 8685, 0, 4449, 1525, 1036, 1722, 0, 0, 
 18639, 14889, 6853, 0, 0, 0, 4318, 0, 1262, 5685, 550, 9, 5489, 
 0, 2336, 15498, 16562, 13826, 8979, 16496, 18723, 4634, 8697, 
 899, 9998, 17414, 2295, 9353, 7708, 0, 2301, 8142, 0, 16550, 
 0, 16790, 0, 7648, 3992, 13222, 11302, 11600, 1527, 0, 0, 7649, 
 7505, 1520, 579, 0, 3493, 18512, 0, 13110, 0, 0, 0, 9007, 1008, 
 2392, 3219, 3293, 16399, 7830, 352, 405, 0, 0, 8068, 1523, 5124, 
 0, 2342, 0, 17696, 0, 1352, 17085, 238, 7481, 0, 0, 5662, 2349, 
 7275, 0, 0, 10246, 327, 4236, 15051, 5284, 7272, 0, 4824, 3245, 
 0, 7487, 0, 2402, 10548, 18583, 18277, 0, 5189, 15624, 0, 18462, 
 7461, 18633, 5205, 15146, 0, 12881, 0, 6119, 853, 0, 18589, 
 0, 4858, 0, 0, 16577, 893, 12273, 0, 6220, 16734, 0, 3324, 
 8086, 15005, 10012, 0, 0, 10312, 0, 711, 0, 3659, 17851, 0, 
 5025, 10016, 0, 3120, 7423, 2458, 6664, 6600, 1805, 6254, 5550, 
 4976, 0, 0, 0, 12761, 7557, 0, 12949, 5906, 5404, 0, 0, 0, 
 16326, 0, 15688, 0, 2751, 0, 7166, 0, 13431, 0, 692, 8141, 
 0, 7166, 12488, 5228, 14019, 0, 0, 18582, 8045, 1777, 977, 9789, 
 4359, 0, 16192, 8138, 9075, 1945, 0, 15763, 2903, 6313, 3352, 
 12834, 14778, 8815, 0, 11451, 14880, 0, 8719, 4587, 0, 3064, 
 16502, 0, 2393, 12576, 7425, 0, 2959, 0, 2316, 0, 0, 0, 2258, 
 1555, 13495, 1505, 6498, 15337, 0, 0, 7015, 17168, 7938, 0, 
 0, 3709, 4894, 17109, 9121, 0, 13613, 7676, 8526, 9809, 0, 7673, 
 0, 0, 7663, 0, 15099, 0, 1529, 0, 14220, 7888, 9597, 2850, 
 18188, 2539, 8155, 9813, 6096, 10387, 7848, 3350, 3599, 11442, 
 2608, 0, 621, 3460, 13890, 0, 0, 0, 7157, 0, 0, 0, 16752, 
 5667, 602, 0, 2159, 0, 2631, 6350, 168, 0, 15701, 1484, 16363, 
 0, 1177, 0, 0, 18088, 0, 15191, 0, 0, 0, 0, 2702, 3268, 0, 
 9581, 0, 6662, 14211, 3859, 6598, 0, 0, 13618, 0, 1362, 278, 
 0, 9970, 5903, 0, 0, 3891, 9810, 18756, 7542, 5177, 0, 2849, 
 13936, 0, 1279, 0, 3123, 2037, 0, 683, 610, 3846, 5811, 7562, 
 0, 9082, 11891, 10541, 1237, 0, 0, 10242, 0, 0, 15124, 0, 14610, 
 0, 0, 63, 8339, 5628, 5631, 11078, 17365, 35, 0, 14272, 14276, 
 6870, 10375, 16934, 18252, 43, 12312, 16003, 328, 55, 7850, 3645, 
 17965, 12323, 48, 15819, 3935, 10125, 0, 0, 2809, 1184, 15765, 
 6788, 15328, 6503, 6685, 2004, 4462, 0, 613, 6853, 0, 0, 0, 
 5765, 10390, 10389, 10392, 10391, 10394, 10393, 6422, 7564, 10398, 
 2085, 2987, 2534, 689, 14790, 0, 5237, 12938, 2997, 16244, 1206, 
 14247, 4082, 1092, 6310, 10382, 18487, 4040, 6961, 10386, 16400, 
 10388, 4253, 0, 0, 8632, 15653, 2997, 0, 12673, 11761, 7160, 
 5383, 0, 15809, 0, 17472, 8895, 6690, 4974, 870, 14445, 10571, 
 0, 1785, 17889, 0, 11035, 4350, 6276, 3372, 8211, 0, 14899, 
 14461, 565, 18571, 0, 17947, 0, 4352, 755, 15280, 0, 16217, 
 0, 0, 18361, 9741, 6881, 3675, 13988, 14596, 0, 0, 8878, 1653, 
 4383, 17771, 5721, 9123, 0, 0, 1998, 1760, 286, 0, 8877, 0, 
 17892, 14204, 0, 1195, 12679, 0, 0, 0, 0, 1947, 0, 0, 0, 3181, 
 5485, 0, 0, 6302, 5513, 9932, 9258, 6740, 0, 17758, 12726, 4075, 
 6902, 17370, 15868, 0, 399, 10153, 17656, 0, 1757, 1694, 0, 
 8559, 16746, 0, 0, 15308, 0, 0, 0, 8134, 8410, 8028, 0, 18149, 
 10525, 0, 2784, 7345, 0, 0, 9296, 3238, 15628, 0, 3093, 8053, 
 4906, 0, 5997, 10488, 5017, 0, 0, 10555, 0, 0, 12360, 10269, 
 12867, 12095, 17083, 0, 2059, 8570, 0, 0, 16678, 0, 1486, 15196, 
 1888, 8572, 0, 219, 810, 190, 14619, 16914, 5745, 14276, 0, 
 6029, 5235, 6927, 8551, 2991, 0, 706, 13257, 0, 14364, 0, 8567, 
 9003, 6289, 6603, 5090, 9854, 0, 7942, 6667, 1225, 9849, 1959, 
 18416, 0, 0, 0, 405, 0, 0, 11841, 4524, 0, 16548, 0, 1430, 
 18095, 0, 0, 17444, 0, 3998, 1560, 14942, 0, 2800, 2666, 1420, 
 5843, 8292, 13653, 4752, 8550, 3276, 14694, 1789, 0, 0, 7449, 
 16089, 16834, 17272, 15942, 3275, 13978, 0, 0, 9020, 0, 0, 0, 
 13891, 17375, 0, 17897, 0, 650, 0, 10108, 0, 17724, 15987, 0, 
 1200, 0, 0, 14111, 16821, 4497, 5161, 5920, 0, 10162, 14210, 
 0, 8720, 2888, 8715, 0, 1656, 14083, 12569, 0, 0, 5383, 0, 
 0, 8024, 13205, 0, 0, 0, 2638, 0, 0, 2815, 4965, 0, 0, 17984, 
 7468, 0, 10071, 16703, 0, 0, 14073, 0, 826, 2615, 0, 0, 9245, 
 8049, 1401, 4124, 0, 18289, 7298, 0, 17000, 8051, 0, 17065, 
 2123, 10304, 7297, 18632, 9987, 16572, 11481, 5860, 4689, 17781, 
 9435, 0, 7006, 4596, 0, 0, 10178, 11603, 149, 6422, 2030, 2797, 
 18693, 12865, 16010, 10492, 3268, 6986, 327, 17803, 0, 0, 3264, 
 0, 6989, 0, 18615, 0, 0, 0, 10737, 0, 13251, 0, 0, 0, 1734, 
 2943, 14523, 7318, 10897, 2799, 0, 12400, 10490, 4776, 0, 1547, 
 7009, 16545, 0, 0, 2783, 11093, 0, 51, 0, 5638, 7774, 0, 6626, 
 5241, 0, 0, 18136, 0, 8557, 133, 16206, 0, 15165, 7295, 0, 
 0, 0, 0, 0, 0, 6011, 9090, 11698, 9265, 0, 8239, 0, 0, 6330, 
 0, 12757, 4191, 0, 0, 6416, 0, 178, 0, 936, 0, 6579, 9630, 
 6762, 0, 6584, 1395, 6390, 0, 9018, 2454, 13153, 0, 0, 6328, 
 0, 11834, 0, 0, 15262, 9023, 17576, 0, 7457, 0, 0, 0, 0, 0, 
 3615, 0, 7181, 16796, 0, 0, 15560, 0, 4479, 2032, 4300, 133, 
 0, 18568, 0, 2860, 15965, 0, 11331, 6116, 0, 8772, 16945, 10398, 
 0, 0, 4477, 7335, 0, 15837, 9216, 9697, 0, 7506, 1992, 3606, 
 18245, 6321, 12208, 14653, 16575, 3750, 16546, 3426, 9130, 12755, 
 0, 4613, 1856, 515, 0, 434, 8802, 10477, 6834, 8029, 1246, 0, 
 0, 4165, 0, 12709, 3924, 11583, 18185, 0, 7451, 14304, 0, 0, 
 0, 4163, 8100, 4239, 0, 4661, 8266, 0, 0, 0, 16641, 13251, 
 3147, 0, 13367, 8276, 9966, 4161, 732, 0, 16745, 0, 5766, 15158, 
 0, 3787, 14533, 8275, 13129, 0, 18047, 9996, 0, 10471, 0, 16915, 
 8267, 0, 0, 18802, 0, 6341, 0, 0, 6386, 4318, 15583, 0, 13695, 
 0, 3667, 10029, 10152, 14477, 7914, 0, 0, 0, 3285, 6270, 3288, 
 0, 946, 9497, 11657, 11896, 0, 0, 3537, 15061, 1590, 7627, 0, 
 11151, 6098, 18720, 10323, 0, 17093, 3005, 9217, 9218, 5432, 
 18233, 9221, 9222, 4474, 0, 9117, 5627, 6002, 0, 0, 1041, 0, 
 0, 8432, 7864, 8603, 0, 0, 7504, 4200, 3764, 16843, 0, 8601, 
 9297, 2969, 0, 6031, 0, 3086, 11824, 257, 8656, 6014, 15328, 
 279, 5149, 0, 0, 9621, 3204, 0, 6189, 11796, 1012, 0, 0, 18397, 
 7567, 0, 0, 4685, 9887, 0, 0, 17436, 5082, 0, 2560, 12696, 
 0, 7286, 6174, 0, 5386, 0, 10466, 0, 0, 0, 16386, 4177, 0, 
 13451, 9040, 13465, 0, 5548, 3835, 0, 13633, 0, 848, 14764, 
 0, 0, 1163, 502, 17270, 0, 2561, 14575, 605, 11991, 10456, 2819, 
 1711, 5887, 0, 4488, 6390, 406, 10682, 9911, 10171, 13174, 2047, 
 18318, 0, 1497, 9734, 0, 0, 4491, 10455, 5605, 1059, 0, 9542, 
 10827, 4344, 8653, 12314, 0, 6750, 0, 0, 5635, 11324, 0, 18062, 
 3269, 18434, 7605, 7455, 14988, 8009, 695, 0, 13037, 0, 1240, 
 14049, 0, 8002, 0, 919, 13134, 0, 4586, 18766, 4381, 0, 0, 
 803, 10952, 17244, 3526, 0, 6076, 8026, 18759, 13280, 6895, 15937, 
 6019, 0, 0, 0, 2167, 0, 0, 3505, 0, 0, 14164, 3063, 0, 0, 
 0, 15314, 0, 6005, 615, 2225, 0, 0, 0, 0, 8313, 223, 9758, 
 2885, 3392, 15257, 0, 4400, 10210, 573, 4737, 15505, 12559, 14268, 
 0, 1860, 1516, 0, 608, 998, 8450, 17715, 14880, 7728, 5464, 
 1750, 5966, 0, 14581, 0, 0, 6924, 3062, 2923, 2601, 15643, 9, 
 9295, 15573, 0, 7035, 0, 0, 0, 8413, 8225, 6536, 0, 2802, 0, 
 14493, 324, 13233, 4226, 0, 0, 0, 8248, 0, 1385, 0, 0, 2872, 
 12161, 0, 10545, 12622, 0, 0, 1129, 0, 16303, 18422, 13931, 
 17221, 7566, 8956, 11991, 0, 8183, 2692, 17166, 0, 0, 10083, 
 13570, 14788, 0, 1905, 6067, 0, 4004, 11869, 0, 699, 0, 6681, 
 4892, 2904, 1086, 10392, 0, 11036, 11217, 811, 7895, 0, 154, 
 13392, 0, 2385, 0, 13628, 9989, 16296, 5650, 256, 13442, 7011, 
 8690, 3316, 5116, 0, 2171, 3510, 8728, 0, 16240, 0, 10332, 13509, 
 0, 0, 3880, 5972, 4570, 0, 0, 13538, 6559, 84, 0, 0, 3883, 
 225, 0, 6623, 7505, 0, 2646, 0, 0, 0, 0, 1807, 8608, 2400, 
 3119, 969, 6269, 4497, 0, 0, 8285, 0, 16350, 4337, 1455, 10932, 
 17791, 9467, 0, 0, 0, 0, 0, 7453, 0, 0, 0, 0, 0, 0, 2151, 
 3731, 3074, 8171, 0, 3961, 0, 0, 0, 8312, 381, 8173, 0, 2907, 
 10896, 8170, 0, 0, 0, 18124, 6163, 0, 0, 0, 0, 9580, 2978, 
 5412, 3209, 5929, 5362, 10686, 12978, 10154, 6238, 2045, 0, 0, 
 12175, 0, 17832, 0, 0, 2861, 0, 135, 0, 2751, 4804, 5928, 4993, 
 10645, 5925, 5248, 5927, 16481, 6349, 14665, 9758, 5928, 11928, 
 0, 6336, 657, 0, 2842, 4054, 0, 9107, 7418, 0, 16348, 3867, 
 11666, 5939, 5938, 5933, 18483, 5935, 3861, 9923, 5741, 5653, 
 0, 9919, 15886, 9993, 2132, 0, 0, 0, 860, 10202, 0, 0, 0, 
 13060, 0, 0, 12720, 18512, 1136, 0, 0, 9106, 0, 5782, 0, 0, 
 4992, 0, 3253, 0, 1135, 9846, 0, 0, 14563, 7434, 195, 4713, 
 0, 13586, 3381, 0, 0, 18729, 16059, 9778, 12437, 0, 3652, 0, 
 16021, 0, 6012, 0, 11735, 0, 7659, 2490, 7662, 7445, 9473, 10415, 
 14217, 0, 7658, 2257, 17857, 15392, 382, 10450, 369, 0, 0, 0, 
 2057, 0, 12429, 9105, 6342, 0, 0, 16244, 0, 0, 7660, 4684, 
 0, 12394, 7010, 17416, 6611, 12399, 0, 7270, 0, 0, 6223, 16865, 
 0, 7266, 2689, 632, 9685, 16269, 11271, 8808, 9649, 0, 1018, 
 0, 0, 7400, 12326, 9319, 0, 9817, 0, 9512, 6012, 882, 485, 
 0, 0, 393, 7450, 10804, 0, 4866, 12749, 4868, 2792, 3433, 15738, 
 16459, 3940, 0, 2183, 0, 12948, 9939, 0, 2342, 18403, 18407, 
 9154, 5702, 0, 2391, 1469, 16020, 0, 0, 11263, 0, 16986, 1261, 
 7584, 8037, 0, 17108, 9271, 5783, 0, 0, 0, 0, 10615, 12216, 
 4788, 0, 1403, 17588, 0, 12064, 0, 1788, 0, 0, 0, 8087, 10669, 
 1259, 1482, 14022, 7192, 929, 0, 18313, 7186, 0, 752, 8644, 
 18672, 0, 3465, 0, 16309, 0, 16061, 0, 869, 0, 5413, 0, 12662, 
 3277, 5463, 0, 0, 0, 2606, 0, 16243, 0, 0, 7642, 0, 7696, 
 0, 0, 0, 0, 0, 15294, 2006, 8170, 0, 160, 5102, 0, 6682, 11418, 
 950, 0, 0, 13279, 5609, 0, 4010, 3425, 0, 6384, 0, 2999, 0, 
 9268, 0, 3413, 0, 0, 17830, 5267, 4219, 6101, 8987, 15720, 0, 
 16037, 311, 0, 4259, 0, 10961, 0, 2967, 5109, 4792, 14728, 0, 
 0, 17181, 0, 13748, 0, 10101, 0, 0, 6944, 0, 1212, 0, 17416, 
 16899, 12709, 16952, 1154, 0, 8658, 0, 0, 0, 7960, 18301, 0, 
 9942, 4736, 0, 17209, 5040, 12110, 0, 8834, 0, 3863, 1719, 2693, 
 10648, 0, 15202, 10073, 2416, 0, 13341, 6276, 0, 1811, 214, 
 10535, 0, 0, 10250, 0, 9005, 2418, 0, 1152, 13058, 0, 0, 0, 
 0, 0, 10323, 3340, 2200, 10109, 155, 15213, 4128, 9820, 0, 11443, 
 15210, 16593, 0, 3393, 0, 13724, 9009, 3344, 1195, 4727, 0, 
 3570, 6069, 2895, 17248, 4705, 1973, 16382, 0, 0, 7351, 0, 4129, 
 16844, 5573, 8453, 0, 7474, 17137, 1349, 0, 9935, 0, 0, 11966, 
 0, 0, 9518, 0, 18077, 9788, 0, 9268, 3565, 355, 5799, 0, 0, 
 0, 0, 0, 0, 4604, 6732, 0, 17990, 134, 6145, 0, 17211, 0, 
 0, 1207, 16370, 5551, 11670, 1868, 1952, 0, 17682, 2549, 13386, 
 0, 5112, 0, 3376, 10146, 0, 0, 0, 7178, 0, 5760, 0, 0, 15055, 
 9511, 6496, 10192, 3602, 16158, 7737, 7176, 7349, 6890, 0, 5900, 
 10941, 13702, 0, 14122, 0, 0, 0, 4373, 2132, 6570, 9872, 0, 
 2671, 0, 0, 0, 4708, 0, 0, 0, 9326, 2072, 85, 0, 903, 0, 
 1625, 0, 3440, 9678, 9789, 0, 0, 0, 0, 0, 3775, 165, 18351, 
 11023, 0, 11429, 9925, 0, 16078, 0, 16594, 11250, 2136, 377, 
 6102, 0, 0, 318, 5520, 0, 14280, 6301, 17582, 17577, 0, 14956, 
 0, 0, 763, 0, 0, 0, 0, 0, 0, 5246, 3354, 0, 7953, 16267, 
 0, 0, 0, 13091, 7524, 0, 9092, 0, 2231, 0, 9089, 0, 0, 2162, 
 15273, 0, 18342, 0, 16252, 18118, 17333, 0, 6345, 359, 14812, 
 0, 14412, 5859, 8938, 0, 0, 4767, 2503, 0, 0, 18006, 0, 0, 
 0, 0, 1320, 14573, 0, 0, 2505, 0, 0, 0, 12186, 0, 2512, 0, 
 8231, 0, 367, 0, 1090, 3045, 0, 13377, 2510, 0, 4570, 658, 
 5454, 9528, 8380, 17155, 6521, 3770, 75, 8593, 6231, 503, 1850, 
 0, 5388, 0, 0, 0, 7324, 6222, 7606, 4891, 5065, 12253, 7133, 
 0, 11457, 6293, 0, 6650, 0, 18281, 13953, 10001, 5426, 0, 6228, 
 5016, 0, 0, 5559, 0, 14172, 0, 12143, 0, 5468, 0, 2718, 6174, 
 0, 5876, 0, 0, 12687, 753, 0, 6128, 5037, 0, 13237, 0, 0, 
 0, 0, 0, 8936, 0, 12249, 0, 11736, 0, 10875, 0, 0, 4998, 5756, 
 2869, 0, 0, 9890, 6187, 0, 0, 2575, 5801, 3271, 7775, 222, 
 244, 4109, 18080, 4105, 17708, 1990, 7199, 0, 0, 15527, 7197, 
 0, 16747, 10064, 0, 8318, 10087, 9697, 0, 17642, 0, 0, 7198, 
 986, 6316, 0, 0, 176, 5556, 0, 0, 0, 0, 0, 0, 15342, 3676, 
 0, 369, 16449, 6836, 2682, 2702, 16029, 14134, 18305, 0, 18812, 
 7139, 17741, 2093, 0, 3603, 15745, 0, 2585, 2578, 0, 10760, 
 83, 0, 0, 2700, 0, 10333, 6939, 0, 0, 0, 13169, 10863, 4241, 
 7607, 8740, 0, 17593, 8109, 17890, 468, 14459, 17161, 0, 0, 
 0, 16640, 666, 0, 0, 3511, 14201, 0, 14442, 2655, 10320, 0, 
 8629, 0, 0, 0, 0, 12998, 8068, 4095, 3641, 0, 0, 6786, 0, 
 0, 2903, 16540, 0, 0, 17309, 0, 12551, 0, 2190, 0, 4296, 4690, 
 7010, 0, 18534, 3584, 0, 6868, 14191, 3206, 3021, 3258, 0, 6647, 
 0, 5555, 5753, 15446, 0, 6977, 0, 17201, 12919, 4999, 0, 1041, 
 3493, 0, 2938, 0, 6113, 8892, 0, 5209, 13959, 5355, 0, 0, 14356, 
 14272, 2888, 10597, 16930, 0, 0, 10046, 2506, 6344, 578, 1428, 
 17317, 0, 1342, 1233, 0, 12754, 9034, 13907, 0, 10339, 0, 8006, 
 5568, 3037, 0, 7738, 1363, 0, 0, 17849, 0, 0, 16595, 0, 1503, 
 17198, 4537, 0, 0, 11995, 0, 0, 8450, 7016, 7221, 812, 2800, 
 3778, 16026, 0, 0, 0, 2960, 0, 0, 5349, 5796, 5229, 6734, 0, 
 297, 4172, 0, 13380, 2419, 0, 6966, 0, 0, 6602, 5848, 0, 0, 
 17409, 9010, 0, 1346, 0, 9543, 6734, 7843, 8227, 0, 0, 0, 3152, 
 14380, 0, 16254, 8268, 4276, 16701, 905, 10258, 596, 2935, 4092, 
 3991, 2217, 8508, 8231, 10890, 6486, 594, 8923, 8613, 13938, 
 2809, 5329, 9360, 4188, 0, 0, 0, 1720, 16306, 1533, 9352, 9353, 
 9354, 8865, 17764, 12480, 6877, 6318, 3503, 9261, 9905, 0, 0, 
 3142, 0, 0, 10196, 16770, 0, 0, 0, 8953, 0, 15875, 0, 10335, 
 0, 0, 6069, 11894, 0, 18347, 0, 11256, 1544, 0, 0, 11061, 1547, 
 1890, 3100, 4792, 0, 7973, 4478, 0, 210, 6427, 15191, 1373, 
 0, 0, 9522, 0, 0, 0, 10312, 18399, 0, 0, 0, 14868, 16727, 
 8135, 16971, 10049, 7288, 0, 16327, 0, 0, 0, 0, 8088, 0, 0, 
 1139, 3664, 47, 0, 397, 737, 1568, 5546, 0, 16231, 4167, 16758, 
 16120, 2012, 5360, 9328, 713, 10211, 8064, 0, 8723, 0, 0, 323, 
 0, 8105, 9041, 0, 0, 4266, 8768, 8548, 0, 0, 6766, 9127, 0, 
 5381, 7989, 0, 0, 0, 0, 0, 7993, 9634, 7991, 7889, 0, 688, 
 3356, 0, 930, 7247, 6160, 1786, 9616, 0, 6165, 3096, 11149, 
 7245, 16768, 2250, 2913, 14595, 0, 0, 0, 0, 8872, 5951, 0, 
 9468, 3661, 0, 11722, 0, 12791, 101, 2476, 0, 5659, 3647, 0, 
 0, 1880, 0, 9206, 9205, 9204, 1784, 208, 14863, 4859, 0, 16526, 
 0, 145, 0, 6989, 6704, 16156, 9198, 7320, 9196, 6411, 10757, 
 5287, 4241, 17410, 1293, 0, 14137, 0, 3719, 0, 6372, 18760, 
 2477, 5360, 0, 6120, 0, 16781, 2229, 0, 10235, 0, 3433, 6304, 
 8530, 6565, 10851, 0, 6074, 6864, 4050, 6721, 0, 0, 0, 16347, 
 18265, 1619, 0, 6322, 0, 6173, 1924, 5423, 5365, 11967, 7514, 
 0, 15129, 460, 0, 736, 0, 81, 9943, 0, 14084, 2811, 12669, 
 0, 496, 8142, 4034, 6689, 0, 559, 2379, 3300, 14733, 0, 1695, 
 558, 5471, 0, 3494, 0, 9689, 0, 0, 0, 0, 7783, 0, 0, 13352, 
 11299, 9800, 0, 16299, 15883, 0, 10292, 0, 0, 6095, 0, 5303, 
 6228, 0, 14468, 10222, 18505, 0, 196, 16853, 0, 0, 12841, 0, 
 2536, 0, 17965, 0, 0, 0, 11001, 0, 5857, 9817, 298, 14629, 
 10409, 8763, 0, 9621, 293, 0, 16871, 9879, 9692, 10343, 5221, 
 17237, 18188, 868, 0, 11576, 7965, 8256, 15826, 8270, 0, 0, 
 0, 8268, 10405, 7093, 0, 1686, 0, 8273, 10012, 0, 0, 4877, 
 10063, 18249, 0, 0, 0, 4201, 0, 7969, 0, 0, 0, 7966, 11678, 
 17501, 0, 7967, 3735, 0, 5804, 0, 3725, 11343, 0, 16522, 3727, 
 11719, 3731, 13043, 10502, 2900, 0, 1077, 8262, 0, 12370, 316, 
 18045, 0, 1916, 0, 1529, 0, 4842, 5472, 14615, 0, 0, 4916, 
 14758, 0, 0, 10305, 0, 3829, 15021, 0, 0, 15518, 15603, 9722, 
 4489, 0, 1915, 17308, 0, 0, 2813, 7672, 7498, 0, 12115, 2742, 
 0, 0, 2863, 3663, 3224, 0, 18537, 4339, 2033, 0, 17644, 3053, 
 6725, 4747, 0, 2439, 17916, 0, 13122, 2873, 10025, 0, 0, 7041, 
 0, 0, 0, 17114, 0, 16257, 17074, 11916, 12858, 7718, 15952, 
 0, 0, 0, 10408, 4474, 16469, 11821, 0, 5473, 0, 4453, 8288, 
 14245, 0, 3554, 0, 1273, 9759, 0, 5790, 0, 0, 0, 5788, 2637, 
 0, 10045, 2073, 6738, 0, 4394, 2294, 0, 0, 3100, 16015, 0, 
 0, 2780, 0, 0, 6753, 5167, 3847, 2997, 6901, 0, 3866, 5822, 
 16653, 1116, 0, 1661, 0, 0, 2538, 16170, 0, 16370, 18660, 7250, 
 2247, 0, 0, 0, 18186, 2875, 0, 14304, 0, 4379, 5630, 13138, 
 5858, 3541, 0, 7692, 15290, 2788, 0, 8234, 0, 5125, 12653, 0, 
 7917, 0, 17308, 7248, 0, 0, 0, 0, 15024, 0, 16517, 0, 12339, 
 175, 8743, 0, 11198, 15224, 11426, 18246, 1671, 2283, 18546, 
 4734, 0, 15787, 0, 17008, 9617, 5906, 0, 3048, 18470, 0, 0, 
 0, 0, 0, 0, 12011, 0, 0, 5789, 0, 0, 0, 0, 9791, 846, 11403, 
 912, 0, 199, 0, 0, 0, 17210, 0, 9438, 0, 0, 0, 1406, 0, 0, 
 15563, 2283, 17429, 636, 10597, 2366, 11338, 0, 8037, 9641, 14822, 
 983, 1634, 0, 0, 9859, 15751, 8476, 0, 521, 0, 0, 0, 0, 2275, 
 6987, 2129, 0, 7568, 0, 0, 0, 0, 9993, 6221, 10422, 3056, 0, 
 7549, 0, 14025, 7534, 1970, 1223, 9138, 0, 0, 10421, 0, 0, 
 4612, 17039, 5640, 0, 8960, 11127, 0, 971, 10536, 892, 14304, 
 0, 4046, 0, 0, 7673, 0, 0, 13853, 15189, 6169, 2596, 0, 0, 
 8665, 0, 2693, 14418, 61, 0, 2703, 0, 0, 0, 0, 3526, 5801, 
 0, 2571, 1950, 1487, 2235, 9004, 10478, 9080, 3525, 18604, 3523, 
 15182, 5276, 4079, 3521, 0, 5247, 6701, 17832, 9674, 0, 4456, 
 11885, 15269, 4305, 17896, 0, 17488, 501, 0, 0, 8922, 2331, 
 10501, 10074, 2424, 3057, 0, 11470, 1467, 13237, 9162, 0, 13445, 
 12531, 0, 0, 7188, 3287, 14588, 0, 5269, 0, 0, 0, 129, 7468, 
 2625, 8522, 0, 2098, 8235, 4847, 16061, 4237, 254, 3694, 0, 
 8194, 2435, 0, 13663, 7741, 7783, 0, 0, 10892, 1908, 3641, 0, 
 3584, 0, 0, 0, 0, 2438, 5564, 739, 0, 344, 8622, 5272, 13344, 
 0, 0, 0, 0, 7381, 9911, 6271, 3399, 2063, 0, 5156, 4723, 18271, 
 15322, 0, 5089, 9174, 14543, 15540, 12724, 10326, 0, 0, 1899, 
 0, 5521, 7822, 2760, 0, 7670, 7847, 7841, 1557, 0, 4176, 13950, 
 12399, 4449, 13622, 0, 0, 5619, 3956, 4052, 0, 0, 0, 15893, 
 2865, 25, 3042, 15324, 9401, 18768, 1151, 3085, 17819, 0, 0, 
 375, 23, 15183, 44, 6267, 0, 5556, 0, 0, 0, 6050, 716, 0, 
 2522, 140, 0, 0, 12191, 5685, 12433, 8892, 9882, 0, 0, 0, 0, 
 7524, 1548, 11624, 17335, 0, 0, 11491, 11674, 5575, 11792, 1727, 
 0, 6017, 13634, 18088, 7294, 10778, 9613, 5707, 16231, 0, 3554, 
 0, 0, 0, 10475, 7584, 10129, 554, 9347, 15835, 0, 0, 1701, 
 4541, 12264, 11092, 6396, 6607, 2549, 4457, 9159, 2412, 10135, 
 1316, 9163, 3343, 0, 154, 17256, 11861, 0, 10210, 11021, 8921, 
 18281, 1201, 0, 8293, 0, 205, 6985, 16805, 0, 14930, 6045, 3634, 
 0, 2084, 10365, 16240, 5708, 3886, 0, 0, 3933, 0, 5562, 10128, 
 0, 12896, 0, 0, 0, 16661, 3461, 2336, 10321, 0, 8014, 0, 0, 
 0, 8263, 4321, 0, 0, 14151, 0, 0, 10504, 0, 14387, 6928, 0, 
 7804, 4683, 12341, 0, 1798, 8939, 0, 0, 0, 10690, 4766, 14129, 
 0, 14590, 0, 0, 0, 10082, 5274, 4695, 0, 0, 6746, 0, 16241, 
 0, 0, 11238, 4236, 3651, 0, 1987, 0, 2669, 10718, 13354, 3595, 
 6178, 1096, 499, 7779, 15089, 3430, 0, 15576, 7140, 0, 0, 5689, 
 0, 12682, 0, 1689, 17294, 7899, 7015, 1830, 1582, 6460, 0, 0, 
 5489, 17847, 17030, 12486, 0, 2956, 9848, 14512, 8805, 3431, 
 0, 9743, 3452, 9040, 98, 2957, 6495, 2851, 14361, 14948, 0, 
 0, 8387, 1416, 4338, 0, 8962, 0, 0, 1224, 0, 3683, 0, 682, 
 1362, 12319, 12095, 9161, 981, 11079, 9219, 9858, 7928, 0, 13575, 
 17459, 8832, 15314, 5592, 0, 5815, 0, 0, 18472, 7192, 10764, 
 18268, 1258, 5002, 591, 15123, 18480, 3914, 5470, 15308, 3556, 
 17074, 0, 4865, 17649, 9070, 3482, 12379, 175, 5401, 2427, 11130, 
 0, 0, 9496, 10826, 8646, 6421, 12664, 3061, 11663, 9499, 7679, 
 3234, 14102, 7177, 0, 3236, 12571, 160, 0, 10191, 4263, 2754, 
 14512, 2723, 0, 9626, 0, 0, 0, 3351, 6053, 9851, 9476, 17425, 
 9826, 0, 0, 0, 8730, 0, 0, 0, 13112, 10080, 3797, 0, 3215, 
 8287, 0, 6626, 8297, 0, 12316, 0, 8073, 1871, 9701, 5884, 1312, 
 7099, 2989, 12006, 0, 2776, 4882, 0, 0, 18458, 58, 0, 0, 13541, 
 6851, 5258, 4080, 0, 0, 6758, 14501, 17210, 7115, 0, 6788, 4069, 
 11421, 2215, 0, 7683, 5806, 8542, 0, 0, 18277, 8545, 6683, 8202, 
 2947, 3364, 0, 0, 714, 0, 0, 7650, 2342, 16853, 1706, 15731, 
 0, 6444, 0, 2618, 0, 507, 7722, 0, 5034, 117, 0, 8359, 2355, 
 0, 0, 10824, 278, 708, 4079, 8361, 16722, 2479, 3779, 14425, 
 4009, 4464, 12901, 0, 4083, 0, 4353, 7904, 0, 2501, 2135, 0, 
 0, 9831, 16271, 9023, 0, 15233, 16237, 3822, 0, 0, 0, 597, 
 0, 0, 2955, 6355, 6097, 0, 8865, 2350, 7845, 0, 0, 13520, 4005, 
 0, 0, 2221, 2847, 0, 0, 4071, 0, 4362, 7685, 9337, 0, 17628, 
 6314, 7547, 3361, 0, 6167, 2040, 0, 2313, 3448, 0, 15069, 10156, 
 0, 0, 0, 0, 1150, 13167, 16624, 18084, 0, 4372, 4621, 17871, 
 0, 16958, 3868, 0, 4858, 0, 16481, 4411, 7448, 15863, 1717, 
 5214, 9046, 0, 8833, 13289, 725, 8106, 0, 0, 300, 0, 9350, 
 12732, 5190, 0, 172, 0, 14712, 2770, 4641, 0, 0, 0, 5192, 16771, 
 18503, 6910, 2899, 18193, 1962, 18195, 4235, 906, 13343, 13318, 
 3314, 6645, 2863, 5902, 18209, 5566, 5907, 8746, 12980, 17115, 
 13915, 3601, 12220, 7400, 5182, 6580, 15735, 0, 10723, 11107, 
 8509, 0, 2910, 4901, 2914, 13364, 0, 1338, 0, 12322, 0, 811, 
 127, 16138, 0, 0, 18792, 0, 15749, 8316, };

  
static const _Py_UnicodeCharacterName aucn[] = 
{
  { "SPACE", 0x20 },
  { "EXCLAMATION MARK", 0x21 },
  { "QUOTATION MARK", 0x22 },
  { "NUMBER SIGN", 0x23 },
  { "DOLLAR SIGN", 0x24 },
  { "PERCENT SIGN", 0x25 },
  { "AMPERSAND", 0x26 },
  { "APOSTROPHE", 0x27 },
  { "LEFT PARENTHESIS", 0x28 },
  { "RIGHT PARENTHESIS", 0x29 },
  { "ASTERISK", 0x2a },
  { "PLUS SIGN", 0x2b },
  { "COMMA", 0x2c },
  { "HYPHEN-MINUS", 0x2d },
  { "FULL STOP", 0x2e },
  { "SOLIDUS", 0x2f },
  { "DIGIT ZERO", 0x30 },
  { "DIGIT ONE", 0x31 },
  { "DIGIT TWO", 0x32 },
  { "DIGIT THREE", 0x33 },
  { "DIGIT FOUR", 0x34 },
  { "DIGIT FIVE", 0x35 },
  { "DIGIT SIX", 0x36 },
  { "DIGIT SEVEN", 0x37 },
  { "DIGIT EIGHT", 0x38 },
  { "DIGIT NINE", 0x39 },
  { "COLON", 0x3a },
  { "SEMICOLON", 0x3b },
  { "LESS-THAN SIGN", 0x3c },
  { "EQUALS SIGN", 0x3d },
  { "GREATER-THAN SIGN", 0x3e },
  { "QUESTION MARK", 0x3f },
  { "COMMERCIAL AT", 0x40 },
  { "LATIN CAPITAL LETTER A", 0x41 },
  { "LATIN CAPITAL LETTER B", 0x42 },
  { "LATIN CAPITAL LETTER C", 0x43 },
  { "LATIN CAPITAL LETTER D", 0x44 },
  { "LATIN CAPITAL LETTER E", 0x45 },
  { "LATIN CAPITAL LETTER F", 0x46 },
  { "LATIN CAPITAL LETTER G", 0x47 },
  { "LATIN CAPITAL LETTER H", 0x48 },
  { "LATIN CAPITAL LETTER I", 0x49 },
  { "LATIN CAPITAL LETTER J", 0x4a },
  { "LATIN CAPITAL LETTER K", 0x4b },
  { "LATIN CAPITAL LETTER L", 0x4c },
  { "LATIN CAPITAL LETTER M", 0x4d },
  { "LATIN CAPITAL LETTER N", 0x4e },
  { "LATIN CAPITAL LETTER O", 0x4f },
  { "LATIN CAPITAL LETTER P", 0x50 },
  { "LATIN CAPITAL LETTER Q", 0x51 },
  { "LATIN CAPITAL LETTER R", 0x52 },
  { "LATIN CAPITAL LETTER S", 0x53 },
  { "LATIN CAPITAL LETTER T", 0x54 },
  { "LATIN CAPITAL LETTER U", 0x55 },
  { "LATIN CAPITAL LETTER V", 0x56 },
  { "LATIN CAPITAL LETTER W", 0x57 },
  { "LATIN CAPITAL LETTER X", 0x58 },
  { "LATIN CAPITAL LETTER Y", 0x59 },
  { "LATIN CAPITAL LETTER Z", 0x5a },
  { "LEFT SQUARE BRACKET", 0x5b },
  { "REVERSE SOLIDUS", 0x5c },
  { "RIGHT SQUARE BRACKET", 0x5d },
  { "CIRCUMFLEX ACCENT", 0x5e },
  { "LOW LINE", 0x5f },
  { "GRAVE ACCENT", 0x60 },
  { "LATIN SMALL LETTER A", 0x61 },
  { "LATIN SMALL LETTER B", 0x62 },
  { "LATIN SMALL LETTER C", 0x63 },
  { "LATIN SMALL LETTER D", 0x64 },
  { "LATIN SMALL LETTER E", 0x65 },
  { "LATIN SMALL LETTER F", 0x66 },
  { "LATIN SMALL LETTER G", 0x67 },
  { "LATIN SMALL LETTER H", 0x68 },
  { "LATIN SMALL LETTER I", 0x69 },
  { "LATIN SMALL LETTER J", 0x6a },
  { "LATIN SMALL LETTER K", 0x6b },
  { "LATIN SMALL LETTER L", 0x6c },
  { "LATIN SMALL LETTER M", 0x6d },
  { "LATIN SMALL LETTER N", 0x6e },
  { "LATIN SMALL LETTER O", 0x6f },
  { "LATIN SMALL LETTER P", 0x70 },
  { "LATIN SMALL LETTER Q", 0x71 },
  { "LATIN SMALL LETTER R", 0x72 },
  { "LATIN SMALL LETTER S", 0x73 },
  { "LATIN SMALL LETTER T", 0x74 },
  { "LATIN SMALL LETTER U", 0x75 },
  { "LATIN SMALL LETTER V", 0x76 },
  { "LATIN SMALL LETTER W", 0x77 },
  { "LATIN SMALL LETTER X", 0x78 },
  { "LATIN SMALL LETTER Y", 0x79 },
  { "LATIN SMALL LETTER Z", 0x7a },
  { "LEFT CURLY BRACKET", 0x7b },
  { "VERTICAL LINE", 0x7c },
  { "RIGHT CURLY BRACKET", 0x7d },
  { "TILDE", 0x7e },
  { "NO-BREAK SPACE", 0xa0 },
  { "INVERTED EXCLAMATION MARK", 0xa1 },
  { "CENT SIGN", 0xa2 },
  { "POUND SIGN", 0xa3 },
  { "CURRENCY SIGN", 0xa4 },
  { "YEN SIGN", 0xa5 },
  { "BROKEN BAR", 0xa6 },
  { "SECTION SIGN", 0xa7 },
  { "DIAERESIS", 0xa8 },
  { "COPYRIGHT SIGN", 0xa9 },
  { "FEMININE ORDINAL INDICATOR", 0xaa },
  { "LEFT-POINTING DOUBLE ANGLE QUOTATION MARK", 0xab },
  { "NOT SIGN", 0xac },
  { "SOFT HYPHEN", 0xad },
  { "REGISTERED SIGN", 0xae },
  { "MACRON", 0xaf },
  { "DEGREE SIGN", 0xb0 },
  { "PLUS-MINUS SIGN", 0xb1 },
  { "SUPERSCRIPT TWO", 0xb2 },
  { "SUPERSCRIPT THREE", 0xb3 },
  { "ACUTE ACCENT", 0xb4 },
  { "MICRO SIGN", 0xb5 },
  { "PILCROW SIGN", 0xb6 },
  { "MIDDLE DOT", 0xb7 },
  { "CEDILLA", 0xb8 },
  { "SUPERSCRIPT ONE", 0xb9 },
  { "MASCULINE ORDINAL INDICATOR", 0xba },
  { "RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK", 0xbb },
  { "VULGAR FRACTION ONE QUARTER", 0xbc },
  { "VULGAR FRACTION ONE HALF", 0xbd },
  { "VULGAR FRACTION THREE QUARTERS", 0xbe },
  { "INVERTED QUESTION MARK", 0xbf },
  { "LATIN CAPITAL LETTER A WITH GRAVE", 0xc0 },
  { "LATIN CAPITAL LETTER A WITH ACUTE", 0xc1 },
  { "LATIN CAPITAL LETTER A WITH CIRCUMFLEX", 0xc2 },
  { "LATIN CAPITAL LETTER A WITH TILDE", 0xc3 },
  { "LATIN CAPITAL LETTER A WITH DIAERESIS", 0xc4 },
  { "LATIN CAPITAL LETTER A WITH RING ABOVE", 0xc5 },
  { "LATIN CAPITAL LETTER AE", 0xc6 },
  { "LATIN CAPITAL LETTER C WITH CEDILLA", 0xc7 },
  { "LATIN CAPITAL LETTER E WITH GRAVE", 0xc8 },
  { "LATIN CAPITAL LETTER E WITH ACUTE", 0xc9 },
  { "LATIN CAPITAL LETTER E WITH CIRCUMFLEX", 0xca },
  { "LATIN CAPITAL LETTER E WITH DIAERESIS", 0xcb },
  { "LATIN CAPITAL LETTER I WITH GRAVE", 0xcc },
  { "LATIN CAPITAL LETTER I WITH ACUTE", 0xcd },
  { "LATIN CAPITAL LETTER I WITH CIRCUMFLEX", 0xce },
  { "LATIN CAPITAL LETTER I WITH DIAERESIS", 0xcf },
  { "LATIN CAPITAL LETTER ETH", 0xd0 },
  { "LATIN CAPITAL LETTER N WITH TILDE", 0xd1 },
  { "LATIN CAPITAL LETTER O WITH GRAVE", 0xd2 },
  { "LATIN CAPITAL LETTER O WITH ACUTE", 0xd3 },
  { "LATIN CAPITAL LETTER O WITH CIRCUMFLEX", 0xd4 },
  { "LATIN CAPITAL LETTER O WITH TILDE", 0xd5 },
  { "LATIN CAPITAL LETTER O WITH DIAERESIS", 0xd6 },
  { "MULTIPLICATION SIGN", 0xd7 },
  { "LATIN CAPITAL LETTER O WITH STROKE", 0xd8 },
  { "LATIN CAPITAL LETTER U WITH GRAVE", 0xd9 },
  { "LATIN CAPITAL LETTER U WITH ACUTE", 0xda },
  { "LATIN CAPITAL LETTER U WITH CIRCUMFLEX", 0xdb },
  { "LATIN CAPITAL LETTER U WITH DIAERESIS", 0xdc },
  { "LATIN CAPITAL LETTER Y WITH ACUTE", 0xdd },
  { "LATIN CAPITAL LETTER THORN", 0xde },
  { "LATIN SMALL LETTER SHARP S", 0xdf },
  { "LATIN SMALL LETTER A WITH GRAVE", 0xe0 },
  { "LATIN SMALL LETTER A WITH ACUTE", 0xe1 },
  { "LATIN SMALL LETTER A WITH CIRCUMFLEX", 0xe2 },
  { "LATIN SMALL LETTER A WITH TILDE", 0xe3 },
  { "LATIN SMALL LETTER A WITH DIAERESIS", 0xe4 },
  { "LATIN SMALL LETTER A WITH RING ABOVE", 0xe5 },
  { "LATIN SMALL LETTER AE", 0xe6 },
  { "LATIN SMALL LETTER C WITH CEDILLA", 0xe7 },
  { "LATIN SMALL LETTER E WITH GRAVE", 0xe8 },
  { "LATIN SMALL LETTER E WITH ACUTE", 0xe9 },
  { "LATIN SMALL LETTER E WITH CIRCUMFLEX", 0xea },
  { "LATIN SMALL LETTER E WITH DIAERESIS", 0xeb },
  { "LATIN SMALL LETTER I WITH GRAVE", 0xec },
  { "LATIN SMALL LETTER I WITH ACUTE", 0xed },
  { "LATIN SMALL LETTER I WITH CIRCUMFLEX", 0xee },
  { "LATIN SMALL LETTER I WITH DIAERESIS", 0xef },
  { "LATIN SMALL LETTER ETH", 0xf0 },
  { "LATIN SMALL LETTER N WITH TILDE", 0xf1 },
  { "LATIN SMALL LETTER O WITH GRAVE", 0xf2 },
  { "LATIN SMALL LETTER O WITH ACUTE", 0xf3 },
  { "LATIN SMALL LETTER O WITH CIRCUMFLEX", 0xf4 },
  { "LATIN SMALL LETTER O WITH TILDE", 0xf5 },
  { "LATIN SMALL LETTER O WITH DIAERESIS", 0xf6 },
  { "DIVISION SIGN", 0xf7 },
  { "LATIN SMALL LETTER O WITH STROKE", 0xf8 },
  { "LATIN SMALL LETTER U WITH GRAVE", 0xf9 },
  { "LATIN SMALL LETTER U WITH ACUTE", 0xfa },
  { "LATIN SMALL LETTER U WITH CIRCUMFLEX", 0xfb },
  { "LATIN SMALL LETTER U WITH DIAERESIS", 0xfc },
  { "LATIN SMALL LETTER Y WITH ACUTE", 0xfd },
  { "LATIN SMALL LETTER THORN", 0xfe },
  { "LATIN SMALL LETTER Y WITH DIAERESIS", 0xff },
  { "LATIN CAPITAL LETTER A WITH MACRON", 0x100 },
  { "LATIN SMALL LETTER A WITH MACRON", 0x101 },
  { "LATIN CAPITAL LETTER A WITH BREVE", 0x102 },
  { "LATIN SMALL LETTER A WITH BREVE", 0x103 },
  { "LATIN CAPITAL LETTER A WITH OGONEK", 0x104 },
  { "LATIN SMALL LETTER A WITH OGONEK", 0x105 },
  { "LATIN CAPITAL LETTER C WITH ACUTE", 0x106 },
  { "LATIN SMALL LETTER C WITH ACUTE", 0x107 },
  { "LATIN CAPITAL LETTER C WITH CIRCUMFLEX", 0x108 },
  { "LATIN SMALL LETTER C WITH CIRCUMFLEX", 0x109 },
  { "LATIN CAPITAL LETTER C WITH DOT ABOVE", 0x10a },
  { "LATIN SMALL LETTER C WITH DOT ABOVE", 0x10b },
  { "LATIN CAPITAL LETTER C WITH CARON", 0x10c },
  { "LATIN SMALL LETTER C WITH CARON", 0x10d },
  { "LATIN CAPITAL LETTER D WITH CARON", 0x10e },
  { "LATIN SMALL LETTER D WITH CARON", 0x10f },
  { "LATIN CAPITAL LETTER D WITH STROKE", 0x110 },
  { "LATIN SMALL LETTER D WITH STROKE", 0x111 },
  { "LATIN CAPITAL LETTER E WITH MACRON", 0x112 },
  { "LATIN SMALL LETTER E WITH MACRON", 0x113 },
  { "LATIN CAPITAL LETTER E WITH BREVE", 0x114 },
  { "LATIN SMALL LETTER E WITH BREVE", 0x115 },
  { "LATIN CAPITAL LETTER E WITH DOT ABOVE", 0x116 },
  { "LATIN SMALL LETTER E WITH DOT ABOVE", 0x117 },
  { "LATIN CAPITAL LETTER E WITH OGONEK", 0x118 },
  { "LATIN SMALL LETTER E WITH OGONEK", 0x119 },
  { "LATIN CAPITAL LETTER E WITH CARON", 0x11a },
  { "LATIN SMALL LETTER E WITH CARON", 0x11b },
  { "LATIN CAPITAL LETTER G WITH CIRCUMFLEX", 0x11c },
  { "LATIN SMALL LETTER G WITH CIRCUMFLEX", 0x11d },
  { "LATIN CAPITAL LETTER G WITH BREVE", 0x11e },
  { "LATIN SMALL LETTER G WITH BREVE", 0x11f },
  { "LATIN CAPITAL LETTER G WITH DOT ABOVE", 0x120 },
  { "LATIN SMALL LETTER G WITH DOT ABOVE", 0x121 },
  { "LATIN CAPITAL LETTER G WITH CEDILLA", 0x122 },
  { "LATIN SMALL LETTER G WITH CEDILLA", 0x123 },
  { "LATIN CAPITAL LETTER H WITH CIRCUMFLEX", 0x124 },
  { "LATIN SMALL LETTER H WITH CIRCUMFLEX", 0x125 },
  { "LATIN CAPITAL LETTER H WITH STROKE", 0x126 },
  { "LATIN SMALL LETTER H WITH STROKE", 0x127 },
  { "LATIN CAPITAL LETTER I WITH TILDE", 0x128 },
  { "LATIN SMALL LETTER I WITH TILDE", 0x129 },
  { "LATIN CAPITAL LETTER I WITH MACRON", 0x12a },
  { "LATIN SMALL LETTER I WITH MACRON", 0x12b },
  { "LATIN CAPITAL LETTER I WITH BREVE", 0x12c },
  { "LATIN SMALL LETTER I WITH BREVE", 0x12d },
  { "LATIN CAPITAL LETTER I WITH OGONEK", 0x12e },
  { "LATIN SMALL LETTER I WITH OGONEK", 0x12f },
  { "LATIN CAPITAL LETTER I WITH DOT ABOVE", 0x130 },
  { "LATIN SMALL LETTER DOTLESS I", 0x131 },
  { "LATIN CAPITAL LIGATURE IJ", 0x132 },
  { "LATIN SMALL LIGATURE IJ", 0x133 },
  { "LATIN CAPITAL LETTER J WITH CIRCUMFLEX", 0x134 },
  { "LATIN SMALL LETTER J WITH CIRCUMFLEX", 0x135 },
  { "LATIN CAPITAL LETTER K WITH CEDILLA", 0x136 },
  { "LATIN SMALL LETTER K WITH CEDILLA", 0x137 },
  { "LATIN SMALL LETTER KRA", 0x138 },
  { "LATIN CAPITAL LETTER L WITH ACUTE", 0x139 },
  { "LATIN SMALL LETTER L WITH ACUTE", 0x13a },
  { "LATIN CAPITAL LETTER L WITH CEDILLA", 0x13b },
  { "LATIN SMALL LETTER L WITH CEDILLA", 0x13c },
  { "LATIN CAPITAL LETTER L WITH CARON", 0x13d },
  { "LATIN SMALL LETTER L WITH CARON", 0x13e },
  { "LATIN CAPITAL LETTER L WITH MIDDLE DOT", 0x13f },
  { "LATIN SMALL LETTER L WITH MIDDLE DOT", 0x140 },
  { "LATIN CAPITAL LETTER L WITH STROKE", 0x141 },
  { "LATIN SMALL LETTER L WITH STROKE", 0x142 },
  { "LATIN CAPITAL LETTER N WITH ACUTE", 0x143 },
  { "LATIN SMALL LETTER N WITH ACUTE", 0x144 },
  { "LATIN CAPITAL LETTER N WITH CEDILLA", 0x145 },
  { "LATIN SMALL LETTER N WITH CEDILLA", 0x146 },
  { "LATIN CAPITAL LETTER N WITH CARON", 0x147 },
  { "LATIN SMALL LETTER N WITH CARON", 0x148 },
  { "LATIN SMALL LETTER N PRECEDED BY APOSTROPHE", 0x149 },
  { "LATIN CAPITAL LETTER ENG", 0x14a },
  { "LATIN SMALL LETTER ENG", 0x14b },
  { "LATIN CAPITAL LETTER O WITH MACRON", 0x14c },
  { "LATIN SMALL LETTER O WITH MACRON", 0x14d },
  { "LATIN CAPITAL LETTER O WITH BREVE", 0x14e },
  { "LATIN SMALL LETTER O WITH BREVE", 0x14f },
  { "LATIN CAPITAL LETTER O WITH DOUBLE ACUTE", 0x150 },
  { "LATIN SMALL LETTER O WITH DOUBLE ACUTE", 0x151 },
  { "LATIN CAPITAL LIGATURE OE", 0x152 },
  { "LATIN SMALL LIGATURE OE", 0x153 },
  { "LATIN CAPITAL LETTER R WITH ACUTE", 0x154 },
  { "LATIN SMALL LETTER R WITH ACUTE", 0x155 },
  { "LATIN CAPITAL LETTER R WITH CEDILLA", 0x156 },
  { "LATIN SMALL LETTER R WITH CEDILLA", 0x157 },
  { "LATIN CAPITAL LETTER R WITH CARON", 0x158 },
  { "LATIN SMALL LETTER R WITH CARON", 0x159 },
  { "LATIN CAPITAL LETTER S WITH ACUTE", 0x15a },
  { "LATIN SMALL LETTER S WITH ACUTE", 0x15b },
  { "LATIN CAPITAL LETTER S WITH CIRCUMFLEX", 0x15c },
  { "LATIN SMALL LETTER S WITH CIRCUMFLEX", 0x15d },
  { "LATIN CAPITAL LETTER S WITH CEDILLA", 0x15e },
  { "LATIN SMALL LETTER S WITH CEDILLA", 0x15f },
  { "LATIN CAPITAL LETTER S WITH CARON", 0x160 },
  { "LATIN SMALL LETTER S WITH CARON", 0x161 },
  { "LATIN CAPITAL LETTER T WITH CEDILLA", 0x162 },
  { "LATIN SMALL LETTER T WITH CEDILLA", 0x163 },
  { "LATIN CAPITAL LETTER T WITH CARON", 0x164 },
  { "LATIN SMALL LETTER T WITH CARON", 0x165 },
  { "LATIN CAPITAL LETTER T WITH STROKE", 0x166 },
  { "LATIN SMALL LETTER T WITH STROKE", 0x167 },
  { "LATIN CAPITAL LETTER U WITH TILDE", 0x168 },
  { "LATIN SMALL LETTER U WITH TILDE", 0x169 },
  { "LATIN CAPITAL LETTER U WITH MACRON", 0x16a },
  { "LATIN SMALL LETTER U WITH MACRON", 0x16b },
  { "LATIN CAPITAL LETTER U WITH BREVE", 0x16c },
  { "LATIN SMALL LETTER U WITH BREVE", 0x16d },
  { "LATIN CAPITAL LETTER U WITH RING ABOVE", 0x16e },
  { "LATIN SMALL LETTER U WITH RING ABOVE", 0x16f },
  { "LATIN CAPITAL LETTER U WITH DOUBLE ACUTE", 0x170 },
  { "LATIN SMALL LETTER U WITH DOUBLE ACUTE", 0x171 },
  { "LATIN CAPITAL LETTER U WITH OGONEK", 0x172 },
  { "LATIN SMALL LETTER U WITH OGONEK", 0x173 },
  { "LATIN CAPITAL LETTER W WITH CIRCUMFLEX", 0x174 },
  { "LATIN SMALL LETTER W WITH CIRCUMFLEX", 0x175 },
  { "LATIN CAPITAL LETTER Y WITH CIRCUMFLEX", 0x176 },
  { "LATIN SMALL LETTER Y WITH CIRCUMFLEX", 0x177 },
  { "LATIN CAPITAL LETTER Y WITH DIAERESIS", 0x178 },
  { "LATIN CAPITAL LETTER Z WITH ACUTE", 0x179 },
  { "LATIN SMALL LETTER Z WITH ACUTE", 0x17a },
  { "LATIN CAPITAL LETTER Z WITH DOT ABOVE", 0x17b },
  { "LATIN SMALL LETTER Z WITH DOT ABOVE", 0x17c },
  { "LATIN CAPITAL LETTER Z WITH CARON", 0x17d },
  { "LATIN SMALL LETTER Z WITH CARON", 0x17e },
  { "LATIN SMALL LETTER LONG S", 0x17f },
  { "LATIN SMALL LETTER B WITH STROKE", 0x180 },
  { "LATIN CAPITAL LETTER B WITH HOOK", 0x181 },
  { "LATIN CAPITAL LETTER B WITH TOPBAR", 0x182 },
  { "LATIN SMALL LETTER B WITH TOPBAR", 0x183 },
  { "LATIN CAPITAL LETTER TONE SIX", 0x184 },
  { "LATIN SMALL LETTER TONE SIX", 0x185 },
  { "LATIN CAPITAL LETTER OPEN O", 0x186 },
  { "LATIN CAPITAL LETTER C WITH HOOK", 0x187 },
  { "LATIN SMALL LETTER C WITH HOOK", 0x188 },
  { "LATIN CAPITAL LETTER AFRICAN D", 0x189 },
  { "LATIN CAPITAL LETTER D WITH HOOK", 0x18a },
  { "LATIN CAPITAL LETTER D WITH TOPBAR", 0x18b },
  { "LATIN SMALL LETTER D WITH TOPBAR", 0x18c },
  { "LATIN SMALL LETTER TURNED DELTA", 0x18d },
  { "LATIN CAPITAL LETTER REVERSED E", 0x18e },
  { "LATIN CAPITAL LETTER SCHWA", 0x18f },
  { "LATIN CAPITAL LETTER OPEN E", 0x190 },
  { "LATIN CAPITAL LETTER F WITH HOOK", 0x191 },
  { "LATIN SMALL LETTER F WITH HOOK", 0x192 },
  { "LATIN CAPITAL LETTER G WITH HOOK", 0x193 },
  { "LATIN CAPITAL LETTER GAMMA", 0x194 },
  { "LATIN SMALL LETTER HV", 0x195 },
  { "LATIN CAPITAL LETTER IOTA", 0x196 },
  { "LATIN CAPITAL LETTER I WITH STROKE", 0x197 },
  { "LATIN CAPITAL LETTER K WITH HOOK", 0x198 },
  { "LATIN SMALL LETTER K WITH HOOK", 0x199 },
  { "LATIN SMALL LETTER L WITH BAR", 0x19a },
  { "LATIN SMALL LETTER LAMBDA WITH STROKE", 0x19b },
  { "LATIN CAPITAL LETTER TURNED M", 0x19c },
  { "LATIN CAPITAL LETTER N WITH LEFT HOOK", 0x19d },
  { "LATIN SMALL LETTER N WITH LONG RIGHT LEG", 0x19e },
  { "LATIN CAPITAL LETTER O WITH MIDDLE TILDE", 0x19f },
  { "LATIN CAPITAL LETTER O WITH HORN", 0x1a0 },
  { "LATIN SMALL LETTER O WITH HORN", 0x1a1 },
  { "LATIN CAPITAL LETTER OI", 0x1a2 },
  { "LATIN SMALL LETTER OI", 0x1a3 },
  { "LATIN CAPITAL LETTER P WITH HOOK", 0x1a4 },
  { "LATIN SMALL LETTER P WITH HOOK", 0x1a5 },
  { "LATIN LETTER YR", 0x1a6 },
  { "LATIN CAPITAL LETTER TONE TWO", 0x1a7 },
  { "LATIN SMALL LETTER TONE TWO", 0x1a8 },
  { "LATIN CAPITAL LETTER ESH", 0x1a9 },
  { "LATIN LETTER REVERSED ESH LOOP", 0x1aa },
  { "LATIN SMALL LETTER T WITH PALATAL HOOK", 0x1ab },
  { "LATIN CAPITAL LETTER T WITH HOOK", 0x1ac },
  { "LATIN SMALL LETTER T WITH HOOK", 0x1ad },
  { "LATIN CAPITAL LETTER T WITH RETROFLEX HOOK", 0x1ae },
  { "LATIN CAPITAL LETTER U WITH HORN", 0x1af },
  { "LATIN SMALL LETTER U WITH HORN", 0x1b0 },
  { "LATIN CAPITAL LETTER UPSILON", 0x1b1 },
  { "LATIN CAPITAL LETTER V WITH HOOK", 0x1b2 },
  { "LATIN CAPITAL LETTER Y WITH HOOK", 0x1b3 },
  { "LATIN SMALL LETTER Y WITH HOOK", 0x1b4 },
  { "LATIN CAPITAL LETTER Z WITH STROKE", 0x1b5 },
  { "LATIN SMALL LETTER Z WITH STROKE", 0x1b6 },
  { "LATIN CAPITAL LETTER EZH", 0x1b7 },
  { "LATIN CAPITAL LETTER EZH REVERSED", 0x1b8 },
  { "LATIN SMALL LETTER EZH REVERSED", 0x1b9 },
  { "LATIN SMALL LETTER EZH WITH TAIL", 0x1ba },
  { "LATIN LETTER TWO WITH STROKE", 0x1bb },
  { "LATIN CAPITAL LETTER TONE FIVE", 0x1bc },
  { "LATIN SMALL LETTER TONE FIVE", 0x1bd },
  { "LATIN LETTER INVERTED GLOTTAL STOP WITH STROKE", 0x1be },
  { "LATIN LETTER WYNN", 0x1bf },
  { "LATIN LETTER DENTAL CLICK", 0x1c0 },
  { "LATIN LETTER LATERAL CLICK", 0x1c1 },
  { "LATIN LETTER ALVEOLAR CLICK", 0x1c2 },
  { "LATIN LETTER RETROFLEX CLICK", 0x1c3 },
  { "LATIN CAPITAL LETTER DZ WITH CARON", 0x1c4 },
  { "LATIN CAPITAL LETTER D WITH SMALL LETTER Z WITH CARON", 0x1c5 },
  { "LATIN SMALL LETTER DZ WITH CARON", 0x1c6 },
  { "LATIN CAPITAL LETTER LJ", 0x1c7 },
  { "LATIN CAPITAL LETTER L WITH SMALL LETTER J", 0x1c8 },
  { "LATIN SMALL LETTER LJ", 0x1c9 },
  { "LATIN CAPITAL LETTER NJ", 0x1ca },
  { "LATIN CAPITAL LETTER N WITH SMALL LETTER J", 0x1cb },
  { "LATIN SMALL LETTER NJ", 0x1cc },
  { "LATIN CAPITAL LETTER A WITH CARON", 0x1cd },
  { "LATIN SMALL LETTER A WITH CARON", 0x1ce },
  { "LATIN CAPITAL LETTER I WITH CARON", 0x1cf },
  { "LATIN SMALL LETTER I WITH CARON", 0x1d0 },
  { "LATIN CAPITAL LETTER O WITH CARON", 0x1d1 },
  { "LATIN SMALL LETTER O WITH CARON", 0x1d2 },
  { "LATIN CAPITAL LETTER U WITH CARON", 0x1d3 },
  { "LATIN SMALL LETTER U WITH CARON", 0x1d4 },
  { "LATIN CAPITAL LETTER U WITH DIAERESIS AND MACRON", 0x1d5 },
  { "LATIN SMALL LETTER U WITH DIAERESIS AND MACRON", 0x1d6 },
  { "LATIN CAPITAL LETTER U WITH DIAERESIS AND ACUTE", 0x1d7 },
  { "LATIN SMALL LETTER U WITH DIAERESIS AND ACUTE", 0x1d8 },
  { "LATIN CAPITAL LETTER U WITH DIAERESIS AND CARON", 0x1d9 },
  { "LATIN SMALL LETTER U WITH DIAERESIS AND CARON", 0x1da },
  { "LATIN CAPITAL LETTER U WITH DIAERESIS AND GRAVE", 0x1db },
  { "LATIN SMALL LETTER U WITH DIAERESIS AND GRAVE", 0x1dc },
  { "LATIN SMALL LETTER TURNED E", 0x1dd },
  { "LATIN CAPITAL LETTER A WITH DIAERESIS AND MACRON", 0x1de },
  { "LATIN SMALL LETTER A WITH DIAERESIS AND MACRON", 0x1df },
  { "LATIN CAPITAL LETTER A WITH DOT ABOVE AND MACRON", 0x1e0 },
  { "LATIN SMALL LETTER A WITH DOT ABOVE AND MACRON", 0x1e1 },
  { "LATIN CAPITAL LETTER AE WITH MACRON", 0x1e2 },
  { "LATIN SMALL LETTER AE WITH MACRON", 0x1e3 },
  { "LATIN CAPITAL LETTER G WITH STROKE", 0x1e4 },
  { "LATIN SMALL LETTER G WITH STROKE", 0x1e5 },
  { "LATIN CAPITAL LETTER G WITH CARON", 0x1e6 },
  { "LATIN SMALL LETTER G WITH CARON", 0x1e7 },
  { "LATIN CAPITAL LETTER K WITH CARON", 0x1e8 },
  { "LATIN SMALL LETTER K WITH CARON", 0x1e9 },
  { "LATIN CAPITAL LETTER O WITH OGONEK", 0x1ea },
  { "LATIN SMALL LETTER O WITH OGONEK", 0x1eb },
  { "LATIN CAPITAL LETTER O WITH OGONEK AND MACRON", 0x1ec },
  { "LATIN SMALL LETTER O WITH OGONEK AND MACRON", 0x1ed },
  { "LATIN CAPITAL LETTER EZH WITH CARON", 0x1ee },
  { "LATIN SMALL LETTER EZH WITH CARON", 0x1ef },
  { "LATIN SMALL LETTER J WITH CARON", 0x1f0 },
  { "LATIN CAPITAL LETTER DZ", 0x1f1 },
  { "LATIN CAPITAL LETTER D WITH SMALL LETTER Z", 0x1f2 },
  { "LATIN SMALL LETTER DZ", 0x1f3 },
  { "LATIN CAPITAL LETTER G WITH ACUTE", 0x1f4 },
  { "LATIN SMALL LETTER G WITH ACUTE", 0x1f5 },
  { "LATIN CAPITAL LETTER HWAIR", 0x1f6 },
  { "LATIN CAPITAL LETTER WYNN", 0x1f7 },
  { "LATIN CAPITAL LETTER N WITH GRAVE", 0x1f8 },
  { "LATIN SMALL LETTER N WITH GRAVE", 0x1f9 },
  { "LATIN CAPITAL LETTER A WITH RING ABOVE AND ACUTE", 0x1fa },
  { "LATIN SMALL LETTER A WITH RING ABOVE AND ACUTE", 0x1fb },
  { "LATIN CAPITAL LETTER AE WITH ACUTE", 0x1fc },
  { "LATIN SMALL LETTER AE WITH ACUTE", 0x1fd },
  { "LATIN CAPITAL LETTER O WITH STROKE AND ACUTE", 0x1fe },
  { "LATIN SMALL LETTER O WITH STROKE AND ACUTE", 0x1ff },
  { "LATIN CAPITAL LETTER A WITH DOUBLE GRAVE", 0x200 },
  { "LATIN SMALL LETTER A WITH DOUBLE GRAVE", 0x201 },
  { "LATIN CAPITAL LETTER A WITH INVERTED BREVE", 0x202 },
  { "LATIN SMALL LETTER A WITH INVERTED BREVE", 0x203 },
  { "LATIN CAPITAL LETTER E WITH DOUBLE GRAVE", 0x204 },
  { "LATIN SMALL LETTER E WITH DOUBLE GRAVE", 0x205 },
  { "LATIN CAPITAL LETTER E WITH INVERTED BREVE", 0x206 },
  { "LATIN SMALL LETTER E WITH INVERTED BREVE", 0x207 },
  { "LATIN CAPITAL LETTER I WITH DOUBLE GRAVE", 0x208 },
  { "LATIN SMALL LETTER I WITH DOUBLE GRAVE", 0x209 },
  { "LATIN CAPITAL LETTER I WITH INVERTED BREVE", 0x20a },
  { "LATIN SMALL LETTER I WITH INVERTED BREVE", 0x20b },
  { "LATIN CAPITAL LETTER O WITH DOUBLE GRAVE", 0x20c },
  { "LATIN SMALL LETTER O WITH DOUBLE GRAVE", 0x20d },
  { "LATIN CAPITAL LETTER O WITH INVERTED BREVE", 0x20e },
  { "LATIN SMALL LETTER O WITH INVERTED BREVE", 0x20f },
  { "LATIN CAPITAL LETTER R WITH DOUBLE GRAVE", 0x210 },
  { "LATIN SMALL LETTER R WITH DOUBLE GRAVE", 0x211 },
  { "LATIN CAPITAL LETTER R WITH INVERTED BREVE", 0x212 },
  { "LATIN SMALL LETTER R WITH INVERTED BREVE", 0x213 },
  { "LATIN CAPITAL LETTER U WITH DOUBLE GRAVE", 0x214 },
  { "LATIN SMALL LETTER U WITH DOUBLE GRAVE", 0x215 },
  { "LATIN CAPITAL LETTER U WITH INVERTED BREVE", 0x216 },
  { "LATIN SMALL LETTER U WITH INVERTED BREVE", 0x217 },
  { "LATIN CAPITAL LETTER S WITH COMMA BELOW", 0x218 },
  { "LATIN SMALL LETTER S WITH COMMA BELOW", 0x219 },
  { "LATIN CAPITAL LETTER T WITH COMMA BELOW", 0x21a },
  { "LATIN SMALL LETTER T WITH COMMA BELOW", 0x21b },
  { "LATIN CAPITAL LETTER YOGH", 0x21c },
  { "LATIN SMALL LETTER YOGH", 0x21d },
  { "LATIN CAPITAL LETTER H WITH CARON", 0x21e },
  { "LATIN SMALL LETTER H WITH CARON", 0x21f },
  { "LATIN CAPITAL LETTER OU", 0x222 },
  { "LATIN SMALL LETTER OU", 0x223 },
  { "LATIN CAPITAL LETTER Z WITH HOOK", 0x224 },
  { "LATIN SMALL LETTER Z WITH HOOK", 0x225 },
  { "LATIN CAPITAL LETTER A WITH DOT ABOVE", 0x226 },
  { "LATIN SMALL LETTER A WITH DOT ABOVE", 0x227 },
  { "LATIN CAPITAL LETTER E WITH CEDILLA", 0x228 },
  { "LATIN SMALL LETTER E WITH CEDILLA", 0x229 },
  { "LATIN CAPITAL LETTER O WITH DIAERESIS AND MACRON", 0x22a },
  { "LATIN SMALL LETTER O WITH DIAERESIS AND MACRON", 0x22b },
  { "LATIN CAPITAL LETTER O WITH TILDE AND MACRON", 0x22c },
  { "LATIN SMALL LETTER O WITH TILDE AND MACRON", 0x22d },
  { "LATIN CAPITAL LETTER O WITH DOT ABOVE", 0x22e },
  { "LATIN SMALL LETTER O WITH DOT ABOVE", 0x22f },
  { "LATIN CAPITAL LETTER O WITH DOT ABOVE AND MACRON", 0x230 },
  { "LATIN SMALL LETTER O WITH DOT ABOVE AND MACRON", 0x231 },
  { "LATIN CAPITAL LETTER Y WITH MACRON", 0x232 },
  { "LATIN SMALL LETTER Y WITH MACRON", 0x233 },
  { "LATIN SMALL LETTER TURNED A", 0x250 },
  { "LATIN SMALL LETTER ALPHA", 0x251 },
  { "LATIN SMALL LETTER TURNED ALPHA", 0x252 },
  { "LATIN SMALL LETTER B WITH HOOK", 0x253 },
  { "LATIN SMALL LETTER OPEN O", 0x254 },
  { "LATIN SMALL LETTER C WITH CURL", 0x255 },
  { "LATIN SMALL LETTER D WITH TAIL", 0x256 },
  { "LATIN SMALL LETTER D WITH HOOK", 0x257 },
  { "LATIN SMALL LETTER REVERSED E", 0x258 },
  { "LATIN SMALL LETTER SCHWA", 0x259 },
  { "LATIN SMALL LETTER SCHWA WITH HOOK", 0x25a },
  { "LATIN SMALL LETTER OPEN E", 0x25b },
  { "LATIN SMALL LETTER REVERSED OPEN E", 0x25c },
  { "LATIN SMALL LETTER REVERSED OPEN E WITH HOOK", 0x25d },
  { "LATIN SMALL LETTER CLOSED REVERSED OPEN E", 0x25e },
  { "LATIN SMALL LETTER DOTLESS J WITH STROKE", 0x25f },
  { "LATIN SMALL LETTER G WITH HOOK", 0x260 },
  { "LATIN SMALL LETTER SCRIPT G", 0x261 },
  { "LATIN LETTER SMALL CAPITAL G", 0x262 },
  { "LATIN SMALL LETTER GAMMA", 0x263 },
  { "LATIN SMALL LETTER RAMS HORN", 0x264 },
  { "LATIN SMALL LETTER TURNED H", 0x265 },
  { "LATIN SMALL LETTER H WITH HOOK", 0x266 },
  { "LATIN SMALL LETTER HENG WITH HOOK", 0x267 },
  { "LATIN SMALL LETTER I WITH STROKE", 0x268 },
  { "LATIN SMALL LETTER IOTA", 0x269 },
  { "LATIN LETTER SMALL CAPITAL I", 0x26a },
  { "LATIN SMALL LETTER L WITH MIDDLE TILDE", 0x26b },
  { "LATIN SMALL LETTER L WITH BELT", 0x26c },
  { "LATIN SMALL LETTER L WITH RETROFLEX HOOK", 0x26d },
  { "LATIN SMALL LETTER LEZH", 0x26e },
  { "LATIN SMALL LETTER TURNED M", 0x26f },
  { "LATIN SMALL LETTER TURNED M WITH LONG LEG", 0x270 },
  { "LATIN SMALL LETTER M WITH HOOK", 0x271 },
  { "LATIN SMALL LETTER N WITH LEFT HOOK", 0x272 },
  { "LATIN SMALL LETTER N WITH RETROFLEX HOOK", 0x273 },
  { "LATIN LETTER SMALL CAPITAL N", 0x274 },
  { "LATIN SMALL LETTER BARRED O", 0x275 },
  { "LATIN LETTER SMALL CAPITAL OE", 0x276 },
  { "LATIN SMALL LETTER CLOSED OMEGA", 0x277 },
  { "LATIN SMALL LETTER PHI", 0x278 },
  { "LATIN SMALL LETTER TURNED R", 0x279 },
  { "LATIN SMALL LETTER TURNED R WITH LONG LEG", 0x27a },
  { "LATIN SMALL LETTER TURNED R WITH HOOK", 0x27b },
  { "LATIN SMALL LETTER R WITH LONG LEG", 0x27c },
  { "LATIN SMALL LETTER R WITH TAIL", 0x27d },
  { "LATIN SMALL LETTER R WITH FISHHOOK", 0x27e },
  { "LATIN SMALL LETTER REVERSED R WITH FISHHOOK", 0x27f },
  { "LATIN LETTER SMALL CAPITAL R", 0x280 },
  { "LATIN LETTER SMALL CAPITAL INVERTED R", 0x281 },
  { "LATIN SMALL LETTER S WITH HOOK", 0x282 },
  { "LATIN SMALL LETTER ESH", 0x283 },
  { "LATIN SMALL LETTER DOTLESS J WITH STROKE AND HOOK", 0x284 },
  { "LATIN SMALL LETTER SQUAT REVERSED ESH", 0x285 },
  { "LATIN SMALL LETTER ESH WITH CURL", 0x286 },
  { "LATIN SMALL LETTER TURNED T", 0x287 },
  { "LATIN SMALL LETTER T WITH RETROFLEX HOOK", 0x288 },
  { "LATIN SMALL LETTER U BAR", 0x289 },
  { "LATIN SMALL LETTER UPSILON", 0x28a },
  { "LATIN SMALL LETTER V WITH HOOK", 0x28b },
  { "LATIN SMALL LETTER TURNED V", 0x28c },
  { "LATIN SMALL LETTER TURNED W", 0x28d },
  { "LATIN SMALL LETTER TURNED Y", 0x28e },
  { "LATIN LETTER SMALL CAPITAL Y", 0x28f },
  { "LATIN SMALL LETTER Z WITH RETROFLEX HOOK", 0x290 },
  { "LATIN SMALL LETTER Z WITH CURL", 0x291 },
  { "LATIN SMALL LETTER EZH", 0x292 },
  { "LATIN SMALL LETTER EZH WITH CURL", 0x293 },
  { "LATIN LETTER GLOTTAL STOP", 0x294 },
  { "LATIN LETTER PHARYNGEAL VOICED FRICATIVE", 0x295 },
  { "LATIN LETTER INVERTED GLOTTAL STOP", 0x296 },
  { "LATIN LETTER STRETCHED C", 0x297 },
  { "LATIN LETTER BILABIAL CLICK", 0x298 },
  { "LATIN LETTER SMALL CAPITAL B", 0x299 },
  { "LATIN SMALL LETTER CLOSED OPEN E", 0x29a },
  { "LATIN LETTER SMALL CAPITAL G WITH HOOK", 0x29b },
  { "LATIN LETTER SMALL CAPITAL H", 0x29c },
  { "LATIN SMALL LETTER J WITH CROSSED-TAIL", 0x29d },
  { "LATIN SMALL LETTER TURNED K", 0x29e },
  { "LATIN LETTER SMALL CAPITAL L", 0x29f },
  { "LATIN SMALL LETTER Q WITH HOOK", 0x2a0 },
  { "LATIN LETTER GLOTTAL STOP WITH STROKE", 0x2a1 },
  { "LATIN LETTER REVERSED GLOTTAL STOP WITH STROKE", 0x2a2 },
  { "LATIN SMALL LETTER DZ DIGRAPH", 0x2a3 },
  { "LATIN SMALL LETTER DEZH DIGRAPH", 0x2a4 },
  { "LATIN SMALL LETTER DZ DIGRAPH WITH CURL", 0x2a5 },
  { "LATIN SMALL LETTER TS DIGRAPH", 0x2a6 },
  { "LATIN SMALL LETTER TESH DIGRAPH", 0x2a7 },
  { "LATIN SMALL LETTER TC DIGRAPH WITH CURL", 0x2a8 },
  { "LATIN SMALL LETTER FENG DIGRAPH", 0x2a9 },
  { "LATIN SMALL LETTER LS DIGRAPH", 0x2aa },
  { "LATIN SMALL LETTER LZ DIGRAPH", 0x2ab },
  { "LATIN LETTER BILABIAL PERCUSSIVE", 0x2ac },
  { "LATIN LETTER BIDENTAL PERCUSSIVE", 0x2ad },
  { "MODIFIER LETTER SMALL H", 0x2b0 },
  { "MODIFIER LETTER SMALL H WITH HOOK", 0x2b1 },
  { "MODIFIER LETTER SMALL J", 0x2b2 },
  { "MODIFIER LETTER SMALL R", 0x2b3 },
  { "MODIFIER LETTER SMALL TURNED R", 0x2b4 },
  { "MODIFIER LETTER SMALL TURNED R WITH HOOK", 0x2b5 },
  { "MODIFIER LETTER SMALL CAPITAL INVERTED R", 0x2b6 },
  { "MODIFIER LETTER SMALL W", 0x2b7 },
  { "MODIFIER LETTER SMALL Y", 0x2b8 },
  { "MODIFIER LETTER PRIME", 0x2b9 },
  { "MODIFIER LETTER DOUBLE PRIME", 0x2ba },
  { "MODIFIER LETTER TURNED COMMA", 0x2bb },
  { "MODIFIER LETTER APOSTROPHE", 0x2bc },
  { "MODIFIER LETTER REVERSED COMMA", 0x2bd },
  { "MODIFIER LETTER RIGHT HALF RING", 0x2be },
  { "MODIFIER LETTER LEFT HALF RING", 0x2bf },
  { "MODIFIER LETTER GLOTTAL STOP", 0x2c0 },
  { "MODIFIER LETTER REVERSED GLOTTAL STOP", 0x2c1 },
  { "MODIFIER LETTER LEFT ARROWHEAD", 0x2c2 },
  { "MODIFIER LETTER RIGHT ARROWHEAD", 0x2c3 },
  { "MODIFIER LETTER UP ARROWHEAD", 0x2c4 },
  { "MODIFIER LETTER DOWN ARROWHEAD", 0x2c5 },
  { "MODIFIER LETTER CIRCUMFLEX ACCENT", 0x2c6 },
  { "CARON", 0x2c7 },
  { "MODIFIER LETTER VERTICAL LINE", 0x2c8 },
  { "MODIFIER LETTER MACRON", 0x2c9 },
  { "MODIFIER LETTER ACUTE ACCENT", 0x2ca },
  { "MODIFIER LETTER GRAVE ACCENT", 0x2cb },
  { "MODIFIER LETTER LOW VERTICAL LINE", 0x2cc },
  { "MODIFIER LETTER LOW MACRON", 0x2cd },
  { "MODIFIER LETTER LOW GRAVE ACCENT", 0x2ce },
  { "MODIFIER LETTER LOW ACUTE ACCENT", 0x2cf },
  { "MODIFIER LETTER TRIANGULAR COLON", 0x2d0 },
  { "MODIFIER LETTER HALF TRIANGULAR COLON", 0x2d1 },
  { "MODIFIER LETTER CENTRED RIGHT HALF RING", 0x2d2 },
  { "MODIFIER LETTER CENTRED LEFT HALF RING", 0x2d3 },
  { "MODIFIER LETTER UP TACK", 0x2d4 },
  { "MODIFIER LETTER DOWN TACK", 0x2d5 },
  { "MODIFIER LETTER PLUS SIGN", 0x2d6 },
  { "MODIFIER LETTER MINUS SIGN", 0x2d7 },
  { "BREVE", 0x2d8 },
  { "DOT ABOVE", 0x2d9 },
  { "RING ABOVE", 0x2da },
  { "OGONEK", 0x2db },
  { "SMALL TILDE", 0x2dc },
  { "DOUBLE ACUTE ACCENT", 0x2dd },
  { "MODIFIER LETTER RHOTIC HOOK", 0x2de },
  { "MODIFIER LETTER CROSS ACCENT", 0x2df },
  { "MODIFIER LETTER SMALL GAMMA", 0x2e0 },
  { "MODIFIER LETTER SMALL L", 0x2e1 },
  { "MODIFIER LETTER SMALL S", 0x2e2 },
  { "MODIFIER LETTER SMALL X", 0x2e3 },
  { "MODIFIER LETTER SMALL REVERSED GLOTTAL STOP", 0x2e4 },
  { "MODIFIER LETTER EXTRA-HIGH TONE BAR", 0x2e5 },
  { "MODIFIER LETTER HIGH TONE BAR", 0x2e6 },
  { "MODIFIER LETTER MID TONE BAR", 0x2e7 },
  { "MODIFIER LETTER LOW TONE BAR", 0x2e8 },
  { "MODIFIER LETTER EXTRA-LOW TONE BAR", 0x2e9 },
  { "MODIFIER LETTER YIN DEPARTING TONE MARK", 0x2ea },
  { "MODIFIER LETTER YANG DEPARTING TONE MARK", 0x2eb },
  { "MODIFIER LETTER VOICING", 0x2ec },
  { "MODIFIER LETTER UNASPIRATED", 0x2ed },
  { "MODIFIER LETTER DOUBLE APOSTROPHE", 0x2ee },
  { "COMBINING GRAVE ACCENT", 0x300 },
  { "COMBINING ACUTE ACCENT", 0x301 },
  { "COMBINING CIRCUMFLEX ACCENT", 0x302 },
  { "COMBINING TILDE", 0x303 },
  { "COMBINING MACRON", 0x304 },
  { "COMBINING OVERLINE", 0x305 },
  { "COMBINING BREVE", 0x306 },
  { "COMBINING DOT ABOVE", 0x307 },
  { "COMBINING DIAERESIS", 0x308 },
  { "COMBINING HOOK ABOVE", 0x309 },
  { "COMBINING RING ABOVE", 0x30a },
  { "COMBINING DOUBLE ACUTE ACCENT", 0x30b },
  { "COMBINING CARON", 0x30c },
  { "COMBINING VERTICAL LINE ABOVE", 0x30d },
  { "COMBINING DOUBLE VERTICAL LINE ABOVE", 0x30e },
  { "COMBINING DOUBLE GRAVE ACCENT", 0x30f },
  { "COMBINING CANDRABINDU", 0x310 },
  { "COMBINING INVERTED BREVE", 0x311 },
  { "COMBINING TURNED COMMA ABOVE", 0x312 },
  { "COMBINING COMMA ABOVE", 0x313 },
  { "COMBINING REVERSED COMMA ABOVE", 0x314 },
  { "COMBINING COMMA ABOVE RIGHT", 0x315 },
  { "COMBINING GRAVE ACCENT BELOW", 0x316 },
  { "COMBINING ACUTE ACCENT BELOW", 0x317 },
  { "COMBINING LEFT TACK BELOW", 0x318 },
  { "COMBINING RIGHT TACK BELOW", 0x319 },
  { "COMBINING LEFT ANGLE ABOVE", 0x31a },
  { "COMBINING HORN", 0x31b },
  { "COMBINING LEFT HALF RING BELOW", 0x31c },
  { "COMBINING UP TACK BELOW", 0x31d },
  { "COMBINING DOWN TACK BELOW", 0x31e },
  { "COMBINING PLUS SIGN BELOW", 0x31f },
  { "COMBINING MINUS SIGN BELOW", 0x320 },
  { "COMBINING PALATALIZED HOOK BELOW", 0x321 },
  { "COMBINING RETROFLEX HOOK BELOW", 0x322 },
  { "COMBINING DOT BELOW", 0x323 },
  { "COMBINING DIAERESIS BELOW", 0x324 },
  { "COMBINING RING BELOW", 0x325 },
  { "COMBINING COMMA BELOW", 0x326 },
  { "COMBINING CEDILLA", 0x327 },
  { "COMBINING OGONEK", 0x328 },
  { "COMBINING VERTICAL LINE BELOW", 0x329 },
  { "COMBINING BRIDGE BELOW", 0x32a },
  { "COMBINING INVERTED DOUBLE ARCH BELOW", 0x32b },
  { "COMBINING CARON BELOW", 0x32c },
  { "COMBINING CIRCUMFLEX ACCENT BELOW", 0x32d },
  { "COMBINING BREVE BELOW", 0x32e },
  { "COMBINING INVERTED BREVE BELOW", 0x32f },
  { "COMBINING TILDE BELOW", 0x330 },
  { "COMBINING MACRON BELOW", 0x331 },
  { "COMBINING LOW LINE", 0x332 },
  { "COMBINING DOUBLE LOW LINE", 0x333 },
  { "COMBINING TILDE OVERLAY", 0x334 },
  { "COMBINING SHORT STROKE OVERLAY", 0x335 },
  { "COMBINING LONG STROKE OVERLAY", 0x336 },
  { "COMBINING SHORT SOLIDUS OVERLAY", 0x337 },
  { "COMBINING LONG SOLIDUS OVERLAY", 0x338 },
  { "COMBINING RIGHT HALF RING BELOW", 0x339 },
  { "COMBINING INVERTED BRIDGE BELOW", 0x33a },
  { "COMBINING SQUARE BELOW", 0x33b },
  { "COMBINING SEAGULL BELOW", 0x33c },
  { "COMBINING X ABOVE", 0x33d },
  { "COMBINING VERTICAL TILDE", 0x33e },
  { "COMBINING DOUBLE OVERLINE", 0x33f },
  { "COMBINING GRAVE TONE MARK", 0x340 },
  { "COMBINING ACUTE TONE MARK", 0x341 },
  { "COMBINING GREEK PERISPOMENI", 0x342 },
  { "COMBINING GREEK KORONIS", 0x343 },
  { "COMBINING GREEK DIALYTIKA TONOS", 0x344 },
  { "COMBINING GREEK YPOGEGRAMMENI", 0x345 },
  { "COMBINING BRIDGE ABOVE", 0x346 },
  { "COMBINING EQUALS SIGN BELOW", 0x347 },
  { "COMBINING DOUBLE VERTICAL LINE BELOW", 0x348 },
  { "COMBINING LEFT ANGLE BELOW", 0x349 },
  { "COMBINING NOT TILDE ABOVE", 0x34a },
  { "COMBINING HOMOTHETIC ABOVE", 0x34b },
  { "COMBINING ALMOST EQUAL TO ABOVE", 0x34c },
  { "COMBINING LEFT RIGHT ARROW BELOW", 0x34d },
  { "COMBINING UPWARDS ARROW BELOW", 0x34e },
  { "COMBINING DOUBLE TILDE", 0x360 },
  { "COMBINING DOUBLE INVERTED BREVE", 0x361 },
  { "COMBINING DOUBLE RIGHTWARDS ARROW BELOW", 0x362 },
  { "GREEK NUMERAL SIGN", 0x374 },
  { "GREEK LOWER NUMERAL SIGN", 0x375 },
  { "GREEK YPOGEGRAMMENI", 0x37a },
  { "GREEK QUESTION MARK", 0x37e },
  { "GREEK TONOS", 0x384 },
  { "GREEK DIALYTIKA TONOS", 0x385 },
  { "GREEK CAPITAL LETTER ALPHA WITH TONOS", 0x386 },
  { "GREEK ANO TELEIA", 0x387 },
  { "GREEK CAPITAL LETTER EPSILON WITH TONOS", 0x388 },
  { "GREEK CAPITAL LETTER ETA WITH TONOS", 0x389 },
  { "GREEK CAPITAL LETTER IOTA WITH TONOS", 0x38a },
  { "GREEK CAPITAL LETTER OMICRON WITH TONOS", 0x38c },
  { "GREEK CAPITAL LETTER UPSILON WITH TONOS", 0x38e },
  { "GREEK CAPITAL LETTER OMEGA WITH TONOS", 0x38f },
  { "GREEK SMALL LETTER IOTA WITH DIALYTIKA AND TONOS", 0x390 },
  { "GREEK CAPITAL LETTER ALPHA", 0x391 },
  { "GREEK CAPITAL LETTER BETA", 0x392 },
  { "GREEK CAPITAL LETTER GAMMA", 0x393 },
  { "GREEK CAPITAL LETTER DELTA", 0x394 },
  { "GREEK CAPITAL LETTER EPSILON", 0x395 },
  { "GREEK CAPITAL LETTER ZETA", 0x396 },
  { "GREEK CAPITAL LETTER ETA", 0x397 },
  { "GREEK CAPITAL LETTER THETA", 0x398 },
  { "GREEK CAPITAL LETTER IOTA", 0x399 },
  { "GREEK CAPITAL LETTER KAPPA", 0x39a },
  { "GREEK CAPITAL LETTER LAMDA", 0x39b },
  { "GREEK CAPITAL LETTER MU", 0x39c },
  { "GREEK CAPITAL LETTER NU", 0x39d },
  { "GREEK CAPITAL LETTER XI", 0x39e },
  { "GREEK CAPITAL LETTER OMICRON", 0x39f },
  { "GREEK CAPITAL LETTER PI", 0x3a0 },
  { "GREEK CAPITAL LETTER RHO", 0x3a1 },
  { "GREEK CAPITAL LETTER SIGMA", 0x3a3 },
  { "GREEK CAPITAL LETTER TAU", 0x3a4 },
  { "GREEK CAPITAL LETTER UPSILON", 0x3a5 },
  { "GREEK CAPITAL LETTER PHI", 0x3a6 },
  { "GREEK CAPITAL LETTER CHI", 0x3a7 },
  { "GREEK CAPITAL LETTER PSI", 0x3a8 },
  { "GREEK CAPITAL LETTER OMEGA", 0x3a9 },
  { "GREEK CAPITAL LETTER IOTA WITH DIALYTIKA", 0x3aa },
  { "GREEK CAPITAL LETTER UPSILON WITH DIALYTIKA", 0x3ab },
  { "GREEK SMALL LETTER ALPHA WITH TONOS", 0x3ac },
  { "GREEK SMALL LETTER EPSILON WITH TONOS", 0x3ad },
  { "GREEK SMALL LETTER ETA WITH TONOS", 0x3ae },
  { "GREEK SMALL LETTER IOTA WITH TONOS", 0x3af },
  { "GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND TONOS", 0x3b0 },
  { "GREEK SMALL LETTER ALPHA", 0x3b1 },
  { "GREEK SMALL LETTER BETA", 0x3b2 },
  { "GREEK SMALL LETTER GAMMA", 0x3b3 },
  { "GREEK SMALL LETTER DELTA", 0x3b4 },
  { "GREEK SMALL LETTER EPSILON", 0x3b5 },
  { "GREEK SMALL LETTER ZETA", 0x3b6 },
  { "GREEK SMALL LETTER ETA", 0x3b7 },
  { "GREEK SMALL LETTER THETA", 0x3b8 },
  { "GREEK SMALL LETTER IOTA", 0x3b9 },
  { "GREEK SMALL LETTER KAPPA", 0x3ba },
  { "GREEK SMALL LETTER LAMDA", 0x3bb },
  { "GREEK SMALL LETTER MU", 0x3bc },
  { "GREEK SMALL LETTER NU", 0x3bd },
  { "GREEK SMALL LETTER XI", 0x3be },
  { "GREEK SMALL LETTER OMICRON", 0x3bf },
  { "GREEK SMALL LETTER PI", 0x3c0 },
  { "GREEK SMALL LETTER RHO", 0x3c1 },
  { "GREEK SMALL LETTER FINAL SIGMA", 0x3c2 },
  { "GREEK SMALL LETTER SIGMA", 0x3c3 },
  { "GREEK SMALL LETTER TAU", 0x3c4 },
  { "GREEK SMALL LETTER UPSILON", 0x3c5 },
  { "GREEK SMALL LETTER PHI", 0x3c6 },
  { "GREEK SMALL LETTER CHI", 0x3c7 },
  { "GREEK SMALL LETTER PSI", 0x3c8 },
  { "GREEK SMALL LETTER OMEGA", 0x3c9 },
  { "GREEK SMALL LETTER IOTA WITH DIALYTIKA", 0x3ca },
  { "GREEK SMALL LETTER UPSILON WITH DIALYTIKA", 0x3cb },
  { "GREEK SMALL LETTER OMICRON WITH TONOS", 0x3cc },
  { "GREEK SMALL LETTER UPSILON WITH TONOS", 0x3cd },
  { "GREEK SMALL LETTER OMEGA WITH TONOS", 0x3ce },
  { "GREEK BETA SYMBOL", 0x3d0 },
  { "GREEK THETA SYMBOL", 0x3d1 },
  { "GREEK UPSILON WITH HOOK SYMBOL", 0x3d2 },
  { "GREEK UPSILON WITH ACUTE AND HOOK SYMBOL", 0x3d3 },
  { "GREEK UPSILON WITH DIAERESIS AND HOOK SYMBOL", 0x3d4 },
  { "GREEK PHI SYMBOL", 0x3d5 },
  { "GREEK PI SYMBOL", 0x3d6 },
  { "GREEK KAI SYMBOL", 0x3d7 },
  { "GREEK LETTER STIGMA", 0x3da },
  { "GREEK SMALL LETTER STIGMA", 0x3db },
  { "GREEK LETTER DIGAMMA", 0x3dc },
  { "GREEK SMALL LETTER DIGAMMA", 0x3dd },
  { "GREEK LETTER KOPPA", 0x3de },
  { "GREEK SMALL LETTER KOPPA", 0x3df },
  { "GREEK LETTER SAMPI", 0x3e0 },
  { "GREEK SMALL LETTER SAMPI", 0x3e1 },
  { "COPTIC CAPITAL LETTER SHEI", 0x3e2 },
  { "COPTIC SMALL LETTER SHEI", 0x3e3 },
  { "COPTIC CAPITAL LETTER FEI", 0x3e4 },
  { "COPTIC SMALL LETTER FEI", 0x3e5 },
  { "COPTIC CAPITAL LETTER KHEI", 0x3e6 },
  { "COPTIC SMALL LETTER KHEI", 0x3e7 },
  { "COPTIC CAPITAL LETTER HORI", 0x3e8 },
  { "COPTIC SMALL LETTER HORI", 0x3e9 },
  { "COPTIC CAPITAL LETTER GANGIA", 0x3ea },
  { "COPTIC SMALL LETTER GANGIA", 0x3eb },
  { "COPTIC CAPITAL LETTER SHIMA", 0x3ec },
  { "COPTIC SMALL LETTER SHIMA", 0x3ed },
  { "COPTIC CAPITAL LETTER DEI", 0x3ee },
  { "COPTIC SMALL LETTER DEI", 0x3ef },
  { "GREEK KAPPA SYMBOL", 0x3f0 },
  { "GREEK RHO SYMBOL", 0x3f1 },
  { "GREEK LUNATE SIGMA SYMBOL", 0x3f2 },
  { "GREEK LETTER YOT", 0x3f3 },
  { "CYRILLIC CAPITAL LETTER IE WITH GRAVE", 0x400 },
  { "CYRILLIC CAPITAL LETTER IO", 0x401 },
  { "CYRILLIC CAPITAL LETTER DJE", 0x402 },
  { "CYRILLIC CAPITAL LETTER GJE", 0x403 },
  { "CYRILLIC CAPITAL LETTER UKRAINIAN IE", 0x404 },
  { "CYRILLIC CAPITAL LETTER DZE", 0x405 },
  { "CYRILLIC CAPITAL LETTER BYELORUSSIAN-UKRAINIAN I", 0x406 },
  { "CYRILLIC CAPITAL LETTER YI", 0x407 },
  { "CYRILLIC CAPITAL LETTER JE", 0x408 },
  { "CYRILLIC CAPITAL LETTER LJE", 0x409 },
  { "CYRILLIC CAPITAL LETTER NJE", 0x40a },
  { "CYRILLIC CAPITAL LETTER TSHE", 0x40b },
  { "CYRILLIC CAPITAL LETTER KJE", 0x40c },
  { "CYRILLIC CAPITAL LETTER I WITH GRAVE", 0x40d },
  { "CYRILLIC CAPITAL LETTER SHORT U", 0x40e },
  { "CYRILLIC CAPITAL LETTER DZHE", 0x40f },
  { "CYRILLIC CAPITAL LETTER A", 0x410 },
  { "CYRILLIC CAPITAL LETTER BE", 0x411 },
  { "CYRILLIC CAPITAL LETTER VE", 0x412 },
  { "CYRILLIC CAPITAL LETTER GHE", 0x413 },
  { "CYRILLIC CAPITAL LETTER DE", 0x414 },
  { "CYRILLIC CAPITAL LETTER IE", 0x415 },
  { "CYRILLIC CAPITAL LETTER ZHE", 0x416 },
  { "CYRILLIC CAPITAL LETTER ZE", 0x417 },
  { "CYRILLIC CAPITAL LETTER I", 0x418 },
  { "CYRILLIC CAPITAL LETTER SHORT I", 0x419 },
  { "CYRILLIC CAPITAL LETTER KA", 0x41a },
  { "CYRILLIC CAPITAL LETTER EL", 0x41b },
  { "CYRILLIC CAPITAL LETTER EM", 0x41c },
  { "CYRILLIC CAPITAL LETTER EN", 0x41d },
  { "CYRILLIC CAPITAL LETTER O", 0x41e },
  { "CYRILLIC CAPITAL LETTER PE", 0x41f },
  { "CYRILLIC CAPITAL LETTER ER", 0x420 },
  { "CYRILLIC CAPITAL LETTER ES", 0x421 },
  { "CYRILLIC CAPITAL LETTER TE", 0x422 },
  { "CYRILLIC CAPITAL LETTER U", 0x423 },
  { "CYRILLIC CAPITAL LETTER EF", 0x424 },
  { "CYRILLIC CAPITAL LETTER HA", 0x425 },
  { "CYRILLIC CAPITAL LETTER TSE", 0x426 },
  { "CYRILLIC CAPITAL LETTER CHE", 0x427 },
  { "CYRILLIC CAPITAL LETTER SHA", 0x428 },
  { "CYRILLIC CAPITAL LETTER SHCHA", 0x429 },
  { "CYRILLIC CAPITAL LETTER HARD SIGN", 0x42a },
  { "CYRILLIC CAPITAL LETTER YERU", 0x42b },
  { "CYRILLIC CAPITAL LETTER SOFT SIGN", 0x42c },
  { "CYRILLIC CAPITAL LETTER E", 0x42d },
  { "CYRILLIC CAPITAL LETTER YU", 0x42e },
  { "CYRILLIC CAPITAL LETTER YA", 0x42f },
  { "CYRILLIC SMALL LETTER A", 0x430 },
  { "CYRILLIC SMALL LETTER BE", 0x431 },
  { "CYRILLIC SMALL LETTER VE", 0x432 },
  { "CYRILLIC SMALL LETTER GHE", 0x433 },
  { "CYRILLIC SMALL LETTER DE", 0x434 },
  { "CYRILLIC SMALL LETTER IE", 0x435 },
  { "CYRILLIC SMALL LETTER ZHE", 0x436 },
  { "CYRILLIC SMALL LETTER ZE", 0x437 },
  { "CYRILLIC SMALL LETTER I", 0x438 },
  { "CYRILLIC SMALL LETTER SHORT I", 0x439 },
  { "CYRILLIC SMALL LETTER KA", 0x43a },
  { "CYRILLIC SMALL LETTER EL", 0x43b },
  { "CYRILLIC SMALL LETTER EM", 0x43c },
  { "CYRILLIC SMALL LETTER EN", 0x43d },
  { "CYRILLIC SMALL LETTER O", 0x43e },
  { "CYRILLIC SMALL LETTER PE", 0x43f },
  { "CYRILLIC SMALL LETTER ER", 0x440 },
  { "CYRILLIC SMALL LETTER ES", 0x441 },
  { "CYRILLIC SMALL LETTER TE", 0x442 },
  { "CYRILLIC SMALL LETTER U", 0x443 },
  { "CYRILLIC SMALL LETTER EF", 0x444 },
  { "CYRILLIC SMALL LETTER HA", 0x445 },
  { "CYRILLIC SMALL LETTER TSE", 0x446 },
  { "CYRILLIC SMALL LETTER CHE", 0x447 },
  { "CYRILLIC SMALL LETTER SHA", 0x448 },
  { "CYRILLIC SMALL LETTER SHCHA", 0x449 },
  { "CYRILLIC SMALL LETTER HARD SIGN", 0x44a },
  { "CYRILLIC SMALL LETTER YERU", 0x44b },
  { "CYRILLIC SMALL LETTER SOFT SIGN", 0x44c },
  { "CYRILLIC SMALL LETTER E", 0x44d },
  { "CYRILLIC SMALL LETTER YU", 0x44e },
  { "CYRILLIC SMALL LETTER YA", 0x44f },
  { "CYRILLIC SMALL LETTER IE WITH GRAVE", 0x450 },
  { "CYRILLIC SMALL LETTER IO", 0x451 },
  { "CYRILLIC SMALL LETTER DJE", 0x452 },
  { "CYRILLIC SMALL LETTER GJE", 0x453 },
  { "CYRILLIC SMALL LETTER UKRAINIAN IE", 0x454 },
  { "CYRILLIC SMALL LETTER DZE", 0x455 },
  { "CYRILLIC SMALL LETTER BYELORUSSIAN-UKRAINIAN I", 0x456 },
  { "CYRILLIC SMALL LETTER YI", 0x457 },
  { "CYRILLIC SMALL LETTER JE", 0x458 },
  { "CYRILLIC SMALL LETTER LJE", 0x459 },
  { "CYRILLIC SMALL LETTER NJE", 0x45a },
  { "CYRILLIC SMALL LETTER TSHE", 0x45b },
  { "CYRILLIC SMALL LETTER KJE", 0x45c },
  { "CYRILLIC SMALL LETTER I WITH GRAVE", 0x45d },
  { "CYRILLIC SMALL LETTER SHORT U", 0x45e },
  { "CYRILLIC SMALL LETTER DZHE", 0x45f },
  { "CYRILLIC CAPITAL LETTER OMEGA", 0x460 },
  { "CYRILLIC SMALL LETTER OMEGA", 0x461 },
  { "CYRILLIC CAPITAL LETTER YAT", 0x462 },
  { "CYRILLIC SMALL LETTER YAT", 0x463 },
  { "CYRILLIC CAPITAL LETTER IOTIFIED E", 0x464 },
  { "CYRILLIC SMALL LETTER IOTIFIED E", 0x465 },
  { "CYRILLIC CAPITAL LETTER LITTLE YUS", 0x466 },
  { "CYRILLIC SMALL LETTER LITTLE YUS", 0x467 },
  { "CYRILLIC CAPITAL LETTER IOTIFIED LITTLE YUS", 0x468 },
  { "CYRILLIC SMALL LETTER IOTIFIED LITTLE YUS", 0x469 },
  { "CYRILLIC CAPITAL LETTER BIG YUS", 0x46a },
  { "CYRILLIC SMALL LETTER BIG YUS", 0x46b },
  { "CYRILLIC CAPITAL LETTER IOTIFIED BIG YUS", 0x46c },
  { "CYRILLIC SMALL LETTER IOTIFIED BIG YUS", 0x46d },
  { "CYRILLIC CAPITAL LETTER KSI", 0x46e },
  { "CYRILLIC SMALL LETTER KSI", 0x46f },
  { "CYRILLIC CAPITAL LETTER PSI", 0x470 },
  { "CYRILLIC SMALL LETTER PSI", 0x471 },
  { "CYRILLIC CAPITAL LETTER FITA", 0x472 },
  { "CYRILLIC SMALL LETTER FITA", 0x473 },
  { "CYRILLIC CAPITAL LETTER IZHITSA", 0x474 },
  { "CYRILLIC SMALL LETTER IZHITSA", 0x475 },
  { "CYRILLIC CAPITAL LETTER IZHITSA WITH DOUBLE GRAVE ACCENT", 0x476 },
  { "CYRILLIC SMALL LETTER IZHITSA WITH DOUBLE GRAVE ACCENT", 0x477 },
  { "CYRILLIC CAPITAL LETTER UK", 0x478 },
  { "CYRILLIC SMALL LETTER UK", 0x479 },
  { "CYRILLIC CAPITAL LETTER ROUND OMEGA", 0x47a },
  { "CYRILLIC SMALL LETTER ROUND OMEGA", 0x47b },
  { "CYRILLIC CAPITAL LETTER OMEGA WITH TITLO", 0x47c },
  { "CYRILLIC SMALL LETTER OMEGA WITH TITLO", 0x47d },
  { "CYRILLIC CAPITAL LETTER OT", 0x47e },
  { "CYRILLIC SMALL LETTER OT", 0x47f },
  { "CYRILLIC CAPITAL LETTER KOPPA", 0x480 },
  { "CYRILLIC SMALL LETTER KOPPA", 0x481 },
  { "CYRILLIC THOUSANDS SIGN", 0x482 },
  { "COMBINING CYRILLIC TITLO", 0x483 },
  { "COMBINING CYRILLIC PALATALIZATION", 0x484 },
  { "COMBINING CYRILLIC DASIA PNEUMATA", 0x485 },
  { "COMBINING CYRILLIC PSILI PNEUMATA", 0x486 },
  { "COMBINING CYRILLIC HUNDRED THOUSANDS SIGN", 0x488 },
  { "COMBINING CYRILLIC MILLIONS SIGN", 0x489 },
  { "CYRILLIC CAPITAL LETTER SEMISOFT SIGN", 0x48c },
  { "CYRILLIC SMALL LETTER SEMISOFT SIGN", 0x48d },
  { "CYRILLIC CAPITAL LETTER ER WITH TICK", 0x48e },
  { "CYRILLIC SMALL LETTER ER WITH TICK", 0x48f },
  { "CYRILLIC CAPITAL LETTER GHE WITH UPTURN", 0x490 },
  { "CYRILLIC SMALL LETTER GHE WITH UPTURN", 0x491 },
  { "CYRILLIC CAPITAL LETTER GHE WITH STROKE", 0x492 },
  { "CYRILLIC SMALL LETTER GHE WITH STROKE", 0x493 },
  { "CYRILLIC CAPITAL LETTER GHE WITH MIDDLE HOOK", 0x494 },
  { "CYRILLIC SMALL LETTER GHE WITH MIDDLE HOOK", 0x495 },
  { "CYRILLIC CAPITAL LETTER ZHE WITH DESCENDER", 0x496 },
  { "CYRILLIC SMALL LETTER ZHE WITH DESCENDER", 0x497 },
  { "CYRILLIC CAPITAL LETTER ZE WITH DESCENDER", 0x498 },
  { "CYRILLIC SMALL LETTER ZE WITH DESCENDER", 0x499 },
  { "CYRILLIC CAPITAL LETTER KA WITH DESCENDER", 0x49a },
  { "CYRILLIC SMALL LETTER KA WITH DESCENDER", 0x49b },
  { "CYRILLIC CAPITAL LETTER KA WITH VERTICAL STROKE", 0x49c },
  { "CYRILLIC SMALL LETTER KA WITH VERTICAL STROKE", 0x49d },
  { "CYRILLIC CAPITAL LETTER KA WITH STROKE", 0x49e },
  { "CYRILLIC SMALL LETTER KA WITH STROKE", 0x49f },
  { "CYRILLIC CAPITAL LETTER BASHKIR KA", 0x4a0 },
  { "CYRILLIC SMALL LETTER BASHKIR KA", 0x4a1 },
  { "CYRILLIC CAPITAL LETTER EN WITH DESCENDER", 0x4a2 },
  { "CYRILLIC SMALL LETTER EN WITH DESCENDER", 0x4a3 },
  { "CYRILLIC CAPITAL LIGATURE EN GHE", 0x4a4 },
  { "CYRILLIC SMALL LIGATURE EN GHE", 0x4a5 },
  { "CYRILLIC CAPITAL LETTER PE WITH MIDDLE HOOK", 0x4a6 },
  { "CYRILLIC SMALL LETTER PE WITH MIDDLE HOOK", 0x4a7 },
  { "CYRILLIC CAPITAL LETTER ABKHASIAN HA", 0x4a8 },
  { "CYRILLIC SMALL LETTER ABKHASIAN HA", 0x4a9 },
  { "CYRILLIC CAPITAL LETTER ES WITH DESCENDER", 0x4aa },
  { "CYRILLIC SMALL LETTER ES WITH DESCENDER", 0x4ab },
  { "CYRILLIC CAPITAL LETTER TE WITH DESCENDER", 0x4ac },
  { "CYRILLIC SMALL LETTER TE WITH DESCENDER", 0x4ad },
  { "CYRILLIC CAPITAL LETTER STRAIGHT U", 0x4ae },
  { "CYRILLIC SMALL LETTER STRAIGHT U", 0x4af },
  { "CYRILLIC CAPITAL LETTER STRAIGHT U WITH STROKE", 0x4b0 },
  { "CYRILLIC SMALL LETTER STRAIGHT U WITH STROKE", 0x4b1 },
  { "CYRILLIC CAPITAL LETTER HA WITH DESCENDER", 0x4b2 },
  { "CYRILLIC SMALL LETTER HA WITH DESCENDER", 0x4b3 },
  { "CYRILLIC CAPITAL LIGATURE TE TSE", 0x4b4 },
  { "CYRILLIC SMALL LIGATURE TE TSE", 0x4b5 },
  { "CYRILLIC CAPITAL LETTER CHE WITH DESCENDER", 0x4b6 },
  { "CYRILLIC SMALL LETTER CHE WITH DESCENDER", 0x4b7 },
  { "CYRILLIC CAPITAL LETTER CHE WITH VERTICAL STROKE", 0x4b8 },
  { "CYRILLIC SMALL LETTER CHE WITH VERTICAL STROKE", 0x4b9 },
  { "CYRILLIC CAPITAL LETTER SHHA", 0x4ba },
  { "CYRILLIC SMALL LETTER SHHA", 0x4bb },
  { "CYRILLIC CAPITAL LETTER ABKHASIAN CHE", 0x4bc },
  { "CYRILLIC SMALL LETTER ABKHASIAN CHE", 0x4bd },
  { "CYRILLIC CAPITAL LETTER ABKHASIAN CHE WITH DESCENDER", 0x4be },
  { "CYRILLIC SMALL LETTER ABKHASIAN CHE WITH DESCENDER", 0x4bf },
  { "CYRILLIC LETTER PALOCHKA", 0x4c0 },
  { "CYRILLIC CAPITAL LETTER ZHE WITH BREVE", 0x4c1 },
  { "CYRILLIC SMALL LETTER ZHE WITH BREVE", 0x4c2 },
  { "CYRILLIC CAPITAL LETTER KA WITH HOOK", 0x4c3 },
  { "CYRILLIC SMALL LETTER KA WITH HOOK", 0x4c4 },
  { "CYRILLIC CAPITAL LETTER EN WITH HOOK", 0x4c7 },
  { "CYRILLIC SMALL LETTER EN WITH HOOK", 0x4c8 },
  { "CYRILLIC CAPITAL LETTER KHAKASSIAN CHE", 0x4cb },
  { "CYRILLIC SMALL LETTER KHAKASSIAN CHE", 0x4cc },
  { "CYRILLIC CAPITAL LETTER A WITH BREVE", 0x4d0 },
  { "CYRILLIC SMALL LETTER A WITH BREVE", 0x4d1 },
  { "CYRILLIC CAPITAL LETTER A WITH DIAERESIS", 0x4d2 },
  { "CYRILLIC SMALL LETTER A WITH DIAERESIS", 0x4d3 },
  { "CYRILLIC CAPITAL LIGATURE A IE", 0x4d4 },
  { "CYRILLIC SMALL LIGATURE A IE", 0x4d5 },
  { "CYRILLIC CAPITAL LETTER IE WITH BREVE", 0x4d6 },
  { "CYRILLIC SMALL LETTER IE WITH BREVE", 0x4d7 },
  { "CYRILLIC CAPITAL LETTER SCHWA", 0x4d8 },
  { "CYRILLIC SMALL LETTER SCHWA", 0x4d9 },
  { "CYRILLIC CAPITAL LETTER SCHWA WITH DIAERESIS", 0x4da },
  { "CYRILLIC SMALL LETTER SCHWA WITH DIAERESIS", 0x4db },
  { "CYRILLIC CAPITAL LETTER ZHE WITH DIAERESIS", 0x4dc },
  { "CYRILLIC SMALL LETTER ZHE WITH DIAERESIS", 0x4dd },
  { "CYRILLIC CAPITAL LETTER ZE WITH DIAERESIS", 0x4de },
  { "CYRILLIC SMALL LETTER ZE WITH DIAERESIS", 0x4df },
  { "CYRILLIC CAPITAL LETTER ABKHASIAN DZE", 0x4e0 },
  { "CYRILLIC SMALL LETTER ABKHASIAN DZE", 0x4e1 },
  { "CYRILLIC CAPITAL LETTER I WITH MACRON", 0x4e2 },
  { "CYRILLIC SMALL LETTER I WITH MACRON", 0x4e3 },
  { "CYRILLIC CAPITAL LETTER I WITH DIAERESIS", 0x4e4 },
  { "CYRILLIC SMALL LETTER I WITH DIAERESIS", 0x4e5 },
  { "CYRILLIC CAPITAL LETTER O WITH DIAERESIS", 0x4e6 },
  { "CYRILLIC SMALL LETTER O WITH DIAERESIS", 0x4e7 },
  { "CYRILLIC CAPITAL LETTER BARRED O", 0x4e8 },
  { "CYRILLIC SMALL LETTER BARRED O", 0x4e9 },
  { "CYRILLIC CAPITAL LETTER BARRED O WITH DIAERESIS", 0x4ea },
  { "CYRILLIC SMALL LETTER BARRED O WITH DIAERESIS", 0x4eb },
  { "CYRILLIC CAPITAL LETTER E WITH DIAERESIS", 0x4ec },
  { "CYRILLIC SMALL LETTER E WITH DIAERESIS", 0x4ed },
  { "CYRILLIC CAPITAL LETTER U WITH MACRON", 0x4ee },
  { "CYRILLIC SMALL LETTER U WITH MACRON", 0x4ef },
  { "CYRILLIC CAPITAL LETTER U WITH DIAERESIS", 0x4f0 },
  { "CYRILLIC SMALL LETTER U WITH DIAERESIS", 0x4f1 },
  { "CYRILLIC CAPITAL LETTER U WITH DOUBLE ACUTE", 0x4f2 },
  { "CYRILLIC SMALL LETTER U WITH DOUBLE ACUTE", 0x4f3 },
  { "CYRILLIC CAPITAL LETTER CHE WITH DIAERESIS", 0x4f4 },
  { "CYRILLIC SMALL LETTER CHE WITH DIAERESIS", 0x4f5 },
  { "CYRILLIC CAPITAL LETTER YERU WITH DIAERESIS", 0x4f8 },
  { "CYRILLIC SMALL LETTER YERU WITH DIAERESIS", 0x4f9 },
  { "ARMENIAN CAPITAL LETTER AYB", 0x531 },
  { "ARMENIAN CAPITAL LETTER BEN", 0x532 },
  { "ARMENIAN CAPITAL LETTER GIM", 0x533 },
  { "ARMENIAN CAPITAL LETTER DA", 0x534 },
  { "ARMENIAN CAPITAL LETTER ECH", 0x535 },
  { "ARMENIAN CAPITAL LETTER ZA", 0x536 },
  { "ARMENIAN CAPITAL LETTER EH", 0x537 },
  { "ARMENIAN CAPITAL LETTER ET", 0x538 },
  { "ARMENIAN CAPITAL LETTER TO", 0x539 },
  { "ARMENIAN CAPITAL LETTER ZHE", 0x53a },
  { "ARMENIAN CAPITAL LETTER INI", 0x53b },
  { "ARMENIAN CAPITAL LETTER LIWN", 0x53c },
  { "ARMENIAN CAPITAL LETTER XEH", 0x53d },
  { "ARMENIAN CAPITAL LETTER CA", 0x53e },
  { "ARMENIAN CAPITAL LETTER KEN", 0x53f },
  { "ARMENIAN CAPITAL LETTER HO", 0x540 },
  { "ARMENIAN CAPITAL LETTER JA", 0x541 },
  { "ARMENIAN CAPITAL LETTER GHAD", 0x542 },
  { "ARMENIAN CAPITAL LETTER CHEH", 0x543 },
  { "ARMENIAN CAPITAL LETTER MEN", 0x544 },
  { "ARMENIAN CAPITAL LETTER YI", 0x545 },
  { "ARMENIAN CAPITAL LETTER NOW", 0x546 },
  { "ARMENIAN CAPITAL LETTER SHA", 0x547 },
  { "ARMENIAN CAPITAL LETTER VO", 0x548 },
  { "ARMENIAN CAPITAL LETTER CHA", 0x549 },
  { "ARMENIAN CAPITAL LETTER PEH", 0x54a },
  { "ARMENIAN CAPITAL LETTER JHEH", 0x54b },
  { "ARMENIAN CAPITAL LETTER RA", 0x54c },
  { "ARMENIAN CAPITAL LETTER SEH", 0x54d },
  { "ARMENIAN CAPITAL LETTER VEW", 0x54e },
  { "ARMENIAN CAPITAL LETTER TIWN", 0x54f },
  { "ARMENIAN CAPITAL LETTER REH", 0x550 },
  { "ARMENIAN CAPITAL LETTER CO", 0x551 },
  { "ARMENIAN CAPITAL LETTER YIWN", 0x552 },
  { "ARMENIAN CAPITAL LETTER PIWR", 0x553 },
  { "ARMENIAN CAPITAL LETTER KEH", 0x554 },
  { "ARMENIAN CAPITAL LETTER OH", 0x555 },
  { "ARMENIAN CAPITAL LETTER FEH", 0x556 },
  { "ARMENIAN MODIFIER LETTER LEFT HALF RING", 0x559 },
  { "ARMENIAN APOSTROPHE", 0x55a },
  { "ARMENIAN EMPHASIS MARK", 0x55b },
  { "ARMENIAN EXCLAMATION MARK", 0x55c },
  { "ARMENIAN COMMA", 0x55d },
  { "ARMENIAN QUESTION MARK", 0x55e },
  { "ARMENIAN ABBREVIATION MARK", 0x55f },
  { "ARMENIAN SMALL LETTER AYB", 0x561 },
  { "ARMENIAN SMALL LETTER BEN", 0x562 },
  { "ARMENIAN SMALL LETTER GIM", 0x563 },
  { "ARMENIAN SMALL LETTER DA", 0x564 },
  { "ARMENIAN SMALL LETTER ECH", 0x565 },
  { "ARMENIAN SMALL LETTER ZA", 0x566 },
  { "ARMENIAN SMALL LETTER EH", 0x567 },
  { "ARMENIAN SMALL LETTER ET", 0x568 },
  { "ARMENIAN SMALL LETTER TO", 0x569 },
  { "ARMENIAN SMALL LETTER ZHE", 0x56a },
  { "ARMENIAN SMALL LETTER INI", 0x56b },
  { "ARMENIAN SMALL LETTER LIWN", 0x56c },
  { "ARMENIAN SMALL LETTER XEH", 0x56d },
  { "ARMENIAN SMALL LETTER CA", 0x56e },
  { "ARMENIAN SMALL LETTER KEN", 0x56f },
  { "ARMENIAN SMALL LETTER HO", 0x570 },
  { "ARMENIAN SMALL LETTER JA", 0x571 },
  { "ARMENIAN SMALL LETTER GHAD", 0x572 },
  { "ARMENIAN SMALL LETTER CHEH", 0x573 },
  { "ARMENIAN SMALL LETTER MEN", 0x574 },
  { "ARMENIAN SMALL LETTER YI", 0x575 },
  { "ARMENIAN SMALL LETTER NOW", 0x576 },
  { "ARMENIAN SMALL LETTER SHA", 0x577 },
  { "ARMENIAN SMALL LETTER VO", 0x578 },
  { "ARMENIAN SMALL LETTER CHA", 0x579 },
  { "ARMENIAN SMALL LETTER PEH", 0x57a },
  { "ARMENIAN SMALL LETTER JHEH", 0x57b },
  { "ARMENIAN SMALL LETTER RA", 0x57c },
  { "ARMENIAN SMALL LETTER SEH", 0x57d },
  { "ARMENIAN SMALL LETTER VEW", 0x57e },
  { "ARMENIAN SMALL LETTER TIWN", 0x57f },
  { "ARMENIAN SMALL LETTER REH", 0x580 },
  { "ARMENIAN SMALL LETTER CO", 0x581 },
  { "ARMENIAN SMALL LETTER YIWN", 0x582 },
  { "ARMENIAN SMALL LETTER PIWR", 0x583 },
  { "ARMENIAN SMALL LETTER KEH", 0x584 },
  { "ARMENIAN SMALL LETTER OH", 0x585 },
  { "ARMENIAN SMALL LETTER FEH", 0x586 },
  { "ARMENIAN SMALL LIGATURE ECH YIWN", 0x587 },
  { "ARMENIAN FULL STOP", 0x589 },
  { "ARMENIAN HYPHEN", 0x58a },
  { "HEBREW ACCENT ETNAHTA", 0x591 },
  { "HEBREW ACCENT SEGOL", 0x592 },
  { "HEBREW ACCENT SHALSHELET", 0x593 },
  { "HEBREW ACCENT ZAQEF QATAN", 0x594 },
  { "HEBREW ACCENT ZAQEF GADOL", 0x595 },
  { "HEBREW ACCENT TIPEHA", 0x596 },
  { "HEBREW ACCENT REVIA", 0x597 },
  { "HEBREW ACCENT ZARQA", 0x598 },
  { "HEBREW ACCENT PASHTA", 0x599 },
  { "HEBREW ACCENT YETIV", 0x59a },
  { "HEBREW ACCENT TEVIR", 0x59b },
  { "HEBREW ACCENT GERESH", 0x59c },
  { "HEBREW ACCENT GERESH MUQDAM", 0x59d },
  { "HEBREW ACCENT GERSHAYIM", 0x59e },
  { "HEBREW ACCENT QARNEY PARA", 0x59f },
  { "HEBREW ACCENT TELISHA GEDOLA", 0x5a0 },
  { "HEBREW ACCENT PAZER", 0x5a1 },
  { "HEBREW ACCENT MUNAH", 0x5a3 },
  { "HEBREW ACCENT MAHAPAKH", 0x5a4 },
  { "HEBREW ACCENT MERKHA", 0x5a5 },
  { "HEBREW ACCENT MERKHA KEFULA", 0x5a6 },
  { "HEBREW ACCENT DARGA", 0x5a7 },
  { "HEBREW ACCENT QADMA", 0x5a8 },
  { "HEBREW ACCENT TELISHA QETANA", 0x5a9 },
  { "HEBREW ACCENT YERAH BEN YOMO", 0x5aa },
  { "HEBREW ACCENT OLE", 0x5ab },
  { "HEBREW ACCENT ILUY", 0x5ac },
  { "HEBREW ACCENT DEHI", 0x5ad },
  { "HEBREW ACCENT ZINOR", 0x5ae },
  { "HEBREW MARK MASORA CIRCLE", 0x5af },
  { "HEBREW POINT SHEVA", 0x5b0 },
  { "HEBREW POINT HATAF SEGOL", 0x5b1 },
  { "HEBREW POINT HATAF PATAH", 0x5b2 },
  { "HEBREW POINT HATAF QAMATS", 0x5b3 },
  { "HEBREW POINT HIRIQ", 0x5b4 },
  { "HEBREW POINT TSERE", 0x5b5 },
  { "HEBREW POINT SEGOL", 0x5b6 },
  { "HEBREW POINT PATAH", 0x5b7 },
  { "HEBREW POINT QAMATS", 0x5b8 },
  { "HEBREW POINT HOLAM", 0x5b9 },
  { "HEBREW POINT QUBUTS", 0x5bb },
  { "HEBREW POINT DAGESH OR MAPIQ", 0x5bc },
  { "HEBREW POINT METEG", 0x5bd },
  { "HEBREW PUNCTUATION MAQAF", 0x5be },
  { "HEBREW POINT RAFE", 0x5bf },
  { "HEBREW PUNCTUATION PASEQ", 0x5c0 },
  { "HEBREW POINT SHIN DOT", 0x5c1 },
  { "HEBREW POINT SIN DOT", 0x5c2 },
  { "HEBREW PUNCTUATION SOF PASUQ", 0x5c3 },
  { "HEBREW MARK UPPER DOT", 0x5c4 },
  { "HEBREW LETTER ALEF", 0x5d0 },
  { "HEBREW LETTER BET", 0x5d1 },
  { "HEBREW LETTER GIMEL", 0x5d2 },
  { "HEBREW LETTER DALET", 0x5d3 },
  { "HEBREW LETTER HE", 0x5d4 },
  { "HEBREW LETTER VAV", 0x5d5 },
  { "HEBREW LETTER ZAYIN", 0x5d6 },
  { "HEBREW LETTER HET", 0x5d7 },
  { "HEBREW LETTER TET", 0x5d8 },
  { "HEBREW LETTER YOD", 0x5d9 },
  { "HEBREW LETTER FINAL KAF", 0x5da },
  { "HEBREW LETTER KAF", 0x5db },
  { "HEBREW LETTER LAMED", 0x5dc },
  { "HEBREW LETTER FINAL MEM", 0x5dd },
  { "HEBREW LETTER MEM", 0x5de },
  { "HEBREW LETTER FINAL NUN", 0x5df },
  { "HEBREW LETTER NUN", 0x5e0 },
  { "HEBREW LETTER SAMEKH", 0x5e1 },
  { "HEBREW LETTER AYIN", 0x5e2 },
  { "HEBREW LETTER FINAL PE", 0x5e3 },
  { "HEBREW LETTER PE", 0x5e4 },
  { "HEBREW LETTER FINAL TSADI", 0x5e5 },
  { "HEBREW LETTER TSADI", 0x5e6 },
  { "HEBREW LETTER QOF", 0x5e7 },
  { "HEBREW LETTER RESH", 0x5e8 },
  { "HEBREW LETTER SHIN", 0x5e9 },
  { "HEBREW LETTER TAV", 0x5ea },
  { "HEBREW LIGATURE YIDDISH DOUBLE VAV", 0x5f0 },
  { "HEBREW LIGATURE YIDDISH VAV YOD", 0x5f1 },
  { "HEBREW LIGATURE YIDDISH DOUBLE YOD", 0x5f2 },
  { "HEBREW PUNCTUATION GERESH", 0x5f3 },
  { "HEBREW PUNCTUATION GERSHAYIM", 0x5f4 },
  { "ARABIC COMMA", 0x60c },
  { "ARABIC SEMICOLON", 0x61b },
  { "ARABIC QUESTION MARK", 0x61f },
  { "ARABIC LETTER HAMZA", 0x621 },
  { "ARABIC LETTER ALEF WITH MADDA ABOVE", 0x622 },
  { "ARABIC LETTER ALEF WITH HAMZA ABOVE", 0x623 },
  { "ARABIC LETTER WAW WITH HAMZA ABOVE", 0x624 },
  { "ARABIC LETTER ALEF WITH HAMZA BELOW", 0x625 },
  { "ARABIC LETTER YEH WITH HAMZA ABOVE", 0x626 },
  { "ARABIC LETTER ALEF", 0x627 },
  { "ARABIC LETTER BEH", 0x628 },
  { "ARABIC LETTER TEH MARBUTA", 0x629 },
  { "ARABIC LETTER TEH", 0x62a },
  { "ARABIC LETTER THEH", 0x62b },
  { "ARABIC LETTER JEEM", 0x62c },
  { "ARABIC LETTER HAH", 0x62d },
  { "ARABIC LETTER KHAH", 0x62e },
  { "ARABIC LETTER DAL", 0x62f },
  { "ARABIC LETTER THAL", 0x630 },
  { "ARABIC LETTER REH", 0x631 },
  { "ARABIC LETTER ZAIN", 0x632 },
  { "ARABIC LETTER SEEN", 0x633 },
  { "ARABIC LETTER SHEEN", 0x634 },
  { "ARABIC LETTER SAD", 0x635 },
  { "ARABIC LETTER DAD", 0x636 },
  { "ARABIC LETTER TAH", 0x637 },
  { "ARABIC LETTER ZAH", 0x638 },
  { "ARABIC LETTER AIN", 0x639 },
  { "ARABIC LETTER GHAIN", 0x63a },
  { "ARABIC TATWEEL", 0x640 },
  { "ARABIC LETTER FEH", 0x641 },
  { "ARABIC LETTER QAF", 0x642 },
  { "ARABIC LETTER KAF", 0x643 },
  { "ARABIC LETTER LAM", 0x644 },
  { "ARABIC LETTER MEEM", 0x645 },
  { "ARABIC LETTER NOON", 0x646 },
  { "ARABIC LETTER HEH", 0x647 },
  { "ARABIC LETTER WAW", 0x648 },
  { "ARABIC LETTER ALEF MAKSURA", 0x649 },
  { "ARABIC LETTER YEH", 0x64a },
  { "ARABIC FATHATAN", 0x64b },
  { "ARABIC DAMMATAN", 0x64c },
  { "ARABIC KASRATAN", 0x64d },
  { "ARABIC FATHA", 0x64e },
  { "ARABIC DAMMA", 0x64f },
  { "ARABIC KASRA", 0x650 },
  { "ARABIC SHADDA", 0x651 },
  { "ARABIC SUKUN", 0x652 },
  { "ARABIC MADDAH ABOVE", 0x653 },
  { "ARABIC HAMZA ABOVE", 0x654 },
  { "ARABIC HAMZA BELOW", 0x655 },
  { "ARABIC-INDIC DIGIT ZERO", 0x660 },
  { "ARABIC-INDIC DIGIT ONE", 0x661 },
  { "ARABIC-INDIC DIGIT TWO", 0x662 },
  { "ARABIC-INDIC DIGIT THREE", 0x663 },
  { "ARABIC-INDIC DIGIT FOUR", 0x664 },
  { "ARABIC-INDIC DIGIT FIVE", 0x665 },
  { "ARABIC-INDIC DIGIT SIX", 0x666 },
  { "ARABIC-INDIC DIGIT SEVEN", 0x667 },
  { "ARABIC-INDIC DIGIT EIGHT", 0x668 },
  { "ARABIC-INDIC DIGIT NINE", 0x669 },
  { "ARABIC PERCENT SIGN", 0x66a },
  { "ARABIC DECIMAL SEPARATOR", 0x66b },
  { "ARABIC THOUSANDS SEPARATOR", 0x66c },
  { "ARABIC FIVE POINTED STAR", 0x66d },
  { "ARABIC LETTER SUPERSCRIPT ALEF", 0x670 },
  { "ARABIC LETTER ALEF WASLA", 0x671 },
  { "ARABIC LETTER ALEF WITH WAVY HAMZA ABOVE", 0x672 },
  { "ARABIC LETTER ALEF WITH WAVY HAMZA BELOW", 0x673 },
  { "ARABIC LETTER HIGH HAMZA", 0x674 },
  { "ARABIC LETTER HIGH HAMZA ALEF", 0x675 },
  { "ARABIC LETTER HIGH HAMZA WAW", 0x676 },
  { "ARABIC LETTER U WITH HAMZA ABOVE", 0x677 },
  { "ARABIC LETTER HIGH HAMZA YEH", 0x678 },
  { "ARABIC LETTER TTEH", 0x679 },
  { "ARABIC LETTER TTEHEH", 0x67a },
  { "ARABIC LETTER BEEH", 0x67b },
  { "ARABIC LETTER TEH WITH RING", 0x67c },
  { "ARABIC LETTER TEH WITH THREE DOTS ABOVE DOWNWARDS", 0x67d },
  { "ARABIC LETTER PEH", 0x67e },
  { "ARABIC LETTER TEHEH", 0x67f },
  { "ARABIC LETTER BEHEH", 0x680 },
  { "ARABIC LETTER HAH WITH HAMZA ABOVE", 0x681 },
  { "ARABIC LETTER HAH WITH TWO DOTS VERTICAL ABOVE", 0x682 },
  { "ARABIC LETTER NYEH", 0x683 },
  { "ARABIC LETTER DYEH", 0x684 },
  { "ARABIC LETTER HAH WITH THREE DOTS ABOVE", 0x685 },
  { "ARABIC LETTER TCHEH", 0x686 },
  { "ARABIC LETTER TCHEHEH", 0x687 },
  { "ARABIC LETTER DDAL", 0x688 },
  { "ARABIC LETTER DAL WITH RING", 0x689 },
  { "ARABIC LETTER DAL WITH DOT BELOW", 0x68a },
  { "ARABIC LETTER DAL WITH DOT BELOW AND SMALL TAH", 0x68b },
  { "ARABIC LETTER DAHAL", 0x68c },
  { "ARABIC LETTER DDAHAL", 0x68d },
  { "ARABIC LETTER DUL", 0x68e },
  { "ARABIC LETTER DAL WITH THREE DOTS ABOVE DOWNWARDS", 0x68f },
  { "ARABIC LETTER DAL WITH FOUR DOTS ABOVE", 0x690 },
  { "ARABIC LETTER RREH", 0x691 },
  { "ARABIC LETTER REH WITH SMALL V", 0x692 },
  { "ARABIC LETTER REH WITH RING", 0x693 },
  { "ARABIC LETTER REH WITH DOT BELOW", 0x694 },
  { "ARABIC LETTER REH WITH SMALL V BELOW", 0x695 },
  { "ARABIC LETTER REH WITH DOT BELOW AND DOT ABOVE", 0x696 },
  { "ARABIC LETTER REH WITH TWO DOTS ABOVE", 0x697 },
  { "ARABIC LETTER JEH", 0x698 },
  { "ARABIC LETTER REH WITH FOUR DOTS ABOVE", 0x699 },
  { "ARABIC LETTER SEEN WITH DOT BELOW AND DOT ABOVE", 0x69a },
  { "ARABIC LETTER SEEN WITH THREE DOTS BELOW", 0x69b },
  { "ARABIC LETTER SEEN WITH THREE DOTS BELOW AND THREE DOTS ABOVE", 0x69c },
  { "ARABIC LETTER SAD WITH TWO DOTS BELOW", 0x69d },
  { "ARABIC LETTER SAD WITH THREE DOTS ABOVE", 0x69e },
  { "ARABIC LETTER TAH WITH THREE DOTS ABOVE", 0x69f },
  { "ARABIC LETTER AIN WITH THREE DOTS ABOVE", 0x6a0 },
  { "ARABIC LETTER DOTLESS FEH", 0x6a1 },
  { "ARABIC LETTER FEH WITH DOT MOVED BELOW", 0x6a2 },
  { "ARABIC LETTER FEH WITH DOT BELOW", 0x6a3 },
  { "ARABIC LETTER VEH", 0x6a4 },
  { "ARABIC LETTER FEH WITH THREE DOTS BELOW", 0x6a5 },
  { "ARABIC LETTER PEHEH", 0x6a6 },
  { "ARABIC LETTER QAF WITH DOT ABOVE", 0x6a7 },
  { "ARABIC LETTER QAF WITH THREE DOTS ABOVE", 0x6a8 },
  { "ARABIC LETTER KEHEH", 0x6a9 },
  { "ARABIC LETTER SWASH KAF", 0x6aa },
  { "ARABIC LETTER KAF WITH RING", 0x6ab },
  { "ARABIC LETTER KAF WITH DOT ABOVE", 0x6ac },
  { "ARABIC LETTER NG", 0x6ad },
  { "ARABIC LETTER KAF WITH THREE DOTS BELOW", 0x6ae },
  { "ARABIC LETTER GAF", 0x6af },
  { "ARABIC LETTER GAF WITH RING", 0x6b0 },
  { "ARABIC LETTER NGOEH", 0x6b1 },
  { "ARABIC LETTER GAF WITH TWO DOTS BELOW", 0x6b2 },
  { "ARABIC LETTER GUEH", 0x6b3 },
  { "ARABIC LETTER GAF WITH THREE DOTS ABOVE", 0x6b4 },
  { "ARABIC LETTER LAM WITH SMALL V", 0x6b5 },
  { "ARABIC LETTER LAM WITH DOT ABOVE", 0x6b6 },
  { "ARABIC LETTER LAM WITH THREE DOTS ABOVE", 0x6b7 },
  { "ARABIC LETTER LAM WITH THREE DOTS BELOW", 0x6b8 },
  { "ARABIC LETTER NOON WITH DOT BELOW", 0x6b9 },
  { "ARABIC LETTER NOON GHUNNA", 0x6ba },
  { "ARABIC LETTER RNOON", 0x6bb },
  { "ARABIC LETTER NOON WITH RING", 0x6bc },
  { "ARABIC LETTER NOON WITH THREE DOTS ABOVE", 0x6bd },
  { "ARABIC LETTER HEH DOACHASHMEE", 0x6be },
  { "ARABIC LETTER TCHEH WITH DOT ABOVE", 0x6bf },
  { "ARABIC LETTER HEH WITH YEH ABOVE", 0x6c0 },
  { "ARABIC LETTER HEH GOAL", 0x6c1 },
  { "ARABIC LETTER HEH GOAL WITH HAMZA ABOVE", 0x6c2 },
  { "ARABIC LETTER TEH MARBUTA GOAL", 0x6c3 },
  { "ARABIC LETTER WAW WITH RING", 0x6c4 },
  { "ARABIC LETTER KIRGHIZ OE", 0x6c5 },
  { "ARABIC LETTER OE", 0x6c6 },
  { "ARABIC LETTER U", 0x6c7 },
  { "ARABIC LETTER YU", 0x6c8 },
  { "ARABIC LETTER KIRGHIZ YU", 0x6c9 },
  { "ARABIC LETTER WAW WITH TWO DOTS ABOVE", 0x6ca },
  { "ARABIC LETTER VE", 0x6cb },
  { "ARABIC LETTER FARSI YEH", 0x6cc },
  { "ARABIC LETTER YEH WITH TAIL", 0x6cd },
  { "ARABIC LETTER YEH WITH SMALL V", 0x6ce },
  { "ARABIC LETTER WAW WITH DOT ABOVE", 0x6cf },
  { "ARABIC LETTER E", 0x6d0 },
  { "ARABIC LETTER YEH WITH THREE DOTS BELOW", 0x6d1 },
  { "ARABIC LETTER YEH BARREE", 0x6d2 },
  { "ARABIC LETTER YEH BARREE WITH HAMZA ABOVE", 0x6d3 },
  { "ARABIC FULL STOP", 0x6d4 },
  { "ARABIC LETTER AE", 0x6d5 },
  { "ARABIC SMALL HIGH LIGATURE SAD WITH LAM WITH ALEF MAKSURA", 0x6d6 },
  { "ARABIC SMALL HIGH LIGATURE QAF WITH LAM WITH ALEF MAKSURA", 0x6d7 },
  { "ARABIC SMALL HIGH MEEM INITIAL FORM", 0x6d8 },
  { "ARABIC SMALL HIGH LAM ALEF", 0x6d9 },
  { "ARABIC SMALL HIGH JEEM", 0x6da },
  { "ARABIC SMALL HIGH THREE DOTS", 0x6db },
  { "ARABIC SMALL HIGH SEEN", 0x6dc },
  { "ARABIC END OF AYAH", 0x6dd },
  { "ARABIC START OF RUB EL HIZB", 0x6de },
  { "ARABIC SMALL HIGH ROUNDED ZERO", 0x6df },
  { "ARABIC SMALL HIGH UPRIGHT RECTANGULAR ZERO", 0x6e0 },
  { "ARABIC SMALL HIGH DOTLESS HEAD OF KHAH", 0x6e1 },
  { "ARABIC SMALL HIGH MEEM ISOLATED FORM", 0x6e2 },
  { "ARABIC SMALL LOW SEEN", 0x6e3 },
  { "ARABIC SMALL HIGH MADDA", 0x6e4 },
  { "ARABIC SMALL WAW", 0x6e5 },
  { "ARABIC SMALL YEH", 0x6e6 },
  { "ARABIC SMALL HIGH YEH", 0x6e7 },
  { "ARABIC SMALL HIGH NOON", 0x6e8 },
  { "ARABIC PLACE OF SAJDAH", 0x6e9 },
  { "ARABIC EMPTY CENTRE LOW STOP", 0x6ea },
  { "ARABIC EMPTY CENTRE HIGH STOP", 0x6eb },
  { "ARABIC ROUNDED HIGH STOP WITH FILLED CENTRE", 0x6ec },
  { "ARABIC SMALL LOW MEEM", 0x6ed },
  { "EXTENDED ARABIC-INDIC DIGIT ZERO", 0x6f0 },
  { "EXTENDED ARABIC-INDIC DIGIT ONE", 0x6f1 },
  { "EXTENDED ARABIC-INDIC DIGIT TWO", 0x6f2 },
  { "EXTENDED ARABIC-INDIC DIGIT THREE", 0x6f3 },
  { "EXTENDED ARABIC-INDIC DIGIT FOUR", 0x6f4 },
  { "EXTENDED ARABIC-INDIC DIGIT FIVE", 0x6f5 },
  { "EXTENDED ARABIC-INDIC DIGIT SIX", 0x6f6 },
  { "EXTENDED ARABIC-INDIC DIGIT SEVEN", 0x6f7 },
  { "EXTENDED ARABIC-INDIC DIGIT EIGHT", 0x6f8 },
  { "EXTENDED ARABIC-INDIC DIGIT NINE", 0x6f9 },
  { "ARABIC LETTER SHEEN WITH DOT BELOW", 0x6fa },
  { "ARABIC LETTER DAD WITH DOT BELOW", 0x6fb },
  { "ARABIC LETTER GHAIN WITH DOT BELOW", 0x6fc },
  { "ARABIC SIGN SINDHI AMPERSAND", 0x6fd },
  { "ARABIC SIGN SINDHI POSTPOSITION MEN", 0x6fe },
  { "SYRIAC END OF PARAGRAPH", 0x700 },
  { "SYRIAC SUPRALINEAR FULL STOP", 0x701 },
  { "SYRIAC SUBLINEAR FULL STOP", 0x702 },
  { "SYRIAC SUPRALINEAR COLON", 0x703 },
  { "SYRIAC SUBLINEAR COLON", 0x704 },
  { "SYRIAC HORIZONTAL COLON", 0x705 },
  { "SYRIAC COLON SKEWED LEFT", 0x706 },
  { "SYRIAC COLON SKEWED RIGHT", 0x707 },
  { "SYRIAC SUPRALINEAR COLON SKEWED LEFT", 0x708 },
  { "SYRIAC SUBLINEAR COLON SKEWED RIGHT", 0x709 },
  { "SYRIAC CONTRACTION", 0x70a },
  { "SYRIAC HARKLEAN OBELUS", 0x70b },
  { "SYRIAC HARKLEAN METOBELUS", 0x70c },
  { "SYRIAC HARKLEAN ASTERISCUS", 0x70d },
  { "SYRIAC ABBREVIATION MARK", 0x70f },
  { "SYRIAC LETTER ALAPH", 0x710 },
  { "SYRIAC LETTER SUPERSCRIPT ALAPH", 0x711 },
  { "SYRIAC LETTER BETH", 0x712 },
  { "SYRIAC LETTER GAMAL", 0x713 },
  { "SYRIAC LETTER GAMAL GARSHUNI", 0x714 },
  { "SYRIAC LETTER DALATH", 0x715 },
  { "SYRIAC LETTER DOTLESS DALATH RISH", 0x716 },
  { "SYRIAC LETTER HE", 0x717 },
  { "SYRIAC LETTER WAW", 0x718 },
  { "SYRIAC LETTER ZAIN", 0x719 },
  { "SYRIAC LETTER HETH", 0x71a },
  { "SYRIAC LETTER TETH", 0x71b },
  { "SYRIAC LETTER TETH GARSHUNI", 0x71c },
  { "SYRIAC LETTER YUDH", 0x71d },
  { "SYRIAC LETTER YUDH HE", 0x71e },
  { "SYRIAC LETTER KAPH", 0x71f },
  { "SYRIAC LETTER LAMADH", 0x720 },
  { "SYRIAC LETTER MIM", 0x721 },
  { "SYRIAC LETTER NUN", 0x722 },
  { "SYRIAC LETTER SEMKATH", 0x723 },
  { "SYRIAC LETTER FINAL SEMKATH", 0x724 },
  { "SYRIAC LETTER E", 0x725 },
  { "SYRIAC LETTER PE", 0x726 },
  { "SYRIAC LETTER REVERSED PE", 0x727 },
  { "SYRIAC LETTER SADHE", 0x728 },
  { "SYRIAC LETTER QAPH", 0x729 },
  { "SYRIAC LETTER RISH", 0x72a },
  { "SYRIAC LETTER SHIN", 0x72b },
  { "SYRIAC LETTER TAW", 0x72c },
  { "SYRIAC PTHAHA ABOVE", 0x730 },
  { "SYRIAC PTHAHA BELOW", 0x731 },
  { "SYRIAC PTHAHA DOTTED", 0x732 },
  { "SYRIAC ZQAPHA ABOVE", 0x733 },
  { "SYRIAC ZQAPHA BELOW", 0x734 },
  { "SYRIAC ZQAPHA DOTTED", 0x735 },
  { "SYRIAC RBASA ABOVE", 0x736 },
  { "SYRIAC RBASA BELOW", 0x737 },
  { "SYRIAC DOTTED ZLAMA HORIZONTAL", 0x738 },
  { "SYRIAC DOTTED ZLAMA ANGULAR", 0x739 },
  { "SYRIAC HBASA ABOVE", 0x73a },
  { "SYRIAC HBASA BELOW", 0x73b },
  { "SYRIAC HBASA-ESASA DOTTED", 0x73c },
  { "SYRIAC ESASA ABOVE", 0x73d },
  { "SYRIAC ESASA BELOW", 0x73e },
  { "SYRIAC RWAHA", 0x73f },
  { "SYRIAC FEMININE DOT", 0x740 },
  { "SYRIAC QUSHSHAYA", 0x741 },
  { "SYRIAC RUKKAKHA", 0x742 },
  { "SYRIAC TWO VERTICAL DOTS ABOVE", 0x743 },
  { "SYRIAC TWO VERTICAL DOTS BELOW", 0x744 },
  { "SYRIAC THREE DOTS ABOVE", 0x745 },
  { "SYRIAC THREE DOTS BELOW", 0x746 },
  { "SYRIAC OBLIQUE LINE ABOVE", 0x747 },
  { "SYRIAC OBLIQUE LINE BELOW", 0x748 },
  { "SYRIAC MUSIC", 0x749 },
  { "SYRIAC BARREKH", 0x74a },
  { "THAANA LETTER HAA", 0x780 },
  { "THAANA LETTER SHAVIYANI", 0x781 },
  { "THAANA LETTER NOONU", 0x782 },
  { "THAANA LETTER RAA", 0x783 },
  { "THAANA LETTER BAA", 0x784 },
  { "THAANA LETTER LHAVIYANI", 0x785 },
  { "THAANA LETTER KAAFU", 0x786 },
  { "THAANA LETTER ALIFU", 0x787 },
  { "THAANA LETTER VAAVU", 0x788 },
  { "THAANA LETTER MEEMU", 0x789 },
  { "THAANA LETTER FAAFU", 0x78a },
  { "THAANA LETTER DHAALU", 0x78b },
  { "THAANA LETTER THAA", 0x78c },
  { "THAANA LETTER LAAMU", 0x78d },
  { "THAANA LETTER GAAFU", 0x78e },
  { "THAANA LETTER GNAVIYANI", 0x78f },
  { "THAANA LETTER SEENU", 0x790 },
  { "THAANA LETTER DAVIYANI", 0x791 },
  { "THAANA LETTER ZAVIYANI", 0x792 },
  { "THAANA LETTER TAVIYANI", 0x793 },
  { "THAANA LETTER YAA", 0x794 },
  { "THAANA LETTER PAVIYANI", 0x795 },
  { "THAANA LETTER JAVIYANI", 0x796 },
  { "THAANA LETTER CHAVIYANI", 0x797 },
  { "THAANA LETTER TTAA", 0x798 },
  { "THAANA LETTER HHAA", 0x799 },
  { "THAANA LETTER KHAA", 0x79a },
  { "THAANA LETTER THAALU", 0x79b },
  { "THAANA LETTER ZAA", 0x79c },
  { "THAANA LETTER SHEENU", 0x79d },
  { "THAANA LETTER SAADHU", 0x79e },
  { "THAANA LETTER DAADHU", 0x79f },
  { "THAANA LETTER TO", 0x7a0 },
  { "THAANA LETTER ZO", 0x7a1 },
  { "THAANA LETTER AINU", 0x7a2 },
  { "THAANA LETTER GHAINU", 0x7a3 },
  { "THAANA LETTER QAAFU", 0x7a4 },
  { "THAANA LETTER WAAVU", 0x7a5 },
  { "THAANA ABAFILI", 0x7a6 },
  { "THAANA AABAAFILI", 0x7a7 },
  { "THAANA IBIFILI", 0x7a8 },
  { "THAANA EEBEEFILI", 0x7a9 },
  { "THAANA UBUFILI", 0x7aa },
  { "THAANA OOBOOFILI", 0x7ab },
  { "THAANA EBEFILI", 0x7ac },
  { "THAANA EYBEYFILI", 0x7ad },
  { "THAANA OBOFILI", 0x7ae },
  { "THAANA OABOAFILI", 0x7af },
  { "THAANA SUKUN", 0x7b0 },
  { "DEVANAGARI SIGN CANDRABINDU", 0x901 },
  { "DEVANAGARI SIGN ANUSVARA", 0x902 },
  { "DEVANAGARI SIGN VISARGA", 0x903 },
  { "DEVANAGARI LETTER A", 0x905 },
  { "DEVANAGARI LETTER AA", 0x906 },
  { "DEVANAGARI LETTER I", 0x907 },
  { "DEVANAGARI LETTER II", 0x908 },
  { "DEVANAGARI LETTER U", 0x909 },
  { "DEVANAGARI LETTER UU", 0x90a },
  { "DEVANAGARI LETTER VOCALIC R", 0x90b },
  { "DEVANAGARI LETTER VOCALIC L", 0x90c },
  { "DEVANAGARI LETTER CANDRA E", 0x90d },
  { "DEVANAGARI LETTER SHORT E", 0x90e },
  { "DEVANAGARI LETTER E", 0x90f },
  { "DEVANAGARI LETTER AI", 0x910 },
  { "DEVANAGARI LETTER CANDRA O", 0x911 },
  { "DEVANAGARI LETTER SHORT O", 0x912 },
  { "DEVANAGARI LETTER O", 0x913 },
  { "DEVANAGARI LETTER AU", 0x914 },
  { "DEVANAGARI LETTER KA", 0x915 },
  { "DEVANAGARI LETTER KHA", 0x916 },
  { "DEVANAGARI LETTER GA", 0x917 },
  { "DEVANAGARI LETTER GHA", 0x918 },
  { "DEVANAGARI LETTER NGA", 0x919 },
  { "DEVANAGARI LETTER CA", 0x91a },
  { "DEVANAGARI LETTER CHA", 0x91b },
  { "DEVANAGARI LETTER JA", 0x91c },
  { "DEVANAGARI LETTER JHA", 0x91d },
  { "DEVANAGARI LETTER NYA", 0x91e },
  { "DEVANAGARI LETTER TTA", 0x91f },
  { "DEVANAGARI LETTER TTHA", 0x920 },
  { "DEVANAGARI LETTER DDA", 0x921 },
  { "DEVANAGARI LETTER DDHA", 0x922 },
  { "DEVANAGARI LETTER NNA", 0x923 },
  { "DEVANAGARI LETTER TA", 0x924 },
  { "DEVANAGARI LETTER THA", 0x925 },
  { "DEVANAGARI LETTER DA", 0x926 },
  { "DEVANAGARI LETTER DHA", 0x927 },
  { "DEVANAGARI LETTER NA", 0x928 },
  { "DEVANAGARI LETTER NNNA", 0x929 },
  { "DEVANAGARI LETTER PA", 0x92a },
  { "DEVANAGARI LETTER PHA", 0x92b },
  { "DEVANAGARI LETTER BA", 0x92c },
  { "DEVANAGARI LETTER BHA", 0x92d },
  { "DEVANAGARI LETTER MA", 0x92e },
  { "DEVANAGARI LETTER YA", 0x92f },
  { "DEVANAGARI LETTER RA", 0x930 },
  { "DEVANAGARI LETTER RRA", 0x931 },
  { "DEVANAGARI LETTER LA", 0x932 },
  { "DEVANAGARI LETTER LLA", 0x933 },
  { "DEVANAGARI LETTER LLLA", 0x934 },
  { "DEVANAGARI LETTER VA", 0x935 },
  { "DEVANAGARI LETTER SHA", 0x936 },
  { "DEVANAGARI LETTER SSA", 0x937 },
  { "DEVANAGARI LETTER SA", 0x938 },
  { "DEVANAGARI LETTER HA", 0x939 },
  { "DEVANAGARI SIGN NUKTA", 0x93c },
  { "DEVANAGARI SIGN AVAGRAHA", 0x93d },
  { "DEVANAGARI VOWEL SIGN AA", 0x93e },
  { "DEVANAGARI VOWEL SIGN I", 0x93f },
  { "DEVANAGARI VOWEL SIGN II", 0x940 },
  { "DEVANAGARI VOWEL SIGN U", 0x941 },
  { "DEVANAGARI VOWEL SIGN UU", 0x942 },
  { "DEVANAGARI VOWEL SIGN VOCALIC R", 0x943 },
  { "DEVANAGARI VOWEL SIGN VOCALIC RR", 0x944 },
  { "DEVANAGARI VOWEL SIGN CANDRA E", 0x945 },
  { "DEVANAGARI VOWEL SIGN SHORT E", 0x946 },
  { "DEVANAGARI VOWEL SIGN E", 0x947 },
  { "DEVANAGARI VOWEL SIGN AI", 0x948 },
  { "DEVANAGARI VOWEL SIGN CANDRA O", 0x949 },
  { "DEVANAGARI VOWEL SIGN SHORT O", 0x94a },
  { "DEVANAGARI VOWEL SIGN O", 0x94b },
  { "DEVANAGARI VOWEL SIGN AU", 0x94c },
  { "DEVANAGARI SIGN VIRAMA", 0x94d },
  { "DEVANAGARI OM", 0x950 },
  { "DEVANAGARI STRESS SIGN UDATTA", 0x951 },
  { "DEVANAGARI STRESS SIGN ANUDATTA", 0x952 },
  { "DEVANAGARI GRAVE ACCENT", 0x953 },
  { "DEVANAGARI ACUTE ACCENT", 0x954 },
  { "DEVANAGARI LETTER QA", 0x958 },
  { "DEVANAGARI LETTER KHHA", 0x959 },
  { "DEVANAGARI LETTER GHHA", 0x95a },
  { "DEVANAGARI LETTER ZA", 0x95b },
  { "DEVANAGARI LETTER DDDHA", 0x95c },
  { "DEVANAGARI LETTER RHA", 0x95d },
  { "DEVANAGARI LETTER FA", 0x95e },
  { "DEVANAGARI LETTER YYA", 0x95f },
  { "DEVANAGARI LETTER VOCALIC RR", 0x960 },
  { "DEVANAGARI LETTER VOCALIC LL", 0x961 },
  { "DEVANAGARI VOWEL SIGN VOCALIC L", 0x962 },
  { "DEVANAGARI VOWEL SIGN VOCALIC LL", 0x963 },
  { "DEVANAGARI DANDA", 0x964 },
  { "DEVANAGARI DOUBLE DANDA", 0x965 },
  { "DEVANAGARI DIGIT ZERO", 0x966 },
  { "DEVANAGARI DIGIT ONE", 0x967 },
  { "DEVANAGARI DIGIT TWO", 0x968 },
  { "DEVANAGARI DIGIT THREE", 0x969 },
  { "DEVANAGARI DIGIT FOUR", 0x96a },
  { "DEVANAGARI DIGIT FIVE", 0x96b },
  { "DEVANAGARI DIGIT SIX", 0x96c },
  { "DEVANAGARI DIGIT SEVEN", 0x96d },
  { "DEVANAGARI DIGIT EIGHT", 0x96e },
  { "DEVANAGARI DIGIT NINE", 0x96f },
  { "DEVANAGARI ABBREVIATION SIGN", 0x970 },
  { "BENGALI SIGN CANDRABINDU", 0x981 },
  { "BENGALI SIGN ANUSVARA", 0x982 },
  { "BENGALI SIGN VISARGA", 0x983 },
  { "BENGALI LETTER A", 0x985 },
  { "BENGALI LETTER AA", 0x986 },
  { "BENGALI LETTER I", 0x987 },
  { "BENGALI LETTER II", 0x988 },
  { "BENGALI LETTER U", 0x989 },
  { "BENGALI LETTER UU", 0x98a },
  { "BENGALI LETTER VOCALIC R", 0x98b },
  { "BENGALI LETTER VOCALIC L", 0x98c },
  { "BENGALI LETTER E", 0x98f },
  { "BENGALI LETTER AI", 0x990 },
  { "BENGALI LETTER O", 0x993 },
  { "BENGALI LETTER AU", 0x994 },
  { "BENGALI LETTER KA", 0x995 },
  { "BENGALI LETTER KHA", 0x996 },
  { "BENGALI LETTER GA", 0x997 },
  { "BENGALI LETTER GHA", 0x998 },
  { "BENGALI LETTER NGA", 0x999 },
  { "BENGALI LETTER CA", 0x99a },
  { "BENGALI LETTER CHA", 0x99b },
  { "BENGALI LETTER JA", 0x99c },
  { "BENGALI LETTER JHA", 0x99d },
  { "BENGALI LETTER NYA", 0x99e },
  { "BENGALI LETTER TTA", 0x99f },
  { "BENGALI LETTER TTHA", 0x9a0 },
  { "BENGALI LETTER DDA", 0x9a1 },
  { "BENGALI LETTER DDHA", 0x9a2 },
  { "BENGALI LETTER NNA", 0x9a3 },
  { "BENGALI LETTER TA", 0x9a4 },
  { "BENGALI LETTER THA", 0x9a5 },
  { "BENGALI LETTER DA", 0x9a6 },
  { "BENGALI LETTER DHA", 0x9a7 },
  { "BENGALI LETTER NA", 0x9a8 },
  { "BENGALI LETTER PA", 0x9aa },
  { "BENGALI LETTER PHA", 0x9ab },
  { "BENGALI LETTER BA", 0x9ac },
  { "BENGALI LETTER BHA", 0x9ad },
  { "BENGALI LETTER MA", 0x9ae },
  { "BENGALI LETTER YA", 0x9af },
  { "BENGALI LETTER RA", 0x9b0 },
  { "BENGALI LETTER LA", 0x9b2 },
  { "BENGALI LETTER SHA", 0x9b6 },
  { "BENGALI LETTER SSA", 0x9b7 },
  { "BENGALI LETTER SA", 0x9b8 },
  { "BENGALI LETTER HA", 0x9b9 },
  { "BENGALI SIGN NUKTA", 0x9bc },
  { "BENGALI VOWEL SIGN AA", 0x9be },
  { "BENGALI VOWEL SIGN I", 0x9bf },
  { "BENGALI VOWEL SIGN II", 0x9c0 },
  { "BENGALI VOWEL SIGN U", 0x9c1 },
  { "BENGALI VOWEL SIGN UU", 0x9c2 },
  { "BENGALI VOWEL SIGN VOCALIC R", 0x9c3 },
  { "BENGALI VOWEL SIGN VOCALIC RR", 0x9c4 },
  { "BENGALI VOWEL SIGN E", 0x9c7 },
  { "BENGALI VOWEL SIGN AI", 0x9c8 },
  { "BENGALI VOWEL SIGN O", 0x9cb },
  { "BENGALI VOWEL SIGN AU", 0x9cc },
  { "BENGALI SIGN VIRAMA", 0x9cd },
  { "BENGALI AU LENGTH MARK", 0x9d7 },
  { "BENGALI LETTER RRA", 0x9dc },
  { "BENGALI LETTER RHA", 0x9dd },
  { "BENGALI LETTER YYA", 0x9df },
  { "BENGALI LETTER VOCALIC RR", 0x9e0 },
  { "BENGALI LETTER VOCALIC LL", 0x9e1 },
  { "BENGALI VOWEL SIGN VOCALIC L", 0x9e2 },
  { "BENGALI VOWEL SIGN VOCALIC LL", 0x9e3 },
  { "BENGALI DIGIT ZERO", 0x9e6 },
  { "BENGALI DIGIT ONE", 0x9e7 },
  { "BENGALI DIGIT TWO", 0x9e8 },
  { "BENGALI DIGIT THREE", 0x9e9 },
  { "BENGALI DIGIT FOUR", 0x9ea },
  { "BENGALI DIGIT FIVE", 0x9eb },
  { "BENGALI DIGIT SIX", 0x9ec },
  { "BENGALI DIGIT SEVEN", 0x9ed },
  { "BENGALI DIGIT EIGHT", 0x9ee },
  { "BENGALI DIGIT NINE", 0x9ef },
  { "BENGALI LETTER RA WITH MIDDLE DIAGONAL", 0x9f0 },
  { "BENGALI LETTER RA WITH LOWER DIAGONAL", 0x9f1 },
  { "BENGALI RUPEE MARK", 0x9f2 },
  { "BENGALI RUPEE SIGN", 0x9f3 },
  { "BENGALI CURRENCY NUMERATOR ONE", 0x9f4 },
  { "BENGALI CURRENCY NUMERATOR TWO", 0x9f5 },
  { "BENGALI CURRENCY NUMERATOR THREE", 0x9f6 },
  { "BENGALI CURRENCY NUMERATOR FOUR", 0x9f7 },
  { "BENGALI CURRENCY NUMERATOR ONE LESS THAN THE DENOMINATOR", 0x9f8 },
  { "BENGALI CURRENCY DENOMINATOR SIXTEEN", 0x9f9 },
  { "BENGALI ISSHAR", 0x9fa },
  { "GURMUKHI SIGN BINDI", 0xa02 },
  { "GURMUKHI LETTER A", 0xa05 },
  { "GURMUKHI LETTER AA", 0xa06 },
  { "GURMUKHI LETTER I", 0xa07 },
  { "GURMUKHI LETTER II", 0xa08 },
  { "GURMUKHI LETTER U", 0xa09 },
  { "GURMUKHI LETTER UU", 0xa0a },
  { "GURMUKHI LETTER EE", 0xa0f },
  { "GURMUKHI LETTER AI", 0xa10 },
  { "GURMUKHI LETTER OO", 0xa13 },
  { "GURMUKHI LETTER AU", 0xa14 },
  { "GURMUKHI LETTER KA", 0xa15 },
  { "GURMUKHI LETTER KHA", 0xa16 },
  { "GURMUKHI LETTER GA", 0xa17 },
  { "GURMUKHI LETTER GHA", 0xa18 },
  { "GURMUKHI LETTER NGA", 0xa19 },
  { "GURMUKHI LETTER CA", 0xa1a },
  { "GURMUKHI LETTER CHA", 0xa1b },
  { "GURMUKHI LETTER JA", 0xa1c },
  { "GURMUKHI LETTER JHA", 0xa1d },
  { "GURMUKHI LETTER NYA", 0xa1e },
  { "GURMUKHI LETTER TTA", 0xa1f },
  { "GURMUKHI LETTER TTHA", 0xa20 },
  { "GURMUKHI LETTER DDA", 0xa21 },
  { "GURMUKHI LETTER DDHA", 0xa22 },
  { "GURMUKHI LETTER NNA", 0xa23 },
  { "GURMUKHI LETTER TA", 0xa24 },
  { "GURMUKHI LETTER THA", 0xa25 },
  { "GURMUKHI LETTER DA", 0xa26 },
  { "GURMUKHI LETTER DHA", 0xa27 },
  { "GURMUKHI LETTER NA", 0xa28 },
  { "GURMUKHI LETTER PA", 0xa2a },
  { "GURMUKHI LETTER PHA", 0xa2b },
  { "GURMUKHI LETTER BA", 0xa2c },
  { "GURMUKHI LETTER BHA", 0xa2d },
  { "GURMUKHI LETTER MA", 0xa2e },
  { "GURMUKHI LETTER YA", 0xa2f },
  { "GURMUKHI LETTER RA", 0xa30 },
  { "GURMUKHI LETTER LA", 0xa32 },
  { "GURMUKHI LETTER LLA", 0xa33 },
  { "GURMUKHI LETTER VA", 0xa35 },
  { "GURMUKHI LETTER SHA", 0xa36 },
  { "GURMUKHI LETTER SA", 0xa38 },
  { "GURMUKHI LETTER HA", 0xa39 },
  { "GURMUKHI SIGN NUKTA", 0xa3c },
  { "GURMUKHI VOWEL SIGN AA", 0xa3e },
  { "GURMUKHI VOWEL SIGN I", 0xa3f },
  { "GURMUKHI VOWEL SIGN II", 0xa40 },
  { "GURMUKHI VOWEL SIGN U", 0xa41 },
  { "GURMUKHI VOWEL SIGN UU", 0xa42 },
  { "GURMUKHI VOWEL SIGN EE", 0xa47 },
  { "GURMUKHI VOWEL SIGN AI", 0xa48 },
  { "GURMUKHI VOWEL SIGN OO", 0xa4b },
  { "GURMUKHI VOWEL SIGN AU", 0xa4c },
  { "GURMUKHI SIGN VIRAMA", 0xa4d },
  { "GURMUKHI LETTER KHHA", 0xa59 },
  { "GURMUKHI LETTER GHHA", 0xa5a },
  { "GURMUKHI LETTER ZA", 0xa5b },
  { "GURMUKHI LETTER RRA", 0xa5c },
  { "GURMUKHI LETTER FA", 0xa5e },
  { "GURMUKHI DIGIT ZERO", 0xa66 },
  { "GURMUKHI DIGIT ONE", 0xa67 },
  { "GURMUKHI DIGIT TWO", 0xa68 },
  { "GURMUKHI DIGIT THREE", 0xa69 },
  { "GURMUKHI DIGIT FOUR", 0xa6a },
  { "GURMUKHI DIGIT FIVE", 0xa6b },
  { "GURMUKHI DIGIT SIX", 0xa6c },
  { "GURMUKHI DIGIT SEVEN", 0xa6d },
  { "GURMUKHI DIGIT EIGHT", 0xa6e },
  { "GURMUKHI DIGIT NINE", 0xa6f },
  { "GURMUKHI TIPPI", 0xa70 },
  { "GURMUKHI ADDAK", 0xa71 },
  { "GURMUKHI IRI", 0xa72 },
  { "GURMUKHI URA", 0xa73 },
  { "GURMUKHI EK ONKAR", 0xa74 },
  { "GUJARATI SIGN CANDRABINDU", 0xa81 },
  { "GUJARATI SIGN ANUSVARA", 0xa82 },
  { "GUJARATI SIGN VISARGA", 0xa83 },
  { "GUJARATI LETTER A", 0xa85 },
  { "GUJARATI LETTER AA", 0xa86 },
  { "GUJARATI LETTER I", 0xa87 },
  { "GUJARATI LETTER II", 0xa88 },
  { "GUJARATI LETTER U", 0xa89 },
  { "GUJARATI LETTER UU", 0xa8a },
  { "GUJARATI LETTER VOCALIC R", 0xa8b },
  { "GUJARATI VOWEL CANDRA E", 0xa8d },
  { "GUJARATI LETTER E", 0xa8f },
  { "GUJARATI LETTER AI", 0xa90 },
  { "GUJARATI VOWEL CANDRA O", 0xa91 },
  { "GUJARATI LETTER O", 0xa93 },
  { "GUJARATI LETTER AU", 0xa94 },
  { "GUJARATI LETTER KA", 0xa95 },
  { "GUJARATI LETTER KHA", 0xa96 },
  { "GUJARATI LETTER GA", 0xa97 },
  { "GUJARATI LETTER GHA", 0xa98 },
  { "GUJARATI LETTER NGA", 0xa99 },
  { "GUJARATI LETTER CA", 0xa9a },
  { "GUJARATI LETTER CHA", 0xa9b },
  { "GUJARATI LETTER JA", 0xa9c },
  { "GUJARATI LETTER JHA", 0xa9d },
  { "GUJARATI LETTER NYA", 0xa9e },
  { "GUJARATI LETTER TTA", 0xa9f },
  { "GUJARATI LETTER TTHA", 0xaa0 },
  { "GUJARATI LETTER DDA", 0xaa1 },
  { "GUJARATI LETTER DDHA", 0xaa2 },
  { "GUJARATI LETTER NNA", 0xaa3 },
  { "GUJARATI LETTER TA", 0xaa4 },
  { "GUJARATI LETTER THA", 0xaa5 },
  { "GUJARATI LETTER DA", 0xaa6 },
  { "GUJARATI LETTER DHA", 0xaa7 },
  { "GUJARATI LETTER NA", 0xaa8 },
  { "GUJARATI LETTER PA", 0xaaa },
  { "GUJARATI LETTER PHA", 0xaab },
  { "GUJARATI LETTER BA", 0xaac },
  { "GUJARATI LETTER BHA", 0xaad },
  { "GUJARATI LETTER MA", 0xaae },
  { "GUJARATI LETTER YA", 0xaaf },
  { "GUJARATI LETTER RA", 0xab0 },
  { "GUJARATI LETTER LA", 0xab2 },
  { "GUJARATI LETTER LLA", 0xab3 },
  { "GUJARATI LETTER VA", 0xab5 },
  { "GUJARATI LETTER SHA", 0xab6 },
  { "GUJARATI LETTER SSA", 0xab7 },
  { "GUJARATI LETTER SA", 0xab8 },
  { "GUJARATI LETTER HA", 0xab9 },
  { "GUJARATI SIGN NUKTA", 0xabc },
  { "GUJARATI SIGN AVAGRAHA", 0xabd },
  { "GUJARATI VOWEL SIGN AA", 0xabe },
  { "GUJARATI VOWEL SIGN I", 0xabf },
  { "GUJARATI VOWEL SIGN II", 0xac0 },
  { "GUJARATI VOWEL SIGN U", 0xac1 },
  { "GUJARATI VOWEL SIGN UU", 0xac2 },
  { "GUJARATI VOWEL SIGN VOCALIC R", 0xac3 },
  { "GUJARATI VOWEL SIGN VOCALIC RR", 0xac4 },
  { "GUJARATI VOWEL SIGN CANDRA E", 0xac5 },
  { "GUJARATI VOWEL SIGN E", 0xac7 },
  { "GUJARATI VOWEL SIGN AI", 0xac8 },
  { "GUJARATI VOWEL SIGN CANDRA O", 0xac9 },
  { "GUJARATI VOWEL SIGN O", 0xacb },
  { "GUJARATI VOWEL SIGN AU", 0xacc },
  { "GUJARATI SIGN VIRAMA", 0xacd },
  { "GUJARATI OM", 0xad0 },
  { "GUJARATI LETTER VOCALIC RR", 0xae0 },
  { "GUJARATI DIGIT ZERO", 0xae6 },
  { "GUJARATI DIGIT ONE", 0xae7 },
  { "GUJARATI DIGIT TWO", 0xae8 },
  { "GUJARATI DIGIT THREE", 0xae9 },
  { "GUJARATI DIGIT FOUR", 0xaea },
  { "GUJARATI DIGIT FIVE", 0xaeb },
  { "GUJARATI DIGIT SIX", 0xaec },
  { "GUJARATI DIGIT SEVEN", 0xaed },
  { "GUJARATI DIGIT EIGHT", 0xaee },
  { "GUJARATI DIGIT NINE", 0xaef },
  { "ORIYA SIGN CANDRABINDU", 0xb01 },
  { "ORIYA SIGN ANUSVARA", 0xb02 },
  { "ORIYA SIGN VISARGA", 0xb03 },
  { "ORIYA LETTER A", 0xb05 },
  { "ORIYA LETTER AA", 0xb06 },
  { "ORIYA LETTER I", 0xb07 },
  { "ORIYA LETTER II", 0xb08 },
  { "ORIYA LETTER U", 0xb09 },
  { "ORIYA LETTER UU", 0xb0a },
  { "ORIYA LETTER VOCALIC R", 0xb0b },
  { "ORIYA LETTER VOCALIC L", 0xb0c },
  { "ORIYA LETTER E", 0xb0f },
  { "ORIYA LETTER AI", 0xb10 },
  { "ORIYA LETTER O", 0xb13 },
  { "ORIYA LETTER AU", 0xb14 },
  { "ORIYA LETTER KA", 0xb15 },
  { "ORIYA LETTER KHA", 0xb16 },
  { "ORIYA LETTER GA", 0xb17 },
  { "ORIYA LETTER GHA", 0xb18 },
  { "ORIYA LETTER NGA", 0xb19 },
  { "ORIYA LETTER CA", 0xb1a },
  { "ORIYA LETTER CHA", 0xb1b },
  { "ORIYA LETTER JA", 0xb1c },
  { "ORIYA LETTER JHA", 0xb1d },
  { "ORIYA LETTER NYA", 0xb1e },
  { "ORIYA LETTER TTA", 0xb1f },
  { "ORIYA LETTER TTHA", 0xb20 },
  { "ORIYA LETTER DDA", 0xb21 },
  { "ORIYA LETTER DDHA", 0xb22 },
  { "ORIYA LETTER NNA", 0xb23 },
  { "ORIYA LETTER TA", 0xb24 },
  { "ORIYA LETTER THA", 0xb25 },
  { "ORIYA LETTER DA", 0xb26 },
  { "ORIYA LETTER DHA", 0xb27 },
  { "ORIYA LETTER NA", 0xb28 },
  { "ORIYA LETTER PA", 0xb2a },
  { "ORIYA LETTER PHA", 0xb2b },
  { "ORIYA LETTER BA", 0xb2c },
  { "ORIYA LETTER BHA", 0xb2d },
  { "ORIYA LETTER MA", 0xb2e },
  { "ORIYA LETTER YA", 0xb2f },
  { "ORIYA LETTER RA", 0xb30 },
  { "ORIYA LETTER LA", 0xb32 },
  { "ORIYA LETTER LLA", 0xb33 },
  { "ORIYA LETTER SHA", 0xb36 },
  { "ORIYA LETTER SSA", 0xb37 },
  { "ORIYA LETTER SA", 0xb38 },
  { "ORIYA LETTER HA", 0xb39 },
  { "ORIYA SIGN NUKTA", 0xb3c },
  { "ORIYA SIGN AVAGRAHA", 0xb3d },
  { "ORIYA VOWEL SIGN AA", 0xb3e },
  { "ORIYA VOWEL SIGN I", 0xb3f },
  { "ORIYA VOWEL SIGN II", 0xb40 },
  { "ORIYA VOWEL SIGN U", 0xb41 },
  { "ORIYA VOWEL SIGN UU", 0xb42 },
  { "ORIYA VOWEL SIGN VOCALIC R", 0xb43 },
  { "ORIYA VOWEL SIGN E", 0xb47 },
  { "ORIYA VOWEL SIGN AI", 0xb48 },
  { "ORIYA VOWEL SIGN O", 0xb4b },
  { "ORIYA VOWEL SIGN AU", 0xb4c },
  { "ORIYA SIGN VIRAMA", 0xb4d },
  { "ORIYA AI LENGTH MARK", 0xb56 },
  { "ORIYA AU LENGTH MARK", 0xb57 },
  { "ORIYA LETTER RRA", 0xb5c },
  { "ORIYA LETTER RHA", 0xb5d },
  { "ORIYA LETTER YYA", 0xb5f },
  { "ORIYA LETTER VOCALIC RR", 0xb60 },
  { "ORIYA LETTER VOCALIC LL", 0xb61 },
  { "ORIYA DIGIT ZERO", 0xb66 },
  { "ORIYA DIGIT ONE", 0xb67 },
  { "ORIYA DIGIT TWO", 0xb68 },
  { "ORIYA DIGIT THREE", 0xb69 },
  { "ORIYA DIGIT FOUR", 0xb6a },
  { "ORIYA DIGIT FIVE", 0xb6b },
  { "ORIYA DIGIT SIX", 0xb6c },
  { "ORIYA DIGIT SEVEN", 0xb6d },
  { "ORIYA DIGIT EIGHT", 0xb6e },
  { "ORIYA DIGIT NINE", 0xb6f },
  { "ORIYA ISSHAR", 0xb70 },
  { "TAMIL SIGN ANUSVARA", 0xb82 },
  { "TAMIL SIGN VISARGA", 0xb83 },
  { "TAMIL LETTER A", 0xb85 },
  { "TAMIL LETTER AA", 0xb86 },
  { "TAMIL LETTER I", 0xb87 },
  { "TAMIL LETTER II", 0xb88 },
  { "TAMIL LETTER U", 0xb89 },
  { "TAMIL LETTER UU", 0xb8a },
  { "TAMIL LETTER E", 0xb8e },
  { "TAMIL LETTER EE", 0xb8f },
  { "TAMIL LETTER AI", 0xb90 },
  { "TAMIL LETTER O", 0xb92 },
  { "TAMIL LETTER OO", 0xb93 },
  { "TAMIL LETTER AU", 0xb94 },
  { "TAMIL LETTER KA", 0xb95 },
  { "TAMIL LETTER NGA", 0xb99 },
  { "TAMIL LETTER CA", 0xb9a },
  { "TAMIL LETTER JA", 0xb9c },
  { "TAMIL LETTER NYA", 0xb9e },
  { "TAMIL LETTER TTA", 0xb9f },
  { "TAMIL LETTER NNA", 0xba3 },
  { "TAMIL LETTER TA", 0xba4 },
  { "TAMIL LETTER NA", 0xba8 },
  { "TAMIL LETTER NNNA", 0xba9 },
  { "TAMIL LETTER PA", 0xbaa },
  { "TAMIL LETTER MA", 0xbae },
  { "TAMIL LETTER YA", 0xbaf },
  { "TAMIL LETTER RA", 0xbb0 },
  { "TAMIL LETTER RRA", 0xbb1 },
  { "TAMIL LETTER LA", 0xbb2 },
  { "TAMIL LETTER LLA", 0xbb3 },
  { "TAMIL LETTER LLLA", 0xbb4 },
  { "TAMIL LETTER VA", 0xbb5 },
  { "TAMIL LETTER SSA", 0xbb7 },
  { "TAMIL LETTER SA", 0xbb8 },
  { "TAMIL LETTER HA", 0xbb9 },
  { "TAMIL VOWEL SIGN AA", 0xbbe },
  { "TAMIL VOWEL SIGN I", 0xbbf },
  { "TAMIL VOWEL SIGN II", 0xbc0 },
  { "TAMIL VOWEL SIGN U", 0xbc1 },
  { "TAMIL VOWEL SIGN UU", 0xbc2 },
  { "TAMIL VOWEL SIGN E", 0xbc6 },
  { "TAMIL VOWEL SIGN EE", 0xbc7 },
  { "TAMIL VOWEL SIGN AI", 0xbc8 },
  { "TAMIL VOWEL SIGN O", 0xbca },
  { "TAMIL VOWEL SIGN OO", 0xbcb },
  { "TAMIL VOWEL SIGN AU", 0xbcc },
  { "TAMIL SIGN VIRAMA", 0xbcd },
  { "TAMIL AU LENGTH MARK", 0xbd7 },
  { "TAMIL DIGIT ONE", 0xbe7 },
  { "TAMIL DIGIT TWO", 0xbe8 },
  { "TAMIL DIGIT THREE", 0xbe9 },
  { "TAMIL DIGIT FOUR", 0xbea },
  { "TAMIL DIGIT FIVE", 0xbeb },
  { "TAMIL DIGIT SIX", 0xbec },
  { "TAMIL DIGIT SEVEN", 0xbed },
  { "TAMIL DIGIT EIGHT", 0xbee },
  { "TAMIL DIGIT NINE", 0xbef },
  { "TAMIL NUMBER TEN", 0xbf0 },
  { "TAMIL NUMBER ONE HUNDRED", 0xbf1 },
  { "TAMIL NUMBER ONE THOUSAND", 0xbf2 },
  { "TELUGU SIGN CANDRABINDU", 0xc01 },
  { "TELUGU SIGN ANUSVARA", 0xc02 },
  { "TELUGU SIGN VISARGA", 0xc03 },
  { "TELUGU LETTER A", 0xc05 },
  { "TELUGU LETTER AA", 0xc06 },
  { "TELUGU LETTER I", 0xc07 },
  { "TELUGU LETTER II", 0xc08 },
  { "TELUGU LETTER U", 0xc09 },
  { "TELUGU LETTER UU", 0xc0a },
  { "TELUGU LETTER VOCALIC R", 0xc0b },
  { "TELUGU LETTER VOCALIC L", 0xc0c },
  { "TELUGU LETTER E", 0xc0e },
  { "TELUGU LETTER EE", 0xc0f },
  { "TELUGU LETTER AI", 0xc10 },
  { "TELUGU LETTER O", 0xc12 },
  { "TELUGU LETTER OO", 0xc13 },
  { "TELUGU LETTER AU", 0xc14 },
  { "TELUGU LETTER KA", 0xc15 },
  { "TELUGU LETTER KHA", 0xc16 },
  { "TELUGU LETTER GA", 0xc17 },
  { "TELUGU LETTER GHA", 0xc18 },
  { "TELUGU LETTER NGA", 0xc19 },
  { "TELUGU LETTER CA", 0xc1a },
  { "TELUGU LETTER CHA", 0xc1b },
  { "TELUGU LETTER JA", 0xc1c },
  { "TELUGU LETTER JHA", 0xc1d },
  { "TELUGU LETTER NYA", 0xc1e },
  { "TELUGU LETTER TTA", 0xc1f },
  { "TELUGU LETTER TTHA", 0xc20 },
  { "TELUGU LETTER DDA", 0xc21 },
  { "TELUGU LETTER DDHA", 0xc22 },
  { "TELUGU LETTER NNA", 0xc23 },
  { "TELUGU LETTER TA", 0xc24 },
  { "TELUGU LETTER THA", 0xc25 },
  { "TELUGU LETTER DA", 0xc26 },
  { "TELUGU LETTER DHA", 0xc27 },
  { "TELUGU LETTER NA", 0xc28 },
  { "TELUGU LETTER PA", 0xc2a },
  { "TELUGU LETTER PHA", 0xc2b },
  { "TELUGU LETTER BA", 0xc2c },
  { "TELUGU LETTER BHA", 0xc2d },
  { "TELUGU LETTER MA", 0xc2e },
  { "TELUGU LETTER YA", 0xc2f },
  { "TELUGU LETTER RA", 0xc30 },
  { "TELUGU LETTER RRA", 0xc31 },
  { "TELUGU LETTER LA", 0xc32 },
  { "TELUGU LETTER LLA", 0xc33 },
  { "TELUGU LETTER VA", 0xc35 },
  { "TELUGU LETTER SHA", 0xc36 },
  { "TELUGU LETTER SSA", 0xc37 },
  { "TELUGU LETTER SA", 0xc38 },
  { "TELUGU LETTER HA", 0xc39 },
  { "TELUGU VOWEL SIGN AA", 0xc3e },
  { "TELUGU VOWEL SIGN I", 0xc3f },
  { "TELUGU VOWEL SIGN II", 0xc40 },
  { "TELUGU VOWEL SIGN U", 0xc41 },
  { "TELUGU VOWEL SIGN UU", 0xc42 },
  { "TELUGU VOWEL SIGN VOCALIC R", 0xc43 },
  { "TELUGU VOWEL SIGN VOCALIC RR", 0xc44 },
  { "TELUGU VOWEL SIGN E", 0xc46 },
  { "TELUGU VOWEL SIGN EE", 0xc47 },
  { "TELUGU VOWEL SIGN AI", 0xc48 },
  { "TELUGU VOWEL SIGN O", 0xc4a },
  { "TELUGU VOWEL SIGN OO", 0xc4b },
  { "TELUGU VOWEL SIGN AU", 0xc4c },
  { "TELUGU SIGN VIRAMA", 0xc4d },
  { "TELUGU LENGTH MARK", 0xc55 },
  { "TELUGU AI LENGTH MARK", 0xc56 },
  { "TELUGU LETTER VOCALIC RR", 0xc60 },
  { "TELUGU LETTER VOCALIC LL", 0xc61 },
  { "TELUGU DIGIT ZERO", 0xc66 },
  { "TELUGU DIGIT ONE", 0xc67 },
  { "TELUGU DIGIT TWO", 0xc68 },
  { "TELUGU DIGIT THREE", 0xc69 },
  { "TELUGU DIGIT FOUR", 0xc6a },
  { "TELUGU DIGIT FIVE", 0xc6b },
  { "TELUGU DIGIT SIX", 0xc6c },
  { "TELUGU DIGIT SEVEN", 0xc6d },
  { "TELUGU DIGIT EIGHT", 0xc6e },
  { "TELUGU DIGIT NINE", 0xc6f },
  { "KANNADA SIGN ANUSVARA", 0xc82 },
  { "KANNADA SIGN VISARGA", 0xc83 },
  { "KANNADA LETTER A", 0xc85 },
  { "KANNADA LETTER AA", 0xc86 },
  { "KANNADA LETTER I", 0xc87 },
  { "KANNADA LETTER II", 0xc88 },
  { "KANNADA LETTER U", 0xc89 },
  { "KANNADA LETTER UU", 0xc8a },
  { "KANNADA LETTER VOCALIC R", 0xc8b },
  { "KANNADA LETTER VOCALIC L", 0xc8c },
  { "KANNADA LETTER E", 0xc8e },
  { "KANNADA LETTER EE", 0xc8f },
  { "KANNADA LETTER AI", 0xc90 },
  { "KANNADA LETTER O", 0xc92 },
  { "KANNADA LETTER OO", 0xc93 },
  { "KANNADA LETTER AU", 0xc94 },
  { "KANNADA LETTER KA", 0xc95 },
  { "KANNADA LETTER KHA", 0xc96 },
  { "KANNADA LETTER GA", 0xc97 },
  { "KANNADA LETTER GHA", 0xc98 },
  { "KANNADA LETTER NGA", 0xc99 },
  { "KANNADA LETTER CA", 0xc9a },
  { "KANNADA LETTER CHA", 0xc9b },
  { "KANNADA LETTER JA", 0xc9c },
  { "KANNADA LETTER JHA", 0xc9d },
  { "KANNADA LETTER NYA", 0xc9e },
  { "KANNADA LETTER TTA", 0xc9f },
  { "KANNADA LETTER TTHA", 0xca0 },
  { "KANNADA LETTER DDA", 0xca1 },
  { "KANNADA LETTER DDHA", 0xca2 },
  { "KANNADA LETTER NNA", 0xca3 },
  { "KANNADA LETTER TA", 0xca4 },
  { "KANNADA LETTER THA", 0xca5 },
  { "KANNADA LETTER DA", 0xca6 },
  { "KANNADA LETTER DHA", 0xca7 },
  { "KANNADA LETTER NA", 0xca8 },
  { "KANNADA LETTER PA", 0xcaa },
  { "KANNADA LETTER PHA", 0xcab },
  { "KANNADA LETTER BA", 0xcac },
  { "KANNADA LETTER BHA", 0xcad },
  { "KANNADA LETTER MA", 0xcae },
  { "KANNADA LETTER YA", 0xcaf },
  { "KANNADA LETTER RA", 0xcb0 },
  { "KANNADA LETTER RRA", 0xcb1 },
  { "KANNADA LETTER LA", 0xcb2 },
  { "KANNADA LETTER LLA", 0xcb3 },
  { "KANNADA LETTER VA", 0xcb5 },
  { "KANNADA LETTER SHA", 0xcb6 },
  { "KANNADA LETTER SSA", 0xcb7 },
  { "KANNADA LETTER SA", 0xcb8 },
  { "KANNADA LETTER HA", 0xcb9 },
  { "KANNADA VOWEL SIGN AA", 0xcbe },
  { "KANNADA VOWEL SIGN I", 0xcbf },
  { "KANNADA VOWEL SIGN II", 0xcc0 },
  { "KANNADA VOWEL SIGN U", 0xcc1 },
  { "KANNADA VOWEL SIGN UU", 0xcc2 },
  { "KANNADA VOWEL SIGN VOCALIC R", 0xcc3 },
  { "KANNADA VOWEL SIGN VOCALIC RR", 0xcc4 },
  { "KANNADA VOWEL SIGN E", 0xcc6 },
  { "KANNADA VOWEL SIGN EE", 0xcc7 },
  { "KANNADA VOWEL SIGN AI", 0xcc8 },
  { "KANNADA VOWEL SIGN O", 0xcca },
  { "KANNADA VOWEL SIGN OO", 0xccb },
  { "KANNADA VOWEL SIGN AU", 0xccc },
  { "KANNADA SIGN VIRAMA", 0xccd },
  { "KANNADA LENGTH MARK", 0xcd5 },
  { "KANNADA AI LENGTH MARK", 0xcd6 },
  { "KANNADA LETTER FA", 0xcde },
  { "KANNADA LETTER VOCALIC RR", 0xce0 },
  { "KANNADA LETTER VOCALIC LL", 0xce1 },
  { "KANNADA DIGIT ZERO", 0xce6 },
  { "KANNADA DIGIT ONE", 0xce7 },
  { "KANNADA DIGIT TWO", 0xce8 },
  { "KANNADA DIGIT THREE", 0xce9 },
  { "KANNADA DIGIT FOUR", 0xcea },
  { "KANNADA DIGIT FIVE", 0xceb },
  { "KANNADA DIGIT SIX", 0xcec },
  { "KANNADA DIGIT SEVEN", 0xced },
  { "KANNADA DIGIT EIGHT", 0xcee },
  { "KANNADA DIGIT NINE", 0xcef },
  { "MALAYALAM SIGN ANUSVARA", 0xd02 },
  { "MALAYALAM SIGN VISARGA", 0xd03 },
  { "MALAYALAM LETTER A", 0xd05 },
  { "MALAYALAM LETTER AA", 0xd06 },
  { "MALAYALAM LETTER I", 0xd07 },
  { "MALAYALAM LETTER II", 0xd08 },
  { "MALAYALAM LETTER U", 0xd09 },
  { "MALAYALAM LETTER UU", 0xd0a },
  { "MALAYALAM LETTER VOCALIC R", 0xd0b },
  { "MALAYALAM LETTER VOCALIC L", 0xd0c },
  { "MALAYALAM LETTER E", 0xd0e },
  { "MALAYALAM LETTER EE", 0xd0f },
  { "MALAYALAM LETTER AI", 0xd10 },
  { "MALAYALAM LETTER O", 0xd12 },
  { "MALAYALAM LETTER OO", 0xd13 },
  { "MALAYALAM LETTER AU", 0xd14 },
  { "MALAYALAM LETTER KA", 0xd15 },
  { "MALAYALAM LETTER KHA", 0xd16 },
  { "MALAYALAM LETTER GA", 0xd17 },
  { "MALAYALAM LETTER GHA", 0xd18 },
  { "MALAYALAM LETTER NGA", 0xd19 },
  { "MALAYALAM LETTER CA", 0xd1a },
  { "MALAYALAM LETTER CHA", 0xd1b },
  { "MALAYALAM LETTER JA", 0xd1c },
  { "MALAYALAM LETTER JHA", 0xd1d },
  { "MALAYALAM LETTER NYA", 0xd1e },
  { "MALAYALAM LETTER TTA", 0xd1f },
  { "MALAYALAM LETTER TTHA", 0xd20 },
  { "MALAYALAM LETTER DDA", 0xd21 },
  { "MALAYALAM LETTER DDHA", 0xd22 },
  { "MALAYALAM LETTER NNA", 0xd23 },
  { "MALAYALAM LETTER TA", 0xd24 },
  { "MALAYALAM LETTER THA", 0xd25 },
  { "MALAYALAM LETTER DA", 0xd26 },
  { "MALAYALAM LETTER DHA", 0xd27 },
  { "MALAYALAM LETTER NA", 0xd28 },
  { "MALAYALAM LETTER PA", 0xd2a },
  { "MALAYALAM LETTER PHA", 0xd2b },
  { "MALAYALAM LETTER BA", 0xd2c },
  { "MALAYALAM LETTER BHA", 0xd2d },
  { "MALAYALAM LETTER MA", 0xd2e },
  { "MALAYALAM LETTER YA", 0xd2f },
  { "MALAYALAM LETTER RA", 0xd30 },
  { "MALAYALAM LETTER RRA", 0xd31 },
  { "MALAYALAM LETTER LA", 0xd32 },
  { "MALAYALAM LETTER LLA", 0xd33 },
  { "MALAYALAM LETTER LLLA", 0xd34 },
  { "MALAYALAM LETTER VA", 0xd35 },
  { "MALAYALAM LETTER SHA", 0xd36 },
  { "MALAYALAM LETTER SSA", 0xd37 },
  { "MALAYALAM LETTER SA", 0xd38 },
  { "MALAYALAM LETTER HA", 0xd39 },
  { "MALAYALAM VOWEL SIGN AA", 0xd3e },
  { "MALAYALAM VOWEL SIGN I", 0xd3f },
  { "MALAYALAM VOWEL SIGN II", 0xd40 },
  { "MALAYALAM VOWEL SIGN U", 0xd41 },
  { "MALAYALAM VOWEL SIGN UU", 0xd42 },
  { "MALAYALAM VOWEL SIGN VOCALIC R", 0xd43 },
  { "MALAYALAM VOWEL SIGN E", 0xd46 },
  { "MALAYALAM VOWEL SIGN EE", 0xd47 },
  { "MALAYALAM VOWEL SIGN AI", 0xd48 },
  { "MALAYALAM VOWEL SIGN O", 0xd4a },
  { "MALAYALAM VOWEL SIGN OO", 0xd4b },
  { "MALAYALAM VOWEL SIGN AU", 0xd4c },
  { "MALAYALAM SIGN VIRAMA", 0xd4d },
  { "MALAYALAM AU LENGTH MARK", 0xd57 },
  { "MALAYALAM LETTER VOCALIC RR", 0xd60 },
  { "MALAYALAM LETTER VOCALIC LL", 0xd61 },
  { "MALAYALAM DIGIT ZERO", 0xd66 },
  { "MALAYALAM DIGIT ONE", 0xd67 },
  { "MALAYALAM DIGIT TWO", 0xd68 },
  { "MALAYALAM DIGIT THREE", 0xd69 },
  { "MALAYALAM DIGIT FOUR", 0xd6a },
  { "MALAYALAM DIGIT FIVE", 0xd6b },
  { "MALAYALAM DIGIT SIX", 0xd6c },
  { "MALAYALAM DIGIT SEVEN", 0xd6d },
  { "MALAYALAM DIGIT EIGHT", 0xd6e },
  { "MALAYALAM DIGIT NINE", 0xd6f },
  { "SINHALA SIGN ANUSVARAYA", 0xd82 },
  { "SINHALA SIGN VISARGAYA", 0xd83 },
  { "SINHALA LETTER AYANNA", 0xd85 },
  { "SINHALA LETTER AAYANNA", 0xd86 },
  { "SINHALA LETTER AEYANNA", 0xd87 },
  { "SINHALA LETTER AEEYANNA", 0xd88 },
  { "SINHALA LETTER IYANNA", 0xd89 },
  { "SINHALA LETTER IIYANNA", 0xd8a },
  { "SINHALA LETTER UYANNA", 0xd8b },
  { "SINHALA LETTER UUYANNA", 0xd8c },
  { "SINHALA LETTER IRUYANNA", 0xd8d },
  { "SINHALA LETTER IRUUYANNA", 0xd8e },
  { "SINHALA LETTER ILUYANNA", 0xd8f },
  { "SINHALA LETTER ILUUYANNA", 0xd90 },
  { "SINHALA LETTER EYANNA", 0xd91 },
  { "SINHALA LETTER EEYANNA", 0xd92 },
  { "SINHALA LETTER AIYANNA", 0xd93 },
  { "SINHALA LETTER OYANNA", 0xd94 },
  { "SINHALA LETTER OOYANNA", 0xd95 },
  { "SINHALA LETTER AUYANNA", 0xd96 },
  { "SINHALA LETTER ALPAPRAANA KAYANNA", 0xd9a },
  { "SINHALA LETTER MAHAAPRAANA KAYANNA", 0xd9b },
  { "SINHALA LETTER ALPAPRAANA GAYANNA", 0xd9c },
  { "SINHALA LETTER MAHAAPRAANA GAYANNA", 0xd9d },
  { "SINHALA LETTER KANTAJA NAASIKYAYA", 0xd9e },
  { "SINHALA LETTER SANYAKA GAYANNA", 0xd9f },
  { "SINHALA LETTER ALPAPRAANA CAYANNA", 0xda0 },
  { "SINHALA LETTER MAHAAPRAANA CAYANNA", 0xda1 },
  { "SINHALA LETTER ALPAPRAANA JAYANNA", 0xda2 },
  { "SINHALA LETTER MAHAAPRAANA JAYANNA", 0xda3 },
  { "SINHALA LETTER TAALUJA NAASIKYAYA", 0xda4 },
  { "SINHALA LETTER TAALUJA SANYOOGA NAAKSIKYAYA", 0xda5 },
  { "SINHALA LETTER SANYAKA JAYANNA", 0xda6 },
  { "SINHALA LETTER ALPAPRAANA TTAYANNA", 0xda7 },
  { "SINHALA LETTER MAHAAPRAANA TTAYANNA", 0xda8 },
  { "SINHALA LETTER ALPAPRAANA DDAYANNA", 0xda9 },
  { "SINHALA LETTER MAHAAPRAANA DDAYANNA", 0xdaa },
  { "SINHALA LETTER MUURDHAJA NAYANNA", 0xdab },
  { "SINHALA LETTER SANYAKA DDAYANNA", 0xdac },
  { "SINHALA LETTER ALPAPRAANA TAYANNA", 0xdad },
  { "SINHALA LETTER MAHAAPRAANA TAYANNA", 0xdae },
  { "SINHALA LETTER ALPAPRAANA DAYANNA", 0xdaf },
  { "SINHALA LETTER MAHAAPRAANA DAYANNA", 0xdb0 },
  { "SINHALA LETTER DANTAJA NAYANNA", 0xdb1 },
  { "SINHALA LETTER SANYAKA DAYANNA", 0xdb3 },
  { "SINHALA LETTER ALPAPRAANA PAYANNA", 0xdb4 },
  { "SINHALA LETTER MAHAAPRAANA PAYANNA", 0xdb5 },
  { "SINHALA LETTER ALPAPRAANA BAYANNA", 0xdb6 },
  { "SINHALA LETTER MAHAAPRAANA BAYANNA", 0xdb7 },
  { "SINHALA LETTER MAYANNA", 0xdb8 },
  { "SINHALA LETTER AMBA BAYANNA", 0xdb9 },
  { "SINHALA LETTER YAYANNA", 0xdba },
  { "SINHALA LETTER RAYANNA", 0xdbb },
  { "SINHALA LETTER DANTAJA LAYANNA", 0xdbd },
  { "SINHALA LETTER VAYANNA", 0xdc0 },
  { "SINHALA LETTER TAALUJA SAYANNA", 0xdc1 },
  { "SINHALA LETTER MUURDHAJA SAYANNA", 0xdc2 },
  { "SINHALA LETTER DANTAJA SAYANNA", 0xdc3 },
  { "SINHALA LETTER HAYANNA", 0xdc4 },
  { "SINHALA LETTER MUURDHAJA LAYANNA", 0xdc5 },
  { "SINHALA LETTER FAYANNA", 0xdc6 },
  { "SINHALA SIGN AL-LAKUNA", 0xdca },
  { "SINHALA VOWEL SIGN AELA-PILLA", 0xdcf },
  { "SINHALA VOWEL SIGN KETTI AEDA-PILLA", 0xdd0 },
  { "SINHALA VOWEL SIGN DIGA AEDA-PILLA", 0xdd1 },
  { "SINHALA VOWEL SIGN KETTI IS-PILLA", 0xdd2 },
  { "SINHALA VOWEL SIGN DIGA IS-PILLA", 0xdd3 },
  { "SINHALA VOWEL SIGN KETTI PAA-PILLA", 0xdd4 },
  { "SINHALA VOWEL SIGN DIGA PAA-PILLA", 0xdd6 },
  { "SINHALA VOWEL SIGN GAETTA-PILLA", 0xdd8 },
  { "SINHALA VOWEL SIGN KOMBUVA", 0xdd9 },
  { "SINHALA VOWEL SIGN DIGA KOMBUVA", 0xdda },
  { "SINHALA VOWEL SIGN KOMBU DEKA", 0xddb },
  { "SINHALA VOWEL SIGN KOMBUVA HAA AELA-PILLA", 0xddc },
  { "SINHALA VOWEL SIGN KOMBUVA HAA DIGA AELA-PILLA", 0xddd },
  { "SINHALA VOWEL SIGN KOMBUVA HAA GAYANUKITTA", 0xdde },
  { "SINHALA VOWEL SIGN GAYANUKITTA", 0xddf },
  { "SINHALA VOWEL SIGN DIGA GAETTA-PILLA", 0xdf2 },
  { "SINHALA VOWEL SIGN DIGA GAYANUKITTA", 0xdf3 },
  { "SINHALA PUNCTUATION KUNDDALIYA", 0xdf4 },
  { "THAI CHARACTER KO KAI", 0xe01 },
  { "THAI CHARACTER KHO KHAI", 0xe02 },
  { "THAI CHARACTER KHO KHUAT", 0xe03 },
  { "THAI CHARACTER KHO KHWAI", 0xe04 },
  { "THAI CHARACTER KHO KHON", 0xe05 },
  { "THAI CHARACTER KHO RAKHANG", 0xe06 },
  { "THAI CHARACTER NGO NGU", 0xe07 },
  { "THAI CHARACTER CHO CHAN", 0xe08 },
  { "THAI CHARACTER CHO CHING", 0xe09 },
  { "THAI CHARACTER CHO CHANG", 0xe0a },
  { "THAI CHARACTER SO SO", 0xe0b },
  { "THAI CHARACTER CHO CHOE", 0xe0c },
  { "THAI CHARACTER YO YING", 0xe0d },
  { "THAI CHARACTER DO CHADA", 0xe0e },
  { "THAI CHARACTER TO PATAK", 0xe0f },
  { "THAI CHARACTER THO THAN", 0xe10 },
  { "THAI CHARACTER THO NANGMONTHO", 0xe11 },
  { "THAI CHARACTER THO PHUTHAO", 0xe12 },
  { "THAI CHARACTER NO NEN", 0xe13 },
  { "THAI CHARACTER DO DEK", 0xe14 },
  { "THAI CHARACTER TO TAO", 0xe15 },
  { "THAI CHARACTER THO THUNG", 0xe16 },
  { "THAI CHARACTER THO THAHAN", 0xe17 },
  { "THAI CHARACTER THO THONG", 0xe18 },
  { "THAI CHARACTER NO NU", 0xe19 },
  { "THAI CHARACTER BO BAIMAI", 0xe1a },
  { "THAI CHARACTER PO PLA", 0xe1b },
  { "THAI CHARACTER PHO PHUNG", 0xe1c },
  { "THAI CHARACTER FO FA", 0xe1d },
  { "THAI CHARACTER PHO PHAN", 0xe1e },
  { "THAI CHARACTER FO FAN", 0xe1f },
  { "THAI CHARACTER PHO SAMPHAO", 0xe20 },
  { "THAI CHARACTER MO MA", 0xe21 },
  { "THAI CHARACTER YO YAK", 0xe22 },
  { "THAI CHARACTER RO RUA", 0xe23 },
  { "THAI CHARACTER RU", 0xe24 },
  { "THAI CHARACTER LO LING", 0xe25 },
  { "THAI CHARACTER LU", 0xe26 },
  { "THAI CHARACTER WO WAEN", 0xe27 },
  { "THAI CHARACTER SO SALA", 0xe28 },
  { "THAI CHARACTER SO RUSI", 0xe29 },
  { "THAI CHARACTER SO SUA", 0xe2a },
  { "THAI CHARACTER HO HIP", 0xe2b },
  { "THAI CHARACTER LO CHULA", 0xe2c },
  { "THAI CHARACTER O ANG", 0xe2d },
  { "THAI CHARACTER HO NOKHUK", 0xe2e },
  { "THAI CHARACTER PAIYANNOI", 0xe2f },
  { "THAI CHARACTER SARA A", 0xe30 },
  { "THAI CHARACTER MAI HAN-AKAT", 0xe31 },
  { "THAI CHARACTER SARA AA", 0xe32 },
  { "THAI CHARACTER SARA AM", 0xe33 },
  { "THAI CHARACTER SARA I", 0xe34 },
  { "THAI CHARACTER SARA II", 0xe35 },
  { "THAI CHARACTER SARA UE", 0xe36 },
  { "THAI CHARACTER SARA UEE", 0xe37 },
  { "THAI CHARACTER SARA U", 0xe38 },
  { "THAI CHARACTER SARA UU", 0xe39 },
  { "THAI CHARACTER PHINTHU", 0xe3a },
  { "THAI CURRENCY SYMBOL BAHT", 0xe3f },
  { "THAI CHARACTER SARA E", 0xe40 },
  { "THAI CHARACTER SARA AE", 0xe41 },
  { "THAI CHARACTER SARA O", 0xe42 },
  { "THAI CHARACTER SARA AI MAIMUAN", 0xe43 },
  { "THAI CHARACTER SARA AI MAIMALAI", 0xe44 },
  { "THAI CHARACTER LAKKHANGYAO", 0xe45 },
  { "THAI CHARACTER MAIYAMOK", 0xe46 },
  { "THAI CHARACTER MAITAIKHU", 0xe47 },
  { "THAI CHARACTER MAI EK", 0xe48 },
  { "THAI CHARACTER MAI THO", 0xe49 },
  { "THAI CHARACTER MAI TRI", 0xe4a },
  { "THAI CHARACTER MAI CHATTAWA", 0xe4b },
  { "THAI CHARACTER THANTHAKHAT", 0xe4c },
  { "THAI CHARACTER NIKHAHIT", 0xe4d },
  { "THAI CHARACTER YAMAKKAN", 0xe4e },
  { "THAI CHARACTER FONGMAN", 0xe4f },
  { "THAI DIGIT ZERO", 0xe50 },
  { "THAI DIGIT ONE", 0xe51 },
  { "THAI DIGIT TWO", 0xe52 },
  { "THAI DIGIT THREE", 0xe53 },
  { "THAI DIGIT FOUR", 0xe54 },
  { "THAI DIGIT FIVE", 0xe55 },
  { "THAI DIGIT SIX", 0xe56 },
  { "THAI DIGIT SEVEN", 0xe57 },
  { "THAI DIGIT EIGHT", 0xe58 },
  { "THAI DIGIT NINE", 0xe59 },
  { "THAI CHARACTER ANGKHANKHU", 0xe5a },
  { "THAI CHARACTER KHOMUT", 0xe5b },
  { "LAO LETTER KO", 0xe81 },
  { "LAO LETTER KHO SUNG", 0xe82 },
  { "LAO LETTER KHO TAM", 0xe84 },
  { "LAO LETTER NGO", 0xe87 },
  { "LAO LETTER CO", 0xe88 },
  { "LAO LETTER SO TAM", 0xe8a },
  { "LAO LETTER NYO", 0xe8d },
  { "LAO LETTER DO", 0xe94 },
  { "LAO LETTER TO", 0xe95 },
  { "LAO LETTER THO SUNG", 0xe96 },
  { "LAO LETTER THO TAM", 0xe97 },
  { "LAO LETTER NO", 0xe99 },
  { "LAO LETTER BO", 0xe9a },
  { "LAO LETTER PO", 0xe9b },
  { "LAO LETTER PHO SUNG", 0xe9c },
  { "LAO LETTER FO TAM", 0xe9d },
  { "LAO LETTER PHO TAM", 0xe9e },
  { "LAO LETTER FO SUNG", 0xe9f },
  { "LAO LETTER MO", 0xea1 },
  { "LAO LETTER YO", 0xea2 },
  { "LAO LETTER LO LING", 0xea3 },
  { "LAO LETTER LO LOOT", 0xea5 },
  { "LAO LETTER WO", 0xea7 },
  { "LAO LETTER SO SUNG", 0xeaa },
  { "LAO LETTER HO SUNG", 0xeab },
  { "LAO LETTER O", 0xead },
  { "LAO LETTER HO TAM", 0xeae },
  { "LAO ELLIPSIS", 0xeaf },
  { "LAO VOWEL SIGN A", 0xeb0 },
  { "LAO VOWEL SIGN MAI KAN", 0xeb1 },
  { "LAO VOWEL SIGN AA", 0xeb2 },
  { "LAO VOWEL SIGN AM", 0xeb3 },
  { "LAO VOWEL SIGN I", 0xeb4 },
  { "LAO VOWEL SIGN II", 0xeb5 },
  { "LAO VOWEL SIGN Y", 0xeb6 },
  { "LAO VOWEL SIGN YY", 0xeb7 },
  { "LAO VOWEL SIGN U", 0xeb8 },
  { "LAO VOWEL SIGN UU", 0xeb9 },
  { "LAO VOWEL SIGN MAI KON", 0xebb },
  { "LAO SEMIVOWEL SIGN LO", 0xebc },
  { "LAO SEMIVOWEL SIGN NYO", 0xebd },
  { "LAO VOWEL SIGN E", 0xec0 },
  { "LAO VOWEL SIGN EI", 0xec1 },
  { "LAO VOWEL SIGN O", 0xec2 },
  { "LAO VOWEL SIGN AY", 0xec3 },
  { "LAO VOWEL SIGN AI", 0xec4 },
  { "LAO KO LA", 0xec6 },
  { "LAO TONE MAI EK", 0xec8 },
  { "LAO TONE MAI THO", 0xec9 },
  { "LAO TONE MAI TI", 0xeca },
  { "LAO TONE MAI CATAWA", 0xecb },
  { "LAO CANCELLATION MARK", 0xecc },
  { "LAO NIGGAHITA", 0xecd },
  { "LAO DIGIT ZERO", 0xed0 },
  { "LAO DIGIT ONE", 0xed1 },
  { "LAO DIGIT TWO", 0xed2 },
  { "LAO DIGIT THREE", 0xed3 },
  { "LAO DIGIT FOUR", 0xed4 },
  { "LAO DIGIT FIVE", 0xed5 },
  { "LAO DIGIT SIX", 0xed6 },
  { "LAO DIGIT SEVEN", 0xed7 },
  { "LAO DIGIT EIGHT", 0xed8 },
  { "LAO DIGIT NINE", 0xed9 },
  { "LAO HO NO", 0xedc },
  { "LAO HO MO", 0xedd },
  { "TIBETAN SYLLABLE OM", 0xf00 },
  { "TIBETAN MARK GTER YIG MGO TRUNCATED A", 0xf01 },
  { "TIBETAN MARK GTER YIG MGO -UM RNAM BCAD MA", 0xf02 },
  { "TIBETAN MARK GTER YIG MGO -UM GTER TSHEG MA", 0xf03 },
  { "TIBETAN MARK INITIAL YIG MGO MDUN MA", 0xf04 },
  { "TIBETAN MARK CLOSING YIG MGO SGAB MA", 0xf05 },
  { "TIBETAN MARK CARET YIG MGO PHUR SHAD MA", 0xf06 },
  { "TIBETAN MARK YIG MGO TSHEG SHAD MA", 0xf07 },
  { "TIBETAN MARK SBRUL SHAD", 0xf08 },
  { "TIBETAN MARK BSKUR YIG MGO", 0xf09 },
  { "TIBETAN MARK BKA- SHOG YIG MGO", 0xf0a },
  { "TIBETAN MARK INTERSYLLABIC TSHEG", 0xf0b },
  { "TIBETAN MARK DELIMITER TSHEG BSTAR", 0xf0c },
  { "TIBETAN MARK SHAD", 0xf0d },
  { "TIBETAN MARK NYIS SHAD", 0xf0e },
  { "TIBETAN MARK TSHEG SHAD", 0xf0f },
  { "TIBETAN MARK NYIS TSHEG SHAD", 0xf10 },
  { "TIBETAN MARK RIN CHEN SPUNGS SHAD", 0xf11 },
  { "TIBETAN MARK RGYA GRAM SHAD", 0xf12 },
  { "TIBETAN MARK CARET -DZUD RTAGS ME LONG CAN", 0xf13 },
  { "TIBETAN MARK GTER TSHEG", 0xf14 },
  { "TIBETAN LOGOTYPE SIGN CHAD RTAGS", 0xf15 },
  { "TIBETAN LOGOTYPE SIGN LHAG RTAGS", 0xf16 },
  { "TIBETAN ASTROLOGICAL SIGN SGRA GCAN -CHAR RTAGS", 0xf17 },
  { "TIBETAN ASTROLOGICAL SIGN -KHYUD PA", 0xf18 },
  { "TIBETAN ASTROLOGICAL SIGN SDONG TSHUGS", 0xf19 },
  { "TIBETAN SIGN RDEL DKAR GCIG", 0xf1a },
  { "TIBETAN SIGN RDEL DKAR GNYIS", 0xf1b },
  { "TIBETAN SIGN RDEL DKAR GSUM", 0xf1c },
  { "TIBETAN SIGN RDEL NAG GCIG", 0xf1d },
  { "TIBETAN SIGN RDEL NAG GNYIS", 0xf1e },
  { "TIBETAN SIGN RDEL DKAR RDEL NAG", 0xf1f },
  { "TIBETAN DIGIT ZERO", 0xf20 },
  { "TIBETAN DIGIT ONE", 0xf21 },
  { "TIBETAN DIGIT TWO", 0xf22 },
  { "TIBETAN DIGIT THREE", 0xf23 },
  { "TIBETAN DIGIT FOUR", 0xf24 },
  { "TIBETAN DIGIT FIVE", 0xf25 },
  { "TIBETAN DIGIT SIX", 0xf26 },
  { "TIBETAN DIGIT SEVEN", 0xf27 },
  { "TIBETAN DIGIT EIGHT", 0xf28 },
  { "TIBETAN DIGIT NINE", 0xf29 },
  { "TIBETAN DIGIT HALF ONE", 0xf2a },
  { "TIBETAN DIGIT HALF TWO", 0xf2b },
  { "TIBETAN DIGIT HALF THREE", 0xf2c },
  { "TIBETAN DIGIT HALF FOUR", 0xf2d },
  { "TIBETAN DIGIT HALF FIVE", 0xf2e },
  { "TIBETAN DIGIT HALF SIX", 0xf2f },
  { "TIBETAN DIGIT HALF SEVEN", 0xf30 },
  { "TIBETAN DIGIT HALF EIGHT", 0xf31 },
  { "TIBETAN DIGIT HALF NINE", 0xf32 },
  { "TIBETAN DIGIT HALF ZERO", 0xf33 },
  { "TIBETAN MARK BSDUS RTAGS", 0xf34 },
  { "TIBETAN MARK NGAS BZUNG NYI ZLA", 0xf35 },
  { "TIBETAN MARK CARET -DZUD RTAGS BZHI MIG CAN", 0xf36 },
  { "TIBETAN MARK NGAS BZUNG SGOR RTAGS", 0xf37 },
  { "TIBETAN MARK CHE MGO", 0xf38 },
  { "TIBETAN MARK TSA -PHRU", 0xf39 },
  { "TIBETAN MARK GUG RTAGS GYON", 0xf3a },
  { "TIBETAN MARK GUG RTAGS GYAS", 0xf3b },
  { "TIBETAN MARK ANG KHANG GYON", 0xf3c },
  { "TIBETAN MARK ANG KHANG GYAS", 0xf3d },
  { "TIBETAN SIGN YAR TSHES", 0xf3e },
  { "TIBETAN SIGN MAR TSHES", 0xf3f },
  { "TIBETAN LETTER KA", 0xf40 },
  { "TIBETAN LETTER KHA", 0xf41 },
  { "TIBETAN LETTER GA", 0xf42 },
  { "TIBETAN LETTER GHA", 0xf43 },
  { "TIBETAN LETTER NGA", 0xf44 },
  { "TIBETAN LETTER CA", 0xf45 },
  { "TIBETAN LETTER CHA", 0xf46 },
  { "TIBETAN LETTER JA", 0xf47 },
  { "TIBETAN LETTER NYA", 0xf49 },
  { "TIBETAN LETTER TTA", 0xf4a },
  { "TIBETAN LETTER TTHA", 0xf4b },
  { "TIBETAN LETTER DDA", 0xf4c },
  { "TIBETAN LETTER DDHA", 0xf4d },
  { "TIBETAN LETTER NNA", 0xf4e },
  { "TIBETAN LETTER TA", 0xf4f },
  { "TIBETAN LETTER THA", 0xf50 },
  { "TIBETAN LETTER DA", 0xf51 },
  { "TIBETAN LETTER DHA", 0xf52 },
  { "TIBETAN LETTER NA", 0xf53 },
  { "TIBETAN LETTER PA", 0xf54 },
  { "TIBETAN LETTER PHA", 0xf55 },
  { "TIBETAN LETTER BA", 0xf56 },
  { "TIBETAN LETTER BHA", 0xf57 },
  { "TIBETAN LETTER MA", 0xf58 },
  { "TIBETAN LETTER TSA", 0xf59 },
  { "TIBETAN LETTER TSHA", 0xf5a },
  { "TIBETAN LETTER DZA", 0xf5b },
  { "TIBETAN LETTER DZHA", 0xf5c },
  { "TIBETAN LETTER WA", 0xf5d },
  { "TIBETAN LETTER ZHA", 0xf5e },
  { "TIBETAN LETTER ZA", 0xf5f },
  { "TIBETAN LETTER -A", 0xf60 },
  { "TIBETAN LETTER YA", 0xf61 },
  { "TIBETAN LETTER RA", 0xf62 },
  { "TIBETAN LETTER LA", 0xf63 },
  { "TIBETAN LETTER SHA", 0xf64 },
  { "TIBETAN LETTER SSA", 0xf65 },
  { "TIBETAN LETTER SA", 0xf66 },
  { "TIBETAN LETTER HA", 0xf67 },
  { "TIBETAN LETTER A", 0xf68 },
  { "TIBETAN LETTER KSSA", 0xf69 },
  { "TIBETAN LETTER FIXED-FORM RA", 0xf6a },
  { "TIBETAN VOWEL SIGN AA", 0xf71 },
  { "TIBETAN VOWEL SIGN I", 0xf72 },
  { "TIBETAN VOWEL SIGN II", 0xf73 },
  { "TIBETAN VOWEL SIGN U", 0xf74 },
  { "TIBETAN VOWEL SIGN UU", 0xf75 },
  { "TIBETAN VOWEL SIGN VOCALIC R", 0xf76 },
  { "TIBETAN VOWEL SIGN VOCALIC RR", 0xf77 },
  { "TIBETAN VOWEL SIGN VOCALIC L", 0xf78 },
  { "TIBETAN VOWEL SIGN VOCALIC LL", 0xf79 },
  { "TIBETAN VOWEL SIGN E", 0xf7a },
  { "TIBETAN VOWEL SIGN EE", 0xf7b },
  { "TIBETAN VOWEL SIGN O", 0xf7c },
  { "TIBETAN VOWEL SIGN OO", 0xf7d },
  { "TIBETAN SIGN RJES SU NGA RO", 0xf7e },
  { "TIBETAN SIGN RNAM BCAD", 0xf7f },
  { "TIBETAN VOWEL SIGN REVERSED I", 0xf80 },
  { "TIBETAN VOWEL SIGN REVERSED II", 0xf81 },
  { "TIBETAN SIGN NYI ZLA NAA DA", 0xf82 },
  { "TIBETAN SIGN SNA LDAN", 0xf83 },
  { "TIBETAN MARK HALANTA", 0xf84 },
  { "TIBETAN MARK PALUTA", 0xf85 },
  { "TIBETAN SIGN LCI RTAGS", 0xf86 },
  { "TIBETAN SIGN YANG RTAGS", 0xf87 },
  { "TIBETAN SIGN LCE TSA CAN", 0xf88 },
  { "TIBETAN SIGN MCHU CAN", 0xf89 },
  { "TIBETAN SIGN GRU CAN RGYINGS", 0xf8a },
  { "TIBETAN SIGN GRU MED RGYINGS", 0xf8b },
  { "TIBETAN SUBJOINED LETTER KA", 0xf90 },
  { "TIBETAN SUBJOINED LETTER KHA", 0xf91 },
  { "TIBETAN SUBJOINED LETTER GA", 0xf92 },
  { "TIBETAN SUBJOINED LETTER GHA", 0xf93 },
  { "TIBETAN SUBJOINED LETTER NGA", 0xf94 },
  { "TIBETAN SUBJOINED LETTER CA", 0xf95 },
  { "TIBETAN SUBJOINED LETTER CHA", 0xf96 },
  { "TIBETAN SUBJOINED LETTER JA", 0xf97 },
  { "TIBETAN SUBJOINED LETTER NYA", 0xf99 },
  { "TIBETAN SUBJOINED LETTER TTA", 0xf9a },
  { "TIBETAN SUBJOINED LETTER TTHA", 0xf9b },
  { "TIBETAN SUBJOINED LETTER DDA", 0xf9c },
  { "TIBETAN SUBJOINED LETTER DDHA", 0xf9d },
  { "TIBETAN SUBJOINED LETTER NNA", 0xf9e },
  { "TIBETAN SUBJOINED LETTER TA", 0xf9f },
  { "TIBETAN SUBJOINED LETTER THA", 0xfa0 },
  { "TIBETAN SUBJOINED LETTER DA", 0xfa1 },
  { "TIBETAN SUBJOINED LETTER DHA", 0xfa2 },
  { "TIBETAN SUBJOINED LETTER NA", 0xfa3 },
  { "TIBETAN SUBJOINED LETTER PA", 0xfa4 },
  { "TIBETAN SUBJOINED LETTER PHA", 0xfa5 },
  { "TIBETAN SUBJOINED LETTER BA", 0xfa6 },
  { "TIBETAN SUBJOINED LETTER BHA", 0xfa7 },
  { "TIBETAN SUBJOINED LETTER MA", 0xfa8 },
  { "TIBETAN SUBJOINED LETTER TSA", 0xfa9 },
  { "TIBETAN SUBJOINED LETTER TSHA", 0xfaa },
  { "TIBETAN SUBJOINED LETTER DZA", 0xfab },
  { "TIBETAN SUBJOINED LETTER DZHA", 0xfac },
  { "TIBETAN SUBJOINED LETTER WA", 0xfad },
  { "TIBETAN SUBJOINED LETTER ZHA", 0xfae },
  { "TIBETAN SUBJOINED LETTER ZA", 0xfaf },
  { "TIBETAN SUBJOINED LETTER -A", 0xfb0 },
  { "TIBETAN SUBJOINED LETTER YA", 0xfb1 },
  { "TIBETAN SUBJOINED LETTER RA", 0xfb2 },
  { "TIBETAN SUBJOINED LETTER LA", 0xfb3 },
  { "TIBETAN SUBJOINED LETTER SHA", 0xfb4 },
  { "TIBETAN SUBJOINED LETTER SSA", 0xfb5 },
  { "TIBETAN SUBJOINED LETTER SA", 0xfb6 },
  { "TIBETAN SUBJOINED LETTER HA", 0xfb7 },
  { "TIBETAN SUBJOINED LETTER A", 0xfb8 },
  { "TIBETAN SUBJOINED LETTER KSSA", 0xfb9 },
  { "TIBETAN SUBJOINED LETTER FIXED-FORM WA", 0xfba },
  { "TIBETAN SUBJOINED LETTER FIXED-FORM YA", 0xfbb },
  { "TIBETAN SUBJOINED LETTER FIXED-FORM RA", 0xfbc },
  { "TIBETAN KU RU KHA", 0xfbe },
  { "TIBETAN KU RU KHA BZHI MIG CAN", 0xfbf },
  { "TIBETAN CANTILLATION SIGN HEAVY BEAT", 0xfc0 },
  { "TIBETAN CANTILLATION SIGN LIGHT BEAT", 0xfc1 },
  { "TIBETAN CANTILLATION SIGN CANG TE-U", 0xfc2 },
  { "TIBETAN CANTILLATION SIGN SBUB -CHAL", 0xfc3 },
  { "TIBETAN SYMBOL DRIL BU", 0xfc4 },
  { "TIBETAN SYMBOL RDO RJE", 0xfc5 },
  { "TIBETAN SYMBOL PADMA GDAN", 0xfc6 },
  { "TIBETAN SYMBOL RDO RJE RGYA GRAM", 0xfc7 },
  { "TIBETAN SYMBOL PHUR PA", 0xfc8 },
  { "TIBETAN SYMBOL NOR BU", 0xfc9 },
  { "TIBETAN SYMBOL NOR BU NYIS -KHYIL", 0xfca },
  { "TIBETAN SYMBOL NOR BU GSUM -KHYIL", 0xfcb },
  { "TIBETAN SYMBOL NOR BU BZHI -KHYIL", 0xfcc },
  { "TIBETAN SIGN RDEL NAG GSUM", 0xfcf },
  { "MYANMAR LETTER KA", 0x1000 },
  { "MYANMAR LETTER KHA", 0x1001 },
  { "MYANMAR LETTER GA", 0x1002 },
  { "MYANMAR LETTER GHA", 0x1003 },
  { "MYANMAR LETTER NGA", 0x1004 },
  { "MYANMAR LETTER CA", 0x1005 },
  { "MYANMAR LETTER CHA", 0x1006 },
  { "MYANMAR LETTER JA", 0x1007 },
  { "MYANMAR LETTER JHA", 0x1008 },
  { "MYANMAR LETTER NYA", 0x1009 },
  { "MYANMAR LETTER NNYA", 0x100a },
  { "MYANMAR LETTER TTA", 0x100b },
  { "MYANMAR LETTER TTHA", 0x100c },
  { "MYANMAR LETTER DDA", 0x100d },
  { "MYANMAR LETTER DDHA", 0x100e },
  { "MYANMAR LETTER NNA", 0x100f },
  { "MYANMAR LETTER TA", 0x1010 },
  { "MYANMAR LETTER THA", 0x1011 },
  { "MYANMAR LETTER DA", 0x1012 },
  { "MYANMAR LETTER DHA", 0x1013 },
  { "MYANMAR LETTER NA", 0x1014 },
  { "MYANMAR LETTER PA", 0x1015 },
  { "MYANMAR LETTER PHA", 0x1016 },
  { "MYANMAR LETTER BA", 0x1017 },
  { "MYANMAR LETTER BHA", 0x1018 },
  { "MYANMAR LETTER MA", 0x1019 },
  { "MYANMAR LETTER YA", 0x101a },
  { "MYANMAR LETTER RA", 0x101b },
  { "MYANMAR LETTER LA", 0x101c },
  { "MYANMAR LETTER WA", 0x101d },
  { "MYANMAR LETTER SA", 0x101e },
  { "MYANMAR LETTER HA", 0x101f },
  { "MYANMAR LETTER LLA", 0x1020 },
  { "MYANMAR LETTER A", 0x1021 },
  { "MYANMAR LETTER I", 0x1023 },
  { "MYANMAR LETTER II", 0x1024 },
  { "MYANMAR LETTER U", 0x1025 },
  { "MYANMAR LETTER UU", 0x1026 },
  { "MYANMAR LETTER E", 0x1027 },
  { "MYANMAR LETTER O", 0x1029 },
  { "MYANMAR LETTER AU", 0x102a },
  { "MYANMAR VOWEL SIGN AA", 0x102c },
  { "MYANMAR VOWEL SIGN I", 0x102d },
  { "MYANMAR VOWEL SIGN II", 0x102e },
  { "MYANMAR VOWEL SIGN U", 0x102f },
  { "MYANMAR VOWEL SIGN UU", 0x1030 },
  { "MYANMAR VOWEL SIGN E", 0x1031 },
  { "MYANMAR VOWEL SIGN AI", 0x1032 },
  { "MYANMAR SIGN ANUSVARA", 0x1036 },
  { "MYANMAR SIGN DOT BELOW", 0x1037 },
  { "MYANMAR SIGN VISARGA", 0x1038 },
  { "MYANMAR SIGN VIRAMA", 0x1039 },
  { "MYANMAR DIGIT ZERO", 0x1040 },
  { "MYANMAR DIGIT ONE", 0x1041 },
  { "MYANMAR DIGIT TWO", 0x1042 },
  { "MYANMAR DIGIT THREE", 0x1043 },
  { "MYANMAR DIGIT FOUR", 0x1044 },
  { "MYANMAR DIGIT FIVE", 0x1045 },
  { "MYANMAR DIGIT SIX", 0x1046 },
  { "MYANMAR DIGIT SEVEN", 0x1047 },
  { "MYANMAR DIGIT EIGHT", 0x1048 },
  { "MYANMAR DIGIT NINE", 0x1049 },
  { "MYANMAR SIGN LITTLE SECTION", 0x104a },
  { "MYANMAR SIGN SECTION", 0x104b },
  { "MYANMAR SYMBOL LOCATIVE", 0x104c },
  { "MYANMAR SYMBOL COMPLETED", 0x104d },
  { "MYANMAR SYMBOL AFOREMENTIONED", 0x104e },
  { "MYANMAR SYMBOL GENITIVE", 0x104f },
  { "MYANMAR LETTER SHA", 0x1050 },
  { "MYANMAR LETTER SSA", 0x1051 },
  { "MYANMAR LETTER VOCALIC R", 0x1052 },
  { "MYANMAR LETTER VOCALIC RR", 0x1053 },
  { "MYANMAR LETTER VOCALIC L", 0x1054 },
  { "MYANMAR LETTER VOCALIC LL", 0x1055 },
  { "MYANMAR VOWEL SIGN VOCALIC R", 0x1056 },
  { "MYANMAR VOWEL SIGN VOCALIC RR", 0x1057 },
  { "MYANMAR VOWEL SIGN VOCALIC L", 0x1058 },
  { "MYANMAR VOWEL SIGN VOCALIC LL", 0x1059 },
  { "GEORGIAN CAPITAL LETTER AN", 0x10a0 },
  { "GEORGIAN CAPITAL LETTER BAN", 0x10a1 },
  { "GEORGIAN CAPITAL LETTER GAN", 0x10a2 },
  { "GEORGIAN CAPITAL LETTER DON", 0x10a3 },
  { "GEORGIAN CAPITAL LETTER EN", 0x10a4 },
  { "GEORGIAN CAPITAL LETTER VIN", 0x10a5 },
  { "GEORGIAN CAPITAL LETTER ZEN", 0x10a6 },
  { "GEORGIAN CAPITAL LETTER TAN", 0x10a7 },
  { "GEORGIAN CAPITAL LETTER IN", 0x10a8 },
  { "GEORGIAN CAPITAL LETTER KAN", 0x10a9 },
  { "GEORGIAN CAPITAL LETTER LAS", 0x10aa },
  { "GEORGIAN CAPITAL LETTER MAN", 0x10ab },
  { "GEORGIAN CAPITAL LETTER NAR", 0x10ac },
  { "GEORGIAN CAPITAL LETTER ON", 0x10ad },
  { "GEORGIAN CAPITAL LETTER PAR", 0x10ae },
  { "GEORGIAN CAPITAL LETTER ZHAR", 0x10af },
  { "GEORGIAN CAPITAL LETTER RAE", 0x10b0 },
  { "GEORGIAN CAPITAL LETTER SAN", 0x10b1 },
  { "GEORGIAN CAPITAL LETTER TAR", 0x10b2 },
  { "GEORGIAN CAPITAL LETTER UN", 0x10b3 },
  { "GEORGIAN CAPITAL LETTER PHAR", 0x10b4 },
  { "GEORGIAN CAPITAL LETTER KHAR", 0x10b5 },
  { "GEORGIAN CAPITAL LETTER GHAN", 0x10b6 },
  { "GEORGIAN CAPITAL LETTER QAR", 0x10b7 },
  { "GEORGIAN CAPITAL LETTER SHIN", 0x10b8 },
  { "GEORGIAN CAPITAL LETTER CHIN", 0x10b9 },
  { "GEORGIAN CAPITAL LETTER CAN", 0x10ba },
  { "GEORGIAN CAPITAL LETTER JIL", 0x10bb },
  { "GEORGIAN CAPITAL LETTER CIL", 0x10bc },
  { "GEORGIAN CAPITAL LETTER CHAR", 0x10bd },
  { "GEORGIAN CAPITAL LETTER XAN", 0x10be },
  { "GEORGIAN CAPITAL LETTER JHAN", 0x10bf },
  { "GEORGIAN CAPITAL LETTER HAE", 0x10c0 },
  { "GEORGIAN CAPITAL LETTER HE", 0x10c1 },
  { "GEORGIAN CAPITAL LETTER HIE", 0x10c2 },
  { "GEORGIAN CAPITAL LETTER WE", 0x10c3 },
  { "GEORGIAN CAPITAL LETTER HAR", 0x10c4 },
  { "GEORGIAN CAPITAL LETTER HOE", 0x10c5 },
  { "GEORGIAN LETTER AN", 0x10d0 },
  { "GEORGIAN LETTER BAN", 0x10d1 },
  { "GEORGIAN LETTER GAN", 0x10d2 },
  { "GEORGIAN LETTER DON", 0x10d3 },
  { "GEORGIAN LETTER EN", 0x10d4 },
  { "GEORGIAN LETTER VIN", 0x10d5 },
  { "GEORGIAN LETTER ZEN", 0x10d6 },
  { "GEORGIAN LETTER TAN", 0x10d7 },
  { "GEORGIAN LETTER IN", 0x10d8 },
  { "GEORGIAN LETTER KAN", 0x10d9 },
  { "GEORGIAN LETTER LAS", 0x10da },
  { "GEORGIAN LETTER MAN", 0x10db },
  { "GEORGIAN LETTER NAR", 0x10dc },
  { "GEORGIAN LETTER ON", 0x10dd },
  { "GEORGIAN LETTER PAR", 0x10de },
  { "GEORGIAN LETTER ZHAR", 0x10df },
  { "GEORGIAN LETTER RAE", 0x10e0 },
  { "GEORGIAN LETTER SAN", 0x10e1 },
  { "GEORGIAN LETTER TAR", 0x10e2 },
  { "GEORGIAN LETTER UN", 0x10e3 },
  { "GEORGIAN LETTER PHAR", 0x10e4 },
  { "GEORGIAN LETTER KHAR", 0x10e5 },
  { "GEORGIAN LETTER GHAN", 0x10e6 },
  { "GEORGIAN LETTER QAR", 0x10e7 },
  { "GEORGIAN LETTER SHIN", 0x10e8 },
  { "GEORGIAN LETTER CHIN", 0x10e9 },
  { "GEORGIAN LETTER CAN", 0x10ea },
  { "GEORGIAN LETTER JIL", 0x10eb },
  { "GEORGIAN LETTER CIL", 0x10ec },
  { "GEORGIAN LETTER CHAR", 0x10ed },
  { "GEORGIAN LETTER XAN", 0x10ee },
  { "GEORGIAN LETTER JHAN", 0x10ef },
  { "GEORGIAN LETTER HAE", 0x10f0 },
  { "GEORGIAN LETTER HE", 0x10f1 },
  { "GEORGIAN LETTER HIE", 0x10f2 },
  { "GEORGIAN LETTER WE", 0x10f3 },
  { "GEORGIAN LETTER HAR", 0x10f4 },
  { "GEORGIAN LETTER HOE", 0x10f5 },
  { "GEORGIAN LETTER FI", 0x10f6 },
  { "GEORGIAN PARAGRAPH SEPARATOR", 0x10fb },
  { "HANGUL CHOSEONG KIYEOK", 0x1100 },
  { "HANGUL CHOSEONG SSANGKIYEOK", 0x1101 },
  { "HANGUL CHOSEONG NIEUN", 0x1102 },
  { "HANGUL CHOSEONG TIKEUT", 0x1103 },
  { "HANGUL CHOSEONG SSANGTIKEUT", 0x1104 },
  { "HANGUL CHOSEONG RIEUL", 0x1105 },
  { "HANGUL CHOSEONG MIEUM", 0x1106 },
  { "HANGUL CHOSEONG PIEUP", 0x1107 },
  { "HANGUL CHOSEONG SSANGPIEUP", 0x1108 },
  { "HANGUL CHOSEONG SIOS", 0x1109 },
  { "HANGUL CHOSEONG SSANGSIOS", 0x110a },
  { "HANGUL CHOSEONG IEUNG", 0x110b },
  { "HANGUL CHOSEONG CIEUC", 0x110c },
  { "HANGUL CHOSEONG SSANGCIEUC", 0x110d },
  { "HANGUL CHOSEONG CHIEUCH", 0x110e },
  { "HANGUL CHOSEONG KHIEUKH", 0x110f },
  { "HANGUL CHOSEONG THIEUTH", 0x1110 },
  { "HANGUL CHOSEONG PHIEUPH", 0x1111 },
  { "HANGUL CHOSEONG HIEUH", 0x1112 },
  { "HANGUL CHOSEONG NIEUN-KIYEOK", 0x1113 },
  { "HANGUL CHOSEONG SSANGNIEUN", 0x1114 },
  { "HANGUL CHOSEONG NIEUN-TIKEUT", 0x1115 },
  { "HANGUL CHOSEONG NIEUN-PIEUP", 0x1116 },
  { "HANGUL CHOSEONG TIKEUT-KIYEOK", 0x1117 },
  { "HANGUL CHOSEONG RIEUL-NIEUN", 0x1118 },
  { "HANGUL CHOSEONG SSANGRIEUL", 0x1119 },
  { "HANGUL CHOSEONG RIEUL-HIEUH", 0x111a },
  { "HANGUL CHOSEONG KAPYEOUNRIEUL", 0x111b },
  { "HANGUL CHOSEONG MIEUM-PIEUP", 0x111c },
  { "HANGUL CHOSEONG KAPYEOUNMIEUM", 0x111d },
  { "HANGUL CHOSEONG PIEUP-KIYEOK", 0x111e },
  { "HANGUL CHOSEONG PIEUP-NIEUN", 0x111f },
  { "HANGUL CHOSEONG PIEUP-TIKEUT", 0x1120 },
  { "HANGUL CHOSEONG PIEUP-SIOS", 0x1121 },
  { "HANGUL CHOSEONG PIEUP-SIOS-KIYEOK", 0x1122 },
  { "HANGUL CHOSEONG PIEUP-SIOS-TIKEUT", 0x1123 },
  { "HANGUL CHOSEONG PIEUP-SIOS-PIEUP", 0x1124 },
  { "HANGUL CHOSEONG PIEUP-SSANGSIOS", 0x1125 },
  { "HANGUL CHOSEONG PIEUP-SIOS-CIEUC", 0x1126 },
  { "HANGUL CHOSEONG PIEUP-CIEUC", 0x1127 },
  { "HANGUL CHOSEONG PIEUP-CHIEUCH", 0x1128 },
  { "HANGUL CHOSEONG PIEUP-THIEUTH", 0x1129 },
  { "HANGUL CHOSEONG PIEUP-PHIEUPH", 0x112a },
  { "HANGUL CHOSEONG KAPYEOUNPIEUP", 0x112b },
  { "HANGUL CHOSEONG KAPYEOUNSSANGPIEUP", 0x112c },
  { "HANGUL CHOSEONG SIOS-KIYEOK", 0x112d },
  { "HANGUL CHOSEONG SIOS-NIEUN", 0x112e },
  { "HANGUL CHOSEONG SIOS-TIKEUT", 0x112f },
  { "HANGUL CHOSEONG SIOS-RIEUL", 0x1130 },
  { "HANGUL CHOSEONG SIOS-MIEUM", 0x1131 },
  { "HANGUL CHOSEONG SIOS-PIEUP", 0x1132 },
  { "HANGUL CHOSEONG SIOS-PIEUP-KIYEOK", 0x1133 },
  { "HANGUL CHOSEONG SIOS-SSANGSIOS", 0x1134 },
  { "HANGUL CHOSEONG SIOS-IEUNG", 0x1135 },
  { "HANGUL CHOSEONG SIOS-CIEUC", 0x1136 },
  { "HANGUL CHOSEONG SIOS-CHIEUCH", 0x1137 },
  { "HANGUL CHOSEONG SIOS-KHIEUKH", 0x1138 },
  { "HANGUL CHOSEONG SIOS-THIEUTH", 0x1139 },
  { "HANGUL CHOSEONG SIOS-PHIEUPH", 0x113a },
  { "HANGUL CHOSEONG SIOS-HIEUH", 0x113b },
  { "HANGUL CHOSEONG CHITUEUMSIOS", 0x113c },
  { "HANGUL CHOSEONG CHITUEUMSSANGSIOS", 0x113d },
  { "HANGUL CHOSEONG CEONGCHIEUMSIOS", 0x113e },
  { "HANGUL CHOSEONG CEONGCHIEUMSSANGSIOS", 0x113f },
  { "HANGUL CHOSEONG PANSIOS", 0x1140 },
  { "HANGUL CHOSEONG IEUNG-KIYEOK", 0x1141 },
  { "HANGUL CHOSEONG IEUNG-TIKEUT", 0x1142 },
  { "HANGUL CHOSEONG IEUNG-MIEUM", 0x1143 },
  { "HANGUL CHOSEONG IEUNG-PIEUP", 0x1144 },
  { "HANGUL CHOSEONG IEUNG-SIOS", 0x1145 },
  { "HANGUL CHOSEONG IEUNG-PANSIOS", 0x1146 },
  { "HANGUL CHOSEONG SSANGIEUNG", 0x1147 },
  { "HANGUL CHOSEONG IEUNG-CIEUC", 0x1148 },
  { "HANGUL CHOSEONG IEUNG-CHIEUCH", 0x1149 },
  { "HANGUL CHOSEONG IEUNG-THIEUTH", 0x114a },
  { "HANGUL CHOSEONG IEUNG-PHIEUPH", 0x114b },
  { "HANGUL CHOSEONG YESIEUNG", 0x114c },
  { "HANGUL CHOSEONG CIEUC-IEUNG", 0x114d },
  { "HANGUL CHOSEONG CHITUEUMCIEUC", 0x114e },
  { "HANGUL CHOSEONG CHITUEUMSSANGCIEUC", 0x114f },
  { "HANGUL CHOSEONG CEONGCHIEUMCIEUC", 0x1150 },
  { "HANGUL CHOSEONG CEONGCHIEUMSSANGCIEUC", 0x1151 },
  { "HANGUL CHOSEONG CHIEUCH-KHIEUKH", 0x1152 },
  { "HANGUL CHOSEONG CHIEUCH-HIEUH", 0x1153 },
  { "HANGUL CHOSEONG CHITUEUMCHIEUCH", 0x1154 },
  { "HANGUL CHOSEONG CEONGCHIEUMCHIEUCH", 0x1155 },
  { "HANGUL CHOSEONG PHIEUPH-PIEUP", 0x1156 },
  { "HANGUL CHOSEONG KAPYEOUNPHIEUPH", 0x1157 },
  { "HANGUL CHOSEONG SSANGHIEUH", 0x1158 },
  { "HANGUL CHOSEONG YEORINHIEUH", 0x1159 },
  { "HANGUL CHOSEONG FILLER", 0x115f },
  { "HANGUL JUNGSEONG FILLER", 0x1160 },
  { "HANGUL JUNGSEONG A", 0x1161 },
  { "HANGUL JUNGSEONG AE", 0x1162 },
  { "HANGUL JUNGSEONG YA", 0x1163 },
  { "HANGUL JUNGSEONG YAE", 0x1164 },
  { "HANGUL JUNGSEONG EO", 0x1165 },
  { "HANGUL JUNGSEONG E", 0x1166 },
  { "HANGUL JUNGSEONG YEO", 0x1167 },
  { "HANGUL JUNGSEONG YE", 0x1168 },
  { "HANGUL JUNGSEONG O", 0x1169 },
  { "HANGUL JUNGSEONG WA", 0x116a },
  { "HANGUL JUNGSEONG WAE", 0x116b },
  { "HANGUL JUNGSEONG OE", 0x116c },
  { "HANGUL JUNGSEONG YO", 0x116d },
  { "HANGUL JUNGSEONG U", 0x116e },
  { "HANGUL JUNGSEONG WEO", 0x116f },
  { "HANGUL JUNGSEONG WE", 0x1170 },
  { "HANGUL JUNGSEONG WI", 0x1171 },
  { "HANGUL JUNGSEONG YU", 0x1172 },
  { "HANGUL JUNGSEONG EU", 0x1173 },
  { "HANGUL JUNGSEONG YI", 0x1174 },
  { "HANGUL JUNGSEONG I", 0x1175 },
  { "HANGUL JUNGSEONG A-O", 0x1176 },
  { "HANGUL JUNGSEONG A-U", 0x1177 },
  { "HANGUL JUNGSEONG YA-O", 0x1178 },
  { "HANGUL JUNGSEONG YA-YO", 0x1179 },
  { "HANGUL JUNGSEONG EO-O", 0x117a },
  { "HANGUL JUNGSEONG EO-U", 0x117b },
  { "HANGUL JUNGSEONG EO-EU", 0x117c },
  { "HANGUL JUNGSEONG YEO-O", 0x117d },
  { "HANGUL JUNGSEONG YEO-U", 0x117e },
  { "HANGUL JUNGSEONG O-EO", 0x117f },
  { "HANGUL JUNGSEONG O-E", 0x1180 },
  { "HANGUL JUNGSEONG O-YE", 0x1181 },
  { "HANGUL JUNGSEONG O-O", 0x1182 },
  { "HANGUL JUNGSEONG O-U", 0x1183 },
  { "HANGUL JUNGSEONG YO-YA", 0x1184 },
  { "HANGUL JUNGSEONG YO-YAE", 0x1185 },
  { "HANGUL JUNGSEONG YO-YEO", 0x1186 },
  { "HANGUL JUNGSEONG YO-O", 0x1187 },
  { "HANGUL JUNGSEONG YO-I", 0x1188 },
  { "HANGUL JUNGSEONG U-A", 0x1189 },
  { "HANGUL JUNGSEONG U-AE", 0x118a },
  { "HANGUL JUNGSEONG U-EO-EU", 0x118b },
  { "HANGUL JUNGSEONG U-YE", 0x118c },
  { "HANGUL JUNGSEONG U-U", 0x118d },
  { "HANGUL JUNGSEONG YU-A", 0x118e },
  { "HANGUL JUNGSEONG YU-EO", 0x118f },
  { "HANGUL JUNGSEONG YU-E", 0x1190 },
  { "HANGUL JUNGSEONG YU-YEO", 0x1191 },
  { "HANGUL JUNGSEONG YU-YE", 0x1192 },
  { "HANGUL JUNGSEONG YU-U", 0x1193 },
  { "HANGUL JUNGSEONG YU-I", 0x1194 },
  { "HANGUL JUNGSEONG EU-U", 0x1195 },
  { "HANGUL JUNGSEONG EU-EU", 0x1196 },
  { "HANGUL JUNGSEONG YI-U", 0x1197 },
  { "HANGUL JUNGSEONG I-A", 0x1198 },
  { "HANGUL JUNGSEONG I-YA", 0x1199 },
  { "HANGUL JUNGSEONG I-O", 0x119a },
  { "HANGUL JUNGSEONG I-U", 0x119b },
  { "HANGUL JUNGSEONG I-EU", 0x119c },
  { "HANGUL JUNGSEONG I-ARAEA", 0x119d },
  { "HANGUL JUNGSEONG ARAEA", 0x119e },
  { "HANGUL JUNGSEONG ARAEA-EO", 0x119f },
  { "HANGUL JUNGSEONG ARAEA-U", 0x11a0 },
  { "HANGUL JUNGSEONG ARAEA-I", 0x11a1 },
  { "HANGUL JUNGSEONG SSANGARAEA", 0x11a2 },
  { "HANGUL JONGSEONG KIYEOK", 0x11a8 },
  { "HANGUL JONGSEONG SSANGKIYEOK", 0x11a9 },
  { "HANGUL JONGSEONG KIYEOK-SIOS", 0x11aa },
  { "HANGUL JONGSEONG NIEUN", 0x11ab },
  { "HANGUL JONGSEONG NIEUN-CIEUC", 0x11ac },
  { "HANGUL JONGSEONG NIEUN-HIEUH", 0x11ad },
  { "HANGUL JONGSEONG TIKEUT", 0x11ae },
  { "HANGUL JONGSEONG RIEUL", 0x11af },
  { "HANGUL JONGSEONG RIEUL-KIYEOK", 0x11b0 },
  { "HANGUL JONGSEONG RIEUL-MIEUM", 0x11b1 },
  { "HANGUL JONGSEONG RIEUL-PIEUP", 0x11b2 },
  { "HANGUL JONGSEONG RIEUL-SIOS", 0x11b3 },
  { "HANGUL JONGSEONG RIEUL-THIEUTH", 0x11b4 },
  { "HANGUL JONGSEONG RIEUL-PHIEUPH", 0x11b5 },
  { "HANGUL JONGSEONG RIEUL-HIEUH", 0x11b6 },
  { "HANGUL JONGSEONG MIEUM", 0x11b7 },
  { "HANGUL JONGSEONG PIEUP", 0x11b8 },
  { "HANGUL JONGSEONG PIEUP-SIOS", 0x11b9 },
  { "HANGUL JONGSEONG SIOS", 0x11ba },
  { "HANGUL JONGSEONG SSANGSIOS", 0x11bb },
  { "HANGUL JONGSEONG IEUNG", 0x11bc },
  { "HANGUL JONGSEONG CIEUC", 0x11bd },
  { "HANGUL JONGSEONG CHIEUCH", 0x11be },
  { "HANGUL JONGSEONG KHIEUKH", 0x11bf },
  { "HANGUL JONGSEONG THIEUTH", 0x11c0 },
  { "HANGUL JONGSEONG PHIEUPH", 0x11c1 },
  { "HANGUL JONGSEONG HIEUH", 0x11c2 },
  { "HANGUL JONGSEONG KIYEOK-RIEUL", 0x11c3 },
  { "HANGUL JONGSEONG KIYEOK-SIOS-KIYEOK", 0x11c4 },
  { "HANGUL JONGSEONG NIEUN-KIYEOK", 0x11c5 },
  { "HANGUL JONGSEONG NIEUN-TIKEUT", 0x11c6 },
  { "HANGUL JONGSEONG NIEUN-SIOS", 0x11c7 },
  { "HANGUL JONGSEONG NIEUN-PANSIOS", 0x11c8 },
  { "HANGUL JONGSEONG NIEUN-THIEUTH", 0x11c9 },
  { "HANGUL JONGSEONG TIKEUT-KIYEOK", 0x11ca },
  { "HANGUL JONGSEONG TIKEUT-RIEUL", 0x11cb },
  { "HANGUL JONGSEONG RIEUL-KIYEOK-SIOS", 0x11cc },
  { "HANGUL JONGSEONG RIEUL-NIEUN", 0x11cd },
  { "HANGUL JONGSEONG RIEUL-TIKEUT", 0x11ce },
  { "HANGUL JONGSEONG RIEUL-TIKEUT-HIEUH", 0x11cf },
  { "HANGUL JONGSEONG SSANGRIEUL", 0x11d0 },
  { "HANGUL JONGSEONG RIEUL-MIEUM-KIYEOK", 0x11d1 },
  { "HANGUL JONGSEONG RIEUL-MIEUM-SIOS", 0x11d2 },
  { "HANGUL JONGSEONG RIEUL-PIEUP-SIOS", 0x11d3 },
  { "HANGUL JONGSEONG RIEUL-PIEUP-HIEUH", 0x11d4 },
  { "HANGUL JONGSEONG RIEUL-KAPYEOUNPIEUP", 0x11d5 },
  { "HANGUL JONGSEONG RIEUL-SSANGSIOS", 0x11d6 },
  { "HANGUL JONGSEONG RIEUL-PANSIOS", 0x11d7 },
  { "HANGUL JONGSEONG RIEUL-KHIEUKH", 0x11d8 },
  { "HANGUL JONGSEONG RIEUL-YEORINHIEUH", 0x11d9 },
  { "HANGUL JONGSEONG MIEUM-KIYEOK", 0x11da },
  { "HANGUL JONGSEONG MIEUM-RIEUL", 0x11db },
  { "HANGUL JONGSEONG MIEUM-PIEUP", 0x11dc },
  { "HANGUL JONGSEONG MIEUM-SIOS", 0x11dd },
  { "HANGUL JONGSEONG MIEUM-SSANGSIOS", 0x11de },
  { "HANGUL JONGSEONG MIEUM-PANSIOS", 0x11df },
  { "HANGUL JONGSEONG MIEUM-CHIEUCH", 0x11e0 },
  { "HANGUL JONGSEONG MIEUM-HIEUH", 0x11e1 },
  { "HANGUL JONGSEONG KAPYEOUNMIEUM", 0x11e2 },
  { "HANGUL JONGSEONG PIEUP-RIEUL", 0x11e3 },
  { "HANGUL JONGSEONG PIEUP-PHIEUPH", 0x11e4 },
  { "HANGUL JONGSEONG PIEUP-HIEUH", 0x11e5 },
  { "HANGUL JONGSEONG KAPYEOUNPIEUP", 0x11e6 },
  { "HANGUL JONGSEONG SIOS-KIYEOK", 0x11e7 },
  { "HANGUL JONGSEONG SIOS-TIKEUT", 0x11e8 },
  { "HANGUL JONGSEONG SIOS-RIEUL", 0x11e9 },
  { "HANGUL JONGSEONG SIOS-PIEUP", 0x11ea },
  { "HANGUL JONGSEONG PANSIOS", 0x11eb },
  { "HANGUL JONGSEONG IEUNG-KIYEOK", 0x11ec },
  { "HANGUL JONGSEONG IEUNG-SSANGKIYEOK", 0x11ed },
  { "HANGUL JONGSEONG SSANGIEUNG", 0x11ee },
  { "HANGUL JONGSEONG IEUNG-KHIEUKH", 0x11ef },
  { "HANGUL JONGSEONG YESIEUNG", 0x11f0 },
  { "HANGUL JONGSEONG YESIEUNG-SIOS", 0x11f1 },
  { "HANGUL JONGSEONG YESIEUNG-PANSIOS", 0x11f2 },
  { "HANGUL JONGSEONG PHIEUPH-PIEUP", 0x11f3 },
  { "HANGUL JONGSEONG KAPYEOUNPHIEUPH", 0x11f4 },
  { "HANGUL JONGSEONG HIEUH-NIEUN", 0x11f5 },
  { "HANGUL JONGSEONG HIEUH-RIEUL", 0x11f6 },
  { "HANGUL JONGSEONG HIEUH-MIEUM", 0x11f7 },
  { "HANGUL JONGSEONG HIEUH-PIEUP", 0x11f8 },
  { "HANGUL JONGSEONG YEORINHIEUH", 0x11f9 },
  { "ETHIOPIC SYLLABLE HA", 0x1200 },
  { "ETHIOPIC SYLLABLE HU", 0x1201 },
  { "ETHIOPIC SYLLABLE HI", 0x1202 },
  { "ETHIOPIC SYLLABLE HAA", 0x1203 },
  { "ETHIOPIC SYLLABLE HEE", 0x1204 },
  { "ETHIOPIC SYLLABLE HE", 0x1205 },
  { "ETHIOPIC SYLLABLE HO", 0x1206 },
  { "ETHIOPIC SYLLABLE LA", 0x1208 },
  { "ETHIOPIC SYLLABLE LU", 0x1209 },
  { "ETHIOPIC SYLLABLE LI", 0x120a },
  { "ETHIOPIC SYLLABLE LAA", 0x120b },
  { "ETHIOPIC SYLLABLE LEE", 0x120c },
  { "ETHIOPIC SYLLABLE LE", 0x120d },
  { "ETHIOPIC SYLLABLE LO", 0x120e },
  { "ETHIOPIC SYLLABLE LWA", 0x120f },
  { "ETHIOPIC SYLLABLE HHA", 0x1210 },
  { "ETHIOPIC SYLLABLE HHU", 0x1211 },
  { "ETHIOPIC SYLLABLE HHI", 0x1212 },
  { "ETHIOPIC SYLLABLE HHAA", 0x1213 },
  { "ETHIOPIC SYLLABLE HHEE", 0x1214 },
  { "ETHIOPIC SYLLABLE HHE", 0x1215 },
  { "ETHIOPIC SYLLABLE HHO", 0x1216 },
  { "ETHIOPIC SYLLABLE HHWA", 0x1217 },
  { "ETHIOPIC SYLLABLE MA", 0x1218 },
  { "ETHIOPIC SYLLABLE MU", 0x1219 },
  { "ETHIOPIC SYLLABLE MI", 0x121a },
  { "ETHIOPIC SYLLABLE MAA", 0x121b },
  { "ETHIOPIC SYLLABLE MEE", 0x121c },
  { "ETHIOPIC SYLLABLE ME", 0x121d },
  { "ETHIOPIC SYLLABLE MO", 0x121e },
  { "ETHIOPIC SYLLABLE MWA", 0x121f },
  { "ETHIOPIC SYLLABLE SZA", 0x1220 },
  { "ETHIOPIC SYLLABLE SZU", 0x1221 },
  { "ETHIOPIC SYLLABLE SZI", 0x1222 },
  { "ETHIOPIC SYLLABLE SZAA", 0x1223 },
  { "ETHIOPIC SYLLABLE SZEE", 0x1224 },
  { "ETHIOPIC SYLLABLE SZE", 0x1225 },
  { "ETHIOPIC SYLLABLE SZO", 0x1226 },
  { "ETHIOPIC SYLLABLE SZWA", 0x1227 },
  { "ETHIOPIC SYLLABLE RA", 0x1228 },
  { "ETHIOPIC SYLLABLE RU", 0x1229 },
  { "ETHIOPIC SYLLABLE RI", 0x122a },
  { "ETHIOPIC SYLLABLE RAA", 0x122b },
  { "ETHIOPIC SYLLABLE REE", 0x122c },
  { "ETHIOPIC SYLLABLE RE", 0x122d },
  { "ETHIOPIC SYLLABLE RO", 0x122e },
  { "ETHIOPIC SYLLABLE RWA", 0x122f },
  { "ETHIOPIC SYLLABLE SA", 0x1230 },
  { "ETHIOPIC SYLLABLE SU", 0x1231 },
  { "ETHIOPIC SYLLABLE SI", 0x1232 },
  { "ETHIOPIC SYLLABLE SAA", 0x1233 },
  { "ETHIOPIC SYLLABLE SEE", 0x1234 },
  { "ETHIOPIC SYLLABLE SE", 0x1235 },
  { "ETHIOPIC SYLLABLE SO", 0x1236 },
  { "ETHIOPIC SYLLABLE SWA", 0x1237 },
  { "ETHIOPIC SYLLABLE SHA", 0x1238 },
  { "ETHIOPIC SYLLABLE SHU", 0x1239 },
  { "ETHIOPIC SYLLABLE SHI", 0x123a },
  { "ETHIOPIC SYLLABLE SHAA", 0x123b },
  { "ETHIOPIC SYLLABLE SHEE", 0x123c },
  { "ETHIOPIC SYLLABLE SHE", 0x123d },
  { "ETHIOPIC SYLLABLE SHO", 0x123e },
  { "ETHIOPIC SYLLABLE SHWA", 0x123f },
  { "ETHIOPIC SYLLABLE QA", 0x1240 },
  { "ETHIOPIC SYLLABLE QU", 0x1241 },
  { "ETHIOPIC SYLLABLE QI", 0x1242 },
  { "ETHIOPIC SYLLABLE QAA", 0x1243 },
  { "ETHIOPIC SYLLABLE QEE", 0x1244 },
  { "ETHIOPIC SYLLABLE QE", 0x1245 },
  { "ETHIOPIC SYLLABLE QO", 0x1246 },
  { "ETHIOPIC SYLLABLE QWA", 0x1248 },
  { "ETHIOPIC SYLLABLE QWI", 0x124a },
  { "ETHIOPIC SYLLABLE QWAA", 0x124b },
  { "ETHIOPIC SYLLABLE QWEE", 0x124c },
  { "ETHIOPIC SYLLABLE QWE", 0x124d },
  { "ETHIOPIC SYLLABLE QHA", 0x1250 },
  { "ETHIOPIC SYLLABLE QHU", 0x1251 },
  { "ETHIOPIC SYLLABLE QHI", 0x1252 },
  { "ETHIOPIC SYLLABLE QHAA", 0x1253 },
  { "ETHIOPIC SYLLABLE QHEE", 0x1254 },
  { "ETHIOPIC SYLLABLE QHE", 0x1255 },
  { "ETHIOPIC SYLLABLE QHO", 0x1256 },
  { "ETHIOPIC SYLLABLE QHWA", 0x1258 },
  { "ETHIOPIC SYLLABLE QHWI", 0x125a },
  { "ETHIOPIC SYLLABLE QHWAA", 0x125b },
  { "ETHIOPIC SYLLABLE QHWEE", 0x125c },
  { "ETHIOPIC SYLLABLE QHWE", 0x125d },
  { "ETHIOPIC SYLLABLE BA", 0x1260 },
  { "ETHIOPIC SYLLABLE BU", 0x1261 },
  { "ETHIOPIC SYLLABLE BI", 0x1262 },
  { "ETHIOPIC SYLLABLE BAA", 0x1263 },
  { "ETHIOPIC SYLLABLE BEE", 0x1264 },
  { "ETHIOPIC SYLLABLE BE", 0x1265 },
  { "ETHIOPIC SYLLABLE BO", 0x1266 },
  { "ETHIOPIC SYLLABLE BWA", 0x1267 },
  { "ETHIOPIC SYLLABLE VA", 0x1268 },
  { "ETHIOPIC SYLLABLE VU", 0x1269 },
  { "ETHIOPIC SYLLABLE VI", 0x126a },
  { "ETHIOPIC SYLLABLE VAA", 0x126b },
  { "ETHIOPIC SYLLABLE VEE", 0x126c },
  { "ETHIOPIC SYLLABLE VE", 0x126d },
  { "ETHIOPIC SYLLABLE VO", 0x126e },
  { "ETHIOPIC SYLLABLE VWA", 0x126f },
  { "ETHIOPIC SYLLABLE TA", 0x1270 },
  { "ETHIOPIC SYLLABLE TU", 0x1271 },
  { "ETHIOPIC SYLLABLE TI", 0x1272 },
  { "ETHIOPIC SYLLABLE TAA", 0x1273 },
  { "ETHIOPIC SYLLABLE TEE", 0x1274 },
  { "ETHIOPIC SYLLABLE TE", 0x1275 },
  { "ETHIOPIC SYLLABLE TO", 0x1276 },
  { "ETHIOPIC SYLLABLE TWA", 0x1277 },
  { "ETHIOPIC SYLLABLE CA", 0x1278 },
  { "ETHIOPIC SYLLABLE CU", 0x1279 },
  { "ETHIOPIC SYLLABLE CI", 0x127a },
  { "ETHIOPIC SYLLABLE CAA", 0x127b },
  { "ETHIOPIC SYLLABLE CEE", 0x127c },
  { "ETHIOPIC SYLLABLE CE", 0x127d },
  { "ETHIOPIC SYLLABLE CO", 0x127e },
  { "ETHIOPIC SYLLABLE CWA", 0x127f },
  { "ETHIOPIC SYLLABLE XA", 0x1280 },
  { "ETHIOPIC SYLLABLE XU", 0x1281 },
  { "ETHIOPIC SYLLABLE XI", 0x1282 },
  { "ETHIOPIC SYLLABLE XAA", 0x1283 },
  { "ETHIOPIC SYLLABLE XEE", 0x1284 },
  { "ETHIOPIC SYLLABLE XE", 0x1285 },
  { "ETHIOPIC SYLLABLE XO", 0x1286 },
  { "ETHIOPIC SYLLABLE XWA", 0x1288 },
  { "ETHIOPIC SYLLABLE XWI", 0x128a },
  { "ETHIOPIC SYLLABLE XWAA", 0x128b },
  { "ETHIOPIC SYLLABLE XWEE", 0x128c },
  { "ETHIOPIC SYLLABLE XWE", 0x128d },
  { "ETHIOPIC SYLLABLE NA", 0x1290 },
  { "ETHIOPIC SYLLABLE NU", 0x1291 },
  { "ETHIOPIC SYLLABLE NI", 0x1292 },
  { "ETHIOPIC SYLLABLE NAA", 0x1293 },
  { "ETHIOPIC SYLLABLE NEE", 0x1294 },
  { "ETHIOPIC SYLLABLE NE", 0x1295 },
  { "ETHIOPIC SYLLABLE NO", 0x1296 },
  { "ETHIOPIC SYLLABLE NWA", 0x1297 },
  { "ETHIOPIC SYLLABLE NYA", 0x1298 },
  { "ETHIOPIC SYLLABLE NYU", 0x1299 },
  { "ETHIOPIC SYLLABLE NYI", 0x129a },
  { "ETHIOPIC SYLLABLE NYAA", 0x129b },
  { "ETHIOPIC SYLLABLE NYEE", 0x129c },
  { "ETHIOPIC SYLLABLE NYE", 0x129d },
  { "ETHIOPIC SYLLABLE NYO", 0x129e },
  { "ETHIOPIC SYLLABLE NYWA", 0x129f },
  { "ETHIOPIC SYLLABLE GLOTTAL A", 0x12a0 },
  { "ETHIOPIC SYLLABLE GLOTTAL U", 0x12a1 },
  { "ETHIOPIC SYLLABLE GLOTTAL I", 0x12a2 },
  { "ETHIOPIC SYLLABLE GLOTTAL AA", 0x12a3 },
  { "ETHIOPIC SYLLABLE GLOTTAL EE", 0x12a4 },
  { "ETHIOPIC SYLLABLE GLOTTAL E", 0x12a5 },
  { "ETHIOPIC SYLLABLE GLOTTAL O", 0x12a6 },
  { "ETHIOPIC SYLLABLE GLOTTAL WA", 0x12a7 },
  { "ETHIOPIC SYLLABLE KA", 0x12a8 },
  { "ETHIOPIC SYLLABLE KU", 0x12a9 },
  { "ETHIOPIC SYLLABLE KI", 0x12aa },
  { "ETHIOPIC SYLLABLE KAA", 0x12ab },
  { "ETHIOPIC SYLLABLE KEE", 0x12ac },
  { "ETHIOPIC SYLLABLE KE", 0x12ad },
  { "ETHIOPIC SYLLABLE KO", 0x12ae },
  { "ETHIOPIC SYLLABLE KWA", 0x12b0 },
  { "ETHIOPIC SYLLABLE KWI", 0x12b2 },
  { "ETHIOPIC SYLLABLE KWAA", 0x12b3 },
  { "ETHIOPIC SYLLABLE KWEE", 0x12b4 },
  { "ETHIOPIC SYLLABLE KWE", 0x12b5 },
  { "ETHIOPIC SYLLABLE KXA", 0x12b8 },
  { "ETHIOPIC SYLLABLE KXU", 0x12b9 },
  { "ETHIOPIC SYLLABLE KXI", 0x12ba },
  { "ETHIOPIC SYLLABLE KXAA", 0x12bb },
  { "ETHIOPIC SYLLABLE KXEE", 0x12bc },
  { "ETHIOPIC SYLLABLE KXE", 0x12bd },
  { "ETHIOPIC SYLLABLE KXO", 0x12be },
  { "ETHIOPIC SYLLABLE KXWA", 0x12c0 },
  { "ETHIOPIC SYLLABLE KXWI", 0x12c2 },
  { "ETHIOPIC SYLLABLE KXWAA", 0x12c3 },
  { "ETHIOPIC SYLLABLE KXWEE", 0x12c4 },
  { "ETHIOPIC SYLLABLE KXWE", 0x12c5 },
  { "ETHIOPIC SYLLABLE WA", 0x12c8 },
  { "ETHIOPIC SYLLABLE WU", 0x12c9 },
  { "ETHIOPIC SYLLABLE WI", 0x12ca },
  { "ETHIOPIC SYLLABLE WAA", 0x12cb },
  { "ETHIOPIC SYLLABLE WEE", 0x12cc },
  { "ETHIOPIC SYLLABLE WE", 0x12cd },
  { "ETHIOPIC SYLLABLE WO", 0x12ce },
  { "ETHIOPIC SYLLABLE PHARYNGEAL A", 0x12d0 },
  { "ETHIOPIC SYLLABLE PHARYNGEAL U", 0x12d1 },
  { "ETHIOPIC SYLLABLE PHARYNGEAL I", 0x12d2 },
  { "ETHIOPIC SYLLABLE PHARYNGEAL AA", 0x12d3 },
  { "ETHIOPIC SYLLABLE PHARYNGEAL EE", 0x12d4 },
  { "ETHIOPIC SYLLABLE PHARYNGEAL E", 0x12d5 },
  { "ETHIOPIC SYLLABLE PHARYNGEAL O", 0x12d6 },
  { "ETHIOPIC SYLLABLE ZA", 0x12d8 },
  { "ETHIOPIC SYLLABLE ZU", 0x12d9 },
  { "ETHIOPIC SYLLABLE ZI", 0x12da },
  { "ETHIOPIC SYLLABLE ZAA", 0x12db },
  { "ETHIOPIC SYLLABLE ZEE", 0x12dc },
  { "ETHIOPIC SYLLABLE ZE", 0x12dd },
  { "ETHIOPIC SYLLABLE ZO", 0x12de },
  { "ETHIOPIC SYLLABLE ZWA", 0x12df },
  { "ETHIOPIC SYLLABLE ZHA", 0x12e0 },
  { "ETHIOPIC SYLLABLE ZHU", 0x12e1 },
  { "ETHIOPIC SYLLABLE ZHI", 0x12e2 },
  { "ETHIOPIC SYLLABLE ZHAA", 0x12e3 },
  { "ETHIOPIC SYLLABLE ZHEE", 0x12e4 },
  { "ETHIOPIC SYLLABLE ZHE", 0x12e5 },
  { "ETHIOPIC SYLLABLE ZHO", 0x12e6 },
  { "ETHIOPIC SYLLABLE ZHWA", 0x12e7 },
  { "ETHIOPIC SYLLABLE YA", 0x12e8 },
  { "ETHIOPIC SYLLABLE YU", 0x12e9 },
  { "ETHIOPIC SYLLABLE YI", 0x12ea },
  { "ETHIOPIC SYLLABLE YAA", 0x12eb },
  { "ETHIOPIC SYLLABLE YEE", 0x12ec },
  { "ETHIOPIC SYLLABLE YE", 0x12ed },
  { "ETHIOPIC SYLLABLE YO", 0x12ee },
  { "ETHIOPIC SYLLABLE DA", 0x12f0 },
  { "ETHIOPIC SYLLABLE DU", 0x12f1 },
  { "ETHIOPIC SYLLABLE DI", 0x12f2 },
  { "ETHIOPIC SYLLABLE DAA", 0x12f3 },
  { "ETHIOPIC SYLLABLE DEE", 0x12f4 },
  { "ETHIOPIC SYLLABLE DE", 0x12f5 },
  { "ETHIOPIC SYLLABLE DO", 0x12f6 },
  { "ETHIOPIC SYLLABLE DWA", 0x12f7 },
  { "ETHIOPIC SYLLABLE DDA", 0x12f8 },
  { "ETHIOPIC SYLLABLE DDU", 0x12f9 },
  { "ETHIOPIC SYLLABLE DDI", 0x12fa },
  { "ETHIOPIC SYLLABLE DDAA", 0x12fb },
  { "ETHIOPIC SYLLABLE DDEE", 0x12fc },
  { "ETHIOPIC SYLLABLE DDE", 0x12fd },
  { "ETHIOPIC SYLLABLE DDO", 0x12fe },
  { "ETHIOPIC SYLLABLE DDWA", 0x12ff },
  { "ETHIOPIC SYLLABLE JA", 0x1300 },
  { "ETHIOPIC SYLLABLE JU", 0x1301 },
  { "ETHIOPIC SYLLABLE JI", 0x1302 },
  { "ETHIOPIC SYLLABLE JAA", 0x1303 },
  { "ETHIOPIC SYLLABLE JEE", 0x1304 },
  { "ETHIOPIC SYLLABLE JE", 0x1305 },
  { "ETHIOPIC SYLLABLE JO", 0x1306 },
  { "ETHIOPIC SYLLABLE JWA", 0x1307 },
  { "ETHIOPIC SYLLABLE GA", 0x1308 },
  { "ETHIOPIC SYLLABLE GU", 0x1309 },
  { "ETHIOPIC SYLLABLE GI", 0x130a },
  { "ETHIOPIC SYLLABLE GAA", 0x130b },
  { "ETHIOPIC SYLLABLE GEE", 0x130c },
  { "ETHIOPIC SYLLABLE GE", 0x130d },
  { "ETHIOPIC SYLLABLE GO", 0x130e },
  { "ETHIOPIC SYLLABLE GWA", 0x1310 },
  { "ETHIOPIC SYLLABLE GWI", 0x1312 },
  { "ETHIOPIC SYLLABLE GWAA", 0x1313 },
  { "ETHIOPIC SYLLABLE GWEE", 0x1314 },
  { "ETHIOPIC SYLLABLE GWE", 0x1315 },
  { "ETHIOPIC SYLLABLE GGA", 0x1318 },
  { "ETHIOPIC SYLLABLE GGU", 0x1319 },
  { "ETHIOPIC SYLLABLE GGI", 0x131a },
  { "ETHIOPIC SYLLABLE GGAA", 0x131b },
  { "ETHIOPIC SYLLABLE GGEE", 0x131c },
  { "ETHIOPIC SYLLABLE GGE", 0x131d },
  { "ETHIOPIC SYLLABLE GGO", 0x131e },
  { "ETHIOPIC SYLLABLE THA", 0x1320 },
  { "ETHIOPIC SYLLABLE THU", 0x1321 },
  { "ETHIOPIC SYLLABLE THI", 0x1322 },
  { "ETHIOPIC SYLLABLE THAA", 0x1323 },
  { "ETHIOPIC SYLLABLE THEE", 0x1324 },
  { "ETHIOPIC SYLLABLE THE", 0x1325 },
  { "ETHIOPIC SYLLABLE THO", 0x1326 },
  { "ETHIOPIC SYLLABLE THWA", 0x1327 },
  { "ETHIOPIC SYLLABLE CHA", 0x1328 },
  { "ETHIOPIC SYLLABLE CHU", 0x1329 },
  { "ETHIOPIC SYLLABLE CHI", 0x132a },
  { "ETHIOPIC SYLLABLE CHAA", 0x132b },
  { "ETHIOPIC SYLLABLE CHEE", 0x132c },
  { "ETHIOPIC SYLLABLE CHE", 0x132d },
  { "ETHIOPIC SYLLABLE CHO", 0x132e },
  { "ETHIOPIC SYLLABLE CHWA", 0x132f },
  { "ETHIOPIC SYLLABLE PHA", 0x1330 },
  { "ETHIOPIC SYLLABLE PHU", 0x1331 },
  { "ETHIOPIC SYLLABLE PHI", 0x1332 },
  { "ETHIOPIC SYLLABLE PHAA", 0x1333 },
  { "ETHIOPIC SYLLABLE PHEE", 0x1334 },
  { "ETHIOPIC SYLLABLE PHE", 0x1335 },
  { "ETHIOPIC SYLLABLE PHO", 0x1336 },
  { "ETHIOPIC SYLLABLE PHWA", 0x1337 },
  { "ETHIOPIC SYLLABLE TSA", 0x1338 },
  { "ETHIOPIC SYLLABLE TSU", 0x1339 },
  { "ETHIOPIC SYLLABLE TSI", 0x133a },
  { "ETHIOPIC SYLLABLE TSAA", 0x133b },
  { "ETHIOPIC SYLLABLE TSEE", 0x133c },
  { "ETHIOPIC SYLLABLE TSE", 0x133d },
  { "ETHIOPIC SYLLABLE TSO", 0x133e },
  { "ETHIOPIC SYLLABLE TSWA", 0x133f },
  { "ETHIOPIC SYLLABLE TZA", 0x1340 },
  { "ETHIOPIC SYLLABLE TZU", 0x1341 },
  { "ETHIOPIC SYLLABLE TZI", 0x1342 },
  { "ETHIOPIC SYLLABLE TZAA", 0x1343 },
  { "ETHIOPIC SYLLABLE TZEE", 0x1344 },
  { "ETHIOPIC SYLLABLE TZE", 0x1345 },
  { "ETHIOPIC SYLLABLE TZO", 0x1346 },
  { "ETHIOPIC SYLLABLE FA", 0x1348 },
  { "ETHIOPIC SYLLABLE FU", 0x1349 },
  { "ETHIOPIC SYLLABLE FI", 0x134a },
  { "ETHIOPIC SYLLABLE FAA", 0x134b },
  { "ETHIOPIC SYLLABLE FEE", 0x134c },
  { "ETHIOPIC SYLLABLE FE", 0x134d },
  { "ETHIOPIC SYLLABLE FO", 0x134e },
  { "ETHIOPIC SYLLABLE FWA", 0x134f },
  { "ETHIOPIC SYLLABLE PA", 0x1350 },
  { "ETHIOPIC SYLLABLE PU", 0x1351 },
  { "ETHIOPIC SYLLABLE PI", 0x1352 },
  { "ETHIOPIC SYLLABLE PAA", 0x1353 },
  { "ETHIOPIC SYLLABLE PEE", 0x1354 },
  { "ETHIOPIC SYLLABLE PE", 0x1355 },
  { "ETHIOPIC SYLLABLE PO", 0x1356 },
  { "ETHIOPIC SYLLABLE PWA", 0x1357 },
  { "ETHIOPIC SYLLABLE RYA", 0x1358 },
  { "ETHIOPIC SYLLABLE MYA", 0x1359 },
  { "ETHIOPIC SYLLABLE FYA", 0x135a },
  { "ETHIOPIC WORDSPACE", 0x1361 },
  { "ETHIOPIC FULL STOP", 0x1362 },
  { "ETHIOPIC COMMA", 0x1363 },
  { "ETHIOPIC SEMICOLON", 0x1364 },
  { "ETHIOPIC COLON", 0x1365 },
  { "ETHIOPIC PREFACE COLON", 0x1366 },
  { "ETHIOPIC QUESTION MARK", 0x1367 },
  { "ETHIOPIC PARAGRAPH SEPARATOR", 0x1368 },
  { "ETHIOPIC DIGIT ONE", 0x1369 },
  { "ETHIOPIC DIGIT TWO", 0x136a },
  { "ETHIOPIC DIGIT THREE", 0x136b },
  { "ETHIOPIC DIGIT FOUR", 0x136c },
  { "ETHIOPIC DIGIT FIVE", 0x136d },
  { "ETHIOPIC DIGIT SIX", 0x136e },
  { "ETHIOPIC DIGIT SEVEN", 0x136f },
  { "ETHIOPIC DIGIT EIGHT", 0x1370 },
  { "ETHIOPIC DIGIT NINE", 0x1371 },
  { "ETHIOPIC NUMBER TEN", 0x1372 },
  { "ETHIOPIC NUMBER TWENTY", 0x1373 },
  { "ETHIOPIC NUMBER THIRTY", 0x1374 },
  { "ETHIOPIC NUMBER FORTY", 0x1375 },
  { "ETHIOPIC NUMBER FIFTY", 0x1376 },
  { "ETHIOPIC NUMBER SIXTY", 0x1377 },
  { "ETHIOPIC NUMBER SEVENTY", 0x1378 },
  { "ETHIOPIC NUMBER EIGHTY", 0x1379 },
  { "ETHIOPIC NUMBER NINETY", 0x137a },
  { "ETHIOPIC NUMBER HUNDRED", 0x137b },
  { "ETHIOPIC NUMBER TEN THOUSAND", 0x137c },
  { "CHEROKEE LETTER A", 0x13a0 },
  { "CHEROKEE LETTER E", 0x13a1 },
  { "CHEROKEE LETTER I", 0x13a2 },
  { "CHEROKEE LETTER O", 0x13a3 },
  { "CHEROKEE LETTER U", 0x13a4 },
  { "CHEROKEE LETTER V", 0x13a5 },
  { "CHEROKEE LETTER GA", 0x13a6 },
  { "CHEROKEE LETTER KA", 0x13a7 },
  { "CHEROKEE LETTER GE", 0x13a8 },
  { "CHEROKEE LETTER GI", 0x13a9 },
  { "CHEROKEE LETTER GO", 0x13aa },
  { "CHEROKEE LETTER GU", 0x13ab },
  { "CHEROKEE LETTER GV", 0x13ac },
  { "CHEROKEE LETTER HA", 0x13ad },
  { "CHEROKEE LETTER HE", 0x13ae },
  { "CHEROKEE LETTER HI", 0x13af },
  { "CHEROKEE LETTER HO", 0x13b0 },
  { "CHEROKEE LETTER HU", 0x13b1 },
  { "CHEROKEE LETTER HV", 0x13b2 },
  { "CHEROKEE LETTER LA", 0x13b3 },
  { "CHEROKEE LETTER LE", 0x13b4 },
  { "CHEROKEE LETTER LI", 0x13b5 },
  { "CHEROKEE LETTER LO", 0x13b6 },
  { "CHEROKEE LETTER LU", 0x13b7 },
  { "CHEROKEE LETTER LV", 0x13b8 },
  { "CHEROKEE LETTER MA", 0x13b9 },
  { "CHEROKEE LETTER ME", 0x13ba },
  { "CHEROKEE LETTER MI", 0x13bb },
  { "CHEROKEE LETTER MO", 0x13bc },
  { "CHEROKEE LETTER MU", 0x13bd },
  { "CHEROKEE LETTER NA", 0x13be },
  { "CHEROKEE LETTER HNA", 0x13bf },
  { "CHEROKEE LETTER NAH", 0x13c0 },
  { "CHEROKEE LETTER NE", 0x13c1 },
  { "CHEROKEE LETTER NI", 0x13c2 },
  { "CHEROKEE LETTER NO", 0x13c3 },
  { "CHEROKEE LETTER NU", 0x13c4 },
  { "CHEROKEE LETTER NV", 0x13c5 },
  { "CHEROKEE LETTER QUA", 0x13c6 },
  { "CHEROKEE LETTER QUE", 0x13c7 },
  { "CHEROKEE LETTER QUI", 0x13c8 },
  { "CHEROKEE LETTER QUO", 0x13c9 },
  { "CHEROKEE LETTER QUU", 0x13ca },
  { "CHEROKEE LETTER QUV", 0x13cb },
  { "CHEROKEE LETTER SA", 0x13cc },
  { "CHEROKEE LETTER S", 0x13cd },
  { "CHEROKEE LETTER SE", 0x13ce },
  { "CHEROKEE LETTER SI", 0x13cf },
  { "CHEROKEE LETTER SO", 0x13d0 },
  { "CHEROKEE LETTER SU", 0x13d1 },
  { "CHEROKEE LETTER SV", 0x13d2 },
  { "CHEROKEE LETTER DA", 0x13d3 },
  { "CHEROKEE LETTER TA", 0x13d4 },
  { "CHEROKEE LETTER DE", 0x13d5 },
  { "CHEROKEE LETTER TE", 0x13d6 },
  { "CHEROKEE LETTER DI", 0x13d7 },
  { "CHEROKEE LETTER TI", 0x13d8 },
  { "CHEROKEE LETTER DO", 0x13d9 },
  { "CHEROKEE LETTER DU", 0x13da },
  { "CHEROKEE LETTER DV", 0x13db },
  { "CHEROKEE LETTER DLA", 0x13dc },
  { "CHEROKEE LETTER TLA", 0x13dd },
  { "CHEROKEE LETTER TLE", 0x13de },
  { "CHEROKEE LETTER TLI", 0x13df },
  { "CHEROKEE LETTER TLO", 0x13e0 },
  { "CHEROKEE LETTER TLU", 0x13e1 },
  { "CHEROKEE LETTER TLV", 0x13e2 },
  { "CHEROKEE LETTER TSA", 0x13e3 },
  { "CHEROKEE LETTER TSE", 0x13e4 },
  { "CHEROKEE LETTER TSI", 0x13e5 },
  { "CHEROKEE LETTER TSO", 0x13e6 },
  { "CHEROKEE LETTER TSU", 0x13e7 },
  { "CHEROKEE LETTER TSV", 0x13e8 },
  { "CHEROKEE LETTER WA", 0x13e9 },
  { "CHEROKEE LETTER WE", 0x13ea },
  { "CHEROKEE LETTER WI", 0x13eb },
  { "CHEROKEE LETTER WO", 0x13ec },
  { "CHEROKEE LETTER WU", 0x13ed },
  { "CHEROKEE LETTER WV", 0x13ee },
  { "CHEROKEE LETTER YA", 0x13ef },
  { "CHEROKEE LETTER YE", 0x13f0 },
  { "CHEROKEE LETTER YI", 0x13f1 },
  { "CHEROKEE LETTER YO", 0x13f2 },
  { "CHEROKEE LETTER YU", 0x13f3 },
  { "CHEROKEE LETTER YV", 0x13f4 },
  { "CANADIAN SYLLABICS E", 0x1401 },
  { "CANADIAN SYLLABICS AAI", 0x1402 },
  { "CANADIAN SYLLABICS I", 0x1403 },
  { "CANADIAN SYLLABICS II", 0x1404 },
  { "CANADIAN SYLLABICS O", 0x1405 },
  { "CANADIAN SYLLABICS OO", 0x1406 },
  { "CANADIAN SYLLABICS Y-CREE OO", 0x1407 },
  { "CANADIAN SYLLABICS CARRIER EE", 0x1408 },
  { "CANADIAN SYLLABICS CARRIER I", 0x1409 },
  { "CANADIAN SYLLABICS A", 0x140a },
  { "CANADIAN SYLLABICS AA", 0x140b },
  { "CANADIAN SYLLABICS WE", 0x140c },
  { "CANADIAN SYLLABICS WEST-CREE WE", 0x140d },
  { "CANADIAN SYLLABICS WI", 0x140e },
  { "CANADIAN SYLLABICS WEST-CREE WI", 0x140f },
  { "CANADIAN SYLLABICS WII", 0x1410 },
  { "CANADIAN SYLLABICS WEST-CREE WII", 0x1411 },
  { "CANADIAN SYLLABICS WO", 0x1412 },
  { "CANADIAN SYLLABICS WEST-CREE WO", 0x1413 },
  { "CANADIAN SYLLABICS WOO", 0x1414 },
  { "CANADIAN SYLLABICS WEST-CREE WOO", 0x1415 },
  { "CANADIAN SYLLABICS NASKAPI WOO", 0x1416 },
  { "CANADIAN SYLLABICS WA", 0x1417 },
  { "CANADIAN SYLLABICS WEST-CREE WA", 0x1418 },
  { "CANADIAN SYLLABICS WAA", 0x1419 },
  { "CANADIAN SYLLABICS WEST-CREE WAA", 0x141a },
  { "CANADIAN SYLLABICS NASKAPI WAA", 0x141b },
  { "CANADIAN SYLLABICS AI", 0x141c },
  { "CANADIAN SYLLABICS Y-CREE W", 0x141d },
  { "CANADIAN SYLLABICS GLOTTAL STOP", 0x141e },
  { "CANADIAN SYLLABICS FINAL ACUTE", 0x141f },
  { "CANADIAN SYLLABICS FINAL GRAVE", 0x1420 },
  { "CANADIAN SYLLABICS FINAL BOTTOM HALF RING", 0x1421 },
  { "CANADIAN SYLLABICS FINAL TOP HALF RING", 0x1422 },
  { "CANADIAN SYLLABICS FINAL RIGHT HALF RING", 0x1423 },
  { "CANADIAN SYLLABICS FINAL RING", 0x1424 },
  { "CANADIAN SYLLABICS FINAL DOUBLE ACUTE", 0x1425 },
  { "CANADIAN SYLLABICS FINAL DOUBLE SHORT VERTICAL STROKES", 0x1426 },
  { "CANADIAN SYLLABICS FINAL MIDDLE DOT", 0x1427 },
  { "CANADIAN SYLLABICS FINAL SHORT HORIZONTAL STROKE", 0x1428 },
  { "CANADIAN SYLLABICS FINAL PLUS", 0x1429 },
  { "CANADIAN SYLLABICS FINAL DOWN TACK", 0x142a },
  { "CANADIAN SYLLABICS EN", 0x142b },
  { "CANADIAN SYLLABICS IN", 0x142c },
  { "CANADIAN SYLLABICS ON", 0x142d },
  { "CANADIAN SYLLABICS AN", 0x142e },
  { "CANADIAN SYLLABICS PE", 0x142f },
  { "CANADIAN SYLLABICS PAAI", 0x1430 },
  { "CANADIAN SYLLABICS PI", 0x1431 },
  { "CANADIAN SYLLABICS PII", 0x1432 },
  { "CANADIAN SYLLABICS PO", 0x1433 },
  { "CANADIAN SYLLABICS POO", 0x1434 },
  { "CANADIAN SYLLABICS Y-CREE POO", 0x1435 },
  { "CANADIAN SYLLABICS CARRIER HEE", 0x1436 },
  { "CANADIAN SYLLABICS CARRIER HI", 0x1437 },
  { "CANADIAN SYLLABICS PA", 0x1438 },
  { "CANADIAN SYLLABICS PAA", 0x1439 },
  { "CANADIAN SYLLABICS PWE", 0x143a },
  { "CANADIAN SYLLABICS WEST-CREE PWE", 0x143b },
  { "CANADIAN SYLLABICS PWI", 0x143c },
  { "CANADIAN SYLLABICS WEST-CREE PWI", 0x143d },
  { "CANADIAN SYLLABICS PWII", 0x143e },
  { "CANADIAN SYLLABICS WEST-CREE PWII", 0x143f },
  { "CANADIAN SYLLABICS PWO", 0x1440 },
  { "CANADIAN SYLLABICS WEST-CREE PWO", 0x1441 },
  { "CANADIAN SYLLABICS PWOO", 0x1442 },
  { "CANADIAN SYLLABICS WEST-CREE PWOO", 0x1443 },
  { "CANADIAN SYLLABICS PWA", 0x1444 },
  { "CANADIAN SYLLABICS WEST-CREE PWA", 0x1445 },
  { "CANADIAN SYLLABICS PWAA", 0x1446 },
  { "CANADIAN SYLLABICS WEST-CREE PWAA", 0x1447 },
  { "CANADIAN SYLLABICS Y-CREE PWAA", 0x1448 },
  { "CANADIAN SYLLABICS P", 0x1449 },
  { "CANADIAN SYLLABICS WEST-CREE P", 0x144a },
  { "CANADIAN SYLLABICS CARRIER H", 0x144b },
  { "CANADIAN SYLLABICS TE", 0x144c },
  { "CANADIAN SYLLABICS TAAI", 0x144d },
  { "CANADIAN SYLLABICS TI", 0x144e },
  { "CANADIAN SYLLABICS TII", 0x144f },
  { "CANADIAN SYLLABICS TO", 0x1450 },
  { "CANADIAN SYLLABICS TOO", 0x1451 },
  { "CANADIAN SYLLABICS Y-CREE TOO", 0x1452 },
  { "CANADIAN SYLLABICS CARRIER DEE", 0x1453 },
  { "CANADIAN SYLLABICS CARRIER DI", 0x1454 },
  { "CANADIAN SYLLABICS TA", 0x1455 },
  { "CANADIAN SYLLABICS TAA", 0x1456 },
  { "CANADIAN SYLLABICS TWE", 0x1457 },
  { "CANADIAN SYLLABICS WEST-CREE TWE", 0x1458 },
  { "CANADIAN SYLLABICS TWI", 0x1459 },
  { "CANADIAN SYLLABICS WEST-CREE TWI", 0x145a },
  { "CANADIAN SYLLABICS TWII", 0x145b },
  { "CANADIAN SYLLABICS WEST-CREE TWII", 0x145c },
  { "CANADIAN SYLLABICS TWO", 0x145d },
  { "CANADIAN SYLLABICS WEST-CREE TWO", 0x145e },
  { "CANADIAN SYLLABICS TWOO", 0x145f },
  { "CANADIAN SYLLABICS WEST-CREE TWOO", 0x1460 },
  { "CANADIAN SYLLABICS TWA", 0x1461 },
  { "CANADIAN SYLLABICS WEST-CREE TWA", 0x1462 },
  { "CANADIAN SYLLABICS TWAA", 0x1463 },
  { "CANADIAN SYLLABICS WEST-CREE TWAA", 0x1464 },
  { "CANADIAN SYLLABICS NASKAPI TWAA", 0x1465 },
  { "CANADIAN SYLLABICS T", 0x1466 },
  { "CANADIAN SYLLABICS TTE", 0x1467 },
  { "CANADIAN SYLLABICS TTI", 0x1468 },
  { "CANADIAN SYLLABICS TTO", 0x1469 },
  { "CANADIAN SYLLABICS TTA", 0x146a },
  { "CANADIAN SYLLABICS KE", 0x146b },
  { "CANADIAN SYLLABICS KAAI", 0x146c },
  { "CANADIAN SYLLABICS KI", 0x146d },
  { "CANADIAN SYLLABICS KII", 0x146e },
  { "CANADIAN SYLLABICS KO", 0x146f },
  { "CANADIAN SYLLABICS KOO", 0x1470 },
  { "CANADIAN SYLLABICS Y-CREE KOO", 0x1471 },
  { "CANADIAN SYLLABICS KA", 0x1472 },
  { "CANADIAN SYLLABICS KAA", 0x1473 },
  { "CANADIAN SYLLABICS KWE", 0x1474 },
  { "CANADIAN SYLLABICS WEST-CREE KWE", 0x1475 },
  { "CANADIAN SYLLABICS KWI", 0x1476 },
  { "CANADIAN SYLLABICS WEST-CREE KWI", 0x1477 },
  { "CANADIAN SYLLABICS KWII", 0x1478 },
  { "CANADIAN SYLLABICS WEST-CREE KWII", 0x1479 },
  { "CANADIAN SYLLABICS KWO", 0x147a },
  { "CANADIAN SYLLABICS WEST-CREE KWO", 0x147b },
  { "CANADIAN SYLLABICS KWOO", 0x147c },
  { "CANADIAN SYLLABICS WEST-CREE KWOO", 0x147d },
  { "CANADIAN SYLLABICS KWA", 0x147e },
  { "CANADIAN SYLLABICS WEST-CREE KWA", 0x147f },
  { "CANADIAN SYLLABICS KWAA", 0x1480 },
  { "CANADIAN SYLLABICS WEST-CREE KWAA", 0x1481 },
  { "CANADIAN SYLLABICS NASKAPI KWAA", 0x1482 },
  { "CANADIAN SYLLABICS K", 0x1483 },
  { "CANADIAN SYLLABICS KW", 0x1484 },
  { "CANADIAN SYLLABICS SOUTH-SLAVEY KEH", 0x1485 },
  { "CANADIAN SYLLABICS SOUTH-SLAVEY KIH", 0x1486 },
  { "CANADIAN SYLLABICS SOUTH-SLAVEY KOH", 0x1487 },
  { "CANADIAN SYLLABICS SOUTH-SLAVEY KAH", 0x1488 },
  { "CANADIAN SYLLABICS CE", 0x1489 },
  { "CANADIAN SYLLABICS CAAI", 0x148a },
  { "CANADIAN SYLLABICS CI", 0x148b },
  { "CANADIAN SYLLABICS CII", 0x148c },
  { "CANADIAN SYLLABICS CO", 0x148d },
  { "CANADIAN SYLLABICS COO", 0x148e },
  { "CANADIAN SYLLABICS Y-CREE COO", 0x148f },
  { "CANADIAN SYLLABICS CA", 0x1490 },
  { "CANADIAN SYLLABICS CAA", 0x1491 },
  { "CANADIAN SYLLABICS CWE", 0x1492 },
  { "CANADIAN SYLLABICS WEST-CREE CWE", 0x1493 },
  { "CANADIAN SYLLABICS CWI", 0x1494 },
  { "CANADIAN SYLLABICS WEST-CREE CWI", 0x1495 },
  { "CANADIAN SYLLABICS CWII", 0x1496 },
  { "CANADIAN SYLLABICS WEST-CREE CWII", 0x1497 },
  { "CANADIAN SYLLABICS CWO", 0x1498 },
  { "CANADIAN SYLLABICS WEST-CREE CWO", 0x1499 },
  { "CANADIAN SYLLABICS CWOO", 0x149a },
  { "CANADIAN SYLLABICS WEST-CREE CWOO", 0x149b },
  { "CANADIAN SYLLABICS CWA", 0x149c },
  { "CANADIAN SYLLABICS WEST-CREE CWA", 0x149d },
  { "CANADIAN SYLLABICS CWAA", 0x149e },
  { "CANADIAN SYLLABICS WEST-CREE CWAA", 0x149f },
  { "CANADIAN SYLLABICS NASKAPI CWAA", 0x14a0 },
  { "CANADIAN SYLLABICS C", 0x14a1 },
  { "CANADIAN SYLLABICS SAYISI TH", 0x14a2 },
  { "CANADIAN SYLLABICS ME", 0x14a3 },
  { "CANADIAN SYLLABICS MAAI", 0x14a4 },
  { "CANADIAN SYLLABICS MI", 0x14a5 },
  { "CANADIAN SYLLABICS MII", 0x14a6 },
  { "CANADIAN SYLLABICS MO", 0x14a7 },
  { "CANADIAN SYLLABICS MOO", 0x14a8 },
  { "CANADIAN SYLLABICS Y-CREE MOO", 0x14a9 },
  { "CANADIAN SYLLABICS MA", 0x14aa },
  { "CANADIAN SYLLABICS MAA", 0x14ab },
  { "CANADIAN SYLLABICS MWE", 0x14ac },
  { "CANADIAN SYLLABICS WEST-CREE MWE", 0x14ad },
  { "CANADIAN SYLLABICS MWI", 0x14ae },
  { "CANADIAN SYLLABICS WEST-CREE MWI", 0x14af },
  { "CANADIAN SYLLABICS MWII", 0x14b0 },
  { "CANADIAN SYLLABICS WEST-CREE MWII", 0x14b1 },
  { "CANADIAN SYLLABICS MWO", 0x14b2 },
  { "CANADIAN SYLLABICS WEST-CREE MWO", 0x14b3 },
  { "CANADIAN SYLLABICS MWOO", 0x14b4 },
  { "CANADIAN SYLLABICS WEST-CREE MWOO", 0x14b5 },
  { "CANADIAN SYLLABICS MWA", 0x14b6 },
  { "CANADIAN SYLLABICS WEST-CREE MWA", 0x14b7 },
  { "CANADIAN SYLLABICS MWAA", 0x14b8 },
  { "CANADIAN SYLLABICS WEST-CREE MWAA", 0x14b9 },
  { "CANADIAN SYLLABICS NASKAPI MWAA", 0x14ba },
  { "CANADIAN SYLLABICS M", 0x14bb },
  { "CANADIAN SYLLABICS WEST-CREE M", 0x14bc },
  { "CANADIAN SYLLABICS MH", 0x14bd },
  { "CANADIAN SYLLABICS ATHAPASCAN M", 0x14be },
  { "CANADIAN SYLLABICS SAYISI M", 0x14bf },
  { "CANADIAN SYLLABICS NE", 0x14c0 },
  { "CANADIAN SYLLABICS NAAI", 0x14c1 },
  { "CANADIAN SYLLABICS NI", 0x14c2 },
  { "CANADIAN SYLLABICS NII", 0x14c3 },
  { "CANADIAN SYLLABICS NO", 0x14c4 },
  { "CANADIAN SYLLABICS NOO", 0x14c5 },
  { "CANADIAN SYLLABICS Y-CREE NOO", 0x14c6 },
  { "CANADIAN SYLLABICS NA", 0x14c7 },
  { "CANADIAN SYLLABICS NAA", 0x14c8 },
  { "CANADIAN SYLLABICS NWE", 0x14c9 },
  { "CANADIAN SYLLABICS WEST-CREE NWE", 0x14ca },
  { "CANADIAN SYLLABICS NWA", 0x14cb },
  { "CANADIAN SYLLABICS WEST-CREE NWA", 0x14cc },
  { "CANADIAN SYLLABICS NWAA", 0x14cd },
  { "CANADIAN SYLLABICS WEST-CREE NWAA", 0x14ce },
  { "CANADIAN SYLLABICS NASKAPI NWAA", 0x14cf },
  { "CANADIAN SYLLABICS N", 0x14d0 },
  { "CANADIAN SYLLABICS CARRIER NG", 0x14d1 },
  { "CANADIAN SYLLABICS NH", 0x14d2 },
  { "CANADIAN SYLLABICS LE", 0x14d3 },
  { "CANADIAN SYLLABICS LAAI", 0x14d4 },
  { "CANADIAN SYLLABICS LI", 0x14d5 },
  { "CANADIAN SYLLABICS LII", 0x14d6 },
  { "CANADIAN SYLLABICS LO", 0x14d7 },
  { "CANADIAN SYLLABICS LOO", 0x14d8 },
  { "CANADIAN SYLLABICS Y-CREE LOO", 0x14d9 },
  { "CANADIAN SYLLABICS LA", 0x14da },
  { "CANADIAN SYLLABICS LAA", 0x14db },
  { "CANADIAN SYLLABICS LWE", 0x14dc },
  { "CANADIAN SYLLABICS WEST-CREE LWE", 0x14dd },
  { "CANADIAN SYLLABICS LWI", 0x14de },
  { "CANADIAN SYLLABICS WEST-CREE LWI", 0x14df },
  { "CANADIAN SYLLABICS LWII", 0x14e0 },
  { "CANADIAN SYLLABICS WEST-CREE LWII", 0x14e1 },
  { "CANADIAN SYLLABICS LWO", 0x14e2 },
  { "CANADIAN SYLLABICS WEST-CREE LWO", 0x14e3 },
  { "CANADIAN SYLLABICS LWOO", 0x14e4 },
  { "CANADIAN SYLLABICS WEST-CREE LWOO", 0x14e5 },
  { "CANADIAN SYLLABICS LWA", 0x14e6 },
  { "CANADIAN SYLLABICS WEST-CREE LWA", 0x14e7 },
  { "CANADIAN SYLLABICS LWAA", 0x14e8 },
  { "CANADIAN SYLLABICS WEST-CREE LWAA", 0x14e9 },
  { "CANADIAN SYLLABICS L", 0x14ea },
  { "CANADIAN SYLLABICS WEST-CREE L", 0x14eb },
  { "CANADIAN SYLLABICS MEDIAL L", 0x14ec },
  { "CANADIAN SYLLABICS SE", 0x14ed },
  { "CANADIAN SYLLABICS SAAI", 0x14ee },
  { "CANADIAN SYLLABICS SI", 0x14ef },
  { "CANADIAN SYLLABICS SII", 0x14f0 },
  { "CANADIAN SYLLABICS SO", 0x14f1 },
  { "CANADIAN SYLLABICS SOO", 0x14f2 },
  { "CANADIAN SYLLABICS Y-CREE SOO", 0x14f3 },
  { "CANADIAN SYLLABICS SA", 0x14f4 },
  { "CANADIAN SYLLABICS SAA", 0x14f5 },
  { "CANADIAN SYLLABICS SWE", 0x14f6 },
  { "CANADIAN SYLLABICS WEST-CREE SWE", 0x14f7 },
  { "CANADIAN SYLLABICS SWI", 0x14f8 },
  { "CANADIAN SYLLABICS WEST-CREE SWI", 0x14f9 },
  { "CANADIAN SYLLABICS SWII", 0x14fa },
  { "CANADIAN SYLLABICS WEST-CREE SWII", 0x14fb },
  { "CANADIAN SYLLABICS SWO", 0x14fc },
  { "CANADIAN SYLLABICS WEST-CREE SWO", 0x14fd },
  { "CANADIAN SYLLABICS SWOO", 0x14fe },
  { "CANADIAN SYLLABICS WEST-CREE SWOO", 0x14ff },
  { "CANADIAN SYLLABICS SWA", 0x1500 },
  { "CANADIAN SYLLABICS WEST-CREE SWA", 0x1501 },
  { "CANADIAN SYLLABICS SWAA", 0x1502 },
  { "CANADIAN SYLLABICS WEST-CREE SWAA", 0x1503 },
  { "CANADIAN SYLLABICS NASKAPI SWAA", 0x1504 },
  { "CANADIAN SYLLABICS S", 0x1505 },
  { "CANADIAN SYLLABICS ATHAPASCAN S", 0x1506 },
  { "CANADIAN SYLLABICS SW", 0x1507 },
  { "CANADIAN SYLLABICS BLACKFOOT S", 0x1508 },
  { "CANADIAN SYLLABICS MOOSE-CREE SK", 0x1509 },
  { "CANADIAN SYLLABICS NASKAPI SKW", 0x150a },
  { "CANADIAN SYLLABICS NASKAPI S-W", 0x150b },
  { "CANADIAN SYLLABICS NASKAPI SPWA", 0x150c },
  { "CANADIAN SYLLABICS NASKAPI STWA", 0x150d },
  { "CANADIAN SYLLABICS NASKAPI SKWA", 0x150e },
  { "CANADIAN SYLLABICS NASKAPI SCWA", 0x150f },
  { "CANADIAN SYLLABICS SHE", 0x1510 },
  { "CANADIAN SYLLABICS SHI", 0x1511 },
  { "CANADIAN SYLLABICS SHII", 0x1512 },
  { "CANADIAN SYLLABICS SHO", 0x1513 },
  { "CANADIAN SYLLABICS SHOO", 0x1514 },
  { "CANADIAN SYLLABICS SHA", 0x1515 },
  { "CANADIAN SYLLABICS SHAA", 0x1516 },
  { "CANADIAN SYLLABICS SHWE", 0x1517 },
  { "CANADIAN SYLLABICS WEST-CREE SHWE", 0x1518 },
  { "CANADIAN SYLLABICS SHWI", 0x1519 },
  { "CANADIAN SYLLABICS WEST-CREE SHWI", 0x151a },
  { "CANADIAN SYLLABICS SHWII", 0x151b },
  { "CANADIAN SYLLABICS WEST-CREE SHWII", 0x151c },
  { "CANADIAN SYLLABICS SHWO", 0x151d },
  { "CANADIAN SYLLABICS WEST-CREE SHWO", 0x151e },
  { "CANADIAN SYLLABICS SHWOO", 0x151f },
  { "CANADIAN SYLLABICS WEST-CREE SHWOO", 0x1520 },
  { "CANADIAN SYLLABICS SHWA", 0x1521 },
  { "CANADIAN SYLLABICS WEST-CREE SHWA", 0x1522 },
  { "CANADIAN SYLLABICS SHWAA", 0x1523 },
  { "CANADIAN SYLLABICS WEST-CREE SHWAA", 0x1524 },
  { "CANADIAN SYLLABICS SH", 0x1525 },
  { "CANADIAN SYLLABICS YE", 0x1526 },
  { "CANADIAN SYLLABICS YAAI", 0x1527 },
  { "CANADIAN SYLLABICS YI", 0x1528 },
  { "CANADIAN SYLLABICS YII", 0x1529 },
  { "CANADIAN SYLLABICS YO", 0x152a },
  { "CANADIAN SYLLABICS YOO", 0x152b },
  { "CANADIAN SYLLABICS Y-CREE YOO", 0x152c },
  { "CANADIAN SYLLABICS YA", 0x152d },
  { "CANADIAN SYLLABICS YAA", 0x152e },
  { "CANADIAN SYLLABICS YWE", 0x152f },
  { "CANADIAN SYLLABICS WEST-CREE YWE", 0x1530 },
  { "CANADIAN SYLLABICS YWI", 0x1531 },
  { "CANADIAN SYLLABICS WEST-CREE YWI", 0x1532 },
  { "CANADIAN SYLLABICS YWII", 0x1533 },
  { "CANADIAN SYLLABICS WEST-CREE YWII", 0x1534 },
  { "CANADIAN SYLLABICS YWO", 0x1535 },
  { "CANADIAN SYLLABICS WEST-CREE YWO", 0x1536 },
  { "CANADIAN SYLLABICS YWOO", 0x1537 },
  { "CANADIAN SYLLABICS WEST-CREE YWOO", 0x1538 },
  { "CANADIAN SYLLABICS YWA", 0x1539 },
  { "CANADIAN SYLLABICS WEST-CREE YWA", 0x153a },
  { "CANADIAN SYLLABICS YWAA", 0x153b },
  { "CANADIAN SYLLABICS WEST-CREE YWAA", 0x153c },
  { "CANADIAN SYLLABICS NASKAPI YWAA", 0x153d },
  { "CANADIAN SYLLABICS Y", 0x153e },
  { "CANADIAN SYLLABICS BIBLE-CREE Y", 0x153f },
  { "CANADIAN SYLLABICS WEST-CREE Y", 0x1540 },
  { "CANADIAN SYLLABICS SAYISI YI", 0x1541 },
  { "CANADIAN SYLLABICS RE", 0x1542 },
  { "CANADIAN SYLLABICS R-CREE RE", 0x1543 },
  { "CANADIAN SYLLABICS WEST-CREE LE", 0x1544 },
  { "CANADIAN SYLLABICS RAAI", 0x1545 },
  { "CANADIAN SYLLABICS RI", 0x1546 },
  { "CANADIAN SYLLABICS RII", 0x1547 },
  { "CANADIAN SYLLABICS RO", 0x1548 },
  { "CANADIAN SYLLABICS ROO", 0x1549 },
  { "CANADIAN SYLLABICS WEST-CREE LO", 0x154a },
  { "CANADIAN SYLLABICS RA", 0x154b },
  { "CANADIAN SYLLABICS RAA", 0x154c },
  { "CANADIAN SYLLABICS WEST-CREE LA", 0x154d },
  { "CANADIAN SYLLABICS RWAA", 0x154e },
  { "CANADIAN SYLLABICS WEST-CREE RWAA", 0x154f },
  { "CANADIAN SYLLABICS R", 0x1550 },
  { "CANADIAN SYLLABICS WEST-CREE R", 0x1551 },
  { "CANADIAN SYLLABICS MEDIAL R", 0x1552 },
  { "CANADIAN SYLLABICS FE", 0x1553 },
  { "CANADIAN SYLLABICS FAAI", 0x1554 },
  { "CANADIAN SYLLABICS FI", 0x1555 },
  { "CANADIAN SYLLABICS FII", 0x1556 },
  { "CANADIAN SYLLABICS FO", 0x1557 },
  { "CANADIAN SYLLABICS FOO", 0x1558 },
  { "CANADIAN SYLLABICS FA", 0x1559 },
  { "CANADIAN SYLLABICS FAA", 0x155a },
  { "CANADIAN SYLLABICS FWAA", 0x155b },
  { "CANADIAN SYLLABICS WEST-CREE FWAA", 0x155c },
  { "CANADIAN SYLLABICS F", 0x155d },
  { "CANADIAN SYLLABICS THE", 0x155e },
  { "CANADIAN SYLLABICS N-CREE THE", 0x155f },
  { "CANADIAN SYLLABICS THI", 0x1560 },
  { "CANADIAN SYLLABICS N-CREE THI", 0x1561 },
  { "CANADIAN SYLLABICS THII", 0x1562 },
  { "CANADIAN SYLLABICS N-CREE THII", 0x1563 },
  { "CANADIAN SYLLABICS THO", 0x1564 },
  { "CANADIAN SYLLABICS THOO", 0x1565 },
  { "CANADIAN SYLLABICS THA", 0x1566 },
  { "CANADIAN SYLLABICS THAA", 0x1567 },
  { "CANADIAN SYLLABICS THWAA", 0x1568 },
  { "CANADIAN SYLLABICS WEST-CREE THWAA", 0x1569 },
  { "CANADIAN SYLLABICS TH", 0x156a },
  { "CANADIAN SYLLABICS TTHE", 0x156b },
  { "CANADIAN SYLLABICS TTHI", 0x156c },
  { "CANADIAN SYLLABICS TTHO", 0x156d },
  { "CANADIAN SYLLABICS TTHA", 0x156e },
  { "CANADIAN SYLLABICS TTH", 0x156f },
  { "CANADIAN SYLLABICS TYE", 0x1570 },
  { "CANADIAN SYLLABICS TYI", 0x1571 },
  { "CANADIAN SYLLABICS TYO", 0x1572 },
  { "CANADIAN SYLLABICS TYA", 0x1573 },
  { "CANADIAN SYLLABICS NUNAVIK HE", 0x1574 },
  { "CANADIAN SYLLABICS NUNAVIK HI", 0x1575 },
  { "CANADIAN SYLLABICS NUNAVIK HII", 0x1576 },
  { "CANADIAN SYLLABICS NUNAVIK HO", 0x1577 },
  { "CANADIAN SYLLABICS NUNAVIK HOO", 0x1578 },
  { "CANADIAN SYLLABICS NUNAVIK HA", 0x1579 },
  { "CANADIAN SYLLABICS NUNAVIK HAA", 0x157a },
  { "CANADIAN SYLLABICS NUNAVIK H", 0x157b },
  { "CANADIAN SYLLABICS NUNAVUT H", 0x157c },
  { "CANADIAN SYLLABICS HK", 0x157d },
  { "CANADIAN SYLLABICS QAAI", 0x157e },
  { "CANADIAN SYLLABICS QI", 0x157f },
  { "CANADIAN SYLLABICS QII", 0x1580 },
  { "CANADIAN SYLLABICS QO", 0x1581 },
  { "CANADIAN SYLLABICS QOO", 0x1582 },
  { "CANADIAN SYLLABICS QA", 0x1583 },
  { "CANADIAN SYLLABICS QAA", 0x1584 },
  { "CANADIAN SYLLABICS Q", 0x1585 },
  { "CANADIAN SYLLABICS TLHE", 0x1586 },
  { "CANADIAN SYLLABICS TLHI", 0x1587 },
  { "CANADIAN SYLLABICS TLHO", 0x1588 },
  { "CANADIAN SYLLABICS TLHA", 0x1589 },
  { "CANADIAN SYLLABICS WEST-CREE RE", 0x158a },
  { "CANADIAN SYLLABICS WEST-CREE RI", 0x158b },
  { "CANADIAN SYLLABICS WEST-CREE RO", 0x158c },
  { "CANADIAN SYLLABICS WEST-CREE RA", 0x158d },
  { "CANADIAN SYLLABICS NGAAI", 0x158e },
  { "CANADIAN SYLLABICS NGI", 0x158f },
  { "CANADIAN SYLLABICS NGII", 0x1590 },
  { "CANADIAN SYLLABICS NGO", 0x1591 },
  { "CANADIAN SYLLABICS NGOO", 0x1592 },
  { "CANADIAN SYLLABICS NGA", 0x1593 },
  { "CANADIAN SYLLABICS NGAA", 0x1594 },
  { "CANADIAN SYLLABICS NG", 0x1595 },
  { "CANADIAN SYLLABICS NNG", 0x1596 },
  { "CANADIAN SYLLABICS SAYISI SHE", 0x1597 },
  { "CANADIAN SYLLABICS SAYISI SHI", 0x1598 },
  { "CANADIAN SYLLABICS SAYISI SHO", 0x1599 },
  { "CANADIAN SYLLABICS SAYISI SHA", 0x159a },
  { "CANADIAN SYLLABICS WOODS-CREE THE", 0x159b },
  { "CANADIAN SYLLABICS WOODS-CREE THI", 0x159c },
  { "CANADIAN SYLLABICS WOODS-CREE THO", 0x159d },
  { "CANADIAN SYLLABICS WOODS-CREE THA", 0x159e },
  { "CANADIAN SYLLABICS WOODS-CREE TH", 0x159f },
  { "CANADIAN SYLLABICS LHI", 0x15a0 },
  { "CANADIAN SYLLABICS LHII", 0x15a1 },
  { "CANADIAN SYLLABICS LHO", 0x15a2 },
  { "CANADIAN SYLLABICS LHOO", 0x15a3 },
  { "CANADIAN SYLLABICS LHA", 0x15a4 },
  { "CANADIAN SYLLABICS LHAA", 0x15a5 },
  { "CANADIAN SYLLABICS LH", 0x15a6 },
  { "CANADIAN SYLLABICS TH-CREE THE", 0x15a7 },
  { "CANADIAN SYLLABICS TH-CREE THI", 0x15a8 },
  { "CANADIAN SYLLABICS TH-CREE THII", 0x15a9 },
  { "CANADIAN SYLLABICS TH-CREE THO", 0x15aa },
  { "CANADIAN SYLLABICS TH-CREE THOO", 0x15ab },
  { "CANADIAN SYLLABICS TH-CREE THA", 0x15ac },
  { "CANADIAN SYLLABICS TH-CREE THAA", 0x15ad },
  { "CANADIAN SYLLABICS TH-CREE TH", 0x15ae },
  { "CANADIAN SYLLABICS AIVILIK B", 0x15af },
  { "CANADIAN SYLLABICS BLACKFOOT E", 0x15b0 },
  { "CANADIAN SYLLABICS BLACKFOOT I", 0x15b1 },
  { "CANADIAN SYLLABICS BLACKFOOT O", 0x15b2 },
  { "CANADIAN SYLLABICS BLACKFOOT A", 0x15b3 },
  { "CANADIAN SYLLABICS BLACKFOOT WE", 0x15b4 },
  { "CANADIAN SYLLABICS BLACKFOOT WI", 0x15b5 },
  { "CANADIAN SYLLABICS BLACKFOOT WO", 0x15b6 },
  { "CANADIAN SYLLABICS BLACKFOOT WA", 0x15b7 },
  { "CANADIAN SYLLABICS BLACKFOOT NE", 0x15b8 },
  { "CANADIAN SYLLABICS BLACKFOOT NI", 0x15b9 },
  { "CANADIAN SYLLABICS BLACKFOOT NO", 0x15ba },
  { "CANADIAN SYLLABICS BLACKFOOT NA", 0x15bb },
  { "CANADIAN SYLLABICS BLACKFOOT KE", 0x15bc },
  { "CANADIAN SYLLABICS BLACKFOOT KI", 0x15bd },
  { "CANADIAN SYLLABICS BLACKFOOT KO", 0x15be },
  { "CANADIAN SYLLABICS BLACKFOOT KA", 0x15bf },
  { "CANADIAN SYLLABICS SAYISI HE", 0x15c0 },
  { "CANADIAN SYLLABICS SAYISI HI", 0x15c1 },
  { "CANADIAN SYLLABICS SAYISI HO", 0x15c2 },
  { "CANADIAN SYLLABICS SAYISI HA", 0x15c3 },
  { "CANADIAN SYLLABICS CARRIER GHU", 0x15c4 },
  { "CANADIAN SYLLABICS CARRIER GHO", 0x15c5 },
  { "CANADIAN SYLLABICS CARRIER GHE", 0x15c6 },
  { "CANADIAN SYLLABICS CARRIER GHEE", 0x15c7 },
  { "CANADIAN SYLLABICS CARRIER GHI", 0x15c8 },
  { "CANADIAN SYLLABICS CARRIER GHA", 0x15c9 },
  { "CANADIAN SYLLABICS CARRIER RU", 0x15ca },
  { "CANADIAN SYLLABICS CARRIER RO", 0x15cb },
  { "CANADIAN SYLLABICS CARRIER RE", 0x15cc },
  { "CANADIAN SYLLABICS CARRIER REE", 0x15cd },
  { "CANADIAN SYLLABICS CARRIER RI", 0x15ce },
  { "CANADIAN SYLLABICS CARRIER RA", 0x15cf },
  { "CANADIAN SYLLABICS CARRIER WU", 0x15d0 },
  { "CANADIAN SYLLABICS CARRIER WO", 0x15d1 },
  { "CANADIAN SYLLABICS CARRIER WE", 0x15d2 },
  { "CANADIAN SYLLABICS CARRIER WEE", 0x15d3 },
  { "CANADIAN SYLLABICS CARRIER WI", 0x15d4 },
  { "CANADIAN SYLLABICS CARRIER WA", 0x15d5 },
  { "CANADIAN SYLLABICS CARRIER HWU", 0x15d6 },
  { "CANADIAN SYLLABICS CARRIER HWO", 0x15d7 },
  { "CANADIAN SYLLABICS CARRIER HWE", 0x15d8 },
  { "CANADIAN SYLLABICS CARRIER HWEE", 0x15d9 },
  { "CANADIAN SYLLABICS CARRIER HWI", 0x15da },
  { "CANADIAN SYLLABICS CARRIER HWA", 0x15db },
  { "CANADIAN SYLLABICS CARRIER THU", 0x15dc },
  { "CANADIAN SYLLABICS CARRIER THO", 0x15dd },
  { "CANADIAN SYLLABICS CARRIER THE", 0x15de },
  { "CANADIAN SYLLABICS CARRIER THEE", 0x15df },
  { "CANADIAN SYLLABICS CARRIER THI", 0x15e0 },
  { "CANADIAN SYLLABICS CARRIER THA", 0x15e1 },
  { "CANADIAN SYLLABICS CARRIER TTU", 0x15e2 },
  { "CANADIAN SYLLABICS CARRIER TTO", 0x15e3 },
  { "CANADIAN SYLLABICS CARRIER TTE", 0x15e4 },
  { "CANADIAN SYLLABICS CARRIER TTEE", 0x15e5 },
  { "CANADIAN SYLLABICS CARRIER TTI", 0x15e6 },
  { "CANADIAN SYLLABICS CARRIER TTA", 0x15e7 },
  { "CANADIAN SYLLABICS CARRIER PU", 0x15e8 },
  { "CANADIAN SYLLABICS CARRIER PO", 0x15e9 },
  { "CANADIAN SYLLABICS CARRIER PE", 0x15ea },
  { "CANADIAN SYLLABICS CARRIER PEE", 0x15eb },
  { "CANADIAN SYLLABICS CARRIER PI", 0x15ec },
  { "CANADIAN SYLLABICS CARRIER PA", 0x15ed },
  { "CANADIAN SYLLABICS CARRIER P", 0x15ee },
  { "CANADIAN SYLLABICS CARRIER GU", 0x15ef },
  { "CANADIAN SYLLABICS CARRIER GO", 0x15f0 },
  { "CANADIAN SYLLABICS CARRIER GE", 0x15f1 },
  { "CANADIAN SYLLABICS CARRIER GEE", 0x15f2 },
  { "CANADIAN SYLLABICS CARRIER GI", 0x15f3 },
  { "CANADIAN SYLLABICS CARRIER GA", 0x15f4 },
  { "CANADIAN SYLLABICS CARRIER KHU", 0x15f5 },
  { "CANADIAN SYLLABICS CARRIER KHO", 0x15f6 },
  { "CANADIAN SYLLABICS CARRIER KHE", 0x15f7 },
  { "CANADIAN SYLLABICS CARRIER KHEE", 0x15f8 },
  { "CANADIAN SYLLABICS CARRIER KHI", 0x15f9 },
  { "CANADIAN SYLLABICS CARRIER KHA", 0x15fa },
  { "CANADIAN SYLLABICS CARRIER KKU", 0x15fb },
  { "CANADIAN SYLLABICS CARRIER KKO", 0x15fc },
  { "CANADIAN SYLLABICS CARRIER KKE", 0x15fd },
  { "CANADIAN SYLLABICS CARRIER KKEE", 0x15fe },
  { "CANADIAN SYLLABICS CARRIER KKI", 0x15ff },
  { "CANADIAN SYLLABICS CARRIER KKA", 0x1600 },
  { "CANADIAN SYLLABICS CARRIER KK", 0x1601 },
  { "CANADIAN SYLLABICS CARRIER NU", 0x1602 },
  { "CANADIAN SYLLABICS CARRIER NO", 0x1603 },
  { "CANADIAN SYLLABICS CARRIER NE", 0x1604 },
  { "CANADIAN SYLLABICS CARRIER NEE", 0x1605 },
  { "CANADIAN SYLLABICS CARRIER NI", 0x1606 },
  { "CANADIAN SYLLABICS CARRIER NA", 0x1607 },
  { "CANADIAN SYLLABICS CARRIER MU", 0x1608 },
  { "CANADIAN SYLLABICS CARRIER MO", 0x1609 },
  { "CANADIAN SYLLABICS CARRIER ME", 0x160a },
  { "CANADIAN SYLLABICS CARRIER MEE", 0x160b },
  { "CANADIAN SYLLABICS CARRIER MI", 0x160c },
  { "CANADIAN SYLLABICS CARRIER MA", 0x160d },
  { "CANADIAN SYLLABICS CARRIER YU", 0x160e },
  { "CANADIAN SYLLABICS CARRIER YO", 0x160f },
  { "CANADIAN SYLLABICS CARRIER YE", 0x1610 },
  { "CANADIAN SYLLABICS CARRIER YEE", 0x1611 },
  { "CANADIAN SYLLABICS CARRIER YI", 0x1612 },
  { "CANADIAN SYLLABICS CARRIER YA", 0x1613 },
  { "CANADIAN SYLLABICS CARRIER JU", 0x1614 },
  { "CANADIAN SYLLABICS SAYISI JU", 0x1615 },
  { "CANADIAN SYLLABICS CARRIER JO", 0x1616 },
  { "CANADIAN SYLLABICS CARRIER JE", 0x1617 },
  { "CANADIAN SYLLABICS CARRIER JEE", 0x1618 },
  { "CANADIAN SYLLABICS CARRIER JI", 0x1619 },
  { "CANADIAN SYLLABICS SAYISI JI", 0x161a },
  { "CANADIAN SYLLABICS CARRIER JA", 0x161b },
  { "CANADIAN SYLLABICS CARRIER JJU", 0x161c },
  { "CANADIAN SYLLABICS CARRIER JJO", 0x161d },
  { "CANADIAN SYLLABICS CARRIER JJE", 0x161e },
  { "CANADIAN SYLLABICS CARRIER JJEE", 0x161f },
  { "CANADIAN SYLLABICS CARRIER JJI", 0x1620 },
  { "CANADIAN SYLLABICS CARRIER JJA", 0x1621 },
  { "CANADIAN SYLLABICS CARRIER LU", 0x1622 },
  { "CANADIAN SYLLABICS CARRIER LO", 0x1623 },
  { "CANADIAN SYLLABICS CARRIER LE", 0x1624 },
  { "CANADIAN SYLLABICS CARRIER LEE", 0x1625 },
  { "CANADIAN SYLLABICS CARRIER LI", 0x1626 },
  { "CANADIAN SYLLABICS CARRIER LA", 0x1627 },
  { "CANADIAN SYLLABICS CARRIER DLU", 0x1628 },
  { "CANADIAN SYLLABICS CARRIER DLO", 0x1629 },
  { "CANADIAN SYLLABICS CARRIER DLE", 0x162a },
  { "CANADIAN SYLLABICS CARRIER DLEE", 0x162b },
  { "CANADIAN SYLLABICS CARRIER DLI", 0x162c },
  { "CANADIAN SYLLABICS CARRIER DLA", 0x162d },
  { "CANADIAN SYLLABICS CARRIER LHU", 0x162e },
  { "CANADIAN SYLLABICS CARRIER LHO", 0x162f },
  { "CANADIAN SYLLABICS CARRIER LHE", 0x1630 },
  { "CANADIAN SYLLABICS CARRIER LHEE", 0x1631 },
  { "CANADIAN SYLLABICS CARRIER LHI", 0x1632 },
  { "CANADIAN SYLLABICS CARRIER LHA", 0x1633 },
  { "CANADIAN SYLLABICS CARRIER TLHU", 0x1634 },
  { "CANADIAN SYLLABICS CARRIER TLHO", 0x1635 },
  { "CANADIAN SYLLABICS CARRIER TLHE", 0x1636 },
  { "CANADIAN SYLLABICS CARRIER TLHEE", 0x1637 },
  { "CANADIAN SYLLABICS CARRIER TLHI", 0x1638 },
  { "CANADIAN SYLLABICS CARRIER TLHA", 0x1639 },
  { "CANADIAN SYLLABICS CARRIER TLU", 0x163a },
  { "CANADIAN SYLLABICS CARRIER TLO", 0x163b },
  { "CANADIAN SYLLABICS CARRIER TLE", 0x163c },
  { "CANADIAN SYLLABICS CARRIER TLEE", 0x163d },
  { "CANADIAN SYLLABICS CARRIER TLI", 0x163e },
  { "CANADIAN SYLLABICS CARRIER TLA", 0x163f },
  { "CANADIAN SYLLABICS CARRIER ZU", 0x1640 },
  { "CANADIAN SYLLABICS CARRIER ZO", 0x1641 },
  { "CANADIAN SYLLABICS CARRIER ZE", 0x1642 },
  { "CANADIAN SYLLABICS CARRIER ZEE", 0x1643 },
  { "CANADIAN SYLLABICS CARRIER ZI", 0x1644 },
  { "CANADIAN SYLLABICS CARRIER ZA", 0x1645 },
  { "CANADIAN SYLLABICS CARRIER Z", 0x1646 },
  { "CANADIAN SYLLABICS CARRIER INITIAL Z", 0x1647 },
  { "CANADIAN SYLLABICS CARRIER DZU", 0x1648 },
  { "CANADIAN SYLLABICS CARRIER DZO", 0x1649 },
  { "CANADIAN SYLLABICS CARRIER DZE", 0x164a },
  { "CANADIAN SYLLABICS CARRIER DZEE", 0x164b },
  { "CANADIAN SYLLABICS CARRIER DZI", 0x164c },
  { "CANADIAN SYLLABICS CARRIER DZA", 0x164d },
  { "CANADIAN SYLLABICS CARRIER SU", 0x164e },
  { "CANADIAN SYLLABICS CARRIER SO", 0x164f },
  { "CANADIAN SYLLABICS CARRIER SE", 0x1650 },
  { "CANADIAN SYLLABICS CARRIER SEE", 0x1651 },
  { "CANADIAN SYLLABICS CARRIER SI", 0x1652 },
  { "CANADIAN SYLLABICS CARRIER SA", 0x1653 },
  { "CANADIAN SYLLABICS CARRIER SHU", 0x1654 },
  { "CANADIAN SYLLABICS CARRIER SHO", 0x1655 },
  { "CANADIAN SYLLABICS CARRIER SHE", 0x1656 },
  { "CANADIAN SYLLABICS CARRIER SHEE", 0x1657 },
  { "CANADIAN SYLLABICS CARRIER SHI", 0x1658 },
  { "CANADIAN SYLLABICS CARRIER SHA", 0x1659 },
  { "CANADIAN SYLLABICS CARRIER SH", 0x165a },
  { "CANADIAN SYLLABICS CARRIER TSU", 0x165b },
  { "CANADIAN SYLLABICS CARRIER TSO", 0x165c },
  { "CANADIAN SYLLABICS CARRIER TSE", 0x165d },
  { "CANADIAN SYLLABICS CARRIER TSEE", 0x165e },
  { "CANADIAN SYLLABICS CARRIER TSI", 0x165f },
  { "CANADIAN SYLLABICS CARRIER TSA", 0x1660 },
  { "CANADIAN SYLLABICS CARRIER CHU", 0x1661 },
  { "CANADIAN SYLLABICS CARRIER CHO", 0x1662 },
  { "CANADIAN SYLLABICS CARRIER CHE", 0x1663 },
  { "CANADIAN SYLLABICS CARRIER CHEE", 0x1664 },
  { "CANADIAN SYLLABICS CARRIER CHI", 0x1665 },
  { "CANADIAN SYLLABICS CARRIER CHA", 0x1666 },
  { "CANADIAN SYLLABICS CARRIER TTSU", 0x1667 },
  { "CANADIAN SYLLABICS CARRIER TTSO", 0x1668 },
  { "CANADIAN SYLLABICS CARRIER TTSE", 0x1669 },
  { "CANADIAN SYLLABICS CARRIER TTSEE", 0x166a },
  { "CANADIAN SYLLABICS CARRIER TTSI", 0x166b },
  { "CANADIAN SYLLABICS CARRIER TTSA", 0x166c },
  { "CANADIAN SYLLABICS CHI SIGN", 0x166d },
  { "CANADIAN SYLLABICS FULL STOP", 0x166e },
  { "CANADIAN SYLLABICS QAI", 0x166f },
  { "CANADIAN SYLLABICS NGAI", 0x1670 },
  { "CANADIAN SYLLABICS NNGI", 0x1671 },
  { "CANADIAN SYLLABICS NNGII", 0x1672 },
  { "CANADIAN SYLLABICS NNGO", 0x1673 },
  { "CANADIAN SYLLABICS NNGOO", 0x1674 },
  { "CANADIAN SYLLABICS NNGA", 0x1675 },
  { "CANADIAN SYLLABICS NNGAA", 0x1676 },
  { "OGHAM SPACE MARK", 0x1680 },
  { "OGHAM LETTER BEITH", 0x1681 },
  { "OGHAM LETTER LUIS", 0x1682 },
  { "OGHAM LETTER FEARN", 0x1683 },
  { "OGHAM LETTER SAIL", 0x1684 },
  { "OGHAM LETTER NION", 0x1685 },
  { "OGHAM LETTER UATH", 0x1686 },
  { "OGHAM LETTER DAIR", 0x1687 },
  { "OGHAM LETTER TINNE", 0x1688 },
  { "OGHAM LETTER COLL", 0x1689 },
  { "OGHAM LETTER CEIRT", 0x168a },
  { "OGHAM LETTER MUIN", 0x168b },
  { "OGHAM LETTER GORT", 0x168c },
  { "OGHAM LETTER NGEADAL", 0x168d },
  { "OGHAM LETTER STRAIF", 0x168e },
  { "OGHAM LETTER RUIS", 0x168f },
  { "OGHAM LETTER AILM", 0x1690 },
  { "OGHAM LETTER ONN", 0x1691 },
  { "OGHAM LETTER UR", 0x1692 },
  { "OGHAM LETTER EADHADH", 0x1693 },
  { "OGHAM LETTER IODHADH", 0x1694 },
  { "OGHAM LETTER EABHADH", 0x1695 },
  { "OGHAM LETTER OR", 0x1696 },
  { "OGHAM LETTER UILLEANN", 0x1697 },
  { "OGHAM LETTER IFIN", 0x1698 },
  { "OGHAM LETTER EAMHANCHOLL", 0x1699 },
  { "OGHAM LETTER PEITH", 0x169a },
  { "OGHAM FEATHER MARK", 0x169b },
  { "OGHAM REVERSED FEATHER MARK", 0x169c },
  { "RUNIC LETTER FEHU FEOH FE F", 0x16a0 },
  { "RUNIC LETTER V", 0x16a1 },
  { "RUNIC LETTER URUZ UR U", 0x16a2 },
  { "RUNIC LETTER YR", 0x16a3 },
  { "RUNIC LETTER Y", 0x16a4 },
  { "RUNIC LETTER W", 0x16a5 },
  { "RUNIC LETTER THURISAZ THURS THORN", 0x16a6 },
  { "RUNIC LETTER ETH", 0x16a7 },
  { "RUNIC LETTER ANSUZ A", 0x16a8 },
  { "RUNIC LETTER OS O", 0x16a9 },
  { "RUNIC LETTER AC A", 0x16aa },
  { "RUNIC LETTER AESC", 0x16ab },
  { "RUNIC LETTER LONG-BRANCH-OSS O", 0x16ac },
  { "RUNIC LETTER SHORT-TWIG-OSS O", 0x16ad },
  { "RUNIC LETTER O", 0x16ae },
  { "RUNIC LETTER OE", 0x16af },
  { "RUNIC LETTER ON", 0x16b0 },
  { "RUNIC LETTER RAIDO RAD REID R", 0x16b1 },
  { "RUNIC LETTER KAUNA", 0x16b2 },
  { "RUNIC LETTER CEN", 0x16b3 },
  { "RUNIC LETTER KAUN K", 0x16b4 },
  { "RUNIC LETTER G", 0x16b5 },
  { "RUNIC LETTER ENG", 0x16b6 },
  { "RUNIC LETTER GEBO GYFU G", 0x16b7 },
  { "RUNIC LETTER GAR", 0x16b8 },
  { "RUNIC LETTER WUNJO WYNN W", 0x16b9 },
  { "RUNIC LETTER HAGLAZ H", 0x16ba },
  { "RUNIC LETTER HAEGL H", 0x16bb },
  { "RUNIC LETTER LONG-BRANCH-HAGALL H", 0x16bc },
  { "RUNIC LETTER SHORT-TWIG-HAGALL H", 0x16bd },
  { "RUNIC LETTER NAUDIZ NYD NAUD N", 0x16be },
  { "RUNIC LETTER SHORT-TWIG-NAUD N", 0x16bf },
  { "RUNIC LETTER DOTTED-N", 0x16c0 },
  { "RUNIC LETTER ISAZ IS ISS I", 0x16c1 },
  { "RUNIC LETTER E", 0x16c2 },
  { "RUNIC LETTER JERAN J", 0x16c3 },
  { "RUNIC LETTER GER", 0x16c4 },
  { "RUNIC LETTER LONG-BRANCH-AR AE", 0x16c5 },
  { "RUNIC LETTER SHORT-TWIG-AR A", 0x16c6 },
  { "RUNIC LETTER IWAZ EOH", 0x16c7 },
  { "RUNIC LETTER PERTHO PEORTH P", 0x16c8 },
  { "RUNIC LETTER ALGIZ EOLHX", 0x16c9 },
  { "RUNIC LETTER SOWILO S", 0x16ca },
  { "RUNIC LETTER SIGEL LONG-BRANCH-SOL S", 0x16cb },
  { "RUNIC LETTER SHORT-TWIG-SOL S", 0x16cc },
  { "RUNIC LETTER C", 0x16cd },
  { "RUNIC LETTER Z", 0x16ce },
  { "RUNIC LETTER TIWAZ TIR TYR T", 0x16cf },
  { "RUNIC LETTER SHORT-TWIG-TYR T", 0x16d0 },
  { "RUNIC LETTER D", 0x16d1 },
  { "RUNIC LETTER BERKANAN BEORC BJARKAN B", 0x16d2 },
  { "RUNIC LETTER SHORT-TWIG-BJARKAN B", 0x16d3 },
  { "RUNIC LETTER DOTTED-P", 0x16d4 },
  { "RUNIC LETTER OPEN-P", 0x16d5 },
  { "RUNIC LETTER EHWAZ EH E", 0x16d6 },
  { "RUNIC LETTER MANNAZ MAN M", 0x16d7 },
  { "RUNIC LETTER LONG-BRANCH-MADR M", 0x16d8 },
  { "RUNIC LETTER SHORT-TWIG-MADR M", 0x16d9 },
  { "RUNIC LETTER LAUKAZ LAGU LOGR L", 0x16da },
  { "RUNIC LETTER DOTTED-L", 0x16db },
  { "RUNIC LETTER INGWAZ", 0x16dc },
  { "RUNIC LETTER ING", 0x16dd },
  { "RUNIC LETTER DAGAZ DAEG D", 0x16de },
  { "RUNIC LETTER OTHALAN ETHEL O", 0x16df },
  { "RUNIC LETTER EAR", 0x16e0 },
  { "RUNIC LETTER IOR", 0x16e1 },
  { "RUNIC LETTER CWEORTH", 0x16e2 },
  { "RUNIC LETTER CALC", 0x16e3 },
  { "RUNIC LETTER CEALC", 0x16e4 },
  { "RUNIC LETTER STAN", 0x16e5 },
  { "RUNIC LETTER LONG-BRANCH-YR", 0x16e6 },
  { "RUNIC LETTER SHORT-TWIG-YR", 0x16e7 },
  { "RUNIC LETTER ICELANDIC-YR", 0x16e8 },
  { "RUNIC LETTER Q", 0x16e9 },
  { "RUNIC LETTER X", 0x16ea },
  { "RUNIC SINGLE PUNCTUATION", 0x16eb },
  { "RUNIC MULTIPLE PUNCTUATION", 0x16ec },
  { "RUNIC CROSS PUNCTUATION", 0x16ed },
  { "RUNIC ARLAUG SYMBOL", 0x16ee },
  { "RUNIC TVIMADUR SYMBOL", 0x16ef },
  { "RUNIC BELGTHOR SYMBOL", 0x16f0 },
  { "KHMER LETTER KA", 0x1780 },
  { "KHMER LETTER KHA", 0x1781 },
  { "KHMER LETTER KO", 0x1782 },
  { "KHMER LETTER KHO", 0x1783 },
  { "KHMER LETTER NGO", 0x1784 },
  { "KHMER LETTER CA", 0x1785 },
  { "KHMER LETTER CHA", 0x1786 },
  { "KHMER LETTER CO", 0x1787 },
  { "KHMER LETTER CHO", 0x1788 },
  { "KHMER LETTER NYO", 0x1789 },
  { "KHMER LETTER DA", 0x178a },
  { "KHMER LETTER TTHA", 0x178b },
  { "KHMER LETTER DO", 0x178c },
  { "KHMER LETTER TTHO", 0x178d },
  { "KHMER LETTER NNO", 0x178e },
  { "KHMER LETTER TA", 0x178f },
  { "KHMER LETTER THA", 0x1790 },
  { "KHMER LETTER TO", 0x1791 },
  { "KHMER LETTER THO", 0x1792 },
  { "KHMER LETTER NO", 0x1793 },
  { "KHMER LETTER BA", 0x1794 },
  { "KHMER LETTER PHA", 0x1795 },
  { "KHMER LETTER PO", 0x1796 },
  { "KHMER LETTER PHO", 0x1797 },
  { "KHMER LETTER MO", 0x1798 },
  { "KHMER LETTER YO", 0x1799 },
  { "KHMER LETTER RO", 0x179a },
  { "KHMER LETTER LO", 0x179b },
  { "KHMER LETTER VO", 0x179c },
  { "KHMER LETTER SHA", 0x179d },
  { "KHMER LETTER SSO", 0x179e },
  { "KHMER LETTER SA", 0x179f },
  { "KHMER LETTER HA", 0x17a0 },
  { "KHMER LETTER LA", 0x17a1 },
  { "KHMER LETTER QA", 0x17a2 },
  { "KHMER INDEPENDENT VOWEL QAQ", 0x17a3 },
  { "KHMER INDEPENDENT VOWEL QAA", 0x17a4 },
  { "KHMER INDEPENDENT VOWEL QI", 0x17a5 },
  { "KHMER INDEPENDENT VOWEL QII", 0x17a6 },
  { "KHMER INDEPENDENT VOWEL QU", 0x17a7 },
  { "KHMER INDEPENDENT VOWEL QUK", 0x17a8 },
  { "KHMER INDEPENDENT VOWEL QUU", 0x17a9 },
  { "KHMER INDEPENDENT VOWEL QUUV", 0x17aa },
  { "KHMER INDEPENDENT VOWEL RY", 0x17ab },
  { "KHMER INDEPENDENT VOWEL RYY", 0x17ac },
  { "KHMER INDEPENDENT VOWEL LY", 0x17ad },
  { "KHMER INDEPENDENT VOWEL LYY", 0x17ae },
  { "KHMER INDEPENDENT VOWEL QE", 0x17af },
  { "KHMER INDEPENDENT VOWEL QAI", 0x17b0 },
  { "KHMER INDEPENDENT VOWEL QOO TYPE ONE", 0x17b1 },
  { "KHMER INDEPENDENT VOWEL QOO TYPE TWO", 0x17b2 },
  { "KHMER INDEPENDENT VOWEL QAU", 0x17b3 },
  { "KHMER VOWEL INHERENT AQ", 0x17b4 },
  { "KHMER VOWEL INHERENT AA", 0x17b5 },
  { "KHMER VOWEL SIGN AA", 0x17b6 },
  { "KHMER VOWEL SIGN I", 0x17b7 },
  { "KHMER VOWEL SIGN II", 0x17b8 },
  { "KHMER VOWEL SIGN Y", 0x17b9 },
  { "KHMER VOWEL SIGN YY", 0x17ba },
  { "KHMER VOWEL SIGN U", 0x17bb },
  { "KHMER VOWEL SIGN UU", 0x17bc },
  { "KHMER VOWEL SIGN UA", 0x17bd },
  { "KHMER VOWEL SIGN OE", 0x17be },
  { "KHMER VOWEL SIGN YA", 0x17bf },
  { "KHMER VOWEL SIGN IE", 0x17c0 },
  { "KHMER VOWEL SIGN E", 0x17c1 },
  { "KHMER VOWEL SIGN AE", 0x17c2 },
  { "KHMER VOWEL SIGN AI", 0x17c3 },
  { "KHMER VOWEL SIGN OO", 0x17c4 },
  { "KHMER VOWEL SIGN AU", 0x17c5 },
  { "KHMER SIGN NIKAHIT", 0x17c6 },
  { "KHMER SIGN REAHMUK", 0x17c7 },
  { "KHMER SIGN YUUKALEAPINTU", 0x17c8 },
  { "KHMER SIGN MUUSIKATOAN", 0x17c9 },
  { "KHMER SIGN TRIISAP", 0x17ca },
  { "KHMER SIGN BANTOC", 0x17cb },
  { "KHMER SIGN ROBAT", 0x17cc },
  { "KHMER SIGN TOANDAKHIAT", 0x17cd },
  { "KHMER SIGN KAKABAT", 0x17ce },
  { "KHMER SIGN AHSDA", 0x17cf },
  { "KHMER SIGN SAMYOK SANNYA", 0x17d0 },
  { "KHMER SIGN VIRIAM", 0x17d1 },
  { "KHMER SIGN COENG", 0x17d2 },
  { "KHMER SIGN BATHAMASAT", 0x17d3 },
  { "KHMER SIGN KHAN", 0x17d4 },
  { "KHMER SIGN BARIYOOSAN", 0x17d5 },
  { "KHMER SIGN CAMNUC PII KUUH", 0x17d6 },
  { "KHMER SIGN LEK TOO", 0x17d7 },
  { "KHMER SIGN BEYYAL", 0x17d8 },
  { "KHMER SIGN PHNAEK MUAN", 0x17d9 },
  { "KHMER SIGN KOOMUUT", 0x17da },
  { "KHMER CURRENCY SYMBOL RIEL", 0x17db },
  { "KHMER SIGN AVAKRAHASANYA", 0x17dc },
  { "KHMER DIGIT ZERO", 0x17e0 },
  { "KHMER DIGIT ONE", 0x17e1 },
  { "KHMER DIGIT TWO", 0x17e2 },
  { "KHMER DIGIT THREE", 0x17e3 },
  { "KHMER DIGIT FOUR", 0x17e4 },
  { "KHMER DIGIT FIVE", 0x17e5 },
  { "KHMER DIGIT SIX", 0x17e6 },
  { "KHMER DIGIT SEVEN", 0x17e7 },
  { "KHMER DIGIT EIGHT", 0x17e8 },
  { "KHMER DIGIT NINE", 0x17e9 },
  { "MONGOLIAN BIRGA", 0x1800 },
  { "MONGOLIAN ELLIPSIS", 0x1801 },
  { "MONGOLIAN COMMA", 0x1802 },
  { "MONGOLIAN FULL STOP", 0x1803 },
  { "MONGOLIAN COLON", 0x1804 },
  { "MONGOLIAN FOUR DOTS", 0x1805 },
  { "MONGOLIAN TODO SOFT HYPHEN", 0x1806 },
  { "MONGOLIAN SIBE SYLLABLE BOUNDARY MARKER", 0x1807 },
  { "MONGOLIAN MANCHU COMMA", 0x1808 },
  { "MONGOLIAN MANCHU FULL STOP", 0x1809 },
  { "MONGOLIAN NIRUGU", 0x180a },
  { "MONGOLIAN FREE VARIATION SELECTOR ONE", 0x180b },
  { "MONGOLIAN FREE VARIATION SELECTOR TWO", 0x180c },
  { "MONGOLIAN FREE VARIATION SELECTOR THREE", 0x180d },
  { "MONGOLIAN VOWEL SEPARATOR", 0x180e },
  { "MONGOLIAN DIGIT ZERO", 0x1810 },
  { "MONGOLIAN DIGIT ONE", 0x1811 },
  { "MONGOLIAN DIGIT TWO", 0x1812 },
  { "MONGOLIAN DIGIT THREE", 0x1813 },
  { "MONGOLIAN DIGIT FOUR", 0x1814 },
  { "MONGOLIAN DIGIT FIVE", 0x1815 },
  { "MONGOLIAN DIGIT SIX", 0x1816 },
  { "MONGOLIAN DIGIT SEVEN", 0x1817 },
  { "MONGOLIAN DIGIT EIGHT", 0x1818 },
  { "MONGOLIAN DIGIT NINE", 0x1819 },
  { "MONGOLIAN LETTER A", 0x1820 },
  { "MONGOLIAN LETTER E", 0x1821 },
  { "MONGOLIAN LETTER I", 0x1822 },
  { "MONGOLIAN LETTER O", 0x1823 },
  { "MONGOLIAN LETTER U", 0x1824 },
  { "MONGOLIAN LETTER OE", 0x1825 },
  { "MONGOLIAN LETTER UE", 0x1826 },
  { "MONGOLIAN LETTER EE", 0x1827 },
  { "MONGOLIAN LETTER NA", 0x1828 },
  { "MONGOLIAN LETTER ANG", 0x1829 },
  { "MONGOLIAN LETTER BA", 0x182a },
  { "MONGOLIAN LETTER PA", 0x182b },
  { "MONGOLIAN LETTER QA", 0x182c },
  { "MONGOLIAN LETTER GA", 0x182d },
  { "MONGOLIAN LETTER MA", 0x182e },
  { "MONGOLIAN LETTER LA", 0x182f },
  { "MONGOLIAN LETTER SA", 0x1830 },
  { "MONGOLIAN LETTER SHA", 0x1831 },
  { "MONGOLIAN LETTER TA", 0x1832 },
  { "MONGOLIAN LETTER DA", 0x1833 },
  { "MONGOLIAN LETTER CHA", 0x1834 },
  { "MONGOLIAN LETTER JA", 0x1835 },
  { "MONGOLIAN LETTER YA", 0x1836 },
  { "MONGOLIAN LETTER RA", 0x1837 },
  { "MONGOLIAN LETTER WA", 0x1838 },
  { "MONGOLIAN LETTER FA", 0x1839 },
  { "MONGOLIAN LETTER KA", 0x183a },
  { "MONGOLIAN LETTER KHA", 0x183b },
  { "MONGOLIAN LETTER TSA", 0x183c },
  { "MONGOLIAN LETTER ZA", 0x183d },
  { "MONGOLIAN LETTER HAA", 0x183e },
  { "MONGOLIAN LETTER ZRA", 0x183f },
  { "MONGOLIAN LETTER LHA", 0x1840 },
  { "MONGOLIAN LETTER ZHI", 0x1841 },
  { "MONGOLIAN LETTER CHI", 0x1842 },
  { "MONGOLIAN LETTER TODO LONG VOWEL SIGN", 0x1843 },
  { "MONGOLIAN LETTER TODO E", 0x1844 },
  { "MONGOLIAN LETTER TODO I", 0x1845 },
  { "MONGOLIAN LETTER TODO O", 0x1846 },
  { "MONGOLIAN LETTER TODO U", 0x1847 },
  { "MONGOLIAN LETTER TODO OE", 0x1848 },
  { "MONGOLIAN LETTER TODO UE", 0x1849 },
  { "MONGOLIAN LETTER TODO ANG", 0x184a },
  { "MONGOLIAN LETTER TODO BA", 0x184b },
  { "MONGOLIAN LETTER TODO PA", 0x184c },
  { "MONGOLIAN LETTER TODO QA", 0x184d },
  { "MONGOLIAN LETTER TODO GA", 0x184e },
  { "MONGOLIAN LETTER TODO MA", 0x184f },
  { "MONGOLIAN LETTER TODO TA", 0x1850 },
  { "MONGOLIAN LETTER TODO DA", 0x1851 },
  { "MONGOLIAN LETTER TODO CHA", 0x1852 },
  { "MONGOLIAN LETTER TODO JA", 0x1853 },
  { "MONGOLIAN LETTER TODO TSA", 0x1854 },
  { "MONGOLIAN LETTER TODO YA", 0x1855 },
  { "MONGOLIAN LETTER TODO WA", 0x1856 },
  { "MONGOLIAN LETTER TODO KA", 0x1857 },
  { "MONGOLIAN LETTER TODO GAA", 0x1858 },
  { "MONGOLIAN LETTER TODO HAA", 0x1859 },
  { "MONGOLIAN LETTER TODO JIA", 0x185a },
  { "MONGOLIAN LETTER TODO NIA", 0x185b },
  { "MONGOLIAN LETTER TODO DZA", 0x185c },
  { "MONGOLIAN LETTER SIBE E", 0x185d },
  { "MONGOLIAN LETTER SIBE I", 0x185e },
  { "MONGOLIAN LETTER SIBE IY", 0x185f },
  { "MONGOLIAN LETTER SIBE UE", 0x1860 },
  { "MONGOLIAN LETTER SIBE U", 0x1861 },
  { "MONGOLIAN LETTER SIBE ANG", 0x1862 },
  { "MONGOLIAN LETTER SIBE KA", 0x1863 },
  { "MONGOLIAN LETTER SIBE GA", 0x1864 },
  { "MONGOLIAN LETTER SIBE HA", 0x1865 },
  { "MONGOLIAN LETTER SIBE PA", 0x1866 },
  { "MONGOLIAN LETTER SIBE SHA", 0x1867 },
  { "MONGOLIAN LETTER SIBE TA", 0x1868 },
  { "MONGOLIAN LETTER SIBE DA", 0x1869 },
  { "MONGOLIAN LETTER SIBE JA", 0x186a },
  { "MONGOLIAN LETTER SIBE FA", 0x186b },
  { "MONGOLIAN LETTER SIBE GAA", 0x186c },
  { "MONGOLIAN LETTER SIBE HAA", 0x186d },
  { "MONGOLIAN LETTER SIBE TSA", 0x186e },
  { "MONGOLIAN LETTER SIBE ZA", 0x186f },
  { "MONGOLIAN LETTER SIBE RAA", 0x1870 },
  { "MONGOLIAN LETTER SIBE CHA", 0x1871 },
  { "MONGOLIAN LETTER SIBE ZHA", 0x1872 },
  { "MONGOLIAN LETTER MANCHU I", 0x1873 },
  { "MONGOLIAN LETTER MANCHU KA", 0x1874 },
  { "MONGOLIAN LETTER MANCHU RA", 0x1875 },
  { "MONGOLIAN LETTER MANCHU FA", 0x1876 },
  { "MONGOLIAN LETTER MANCHU ZHA", 0x1877 },
  { "MONGOLIAN LETTER ALI GALI ANUSVARA ONE", 0x1880 },
  { "MONGOLIAN LETTER ALI GALI VISARGA ONE", 0x1881 },
  { "MONGOLIAN LETTER ALI GALI DAMARU", 0x1882 },
  { "MONGOLIAN LETTER ALI GALI UBADAMA", 0x1883 },
  { "MONGOLIAN LETTER ALI GALI INVERTED UBADAMA", 0x1884 },
  { "MONGOLIAN LETTER ALI GALI BALUDA", 0x1885 },
  { "MONGOLIAN LETTER ALI GALI THREE BALUDA", 0x1886 },
  { "MONGOLIAN LETTER ALI GALI A", 0x1887 },
  { "MONGOLIAN LETTER ALI GALI I", 0x1888 },
  { "MONGOLIAN LETTER ALI GALI KA", 0x1889 },
  { "MONGOLIAN LETTER ALI GALI NGA", 0x188a },
  { "MONGOLIAN LETTER ALI GALI CA", 0x188b },
  { "MONGOLIAN LETTER ALI GALI TTA", 0x188c },
  { "MONGOLIAN LETTER ALI GALI TTHA", 0x188d },
  { "MONGOLIAN LETTER ALI GALI DDA", 0x188e },
  { "MONGOLIAN LETTER ALI GALI NNA", 0x188f },
  { "MONGOLIAN LETTER ALI GALI TA", 0x1890 },
  { "MONGOLIAN LETTER ALI GALI DA", 0x1891 },
  { "MONGOLIAN LETTER ALI GALI PA", 0x1892 },
  { "MONGOLIAN LETTER ALI GALI PHA", 0x1893 },
  { "MONGOLIAN LETTER ALI GALI SSA", 0x1894 },
  { "MONGOLIAN LETTER ALI GALI ZHA", 0x1895 },
  { "MONGOLIAN LETTER ALI GALI ZA", 0x1896 },
  { "MONGOLIAN LETTER ALI GALI AH", 0x1897 },
  { "MONGOLIAN LETTER TODO ALI GALI TA", 0x1898 },
  { "MONGOLIAN LETTER TODO ALI GALI ZHA", 0x1899 },
  { "MONGOLIAN LETTER MANCHU ALI GALI GHA", 0x189a },
  { "MONGOLIAN LETTER MANCHU ALI GALI NGA", 0x189b },
  { "MONGOLIAN LETTER MANCHU ALI GALI CA", 0x189c },
  { "MONGOLIAN LETTER MANCHU ALI GALI JHA", 0x189d },
  { "MONGOLIAN LETTER MANCHU ALI GALI TTA", 0x189e },
  { "MONGOLIAN LETTER MANCHU ALI GALI DDHA", 0x189f },
  { "MONGOLIAN LETTER MANCHU ALI GALI TA", 0x18a0 },
  { "MONGOLIAN LETTER MANCHU ALI GALI DHA", 0x18a1 },
  { "MONGOLIAN LETTER MANCHU ALI GALI SSA", 0x18a2 },
  { "MONGOLIAN LETTER MANCHU ALI GALI CYA", 0x18a3 },
  { "MONGOLIAN LETTER MANCHU ALI GALI ZHA", 0x18a4 },
  { "MONGOLIAN LETTER MANCHU ALI GALI ZA", 0x18a5 },
  { "MONGOLIAN LETTER ALI GALI HALF U", 0x18a6 },
  { "MONGOLIAN LETTER ALI GALI HALF YA", 0x18a7 },
  { "MONGOLIAN LETTER MANCHU ALI GALI BHA", 0x18a8 },
  { "MONGOLIAN LETTER ALI GALI DAGALGA", 0x18a9 },
  { "LATIN CAPITAL LETTER A WITH RING BELOW", 0x1e00 },
  { "LATIN SMALL LETTER A WITH RING BELOW", 0x1e01 },
  { "LATIN CAPITAL LETTER B WITH DOT ABOVE", 0x1e02 },
  { "LATIN SMALL LETTER B WITH DOT ABOVE", 0x1e03 },
  { "LATIN CAPITAL LETTER B WITH DOT BELOW", 0x1e04 },
  { "LATIN SMALL LETTER B WITH DOT BELOW", 0x1e05 },
  { "LATIN CAPITAL LETTER B WITH LINE BELOW", 0x1e06 },
  { "LATIN SMALL LETTER B WITH LINE BELOW", 0x1e07 },
  { "LATIN CAPITAL LETTER C WITH CEDILLA AND ACUTE", 0x1e08 },
  { "LATIN SMALL LETTER C WITH CEDILLA AND ACUTE", 0x1e09 },
  { "LATIN CAPITAL LETTER D WITH DOT ABOVE", 0x1e0a },
  { "LATIN SMALL LETTER D WITH DOT ABOVE", 0x1e0b },
  { "LATIN CAPITAL LETTER D WITH DOT BELOW", 0x1e0c },
  { "LATIN SMALL LETTER D WITH DOT BELOW", 0x1e0d },
  { "LATIN CAPITAL LETTER D WITH LINE BELOW", 0x1e0e },
  { "LATIN SMALL LETTER D WITH LINE BELOW", 0x1e0f },
  { "LATIN CAPITAL LETTER D WITH CEDILLA", 0x1e10 },
  { "LATIN SMALL LETTER D WITH CEDILLA", 0x1e11 },
  { "LATIN CAPITAL LETTER D WITH CIRCUMFLEX BELOW", 0x1e12 },
  { "LATIN SMALL LETTER D WITH CIRCUMFLEX BELOW", 0x1e13 },
  { "LATIN CAPITAL LETTER E WITH MACRON AND GRAVE", 0x1e14 },
  { "LATIN SMALL LETTER E WITH MACRON AND GRAVE", 0x1e15 },
  { "LATIN CAPITAL LETTER E WITH MACRON AND ACUTE", 0x1e16 },
  { "LATIN SMALL LETTER E WITH MACRON AND ACUTE", 0x1e17 },
  { "LATIN CAPITAL LETTER E WITH CIRCUMFLEX BELOW", 0x1e18 },
  { "LATIN SMALL LETTER E WITH CIRCUMFLEX BELOW", 0x1e19 },
  { "LATIN CAPITAL LETTER E WITH TILDE BELOW", 0x1e1a },
  { "LATIN SMALL LETTER E WITH TILDE BELOW", 0x1e1b },
  { "LATIN CAPITAL LETTER E WITH CEDILLA AND BREVE", 0x1e1c },
  { "LATIN SMALL LETTER E WITH CEDILLA AND BREVE", 0x1e1d },
  { "LATIN CAPITAL LETTER F WITH DOT ABOVE", 0x1e1e },
  { "LATIN SMALL LETTER F WITH DOT ABOVE", 0x1e1f },
  { "LATIN CAPITAL LETTER G WITH MACRON", 0x1e20 },
  { "LATIN SMALL LETTER G WITH MACRON", 0x1e21 },
  { "LATIN CAPITAL LETTER H WITH DOT ABOVE", 0x1e22 },
  { "LATIN SMALL LETTER H WITH DOT ABOVE", 0x1e23 },
  { "LATIN CAPITAL LETTER H WITH DOT BELOW", 0x1e24 },
  { "LATIN SMALL LETTER H WITH DOT BELOW", 0x1e25 },
  { "LATIN CAPITAL LETTER H WITH DIAERESIS", 0x1e26 },
  { "LATIN SMALL LETTER H WITH DIAERESIS", 0x1e27 },
  { "LATIN CAPITAL LETTER H WITH CEDILLA", 0x1e28 },
  { "LATIN SMALL LETTER H WITH CEDILLA", 0x1e29 },
  { "LATIN CAPITAL LETTER H WITH BREVE BELOW", 0x1e2a },
  { "LATIN SMALL LETTER H WITH BREVE BELOW", 0x1e2b },
  { "LATIN CAPITAL LETTER I WITH TILDE BELOW", 0x1e2c },
  { "LATIN SMALL LETTER I WITH TILDE BELOW", 0x1e2d },
  { "LATIN CAPITAL LETTER I WITH DIAERESIS AND ACUTE", 0x1e2e },
  { "LATIN SMALL LETTER I WITH DIAERESIS AND ACUTE", 0x1e2f },
  { "LATIN CAPITAL LETTER K WITH ACUTE", 0x1e30 },
  { "LATIN SMALL LETTER K WITH ACUTE", 0x1e31 },
  { "LATIN CAPITAL LETTER K WITH DOT BELOW", 0x1e32 },
  { "LATIN SMALL LETTER K WITH DOT BELOW", 0x1e33 },
  { "LATIN CAPITAL LETTER K WITH LINE BELOW", 0x1e34 },
  { "LATIN SMALL LETTER K WITH LINE BELOW", 0x1e35 },
  { "LATIN CAPITAL LETTER L WITH DOT BELOW", 0x1e36 },
  { "LATIN SMALL LETTER L WITH DOT BELOW", 0x1e37 },
  { "LATIN CAPITAL LETTER L WITH DOT BELOW AND MACRON", 0x1e38 },
  { "LATIN SMALL LETTER L WITH DOT BELOW AND MACRON", 0x1e39 },
  { "LATIN CAPITAL LETTER L WITH LINE BELOW", 0x1e3a },
  { "LATIN SMALL LETTER L WITH LINE BELOW", 0x1e3b },
  { "LATIN CAPITAL LETTER L WITH CIRCUMFLEX BELOW", 0x1e3c },
  { "LATIN SMALL LETTER L WITH CIRCUMFLEX BELOW", 0x1e3d },
  { "LATIN CAPITAL LETTER M WITH ACUTE", 0x1e3e },
  { "LATIN SMALL LETTER M WITH ACUTE", 0x1e3f },
  { "LATIN CAPITAL LETTER M WITH DOT ABOVE", 0x1e40 },
  { "LATIN SMALL LETTER M WITH DOT ABOVE", 0x1e41 },
  { "LATIN CAPITAL LETTER M WITH DOT BELOW", 0x1e42 },
  { "LATIN SMALL LETTER M WITH DOT BELOW", 0x1e43 },
  { "LATIN CAPITAL LETTER N WITH DOT ABOVE", 0x1e44 },
  { "LATIN SMALL LETTER N WITH DOT ABOVE", 0x1e45 },
  { "LATIN CAPITAL LETTER N WITH DOT BELOW", 0x1e46 },
  { "LATIN SMALL LETTER N WITH DOT BELOW", 0x1e47 },
  { "LATIN CAPITAL LETTER N WITH LINE BELOW", 0x1e48 },
  { "LATIN SMALL LETTER N WITH LINE BELOW", 0x1e49 },
  { "LATIN CAPITAL LETTER N WITH CIRCUMFLEX BELOW", 0x1e4a },
  { "LATIN SMALL LETTER N WITH CIRCUMFLEX BELOW", 0x1e4b },
  { "LATIN CAPITAL LETTER O WITH TILDE AND ACUTE", 0x1e4c },
  { "LATIN SMALL LETTER O WITH TILDE AND ACUTE", 0x1e4d },
  { "LATIN CAPITAL LETTER O WITH TILDE AND DIAERESIS", 0x1e4e },
  { "LATIN SMALL LETTER O WITH TILDE AND DIAERESIS", 0x1e4f },
  { "LATIN CAPITAL LETTER O WITH MACRON AND GRAVE", 0x1e50 },
  { "LATIN SMALL LETTER O WITH MACRON AND GRAVE", 0x1e51 },
  { "LATIN CAPITAL LETTER O WITH MACRON AND ACUTE", 0x1e52 },
  { "LATIN SMALL LETTER O WITH MACRON AND ACUTE", 0x1e53 },
  { "LATIN CAPITAL LETTER P WITH ACUTE", 0x1e54 },
  { "LATIN SMALL LETTER P WITH ACUTE", 0x1e55 },
  { "LATIN CAPITAL LETTER P WITH DOT ABOVE", 0x1e56 },
  { "LATIN SMALL LETTER P WITH DOT ABOVE", 0x1e57 },
  { "LATIN CAPITAL LETTER R WITH DOT ABOVE", 0x1e58 },
  { "LATIN SMALL LETTER R WITH DOT ABOVE", 0x1e59 },
  { "LATIN CAPITAL LETTER R WITH DOT BELOW", 0x1e5a },
  { "LATIN SMALL LETTER R WITH DOT BELOW", 0x1e5b },
  { "LATIN CAPITAL LETTER R WITH DOT BELOW AND MACRON", 0x1e5c },
  { "LATIN SMALL LETTER R WITH DOT BELOW AND MACRON", 0x1e5d },
  { "LATIN CAPITAL LETTER R WITH LINE BELOW", 0x1e5e },
  { "LATIN SMALL LETTER R WITH LINE BELOW", 0x1e5f },
  { "LATIN CAPITAL LETTER S WITH DOT ABOVE", 0x1e60 },
  { "LATIN SMALL LETTER S WITH DOT ABOVE", 0x1e61 },
  { "LATIN CAPITAL LETTER S WITH DOT BELOW", 0x1e62 },
  { "LATIN SMALL LETTER S WITH DOT BELOW", 0x1e63 },
  { "LATIN CAPITAL LETTER S WITH ACUTE AND DOT ABOVE", 0x1e64 },
  { "LATIN SMALL LETTER S WITH ACUTE AND DOT ABOVE", 0x1e65 },
  { "LATIN CAPITAL LETTER S WITH CARON AND DOT ABOVE", 0x1e66 },
  { "LATIN SMALL LETTER S WITH CARON AND DOT ABOVE", 0x1e67 },
  { "LATIN CAPITAL LETTER S WITH DOT BELOW AND DOT ABOVE", 0x1e68 },
  { "LATIN SMALL LETTER S WITH DOT BELOW AND DOT ABOVE", 0x1e69 },
  { "LATIN CAPITAL LETTER T WITH DOT ABOVE", 0x1e6a },
  { "LATIN SMALL LETTER T WITH DOT ABOVE", 0x1e6b },
  { "LATIN CAPITAL LETTER T WITH DOT BELOW", 0x1e6c },
  { "LATIN SMALL LETTER T WITH DOT BELOW", 0x1e6d },
  { "LATIN CAPITAL LETTER T WITH LINE BELOW", 0x1e6e },
  { "LATIN SMALL LETTER T WITH LINE BELOW", 0x1e6f },
  { "LATIN CAPITAL LETTER T WITH CIRCUMFLEX BELOW", 0x1e70 },
  { "LATIN SMALL LETTER T WITH CIRCUMFLEX BELOW", 0x1e71 },
  { "LATIN CAPITAL LETTER U WITH DIAERESIS BELOW", 0x1e72 },
  { "LATIN SMALL LETTER U WITH DIAERESIS BELOW", 0x1e73 },
  { "LATIN CAPITAL LETTER U WITH TILDE BELOW", 0x1e74 },
  { "LATIN SMALL LETTER U WITH TILDE BELOW", 0x1e75 },
  { "LATIN CAPITAL LETTER U WITH CIRCUMFLEX BELOW", 0x1e76 },
  { "LATIN SMALL LETTER U WITH CIRCUMFLEX BELOW", 0x1e77 },
  { "LATIN CAPITAL LETTER U WITH TILDE AND ACUTE", 0x1e78 },
  { "LATIN SMALL LETTER U WITH TILDE AND ACUTE", 0x1e79 },
  { "LATIN CAPITAL LETTER U WITH MACRON AND DIAERESIS", 0x1e7a },
  { "LATIN SMALL LETTER U WITH MACRON AND DIAERESIS", 0x1e7b },
  { "LATIN CAPITAL LETTER V WITH TILDE", 0x1e7c },
  { "LATIN SMALL LETTER V WITH TILDE", 0x1e7d },
  { "LATIN CAPITAL LETTER V WITH DOT BELOW", 0x1e7e },
  { "LATIN SMALL LETTER V WITH DOT BELOW", 0x1e7f },
  { "LATIN CAPITAL LETTER W WITH GRAVE", 0x1e80 },
  { "LATIN SMALL LETTER W WITH GRAVE", 0x1e81 },
  { "LATIN CAPITAL LETTER W WITH ACUTE", 0x1e82 },
  { "LATIN SMALL LETTER W WITH ACUTE", 0x1e83 },
  { "LATIN CAPITAL LETTER W WITH DIAERESIS", 0x1e84 },
  { "LATIN SMALL LETTER W WITH DIAERESIS", 0x1e85 },
  { "LATIN CAPITAL LETTER W WITH DOT ABOVE", 0x1e86 },
  { "LATIN SMALL LETTER W WITH DOT ABOVE", 0x1e87 },
  { "LATIN CAPITAL LETTER W WITH DOT BELOW", 0x1e88 },
  { "LATIN SMALL LETTER W WITH DOT BELOW", 0x1e89 },
  { "LATIN CAPITAL LETTER X WITH DOT ABOVE", 0x1e8a },
  { "LATIN SMALL LETTER X WITH DOT ABOVE", 0x1e8b },
  { "LATIN CAPITAL LETTER X WITH DIAERESIS", 0x1e8c },
  { "LATIN SMALL LETTER X WITH DIAERESIS", 0x1e8d },
  { "LATIN CAPITAL LETTER Y WITH DOT ABOVE", 0x1e8e },
  { "LATIN SMALL LETTER Y WITH DOT ABOVE", 0x1e8f },
  { "LATIN CAPITAL LETTER Z WITH CIRCUMFLEX", 0x1e90 },
  { "LATIN SMALL LETTER Z WITH CIRCUMFLEX", 0x1e91 },
  { "LATIN CAPITAL LETTER Z WITH DOT BELOW", 0x1e92 },
  { "LATIN SMALL LETTER Z WITH DOT BELOW", 0x1e93 },
  { "LATIN CAPITAL LETTER Z WITH LINE BELOW", 0x1e94 },
  { "LATIN SMALL LETTER Z WITH LINE BELOW", 0x1e95 },
  { "LATIN SMALL LETTER H WITH LINE BELOW", 0x1e96 },
  { "LATIN SMALL LETTER T WITH DIAERESIS", 0x1e97 },
  { "LATIN SMALL LETTER W WITH RING ABOVE", 0x1e98 },
  { "LATIN SMALL LETTER Y WITH RING ABOVE", 0x1e99 },
  { "LATIN SMALL LETTER A WITH RIGHT HALF RING", 0x1e9a },
  { "LATIN SMALL LETTER LONG S WITH DOT ABOVE", 0x1e9b },
  { "LATIN CAPITAL LETTER A WITH DOT BELOW", 0x1ea0 },
  { "LATIN SMALL LETTER A WITH DOT BELOW", 0x1ea1 },
  { "LATIN CAPITAL LETTER A WITH HOOK ABOVE", 0x1ea2 },
  { "LATIN SMALL LETTER A WITH HOOK ABOVE", 0x1ea3 },
  { "LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND ACUTE", 0x1ea4 },
  { "LATIN SMALL LETTER A WITH CIRCUMFLEX AND ACUTE", 0x1ea5 },
  { "LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND GRAVE", 0x1ea6 },
  { "LATIN SMALL LETTER A WITH CIRCUMFLEX AND GRAVE", 0x1ea7 },
  { "LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND HOOK ABOVE", 0x1ea8 },
  { "LATIN SMALL LETTER A WITH CIRCUMFLEX AND HOOK ABOVE", 0x1ea9 },
  { "LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND TILDE", 0x1eaa },
  { "LATIN SMALL LETTER A WITH CIRCUMFLEX AND TILDE", 0x1eab },
  { "LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND DOT BELOW", 0x1eac },
  { "LATIN SMALL LETTER A WITH CIRCUMFLEX AND DOT BELOW", 0x1ead },
  { "LATIN CAPITAL LETTER A WITH BREVE AND ACUTE", 0x1eae },
  { "LATIN SMALL LETTER A WITH BREVE AND ACUTE", 0x1eaf },
  { "LATIN CAPITAL LETTER A WITH BREVE AND GRAVE", 0x1eb0 },
  { "LATIN SMALL LETTER A WITH BREVE AND GRAVE", 0x1eb1 },
  { "LATIN CAPITAL LETTER A WITH BREVE AND HOOK ABOVE", 0x1eb2 },
  { "LATIN SMALL LETTER A WITH BREVE AND HOOK ABOVE", 0x1eb3 },
  { "LATIN CAPITAL LETTER A WITH BREVE AND TILDE", 0x1eb4 },
  { "LATIN SMALL LETTER A WITH BREVE AND TILDE", 0x1eb5 },
  { "LATIN CAPITAL LETTER A WITH BREVE AND DOT BELOW", 0x1eb6 },
  { "LATIN SMALL LETTER A WITH BREVE AND DOT BELOW", 0x1eb7 },
  { "LATIN CAPITAL LETTER E WITH DOT BELOW", 0x1eb8 },
  { "LATIN SMALL LETTER E WITH DOT BELOW", 0x1eb9 },
  { "LATIN CAPITAL LETTER E WITH HOOK ABOVE", 0x1eba },
  { "LATIN SMALL LETTER E WITH HOOK ABOVE", 0x1ebb },
  { "LATIN CAPITAL LETTER E WITH TILDE", 0x1ebc },
  { "LATIN SMALL LETTER E WITH TILDE", 0x1ebd },
  { "LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND ACUTE", 0x1ebe },
  { "LATIN SMALL LETTER E WITH CIRCUMFLEX AND ACUTE", 0x1ebf },
  { "LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND GRAVE", 0x1ec0 },
  { "LATIN SMALL LETTER E WITH CIRCUMFLEX AND GRAVE", 0x1ec1 },
  { "LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND HOOK ABOVE", 0x1ec2 },
  { "LATIN SMALL LETTER E WITH CIRCUMFLEX AND HOOK ABOVE", 0x1ec3 },
  { "LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND TILDE", 0x1ec4 },
  { "LATIN SMALL LETTER E WITH CIRCUMFLEX AND TILDE", 0x1ec5 },
  { "LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND DOT BELOW", 0x1ec6 },
  { "LATIN SMALL LETTER E WITH CIRCUMFLEX AND DOT BELOW", 0x1ec7 },
  { "LATIN CAPITAL LETTER I WITH HOOK ABOVE", 0x1ec8 },
  { "LATIN SMALL LETTER I WITH HOOK ABOVE", 0x1ec9 },
  { "LATIN CAPITAL LETTER I WITH DOT BELOW", 0x1eca },
  { "LATIN SMALL LETTER I WITH DOT BELOW", 0x1ecb },
  { "LATIN CAPITAL LETTER O WITH DOT BELOW", 0x1ecc },
  { "LATIN SMALL LETTER O WITH DOT BELOW", 0x1ecd },
  { "LATIN CAPITAL LETTER O WITH HOOK ABOVE", 0x1ece },
  { "LATIN SMALL LETTER O WITH HOOK ABOVE", 0x1ecf },
  { "LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND ACUTE", 0x1ed0 },
  { "LATIN SMALL LETTER O WITH CIRCUMFLEX AND ACUTE", 0x1ed1 },
  { "LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND GRAVE", 0x1ed2 },
  { "LATIN SMALL LETTER O WITH CIRCUMFLEX AND GRAVE", 0x1ed3 },
  { "LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND HOOK ABOVE", 0x1ed4 },
  { "LATIN SMALL LETTER O WITH CIRCUMFLEX AND HOOK ABOVE", 0x1ed5 },
  { "LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND TILDE", 0x1ed6 },
  { "LATIN SMALL LETTER O WITH CIRCUMFLEX AND TILDE", 0x1ed7 },
  { "LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND DOT BELOW", 0x1ed8 },
  { "LATIN SMALL LETTER O WITH CIRCUMFLEX AND DOT BELOW", 0x1ed9 },
  { "LATIN CAPITAL LETTER O WITH HORN AND ACUTE", 0x1eda },
  { "LATIN SMALL LETTER O WITH HORN AND ACUTE", 0x1edb },
  { "LATIN CAPITAL LETTER O WITH HORN AND GRAVE", 0x1edc },
  { "LATIN SMALL LETTER O WITH HORN AND GRAVE", 0x1edd },
  { "LATIN CAPITAL LETTER O WITH HORN AND HOOK ABOVE", 0x1ede },
  { "LATIN SMALL LETTER O WITH HORN AND HOOK ABOVE", 0x1edf },
  { "LATIN CAPITAL LETTER O WITH HORN AND TILDE", 0x1ee0 },
  { "LATIN SMALL LETTER O WITH HORN AND TILDE", 0x1ee1 },
  { "LATIN CAPITAL LETTER O WITH HORN AND DOT BELOW", 0x1ee2 },
  { "LATIN SMALL LETTER O WITH HORN AND DOT BELOW", 0x1ee3 },
  { "LATIN CAPITAL LETTER U WITH DOT BELOW", 0x1ee4 },
  { "LATIN SMALL LETTER U WITH DOT BELOW", 0x1ee5 },
  { "LATIN CAPITAL LETTER U WITH HOOK ABOVE", 0x1ee6 },
  { "LATIN SMALL LETTER U WITH HOOK ABOVE", 0x1ee7 },
  { "LATIN CAPITAL LETTER U WITH HORN AND ACUTE", 0x1ee8 },
  { "LATIN SMALL LETTER U WITH HORN AND ACUTE", 0x1ee9 },
  { "LATIN CAPITAL LETTER U WITH HORN AND GRAVE", 0x1eea },
  { "LATIN SMALL LETTER U WITH HORN AND GRAVE", 0x1eeb },
  { "LATIN CAPITAL LETTER U WITH HORN AND HOOK ABOVE", 0x1eec },
  { "LATIN SMALL LETTER U WITH HORN AND HOOK ABOVE", 0x1eed },
  { "LATIN CAPITAL LETTER U WITH HORN AND TILDE", 0x1eee },
  { "LATIN SMALL LETTER U WITH HORN AND TILDE", 0x1eef },
  { "LATIN CAPITAL LETTER U WITH HORN AND DOT BELOW", 0x1ef0 },
  { "LATIN SMALL LETTER U WITH HORN AND DOT BELOW", 0x1ef1 },
  { "LATIN CAPITAL LETTER Y WITH GRAVE", 0x1ef2 },
  { "LATIN SMALL LETTER Y WITH GRAVE", 0x1ef3 },
  { "LATIN CAPITAL LETTER Y WITH DOT BELOW", 0x1ef4 },
  { "LATIN SMALL LETTER Y WITH DOT BELOW", 0x1ef5 },
  { "LATIN CAPITAL LETTER Y WITH HOOK ABOVE", 0x1ef6 },
  { "LATIN SMALL LETTER Y WITH HOOK ABOVE", 0x1ef7 },
  { "LATIN CAPITAL LETTER Y WITH TILDE", 0x1ef8 },
  { "LATIN SMALL LETTER Y WITH TILDE", 0x1ef9 },
  { "GREEK SMALL LETTER ALPHA WITH PSILI", 0x1f00 },
  { "GREEK SMALL LETTER ALPHA WITH DASIA", 0x1f01 },
  { "GREEK SMALL LETTER ALPHA WITH PSILI AND VARIA", 0x1f02 },
  { "GREEK SMALL LETTER ALPHA WITH DASIA AND VARIA", 0x1f03 },
  { "GREEK SMALL LETTER ALPHA WITH PSILI AND OXIA", 0x1f04 },
  { "GREEK SMALL LETTER ALPHA WITH DASIA AND OXIA", 0x1f05 },
  { "GREEK SMALL LETTER ALPHA WITH PSILI AND PERISPOMENI", 0x1f06 },
  { "GREEK SMALL LETTER ALPHA WITH DASIA AND PERISPOMENI", 0x1f07 },
  { "GREEK CAPITAL LETTER ALPHA WITH PSILI", 0x1f08 },
  { "GREEK CAPITAL LETTER ALPHA WITH DASIA", 0x1f09 },
  { "GREEK CAPITAL LETTER ALPHA WITH PSILI AND VARIA", 0x1f0a },
  { "GREEK CAPITAL LETTER ALPHA WITH DASIA AND VARIA", 0x1f0b },
  { "GREEK CAPITAL LETTER ALPHA WITH PSILI AND OXIA", 0x1f0c },
  { "GREEK CAPITAL LETTER ALPHA WITH DASIA AND OXIA", 0x1f0d },
  { "GREEK CAPITAL LETTER ALPHA WITH PSILI AND PERISPOMENI", 0x1f0e },
  { "GREEK CAPITAL LETTER ALPHA WITH DASIA AND PERISPOMENI", 0x1f0f },
  { "GREEK SMALL LETTER EPSILON WITH PSILI", 0x1f10 },
  { "GREEK SMALL LETTER EPSILON WITH DASIA", 0x1f11 },
  { "GREEK SMALL LETTER EPSILON WITH PSILI AND VARIA", 0x1f12 },
  { "GREEK SMALL LETTER EPSILON WITH DASIA AND VARIA", 0x1f13 },
  { "GREEK SMALL LETTER EPSILON WITH PSILI AND OXIA", 0x1f14 },
  { "GREEK SMALL LETTER EPSILON WITH DASIA AND OXIA", 0x1f15 },
  { "GREEK CAPITAL LETTER EPSILON WITH PSILI", 0x1f18 },
  { "GREEK CAPITAL LETTER EPSILON WITH DASIA", 0x1f19 },
  { "GREEK CAPITAL LETTER EPSILON WITH PSILI AND VARIA", 0x1f1a },
  { "GREEK CAPITAL LETTER EPSILON WITH DASIA AND VARIA", 0x1f1b },
  { "GREEK CAPITAL LETTER EPSILON WITH PSILI AND OXIA", 0x1f1c },
  { "GREEK CAPITAL LETTER EPSILON WITH DASIA AND OXIA", 0x1f1d },
  { "GREEK SMALL LETTER ETA WITH PSILI", 0x1f20 },
  { "GREEK SMALL LETTER ETA WITH DASIA", 0x1f21 },
  { "GREEK SMALL LETTER ETA WITH PSILI AND VARIA", 0x1f22 },
  { "GREEK SMALL LETTER ETA WITH DASIA AND VARIA", 0x1f23 },
  { "GREEK SMALL LETTER ETA WITH PSILI AND OXIA", 0x1f24 },
  { "GREEK SMALL LETTER ETA WITH DASIA AND OXIA", 0x1f25 },
  { "GREEK SMALL LETTER ETA WITH PSILI AND PERISPOMENI", 0x1f26 },
  { "GREEK SMALL LETTER ETA WITH DASIA AND PERISPOMENI", 0x1f27 },
  { "GREEK CAPITAL LETTER ETA WITH PSILI", 0x1f28 },
  { "GREEK CAPITAL LETTER ETA WITH DASIA", 0x1f29 },
  { "GREEK CAPITAL LETTER ETA WITH PSILI AND VARIA", 0x1f2a },
  { "GREEK CAPITAL LETTER ETA WITH DASIA AND VARIA", 0x1f2b },
  { "GREEK CAPITAL LETTER ETA WITH PSILI AND OXIA", 0x1f2c },
  { "GREEK CAPITAL LETTER ETA WITH DASIA AND OXIA", 0x1f2d },
  { "GREEK CAPITAL LETTER ETA WITH PSILI AND PERISPOMENI", 0x1f2e },
  { "GREEK CAPITAL LETTER ETA WITH DASIA AND PERISPOMENI", 0x1f2f },
  { "GREEK SMALL LETTER IOTA WITH PSILI", 0x1f30 },
  { "GREEK SMALL LETTER IOTA WITH DASIA", 0x1f31 },
  { "GREEK SMALL LETTER IOTA WITH PSILI AND VARIA", 0x1f32 },
  { "GREEK SMALL LETTER IOTA WITH DASIA AND VARIA", 0x1f33 },
  { "GREEK SMALL LETTER IOTA WITH PSILI AND OXIA", 0x1f34 },
  { "GREEK SMALL LETTER IOTA WITH DASIA AND OXIA", 0x1f35 },
  { "GREEK SMALL LETTER IOTA WITH PSILI AND PERISPOMENI", 0x1f36 },
  { "GREEK SMALL LETTER IOTA WITH DASIA AND PERISPOMENI", 0x1f37 },
  { "GREEK CAPITAL LETTER IOTA WITH PSILI", 0x1f38 },
  { "GREEK CAPITAL LETTER IOTA WITH DASIA", 0x1f39 },
  { "GREEK CAPITAL LETTER IOTA WITH PSILI AND VARIA", 0x1f3a },
  { "GREEK CAPITAL LETTER IOTA WITH DASIA AND VARIA", 0x1f3b },
  { "GREEK CAPITAL LETTER IOTA WITH PSILI AND OXIA", 0x1f3c },
  { "GREEK CAPITAL LETTER IOTA WITH DASIA AND OXIA", 0x1f3d },
  { "GREEK CAPITAL LETTER IOTA WITH PSILI AND PERISPOMENI", 0x1f3e },
  { "GREEK CAPITAL LETTER IOTA WITH DASIA AND PERISPOMENI", 0x1f3f },
  { "GREEK SMALL LETTER OMICRON WITH PSILI", 0x1f40 },
  { "GREEK SMALL LETTER OMICRON WITH DASIA", 0x1f41 },
  { "GREEK SMALL LETTER OMICRON WITH PSILI AND VARIA", 0x1f42 },
  { "GREEK SMALL LETTER OMICRON WITH DASIA AND VARIA", 0x1f43 },
  { "GREEK SMALL LETTER OMICRON WITH PSILI AND OXIA", 0x1f44 },
  { "GREEK SMALL LETTER OMICRON WITH DASIA AND OXIA", 0x1f45 },
  { "GREEK CAPITAL LETTER OMICRON WITH PSILI", 0x1f48 },
  { "GREEK CAPITAL LETTER OMICRON WITH DASIA", 0x1f49 },
  { "GREEK CAPITAL LETTER OMICRON WITH PSILI AND VARIA", 0x1f4a },
  { "GREEK CAPITAL LETTER OMICRON WITH DASIA AND VARIA", 0x1f4b },
  { "GREEK CAPITAL LETTER OMICRON WITH PSILI AND OXIA", 0x1f4c },
  { "GREEK CAPITAL LETTER OMICRON WITH DASIA AND OXIA", 0x1f4d },
  { "GREEK SMALL LETTER UPSILON WITH PSILI", 0x1f50 },
  { "GREEK SMALL LETTER UPSILON WITH DASIA", 0x1f51 },
  { "GREEK SMALL LETTER UPSILON WITH PSILI AND VARIA", 0x1f52 },
  { "GREEK SMALL LETTER UPSILON WITH DASIA AND VARIA", 0x1f53 },
  { "GREEK SMALL LETTER UPSILON WITH PSILI AND OXIA", 0x1f54 },
  { "GREEK SMALL LETTER UPSILON WITH DASIA AND OXIA", 0x1f55 },
  { "GREEK SMALL LETTER UPSILON WITH PSILI AND PERISPOMENI", 0x1f56 },
  { "GREEK SMALL LETTER UPSILON WITH DASIA AND PERISPOMENI", 0x1f57 },
  { "GREEK CAPITAL LETTER UPSILON WITH DASIA", 0x1f59 },
  { "GREEK CAPITAL LETTER UPSILON WITH DASIA AND VARIA", 0x1f5b },
  { "GREEK CAPITAL LETTER UPSILON WITH DASIA AND OXIA", 0x1f5d },
  { "GREEK CAPITAL LETTER UPSILON WITH DASIA AND PERISPOMENI", 0x1f5f },
  { "GREEK SMALL LETTER OMEGA WITH PSILI", 0x1f60 },
  { "GREEK SMALL LETTER OMEGA WITH DASIA", 0x1f61 },
  { "GREEK SMALL LETTER OMEGA WITH PSILI AND VARIA", 0x1f62 },
  { "GREEK SMALL LETTER OMEGA WITH DASIA AND VARIA", 0x1f63 },
  { "GREEK SMALL LETTER OMEGA WITH PSILI AND OXIA", 0x1f64 },
  { "GREEK SMALL LETTER OMEGA WITH DASIA AND OXIA", 0x1f65 },
  { "GREEK SMALL LETTER OMEGA WITH PSILI AND PERISPOMENI", 0x1f66 },
  { "GREEK SMALL LETTER OMEGA WITH DASIA AND PERISPOMENI", 0x1f67 },
  { "GREEK CAPITAL LETTER OMEGA WITH PSILI", 0x1f68 },
  { "GREEK CAPITAL LETTER OMEGA WITH DASIA", 0x1f69 },
  { "GREEK CAPITAL LETTER OMEGA WITH PSILI AND VARIA", 0x1f6a },
  { "GREEK CAPITAL LETTER OMEGA WITH DASIA AND VARIA", 0x1f6b },
  { "GREEK CAPITAL LETTER OMEGA WITH PSILI AND OXIA", 0x1f6c },
  { "GREEK CAPITAL LETTER OMEGA WITH DASIA AND OXIA", 0x1f6d },
  { "GREEK CAPITAL LETTER OMEGA WITH PSILI AND PERISPOMENI", 0x1f6e },
  { "GREEK CAPITAL LETTER OMEGA WITH DASIA AND PERISPOMENI", 0x1f6f },
  { "GREEK SMALL LETTER ALPHA WITH VARIA", 0x1f70 },
  { "GREEK SMALL LETTER ALPHA WITH OXIA", 0x1f71 },
  { "GREEK SMALL LETTER EPSILON WITH VARIA", 0x1f72 },
  { "GREEK SMALL LETTER EPSILON WITH OXIA", 0x1f73 },
  { "GREEK SMALL LETTER ETA WITH VARIA", 0x1f74 },
  { "GREEK SMALL LETTER ETA WITH OXIA", 0x1f75 },
  { "GREEK SMALL LETTER IOTA WITH VARIA", 0x1f76 },
  { "GREEK SMALL LETTER IOTA WITH OXIA", 0x1f77 },
  { "GREEK SMALL LETTER OMICRON WITH VARIA", 0x1f78 },
  { "GREEK SMALL LETTER OMICRON WITH OXIA", 0x1f79 },
  { "GREEK SMALL LETTER UPSILON WITH VARIA", 0x1f7a },
  { "GREEK SMALL LETTER UPSILON WITH OXIA", 0x1f7b },
  { "GREEK SMALL LETTER OMEGA WITH VARIA", 0x1f7c },
  { "GREEK SMALL LETTER OMEGA WITH OXIA", 0x1f7d },
  { "GREEK SMALL LETTER ALPHA WITH PSILI AND YPOGEGRAMMENI", 0x1f80 },
  { "GREEK SMALL LETTER ALPHA WITH DASIA AND YPOGEGRAMMENI", 0x1f81 },
  { "GREEK SMALL LETTER ALPHA WITH PSILI AND VARIA AND YPOGEGRAMMENI", 0x1f82 },
  { "GREEK SMALL LETTER ALPHA WITH DASIA AND VARIA AND YPOGEGRAMMENI", 0x1f83 },
  { "GREEK SMALL LETTER ALPHA WITH PSILI AND OXIA AND YPOGEGRAMMENI", 0x1f84 },
  { "GREEK SMALL LETTER ALPHA WITH DASIA AND OXIA AND YPOGEGRAMMENI", 0x1f85 },
  { "GREEK SMALL LETTER ALPHA WITH PSILI AND PERISPOMENI AND YPOGEGRAMMENI", 0x1f86 },
  { "GREEK SMALL LETTER ALPHA WITH DASIA AND PERISPOMENI AND YPOGEGRAMMENI", 0x1f87 },
  { "GREEK CAPITAL LETTER ALPHA WITH PSILI AND PROSGEGRAMMENI", 0x1f88 },
  { "GREEK CAPITAL LETTER ALPHA WITH DASIA AND PROSGEGRAMMENI", 0x1f89 },
  { "GREEK CAPITAL LETTER ALPHA WITH PSILI AND VARIA AND PROSGEGRAMMENI", 0x1f8a },
  { "GREEK CAPITAL LETTER ALPHA WITH DASIA AND VARIA AND PROSGEGRAMMENI", 0x1f8b },
  { "GREEK CAPITAL LETTER ALPHA WITH PSILI AND OXIA AND PROSGEGRAMMENI", 0x1f8c },
  { "GREEK CAPITAL LETTER ALPHA WITH DASIA AND OXIA AND PROSGEGRAMMENI", 0x1f8d },
  { "GREEK CAPITAL LETTER ALPHA WITH PSILI AND PERISPOMENI AND PROSGEGRAMMENI", 0x1f8e },
  { "GREEK CAPITAL LETTER ALPHA WITH DASIA AND PERISPOMENI AND PROSGEGRAMMENI", 0x1f8f },
  { "GREEK SMALL LETTER ETA WITH PSILI AND YPOGEGRAMMENI", 0x1f90 },
  { "GREEK SMALL LETTER ETA WITH DASIA AND YPOGEGRAMMENI", 0x1f91 },
  { "GREEK SMALL LETTER ETA WITH PSILI AND VARIA AND YPOGEGRAMMENI", 0x1f92 },
  { "GREEK SMALL LETTER ETA WITH DASIA AND VARIA AND YPOGEGRAMMENI", 0x1f93 },
  { "GREEK SMALL LETTER ETA WITH PSILI AND OXIA AND YPOGEGRAMMENI", 0x1f94 },
  { "GREEK SMALL LETTER ETA WITH DASIA AND OXIA AND YPOGEGRAMMENI", 0x1f95 },
  { "GREEK SMALL LETTER ETA WITH PSILI AND PERISPOMENI AND YPOGEGRAMMENI", 0x1f96 },
  { "GREEK SMALL LETTER ETA WITH DASIA AND PERISPOMENI AND YPOGEGRAMMENI", 0x1f97 },
  { "GREEK CAPITAL LETTER ETA WITH PSILI AND PROSGEGRAMMENI", 0x1f98 },
  { "GREEK CAPITAL LETTER ETA WITH DASIA AND PROSGEGRAMMENI", 0x1f99 },
  { "GREEK CAPITAL LETTER ETA WITH PSILI AND VARIA AND PROSGEGRAMMENI", 0x1f9a },
  { "GREEK CAPITAL LETTER ETA WITH DASIA AND VARIA AND PROSGEGRAMMENI", 0x1f9b },
  { "GREEK CAPITAL LETTER ETA WITH PSILI AND OXIA AND PROSGEGRAMMENI", 0x1f9c },
  { "GREEK CAPITAL LETTER ETA WITH DASIA AND OXIA AND PROSGEGRAMMENI", 0x1f9d },
  { "GREEK CAPITAL LETTER ETA WITH PSILI AND PERISPOMENI AND PROSGEGRAMMENI", 0x1f9e },
  { "GREEK CAPITAL LETTER ETA WITH DASIA AND PERISPOMENI AND PROSGEGRAMMENI", 0x1f9f },
  { "GREEK SMALL LETTER OMEGA WITH PSILI AND YPOGEGRAMMENI", 0x1fa0 },
  { "GREEK SMALL LETTER OMEGA WITH DASIA AND YPOGEGRAMMENI", 0x1fa1 },
  { "GREEK SMALL LETTER OMEGA WITH PSILI AND VARIA AND YPOGEGRAMMENI", 0x1fa2 },
  { "GREEK SMALL LETTER OMEGA WITH DASIA AND VARIA AND YPOGEGRAMMENI", 0x1fa3 },
  { "GREEK SMALL LETTER OMEGA WITH PSILI AND OXIA AND YPOGEGRAMMENI", 0x1fa4 },
  { "GREEK SMALL LETTER OMEGA WITH DASIA AND OXIA AND YPOGEGRAMMENI", 0x1fa5 },
  { "GREEK SMALL LETTER OMEGA WITH PSILI AND PERISPOMENI AND YPOGEGRAMMENI", 0x1fa6 },
  { "GREEK SMALL LETTER OMEGA WITH DASIA AND PERISPOMENI AND YPOGEGRAMMENI", 0x1fa7 },
  { "GREEK CAPITAL LETTER OMEGA WITH PSILI AND PROSGEGRAMMENI", 0x1fa8 },
  { "GREEK CAPITAL LETTER OMEGA WITH DASIA AND PROSGEGRAMMENI", 0x1fa9 },
  { "GREEK CAPITAL LETTER OMEGA WITH PSILI AND VARIA AND PROSGEGRAMMENI", 0x1faa },
  { "GREEK CAPITAL LETTER OMEGA WITH DASIA AND VARIA AND PROSGEGRAMMENI", 0x1fab },
  { "GREEK CAPITAL LETTER OMEGA WITH PSILI AND OXIA AND PROSGEGRAMMENI", 0x1fac },
  { "GREEK CAPITAL LETTER OMEGA WITH DASIA AND OXIA AND PROSGEGRAMMENI", 0x1fad },
  { "GREEK CAPITAL LETTER OMEGA WITH PSILI AND PERISPOMENI AND PROSGEGRAMMENI", 0x1fae },
  { "GREEK CAPITAL LETTER OMEGA WITH DASIA AND PERISPOMENI AND PROSGEGRAMMENI", 0x1faf },
  { "GREEK SMALL LETTER ALPHA WITH VRACHY", 0x1fb0 },
  { "GREEK SMALL LETTER ALPHA WITH MACRON", 0x1fb1 },
  { "GREEK SMALL LETTER ALPHA WITH VARIA AND YPOGEGRAMMENI", 0x1fb2 },
  { "GREEK SMALL LETTER ALPHA WITH YPOGEGRAMMENI", 0x1fb3 },
  { "GREEK SMALL LETTER ALPHA WITH OXIA AND YPOGEGRAMMENI", 0x1fb4 },
  { "GREEK SMALL LETTER ALPHA WITH PERISPOMENI", 0x1fb6 },
  { "GREEK SMALL LETTER ALPHA WITH PERISPOMENI AND YPOGEGRAMMENI", 0x1fb7 },
  { "GREEK CAPITAL LETTER ALPHA WITH VRACHY", 0x1fb8 },
  { "GREEK CAPITAL LETTER ALPHA WITH MACRON", 0x1fb9 },
  { "GREEK CAPITAL LETTER ALPHA WITH VARIA", 0x1fba },
  { "GREEK CAPITAL LETTER ALPHA WITH OXIA", 0x1fbb },
  { "GREEK CAPITAL LETTER ALPHA WITH PROSGEGRAMMENI", 0x1fbc },
  { "GREEK KORONIS", 0x1fbd },
  { "GREEK PROSGEGRAMMENI", 0x1fbe },
  { "GREEK PSILI", 0x1fbf },
  { "GREEK PERISPOMENI", 0x1fc0 },
  { "GREEK DIALYTIKA AND PERISPOMENI", 0x1fc1 },
  { "GREEK SMALL LETTER ETA WITH VARIA AND YPOGEGRAMMENI", 0x1fc2 },
  { "GREEK SMALL LETTER ETA WITH YPOGEGRAMMENI", 0x1fc3 },
  { "GREEK SMALL LETTER ETA WITH OXIA AND YPOGEGRAMMENI", 0x1fc4 },
  { "GREEK SMALL LETTER ETA WITH PERISPOMENI", 0x1fc6 },
  { "GREEK SMALL LETTER ETA WITH PERISPOMENI AND YPOGEGRAMMENI", 0x1fc7 },
  { "GREEK CAPITAL LETTER EPSILON WITH VARIA", 0x1fc8 },
  { "GREEK CAPITAL LETTER EPSILON WITH OXIA", 0x1fc9 },
  { "GREEK CAPITAL LETTER ETA WITH VARIA", 0x1fca },
  { "GREEK CAPITAL LETTER ETA WITH OXIA", 0x1fcb },
  { "GREEK CAPITAL LETTER ETA WITH PROSGEGRAMMENI", 0x1fcc },
  { "GREEK PSILI AND VARIA", 0x1fcd },
  { "GREEK PSILI AND OXIA", 0x1fce },
  { "GREEK PSILI AND PERISPOMENI", 0x1fcf },
  { "GREEK SMALL LETTER IOTA WITH VRACHY", 0x1fd0 },
  { "GREEK SMALL LETTER IOTA WITH MACRON", 0x1fd1 },
  { "GREEK SMALL LETTER IOTA WITH DIALYTIKA AND VARIA", 0x1fd2 },
  { "GREEK SMALL LETTER IOTA WITH DIALYTIKA AND OXIA", 0x1fd3 },
  { "GREEK SMALL LETTER IOTA WITH PERISPOMENI", 0x1fd6 },
  { "GREEK SMALL LETTER IOTA WITH DIALYTIKA AND PERISPOMENI", 0x1fd7 },
  { "GREEK CAPITAL LETTER IOTA WITH VRACHY", 0x1fd8 },
  { "GREEK CAPITAL LETTER IOTA WITH MACRON", 0x1fd9 },
  { "GREEK CAPITAL LETTER IOTA WITH VARIA", 0x1fda },
  { "GREEK CAPITAL LETTER IOTA WITH OXIA", 0x1fdb },
  { "GREEK DASIA AND VARIA", 0x1fdd },
  { "GREEK DASIA AND OXIA", 0x1fde },
  { "GREEK DASIA AND PERISPOMENI", 0x1fdf },
  { "GREEK SMALL LETTER UPSILON WITH VRACHY", 0x1fe0 },
  { "GREEK SMALL LETTER UPSILON WITH MACRON", 0x1fe1 },
  { "GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND VARIA", 0x1fe2 },
  { "GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND OXIA", 0x1fe3 },
  { "GREEK SMALL LETTER RHO WITH PSILI", 0x1fe4 },
  { "GREEK SMALL LETTER RHO WITH DASIA", 0x1fe5 },
  { "GREEK SMALL LETTER UPSILON WITH PERISPOMENI", 0x1fe6 },
  { "GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND PERISPOMENI", 0x1fe7 },
  { "GREEK CAPITAL LETTER UPSILON WITH VRACHY", 0x1fe8 },
  { "GREEK CAPITAL LETTER UPSILON WITH MACRON", 0x1fe9 },
  { "GREEK CAPITAL LETTER UPSILON WITH VARIA", 0x1fea },
  { "GREEK CAPITAL LETTER UPSILON WITH OXIA", 0x1feb },
  { "GREEK CAPITAL LETTER RHO WITH DASIA", 0x1fec },
  { "GREEK DIALYTIKA AND VARIA", 0x1fed },
  { "GREEK DIALYTIKA AND OXIA", 0x1fee },
  { "GREEK VARIA", 0x1fef },
  { "GREEK SMALL LETTER OMEGA WITH VARIA AND YPOGEGRAMMENI", 0x1ff2 },
  { "GREEK SMALL LETTER OMEGA WITH YPOGEGRAMMENI", 0x1ff3 },
  { "GREEK SMALL LETTER OMEGA WITH OXIA AND YPOGEGRAMMENI", 0x1ff4 },
  { "GREEK SMALL LETTER OMEGA WITH PERISPOMENI", 0x1ff6 },
  { "GREEK SMALL LETTER OMEGA WITH PERISPOMENI AND YPOGEGRAMMENI", 0x1ff7 },
  { "GREEK CAPITAL LETTER OMICRON WITH VARIA", 0x1ff8 },
  { "GREEK CAPITAL LETTER OMICRON WITH OXIA", 0x1ff9 },
  { "GREEK CAPITAL LETTER OMEGA WITH VARIA", 0x1ffa },
  { "GREEK CAPITAL LETTER OMEGA WITH OXIA", 0x1ffb },
  { "GREEK CAPITAL LETTER OMEGA WITH PROSGEGRAMMENI", 0x1ffc },
  { "GREEK OXIA", 0x1ffd },
  { "GREEK DASIA", 0x1ffe },
  { "EN QUAD", 0x2000 },
  { "EM QUAD", 0x2001 },
  { "EN SPACE", 0x2002 },
  { "EM SPACE", 0x2003 },
  { "THREE-PER-EM SPACE", 0x2004 },
  { "FOUR-PER-EM SPACE", 0x2005 },
  { "SIX-PER-EM SPACE", 0x2006 },
  { "FIGURE SPACE", 0x2007 },
  { "PUNCTUATION SPACE", 0x2008 },
  { "THIN SPACE", 0x2009 },
  { "HAIR SPACE", 0x200a },
  { "ZERO WIDTH SPACE", 0x200b },
  { "ZERO WIDTH NON-JOINER", 0x200c },
  { "ZERO WIDTH JOINER", 0x200d },
  { "LEFT-TO-RIGHT MARK", 0x200e },
  { "RIGHT-TO-LEFT MARK", 0x200f },
  { "HYPHEN", 0x2010 },
  { "NON-BREAKING HYPHEN", 0x2011 },
  { "FIGURE DASH", 0x2012 },
  { "EN DASH", 0x2013 },
  { "EM DASH", 0x2014 },
  { "HORIZONTAL BAR", 0x2015 },
  { "DOUBLE VERTICAL LINE", 0x2016 },
  { "DOUBLE LOW LINE", 0x2017 },
  { "LEFT SINGLE QUOTATION MARK", 0x2018 },
  { "RIGHT SINGLE QUOTATION MARK", 0x2019 },
  { "SINGLE LOW-9 QUOTATION MARK", 0x201a },
  { "SINGLE HIGH-REVERSED-9 QUOTATION MARK", 0x201b },
  { "LEFT DOUBLE QUOTATION MARK", 0x201c },
  { "RIGHT DOUBLE QUOTATION MARK", 0x201d },
  { "DOUBLE LOW-9 QUOTATION MARK", 0x201e },
  { "DOUBLE HIGH-REVERSED-9 QUOTATION MARK", 0x201f },
  { "DAGGER", 0x2020 },
  { "DOUBLE DAGGER", 0x2021 },
  { "BULLET", 0x2022 },
  { "TRIANGULAR BULLET", 0x2023 },
  { "ONE DOT LEADER", 0x2024 },
  { "TWO DOT LEADER", 0x2025 },
  { "HORIZONTAL ELLIPSIS", 0x2026 },
  { "HYPHENATION POINT", 0x2027 },
  { "LINE SEPARATOR", 0x2028 },
  { "PARAGRAPH SEPARATOR", 0x2029 },
  { "LEFT-TO-RIGHT EMBEDDING", 0x202a },
  { "RIGHT-TO-LEFT EMBEDDING", 0x202b },
  { "POP DIRECTIONAL FORMATTING", 0x202c },
  { "LEFT-TO-RIGHT OVERRIDE", 0x202d },
  { "RIGHT-TO-LEFT OVERRIDE", 0x202e },
  { "NARROW NO-BREAK SPACE", 0x202f },
  { "PER MILLE SIGN", 0x2030 },
  { "PER TEN THOUSAND SIGN", 0x2031 },
  { "PRIME", 0x2032 },
  { "DOUBLE PRIME", 0x2033 },
  { "TRIPLE PRIME", 0x2034 },
  { "REVERSED PRIME", 0x2035 },
  { "REVERSED DOUBLE PRIME", 0x2036 },
  { "REVERSED TRIPLE PRIME", 0x2037 },
  { "CARET", 0x2038 },
  { "SINGLE LEFT-POINTING ANGLE QUOTATION MARK", 0x2039 },
  { "SINGLE RIGHT-POINTING ANGLE QUOTATION MARK", 0x203a },
  { "REFERENCE MARK", 0x203b },
  { "DOUBLE EXCLAMATION MARK", 0x203c },
  { "INTERROBANG", 0x203d },
  { "OVERLINE", 0x203e },
  { "UNDERTIE", 0x203f },
  { "CHARACTER TIE", 0x2040 },
  { "CARET INSERTION POINT", 0x2041 },
  { "ASTERISM", 0x2042 },
  { "HYPHEN BULLET", 0x2043 },
  { "FRACTION SLASH", 0x2044 },
  { "LEFT SQUARE BRACKET WITH QUILL", 0x2045 },
  { "RIGHT SQUARE BRACKET WITH QUILL", 0x2046 },
  { "QUESTION EXCLAMATION MARK", 0x2048 },
  { "EXCLAMATION QUESTION MARK", 0x2049 },
  { "TIRONIAN SIGN ET", 0x204a },
  { "REVERSED PILCROW SIGN", 0x204b },
  { "BLACK LEFTWARDS BULLET", 0x204c },
  { "BLACK RIGHTWARDS BULLET", 0x204d },
  { "INHIBIT SYMMETRIC SWAPPING", 0x206a },
  { "ACTIVATE SYMMETRIC SWAPPING", 0x206b },
  { "INHIBIT ARABIC FORM SHAPING", 0x206c },
  { "ACTIVATE ARABIC FORM SHAPING", 0x206d },
  { "NATIONAL DIGIT SHAPES", 0x206e },
  { "NOMINAL DIGIT SHAPES", 0x206f },
  { "SUPERSCRIPT ZERO", 0x2070 },
  { "SUPERSCRIPT FOUR", 0x2074 },
  { "SUPERSCRIPT FIVE", 0x2075 },
  { "SUPERSCRIPT SIX", 0x2076 },
  { "SUPERSCRIPT SEVEN", 0x2077 },
  { "SUPERSCRIPT EIGHT", 0x2078 },
  { "SUPERSCRIPT NINE", 0x2079 },
  { "SUPERSCRIPT PLUS SIGN", 0x207a },
  { "SUPERSCRIPT MINUS", 0x207b },
  { "SUPERSCRIPT EQUALS SIGN", 0x207c },
  { "SUPERSCRIPT LEFT PARENTHESIS", 0x207d },
  { "SUPERSCRIPT RIGHT PARENTHESIS", 0x207e },
  { "SUPERSCRIPT LATIN SMALL LETTER N", 0x207f },
  { "SUBSCRIPT ZERO", 0x2080 },
  { "SUBSCRIPT ONE", 0x2081 },
  { "SUBSCRIPT TWO", 0x2082 },
  { "SUBSCRIPT THREE", 0x2083 },
  { "SUBSCRIPT FOUR", 0x2084 },
  { "SUBSCRIPT FIVE", 0x2085 },
  { "SUBSCRIPT SIX", 0x2086 },
  { "SUBSCRIPT SEVEN", 0x2087 },
  { "SUBSCRIPT EIGHT", 0x2088 },
  { "SUBSCRIPT NINE", 0x2089 },
  { "SUBSCRIPT PLUS SIGN", 0x208a },
  { "SUBSCRIPT MINUS", 0x208b },
  { "SUBSCRIPT EQUALS SIGN", 0x208c },
  { "SUBSCRIPT LEFT PARENTHESIS", 0x208d },
  { "SUBSCRIPT RIGHT PARENTHESIS", 0x208e },
  { "EURO-CURRENCY SIGN", 0x20a0 },
  { "COLON SIGN", 0x20a1 },
  { "CRUZEIRO SIGN", 0x20a2 },
  { "FRENCH FRANC SIGN", 0x20a3 },
  { "LIRA SIGN", 0x20a4 },
  { "MILL SIGN", 0x20a5 },
  { "NAIRA SIGN", 0x20a6 },
  { "PESETA SIGN", 0x20a7 },
  { "RUPEE SIGN", 0x20a8 },
  { "WON SIGN", 0x20a9 },
  { "NEW SHEQEL SIGN", 0x20aa },
  { "DONG SIGN", 0x20ab },
  { "EURO SIGN", 0x20ac },
  { "KIP SIGN", 0x20ad },
  { "TUGRIK SIGN", 0x20ae },
  { "DRACHMA SIGN", 0x20af },
  { "COMBINING LEFT HARPOON ABOVE", 0x20d0 },
  { "COMBINING RIGHT HARPOON ABOVE", 0x20d1 },
  { "COMBINING LONG VERTICAL LINE OVERLAY", 0x20d2 },
  { "COMBINING SHORT VERTICAL LINE OVERLAY", 0x20d3 },
  { "COMBINING ANTICLOCKWISE ARROW ABOVE", 0x20d4 },
  { "COMBINING CLOCKWISE ARROW ABOVE", 0x20d5 },
  { "COMBINING LEFT ARROW ABOVE", 0x20d6 },
  { "COMBINING RIGHT ARROW ABOVE", 0x20d7 },
  { "COMBINING RING OVERLAY", 0x20d8 },
  { "COMBINING CLOCKWISE RING OVERLAY", 0x20d9 },
  { "COMBINING ANTICLOCKWISE RING OVERLAY", 0x20da },
  { "COMBINING THREE DOTS ABOVE", 0x20db },
  { "COMBINING FOUR DOTS ABOVE", 0x20dc },
  { "COMBINING ENCLOSING CIRCLE", 0x20dd },
  { "COMBINING ENCLOSING SQUARE", 0x20de },
  { "COMBINING ENCLOSING DIAMOND", 0x20df },
  { "COMBINING ENCLOSING CIRCLE BACKSLASH", 0x20e0 },
  { "COMBINING LEFT RIGHT ARROW ABOVE", 0x20e1 },
  { "COMBINING ENCLOSING SCREEN", 0x20e2 },
  { "COMBINING ENCLOSING KEYCAP", 0x20e3 },
  { "ACCOUNT OF", 0x2100 },
  { "ADDRESSED TO THE SUBJECT", 0x2101 },
  { "DOUBLE-STRUCK CAPITAL C", 0x2102 },
  { "DEGREE CELSIUS", 0x2103 },
  { "CENTRE LINE SYMBOL", 0x2104 },
  { "CARE OF", 0x2105 },
  { "CADA UNA", 0x2106 },
  { "EULER CONSTANT", 0x2107 },
  { "SCRUPLE", 0x2108 },
  { "DEGREE FAHRENHEIT", 0x2109 },
  { "SCRIPT SMALL G", 0x210a },
  { "SCRIPT CAPITAL H", 0x210b },
  { "BLACK-LETTER CAPITAL H", 0x210c },
  { "DOUBLE-STRUCK CAPITAL H", 0x210d },
  { "PLANCK CONSTANT", 0x210e },
  { "PLANCK CONSTANT OVER TWO PI", 0x210f },
  { "SCRIPT CAPITAL I", 0x2110 },
  { "BLACK-LETTER CAPITAL I", 0x2111 },
  { "SCRIPT CAPITAL L", 0x2112 },
  { "SCRIPT SMALL L", 0x2113 },
  { "L B BAR SYMBOL", 0x2114 },
  { "DOUBLE-STRUCK CAPITAL N", 0x2115 },
  { "NUMERO SIGN", 0x2116 },
  { "SOUND RECORDING COPYRIGHT", 0x2117 },
  { "SCRIPT CAPITAL P", 0x2118 },
  { "DOUBLE-STRUCK CAPITAL P", 0x2119 },
  { "DOUBLE-STRUCK CAPITAL Q", 0x211a },
  { "SCRIPT CAPITAL R", 0x211b },
  { "BLACK-LETTER CAPITAL R", 0x211c },
  { "DOUBLE-STRUCK CAPITAL R", 0x211d },
  { "PRESCRIPTION TAKE", 0x211e },
  { "RESPONSE", 0x211f },
  { "SERVICE MARK", 0x2120 },
  { "TELEPHONE SIGN", 0x2121 },
  { "TRADE MARK SIGN", 0x2122 },
  { "VERSICLE", 0x2123 },
  { "DOUBLE-STRUCK CAPITAL Z", 0x2124 },
  { "OUNCE SIGN", 0x2125 },
  { "OHM SIGN", 0x2126 },
  { "INVERTED OHM SIGN", 0x2127 },
  { "BLACK-LETTER CAPITAL Z", 0x2128 },
  { "TURNED GREEK SMALL LETTER IOTA", 0x2129 },
  { "KELVIN SIGN", 0x212a },
  { "ANGSTROM SIGN", 0x212b },
  { "SCRIPT CAPITAL B", 0x212c },
  { "BLACK-LETTER CAPITAL C", 0x212d },
  { "ESTIMATED SYMBOL", 0x212e },
  { "SCRIPT SMALL E", 0x212f },
  { "SCRIPT CAPITAL E", 0x2130 },
  { "SCRIPT CAPITAL F", 0x2131 },
  { "TURNED CAPITAL F", 0x2132 },
  { "SCRIPT CAPITAL M", 0x2133 },
  { "SCRIPT SMALL O", 0x2134 },
  { "ALEF SYMBOL", 0x2135 },
  { "BET SYMBOL", 0x2136 },
  { "GIMEL SYMBOL", 0x2137 },
  { "DALET SYMBOL", 0x2138 },
  { "INFORMATION SOURCE", 0x2139 },
  { "ROTATED CAPITAL Q", 0x213a },
  { "VULGAR FRACTION ONE THIRD", 0x2153 },
  { "VULGAR FRACTION TWO THIRDS", 0x2154 },
  { "VULGAR FRACTION ONE FIFTH", 0x2155 },
  { "VULGAR FRACTION TWO FIFTHS", 0x2156 },
  { "VULGAR FRACTION THREE FIFTHS", 0x2157 },
  { "VULGAR FRACTION FOUR FIFTHS", 0x2158 },
  { "VULGAR FRACTION ONE SIXTH", 0x2159 },
  { "VULGAR FRACTION FIVE SIXTHS", 0x215a },
  { "VULGAR FRACTION ONE EIGHTH", 0x215b },
  { "VULGAR FRACTION THREE EIGHTHS", 0x215c },
  { "VULGAR FRACTION FIVE EIGHTHS", 0x215d },
  { "VULGAR FRACTION SEVEN EIGHTHS", 0x215e },
  { "FRACTION NUMERATOR ONE", 0x215f },
  { "ROMAN NUMERAL ONE", 0x2160 },
  { "ROMAN NUMERAL TWO", 0x2161 },
  { "ROMAN NUMERAL THREE", 0x2162 },
  { "ROMAN NUMERAL FOUR", 0x2163 },
  { "ROMAN NUMERAL FIVE", 0x2164 },
  { "ROMAN NUMERAL SIX", 0x2165 },
  { "ROMAN NUMERAL SEVEN", 0x2166 },
  { "ROMAN NUMERAL EIGHT", 0x2167 },
  { "ROMAN NUMERAL NINE", 0x2168 },
  { "ROMAN NUMERAL TEN", 0x2169 },
  { "ROMAN NUMERAL ELEVEN", 0x216a },
  { "ROMAN NUMERAL TWELVE", 0x216b },
  { "ROMAN NUMERAL FIFTY", 0x216c },
  { "ROMAN NUMERAL ONE HUNDRED", 0x216d },
  { "ROMAN NUMERAL FIVE HUNDRED", 0x216e },
  { "ROMAN NUMERAL ONE THOUSAND", 0x216f },
  { "SMALL ROMAN NUMERAL ONE", 0x2170 },
  { "SMALL ROMAN NUMERAL TWO", 0x2171 },
  { "SMALL ROMAN NUMERAL THREE", 0x2172 },
  { "SMALL ROMAN NUMERAL FOUR", 0x2173 },
  { "SMALL ROMAN NUMERAL FIVE", 0x2174 },
  { "SMALL ROMAN NUMERAL SIX", 0x2175 },
  { "SMALL ROMAN NUMERAL SEVEN", 0x2176 },
  { "SMALL ROMAN NUMERAL EIGHT", 0x2177 },
  { "SMALL ROMAN NUMERAL NINE", 0x2178 },
  { "SMALL ROMAN NUMERAL TEN", 0x2179 },
  { "SMALL ROMAN NUMERAL ELEVEN", 0x217a },
  { "SMALL ROMAN NUMERAL TWELVE", 0x217b },
  { "SMALL ROMAN NUMERAL FIFTY", 0x217c },
  { "SMALL ROMAN NUMERAL ONE HUNDRED", 0x217d },
  { "SMALL ROMAN NUMERAL FIVE HUNDRED", 0x217e },
  { "SMALL ROMAN NUMERAL ONE THOUSAND", 0x217f },
  { "ROMAN NUMERAL ONE THOUSAND C D", 0x2180 },
  { "ROMAN NUMERAL FIVE THOUSAND", 0x2181 },
  { "ROMAN NUMERAL TEN THOUSAND", 0x2182 },
  { "ROMAN NUMERAL REVERSED ONE HUNDRED", 0x2183 },
  { "LEFTWARDS ARROW", 0x2190 },
  { "UPWARDS ARROW", 0x2191 },
  { "RIGHTWARDS ARROW", 0x2192 },
  { "DOWNWARDS ARROW", 0x2193 },
  { "LEFT RIGHT ARROW", 0x2194 },
  { "UP DOWN ARROW", 0x2195 },
  { "NORTH WEST ARROW", 0x2196 },
  { "NORTH EAST ARROW", 0x2197 },
  { "SOUTH EAST ARROW", 0x2198 },
  { "SOUTH WEST ARROW", 0x2199 },
  { "LEFTWARDS ARROW WITH STROKE", 0x219a },
  { "RIGHTWARDS ARROW WITH STROKE", 0x219b },
  { "LEFTWARDS WAVE ARROW", 0x219c },
  { "RIGHTWARDS WAVE ARROW", 0x219d },
  { "LEFTWARDS TWO HEADED ARROW", 0x219e },
  { "UPWARDS TWO HEADED ARROW", 0x219f },
  { "RIGHTWARDS TWO HEADED ARROW", 0x21a0 },
  { "DOWNWARDS TWO HEADED ARROW", 0x21a1 },
  { "LEFTWARDS ARROW WITH TAIL", 0x21a2 },
  { "RIGHTWARDS ARROW WITH TAIL", 0x21a3 },
  { "LEFTWARDS ARROW FROM BAR", 0x21a4 },
  { "UPWARDS ARROW FROM BAR", 0x21a5 },
  { "RIGHTWARDS ARROW FROM BAR", 0x21a6 },
  { "DOWNWARDS ARROW FROM BAR", 0x21a7 },
  { "UP DOWN ARROW WITH BASE", 0x21a8 },
  { "LEFTWARDS ARROW WITH HOOK", 0x21a9 },
  { "RIGHTWARDS ARROW WITH HOOK", 0x21aa },
  { "LEFTWARDS ARROW WITH LOOP", 0x21ab },
  { "RIGHTWARDS ARROW WITH LOOP", 0x21ac },
  { "LEFT RIGHT WAVE ARROW", 0x21ad },
  { "LEFT RIGHT ARROW WITH STROKE", 0x21ae },
  { "DOWNWARDS ZIGZAG ARROW", 0x21af },
  { "UPWARDS ARROW WITH TIP LEFTWARDS", 0x21b0 },
  { "UPWARDS ARROW WITH TIP RIGHTWARDS", 0x21b1 },
  { "DOWNWARDS ARROW WITH TIP LEFTWARDS", 0x21b2 },
  { "DOWNWARDS ARROW WITH TIP RIGHTWARDS", 0x21b3 },
  { "RIGHTWARDS ARROW WITH CORNER DOWNWARDS", 0x21b4 },
  { "DOWNWARDS ARROW WITH CORNER LEFTWARDS", 0x21b5 },
  { "ANTICLOCKWISE TOP SEMICIRCLE ARROW", 0x21b6 },
  { "CLOCKWISE TOP SEMICIRCLE ARROW", 0x21b7 },
  { "NORTH WEST ARROW TO LONG BAR", 0x21b8 },
  { "LEFTWARDS ARROW TO BAR OVER RIGHTWARDS ARROW TO BAR", 0x21b9 },
  { "ANTICLOCKWISE OPEN CIRCLE ARROW", 0x21ba },
  { "CLOCKWISE OPEN CIRCLE ARROW", 0x21bb },
  { "LEFTWARDS HARPOON WITH BARB UPWARDS", 0x21bc },
  { "LEFTWARDS HARPOON WITH BARB DOWNWARDS", 0x21bd },
  { "UPWARDS HARPOON WITH BARB RIGHTWARDS", 0x21be },
  { "UPWARDS HARPOON WITH BARB LEFTWARDS", 0x21bf },
  { "RIGHTWARDS HARPOON WITH BARB UPWARDS", 0x21c0 },
  { "RIGHTWARDS HARPOON WITH BARB DOWNWARDS", 0x21c1 },
  { "DOWNWARDS HARPOON WITH BARB RIGHTWARDS", 0x21c2 },
  { "DOWNWARDS HARPOON WITH BARB LEFTWARDS", 0x21c3 },
  { "RIGHTWARDS ARROW OVER LEFTWARDS ARROW", 0x21c4 },
  { "UPWARDS ARROW LEFTWARDS OF DOWNWARDS ARROW", 0x21c5 },
  { "LEFTWARDS ARROW OVER RIGHTWARDS ARROW", 0x21c6 },
  { "LEFTWARDS PAIRED ARROWS", 0x21c7 },
  { "UPWARDS PAIRED ARROWS", 0x21c8 },
  { "RIGHTWARDS PAIRED ARROWS", 0x21c9 },
  { "DOWNWARDS PAIRED ARROWS", 0x21ca },
  { "LEFTWARDS HARPOON OVER RIGHTWARDS HARPOON", 0x21cb },
  { "RIGHTWARDS HARPOON OVER LEFTWARDS HARPOON", 0x21cc },
  { "LEFTWARDS DOUBLE ARROW WITH STROKE", 0x21cd },
  { "LEFT RIGHT DOUBLE ARROW WITH STROKE", 0x21ce },
  { "RIGHTWARDS DOUBLE ARROW WITH STROKE", 0x21cf },
  { "LEFTWARDS DOUBLE ARROW", 0x21d0 },
  { "UPWARDS DOUBLE ARROW", 0x21d1 },
  { "RIGHTWARDS DOUBLE ARROW", 0x21d2 },
  { "DOWNWARDS DOUBLE ARROW", 0x21d3 },
  { "LEFT RIGHT DOUBLE ARROW", 0x21d4 },
  { "UP DOWN DOUBLE ARROW", 0x21d5 },
  { "NORTH WEST DOUBLE ARROW", 0x21d6 },
  { "NORTH EAST DOUBLE ARROW", 0x21d7 },
  { "SOUTH EAST DOUBLE ARROW", 0x21d8 },
  { "SOUTH WEST DOUBLE ARROW", 0x21d9 },
  { "LEFTWARDS TRIPLE ARROW", 0x21da },
  { "RIGHTWARDS TRIPLE ARROW", 0x21db },
  { "LEFTWARDS SQUIGGLE ARROW", 0x21dc },
  { "RIGHTWARDS SQUIGGLE ARROW", 0x21dd },
  { "UPWARDS ARROW WITH DOUBLE STROKE", 0x21de },
  { "DOWNWARDS ARROW WITH DOUBLE STROKE", 0x21df },
  { "LEFTWARDS DASHED ARROW", 0x21e0 },
  { "UPWARDS DASHED ARROW", 0x21e1 },
  { "RIGHTWARDS DASHED ARROW", 0x21e2 },
  { "DOWNWARDS DASHED ARROW", 0x21e3 },
  { "LEFTWARDS ARROW TO BAR", 0x21e4 },
  { "RIGHTWARDS ARROW TO BAR", 0x21e5 },
  { "LEFTWARDS WHITE ARROW", 0x21e6 },
  { "UPWARDS WHITE ARROW", 0x21e7 },
  { "RIGHTWARDS WHITE ARROW", 0x21e8 },
  { "DOWNWARDS WHITE ARROW", 0x21e9 },
  { "UPWARDS WHITE ARROW FROM BAR", 0x21ea },
  { "UPWARDS WHITE ARROW ON PEDESTAL", 0x21eb },
  { "UPWARDS WHITE ARROW ON PEDESTAL WITH HORIZONTAL BAR", 0x21ec },
  { "UPWARDS WHITE ARROW ON PEDESTAL WITH VERTICAL BAR", 0x21ed },
  { "UPWARDS WHITE DOUBLE ARROW", 0x21ee },
  { "UPWARDS WHITE DOUBLE ARROW ON PEDESTAL", 0x21ef },
  { "RIGHTWARDS WHITE ARROW FROM WALL", 0x21f0 },
  { "NORTH WEST ARROW TO CORNER", 0x21f1 },
  { "SOUTH EAST ARROW TO CORNER", 0x21f2 },
  { "UP DOWN WHITE ARROW", 0x21f3 },
  { "FOR ALL", 0x2200 },
  { "COMPLEMENT", 0x2201 },
  { "PARTIAL DIFFERENTIAL", 0x2202 },
  { "THERE EXISTS", 0x2203 },
  { "THERE DOES NOT EXIST", 0x2204 },
  { "EMPTY SET", 0x2205 },
  { "INCREMENT", 0x2206 },
  { "NABLA", 0x2207 },
  { "ELEMENT OF", 0x2208 },
  { "NOT AN ELEMENT OF", 0x2209 },
  { "SMALL ELEMENT OF", 0x220a },
  { "CONTAINS AS MEMBER", 0x220b },
  { "DOES NOT CONTAIN AS MEMBER", 0x220c },
  { "SMALL CONTAINS AS MEMBER", 0x220d },
  { "END OF PROOF", 0x220e },
  { "N-ARY PRODUCT", 0x220f },
  { "N-ARY COPRODUCT", 0x2210 },
  { "N-ARY SUMMATION", 0x2211 },
  { "MINUS SIGN", 0x2212 },
  { "MINUS-OR-PLUS SIGN", 0x2213 },
  { "DOT PLUS", 0x2214 },
  { "DIVISION SLASH", 0x2215 },
  { "SET MINUS", 0x2216 },
  { "ASTERISK OPERATOR", 0x2217 },
  { "RING OPERATOR", 0x2218 },
  { "BULLET OPERATOR", 0x2219 },
  { "SQUARE ROOT", 0x221a },
  { "CUBE ROOT", 0x221b },
  { "FOURTH ROOT", 0x221c },
  { "PROPORTIONAL TO", 0x221d },
  { "INFINITY", 0x221e },
  { "RIGHT ANGLE", 0x221f },
  { "ANGLE", 0x2220 },
  { "MEASURED ANGLE", 0x2221 },
  { "SPHERICAL ANGLE", 0x2222 },
  { "DIVIDES", 0x2223 },
  { "DOES NOT DIVIDE", 0x2224 },
  { "PARALLEL TO", 0x2225 },
  { "NOT PARALLEL TO", 0x2226 },
  { "LOGICAL AND", 0x2227 },
  { "LOGICAL OR", 0x2228 },
  { "INTERSECTION", 0x2229 },
  { "UNION", 0x222a },
  { "INTEGRAL", 0x222b },
  { "DOUBLE INTEGRAL", 0x222c },
  { "TRIPLE INTEGRAL", 0x222d },
  { "CONTOUR INTEGRAL", 0x222e },
  { "SURFACE INTEGRAL", 0x222f },
  { "VOLUME INTEGRAL", 0x2230 },
  { "CLOCKWISE INTEGRAL", 0x2231 },
  { "CLOCKWISE CONTOUR INTEGRAL", 0x2232 },
  { "ANTICLOCKWISE CONTOUR INTEGRAL", 0x2233 },
  { "THEREFORE", 0x2234 },
  { "BECAUSE", 0x2235 },
  { "RATIO", 0x2236 },
  { "PROPORTION", 0x2237 },
  { "DOT MINUS", 0x2238 },
  { "EXCESS", 0x2239 },
  { "GEOMETRIC PROPORTION", 0x223a },
  { "HOMOTHETIC", 0x223b },
  { "TILDE OPERATOR", 0x223c },
  { "REVERSED TILDE", 0x223d },
  { "INVERTED LAZY S", 0x223e },
  { "SINE WAVE", 0x223f },
  { "WREATH PRODUCT", 0x2240 },
  { "NOT TILDE", 0x2241 },
  { "MINUS TILDE", 0x2242 },
  { "ASYMPTOTICALLY EQUAL TO", 0x2243 },
  { "NOT ASYMPTOTICALLY EQUAL TO", 0x2244 },
  { "APPROXIMATELY EQUAL TO", 0x2245 },
  { "APPROXIMATELY BUT NOT ACTUALLY EQUAL TO", 0x2246 },
  { "NEITHER APPROXIMATELY NOR ACTUALLY EQUAL TO", 0x2247 },
  { "ALMOST EQUAL TO", 0x2248 },
  { "NOT ALMOST EQUAL TO", 0x2249 },
  { "ALMOST EQUAL OR EQUAL TO", 0x224a },
  { "TRIPLE TILDE", 0x224b },
  { "ALL EQUAL TO", 0x224c },
  { "EQUIVALENT TO", 0x224d },
  { "GEOMETRICALLY EQUIVALENT TO", 0x224e },
  { "DIFFERENCE BETWEEN", 0x224f },
  { "APPROACHES THE LIMIT", 0x2250 },
  { "GEOMETRICALLY EQUAL TO", 0x2251 },
  { "APPROXIMATELY EQUAL TO OR THE IMAGE OF", 0x2252 },
  { "IMAGE OF OR APPROXIMATELY EQUAL TO", 0x2253 },
  { "COLON EQUALS", 0x2254 },
  { "EQUALS COLON", 0x2255 },
  { "RING IN EQUAL TO", 0x2256 },
  { "RING EQUAL TO", 0x2257 },
  { "CORRESPONDS TO", 0x2258 },
  { "ESTIMATES", 0x2259 },
  { "EQUIANGULAR TO", 0x225a },
  { "STAR EQUALS", 0x225b },
  { "DELTA EQUAL TO", 0x225c },
  { "EQUAL TO BY DEFINITION", 0x225d },
  { "MEASURED BY", 0x225e },
  { "QUESTIONED EQUAL TO", 0x225f },
  { "NOT EQUAL TO", 0x2260 },
  { "IDENTICAL TO", 0x2261 },
  { "NOT IDENTICAL TO", 0x2262 },
  { "STRICTLY EQUIVALENT TO", 0x2263 },
  { "LESS-THAN OR EQUAL TO", 0x2264 },
  { "GREATER-THAN OR EQUAL TO", 0x2265 },
  { "LESS-THAN OVER EQUAL TO", 0x2266 },
  { "GREATER-THAN OVER EQUAL TO", 0x2267 },
  { "LESS-THAN BUT NOT EQUAL TO", 0x2268 },
  { "GREATER-THAN BUT NOT EQUAL TO", 0x2269 },
  { "MUCH LESS-THAN", 0x226a },
  { "MUCH GREATER-THAN", 0x226b },
  { "BETWEEN", 0x226c },
  { "NOT EQUIVALENT TO", 0x226d },
  { "NOT LESS-THAN", 0x226e },
  { "NOT GREATER-THAN", 0x226f },
  { "NEITHER LESS-THAN NOR EQUAL TO", 0x2270 },
  { "NEITHER GREATER-THAN NOR EQUAL TO", 0x2271 },
  { "LESS-THAN OR EQUIVALENT TO", 0x2272 },
  { "GREATER-THAN OR EQUIVALENT TO", 0x2273 },
  { "NEITHER LESS-THAN NOR EQUIVALENT TO", 0x2274 },
  { "NEITHER GREATER-THAN NOR EQUIVALENT TO", 0x2275 },
  { "LESS-THAN OR GREATER-THAN", 0x2276 },
  { "GREATER-THAN OR LESS-THAN", 0x2277 },
  { "NEITHER LESS-THAN NOR GREATER-THAN", 0x2278 },
  { "NEITHER GREATER-THAN NOR LESS-THAN", 0x2279 },
  { "PRECEDES", 0x227a },
  { "SUCCEEDS", 0x227b },
  { "PRECEDES OR EQUAL TO", 0x227c },
  { "SUCCEEDS OR EQUAL TO", 0x227d },
  { "PRECEDES OR EQUIVALENT TO", 0x227e },
  { "SUCCEEDS OR EQUIVALENT TO", 0x227f },
  { "DOES NOT PRECEDE", 0x2280 },
  { "DOES NOT SUCCEED", 0x2281 },
  { "SUBSET OF", 0x2282 },
  { "SUPERSET OF", 0x2283 },
  { "NOT A SUBSET OF", 0x2284 },
  { "NOT A SUPERSET OF", 0x2285 },
  { "SUBSET OF OR EQUAL TO", 0x2286 },
  { "SUPERSET OF OR EQUAL TO", 0x2287 },
  { "NEITHER A SUBSET OF NOR EQUAL TO", 0x2288 },
  { "NEITHER A SUPERSET OF NOR EQUAL TO", 0x2289 },
  { "SUBSET OF WITH NOT EQUAL TO", 0x228a },
  { "SUPERSET OF WITH NOT EQUAL TO", 0x228b },
  { "MULTISET", 0x228c },
  { "MULTISET MULTIPLICATION", 0x228d },
  { "MULTISET UNION", 0x228e },
  { "SQUARE IMAGE OF", 0x228f },
  { "SQUARE ORIGINAL OF", 0x2290 },
  { "SQUARE IMAGE OF OR EQUAL TO", 0x2291 },
  { "SQUARE ORIGINAL OF OR EQUAL TO", 0x2292 },
  { "SQUARE CAP", 0x2293 },
  { "SQUARE CUP", 0x2294 },
  { "CIRCLED PLUS", 0x2295 },
  { "CIRCLED MINUS", 0x2296 },
  { "CIRCLED TIMES", 0x2297 },
  { "CIRCLED DIVISION SLASH", 0x2298 },
  { "CIRCLED DOT OPERATOR", 0x2299 },
  { "CIRCLED RING OPERATOR", 0x229a },
  { "CIRCLED ASTERISK OPERATOR", 0x229b },
  { "CIRCLED EQUALS", 0x229c },
  { "CIRCLED DASH", 0x229d },
  { "SQUARED PLUS", 0x229e },
  { "SQUARED MINUS", 0x229f },
  { "SQUARED TIMES", 0x22a0 },
  { "SQUARED DOT OPERATOR", 0x22a1 },
  { "RIGHT TACK", 0x22a2 },
  { "LEFT TACK", 0x22a3 },
  { "DOWN TACK", 0x22a4 },
  { "UP TACK", 0x22a5 },
  { "ASSERTION", 0x22a6 },
  { "MODELS", 0x22a7 },
  { "TRUE", 0x22a8 },
  { "FORCES", 0x22a9 },
  { "TRIPLE VERTICAL BAR RIGHT TURNSTILE", 0x22aa },
  { "DOUBLE VERTICAL BAR DOUBLE RIGHT TURNSTILE", 0x22ab },
  { "DOES NOT PROVE", 0x22ac },
  { "NOT TRUE", 0x22ad },
  { "DOES NOT FORCE", 0x22ae },
  { "NEGATED DOUBLE VERTICAL BAR DOUBLE RIGHT TURNSTILE", 0x22af },
  { "PRECEDES UNDER RELATION", 0x22b0 },
  { "SUCCEEDS UNDER RELATION", 0x22b1 },
  { "NORMAL SUBGROUP OF", 0x22b2 },
  { "CONTAINS AS NORMAL SUBGROUP", 0x22b3 },
  { "NORMAL SUBGROUP OF OR EQUAL TO", 0x22b4 },
  { "CONTAINS AS NORMAL SUBGROUP OR EQUAL TO", 0x22b5 },
  { "ORIGINAL OF", 0x22b6 },
  { "IMAGE OF", 0x22b7 },
  { "MULTIMAP", 0x22b8 },
  { "HERMITIAN CONJUGATE MATRIX", 0x22b9 },
  { "INTERCALATE", 0x22ba },
  { "XOR", 0x22bb },
  { "NAND", 0x22bc },
  { "NOR", 0x22bd },
  { "RIGHT ANGLE WITH ARC", 0x22be },
  { "RIGHT TRIANGLE", 0x22bf },
  { "N-ARY LOGICAL AND", 0x22c0 },
  { "N-ARY LOGICAL OR", 0x22c1 },
  { "N-ARY INTERSECTION", 0x22c2 },
  { "N-ARY UNION", 0x22c3 },
  { "DIAMOND OPERATOR", 0x22c4 },
  { "DOT OPERATOR", 0x22c5 },
  { "STAR OPERATOR", 0x22c6 },
  { "DIVISION TIMES", 0x22c7 },
  { "BOWTIE", 0x22c8 },
  { "LEFT NORMAL FACTOR SEMIDIRECT PRODUCT", 0x22c9 },
  { "RIGHT NORMAL FACTOR SEMIDIRECT PRODUCT", 0x22ca },
  { "LEFT SEMIDIRECT PRODUCT", 0x22cb },
  { "RIGHT SEMIDIRECT PRODUCT", 0x22cc },
  { "REVERSED TILDE EQUALS", 0x22cd },
  { "CURLY LOGICAL OR", 0x22ce },
  { "CURLY LOGICAL AND", 0x22cf },
  { "DOUBLE SUBSET", 0x22d0 },
  { "DOUBLE SUPERSET", 0x22d1 },
  { "DOUBLE INTERSECTION", 0x22d2 },
  { "DOUBLE UNION", 0x22d3 },
  { "PITCHFORK", 0x22d4 },
  { "EQUAL AND PARALLEL TO", 0x22d5 },
  { "LESS-THAN WITH DOT", 0x22d6 },
  { "GREATER-THAN WITH DOT", 0x22d7 },
  { "VERY MUCH LESS-THAN", 0x22d8 },
  { "VERY MUCH GREATER-THAN", 0x22d9 },
  { "LESS-THAN EQUAL TO OR GREATER-THAN", 0x22da },
  { "GREATER-THAN EQUAL TO OR LESS-THAN", 0x22db },
  { "EQUAL TO OR LESS-THAN", 0x22dc },
  { "EQUAL TO OR GREATER-THAN", 0x22dd },
  { "EQUAL TO OR PRECEDES", 0x22de },
  { "EQUAL TO OR SUCCEEDS", 0x22df },
  { "DOES NOT PRECEDE OR EQUAL", 0x22e0 },
  { "DOES NOT SUCCEED OR EQUAL", 0x22e1 },
  { "NOT SQUARE IMAGE OF OR EQUAL TO", 0x22e2 },
  { "NOT SQUARE ORIGINAL OF OR EQUAL TO", 0x22e3 },
  { "SQUARE IMAGE OF OR NOT EQUAL TO", 0x22e4 },
  { "SQUARE ORIGINAL OF OR NOT EQUAL TO", 0x22e5 },
  { "LESS-THAN BUT NOT EQUIVALENT TO", 0x22e6 },
  { "GREATER-THAN BUT NOT EQUIVALENT TO", 0x22e7 },
  { "PRECEDES BUT NOT EQUIVALENT TO", 0x22e8 },
  { "SUCCEEDS BUT NOT EQUIVALENT TO", 0x22e9 },
  { "NOT NORMAL SUBGROUP OF", 0x22ea },
  { "DOES NOT CONTAIN AS NORMAL SUBGROUP", 0x22eb },
  { "NOT NORMAL SUBGROUP OF OR EQUAL TO", 0x22ec },
  { "DOES NOT CONTAIN AS NORMAL SUBGROUP OR EQUAL", 0x22ed },
  { "VERTICAL ELLIPSIS", 0x22ee },
  { "MIDLINE HORIZONTAL ELLIPSIS", 0x22ef },
  { "UP RIGHT DIAGONAL ELLIPSIS", 0x22f0 },
  { "DOWN RIGHT DIAGONAL ELLIPSIS", 0x22f1 },
  { "DIAMETER SIGN", 0x2300 },
  { "ELECTRIC ARROW", 0x2301 },
  { "HOUSE", 0x2302 },
  { "UP ARROWHEAD", 0x2303 },
  { "DOWN ARROWHEAD", 0x2304 },
  { "PROJECTIVE", 0x2305 },
  { "PERSPECTIVE", 0x2306 },
  { "WAVY LINE", 0x2307 },
  { "LEFT CEILING", 0x2308 },
  { "RIGHT CEILING", 0x2309 },
  { "LEFT FLOOR", 0x230a },
  { "RIGHT FLOOR", 0x230b },
  { "BOTTOM RIGHT CROP", 0x230c },
  { "BOTTOM LEFT CROP", 0x230d },
  { "TOP RIGHT CROP", 0x230e },
  { "TOP LEFT CROP", 0x230f },
  { "REVERSED NOT SIGN", 0x2310 },
  { "SQUARE LOZENGE", 0x2311 },
  { "ARC", 0x2312 },
  { "SEGMENT", 0x2313 },
  { "SECTOR", 0x2314 },
  { "TELEPHONE RECORDER", 0x2315 },
  { "POSITION INDICATOR", 0x2316 },
  { "VIEWDATA SQUARE", 0x2317 },
  { "PLACE OF INTEREST SIGN", 0x2318 },
  { "TURNED NOT SIGN", 0x2319 },
  { "WATCH", 0x231a },
  { "HOURGLASS", 0x231b },
  { "TOP LEFT CORNER", 0x231c },
  { "TOP RIGHT CORNER", 0x231d },
  { "BOTTOM LEFT CORNER", 0x231e },
  { "BOTTOM RIGHT CORNER", 0x231f },
  { "TOP HALF INTEGRAL", 0x2320 },
  { "BOTTOM HALF INTEGRAL", 0x2321 },
  { "FROWN", 0x2322 },
  { "SMILE", 0x2323 },
  { "UP ARROWHEAD BETWEEN TWO HORIZONTAL BARS", 0x2324 },
  { "OPTION KEY", 0x2325 },
  { "ERASE TO THE RIGHT", 0x2326 },
  { "X IN A RECTANGLE BOX", 0x2327 },
  { "KEYBOARD", 0x2328 },
  { "LEFT-POINTING ANGLE BRACKET", 0x2329 },
  { "RIGHT-POINTING ANGLE BRACKET", 0x232a },
  { "ERASE TO THE LEFT", 0x232b },
  { "BENZENE RING", 0x232c },
  { "CYLINDRICITY", 0x232d },
  { "ALL AROUND-PROFILE", 0x232e },
  { "SYMMETRY", 0x232f },
  { "TOTAL RUNOUT", 0x2330 },
  { "DIMENSION ORIGIN", 0x2331 },
  { "CONICAL TAPER", 0x2332 },
  { "SLOPE", 0x2333 },
  { "COUNTERBORE", 0x2334 },
  { "COUNTERSINK", 0x2335 },
  { "APL FUNCTIONAL SYMBOL I-BEAM", 0x2336 },
  { "APL FUNCTIONAL SYMBOL SQUISH QUAD", 0x2337 },
  { "APL FUNCTIONAL SYMBOL QUAD EQUAL", 0x2338 },
  { "APL FUNCTIONAL SYMBOL QUAD DIVIDE", 0x2339 },
  { "APL FUNCTIONAL SYMBOL QUAD DIAMOND", 0x233a },
  { "APL FUNCTIONAL SYMBOL QUAD JOT", 0x233b },
  { "APL FUNCTIONAL SYMBOL QUAD CIRCLE", 0x233c },
  { "APL FUNCTIONAL SYMBOL CIRCLE STILE", 0x233d },
  { "APL FUNCTIONAL SYMBOL CIRCLE JOT", 0x233e },
  { "APL FUNCTIONAL SYMBOL SLASH BAR", 0x233f },
  { "APL FUNCTIONAL SYMBOL BACKSLASH BAR", 0x2340 },
  { "APL FUNCTIONAL SYMBOL QUAD SLASH", 0x2341 },
  { "APL FUNCTIONAL SYMBOL QUAD BACKSLASH", 0x2342 },
  { "APL FUNCTIONAL SYMBOL QUAD LESS-THAN", 0x2343 },
  { "APL FUNCTIONAL SYMBOL QUAD GREATER-THAN", 0x2344 },
  { "APL FUNCTIONAL SYMBOL LEFTWARDS VANE", 0x2345 },
  { "APL FUNCTIONAL SYMBOL RIGHTWARDS VANE", 0x2346 },
  { "APL FUNCTIONAL SYMBOL QUAD LEFTWARDS ARROW", 0x2347 },
  { "APL FUNCTIONAL SYMBOL QUAD RIGHTWARDS ARROW", 0x2348 },
  { "APL FUNCTIONAL SYMBOL CIRCLE BACKSLASH", 0x2349 },
  { "APL FUNCTIONAL SYMBOL DOWN TACK UNDERBAR", 0x234a },
  { "APL FUNCTIONAL SYMBOL DELTA STILE", 0x234b },
  { "APL FUNCTIONAL SYMBOL QUAD DOWN CARET", 0x234c },
  { "APL FUNCTIONAL SYMBOL QUAD DELTA", 0x234d },
  { "APL FUNCTIONAL SYMBOL DOWN TACK JOT", 0x234e },
  { "APL FUNCTIONAL SYMBOL UPWARDS VANE", 0x234f },
  { "APL FUNCTIONAL SYMBOL QUAD UPWARDS ARROW", 0x2350 },
  { "APL FUNCTIONAL SYMBOL UP TACK OVERBAR", 0x2351 },
  { "APL FUNCTIONAL SYMBOL DEL STILE", 0x2352 },
  { "APL FUNCTIONAL SYMBOL QUAD UP CARET", 0x2353 },
  { "APL FUNCTIONAL SYMBOL QUAD DEL", 0x2354 },
  { "APL FUNCTIONAL SYMBOL UP TACK JOT", 0x2355 },
  { "APL FUNCTIONAL SYMBOL DOWNWARDS VANE", 0x2356 },
  { "APL FUNCTIONAL SYMBOL QUAD DOWNWARDS ARROW", 0x2357 },
  { "APL FUNCTIONAL SYMBOL QUOTE UNDERBAR", 0x2358 },
  { "APL FUNCTIONAL SYMBOL DELTA UNDERBAR", 0x2359 },
  { "APL FUNCTIONAL SYMBOL DIAMOND UNDERBAR", 0x235a },
  { "APL FUNCTIONAL SYMBOL JOT UNDERBAR", 0x235b },
  { "APL FUNCTIONAL SYMBOL CIRCLE UNDERBAR", 0x235c },
  { "APL FUNCTIONAL SYMBOL UP SHOE JOT", 0x235d },
  { "APL FUNCTIONAL SYMBOL QUOTE QUAD", 0x235e },
  { "APL FUNCTIONAL SYMBOL CIRCLE STAR", 0x235f },
  { "APL FUNCTIONAL SYMBOL QUAD COLON", 0x2360 },
  { "APL FUNCTIONAL SYMBOL UP TACK DIAERESIS", 0x2361 },
  { "APL FUNCTIONAL SYMBOL DEL DIAERESIS", 0x2362 },
  { "APL FUNCTIONAL SYMBOL STAR DIAERESIS", 0x2363 },
  { "APL FUNCTIONAL SYMBOL JOT DIAERESIS", 0x2364 },
  { "APL FUNCTIONAL SYMBOL CIRCLE DIAERESIS", 0x2365 },
  { "APL FUNCTIONAL SYMBOL DOWN SHOE STILE", 0x2366 },
  { "APL FUNCTIONAL SYMBOL LEFT SHOE STILE", 0x2367 },
  { "APL FUNCTIONAL SYMBOL TILDE DIAERESIS", 0x2368 },
  { "APL FUNCTIONAL SYMBOL GREATER-THAN DIAERESIS", 0x2369 },
  { "APL FUNCTIONAL SYMBOL COMMA BAR", 0x236a },
  { "APL FUNCTIONAL SYMBOL DEL TILDE", 0x236b },
  { "APL FUNCTIONAL SYMBOL ZILDE", 0x236c },
  { "APL FUNCTIONAL SYMBOL STILE TILDE", 0x236d },
  { "APL FUNCTIONAL SYMBOL SEMICOLON UNDERBAR", 0x236e },
  { "APL FUNCTIONAL SYMBOL QUAD NOT EQUAL", 0x236f },
  { "APL FUNCTIONAL SYMBOL QUAD QUESTION", 0x2370 },
  { "APL FUNCTIONAL SYMBOL DOWN CARET TILDE", 0x2371 },
  { "APL FUNCTIONAL SYMBOL UP CARET TILDE", 0x2372 },
  { "APL FUNCTIONAL SYMBOL IOTA", 0x2373 },
  { "APL FUNCTIONAL SYMBOL RHO", 0x2374 },
  { "APL FUNCTIONAL SYMBOL OMEGA", 0x2375 },
  { "APL FUNCTIONAL SYMBOL ALPHA UNDERBAR", 0x2376 },
  { "APL FUNCTIONAL SYMBOL EPSILON UNDERBAR", 0x2377 },
  { "APL FUNCTIONAL SYMBOL IOTA UNDERBAR", 0x2378 },
  { "APL FUNCTIONAL SYMBOL OMEGA UNDERBAR", 0x2379 },
  { "APL FUNCTIONAL SYMBOL ALPHA", 0x237a },
  { "NOT CHECK MARK", 0x237b },
  { "SHOULDERED OPEN BOX", 0x237d },
  { "BELL SYMBOL", 0x237e },
  { "VERTICAL LINE WITH MIDDLE DOT", 0x237f },
  { "INSERTION SYMBOL", 0x2380 },
  { "CONTINUOUS UNDERLINE SYMBOL", 0x2381 },
  { "DISCONTINUOUS UNDERLINE SYMBOL", 0x2382 },
  { "EMPHASIS SYMBOL", 0x2383 },
  { "COMPOSITION SYMBOL", 0x2384 },
  { "WHITE SQUARE WITH CENTRE VERTICAL LINE", 0x2385 },
  { "ENTER SYMBOL", 0x2386 },
  { "ALTERNATIVE KEY SYMBOL", 0x2387 },
  { "HELM SYMBOL", 0x2388 },
  { "CIRCLED HORIZONTAL BAR WITH NOTCH", 0x2389 },
  { "CIRCLED TRIANGLE DOWN", 0x238a },
  { "BROKEN CIRCLE WITH NORTHWEST ARROW", 0x238b },
  { "UNDO SYMBOL", 0x238c },
  { "MONOSTABLE SYMBOL", 0x238d },
  { "HYSTERESIS SYMBOL", 0x238e },
  { "OPEN-CIRCUIT-OUTPUT H-TYPE SYMBOL", 0x238f },
  { "OPEN-CIRCUIT-OUTPUT L-TYPE SYMBOL", 0x2390 },
  { "PASSIVE-PULL-DOWN-OUTPUT SYMBOL", 0x2391 },
  { "PASSIVE-PULL-UP-OUTPUT SYMBOL", 0x2392 },
  { "DIRECT CURRENT SYMBOL FORM TWO", 0x2393 },
  { "SOFTWARE-FUNCTION SYMBOL", 0x2394 },
  { "APL FUNCTIONAL SYMBOL QUAD", 0x2395 },
  { "DECIMAL SEPARATOR KEY SYMBOL", 0x2396 },
  { "PREVIOUS PAGE", 0x2397 },
  { "NEXT PAGE", 0x2398 },
  { "PRINT SCREEN SYMBOL", 0x2399 },
  { "CLEAR SCREEN SYMBOL", 0x239a },
  { "SYMBOL FOR NULL", 0x2400 },
  { "SYMBOL FOR START OF HEADING", 0x2401 },
  { "SYMBOL FOR START OF TEXT", 0x2402 },
  { "SYMBOL FOR END OF TEXT", 0x2403 },
  { "SYMBOL FOR END OF TRANSMISSION", 0x2404 },
  { "SYMBOL FOR ENQUIRY", 0x2405 },
  { "SYMBOL FOR ACKNOWLEDGE", 0x2406 },
  { "SYMBOL FOR BELL", 0x2407 },
  { "SYMBOL FOR BACKSPACE", 0x2408 },
  { "SYMBOL FOR HORIZONTAL TABULATION", 0x2409 },
  { "SYMBOL FOR LINE FEED", 0x240a },
  { "SYMBOL FOR VERTICAL TABULATION", 0x240b },
  { "SYMBOL FOR FORM FEED", 0x240c },
  { "SYMBOL FOR CARRIAGE RETURN", 0x240d },
  { "SYMBOL FOR SHIFT OUT", 0x240e },
  { "SYMBOL FOR SHIFT IN", 0x240f },
  { "SYMBOL FOR DATA LINK ESCAPE", 0x2410 },
  { "SYMBOL FOR DEVICE CONTROL ONE", 0x2411 },
  { "SYMBOL FOR DEVICE CONTROL TWO", 0x2412 },
  { "SYMBOL FOR DEVICE CONTROL THREE", 0x2413 },
  { "SYMBOL FOR DEVICE CONTROL FOUR", 0x2414 },
  { "SYMBOL FOR NEGATIVE ACKNOWLEDGE", 0x2415 },
  { "SYMBOL FOR SYNCHRONOUS IDLE", 0x2416 },
  { "SYMBOL FOR END OF TRANSMISSION BLOCK", 0x2417 },
  { "SYMBOL FOR CANCEL", 0x2418 },
  { "SYMBOL FOR END OF MEDIUM", 0x2419 },
  { "SYMBOL FOR SUBSTITUTE", 0x241a },
  { "SYMBOL FOR ESCAPE", 0x241b },
  { "SYMBOL FOR FILE SEPARATOR", 0x241c },
  { "SYMBOL FOR GROUP SEPARATOR", 0x241d },
  { "SYMBOL FOR RECORD SEPARATOR", 0x241e },
  { "SYMBOL FOR UNIT SEPARATOR", 0x241f },
  { "SYMBOL FOR SPACE", 0x2420 },
  { "SYMBOL FOR DELETE", 0x2421 },
  { "BLANK SYMBOL", 0x2422 },
  { "OPEN BOX", 0x2423 },
  { "SYMBOL FOR NEWLINE", 0x2424 },
  { "SYMBOL FOR DELETE FORM TWO", 0x2425 },
  { "SYMBOL FOR SUBSTITUTE FORM TWO", 0x2426 },
  { "OCR HOOK", 0x2440 },
  { "OCR CHAIR", 0x2441 },
  { "OCR FORK", 0x2442 },
  { "OCR INVERTED FORK", 0x2443 },
  { "OCR BELT BUCKLE", 0x2444 },
  { "OCR BOW TIE", 0x2445 },
  { "OCR BRANCH BANK IDENTIFICATION", 0x2446 },
  { "OCR AMOUNT OF CHECK", 0x2447 },
  { "OCR DASH", 0x2448 },
  { "OCR CUSTOMER ACCOUNT NUMBER", 0x2449 },
  { "OCR DOUBLE BACKSLASH", 0x244a },
  { "CIRCLED DIGIT ONE", 0x2460 },
  { "CIRCLED DIGIT TWO", 0x2461 },
  { "CIRCLED DIGIT THREE", 0x2462 },
  { "CIRCLED DIGIT FOUR", 0x2463 },
  { "CIRCLED DIGIT FIVE", 0x2464 },
  { "CIRCLED DIGIT SIX", 0x2465 },
  { "CIRCLED DIGIT SEVEN", 0x2466 },
  { "CIRCLED DIGIT EIGHT", 0x2467 },
  { "CIRCLED DIGIT NINE", 0x2468 },
  { "CIRCLED NUMBER TEN", 0x2469 },
  { "CIRCLED NUMBER ELEVEN", 0x246a },
  { "CIRCLED NUMBER TWELVE", 0x246b },
  { "CIRCLED NUMBER THIRTEEN", 0x246c },
  { "CIRCLED NUMBER FOURTEEN", 0x246d },
  { "CIRCLED NUMBER FIFTEEN", 0x246e },
  { "CIRCLED NUMBER SIXTEEN", 0x246f },
  { "CIRCLED NUMBER SEVENTEEN", 0x2470 },
  { "CIRCLED NUMBER EIGHTEEN", 0x2471 },
  { "CIRCLED NUMBER NINETEEN", 0x2472 },
  { "CIRCLED NUMBER TWENTY", 0x2473 },
  { "PARENTHESIZED DIGIT ONE", 0x2474 },
  { "PARENTHESIZED DIGIT TWO", 0x2475 },
  { "PARENTHESIZED DIGIT THREE", 0x2476 },
  { "PARENTHESIZED DIGIT FOUR", 0x2477 },
  { "PARENTHESIZED DIGIT FIVE", 0x2478 },
  { "PARENTHESIZED DIGIT SIX", 0x2479 },
  { "PARENTHESIZED DIGIT SEVEN", 0x247a },
  { "PARENTHESIZED DIGIT EIGHT", 0x247b },
  { "PARENTHESIZED DIGIT NINE", 0x247c },
  { "PARENTHESIZED NUMBER TEN", 0x247d },
  { "PARENTHESIZED NUMBER ELEVEN", 0x247e },
  { "PARENTHESIZED NUMBER TWELVE", 0x247f },
  { "PARENTHESIZED NUMBER THIRTEEN", 0x2480 },
  { "PARENTHESIZED NUMBER FOURTEEN", 0x2481 },
  { "PARENTHESIZED NUMBER FIFTEEN", 0x2482 },
  { "PARENTHESIZED NUMBER SIXTEEN", 0x2483 },
  { "PARENTHESIZED NUMBER SEVENTEEN", 0x2484 },
  { "PARENTHESIZED NUMBER EIGHTEEN", 0x2485 },
  { "PARENTHESIZED NUMBER NINETEEN", 0x2486 },
  { "PARENTHESIZED NUMBER TWENTY", 0x2487 },
  { "DIGIT ONE FULL STOP", 0x2488 },
  { "DIGIT TWO FULL STOP", 0x2489 },
  { "DIGIT THREE FULL STOP", 0x248a },
  { "DIGIT FOUR FULL STOP", 0x248b },
  { "DIGIT FIVE FULL STOP", 0x248c },
  { "DIGIT SIX FULL STOP", 0x248d },
  { "DIGIT SEVEN FULL STOP", 0x248e },
  { "DIGIT EIGHT FULL STOP", 0x248f },
  { "DIGIT NINE FULL STOP", 0x2490 },
  { "NUMBER TEN FULL STOP", 0x2491 },
  { "NUMBER ELEVEN FULL STOP", 0x2492 },
  { "NUMBER TWELVE FULL STOP", 0x2493 },
  { "NUMBER THIRTEEN FULL STOP", 0x2494 },
  { "NUMBER FOURTEEN FULL STOP", 0x2495 },
  { "NUMBER FIFTEEN FULL STOP", 0x2496 },
  { "NUMBER SIXTEEN FULL STOP", 0x2497 },
  { "NUMBER SEVENTEEN FULL STOP", 0x2498 },
  { "NUMBER EIGHTEEN FULL STOP", 0x2499 },
  { "NUMBER NINETEEN FULL STOP", 0x249a },
  { "NUMBER TWENTY FULL STOP", 0x249b },
  { "PARENTHESIZED LATIN SMALL LETTER A", 0x249c },
  { "PARENTHESIZED LATIN SMALL LETTER B", 0x249d },
  { "PARENTHESIZED LATIN SMALL LETTER C", 0x249e },
  { "PARENTHESIZED LATIN SMALL LETTER D", 0x249f },
  { "PARENTHESIZED LATIN SMALL LETTER E", 0x24a0 },
  { "PARENTHESIZED LATIN SMALL LETTER F", 0x24a1 },
  { "PARENTHESIZED LATIN SMALL LETTER G", 0x24a2 },
  { "PARENTHESIZED LATIN SMALL LETTER H", 0x24a3 },
  { "PARENTHESIZED LATIN SMALL LETTER I", 0x24a4 },
  { "PARENTHESIZED LATIN SMALL LETTER J", 0x24a5 },
  { "PARENTHESIZED LATIN SMALL LETTER K", 0x24a6 },
  { "PARENTHESIZED LATIN SMALL LETTER L", 0x24a7 },
  { "PARENTHESIZED LATIN SMALL LETTER M", 0x24a8 },
  { "PARENTHESIZED LATIN SMALL LETTER N", 0x24a9 },
  { "PARENTHESIZED LATIN SMALL LETTER O", 0x24aa },
  { "PARENTHESIZED LATIN SMALL LETTER P", 0x24ab },
  { "PARENTHESIZED LATIN SMALL LETTER Q", 0x24ac },
  { "PARENTHESIZED LATIN SMALL LETTER R", 0x24ad },
  { "PARENTHESIZED LATIN SMALL LETTER S", 0x24ae },
  { "PARENTHESIZED LATIN SMALL LETTER T", 0x24af },
  { "PARENTHESIZED LATIN SMALL LETTER U", 0x24b0 },
  { "PARENTHESIZED LATIN SMALL LETTER V", 0x24b1 },
  { "PARENTHESIZED LATIN SMALL LETTER W", 0x24b2 },
  { "PARENTHESIZED LATIN SMALL LETTER X", 0x24b3 },
  { "PARENTHESIZED LATIN SMALL LETTER Y", 0x24b4 },
  { "PARENTHESIZED LATIN SMALL LETTER Z", 0x24b5 },
  { "CIRCLED LATIN CAPITAL LETTER A", 0x24b6 },
  { "CIRCLED LATIN CAPITAL LETTER B", 0x24b7 },
  { "CIRCLED LATIN CAPITAL LETTER C", 0x24b8 },
  { "CIRCLED LATIN CAPITAL LETTER D", 0x24b9 },
  { "CIRCLED LATIN CAPITAL LETTER E", 0x24ba },
  { "CIRCLED LATIN CAPITAL LETTER F", 0x24bb },
  { "CIRCLED LATIN CAPITAL LETTER G", 0x24bc },
  { "CIRCLED LATIN CAPITAL LETTER H", 0x24bd },
  { "CIRCLED LATIN CAPITAL LETTER I", 0x24be },
  { "CIRCLED LATIN CAPITAL LETTER J", 0x24bf },
  { "CIRCLED LATIN CAPITAL LETTER K", 0x24c0 },
  { "CIRCLED LATIN CAPITAL LETTER L", 0x24c1 },
  { "CIRCLED LATIN CAPITAL LETTER M", 0x24c2 },
  { "CIRCLED LATIN CAPITAL LETTER N", 0x24c3 },
  { "CIRCLED LATIN CAPITAL LETTER O", 0x24c4 },
  { "CIRCLED LATIN CAPITAL LETTER P", 0x24c5 },
  { "CIRCLED LATIN CAPITAL LETTER Q", 0x24c6 },
  { "CIRCLED LATIN CAPITAL LETTER R", 0x24c7 },
  { "CIRCLED LATIN CAPITAL LETTER S", 0x24c8 },
  { "CIRCLED LATIN CAPITAL LETTER T", 0x24c9 },
  { "CIRCLED LATIN CAPITAL LETTER U", 0x24ca },
  { "CIRCLED LATIN CAPITAL LETTER V", 0x24cb },
  { "CIRCLED LATIN CAPITAL LETTER W", 0x24cc },
  { "CIRCLED LATIN CAPITAL LETTER X", 0x24cd },
  { "CIRCLED LATIN CAPITAL LETTER Y", 0x24ce },
  { "CIRCLED LATIN CAPITAL LETTER Z", 0x24cf },
  { "CIRCLED LATIN SMALL LETTER A", 0x24d0 },
  { "CIRCLED LATIN SMALL LETTER B", 0x24d1 },
  { "CIRCLED LATIN SMALL LETTER C", 0x24d2 },
  { "CIRCLED LATIN SMALL LETTER D", 0x24d3 },
  { "CIRCLED LATIN SMALL LETTER E", 0x24d4 },
  { "CIRCLED LATIN SMALL LETTER F", 0x24d5 },
  { "CIRCLED LATIN SMALL LETTER G", 0x24d6 },
  { "CIRCLED LATIN SMALL LETTER H", 0x24d7 },
  { "CIRCLED LATIN SMALL LETTER I", 0x24d8 },
  { "CIRCLED LATIN SMALL LETTER J", 0x24d9 },
  { "CIRCLED LATIN SMALL LETTER K", 0x24da },
  { "CIRCLED LATIN SMALL LETTER L", 0x24db },
  { "CIRCLED LATIN SMALL LETTER M", 0x24dc },
  { "CIRCLED LATIN SMALL LETTER N", 0x24dd },
  { "CIRCLED LATIN SMALL LETTER O", 0x24de },
  { "CIRCLED LATIN SMALL LETTER P", 0x24df },
  { "CIRCLED LATIN SMALL LETTER Q", 0x24e0 },
  { "CIRCLED LATIN SMALL LETTER R", 0x24e1 },
  { "CIRCLED LATIN SMALL LETTER S", 0x24e2 },
  { "CIRCLED LATIN SMALL LETTER T", 0x24e3 },
  { "CIRCLED LATIN SMALL LETTER U", 0x24e4 },
  { "CIRCLED LATIN SMALL LETTER V", 0x24e5 },
  { "CIRCLED LATIN SMALL LETTER W", 0x24e6 },
  { "CIRCLED LATIN SMALL LETTER X", 0x24e7 },
  { "CIRCLED LATIN SMALL LETTER Y", 0x24e8 },
  { "CIRCLED LATIN SMALL LETTER Z", 0x24e9 },
  { "CIRCLED DIGIT ZERO", 0x24ea },
  { "BOX DRAWINGS LIGHT HORIZONTAL", 0x2500 },
  { "BOX DRAWINGS HEAVY HORIZONTAL", 0x2501 },
  { "BOX DRAWINGS LIGHT VERTICAL", 0x2502 },
  { "BOX DRAWINGS HEAVY VERTICAL", 0x2503 },
  { "BOX DRAWINGS LIGHT TRIPLE DASH HORIZONTAL", 0x2504 },
  { "BOX DRAWINGS HEAVY TRIPLE DASH HORIZONTAL", 0x2505 },
  { "BOX DRAWINGS LIGHT TRIPLE DASH VERTICAL", 0x2506 },
  { "BOX DRAWINGS HEAVY TRIPLE DASH VERTICAL", 0x2507 },
  { "BOX DRAWINGS LIGHT QUADRUPLE DASH HORIZONTAL", 0x2508 },
  { "BOX DRAWINGS HEAVY QUADRUPLE DASH HORIZONTAL", 0x2509 },
  { "BOX DRAWINGS LIGHT QUADRUPLE DASH VERTICAL", 0x250a },
  { "BOX DRAWINGS HEAVY QUADRUPLE DASH VERTICAL", 0x250b },
  { "BOX DRAWINGS LIGHT DOWN AND RIGHT", 0x250c },
  { "BOX DRAWINGS DOWN LIGHT AND RIGHT HEAVY", 0x250d },
  { "BOX DRAWINGS DOWN HEAVY AND RIGHT LIGHT", 0x250e },
  { "BOX DRAWINGS HEAVY DOWN AND RIGHT", 0x250f },
  { "BOX DRAWINGS LIGHT DOWN AND LEFT", 0x2510 },
  { "BOX DRAWINGS DOWN LIGHT AND LEFT HEAVY", 0x2511 },
  { "BOX DRAWINGS DOWN HEAVY AND LEFT LIGHT", 0x2512 },
  { "BOX DRAWINGS HEAVY DOWN AND LEFT", 0x2513 },
  { "BOX DRAWINGS LIGHT UP AND RIGHT", 0x2514 },
  { "BOX DRAWINGS UP LIGHT AND RIGHT HEAVY", 0x2515 },
  { "BOX DRAWINGS UP HEAVY AND RIGHT LIGHT", 0x2516 },
  { "BOX DRAWINGS HEAVY UP AND RIGHT", 0x2517 },
  { "BOX DRAWINGS LIGHT UP AND LEFT", 0x2518 },
  { "BOX DRAWINGS UP LIGHT AND LEFT HEAVY", 0x2519 },
  { "BOX DRAWINGS UP HEAVY AND LEFT LIGHT", 0x251a },
  { "BOX DRAWINGS HEAVY UP AND LEFT", 0x251b },
  { "BOX DRAWINGS LIGHT VERTICAL AND RIGHT", 0x251c },
  { "BOX DRAWINGS VERTICAL LIGHT AND RIGHT HEAVY", 0x251d },
  { "BOX DRAWINGS UP HEAVY AND RIGHT DOWN LIGHT", 0x251e },
  { "BOX DRAWINGS DOWN HEAVY AND RIGHT UP LIGHT", 0x251f },
  { "BOX DRAWINGS VERTICAL HEAVY AND RIGHT LIGHT", 0x2520 },
  { "BOX DRAWINGS DOWN LIGHT AND RIGHT UP HEAVY", 0x2521 },
  { "BOX DRAWINGS UP LIGHT AND RIGHT DOWN HEAVY", 0x2522 },
  { "BOX DRAWINGS HEAVY VERTICAL AND RIGHT", 0x2523 },
  { "BOX DRAWINGS LIGHT VERTICAL AND LEFT", 0x2524 },
  { "BOX DRAWINGS VERTICAL LIGHT AND LEFT HEAVY", 0x2525 },
  { "BOX DRAWINGS UP HEAVY AND LEFT DOWN LIGHT", 0x2526 },
  { "BOX DRAWINGS DOWN HEAVY AND LEFT UP LIGHT", 0x2527 },
  { "BOX DRAWINGS VERTICAL HEAVY AND LEFT LIGHT", 0x2528 },
  { "BOX DRAWINGS DOWN LIGHT AND LEFT UP HEAVY", 0x2529 },
  { "BOX DRAWINGS UP LIGHT AND LEFT DOWN HEAVY", 0x252a },
  { "BOX DRAWINGS HEAVY VERTICAL AND LEFT", 0x252b },
  { "BOX DRAWINGS LIGHT DOWN AND HORIZONTAL", 0x252c },
  { "BOX DRAWINGS LEFT HEAVY AND RIGHT DOWN LIGHT", 0x252d },
  { "BOX DRAWINGS RIGHT HEAVY AND LEFT DOWN LIGHT", 0x252e },
  { "BOX DRAWINGS DOWN LIGHT AND HORIZONTAL HEAVY", 0x252f },
  { "BOX DRAWINGS DOWN HEAVY AND HORIZONTAL LIGHT", 0x2530 },
  { "BOX DRAWINGS RIGHT LIGHT AND LEFT DOWN HEAVY", 0x2531 },
  { "BOX DRAWINGS LEFT LIGHT AND RIGHT DOWN HEAVY", 0x2532 },
  { "BOX DRAWINGS HEAVY DOWN AND HORIZONTAL", 0x2533 },
  { "BOX DRAWINGS LIGHT UP AND HORIZONTAL", 0x2534 },
  { "BOX DRAWINGS LEFT HEAVY AND RIGHT UP LIGHT", 0x2535 },
  { "BOX DRAWINGS RIGHT HEAVY AND LEFT UP LIGHT", 0x2536 },
  { "BOX DRAWINGS UP LIGHT AND HORIZONTAL HEAVY", 0x2537 },
  { "BOX DRAWINGS UP HEAVY AND HORIZONTAL LIGHT", 0x2538 },
  { "BOX DRAWINGS RIGHT LIGHT AND LEFT UP HEAVY", 0x2539 },
  { "BOX DRAWINGS LEFT LIGHT AND RIGHT UP HEAVY", 0x253a },
  { "BOX DRAWINGS HEAVY UP AND HORIZONTAL", 0x253b },
  { "BOX DRAWINGS LIGHT VERTICAL AND HORIZONTAL", 0x253c },
  { "BOX DRAWINGS LEFT HEAVY AND RIGHT VERTICAL LIGHT", 0x253d },
  { "BOX DRAWINGS RIGHT HEAVY AND LEFT VERTICAL LIGHT", 0x253e },
  { "BOX DRAWINGS VERTICAL LIGHT AND HORIZONTAL HEAVY", 0x253f },
  { "BOX DRAWINGS UP HEAVY AND DOWN HORIZONTAL LIGHT", 0x2540 },
  { "BOX DRAWINGS DOWN HEAVY AND UP HORIZONTAL LIGHT", 0x2541 },
  { "BOX DRAWINGS VERTICAL HEAVY AND HORIZONTAL LIGHT", 0x2542 },
  { "BOX DRAWINGS LEFT UP HEAVY AND RIGHT DOWN LIGHT", 0x2543 },
  { "BOX DRAWINGS RIGHT UP HEAVY AND LEFT DOWN LIGHT", 0x2544 },
  { "BOX DRAWINGS LEFT DOWN HEAVY AND RIGHT UP LIGHT", 0x2545 },
  { "BOX DRAWINGS RIGHT DOWN HEAVY AND LEFT UP LIGHT", 0x2546 },
  { "BOX DRAWINGS DOWN LIGHT AND UP HORIZONTAL HEAVY", 0x2547 },
  { "BOX DRAWINGS UP LIGHT AND DOWN HORIZONTAL HEAVY", 0x2548 },
  { "BOX DRAWINGS RIGHT LIGHT AND LEFT VERTICAL HEAVY", 0x2549 },
  { "BOX DRAWINGS LEFT LIGHT AND RIGHT VERTICAL HEAVY", 0x254a },
  { "BOX DRAWINGS HEAVY VERTICAL AND HORIZONTAL", 0x254b },
  { "BOX DRAWINGS LIGHT DOUBLE DASH HORIZONTAL", 0x254c },
  { "BOX DRAWINGS HEAVY DOUBLE DASH HORIZONTAL", 0x254d },
  { "BOX DRAWINGS LIGHT DOUBLE DASH VERTICAL", 0x254e },
  { "BOX DRAWINGS HEAVY DOUBLE DASH VERTICAL", 0x254f },
  { "BOX DRAWINGS DOUBLE HORIZONTAL", 0x2550 },
  { "BOX DRAWINGS DOUBLE VERTICAL", 0x2551 },
  { "BOX DRAWINGS DOWN SINGLE AND RIGHT DOUBLE", 0x2552 },
  { "BOX DRAWINGS DOWN DOUBLE AND RIGHT SINGLE", 0x2553 },
  { "BOX DRAWINGS DOUBLE DOWN AND RIGHT", 0x2554 },
  { "BOX DRAWINGS DOWN SINGLE AND LEFT DOUBLE", 0x2555 },
  { "BOX DRAWINGS DOWN DOUBLE AND LEFT SINGLE", 0x2556 },
  { "BOX DRAWINGS DOUBLE DOWN AND LEFT", 0x2557 },
  { "BOX DRAWINGS UP SINGLE AND RIGHT DOUBLE", 0x2558 },
  { "BOX DRAWINGS UP DOUBLE AND RIGHT SINGLE", 0x2559 },
  { "BOX DRAWINGS DOUBLE UP AND RIGHT", 0x255a },
  { "BOX DRAWINGS UP SINGLE AND LEFT DOUBLE", 0x255b },
  { "BOX DRAWINGS UP DOUBLE AND LEFT SINGLE", 0x255c },
  { "BOX DRAWINGS DOUBLE UP AND LEFT", 0x255d },
  { "BOX DRAWINGS VERTICAL SINGLE AND RIGHT DOUBLE", 0x255e },
  { "BOX DRAWINGS VERTICAL DOUBLE AND RIGHT SINGLE", 0x255f },
  { "BOX DRAWINGS DOUBLE VERTICAL AND RIGHT", 0x2560 },
  { "BOX DRAWINGS VERTICAL SINGLE AND LEFT DOUBLE", 0x2561 },
  { "BOX DRAWINGS VERTICAL DOUBLE AND LEFT SINGLE", 0x2562 },
  { "BOX DRAWINGS DOUBLE VERTICAL AND LEFT", 0x2563 },
  { "BOX DRAWINGS DOWN SINGLE AND HORIZONTAL DOUBLE", 0x2564 },
  { "BOX DRAWINGS DOWN DOUBLE AND HORIZONTAL SINGLE", 0x2565 },
  { "BOX DRAWINGS DOUBLE DOWN AND HORIZONTAL", 0x2566 },
  { "BOX DRAWINGS UP SINGLE AND HORIZONTAL DOUBLE", 0x2567 },
  { "BOX DRAWINGS UP DOUBLE AND HORIZONTAL SINGLE", 0x2568 },
  { "BOX DRAWINGS DOUBLE UP AND HORIZONTAL", 0x2569 },
  { "BOX DRAWINGS VERTICAL SINGLE AND HORIZONTAL DOUBLE", 0x256a },
  { "BOX DRAWINGS VERTICAL DOUBLE AND HORIZONTAL SINGLE", 0x256b },
  { "BOX DRAWINGS DOUBLE VERTICAL AND HORIZONTAL", 0x256c },
  { "BOX DRAWINGS LIGHT ARC DOWN AND RIGHT", 0x256d },
  { "BOX DRAWINGS LIGHT ARC DOWN AND LEFT", 0x256e },
  { "BOX DRAWINGS LIGHT ARC UP AND LEFT", 0x256f },
  { "BOX DRAWINGS LIGHT ARC UP AND RIGHT", 0x2570 },
  { "BOX DRAWINGS LIGHT DIAGONAL UPPER RIGHT TO LOWER LEFT", 0x2571 },
  { "BOX DRAWINGS LIGHT DIAGONAL UPPER LEFT TO LOWER RIGHT", 0x2572 },
  { "BOX DRAWINGS LIGHT DIAGONAL CROSS", 0x2573 },
  { "BOX DRAWINGS LIGHT LEFT", 0x2574 },
  { "BOX DRAWINGS LIGHT UP", 0x2575 },
  { "BOX DRAWINGS LIGHT RIGHT", 0x2576 },
  { "BOX DRAWINGS LIGHT DOWN", 0x2577 },
  { "BOX DRAWINGS HEAVY LEFT", 0x2578 },
  { "BOX DRAWINGS HEAVY UP", 0x2579 },
  { "BOX DRAWINGS HEAVY RIGHT", 0x257a },
  { "BOX DRAWINGS HEAVY DOWN", 0x257b },
  { "BOX DRAWINGS LIGHT LEFT AND HEAVY RIGHT", 0x257c },
  { "BOX DRAWINGS LIGHT UP AND HEAVY DOWN", 0x257d },
  { "BOX DRAWINGS HEAVY LEFT AND LIGHT RIGHT", 0x257e },
  { "BOX DRAWINGS HEAVY UP AND LIGHT DOWN", 0x257f },
  { "UPPER HALF BLOCK", 0x2580 },
  { "LOWER ONE EIGHTH BLOCK", 0x2581 },
  { "LOWER ONE QUARTER BLOCK", 0x2582 },
  { "LOWER THREE EIGHTHS BLOCK", 0x2583 },
  { "LOWER HALF BLOCK", 0x2584 },
  { "LOWER FIVE EIGHTHS BLOCK", 0x2585 },
  { "LOWER THREE QUARTERS BLOCK", 0x2586 },
  { "LOWER SEVEN EIGHTHS BLOCK", 0x2587 },
  { "FULL BLOCK", 0x2588 },
  { "LEFT SEVEN EIGHTHS BLOCK", 0x2589 },
  { "LEFT THREE QUARTERS BLOCK", 0x258a },
  { "LEFT FIVE EIGHTHS BLOCK", 0x258b },
  { "LEFT HALF BLOCK", 0x258c },
  { "LEFT THREE EIGHTHS BLOCK", 0x258d },
  { "LEFT ONE QUARTER BLOCK", 0x258e },
  { "LEFT ONE EIGHTH BLOCK", 0x258f },
  { "RIGHT HALF BLOCK", 0x2590 },
  { "LIGHT SHADE", 0x2591 },
  { "MEDIUM SHADE", 0x2592 },
  { "DARK SHADE", 0x2593 },
  { "UPPER ONE EIGHTH BLOCK", 0x2594 },
  { "RIGHT ONE EIGHTH BLOCK", 0x2595 },
  { "BLACK SQUARE", 0x25a0 },
  { "WHITE SQUARE", 0x25a1 },
  { "WHITE SQUARE WITH ROUNDED CORNERS", 0x25a2 },
  { "WHITE SQUARE CONTAINING BLACK SMALL SQUARE", 0x25a3 },
  { "SQUARE WITH HORIZONTAL FILL", 0x25a4 },
  { "SQUARE WITH VERTICAL FILL", 0x25a5 },
  { "SQUARE WITH ORTHOGONAL CROSSHATCH FILL", 0x25a6 },
  { "SQUARE WITH UPPER LEFT TO LOWER RIGHT FILL", 0x25a7 },
  { "SQUARE WITH UPPER RIGHT TO LOWER LEFT FILL", 0x25a8 },
  { "SQUARE WITH DIAGONAL CROSSHATCH FILL", 0x25a9 },
  { "BLACK SMALL SQUARE", 0x25aa },
  { "WHITE SMALL SQUARE", 0x25ab },
  { "BLACK RECTANGLE", 0x25ac },
  { "WHITE RECTANGLE", 0x25ad },
  { "BLACK VERTICAL RECTANGLE", 0x25ae },
  { "WHITE VERTICAL RECTANGLE", 0x25af },
  { "BLACK PARALLELOGRAM", 0x25b0 },
  { "WHITE PARALLELOGRAM", 0x25b1 },
  { "BLACK UP-POINTING TRIANGLE", 0x25b2 },
  { "WHITE UP-POINTING TRIANGLE", 0x25b3 },
  { "BLACK UP-POINTING SMALL TRIANGLE", 0x25b4 },
  { "WHITE UP-POINTING SMALL TRIANGLE", 0x25b5 },
  { "BLACK RIGHT-POINTING TRIANGLE", 0x25b6 },
  { "WHITE RIGHT-POINTING TRIANGLE", 0x25b7 },
  { "BLACK RIGHT-POINTING SMALL TRIANGLE", 0x25b8 },
  { "WHITE RIGHT-POINTING SMALL TRIANGLE", 0x25b9 },
  { "BLACK RIGHT-POINTING POINTER", 0x25ba },
  { "WHITE RIGHT-POINTING POINTER", 0x25bb },
  { "BLACK DOWN-POINTING TRIANGLE", 0x25bc },
  { "WHITE DOWN-POINTING TRIANGLE", 0x25bd },
  { "BLACK DOWN-POINTING SMALL TRIANGLE", 0x25be },
  { "WHITE DOWN-POINTING SMALL TRIANGLE", 0x25bf },
  { "BLACK LEFT-POINTING TRIANGLE", 0x25c0 },
  { "WHITE LEFT-POINTING TRIANGLE", 0x25c1 },
  { "BLACK LEFT-POINTING SMALL TRIANGLE", 0x25c2 },
  { "WHITE LEFT-POINTING SMALL TRIANGLE", 0x25c3 },
  { "BLACK LEFT-POINTING POINTER", 0x25c4 },
  { "WHITE LEFT-POINTING POINTER", 0x25c5 },
  { "BLACK DIAMOND", 0x25c6 },
  { "WHITE DIAMOND", 0x25c7 },
  { "WHITE DIAMOND CONTAINING BLACK SMALL DIAMOND", 0x25c8 },
  { "FISHEYE", 0x25c9 },
  { "LOZENGE", 0x25ca },
  { "WHITE CIRCLE", 0x25cb },
  { "DOTTED CIRCLE", 0x25cc },
  { "CIRCLE WITH VERTICAL FILL", 0x25cd },
  { "BULLSEYE", 0x25ce },
  { "BLACK CIRCLE", 0x25cf },
  { "CIRCLE WITH LEFT HALF BLACK", 0x25d0 },
  { "CIRCLE WITH RIGHT HALF BLACK", 0x25d1 },
  { "CIRCLE WITH LOWER HALF BLACK", 0x25d2 },
  { "CIRCLE WITH UPPER HALF BLACK", 0x25d3 },
  { "CIRCLE WITH UPPER RIGHT QUADRANT BLACK", 0x25d4 },
  { "CIRCLE WITH ALL BUT UPPER LEFT QUADRANT BLACK", 0x25d5 },
  { "LEFT HALF BLACK CIRCLE", 0x25d6 },
  { "RIGHT HALF BLACK CIRCLE", 0x25d7 },
  { "INVERSE BULLET", 0x25d8 },
  { "INVERSE WHITE CIRCLE", 0x25d9 },
  { "UPPER HALF INVERSE WHITE CIRCLE", 0x25da },
  { "LOWER HALF INVERSE WHITE CIRCLE", 0x25db },
  { "UPPER LEFT QUADRANT CIRCULAR ARC", 0x25dc },
  { "UPPER RIGHT QUADRANT CIRCULAR ARC", 0x25dd },
  { "LOWER RIGHT QUADRANT CIRCULAR ARC", 0x25de },
  { "LOWER LEFT QUADRANT CIRCULAR ARC", 0x25df },
  { "UPPER HALF CIRCLE", 0x25e0 },
  { "LOWER HALF CIRCLE", 0x25e1 },
  { "BLACK LOWER RIGHT TRIANGLE", 0x25e2 },
  { "BLACK LOWER LEFT TRIANGLE", 0x25e3 },
  { "BLACK UPPER LEFT TRIANGLE", 0x25e4 },
  { "BLACK UPPER RIGHT TRIANGLE", 0x25e5 },
  { "WHITE BULLET", 0x25e6 },
  { "SQUARE WITH LEFT HALF BLACK", 0x25e7 },
  { "SQUARE WITH RIGHT HALF BLACK", 0x25e8 },
  { "SQUARE WITH UPPER LEFT DIAGONAL HALF BLACK", 0x25e9 },
  { "SQUARE WITH LOWER RIGHT DIAGONAL HALF BLACK", 0x25ea },
  { "WHITE SQUARE WITH VERTICAL BISECTING LINE", 0x25eb },
  { "WHITE UP-POINTING TRIANGLE WITH DOT", 0x25ec },
  { "UP-POINTING TRIANGLE WITH LEFT HALF BLACK", 0x25ed },
  { "UP-POINTING TRIANGLE WITH RIGHT HALF BLACK", 0x25ee },
  { "LARGE CIRCLE", 0x25ef },
  { "WHITE SQUARE WITH UPPER LEFT QUADRANT", 0x25f0 },
  { "WHITE SQUARE WITH LOWER LEFT QUADRANT", 0x25f1 },
  { "WHITE SQUARE WITH LOWER RIGHT QUADRANT", 0x25f2 },
  { "WHITE SQUARE WITH UPPER RIGHT QUADRANT", 0x25f3 },
  { "WHITE CIRCLE WITH UPPER LEFT QUADRANT", 0x25f4 },
  { "WHITE CIRCLE WITH LOWER LEFT QUADRANT", 0x25f5 },
  { "WHITE CIRCLE WITH LOWER RIGHT QUADRANT", 0x25f6 },
  { "WHITE CIRCLE WITH UPPER RIGHT QUADRANT", 0x25f7 },
  { "BLACK SUN WITH RAYS", 0x2600 },
  { "CLOUD", 0x2601 },
  { "UMBRELLA", 0x2602 },
  { "SNOWMAN", 0x2603 },
  { "COMET", 0x2604 },
  { "BLACK STAR", 0x2605 },
  { "WHITE STAR", 0x2606 },
  { "LIGHTNING", 0x2607 },
  { "THUNDERSTORM", 0x2608 },
  { "SUN", 0x2609 },
  { "ASCENDING NODE", 0x260a },
  { "DESCENDING NODE", 0x260b },
  { "CONJUNCTION", 0x260c },
  { "OPPOSITION", 0x260d },
  { "BLACK TELEPHONE", 0x260e },
  { "WHITE TELEPHONE", 0x260f },
  { "BALLOT BOX", 0x2610 },
  { "BALLOT BOX WITH CHECK", 0x2611 },
  { "BALLOT BOX WITH X", 0x2612 },
  { "SALTIRE", 0x2613 },
  { "REVERSED ROTATED FLORAL HEART BULLET", 0x2619 },
  { "BLACK LEFT POINTING INDEX", 0x261a },
  { "BLACK RIGHT POINTING INDEX", 0x261b },
  { "WHITE LEFT POINTING INDEX", 0x261c },
  { "WHITE UP POINTING INDEX", 0x261d },
  { "WHITE RIGHT POINTING INDEX", 0x261e },
  { "WHITE DOWN POINTING INDEX", 0x261f },
  { "SKULL AND CROSSBONES", 0x2620 },
  { "CAUTION SIGN", 0x2621 },
  { "RADIOACTIVE SIGN", 0x2622 },
  { "BIOHAZARD SIGN", 0x2623 },
  { "CADUCEUS", 0x2624 },
  { "ANKH", 0x2625 },
  { "ORTHODOX CROSS", 0x2626 },
  { "CHI RHO", 0x2627 },
  { "CROSS OF LORRAINE", 0x2628 },
  { "CROSS OF JERUSALEM", 0x2629 },
  { "STAR AND CRESCENT", 0x262a },
  { "FARSI SYMBOL", 0x262b },
  { "ADI SHAKTI", 0x262c },
  { "HAMMER AND SICKLE", 0x262d },
  { "PEACE SYMBOL", 0x262e },
  { "YIN YANG", 0x262f },
  { "TRIGRAM FOR HEAVEN", 0x2630 },
  { "TRIGRAM FOR LAKE", 0x2631 },
  { "TRIGRAM FOR FIRE", 0x2632 },
  { "TRIGRAM FOR THUNDER", 0x2633 },
  { "TRIGRAM FOR WIND", 0x2634 },
  { "TRIGRAM FOR WATER", 0x2635 },
  { "TRIGRAM FOR MOUNTAIN", 0x2636 },
  { "TRIGRAM FOR EARTH", 0x2637 },
  { "WHEEL OF DHARMA", 0x2638 },
  { "WHITE FROWNING FACE", 0x2639 },
  { "WHITE SMILING FACE", 0x263a },
  { "BLACK SMILING FACE", 0x263b },
  { "WHITE SUN WITH RAYS", 0x263c },
  { "FIRST QUARTER MOON", 0x263d },
  { "LAST QUARTER MOON", 0x263e },
  { "MERCURY", 0x263f },
  { "FEMALE SIGN", 0x2640 },
  { "EARTH", 0x2641 },
  { "MALE SIGN", 0x2642 },
  { "JUPITER", 0x2643 },
  { "SATURN", 0x2644 },
  { "URANUS", 0x2645 },
  { "NEPTUNE", 0x2646 },
  { "PLUTO", 0x2647 },
  { "ARIES", 0x2648 },
  { "TAURUS", 0x2649 },
  { "GEMINI", 0x264a },
  { "CANCER", 0x264b },
  { "LEO", 0x264c },
  { "VIRGO", 0x264d },
  { "LIBRA", 0x264e },
  { "SCORPIUS", 0x264f },
  { "SAGITTARIUS", 0x2650 },
  { "CAPRICORN", 0x2651 },
  { "AQUARIUS", 0x2652 },
  { "PISCES", 0x2653 },
  { "WHITE CHESS KING", 0x2654 },
  { "WHITE CHESS QUEEN", 0x2655 },
  { "WHITE CHESS ROOK", 0x2656 },
  { "WHITE CHESS BISHOP", 0x2657 },
  { "WHITE CHESS KNIGHT", 0x2658 },
  { "WHITE CHESS PAWN", 0x2659 },
  { "BLACK CHESS KING", 0x265a },
  { "BLACK CHESS QUEEN", 0x265b },
  { "BLACK CHESS ROOK", 0x265c },
  { "BLACK CHESS BISHOP", 0x265d },
  { "BLACK CHESS KNIGHT", 0x265e },
  { "BLACK CHESS PAWN", 0x265f },
  { "BLACK SPADE SUIT", 0x2660 },
  { "WHITE HEART SUIT", 0x2661 },
  { "WHITE DIAMOND SUIT", 0x2662 },
  { "BLACK CLUB SUIT", 0x2663 },
  { "WHITE SPADE SUIT", 0x2664 },
  { "BLACK HEART SUIT", 0x2665 },
  { "BLACK DIAMOND SUIT", 0x2666 },
  { "WHITE CLUB SUIT", 0x2667 },
  { "HOT SPRINGS", 0x2668 },
  { "QUARTER NOTE", 0x2669 },
  { "EIGHTH NOTE", 0x266a },
  { "BEAMED EIGHTH NOTES", 0x266b },
  { "BEAMED SIXTEENTH NOTES", 0x266c },
  { "MUSIC FLAT SIGN", 0x266d },
  { "MUSIC NATURAL SIGN", 0x266e },
  { "MUSIC SHARP SIGN", 0x266f },
  { "WEST SYRIAC CROSS", 0x2670 },
  { "EAST SYRIAC CROSS", 0x2671 },
  { "UPPER BLADE SCISSORS", 0x2701 },
  { "BLACK SCISSORS", 0x2702 },
  { "LOWER BLADE SCISSORS", 0x2703 },
  { "WHITE SCISSORS", 0x2704 },
  { "TELEPHONE LOCATION SIGN", 0x2706 },
  { "TAPE DRIVE", 0x2707 },
  { "AIRPLANE", 0x2708 },
  { "ENVELOPE", 0x2709 },
  { "VICTORY HAND", 0x270c },
  { "WRITING HAND", 0x270d },
  { "LOWER RIGHT PENCIL", 0x270e },
  { "PENCIL", 0x270f },
  { "UPPER RIGHT PENCIL", 0x2710 },
  { "WHITE NIB", 0x2711 },
  { "BLACK NIB", 0x2712 },
  { "CHECK MARK", 0x2713 },
  { "HEAVY CHECK MARK", 0x2714 },
  { "MULTIPLICATION X", 0x2715 },
  { "HEAVY MULTIPLICATION X", 0x2716 },
  { "BALLOT X", 0x2717 },
  { "HEAVY BALLOT X", 0x2718 },
  { "OUTLINED GREEK CROSS", 0x2719 },
  { "HEAVY GREEK CROSS", 0x271a },
  { "OPEN CENTRE CROSS", 0x271b },
  { "HEAVY OPEN CENTRE CROSS", 0x271c },
  { "LATIN CROSS", 0x271d },
  { "SHADOWED WHITE LATIN CROSS", 0x271e },
  { "OUTLINED LATIN CROSS", 0x271f },
  { "MALTESE CROSS", 0x2720 },
  { "STAR OF DAVID", 0x2721 },
  { "FOUR TEARDROP-SPOKED ASTERISK", 0x2722 },
  { "FOUR BALLOON-SPOKED ASTERISK", 0x2723 },
  { "HEAVY FOUR BALLOON-SPOKED ASTERISK", 0x2724 },
  { "FOUR CLUB-SPOKED ASTERISK", 0x2725 },
  { "BLACK FOUR POINTED STAR", 0x2726 },
  { "WHITE FOUR POINTED STAR", 0x2727 },
  { "STRESS OUTLINED WHITE STAR", 0x2729 },
  { "CIRCLED WHITE STAR", 0x272a },
  { "OPEN CENTRE BLACK STAR", 0x272b },
  { "BLACK CENTRE WHITE STAR", 0x272c },
  { "OUTLINED BLACK STAR", 0x272d },
  { "HEAVY OUTLINED BLACK STAR", 0x272e },
  { "PINWHEEL STAR", 0x272f },
  { "SHADOWED WHITE STAR", 0x2730 },
  { "HEAVY ASTERISK", 0x2731 },
  { "OPEN CENTRE ASTERISK", 0x2732 },
  { "EIGHT SPOKED ASTERISK", 0x2733 },
  { "EIGHT POINTED BLACK STAR", 0x2734 },
  { "EIGHT POINTED PINWHEEL STAR", 0x2735 },
  { "SIX POINTED BLACK STAR", 0x2736 },
  { "EIGHT POINTED RECTILINEAR BLACK STAR", 0x2737 },
  { "HEAVY EIGHT POINTED RECTILINEAR BLACK STAR", 0x2738 },
  { "TWELVE POINTED BLACK STAR", 0x2739 },
  { "SIXTEEN POINTED ASTERISK", 0x273a },
  { "TEARDROP-SPOKED ASTERISK", 0x273b },
  { "OPEN CENTRE TEARDROP-SPOKED ASTERISK", 0x273c },
  { "HEAVY TEARDROP-SPOKED ASTERISK", 0x273d },
  { "SIX PETALLED BLACK AND WHITE FLORETTE", 0x273e },
  { "BLACK FLORETTE", 0x273f },
  { "WHITE FLORETTE", 0x2740 },
  { "EIGHT PETALLED OUTLINED BLACK FLORETTE", 0x2741 },
  { "CIRCLED OPEN CENTRE EIGHT POINTED STAR", 0x2742 },
  { "HEAVY TEARDROP-SPOKED PINWHEEL ASTERISK", 0x2743 },
  { "SNOWFLAKE", 0x2744 },
  { "TIGHT TRIFOLIATE SNOWFLAKE", 0x2745 },
  { "HEAVY CHEVRON SNOWFLAKE", 0x2746 },
  { "SPARKLE", 0x2747 },
  { "HEAVY SPARKLE", 0x2748 },
  { "BALLOON-SPOKED ASTERISK", 0x2749 },
  { "EIGHT TEARDROP-SPOKED PROPELLER ASTERISK", 0x274a },
  { "HEAVY EIGHT TEARDROP-SPOKED PROPELLER ASTERISK", 0x274b },
  { "SHADOWED WHITE CIRCLE", 0x274d },
  { "LOWER RIGHT DROP-SHADOWED WHITE SQUARE", 0x274f },
  { "UPPER RIGHT DROP-SHADOWED WHITE SQUARE", 0x2750 },
  { "LOWER RIGHT SHADOWED WHITE SQUARE", 0x2751 },
  { "UPPER RIGHT SHADOWED WHITE SQUARE", 0x2752 },
  { "BLACK DIAMOND MINUS WHITE X", 0x2756 },
  { "LIGHT VERTICAL BAR", 0x2758 },
  { "MEDIUM VERTICAL BAR", 0x2759 },
  { "HEAVY VERTICAL BAR", 0x275a },
  { "HEAVY SINGLE TURNED COMMA QUOTATION MARK ORNAMENT", 0x275b },
  { "HEAVY SINGLE COMMA QUOTATION MARK ORNAMENT", 0x275c },
  { "HEAVY DOUBLE TURNED COMMA QUOTATION MARK ORNAMENT", 0x275d },
  { "HEAVY DOUBLE COMMA QUOTATION MARK ORNAMENT", 0x275e },
  { "CURVED STEM PARAGRAPH SIGN ORNAMENT", 0x2761 },
  { "HEAVY EXCLAMATION MARK ORNAMENT", 0x2762 },
  { "HEAVY HEART EXCLAMATION MARK ORNAMENT", 0x2763 },
  { "HEAVY BLACK HEART", 0x2764 },
  { "ROTATED HEAVY BLACK HEART BULLET", 0x2765 },
  { "FLORAL HEART", 0x2766 },
  { "ROTATED FLORAL HEART BULLET", 0x2767 },
  { "DINGBAT NEGATIVE CIRCLED DIGIT ONE", 0x2776 },
  { "DINGBAT NEGATIVE CIRCLED DIGIT TWO", 0x2777 },
  { "DINGBAT NEGATIVE CIRCLED DIGIT THREE", 0x2778 },
  { "DINGBAT NEGATIVE CIRCLED DIGIT FOUR", 0x2779 },
  { "DINGBAT NEGATIVE CIRCLED DIGIT FIVE", 0x277a },
  { "DINGBAT NEGATIVE CIRCLED DIGIT SIX", 0x277b },
  { "DINGBAT NEGATIVE CIRCLED DIGIT SEVEN", 0x277c },
  { "DINGBAT NEGATIVE CIRCLED DIGIT EIGHT", 0x277d },
  { "DINGBAT NEGATIVE CIRCLED DIGIT NINE", 0x277e },
  { "DINGBAT NEGATIVE CIRCLED NUMBER TEN", 0x277f },
  { "DINGBAT CIRCLED SANS-SERIF DIGIT ONE", 0x2780 },
  { "DINGBAT CIRCLED SANS-SERIF DIGIT TWO", 0x2781 },
  { "DINGBAT CIRCLED SANS-SERIF DIGIT THREE", 0x2782 },
  { "DINGBAT CIRCLED SANS-SERIF DIGIT FOUR", 0x2783 },
  { "DINGBAT CIRCLED SANS-SERIF DIGIT FIVE", 0x2784 },
  { "DINGBAT CIRCLED SANS-SERIF DIGIT SIX", 0x2785 },
  { "DINGBAT CIRCLED SANS-SERIF DIGIT SEVEN", 0x2786 },
  { "DINGBAT CIRCLED SANS-SERIF DIGIT EIGHT", 0x2787 },
  { "DINGBAT CIRCLED SANS-SERIF DIGIT NINE", 0x2788 },
  { "DINGBAT CIRCLED SANS-SERIF NUMBER TEN", 0x2789 },
  { "DINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT ONE", 0x278a },
  { "DINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT TWO", 0x278b },
  { "DINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT THREE", 0x278c },
  { "DINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT FOUR", 0x278d },
  { "DINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT FIVE", 0x278e },
  { "DINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT SIX", 0x278f },
  { "DINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT SEVEN", 0x2790 },
  { "DINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT EIGHT", 0x2791 },
  { "DINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT NINE", 0x2792 },
  { "DINGBAT NEGATIVE CIRCLED SANS-SERIF NUMBER TEN", 0x2793 },
  { "HEAVY WIDE-HEADED RIGHTWARDS ARROW", 0x2794 },
  { "HEAVY SOUTH EAST ARROW", 0x2798 },
  { "HEAVY RIGHTWARDS ARROW", 0x2799 },
  { "HEAVY NORTH EAST ARROW", 0x279a },
  { "DRAFTING POINT RIGHTWARDS ARROW", 0x279b },
  { "HEAVY ROUND-TIPPED RIGHTWARDS ARROW", 0x279c },
  { "TRIANGLE-HEADED RIGHTWARDS ARROW", 0x279d },
  { "HEAVY TRIANGLE-HEADED RIGHTWARDS ARROW", 0x279e },
  { "DASHED TRIANGLE-HEADED RIGHTWARDS ARROW", 0x279f },
  { "HEAVY DASHED TRIANGLE-HEADED RIGHTWARDS ARROW", 0x27a0 },
  { "BLACK RIGHTWARDS ARROW", 0x27a1 },
  { "THREE-D TOP-LIGHTED RIGHTWARDS ARROWHEAD", 0x27a2 },
  { "THREE-D BOTTOM-LIGHTED RIGHTWARDS ARROWHEAD", 0x27a3 },
  { "BLACK RIGHTWARDS ARROWHEAD", 0x27a4 },
  { "HEAVY BLACK CURVED DOWNWARDS AND RIGHTWARDS ARROW", 0x27a5 },
  { "HEAVY BLACK CURVED UPWARDS AND RIGHTWARDS ARROW", 0x27a6 },
  { "SQUAT BLACK RIGHTWARDS ARROW", 0x27a7 },
  { "HEAVY CONCAVE-POINTED BLACK RIGHTWARDS ARROW", 0x27a8 },
  { "RIGHT-SHADED WHITE RIGHTWARDS ARROW", 0x27a9 },
  { "LEFT-SHADED WHITE RIGHTWARDS ARROW", 0x27aa },
  { "BACK-TILTED SHADOWED WHITE RIGHTWARDS ARROW", 0x27ab },
  { "FRONT-TILTED SHADOWED WHITE RIGHTWARDS ARROW", 0x27ac },
  { "HEAVY LOWER RIGHT-SHADOWED WHITE RIGHTWARDS ARROW", 0x27ad },
  { "HEAVY UPPER RIGHT-SHADOWED WHITE RIGHTWARDS ARROW", 0x27ae },
  { "NOTCHED LOWER RIGHT-SHADOWED WHITE RIGHTWARDS ARROW", 0x27af },
  { "NOTCHED UPPER RIGHT-SHADOWED WHITE RIGHTWARDS ARROW", 0x27b1 },
  { "CIRCLED HEAVY WHITE RIGHTWARDS ARROW", 0x27b2 },
  { "WHITE-FEATHERED RIGHTWARDS ARROW", 0x27b3 },
  { "BLACK-FEATHERED SOUTH EAST ARROW", 0x27b4 },
  { "BLACK-FEATHERED RIGHTWARDS ARROW", 0x27b5 },
  { "BLACK-FEATHERED NORTH EAST ARROW", 0x27b6 },
  { "HEAVY BLACK-FEATHERED SOUTH EAST ARROW", 0x27b7 },
  { "HEAVY BLACK-FEATHERED RIGHTWARDS ARROW", 0x27b8 },
  { "HEAVY BLACK-FEATHERED NORTH EAST ARROW", 0x27b9 },
  { "TEARDROP-BARBED RIGHTWARDS ARROW", 0x27ba },
  { "HEAVY TEARDROP-SHANKED RIGHTWARDS ARROW", 0x27bb },
  { "WEDGE-TAILED RIGHTWARDS ARROW", 0x27bc },
  { "HEAVY WEDGE-TAILED RIGHTWARDS ARROW", 0x27bd },
  { "OPEN-OUTLINED RIGHTWARDS ARROW", 0x27be },
  { "BRAILLE PATTERN BLANK", 0x2800 },
  { "BRAILLE PATTERN DOTS-1", 0x2801 },
  { "BRAILLE PATTERN DOTS-2", 0x2802 },
  { "BRAILLE PATTERN DOTS-12", 0x2803 },
  { "BRAILLE PATTERN DOTS-3", 0x2804 },
  { "BRAILLE PATTERN DOTS-13", 0x2805 },
  { "BRAILLE PATTERN DOTS-23", 0x2806 },
  { "BRAILLE PATTERN DOTS-123", 0x2807 },
  { "BRAILLE PATTERN DOTS-4", 0x2808 },
  { "BRAILLE PATTERN DOTS-14", 0x2809 },
  { "BRAILLE PATTERN DOTS-24", 0x280a },
  { "BRAILLE PATTERN DOTS-124", 0x280b },
  { "BRAILLE PATTERN DOTS-34", 0x280c },
  { "BRAILLE PATTERN DOTS-134", 0x280d },
  { "BRAILLE PATTERN DOTS-234", 0x280e },
  { "BRAILLE PATTERN DOTS-1234", 0x280f },
  { "BRAILLE PATTERN DOTS-5", 0x2810 },
  { "BRAILLE PATTERN DOTS-15", 0x2811 },
  { "BRAILLE PATTERN DOTS-25", 0x2812 },
  { "BRAILLE PATTERN DOTS-125", 0x2813 },
  { "BRAILLE PATTERN DOTS-35", 0x2814 },
  { "BRAILLE PATTERN DOTS-135", 0x2815 },
  { "BRAILLE PATTERN DOTS-235", 0x2816 },
  { "BRAILLE PATTERN DOTS-1235", 0x2817 },
  { "BRAILLE PATTERN DOTS-45", 0x2818 },
  { "BRAILLE PATTERN DOTS-145", 0x2819 },
  { "BRAILLE PATTERN DOTS-245", 0x281a },
  { "BRAILLE PATTERN DOTS-1245", 0x281b },
  { "BRAILLE PATTERN DOTS-345", 0x281c },
  { "BRAILLE PATTERN DOTS-1345", 0x281d },
  { "BRAILLE PATTERN DOTS-2345", 0x281e },
  { "BRAILLE PATTERN DOTS-12345", 0x281f },
  { "BRAILLE PATTERN DOTS-6", 0x2820 },
  { "BRAILLE PATTERN DOTS-16", 0x2821 },
  { "BRAILLE PATTERN DOTS-26", 0x2822 },
  { "BRAILLE PATTERN DOTS-126", 0x2823 },
  { "BRAILLE PATTERN DOTS-36", 0x2824 },
  { "BRAILLE PATTERN DOTS-136", 0x2825 },
  { "BRAILLE PATTERN DOTS-236", 0x2826 },
  { "BRAILLE PATTERN DOTS-1236", 0x2827 },
  { "BRAILLE PATTERN DOTS-46", 0x2828 },
  { "BRAILLE PATTERN DOTS-146", 0x2829 },
  { "BRAILLE PATTERN DOTS-246", 0x282a },
  { "BRAILLE PATTERN DOTS-1246", 0x282b },
  { "BRAILLE PATTERN DOTS-346", 0x282c },
  { "BRAILLE PATTERN DOTS-1346", 0x282d },
  { "BRAILLE PATTERN DOTS-2346", 0x282e },
  { "BRAILLE PATTERN DOTS-12346", 0x282f },
  { "BRAILLE PATTERN DOTS-56", 0x2830 },
  { "BRAILLE PATTERN DOTS-156", 0x2831 },
  { "BRAILLE PATTERN DOTS-256", 0x2832 },
  { "BRAILLE PATTERN DOTS-1256", 0x2833 },
  { "BRAILLE PATTERN DOTS-356", 0x2834 },
  { "BRAILLE PATTERN DOTS-1356", 0x2835 },
  { "BRAILLE PATTERN DOTS-2356", 0x2836 },
  { "BRAILLE PATTERN DOTS-12356", 0x2837 },
  { "BRAILLE PATTERN DOTS-456", 0x2838 },
  { "BRAILLE PATTERN DOTS-1456", 0x2839 },
  { "BRAILLE PATTERN DOTS-2456", 0x283a },
  { "BRAILLE PATTERN DOTS-12456", 0x283b },
  { "BRAILLE PATTERN DOTS-3456", 0x283c },
  { "BRAILLE PATTERN DOTS-13456", 0x283d },
  { "BRAILLE PATTERN DOTS-23456", 0x283e },
  { "BRAILLE PATTERN DOTS-123456", 0x283f },
  { "BRAILLE PATTERN DOTS-7", 0x2840 },
  { "BRAILLE PATTERN DOTS-17", 0x2841 },
  { "BRAILLE PATTERN DOTS-27", 0x2842 },
  { "BRAILLE PATTERN DOTS-127", 0x2843 },
  { "BRAILLE PATTERN DOTS-37", 0x2844 },
  { "BRAILLE PATTERN DOTS-137", 0x2845 },
  { "BRAILLE PATTERN DOTS-237", 0x2846 },
  { "BRAILLE PATTERN DOTS-1237", 0x2847 },
  { "BRAILLE PATTERN DOTS-47", 0x2848 },
  { "BRAILLE PATTERN DOTS-147", 0x2849 },
  { "BRAILLE PATTERN DOTS-247", 0x284a },
  { "BRAILLE PATTERN DOTS-1247", 0x284b },
  { "BRAILLE PATTERN DOTS-347", 0x284c },
  { "BRAILLE PATTERN DOTS-1347", 0x284d },
  { "BRAILLE PATTERN DOTS-2347", 0x284e },
  { "BRAILLE PATTERN DOTS-12347", 0x284f },
  { "BRAILLE PATTERN DOTS-57", 0x2850 },
  { "BRAILLE PATTERN DOTS-157", 0x2851 },
  { "BRAILLE PATTERN DOTS-257", 0x2852 },
  { "BRAILLE PATTERN DOTS-1257", 0x2853 },
  { "BRAILLE PATTERN DOTS-357", 0x2854 },
  { "BRAILLE PATTERN DOTS-1357", 0x2855 },
  { "BRAILLE PATTERN DOTS-2357", 0x2856 },
  { "BRAILLE PATTERN DOTS-12357", 0x2857 },
  { "BRAILLE PATTERN DOTS-457", 0x2858 },
  { "BRAILLE PATTERN DOTS-1457", 0x2859 },
  { "BRAILLE PATTERN DOTS-2457", 0x285a },
  { "BRAILLE PATTERN DOTS-12457", 0x285b },
  { "BRAILLE PATTERN DOTS-3457", 0x285c },
  { "BRAILLE PATTERN DOTS-13457", 0x285d },
  { "BRAILLE PATTERN DOTS-23457", 0x285e },
  { "BRAILLE PATTERN DOTS-123457", 0x285f },
  { "BRAILLE PATTERN DOTS-67", 0x2860 },
  { "BRAILLE PATTERN DOTS-167", 0x2861 },
  { "BRAILLE PATTERN DOTS-267", 0x2862 },
  { "BRAILLE PATTERN DOTS-1267", 0x2863 },
  { "BRAILLE PATTERN DOTS-367", 0x2864 },
  { "BRAILLE PATTERN DOTS-1367", 0x2865 },
  { "BRAILLE PATTERN DOTS-2367", 0x2866 },
  { "BRAILLE PATTERN DOTS-12367", 0x2867 },
  { "BRAILLE PATTERN DOTS-467", 0x2868 },
  { "BRAILLE PATTERN DOTS-1467", 0x2869 },
  { "BRAILLE PATTERN DOTS-2467", 0x286a },
  { "BRAILLE PATTERN DOTS-12467", 0x286b },
  { "BRAILLE PATTERN DOTS-3467", 0x286c },
  { "BRAILLE PATTERN DOTS-13467", 0x286d },
  { "BRAILLE PATTERN DOTS-23467", 0x286e },
  { "BRAILLE PATTERN DOTS-123467", 0x286f },
  { "BRAILLE PATTERN DOTS-567", 0x2870 },
  { "BRAILLE PATTERN DOTS-1567", 0x2871 },
  { "BRAILLE PATTERN DOTS-2567", 0x2872 },
  { "BRAILLE PATTERN DOTS-12567", 0x2873 },
  { "BRAILLE PATTERN DOTS-3567", 0x2874 },
  { "BRAILLE PATTERN DOTS-13567", 0x2875 },
  { "BRAILLE PATTERN DOTS-23567", 0x2876 },
  { "BRAILLE PATTERN DOTS-123567", 0x2877 },
  { "BRAILLE PATTERN DOTS-4567", 0x2878 },
  { "BRAILLE PATTERN DOTS-14567", 0x2879 },
  { "BRAILLE PATTERN DOTS-24567", 0x287a },
  { "BRAILLE PATTERN DOTS-124567", 0x287b },
  { "BRAILLE PATTERN DOTS-34567", 0x287c },
  { "BRAILLE PATTERN DOTS-134567", 0x287d },
  { "BRAILLE PATTERN DOTS-234567", 0x287e },
  { "BRAILLE PATTERN DOTS-1234567", 0x287f },
  { "BRAILLE PATTERN DOTS-8", 0x2880 },
  { "BRAILLE PATTERN DOTS-18", 0x2881 },
  { "BRAILLE PATTERN DOTS-28", 0x2882 },
  { "BRAILLE PATTERN DOTS-128", 0x2883 },
  { "BRAILLE PATTERN DOTS-38", 0x2884 },
  { "BRAILLE PATTERN DOTS-138", 0x2885 },
  { "BRAILLE PATTERN DOTS-238", 0x2886 },
  { "BRAILLE PATTERN DOTS-1238", 0x2887 },
  { "BRAILLE PATTERN DOTS-48", 0x2888 },
  { "BRAILLE PATTERN DOTS-148", 0x2889 },
  { "BRAILLE PATTERN DOTS-248", 0x288a },
  { "BRAILLE PATTERN DOTS-1248", 0x288b },
  { "BRAILLE PATTERN DOTS-348", 0x288c },
  { "BRAILLE PATTERN DOTS-1348", 0x288d },
  { "BRAILLE PATTERN DOTS-2348", 0x288e },
  { "BRAILLE PATTERN DOTS-12348", 0x288f },
  { "BRAILLE PATTERN DOTS-58", 0x2890 },
  { "BRAILLE PATTERN DOTS-158", 0x2891 },
  { "BRAILLE PATTERN DOTS-258", 0x2892 },
  { "BRAILLE PATTERN DOTS-1258", 0x2893 },
  { "BRAILLE PATTERN DOTS-358", 0x2894 },
  { "BRAILLE PATTERN DOTS-1358", 0x2895 },
  { "BRAILLE PATTERN DOTS-2358", 0x2896 },
  { "BRAILLE PATTERN DOTS-12358", 0x2897 },
  { "BRAILLE PATTERN DOTS-458", 0x2898 },
  { "BRAILLE PATTERN DOTS-1458", 0x2899 },
  { "BRAILLE PATTERN DOTS-2458", 0x289a },
  { "BRAILLE PATTERN DOTS-12458", 0x289b },
  { "BRAILLE PATTERN DOTS-3458", 0x289c },
  { "BRAILLE PATTERN DOTS-13458", 0x289d },
  { "BRAILLE PATTERN DOTS-23458", 0x289e },
  { "BRAILLE PATTERN DOTS-123458", 0x289f },
  { "BRAILLE PATTERN DOTS-68", 0x28a0 },
  { "BRAILLE PATTERN DOTS-168", 0x28a1 },
  { "BRAILLE PATTERN DOTS-268", 0x28a2 },
  { "BRAILLE PATTERN DOTS-1268", 0x28a3 },
  { "BRAILLE PATTERN DOTS-368", 0x28a4 },
  { "BRAILLE PATTERN DOTS-1368", 0x28a5 },
  { "BRAILLE PATTERN DOTS-2368", 0x28a6 },
  { "BRAILLE PATTERN DOTS-12368", 0x28a7 },
  { "BRAILLE PATTERN DOTS-468", 0x28a8 },
  { "BRAILLE PATTERN DOTS-1468", 0x28a9 },
  { "BRAILLE PATTERN DOTS-2468", 0x28aa },
  { "BRAILLE PATTERN DOTS-12468", 0x28ab },
  { "BRAILLE PATTERN DOTS-3468", 0x28ac },
  { "BRAILLE PATTERN DOTS-13468", 0x28ad },
  { "BRAILLE PATTERN DOTS-23468", 0x28ae },
  { "BRAILLE PATTERN DOTS-123468", 0x28af },
  { "BRAILLE PATTERN DOTS-568", 0x28b0 },
  { "BRAILLE PATTERN DOTS-1568", 0x28b1 },
  { "BRAILLE PATTERN DOTS-2568", 0x28b2 },
  { "BRAILLE PATTERN DOTS-12568", 0x28b3 },
  { "BRAILLE PATTERN DOTS-3568", 0x28b4 },
  { "BRAILLE PATTERN DOTS-13568", 0x28b5 },
  { "BRAILLE PATTERN DOTS-23568", 0x28b6 },
  { "BRAILLE PATTERN DOTS-123568", 0x28b7 },
  { "BRAILLE PATTERN DOTS-4568", 0x28b8 },
  { "BRAILLE PATTERN DOTS-14568", 0x28b9 },
  { "BRAILLE PATTERN DOTS-24568", 0x28ba },
  { "BRAILLE PATTERN DOTS-124568", 0x28bb },
  { "BRAILLE PATTERN DOTS-34568", 0x28bc },
  { "BRAILLE PATTERN DOTS-134568", 0x28bd },
  { "BRAILLE PATTERN DOTS-234568", 0x28be },
  { "BRAILLE PATTERN DOTS-1234568", 0x28bf },
  { "BRAILLE PATTERN DOTS-78", 0x28c0 },
  { "BRAILLE PATTERN DOTS-178", 0x28c1 },
  { "BRAILLE PATTERN DOTS-278", 0x28c2 },
  { "BRAILLE PATTERN DOTS-1278", 0x28c3 },
  { "BRAILLE PATTERN DOTS-378", 0x28c4 },
  { "BRAILLE PATTERN DOTS-1378", 0x28c5 },
  { "BRAILLE PATTERN DOTS-2378", 0x28c6 },
  { "BRAILLE PATTERN DOTS-12378", 0x28c7 },
  { "BRAILLE PATTERN DOTS-478", 0x28c8 },
  { "BRAILLE PATTERN DOTS-1478", 0x28c9 },
  { "BRAILLE PATTERN DOTS-2478", 0x28ca },
  { "BRAILLE PATTERN DOTS-12478", 0x28cb },
  { "BRAILLE PATTERN DOTS-3478", 0x28cc },
  { "BRAILLE PATTERN DOTS-13478", 0x28cd },
  { "BRAILLE PATTERN DOTS-23478", 0x28ce },
  { "BRAILLE PATTERN DOTS-123478", 0x28cf },
  { "BRAILLE PATTERN DOTS-578", 0x28d0 },
  { "BRAILLE PATTERN DOTS-1578", 0x28d1 },
  { "BRAILLE PATTERN DOTS-2578", 0x28d2 },
  { "BRAILLE PATTERN DOTS-12578", 0x28d3 },
  { "BRAILLE PATTERN DOTS-3578", 0x28d4 },
  { "BRAILLE PATTERN DOTS-13578", 0x28d5 },
  { "BRAILLE PATTERN DOTS-23578", 0x28d6 },
  { "BRAILLE PATTERN DOTS-123578", 0x28d7 },
  { "BRAILLE PATTERN DOTS-4578", 0x28d8 },
  { "BRAILLE PATTERN DOTS-14578", 0x28d9 },
  { "BRAILLE PATTERN DOTS-24578", 0x28da },
  { "BRAILLE PATTERN DOTS-124578", 0x28db },
  { "BRAILLE PATTERN DOTS-34578", 0x28dc },
  { "BRAILLE PATTERN DOTS-134578", 0x28dd },
  { "BRAILLE PATTERN DOTS-234578", 0x28de },
  { "BRAILLE PATTERN DOTS-1234578", 0x28df },
  { "BRAILLE PATTERN DOTS-678", 0x28e0 },
  { "BRAILLE PATTERN DOTS-1678", 0x28e1 },
  { "BRAILLE PATTERN DOTS-2678", 0x28e2 },
  { "BRAILLE PATTERN DOTS-12678", 0x28e3 },
  { "BRAILLE PATTERN DOTS-3678", 0x28e4 },
  { "BRAILLE PATTERN DOTS-13678", 0x28e5 },
  { "BRAILLE PATTERN DOTS-23678", 0x28e6 },
  { "BRAILLE PATTERN DOTS-123678", 0x28e7 },
  { "BRAILLE PATTERN DOTS-4678", 0x28e8 },
  { "BRAILLE PATTERN DOTS-14678", 0x28e9 },
  { "BRAILLE PATTERN DOTS-24678", 0x28ea },
  { "BRAILLE PATTERN DOTS-124678", 0x28eb },
  { "BRAILLE PATTERN DOTS-34678", 0x28ec },
  { "BRAILLE PATTERN DOTS-134678", 0x28ed },
  { "BRAILLE PATTERN DOTS-234678", 0x28ee },
  { "BRAILLE PATTERN DOTS-1234678", 0x28ef },
  { "BRAILLE PATTERN DOTS-5678", 0x28f0 },
  { "BRAILLE PATTERN DOTS-15678", 0x28f1 },
  { "BRAILLE PATTERN DOTS-25678", 0x28f2 },
  { "BRAILLE PATTERN DOTS-125678", 0x28f3 },
  { "BRAILLE PATTERN DOTS-35678", 0x28f4 },
  { "BRAILLE PATTERN DOTS-135678", 0x28f5 },
  { "BRAILLE PATTERN DOTS-235678", 0x28f6 },
  { "BRAILLE PATTERN DOTS-1235678", 0x28f7 },
  { "BRAILLE PATTERN DOTS-45678", 0x28f8 },
  { "BRAILLE PATTERN DOTS-145678", 0x28f9 },
  { "BRAILLE PATTERN DOTS-245678", 0x28fa },
  { "BRAILLE PATTERN DOTS-1245678", 0x28fb },
  { "BRAILLE PATTERN DOTS-345678", 0x28fc },
  { "BRAILLE PATTERN DOTS-1345678", 0x28fd },
  { "BRAILLE PATTERN DOTS-2345678", 0x28fe },
  { "BRAILLE PATTERN DOTS-12345678", 0x28ff },
  { "CJK RADICAL REPEAT", 0x2e80 },
  { "CJK RADICAL CLIFF", 0x2e81 },
  { "CJK RADICAL SECOND ONE", 0x2e82 },
  { "CJK RADICAL SECOND TWO", 0x2e83 },
  { "CJK RADICAL SECOND THREE", 0x2e84 },
  { "CJK RADICAL PERSON", 0x2e85 },
  { "CJK RADICAL BOX", 0x2e86 },
  { "CJK RADICAL TABLE", 0x2e87 },
  { "CJK RADICAL KNIFE ONE", 0x2e88 },
  { "CJK RADICAL KNIFE TWO", 0x2e89 },
  { "CJK RADICAL DIVINATION", 0x2e8a },
  { "CJK RADICAL SEAL", 0x2e8b },
  { "CJK RADICAL SMALL ONE", 0x2e8c },
  { "CJK RADICAL SMALL TWO", 0x2e8d },
  { "CJK RADICAL LAME ONE", 0x2e8e },
  { "CJK RADICAL LAME TWO", 0x2e8f },
  { "CJK RADICAL LAME THREE", 0x2e90 },
  { "CJK RADICAL LAME FOUR", 0x2e91 },
  { "CJK RADICAL SNAKE", 0x2e92 },
  { "CJK RADICAL THREAD", 0x2e93 },
  { "CJK RADICAL SNOUT ONE", 0x2e94 },
  { "CJK RADICAL SNOUT TWO", 0x2e95 },
  { "CJK RADICAL HEART ONE", 0x2e96 },
  { "CJK RADICAL HEART TWO", 0x2e97 },
  { "CJK RADICAL HAND", 0x2e98 },
  { "CJK RADICAL RAP", 0x2e99 },
  { "CJK RADICAL CHOKE", 0x2e9b },
  { "CJK RADICAL SUN", 0x2e9c },
  { "CJK RADICAL MOON", 0x2e9d },
  { "CJK RADICAL DEATH", 0x2e9e },
  { "CJK RADICAL MOTHER", 0x2e9f },
  { "CJK RADICAL CIVILIAN", 0x2ea0 },
  { "CJK RADICAL WATER ONE", 0x2ea1 },
  { "CJK RADICAL WATER TWO", 0x2ea2 },
  { "CJK RADICAL FIRE", 0x2ea3 },
  { "CJK RADICAL PAW ONE", 0x2ea4 },
  { "CJK RADICAL PAW TWO", 0x2ea5 },
  { "CJK RADICAL SIMPLIFIED HALF TREE TRUNK", 0x2ea6 },
  { "CJK RADICAL COW", 0x2ea7 },
  { "CJK RADICAL DOG", 0x2ea8 },
  { "CJK RADICAL JADE", 0x2ea9 },
  { "CJK RADICAL BOLT OF CLOTH", 0x2eaa },
  { "CJK RADICAL EYE", 0x2eab },
  { "CJK RADICAL SPIRIT ONE", 0x2eac },
  { "CJK RADICAL SPIRIT TWO", 0x2ead },
  { "CJK RADICAL BAMBOO", 0x2eae },
  { "CJK RADICAL SILK", 0x2eaf },
  { "CJK RADICAL C-SIMPLIFIED SILK", 0x2eb0 },
  { "CJK RADICAL NET ONE", 0x2eb1 },
  { "CJK RADICAL NET TWO", 0x2eb2 },
  { "CJK RADICAL NET THREE", 0x2eb3 },
  { "CJK RADICAL NET FOUR", 0x2eb4 },
  { "CJK RADICAL MESH", 0x2eb5 },
  { "CJK RADICAL SHEEP", 0x2eb6 },
  { "CJK RADICAL RAM", 0x2eb7 },
  { "CJK RADICAL EWE", 0x2eb8 },
  { "CJK RADICAL OLD", 0x2eb9 },
  { "CJK RADICAL BRUSH ONE", 0x2eba },
  { "CJK RADICAL BRUSH TWO", 0x2ebb },
  { "CJK RADICAL MEAT", 0x2ebc },
  { "CJK RADICAL MORTAR", 0x2ebd },
  { "CJK RADICAL GRASS ONE", 0x2ebe },
  { "CJK RADICAL GRASS TWO", 0x2ebf },
  { "CJK RADICAL GRASS THREE", 0x2ec0 },
  { "CJK RADICAL TIGER", 0x2ec1 },
  { "CJK RADICAL CLOTHES", 0x2ec2 },
  { "CJK RADICAL WEST ONE", 0x2ec3 },
  { "CJK RADICAL WEST TWO", 0x2ec4 },
  { "CJK RADICAL C-SIMPLIFIED SEE", 0x2ec5 },
  { "CJK RADICAL SIMPLIFIED HORN", 0x2ec6 },
  { "CJK RADICAL HORN", 0x2ec7 },
  { "CJK RADICAL C-SIMPLIFIED SPEECH", 0x2ec8 },
  { "CJK RADICAL C-SIMPLIFIED SHELL", 0x2ec9 },
  { "CJK RADICAL FOOT", 0x2eca },
  { "CJK RADICAL C-SIMPLIFIED CART", 0x2ecb },
  { "CJK RADICAL SIMPLIFIED WALK", 0x2ecc },
  { "CJK RADICAL WALK ONE", 0x2ecd },
  { "CJK RADICAL WALK TWO", 0x2ece },
  { "CJK RADICAL CITY", 0x2ecf },
  { "CJK RADICAL C-SIMPLIFIED GOLD", 0x2ed0 },
  { "CJK RADICAL LONG ONE", 0x2ed1 },
  { "CJK RADICAL LONG TWO", 0x2ed2 },
  { "CJK RADICAL C-SIMPLIFIED LONG", 0x2ed3 },
  { "CJK RADICAL C-SIMPLIFIED GATE", 0x2ed4 },
  { "CJK RADICAL MOUND ONE", 0x2ed5 },
  { "CJK RADICAL MOUND TWO", 0x2ed6 },
  { "CJK RADICAL RAIN", 0x2ed7 },
  { "CJK RADICAL BLUE", 0x2ed8 },
  { "CJK RADICAL C-SIMPLIFIED TANNED LEATHER", 0x2ed9 },
  { "CJK RADICAL C-SIMPLIFIED LEAF", 0x2eda },
  { "CJK RADICAL C-SIMPLIFIED WIND", 0x2edb },
  { "CJK RADICAL C-SIMPLIFIED FLY", 0x2edc },
  { "CJK RADICAL EAT ONE", 0x2edd },
  { "CJK RADICAL EAT TWO", 0x2ede },
  { "CJK RADICAL EAT THREE", 0x2edf },
  { "CJK RADICAL C-SIMPLIFIED EAT", 0x2ee0 },
  { "CJK RADICAL HEAD", 0x2ee1 },
  { "CJK RADICAL C-SIMPLIFIED HORSE", 0x2ee2 },
  { "CJK RADICAL BONE", 0x2ee3 },
  { "CJK RADICAL GHOST", 0x2ee4 },
  { "CJK RADICAL C-SIMPLIFIED FISH", 0x2ee5 },
  { "CJK RADICAL C-SIMPLIFIED BIRD", 0x2ee6 },
  { "CJK RADICAL C-SIMPLIFIED SALT", 0x2ee7 },
  { "CJK RADICAL SIMPLIFIED WHEAT", 0x2ee8 },
  { "CJK RADICAL SIMPLIFIED YELLOW", 0x2ee9 },
  { "CJK RADICAL C-SIMPLIFIED FROG", 0x2eea },
  { "CJK RADICAL J-SIMPLIFIED EVEN", 0x2eeb },
  { "CJK RADICAL C-SIMPLIFIED EVEN", 0x2eec },
  { "CJK RADICAL J-SIMPLIFIED TOOTH", 0x2eed },
  { "CJK RADICAL C-SIMPLIFIED TOOTH", 0x2eee },
  { "CJK RADICAL J-SIMPLIFIED DRAGON", 0x2eef },
  { "CJK RADICAL C-SIMPLIFIED DRAGON", 0x2ef0 },
  { "CJK RADICAL TURTLE", 0x2ef1 },
  { "CJK RADICAL J-SIMPLIFIED TURTLE", 0x2ef2 },
  { "CJK RADICAL C-SIMPLIFIED TURTLE", 0x2ef3 },
  { "KANGXI RADICAL ONE", 0x2f00 },
  { "KANGXI RADICAL LINE", 0x2f01 },
  { "KANGXI RADICAL DOT", 0x2f02 },
  { "KANGXI RADICAL SLASH", 0x2f03 },
  { "KANGXI RADICAL SECOND", 0x2f04 },
  { "KANGXI RADICAL HOOK", 0x2f05 },
  { "KANGXI RADICAL TWO", 0x2f06 },
  { "KANGXI RADICAL LID", 0x2f07 },
  { "KANGXI RADICAL MAN", 0x2f08 },
  { "KANGXI RADICAL LEGS", 0x2f09 },
  { "KANGXI RADICAL ENTER", 0x2f0a },
  { "KANGXI RADICAL EIGHT", 0x2f0b },
  { "KANGXI RADICAL DOWN BOX", 0x2f0c },
  { "KANGXI RADICAL COVER", 0x2f0d },
  { "KANGXI RADICAL ICE", 0x2f0e },
  { "KANGXI RADICAL TABLE", 0x2f0f },
  { "KANGXI RADICAL OPEN BOX", 0x2f10 },
  { "KANGXI RADICAL KNIFE", 0x2f11 },
  { "KANGXI RADICAL POWER", 0x2f12 },
  { "KANGXI RADICAL WRAP", 0x2f13 },
  { "KANGXI RADICAL SPOON", 0x2f14 },
  { "KANGXI RADICAL RIGHT OPEN BOX", 0x2f15 },
  { "KANGXI RADICAL HIDING ENCLOSURE", 0x2f16 },
  { "KANGXI RADICAL TEN", 0x2f17 },
  { "KANGXI RADICAL DIVINATION", 0x2f18 },
  { "KANGXI RADICAL SEAL", 0x2f19 },
  { "KANGXI RADICAL CLIFF", 0x2f1a },
  { "KANGXI RADICAL PRIVATE", 0x2f1b },
  { "KANGXI RADICAL AGAIN", 0x2f1c },
  { "KANGXI RADICAL MOUTH", 0x2f1d },
  { "KANGXI RADICAL ENCLOSURE", 0x2f1e },
  { "KANGXI RADICAL EARTH", 0x2f1f },
  { "KANGXI RADICAL SCHOLAR", 0x2f20 },
  { "KANGXI RADICAL GO", 0x2f21 },
  { "KANGXI RADICAL GO SLOWLY", 0x2f22 },
  { "KANGXI RADICAL EVENING", 0x2f23 },
  { "KANGXI RADICAL BIG", 0x2f24 },
  { "KANGXI RADICAL WOMAN", 0x2f25 },
  { "KANGXI RADICAL CHILD", 0x2f26 },
  { "KANGXI RADICAL ROOF", 0x2f27 },
  { "KANGXI RADICAL INCH", 0x2f28 },
  { "KANGXI RADICAL SMALL", 0x2f29 },
  { "KANGXI RADICAL LAME", 0x2f2a },
  { "KANGXI RADICAL CORPSE", 0x2f2b },
  { "KANGXI RADICAL SPROUT", 0x2f2c },
  { "KANGXI RADICAL MOUNTAIN", 0x2f2d },
  { "KANGXI RADICAL RIVER", 0x2f2e },
  { "KANGXI RADICAL WORK", 0x2f2f },
  { "KANGXI RADICAL ONESELF", 0x2f30 },
  { "KANGXI RADICAL TURBAN", 0x2f31 },
  { "KANGXI RADICAL DRY", 0x2f32 },
  { "KANGXI RADICAL SHORT THREAD", 0x2f33 },
  { "KANGXI RADICAL DOTTED CLIFF", 0x2f34 },
  { "KANGXI RADICAL LONG STRIDE", 0x2f35 },
  { "KANGXI RADICAL TWO HANDS", 0x2f36 },
  { "KANGXI RADICAL SHOOT", 0x2f37 },
  { "KANGXI RADICAL BOW", 0x2f38 },
  { "KANGXI RADICAL SNOUT", 0x2f39 },
  { "KANGXI RADICAL BRISTLE", 0x2f3a },
  { "KANGXI RADICAL STEP", 0x2f3b },
  { "KANGXI RADICAL HEART", 0x2f3c },
  { "KANGXI RADICAL HALBERD", 0x2f3d },
  { "KANGXI RADICAL DOOR", 0x2f3e },
  { "KANGXI RADICAL HAND", 0x2f3f },
  { "KANGXI RADICAL BRANCH", 0x2f40 },
  { "KANGXI RADICAL RAP", 0x2f41 },
  { "KANGXI RADICAL SCRIPT", 0x2f42 },
  { "KANGXI RADICAL DIPPER", 0x2f43 },
  { "KANGXI RADICAL AXE", 0x2f44 },
  { "KANGXI RADICAL SQUARE", 0x2f45 },
  { "KANGXI RADICAL NOT", 0x2f46 },
  { "KANGXI RADICAL SUN", 0x2f47 },
  { "KANGXI RADICAL SAY", 0x2f48 },
  { "KANGXI RADICAL MOON", 0x2f49 },
  { "KANGXI RADICAL TREE", 0x2f4a },
  { "KANGXI RADICAL LACK", 0x2f4b },
  { "KANGXI RADICAL STOP", 0x2f4c },
  { "KANGXI RADICAL DEATH", 0x2f4d },
  { "KANGXI RADICAL WEAPON", 0x2f4e },
  { "KANGXI RADICAL DO NOT", 0x2f4f },
  { "KANGXI RADICAL COMPARE", 0x2f50 },
  { "KANGXI RADICAL FUR", 0x2f51 },
  { "KANGXI RADICAL CLAN", 0x2f52 },
  { "KANGXI RADICAL STEAM", 0x2f53 },
  { "KANGXI RADICAL WATER", 0x2f54 },
  { "KANGXI RADICAL FIRE", 0x2f55 },
  { "KANGXI RADICAL CLAW", 0x2f56 },
  { "KANGXI RADICAL FATHER", 0x2f57 },
  { "KANGXI RADICAL DOUBLE X", 0x2f58 },
  { "KANGXI RADICAL HALF TREE TRUNK", 0x2f59 },
  { "KANGXI RADICAL SLICE", 0x2f5a },
  { "KANGXI RADICAL FANG", 0x2f5b },
  { "KANGXI RADICAL COW", 0x2f5c },
  { "KANGXI RADICAL DOG", 0x2f5d },
  { "KANGXI RADICAL PROFOUND", 0x2f5e },
  { "KANGXI RADICAL JADE", 0x2f5f },
  { "KANGXI RADICAL MELON", 0x2f60 },
  { "KANGXI RADICAL TILE", 0x2f61 },
  { "KANGXI RADICAL SWEET", 0x2f62 },
  { "KANGXI RADICAL LIFE", 0x2f63 },
  { "KANGXI RADICAL USE", 0x2f64 },
  { "KANGXI RADICAL FIELD", 0x2f65 },
  { "KANGXI RADICAL BOLT OF CLOTH", 0x2f66 },
  { "KANGXI RADICAL SICKNESS", 0x2f67 },
  { "KANGXI RADICAL DOTTED TENT", 0x2f68 },
  { "KANGXI RADICAL WHITE", 0x2f69 },
  { "KANGXI RADICAL SKIN", 0x2f6a },
  { "KANGXI RADICAL DISH", 0x2f6b },
  { "KANGXI RADICAL EYE", 0x2f6c },
  { "KANGXI RADICAL SPEAR", 0x2f6d },
  { "KANGXI RADICAL ARROW", 0x2f6e },
  { "KANGXI RADICAL STONE", 0x2f6f },
  { "KANGXI RADICAL SPIRIT", 0x2f70 },
  { "KANGXI RADICAL TRACK", 0x2f71 },
  { "KANGXI RADICAL GRAIN", 0x2f72 },
  { "KANGXI RADICAL CAVE", 0x2f73 },
  { "KANGXI RADICAL STAND", 0x2f74 },
  { "KANGXI RADICAL BAMBOO", 0x2f75 },
  { "KANGXI RADICAL RICE", 0x2f76 },
  { "KANGXI RADICAL SILK", 0x2f77 },
  { "KANGXI RADICAL JAR", 0x2f78 },
  { "KANGXI RADICAL NET", 0x2f79 },
  { "KANGXI RADICAL SHEEP", 0x2f7a },
  { "KANGXI RADICAL FEATHER", 0x2f7b },
  { "KANGXI RADICAL OLD", 0x2f7c },
  { "KANGXI RADICAL AND", 0x2f7d },
  { "KANGXI RADICAL PLOW", 0x2f7e },
  { "KANGXI RADICAL EAR", 0x2f7f },
  { "KANGXI RADICAL BRUSH", 0x2f80 },
  { "KANGXI RADICAL MEAT", 0x2f81 },
  { "KANGXI RADICAL MINISTER", 0x2f82 },
  { "KANGXI RADICAL SELF", 0x2f83 },
  { "KANGXI RADICAL ARRIVE", 0x2f84 },
  { "KANGXI RADICAL MORTAR", 0x2f85 },
  { "KANGXI RADICAL TONGUE", 0x2f86 },
  { "KANGXI RADICAL OPPOSE", 0x2f87 },
  { "KANGXI RADICAL BOAT", 0x2f88 },
  { "KANGXI RADICAL STOPPING", 0x2f89 },
  { "KANGXI RADICAL COLOR", 0x2f8a },
  { "KANGXI RADICAL GRASS", 0x2f8b },
  { "KANGXI RADICAL TIGER", 0x2f8c },
  { "KANGXI RADICAL INSECT", 0x2f8d },
  { "KANGXI RADICAL BLOOD", 0x2f8e },
  { "KANGXI RADICAL WALK ENCLOSURE", 0x2f8f },
  { "KANGXI RADICAL CLOTHES", 0x2f90 },
  { "KANGXI RADICAL WEST", 0x2f91 },
  { "KANGXI RADICAL SEE", 0x2f92 },
  { "KANGXI RADICAL HORN", 0x2f93 },
  { "KANGXI RADICAL SPEECH", 0x2f94 },
  { "KANGXI RADICAL VALLEY", 0x2f95 },
  { "KANGXI RADICAL BEAN", 0x2f96 },
  { "KANGXI RADICAL PIG", 0x2f97 },
  { "KANGXI RADICAL BADGER", 0x2f98 },
  { "KANGXI RADICAL SHELL", 0x2f99 },
  { "KANGXI RADICAL RED", 0x2f9a },
  { "KANGXI RADICAL RUN", 0x2f9b },
  { "KANGXI RADICAL FOOT", 0x2f9c },
  { "KANGXI RADICAL BODY", 0x2f9d },
  { "KANGXI RADICAL CART", 0x2f9e },
  { "KANGXI RADICAL BITTER", 0x2f9f },
  { "KANGXI RADICAL MORNING", 0x2fa0 },
  { "KANGXI RADICAL WALK", 0x2fa1 },
  { "KANGXI RADICAL CITY", 0x2fa2 },
  { "KANGXI RADICAL WINE", 0x2fa3 },
  { "KANGXI RADICAL DISTINGUISH", 0x2fa4 },
  { "KANGXI RADICAL VILLAGE", 0x2fa5 },
  { "KANGXI RADICAL GOLD", 0x2fa6 },
  { "KANGXI RADICAL LONG", 0x2fa7 },
  { "KANGXI RADICAL GATE", 0x2fa8 },
  { "KANGXI RADICAL MOUND", 0x2fa9 },
  { "KANGXI RADICAL SLAVE", 0x2faa },
  { "KANGXI RADICAL SHORT TAILED BIRD", 0x2fab },
  { "KANGXI RADICAL RAIN", 0x2fac },
  { "KANGXI RADICAL BLUE", 0x2fad },
  { "KANGXI RADICAL WRONG", 0x2fae },
  { "KANGXI RADICAL FACE", 0x2faf },
  { "KANGXI RADICAL LEATHER", 0x2fb0 },
  { "KANGXI RADICAL TANNED LEATHER", 0x2fb1 },
  { "KANGXI RADICAL LEEK", 0x2fb2 },
  { "KANGXI RADICAL SOUND", 0x2fb3 },
  { "KANGXI RADICAL LEAF", 0x2fb4 },
  { "KANGXI RADICAL WIND", 0x2fb5 },
  { "KANGXI RADICAL FLY", 0x2fb6 },
  { "KANGXI RADICAL EAT", 0x2fb7 },
  { "KANGXI RADICAL HEAD", 0x2fb8 },
  { "KANGXI RADICAL FRAGRANT", 0x2fb9 },
  { "KANGXI RADICAL HORSE", 0x2fba },
  { "KANGXI RADICAL BONE", 0x2fbb },
  { "KANGXI RADICAL TALL", 0x2fbc },
  { "KANGXI RADICAL HAIR", 0x2fbd },
  { "KANGXI RADICAL FIGHT", 0x2fbe },
  { "KANGXI RADICAL SACRIFICIAL WINE", 0x2fbf },
  { "KANGXI RADICAL CAULDRON", 0x2fc0 },
  { "KANGXI RADICAL GHOST", 0x2fc1 },
  { "KANGXI RADICAL FISH", 0x2fc2 },
  { "KANGXI RADICAL BIRD", 0x2fc3 },
  { "KANGXI RADICAL SALT", 0x2fc4 },
  { "KANGXI RADICAL DEER", 0x2fc5 },
  { "KANGXI RADICAL WHEAT", 0x2fc6 },
  { "KANGXI RADICAL HEMP", 0x2fc7 },
  { "KANGXI RADICAL YELLOW", 0x2fc8 },
  { "KANGXI RADICAL MILLET", 0x2fc9 },
  { "KANGXI RADICAL BLACK", 0x2fca },
  { "KANGXI RADICAL EMBROIDERY", 0x2fcb },
  { "KANGXI RADICAL FROG", 0x2fcc },
  { "KANGXI RADICAL TRIPOD", 0x2fcd },
  { "KANGXI RADICAL DRUM", 0x2fce },
  { "KANGXI RADICAL RAT", 0x2fcf },
  { "KANGXI RADICAL NOSE", 0x2fd0 },
  { "KANGXI RADICAL EVEN", 0x2fd1 },
  { "KANGXI RADICAL TOOTH", 0x2fd2 },
  { "KANGXI RADICAL DRAGON", 0x2fd3 },
  { "KANGXI RADICAL TURTLE", 0x2fd4 },
  { "KANGXI RADICAL FLUTE", 0x2fd5 },
  { "IDEOGRAPHIC DESCRIPTION CHARACTER LEFT TO RIGHT", 0x2ff0 },
  { "IDEOGRAPHIC DESCRIPTION CHARACTER ABOVE TO BELOW", 0x2ff1 },
  { "IDEOGRAPHIC DESCRIPTION CHARACTER LEFT TO MIDDLE AND RIGHT", 0x2ff2 },
  { "IDEOGRAPHIC DESCRIPTION CHARACTER ABOVE TO MIDDLE AND BELOW", 0x2ff3 },
  { "IDEOGRAPHIC DESCRIPTION CHARACTER FULL SURROUND", 0x2ff4 },
  { "IDEOGRAPHIC DESCRIPTION CHARACTER SURROUND FROM ABOVE", 0x2ff5 },
  { "IDEOGRAPHIC DESCRIPTION CHARACTER SURROUND FROM BELOW", 0x2ff6 },
  { "IDEOGRAPHIC DESCRIPTION CHARACTER SURROUND FROM LEFT", 0x2ff7 },
  { "IDEOGRAPHIC DESCRIPTION CHARACTER SURROUND FROM UPPER LEFT", 0x2ff8 },
  { "IDEOGRAPHIC DESCRIPTION CHARACTER SURROUND FROM UPPER RIGHT", 0x2ff9 },
  { "IDEOGRAPHIC DESCRIPTION CHARACTER SURROUND FROM LOWER LEFT", 0x2ffa },
  { "IDEOGRAPHIC DESCRIPTION CHARACTER OVERLAID", 0x2ffb },
  { "IDEOGRAPHIC SPACE", 0x3000 },
  { "IDEOGRAPHIC COMMA", 0x3001 },
  { "IDEOGRAPHIC FULL STOP", 0x3002 },
  { "DITTO MARK", 0x3003 },
  { "JAPANESE INDUSTRIAL STANDARD SYMBOL", 0x3004 },
  { "IDEOGRAPHIC ITERATION MARK", 0x3005 },
  { "IDEOGRAPHIC CLOSING MARK", 0x3006 },
  { "IDEOGRAPHIC NUMBER ZERO", 0x3007 },
  { "LEFT ANGLE BRACKET", 0x3008 },
  { "RIGHT ANGLE BRACKET", 0x3009 },
  { "LEFT DOUBLE ANGLE BRACKET", 0x300a },
  { "RIGHT DOUBLE ANGLE BRACKET", 0x300b },
  { "LEFT CORNER BRACKET", 0x300c },
  { "RIGHT CORNER BRACKET", 0x300d },
  { "LEFT WHITE CORNER BRACKET", 0x300e },
  { "RIGHT WHITE CORNER BRACKET", 0x300f },
  { "LEFT BLACK LENTICULAR BRACKET", 0x3010 },
  { "RIGHT BLACK LENTICULAR BRACKET", 0x3011 },
  { "POSTAL MARK", 0x3012 },
  { "GETA MARK", 0x3013 },
  { "LEFT TORTOISE SHELL BRACKET", 0x3014 },
  { "RIGHT TORTOISE SHELL BRACKET", 0x3015 },
  { "LEFT WHITE LENTICULAR BRACKET", 0x3016 },
  { "RIGHT WHITE LENTICULAR BRACKET", 0x3017 },
  { "LEFT WHITE TORTOISE SHELL BRACKET", 0x3018 },
  { "RIGHT WHITE TORTOISE SHELL BRACKET", 0x3019 },
  { "LEFT WHITE SQUARE BRACKET", 0x301a },
  { "RIGHT WHITE SQUARE BRACKET", 0x301b },
  { "WAVE DASH", 0x301c },
  { "REVERSED DOUBLE PRIME QUOTATION MARK", 0x301d },
  { "DOUBLE PRIME QUOTATION MARK", 0x301e },
  { "LOW DOUBLE PRIME QUOTATION MARK", 0x301f },
  { "POSTAL MARK FACE", 0x3020 },
  { "HANGZHOU NUMERAL ONE", 0x3021 },
  { "HANGZHOU NUMERAL TWO", 0x3022 },
  { "HANGZHOU NUMERAL THREE", 0x3023 },
  { "HANGZHOU NUMERAL FOUR", 0x3024 },
  { "HANGZHOU NUMERAL FIVE", 0x3025 },
  { "HANGZHOU NUMERAL SIX", 0x3026 },
  { "HANGZHOU NUMERAL SEVEN", 0x3027 },
  { "HANGZHOU NUMERAL EIGHT", 0x3028 },
  { "HANGZHOU NUMERAL NINE", 0x3029 },
  { "IDEOGRAPHIC LEVEL TONE MARK", 0x302a },
  { "IDEOGRAPHIC RISING TONE MARK", 0x302b },
  { "IDEOGRAPHIC DEPARTING TONE MARK", 0x302c },
  { "IDEOGRAPHIC ENTERING TONE MARK", 0x302d },
  { "HANGUL SINGLE DOT TONE MARK", 0x302e },
  { "HANGUL DOUBLE DOT TONE MARK", 0x302f },
  { "WAVY DASH", 0x3030 },
  { "VERTICAL KANA REPEAT MARK", 0x3031 },
  { "VERTICAL KANA REPEAT WITH VOICED SOUND MARK", 0x3032 },
  { "VERTICAL KANA REPEAT MARK UPPER HALF", 0x3033 },
  { "VERTICAL KANA REPEAT WITH VOICED SOUND MARK UPPER HALF", 0x3034 },
  { "VERTICAL KANA REPEAT MARK LOWER HALF", 0x3035 },
  { "CIRCLED POSTAL MARK", 0x3036 },
  { "IDEOGRAPHIC TELEGRAPH LINE FEED SEPARATOR SYMBOL", 0x3037 },
  { "HANGZHOU NUMERAL TEN", 0x3038 },
  { "HANGZHOU NUMERAL TWENTY", 0x3039 },
  { "HANGZHOU NUMERAL THIRTY", 0x303a },
  { "IDEOGRAPHIC VARIATION INDICATOR", 0x303e },
  { "IDEOGRAPHIC HALF FILL SPACE", 0x303f },
  { "HIRAGANA LETTER SMALL A", 0x3041 },
  { "HIRAGANA LETTER A", 0x3042 },
  { "HIRAGANA LETTER SMALL I", 0x3043 },
  { "HIRAGANA LETTER I", 0x3044 },
  { "HIRAGANA LETTER SMALL U", 0x3045 },
  { "HIRAGANA LETTER U", 0x3046 },
  { "HIRAGANA LETTER SMALL E", 0x3047 },
  { "HIRAGANA LETTER E", 0x3048 },
  { "HIRAGANA LETTER SMALL O", 0x3049 },
  { "HIRAGANA LETTER O", 0x304a },
  { "HIRAGANA LETTER KA", 0x304b },
  { "HIRAGANA LETTER GA", 0x304c },
  { "HIRAGANA LETTER KI", 0x304d },
  { "HIRAGANA LETTER GI", 0x304e },
  { "HIRAGANA LETTER KU", 0x304f },
  { "HIRAGANA LETTER GU", 0x3050 },
  { "HIRAGANA LETTER KE", 0x3051 },
  { "HIRAGANA LETTER GE", 0x3052 },
  { "HIRAGANA LETTER KO", 0x3053 },
  { "HIRAGANA LETTER GO", 0x3054 },
  { "HIRAGANA LETTER SA", 0x3055 },
  { "HIRAGANA LETTER ZA", 0x3056 },
  { "HIRAGANA LETTER SI", 0x3057 },
  { "HIRAGANA LETTER ZI", 0x3058 },
  { "HIRAGANA LETTER SU", 0x3059 },
  { "HIRAGANA LETTER ZU", 0x305a },
  { "HIRAGANA LETTER SE", 0x305b },
  { "HIRAGANA LETTER ZE", 0x305c },
  { "HIRAGANA LETTER SO", 0x305d },
  { "HIRAGANA LETTER ZO", 0x305e },
  { "HIRAGANA LETTER TA", 0x305f },
  { "HIRAGANA LETTER DA", 0x3060 },
  { "HIRAGANA LETTER TI", 0x3061 },
  { "HIRAGANA LETTER DI", 0x3062 },
  { "HIRAGANA LETTER SMALL TU", 0x3063 },
  { "HIRAGANA LETTER TU", 0x3064 },
  { "HIRAGANA LETTER DU", 0x3065 },
  { "HIRAGANA LETTER TE", 0x3066 },
  { "HIRAGANA LETTER DE", 0x3067 },
  { "HIRAGANA LETTER TO", 0x3068 },
  { "HIRAGANA LETTER DO", 0x3069 },
  { "HIRAGANA LETTER NA", 0x306a },
  { "HIRAGANA LETTER NI", 0x306b },
  { "HIRAGANA LETTER NU", 0x306c },
  { "HIRAGANA LETTER NE", 0x306d },
  { "HIRAGANA LETTER NO", 0x306e },
  { "HIRAGANA LETTER HA", 0x306f },
  { "HIRAGANA LETTER BA", 0x3070 },
  { "HIRAGANA LETTER PA", 0x3071 },
  { "HIRAGANA LETTER HI", 0x3072 },
  { "HIRAGANA LETTER BI", 0x3073 },
  { "HIRAGANA LETTER PI", 0x3074 },
  { "HIRAGANA LETTER HU", 0x3075 },
  { "HIRAGANA LETTER BU", 0x3076 },
  { "HIRAGANA LETTER PU", 0x3077 },
  { "HIRAGANA LETTER HE", 0x3078 },
  { "HIRAGANA LETTER BE", 0x3079 },
  { "HIRAGANA LETTER PE", 0x307a },
  { "HIRAGANA LETTER HO", 0x307b },
  { "HIRAGANA LETTER BO", 0x307c },
  { "HIRAGANA LETTER PO", 0x307d },
  { "HIRAGANA LETTER MA", 0x307e },
  { "HIRAGANA LETTER MI", 0x307f },
  { "HIRAGANA LETTER MU", 0x3080 },
  { "HIRAGANA LETTER ME", 0x3081 },
  { "HIRAGANA LETTER MO", 0x3082 },
  { "HIRAGANA LETTER SMALL YA", 0x3083 },
  { "HIRAGANA LETTER YA", 0x3084 },
  { "HIRAGANA LETTER SMALL YU", 0x3085 },
  { "HIRAGANA LETTER YU", 0x3086 },
  { "HIRAGANA LETTER SMALL YO", 0x3087 },
  { "HIRAGANA LETTER YO", 0x3088 },
  { "HIRAGANA LETTER RA", 0x3089 },
  { "HIRAGANA LETTER RI", 0x308a },
  { "HIRAGANA LETTER RU", 0x308b },
  { "HIRAGANA LETTER RE", 0x308c },
  { "HIRAGANA LETTER RO", 0x308d },
  { "HIRAGANA LETTER SMALL WA", 0x308e },
  { "HIRAGANA LETTER WA", 0x308f },
  { "HIRAGANA LETTER WI", 0x3090 },
  { "HIRAGANA LETTER WE", 0x3091 },
  { "HIRAGANA LETTER WO", 0x3092 },
  { "HIRAGANA LETTER N", 0x3093 },
  { "HIRAGANA LETTER VU", 0x3094 },
  { "COMBINING KATAKANA-HIRAGANA VOICED SOUND MARK", 0x3099 },
  { "COMBINING KATAKANA-HIRAGANA SEMI-VOICED SOUND MARK", 0x309a },
  { "KATAKANA-HIRAGANA VOICED SOUND MARK", 0x309b },
  { "KATAKANA-HIRAGANA SEMI-VOICED SOUND MARK", 0x309c },
  { "HIRAGANA ITERATION MARK", 0x309d },
  { "HIRAGANA VOICED ITERATION MARK", 0x309e },
  { "KATAKANA LETTER SMALL A", 0x30a1 },
  { "KATAKANA LETTER A", 0x30a2 },
  { "KATAKANA LETTER SMALL I", 0x30a3 },
  { "KATAKANA LETTER I", 0x30a4 },
  { "KATAKANA LETTER SMALL U", 0x30a5 },
  { "KATAKANA LETTER U", 0x30a6 },
  { "KATAKANA LETTER SMALL E", 0x30a7 },
  { "KATAKANA LETTER E", 0x30a8 },
  { "KATAKANA LETTER SMALL O", 0x30a9 },
  { "KATAKANA LETTER O", 0x30aa },
  { "KATAKANA LETTER KA", 0x30ab },
  { "KATAKANA LETTER GA", 0x30ac },
  { "KATAKANA LETTER KI", 0x30ad },
  { "KATAKANA LETTER GI", 0x30ae },
  { "KATAKANA LETTER KU", 0x30af },
  { "KATAKANA LETTER GU", 0x30b0 },
  { "KATAKANA LETTER KE", 0x30b1 },
  { "KATAKANA LETTER GE", 0x30b2 },
  { "KATAKANA LETTER KO", 0x30b3 },
  { "KATAKANA LETTER GO", 0x30b4 },
  { "KATAKANA LETTER SA", 0x30b5 },
  { "KATAKANA LETTER ZA", 0x30b6 },
  { "KATAKANA LETTER SI", 0x30b7 },
  { "KATAKANA LETTER ZI", 0x30b8 },
  { "KATAKANA LETTER SU", 0x30b9 },
  { "KATAKANA LETTER ZU", 0x30ba },
  { "KATAKANA LETTER SE", 0x30bb },
  { "KATAKANA LETTER ZE", 0x30bc },
  { "KATAKANA LETTER SO", 0x30bd },
  { "KATAKANA LETTER ZO", 0x30be },
  { "KATAKANA LETTER TA", 0x30bf },
  { "KATAKANA LETTER DA", 0x30c0 },
  { "KATAKANA LETTER TI", 0x30c1 },
  { "KATAKANA LETTER DI", 0x30c2 },
  { "KATAKANA LETTER SMALL TU", 0x30c3 },
  { "KATAKANA LETTER TU", 0x30c4 },
  { "KATAKANA LETTER DU", 0x30c5 },
  { "KATAKANA LETTER TE", 0x30c6 },
  { "KATAKANA LETTER DE", 0x30c7 },
  { "KATAKANA LETTER TO", 0x30c8 },
  { "KATAKANA LETTER DO", 0x30c9 },
  { "KATAKANA LETTER NA", 0x30ca },
  { "KATAKANA LETTER NI", 0x30cb },
  { "KATAKANA LETTER NU", 0x30cc },
  { "KATAKANA LETTER NE", 0x30cd },
  { "KATAKANA LETTER NO", 0x30ce },
  { "KATAKANA LETTER HA", 0x30cf },
  { "KATAKANA LETTER BA", 0x30d0 },
  { "KATAKANA LETTER PA", 0x30d1 },
  { "KATAKANA LETTER HI", 0x30d2 },
  { "KATAKANA LETTER BI", 0x30d3 },
  { "KATAKANA LETTER PI", 0x30d4 },
  { "KATAKANA LETTER HU", 0x30d5 },
  { "KATAKANA LETTER BU", 0x30d6 },
  { "KATAKANA LETTER PU", 0x30d7 },
  { "KATAKANA LETTER HE", 0x30d8 },
  { "KATAKANA LETTER BE", 0x30d9 },
  { "KATAKANA LETTER PE", 0x30da },
  { "KATAKANA LETTER HO", 0x30db },
  { "KATAKANA LETTER BO", 0x30dc },
  { "KATAKANA LETTER PO", 0x30dd },
  { "KATAKANA LETTER MA", 0x30de },
  { "KATAKANA LETTER MI", 0x30df },
  { "KATAKANA LETTER MU", 0x30e0 },
  { "KATAKANA LETTER ME", 0x30e1 },
  { "KATAKANA LETTER MO", 0x30e2 },
  { "KATAKANA LETTER SMALL YA", 0x30e3 },
  { "KATAKANA LETTER YA", 0x30e4 },
  { "KATAKANA LETTER SMALL YU", 0x30e5 },
  { "KATAKANA LETTER YU", 0x30e6 },
  { "KATAKANA LETTER SMALL YO", 0x30e7 },
  { "KATAKANA LETTER YO", 0x30e8 },
  { "KATAKANA LETTER RA", 0x30e9 },
  { "KATAKANA LETTER RI", 0x30ea },
  { "KATAKANA LETTER RU", 0x30eb },
  { "KATAKANA LETTER RE", 0x30ec },
  { "KATAKANA LETTER RO", 0x30ed },
  { "KATAKANA LETTER SMALL WA", 0x30ee },
  { "KATAKANA LETTER WA", 0x30ef },
  { "KATAKANA LETTER WI", 0x30f0 },
  { "KATAKANA LETTER WE", 0x30f1 },
  { "KATAKANA LETTER WO", 0x30f2 },
  { "KATAKANA LETTER N", 0x30f3 },
  { "KATAKANA LETTER VU", 0x30f4 },
  { "KATAKANA LETTER SMALL KA", 0x30f5 },
  { "KATAKANA LETTER SMALL KE", 0x30f6 },
  { "KATAKANA LETTER VA", 0x30f7 },
  { "KATAKANA LETTER VI", 0x30f8 },
  { "KATAKANA LETTER VE", 0x30f9 },
  { "KATAKANA LETTER VO", 0x30fa },
  { "KATAKANA MIDDLE DOT", 0x30fb },
  { "KATAKANA-HIRAGANA PROLONGED SOUND MARK", 0x30fc },
  { "KATAKANA ITERATION MARK", 0x30fd },
  { "KATAKANA VOICED ITERATION MARK", 0x30fe },
  { "BOPOMOFO LETTER B", 0x3105 },
  { "BOPOMOFO LETTER P", 0x3106 },
  { "BOPOMOFO LETTER M", 0x3107 },
  { "BOPOMOFO LETTER F", 0x3108 },
  { "BOPOMOFO LETTER D", 0x3109 },
  { "BOPOMOFO LETTER T", 0x310a },
  { "BOPOMOFO LETTER N", 0x310b },
  { "BOPOMOFO LETTER L", 0x310c },
  { "BOPOMOFO LETTER G", 0x310d },
  { "BOPOMOFO LETTER K", 0x310e },
  { "BOPOMOFO LETTER H", 0x310f },
  { "BOPOMOFO LETTER J", 0x3110 },
  { "BOPOMOFO LETTER Q", 0x3111 },
  { "BOPOMOFO LETTER X", 0x3112 },
  { "BOPOMOFO LETTER ZH", 0x3113 },
  { "BOPOMOFO LETTER CH", 0x3114 },
  { "BOPOMOFO LETTER SH", 0x3115 },
  { "BOPOMOFO LETTER R", 0x3116 },
  { "BOPOMOFO LETTER Z", 0x3117 },
  { "BOPOMOFO LETTER C", 0x3118 },
  { "BOPOMOFO LETTER S", 0x3119 },
  { "BOPOMOFO LETTER A", 0x311a },
  { "BOPOMOFO LETTER O", 0x311b },
  { "BOPOMOFO LETTER E", 0x311c },
  { "BOPOMOFO LETTER EH", 0x311d },
  { "BOPOMOFO LETTER AI", 0x311e },
  { "BOPOMOFO LETTER EI", 0x311f },
  { "BOPOMOFO LETTER AU", 0x3120 },
  { "BOPOMOFO LETTER OU", 0x3121 },
  { "BOPOMOFO LETTER AN", 0x3122 },
  { "BOPOMOFO LETTER EN", 0x3123 },
  { "BOPOMOFO LETTER ANG", 0x3124 },
  { "BOPOMOFO LETTER ENG", 0x3125 },
  { "BOPOMOFO LETTER ER", 0x3126 },
  { "BOPOMOFO LETTER I", 0x3127 },
  { "BOPOMOFO LETTER U", 0x3128 },
  { "BOPOMOFO LETTER IU", 0x3129 },
  { "BOPOMOFO LETTER V", 0x312a },
  { "BOPOMOFO LETTER NG", 0x312b },
  { "BOPOMOFO LETTER GN", 0x312c },
  { "HANGUL LETTER KIYEOK", 0x3131 },
  { "HANGUL LETTER SSANGKIYEOK", 0x3132 },
  { "HANGUL LETTER KIYEOK-SIOS", 0x3133 },
  { "HANGUL LETTER NIEUN", 0x3134 },
  { "HANGUL LETTER NIEUN-CIEUC", 0x3135 },
  { "HANGUL LETTER NIEUN-HIEUH", 0x3136 },
  { "HANGUL LETTER TIKEUT", 0x3137 },
  { "HANGUL LETTER SSANGTIKEUT", 0x3138 },
  { "HANGUL LETTER RIEUL", 0x3139 },
  { "HANGUL LETTER RIEUL-KIYEOK", 0x313a },
  { "HANGUL LETTER RIEUL-MIEUM", 0x313b },
  { "HANGUL LETTER RIEUL-PIEUP", 0x313c },
  { "HANGUL LETTER RIEUL-SIOS", 0x313d },
  { "HANGUL LETTER RIEUL-THIEUTH", 0x313e },
  { "HANGUL LETTER RIEUL-PHIEUPH", 0x313f },
  { "HANGUL LETTER RIEUL-HIEUH", 0x3140 },
  { "HANGUL LETTER MIEUM", 0x3141 },
  { "HANGUL LETTER PIEUP", 0x3142 },
  { "HANGUL LETTER SSANGPIEUP", 0x3143 },
  { "HANGUL LETTER PIEUP-SIOS", 0x3144 },
  { "HANGUL LETTER SIOS", 0x3145 },
  { "HANGUL LETTER SSANGSIOS", 0x3146 },
  { "HANGUL LETTER IEUNG", 0x3147 },
  { "HANGUL LETTER CIEUC", 0x3148 },
  { "HANGUL LETTER SSANGCIEUC", 0x3149 },
  { "HANGUL LETTER CHIEUCH", 0x314a },
  { "HANGUL LETTER KHIEUKH", 0x314b },
  { "HANGUL LETTER THIEUTH", 0x314c },
  { "HANGUL LETTER PHIEUPH", 0x314d },
  { "HANGUL LETTER HIEUH", 0x314e },
  { "HANGUL LETTER A", 0x314f },
  { "HANGUL LETTER AE", 0x3150 },
  { "HANGUL LETTER YA", 0x3151 },
  { "HANGUL LETTER YAE", 0x3152 },
  { "HANGUL LETTER EO", 0x3153 },
  { "HANGUL LETTER E", 0x3154 },
  { "HANGUL LETTER YEO", 0x3155 },
  { "HANGUL LETTER YE", 0x3156 },
  { "HANGUL LETTER O", 0x3157 },
  { "HANGUL LETTER WA", 0x3158 },
  { "HANGUL LETTER WAE", 0x3159 },
  { "HANGUL LETTER OE", 0x315a },
  { "HANGUL LETTER YO", 0x315b },
  { "HANGUL LETTER U", 0x315c },
  { "HANGUL LETTER WEO", 0x315d },
  { "HANGUL LETTER WE", 0x315e },
  { "HANGUL LETTER WI", 0x315f },
  { "HANGUL LETTER YU", 0x3160 },
  { "HANGUL LETTER EU", 0x3161 },
  { "HANGUL LETTER YI", 0x3162 },
  { "HANGUL LETTER I", 0x3163 },
  { "HANGUL FILLER", 0x3164 },
  { "HANGUL LETTER SSANGNIEUN", 0x3165 },
  { "HANGUL LETTER NIEUN-TIKEUT", 0x3166 },
  { "HANGUL LETTER NIEUN-SIOS", 0x3167 },
  { "HANGUL LETTER NIEUN-PANSIOS", 0x3168 },
  { "HANGUL LETTER RIEUL-KIYEOK-SIOS", 0x3169 },
  { "HANGUL LETTER RIEUL-TIKEUT", 0x316a },
  { "HANGUL LETTER RIEUL-PIEUP-SIOS", 0x316b },
  { "HANGUL LETTER RIEUL-PANSIOS", 0x316c },
  { "HANGUL LETTER RIEUL-YEORINHIEUH", 0x316d },
  { "HANGUL LETTER MIEUM-PIEUP", 0x316e },
  { "HANGUL LETTER MIEUM-SIOS", 0x316f },
  { "HANGUL LETTER MIEUM-PANSIOS", 0x3170 },
  { "HANGUL LETTER KAPYEOUNMIEUM", 0x3171 },
  { "HANGUL LETTER PIEUP-KIYEOK", 0x3172 },
  { "HANGUL LETTER PIEUP-TIKEUT", 0x3173 },
  { "HANGUL LETTER PIEUP-SIOS-KIYEOK", 0x3174 },
  { "HANGUL LETTER PIEUP-SIOS-TIKEUT", 0x3175 },
  { "HANGUL LETTER PIEUP-CIEUC", 0x3176 },
  { "HANGUL LETTER PIEUP-THIEUTH", 0x3177 },
  { "HANGUL LETTER KAPYEOUNPIEUP", 0x3178 },
  { "HANGUL LETTER KAPYEOUNSSANGPIEUP", 0x3179 },
  { "HANGUL LETTER SIOS-KIYEOK", 0x317a },
  { "HANGUL LETTER SIOS-NIEUN", 0x317b },
  { "HANGUL LETTER SIOS-TIKEUT", 0x317c },
  { "HANGUL LETTER SIOS-PIEUP", 0x317d },
  { "HANGUL LETTER SIOS-CIEUC", 0x317e },
  { "HANGUL LETTER PANSIOS", 0x317f },
  { "HANGUL LETTER SSANGIEUNG", 0x3180 },
  { "HANGUL LETTER YESIEUNG", 0x3181 },
  { "HANGUL LETTER YESIEUNG-SIOS", 0x3182 },
  { "HANGUL LETTER YESIEUNG-PANSIOS", 0x3183 },
  { "HANGUL LETTER KAPYEOUNPHIEUPH", 0x3184 },
  { "HANGUL LETTER SSANGHIEUH", 0x3185 },
  { "HANGUL LETTER YEORINHIEUH", 0x3186 },
  { "HANGUL LETTER YO-YA", 0x3187 },
  { "HANGUL LETTER YO-YAE", 0x3188 },
  { "HANGUL LETTER YO-I", 0x3189 },
  { "HANGUL LETTER YU-YEO", 0x318a },
  { "HANGUL LETTER YU-YE", 0x318b },
  { "HANGUL LETTER YU-I", 0x318c },
  { "HANGUL LETTER ARAEA", 0x318d },
  { "HANGUL LETTER ARAEAE", 0x318e },
  { "IDEOGRAPHIC ANNOTATION LINKING MARK", 0x3190 },
  { "IDEOGRAPHIC ANNOTATION REVERSE MARK", 0x3191 },
  { "IDEOGRAPHIC ANNOTATION ONE MARK", 0x3192 },
  { "IDEOGRAPHIC ANNOTATION TWO MARK", 0x3193 },
  { "IDEOGRAPHIC ANNOTATION THREE MARK", 0x3194 },
  { "IDEOGRAPHIC ANNOTATION FOUR MARK", 0x3195 },
  { "IDEOGRAPHIC ANNOTATION TOP MARK", 0x3196 },
  { "IDEOGRAPHIC ANNOTATION MIDDLE MARK", 0x3197 },
  { "IDEOGRAPHIC ANNOTATION BOTTOM MARK", 0x3198 },
  { "IDEOGRAPHIC ANNOTATION FIRST MARK", 0x3199 },
  { "IDEOGRAPHIC ANNOTATION SECOND MARK", 0x319a },
  { "IDEOGRAPHIC ANNOTATION THIRD MARK", 0x319b },
  { "IDEOGRAPHIC ANNOTATION FOURTH MARK", 0x319c },
  { "IDEOGRAPHIC ANNOTATION HEAVEN MARK", 0x319d },
  { "IDEOGRAPHIC ANNOTATION EARTH MARK", 0x319e },
  { "IDEOGRAPHIC ANNOTATION MAN MARK", 0x319f },
  { "BOPOMOFO LETTER BU", 0x31a0 },
  { "BOPOMOFO LETTER ZI", 0x31a1 },
  { "BOPOMOFO LETTER JI", 0x31a2 },
  { "BOPOMOFO LETTER GU", 0x31a3 },
  { "BOPOMOFO LETTER EE", 0x31a4 },
  { "BOPOMOFO LETTER ENN", 0x31a5 },
  { "BOPOMOFO LETTER OO", 0x31a6 },
  { "BOPOMOFO LETTER ONN", 0x31a7 },
  { "BOPOMOFO LETTER IR", 0x31a8 },
  { "BOPOMOFO LETTER ANN", 0x31a9 },
  { "BOPOMOFO LETTER INN", 0x31aa },
  { "BOPOMOFO LETTER UNN", 0x31ab },
  { "BOPOMOFO LETTER IM", 0x31ac },
  { "BOPOMOFO LETTER NGG", 0x31ad },
  { "BOPOMOFO LETTER AINN", 0x31ae },
  { "BOPOMOFO LETTER AUNN", 0x31af },
  { "BOPOMOFO LETTER AM", 0x31b0 },
  { "BOPOMOFO LETTER OM", 0x31b1 },
  { "BOPOMOFO LETTER ONG", 0x31b2 },
  { "BOPOMOFO LETTER INNN", 0x31b3 },
  { "BOPOMOFO FINAL LETTER P", 0x31b4 },
  { "BOPOMOFO FINAL LETTER T", 0x31b5 },
  { "BOPOMOFO FINAL LETTER K", 0x31b6 },
  { "BOPOMOFO FINAL LETTER H", 0x31b7 },
  { "PARENTHESIZED HANGUL KIYEOK", 0x3200 },
  { "PARENTHESIZED HANGUL NIEUN", 0x3201 },
  { "PARENTHESIZED HANGUL TIKEUT", 0x3202 },
  { "PARENTHESIZED HANGUL RIEUL", 0x3203 },
  { "PARENTHESIZED HANGUL MIEUM", 0x3204 },
  { "PARENTHESIZED HANGUL PIEUP", 0x3205 },
  { "PARENTHESIZED HANGUL SIOS", 0x3206 },
  { "PARENTHESIZED HANGUL IEUNG", 0x3207 },
  { "PARENTHESIZED HANGUL CIEUC", 0x3208 },
  { "PARENTHESIZED HANGUL CHIEUCH", 0x3209 },
  { "PARENTHESIZED HANGUL KHIEUKH", 0x320a },
  { "PARENTHESIZED HANGUL THIEUTH", 0x320b },
  { "PARENTHESIZED HANGUL PHIEUPH", 0x320c },
  { "PARENTHESIZED HANGUL HIEUH", 0x320d },
  { "PARENTHESIZED HANGUL KIYEOK A", 0x320e },
  { "PARENTHESIZED HANGUL NIEUN A", 0x320f },
  { "PARENTHESIZED HANGUL TIKEUT A", 0x3210 },
  { "PARENTHESIZED HANGUL RIEUL A", 0x3211 },
  { "PARENTHESIZED HANGUL MIEUM A", 0x3212 },
  { "PARENTHESIZED HANGUL PIEUP A", 0x3213 },
  { "PARENTHESIZED HANGUL SIOS A", 0x3214 },
  { "PARENTHESIZED HANGUL IEUNG A", 0x3215 },
  { "PARENTHESIZED HANGUL CIEUC A", 0x3216 },
  { "PARENTHESIZED HANGUL CHIEUCH A", 0x3217 },
  { "PARENTHESIZED HANGUL KHIEUKH A", 0x3218 },
  { "PARENTHESIZED HANGUL THIEUTH A", 0x3219 },
  { "PARENTHESIZED HANGUL PHIEUPH A", 0x321a },
  { "PARENTHESIZED HANGUL HIEUH A", 0x321b },
  { "PARENTHESIZED HANGUL CIEUC U", 0x321c },
  { "PARENTHESIZED IDEOGRAPH ONE", 0x3220 },
  { "PARENTHESIZED IDEOGRAPH TWO", 0x3221 },
  { "PARENTHESIZED IDEOGRAPH THREE", 0x3222 },
  { "PARENTHESIZED IDEOGRAPH FOUR", 0x3223 },
  { "PARENTHESIZED IDEOGRAPH FIVE", 0x3224 },
  { "PARENTHESIZED IDEOGRAPH SIX", 0x3225 },
  { "PARENTHESIZED IDEOGRAPH SEVEN", 0x3226 },
  { "PARENTHESIZED IDEOGRAPH EIGHT", 0x3227 },
  { "PARENTHESIZED IDEOGRAPH NINE", 0x3228 },
  { "PARENTHESIZED IDEOGRAPH TEN", 0x3229 },
  { "PARENTHESIZED IDEOGRAPH MOON", 0x322a },
  { "PARENTHESIZED IDEOGRAPH FIRE", 0x322b },
  { "PARENTHESIZED IDEOGRAPH WATER", 0x322c },
  { "PARENTHESIZED IDEOGRAPH WOOD", 0x322d },
  { "PARENTHESIZED IDEOGRAPH METAL", 0x322e },
  { "PARENTHESIZED IDEOGRAPH EARTH", 0x322f },
  { "PARENTHESIZED IDEOGRAPH SUN", 0x3230 },
  { "PARENTHESIZED IDEOGRAPH STOCK", 0x3231 },
  { "PARENTHESIZED IDEOGRAPH HAVE", 0x3232 },
  { "PARENTHESIZED IDEOGRAPH SOCIETY", 0x3233 },
  { "PARENTHESIZED IDEOGRAPH NAME", 0x3234 },
  { "PARENTHESIZED IDEOGRAPH SPECIAL", 0x3235 },
  { "PARENTHESIZED IDEOGRAPH FINANCIAL", 0x3236 },
  { "PARENTHESIZED IDEOGRAPH CONGRATULATION", 0x3237 },
  { "PARENTHESIZED IDEOGRAPH LABOR", 0x3238 },
  { "PARENTHESIZED IDEOGRAPH REPRESENT", 0x3239 },
  { "PARENTHESIZED IDEOGRAPH CALL", 0x323a },
  { "PARENTHESIZED IDEOGRAPH STUDY", 0x323b },
  { "PARENTHESIZED IDEOGRAPH SUPERVISE", 0x323c },
  { "PARENTHESIZED IDEOGRAPH ENTERPRISE", 0x323d },
  { "PARENTHESIZED IDEOGRAPH RESOURCE", 0x323e },
  { "PARENTHESIZED IDEOGRAPH ALLIANCE", 0x323f },
  { "PARENTHESIZED IDEOGRAPH FESTIVAL", 0x3240 },
  { "PARENTHESIZED IDEOGRAPH REST", 0x3241 },
  { "PARENTHESIZED IDEOGRAPH SELF", 0x3242 },
  { "PARENTHESIZED IDEOGRAPH REACH", 0x3243 },
  { "CIRCLED HANGUL KIYEOK", 0x3260 },
  { "CIRCLED HANGUL NIEUN", 0x3261 },
  { "CIRCLED HANGUL TIKEUT", 0x3262 },
  { "CIRCLED HANGUL RIEUL", 0x3263 },
  { "CIRCLED HANGUL MIEUM", 0x3264 },
  { "CIRCLED HANGUL PIEUP", 0x3265 },
  { "CIRCLED HANGUL SIOS", 0x3266 },
  { "CIRCLED HANGUL IEUNG", 0x3267 },
  { "CIRCLED HANGUL CIEUC", 0x3268 },
  { "CIRCLED HANGUL CHIEUCH", 0x3269 },
  { "CIRCLED HANGUL KHIEUKH", 0x326a },
  { "CIRCLED HANGUL THIEUTH", 0x326b },
  { "CIRCLED HANGUL PHIEUPH", 0x326c },
  { "CIRCLED HANGUL HIEUH", 0x326d },
  { "CIRCLED HANGUL KIYEOK A", 0x326e },
  { "CIRCLED HANGUL NIEUN A", 0x326f },
  { "CIRCLED HANGUL TIKEUT A", 0x3270 },
  { "CIRCLED HANGUL RIEUL A", 0x3271 },
  { "CIRCLED HANGUL MIEUM A", 0x3272 },
  { "CIRCLED HANGUL PIEUP A", 0x3273 },
  { "CIRCLED HANGUL SIOS A", 0x3274 },
  { "CIRCLED HANGUL IEUNG A", 0x3275 },
  { "CIRCLED HANGUL CIEUC A", 0x3276 },
  { "CIRCLED HANGUL CHIEUCH A", 0x3277 },
  { "CIRCLED HANGUL KHIEUKH A", 0x3278 },
  { "CIRCLED HANGUL THIEUTH A", 0x3279 },
  { "CIRCLED HANGUL PHIEUPH A", 0x327a },
  { "CIRCLED HANGUL HIEUH A", 0x327b },
  { "KOREAN STANDARD SYMBOL", 0x327f },
  { "CIRCLED IDEOGRAPH ONE", 0x3280 },
  { "CIRCLED IDEOGRAPH TWO", 0x3281 },
  { "CIRCLED IDEOGRAPH THREE", 0x3282 },
  { "CIRCLED IDEOGRAPH FOUR", 0x3283 },
  { "CIRCLED IDEOGRAPH FIVE", 0x3284 },
  { "CIRCLED IDEOGRAPH SIX", 0x3285 },
  { "CIRCLED IDEOGRAPH SEVEN", 0x3286 },
  { "CIRCLED IDEOGRAPH EIGHT", 0x3287 },
  { "CIRCLED IDEOGRAPH NINE", 0x3288 },
  { "CIRCLED IDEOGRAPH TEN", 0x3289 },
  { "CIRCLED IDEOGRAPH MOON", 0x328a },
  { "CIRCLED IDEOGRAPH FIRE", 0x328b },
  { "CIRCLED IDEOGRAPH WATER", 0x328c },
  { "CIRCLED IDEOGRAPH WOOD", 0x328d },
  { "CIRCLED IDEOGRAPH METAL", 0x328e },
  { "CIRCLED IDEOGRAPH EARTH", 0x328f },
  { "CIRCLED IDEOGRAPH SUN", 0x3290 },
  { "CIRCLED IDEOGRAPH STOCK", 0x3291 },
  { "CIRCLED IDEOGRAPH HAVE", 0x3292 },
  { "CIRCLED IDEOGRAPH SOCIETY", 0x3293 },
  { "CIRCLED IDEOGRAPH NAME", 0x3294 },
  { "CIRCLED IDEOGRAPH SPECIAL", 0x3295 },
  { "CIRCLED IDEOGRAPH FINANCIAL", 0x3296 },
  { "CIRCLED IDEOGRAPH CONGRATULATION", 0x3297 },
  { "CIRCLED IDEOGRAPH LABOR", 0x3298 },
  { "CIRCLED IDEOGRAPH SECRET", 0x3299 },
  { "CIRCLED IDEOGRAPH MALE", 0x329a },
  { "CIRCLED IDEOGRAPH FEMALE", 0x329b },
  { "CIRCLED IDEOGRAPH SUITABLE", 0x329c },
  { "CIRCLED IDEOGRAPH EXCELLENT", 0x329d },
  { "CIRCLED IDEOGRAPH PRINT", 0x329e },
  { "CIRCLED IDEOGRAPH ATTENTION", 0x329f },
  { "CIRCLED IDEOGRAPH ITEM", 0x32a0 },
  { "CIRCLED IDEOGRAPH REST", 0x32a1 },
  { "CIRCLED IDEOGRAPH COPY", 0x32a2 },
  { "CIRCLED IDEOGRAPH CORRECT", 0x32a3 },
  { "CIRCLED IDEOGRAPH HIGH", 0x32a4 },
  { "CIRCLED IDEOGRAPH CENTRE", 0x32a5 },
  { "CIRCLED IDEOGRAPH LOW", 0x32a6 },
  { "CIRCLED IDEOGRAPH LEFT", 0x32a7 },
  { "CIRCLED IDEOGRAPH RIGHT", 0x32a8 },
  { "CIRCLED IDEOGRAPH MEDICINE", 0x32a9 },
  { "CIRCLED IDEOGRAPH RELIGION", 0x32aa },
  { "CIRCLED IDEOGRAPH STUDY", 0x32ab },
  { "CIRCLED IDEOGRAPH SUPERVISE", 0x32ac },
  { "CIRCLED IDEOGRAPH ENTERPRISE", 0x32ad },
  { "CIRCLED IDEOGRAPH RESOURCE", 0x32ae },
  { "CIRCLED IDEOGRAPH ALLIANCE", 0x32af },
  { "CIRCLED IDEOGRAPH NIGHT", 0x32b0 },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR JANUARY", 0x32c0 },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR FEBRUARY", 0x32c1 },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR MARCH", 0x32c2 },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR APRIL", 0x32c3 },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR MAY", 0x32c4 },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR JUNE", 0x32c5 },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR JULY", 0x32c6 },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR AUGUST", 0x32c7 },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR SEPTEMBER", 0x32c8 },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR OCTOBER", 0x32c9 },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR NOVEMBER", 0x32ca },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DECEMBER", 0x32cb },
  { "CIRCLED KATAKANA A", 0x32d0 },
  { "CIRCLED KATAKANA I", 0x32d1 },
  { "CIRCLED KATAKANA U", 0x32d2 },
  { "CIRCLED KATAKANA E", 0x32d3 },
  { "CIRCLED KATAKANA O", 0x32d4 },
  { "CIRCLED KATAKANA KA", 0x32d5 },
  { "CIRCLED KATAKANA KI", 0x32d6 },
  { "CIRCLED KATAKANA KU", 0x32d7 },
  { "CIRCLED KATAKANA KE", 0x32d8 },
  { "CIRCLED KATAKANA KO", 0x32d9 },
  { "CIRCLED KATAKANA SA", 0x32da },
  { "CIRCLED KATAKANA SI", 0x32db },
  { "CIRCLED KATAKANA SU", 0x32dc },
  { "CIRCLED KATAKANA SE", 0x32dd },
  { "CIRCLED KATAKANA SO", 0x32de },
  { "CIRCLED KATAKANA TA", 0x32df },
  { "CIRCLED KATAKANA TI", 0x32e0 },
  { "CIRCLED KATAKANA TU", 0x32e1 },
  { "CIRCLED KATAKANA TE", 0x32e2 },
  { "CIRCLED KATAKANA TO", 0x32e3 },
  { "CIRCLED KATAKANA NA", 0x32e4 },
  { "CIRCLED KATAKANA NI", 0x32e5 },
  { "CIRCLED KATAKANA NU", 0x32e6 },
  { "CIRCLED KATAKANA NE", 0x32e7 },
  { "CIRCLED KATAKANA NO", 0x32e8 },
  { "CIRCLED KATAKANA HA", 0x32e9 },
  { "CIRCLED KATAKANA HI", 0x32ea },
  { "CIRCLED KATAKANA HU", 0x32eb },
  { "CIRCLED KATAKANA HE", 0x32ec },
  { "CIRCLED KATAKANA HO", 0x32ed },
  { "CIRCLED KATAKANA MA", 0x32ee },
  { "CIRCLED KATAKANA MI", 0x32ef },
  { "CIRCLED KATAKANA MU", 0x32f0 },
  { "CIRCLED KATAKANA ME", 0x32f1 },
  { "CIRCLED KATAKANA MO", 0x32f2 },
  { "CIRCLED KATAKANA YA", 0x32f3 },
  { "CIRCLED KATAKANA YU", 0x32f4 },
  { "CIRCLED KATAKANA YO", 0x32f5 },
  { "CIRCLED KATAKANA RA", 0x32f6 },
  { "CIRCLED KATAKANA RI", 0x32f7 },
  { "CIRCLED KATAKANA RU", 0x32f8 },
  { "CIRCLED KATAKANA RE", 0x32f9 },
  { "CIRCLED KATAKANA RO", 0x32fa },
  { "CIRCLED KATAKANA WA", 0x32fb },
  { "CIRCLED KATAKANA WI", 0x32fc },
  { "CIRCLED KATAKANA WE", 0x32fd },
  { "CIRCLED KATAKANA WO", 0x32fe },
  { "SQUARE APAATO", 0x3300 },
  { "SQUARE ARUHUA", 0x3301 },
  { "SQUARE ANPEA", 0x3302 },
  { "SQUARE AARU", 0x3303 },
  { "SQUARE ININGU", 0x3304 },
  { "SQUARE INTI", 0x3305 },
  { "SQUARE UON", 0x3306 },
  { "SQUARE ESUKUUDO", 0x3307 },
  { "SQUARE EEKAA", 0x3308 },
  { "SQUARE ONSU", 0x3309 },
  { "SQUARE OOMU", 0x330a },
  { "SQUARE KAIRI", 0x330b },
  { "SQUARE KARATTO", 0x330c },
  { "SQUARE KARORII", 0x330d },
  { "SQUARE GARON", 0x330e },
  { "SQUARE GANMA", 0x330f },
  { "SQUARE GIGA", 0x3310 },
  { "SQUARE GINII", 0x3311 },
  { "SQUARE KYURII", 0x3312 },
  { "SQUARE GIRUDAA", 0x3313 },
  { "SQUARE KIRO", 0x3314 },
  { "SQUARE KIROGURAMU", 0x3315 },
  { "SQUARE KIROMEETORU", 0x3316 },
  { "SQUARE KIROWATTO", 0x3317 },
  { "SQUARE GURAMU", 0x3318 },
  { "SQUARE GURAMUTON", 0x3319 },
  { "SQUARE KURUZEIRO", 0x331a },
  { "SQUARE KUROONE", 0x331b },
  { "SQUARE KEESU", 0x331c },
  { "SQUARE KORUNA", 0x331d },
  { "SQUARE KOOPO", 0x331e },
  { "SQUARE SAIKURU", 0x331f },
  { "SQUARE SANTIIMU", 0x3320 },
  { "SQUARE SIRINGU", 0x3321 },
  { "SQUARE SENTI", 0x3322 },
  { "SQUARE SENTO", 0x3323 },
  { "SQUARE DAASU", 0x3324 },
  { "SQUARE DESI", 0x3325 },
  { "SQUARE DORU", 0x3326 },
  { "SQUARE TON", 0x3327 },
  { "SQUARE NANO", 0x3328 },
  { "SQUARE NOTTO", 0x3329 },
  { "SQUARE HAITU", 0x332a },
  { "SQUARE PAASENTO", 0x332b },
  { "SQUARE PAATU", 0x332c },
  { "SQUARE BAARERU", 0x332d },
  { "SQUARE PIASUTORU", 0x332e },
  { "SQUARE PIKURU", 0x332f },
  { "SQUARE PIKO", 0x3330 },
  { "SQUARE BIRU", 0x3331 },
  { "SQUARE HUARADDO", 0x3332 },
  { "SQUARE HUIITO", 0x3333 },
  { "SQUARE BUSSYERU", 0x3334 },
  { "SQUARE HURAN", 0x3335 },
  { "SQUARE HEKUTAARU", 0x3336 },
  { "SQUARE PESO", 0x3337 },
  { "SQUARE PENIHI", 0x3338 },
  { "SQUARE HERUTU", 0x3339 },
  { "SQUARE PENSU", 0x333a },
  { "SQUARE PEEZI", 0x333b },
  { "SQUARE BEETA", 0x333c },
  { "SQUARE POINTO", 0x333d },
  { "SQUARE BORUTO", 0x333e },
  { "SQUARE HON", 0x333f },
  { "SQUARE PONDO", 0x3340 },
  { "SQUARE HOORU", 0x3341 },
  { "SQUARE HOON", 0x3342 },
  { "SQUARE MAIKURO", 0x3343 },
  { "SQUARE MAIRU", 0x3344 },
  { "SQUARE MAHHA", 0x3345 },
  { "SQUARE MARUKU", 0x3346 },
  { "SQUARE MANSYON", 0x3347 },
  { "SQUARE MIKURON", 0x3348 },
  { "SQUARE MIRI", 0x3349 },
  { "SQUARE MIRIBAARU", 0x334a },
  { "SQUARE MEGA", 0x334b },
  { "SQUARE MEGATON", 0x334c },
  { "SQUARE MEETORU", 0x334d },
  { "SQUARE YAADO", 0x334e },
  { "SQUARE YAARU", 0x334f },
  { "SQUARE YUAN", 0x3350 },
  { "SQUARE RITTORU", 0x3351 },
  { "SQUARE RIRA", 0x3352 },
  { "SQUARE RUPII", 0x3353 },
  { "SQUARE RUUBURU", 0x3354 },
  { "SQUARE REMU", 0x3355 },
  { "SQUARE RENTOGEN", 0x3356 },
  { "SQUARE WATTO", 0x3357 },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR ZERO", 0x3358 },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR ONE", 0x3359 },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWO", 0x335a },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR THREE", 0x335b },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR FOUR", 0x335c },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR FIVE", 0x335d },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR SIX", 0x335e },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR SEVEN", 0x335f },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR EIGHT", 0x3360 },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR NINE", 0x3361 },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TEN", 0x3362 },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR ELEVEN", 0x3363 },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWELVE", 0x3364 },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR THIRTEEN", 0x3365 },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR FOURTEEN", 0x3366 },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR FIFTEEN", 0x3367 },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR SIXTEEN", 0x3368 },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR SEVENTEEN", 0x3369 },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR EIGHTEEN", 0x336a },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR NINETEEN", 0x336b },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWENTY", 0x336c },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWENTY-ONE", 0x336d },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWENTY-TWO", 0x336e },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWENTY-THREE", 0x336f },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWENTY-FOUR", 0x3370 },
  { "SQUARE HPA", 0x3371 },
  { "SQUARE DA", 0x3372 },
  { "SQUARE AU", 0x3373 },
  { "SQUARE BAR", 0x3374 },
  { "SQUARE OV", 0x3375 },
  { "SQUARE PC", 0x3376 },
  { "SQUARE ERA NAME HEISEI", 0x337b },
  { "SQUARE ERA NAME SYOUWA", 0x337c },
  { "SQUARE ERA NAME TAISYOU", 0x337d },
  { "SQUARE ERA NAME MEIZI", 0x337e },
  { "SQUARE CORPORATION", 0x337f },
  { "SQUARE PA AMPS", 0x3380 },
  { "SQUARE NA", 0x3381 },
  { "SQUARE MU A", 0x3382 },
  { "SQUARE MA", 0x3383 },
  { "SQUARE KA", 0x3384 },
  { "SQUARE KB", 0x3385 },
  { "SQUARE MB", 0x3386 },
  { "SQUARE GB", 0x3387 },
  { "SQUARE CAL", 0x3388 },
  { "SQUARE KCAL", 0x3389 },
  { "SQUARE PF", 0x338a },
  { "SQUARE NF", 0x338b },
  { "SQUARE MU F", 0x338c },
  { "SQUARE MU G", 0x338d },
  { "SQUARE MG", 0x338e },
  { "SQUARE KG", 0x338f },
  { "SQUARE HZ", 0x3390 },
  { "SQUARE KHZ", 0x3391 },
  { "SQUARE MHZ", 0x3392 },
  { "SQUARE GHZ", 0x3393 },
  { "SQUARE THZ", 0x3394 },
  { "SQUARE MU L", 0x3395 },
  { "SQUARE ML", 0x3396 },
  { "SQUARE DL", 0x3397 },
  { "SQUARE KL", 0x3398 },
  { "SQUARE FM", 0x3399 },
  { "SQUARE NM", 0x339a },
  { "SQUARE MU M", 0x339b },
  { "SQUARE MM", 0x339c },
  { "SQUARE CM", 0x339d },
  { "SQUARE KM", 0x339e },
  { "SQUARE MM SQUARED", 0x339f },
  { "SQUARE CM SQUARED", 0x33a0 },
  { "SQUARE M SQUARED", 0x33a1 },
  { "SQUARE KM SQUARED", 0x33a2 },
  { "SQUARE MM CUBED", 0x33a3 },
  { "SQUARE CM CUBED", 0x33a4 },
  { "SQUARE M CUBED", 0x33a5 },
  { "SQUARE KM CUBED", 0x33a6 },
  { "SQUARE M OVER S", 0x33a7 },
  { "SQUARE M OVER S SQUARED", 0x33a8 },
  { "SQUARE PA", 0x33a9 },
  { "SQUARE KPA", 0x33aa },
  { "SQUARE MPA", 0x33ab },
  { "SQUARE GPA", 0x33ac },
  { "SQUARE RAD", 0x33ad },
  { "SQUARE RAD OVER S", 0x33ae },
  { "SQUARE RAD OVER S SQUARED", 0x33af },
  { "SQUARE PS", 0x33b0 },
  { "SQUARE NS", 0x33b1 },
  { "SQUARE MU S", 0x33b2 },
  { "SQUARE MS", 0x33b3 },
  { "SQUARE PV", 0x33b4 },
  { "SQUARE NV", 0x33b5 },
  { "SQUARE MU V", 0x33b6 },
  { "SQUARE MV", 0x33b7 },
  { "SQUARE KV", 0x33b8 },
  { "SQUARE MV MEGA", 0x33b9 },
  { "SQUARE PW", 0x33ba },
  { "SQUARE NW", 0x33bb },
  { "SQUARE MU W", 0x33bc },
  { "SQUARE MW", 0x33bd },
  { "SQUARE KW", 0x33be },
  { "SQUARE MW MEGA", 0x33bf },
  { "SQUARE K OHM", 0x33c0 },
  { "SQUARE M OHM", 0x33c1 },
  { "SQUARE AM", 0x33c2 },
  { "SQUARE BQ", 0x33c3 },
  { "SQUARE CC", 0x33c4 },
  { "SQUARE CD", 0x33c5 },
  { "SQUARE C OVER KG", 0x33c6 },
  { "SQUARE CO", 0x33c7 },
  { "SQUARE DB", 0x33c8 },
  { "SQUARE GY", 0x33c9 },
  { "SQUARE HA", 0x33ca },
  { "SQUARE HP", 0x33cb },
  { "SQUARE IN", 0x33cc },
  { "SQUARE KK", 0x33cd },
  { "SQUARE KM CAPITAL", 0x33ce },
  { "SQUARE KT", 0x33cf },
  { "SQUARE LM", 0x33d0 },
  { "SQUARE LN", 0x33d1 },
  { "SQUARE LOG", 0x33d2 },
  { "SQUARE LX", 0x33d3 },
  { "SQUARE MB SMALL", 0x33d4 },
  { "SQUARE MIL", 0x33d5 },
  { "SQUARE MOL", 0x33d6 },
  { "SQUARE PH", 0x33d7 },
  { "SQUARE PM", 0x33d8 },
  { "SQUARE PPM", 0x33d9 },
  { "SQUARE PR", 0x33da },
  { "SQUARE SR", 0x33db },
  { "SQUARE SV", 0x33dc },
  { "SQUARE WB", 0x33dd },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY ONE", 0x33e0 },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWO", 0x33e1 },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY THREE", 0x33e2 },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY FOUR", 0x33e3 },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY FIVE", 0x33e4 },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY SIX", 0x33e5 },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY SEVEN", 0x33e6 },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY EIGHT", 0x33e7 },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY NINE", 0x33e8 },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TEN", 0x33e9 },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY ELEVEN", 0x33ea },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWELVE", 0x33eb },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY THIRTEEN", 0x33ec },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY FOURTEEN", 0x33ed },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY FIFTEEN", 0x33ee },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY SIXTEEN", 0x33ef },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY SEVENTEEN", 0x33f0 },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY EIGHTEEN", 0x33f1 },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY NINETEEN", 0x33f2 },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY", 0x33f3 },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-ONE", 0x33f4 },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-TWO", 0x33f5 },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-THREE", 0x33f6 },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-FOUR", 0x33f7 },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-FIVE", 0x33f8 },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-SIX", 0x33f9 },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-SEVEN", 0x33fa },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-EIGHT", 0x33fb },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-NINE", 0x33fc },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY THIRTY", 0x33fd },
  { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY THIRTY-ONE", 0x33fe },
  { "YI SYLLABLE IT", 0xa000 },
  { "YI SYLLABLE IX", 0xa001 },
  { "YI SYLLABLE I", 0xa002 },
  { "YI SYLLABLE IP", 0xa003 },
  { "YI SYLLABLE IET", 0xa004 },
  { "YI SYLLABLE IEX", 0xa005 },
  { "YI SYLLABLE IE", 0xa006 },
  { "YI SYLLABLE IEP", 0xa007 },
  { "YI SYLLABLE AT", 0xa008 },
  { "YI SYLLABLE AX", 0xa009 },
  { "YI SYLLABLE A", 0xa00a },
  { "YI SYLLABLE AP", 0xa00b },
  { "YI SYLLABLE UOX", 0xa00c },
  { "YI SYLLABLE UO", 0xa00d },
  { "YI SYLLABLE UOP", 0xa00e },
  { "YI SYLLABLE OT", 0xa00f },
  { "YI SYLLABLE OX", 0xa010 },
  { "YI SYLLABLE O", 0xa011 },
  { "YI SYLLABLE OP", 0xa012 },
  { "YI SYLLABLE EX", 0xa013 },
  { "YI SYLLABLE E", 0xa014 },
  { "YI SYLLABLE WU", 0xa015 },
  { "YI SYLLABLE BIT", 0xa016 },
  { "YI SYLLABLE BIX", 0xa017 },
  { "YI SYLLABLE BI", 0xa018 },
  { "YI SYLLABLE BIP", 0xa019 },
  { "YI SYLLABLE BIET", 0xa01a },
  { "YI SYLLABLE BIEX", 0xa01b },
  { "YI SYLLABLE BIE", 0xa01c },
  { "YI SYLLABLE BIEP", 0xa01d },
  { "YI SYLLABLE BAT", 0xa01e },
  { "YI SYLLABLE BAX", 0xa01f },
  { "YI SYLLABLE BA", 0xa020 },
  { "YI SYLLABLE BAP", 0xa021 },
  { "YI SYLLABLE BUOX", 0xa022 },
  { "YI SYLLABLE BUO", 0xa023 },
  { "YI SYLLABLE BUOP", 0xa024 },
  { "YI SYLLABLE BOT", 0xa025 },
  { "YI SYLLABLE BOX", 0xa026 },
  { "YI SYLLABLE BO", 0xa027 },
  { "YI SYLLABLE BOP", 0xa028 },
  { "YI SYLLABLE BEX", 0xa029 },
  { "YI SYLLABLE BE", 0xa02a },
  { "YI SYLLABLE BEP", 0xa02b },
  { "YI SYLLABLE BUT", 0xa02c },
  { "YI SYLLABLE BUX", 0xa02d },
  { "YI SYLLABLE BU", 0xa02e },
  { "YI SYLLABLE BUP", 0xa02f },
  { "YI SYLLABLE BURX", 0xa030 },
  { "YI SYLLABLE BUR", 0xa031 },
  { "YI SYLLABLE BYT", 0xa032 },
  { "YI SYLLABLE BYX", 0xa033 },
  { "YI SYLLABLE BY", 0xa034 },
  { "YI SYLLABLE BYP", 0xa035 },
  { "YI SYLLABLE BYRX", 0xa036 },
  { "YI SYLLABLE BYR", 0xa037 },
  { "YI SYLLABLE PIT", 0xa038 },
  { "YI SYLLABLE PIX", 0xa039 },
  { "YI SYLLABLE PI", 0xa03a },
  { "YI SYLLABLE PIP", 0xa03b },
  { "YI SYLLABLE PIEX", 0xa03c },
  { "YI SYLLABLE PIE", 0xa03d },
  { "YI SYLLABLE PIEP", 0xa03e },
  { "YI SYLLABLE PAT", 0xa03f },
  { "YI SYLLABLE PAX", 0xa040 },
  { "YI SYLLABLE PA", 0xa041 },
  { "YI SYLLABLE PAP", 0xa042 },
  { "YI SYLLABLE PUOX", 0xa043 },
  { "YI SYLLABLE PUO", 0xa044 },
  { "YI SYLLABLE PUOP", 0xa045 },
  { "YI SYLLABLE POT", 0xa046 },
  { "YI SYLLABLE POX", 0xa047 },
  { "YI SYLLABLE PO", 0xa048 },
  { "YI SYLLABLE POP", 0xa049 },
  { "YI SYLLABLE PUT", 0xa04a },
  { "YI SYLLABLE PUX", 0xa04b },
  { "YI SYLLABLE PU", 0xa04c },
  { "YI SYLLABLE PUP", 0xa04d },
  { "YI SYLLABLE PURX", 0xa04e },
  { "YI SYLLABLE PUR", 0xa04f },
  { "YI SYLLABLE PYT", 0xa050 },
  { "YI SYLLABLE PYX", 0xa051 },
  { "YI SYLLABLE PY", 0xa052 },
  { "YI SYLLABLE PYP", 0xa053 },
  { "YI SYLLABLE PYRX", 0xa054 },
  { "YI SYLLABLE PYR", 0xa055 },
  { "YI SYLLABLE BBIT", 0xa056 },
  { "YI SYLLABLE BBIX", 0xa057 },
  { "YI SYLLABLE BBI", 0xa058 },
  { "YI SYLLABLE BBIP", 0xa059 },
  { "YI SYLLABLE BBIET", 0xa05a },
  { "YI SYLLABLE BBIEX", 0xa05b },
  { "YI SYLLABLE BBIE", 0xa05c },
  { "YI SYLLABLE BBIEP", 0xa05d },
  { "YI SYLLABLE BBAT", 0xa05e },
  { "YI SYLLABLE BBAX", 0xa05f },
  { "YI SYLLABLE BBA", 0xa060 },
  { "YI SYLLABLE BBAP", 0xa061 },
  { "YI SYLLABLE BBUOX", 0xa062 },
  { "YI SYLLABLE BBUO", 0xa063 },
  { "YI SYLLABLE BBUOP", 0xa064 },
  { "YI SYLLABLE BBOT", 0xa065 },
  { "YI SYLLABLE BBOX", 0xa066 },
  { "YI SYLLABLE BBO", 0xa067 },
  { "YI SYLLABLE BBOP", 0xa068 },
  { "YI SYLLABLE BBEX", 0xa069 },
  { "YI SYLLABLE BBE", 0xa06a },
  { "YI SYLLABLE BBEP", 0xa06b },
  { "YI SYLLABLE BBUT", 0xa06c },
  { "YI SYLLABLE BBUX", 0xa06d },
  { "YI SYLLABLE BBU", 0xa06e },
  { "YI SYLLABLE BBUP", 0xa06f },
  { "YI SYLLABLE BBURX", 0xa070 },
  { "YI SYLLABLE BBUR", 0xa071 },
  { "YI SYLLABLE BBYT", 0xa072 },
  { "YI SYLLABLE BBYX", 0xa073 },
  { "YI SYLLABLE BBY", 0xa074 },
  { "YI SYLLABLE BBYP", 0xa075 },
  { "YI SYLLABLE NBIT", 0xa076 },
  { "YI SYLLABLE NBIX", 0xa077 },
  { "YI SYLLABLE NBI", 0xa078 },
  { "YI SYLLABLE NBIP", 0xa079 },
  { "YI SYLLABLE NBIEX", 0xa07a },
  { "YI SYLLABLE NBIE", 0xa07b },
  { "YI SYLLABLE NBIEP", 0xa07c },
  { "YI SYLLABLE NBAT", 0xa07d },
  { "YI SYLLABLE NBAX", 0xa07e },
  { "YI SYLLABLE NBA", 0xa07f },
  { "YI SYLLABLE NBAP", 0xa080 },
  { "YI SYLLABLE NBOT", 0xa081 },
  { "YI SYLLABLE NBOX", 0xa082 },
  { "YI SYLLABLE NBO", 0xa083 },
  { "YI SYLLABLE NBOP", 0xa084 },
  { "YI SYLLABLE NBUT", 0xa085 },
  { "YI SYLLABLE NBUX", 0xa086 },
  { "YI SYLLABLE NBU", 0xa087 },
  { "YI SYLLABLE NBUP", 0xa088 },
  { "YI SYLLABLE NBURX", 0xa089 },
  { "YI SYLLABLE NBUR", 0xa08a },
  { "YI SYLLABLE NBYT", 0xa08b },
  { "YI SYLLABLE NBYX", 0xa08c },
  { "YI SYLLABLE NBY", 0xa08d },
  { "YI SYLLABLE NBYP", 0xa08e },
  { "YI SYLLABLE NBYRX", 0xa08f },
  { "YI SYLLABLE NBYR", 0xa090 },
  { "YI SYLLABLE HMIT", 0xa091 },
  { "YI SYLLABLE HMIX", 0xa092 },
  { "YI SYLLABLE HMI", 0xa093 },
  { "YI SYLLABLE HMIP", 0xa094 },
  { "YI SYLLABLE HMIEX", 0xa095 },
  { "YI SYLLABLE HMIE", 0xa096 },
  { "YI SYLLABLE HMIEP", 0xa097 },
  { "YI SYLLABLE HMAT", 0xa098 },
  { "YI SYLLABLE HMAX", 0xa099 },
  { "YI SYLLABLE HMA", 0xa09a },
  { "YI SYLLABLE HMAP", 0xa09b },
  { "YI SYLLABLE HMUOX", 0xa09c },
  { "YI SYLLABLE HMUO", 0xa09d },
  { "YI SYLLABLE HMUOP", 0xa09e },
  { "YI SYLLABLE HMOT", 0xa09f },
  { "YI SYLLABLE HMOX", 0xa0a0 },
  { "YI SYLLABLE HMO", 0xa0a1 },
  { "YI SYLLABLE HMOP", 0xa0a2 },
  { "YI SYLLABLE HMUT", 0xa0a3 },
  { "YI SYLLABLE HMUX", 0xa0a4 },
  { "YI SYLLABLE HMU", 0xa0a5 },
  { "YI SYLLABLE HMUP", 0xa0a6 },
  { "YI SYLLABLE HMURX", 0xa0a7 },
  { "YI SYLLABLE HMUR", 0xa0a8 },
  { "YI SYLLABLE HMYX", 0xa0a9 },
  { "YI SYLLABLE HMY", 0xa0aa },
  { "YI SYLLABLE HMYP", 0xa0ab },
  { "YI SYLLABLE HMYRX", 0xa0ac },
  { "YI SYLLABLE HMYR", 0xa0ad },
  { "YI SYLLABLE MIT", 0xa0ae },
  { "YI SYLLABLE MIX", 0xa0af },
  { "YI SYLLABLE MI", 0xa0b0 },
  { "YI SYLLABLE MIP", 0xa0b1 },
  { "YI SYLLABLE MIEX", 0xa0b2 },
  { "YI SYLLABLE MIE", 0xa0b3 },
  { "YI SYLLABLE MIEP", 0xa0b4 },
  { "YI SYLLABLE MAT", 0xa0b5 },
  { "YI SYLLABLE MAX", 0xa0b6 },
  { "YI SYLLABLE MA", 0xa0b7 },
  { "YI SYLLABLE MAP", 0xa0b8 },
  { "YI SYLLABLE MUOT", 0xa0b9 },
  { "YI SYLLABLE MUOX", 0xa0ba },
  { "YI SYLLABLE MUO", 0xa0bb },
  { "YI SYLLABLE MUOP", 0xa0bc },
  { "YI SYLLABLE MOT", 0xa0bd },
  { "YI SYLLABLE MOX", 0xa0be },
  { "YI SYLLABLE MO", 0xa0bf },
  { "YI SYLLABLE MOP", 0xa0c0 },
  { "YI SYLLABLE MEX", 0xa0c1 },
  { "YI SYLLABLE ME", 0xa0c2 },
  { "YI SYLLABLE MUT", 0xa0c3 },
  { "YI SYLLABLE MUX", 0xa0c4 },
  { "YI SYLLABLE MU", 0xa0c5 },
  { "YI SYLLABLE MUP", 0xa0c6 },
  { "YI SYLLABLE MURX", 0xa0c7 },
  { "YI SYLLABLE MUR", 0xa0c8 },
  { "YI SYLLABLE MYT", 0xa0c9 },
  { "YI SYLLABLE MYX", 0xa0ca },
  { "YI SYLLABLE MY", 0xa0cb },
  { "YI SYLLABLE MYP", 0xa0cc },
  { "YI SYLLABLE FIT", 0xa0cd },
  { "YI SYLLABLE FIX", 0xa0ce },
  { "YI SYLLABLE FI", 0xa0cf },
  { "YI SYLLABLE FIP", 0xa0d0 },
  { "YI SYLLABLE FAT", 0xa0d1 },
  { "YI SYLLABLE FAX", 0xa0d2 },
  { "YI SYLLABLE FA", 0xa0d3 },
  { "YI SYLLABLE FAP", 0xa0d4 },
  { "YI SYLLABLE FOX", 0xa0d5 },
  { "YI SYLLABLE FO", 0xa0d6 },
  { "YI SYLLABLE FOP", 0xa0d7 },
  { "YI SYLLABLE FUT", 0xa0d8 },
  { "YI SYLLABLE FUX", 0xa0d9 },
  { "YI SYLLABLE FU", 0xa0da },
  { "YI SYLLABLE FUP", 0xa0db },
  { "YI SYLLABLE FURX", 0xa0dc },
  { "YI SYLLABLE FUR", 0xa0dd },
  { "YI SYLLABLE FYT", 0xa0de },
  { "YI SYLLABLE FYX", 0xa0df },
  { "YI SYLLABLE FY", 0xa0e0 },
  { "YI SYLLABLE FYP", 0xa0e1 },
  { "YI SYLLABLE VIT", 0xa0e2 },
  { "YI SYLLABLE VIX", 0xa0e3 },
  { "YI SYLLABLE VI", 0xa0e4 },
  { "YI SYLLABLE VIP", 0xa0e5 },
  { "YI SYLLABLE VIET", 0xa0e6 },
  { "YI SYLLABLE VIEX", 0xa0e7 },
  { "YI SYLLABLE VIE", 0xa0e8 },
  { "YI SYLLABLE VIEP", 0xa0e9 },
  { "YI SYLLABLE VAT", 0xa0ea },
  { "YI SYLLABLE VAX", 0xa0eb },
  { "YI SYLLABLE VA", 0xa0ec },
  { "YI SYLLABLE VAP", 0xa0ed },
  { "YI SYLLABLE VOT", 0xa0ee },
  { "YI SYLLABLE VOX", 0xa0ef },
  { "YI SYLLABLE VO", 0xa0f0 },
  { "YI SYLLABLE VOP", 0xa0f1 },
  { "YI SYLLABLE VEX", 0xa0f2 },
  { "YI SYLLABLE VEP", 0xa0f3 },
  { "YI SYLLABLE VUT", 0xa0f4 },
  { "YI SYLLABLE VUX", 0xa0f5 },
  { "YI SYLLABLE VU", 0xa0f6 },
  { "YI SYLLABLE VUP", 0xa0f7 },
  { "YI SYLLABLE VURX", 0xa0f8 },
  { "YI SYLLABLE VUR", 0xa0f9 },
  { "YI SYLLABLE VYT", 0xa0fa },
  { "YI SYLLABLE VYX", 0xa0fb },
  { "YI SYLLABLE VY", 0xa0fc },
  { "YI SYLLABLE VYP", 0xa0fd },
  { "YI SYLLABLE VYRX", 0xa0fe },
  { "YI SYLLABLE VYR", 0xa0ff },
  { "YI SYLLABLE DIT", 0xa100 },
  { "YI SYLLABLE DIX", 0xa101 },
  { "YI SYLLABLE DI", 0xa102 },
  { "YI SYLLABLE DIP", 0xa103 },
  { "YI SYLLABLE DIEX", 0xa104 },
  { "YI SYLLABLE DIE", 0xa105 },
  { "YI SYLLABLE DIEP", 0xa106 },
  { "YI SYLLABLE DAT", 0xa107 },
  { "YI SYLLABLE DAX", 0xa108 },
  { "YI SYLLABLE DA", 0xa109 },
  { "YI SYLLABLE DAP", 0xa10a },
  { "YI SYLLABLE DUOX", 0xa10b },
  { "YI SYLLABLE DUO", 0xa10c },
  { "YI SYLLABLE DOT", 0xa10d },
  { "YI SYLLABLE DOX", 0xa10e },
  { "YI SYLLABLE DO", 0xa10f },
  { "YI SYLLABLE DOP", 0xa110 },
  { "YI SYLLABLE DEX", 0xa111 },
  { "YI SYLLABLE DE", 0xa112 },
  { "YI SYLLABLE DEP", 0xa113 },
  { "YI SYLLABLE DUT", 0xa114 },
  { "YI SYLLABLE DUX", 0xa115 },
  { "YI SYLLABLE DU", 0xa116 },
  { "YI SYLLABLE DUP", 0xa117 },
  { "YI SYLLABLE DURX", 0xa118 },
  { "YI SYLLABLE DUR", 0xa119 },
  { "YI SYLLABLE TIT", 0xa11a },
  { "YI SYLLABLE TIX", 0xa11b },
  { "YI SYLLABLE TI", 0xa11c },
  { "YI SYLLABLE TIP", 0xa11d },
  { "YI SYLLABLE TIEX", 0xa11e },
  { "YI SYLLABLE TIE", 0xa11f },
  { "YI SYLLABLE TIEP", 0xa120 },
  { "YI SYLLABLE TAT", 0xa121 },
  { "YI SYLLABLE TAX", 0xa122 },
  { "YI SYLLABLE TA", 0xa123 },
  { "YI SYLLABLE TAP", 0xa124 },
  { "YI SYLLABLE TUOT", 0xa125 },
  { "YI SYLLABLE TUOX", 0xa126 },
  { "YI SYLLABLE TUO", 0xa127 },
  { "YI SYLLABLE TUOP", 0xa128 },
  { "YI SYLLABLE TOT", 0xa129 },
  { "YI SYLLABLE TOX", 0xa12a },
  { "YI SYLLABLE TO", 0xa12b },
  { "YI SYLLABLE TOP", 0xa12c },
  { "YI SYLLABLE TEX", 0xa12d },
  { "YI SYLLABLE TE", 0xa12e },
  { "YI SYLLABLE TEP", 0xa12f },
  { "YI SYLLABLE TUT", 0xa130 },
  { "YI SYLLABLE TUX", 0xa131 },
  { "YI SYLLABLE TU", 0xa132 },
  { "YI SYLLABLE TUP", 0xa133 },
  { "YI SYLLABLE TURX", 0xa134 },
  { "YI SYLLABLE TUR", 0xa135 },
  { "YI SYLLABLE DDIT", 0xa136 },
  { "YI SYLLABLE DDIX", 0xa137 },
  { "YI SYLLABLE DDI", 0xa138 },
  { "YI SYLLABLE DDIP", 0xa139 },
  { "YI SYLLABLE DDIEX", 0xa13a },
  { "YI SYLLABLE DDIE", 0xa13b },
  { "YI SYLLABLE DDIEP", 0xa13c },
  { "YI SYLLABLE DDAT", 0xa13d },
  { "YI SYLLABLE DDAX", 0xa13e },
  { "YI SYLLABLE DDA", 0xa13f },
  { "YI SYLLABLE DDAP", 0xa140 },
  { "YI SYLLABLE DDUOX", 0xa141 },
  { "YI SYLLABLE DDUO", 0xa142 },
  { "YI SYLLABLE DDUOP", 0xa143 },
  { "YI SYLLABLE DDOT", 0xa144 },
  { "YI SYLLABLE DDOX", 0xa145 },
  { "YI SYLLABLE DDO", 0xa146 },
  { "YI SYLLABLE DDOP", 0xa147 },
  { "YI SYLLABLE DDEX", 0xa148 },
  { "YI SYLLABLE DDE", 0xa149 },
  { "YI SYLLABLE DDEP", 0xa14a },
  { "YI SYLLABLE DDUT", 0xa14b },
  { "YI SYLLABLE DDUX", 0xa14c },
  { "YI SYLLABLE DDU", 0xa14d },
  { "YI SYLLABLE DDUP", 0xa14e },
  { "YI SYLLABLE DDURX", 0xa14f },
  { "YI SYLLABLE DDUR", 0xa150 },
  { "YI SYLLABLE NDIT", 0xa151 },
  { "YI SYLLABLE NDIX", 0xa152 },
  { "YI SYLLABLE NDI", 0xa153 },
  { "YI SYLLABLE NDIP", 0xa154 },
  { "YI SYLLABLE NDIEX", 0xa155 },
  { "YI SYLLABLE NDIE", 0xa156 },
  { "YI SYLLABLE NDAT", 0xa157 },
  { "YI SYLLABLE NDAX", 0xa158 },
  { "YI SYLLABLE NDA", 0xa159 },
  { "YI SYLLABLE NDAP", 0xa15a },
  { "YI SYLLABLE NDOT", 0xa15b },
  { "YI SYLLABLE NDOX", 0xa15c },
  { "YI SYLLABLE NDO", 0xa15d },
  { "YI SYLLABLE NDOP", 0xa15e },
  { "YI SYLLABLE NDEX", 0xa15f },
  { "YI SYLLABLE NDE", 0xa160 },
  { "YI SYLLABLE NDEP", 0xa161 },
  { "YI SYLLABLE NDUT", 0xa162 },
  { "YI SYLLABLE NDUX", 0xa163 },
  { "YI SYLLABLE NDU", 0xa164 },
  { "YI SYLLABLE NDUP", 0xa165 },
  { "YI SYLLABLE NDURX", 0xa166 },
  { "YI SYLLABLE NDUR", 0xa167 },
  { "YI SYLLABLE HNIT", 0xa168 },
  { "YI SYLLABLE HNIX", 0xa169 },
  { "YI SYLLABLE HNI", 0xa16a },
  { "YI SYLLABLE HNIP", 0xa16b },
  { "YI SYLLABLE HNIET", 0xa16c },
  { "YI SYLLABLE HNIEX", 0xa16d },
  { "YI SYLLABLE HNIE", 0xa16e },
  { "YI SYLLABLE HNIEP", 0xa16f },
  { "YI SYLLABLE HNAT", 0xa170 },
  { "YI SYLLABLE HNAX", 0xa171 },
  { "YI SYLLABLE HNA", 0xa172 },
  { "YI SYLLABLE HNAP", 0xa173 },
  { "YI SYLLABLE HNUOX", 0xa174 },
  { "YI SYLLABLE HNUO", 0xa175 },
  { "YI SYLLABLE HNOT", 0xa176 },
  { "YI SYLLABLE HNOX", 0xa177 },
  { "YI SYLLABLE HNOP", 0xa178 },
  { "YI SYLLABLE HNEX", 0xa179 },
  { "YI SYLLABLE HNE", 0xa17a },
  { "YI SYLLABLE HNEP", 0xa17b },
  { "YI SYLLABLE HNUT", 0xa17c },
  { "YI SYLLABLE NIT", 0xa17d },
  { "YI SYLLABLE NIX", 0xa17e },
  { "YI SYLLABLE NI", 0xa17f },
  { "YI SYLLABLE NIP", 0xa180 },
  { "YI SYLLABLE NIEX", 0xa181 },
  { "YI SYLLABLE NIE", 0xa182 },
  { "YI SYLLABLE NIEP", 0xa183 },
  { "YI SYLLABLE NAX", 0xa184 },
  { "YI SYLLABLE NA", 0xa185 },
  { "YI SYLLABLE NAP", 0xa186 },
  { "YI SYLLABLE NUOX", 0xa187 },
  { "YI SYLLABLE NUO", 0xa188 },
  { "YI SYLLABLE NUOP", 0xa189 },
  { "YI SYLLABLE NOT", 0xa18a },
  { "YI SYLLABLE NOX", 0xa18b },
  { "YI SYLLABLE NO", 0xa18c },
  { "YI SYLLABLE NOP", 0xa18d },
  { "YI SYLLABLE NEX", 0xa18e },
  { "YI SYLLABLE NE", 0xa18f },
  { "YI SYLLABLE NEP", 0xa190 },
  { "YI SYLLABLE NUT", 0xa191 },
  { "YI SYLLABLE NUX", 0xa192 },
  { "YI SYLLABLE NU", 0xa193 },
  { "YI SYLLABLE NUP", 0xa194 },
  { "YI SYLLABLE NURX", 0xa195 },
  { "YI SYLLABLE NUR", 0xa196 },
  { "YI SYLLABLE HLIT", 0xa197 },
  { "YI SYLLABLE HLIX", 0xa198 },
  { "YI SYLLABLE HLI", 0xa199 },
  { "YI SYLLABLE HLIP", 0xa19a },
  { "YI SYLLABLE HLIEX", 0xa19b },
  { "YI SYLLABLE HLIE", 0xa19c },
  { "YI SYLLABLE HLIEP", 0xa19d },
  { "YI SYLLABLE HLAT", 0xa19e },
  { "YI SYLLABLE HLAX", 0xa19f },
  { "YI SYLLABLE HLA", 0xa1a0 },
  { "YI SYLLABLE HLAP", 0xa1a1 },
  { "YI SYLLABLE HLUOX", 0xa1a2 },
  { "YI SYLLABLE HLUO", 0xa1a3 },
  { "YI SYLLABLE HLUOP", 0xa1a4 },
  { "YI SYLLABLE HLOX", 0xa1a5 },
  { "YI SYLLABLE HLO", 0xa1a6 },
  { "YI SYLLABLE HLOP", 0xa1a7 },
  { "YI SYLLABLE HLEX", 0xa1a8 },
  { "YI SYLLABLE HLE", 0xa1a9 },
  { "YI SYLLABLE HLEP", 0xa1aa },
  { "YI SYLLABLE HLUT", 0xa1ab },
  { "YI SYLLABLE HLUX", 0xa1ac },
  { "YI SYLLABLE HLU", 0xa1ad },
  { "YI SYLLABLE HLUP", 0xa1ae },
  { "YI SYLLABLE HLURX", 0xa1af },
  { "YI SYLLABLE HLUR", 0xa1b0 },
  { "YI SYLLABLE HLYT", 0xa1b1 },
  { "YI SYLLABLE HLYX", 0xa1b2 },
  { "YI SYLLABLE HLY", 0xa1b3 },
  { "YI SYLLABLE HLYP", 0xa1b4 },
  { "YI SYLLABLE HLYRX", 0xa1b5 },
  { "YI SYLLABLE HLYR", 0xa1b6 },
  { "YI SYLLABLE LIT", 0xa1b7 },
  { "YI SYLLABLE LIX", 0xa1b8 },
  { "YI SYLLABLE LI", 0xa1b9 },
  { "YI SYLLABLE LIP", 0xa1ba },
  { "YI SYLLABLE LIET", 0xa1bb },
  { "YI SYLLABLE LIEX", 0xa1bc },
  { "YI SYLLABLE LIE", 0xa1bd },
  { "YI SYLLABLE LIEP", 0xa1be },
  { "YI SYLLABLE LAT", 0xa1bf },
  { "YI SYLLABLE LAX", 0xa1c0 },
  { "YI SYLLABLE LA", 0xa1c1 },
  { "YI SYLLABLE LAP", 0xa1c2 },
  { "YI SYLLABLE LUOT", 0xa1c3 },
  { "YI SYLLABLE LUOX", 0xa1c4 },
  { "YI SYLLABLE LUO", 0xa1c5 },
  { "YI SYLLABLE LUOP", 0xa1c6 },
  { "YI SYLLABLE LOT", 0xa1c7 },
  { "YI SYLLABLE LOX", 0xa1c8 },
  { "YI SYLLABLE LO", 0xa1c9 },
  { "YI SYLLABLE LOP", 0xa1ca },
  { "YI SYLLABLE LEX", 0xa1cb },
  { "YI SYLLABLE LE", 0xa1cc },
  { "YI SYLLABLE LEP", 0xa1cd },
  { "YI SYLLABLE LUT", 0xa1ce },
  { "YI SYLLABLE LUX", 0xa1cf },
  { "YI SYLLABLE LU", 0xa1d0 },
  { "YI SYLLABLE LUP", 0xa1d1 },
  { "YI SYLLABLE LURX", 0xa1d2 },
  { "YI SYLLABLE LUR", 0xa1d3 },
  { "YI SYLLABLE LYT", 0xa1d4 },
  { "YI SYLLABLE LYX", 0xa1d5 },
  { "YI SYLLABLE LY", 0xa1d6 },
  { "YI SYLLABLE LYP", 0xa1d7 },
  { "YI SYLLABLE LYRX", 0xa1d8 },
  { "YI SYLLABLE LYR", 0xa1d9 },
  { "YI SYLLABLE GIT", 0xa1da },
  { "YI SYLLABLE GIX", 0xa1db },
  { "YI SYLLABLE GI", 0xa1dc },
  { "YI SYLLABLE GIP", 0xa1dd },
  { "YI SYLLABLE GIET", 0xa1de },
  { "YI SYLLABLE GIEX", 0xa1df },
  { "YI SYLLABLE GIE", 0xa1e0 },
  { "YI SYLLABLE GIEP", 0xa1e1 },
  { "YI SYLLABLE GAT", 0xa1e2 },
  { "YI SYLLABLE GAX", 0xa1e3 },
  { "YI SYLLABLE GA", 0xa1e4 },
  { "YI SYLLABLE GAP", 0xa1e5 },
  { "YI SYLLABLE GUOT", 0xa1e6 },
  { "YI SYLLABLE GUOX", 0xa1e7 },
  { "YI SYLLABLE GUO", 0xa1e8 },
  { "YI SYLLABLE GUOP", 0xa1e9 },
  { "YI SYLLABLE GOT", 0xa1ea },
  { "YI SYLLABLE GOX", 0xa1eb },
  { "YI SYLLABLE GO", 0xa1ec },
  { "YI SYLLABLE GOP", 0xa1ed },
  { "YI SYLLABLE GET", 0xa1ee },
  { "YI SYLLABLE GEX", 0xa1ef },
  { "YI SYLLABLE GE", 0xa1f0 },
  { "YI SYLLABLE GEP", 0xa1f1 },
  { "YI SYLLABLE GUT", 0xa1f2 },
  { "YI SYLLABLE GUX", 0xa1f3 },
  { "YI SYLLABLE GU", 0xa1f4 },
  { "YI SYLLABLE GUP", 0xa1f5 },
  { "YI SYLLABLE GURX", 0xa1f6 },
  { "YI SYLLABLE GUR", 0xa1f7 },
  { "YI SYLLABLE KIT", 0xa1f8 },
  { "YI SYLLABLE KIX", 0xa1f9 },
  { "YI SYLLABLE KI", 0xa1fa },
  { "YI SYLLABLE KIP", 0xa1fb },
  { "YI SYLLABLE KIEX", 0xa1fc },
  { "YI SYLLABLE KIE", 0xa1fd },
  { "YI SYLLABLE KIEP", 0xa1fe },
  { "YI SYLLABLE KAT", 0xa1ff },
  { "YI SYLLABLE KAX", 0xa200 },
  { "YI SYLLABLE KA", 0xa201 },
  { "YI SYLLABLE KAP", 0xa202 },
  { "YI SYLLABLE KUOX", 0xa203 },
  { "YI SYLLABLE KUO", 0xa204 },
  { "YI SYLLABLE KUOP", 0xa205 },
  { "YI SYLLABLE KOT", 0xa206 },
  { "YI SYLLABLE KOX", 0xa207 },
  { "YI SYLLABLE KO", 0xa208 },
  { "YI SYLLABLE KOP", 0xa209 },
  { "YI SYLLABLE KET", 0xa20a },
  { "YI SYLLABLE KEX", 0xa20b },
  { "YI SYLLABLE KE", 0xa20c },
  { "YI SYLLABLE KEP", 0xa20d },
  { "YI SYLLABLE KUT", 0xa20e },
  { "YI SYLLABLE KUX", 0xa20f },
  { "YI SYLLABLE KU", 0xa210 },
  { "YI SYLLABLE KUP", 0xa211 },
  { "YI SYLLABLE KURX", 0xa212 },
  { "YI SYLLABLE KUR", 0xa213 },
  { "YI SYLLABLE GGIT", 0xa214 },
  { "YI SYLLABLE GGIX", 0xa215 },
  { "YI SYLLABLE GGI", 0xa216 },
  { "YI SYLLABLE GGIEX", 0xa217 },
  { "YI SYLLABLE GGIE", 0xa218 },
  { "YI SYLLABLE GGIEP", 0xa219 },
  { "YI SYLLABLE GGAT", 0xa21a },
  { "YI SYLLABLE GGAX", 0xa21b },
  { "YI SYLLABLE GGA", 0xa21c },
  { "YI SYLLABLE GGAP", 0xa21d },
  { "YI SYLLABLE GGUOT", 0xa21e },
  { "YI SYLLABLE GGUOX", 0xa21f },
  { "YI SYLLABLE GGUO", 0xa220 },
  { "YI SYLLABLE GGUOP", 0xa221 },
  { "YI SYLLABLE GGOT", 0xa222 },
  { "YI SYLLABLE GGOX", 0xa223 },
  { "YI SYLLABLE GGO", 0xa224 },
  { "YI SYLLABLE GGOP", 0xa225 },
  { "YI SYLLABLE GGET", 0xa226 },
  { "YI SYLLABLE GGEX", 0xa227 },
  { "YI SYLLABLE GGE", 0xa228 },
  { "YI SYLLABLE GGEP", 0xa229 },
  { "YI SYLLABLE GGUT", 0xa22a },
  { "YI SYLLABLE GGUX", 0xa22b },
  { "YI SYLLABLE GGU", 0xa22c },
  { "YI SYLLABLE GGUP", 0xa22d },
  { "YI SYLLABLE GGURX", 0xa22e },
  { "YI SYLLABLE GGUR", 0xa22f },
  { "YI SYLLABLE MGIEX", 0xa230 },
  { "YI SYLLABLE MGIE", 0xa231 },
  { "YI SYLLABLE MGAT", 0xa232 },
  { "YI SYLLABLE MGAX", 0xa233 },
  { "YI SYLLABLE MGA", 0xa234 },
  { "YI SYLLABLE MGAP", 0xa235 },
  { "YI SYLLABLE MGUOX", 0xa236 },
  { "YI SYLLABLE MGUO", 0xa237 },
  { "YI SYLLABLE MGUOP", 0xa238 },
  { "YI SYLLABLE MGOT", 0xa239 },
  { "YI SYLLABLE MGOX", 0xa23a },
  { "YI SYLLABLE MGO", 0xa23b },
  { "YI SYLLABLE MGOP", 0xa23c },
  { "YI SYLLABLE MGEX", 0xa23d },
  { "YI SYLLABLE MGE", 0xa23e },
  { "YI SYLLABLE MGEP", 0xa23f },
  { "YI SYLLABLE MGUT", 0xa240 },
  { "YI SYLLABLE MGUX", 0xa241 },
  { "YI SYLLABLE MGU", 0xa242 },
  { "YI SYLLABLE MGUP", 0xa243 },
  { "YI SYLLABLE MGURX", 0xa244 },
  { "YI SYLLABLE MGUR", 0xa245 },
  { "YI SYLLABLE HXIT", 0xa246 },
  { "YI SYLLABLE HXIX", 0xa247 },
  { "YI SYLLABLE HXI", 0xa248 },
  { "YI SYLLABLE HXIP", 0xa249 },
  { "YI SYLLABLE HXIET", 0xa24a },
  { "YI SYLLABLE HXIEX", 0xa24b },
  { "YI SYLLABLE HXIE", 0xa24c },
  { "YI SYLLABLE HXIEP", 0xa24d },
  { "YI SYLLABLE HXAT", 0xa24e },
  { "YI SYLLABLE HXAX", 0xa24f },
  { "YI SYLLABLE HXA", 0xa250 },
  { "YI SYLLABLE HXAP", 0xa251 },
  { "YI SYLLABLE HXUOT", 0xa252 },
  { "YI SYLLABLE HXUOX", 0xa253 },
  { "YI SYLLABLE HXUO", 0xa254 },
  { "YI SYLLABLE HXUOP", 0xa255 },
  { "YI SYLLABLE HXOT", 0xa256 },
  { "YI SYLLABLE HXOX", 0xa257 },
  { "YI SYLLABLE HXO", 0xa258 },
  { "YI SYLLABLE HXOP", 0xa259 },
  { "YI SYLLABLE HXEX", 0xa25a },
  { "YI SYLLABLE HXE", 0xa25b },
  { "YI SYLLABLE HXEP", 0xa25c },
  { "YI SYLLABLE NGIEX", 0xa25d },
  { "YI SYLLABLE NGIE", 0xa25e },
  { "YI SYLLABLE NGIEP", 0xa25f },
  { "YI SYLLABLE NGAT", 0xa260 },
  { "YI SYLLABLE NGAX", 0xa261 },
  { "YI SYLLABLE NGA", 0xa262 },
  { "YI SYLLABLE NGAP", 0xa263 },
  { "YI SYLLABLE NGUOT", 0xa264 },
  { "YI SYLLABLE NGUOX", 0xa265 },
  { "YI SYLLABLE NGUO", 0xa266 },
  { "YI SYLLABLE NGOT", 0xa267 },
  { "YI SYLLABLE NGOX", 0xa268 },
  { "YI SYLLABLE NGO", 0xa269 },
  { "YI SYLLABLE NGOP", 0xa26a },
  { "YI SYLLABLE NGEX", 0xa26b },
  { "YI SYLLABLE NGE", 0xa26c },
  { "YI SYLLABLE NGEP", 0xa26d },
  { "YI SYLLABLE HIT", 0xa26e },
  { "YI SYLLABLE HIEX", 0xa26f },
  { "YI SYLLABLE HIE", 0xa270 },
  { "YI SYLLABLE HAT", 0xa271 },
  { "YI SYLLABLE HAX", 0xa272 },
  { "YI SYLLABLE HA", 0xa273 },
  { "YI SYLLABLE HAP", 0xa274 },
  { "YI SYLLABLE HUOT", 0xa275 },
  { "YI SYLLABLE HUOX", 0xa276 },
  { "YI SYLLABLE HUO", 0xa277 },
  { "YI SYLLABLE HUOP", 0xa278 },
  { "YI SYLLABLE HOT", 0xa279 },
  { "YI SYLLABLE HOX", 0xa27a },
  { "YI SYLLABLE HO", 0xa27b },
  { "YI SYLLABLE HOP", 0xa27c },
  { "YI SYLLABLE HEX", 0xa27d },
  { "YI SYLLABLE HE", 0xa27e },
  { "YI SYLLABLE HEP", 0xa27f },
  { "YI SYLLABLE WAT", 0xa280 },
  { "YI SYLLABLE WAX", 0xa281 },
  { "YI SYLLABLE WA", 0xa282 },
  { "YI SYLLABLE WAP", 0xa283 },
  { "YI SYLLABLE WUOX", 0xa284 },
  { "YI SYLLABLE WUO", 0xa285 },
  { "YI SYLLABLE WUOP", 0xa286 },
  { "YI SYLLABLE WOX", 0xa287 },
  { "YI SYLLABLE WO", 0xa288 },
  { "YI SYLLABLE WOP", 0xa289 },
  { "YI SYLLABLE WEX", 0xa28a },
  { "YI SYLLABLE WE", 0xa28b },
  { "YI SYLLABLE WEP", 0xa28c },
  { "YI SYLLABLE ZIT", 0xa28d },
  { "YI SYLLABLE ZIX", 0xa28e },
  { "YI SYLLABLE ZI", 0xa28f },
  { "YI SYLLABLE ZIP", 0xa290 },
  { "YI SYLLABLE ZIEX", 0xa291 },
  { "YI SYLLABLE ZIE", 0xa292 },
  { "YI SYLLABLE ZIEP", 0xa293 },
  { "YI SYLLABLE ZAT", 0xa294 },
  { "YI SYLLABLE ZAX", 0xa295 },
  { "YI SYLLABLE ZA", 0xa296 },
  { "YI SYLLABLE ZAP", 0xa297 },
  { "YI SYLLABLE ZUOX", 0xa298 },
  { "YI SYLLABLE ZUO", 0xa299 },
  { "YI SYLLABLE ZUOP", 0xa29a },
  { "YI SYLLABLE ZOT", 0xa29b },
  { "YI SYLLABLE ZOX", 0xa29c },
  { "YI SYLLABLE ZO", 0xa29d },
  { "YI SYLLABLE ZOP", 0xa29e },
  { "YI SYLLABLE ZEX", 0xa29f },
  { "YI SYLLABLE ZE", 0xa2a0 },
  { "YI SYLLABLE ZEP", 0xa2a1 },
  { "YI SYLLABLE ZUT", 0xa2a2 },
  { "YI SYLLABLE ZUX", 0xa2a3 },
  { "YI SYLLABLE ZU", 0xa2a4 },
  { "YI SYLLABLE ZUP", 0xa2a5 },
  { "YI SYLLABLE ZURX", 0xa2a6 },
  { "YI SYLLABLE ZUR", 0xa2a7 },
  { "YI SYLLABLE ZYT", 0xa2a8 },
  { "YI SYLLABLE ZYX", 0xa2a9 },
  { "YI SYLLABLE ZY", 0xa2aa },
  { "YI SYLLABLE ZYP", 0xa2ab },
  { "YI SYLLABLE ZYRX", 0xa2ac },
  { "YI SYLLABLE ZYR", 0xa2ad },
  { "YI SYLLABLE CIT", 0xa2ae },
  { "YI SYLLABLE CIX", 0xa2af },
  { "YI SYLLABLE CI", 0xa2b0 },
  { "YI SYLLABLE CIP", 0xa2b1 },
  { "YI SYLLABLE CIET", 0xa2b2 },
  { "YI SYLLABLE CIEX", 0xa2b3 },
  { "YI SYLLABLE CIE", 0xa2b4 },
  { "YI SYLLABLE CIEP", 0xa2b5 },
  { "YI SYLLABLE CAT", 0xa2b6 },
  { "YI SYLLABLE CAX", 0xa2b7 },
  { "YI SYLLABLE CA", 0xa2b8 },
  { "YI SYLLABLE CAP", 0xa2b9 },
  { "YI SYLLABLE CUOX", 0xa2ba },
  { "YI SYLLABLE CUO", 0xa2bb },
  { "YI SYLLABLE CUOP", 0xa2bc },
  { "YI SYLLABLE COT", 0xa2bd },
  { "YI SYLLABLE COX", 0xa2be },
  { "YI SYLLABLE CO", 0xa2bf },
  { "YI SYLLABLE COP", 0xa2c0 },
  { "YI SYLLABLE CEX", 0xa2c1 },
  { "YI SYLLABLE CE", 0xa2c2 },
  { "YI SYLLABLE CEP", 0xa2c3 },
  { "YI SYLLABLE CUT", 0xa2c4 },
  { "YI SYLLABLE CUX", 0xa2c5 },
  { "YI SYLLABLE CU", 0xa2c6 },
  { "YI SYLLABLE CUP", 0xa2c7 },
  { "YI SYLLABLE CURX", 0xa2c8 },
  { "YI SYLLABLE CUR", 0xa2c9 },
  { "YI SYLLABLE CYT", 0xa2ca },
  { "YI SYLLABLE CYX", 0xa2cb },
  { "YI SYLLABLE CY", 0xa2cc },
  { "YI SYLLABLE CYP", 0xa2cd },
  { "YI SYLLABLE CYRX", 0xa2ce },
  { "YI SYLLABLE CYR", 0xa2cf },
  { "YI SYLLABLE ZZIT", 0xa2d0 },
  { "YI SYLLABLE ZZIX", 0xa2d1 },
  { "YI SYLLABLE ZZI", 0xa2d2 },
  { "YI SYLLABLE ZZIP", 0xa2d3 },
  { "YI SYLLABLE ZZIET", 0xa2d4 },
  { "YI SYLLABLE ZZIEX", 0xa2d5 },
  { "YI SYLLABLE ZZIE", 0xa2d6 },
  { "YI SYLLABLE ZZIEP", 0xa2d7 },
  { "YI SYLLABLE ZZAT", 0xa2d8 },
  { "YI SYLLABLE ZZAX", 0xa2d9 },
  { "YI SYLLABLE ZZA", 0xa2da },
  { "YI SYLLABLE ZZAP", 0xa2db },
  { "YI SYLLABLE ZZOX", 0xa2dc },
  { "YI SYLLABLE ZZO", 0xa2dd },
  { "YI SYLLABLE ZZOP", 0xa2de },
  { "YI SYLLABLE ZZEX", 0xa2df },
  { "YI SYLLABLE ZZE", 0xa2e0 },
  { "YI SYLLABLE ZZEP", 0xa2e1 },
  { "YI SYLLABLE ZZUX", 0xa2e2 },
  { "YI SYLLABLE ZZU", 0xa2e3 },
  { "YI SYLLABLE ZZUP", 0xa2e4 },
  { "YI SYLLABLE ZZURX", 0xa2e5 },
  { "YI SYLLABLE ZZUR", 0xa2e6 },
  { "YI SYLLABLE ZZYT", 0xa2e7 },
  { "YI SYLLABLE ZZYX", 0xa2e8 },
  { "YI SYLLABLE ZZY", 0xa2e9 },
  { "YI SYLLABLE ZZYP", 0xa2ea },
  { "YI SYLLABLE ZZYRX", 0xa2eb },
  { "YI SYLLABLE ZZYR", 0xa2ec },
  { "YI SYLLABLE NZIT", 0xa2ed },
  { "YI SYLLABLE NZIX", 0xa2ee },
  { "YI SYLLABLE NZI", 0xa2ef },
  { "YI SYLLABLE NZIP", 0xa2f0 },
  { "YI SYLLABLE NZIEX", 0xa2f1 },
  { "YI SYLLABLE NZIE", 0xa2f2 },
  { "YI SYLLABLE NZIEP", 0xa2f3 },
  { "YI SYLLABLE NZAT", 0xa2f4 },
  { "YI SYLLABLE NZAX", 0xa2f5 },
  { "YI SYLLABLE NZA", 0xa2f6 },
  { "YI SYLLABLE NZAP", 0xa2f7 },
  { "YI SYLLABLE NZUOX", 0xa2f8 },
  { "YI SYLLABLE NZUO", 0xa2f9 },
  { "YI SYLLABLE NZOX", 0xa2fa },
  { "YI SYLLABLE NZOP", 0xa2fb },
  { "YI SYLLABLE NZEX", 0xa2fc },
  { "YI SYLLABLE NZE", 0xa2fd },
  { "YI SYLLABLE NZUX", 0xa2fe },
  { "YI SYLLABLE NZU", 0xa2ff },
  { "YI SYLLABLE NZUP", 0xa300 },
  { "YI SYLLABLE NZURX", 0xa301 },
  { "YI SYLLABLE NZUR", 0xa302 },
  { "YI SYLLABLE NZYT", 0xa303 },
  { "YI SYLLABLE NZYX", 0xa304 },
  { "YI SYLLABLE NZY", 0xa305 },
  { "YI SYLLABLE NZYP", 0xa306 },
  { "YI SYLLABLE NZYRX", 0xa307 },
  { "YI SYLLABLE NZYR", 0xa308 },
  { "YI SYLLABLE SIT", 0xa309 },
  { "YI SYLLABLE SIX", 0xa30a },
  { "YI SYLLABLE SI", 0xa30b },
  { "YI SYLLABLE SIP", 0xa30c },
  { "YI SYLLABLE SIEX", 0xa30d },
  { "YI SYLLABLE SIE", 0xa30e },
  { "YI SYLLABLE SIEP", 0xa30f },
  { "YI SYLLABLE SAT", 0xa310 },
  { "YI SYLLABLE SAX", 0xa311 },
  { "YI SYLLABLE SA", 0xa312 },
  { "YI SYLLABLE SAP", 0xa313 },
  { "YI SYLLABLE SUOX", 0xa314 },
  { "YI SYLLABLE SUO", 0xa315 },
  { "YI SYLLABLE SUOP", 0xa316 },
  { "YI SYLLABLE SOT", 0xa317 },
  { "YI SYLLABLE SOX", 0xa318 },
  { "YI SYLLABLE SO", 0xa319 },
  { "YI SYLLABLE SOP", 0xa31a },
  { "YI SYLLABLE SEX", 0xa31b },
  { "YI SYLLABLE SE", 0xa31c },
  { "YI SYLLABLE SEP", 0xa31d },
  { "YI SYLLABLE SUT", 0xa31e },
  { "YI SYLLABLE SUX", 0xa31f },
  { "YI SYLLABLE SU", 0xa320 },
  { "YI SYLLABLE SUP", 0xa321 },
  { "YI SYLLABLE SURX", 0xa322 },
  { "YI SYLLABLE SUR", 0xa323 },
  { "YI SYLLABLE SYT", 0xa324 },
  { "YI SYLLABLE SYX", 0xa325 },
  { "YI SYLLABLE SY", 0xa326 },
  { "YI SYLLABLE SYP", 0xa327 },
  { "YI SYLLABLE SYRX", 0xa328 },
  { "YI SYLLABLE SYR", 0xa329 },
  { "YI SYLLABLE SSIT", 0xa32a },
  { "YI SYLLABLE SSIX", 0xa32b },
  { "YI SYLLABLE SSI", 0xa32c },
  { "YI SYLLABLE SSIP", 0xa32d },
  { "YI SYLLABLE SSIEX", 0xa32e },
  { "YI SYLLABLE SSIE", 0xa32f },
  { "YI SYLLABLE SSIEP", 0xa330 },
  { "YI SYLLABLE SSAT", 0xa331 },
  { "YI SYLLABLE SSAX", 0xa332 },
  { "YI SYLLABLE SSA", 0xa333 },
  { "YI SYLLABLE SSAP", 0xa334 },
  { "YI SYLLABLE SSOT", 0xa335 },
  { "YI SYLLABLE SSOX", 0xa336 },
  { "YI SYLLABLE SSO", 0xa337 },
  { "YI SYLLABLE SSOP", 0xa338 },
  { "YI SYLLABLE SSEX", 0xa339 },
  { "YI SYLLABLE SSE", 0xa33a },
  { "YI SYLLABLE SSEP", 0xa33b },
  { "YI SYLLABLE SSUT", 0xa33c },
  { "YI SYLLABLE SSUX", 0xa33d },
  { "YI SYLLABLE SSU", 0xa33e },
  { "YI SYLLABLE SSUP", 0xa33f },
  { "YI SYLLABLE SSYT", 0xa340 },
  { "YI SYLLABLE SSYX", 0xa341 },
  { "YI SYLLABLE SSY", 0xa342 },
  { "YI SYLLABLE SSYP", 0xa343 },
  { "YI SYLLABLE SSYRX", 0xa344 },
  { "YI SYLLABLE SSYR", 0xa345 },
  { "YI SYLLABLE ZHAT", 0xa346 },
  { "YI SYLLABLE ZHAX", 0xa347 },
  { "YI SYLLABLE ZHA", 0xa348 },
  { "YI SYLLABLE ZHAP", 0xa349 },
  { "YI SYLLABLE ZHUOX", 0xa34a },
  { "YI SYLLABLE ZHUO", 0xa34b },
  { "YI SYLLABLE ZHUOP", 0xa34c },
  { "YI SYLLABLE ZHOT", 0xa34d },
  { "YI SYLLABLE ZHOX", 0xa34e },
  { "YI SYLLABLE ZHO", 0xa34f },
  { "YI SYLLABLE ZHOP", 0xa350 },
  { "YI SYLLABLE ZHET", 0xa351 },
  { "YI SYLLABLE ZHEX", 0xa352 },
  { "YI SYLLABLE ZHE", 0xa353 },
  { "YI SYLLABLE ZHEP", 0xa354 },
  { "YI SYLLABLE ZHUT", 0xa355 },
  { "YI SYLLABLE ZHUX", 0xa356 },
  { "YI SYLLABLE ZHU", 0xa357 },
  { "YI SYLLABLE ZHUP", 0xa358 },
  { "YI SYLLABLE ZHURX", 0xa359 },
  { "YI SYLLABLE ZHUR", 0xa35a },
  { "YI SYLLABLE ZHYT", 0xa35b },
  { "YI SYLLABLE ZHYX", 0xa35c },
  { "YI SYLLABLE ZHY", 0xa35d },
  { "YI SYLLABLE ZHYP", 0xa35e },
  { "YI SYLLABLE ZHYRX", 0xa35f },
  { "YI SYLLABLE ZHYR", 0xa360 },
  { "YI SYLLABLE CHAT", 0xa361 },
  { "YI SYLLABLE CHAX", 0xa362 },
  { "YI SYLLABLE CHA", 0xa363 },
  { "YI SYLLABLE CHAP", 0xa364 },
  { "YI SYLLABLE CHUOT", 0xa365 },
  { "YI SYLLABLE CHUOX", 0xa366 },
  { "YI SYLLABLE CHUO", 0xa367 },
  { "YI SYLLABLE CHUOP", 0xa368 },
  { "YI SYLLABLE CHOT", 0xa369 },
  { "YI SYLLABLE CHOX", 0xa36a },
  { "YI SYLLABLE CHO", 0xa36b },
  { "YI SYLLABLE CHOP", 0xa36c },
  { "YI SYLLABLE CHET", 0xa36d },
  { "YI SYLLABLE CHEX", 0xa36e },
  { "YI SYLLABLE CHE", 0xa36f },
  { "YI SYLLABLE CHEP", 0xa370 },
  { "YI SYLLABLE CHUX", 0xa371 },
  { "YI SYLLABLE CHU", 0xa372 },
  { "YI SYLLABLE CHUP", 0xa373 },
  { "YI SYLLABLE CHURX", 0xa374 },
  { "YI SYLLABLE CHUR", 0xa375 },
  { "YI SYLLABLE CHYT", 0xa376 },
  { "YI SYLLABLE CHYX", 0xa377 },
  { "YI SYLLABLE CHY", 0xa378 },
  { "YI SYLLABLE CHYP", 0xa379 },
  { "YI SYLLABLE CHYRX", 0xa37a },
  { "YI SYLLABLE CHYR", 0xa37b },
  { "YI SYLLABLE RRAX", 0xa37c },
  { "YI SYLLABLE RRA", 0xa37d },
  { "YI SYLLABLE RRUOX", 0xa37e },
  { "YI SYLLABLE RRUO", 0xa37f },
  { "YI SYLLABLE RROT", 0xa380 },
  { "YI SYLLABLE RROX", 0xa381 },
  { "YI SYLLABLE RRO", 0xa382 },
  { "YI SYLLABLE RROP", 0xa383 },
  { "YI SYLLABLE RRET", 0xa384 },
  { "YI SYLLABLE RREX", 0xa385 },
  { "YI SYLLABLE RRE", 0xa386 },
  { "YI SYLLABLE RREP", 0xa387 },
  { "YI SYLLABLE RRUT", 0xa388 },
  { "YI SYLLABLE RRUX", 0xa389 },
  { "YI SYLLABLE RRU", 0xa38a },
  { "YI SYLLABLE RRUP", 0xa38b },
  { "YI SYLLABLE RRURX", 0xa38c },
  { "YI SYLLABLE RRUR", 0xa38d },
  { "YI SYLLABLE RRYT", 0xa38e },
  { "YI SYLLABLE RRYX", 0xa38f },
  { "YI SYLLABLE RRY", 0xa390 },
  { "YI SYLLABLE RRYP", 0xa391 },
  { "YI SYLLABLE RRYRX", 0xa392 },
  { "YI SYLLABLE RRYR", 0xa393 },
  { "YI SYLLABLE NRAT", 0xa394 },
  { "YI SYLLABLE NRAX", 0xa395 },
  { "YI SYLLABLE NRA", 0xa396 },
  { "YI SYLLABLE NRAP", 0xa397 },
  { "YI SYLLABLE NROX", 0xa398 },
  { "YI SYLLABLE NRO", 0xa399 },
  { "YI SYLLABLE NROP", 0xa39a },
  { "YI SYLLABLE NRET", 0xa39b },
  { "YI SYLLABLE NREX", 0xa39c },
  { "YI SYLLABLE NRE", 0xa39d },
  { "YI SYLLABLE NREP", 0xa39e },
  { "YI SYLLABLE NRUT", 0xa39f },
  { "YI SYLLABLE NRUX", 0xa3a0 },
  { "YI SYLLABLE NRU", 0xa3a1 },
  { "YI SYLLABLE NRUP", 0xa3a2 },
  { "YI SYLLABLE NRURX", 0xa3a3 },
  { "YI SYLLABLE NRUR", 0xa3a4 },
  { "YI SYLLABLE NRYT", 0xa3a5 },
  { "YI SYLLABLE NRYX", 0xa3a6 },
  { "YI SYLLABLE NRY", 0xa3a7 },
  { "YI SYLLABLE NRYP", 0xa3a8 },
  { "YI SYLLABLE NRYRX", 0xa3a9 },
  { "YI SYLLABLE NRYR", 0xa3aa },
  { "YI SYLLABLE SHAT", 0xa3ab },
  { "YI SYLLABLE SHAX", 0xa3ac },
  { "YI SYLLABLE SHA", 0xa3ad },
  { "YI SYLLABLE SHAP", 0xa3ae },
  { "YI SYLLABLE SHUOX", 0xa3af },
  { "YI SYLLABLE SHUO", 0xa3b0 },
  { "YI SYLLABLE SHUOP", 0xa3b1 },
  { "YI SYLLABLE SHOT", 0xa3b2 },
  { "YI SYLLABLE SHOX", 0xa3b3 },
  { "YI SYLLABLE SHO", 0xa3b4 },
  { "YI SYLLABLE SHOP", 0xa3b5 },
  { "YI SYLLABLE SHET", 0xa3b6 },
  { "YI SYLLABLE SHEX", 0xa3b7 },
  { "YI SYLLABLE SHE", 0xa3b8 },
  { "YI SYLLABLE SHEP", 0xa3b9 },
  { "YI SYLLABLE SHUT", 0xa3ba },
  { "YI SYLLABLE SHUX", 0xa3bb },
  { "YI SYLLABLE SHU", 0xa3bc },
  { "YI SYLLABLE SHUP", 0xa3bd },
  { "YI SYLLABLE SHURX", 0xa3be },
  { "YI SYLLABLE SHUR", 0xa3bf },
  { "YI SYLLABLE SHYT", 0xa3c0 },
  { "YI SYLLABLE SHYX", 0xa3c1 },
  { "YI SYLLABLE SHY", 0xa3c2 },
  { "YI SYLLABLE SHYP", 0xa3c3 },
  { "YI SYLLABLE SHYRX", 0xa3c4 },
  { "YI SYLLABLE SHYR", 0xa3c5 },
  { "YI SYLLABLE RAT", 0xa3c6 },
  { "YI SYLLABLE RAX", 0xa3c7 },
  { "YI SYLLABLE RA", 0xa3c8 },
  { "YI SYLLABLE RAP", 0xa3c9 },
  { "YI SYLLABLE RUOX", 0xa3ca },
  { "YI SYLLABLE RUO", 0xa3cb },
  { "YI SYLLABLE RUOP", 0xa3cc },
  { "YI SYLLABLE ROT", 0xa3cd },
  { "YI SYLLABLE ROX", 0xa3ce },
  { "YI SYLLABLE RO", 0xa3cf },
  { "YI SYLLABLE ROP", 0xa3d0 },
  { "YI SYLLABLE REX", 0xa3d1 },
  { "YI SYLLABLE RE", 0xa3d2 },
  { "YI SYLLABLE REP", 0xa3d3 },
  { "YI SYLLABLE RUT", 0xa3d4 },
  { "YI SYLLABLE RUX", 0xa3d5 },
  { "YI SYLLABLE RU", 0xa3d6 },
  { "YI SYLLABLE RUP", 0xa3d7 },
  { "YI SYLLABLE RURX", 0xa3d8 },
  { "YI SYLLABLE RUR", 0xa3d9 },
  { "YI SYLLABLE RYT", 0xa3da },
  { "YI SYLLABLE RYX", 0xa3db },
  { "YI SYLLABLE RY", 0xa3dc },
  { "YI SYLLABLE RYP", 0xa3dd },
  { "YI SYLLABLE RYRX", 0xa3de },
  { "YI SYLLABLE RYR", 0xa3df },
  { "YI SYLLABLE JIT", 0xa3e0 },
  { "YI SYLLABLE JIX", 0xa3e1 },
  { "YI SYLLABLE JI", 0xa3e2 },
  { "YI SYLLABLE JIP", 0xa3e3 },
  { "YI SYLLABLE JIET", 0xa3e4 },
  { "YI SYLLABLE JIEX", 0xa3e5 },
  { "YI SYLLABLE JIE", 0xa3e6 },
  { "YI SYLLABLE JIEP", 0xa3e7 },
  { "YI SYLLABLE JUOT", 0xa3e8 },
  { "YI SYLLABLE JUOX", 0xa3e9 },
  { "YI SYLLABLE JUO", 0xa3ea },
  { "YI SYLLABLE JUOP", 0xa3eb },
  { "YI SYLLABLE JOT", 0xa3ec },
  { "YI SYLLABLE JOX", 0xa3ed },
  { "YI SYLLABLE JO", 0xa3ee },
  { "YI SYLLABLE JOP", 0xa3ef },
  { "YI SYLLABLE JUT", 0xa3f0 },
  { "YI SYLLABLE JUX", 0xa3f1 },
  { "YI SYLLABLE JU", 0xa3f2 },
  { "YI SYLLABLE JUP", 0xa3f3 },
  { "YI SYLLABLE JURX", 0xa3f4 },
  { "YI SYLLABLE JUR", 0xa3f5 },
  { "YI SYLLABLE JYT", 0xa3f6 },
  { "YI SYLLABLE JYX", 0xa3f7 },
  { "YI SYLLABLE JY", 0xa3f8 },
  { "YI SYLLABLE JYP", 0xa3f9 },
  { "YI SYLLABLE JYRX", 0xa3fa },
  { "YI SYLLABLE JYR", 0xa3fb },
  { "YI SYLLABLE QIT", 0xa3fc },
  { "YI SYLLABLE QIX", 0xa3fd },
  { "YI SYLLABLE QI", 0xa3fe },
  { "YI SYLLABLE QIP", 0xa3ff },
  { "YI SYLLABLE QIET", 0xa400 },
  { "YI SYLLABLE QIEX", 0xa401 },
  { "YI SYLLABLE QIE", 0xa402 },
  { "YI SYLLABLE QIEP", 0xa403 },
  { "YI SYLLABLE QUOT", 0xa404 },
  { "YI SYLLABLE QUOX", 0xa405 },
  { "YI SYLLABLE QUO", 0xa406 },
  { "YI SYLLABLE QUOP", 0xa407 },
  { "YI SYLLABLE QOT", 0xa408 },
  { "YI SYLLABLE QOX", 0xa409 },
  { "YI SYLLABLE QO", 0xa40a },
  { "YI SYLLABLE QOP", 0xa40b },
  { "YI SYLLABLE QUT", 0xa40c },
  { "YI SYLLABLE QUX", 0xa40d },
  { "YI SYLLABLE QU", 0xa40e },
  { "YI SYLLABLE QUP", 0xa40f },
  { "YI SYLLABLE QURX", 0xa410 },
  { "YI SYLLABLE QUR", 0xa411 },
  { "YI SYLLABLE QYT", 0xa412 },
  { "YI SYLLABLE QYX", 0xa413 },
  { "YI SYLLABLE QY", 0xa414 },
  { "YI SYLLABLE QYP", 0xa415 },
  { "YI SYLLABLE QYRX", 0xa416 },
  { "YI SYLLABLE QYR", 0xa417 },
  { "YI SYLLABLE JJIT", 0xa418 },
  { "YI SYLLABLE JJIX", 0xa419 },
  { "YI SYLLABLE JJI", 0xa41a },
  { "YI SYLLABLE JJIP", 0xa41b },
  { "YI SYLLABLE JJIET", 0xa41c },
  { "YI SYLLABLE JJIEX", 0xa41d },
  { "YI SYLLABLE JJIE", 0xa41e },
  { "YI SYLLABLE JJIEP", 0xa41f },
  { "YI SYLLABLE JJUOX", 0xa420 },
  { "YI SYLLABLE JJUO", 0xa421 },
  { "YI SYLLABLE JJUOP", 0xa422 },
  { "YI SYLLABLE JJOT", 0xa423 },
  { "YI SYLLABLE JJOX", 0xa424 },
  { "YI SYLLABLE JJO", 0xa425 },
  { "YI SYLLABLE JJOP", 0xa426 },
  { "YI SYLLABLE JJUT", 0xa427 },
  { "YI SYLLABLE JJUX", 0xa428 },
  { "YI SYLLABLE JJU", 0xa429 },
  { "YI SYLLABLE JJUP", 0xa42a },
  { "YI SYLLABLE JJURX", 0xa42b },
  { "YI SYLLABLE JJUR", 0xa42c },
  { "YI SYLLABLE JJYT", 0xa42d },
  { "YI SYLLABLE JJYX", 0xa42e },
  { "YI SYLLABLE JJY", 0xa42f },
  { "YI SYLLABLE JJYP", 0xa430 },
  { "YI SYLLABLE NJIT", 0xa431 },
  { "YI SYLLABLE NJIX", 0xa432 },
  { "YI SYLLABLE NJI", 0xa433 },
  { "YI SYLLABLE NJIP", 0xa434 },
  { "YI SYLLABLE NJIET", 0xa435 },
  { "YI SYLLABLE NJIEX", 0xa436 },
  { "YI SYLLABLE NJIE", 0xa437 },
  { "YI SYLLABLE NJIEP", 0xa438 },
  { "YI SYLLABLE NJUOX", 0xa439 },
  { "YI SYLLABLE NJUO", 0xa43a },
  { "YI SYLLABLE NJOT", 0xa43b },
  { "YI SYLLABLE NJOX", 0xa43c },
  { "YI SYLLABLE NJO", 0xa43d },
  { "YI SYLLABLE NJOP", 0xa43e },
  { "YI SYLLABLE NJUX", 0xa43f },
  { "YI SYLLABLE NJU", 0xa440 },
  { "YI SYLLABLE NJUP", 0xa441 },
  { "YI SYLLABLE NJURX", 0xa442 },
  { "YI SYLLABLE NJUR", 0xa443 },
  { "YI SYLLABLE NJYT", 0xa444 },
  { "YI SYLLABLE NJYX", 0xa445 },
  { "YI SYLLABLE NJY", 0xa446 },
  { "YI SYLLABLE NJYP", 0xa447 },
  { "YI SYLLABLE NJYRX", 0xa448 },
  { "YI SYLLABLE NJYR", 0xa449 },
  { "YI SYLLABLE NYIT", 0xa44a },
  { "YI SYLLABLE NYIX", 0xa44b },
  { "YI SYLLABLE NYI", 0xa44c },
  { "YI SYLLABLE NYIP", 0xa44d },
  { "YI SYLLABLE NYIET", 0xa44e },
  { "YI SYLLABLE NYIEX", 0xa44f },
  { "YI SYLLABLE NYIE", 0xa450 },
  { "YI SYLLABLE NYIEP", 0xa451 },
  { "YI SYLLABLE NYUOX", 0xa452 },
  { "YI SYLLABLE NYUO", 0xa453 },
  { "YI SYLLABLE NYUOP", 0xa454 },
  { "YI SYLLABLE NYOT", 0xa455 },
  { "YI SYLLABLE NYOX", 0xa456 },
  { "YI SYLLABLE NYO", 0xa457 },
  { "YI SYLLABLE NYOP", 0xa458 },
  { "YI SYLLABLE NYUT", 0xa459 },
  { "YI SYLLABLE NYUX", 0xa45a },
  { "YI SYLLABLE NYU", 0xa45b },
  { "YI SYLLABLE NYUP", 0xa45c },
  { "YI SYLLABLE XIT", 0xa45d },
  { "YI SYLLABLE XIX", 0xa45e },
  { "YI SYLLABLE XI", 0xa45f },
  { "YI SYLLABLE XIP", 0xa460 },
  { "YI SYLLABLE XIET", 0xa461 },
  { "YI SYLLABLE XIEX", 0xa462 },
  { "YI SYLLABLE XIE", 0xa463 },
  { "YI SYLLABLE XIEP", 0xa464 },
  { "YI SYLLABLE XUOX", 0xa465 },
  { "YI SYLLABLE XUO", 0xa466 },
  { "YI SYLLABLE XOT", 0xa467 },
  { "YI SYLLABLE XOX", 0xa468 },
  { "YI SYLLABLE XO", 0xa469 },
  { "YI SYLLABLE XOP", 0xa46a },
  { "YI SYLLABLE XYT", 0xa46b },
  { "YI SYLLABLE XYX", 0xa46c },
  { "YI SYLLABLE XY", 0xa46d },
  { "YI SYLLABLE XYP", 0xa46e },
  { "YI SYLLABLE XYRX", 0xa46f },
  { "YI SYLLABLE XYR", 0xa470 },
  { "YI SYLLABLE YIT", 0xa471 },
  { "YI SYLLABLE YIX", 0xa472 },
  { "YI SYLLABLE YI", 0xa473 },
  { "YI SYLLABLE YIP", 0xa474 },
  { "YI SYLLABLE YIET", 0xa475 },
  { "YI SYLLABLE YIEX", 0xa476 },
  { "YI SYLLABLE YIE", 0xa477 },
  { "YI SYLLABLE YIEP", 0xa478 },
  { "YI SYLLABLE YUOT", 0xa479 },
  { "YI SYLLABLE YUOX", 0xa47a },
  { "YI SYLLABLE YUO", 0xa47b },
  { "YI SYLLABLE YUOP", 0xa47c },
  { "YI SYLLABLE YOT", 0xa47d },
  { "YI SYLLABLE YOX", 0xa47e },
  { "YI SYLLABLE YO", 0xa47f },
  { "YI SYLLABLE YOP", 0xa480 },
  { "YI SYLLABLE YUT", 0xa481 },
  { "YI SYLLABLE YUX", 0xa482 },
  { "YI SYLLABLE YU", 0xa483 },
  { "YI SYLLABLE YUP", 0xa484 },
  { "YI SYLLABLE YURX", 0xa485 },
  { "YI SYLLABLE YUR", 0xa486 },
  { "YI SYLLABLE YYT", 0xa487 },
  { "YI SYLLABLE YYX", 0xa488 },
  { "YI SYLLABLE YY", 0xa489 },
  { "YI SYLLABLE YYP", 0xa48a },
  { "YI SYLLABLE YYRX", 0xa48b },
  { "YI SYLLABLE YYR", 0xa48c },
  { "YI RADICAL QOT", 0xa490 },
  { "YI RADICAL LI", 0xa491 },
  { "YI RADICAL KIT", 0xa492 },
  { "YI RADICAL NYIP", 0xa493 },
  { "YI RADICAL CYP", 0xa494 },
  { "YI RADICAL SSI", 0xa495 },
  { "YI RADICAL GGOP", 0xa496 },
  { "YI RADICAL GEP", 0xa497 },
  { "YI RADICAL MI", 0xa498 },
  { "YI RADICAL HXIT", 0xa499 },
  { "YI RADICAL LYR", 0xa49a },
  { "YI RADICAL BBUT", 0xa49b },
  { "YI RADICAL MOP", 0xa49c },
  { "YI RADICAL YO", 0xa49d },
  { "YI RADICAL PUT", 0xa49e },
  { "YI RADICAL HXUO", 0xa49f },
  { "YI RADICAL TAT", 0xa4a0 },
  { "YI RADICAL GA", 0xa4a1 },
  { "YI RADICAL DDUR", 0xa4a4 },
  { "YI RADICAL BUR", 0xa4a5 },
  { "YI RADICAL GGUO", 0xa4a6 },
  { "YI RADICAL NYOP", 0xa4a7 },
  { "YI RADICAL TU", 0xa4a8 },
  { "YI RADICAL OP", 0xa4a9 },
  { "YI RADICAL JJUT", 0xa4aa },
  { "YI RADICAL ZOT", 0xa4ab },
  { "YI RADICAL PYT", 0xa4ac },
  { "YI RADICAL HMO", 0xa4ad },
  { "YI RADICAL YIT", 0xa4ae },
  { "YI RADICAL VUR", 0xa4af },
  { "YI RADICAL SHY", 0xa4b0 },
  { "YI RADICAL VEP", 0xa4b1 },
  { "YI RADICAL ZA", 0xa4b2 },
  { "YI RADICAL JO", 0xa4b3 },
  { "YI RADICAL JJY", 0xa4b5 },
  { "YI RADICAL GOT", 0xa4b6 },
  { "YI RADICAL JJIE", 0xa4b7 },
  { "YI RADICAL WO", 0xa4b8 },
  { "YI RADICAL DU", 0xa4b9 },
  { "YI RADICAL SHUR", 0xa4ba },
  { "YI RADICAL LIE", 0xa4bb },
  { "YI RADICAL CY", 0xa4bc },
  { "YI RADICAL CUOP", 0xa4bd },
  { "YI RADICAL CIP", 0xa4be },
  { "YI RADICAL HXOP", 0xa4bf },
  { "YI RADICAL SHAT", 0xa4c0 },
  { "YI RADICAL SHOP", 0xa4c2 },
  { "YI RADICAL CHE", 0xa4c3 },
  { "YI RADICAL ZZIET", 0xa4c4 },
  { "YI RADICAL KE", 0xa4c6 },
  { "CJK COMPATIBILITY IDEOGRAPH-F900", 0xf900 },
  { "CJK COMPATIBILITY IDEOGRAPH-F901", 0xf901 },
  { "CJK COMPATIBILITY IDEOGRAPH-F902", 0xf902 },
  { "CJK COMPATIBILITY IDEOGRAPH-F903", 0xf903 },
  { "CJK COMPATIBILITY IDEOGRAPH-F904", 0xf904 },
  { "CJK COMPATIBILITY IDEOGRAPH-F905", 0xf905 },
  { "CJK COMPATIBILITY IDEOGRAPH-F906", 0xf906 },
  { "CJK COMPATIBILITY IDEOGRAPH-F907", 0xf907 },
  { "CJK COMPATIBILITY IDEOGRAPH-F908", 0xf908 },
  { "CJK COMPATIBILITY IDEOGRAPH-F909", 0xf909 },
  { "CJK COMPATIBILITY IDEOGRAPH-F90A", 0xf90a },
  { "CJK COMPATIBILITY IDEOGRAPH-F90B", 0xf90b },
  { "CJK COMPATIBILITY IDEOGRAPH-F90C", 0xf90c },
  { "CJK COMPATIBILITY IDEOGRAPH-F90D", 0xf90d },
  { "CJK COMPATIBILITY IDEOGRAPH-F90E", 0xf90e },
  { "CJK COMPATIBILITY IDEOGRAPH-F90F", 0xf90f },
  { "CJK COMPATIBILITY IDEOGRAPH-F910", 0xf910 },
  { "CJK COMPATIBILITY IDEOGRAPH-F911", 0xf911 },
  { "CJK COMPATIBILITY IDEOGRAPH-F912", 0xf912 },
  { "CJK COMPATIBILITY IDEOGRAPH-F913", 0xf913 },
  { "CJK COMPATIBILITY IDEOGRAPH-F914", 0xf914 },
  { "CJK COMPATIBILITY IDEOGRAPH-F915", 0xf915 },
  { "CJK COMPATIBILITY IDEOGRAPH-F916", 0xf916 },
  { "CJK COMPATIBILITY IDEOGRAPH-F917", 0xf917 },
  { "CJK COMPATIBILITY IDEOGRAPH-F918", 0xf918 },
  { "CJK COMPATIBILITY IDEOGRAPH-F919", 0xf919 },
  { "CJK COMPATIBILITY IDEOGRAPH-F91A", 0xf91a },
  { "CJK COMPATIBILITY IDEOGRAPH-F91B", 0xf91b },
  { "CJK COMPATIBILITY IDEOGRAPH-F91C", 0xf91c },
  { "CJK COMPATIBILITY IDEOGRAPH-F91D", 0xf91d },
  { "CJK COMPATIBILITY IDEOGRAPH-F91E", 0xf91e },
  { "CJK COMPATIBILITY IDEOGRAPH-F91F", 0xf91f },
  { "CJK COMPATIBILITY IDEOGRAPH-F920", 0xf920 },
  { "CJK COMPATIBILITY IDEOGRAPH-F921", 0xf921 },
  { "CJK COMPATIBILITY IDEOGRAPH-F922", 0xf922 },
  { "CJK COMPATIBILITY IDEOGRAPH-F923", 0xf923 },
  { "CJK COMPATIBILITY IDEOGRAPH-F924", 0xf924 },
  { "CJK COMPATIBILITY IDEOGRAPH-F925", 0xf925 },
  { "CJK COMPATIBILITY IDEOGRAPH-F926", 0xf926 },
  { "CJK COMPATIBILITY IDEOGRAPH-F927", 0xf927 },
  { "CJK COMPATIBILITY IDEOGRAPH-F928", 0xf928 },
  { "CJK COMPATIBILITY IDEOGRAPH-F929", 0xf929 },
  { "CJK COMPATIBILITY IDEOGRAPH-F92A", 0xf92a },
  { "CJK COMPATIBILITY IDEOGRAPH-F92B", 0xf92b },
  { "CJK COMPATIBILITY IDEOGRAPH-F92C", 0xf92c },
  { "CJK COMPATIBILITY IDEOGRAPH-F92D", 0xf92d },
  { "CJK COMPATIBILITY IDEOGRAPH-F92E", 0xf92e },
  { "CJK COMPATIBILITY IDEOGRAPH-F92F", 0xf92f },
  { "CJK COMPATIBILITY IDEOGRAPH-F930", 0xf930 },
  { "CJK COMPATIBILITY IDEOGRAPH-F931", 0xf931 },
  { "CJK COMPATIBILITY IDEOGRAPH-F932", 0xf932 },
  { "CJK COMPATIBILITY IDEOGRAPH-F933", 0xf933 },
  { "CJK COMPATIBILITY IDEOGRAPH-F934", 0xf934 },
  { "CJK COMPATIBILITY IDEOGRAPH-F935", 0xf935 },
  { "CJK COMPATIBILITY IDEOGRAPH-F936", 0xf936 },
  { "CJK COMPATIBILITY IDEOGRAPH-F937", 0xf937 },
  { "CJK COMPATIBILITY IDEOGRAPH-F938", 0xf938 },
  { "CJK COMPATIBILITY IDEOGRAPH-F939", 0xf939 },
  { "CJK COMPATIBILITY IDEOGRAPH-F93A", 0xf93a },
  { "CJK COMPATIBILITY IDEOGRAPH-F93B", 0xf93b },
  { "CJK COMPATIBILITY IDEOGRAPH-F93C", 0xf93c },
  { "CJK COMPATIBILITY IDEOGRAPH-F93D", 0xf93d },
  { "CJK COMPATIBILITY IDEOGRAPH-F93E", 0xf93e },
  { "CJK COMPATIBILITY IDEOGRAPH-F93F", 0xf93f },
  { "CJK COMPATIBILITY IDEOGRAPH-F940", 0xf940 },
  { "CJK COMPATIBILITY IDEOGRAPH-F941", 0xf941 },
  { "CJK COMPATIBILITY IDEOGRAPH-F942", 0xf942 },
  { "CJK COMPATIBILITY IDEOGRAPH-F943", 0xf943 },
  { "CJK COMPATIBILITY IDEOGRAPH-F944", 0xf944 },
  { "CJK COMPATIBILITY IDEOGRAPH-F945", 0xf945 },
  { "CJK COMPATIBILITY IDEOGRAPH-F946", 0xf946 },
  { "CJK COMPATIBILITY IDEOGRAPH-F947", 0xf947 },
  { "CJK COMPATIBILITY IDEOGRAPH-F948", 0xf948 },
  { "CJK COMPATIBILITY IDEOGRAPH-F949", 0xf949 },
  { "CJK COMPATIBILITY IDEOGRAPH-F94A", 0xf94a },
  { "CJK COMPATIBILITY IDEOGRAPH-F94B", 0xf94b },
  { "CJK COMPATIBILITY IDEOGRAPH-F94C", 0xf94c },
  { "CJK COMPATIBILITY IDEOGRAPH-F94D", 0xf94d },
  { "CJK COMPATIBILITY IDEOGRAPH-F94E", 0xf94e },
  { "CJK COMPATIBILITY IDEOGRAPH-F94F", 0xf94f },
  { "CJK COMPATIBILITY IDEOGRAPH-F950", 0xf950 },
  { "CJK COMPATIBILITY IDEOGRAPH-F951", 0xf951 },
  { "CJK COMPATIBILITY IDEOGRAPH-F952", 0xf952 },
  { "CJK COMPATIBILITY IDEOGRAPH-F953", 0xf953 },
  { "CJK COMPATIBILITY IDEOGRAPH-F954", 0xf954 },
  { "CJK COMPATIBILITY IDEOGRAPH-F955", 0xf955 },
  { "CJK COMPATIBILITY IDEOGRAPH-F956", 0xf956 },
  { "CJK COMPATIBILITY IDEOGRAPH-F957", 0xf957 },
  { "CJK COMPATIBILITY IDEOGRAPH-F958", 0xf958 },
  { "CJK COMPATIBILITY IDEOGRAPH-F959", 0xf959 },
  { "CJK COMPATIBILITY IDEOGRAPH-F95A", 0xf95a },
  { "CJK COMPATIBILITY IDEOGRAPH-F95B", 0xf95b },
  { "CJK COMPATIBILITY IDEOGRAPH-F95C", 0xf95c },
  { "CJK COMPATIBILITY IDEOGRAPH-F95D", 0xf95d },
  { "CJK COMPATIBILITY IDEOGRAPH-F95E", 0xf95e },
  { "CJK COMPATIBILITY IDEOGRAPH-F95F", 0xf95f },
  { "CJK COMPATIBILITY IDEOGRAPH-F960", 0xf960 },
  { "CJK COMPATIBILITY IDEOGRAPH-F961", 0xf961 },
  { "CJK COMPATIBILITY IDEOGRAPH-F962", 0xf962 },
  { "CJK COMPATIBILITY IDEOGRAPH-F963", 0xf963 },
  { "CJK COMPATIBILITY IDEOGRAPH-F964", 0xf964 },
  { "CJK COMPATIBILITY IDEOGRAPH-F965", 0xf965 },
  { "CJK COMPATIBILITY IDEOGRAPH-F966", 0xf966 },
  { "CJK COMPATIBILITY IDEOGRAPH-F967", 0xf967 },
  { "CJK COMPATIBILITY IDEOGRAPH-F968", 0xf968 },
  { "CJK COMPATIBILITY IDEOGRAPH-F969", 0xf969 },
  { "CJK COMPATIBILITY IDEOGRAPH-F96A", 0xf96a },
  { "CJK COMPATIBILITY IDEOGRAPH-F96B", 0xf96b },
  { "CJK COMPATIBILITY IDEOGRAPH-F96C", 0xf96c },
  { "CJK COMPATIBILITY IDEOGRAPH-F96D", 0xf96d },
  { "CJK COMPATIBILITY IDEOGRAPH-F96E", 0xf96e },
  { "CJK COMPATIBILITY IDEOGRAPH-F96F", 0xf96f },
  { "CJK COMPATIBILITY IDEOGRAPH-F970", 0xf970 },
  { "CJK COMPATIBILITY IDEOGRAPH-F971", 0xf971 },
  { "CJK COMPATIBILITY IDEOGRAPH-F972", 0xf972 },
  { "CJK COMPATIBILITY IDEOGRAPH-F973", 0xf973 },
  { "CJK COMPATIBILITY IDEOGRAPH-F974", 0xf974 },
  { "CJK COMPATIBILITY IDEOGRAPH-F975", 0xf975 },
  { "CJK COMPATIBILITY IDEOGRAPH-F976", 0xf976 },
  { "CJK COMPATIBILITY IDEOGRAPH-F977", 0xf977 },
  { "CJK COMPATIBILITY IDEOGRAPH-F978", 0xf978 },
  { "CJK COMPATIBILITY IDEOGRAPH-F979", 0xf979 },
  { "CJK COMPATIBILITY IDEOGRAPH-F97A", 0xf97a },
  { "CJK COMPATIBILITY IDEOGRAPH-F97B", 0xf97b },
  { "CJK COMPATIBILITY IDEOGRAPH-F97C", 0xf97c },
  { "CJK COMPATIBILITY IDEOGRAPH-F97D", 0xf97d },
  { "CJK COMPATIBILITY IDEOGRAPH-F97E", 0xf97e },
  { "CJK COMPATIBILITY IDEOGRAPH-F97F", 0xf97f },
  { "CJK COMPATIBILITY IDEOGRAPH-F980", 0xf980 },
  { "CJK COMPATIBILITY IDEOGRAPH-F981", 0xf981 },
  { "CJK COMPATIBILITY IDEOGRAPH-F982", 0xf982 },
  { "CJK COMPATIBILITY IDEOGRAPH-F983", 0xf983 },
  { "CJK COMPATIBILITY IDEOGRAPH-F984", 0xf984 },
  { "CJK COMPATIBILITY IDEOGRAPH-F985", 0xf985 },
  { "CJK COMPATIBILITY IDEOGRAPH-F986", 0xf986 },
  { "CJK COMPATIBILITY IDEOGRAPH-F987", 0xf987 },
  { "CJK COMPATIBILITY IDEOGRAPH-F988", 0xf988 },
  { "CJK COMPATIBILITY IDEOGRAPH-F989", 0xf989 },
  { "CJK COMPATIBILITY IDEOGRAPH-F98A", 0xf98a },
  { "CJK COMPATIBILITY IDEOGRAPH-F98B", 0xf98b },
  { "CJK COMPATIBILITY IDEOGRAPH-F98C", 0xf98c },
  { "CJK COMPATIBILITY IDEOGRAPH-F98D", 0xf98d },
  { "CJK COMPATIBILITY IDEOGRAPH-F98E", 0xf98e },
  { "CJK COMPATIBILITY IDEOGRAPH-F98F", 0xf98f },
  { "CJK COMPATIBILITY IDEOGRAPH-F990", 0xf990 },
  { "CJK COMPATIBILITY IDEOGRAPH-F991", 0xf991 },
  { "CJK COMPATIBILITY IDEOGRAPH-F992", 0xf992 },
  { "CJK COMPATIBILITY IDEOGRAPH-F993", 0xf993 },
  { "CJK COMPATIBILITY IDEOGRAPH-F994", 0xf994 },
  { "CJK COMPATIBILITY IDEOGRAPH-F995", 0xf995 },
  { "CJK COMPATIBILITY IDEOGRAPH-F996", 0xf996 },
  { "CJK COMPATIBILITY IDEOGRAPH-F997", 0xf997 },
  { "CJK COMPATIBILITY IDEOGRAPH-F998", 0xf998 },
  { "CJK COMPATIBILITY IDEOGRAPH-F999", 0xf999 },
  { "CJK COMPATIBILITY IDEOGRAPH-F99A", 0xf99a },
  { "CJK COMPATIBILITY IDEOGRAPH-F99B", 0xf99b },
  { "CJK COMPATIBILITY IDEOGRAPH-F99C", 0xf99c },
  { "CJK COMPATIBILITY IDEOGRAPH-F99D", 0xf99d },
  { "CJK COMPATIBILITY IDEOGRAPH-F99E", 0xf99e },
  { "CJK COMPATIBILITY IDEOGRAPH-F99F", 0xf99f },
  { "CJK COMPATIBILITY IDEOGRAPH-F9A0", 0xf9a0 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9A1", 0xf9a1 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9A2", 0xf9a2 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9A3", 0xf9a3 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9A4", 0xf9a4 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9A5", 0xf9a5 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9A6", 0xf9a6 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9A7", 0xf9a7 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9A8", 0xf9a8 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9A9", 0xf9a9 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9AA", 0xf9aa },
  { "CJK COMPATIBILITY IDEOGRAPH-F9AB", 0xf9ab },
  { "CJK COMPATIBILITY IDEOGRAPH-F9AC", 0xf9ac },
  { "CJK COMPATIBILITY IDEOGRAPH-F9AD", 0xf9ad },
  { "CJK COMPATIBILITY IDEOGRAPH-F9AE", 0xf9ae },
  { "CJK COMPATIBILITY IDEOGRAPH-F9AF", 0xf9af },
  { "CJK COMPATIBILITY IDEOGRAPH-F9B0", 0xf9b0 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9B1", 0xf9b1 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9B2", 0xf9b2 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9B3", 0xf9b3 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9B4", 0xf9b4 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9B5", 0xf9b5 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9B6", 0xf9b6 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9B7", 0xf9b7 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9B8", 0xf9b8 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9B9", 0xf9b9 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9BA", 0xf9ba },
  { "CJK COMPATIBILITY IDEOGRAPH-F9BB", 0xf9bb },
  { "CJK COMPATIBILITY IDEOGRAPH-F9BC", 0xf9bc },
  { "CJK COMPATIBILITY IDEOGRAPH-F9BD", 0xf9bd },
  { "CJK COMPATIBILITY IDEOGRAPH-F9BE", 0xf9be },
  { "CJK COMPATIBILITY IDEOGRAPH-F9BF", 0xf9bf },
  { "CJK COMPATIBILITY IDEOGRAPH-F9C0", 0xf9c0 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9C1", 0xf9c1 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9C2", 0xf9c2 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9C3", 0xf9c3 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9C4", 0xf9c4 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9C5", 0xf9c5 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9C6", 0xf9c6 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9C7", 0xf9c7 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9C8", 0xf9c8 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9C9", 0xf9c9 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9CA", 0xf9ca },
  { "CJK COMPATIBILITY IDEOGRAPH-F9CB", 0xf9cb },
  { "CJK COMPATIBILITY IDEOGRAPH-F9CC", 0xf9cc },
  { "CJK COMPATIBILITY IDEOGRAPH-F9CD", 0xf9cd },
  { "CJK COMPATIBILITY IDEOGRAPH-F9CE", 0xf9ce },
  { "CJK COMPATIBILITY IDEOGRAPH-F9CF", 0xf9cf },
  { "CJK COMPATIBILITY IDEOGRAPH-F9D0", 0xf9d0 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9D1", 0xf9d1 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9D2", 0xf9d2 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9D3", 0xf9d3 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9D4", 0xf9d4 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9D5", 0xf9d5 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9D6", 0xf9d6 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9D7", 0xf9d7 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9D8", 0xf9d8 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9D9", 0xf9d9 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9DA", 0xf9da },
  { "CJK COMPATIBILITY IDEOGRAPH-F9DB", 0xf9db },
  { "CJK COMPATIBILITY IDEOGRAPH-F9DC", 0xf9dc },
  { "CJK COMPATIBILITY IDEOGRAPH-F9DD", 0xf9dd },
  { "CJK COMPATIBILITY IDEOGRAPH-F9DE", 0xf9de },
  { "CJK COMPATIBILITY IDEOGRAPH-F9DF", 0xf9df },
  { "CJK COMPATIBILITY IDEOGRAPH-F9E0", 0xf9e0 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9E1", 0xf9e1 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9E2", 0xf9e2 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9E3", 0xf9e3 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9E4", 0xf9e4 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9E5", 0xf9e5 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9E6", 0xf9e6 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9E7", 0xf9e7 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9E8", 0xf9e8 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9E9", 0xf9e9 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9EA", 0xf9ea },
  { "CJK COMPATIBILITY IDEOGRAPH-F9EB", 0xf9eb },
  { "CJK COMPATIBILITY IDEOGRAPH-F9EC", 0xf9ec },
  { "CJK COMPATIBILITY IDEOGRAPH-F9ED", 0xf9ed },
  { "CJK COMPATIBILITY IDEOGRAPH-F9EE", 0xf9ee },
  { "CJK COMPATIBILITY IDEOGRAPH-F9EF", 0xf9ef },
  { "CJK COMPATIBILITY IDEOGRAPH-F9F0", 0xf9f0 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9F1", 0xf9f1 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9F2", 0xf9f2 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9F3", 0xf9f3 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9F4", 0xf9f4 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9F5", 0xf9f5 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9F6", 0xf9f6 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9F7", 0xf9f7 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9F8", 0xf9f8 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9F9", 0xf9f9 },
  { "CJK COMPATIBILITY IDEOGRAPH-F9FA", 0xf9fa },
  { "CJK COMPATIBILITY IDEOGRAPH-F9FB", 0xf9fb },
  { "CJK COMPATIBILITY IDEOGRAPH-F9FC", 0xf9fc },
  { "CJK COMPATIBILITY IDEOGRAPH-F9FD", 0xf9fd },
  { "CJK COMPATIBILITY IDEOGRAPH-F9FE", 0xf9fe },
  { "CJK COMPATIBILITY IDEOGRAPH-F9FF", 0xf9ff },
  { "CJK COMPATIBILITY IDEOGRAPH-FA00", 0xfa00 },
  { "CJK COMPATIBILITY IDEOGRAPH-FA01", 0xfa01 },
  { "CJK COMPATIBILITY IDEOGRAPH-FA02", 0xfa02 },
  { "CJK COMPATIBILITY IDEOGRAPH-FA03", 0xfa03 },
  { "CJK COMPATIBILITY IDEOGRAPH-FA04", 0xfa04 },
  { "CJK COMPATIBILITY IDEOGRAPH-FA05", 0xfa05 },
  { "CJK COMPATIBILITY IDEOGRAPH-FA06", 0xfa06 },
  { "CJK COMPATIBILITY IDEOGRAPH-FA07", 0xfa07 },
  { "CJK COMPATIBILITY IDEOGRAPH-FA08", 0xfa08 },
  { "CJK COMPATIBILITY IDEOGRAPH-FA09", 0xfa09 },
  { "CJK COMPATIBILITY IDEOGRAPH-FA0A", 0xfa0a },
  { "CJK COMPATIBILITY IDEOGRAPH-FA0B", 0xfa0b },
  { "CJK COMPATIBILITY IDEOGRAPH-FA0C", 0xfa0c },
  { "CJK COMPATIBILITY IDEOGRAPH-FA0D", 0xfa0d },
  { "CJK COMPATIBILITY IDEOGRAPH-FA0E", 0xfa0e },
  { "CJK COMPATIBILITY IDEOGRAPH-FA0F", 0xfa0f },
  { "CJK COMPATIBILITY IDEOGRAPH-FA10", 0xfa10 },
  { "CJK COMPATIBILITY IDEOGRAPH-FA11", 0xfa11 },
  { "CJK COMPATIBILITY IDEOGRAPH-FA12", 0xfa12 },
  { "CJK COMPATIBILITY IDEOGRAPH-FA13", 0xfa13 },
  { "CJK COMPATIBILITY IDEOGRAPH-FA14", 0xfa14 },
  { "CJK COMPATIBILITY IDEOGRAPH-FA15", 0xfa15 },
  { "CJK COMPATIBILITY IDEOGRAPH-FA16", 0xfa16 },
  { "CJK COMPATIBILITY IDEOGRAPH-FA17", 0xfa17 },
  { "CJK COMPATIBILITY IDEOGRAPH-FA18", 0xfa18 },
  { "CJK COMPATIBILITY IDEOGRAPH-FA19", 0xfa19 },
  { "CJK COMPATIBILITY IDEOGRAPH-FA1A", 0xfa1a },
  { "CJK COMPATIBILITY IDEOGRAPH-FA1B", 0xfa1b },
  { "CJK COMPATIBILITY IDEOGRAPH-FA1C", 0xfa1c },
  { "CJK COMPATIBILITY IDEOGRAPH-FA1D", 0xfa1d },
  { "CJK COMPATIBILITY IDEOGRAPH-FA1E", 0xfa1e },
  { "CJK COMPATIBILITY IDEOGRAPH-FA1F", 0xfa1f },
  { "CJK COMPATIBILITY IDEOGRAPH-FA20", 0xfa20 },
  { "CJK COMPATIBILITY IDEOGRAPH-FA21", 0xfa21 },
  { "CJK COMPATIBILITY IDEOGRAPH-FA22", 0xfa22 },
  { "CJK COMPATIBILITY IDEOGRAPH-FA23", 0xfa23 },
  { "CJK COMPATIBILITY IDEOGRAPH-FA24", 0xfa24 },
  { "CJK COMPATIBILITY IDEOGRAPH-FA25", 0xfa25 },
  { "CJK COMPATIBILITY IDEOGRAPH-FA26", 0xfa26 },
  { "CJK COMPATIBILITY IDEOGRAPH-FA27", 0xfa27 },
  { "CJK COMPATIBILITY IDEOGRAPH-FA28", 0xfa28 },
  { "CJK COMPATIBILITY IDEOGRAPH-FA29", 0xfa29 },
  { "CJK COMPATIBILITY IDEOGRAPH-FA2A", 0xfa2a },
  { "CJK COMPATIBILITY IDEOGRAPH-FA2B", 0xfa2b },
  { "CJK COMPATIBILITY IDEOGRAPH-FA2C", 0xfa2c },
  { "CJK COMPATIBILITY IDEOGRAPH-FA2D", 0xfa2d },
  { "LATIN SMALL LIGATURE FF", 0xfb00 },
  { "LATIN SMALL LIGATURE FI", 0xfb01 },
  { "LATIN SMALL LIGATURE FL", 0xfb02 },
  { "LATIN SMALL LIGATURE FFI", 0xfb03 },
  { "LATIN SMALL LIGATURE FFL", 0xfb04 },
  { "LATIN SMALL LIGATURE LONG S T", 0xfb05 },
  { "LATIN SMALL LIGATURE ST", 0xfb06 },
  { "ARMENIAN SMALL LIGATURE MEN NOW", 0xfb13 },
  { "ARMENIAN SMALL LIGATURE MEN ECH", 0xfb14 },
  { "ARMENIAN SMALL LIGATURE MEN INI", 0xfb15 },
  { "ARMENIAN SMALL LIGATURE VEW NOW", 0xfb16 },
  { "ARMENIAN SMALL LIGATURE MEN XEH", 0xfb17 },
  { "HEBREW LETTER YOD WITH HIRIQ", 0xfb1d },
  { "HEBREW POINT JUDEO-SPANISH VARIKA", 0xfb1e },
  { "HEBREW LIGATURE YIDDISH YOD YOD PATAH", 0xfb1f },
  { "HEBREW LETTER ALTERNATIVE AYIN", 0xfb20 },
  { "HEBREW LETTER WIDE ALEF", 0xfb21 },
  { "HEBREW LETTER WIDE DALET", 0xfb22 },
  { "HEBREW LETTER WIDE HE", 0xfb23 },
  { "HEBREW LETTER WIDE KAF", 0xfb24 },
  { "HEBREW LETTER WIDE LAMED", 0xfb25 },
  { "HEBREW LETTER WIDE FINAL MEM", 0xfb26 },
  { "HEBREW LETTER WIDE RESH", 0xfb27 },
  { "HEBREW LETTER WIDE TAV", 0xfb28 },
  { "HEBREW LETTER ALTERNATIVE PLUS SIGN", 0xfb29 },
  { "HEBREW LETTER SHIN WITH SHIN DOT", 0xfb2a },
  { "HEBREW LETTER SHIN WITH SIN DOT", 0xfb2b },
  { "HEBREW LETTER SHIN WITH DAGESH AND SHIN DOT", 0xfb2c },
  { "HEBREW LETTER SHIN WITH DAGESH AND SIN DOT", 0xfb2d },
  { "HEBREW LETTER ALEF WITH PATAH", 0xfb2e },
  { "HEBREW LETTER ALEF WITH QAMATS", 0xfb2f },
  { "HEBREW LETTER ALEF WITH MAPIQ", 0xfb30 },
  { "HEBREW LETTER BET WITH DAGESH", 0xfb31 },
  { "HEBREW LETTER GIMEL WITH DAGESH", 0xfb32 },
  { "HEBREW LETTER DALET WITH DAGESH", 0xfb33 },
  { "HEBREW LETTER HE WITH MAPIQ", 0xfb34 },
  { "HEBREW LETTER VAV WITH DAGESH", 0xfb35 },
  { "HEBREW LETTER ZAYIN WITH DAGESH", 0xfb36 },
  { "HEBREW LETTER TET WITH DAGESH", 0xfb38 },
  { "HEBREW LETTER YOD WITH DAGESH", 0xfb39 },
  { "HEBREW LETTER FINAL KAF WITH DAGESH", 0xfb3a },
  { "HEBREW LETTER KAF WITH DAGESH", 0xfb3b },
  { "HEBREW LETTER LAMED WITH DAGESH", 0xfb3c },
  { "HEBREW LETTER MEM WITH DAGESH", 0xfb3e },
  { "HEBREW LETTER NUN WITH DAGESH", 0xfb40 },
  { "HEBREW LETTER SAMEKH WITH DAGESH", 0xfb41 },
  { "HEBREW LETTER FINAL PE WITH DAGESH", 0xfb43 },
  { "HEBREW LETTER PE WITH DAGESH", 0xfb44 },
  { "HEBREW LETTER TSADI WITH DAGESH", 0xfb46 },
  { "HEBREW LETTER QOF WITH DAGESH", 0xfb47 },
  { "HEBREW LETTER RESH WITH DAGESH", 0xfb48 },
  { "HEBREW LETTER SHIN WITH DAGESH", 0xfb49 },
  { "HEBREW LETTER TAV WITH DAGESH", 0xfb4a },
  { "HEBREW LETTER VAV WITH HOLAM", 0xfb4b },
  { "HEBREW LETTER BET WITH RAFE", 0xfb4c },
  { "HEBREW LETTER KAF WITH RAFE", 0xfb4d },
  { "HEBREW LETTER PE WITH RAFE", 0xfb4e },
  { "HEBREW LIGATURE ALEF LAMED", 0xfb4f },
  { "ARABIC LETTER ALEF WASLA ISOLATED FORM", 0xfb50 },
  { "ARABIC LETTER ALEF WASLA FINAL FORM", 0xfb51 },
  { "ARABIC LETTER BEEH ISOLATED FORM", 0xfb52 },
  { "ARABIC LETTER BEEH FINAL FORM", 0xfb53 },
  { "ARABIC LETTER BEEH INITIAL FORM", 0xfb54 },
  { "ARABIC LETTER BEEH MEDIAL FORM", 0xfb55 },
  { "ARABIC LETTER PEH ISOLATED FORM", 0xfb56 },
  { "ARABIC LETTER PEH FINAL FORM", 0xfb57 },
  { "ARABIC LETTER PEH INITIAL FORM", 0xfb58 },
  { "ARABIC LETTER PEH MEDIAL FORM", 0xfb59 },
  { "ARABIC LETTER BEHEH ISOLATED FORM", 0xfb5a },
  { "ARABIC LETTER BEHEH FINAL FORM", 0xfb5b },
  { "ARABIC LETTER BEHEH INITIAL FORM", 0xfb5c },
  { "ARABIC LETTER BEHEH MEDIAL FORM", 0xfb5d },
  { "ARABIC LETTER TTEHEH ISOLATED FORM", 0xfb5e },
  { "ARABIC LETTER TTEHEH FINAL FORM", 0xfb5f },
  { "ARABIC LETTER TTEHEH INITIAL FORM", 0xfb60 },
  { "ARABIC LETTER TTEHEH MEDIAL FORM", 0xfb61 },
  { "ARABIC LETTER TEHEH ISOLATED FORM", 0xfb62 },
  { "ARABIC LETTER TEHEH FINAL FORM", 0xfb63 },
  { "ARABIC LETTER TEHEH INITIAL FORM", 0xfb64 },
  { "ARABIC LETTER TEHEH MEDIAL FORM", 0xfb65 },
  { "ARABIC LETTER TTEH ISOLATED FORM", 0xfb66 },
  { "ARABIC LETTER TTEH FINAL FORM", 0xfb67 },
  { "ARABIC LETTER TTEH INITIAL FORM", 0xfb68 },
  { "ARABIC LETTER TTEH MEDIAL FORM", 0xfb69 },
  { "ARABIC LETTER VEH ISOLATED FORM", 0xfb6a },
  { "ARABIC LETTER VEH FINAL FORM", 0xfb6b },
  { "ARABIC LETTER VEH INITIAL FORM", 0xfb6c },
  { "ARABIC LETTER VEH MEDIAL FORM", 0xfb6d },
  { "ARABIC LETTER PEHEH ISOLATED FORM", 0xfb6e },
  { "ARABIC LETTER PEHEH FINAL FORM", 0xfb6f },
  { "ARABIC LETTER PEHEH INITIAL FORM", 0xfb70 },
  { "ARABIC LETTER PEHEH MEDIAL FORM", 0xfb71 },
  { "ARABIC LETTER DYEH ISOLATED FORM", 0xfb72 },
  { "ARABIC LETTER DYEH FINAL FORM", 0xfb73 },
  { "ARABIC LETTER DYEH INITIAL FORM", 0xfb74 },
  { "ARABIC LETTER DYEH MEDIAL FORM", 0xfb75 },
  { "ARABIC LETTER NYEH ISOLATED FORM", 0xfb76 },
  { "ARABIC LETTER NYEH FINAL FORM", 0xfb77 },
  { "ARABIC LETTER NYEH INITIAL FORM", 0xfb78 },
  { "ARABIC LETTER NYEH MEDIAL FORM", 0xfb79 },
  { "ARABIC LETTER TCHEH ISOLATED FORM", 0xfb7a },
  { "ARABIC LETTER TCHEH FINAL FORM", 0xfb7b },
  { "ARABIC LETTER TCHEH INITIAL FORM", 0xfb7c },
  { "ARABIC LETTER TCHEH MEDIAL FORM", 0xfb7d },
  { "ARABIC LETTER TCHEHEH ISOLATED FORM", 0xfb7e },
  { "ARABIC LETTER TCHEHEH FINAL FORM", 0xfb7f },
  { "ARABIC LETTER TCHEHEH INITIAL FORM", 0xfb80 },
  { "ARABIC LETTER TCHEHEH MEDIAL FORM", 0xfb81 },
  { "ARABIC LETTER DDAHAL ISOLATED FORM", 0xfb82 },
  { "ARABIC LETTER DDAHAL FINAL FORM", 0xfb83 },
  { "ARABIC LETTER DAHAL ISOLATED FORM", 0xfb84 },
  { "ARABIC LETTER DAHAL FINAL FORM", 0xfb85 },
  { "ARABIC LETTER DUL ISOLATED FORM", 0xfb86 },
  { "ARABIC LETTER DUL FINAL FORM", 0xfb87 },
  { "ARABIC LETTER DDAL ISOLATED FORM", 0xfb88 },
  { "ARABIC LETTER DDAL FINAL FORM", 0xfb89 },
  { "ARABIC LETTER JEH ISOLATED FORM", 0xfb8a },
  { "ARABIC LETTER JEH FINAL FORM", 0xfb8b },
  { "ARABIC LETTER RREH ISOLATED FORM", 0xfb8c },
  { "ARABIC LETTER RREH FINAL FORM", 0xfb8d },
  { "ARABIC LETTER KEHEH ISOLATED FORM", 0xfb8e },
  { "ARABIC LETTER KEHEH FINAL FORM", 0xfb8f },
  { "ARABIC LETTER KEHEH INITIAL FORM", 0xfb90 },
  { "ARABIC LETTER KEHEH MEDIAL FORM", 0xfb91 },
  { "ARABIC LETTER GAF ISOLATED FORM", 0xfb92 },
  { "ARABIC LETTER GAF FINAL FORM", 0xfb93 },
  { "ARABIC LETTER GAF INITIAL FORM", 0xfb94 },
  { "ARABIC LETTER GAF MEDIAL FORM", 0xfb95 },
  { "ARABIC LETTER GUEH ISOLATED FORM", 0xfb96 },
  { "ARABIC LETTER GUEH FINAL FORM", 0xfb97 },
  { "ARABIC LETTER GUEH INITIAL FORM", 0xfb98 },
  { "ARABIC LETTER GUEH MEDIAL FORM", 0xfb99 },
  { "ARABIC LETTER NGOEH ISOLATED FORM", 0xfb9a },
  { "ARABIC LETTER NGOEH FINAL FORM", 0xfb9b },
  { "ARABIC LETTER NGOEH INITIAL FORM", 0xfb9c },
  { "ARABIC LETTER NGOEH MEDIAL FORM", 0xfb9d },
  { "ARABIC LETTER NOON GHUNNA ISOLATED FORM", 0xfb9e },
  { "ARABIC LETTER NOON GHUNNA FINAL FORM", 0xfb9f },
  { "ARABIC LETTER RNOON ISOLATED FORM", 0xfba0 },
  { "ARABIC LETTER RNOON FINAL FORM", 0xfba1 },
  { "ARABIC LETTER RNOON INITIAL FORM", 0xfba2 },
  { "ARABIC LETTER RNOON MEDIAL FORM", 0xfba3 },
  { "ARABIC LETTER HEH WITH YEH ABOVE ISOLATED FORM", 0xfba4 },
  { "ARABIC LETTER HEH WITH YEH ABOVE FINAL FORM", 0xfba5 },
  { "ARABIC LETTER HEH GOAL ISOLATED FORM", 0xfba6 },
  { "ARABIC LETTER HEH GOAL FINAL FORM", 0xfba7 },
  { "ARABIC LETTER HEH GOAL INITIAL FORM", 0xfba8 },
  { "ARABIC LETTER HEH GOAL MEDIAL FORM", 0xfba9 },
  { "ARABIC LETTER HEH DOACHASHMEE ISOLATED FORM", 0xfbaa },
  { "ARABIC LETTER HEH DOACHASHMEE FINAL FORM", 0xfbab },
  { "ARABIC LETTER HEH DOACHASHMEE INITIAL FORM", 0xfbac },
  { "ARABIC LETTER HEH DOACHASHMEE MEDIAL FORM", 0xfbad },
  { "ARABIC LETTER YEH BARREE ISOLATED FORM", 0xfbae },
  { "ARABIC LETTER YEH BARREE FINAL FORM", 0xfbaf },
  { "ARABIC LETTER YEH BARREE WITH HAMZA ABOVE ISOLATED FORM", 0xfbb0 },
  { "ARABIC LETTER YEH BARREE WITH HAMZA ABOVE FINAL FORM", 0xfbb1 },
  { "ARABIC LETTER NG ISOLATED FORM", 0xfbd3 },
  { "ARABIC LETTER NG FINAL FORM", 0xfbd4 },
  { "ARABIC LETTER NG INITIAL FORM", 0xfbd5 },
  { "ARABIC LETTER NG MEDIAL FORM", 0xfbd6 },
  { "ARABIC LETTER U ISOLATED FORM", 0xfbd7 },
  { "ARABIC LETTER U FINAL FORM", 0xfbd8 },
  { "ARABIC LETTER OE ISOLATED FORM", 0xfbd9 },
  { "ARABIC LETTER OE FINAL FORM", 0xfbda },
  { "ARABIC LETTER YU ISOLATED FORM", 0xfbdb },
  { "ARABIC LETTER YU FINAL FORM", 0xfbdc },
  { "ARABIC LETTER U WITH HAMZA ABOVE ISOLATED FORM", 0xfbdd },
  { "ARABIC LETTER VE ISOLATED FORM", 0xfbde },
  { "ARABIC LETTER VE FINAL FORM", 0xfbdf },
  { "ARABIC LETTER KIRGHIZ OE ISOLATED FORM", 0xfbe0 },
  { "ARABIC LETTER KIRGHIZ OE FINAL FORM", 0xfbe1 },
  { "ARABIC LETTER KIRGHIZ YU ISOLATED FORM", 0xfbe2 },
  { "ARABIC LETTER KIRGHIZ YU FINAL FORM", 0xfbe3 },
  { "ARABIC LETTER E ISOLATED FORM", 0xfbe4 },
  { "ARABIC LETTER E FINAL FORM", 0xfbe5 },
  { "ARABIC LETTER E INITIAL FORM", 0xfbe6 },
  { "ARABIC LETTER E MEDIAL FORM", 0xfbe7 },
  { "ARABIC LETTER UIGHUR KAZAKH KIRGHIZ ALEF MAKSURA INITIAL FORM", 0xfbe8 },
  { "ARABIC LETTER UIGHUR KAZAKH KIRGHIZ ALEF MAKSURA MEDIAL FORM", 0xfbe9 },
  { "ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH ALEF ISOLATED FORM", 0xfbea },
  { "ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH ALEF FINAL FORM", 0xfbeb },
  { "ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH AE ISOLATED FORM", 0xfbec },
  { "ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH AE FINAL FORM", 0xfbed },
  { "ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH WAW ISOLATED FORM", 0xfbee },
  { "ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH WAW FINAL FORM", 0xfbef },
  { "ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH U ISOLATED FORM", 0xfbf0 },
  { "ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH U FINAL FORM", 0xfbf1 },
  { "ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH OE ISOLATED FORM", 0xfbf2 },
  { "ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH OE FINAL FORM", 0xfbf3 },
  { "ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH YU ISOLATED FORM", 0xfbf4 },
  { "ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH YU FINAL FORM", 0xfbf5 },
  { "ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH E ISOLATED FORM", 0xfbf6 },
  { "ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH E FINAL FORM", 0xfbf7 },
  { "ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH E INITIAL FORM", 0xfbf8 },
  { "ARABIC LIGATURE UIGHUR KIRGHIZ YEH WITH HAMZA ABOVE WITH ALEF MAKSURA ISOLATED FORM", 0xfbf9 },
  { "ARABIC LIGATURE UIGHUR KIRGHIZ YEH WITH HAMZA ABOVE WITH ALEF MAKSURA FINAL FORM", 0xfbfa },
  { "ARABIC LIGATURE UIGHUR KIRGHIZ YEH WITH HAMZA ABOVE WITH ALEF MAKSURA INITIAL FORM", 0xfbfb },
  { "ARABIC LETTER FARSI YEH ISOLATED FORM", 0xfbfc },
  { "ARABIC LETTER FARSI YEH FINAL FORM", 0xfbfd },
  { "ARABIC LETTER FARSI YEH INITIAL FORM", 0xfbfe },
  { "ARABIC LETTER FARSI YEH MEDIAL FORM", 0xfbff },
  { "ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH JEEM ISOLATED FORM", 0xfc00 },
  { "ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH HAH ISOLATED FORM", 0xfc01 },
  { "ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH MEEM ISOLATED FORM", 0xfc02 },
  { "ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH ALEF MAKSURA ISOLATED FORM", 0xfc03 },
  { "ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH YEH ISOLATED FORM", 0xfc04 },
  { "ARABIC LIGATURE BEH WITH JEEM ISOLATED FORM", 0xfc05 },
  { "ARABIC LIGATURE BEH WITH HAH ISOLATED FORM", 0xfc06 },
  { "ARABIC LIGATURE BEH WITH KHAH ISOLATED FORM", 0xfc07 },
  { "ARABIC LIGATURE BEH WITH MEEM ISOLATED FORM", 0xfc08 },
  { "ARABIC LIGATURE BEH WITH ALEF MAKSURA ISOLATED FORM", 0xfc09 },
  { "ARABIC LIGATURE BEH WITH YEH ISOLATED FORM", 0xfc0a },
  { "ARABIC LIGATURE TEH WITH JEEM ISOLATED FORM", 0xfc0b },
  { "ARABIC LIGATURE TEH WITH HAH ISOLATED FORM", 0xfc0c },
  { "ARABIC LIGATURE TEH WITH KHAH ISOLATED FORM", 0xfc0d },
  { "ARABIC LIGATURE TEH WITH MEEM ISOLATED FORM", 0xfc0e },
  { "ARABIC LIGATURE TEH WITH ALEF MAKSURA ISOLATED FORM", 0xfc0f },
  { "ARABIC LIGATURE TEH WITH YEH ISOLATED FORM", 0xfc10 },
  { "ARABIC LIGATURE THEH WITH JEEM ISOLATED FORM", 0xfc11 },
  { "ARABIC LIGATURE THEH WITH MEEM ISOLATED FORM", 0xfc12 },
  { "ARABIC LIGATURE THEH WITH ALEF MAKSURA ISOLATED FORM", 0xfc13 },
  { "ARABIC LIGATURE THEH WITH YEH ISOLATED FORM", 0xfc14 },
  { "ARABIC LIGATURE JEEM WITH HAH ISOLATED FORM", 0xfc15 },
  { "ARABIC LIGATURE JEEM WITH MEEM ISOLATED FORM", 0xfc16 },
  { "ARABIC LIGATURE HAH WITH JEEM ISOLATED FORM", 0xfc17 },
  { "ARABIC LIGATURE HAH WITH MEEM ISOLATED FORM", 0xfc18 },
  { "ARABIC LIGATURE KHAH WITH JEEM ISOLATED FORM", 0xfc19 },
  { "ARABIC LIGATURE KHAH WITH HAH ISOLATED FORM", 0xfc1a },
  { "ARABIC LIGATURE KHAH WITH MEEM ISOLATED FORM", 0xfc1b },
  { "ARABIC LIGATURE SEEN WITH JEEM ISOLATED FORM", 0xfc1c },
  { "ARABIC LIGATURE SEEN WITH HAH ISOLATED FORM", 0xfc1d },
  { "ARABIC LIGATURE SEEN WITH KHAH ISOLATED FORM", 0xfc1e },
  { "ARABIC LIGATURE SEEN WITH MEEM ISOLATED FORM", 0xfc1f },
  { "ARABIC LIGATURE SAD WITH HAH ISOLATED FORM", 0xfc20 },
  { "ARABIC LIGATURE SAD WITH MEEM ISOLATED FORM", 0xfc21 },
  { "ARABIC LIGATURE DAD WITH JEEM ISOLATED FORM", 0xfc22 },
  { "ARABIC LIGATURE DAD WITH HAH ISOLATED FORM", 0xfc23 },
  { "ARABIC LIGATURE DAD WITH KHAH ISOLATED FORM", 0xfc24 },
  { "ARABIC LIGATURE DAD WITH MEEM ISOLATED FORM", 0xfc25 },
  { "ARABIC LIGATURE TAH WITH HAH ISOLATED FORM", 0xfc26 },
  { "ARABIC LIGATURE TAH WITH MEEM ISOLATED FORM", 0xfc27 },
  { "ARABIC LIGATURE ZAH WITH MEEM ISOLATED FORM", 0xfc28 },
  { "ARABIC LIGATURE AIN WITH JEEM ISOLATED FORM", 0xfc29 },
  { "ARABIC LIGATURE AIN WITH MEEM ISOLATED FORM", 0xfc2a },
  { "ARABIC LIGATURE GHAIN WITH JEEM ISOLATED FORM", 0xfc2b },
  { "ARABIC LIGATURE GHAIN WITH MEEM ISOLATED FORM", 0xfc2c },
  { "ARABIC LIGATURE FEH WITH JEEM ISOLATED FORM", 0xfc2d },
  { "ARABIC LIGATURE FEH WITH HAH ISOLATED FORM", 0xfc2e },
  { "ARABIC LIGATURE FEH WITH KHAH ISOLATED FORM", 0xfc2f },
  { "ARABIC LIGATURE FEH WITH MEEM ISOLATED FORM", 0xfc30 },
  { "ARABIC LIGATURE FEH WITH ALEF MAKSURA ISOLATED FORM", 0xfc31 },
  { "ARABIC LIGATURE FEH WITH YEH ISOLATED FORM", 0xfc32 },
  { "ARABIC LIGATURE QAF WITH HAH ISOLATED FORM", 0xfc33 },
  { "ARABIC LIGATURE QAF WITH MEEM ISOLATED FORM", 0xfc34 },
  { "ARABIC LIGATURE QAF WITH ALEF MAKSURA ISOLATED FORM", 0xfc35 },
  { "ARABIC LIGATURE QAF WITH YEH ISOLATED FORM", 0xfc36 },
  { "ARABIC LIGATURE KAF WITH ALEF ISOLATED FORM", 0xfc37 },
  { "ARABIC LIGATURE KAF WITH JEEM ISOLATED FORM", 0xfc38 },
  { "ARABIC LIGATURE KAF WITH HAH ISOLATED FORM", 0xfc39 },
  { "ARABIC LIGATURE KAF WITH KHAH ISOLATED FORM", 0xfc3a },
  { "ARABIC LIGATURE KAF WITH LAM ISOLATED FORM", 0xfc3b },
  { "ARABIC LIGATURE KAF WITH MEEM ISOLATED FORM", 0xfc3c },
  { "ARABIC LIGATURE KAF WITH ALEF MAKSURA ISOLATED FORM", 0xfc3d },
  { "ARABIC LIGATURE KAF WITH YEH ISOLATED FORM", 0xfc3e },
  { "ARABIC LIGATURE LAM WITH JEEM ISOLATED FORM", 0xfc3f },
  { "ARABIC LIGATURE LAM WITH HAH ISOLATED FORM", 0xfc40 },
  { "ARABIC LIGATURE LAM WITH KHAH ISOLATED FORM", 0xfc41 },
  { "ARABIC LIGATURE LAM WITH MEEM ISOLATED FORM", 0xfc42 },
  { "ARABIC LIGATURE LAM WITH ALEF MAKSURA ISOLATED FORM", 0xfc43 },
  { "ARABIC LIGATURE LAM WITH YEH ISOLATED FORM", 0xfc44 },
  { "ARABIC LIGATURE MEEM WITH JEEM ISOLATED FORM", 0xfc45 },
  { "ARABIC LIGATURE MEEM WITH HAH ISOLATED FORM", 0xfc46 },
  { "ARABIC LIGATURE MEEM WITH KHAH ISOLATED FORM", 0xfc47 },
  { "ARABIC LIGATURE MEEM WITH MEEM ISOLATED FORM", 0xfc48 },
  { "ARABIC LIGATURE MEEM WITH ALEF MAKSURA ISOLATED FORM", 0xfc49 },
  { "ARABIC LIGATURE MEEM WITH YEH ISOLATED FORM", 0xfc4a },
  { "ARABIC LIGATURE NOON WITH JEEM ISOLATED FORM", 0xfc4b },
  { "ARABIC LIGATURE NOON WITH HAH ISOLATED FORM", 0xfc4c },
  { "ARABIC LIGATURE NOON WITH KHAH ISOLATED FORM", 0xfc4d },
  { "ARABIC LIGATURE NOON WITH MEEM ISOLATED FORM", 0xfc4e },
  { "ARABIC LIGATURE NOON WITH ALEF MAKSURA ISOLATED FORM", 0xfc4f },
  { "ARABIC LIGATURE NOON WITH YEH ISOLATED FORM", 0xfc50 },
  { "ARABIC LIGATURE HEH WITH JEEM ISOLATED FORM", 0xfc51 },
  { "ARABIC LIGATURE HEH WITH MEEM ISOLATED FORM", 0xfc52 },
  { "ARABIC LIGATURE HEH WITH ALEF MAKSURA ISOLATED FORM", 0xfc53 },
  { "ARABIC LIGATURE HEH WITH YEH ISOLATED FORM", 0xfc54 },
  { "ARABIC LIGATURE YEH WITH JEEM ISOLATED FORM", 0xfc55 },
  { "ARABIC LIGATURE YEH WITH HAH ISOLATED FORM", 0xfc56 },
  { "ARABIC LIGATURE YEH WITH KHAH ISOLATED FORM", 0xfc57 },
  { "ARABIC LIGATURE YEH WITH MEEM ISOLATED FORM", 0xfc58 },
  { "ARABIC LIGATURE YEH WITH ALEF MAKSURA ISOLATED FORM", 0xfc59 },
  { "ARABIC LIGATURE YEH WITH YEH ISOLATED FORM", 0xfc5a },
  { "ARABIC LIGATURE THAL WITH SUPERSCRIPT ALEF ISOLATED FORM", 0xfc5b },
  { "ARABIC LIGATURE REH WITH SUPERSCRIPT ALEF ISOLATED FORM", 0xfc5c },
  { "ARABIC LIGATURE ALEF MAKSURA WITH SUPERSCRIPT ALEF ISOLATED FORM", 0xfc5d },
  { "ARABIC LIGATURE SHADDA WITH DAMMATAN ISOLATED FORM", 0xfc5e },
  { "ARABIC LIGATURE SHADDA WITH KASRATAN ISOLATED FORM", 0xfc5f },
  { "ARABIC LIGATURE SHADDA WITH FATHA ISOLATED FORM", 0xfc60 },
  { "ARABIC LIGATURE SHADDA WITH DAMMA ISOLATED FORM", 0xfc61 },
  { "ARABIC LIGATURE SHADDA WITH KASRA ISOLATED FORM", 0xfc62 },
  { "ARABIC LIGATURE SHADDA WITH SUPERSCRIPT ALEF ISOLATED FORM", 0xfc63 },
  { "ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH REH FINAL FORM", 0xfc64 },
  { "ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH ZAIN FINAL FORM", 0xfc65 },
  { "ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH MEEM FINAL FORM", 0xfc66 },
  { "ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH NOON FINAL FORM", 0xfc67 },
  { "ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH ALEF MAKSURA FINAL FORM", 0xfc68 },
  { "ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH YEH FINAL FORM", 0xfc69 },
  { "ARABIC LIGATURE BEH WITH REH FINAL FORM", 0xfc6a },
  { "ARABIC LIGATURE BEH WITH ZAIN FINAL FORM", 0xfc6b },
  { "ARABIC LIGATURE BEH WITH MEEM FINAL FORM", 0xfc6c },
  { "ARABIC LIGATURE BEH WITH NOON FINAL FORM", 0xfc6d },
  { "ARABIC LIGATURE BEH WITH ALEF MAKSURA FINAL FORM", 0xfc6e },
  { "ARABIC LIGATURE BEH WITH YEH FINAL FORM", 0xfc6f },
  { "ARABIC LIGATURE TEH WITH REH FINAL FORM", 0xfc70 },
  { "ARABIC LIGATURE TEH WITH ZAIN FINAL FORM", 0xfc71 },
  { "ARABIC LIGATURE TEH WITH MEEM FINAL FORM", 0xfc72 },
  { "ARABIC LIGATURE TEH WITH NOON FINAL FORM", 0xfc73 },
  { "ARABIC LIGATURE TEH WITH ALEF MAKSURA FINAL FORM", 0xfc74 },
  { "ARABIC LIGATURE TEH WITH YEH FINAL FORM", 0xfc75 },
  { "ARABIC LIGATURE THEH WITH REH FINAL FORM", 0xfc76 },
  { "ARABIC LIGATURE THEH WITH ZAIN FINAL FORM", 0xfc77 },
  { "ARABIC LIGATURE THEH WITH MEEM FINAL FORM", 0xfc78 },
  { "ARABIC LIGATURE THEH WITH NOON FINAL FORM", 0xfc79 },
  { "ARABIC LIGATURE THEH WITH ALEF MAKSURA FINAL FORM", 0xfc7a },
  { "ARABIC LIGATURE THEH WITH YEH FINAL FORM", 0xfc7b },
  { "ARABIC LIGATURE FEH WITH ALEF MAKSURA FINAL FORM", 0xfc7c },
  { "ARABIC LIGATURE FEH WITH YEH FINAL FORM", 0xfc7d },
  { "ARABIC LIGATURE QAF WITH ALEF MAKSURA FINAL FORM", 0xfc7e },
  { "ARABIC LIGATURE QAF WITH YEH FINAL FORM", 0xfc7f },
  { "ARABIC LIGATURE KAF WITH ALEF FINAL FORM", 0xfc80 },
  { "ARABIC LIGATURE KAF WITH LAM FINAL FORM", 0xfc81 },
  { "ARABIC LIGATURE KAF WITH MEEM FINAL FORM", 0xfc82 },
  { "ARABIC LIGATURE KAF WITH ALEF MAKSURA FINAL FORM", 0xfc83 },
  { "ARABIC LIGATURE KAF WITH YEH FINAL FORM", 0xfc84 },
  { "ARABIC LIGATURE LAM WITH MEEM FINAL FORM", 0xfc85 },
  { "ARABIC LIGATURE LAM WITH ALEF MAKSURA FINAL FORM", 0xfc86 },
  { "ARABIC LIGATURE LAM WITH YEH FINAL FORM", 0xfc87 },
  { "ARABIC LIGATURE MEEM WITH ALEF FINAL FORM", 0xfc88 },
  { "ARABIC LIGATURE MEEM WITH MEEM FINAL FORM", 0xfc89 },
  { "ARABIC LIGATURE NOON WITH REH FINAL FORM", 0xfc8a },
  { "ARABIC LIGATURE NOON WITH ZAIN FINAL FORM", 0xfc8b },
  { "ARABIC LIGATURE NOON WITH MEEM FINAL FORM", 0xfc8c },
  { "ARABIC LIGATURE NOON WITH NOON FINAL FORM", 0xfc8d },
  { "ARABIC LIGATURE NOON WITH ALEF MAKSURA FINAL FORM", 0xfc8e },
  { "ARABIC LIGATURE NOON WITH YEH FINAL FORM", 0xfc8f },
  { "ARABIC LIGATURE ALEF MAKSURA WITH SUPERSCRIPT ALEF FINAL FORM", 0xfc90 },
  { "ARABIC LIGATURE YEH WITH REH FINAL FORM", 0xfc91 },
  { "ARABIC LIGATURE YEH WITH ZAIN FINAL FORM", 0xfc92 },
  { "ARABIC LIGATURE YEH WITH MEEM FINAL FORM", 0xfc93 },
  { "ARABIC LIGATURE YEH WITH NOON FINAL FORM", 0xfc94 },
  { "ARABIC LIGATURE YEH WITH ALEF MAKSURA FINAL FORM", 0xfc95 },
  { "ARABIC LIGATURE YEH WITH YEH FINAL FORM", 0xfc96 },
  { "ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH JEEM INITIAL FORM", 0xfc97 },
  { "ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH HAH INITIAL FORM", 0xfc98 },
  { "ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH KHAH INITIAL FORM", 0xfc99 },
  { "ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH MEEM INITIAL FORM", 0xfc9a },
  { "ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH HEH INITIAL FORM", 0xfc9b },
  { "ARABIC LIGATURE BEH WITH JEEM INITIAL FORM", 0xfc9c },
  { "ARABIC LIGATURE BEH WITH HAH INITIAL FORM", 0xfc9d },
  { "ARABIC LIGATURE BEH WITH KHAH INITIAL FORM", 0xfc9e },
  { "ARABIC LIGATURE BEH WITH MEEM INITIAL FORM", 0xfc9f },
  { "ARABIC LIGATURE BEH WITH HEH INITIAL FORM", 0xfca0 },
  { "ARABIC LIGATURE TEH WITH JEEM INITIAL FORM", 0xfca1 },
  { "ARABIC LIGATURE TEH WITH HAH INITIAL FORM", 0xfca2 },
  { "ARABIC LIGATURE TEH WITH KHAH INITIAL FORM", 0xfca3 },
  { "ARABIC LIGATURE TEH WITH MEEM INITIAL FORM", 0xfca4 },
  { "ARABIC LIGATURE TEH WITH HEH INITIAL FORM", 0xfca5 },
  { "ARABIC LIGATURE THEH WITH MEEM INITIAL FORM", 0xfca6 },
  { "ARABIC LIGATURE JEEM WITH HAH INITIAL FORM", 0xfca7 },
  { "ARABIC LIGATURE JEEM WITH MEEM INITIAL FORM", 0xfca8 },
  { "ARABIC LIGATURE HAH WITH JEEM INITIAL FORM", 0xfca9 },
  { "ARABIC LIGATURE HAH WITH MEEM INITIAL FORM", 0xfcaa },
  { "ARABIC LIGATURE KHAH WITH JEEM INITIAL FORM", 0xfcab },
  { "ARABIC LIGATURE KHAH WITH MEEM INITIAL FORM", 0xfcac },
  { "ARABIC LIGATURE SEEN WITH JEEM INITIAL FORM", 0xfcad },
  { "ARABIC LIGATURE SEEN WITH HAH INITIAL FORM", 0xfcae },
  { "ARABIC LIGATURE SEEN WITH KHAH INITIAL FORM", 0xfcaf },
  { "ARABIC LIGATURE SEEN WITH MEEM INITIAL FORM", 0xfcb0 },
  { "ARABIC LIGATURE SAD WITH HAH INITIAL FORM", 0xfcb1 },
  { "ARABIC LIGATURE SAD WITH KHAH INITIAL FORM", 0xfcb2 },
  { "ARABIC LIGATURE SAD WITH MEEM INITIAL FORM", 0xfcb3 },
  { "ARABIC LIGATURE DAD WITH JEEM INITIAL FORM", 0xfcb4 },
  { "ARABIC LIGATURE DAD WITH HAH INITIAL FORM", 0xfcb5 },
  { "ARABIC LIGATURE DAD WITH KHAH INITIAL FORM", 0xfcb6 },
  { "ARABIC LIGATURE DAD WITH MEEM INITIAL FORM", 0xfcb7 },
  { "ARABIC LIGATURE TAH WITH HAH INITIAL FORM", 0xfcb8 },
  { "ARABIC LIGATURE ZAH WITH MEEM INITIAL FORM", 0xfcb9 },
  { "ARABIC LIGATURE AIN WITH JEEM INITIAL FORM", 0xfcba },
  { "ARABIC LIGATURE AIN WITH MEEM INITIAL FORM", 0xfcbb },
  { "ARABIC LIGATURE GHAIN WITH JEEM INITIAL FORM", 0xfcbc },
  { "ARABIC LIGATURE GHAIN WITH MEEM INITIAL FORM", 0xfcbd },
  { "ARABIC LIGATURE FEH WITH JEEM INITIAL FORM", 0xfcbe },
  { "ARABIC LIGATURE FEH WITH HAH INITIAL FORM", 0xfcbf },
  { "ARABIC LIGATURE FEH WITH KHAH INITIAL FORM", 0xfcc0 },
  { "ARABIC LIGATURE FEH WITH MEEM INITIAL FORM", 0xfcc1 },
  { "ARABIC LIGATURE QAF WITH HAH INITIAL FORM", 0xfcc2 },
  { "ARABIC LIGATURE QAF WITH MEEM INITIAL FORM", 0xfcc3 },
  { "ARABIC LIGATURE KAF WITH JEEM INITIAL FORM", 0xfcc4 },
  { "ARABIC LIGATURE KAF WITH HAH INITIAL FORM", 0xfcc5 },
  { "ARABIC LIGATURE KAF WITH KHAH INITIAL FORM", 0xfcc6 },
  { "ARABIC LIGATURE KAF WITH LAM INITIAL FORM", 0xfcc7 },
  { "ARABIC LIGATURE KAF WITH MEEM INITIAL FORM", 0xfcc8 },
  { "ARABIC LIGATURE LAM WITH JEEM INITIAL FORM", 0xfcc9 },
  { "ARABIC LIGATURE LAM WITH HAH INITIAL FORM", 0xfcca },
  { "ARABIC LIGATURE LAM WITH KHAH INITIAL FORM", 0xfccb },
  { "ARABIC LIGATURE LAM WITH MEEM INITIAL FORM", 0xfccc },
  { "ARABIC LIGATURE LAM WITH HEH INITIAL FORM", 0xfccd },
  { "ARABIC LIGATURE MEEM WITH JEEM INITIAL FORM", 0xfcce },
  { "ARABIC LIGATURE MEEM WITH HAH INITIAL FORM", 0xfccf },
  { "ARABIC LIGATURE MEEM WITH KHAH INITIAL FORM", 0xfcd0 },
  { "ARABIC LIGATURE MEEM WITH MEEM INITIAL FORM", 0xfcd1 },
  { "ARABIC LIGATURE NOON WITH JEEM INITIAL FORM", 0xfcd2 },
  { "ARABIC LIGATURE NOON WITH HAH INITIAL FORM", 0xfcd3 },
  { "ARABIC LIGATURE NOON WITH KHAH INITIAL FORM", 0xfcd4 },
  { "ARABIC LIGATURE NOON WITH MEEM INITIAL FORM", 0xfcd5 },
  { "ARABIC LIGATURE NOON WITH HEH INITIAL FORM", 0xfcd6 },
  { "ARABIC LIGATURE HEH WITH JEEM INITIAL FORM", 0xfcd7 },
  { "ARABIC LIGATURE HEH WITH MEEM INITIAL FORM", 0xfcd8 },
  { "ARABIC LIGATURE HEH WITH SUPERSCRIPT ALEF INITIAL FORM", 0xfcd9 },
  { "ARABIC LIGATURE YEH WITH JEEM INITIAL FORM", 0xfcda },
  { "ARABIC LIGATURE YEH WITH HAH INITIAL FORM", 0xfcdb },
  { "ARABIC LIGATURE YEH WITH KHAH INITIAL FORM", 0xfcdc },
  { "ARABIC LIGATURE YEH WITH MEEM INITIAL FORM", 0xfcdd },
  { "ARABIC LIGATURE YEH WITH HEH INITIAL FORM", 0xfcde },
  { "ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH MEEM MEDIAL FORM", 0xfcdf },
  { "ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH HEH MEDIAL FORM", 0xfce0 },
  { "ARABIC LIGATURE BEH WITH MEEM MEDIAL FORM", 0xfce1 },
  { "ARABIC LIGATURE BEH WITH HEH MEDIAL FORM", 0xfce2 },
  { "ARABIC LIGATURE TEH WITH MEEM MEDIAL FORM", 0xfce3 },
  { "ARABIC LIGATURE TEH WITH HEH MEDIAL FORM", 0xfce4 },
  { "ARABIC LIGATURE THEH WITH MEEM MEDIAL FORM", 0xfce5 },
  { "ARABIC LIGATURE THEH WITH HEH MEDIAL FORM", 0xfce6 },
  { "ARABIC LIGATURE SEEN WITH MEEM MEDIAL FORM", 0xfce7 },
  { "ARABIC LIGATURE SEEN WITH HEH MEDIAL FORM", 0xfce8 },
  { "ARABIC LIGATURE SHEEN WITH MEEM MEDIAL FORM", 0xfce9 },
  { "ARABIC LIGATURE SHEEN WITH HEH MEDIAL FORM", 0xfcea },
  { "ARABIC LIGATURE KAF WITH LAM MEDIAL FORM", 0xfceb },
  { "ARABIC LIGATURE KAF WITH MEEM MEDIAL FORM", 0xfcec },
  { "ARABIC LIGATURE LAM WITH MEEM MEDIAL FORM", 0xfced },
  { "ARABIC LIGATURE NOON WITH MEEM MEDIAL FORM", 0xfcee },
  { "ARABIC LIGATURE NOON WITH HEH MEDIAL FORM", 0xfcef },
  { "ARABIC LIGATURE YEH WITH MEEM MEDIAL FORM", 0xfcf0 },
  { "ARABIC LIGATURE YEH WITH HEH MEDIAL FORM", 0xfcf1 },
  { "ARABIC LIGATURE SHADDA WITH FATHA MEDIAL FORM", 0xfcf2 },
  { "ARABIC LIGATURE SHADDA WITH DAMMA MEDIAL FORM", 0xfcf3 },
  { "ARABIC LIGATURE SHADDA WITH KASRA MEDIAL FORM", 0xfcf4 },
  { "ARABIC LIGATURE TAH WITH ALEF MAKSURA ISOLATED FORM", 0xfcf5 },
  { "ARABIC LIGATURE TAH WITH YEH ISOLATED FORM", 0xfcf6 },
  { "ARABIC LIGATURE AIN WITH ALEF MAKSURA ISOLATED FORM", 0xfcf7 },
  { "ARABIC LIGATURE AIN WITH YEH ISOLATED FORM", 0xfcf8 },
  { "ARABIC LIGATURE GHAIN WITH ALEF MAKSURA ISOLATED FORM", 0xfcf9 },
  { "ARABIC LIGATURE GHAIN WITH YEH ISOLATED FORM", 0xfcfa },
  { "ARABIC LIGATURE SEEN WITH ALEF MAKSURA ISOLATED FORM", 0xfcfb },
  { "ARABIC LIGATURE SEEN WITH YEH ISOLATED FORM", 0xfcfc },
  { "ARABIC LIGATURE SHEEN WITH ALEF MAKSURA ISOLATED FORM", 0xfcfd },
  { "ARABIC LIGATURE SHEEN WITH YEH ISOLATED FORM", 0xfcfe },
  { "ARABIC LIGATURE HAH WITH ALEF MAKSURA ISOLATED FORM", 0xfcff },
  { "ARABIC LIGATURE HAH WITH YEH ISOLATED FORM", 0xfd00 },
  { "ARABIC LIGATURE JEEM WITH ALEF MAKSURA ISOLATED FORM", 0xfd01 },
  { "ARABIC LIGATURE JEEM WITH YEH ISOLATED FORM", 0xfd02 },
  { "ARABIC LIGATURE KHAH WITH ALEF MAKSURA ISOLATED FORM", 0xfd03 },
  { "ARABIC LIGATURE KHAH WITH YEH ISOLATED FORM", 0xfd04 },
  { "ARABIC LIGATURE SAD WITH ALEF MAKSURA ISOLATED FORM", 0xfd05 },
  { "ARABIC LIGATURE SAD WITH YEH ISOLATED FORM", 0xfd06 },
  { "ARABIC LIGATURE DAD WITH ALEF MAKSURA ISOLATED FORM", 0xfd07 },
  { "ARABIC LIGATURE DAD WITH YEH ISOLATED FORM", 0xfd08 },
  { "ARABIC LIGATURE SHEEN WITH JEEM ISOLATED FORM", 0xfd09 },
  { "ARABIC LIGATURE SHEEN WITH HAH ISOLATED FORM", 0xfd0a },
  { "ARABIC LIGATURE SHEEN WITH KHAH ISOLATED FORM", 0xfd0b },
  { "ARABIC LIGATURE SHEEN WITH MEEM ISOLATED FORM", 0xfd0c },
  { "ARABIC LIGATURE SHEEN WITH REH ISOLATED FORM", 0xfd0d },
  { "ARABIC LIGATURE SEEN WITH REH ISOLATED FORM", 0xfd0e },
  { "ARABIC LIGATURE SAD WITH REH ISOLATED FORM", 0xfd0f },
  { "ARABIC LIGATURE DAD WITH REH ISOLATED FORM", 0xfd10 },
  { "ARABIC LIGATURE TAH WITH ALEF MAKSURA FINAL FORM", 0xfd11 },
  { "ARABIC LIGATURE TAH WITH YEH FINAL FORM", 0xfd12 },
  { "ARABIC LIGATURE AIN WITH ALEF MAKSURA FINAL FORM", 0xfd13 },
  { "ARABIC LIGATURE AIN WITH YEH FINAL FORM", 0xfd14 },
  { "ARABIC LIGATURE GHAIN WITH ALEF MAKSURA FINAL FORM", 0xfd15 },
  { "ARABIC LIGATURE GHAIN WITH YEH FINAL FORM", 0xfd16 },
  { "ARABIC LIGATURE SEEN WITH ALEF MAKSURA FINAL FORM", 0xfd17 },
  { "ARABIC LIGATURE SEEN WITH YEH FINAL FORM", 0xfd18 },
  { "ARABIC LIGATURE SHEEN WITH ALEF MAKSURA FINAL FORM", 0xfd19 },
  { "ARABIC LIGATURE SHEEN WITH YEH FINAL FORM", 0xfd1a },
  { "ARABIC LIGATURE HAH WITH ALEF MAKSURA FINAL FORM", 0xfd1b },
  { "ARABIC LIGATURE HAH WITH YEH FINAL FORM", 0xfd1c },
  { "ARABIC LIGATURE JEEM WITH ALEF MAKSURA FINAL FORM", 0xfd1d },
  { "ARABIC LIGATURE JEEM WITH YEH FINAL FORM", 0xfd1e },
  { "ARABIC LIGATURE KHAH WITH ALEF MAKSURA FINAL FORM", 0xfd1f },
  { "ARABIC LIGATURE KHAH WITH YEH FINAL FORM", 0xfd20 },
  { "ARABIC LIGATURE SAD WITH ALEF MAKSURA FINAL FORM", 0xfd21 },
  { "ARABIC LIGATURE SAD WITH YEH FINAL FORM", 0xfd22 },
  { "ARABIC LIGATURE DAD WITH ALEF MAKSURA FINAL FORM", 0xfd23 },
  { "ARABIC LIGATURE DAD WITH YEH FINAL FORM", 0xfd24 },
  { "ARABIC LIGATURE SHEEN WITH JEEM FINAL FORM", 0xfd25 },
  { "ARABIC LIGATURE SHEEN WITH HAH FINAL FORM", 0xfd26 },
  { "ARABIC LIGATURE SHEEN WITH KHAH FINAL FORM", 0xfd27 },
  { "ARABIC LIGATURE SHEEN WITH MEEM FINAL FORM", 0xfd28 },
  { "ARABIC LIGATURE SHEEN WITH REH FINAL FORM", 0xfd29 },
  { "ARABIC LIGATURE SEEN WITH REH FINAL FORM", 0xfd2a },
  { "ARABIC LIGATURE SAD WITH REH FINAL FORM", 0xfd2b },
  { "ARABIC LIGATURE DAD WITH REH FINAL FORM", 0xfd2c },
  { "ARABIC LIGATURE SHEEN WITH JEEM INITIAL FORM", 0xfd2d },
  { "ARABIC LIGATURE SHEEN WITH HAH INITIAL FORM", 0xfd2e },
  { "ARABIC LIGATURE SHEEN WITH KHAH INITIAL FORM", 0xfd2f },
  { "ARABIC LIGATURE SHEEN WITH MEEM INITIAL FORM", 0xfd30 },
  { "ARABIC LIGATURE SEEN WITH HEH INITIAL FORM", 0xfd31 },
  { "ARABIC LIGATURE SHEEN WITH HEH INITIAL FORM", 0xfd32 },
  { "ARABIC LIGATURE TAH WITH MEEM INITIAL FORM", 0xfd33 },
  { "ARABIC LIGATURE SEEN WITH JEEM MEDIAL FORM", 0xfd34 },
  { "ARABIC LIGATURE SEEN WITH HAH MEDIAL FORM", 0xfd35 },
  { "ARABIC LIGATURE SEEN WITH KHAH MEDIAL FORM", 0xfd36 },
  { "ARABIC LIGATURE SHEEN WITH JEEM MEDIAL FORM", 0xfd37 },
  { "ARABIC LIGATURE SHEEN WITH HAH MEDIAL FORM", 0xfd38 },
  { "ARABIC LIGATURE SHEEN WITH KHAH MEDIAL FORM", 0xfd39 },
  { "ARABIC LIGATURE TAH WITH MEEM MEDIAL FORM", 0xfd3a },
  { "ARABIC LIGATURE ZAH WITH MEEM MEDIAL FORM", 0xfd3b },
  { "ARABIC LIGATURE ALEF WITH FATHATAN FINAL FORM", 0xfd3c },
  { "ARABIC LIGATURE ALEF WITH FATHATAN ISOLATED FORM", 0xfd3d },
  { "ORNATE LEFT PARENTHESIS", 0xfd3e },
  { "ORNATE RIGHT PARENTHESIS", 0xfd3f },
  { "ARABIC LIGATURE TEH WITH JEEM WITH MEEM INITIAL FORM", 0xfd50 },
  { "ARABIC LIGATURE TEH WITH HAH WITH JEEM FINAL FORM", 0xfd51 },
  { "ARABIC LIGATURE TEH WITH HAH WITH JEEM INITIAL FORM", 0xfd52 },
  { "ARABIC LIGATURE TEH WITH HAH WITH MEEM INITIAL FORM", 0xfd53 },
  { "ARABIC LIGATURE TEH WITH KHAH WITH MEEM INITIAL FORM", 0xfd54 },
  { "ARABIC LIGATURE TEH WITH MEEM WITH JEEM INITIAL FORM", 0xfd55 },
  { "ARABIC LIGATURE TEH WITH MEEM WITH HAH INITIAL FORM", 0xfd56 },
  { "ARABIC LIGATURE TEH WITH MEEM WITH KHAH INITIAL FORM", 0xfd57 },
  { "ARABIC LIGATURE JEEM WITH MEEM WITH HAH FINAL FORM", 0xfd58 },
  { "ARABIC LIGATURE JEEM WITH MEEM WITH HAH INITIAL FORM", 0xfd59 },
  { "ARABIC LIGATURE HAH WITH MEEM WITH YEH FINAL FORM", 0xfd5a },
  { "ARABIC LIGATURE HAH WITH MEEM WITH ALEF MAKSURA FINAL FORM", 0xfd5b },
  { "ARABIC LIGATURE SEEN WITH HAH WITH JEEM INITIAL FORM", 0xfd5c },
  { "ARABIC LIGATURE SEEN WITH JEEM WITH HAH INITIAL FORM", 0xfd5d },
  { "ARABIC LIGATURE SEEN WITH JEEM WITH ALEF MAKSURA FINAL FORM", 0xfd5e },
  { "ARABIC LIGATURE SEEN WITH MEEM WITH HAH FINAL FORM", 0xfd5f },
  { "ARABIC LIGATURE SEEN WITH MEEM WITH HAH INITIAL FORM", 0xfd60 },
  { "ARABIC LIGATURE SEEN WITH MEEM WITH JEEM INITIAL FORM", 0xfd61 },
  { "ARABIC LIGATURE SEEN WITH MEEM WITH MEEM FINAL FORM", 0xfd62 },
  { "ARABIC LIGATURE SEEN WITH MEEM WITH MEEM INITIAL FORM", 0xfd63 },
  { "ARABIC LIGATURE SAD WITH HAH WITH HAH FINAL FORM", 0xfd64 },
  { "ARABIC LIGATURE SAD WITH HAH WITH HAH INITIAL FORM", 0xfd65 },
  { "ARABIC LIGATURE SAD WITH MEEM WITH MEEM FINAL FORM", 0xfd66 },
  { "ARABIC LIGATURE SHEEN WITH HAH WITH MEEM FINAL FORM", 0xfd67 },
  { "ARABIC LIGATURE SHEEN WITH HAH WITH MEEM INITIAL FORM", 0xfd68 },
  { "ARABIC LIGATURE SHEEN WITH JEEM WITH YEH FINAL FORM", 0xfd69 },
  { "ARABIC LIGATURE SHEEN WITH MEEM WITH KHAH FINAL FORM", 0xfd6a },
  { "ARABIC LIGATURE SHEEN WITH MEEM WITH KHAH INITIAL FORM", 0xfd6b },
  { "ARABIC LIGATURE SHEEN WITH MEEM WITH MEEM FINAL FORM", 0xfd6c },
  { "ARABIC LIGATURE SHEEN WITH MEEM WITH MEEM INITIAL FORM", 0xfd6d },
  { "ARABIC LIGATURE DAD WITH HAH WITH ALEF MAKSURA FINAL FORM", 0xfd6e },
  { "ARABIC LIGATURE DAD WITH KHAH WITH MEEM FINAL FORM", 0xfd6f },
  { "ARABIC LIGATURE DAD WITH KHAH WITH MEEM INITIAL FORM", 0xfd70 },
  { "ARABIC LIGATURE TAH WITH MEEM WITH HAH FINAL FORM", 0xfd71 },
  { "ARABIC LIGATURE TAH WITH MEEM WITH HAH INITIAL FORM", 0xfd72 },
  { "ARABIC LIGATURE TAH WITH MEEM WITH MEEM INITIAL FORM", 0xfd73 },
  { "ARABIC LIGATURE TAH WITH MEEM WITH YEH FINAL FORM", 0xfd74 },
  { "ARABIC LIGATURE AIN WITH JEEM WITH MEEM FINAL FORM", 0xfd75 },
  { "ARABIC LIGATURE AIN WITH MEEM WITH MEEM FINAL FORM", 0xfd76 },
  { "ARABIC LIGATURE AIN WITH MEEM WITH MEEM INITIAL FORM", 0xfd77 },
  { "ARABIC LIGATURE AIN WITH MEEM WITH ALEF MAKSURA FINAL FORM", 0xfd78 },
  { "ARABIC LIGATURE GHAIN WITH MEEM WITH MEEM FINAL FORM", 0xfd79 },
  { "ARABIC LIGATURE GHAIN WITH MEEM WITH YEH FINAL FORM", 0xfd7a },
  { "ARABIC LIGATURE GHAIN WITH MEEM WITH ALEF MAKSURA FINAL FORM", 0xfd7b },
  { "ARABIC LIGATURE FEH WITH KHAH WITH MEEM FINAL FORM", 0xfd7c },
  { "ARABIC LIGATURE FEH WITH KHAH WITH MEEM INITIAL FORM", 0xfd7d },
  { "ARABIC LIGATURE QAF WITH MEEM WITH HAH FINAL FORM", 0xfd7e },
  { "ARABIC LIGATURE QAF WITH MEEM WITH MEEM FINAL FORM", 0xfd7f },
  { "ARABIC LIGATURE LAM WITH HAH WITH MEEM FINAL FORM", 0xfd80 },
  { "ARABIC LIGATURE LAM WITH HAH WITH YEH FINAL FORM", 0xfd81 },
  { "ARABIC LIGATURE LAM WITH HAH WITH ALEF MAKSURA FINAL FORM", 0xfd82 },
  { "ARABIC LIGATURE LAM WITH JEEM WITH JEEM INITIAL FORM", 0xfd83 },
  { "ARABIC LIGATURE LAM WITH JEEM WITH JEEM FINAL FORM", 0xfd84 },
  { "ARABIC LIGATURE LAM WITH KHAH WITH MEEM FINAL FORM", 0xfd85 },
  { "ARABIC LIGATURE LAM WITH KHAH WITH MEEM INITIAL FORM", 0xfd86 },
  { "ARABIC LIGATURE LAM WITH MEEM WITH HAH FINAL FORM", 0xfd87 },
  { "ARABIC LIGATURE LAM WITH MEEM WITH HAH INITIAL FORM", 0xfd88 },
  { "ARABIC LIGATURE MEEM WITH HAH WITH JEEM INITIAL FORM", 0xfd89 },
  { "ARABIC LIGATURE MEEM WITH HAH WITH MEEM INITIAL FORM", 0xfd8a },
  { "ARABIC LIGATURE MEEM WITH HAH WITH YEH FINAL FORM", 0xfd8b },
  { "ARABIC LIGATURE MEEM WITH JEEM WITH HAH INITIAL FORM", 0xfd8c },
  { "ARABIC LIGATURE MEEM WITH JEEM WITH MEEM INITIAL FORM", 0xfd8d },
  { "ARABIC LIGATURE MEEM WITH KHAH WITH JEEM INITIAL FORM", 0xfd8e },
  { "ARABIC LIGATURE MEEM WITH KHAH WITH MEEM INITIAL FORM", 0xfd8f },
  { "ARABIC LIGATURE MEEM WITH JEEM WITH KHAH INITIAL FORM", 0xfd92 },
  { "ARABIC LIGATURE HEH WITH MEEM WITH JEEM INITIAL FORM", 0xfd93 },
  { "ARABIC LIGATURE HEH WITH MEEM WITH MEEM INITIAL FORM", 0xfd94 },
  { "ARABIC LIGATURE NOON WITH HAH WITH MEEM INITIAL FORM", 0xfd95 },
  { "ARABIC LIGATURE NOON WITH HAH WITH ALEF MAKSURA FINAL FORM", 0xfd96 },
  { "ARABIC LIGATURE NOON WITH JEEM WITH MEEM FINAL FORM", 0xfd97 },
  { "ARABIC LIGATURE NOON WITH JEEM WITH MEEM INITIAL FORM", 0xfd98 },
  { "ARABIC LIGATURE NOON WITH JEEM WITH ALEF MAKSURA FINAL FORM", 0xfd99 },
  { "ARABIC LIGATURE NOON WITH MEEM WITH YEH FINAL FORM", 0xfd9a },
  { "ARABIC LIGATURE NOON WITH MEEM WITH ALEF MAKSURA FINAL FORM", 0xfd9b },
  { "ARABIC LIGATURE YEH WITH MEEM WITH MEEM FINAL FORM", 0xfd9c },
  { "ARABIC LIGATURE YEH WITH MEEM WITH MEEM INITIAL FORM", 0xfd9d },
  { "ARABIC LIGATURE BEH WITH KHAH WITH YEH FINAL FORM", 0xfd9e },
  { "ARABIC LIGATURE TEH WITH JEEM WITH YEH FINAL FORM", 0xfd9f },
  { "ARABIC LIGATURE TEH WITH JEEM WITH ALEF MAKSURA FINAL FORM", 0xfda0 },
  { "ARABIC LIGATURE TEH WITH KHAH WITH YEH FINAL FORM", 0xfda1 },
  { "ARABIC LIGATURE TEH WITH KHAH WITH ALEF MAKSURA FINAL FORM", 0xfda2 },
  { "ARABIC LIGATURE TEH WITH MEEM WITH YEH FINAL FORM", 0xfda3 },
  { "ARABIC LIGATURE TEH WITH MEEM WITH ALEF MAKSURA FINAL FORM", 0xfda4 },
  { "ARABIC LIGATURE JEEM WITH MEEM WITH YEH FINAL FORM", 0xfda5 },
  { "ARABIC LIGATURE JEEM WITH HAH WITH ALEF MAKSURA FINAL FORM", 0xfda6 },
  { "ARABIC LIGATURE JEEM WITH MEEM WITH ALEF MAKSURA FINAL FORM", 0xfda7 },
  { "ARABIC LIGATURE SEEN WITH KHAH WITH ALEF MAKSURA FINAL FORM", 0xfda8 },
  { "ARABIC LIGATURE SAD WITH HAH WITH YEH FINAL FORM", 0xfda9 },
  { "ARABIC LIGATURE SHEEN WITH HAH WITH YEH FINAL FORM", 0xfdaa },
  { "ARABIC LIGATURE DAD WITH HAH WITH YEH FINAL FORM", 0xfdab },
  { "ARABIC LIGATURE LAM WITH JEEM WITH YEH FINAL FORM", 0xfdac },
  { "ARABIC LIGATURE LAM WITH MEEM WITH YEH FINAL FORM", 0xfdad },
  { "ARABIC LIGATURE YEH WITH HAH WITH YEH FINAL FORM", 0xfdae },
  { "ARABIC LIGATURE YEH WITH JEEM WITH YEH FINAL FORM", 0xfdaf },
  { "ARABIC LIGATURE YEH WITH MEEM WITH YEH FINAL FORM", 0xfdb0 },
  { "ARABIC LIGATURE MEEM WITH MEEM WITH YEH FINAL FORM", 0xfdb1 },
  { "ARABIC LIGATURE QAF WITH MEEM WITH YEH FINAL FORM", 0xfdb2 },
  { "ARABIC LIGATURE NOON WITH HAH WITH YEH FINAL FORM", 0xfdb3 },
  { "ARABIC LIGATURE QAF WITH MEEM WITH HAH INITIAL FORM", 0xfdb4 },
  { "ARABIC LIGATURE LAM WITH HAH WITH MEEM INITIAL FORM", 0xfdb5 },
  { "ARABIC LIGATURE AIN WITH MEEM WITH YEH FINAL FORM", 0xfdb6 },
  { "ARABIC LIGATURE KAF WITH MEEM WITH YEH FINAL FORM", 0xfdb7 },
  { "ARABIC LIGATURE NOON WITH JEEM WITH HAH INITIAL FORM", 0xfdb8 },
  { "ARABIC LIGATURE MEEM WITH KHAH WITH YEH FINAL FORM", 0xfdb9 },
  { "ARABIC LIGATURE LAM WITH JEEM WITH MEEM INITIAL FORM", 0xfdba },
  { "ARABIC LIGATURE KAF WITH MEEM WITH MEEM FINAL FORM", 0xfdbb },
  { "ARABIC LIGATURE LAM WITH JEEM WITH MEEM FINAL FORM", 0xfdbc },
  { "ARABIC LIGATURE NOON WITH JEEM WITH HAH FINAL FORM", 0xfdbd },
  { "ARABIC LIGATURE JEEM WITH HAH WITH YEH FINAL FORM", 0xfdbe },
  { "ARABIC LIGATURE HAH WITH JEEM WITH YEH FINAL FORM", 0xfdbf },
  { "ARABIC LIGATURE MEEM WITH JEEM WITH YEH FINAL FORM", 0xfdc0 },
  { "ARABIC LIGATURE FEH WITH MEEM WITH YEH FINAL FORM", 0xfdc1 },
  { "ARABIC LIGATURE BEH WITH HAH WITH YEH FINAL FORM", 0xfdc2 },
  { "ARABIC LIGATURE KAF WITH MEEM WITH MEEM INITIAL FORM", 0xfdc3 },
  { "ARABIC LIGATURE AIN WITH JEEM WITH MEEM INITIAL FORM", 0xfdc4 },
  { "ARABIC LIGATURE SAD WITH MEEM WITH MEEM INITIAL FORM", 0xfdc5 },
  { "ARABIC LIGATURE SEEN WITH KHAH WITH YEH FINAL FORM", 0xfdc6 },
  { "ARABIC LIGATURE NOON WITH JEEM WITH YEH FINAL FORM", 0xfdc7 },
  { "ARABIC LIGATURE SALLA USED AS KORANIC STOP SIGN ISOLATED FORM", 0xfdf0 },
  { "ARABIC LIGATURE QALA USED AS KORANIC STOP SIGN ISOLATED FORM", 0xfdf1 },
  { "ARABIC LIGATURE ALLAH ISOLATED FORM", 0xfdf2 },
  { "ARABIC LIGATURE AKBAR ISOLATED FORM", 0xfdf3 },
  { "ARABIC LIGATURE MOHAMMAD ISOLATED FORM", 0xfdf4 },
  { "ARABIC LIGATURE SALAM ISOLATED FORM", 0xfdf5 },
  { "ARABIC LIGATURE RASOUL ISOLATED FORM", 0xfdf6 },
  { "ARABIC LIGATURE ALAYHE ISOLATED FORM", 0xfdf7 },
  { "ARABIC LIGATURE WASALLAM ISOLATED FORM", 0xfdf8 },
  { "ARABIC LIGATURE SALLA ISOLATED FORM", 0xfdf9 },
  { "ARABIC LIGATURE SALLALLAHOU ALAYHE WASALLAM", 0xfdfa },
  { "ARABIC LIGATURE JALLAJALALOUHOU", 0xfdfb },
  { "COMBINING LIGATURE LEFT HALF", 0xfe20 },
  { "COMBINING LIGATURE RIGHT HALF", 0xfe21 },
  { "COMBINING DOUBLE TILDE LEFT HALF", 0xfe22 },
  { "COMBINING DOUBLE TILDE RIGHT HALF", 0xfe23 },
  { "PRESENTATION FORM FOR VERTICAL TWO DOT LEADER", 0xfe30 },
  { "PRESENTATION FORM FOR VERTICAL EM DASH", 0xfe31 },
  { "PRESENTATION FORM FOR VERTICAL EN DASH", 0xfe32 },
  { "PRESENTATION FORM FOR VERTICAL LOW LINE", 0xfe33 },
  { "PRESENTATION FORM FOR VERTICAL WAVY LOW LINE", 0xfe34 },
  { "PRESENTATION FORM FOR VERTICAL LEFT PARENTHESIS", 0xfe35 },
  { "PRESENTATION FORM FOR VERTICAL RIGHT PARENTHESIS", 0xfe36 },
  { "PRESENTATION FORM FOR VERTICAL LEFT CURLY BRACKET", 0xfe37 },
  { "PRESENTATION FORM FOR VERTICAL RIGHT CURLY BRACKET", 0xfe38 },
  { "PRESENTATION FORM FOR VERTICAL LEFT TORTOISE SHELL BRACKET", 0xfe39 },
  { "PRESENTATION FORM FOR VERTICAL RIGHT TORTOISE SHELL BRACKET", 0xfe3a },
  { "PRESENTATION FORM FOR VERTICAL LEFT BLACK LENTICULAR BRACKET", 0xfe3b },
  { "PRESENTATION FORM FOR VERTICAL RIGHT BLACK LENTICULAR BRACKET", 0xfe3c },
  { "PRESENTATION FORM FOR VERTICAL LEFT DOUBLE ANGLE BRACKET", 0xfe3d },
  { "PRESENTATION FORM FOR VERTICAL RIGHT DOUBLE ANGLE BRACKET", 0xfe3e },
  { "PRESENTATION FORM FOR VERTICAL LEFT ANGLE BRACKET", 0xfe3f },
  { "PRESENTATION FORM FOR VERTICAL RIGHT ANGLE BRACKET", 0xfe40 },
  { "PRESENTATION FORM FOR VERTICAL LEFT CORNER BRACKET", 0xfe41 },
  { "PRESENTATION FORM FOR VERTICAL RIGHT CORNER BRACKET", 0xfe42 },
  { "PRESENTATION FORM FOR VERTICAL LEFT WHITE CORNER BRACKET", 0xfe43 },
  { "PRESENTATION FORM FOR VERTICAL RIGHT WHITE CORNER BRACKET", 0xfe44 },
  { "DASHED OVERLINE", 0xfe49 },
  { "CENTRELINE OVERLINE", 0xfe4a },
  { "WAVY OVERLINE", 0xfe4b },
  { "DOUBLE WAVY OVERLINE", 0xfe4c },
  { "DASHED LOW LINE", 0xfe4d },
  { "CENTRELINE LOW LINE", 0xfe4e },
  { "WAVY LOW LINE", 0xfe4f },
  { "SMALL COMMA", 0xfe50 },
  { "SMALL IDEOGRAPHIC COMMA", 0xfe51 },
  { "SMALL FULL STOP", 0xfe52 },
  { "SMALL SEMICOLON", 0xfe54 },
  { "SMALL COLON", 0xfe55 },
  { "SMALL QUESTION MARK", 0xfe56 },
  { "SMALL EXCLAMATION MARK", 0xfe57 },
  { "SMALL EM DASH", 0xfe58 },
  { "SMALL LEFT PARENTHESIS", 0xfe59 },
  { "SMALL RIGHT PARENTHESIS", 0xfe5a },
  { "SMALL LEFT CURLY BRACKET", 0xfe5b },
  { "SMALL RIGHT CURLY BRACKET", 0xfe5c },
  { "SMALL LEFT TORTOISE SHELL BRACKET", 0xfe5d },
  { "SMALL RIGHT TORTOISE SHELL BRACKET", 0xfe5e },
  { "SMALL NUMBER SIGN", 0xfe5f },
  { "SMALL AMPERSAND", 0xfe60 },
  { "SMALL ASTERISK", 0xfe61 },
  { "SMALL PLUS SIGN", 0xfe62 },
  { "SMALL HYPHEN-MINUS", 0xfe63 },
  { "SMALL LESS-THAN SIGN", 0xfe64 },
  { "SMALL GREATER-THAN SIGN", 0xfe65 },
  { "SMALL EQUALS SIGN", 0xfe66 },
  { "SMALL REVERSE SOLIDUS", 0xfe68 },
  { "SMALL DOLLAR SIGN", 0xfe69 },
  { "SMALL PERCENT SIGN", 0xfe6a },
  { "SMALL COMMERCIAL AT", 0xfe6b },
  { "ARABIC FATHATAN ISOLATED FORM", 0xfe70 },
  { "ARABIC TATWEEL WITH FATHATAN ABOVE", 0xfe71 },
  { "ARABIC DAMMATAN ISOLATED FORM", 0xfe72 },
  { "ARABIC KASRATAN ISOLATED FORM", 0xfe74 },
  { "ARABIC FATHA ISOLATED FORM", 0xfe76 },
  { "ARABIC FATHA MEDIAL FORM", 0xfe77 },
  { "ARABIC DAMMA ISOLATED FORM", 0xfe78 },
  { "ARABIC DAMMA MEDIAL FORM", 0xfe79 },
  { "ARABIC KASRA ISOLATED FORM", 0xfe7a },
  { "ARABIC KASRA MEDIAL FORM", 0xfe7b },
  { "ARABIC SHADDA ISOLATED FORM", 0xfe7c },
  { "ARABIC SHADDA MEDIAL FORM", 0xfe7d },
  { "ARABIC SUKUN ISOLATED FORM", 0xfe7e },
  { "ARABIC SUKUN MEDIAL FORM", 0xfe7f },
  { "ARABIC LETTER HAMZA ISOLATED FORM", 0xfe80 },
  { "ARABIC LETTER ALEF WITH MADDA ABOVE ISOLATED FORM", 0xfe81 },
  { "ARABIC LETTER ALEF WITH MADDA ABOVE FINAL FORM", 0xfe82 },
  { "ARABIC LETTER ALEF WITH HAMZA ABOVE ISOLATED FORM", 0xfe83 },
  { "ARABIC LETTER ALEF WITH HAMZA ABOVE FINAL FORM", 0xfe84 },
  { "ARABIC LETTER WAW WITH HAMZA ABOVE ISOLATED FORM", 0xfe85 },
  { "ARABIC LETTER WAW WITH HAMZA ABOVE FINAL FORM", 0xfe86 },
  { "ARABIC LETTER ALEF WITH HAMZA BELOW ISOLATED FORM", 0xfe87 },
  { "ARABIC LETTER ALEF WITH HAMZA BELOW FINAL FORM", 0xfe88 },
  { "ARABIC LETTER YEH WITH HAMZA ABOVE ISOLATED FORM", 0xfe89 },
  { "ARABIC LETTER YEH WITH HAMZA ABOVE FINAL FORM", 0xfe8a },
  { "ARABIC LETTER YEH WITH HAMZA ABOVE INITIAL FORM", 0xfe8b },
  { "ARABIC LETTER YEH WITH HAMZA ABOVE MEDIAL FORM", 0xfe8c },
  { "ARABIC LETTER ALEF ISOLATED FORM", 0xfe8d },
  { "ARABIC LETTER ALEF FINAL FORM", 0xfe8e },
  { "ARABIC LETTER BEH ISOLATED FORM", 0xfe8f },
  { "ARABIC LETTER BEH FINAL FORM", 0xfe90 },
  { "ARABIC LETTER BEH INITIAL FORM", 0xfe91 },
  { "ARABIC LETTER BEH MEDIAL FORM", 0xfe92 },
  { "ARABIC LETTER TEH MARBUTA ISOLATED FORM", 0xfe93 },
  { "ARABIC LETTER TEH MARBUTA FINAL FORM", 0xfe94 },
  { "ARABIC LETTER TEH ISOLATED FORM", 0xfe95 },
  { "ARABIC LETTER TEH FINAL FORM", 0xfe96 },
  { "ARABIC LETTER TEH INITIAL FORM", 0xfe97 },
  { "ARABIC LETTER TEH MEDIAL FORM", 0xfe98 },
  { "ARABIC LETTER THEH ISOLATED FORM", 0xfe99 },
  { "ARABIC LETTER THEH FINAL FORM", 0xfe9a },
  { "ARABIC LETTER THEH INITIAL FORM", 0xfe9b },
  { "ARABIC LETTER THEH MEDIAL FORM", 0xfe9c },
  { "ARABIC LETTER JEEM ISOLATED FORM", 0xfe9d },
  { "ARABIC LETTER JEEM FINAL FORM", 0xfe9e },
  { "ARABIC LETTER JEEM INITIAL FORM", 0xfe9f },
  { "ARABIC LETTER JEEM MEDIAL FORM", 0xfea0 },
  { "ARABIC LETTER HAH ISOLATED FORM", 0xfea1 },
  { "ARABIC LETTER HAH FINAL FORM", 0xfea2 },
  { "ARABIC LETTER HAH INITIAL FORM", 0xfea3 },
  { "ARABIC LETTER HAH MEDIAL FORM", 0xfea4 },
  { "ARABIC LETTER KHAH ISOLATED FORM", 0xfea5 },
  { "ARABIC LETTER KHAH FINAL FORM", 0xfea6 },
  { "ARABIC LETTER KHAH INITIAL FORM", 0xfea7 },
  { "ARABIC LETTER KHAH MEDIAL FORM", 0xfea8 },
  { "ARABIC LETTER DAL ISOLATED FORM", 0xfea9 },
  { "ARABIC LETTER DAL FINAL FORM", 0xfeaa },
  { "ARABIC LETTER THAL ISOLATED FORM", 0xfeab },
  { "ARABIC LETTER THAL FINAL FORM", 0xfeac },
  { "ARABIC LETTER REH ISOLATED FORM", 0xfead },
  { "ARABIC LETTER REH FINAL FORM", 0xfeae },
  { "ARABIC LETTER ZAIN ISOLATED FORM", 0xfeaf },
  { "ARABIC LETTER ZAIN FINAL FORM", 0xfeb0 },
  { "ARABIC LETTER SEEN ISOLATED FORM", 0xfeb1 },
  { "ARABIC LETTER SEEN FINAL FORM", 0xfeb2 },
  { "ARABIC LETTER SEEN INITIAL FORM", 0xfeb3 },
  { "ARABIC LETTER SEEN MEDIAL FORM", 0xfeb4 },
  { "ARABIC LETTER SHEEN ISOLATED FORM", 0xfeb5 },
  { "ARABIC LETTER SHEEN FINAL FORM", 0xfeb6 },
  { "ARABIC LETTER SHEEN INITIAL FORM", 0xfeb7 },
  { "ARABIC LETTER SHEEN MEDIAL FORM", 0xfeb8 },
  { "ARABIC LETTER SAD ISOLATED FORM", 0xfeb9 },
  { "ARABIC LETTER SAD FINAL FORM", 0xfeba },
  { "ARABIC LETTER SAD INITIAL FORM", 0xfebb },
  { "ARABIC LETTER SAD MEDIAL FORM", 0xfebc },
  { "ARABIC LETTER DAD ISOLATED FORM", 0xfebd },
  { "ARABIC LETTER DAD FINAL FORM", 0xfebe },
  { "ARABIC LETTER DAD INITIAL FORM", 0xfebf },
  { "ARABIC LETTER DAD MEDIAL FORM", 0xfec0 },
  { "ARABIC LETTER TAH ISOLATED FORM", 0xfec1 },
  { "ARABIC LETTER TAH FINAL FORM", 0xfec2 },
  { "ARABIC LETTER TAH INITIAL FORM", 0xfec3 },
  { "ARABIC LETTER TAH MEDIAL FORM", 0xfec4 },
  { "ARABIC LETTER ZAH ISOLATED FORM", 0xfec5 },
  { "ARABIC LETTER ZAH FINAL FORM", 0xfec6 },
  { "ARABIC LETTER ZAH INITIAL FORM", 0xfec7 },
  { "ARABIC LETTER ZAH MEDIAL FORM", 0xfec8 },
  { "ARABIC LETTER AIN ISOLATED FORM", 0xfec9 },
  { "ARABIC LETTER AIN FINAL FORM", 0xfeca },
  { "ARABIC LETTER AIN INITIAL FORM", 0xfecb },
  { "ARABIC LETTER AIN MEDIAL FORM", 0xfecc },
  { "ARABIC LETTER GHAIN ISOLATED FORM", 0xfecd },
  { "ARABIC LETTER GHAIN FINAL FORM", 0xfece },
  { "ARABIC LETTER GHAIN INITIAL FORM", 0xfecf },
  { "ARABIC LETTER GHAIN MEDIAL FORM", 0xfed0 },
  { "ARABIC LETTER FEH ISOLATED FORM", 0xfed1 },
  { "ARABIC LETTER FEH FINAL FORM", 0xfed2 },
  { "ARABIC LETTER FEH INITIAL FORM", 0xfed3 },
  { "ARABIC LETTER FEH MEDIAL FORM", 0xfed4 },
  { "ARABIC LETTER QAF ISOLATED FORM", 0xfed5 },
  { "ARABIC LETTER QAF FINAL FORM", 0xfed6 },
  { "ARABIC LETTER QAF INITIAL FORM", 0xfed7 },
  { "ARABIC LETTER QAF MEDIAL FORM", 0xfed8 },
  { "ARABIC LETTER KAF ISOLATED FORM", 0xfed9 },
  { "ARABIC LETTER KAF FINAL FORM", 0xfeda },
  { "ARABIC LETTER KAF INITIAL FORM", 0xfedb },
  { "ARABIC LETTER KAF MEDIAL FORM", 0xfedc },
  { "ARABIC LETTER LAM ISOLATED FORM", 0xfedd },
  { "ARABIC LETTER LAM FINAL FORM", 0xfede },
  { "ARABIC LETTER LAM INITIAL FORM", 0xfedf },
  { "ARABIC LETTER LAM MEDIAL FORM", 0xfee0 },
  { "ARABIC LETTER MEEM ISOLATED FORM", 0xfee1 },
  { "ARABIC LETTER MEEM FINAL FORM", 0xfee2 },
  { "ARABIC LETTER MEEM INITIAL FORM", 0xfee3 },
  { "ARABIC LETTER MEEM MEDIAL FORM", 0xfee4 },
  { "ARABIC LETTER NOON ISOLATED FORM", 0xfee5 },
  { "ARABIC LETTER NOON FINAL FORM", 0xfee6 },
  { "ARABIC LETTER NOON INITIAL FORM", 0xfee7 },
  { "ARABIC LETTER NOON MEDIAL FORM", 0xfee8 },
  { "ARABIC LETTER HEH ISOLATED FORM", 0xfee9 },
  { "ARABIC LETTER HEH FINAL FORM", 0xfeea },
  { "ARABIC LETTER HEH INITIAL FORM", 0xfeeb },
  { "ARABIC LETTER HEH MEDIAL FORM", 0xfeec },
  { "ARABIC LETTER WAW ISOLATED FORM", 0xfeed },
  { "ARABIC LETTER WAW FINAL FORM", 0xfeee },
  { "ARABIC LETTER ALEF MAKSURA ISOLATED FORM", 0xfeef },
  { "ARABIC LETTER ALEF MAKSURA FINAL FORM", 0xfef0 },
  { "ARABIC LETTER YEH ISOLATED FORM", 0xfef1 },
  { "ARABIC LETTER YEH FINAL FORM", 0xfef2 },
  { "ARABIC LETTER YEH INITIAL FORM", 0xfef3 },
  { "ARABIC LETTER YEH MEDIAL FORM", 0xfef4 },
  { "ARABIC LIGATURE LAM WITH ALEF WITH MADDA ABOVE ISOLATED FORM", 0xfef5 },
  { "ARABIC LIGATURE LAM WITH ALEF WITH MADDA ABOVE FINAL FORM", 0xfef6 },
  { "ARABIC LIGATURE LAM WITH ALEF WITH HAMZA ABOVE ISOLATED FORM", 0xfef7 },
  { "ARABIC LIGATURE LAM WITH ALEF WITH HAMZA ABOVE FINAL FORM", 0xfef8 },
  { "ARABIC LIGATURE LAM WITH ALEF WITH HAMZA BELOW ISOLATED FORM", 0xfef9 },
  { "ARABIC LIGATURE LAM WITH ALEF WITH HAMZA BELOW FINAL FORM", 0xfefa },
  { "ARABIC LIGATURE LAM WITH ALEF ISOLATED FORM", 0xfefb },
  { "ARABIC LIGATURE LAM WITH ALEF FINAL FORM", 0xfefc },
  { "ZERO WIDTH NO-BREAK SPACE", 0xfeff },
  { "FULLWIDTH EXCLAMATION MARK", 0xff01 },
  { "FULLWIDTH QUOTATION MARK", 0xff02 },
  { "FULLWIDTH NUMBER SIGN", 0xff03 },
  { "FULLWIDTH DOLLAR SIGN", 0xff04 },
  { "FULLWIDTH PERCENT SIGN", 0xff05 },
  { "FULLWIDTH AMPERSAND", 0xff06 },
  { "FULLWIDTH APOSTROPHE", 0xff07 },
  { "FULLWIDTH LEFT PARENTHESIS", 0xff08 },
  { "FULLWIDTH RIGHT PARENTHESIS", 0xff09 },
  { "FULLWIDTH ASTERISK", 0xff0a },
  { "FULLWIDTH PLUS SIGN", 0xff0b },
  { "FULLWIDTH COMMA", 0xff0c },
  { "FULLWIDTH HYPHEN-MINUS", 0xff0d },
  { "FULLWIDTH FULL STOP", 0xff0e },
  { "FULLWIDTH SOLIDUS", 0xff0f },
  { "FULLWIDTH DIGIT ZERO", 0xff10 },
  { "FULLWIDTH DIGIT ONE", 0xff11 },
  { "FULLWIDTH DIGIT TWO", 0xff12 },
  { "FULLWIDTH DIGIT THREE", 0xff13 },
  { "FULLWIDTH DIGIT FOUR", 0xff14 },
  { "FULLWIDTH DIGIT FIVE", 0xff15 },
  { "FULLWIDTH DIGIT SIX", 0xff16 },
  { "FULLWIDTH DIGIT SEVEN", 0xff17 },
  { "FULLWIDTH DIGIT EIGHT", 0xff18 },
  { "FULLWIDTH DIGIT NINE", 0xff19 },
  { "FULLWIDTH COLON", 0xff1a },
  { "FULLWIDTH SEMICOLON", 0xff1b },
  { "FULLWIDTH LESS-THAN SIGN", 0xff1c },
  { "FULLWIDTH EQUALS SIGN", 0xff1d },
  { "FULLWIDTH GREATER-THAN SIGN", 0xff1e },
  { "FULLWIDTH QUESTION MARK", 0xff1f },
  { "FULLWIDTH COMMERCIAL AT", 0xff20 },
  { "FULLWIDTH LATIN CAPITAL LETTER A", 0xff21 },
  { "FULLWIDTH LATIN CAPITAL LETTER B", 0xff22 },
  { "FULLWIDTH LATIN CAPITAL LETTER C", 0xff23 },
  { "FULLWIDTH LATIN CAPITAL LETTER D", 0xff24 },
  { "FULLWIDTH LATIN CAPITAL LETTER E", 0xff25 },
  { "FULLWIDTH LATIN CAPITAL LETTER F", 0xff26 },
  { "FULLWIDTH LATIN CAPITAL LETTER G", 0xff27 },
  { "FULLWIDTH LATIN CAPITAL LETTER H", 0xff28 },
  { "FULLWIDTH LATIN CAPITAL LETTER I", 0xff29 },
  { "FULLWIDTH LATIN CAPITAL LETTER J", 0xff2a },
  { "FULLWIDTH LATIN CAPITAL LETTER K", 0xff2b },
  { "FULLWIDTH LATIN CAPITAL LETTER L", 0xff2c },
  { "FULLWIDTH LATIN CAPITAL LETTER M", 0xff2d },
  { "FULLWIDTH LATIN CAPITAL LETTER N", 0xff2e },
  { "FULLWIDTH LATIN CAPITAL LETTER O", 0xff2f },
  { "FULLWIDTH LATIN CAPITAL LETTER P", 0xff30 },
  { "FULLWIDTH LATIN CAPITAL LETTER Q", 0xff31 },
  { "FULLWIDTH LATIN CAPITAL LETTER R", 0xff32 },
  { "FULLWIDTH LATIN CAPITAL LETTER S", 0xff33 },
  { "FULLWIDTH LATIN CAPITAL LETTER T", 0xff34 },
  { "FULLWIDTH LATIN CAPITAL LETTER U", 0xff35 },
  { "FULLWIDTH LATIN CAPITAL LETTER V", 0xff36 },
  { "FULLWIDTH LATIN CAPITAL LETTER W", 0xff37 },
  { "FULLWIDTH LATIN CAPITAL LETTER X", 0xff38 },
  { "FULLWIDTH LATIN CAPITAL LETTER Y", 0xff39 },
  { "FULLWIDTH LATIN CAPITAL LETTER Z", 0xff3a },
  { "FULLWIDTH LEFT SQUARE BRACKET", 0xff3b },
  { "FULLWIDTH REVERSE SOLIDUS", 0xff3c },
  { "FULLWIDTH RIGHT SQUARE BRACKET", 0xff3d },
  { "FULLWIDTH CIRCUMFLEX ACCENT", 0xff3e },
  { "FULLWIDTH LOW LINE", 0xff3f },
  { "FULLWIDTH GRAVE ACCENT", 0xff40 },
  { "FULLWIDTH LATIN SMALL LETTER A", 0xff41 },
  { "FULLWIDTH LATIN SMALL LETTER B", 0xff42 },
  { "FULLWIDTH LATIN SMALL LETTER C", 0xff43 },
  { "FULLWIDTH LATIN SMALL LETTER D", 0xff44 },
  { "FULLWIDTH LATIN SMALL LETTER E", 0xff45 },
  { "FULLWIDTH LATIN SMALL LETTER F", 0xff46 },
  { "FULLWIDTH LATIN SMALL LETTER G", 0xff47 },
  { "FULLWIDTH LATIN SMALL LETTER H", 0xff48 },
  { "FULLWIDTH LATIN SMALL LETTER I", 0xff49 },
  { "FULLWIDTH LATIN SMALL LETTER J", 0xff4a },
  { "FULLWIDTH LATIN SMALL LETTER K", 0xff4b },
  { "FULLWIDTH LATIN SMALL LETTER L", 0xff4c },
  { "FULLWIDTH LATIN SMALL LETTER M", 0xff4d },
  { "FULLWIDTH LATIN SMALL LETTER N", 0xff4e },
  { "FULLWIDTH LATIN SMALL LETTER O", 0xff4f },
  { "FULLWIDTH LATIN SMALL LETTER P", 0xff50 },
  { "FULLWIDTH LATIN SMALL LETTER Q", 0xff51 },
  { "FULLWIDTH LATIN SMALL LETTER R", 0xff52 },
  { "FULLWIDTH LATIN SMALL LETTER S", 0xff53 },
  { "FULLWIDTH LATIN SMALL LETTER T", 0xff54 },
  { "FULLWIDTH LATIN SMALL LETTER U", 0xff55 },
  { "FULLWIDTH LATIN SMALL LETTER V", 0xff56 },
  { "FULLWIDTH LATIN SMALL LETTER W", 0xff57 },
  { "FULLWIDTH LATIN SMALL LETTER X", 0xff58 },
  { "FULLWIDTH LATIN SMALL LETTER Y", 0xff59 },
  { "FULLWIDTH LATIN SMALL LETTER Z", 0xff5a },
  { "FULLWIDTH LEFT CURLY BRACKET", 0xff5b },
  { "FULLWIDTH VERTICAL LINE", 0xff5c },
  { "FULLWIDTH RIGHT CURLY BRACKET", 0xff5d },
  { "FULLWIDTH TILDE", 0xff5e },
  { "HALFWIDTH IDEOGRAPHIC FULL STOP", 0xff61 },
  { "HALFWIDTH LEFT CORNER BRACKET", 0xff62 },
  { "HALFWIDTH RIGHT CORNER BRACKET", 0xff63 },
  { "HALFWIDTH IDEOGRAPHIC COMMA", 0xff64 },
  { "HALFWIDTH KATAKANA MIDDLE DOT", 0xff65 },
  { "HALFWIDTH KATAKANA LETTER WO", 0xff66 },
  { "HALFWIDTH KATAKANA LETTER SMALL A", 0xff67 },
  { "HALFWIDTH KATAKANA LETTER SMALL I", 0xff68 },
  { "HALFWIDTH KATAKANA LETTER SMALL U", 0xff69 },
  { "HALFWIDTH KATAKANA LETTER SMALL E", 0xff6a },
  { "HALFWIDTH KATAKANA LETTER SMALL O", 0xff6b },
  { "HALFWIDTH KATAKANA LETTER SMALL YA", 0xff6c },
  { "HALFWIDTH KATAKANA LETTER SMALL YU", 0xff6d },
  { "HALFWIDTH KATAKANA LETTER SMALL YO", 0xff6e },
  { "HALFWIDTH KATAKANA LETTER SMALL TU", 0xff6f },
  { "HALFWIDTH KATAKANA-HIRAGANA PROLONGED SOUND MARK", 0xff70 },
  { "HALFWIDTH KATAKANA LETTER A", 0xff71 },
  { "HALFWIDTH KATAKANA LETTER I", 0xff72 },
  { "HALFWIDTH KATAKANA LETTER U", 0xff73 },
  { "HALFWIDTH KATAKANA LETTER E", 0xff74 },
  { "HALFWIDTH KATAKANA LETTER O", 0xff75 },
  { "HALFWIDTH KATAKANA LETTER KA", 0xff76 },
  { "HALFWIDTH KATAKANA LETTER KI", 0xff77 },
  { "HALFWIDTH KATAKANA LETTER KU", 0xff78 },
  { "HALFWIDTH KATAKANA LETTER KE", 0xff79 },
  { "HALFWIDTH KATAKANA LETTER KO", 0xff7a },
  { "HALFWIDTH KATAKANA LETTER SA", 0xff7b },
  { "HALFWIDTH KATAKANA LETTER SI", 0xff7c },
  { "HALFWIDTH KATAKANA LETTER SU", 0xff7d },
  { "HALFWIDTH KATAKANA LETTER SE", 0xff7e },
  { "HALFWIDTH KATAKANA LETTER SO", 0xff7f },
  { "HALFWIDTH KATAKANA LETTER TA", 0xff80 },
  { "HALFWIDTH KATAKANA LETTER TI", 0xff81 },
  { "HALFWIDTH KATAKANA LETTER TU", 0xff82 },
  { "HALFWIDTH KATAKANA LETTER TE", 0xff83 },
  { "HALFWIDTH KATAKANA LETTER TO", 0xff84 },
  { "HALFWIDTH KATAKANA LETTER NA", 0xff85 },
  { "HALFWIDTH KATAKANA LETTER NI", 0xff86 },
  { "HALFWIDTH KATAKANA LETTER NU", 0xff87 },
  { "HALFWIDTH KATAKANA LETTER NE", 0xff88 },
  { "HALFWIDTH KATAKANA LETTER NO", 0xff89 },
  { "HALFWIDTH KATAKANA LETTER HA", 0xff8a },
  { "HALFWIDTH KATAKANA LETTER HI", 0xff8b },
  { "HALFWIDTH KATAKANA LETTER HU", 0xff8c },
  { "HALFWIDTH KATAKANA LETTER HE", 0xff8d },
  { "HALFWIDTH KATAKANA LETTER HO", 0xff8e },
  { "HALFWIDTH KATAKANA LETTER MA", 0xff8f },
  { "HALFWIDTH KATAKANA LETTER MI", 0xff90 },
  { "HALFWIDTH KATAKANA LETTER MU", 0xff91 },
  { "HALFWIDTH KATAKANA LETTER ME", 0xff92 },
  { "HALFWIDTH KATAKANA LETTER MO", 0xff93 },
  { "HALFWIDTH KATAKANA LETTER YA", 0xff94 },
  { "HALFWIDTH KATAKANA LETTER YU", 0xff95 },
  { "HALFWIDTH KATAKANA LETTER YO", 0xff96 },
  { "HALFWIDTH KATAKANA LETTER RA", 0xff97 },
  { "HALFWIDTH KATAKANA LETTER RI", 0xff98 },
  { "HALFWIDTH KATAKANA LETTER RU", 0xff99 },
  { "HALFWIDTH KATAKANA LETTER RE", 0xff9a },
  { "HALFWIDTH KATAKANA LETTER RO", 0xff9b },
  { "HALFWIDTH KATAKANA LETTER WA", 0xff9c },
  { "HALFWIDTH KATAKANA LETTER N", 0xff9d },
  { "HALFWIDTH KATAKANA VOICED SOUND MARK", 0xff9e },
  { "HALFWIDTH KATAKANA SEMI-VOICED SOUND MARK", 0xff9f },
  { "HALFWIDTH HANGUL FILLER", 0xffa0 },
  { "HALFWIDTH HANGUL LETTER KIYEOK", 0xffa1 },
  { "HALFWIDTH HANGUL LETTER SSANGKIYEOK", 0xffa2 },
  { "HALFWIDTH HANGUL LETTER KIYEOK-SIOS", 0xffa3 },
  { "HALFWIDTH HANGUL LETTER NIEUN", 0xffa4 },
  { "HALFWIDTH HANGUL LETTER NIEUN-CIEUC", 0xffa5 },
  { "HALFWIDTH HANGUL LETTER NIEUN-HIEUH", 0xffa6 },
  { "HALFWIDTH HANGUL LETTER TIKEUT", 0xffa7 },
  { "HALFWIDTH HANGUL LETTER SSANGTIKEUT", 0xffa8 },
  { "HALFWIDTH HANGUL LETTER RIEUL", 0xffa9 },
  { "HALFWIDTH HANGUL LETTER RIEUL-KIYEOK", 0xffaa },
  { "HALFWIDTH HANGUL LETTER RIEUL-MIEUM", 0xffab },
  { "HALFWIDTH HANGUL LETTER RIEUL-PIEUP", 0xffac },
  { "HALFWIDTH HANGUL LETTER RIEUL-SIOS", 0xffad },
  { "HALFWIDTH HANGUL LETTER RIEUL-THIEUTH", 0xffae },
  { "HALFWIDTH HANGUL LETTER RIEUL-PHIEUPH", 0xffaf },
  { "HALFWIDTH HANGUL LETTER RIEUL-HIEUH", 0xffb0 },
  { "HALFWIDTH HANGUL LETTER MIEUM", 0xffb1 },
  { "HALFWIDTH HANGUL LETTER PIEUP", 0xffb2 },
  { "HALFWIDTH HANGUL LETTER SSANGPIEUP", 0xffb3 },
  { "HALFWIDTH HANGUL LETTER PIEUP-SIOS", 0xffb4 },
  { "HALFWIDTH HANGUL LETTER SIOS", 0xffb5 },
  { "HALFWIDTH HANGUL LETTER SSANGSIOS", 0xffb6 },
  { "HALFWIDTH HANGUL LETTER IEUNG", 0xffb7 },
  { "HALFWIDTH HANGUL LETTER CIEUC", 0xffb8 },
  { "HALFWIDTH HANGUL LETTER SSANGCIEUC", 0xffb9 },
  { "HALFWIDTH HANGUL LETTER CHIEUCH", 0xffba },
  { "HALFWIDTH HANGUL LETTER KHIEUKH", 0xffbb },
  { "HALFWIDTH HANGUL LETTER THIEUTH", 0xffbc },
  { "HALFWIDTH HANGUL LETTER PHIEUPH", 0xffbd },
  { "HALFWIDTH HANGUL LETTER HIEUH", 0xffbe },
  { "HALFWIDTH HANGUL LETTER A", 0xffc2 },
  { "HALFWIDTH HANGUL LETTER AE", 0xffc3 },
  { "HALFWIDTH HANGUL LETTER YA", 0xffc4 },
  { "HALFWIDTH HANGUL LETTER YAE", 0xffc5 },
  { "HALFWIDTH HANGUL LETTER EO", 0xffc6 },
  { "HALFWIDTH HANGUL LETTER E", 0xffc7 },
  { "HALFWIDTH HANGUL LETTER YEO", 0xffca },
  { "HALFWIDTH HANGUL LETTER YE", 0xffcb },
  { "HALFWIDTH HANGUL LETTER O", 0xffcc },
  { "HALFWIDTH HANGUL LETTER WA", 0xffcd },
  { "HALFWIDTH HANGUL LETTER WAE", 0xffce },
  { "HALFWIDTH HANGUL LETTER OE", 0xffcf },
  { "HALFWIDTH HANGUL LETTER YO", 0xffd2 },
  { "HALFWIDTH HANGUL LETTER U", 0xffd3 },
  { "HALFWIDTH HANGUL LETTER WEO", 0xffd4 },
  { "HALFWIDTH HANGUL LETTER WE", 0xffd5 },
  { "HALFWIDTH HANGUL LETTER WI", 0xffd6 },
  { "HALFWIDTH HANGUL LETTER YU", 0xffd7 },
  { "HALFWIDTH HANGUL LETTER EU", 0xffda },
  { "HALFWIDTH HANGUL LETTER YI", 0xffdb },
  { "HALFWIDTH HANGUL LETTER I", 0xffdc },
  { "FULLWIDTH CENT SIGN", 0xffe0 },
  { "FULLWIDTH POUND SIGN", 0xffe1 },
  { "FULLWIDTH NOT SIGN", 0xffe2 },
  { "FULLWIDTH MACRON", 0xffe3 },
  { "FULLWIDTH BROKEN BAR", 0xffe4 },
  { "FULLWIDTH YEN SIGN", 0xffe5 },
  { "FULLWIDTH WON SIGN", 0xffe6 },
  { "HALFWIDTH FORMS LIGHT VERTICAL", 0xffe8 },
  { "HALFWIDTH LEFTWARDS ARROW", 0xffe9 },
  { "HALFWIDTH UPWARDS ARROW", 0xffea },
  { "HALFWIDTH RIGHTWARDS ARROW", 0xffeb },
  { "HALFWIDTH DOWNWARDS ARROW", 0xffec },
  { "HALFWIDTH BLACK SQUARE", 0xffed },
  { "HALFWIDTH WHITE CIRCLE", 0xffee },
  { "INTERLINEAR ANNOTATION ANCHOR", 0xfff9 },
  { "INTERLINEAR ANNOTATION SEPARATOR", 0xfffa },
  { "INTERLINEAR ANNOTATION TERMINATOR", 0xfffb },
  { "OBJECT REPLACEMENT CHARACTER", 0xfffc },
  { "REPLACEMENT CHARACTER", 0xfffd },
};

