require "socket"
require "rlang"
require "timeout"
require "my/mystring"

class MockSerialDevice
	def initialize(comm_port)
		@comm_port = comm_port
	end

	def has_data?
		return false
	end

	def get_data
		""
	end

	def write_data(data)
	end

	def write_event(event)
	end
end

class MockRedirector
	COMMAND_PORT_BASE = 8000
	DATA_PORT_BASE = 8100
	
	DATA_CHUNK_SIZE = 1024		# maximum size of chunk of data to write

	IN_EVENT_HASH = {
		"C" => :cts_high,
		"c" => :cts_low,
		"S" => :dsr_high,
		"s" => :dsr_low,
		"D" => :dcd_high,
		"d" => :dcd_low,
		"R" => :ring_high,
		"r" => :ring_low,
		"B" => :break_detected
	}

	OUT_EVENT_HASH = {
		:dtr_high => "D",
		:dtr_low => "d",
		:rts_high => "R",
		:rts_low => "r",
		:set_uart => "U%d-%d-%s-%d-%s",
		:send_break => "B%d"
	}

	PARITY_HASH = {
		:none => "N",
		:even => "E",
		:odd => "O",
		:mark => "M",
		:space => "S"
	}

	FLOW_CONTROL_HASH = {
		:none => "N",
		:hardware => "H",
		:software => "S"
	}
	
	acts_as_rlangable

	interface :connect, [:pid, :host]
	message :connect, [[:pid, :_any, :host, :_any]]

	interface :disconnect, [:pid]
	message :disconnect, [[:pid, :_any]]

	interface :send_data, [:pid, :data]	
	message :send_data, [[:pid, :_any], [:data, :_any]]

	interface :change_signal, [:pid, :event]
	message :change_signal, [[:pid, :_any], [:event, :_any]]

	interface :set_uart, [:pid, :baud_rate, :data_bits, :parity, :stop_bits]
	message :set_uart, [[:pid, :_any], [:baud_rate, :_any], [:data_bits, :_any], [:parity, :_any], [:stop_bits, :_any], [:flow_control, :_any]]

	interface :send_break, [:pid, :break_length]
	message :send_break, [[:pid, :_any], [:break_length, :_any]]
	
	thread_func	:run

	attr_reader :comm_port
	
	def initialize(comm_port)
		@comm_port = comm_port
		@connected_lock = Mutex.new
		@command_data = ""
		@connected = false
		@out_command_queue = []
	end

	def connect(listener, host)
		@listener = listener
		
		$TRACE.debug 5, "attempting to connect to '#{host}'"
		mock_serial_device_class_str = host
		mock_serial_device_filename = "#{host}.rb"
		unless File.exist?(mock_serial_device_filename)
			@listener << {:connected => "mock device: '#{mock_serial_device_filename}' doesn't exist"}
		end
		$TRACE.debug 5, "connection suceeded!'"

		load (mock_serial_device_filename) unless Object.const_defined?(mock_serial_device_class_str)
		mock_serial_device_class = eval(mock_serial_device_class_str)

		@modem = mock_serial_device_class.new(@comm_port)
		
		@connected = true

		@listener << {:connected => true}
	end

	def disconnect(listener)
		@listener << {:connected => false}
	end

	def send_data(sender, data)
		index = 0
		$TRACE.debug 5, "send (#{data.size})='#{data[0..80].x}'"
		while (data.size - index) > DATA_CHUNK_SIZE 
			@out_command_queue << {:data => data[index..(index+DATA_CHUNK_SIZE-1)]}
			index += DATA_CHUNK_SIZE
		end
		@out_command_queue << {:data => data[index..-1]}
	end

	def send_break(sender, break_length)
		$TRACE.debug 0, "send_break: #{break_length} milliseconds"
		@out_command_queue << {:event => [:send_break, break_length]}
	end
	
	def change_signal(sender, event)
		$TRACE.debug 0, "change_signal: #{event.inspect}"
		@out_command_queue << {:event => event}
	end

	def set_uart(sender, baud_rate, data_bits, parity, stop_bits, flow_control)
		@out_command_queue << {:event => [:set_uart, baud_rate, data_bits, PARITY_HASH[parity], stop_bits, FLOW_CONTROL_HASH[flow_control]]}
	end
	
	def got_data(data)
		@listener << {:data => data}
	end
	
	def run		
		return unless @connected

		if @modem.has_data? then
			data = @modem.get_data
			got_data(data)
		end
		
		unless @out_command_queue.empty?
			$TRACE.debug 9, "out_command_queue = #{@out_command_queue.inspect}"
			next_out_command = @out_command_queue.first
			if next_out_command.has_key?(:data) then
				@modem.write_data(next_out_command[:data])
				$TRACE.debug 5, "+DATA: '#{next_out_command[:data].x}'"
				@out_command_queue.shift
			elsif next_out_command.has_key?(:event) then
				@modem.write_event(next_out_command[:event])
				@out_command_queue.shift
			end
		end
	end
end


