class PortWriter
	class Log
		def initialize(filename, output_block)
			@file = File.new(filename, "w")
			@output_block = output_block
		end

		def timestamp
			Time.now.strftime("%H:%M:%S: ")
		end

		def put_prefix(direction, args)
			if direction.is_a?(String) then
				args.insert(0,direction)
				direction = :status
			end
			low_level_write(timestamp)
			case direction
			when :in
				low_level_write("<-- ")
			when :out
				low_level_write("--> ")
			when :status
				low_level_write("--- ")
			when :error
				low_level_write("EEE ")
			else
				raise "invalid direction: '#{direction}'"
			end
		end
		
		def puts(direction, *args)
			put_prefix(direction, args)
			low_level_write(*args)
			low_level_write("\n")
		end

		def write(direction, *args)
			put_prefix(direction, args)
			low_level_write(*args)
		end
		
		def printf(direction, *args)
			put_prefix(direction, args)
			str = sprintf(*args)
			low_level_write str
		end

		def put_multiple_lines(direction, str)
			#$stdout.puts("put_multiple_lines str = #{str.x}")
			lines = str.split(/\n/)

			lines.each_with_index do |line, index|
				put_prefix(direction, line)

				# add back in the \n on all but the last line
				line2 = (index != lines.size - 1) ? line + "\n" : line

				low_level_write(line2.x)
				low_level_write("\n")
			end
		end

		def low_level_write(str)
			#$stdout.write str
			@output_block.call(str)
			@file.write str
		end

		def close
			@file.close
		end
	end

	class Error < StandardError
	end

	class TimeoutError < Error
		attr_reader :regex, :data
		def initialize(message, regex, data)
			@regex = regex
			@data = data
			super(message)
		end
	end

	class SyncTimeoutError < Error
		attr_reader :filename, :num_wanted
		def initialize(message, filename, num_wanted)
			@filename = filename
			@num_wanted = num_wanted
			super(message)
		end
	end

	class UserAbortError < Error
	end

	SYNC_FILE = "__SYNC__%d.txt"
	
	attr_reader :log, :last_match

	INITIALIZE_OPTIONS = {
		:log_filename => "pwlog.txt",
		:output_to => proc{|data| $stdout.write(data)},
		:status_proc => proc{|status_hash|},
		:options => {}
	}

	#
	# set up a new non-connected port writer through the passed in re-director
	#
	def initialize(redirector, options={})
		options = INITIALIZE_OPTIONS.merge(options)
		
		@redirector = redirector
		@options = options[:options]
		@log = Log.new(options[:log_filename], options[:output_to])
		@status_proc = options[:status_proc]
		@data_received = ""
		@sync_num = 0
		@user_aborted = false

		# default settings for serial port
		@baud_rate = 115200
		@data_bits = 8
		@parity = :none
		@stop_bits = 1
		@flow_control = :hardware

		Thread.current[:trace_thread_id] = redirector.comm_port

		begin
			File.delete(SYNC_FILE % @redirector.comm_port)
		rescue Errno::ENOENT
		end
	end

	#
	# connect to the tester redirector
	#
	def connect(host, &block)
		@redirector.connect(Thread.current, host)
		$TRACE.debug 5, "[#{@redirector.comm_port}]: after sending connect message, block = #{block.inspect}"
		if block then
			m = wait_until_connected
			raise "unable to connect to host '#{host}' because '#{m[:connected]}'" if m[:connected].kind_of?(String)
			begin
				block.call(self, @options)
			ensure
				@redirector.disconnect(Thread.current)
				wait_until_disconnected
			end
		end
	end

	#
	# abort a wait_for by raising an exception
	#
	def abort_wait_for
		@user_aborted = true
	end
	
	#
	# Pass status info to the UI
	#
	def status_info(status_hash)
		@status_proc.call(status_hash)
	end

	#
	# Write data to the port
	#
	def write(data, options={})
		@redirector.send_data(Thread.current, data)
		if options[:display_byte_count] then
			@log.printf(:out, "%d bytes\n", data.size)
		else
			@log.printf(:out, "#{data.x}\n")
		end
	end

	#
	# Write a break to the port
	#
	def write_break(break_length=100)
		@redirector.send_break(Thread.current, break_length)
		@log.printf(:out, "<BREAK:%d>\n", break_length)
	end

	WAIT_FOR_DEFAULT_OPTIONS = {
		:timeout => 5,							# timeout is for 5 seconds
		:display_byte_count => false,		# display byte count for large downloads
		:debug => false,						# display what we are waiting for
		:match_also => []						# this is a list of regex, proc pairs that allow extra
													# stuff to be done when a regex matches
	}

	#
	# drop a modem signal (either :dtr or :rts)
	#
	def drop_signal(signal)
		case signal
		when :dtr: @redirector.change_signal(Thread.current, :dtr_low)
		when :rts: @redirector.change_signal(Thread.current, :rts_low)
		end
	end

	#
	# raise a modem signal (either :dtr or :rts)
	#
	def raise_signal(signal)
		case signal
		when :dtr: @redirector.change_signal(Thread.current, :dtr_high)
		when :rts: @redirector.change_signal(Thread.current, :rts_high)
		end
	end

	def baud_rate=(new_baud_rate)
		@baud_rate = new_baud_rate
		set_uart
	end

	def parity=(new_parity)
		@parity = new_parity
		set_uart
	end
	
	def stop_bits=(new_stop_bits)
		@stop_bits = new_stop_bits
		set_uart
	end

	def data_bits=(new_data_bits)
		@data_bits = new_data_bits
		set_uart
	end

	def wait_for_signal(signal, state)
	end
	
	def clear_received_data
		log.puts "clearing data received: #{@data_received.x}"
		@data_received = ""
	end

	def clear_transmit_data
		@redirector.clear_transmit_data(Thread.current)	
	end

	#
	# wait for a regular expression to be satisfied or until a timeout happens. When
	# waiting on binary data, set options[:display_byte_count] to true so that only
	# byte counts and not the binary data is displayed.
	#
	def wait_for(regex,  options={})
		options = WAIT_FOR_DEFAULT_OPTIONS.merge(options)
		
		$TRACE.debug 5, "[#{@redirector.comm_port}]: wait_for(begin): #{regex.source}, #{options[:timeout]}"
		$TRACE.debug 5, "[#{@redirector.comm_port}]: in wait_for with data_received = #{@data_received.inspect}"

		@match_also_indexes = {}

		# if last call to this was to display byte count and this one isn't
		# and there is data in the data buffer
		if @was_display_byte_count && !options[:display_byte_count] && 
		   @data_received then
			@log.printf(:in, "%s\n", @data_received.x)
		end
		m = check_match(regex, options)
		return m if m

		@was_display_byte_count = options[:display_byte_count]
		
		$TRACE.debug 5, "[#{@redirector.comm_port}]: no initial match"
		
		done = false
		last_size = 0
		size = 0
		begin
			timeout(options[:timeout]) do
				while !done
					sleep 0.05

					begin
						msg = {:data => ""}
						msg = wait_for_message :data => :_any
					ensure
						@data_received << msg[:data]
					end
					if options[:display_byte_count] then
						size += msg[:data].size
						((last_size+1)..(size / 1000)).each do |val|		#/
							@log.printf(:in, "[%d] bytes received\n", val * 1000)
						end
						last_size = size / 1000	# /
					else
						@log.put_multiple_lines(:in, msg[:data])
					end

					$TRACE.debug 5, "[#{@redirector.comm_port}]: got data size = #{msg[:data].size}"
					$TRACE.debug 5, "[#{@redirector.comm_port}]: got data = '#{msg[:data].x}'"


					m = check_match(regex, options)
					if m then
						@log.printf(:in, "[%d] bytes received (total)\n", size - @data_received.size)	if options[:display_byte_count]
						return m
					end
				end
			end
		rescue Timeout::Error
			message = "ERROR (#{options[:timeout]} secs): no '#{regex.source.x}' in "
			if options[:display_byte_count] then
				@log.puts(:status, message + "#{@data_received.size} bytes")
			else
				@log.puts(:status, message + "'#{@data_received.x}'")
			end
			raise TimeoutError.new(message, regex, @data_received)
		end
	end

	#
	# write a list of strings waiting for a regex after each one
	#
	def write_many_and_wait_for(*args)
		args = args.flatten
		command_list = []
		while args.first.is_a?(String) 
			command_list.push(args.shift)
		end

		if args.empty? || !(args.first.is_a?(Regexp)) then
			raise ArgumentError.new("Requires a Regexp argument after the strings")
		else
			regex = args.shift
		end

		if !args.empty? then
			if args.first.is_a?(Hash) then
				options = args.shift
				raise ArgumentError.new("Too many arguments") if !args.empty
			else
				raise ArgumentError.new("Can only have an optional options hash after the Regexp argument")
			end
		else
			options = {}
		end

		command_list.each do |command|
			write_and_wait_for(command, regex, options)
		end
	end


	WRITE_AND_WAIT_FOR_DEFAULT_OPTIONS = {
		:timeout => 5,							# timeout is for 5 seconds
		:num_tries => 3,						# display byte count for large downloads
		:time_between_tries => 1,
	}

	#
	# write data to the port and wait for a regular expression to be satisfied
	# or timeout. This will also retry a command options[:num_tries] times with
	# a delay of options[:time_between_tries] seconds before trying again.
	#
	def write_and_wait_for(data, regex, options={})
		options = WRITE_AND_WAIT_FOR_DEFAULT_OPTIONS.merge(options)

		num_tries = options[:num_tries]
		time_between_tries = options[:time_between_tries]
		time_between_tries = options[:time_between_tries]
		
		timeout_exception = nil
		num_tries.times do
			timeout_exception = nil
			write(data, options)
			begin
				m = wait_for(regex, options)
			rescue TimeoutError => timeout_exception
			end
			
			return m unless timeout_exception
			sleep(time_between_tries)
		end

		raise timeout_exception
	end

	SYNC_OPTIONS = {
		:timeout => 30						# timeout is for 30 seconds for sync to be fulfilled
	}

	#
	# Synchronize with another test on another port
	#
	def sync(other_port, options={})
		options = SYNC_OPTIONS.merge(options)
		write_filename = SYNC_FILE % other_port
		read_filename = SYNC_FILE % @redirector.comm_port

		$TRACE.debug 9, "[#{@redirector.comm_port}]: writing to '#{write_filename}'"
		File.open(write_filename, "w") do |f|
			begin
				lock_file(f, read_filename) do
					$TRACE.debug 5, "[#{@redirector.comm_port}]: LOCKED writing #{@sync_num+1} from '#{read_filename}'"
					f.puts(@sync_num+1)
				end
			end
		end

		(options[:timeout]*10).times do
			$TRACE.debug 9, "[#{@redirector.comm_port}]: reading '#{read_filename}'"
			begin
				File.open(read_filename, "r") do |f|
					lock_file(f, write_filename) do
						sync_num = f.gets.to_i
						$TRACE.debug 5, "[#{@redirector.comm_port}]: LOCKED reading #{sync_num} from '#{read_filename}'"
						if sync_num >= @sync_num+1 then
							@log.puts(:status, "port #{@redirector.comm_port} synced with #{other_port}")
							@sync_num += 1
							return
						end
					end
				end
			rescue Errno::ENOENT
				$TRACE.debug 5, "[#{@redirector.comm_port}]: read file '#{read_filename}' not found."
			end
	
			sleep(0.1)
		end

		message = "timed out waiting for #{@sync_num+1} in '#{read_filename}'"
		@log.puts(:error, message)
		raise SyncTimeoutError.new(message, read_filename, @sync_num+1)
	end

	def close
		@log.close
	end


private
	#
	# wait for the passed in message_spec and throw away all other messages from the other threads
	#
	def wait_for_message(message_spec)
		$TRACE.debug 5, "[#{@redirector.comm_port}]: looking for message: #{message_spec.inspect} in thread #{Thread.current.inspect}"
		while true
			Thread.current.receive
			if Thread.current.message then
				$TRACE.debug 5, "[#{@redirector.comm_port}]: got message: #{Thread.current.message.inspect}"
				if message_spec === Thread.current.message then
					$TRACE.debug 5, "[#{@redirector.comm_port}]: matched message: #{Thread.current.message.inspect}"
					return Thread.current.message
				end
			end

			if @user_aborted then
				@user_aborted = false
				raise UserAbortError.new("User aborted wait_for")
			end
		end
	end

	#
	# wait until we receive the connected message
	#
	def wait_until_connected
		wait_for_message :connected => :_any
	end

	#
	# wait until we receive the disconnected message
	#
	def wait_until_disconnected
		wait_for_message :connected => false
	end

	#
	# checks if a match has occured either on the main regular expression (regex) or
	# on the options[:match_also] regular expressions
	#
	def check_match(regex, options)
		# check for match also's
		options[:match_also].each do |regex_also, p|
			# for all matches we can find
			while true
				# if we have already found a match
				if start_index = @match_also_indexes[regex.source] then
					m = regex_also.match(@data_received[start_index..-1])
				else
					m = regex_also.match(@data_received)
				end

				if m then
					p.call(m)

					@match_also_indexes[regex.source] = @data_received.size - m.post_match.size
				else
					break
				end
			end
		end
		
		if m = regex.match(@data_received) then
			$TRACE.debug 5, "[#{@redirector.comm_port}]: matched #{regex.source} to #{@data_received.inspect} with post match #{m.post_match.inspect}"
			@data_received = m.post_match
			@data_received = "" unless @data_received
			$TRACE.debug 5, "[#{@redirector.comm_port}]: data received now = #{@data_received.inspect}"
			@last_match = m
			return m
		end
		return nil
	end

	#
	# utility function to lock the file and make sure it is unlocked on exit
	#
	def lock_file(f, filename="")
		begin
			begin
				f.flock(File::LOCK_EX)
			rescue Errno::EINVAL
				$TRACE.debug 1, "[#{@redirector.comm_port}]: Invalid argument: on lock - filename: '#{filename}'"
			end
			yield
		ensure
			begin
				f.flock(File::LOCK_UN)
			rescue Errno::EINVAL
				$TRACE.debug 1, "[#{@redirector.comm_port}]: Invalid argument: on unlock - filename: '#{filename}'"
			end
		end
	end

	#
	# set the uart up according to the 
	#
	def set_uart
		@redirector.set_uart(Thread.current, @baud_rate, @data_bits, @parity, @stop_bits, @flow_control)
	end
end

