require "testerredirector"
require "rfc2217redirector"
require "mockredirector"
require "portwriter"

module RSerialPort
	OK = 				/\r\nOK\r\n/
	CONNECT = 		/\r\nCONNECT\r\n/
	X4_CONNECT =	/\r\nCONNECT (\d+) (\w+)\r\n/
	ERROR = 			/\r\nERROR\r\n/
	OK_OR_ERROR = 	/\r\n(OK|ERROR)\r\n/

	ETX =				"\x03"
	DLE =				"\x10"
	XON =				"\x11"
	DC2 =				"\x12"
	XOFF =			"\x13"

	#TimeoutError = PortWriter::TimeoutError
	
	def with_port(*args, &block)
		SerialPort.new(*args, &block)
	end

	class SerialPort
		attr_reader :redirector, :port_writer

		INITIALIZE_OPTIONS = {
			:host => "localhost",
			:host_type => :tester,
			:log => :use_time_stamped		# or :use_same_log
			#:output_to => 					# use PortWriter default (which is stdout)
		}

		def initialize(port_name, options={}, &block)
			options = INITIALIZE_OPTIONS.merge(options)
			case options[:host_type]
			when :tester
				@redirector = TesterRedirector.new(port_name)
			when :rfc2217
				@redirector = RFC2217Redirector.new(port_name)
			when :mock
				@redirector = MockRedirector.new(port_name)
			else
				raise "unknown host_type: #{options[:host_type].inspect}"
			end

			port_writer_options = {}
			if options[:log] == :use_time_stamped then
				port_writer_options = {:log_filename => "pw-log-#{port_name}-#{Time.now.strftime("%m-%d--%H-%M-%S")}.txt"}
			end
			[:output_to, :options, :status_proc].each do |key|
				port_writer_options[key] = options[key] if options[key]
			end
			
			@port_writer = PortWriter.new(@redirector, port_writer_options)

			if options[:log] == :use_time_stamped then
				@port_writer.log.puts("Filename: #{port_writer_options[:log_filename]}")
			end

			begin
				@redirector.start
				@port_writer.connect(options[:host], &block)
			rescue PortWriter::TimeoutError
			rescue PortWriter::SyncTimeoutError
			rescue Exception => e
				@port_writer.log.puts e.message
				e.backtrace.each do |bt|
					@port_writer.log.puts bt
				end
				@port_writer.log.puts('-------------- errored end of test -------------')
			ensure
				close if block
			end
		end

		def close
			@port_writer.close
		end
	end

end

include RSerialPort

