require "socket"
require "rlang"
require "timeout"
require "my/mystring"

class TesterRedirector
	COMMAND_PORT_BASE = 8000
	DATA_PORT_BASE = 8100
	
	DATA_CHUNK_SIZE = 1024		# maximum size of chunk of data to write

	IN_EVENT_HASH = {
		"C" => :cts_high,
		"c" => :cts_low,
		"S" => :dsr_high,
		"s" => :dsr_low,
		"D" => :dcd_high,
		"d" => :dcd_low,
		"R" => :ring_high,
		"r" => :ring_low,
		"B" => :break_detected
	}

	OUT_EVENT_HASH = {
		:dtr_high => "D",
		:dtr_low => "d",
		:rts_high => "R",
		:rts_low => "r",
		:set_uart => "U%d-%d-%s-%d-%s",
		:send_break => "B%d"
	}

	PARITY_HASH = {
		:none => "N",
		:even => "E",
		:odd => "O",
		:mark => "M",
		:space => "S"
	}

	FLOW_CONTROL_HASH = {
		:none => "N",
		:hardware => "H",
		:software => "S"
	}
	
	acts_as_rlangable

	interface :connect, [:pid, :host]
	message :connect, [[:pid, :_any, :host, :_any]]

	interface :disconnect, [:pid]
	message :disconnect, [[:pid, :_any]]

	interface :send_data, [:pid, :data]	
	message :send_data, [[:pid, :_any], [:data, :_any]]

	interface :change_signal, [:pid, :event]
	message :change_signal, [[:pid, :_any], [:event, :_any]]

	interface :set_uart, [:pid, :baud_rate, :data_bits, :parity, :stop_bits]
	message :set_uart, [[:pid, :_any], [:baud_rate, :_any], [:data_bits, :_any], [:parity, :_any], [:stop_bits, :_any], [:flow_control, :_any]]

	interface :send_break, [:pid, :break_length]
	message :send_break, [[:pid, :_any], [:break_length, :_any]]

	interface :clear_transmit_data, [:pid]
	message :clear_transmit_data, [[:pid, :_any]]

	thread_func	:run

	attr_reader :comm_port
	
	def initialize(comm_port)
		@comm_port = comm_port
		@connected_lock = Mutex.new
		@command_data = ""
		@receive_data = ""
		@connected = false
		@in_command_queue = []
		@out_command_queue = []
	end

	def connect(listener, host)
		#@rx_data_file = File.open("rx2.bin", "wb")
		
		@listener = listener
		
		begin
			$TRACE.debug 5, "attempting to connect to '#{host}'"
			@command_port = TCPSocket.new(host, COMMAND_PORT_BASE + @comm_port)
			#puts "connected to command port"
			@data_port = TCPSocket.new(host, DATA_PORT_BASE + @comm_port)
			#puts "connected to data port"
			$TRACE.debug 5, "connection suceeded!'"
			
		rescue Exception => e
			$TRACE.debug 5, "connection failed because of '#{e.message}''"
			@listener << {:connected => e.message}
			return
		end
		
		#@connected_lock.try_lock do
			@connected = true
		#end

		@listener << {:connected => true}
	end

	def disconnect(listener)
		# wait until the command queue has been emptied out before closing the ports
		while !@out_command_queue.empty?
			run
		end
		
		@command_port.close
		@data_port.close
		
		@command_port = nil
		@data_port = nil
		@connected = false

		#@rx_data_file.close

		listener << {:connected => false}
	end

	def clear_transmit_data(sender)
	end

	def send_data(sender, data)
		index = 0
		$TRACE.debug 5, "send (#{data.size})='#{data[0..80].x}'"
		while (data.size - index) > DATA_CHUNK_SIZE 
			@out_command_queue << {:data => data[index..(index+DATA_CHUNK_SIZE-1)]}
			index += DATA_CHUNK_SIZE
		end
		@out_command_queue << {:data => data[index..-1]}
	end

	def send_break(sender, break_length)
		$TRACE.debug 0, "send_break: #{break_length} milliseconds"
		@out_command_queue << {:event => [:send_break, break_length]}
	end
	
	def change_signal(sender, event)
		$TRACE.debug 0, "change_signal: #{event.inspect}"
		@out_command_queue << {:event => event}
	end

	def set_uart(sender, baud_rate, data_bits, parity, stop_bits, flow_control)
		@out_command_queue << {:event => [:set_uart, baud_rate, data_bits, PARITY_HASH[parity], stop_bits, FLOW_CONTROL_HASH[flow_control]]}
	end
	
	def process_commands
		$TRACE.debug 5, "process_commands(begin): @comamnd_data = #{@command_data.x}, queue = #{@in_command_queue.inspect}"
		# while there are complete commands available
		while index = @command_data.index(/\n/)
			$TRACE.debug 9, "process_commands(parse): #{index}"

			command = @command_data[0..(index-1)]
			@command_data = @command_data[index+1..-1]

			$TRACE.debug 9, "process_commands(parse2): #{command.x}, #{@command_data.x}"

			case command.chomp
			when /^D(\d+)$/
				#@num_data_bytes = $1.to_i
				$TRACE.debug 9, "process_commands(parse3): got data command #{$1}"
				@in_command_queue << {:data => $1.to_i}
			when /^E([CcDdSsRrB])$/
				raise "unknown event: #{$1}" unless IN_EVENT_HASH[$1]
				$TRACE.debug 9, "process_commands(parse3): got event command #{IN_EVENT_HASH[$1]}"
				@in_command_queue << {:event=> IN_EVENT_HASH[$1]}
			else
			end
		end

		$TRACE.debug 5, "process_commands(mid): #{@in_command_queue.inspect}"

		num_data_bytes = 0
		commands_processed = 0
		@in_command_queue.each_with_index do |command, index|
			$TRACE.debug 9, "process_commands(mid): each command: command=#{command.inspect}, index=#{index}, processed = #{commands_processed}, num_bytes=#{num_data_bytes}"
			if command[:event] then
				if num_data_bytes > 0
					$TRACE.debug 9, "process_commands(mid): send send #{num_data_bytes} bytes because of event"
					send_data_message(num_data_bytes)
					num_data_bytes = 0
				end
					
				@listener << command
				commands_processed += 1
			else
				if command[:data] + num_data_bytes <= @receive_data.size then
					num_data_bytes += command[:data]
					commands_processed += 1
				else
					if num_data_bytes > 0
						$TRACE.debug 9, "process_commands(mid): send send #{num_data_bytes} bytes because of not enough data"
						send_data_message(num_data_bytes) 
						num_data_bytes = 0
					end
					break
				end
			end
		end

		if num_data_bytes > 0 then
			$TRACE.debug 9, "process_commands(mid): send #{num_data_bytes} bytes because enough data"
			send_data_message(num_data_bytes) 
		end

		@in_command_queue = @in_command_queue[commands_processed..-1]

		$TRACE.debug 5, "process_commands(end): #{@in_command_queue.inspect}"
	end

	def send_data_message(num_data_bytes)
		$TRACE.debug 9, "num_data_bytes = #{num_data_bytes}"
		$TRACE.debug 9, "receive_data(before) = #{@receive_data.inspect}"
		data_message = {:data => @receive_data[0..(num_data_bytes-1)]}
		@receive_data = @receive_data[num_data_bytes..-1]
		@receive_data = "" unless @receive_data
		$TRACE.debug 5, "data_message = #{data_message.inspect}"
		#$TRACE.debug 9, "receive_data(after) = #{@receive_data.inspect}"
		@listener << data_message
	end
	
	def got_cmd(data)
		@command_data << data
		process_commands
	end

	def got_data(data)
		#@rx_data_file.write(data)
		
		#$TRACE.debug 9, "got_data: receive_data(before) = #{@receive_data.inspect}, data = #{data.inspect}"
		@receive_data << data
		#$TRACE.debug 9, "got_data: receive_data(after) = #{@receive_data.inspect}"
		process_commands
	end
	
	def run		
		#$TRACE.debug 5, "in run: connected = #{@connected}"
		#@connected_lock.try_lock do
			return unless @connected
		#end

		#$TRACE.debug 9, "before select"
		fds = select([@command_port, @data_port], @out_command_queue.empty? ? [] : [@command_port, @data_port], nil, 0.1)
		#$TRACE.debug 9, "after select fds = #{fds.inspect}"
		if fds then
			read_ports = fds[0]
			write_ports = fds[1]
			read_ports.each do |port|
				#$TRACE.debug 9, "before port.sysread"
				data = port.sysread(1024)
				#$TRACE.debug 9, "after port.sysread"
				if port == @command_port
					$TRACE.debug 5, "-CMD: '#{data.x}'"
					got_cmd(data)
				elsif port == @data_port 
					$TRACE.debug 5, "-DATA: '#{data.x}'"
					got_data(data)
				else
					$TRACE.debug 0, "unknown result from select read, port = #{port.inspect}"
				end
			end

			unless @out_command_queue.empty?
				$TRACE.debug 9, "out_command_queue = #{@out_command_queue.inspect}"
				next_out_command = @out_command_queue.first
				if next_out_command.has_key?(:data) && write_ports.size == 2  then
					$TRACE.debug 5, "+DATA: '#{next_out_command[:data].x}'"
					@data_port.write(next_out_command[:data])
					cmd_string = "D#{next_out_command[:data].size}"
					$TRACE.debug 5, "+CMD: '#{next_out_command[:data]}'"
					@command_port.write("#{cmd_string}\n")
					@out_command_queue.shift
				elsif next_out_command.has_key?(:event) && write_ports.include?(@command_port) then
					event = next_out_command[:event]
					event_str = "E"
					if event.is_a?(Array) then
						event_str += OUT_EVENT_HASH[event[0]] % event[1..-1]
					else
						event_str += OUT_EVENT_HASH[event]
					end
					$TRACE.debug 5, "+CMD: '#{event_str}'"
					@command_port.write("#{event_str}\n")
					@out_command_queue.shift
				end
			end
		end
	end
end


