require "rserialport"
require "usage"
require "optparse"
require "inifile"

class TesterINIFile < INIFile
	def sections
		s = super
		s.delete("")
		if s.size > 1 then
			s.delete("global")
			return s
		elsif s.size == 1 && s.first.downcase == "global"
			raise "starting-port key required in file" unless @info["global"]["starting-port"]
			return starting_port_list(@info["global"]["starting-port"])
		else
			return []
		end
	end

	def info_hash(section)
		if (@info[section])
			if @info["global"]
				return @info["global"].merge(@info[section])
			else
				return @info[section]
			end
		elsif (@info["global"])
			return @info["global"]
		else
			return {}
		end
	end

	def info(key, section="")
		key = key.to_s.downcase
		section = section.downcase
		$TRACE.debug 9, "looking for '#{section}:#{key}' in info hash = #{@info.inspect}"
		if (@info[section] && @info[section].has_key?(key)) then
			return @info[section][key]
		elsif (@info["global"].has_key?(key)) then
			return @info["global"][key]
		else
			return @defaults[key]
		end
	end

	def starting_port_list(port_list)
		port_list.split(/,/).inject([]) do |sum, entry| 
			case entry
			when /^(\d+)$/
				sum + [$1]
			when /^(\d+)-(\d+)$/
				sum + (($1.to_i)..($2.to_i)).to_a.map{|x| x.to_s}
			else
				raise "Invalid entry in starting-port list: '#{entry}'"
			end
		end
	end
end

class RunnerOptionParser < OptionParser
	attr_reader :value_hash
	def initialize(*args, &block)
		@value_hash = {}
		super(args, &block)
	end

	def parse_args_in_order(*args, &block)
		begin
			self.order(*args)
		rescue Exception => e
			puts "exception: #{e.message}"
			block.call(e)
		end
	end
end

class String
	def has_extension
		m = /\.(.*)$/.match(self)

		if m then
			return m[1]
		else
			return nil
		end
	end

	def new_extension(extension)
		self.gsub(/\..*$/, ".#{extension}")
	end

	def without_extension
		if /^(.*)\.([^\.]*)$/.match(self) then
			return $1
		else
			return self
		end
	end
end

DEFAULTS = [
	["b", "baud-rate",		"speed", 			33600,				Integer],
	["c", "call-type", 		"type", 				"answer", 			String],
	["C", "compare-files",	nil,					false,				nil],
	["f", "fax-class", 		"class", 			"2.1", 				String],
	["d", "data-file", 		"filename", 		"testfax0.fax", 	String],
	["i", "ip-address", 		"address", 			"localhost", 		String],
	["m", "multi-page-fax",	"filename",			false,				nil],
	["n", "num-cycles", 		"number", 			1,						Integer],
	["N", "num-pages",		"number",			1,						Integer],
	["p", "phone-num",		"phone-number",	"",				String],
	["r", "redirector-type",	"type", 			"tester", 			String],
	["R", "reference-file",	"filename",			"testfax0.fax",	String],	
	["s", "starting-port",	"port-number",		"1", 					String],
	["t", "trace-level",		"level", 			"0", 					Integer],
	["T", "test-file",		"filename",			nil,					String],
]

class Runner
	def initialize(ui)
		@ui = ui

		test_name = nil
		@port_infos = []
		@options = {}
		@test_paused = false
		@port_writers_mutex = Mutex.new
		@port_writers = []
		req_args = nil

		opts = RunnerOptionParser.new do |opts|
			DEFAULTS.each do |short_name, long_name, argument, default, arg_type|
				short_option = "-#{short_name}"
				long_option = "--#{long_name}" + (argument ? " #{argument.upcase}": "")
				#puts "short_option = '#{short_option}', long_option = '#{long_option}'"
				opts.on(short_option, long_option, arg_type) do |value|
					opts.value_hash[long_name] = value
				end
			end
		end

		req_args = opts.parse_args_in_order(ARGV) {|e| error(e.message)}

		if opts.value_hash.has_key?("trace-level") then
			$TRACE.set_level opts.value_hash["trace-level"]
			$TRACE.set_output_filename "log.txt"
		end
		
		$TRACE.debug 5, "after initial parse req_args =#{req_args.inspect}"

		# if there is a testfile as an argument
		if req_args.empty? then
			error("expected either a testfile argument or test script and port number pairs")
			
		elsif req_args.size == 1 then
			test_file_name = req_args.shift
			# if no extension on the filename
			if !(extension = test_file_name.has_extension) then
				test_file_name += ".tst"
			else
				if extension.downcase != "tst" then
					error("when run with one argument, it requires a .tst file")
				end
			end

			if !File.exist?(test_file_name) then
				error("unable to open test file '#{test_file_name}'")
			end

			defaults = {}
			DEFAULTS.each {|short_name, long_name, name, default| defaults[long_name] = default}

#$TRACE.set_level 9 do
			tester_file = TesterINIFile.new(test_file_name, defaults)
#end
			tester_file.sections.each do |section|
				$TRACE.debug 5, "options = #{tester_file.info_hash(section.inspect)}, command line = #{opts.value_hash.inspect}"
				@options[section] = tester_file.info_hash(section).merge(opts.value_hash)
				unknown_keys = @options[section].keys - defaults.keys
				unless unknown_keys.empty?
					error("unknown key entries in test file '#{test_file_name}': #{unknown_keys.join(', ')}")
				end
				@options[section]["port-number"] = section.to_i
			end

		# other wise the arguments are test-file port-num [test-file port-num]...
		else  #if req_args.size > 0 && req_args.size % 2 == 0

			loop do
				test_file = req_args.shift or error("test script argument expected")
				port_num_str = req_args.shift or error("port number script expected")

				if !test_file.has_extension then
					test_file += ".rb"
				end
				port_num = port_num_str.to_i

				error("test script '#{test_file}' doesn't exist") unless File.exist?(test_file)
				error ("port number argument '#{port_num_str}' is not a number") unless /^(\d+)$/.match(port_num_str)

				@options[port_num_str] = opts.value_hash.dup
				@options[port_num_str]["test-file"] = test_file
				@options[port_num_str]["port-number"] = port_num_str.to_i

				break if req_args.empty?

				req_args = opts.parse_args_in_order(req_args) {|e| error(e.message)}
			end
		end

		@options.keys.sort.each do |port_num|
			@port_infos.push(["Port #{port_num}", port_num.to_i])
		end
		
=begin
		@usage = Usage.new("[-r win|unix] [-i ip_address] test_files_equal_port_nums...")
		@usage.test_files_equal_port_nums.each do |test_file_or_port_num|
			if /^(\d+)$/.match(test_file_or_port_num) then
				@usage.die "no test name set for port #{test_file_or_port_num}" unless test_name
				@port_infos.push(["port #{$1}", $1.to_i, test_name])
			else
				test_name = test_file_or_port_num
				if test_name !~ /\.rb$/ then
					test_name += ".rb"
				end
			end
		end

		@ip_address = @usage.ip_address if @usage.ip_address
		@host_type = HOST_TYPE_HASH[@usage.dash_r] if @usage.dash_r
=end
	end

	def error(str, display_error=true)
		puts "USAGE: #$0: options (.tst_file | ruby_test_file port_num [ruby_test_file port_num]...)"
		if display_error then
			puts
			puts "ERROR: #{str}"
		end
		exit
	end
	
	HOST_TYPE_HASH = {
		"tester" => :tester,
		"rfc2217" => :rfc2217,
		"mock" => :mock
	}

	def defaults_hash
		hash = {}
		DEFAULTS.each do |short_name, long_name, argument, default, type|
			hash[long_name] = default
		end
		hash
	end

	def run
		$TRACE.debug 5, "@options = #{@options.inspect}"

		#options = {:command => "X0"}
		test_options = {}
		
		window = @ui.show_window(self, @port_infos)
		threads = []
		num_tests = @options.keys.size
		cycle_done_counter = 0
		cycle_done_mutex = Mutex.new
		
		@options.keys.each do |port_num_str|
		#@port_infos.each do |name, port, test_file|
		#	output_to_proc = @ui.output_to_proc(window, port)

			output_to_proc = @ui.output_to_proc(window, port_num_str.to_i)
			status_proc = @ui.status_proc(window, port_num_str.to_i)
			port_num_options = defaults_hash.merge(@options[port_num_str])

			test_file = port_num_options["test-file"]
			$TRACE.debug 5, "@port_num_options = #{port_num_options.inspect}"

			with_port_options = {}
			with_port_options[:options] = port_num_options
			with_port_options[:host] = port_num_options["ip-address"]
			with_port_options[:host_type] = HOST_TYPE_HASH[port_num_options["redirector-type"]]
			with_port_options[:output_to] = output_to_proc
			with_port_options[:status_proc] = status_proc

#			with_port_options = {}
#			with_port_options[:options] = test_options
#			with_port_options[:host] = @ip_address if @ip_address
#			with_port_options[:host_type] = @host_type
#			with_port_options[:output_to] = output_to_proc
			thread = Thread.new do
				begin
=begin
					eval(<<-EOT)
						with_port(#{port},
										with_port_options) do |sp,options| 
							#{File.read(test_file)}
							sp.log.puts('-------------- end of test -------------')
						end
					EOT
=end
					with_port(port_num_str.to_i, with_port_options) do |sp, options|
						# save the port writer
						@port_writers_mutex.lock
						@port_writers.push(sp)
						@port_writers_mutex.unlock
						
						begin
							sp.instance_eval(<<-EOT)
								#{File.read(test_file)}
							EOT

							if sp.respond_to?(:init_cycles) then
								sp.init_cycles(options)
							end

							if sp.respond_to?(:run_cycle) then
								options["num-cycles"].to_i.times do |i|
									sp.status_info("cycle_num" => i+1, "num_cycles" => options["num-cycles"])

									begin
										sp.log.puts "------------- Start Cycle #{i+1} -----------------"
										sp.run_cycle(i+1, options)

									rescue PortWriter::Error => e
										sp.log.puts "---------------------------------------------------------------------------"
										sp.log.puts "ERROR: #{e.message}"
										sp.log.puts "---------------------------------------------------------------------------"
									ensure
										cycle_done_mutex.lock
											cycle_done_counter = (cycle_done_counter + 1) % num_tests
										cycle_done_mutex.unlock
										sp.log.puts "------------- End Cycle #{i+1} -----------------"
									end

									if @test_paused then
										sp.log.puts "------------------------ Paused -----------------------"
									end

									sp.log.puts "------------- Waiting for sync #{cycle_done_counter} -----------------"
									sleep 0.2 while cycle_done_counter != 0 || @test_paused
									if @test_stopped then
										sp.log.puts "---------------------------------------------------------------------------"
										sp.log.puts "TEST ABORTED"
										sp.log.puts "---------------------------------------------------------------------------"
										break
									end
								end
							end
						ensure
							sp.log.puts('-------------- end of test -------------')
						end
					end
				rescue => e
					$stdout.puts e.message
					$stdout.puts e.backtrace.join("\n")
					$stdout.flush
				end
			end
			threads.push(thread)
		end

		@ui.main_loop(threads)
	end

	def pause_test
		@test_paused = true
	end

	def resume_test
		@test_paused = false
	end

	def stop_cycle
		@port_writers.each do |pw|
			pw.abort_wait_for
		end
	end

	def stop_test
		stop_cycle
		@test_stopped = true
		@test_paused = false
	end
end

