/* vi: ts=8 sw=8
 *
 * TI 3410/5052 USB Compatibility Header
 *
 * Copyright (C) 2004 Texas Instruments
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * For questions or problems with this driver, contact Texas Instruments
 * technical support, or Al Borchers <alborchers@steinerpoint.com>, or
 * Peter Berger <pberger@brimson.com>.
 */

#ifndef _TI_COMPATIBILITY_H_
#define _TI_COMPATIBILITY_H_

#include <linux/version.h>

#define KERNEL_VERSION_ENDIAN_TYPES		9
#define KERNEL_VERSION_MODULE_PARAM_ARRAY	10
#define KERNEL_VERSION_USB_SERIAL_DEBUG		8
#define KERNEL_VERSION_TTY_WAKEUP		9
#define KERNEL_VERSION_USB_KILL_URB		8
#define KERNEL_VERSION_MSLEEP			7
#define KERNEL_VERSION_MSLEEP_INTERRUPTIBLE	9
#define KERNEL_VERSION_RESET_OK			7
#define KERNEL_VERSION_RESET_NO_LOCK		10
#define KERNEL_VERSION_WRITE_KERNEL		10

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,KERNEL_VERSION_ENDIAN_TYPES)
typedef __u16 __be16;
typedef __u16 __le16;
#endif

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,KERNEL_VERSION_MODULE_PARAM_ARRAY)
#define module_param_array_new(name, type, nump, perm) \
	module_param_array(name, type, (*nump), perm)
#else
#define module_param_array_new(name, type, nump, perm) \
	module_param_array(name, type, nump, perm)
#endif

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,KERNEL_VERSION_USB_SERIAL_DEBUG)
extern int debug;
#define usb_serial_debug_data_new(debug, dev, function, size, data) \
	usb_serial_debug_data(__FILE__, function, size, data)
#else
#define usb_serial_debug_data_new(debug, dev, function, size, data) \
	usb_serial_debug_data(debug, dev, function, size, data)
#endif

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,KERNEL_VERSION_TTY_WAKEUP)
void tty_wakeup(struct tty_struct *tty)
{
	/* let the tty driver wakeup if it has a special write_wakeup function */
	if ((tty->flags & (1 << TTY_DO_WRITE_WAKEUP)) && tty->ldisc.write_wakeup)
		(tty->ldisc.write_wakeup)(tty);

	/* tell the tty driver that something has changed */
	wake_up_interruptible(&tty->write_wait);
}
#endif

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,KERNEL_VERSION_USB_KILL_URB)
#define usb_kill_urb(urb) usb_unlink_urb(urb)
#endif

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,KERNEL_VERSION_MSLEEP)
#define MAX_JIFFY_OFFSET ((~0UL >> 1)-1)

static inline unsigned int jiffies_to_msecs(const unsigned long j)
{
#if HZ <= 1000 && !(1000 % HZ)
    return (1000 / HZ) * j;
#elif HZ > 1000 && !(HZ % 1000)
    return (j + (HZ / 1000) - 1)/(HZ / 1000);
#else
    return (j * 1000) / HZ;
#endif
}

static inline unsigned long msecs_to_jiffies(const unsigned int m)
{
    if (m > jiffies_to_msecs(MAX_JIFFY_OFFSET))
        return MAX_JIFFY_OFFSET;
#if HZ <= 1000 && !(1000 % HZ)
    return (m + (1000 / HZ) - 1) / (1000 / HZ);
#elif HZ > 1000 && !(HZ % 1000)
    return m * (HZ / 1000);
#else
    return (m * HZ + 999) / 1000;
#endif
}

static inline void msleep(unsigned int msecs)
{
	unsigned long timeout = msecs_to_jiffies(msecs) + 1;

	while (timeout) {
		set_current_state(TASK_UNINTERRUPTIBLE);
		timeout = schedule_timeout(timeout);
	}
}

static inline unsigned long msleep_interruptible(unsigned int msecs)
{
	unsigned long timeout = msecs_to_jiffies(msecs) + 1;

	while (timeout && !signal_pending(current)) {
		set_current_state(TASK_INTERRUPTIBLE);
		timeout = schedule_timeout(timeout);
	}
	return jiffies_to_msecs(timeout);
}
#elif LINUX_VERSION_CODE < KERNEL_VERSION(2,6,KERNEL_VERSION_MSLEEP_INTERRUPTIBLE)
#define msleep_interruptible(delay) msleep(delay)
#endif

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,KERNEL_VERSION_RESET_NO_LOCK)
#define usb_reset_device(delay) __usb_reset_device(delay)
#endif

#endif /* _TI_COMPATIBILITY_H_ */
