/* ===[ $RCSfile: tasks.c,v $ ]===========================================

    This item is the property of GTECH Corporation, West Greenwich,
    Rhode Island, and contains confidential and trade secret information.
    It may not be transferred from the custody or control of GTECH except
    as authorized in writing by an officer of GTECH.  Neither this item
    nor the information it contains may be used, transferred, reproduced,
    published, or disclosed, in whole or in part, and directly or
    indirectly, except as expressly authorized by an officer of GTECH,
    pursuant to written agreement.

    Copyright (c) 2002-2005 GTECH Corporation.  All rights reserved.

   ======================================================================= */

/** \file

 $Id: tasks.c,v 1.9 2005/03/22 20:24:40 cmayncvs Exp $

 \brief Reads the config for devcom and starts the other comm processes.
 */
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include "gassert.h"
#include "libcom/list.h"
#include "libcom/stats.h"
#include "devcom_debug.h"

/** \brief Determine if security code should be compiled in or not. */
#define ENABLE_SECURITY         0
/** \brief Devcomm debug level */
unsigned char devcom_dbg_lvl = ~0;
/** \brief Argument list for starting the stty module. */
static char *stty_argv[] = {"/bin/stty-inf", "stty", "x42pp", 0};
/** \brief Argument list for starting the X.42pp module. */
static char *x42pp_argv[] = {"/bin/x42pp", "x42pp", "escp", "stty", 0};
/** \brief Argument list for starting the mesc module. */
static char *mesc_argv[] = {"/bin/mesc", "escp", "x42pp", "ethmon", "pad", 0};
/** \brief Argument list for starting the ethmon module. */
static char *ethmon_argv[] = {"/bin/ethmon", "ethmon", "escp", 0};
#if ENABLE_SECURITY /* TODO: Pending R11 Security Audit */
// NEW_TM
/** \brief Argument list for starting the msecp module. */
static char *msecp_argv[] = {"/bin/msecp", "msecp", "escp", 0};
#endif
/** \brief Argument list for starting the pad module. */
static char *pad_argv[] = {"/bin/pad", "pad", "escp", 0};
/** \brief A list of process ID's for each task started. */
static list pid_list = 0;

/**
 \brief Start a unix process.
 \param argv array of argument strings.
 */
static void
start_task (char *argv[])
{
    pid_t pid;

    assert ((pid = fork ()) != -1);
    if (pid == 0)
    {
        assert (execvp (argv[0], argv) != -1);
    }
    else
    {
        info ("Started %s with PID (%d)", argv[0], pid);
        pid_list = acons ((void *)pid, argv[0], pid_list);
    }
}

/**
 \brief Catch SIGCHLD.
 \param signo Not used.
 */
static void
signal_handler (int signo)
{
    pid_t pid;
    int status;
    list t;
    char str[256];

    int eq (void *a, void *b) { return a == b; }

    signo = 0;
    pid = wait (&status);
    t = assoc ((void *)pid, pid_list, eq);

    warn ("Caught SIGCHLD for %s was PID (%d) with Status (%X)", (char *)nth (1, t), pid, status);
    if (WIFEXITED (status)) snprintf (str, sizeof (str), "Exited normally");
    else if (WIFSIGNALED (status)) snprintf (str, sizeof (str), "Uncaught Signal (%d)", WTERMSIG (status));
    else if (WCOREDUMP (status)) snprintf (str, sizeof (str), "Core Dumped");
    else snprintf (str, sizeof (str), "Unknown Reason");
    warn ("%s", str);
    f_fault_write_log ("Caught SIGCHLD for %s was PID (%d) with Status: %s (%X)", (char *)nth (1, t), pid, str, status);
    exit (0);
    //assert ("Caught SIGCHLD" == 0);
}



/**
 \brief Start processes for configured protocol.
 */
void
tasks_start (void)
{
    struct sigaction sa;

    memset (&sa, 0, sizeof (struct sigaction));
    sa.sa_handler = signal_handler;
    assert (sigaction (SIGCHLD, &sa, 0) != -1);

    start_task (stty_argv);
    start_task (mesc_argv);
    start_task (ethmon_argv);
#if ENABLE_SECURITY /* TODO: Pending R11 Security Audit */
// NEW_TM
    start_task (msecp_argv);
#endif
    start_task (pad_argv);
    start_task (x42pp_argv);
}

/**
 \brief Devcom main entry point.
 \return 0, always.  (Should never return.)
 */
int
main (void)
{
    sleep (1);
    while (get_pid ("getty") != 0 || get_pid ("login") != 0)
    {
        sleep (4);
        warn ("Waiting for getty to exit");
    }

    info ("==========[ Starting up all tasks ... ]==========");
    tasks_start ();
    info ("Done.  Waiting to catch any dead children ...");

    for (;;)
        pause ();

    return 0;
}


/*
 * End of $Id: tasks.c,v 1.9 2005/03/22 20:24:40 cmayncvs Exp $
 */

