/* ===[ $RCSfile: ethmon.c,v $ ]==========================================

    This item is the property of GTECH Corporation, West Greenwich,
    Rhode Island, and contains confidential and trade secret information.
    It may not be transferred from the custody or control of GTECH except
    as authorized in writing by an officer of GTECH.  Neither this item
    nor the information it contains may be used, transferred, reproduced,
    published, or disclosed, in whole or in part, and directly or
    indirectly, except as expressly authorized by an officer of GTECH,
    pursuant to written agreement.

    Copyright (c) 2005 GTECH Corporation.  All rights reserved.

   ======================================================================= */

/** \file
 *
 *  "$Id: ethmon.c,v 1.3 2005/03/23 20:41:27 cmayncvs Exp $"
 *
 *  \brief This module monitors the Eth0 interface and reports up/down status.
 *
 */
/* ======================================================================= */

/* ============= */
/* Include Files */
/* ============= */
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <netinet/in.h>
#include <net/if.h>
#include <arpa/inet.h>
#include "libcom/buf.h"
#include "libcom/mq.h"
#include "libcom/msg.h"
#include "libcom/stats.h"
#include "libcom/timer.h"
#include "ethmon.h"
#include "ethmon_imp.h"
#include "gdebug.h"
#include "gassert.h"

/** Indicates no check for status change: send unconditionally. */
#define _DONT_CHECK_FOR_CHANGE  0
/** Indicates check for status change: send only if status changed. */
#define _CHECK_FOR_CHANGE       1

/** The debug level for the ethmon */
u_int8_t ethmon_dbg_level = PRINT_LEVEL;

/** Typedef for ethmon information */
typedef struct {
    int status;                 /**< The link status as DOWN or ONLINE */
    timer tmr;                  /**< The ethmon poll timer */
    int sock;                   /**< The socket ethmon uses to get status */
    struct ifreq ifr;           /**< Socket interface info */
    struct sockaddr_in *addr;   /**< Socket address */
} _ethmon_t;

/** The ethmon information */
static _ethmon_t _ethmon;

#if 0 /* !defined ARCH_ESPAD */
static void _ethmon_debug_init(void);
#endif /* !ARCH_ESPAD */
static void _ethmon_init(void);
static u_int8_t _ethmon_dbg_fw2comm(u_int8_t level);
static char *_ethmon_dbg_level2str(u_int8_t level);
static void _ethmon_send_status(int check);
static void _ethmon_get_flags(void);
static void _ethmon_timer_start(void);
static void _ethmon_stat_print(FILE *f);

/* ======================================================================= */
/**
 *  \brief ETHMON entry point.
 *  \param argc Should be 2 or 3.  More than 3 are ignored.
 *  \param argv - argv[1]=local queue, argv[2]=upper queue, optional argv[3,4]
 *      holds the debug level.
 *
 *  \return Never returns.
 */
/* ======================================================================= */
int main(int argc, char *argv[])
{
    #if 0 /* !defined ARCH_ESPAD */
    _ethmon_debug_init();
    #endif /* !ARCH_ESPAD */

    if ( argc < 3 )
    {
        info("Usage: ethmon <local_queue> <upper_queue> [-d dbglvl]\n");
        exit(0);
    }

    info("ethmon(%d) Started.", getpid());
    ethmon_imp_init(argv[1], argv[2]);
    _ethmon_init();

    while ( 1 )
        msg_dispatch(mq_read());

    return (0);

} /* main() */


/* ======================================================================= */
/**
 *  \brief Handle timeout by reading the Eth0 status, then restart the timer.
 *  \return Nothing.
 */
/* ======================================================================= */
void ethmon_process_timeout(void)
{
    /* Get a snapshot of the flags, then report the status, if it changed */
    _ethmon_get_flags();
    _ethmon_send_status(_CHECK_FOR_CHANGE);

    /* Start the ethmon poll timer for the next status report */
    _ethmon_timer_start();
} /* ethmon_process_timeout() */


/* ======================================================================= */
/**
 *  \brief Set debug level to \a level, after converting to comm level.
 *  \param level The firmware-style debug level to change to.
 *  \return Nothing.
 */
/* ======================================================================= */
void ethmon_dbg_set_level(u_int8_t level)
{
    ethmon_dbg_level = _ethmon_dbg_fw2comm(level);
    warn("Debug level change to %s (%X).",
        _ethmon_dbg_level2str(ethmon_dbg_level), ethmon_dbg_level);
} /* ethmon_dbg_set_level() */

                            /* =============== */
                            /* Local Functions */
                            /* =============== */

#if 0 /* !defined ARCH_ESPAD */
/* ======================================================================= */
/**
 *  \brief Redirects output from stdout to the circular debug log.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _ethmon_debug_init(void)
{
    int debug_fd;

    /* Force kernel debug console_loglevel to lowest possible value */
    system("echo 0 > /proc/sys/kernel/printk");

    /*
     * If f_debug_init() returns -1, assume it's because we're not working
     * on actual target hardware, but a linux development box instead, with no
     * gboard installed.  In that case, don't redirect output.
     */
    if ( (debug_fd = f_debug_init()) != -1 )
    {
        dup2(debug_fd, STDOUT_FILENO);
        dup2(debug_fd, STDERR_FILENO);
    }
} /* _ethmon_debug_init() */
#endif /* !ARCH_ESPAD */

/* ======================================================================= */
/**
 *  \brief Initialize Ethmon.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _ethmon_init(void)
{
    int s;

    /* Create a dummy socket. */
    while ( (s = socket(AF_INET, SOCK_DGRAM, 0)) == -1 )
    {
        warn("A problem occurred trying to create a socket.");
        sleep(1);
    }

    /* Initialize stuff */
    stats_add_proc(_ethmon_stat_print);
    memset(&_ethmon, 0, sizeof(_ethmon));
    _ethmon.status = ETHMON_IMP_ONLINE;
    _ethmon.tmr = ETHMON_TMR_ID;
    _ethmon.sock = s;
    strcpy(_ethmon.ifr.ifr_name, "eth0");
    _ethmon.ifr.ifr_addr.sa_family = AF_INET;

    /* Get a snapshot of the flags and report an initial UP/DOWN status */
    _ethmon_get_flags();
    _ethmon_send_status(_DONT_CHECK_FOR_CHANGE);

    /* Start the ethmon poll timer for the next status report */
    _ethmon_timer_start();

} /* _ethmon_init() */


/* ======================================================================= */
/**
 *  \brief Convert firmware debug level to comm debug level.
 *  \param level The firmware debug level (see debug.h for firmware values).
 *  \return The comm debug level (see debug.h for comm values).
 */
/* ======================================================================= */
static u_int8_t _ethmon_dbg_fw2comm(u_int8_t level)
{
    static u_int8_t _fw2comm[8] = {
        [DBG_NONE] NONE_LEVEL,
        [DBG_ALL] PRINT_LEVEL,
        [DBG_INFO4] PRINT_LEVEL,
        [DBG_INFO3] TRACE_LEVEL,
        [DBG_INFO2] TRACE_LEVEL,
        [DBG_INFO1] INFO_LEVEL,
        [DBG_WARN] WARN_LEVEL,
        [DBG_ERR] WARN_LEVEL
    };
    return (_fw2comm[level & 0x07]);

} /* _ethmon_dbg_fw2comm() */


/* ======================================================================= */
/**
 *  \brief Converts the L2 status from its integer value to a readable string.
 *  \param status The L2 status, as either DOWN or ONLINE.
 *  \return The string representing the L2 status.
 */
/* ======================================================================= */
static char *l2status2str(u_int8_t status)
{
    if ( status == ETHMON_IMP_DOWN )
        return ("DOWN");
    if ( status == ETHMON_IMP_ONLINE )
        return ("ONLINE");
    return ("UNKNOWN");
} /* l2status2str() */


/* ======================================================================= */
/**
 *  \brief Returns a human-readable text string of the debug level.
 *  \param level The debug level.
 *  \return The string.
 */
/* ======================================================================= */
static char *_ethmon_dbg_level2str(u_int8_t level)
{
    switch ( level )
    {
        case NONE_LEVEL:    return ("NONE");
        case WARN_LEVEL:    return ("WARN");
        case INFO_LEVEL:    return ("INFO");
        case TRACE_LEVEL:   return ("TRACE");
        case PRINT_LEVEL:   return ("PRINT");
        default:            return ("UNKNOWN");
    }
} /* _ethmon_dbg_level2str() */


/* ======================================================================= */
/**
 *  \brief Send a comm status, but only if there's a change, unless \a check
 *      indicates to send unconditionally.
 *  \param check A boolean flag indicating if we should check for a change in
 *      status before sending, or simply send unconditionally.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _ethmon_send_status(int check)
{
    int status = (_ethmon.ifr.ifr_flags & IFF_RUNNING) ?
        ETHMON_IMP_ONLINE : ETHMON_IMP_DOWN;

    if ( check )
    {
        if ( status != _ethmon.status )
        {
            info("L2 Status change: %s -> %s.",
                l2status2str(_ethmon.status), l2status2str(status));
            _ethmon.status = status;
            ethmon_imp_send_status(_ethmon.status);
        }
    }
    else    /* send unconditionally */
    {
        info("L2 Status change: %s -> %s.",
            l2status2str(_ethmon.status), l2status2str(status));
        _ethmon.status = status;
        ethmon_imp_send_status(_ethmon.status);
    }

} /* _ethmon_send_status() */


/* ======================================================================= */
/**
 *  \brief Queries the socket for all the flags.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _ethmon_get_flags(void)
{
    /* Get flags */
    while ( ioctl(_ethmon.sock, SIOCGIFFLAGS, &_ethmon.ifr) == -1 )
    {
        warn("Oops - Flags failure.\n");
        sleep(1);
    }

} /* _ethmon_get_flags() */


/* ======================================================================= */
/**
 *  \brief Starts the ethmon poll timer (It ensures it's stopped first).
 *  \return Nothing.
 */
/* ======================================================================= */
static void _ethmon_timer_start(void)
{
    timer_stop(_ethmon.tmr);
    _ethmon.tmr = timer_start(1000, ETHMON_TMR_ID);
} /* _ethmon_timer_start() */


/* ======================================================================= */
/**
 *  \brief Print ETHMON information to a file
 *  \param f The file to write ETHMON information to.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _ethmon_stat_print(FILE *f)
{
    if ( f == NULL )
        return;

    fprintf(f, "---[ ETHERNET MONITORING STATISTICS ]-------------"
        "-------------\n\n");
    fprintf(f, "Eth0 Interface Flags (%08X):\n", _ethmon.ifr.ifr_flags);
    fprintf(f, "   Up?                                     -> %s\n",
        (_ethmon.ifr.ifr_flags & IFF_UP) ? "YES" : "NO");
    fprintf(f, "   Broadcast address valid?                -> %s\n",
        (_ethmon.ifr.ifr_flags & IFF_BROADCAST) ? "YES" : "NO");
    fprintf(f, "   Debugging turned on?                    -> %s\n",
        (_ethmon.ifr.ifr_flags & IFF_DEBUG) ? "YES" : "NO");
    fprintf(f, "   Is a loopback net?                      -> %s\n",
        (_ethmon.ifr.ifr_flags & IFF_LOOPBACK) ? "YES" : "NO");
    /* fprintf(f, "   Is/Has a p-p link?                      -> %s\n",
            (_ethmon.ifr.ifr_flags & IFF_POINTTOPOINT) ? "YES" : "NO"); */
    fprintf(f, "   Avoid use of trailers?                  -> %s\n",
        (_ethmon.ifr.ifr_flags & IFF_NOTRAILERS) ? "YES" : "NO");
    fprintf(f, "   Resources allocated?                    -> %s\n",
        (_ethmon.ifr.ifr_flags & IFF_RUNNING) ? "YES" : "NO");
    fprintf(f, "   No ARP protocol?                        -> %s\n",
        (_ethmon.ifr.ifr_flags & IFF_NOARP) ? "YES" : "NO");
    fprintf(f, "   Receive all packets?                    -> %s\n",
        (_ethmon.ifr.ifr_flags & IFF_PROMISC) ? "YES" : "NO");
    fprintf(f, "   Receive all multicast packets?          -> %s\n",
        (_ethmon.ifr.ifr_flags & IFF_ALLMULTI) ? "YES" : "NO");
    fprintf(f, "   Master of a load balancer?              -> %s\n",
        (_ethmon.ifr.ifr_flags & IFF_MASTER) ? "YES" : "NO");
    fprintf(f, "   Slave of a load balancer?               -> %s\n",
        (_ethmon.ifr.ifr_flags & IFF_SLAVE) ? "YES" : "NO");
    fprintf(f, "   Supports multicast?                     -> %s\n",
        (_ethmon.ifr.ifr_flags & IFF_MULTICAST) ? "YES" : "NO");
    /* This next one is simply a combination of other flags: */
    /* fprintf(f, "   Volatile?                               -> %s\n",
            (_ethmon.ifr.ifr_flags & IFF_VOLATILE) ? "YES" : "NO"); */
    fprintf(f, "   Can set media type?                     -> %s\n",
        (_ethmon.ifr.ifr_flags & IFF_PORTSEL) ? "YES" : "NO");
    fprintf(f, "   Auto media select active?               -> %s\n",
        (_ethmon.ifr.ifr_flags & IFF_AUTOMEDIA) ? "YES" : "NO");
    /* fprintf(f, "   Dialup device with changing addresses?  -> %s\n",
            (_ethmon.ifr.ifr_flags & IFF_DYNAMIC) ? "YES" : "NO"); */
    fprintf(f, "\n");
} /* _ethmon_stat_print() */


/*
 * End of "$Id: ethmon.c,v 1.3 2005/03/23 20:41:27 cmayncvs Exp $".
 */


