/* ===[ $RCSfile: ethmon_imp.c,v $ ]======================================

    This item is the property of GTECH Corporation, West Greenwich,
    Rhode Island, and contains confidential and trade secret information.
    It may not be transferred from the custody or control of GTECH except
    as authorized in writing by an officer of GTECH.  Neither this item
    nor the information it contains may be used, transferred, reproduced,
    published, or disclosed, in whole or in part, and directly or
    indirectly, except as expressly authorized by an officer of GTECH,
    pursuant to written agreement.

    Copyright (c) 2005 GTECH Corporation.  All rights reserved.

   ======================================================================= */

/** \file
 *
 *  "$Id: ethmon_imp.c,v 1.1 2005/03/22 20:29:27 cmayncvs Exp $"
 *
 *  \brief Handle ETHMON Inter-Module Protocol (IMP) messages.
 */
/* ======================================================================= */

/* ============= */
/* Include Files */
/* ============= */
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <unistd.h>
#include <sys/reboot.h>
#include <linux/reboot.h>
#include <string.h>
#include <signal.h>
#include "libcom/buf.h"
#include "libcom/mq.h"
#include "libcom/msg.h"
#include "libcom/stats.h"
#include "mesc/mesc_cfg.h"
#include "ethmon.h"
#include "ethmon_imp.h"
#include "gassert.h"

/* =========================== */
/* Local functions & variables */
/* =========================== */
static char *ethmon_imp_upper_q;    /**< Ethmon's upper_q */

static void _ethmon_imp_process_timeout(buf_t b);
static void _ethmon_imp_set_debug_level(buf_t b);
static void _ethmon_imp_stop(buf_t b);
static void _ethmon_imp_stats(buf_t b);
static void _ethmon_imp_default_proc(buf_t b, const char *file, int line);


/* ======================================================================= */
/**
 *  \brief Saves upper queue, creates ethmon's queue, registers functions
 *      to handle imp messages, and synchronizes with upper queue.
 *  \param local_q ethmon queue name.
 *  \param upper_q The upper queue name.
 *
 *  \return Nothing.
 */
/* ======================================================================= */
void ethmon_imp_init(const char *local_q, const char *upper_q)
{
    assert(local_q && upper_q);

    /* Take care of the queue's */
    ethmon_imp_upper_q = (char *)upper_q;
    mq_create((char *)local_q);

    /* Synch with upper q */
    info("imp_init(): Waiting for %s.", upper_q);
    mq_sync_with((char *)upper_q);

    /* Register messages */
    msg_dispatch_proc("timeout", _ethmon_imp_process_timeout);
    msg_dispatch_proc("set-debug-level", _ethmon_imp_set_debug_level);
    msg_dispatch_proc("stop", _ethmon_imp_stop);
    msg_dispatch_proc("stats", _ethmon_imp_stats);
    msg_default_dispatch_proc(_ethmon_imp_default_proc);
    info("imp_init(): Done.");

} /* ethmon_imp_init() */


/* ======================================================================= */
/**
 *  \brief Send status update to upper queue.  Status is either DOWN or ONLINE.
 *      Format is: comm-status status-update
 *  \param status Status.
 *  \return Nothing.
 */
/* ======================================================================= */
void ethmon_imp_send_status(u_int8_t status)
{
    buf_t b;

    b = msg_create(ethmon_imp_upper_q, 0, "comm-status");
    b = msg_add_field(b, "status-update", &status, 1);
    mq_write(ethmon_imp_upper_q, b);

} /* ethmon_imp_send_status() */

                        /* =============== */
                        /* LOCAL FUNCTIONS */
                        /* =============== */

/* ======================================================================= */
/**
 *  \brief Process a timeout message.  There are 3 timers per drop, response,
 *      keepalive, and disabled timer, each with a unique id.
 *      Format is: "timeout id"
 *  \param b The IMP message.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _ethmon_imp_process_timeout(buf_t b)
{
    u_int32_t *id;
    int length;

    /* Get and validate the timer ID */
    id = (u_int32_t *)msg_get_field(b, "id", &length);
    if ( (id == NULL) || (*id != ETHMON_TMR_ID) )
    {
        warn("Bad timer ID: 0x%08X.", *id);
        return;
    }

    ethmon_process_timeout();

} /* _ethmon_imp_process_timeout() */


/* ======================================================================= */
/**
 *  \brief Allows us to dynamically set a new debug level, while the program
 *      is running.  Format is: "set-debug-level level"
 *  \param b The buffer holding the debug level change message.
 *
 *  \return Nothing.
 */
/* ======================================================================= */
static void _ethmon_imp_set_debug_level(buf_t b)
{
    char *field;
    unsigned int field_len;

    /* Verify message format */
    field = msg_get_field(b, "level", &field_len);
    if ( field == NULL || field_len != 1 )
        return;

    /* Message looks good - handle it */
    ethmon_dbg_set_level((u_int8_t)*field);

} /* _ethmon_imp_set_debug_level() */


/* ======================================================================= */
/**
 *  \brief Allows us to gracefully shut down via the IMP interface.
 *  \param b The buffer holding the stop message.
 *
 *  \return Nothing - Never returns!
 */
/* ======================================================================= */
static void _ethmon_imp_stop(buf_t b)
{
    warn("imp_stop(): Shutting down...\n");
    buf_free(b);
    exit(0);

} /* _ethmon_imp_stop() */


/* ======================================================================= */
/**
 *  \brief Allows us to send ourselves a SIGUSR2 to generate statistics.
 *  \param b The buffer holding the stats message.
 *
 *  \return Nothing
 */
/* ======================================================================= */
static void _ethmon_imp_stats(buf_t b)
{
    info("Statistics written to /tmp/linux-comm-stats.");
    raise(SIGUSR2);
} /* _ethmon_imp_stats() */


/* ======================================================================= */
/**
 *  \brief Default message dispatch procedure.
 *  \param b The IMP message.
 *  \param file The file this function was called from.
 *  \param line The line number this function was called from.
 *
 *  This replaces the standard default procedure and prevents the unimplemented
 *  messages from being written to the debug log.
 */
/* ======================================================================= */
static void _ethmon_imp_default_proc(buf_t b, const char *file, int line)
{
} /* _ethmon_imp_default_proc() */


/*
 * End of "$Id: ethmon_imp.c,v 1.1 2005/03/22 20:29:27 cmayncvs Exp $".
 */


