/* ===[ $RCSfile: gfault.c,v $ ]==========================================

    This item is the property of GTECH Corporation, West Greenwich,
    Rhode Island, and contains confidential and trade secret information.
    It may not be transferred from the custody or control of GTECH except
    as authorized in writing by an officer of GTECH.  Neither this item
    nor the information it contains may be used, transferred, reproduced,
    published, or disclosed, in whole or in part, and directly or
    indirectly, except as expressly authorized by an officer of GTECH,
    pursuant to written agreement.

    Copyright (c) 2002-2004 GTECH Corporation.  All rights reserved.

   ======================================================================= */

/** \file

  $Id: gfault.c,v 1.4 2005/02/09 19:20:30 cmayncvs Exp $

   Re-implementation of the Altura gfault library.

   A big difference between this and the original is that the fault log
   file is kept in shared memory., where it's easier to keep it tidy.
*/
#if BUILD_LOCAL
/** don't reboot a development workstation! */
#define DO_REAL_REBOOT 0
#else
/** do actual reboot on Altura terminal */
#define DO_REAL_REBOOT 1
#endif

#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <stdlib.h>
#include <stdarg.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <time.h>
#include <dirent.h>
#include <signal.h>    /* for sigprocmask(2) */
#include <sys/ipc.h>
#include <sys/shm.h>
#include <sys/sem.h>

#include "gfault.h"
#include "gpaths.h"

#if defined(__GNU_LIBRARY__) && !defined(_SEM_SEMUN_UNDEFINED)
/* union semun is defined by including <sys/sem.h> */
#else
/** According to X/OPEN we have to define it ourselves */
union semun {
   int val;                  /**< value for SETVAL */
   struct semid_ds *buf;     /**< buffer for IPC_STAT, IPC_SET */
   unsigned short *array;    /**< array for GETALL, SETALL
                                  Linux specific part: */
   struct seminfo *__buf;    /**< buffer for IPC_INFO */
};
#endif

#ifndef G_DBG_LVL
/** declare a variable to hold our debug level */
#define G_DBG_LVL g_fault_dbglevel
#endif

#include <gdebug.h>

/** Holds Gtech-specific errors akin to errno */
int gerrno;

/** build data structures needed for debug logging */
G_DBG_DEFDATA

/** IPC keys -- shared memory key */
#define FLT_SHMKEY 0x4746
/** IPC keys -- semaphore key */
#define FLT_SEMKEY (FLT_SHMKEY + 1)

/** per-process shared-memory ID */
static int shmid;
/** per-process semaphore ID */
static int semid;
/** per-process initialized flag */
static int process_initialized;


/** all signals are blocked while holding semaphore lock, restored on unlock */
static sigset_t prev_sigmask;  /* blocked sigset before sem_lock() */

/** per-process PID */
static int mypid;
/** per-process name */
static char myname[16];

/** Structure of each fault record. */
typedef struct {
        unsigned char    valid;              /**< boolean */
        char             msg[FAULTLOGSIZE];  /**< text of msg (w/ timestamp) */
} log_msg_t;

/** Structure of the entire fault log file */
typedef struct {
        log_msg_t  msgs[MAXFAULTS];          /**< array of messages */
} log_file_t;

/** Structure of the global shared memory */
typedef struct {
        unsigned char    initialized;  /**< boolean */
        char             path[255];    /**< set by f_fault_initdir() */
        log_file_t       logfile;      /**< the fault log */
} shared_mem_t;

/** per-process pointer to attached shared memory segment */
shared_mem_t  *shmptr;

/* prototypes for static functions */
#if 0 /* may end up consuming too much nvram */
static void read_proc(const char *outfile);
static void read_proc_entry(FILE *outfp, char *entry);
#endif
static int  sem_lock(void);
static int  sem_unlock(void);
static void make_room(void);
static void save_myname(void);
static void print_log_to_file(FILE *fp);
#if BUILD_LOCAL
static void destroy_shared_memory(void);
#endif /* BUILD_LOCAL */

/***********************************************************************
 *                        Exported Functions                           *
 ***********************************************************************/

/**
 \brief set the directory where the fault log is located.  Called ONCE.
 \param dir the directory to use for the fault log.

 Initialize the global shared memory.  If the fault log file exists,
 read it onto an area in the shared memory.  If it doesn't exist,
 initialize the area in the shared memory to reflect an empty log.

 \warning This function is intended to be called once by a single running
    process at system initialization time.  Presumably this process will be
    System Manager, but on a development environment when testing without
    System Manager running, it will obviously need to be done by another one.

 \return OK, or ERROR, setting gerrno appropriately
*/
int
f_fault_initdir(char *dir)
{
        int fd, rv;
        union semun semopts = {val: 1};

        f_debug_init();
        f_fault_set_debug_level(-1);
        mypid = getpid();
        save_myname();

        /*
         *  we can read the initialized member without being under the
         *  semaphore:  it is written only once.
         */
        if ((shmptr != NULL) && (shmptr != (shared_mem_t*) -1) &&
                            (shmptr->initialized == 1)) {
                DBGPRT(DBG_WARN, "pid %d:  initdir already called\n", mypid);
                return OK;
        }
        /* create (if necessary) the shared memory and attach */
        if ((shmid = shmget(FLT_SHMKEY, sizeof(shared_mem_t),
                            IPC_CREAT | 0777)) < 0) {
                DBGPRT(DBG_ERR, "pid %d. shmget failed:  %s\n",
                       mypid, strerror(errno));
                gerrno = FLTERR_SHMCREAT;
                return ERROR;
        }
        shmptr = shmat(shmid, NULL, 0);
        if (shmptr == (shared_mem_t*) -1) {
                DBGPRT(DBG_ERR, "pid %d, shmat failed:  %s\n",
                       mypid, strerror(errno));
                gerrno = FLTERR_SHMCREAT;
                return ERROR;
        }
        if ((semid = semget(FLT_SEMKEY, 1, IPC_CREAT | 0777)) < 0) {
                DBGPRT(DBG_ERR, "pid %d, semget failed:  %s\n",
                       mypid, strerror(errno));
                if (shmdt(shmptr) == -1) {
                        DBGPRT(DBG_ERR, "pid %d, shmdt failed:  %s\n",
                               mypid, strerror(errno));
                }
                shmptr = NULL;
                gerrno = FLTERR_SEMCREAT;
                return ERROR;
        }

        /* init the semaphore */
        semctl(semid, 0, SETVAL, &semopts);

        /* now, lock it */
        if (sem_lock() != OK)
                return ERROR;   /* gerrno set by sem_lock() */

        /* put the full path to the fault log file in shared memory */
        snprintf(shmptr->path, sizeof(shmptr->path), "%s/fault.log", dir);
        /* try to read in the fault log file, if it exists */
        if ((fd = open(shmptr->path, O_RDONLY)) >= 0) {
                while (1) {
                        /* we should be able to read the whole thing at once */
                        rv = read(fd, &shmptr->logfile, sizeof(log_file_t));
                        if (rv < 0) {
                                if (errno == EINTR)
                                        continue;
                                DBGPRT(DBG_ERR, "read error on %s:  %s\n",
                                       shmptr->path, strerror(errno));
                                break;
                        }
                        break;
                }
                close(fd);
        }
        /*
         *  else
         *      No need to create one in nvram as we work with the shared
         *      memory copy for everything until it's time to either log a
         *      fault or erase the fault log.
         */

        shmptr->initialized = 1;  /* mark global structure initialized */
        (void) sem_unlock();

        /* set the per-process initialized flag */
        process_initialized = 1;
        DBGPRT(DBG_INFO1, "pid %d:  initdir success\n", mypid);
        return OK;
} /* f_fault_initdir() */

/**
 \brief per-process library initialization
 \return OK or ERROR, with gerrno set appropriately

 Attach to shared memory and set the per-process 'mypid' variable
*/
int
f_fault_init(void)
{
        f_debug_init();
        f_fault_set_debug_level(-1);
        mypid = getpid();
        save_myname();

        if (process_initialized) {
                DBGPRT(DBG_WARN, "pid %d:  already initialized\n", mypid);
                return OK;     /* make idempotent */
        }
        process_initialized = 1;    /* being optimistic */

        shmid = shmget(FLT_SHMKEY, sizeof(shared_mem_t), 0777);
        if (shmid < 0) {
                DBGPRT(DBG_ERR, "pid %d, cannot access shared memory:  %s\n",
                       mypid, strerror(errno));
                gerrno = FLTERR_SHMGET;
                process_initialized = 0;
                return ERROR;
        }
        shmptr = shmat(shmid, 0, 0);
        if (shmptr == (shared_mem_t*) -1) {
                DBGPRT(DBG_ERR, "pid %d, shmat failed:  %s\n",
                       mypid, strerror(errno));
                gerrno = FLTERR_SHMGET;
                process_initialized = 0;
                return ERROR;
        }
        if ((semid = semget(FLT_SEMKEY, 0, 0770)) < 0) {
                DBGPRT(DBG_ERR, "pid %d, cannot access semaphore:  %s\n",
                       mypid, strerror(errno));
                gerrno = FLTERR_SEMGET;
                if (shmdt(shmptr) == -1) {
                        DBGPRT(DBG_ERR, "pid %d, shmdt failed:  %s\n",
                               mypid, strerror(errno));
                }
                shmptr = NULL;
                process_initialized = 0;
                return ERROR;
        }
        DBGPRT(DBG_INFO1, "pid %d:  init success\n", mypid);
        return OK;
} /* f_fault_init() */

/**
 \brief Write timestamped entry to fault log and reboot
 \param fmt The format string controlling output of any remaining arguments.

    Writes a log and reboots.  If the library hasn't been initialized,
    it will attempt to so with f_config_init().  If that fails, a call to
    f_config_initdir() is made using the default fault log directory as
    defined in "gpaths.h".

    Does not return.  On an Altura terminal, a reboot will occur; on a
    development workstation if built with BUILD_LOCAL=1, process termination
    will occur and shared memory resources will be deleted, but the
    workstation will not be rebooted.

*/
void
f_fault_write_log(char *fmt, ...)
{
        int     i;
        char    tmpbuf[FAULTLOGSIZE];
        va_list ap;
        time_t  now;
        FILE   *fp;
        int     gotlock = 0;

        /*
         *  Do a formatted print into our tmpbuf.  We'll prepend a timestamp
         *  in the actual message buffer later.
         */
        va_start(ap, fmt);
        vsnprintf(tmpbuf, sizeof(tmpbuf), fmt, ap);
        va_end(ap);
        time(&now);

        if (!process_initialized) {
                DBGPRT(DBG_WARN, "init not called before write_log\n");
                if ((f_fault_init() != OK) &&
                            (f_fault_initdir(G_FAULT_LOG_DIR) != OK)) {
                        goto do_reboot;
                }
        }
        if (sem_lock() != OK)
                goto do_reboot;
        gotlock = 1;

        /* find an empty slot in the log.  If none, make one */
        for (i = 0; i < MAXFAULTS; i++) {
                if (shmptr->logfile.msgs[i].valid == 0)
                        break;
        }
        if (i >= MAXFAULTS) {
                make_room();
                i = MAXFAULTS - 1;  /* use the last slot */
        }

        /* put a timestamp at start of log msg then append the caller's msg */
        snprintf(shmptr->logfile.msgs[i].msg, FAULTLOGSIZE, "%s%s",
                ctime(&now), tmpbuf);
        shmptr->logfile.msgs[i].valid = 1;

        /* try to write the entire fault log to the designated file */
        if ((fp = fopen(shmptr->path, "w")) == NULL) {
                DBGPRT(DBG_ERR, "can't write %s:  %s\n",
                       shmptr->path, strerror(errno));
                goto do_reboot;
        }
        if (fwrite(&shmptr->logfile, sizeof(log_file_t), 1, fp) != 1) {
                DBGPRT(DBG_ERR, "write error on %s:  %s\n",
                       shmptr->path, strerror(errno));
                fclose(fp);
                goto do_reboot;
        }
        fclose(fp);

        system ("saveconf \n");

 do_reboot:
        /*
         * Always write a basic fault log to the circular debug log as well,
         * but don't write the proc entry stuff there so we can preserve a
         * trace of what happened BEFORE the fault occurred.
         */
        f_debug_writemsg("***** FAULT *****\n%s%s\n\n", ctime(&now), tmpbuf);

        #if 0
        /*
         * This seemed like a good idea at the time, but it may end up
         * consuming far too much space in NVRAM.  What we would need is to be
         * able to save this information off somewhere other than NVRAM, but
         * where?  For now, just forget it.
         */
        /* read_proc("stdout"); */
        read_proc(G_FAULT_LOG_DIR"/fault.rpt");
        #endif

        if (gotlock)
                (void) sem_unlock();
#if DO_REAL_REBOOT
        while (1) {
                system("/sbin/reboot");
                /* notreached ?? */
                sleep(10);
        }
#else
        DBGPRT(DBG_INFO1, "<simulated terminal reboot in progress>\n");
        #if 0   /* Set to 0 when testing with the fault_tester.c file */
        destroy_shared_memory();
        #endif
        exit(0);
#endif /* DO_REAL_REBOOT */

} /* f_fault_write_log() */

/**
 \brief erase the entire fault log by populating with zeros
 \return OK or ERROR, with gerrno set appropriately
*/
int
f_fault_erase(void)
{
        FILE  *fp;

        if (!process_initialized) {
                gerrno = FLTERR_INIT;
                return ERROR;
        }
        if (sem_lock() != OK)   /* sem_lock() sets gerrno */
                return ERROR;
        memset(&shmptr->logfile, 0, sizeof(log_file_t));
        if ((fp = fopen(shmptr->path, "w")) == NULL) {
                DBGPRT(DBG_ERR, "can't write %s:  %s\n", shmptr->path,
                       strerror(errno));
                (void) sem_unlock();
                gerrno = FLTERR_OPEN;
                return ERROR;
        }
        if (fwrite(&shmptr->logfile, sizeof(log_file_t), 1, fp) != 1) {
                DBGPRT(DBG_ERR, "write error erasing %s:  %s\n",
                       shmptr->path, strerror(errno));
                fclose(fp);
                (void) sem_unlock();
                gerrno = FLTERR_ERASE;
                return ERROR;
        }
        (void) sem_unlock();
        return OK;
} /* f_fault_erase() */

/**
 \brief get the count of messages in the fault log
 \return return the count
*/
int
f_fault_get_count(void)
{
        int i;

        if (!process_initialized) {
                gerrno = FLTERR_INIT;
                return ERROR;
        }
        if (sem_lock() != OK)
                return ERROR;

        for (i = 0; i < MAXFAULTS; i++) {
                if (shmptr->logfile.msgs[i].valid == 0)
                        break;
        }
        (void) sem_unlock();
        return i;
} /* f_fault_get_count() */

/**
 \brief dump the fault log to the console, a file, or a buffer
 \param dest 0 for console, 1 for a file, 2 for a buffer

 If \a dest is 0, then any other args present are ignored.
 If \a dest is 1, then a second arg is file to dump to.
 If \a dest is 2, then a second arg is pointer to the buffer, AND a
      THIRD arg is the number of log messages to dump.

 \return Upon success, returns OK for dest 0 and 1, but returns the number
    of fault log messages copied to the buffer if dest is 2.  Upon failure,
    returns ERROR, with gerrno set appropriately.
*/

int
f_fault_dump(char dest, ...)
{
        va_list  ap;
        FILE    *fp;
        char    *arg2;  /* either file name or buffer */
        int      nrec = 0;  /* if dest is 2, then count of records to dump */
        int      i, num_valid;

        if (!process_initialized) {
                gerrno = FLTERR_INIT;
                return ERROR;
        }
        if (sem_lock() != OK)
                return ERROR;

        switch (dest) {

        case DUMP_TO_CONSOLE:   /* write to "console" (meaning "stdout") */
                print_log_to_file(stdout);
                fflush(stdout);
                break;

        case DUMP_TO_FILE:      /* write to file */
                va_start(ap, dest);
                arg2 = va_arg(ap, char*);
                va_end(ap);
                if ((fp = fopen(arg2, "w")) == NULL) {
                        DBGPRT(DBG_ERR, "error creating fault dump file %s:  "
                               "%s\n", arg2, strerror(errno));
                        (void) sem_unlock();
                        gerrno = FLTERR_OPEN;
                        return ERROR;
                }
                print_log_to_file(fp);
                fclose(fp);
                break;

        case DUMP_TO_BUFFER:    /* write latest N messages to a buffer */

                /* first, find out how many valid records exist */
                for (num_valid = 0; num_valid < MAXFAULTS; num_valid++) {
                        if (shmptr->logfile.msgs[num_valid].valid == 0)
                                break;
                }
                va_start(ap, dest);
                arg2 = va_arg(ap, char*);
                nrec = va_arg(ap, int);
                va_end(ap);
                if (arg2 == NULL) {
                        DBGPRT(DBG_ERR, "invalid buffer pointer\n");
                        (void) sem_unlock();
                        gerrno = FLTERR_INVAL;
                        return ERROR;
                }
                if (nrec < 0) {
                        DBGPRT(DBG_ERR, "invalid number of records to "
                               "dump %d\n", nrec);
                        (void) sem_unlock();
                        gerrno = FLTERR_INVAL;
                        return ERROR;
                }
                /* if nrec greater than the number of valid records, reduce */
                if (nrec > num_valid)
                        nrec = num_valid;

                /* start with the latest of the ones the caller wants */
                for (i = num_valid - 1; i >= (num_valid - nrec); i--) {
                        if (shmptr->logfile.msgs[i].valid == 0)
                                break;
                        /* print the records at FAULTLOGSIZE intervals
                         *  in caller's buffer.  Since the memory records
                         *  don't have index numbers, we add them.
                         */
                        shmptr->logfile.msgs[i].msg[FAULTLOGSIZE-1] = '\0';
                        snprintf(arg2, FAULTLOGSIZE, "%d) %s",
                                 i+1, shmptr->logfile.msgs[i].msg);
                        /*strncpy(&arg2[FAULTLOGSIZE-3], "\n\n", 3);*/
                        /* increment buffer pointer */
                        arg2 += FAULTLOGSIZE;
                }
                break;

        default:
                DBGPRT(DBG_ERR, "invalid first arg to f_fault_dump %d\n",
                       dest);
                (void) sem_unlock();
                gerrno = FLTERR_INVAL;
                return ERROR;
        }
        (void) sem_unlock();
        if ( dest == DUMP_TO_BUFFER )
            return nrec;
        return OK;
} /* f_fault_dump() */


/**
 \brief Sets the debug level for the fault library
 \param level the debug level (0-7), defined in gdebug.h, or \n
        (-1) indicating that we're calling it internally.

    If the debug level hasn't been set yet and we call it internally, then
    go ahead and turn debug off, but as soon as a call is made to set the
    debug level, then don't ever change it internally.
 \return nothing
*/
void
f_fault_set_debug_level(int level)
{
    static int f_fault_debug_level_not_set = 1;

        if ( level >= 0 ) {
                f_fault_debug_level_not_set = 0;
                G_DBG_SETLEVEL(level & 0x07);
        }
        else if ( f_fault_debug_level_not_set )
                G_DBG_SETLEVEL(DBG_NONE);
} /* f_fault_set_debug_level() */


/**
 \brief Uses the process ID of the calling process to look up its name.
 \return The plain text process name from /proc/self/exe.
*/
char *
f_fault_read_pname(void)
{
    save_myname();
    return (myname);
} /* f_fault_read_pname() */


/***********************************************************************
 *                   Local (static) functions                          *
 ***********************************************************************/

/**
 \brief print the valid log messages to a file
 \param fp the FILE
*/
static void
print_log_to_file(FILE *fp)
{
        int   i;

        for (i = MAXFAULTS - 1; i >= 0; i--) {
                if (shmptr->logfile.msgs[i].valid != 0) {
                        shmptr->logfile.msgs[i].msg[FAULTLOGSIZE-1] = '\0';
                        fprintf(fp, "%d) %s\n\n",
                                i+1, shmptr->logfile.msgs[i].msg);
                }
        }

} /* print_log_to_file() */

/**
 \brief If the log is full, shift all but one upwards in the memory region, \n
    then mark the last entry as free.
 \warning Assumes that we're attached to shared memory ('shmptr' is valid).
 \return Nothing.
*/
static void
make_room(void)
{
        /* see if the last one is valid.  If so, delete first */
        if (shmptr->logfile.msgs[MAXFAULTS - 1].valid != 0) {
                memmove(&shmptr->logfile.msgs[0], &shmptr->logfile.msgs[1],
                        sizeof(log_msg_t) * (MAXFAULTS - 1));
                shmptr->logfile.msgs[MAXFAULTS - 1].valid = 0;
        }

} /* make_room() */


/* ==================================================================== */
/**
 *  \brief Save the caller's process name to better identify it in the log.
 *  \return Nothing.
 */
/* ==================================================================== */
static void save_myname(void)
{
    char tmp[256];
    char *ptr;
    int len;

    if ( myname[0] )    /* don't bother saving more than once. */
        return;
    if ( (len = readlink("/proc/self/exe", tmp, sizeof(tmp)-1)) <= 0 )
        strcpy(myname, "?");
    else
    {
        tmp[len] = '\0';
        /* try to skip past the path to save space in the log */
        if ( (ptr = strrchr(tmp, '/')) != NULL )
        {
            if ( (++ptr == NULL) || (*ptr == '\0') )
                strncpy(myname, tmp, sizeof(myname));
            else
                strncpy(myname, ptr, sizeof(myname));
        }
        else    /* '/' not found */
            strncpy(myname, tmp, sizeof(myname));
        myname[sizeof(myname)-1] = '\0';
    }
} /* save_myname() */


#if 0
/* ==================================================================== */
/*                      PROC FILE SYSTEM DEBUG STUFF                    */
/* ==================================================================== */
/**
 *  \brief Reads all interesting proc info for debug purposes.
 *  \param outfile Indicates where to output the proc information.
 *
 *  If outfile is "stdout", then information is printed to stdout, otherwise
 *  it is written to the file provided by \a outfile.  Note that \a outfile
 *  will be overwritten each time this function is called.
 *
 *  \return Nothing.
 */
/* ==================================================================== */
static void read_proc(const char *outfile)
{
    FILE *outfp;
    int i;
    int so;
    char *pid_entry[] = {
        "/proc/self/cmdline",
        "/proc/self/maps",
        "/proc/self/mem",
        "/proc/self/stat",
        "/proc/self/statm",
        "/proc/self/status",
        NULL
    };
    char *proc_entry[] = {
        "/proc/gtech/gboard/temperature",
        "/proc/gzs_debug",
        "/proc/interrupts",
        "/proc/lc_scc_debug",
        "/proc/loadavg",
        "/proc/locks",
        "/proc/net/arp",
        "/proc/net/sockstat",
        "/proc/meminfo",
        "/proc/misc",
        "/proc/uptime",
        "/proc/version",
        NULL
    };

    /* Open output file */
    so = !strcmp(outfile, "stdout");
    if ( so )
        outfp = stdout;
    else if ( (outfp = fopen(outfile, "w")) == NULL )
        return;

    /* ==================== */
    /* Read all the entries */
    /* ==================== */
    /* pid entries we're interested in */
    for ( i = 0; pid_entry[i]; i++ )
        read_proc_entry(outfp, pid_entry[i]);

    /* proc entries we're interested in */
    for ( i = 0; proc_entry[i]; i++ )
        read_proc_entry(outfp, proc_entry[i]);
    /* ==================== */

    /* Close */
    if ( so )
        fflush(outfp);
    else
        fclose(outfp);
} /* read_proc() */


/* ==================================================================== */
/**
 *  \brief Reads the proc entry from the file and outputs to the \a outfp.
 *  \param outfp The file to write the entry data to.
 *  \param entry The proc entry to read from.
 *  \return Nothing.
 */
/* ==================================================================== */
static void read_proc_entry(FILE *outfp, char *entry)
{
    FILE *infp;
    int len;
    char buf[256];

    /* Open Input file */
    if ( (infp = fopen(entry, "r")) == NULL )
        return;

    /* Read/Write */
    snprintf(buf, sizeof(buf), "%s\n", entry);
    fwrite(buf, strlen(buf), 1, outfp);
    while ( (len = fread(buf, 1, sizeof(buf), infp)) > 0 )
        fwrite(buf, len, 1, outfp);
    buf[0] = buf[1] = '\n';
    fwrite(buf, 2, 1, outfp);
} /* read_proc_entry() */
#endif

/*************************************************************************
 *                             Semaphore                                 *
 *                                                                       *
 *  The original implementation used the semaphore to protect access     *
 *  to the shared memory area.  We extend this to:                       *
 *     -> block all signals while accessing config directories           *
 *     -> allow only one process at a time access to config directories. *
 *  These additions should make the configuration management system      *
 *  more robust.                                                         *
 *************************************************************************/

/**
 \brief lock the global semaphore in per-process variable 'cfg_semid'
 \return OK on success or ERROR on failure (with gerrno set properly)

 Care is taken to properly handle EINTR 'error' returns and to ensure
 that the lock is undone should this process die with the lock held.
*/
static int
sem_lock(void)
{
        struct sembuf    sb;
        sigset_t         all_sigsmask;

        sigfillset(&all_sigsmask);   /* block ALL signals when lock acquired */

        while (1) {
                sb.sem_num = 0;
                sb.sem_op = -1;
                sb.sem_flg = SEM_UNDO; /* assure lock is released if we die! */
                if (semop(semid, &sb, 1) == 0) {
                        break;
                }
                if (errno == EINTR) {
                        continue;    /* interrupted by signal */
                } else {
                        DBGPRT(DBG_ERR, "semaphore lock failed:  %s\n",
                               strerror(errno));
                        gerrno = FLTERR_SEMLOCK;
                        return ERROR;
                }
        }
        /* block all signals while we hold the lock */
        (void) sigprocmask(SIG_BLOCK, &all_sigsmask, &prev_sigmask);
        return OK;
} /* sem_lock() */

/**
 \brief unlock the global semaphore in per-process variable 'cfg_semid'
 \return OK on success or ERROR on failure (with gerrno set properly).

 Once the semaphore is released, the process's previous sigmask restored.
*/
static int
sem_unlock(void)
{
        struct sembuf sb;

        while (1) {
                sb.sem_num = 0;
                sb.sem_op = 1;
                sb.sem_flg = 0;
                if (semop(semid, &sb, 1) == 0)
                        break;
                /*
                 *  Even though all signals are blocked, we still check EINTR
                 */
                if (errno == EINTR) {
                        continue; /* interrupt by signal (should not happen) */
                } else {
                        DBGPRT(DBG_ERR, "semaphore unlock failed:  %s\n",
                               strerror(errno));
                        gerrno = FLTERR_SEMUNLOCK;
                        return ERROR;
                }
        }
        /* restore previous signal mask */
        (void) sigprocmask(SIG_SETMASK, &prev_sigmask, NULL);
        return OK;
} /* sem_unlock() */


#if BUILD_LOCAL
#include <sys/wait.h>

#if 0 /* defined(__GNU_LIBRARY__) && !defined(_SEM_SEMUN_UNDEFINED) */
/* union semun is defined by including <sys/sem.h> */
#else
/** according to X/OPEN we have to define it ourselves */
union semun {
        int val;                    /**< value for SETVAL */
        struct semid_ds *buf;       /**< buffer for IPC_STAT, IPC_SET */
         unsigned short int *array; /**< array for GETALL, SETALL */
        struct seminfo *__buf;      /**< buffer for IPC_INFO */
};
#endif

/***********************************************************************
 *              Local (static) functions used for workstation only!    *
 ***********************************************************************/

/**
    \brief Destroys our shared memory resources to simulate the conditions \n
        present when the fault log is normally initialized.

    On a development environment that is not actually rebooted after a fault,
    there could be left-over shared memory resources that need to be deleted.
 */
static void
destroy_shared_memory(void)
{
        int rv;
        int id;
        struct shmid_ds buf;
        union semun sumun;
        int kidpid;
        int forktries = 5;
        int status;

        /* run the destroy stuff in a child to assure destruction */
        while ((kidpid = fork()) < 0) {
                if (--forktries < 0) {
                        printf("fork failed after retries:  %s\n",
                                strerror(errno));
                        exit(1);
                }
                sleep(1);   /* most likely EAGAIN */
        }
        if (kidpid != 0) {
                /* parent --- wait for child to finish */
                while (1) {
                        if (waitpid(kidpid, &status, 0) == kidpid) {
                                if (WIFEXITED(status) ||
                                    WIFSIGNALED(status)) {
                                        break;
                                }
                        }
                }
                return;  /* parent returns */
        }

        /* child --- try to destroy the shared memory */
        /* make stdout unbuffered */
        setvbuf(stdout, NULL,_IONBF, 0);
        if ((id = shmget((key_t) FLT_SHMKEY, 2048, 0)) < 0) {
                printf("shmget failed:  %s\n", strerror(errno));
        } else {
                if ((rv = shmctl(id, IPC_RMID, &buf)) != 0) {
                        printf("shmctl failed:  %s\n",
                                strerror(errno));
                } else {
                        printf("shmctl destruction assured\n");
                }
        }

        if ((id = semget((key_t) FLT_SEMKEY, 1, 0)) < 0) {
                printf("semget failed:  %s\n", strerror(errno));
        } else {
                if ((rv = semctl(id, 0, IPC_RMID, &sumun)) < 0) {
                        printf("semctl failed:  %s\n",
                                strerror(errno));
                } else {
                        printf("semctl destruction assured\n");
                }
        }
        exit(0);   /* child process exits */
}
#endif /* BUILD_LOCAL */


/*
 * End of "$Id: gfault.c,v 1.4 2005/02/09 19:20:30 cmayncvs Exp $".
 */


