#ifndef INC_gdebug_h
 #define INC_gdebug_h

#include <stdio.h>

/* common definitions */
#ifndef ERROR
 #define ERROR -1
#endif

#ifndef OK
 #define OK 0
#endif

/* debug print error levels */
#define DBG_NONE        0
#define DBG_ALL         1
#define DBG_INFO4       2
#define DBG_INFO3       3
#define DBG_INFO2       4
#define DBG_INFO1       5
#define DBG_WARN        6
#define DBG_ERR         7

/* G_DBG_LVL is a generic symbol name that holds the debug level.
 * Each application must define a unique name for it. 
 * e.g. #define G_DBG_LVL g_dbg_485drvLevel
 *
 * The application must include this header file *gdebug.h* after G_DBG_LVL 
 * is defined.

 * The _DEFDATA_ macros declares the debug level. The main source file of the 
 * application will declare and export a debug level to be used by the other
 * source files.
 */

/* Use macro G_DBG_DEFDATA in the main source file of the application. The macro 
 * defines the debug level for the whole application.
 */
#define G_DBG_DEFDATA \
    static const char * dbg_text[8] = {"DUM0", "DUM1", "I4  ", "I3  ", "I2  ", "I1  ", "WRN ", "ERR "};\
    unsigned char G_DBG_LVL = 0;

/* Use macro G_DBG_DEFDATA_EXTERN in sub-modules (files) of the application. The macro 
 * declares the debug level as extern (to be imported from the main file).
 */
#define G_DBG_DEFDATA_EXTERN \
    static const char * dbg_text[8] = {"DUM0", "DUM1", "I4  ", "I3  ", "I2  ", "I1  ", "WRN ", "ERR "};\
    extern unsigned char G_DBG_LVL; 

#define G_DBG_SETLEVEL(level) G_DBG_LVL = level
#define G_DBG_GETLEVEL G_DBG_LVL

extern int gerrno;

/* prototypes for the files using debug library */
#define f_debug_init() open("/dev/debug_log", O_RDWR)
#define f_debug_writemsg(fmt, args...) do {printf(fmt , ##args); fflush(stdout);} while (0)

/* DBGPRT is a compound statment, hence use of braces is needed whereever applicable */
#define DBGPRT(level, fmt, args...) \
        if((G_DBG_LVL != DBG_NONE) && (level != DBG_NONE) && (level >= G_DBG_LVL)) \
             f_debug_writemsg("%s %s %d " fmt, dbg_text[level], __FILE__, __LINE__ , ##args) 

#define DBGINIT \
        f_debug_init()

#endif
