/* ===[ $RCSfile: buf.c,v $ ]================================================

    This item is the property of GTECH Corporation, West Greenwich,
    Rhode Island, and contains confidential and trade secret information.
    It may not be transferred from the custody or control of GTECH except
    as authorized in writing by an officer of GTECH.  Neither this item
    nor the information it contains may be used, transferred, reproduced,
    published, or disclosed, in whole or in part, and directly or
    indirectly, except as expressly authorized by an officer of GTECH,
    pursuant to written agreement.

    Copyright (c) 2002-2005 GTECH Corporation.  All rights reserved.

   ======================================================================= */

/** \file

 $Id: buf.c,v 1.3 2005/07/18 19:01:48 cmayncvs Exp $

 \brief Implements the standard buffer management functions.

 All buffers are stored in contiguous blocks of memory. The API is keep as
 simple as possible. The idiom used is that functions return the result of
 their operation and not error codes. Errors within a buffer function are
 considered fatal.

 There is no need to explicitly allocate a new buffer before putting data in
 it. If a \c null pointer is passed to buf_append a new buffer will be
 allocated automatically.

 Semantically all for all functions that take a buffer and return a buffer, a
 new buffer is created by the call and the old buffer is no longer valid.
 */
#include <malloc.h>
#include <string.h>
#include <stdio.h>
#include "buf.h"
#include "mq.h"
#include "gassert.h"
#include "gdebug.h"

/** Size of small buffer. */
#define SLAB_SMALL  512
/** Size of large buffer. */
#define SLAB_BIG    MAX_MESSAGE_LENGTH

/** Buffer structure. */
struct buf
{
    unsigned int tag;       ///< A tag associated with the buffer.
    unsigned char misc[8];  ///< Misc data that can ride along with the buffer
    size_t length;          ///< Amount of data in data area.
    size_t size;            ///< Length of data area.
    struct buf *next;       ///< Pointer to next buffer in a queue.
    char data[0];           ///< Pointer to data area.
};

/** Sizeof buffer header. */
static const int header_size = sizeof (struct buf);

/** Free small buffer list. */
static buf_t free_list_small = 0;
/** Free big buffer list. */
static buf_t free_list_big = 0;

/** Buffer statistics. */
static struct
{
    int curr_allocated_small;
    int max_allocated_small;
    int curr_allocated_big;
    int max_allocated_big;
    int dup_frees;
} stats;

/**
 \brief Allocate a small buffer.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 \return Allocated buffer.
 */
static buf_t
alloc_small (const char *file, int line)
{
    buf_t b;
    if (free_list_small)
    {
        b = free_list_small;
        free_list_small = b->next;
    }
    else
    {
        libassert (b = malloc (header_size + SLAB_SMALL));
        b->size = SLAB_SMALL;
    }
    stats.curr_allocated_small++;
    stats.max_allocated_small = MAX (stats.curr_allocated_small, stats.max_allocated_small);
    return b;
}

/**
 \brief Allocate a big buffer.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 \return Allocated buffer.
 */
static buf_t
alloc_big (const char *file, int line)
{
    buf_t b;
    if (free_list_big)
    {
        b = free_list_big;
        free_list_big = b->next;
    }
    else
    {
        libassert (b = malloc (header_size + SLAB_BIG));
        b->size = SLAB_BIG;
    }
    stats.curr_allocated_big++;
    stats.max_allocated_big = MAX (stats.curr_allocated_big, stats.max_allocated_big);
    return b;
}

/**
 \brief Allocate a new buffer.
 \param size Requested buffer size.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 \return Allocated buffer.
 */
static buf_t
alloc_buf (size_t size, const char *file, int line)
{
    buf_t b;

    libassert (size <= SLAB_BIG);
    b = (size > SLAB_SMALL) ? alloc_big (file, line) : alloc_small (file, line);
    b->next = 0;
    b->length = 0;
    b->tag = 0;
    return b;
}

/**
 \brief Copy from a small buffer to a big buffer.
 \param d Big buffer to fill.
 \param s Small buffer to copy from.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 \return Filled big buffer.
 */
static buf_t
copy_and_free (buf_t d, buf_t s, const char *file, int line)
{
    memcpy (d->data, s->data, s->length);
    d->tag = s->tag;
    d->length = s->length;
    d->next = s->next;
    _buf_free (s, file, line);
    return d;
}

/**
 \brief Add data to the end of a buffer.
 \param b Buffer to add to.
 \param vdata Data to add.
 \param length Amount of data to add.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 \return Filled buffer.

 Appends data to end of a buffer. If \a b is \c null, a new buffer is allocated.
 */
buf_t
_buf_append (buf_t b, const void *vdata, size_t length, const char *file, int line)
{
    const char *data = vdata;

    if (b)
    {
        libassert (data && ((b->length + length) <= SLAB_BIG));
        if ((b->length + length) > b->size)
            b = copy_and_free (alloc_buf (b->length + length, file, line), b, file, line);
    }
    else
    {
        libassert (data && (length <= SLAB_BIG));
        b = alloc_buf (length, file, line);
    }
    memcpy (&b->data [b->length], data, length);
    b->length += length;
    return b;
}

/**
 \brief Return pointer to start of data.
 \param b Buffer to get pointer from.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 \return Pointer to the start of data.
 */
void *
_buf_data (buf_t b, const char *file, int line)
{
    libassert (b);
    return b->data;
}

/**
 \brief Get buffer length.
 \param b Buffer to get length from.
 \return Length of data in the buffer.
 */
size_t
buf_length (buf_t b)
{
    if (b)
        return b->length;
    return 0;
}

/**
 \brief Join together two buffers into one.
 \param head Head buffer.
 \param tail Tail buffer.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 \return Joined buffer.

 \warning \a Head and \a tail are freed.
 */
buf_t
_buf_join (buf_t head, buf_t tail, const char *file, int line)
{
    head = _buf_append (head, _buf_data (tail, file, line), buf_length (tail), file, line);
    _buf_free (tail, file, line);
    return head;
}

/**
 \brief Duplicate a buffer.
 \param b Buffer to duplicate.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 \return Duplicate buffer.
 \note _buf_dup duplicate the object and not the reference to the object.
 */
buf_t
_buf_dup (buf_t b, const char *file, int line)
{
    buf_t t;
    unsigned int tag;

    tag = buf_get_tag (b);
    t = _buf_append (0, _buf_data (b, file, line), buf_length (b), file, line);
    buf_set_tag (t, tag);
    return t;
}

/**
 \brief Look for a buffer on the free list.
 \param b Buffer to look for.
 \param fl Free list to search.
 */
static int
already_freed (buf_t b, buf_t fl)
{
    while (fl)
    {
        if (b == fl)
            return 1;
        fl = fl->next;
    }
    return 0;
}

/**
 \brief Free a buffer.
 \param b Buffer to free.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 */
buf_t
_buf_free (buf_t b, const char *file, int line)
{
    if (b)
    {
        libassert ((b->size == SLAB_SMALL) || (b->size == SLAB_BIG));
        if (already_freed (b, (b->size == SLAB_SMALL) ? free_list_small : free_list_big))
        {
            //printf ("***WARNING*** You already freed buffer %p\n", b);
            stats.dup_frees++;
            return 0;
        }
        if (b->size == SLAB_BIG)
        {
            b->next = free_list_big;
            free_list_big = b;
            stats.curr_allocated_big--;
        }
        else
        {
            b->next = free_list_small;
            free_list_small = b;
            stats.curr_allocated_small--;
        }
    }
    return 0;
}

/**
 * \brief Get a buffer's tag.
 * \param b Buffer.
 * \return Buffer tag.
 */
unsigned int
buf_get_tag (buf_t b)
{
    if (b)
        return b->tag;
    return 0;
}

/**
 * \brief Set a buffer's tag.
 * \param b Buffer.
 * \param tag Tag.
 **/
void
buf_set_tag (buf_t b, unsigned int tag)
{
    if (b)
        b->tag = tag;
}

/**
 * \brief Get a value from the misc data area.
 * \param b Buffer.
 * \param index Index into the misc array (range: 0-7)
 * \return The value of the misc data array at the index specified.
 **/
unsigned char
buf_get_val (buf_t b, unsigned int index)
{
    if (b)
        return (b->misc[index & 0x07]);
    return (0);
}

/**
 * \brief Set a value in the misc data area.
 * \param b Buffer.
 * \param index Index into the misc array (range: 0-7)
 * \param val The value to write into the given array index
 **/
void
buf_set_val (buf_t b, unsigned int index, unsigned char val)
{
    if (b)
        b->misc[index & 0x07] = val;
}

/**
 \brief Put buffer in a queue.
 \param q Queue to add to.
 \param b Buffer to add.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 */
void
_buf_queue (buf_q *q, buf_t b, const char *file, int line)
{
    buf_q t;

    if (!b)
        return;

    libassert (q);
    t = *q;

    if (t)
    {
        while (t->next)
            t = t->next;
        t->next = b;
    }
    else
    {
        *q = b;
    }
}

/**
 \brief get buffer from head of a queue.
 \param q Queue to get from.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 \return A buffer.

 _buf_dequeue will return \c null if the queue is empty.
 */
buf_t
_buf_dequeue (buf_q *q, const char *file, int line)
{
    buf_t b = 0;
    buf_q t;

    libassert (q);
    t = *q;

    if (t)
    {
        b = t;
        *q = t->next;
    }
    return b;
}

/** \def P Macro for print function. */
#define P f_debug_writemsg
/** \def HEXDIGIT Macro for is printable. */
#define HEXDIGIT(x) (((x) < 10) ? ((x) + '0') : ((x) - 10 + 'A'))

/**
 \brief Pretty print a chunk of data.
 \param d Data to print.
 \param l Length of data.
 */
void
print_hex_data (unsigned char *d, int l)
{
    char  line[ 76 ];           /* construct a complete line, then print */
    char *linep = line;         /* HEX digits */
    char *ascii = line + 56;    /* ASCII encoding of printable */
    int  i, n = 0;

    //#define P printf

    memset(line, ' ', sizeof(line));   /* prefill with blanks */
    for (i = 0; i < l; i++, d++) {
        *linep++ = HEXDIGIT((*d >> 4) & 0xf);
        *linep++ = HEXDIGIT(*d & 0xf);
        linep++;
        *ascii++ = (*d >= ' ' && *d <= '~') ? *d : '.';
        if (++n == 8) {
            *linep = ':'; /* put ':' halfway through HEX */
            linep += 2;
            ascii++;      /* leave space halfway through ASCII */
        }
        if (n >= 16) {
            *ascii = 0;  /* delimit for printing */
            P ("%s\n", line);
            memset(line, ' ', sizeof(line));
            ascii = line + 56;
            linep = line;
            n = 0;
        }
    }
    if (n) {
        *ascii = 0;
        P ("%s\n", line);
    }
    #undef HEXDIGIT
}

/**
 \brief Pretty print n bytes of a buffer.
 \param b Buffer to print.
 \param num_bytes Number of bytes of data to print.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 */
void
_buf_nprint (buf_t b, size_t num_bytes, const char *file, int line)
{
    libassert (b);

    //#define P printf
    #define P f_debug_writemsg

    P ("(buffer (:buf-address %p)\n", b);
    P ("        (:data-address %p)\n", b->data);
    P ("        (:length %d)\n", b->length);
    P ("        (:size %d))\n", b->size);
    print_hex_data (_buf_data (b, file, line), (num_bytes > buf_length (b)) ? buf_length (b) : num_bytes);

    #undef P
}

/**
 \brief Pretty print a buffer.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 \param b Buffer to print.
 */
void
_buf_print (buf_t b, const char *file, int line)
{
    //#define P printf
    #define P f_debug_writemsg

    if ( buf_length (b) > 0 )
        _buf_nprint (b, buf_length (b), file, line);
    else
        P ("buffer is empty - nothing to print.\n");
}

/**
 \brief Get length of a buffer list.
 \param b Buffer list.
 */
static unsigned int
list_length (buf_t b)
{
    int i = 0;

    while (b)
    {
        i++;
        b = b->next;
    }
    return i;
}

/**
 \brief Print buffer statistics to a file.
 \param f File to print to.
 */
void
buf_print_stats (FILE *f)
{
    fprintf (f, "---Buffer Statistics-------------------------------------------\n");
    fprintf (f, "   Currently allocated from small pool     -> %d\n", stats.curr_allocated_small);
    fprintf (f, "   Maximum allocated from small pool       -> %d\n", stats.max_allocated_small);
    fprintf (f, "   Currently allocated from big pool       -> %d\n", stats.curr_allocated_big);
    fprintf (f, "   Maximum allocated big from pool         -> %d\n", stats.max_allocated_big);
    fprintf (f, "   Number of duplicate frees               -> %d\n", stats.dup_frees);
    fprintf (f, "   Small free list length                  -> %d\n", list_length (free_list_small));
    fprintf (f, "   Big free list length                    -> %d\n\n", list_length (free_list_big));
}

/**
 \brief Get the number of buffers in a queue.
 \param q Buffer queue.
 \return Number of buffers in the queue.
 */
unsigned int
buf_queue_length (buf_q q)
{
    return list_length (q);
}


#if 0

void
pb (buf_t b)
{
    int i;

    if (b)
    {
        printf ("(buf_t %d %d %p \"", buf_length (b), b->size, b->next);
        for (i = 0; i < (int)buf_length (b); i++)
            printf ("%c", buf_data (b)[i]);
        printf ("\")\n");
    }
    else
        printf ("(buf_t)\n");
}

void
pq (buf_q q)
{
    printf ("(queue ");
    if (q)
    {
        pb (q);
        q = q->next;
    }
    while (q)
    {
        printf ("       ");
        pb (q);
        q = q->next;
    }
    printf ("\b)\n");
}

void
pfl (void)
{
    int i = 0;
    buf_t b = free_list_big;

    for (i = 0; b; i++)
        b = b->next;

    printf ("(free-list-length-big %d)\n", i);

    b = free_list_small;

    for (i = 0; b; i++)
        b = b->next;

    printf ("(free-list-length-small %d)\n", i);

}

int
main (void)
{
    buf_t b = 0, h = 0, t = 0;
    buf_q q = 0;
    int i;
    buf_t ba[1000];

    for (i = 0; i < 1000; i++)
    {
        b = buf_append (0, "TESTING", 8);
        buf_free (b);
    }
    pfl ();

    for (i = 0; i < 1000; i++)
        ba[i] = buf_append (0, "TEST TEST TEST TEST", 20);
    pfl ();
    for (i = 0; i < 1000; i++)
        buf_free (ba[i]);
    pfl ();

    h = buf_append (h, "A ", 2);
    pb (h);
    h = buf_append (h, "B ", 2);
    pb (h);
    h = buf_append (h, "C ", 2);
    pb (h);

    t = buf_append (t, "1 ", 2);
    pb (t);
    t = buf_append (t, "2 ", 2);
    pb (t);
    t = buf_append (t, "3 ", 2);
    pb (t);

    b = buf_join (h, t);
    pb (b);
    buf_free (b);

    b = buf_append (0, " MORE", 5);
    pb (b);
    b = buf_append (b, " EVEN MORE", 10);
    pb (b);
    buf_free (b);

    #define add(s) do { b = buf_append (0, s, strlen (s)); } while (0)

    pq (q);

    add ("TEST1");
    buf_queue (q, b);
    pq (q);

    add ("TEST2");
    buf_queue (q, b);
    pq (q);

    add ("TEST3");
    buf_queue (q, b);
    pq (q);

    printf ("(Queue Length %d)\n", buf_queue_length (q));
    for (i = 0; i < 4; i++)
    {
        b = buf_dequeue (q);
        pb (b);
        buf_free (b);
        buf_free (b);
        pq (q);
    }

    printf ("(Queue Length %d)\n", buf_queue_length (q));
    pfl ();

    b = buf_append (0, "TEST", 5);
    buf_print (b);
    b = buf_dup (b);
    buf_print (b);

    buf_print_stats (stdout);

    b = buf_append (0, "TEST", 5);
    buf_set_tag (b, 'A');
    printf ("(Tag %c)\n", buf_get_tag (b));
    b = buf_dup (b);
    printf ("(Tag %c)\n", buf_get_tag (b));
    return 0;
}

#endif

/*
 * End of $Id: buf.c,v 1.3 2005/07/18 19:01:48 cmayncvs Exp $
 */


