/* ===[ $RCSfile: c_config.c,v $ ]========================================

    This item is the property of GTECH Corporation, West Greenwich,
    Rhode Island, and contains confidential and trade secret information.
    It may not be transferred from the custody or control of GTECH except
    as authorized in writing by an officer of GTECH.  Neither this item
    nor the information it contains may be used, transferred, reproduced,
    published, or disclosed, in whole or in part, and directly or
    indirectly, except as expressly authorized by an officer of GTECH,
    pursuant to written agreement.

    Copyright (c) 2002-2005 GTECH Corporation.  All rights reserved.

   ======================================================================= */

/** \file
 *
 *  "$Id: c_config.c,v 1.5 2005/02/09 17:13:34 cmayncvs Exp $"
 *
 *  \brief Implements the standard comm parameter read/write interface.
 *
 *  This interface isolates comm from parameter reading and writing details
 *  and allows for easy update/expansion of parameters in the field.  It
 *  utilizes a local version of the Altura firmware config library.
 *
 *  \warning In order to ensure that this interface to the communications
 *  parameters always works, the following convention \b must \b be
 *  \b followed regarding the structs that hold the parameters:
 *      \li Never delete parameter members from the struct
 *      \li Always add new parameter members to the end of the struct
 *
 *    Check record sizes greater than 512 bytes.
 *    Check record size limited to 4 chars (ex "MESC"  MESC.conf).
 *
 */
/* ======================================================================= */

/*===============*/
/* Include Files */
/*===============*/
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <unistd.h>
#include <string.h>
#include "gassert.h"
#include "c_config.h"

/*=============================*/
/* Local functions & variables */
/*=============================*/
static void config_init(const char *file, int line);
/** Flag to indicate if the gconfig library has been initialized or not. */
static int c_config_initialized = 0;

/* Gconfig stuff for keeping ESPAD as closely sync'd to Altura as possible */
/** Macro to pass file & line # from caller to _f_config_write_record()
 *  library function. */
#define f_config_write_record(rec, src, size)   \
    _f_config_write_record(rec, src, size, file, line)

static int f_config_initdir(char *pdir, char *sdir, char *tdir, char *ddir);
static int f_config_init(void);
static int f_config_read_record(char* recordName, char* destPtr, int bufSize);
static int f_config_read_default_record(char* recordName, char* destPtr, int bufSize);
static int _f_config_write_record(char* recordName, char* srcPtr, int bufSize,
    const char *file, int line);
static int f_config_commit_record(char *recordName);
static int f_config_rollback_record(char *recordName);
#if 0 /* Not used ... yet */
static int f_config_erase(void);
static int f_config_erase_record(char* recordName);
static int f_config_backup(char *dir);
#endif /* 0 */
static void f_config_set_debug_level(int level);

static void sem_lock(const char *file, int line);
static void sem_unlock(const char *file, int line);

/* ======================================================================= */
/**
 *  \brief Reads \a size bytes from \a record into \a dest.
 *  \param record The record name to read from.
 *  \param dest The destination memory area.
 *  \param defaults A default set of parameters to use as a starting point.
 *  \param size How much data to read.
 *  \param file The file this call was made from.
 *  \param line The line in the file where this call was made.
 *
 *  Attempts to read \a size bytes from \a record into \a dest.  If \a size
 *  is less than the available data in the current record, the extra data in
 *  the current record is ignored.  If \a size is greater than the size of the
 *  current record, \a defaults is used to fill in the extra data.  A new
 *  record is never written.
 *
 *  If no valid current record exists, a rollback is done and the old record
 *  is tried.  If that fails, the factory default record is tried.  If that
 *  also fails, \a dest will contain all user defaults from \a defaults.
 *
 *  \return A pointer to \a dest, guaranteed to be non-NULL, and guaranteed to
 *      be of size \a size.
 */
/* ======================================================================= */
void *_c_config_read(const char *record, void *dest, const void *defaults,
    size_t size, const char *file, int line)
{
    int retval;

    libassert(record && dest && defaults);
    memcpy(dest, defaults, size);

    /* Make sure the initialization has been done. */
    config_init(file, line);

    retval = f_config_read_record((char *)record, (char *)dest, (int)size);
    if ( retval == ERROR )
    {
        libassert(gerrno == (CFGERR_OPEN));

        /* Error - rollback and try again. */
        retval = f_config_rollback_record((char *)record);
        if ( retval == 0 )
        {
            retval = f_config_read_record((char *)record, (char *)dest,
                (int)size);
            if ( retval == ERROR )
            {
                libassert(gerrno == (CFGERR_OPEN));

                /* Error in rollback'd record too - use defaults */
                _c_config_read_defaults(record, dest, defaults, size, file, line);
            }
        }
        else
        {
            libassert(gerrno == (CFGERR_NOOLDCFG));

            /* Rollback failed - use defaults. */
            _c_config_read_defaults(record, dest, defaults, size, file, line);
        }
    }
    return (dest);

} /* _c_config_read() */


/* ======================================================================= */
/**
 *  \brief Reads \a size bytes from default record \a record into \a dest.
 *  \param record The default record name to read from.
 *  \param dest The destination memory area.
 *  \param defaults A default set of parameters to use as a starting point.
 *  \param size How much data to read.
 *  \param file The file this call was made from.
 *  \param line The line in the file where this call was made.
 *
 *  Attempts to read \a size bytes from the factory default record \a record
 *  into \a dest.  If \a size is greater than the size of the factory default
 *  record, \a defaults is used to fill in the extra data.  If \a size is less
 *  than the available data in the factory default record, the additional
 *  factory defaults are ignored.
 *
 *  If no valid factory default record exists, \a dest will contain all user
 *  defaults from \a defaults.
 *
 *  \return A pointer to \a dest, guaranteed to be non-NULL, and guaranteed to
 *      be of size \a size.
 */
/* ======================================================================= */
void *_c_config_read_defaults(const char *record, void *dest,
    const void *defaults, size_t size, const char *file, int line)
{
    int retval;

    libassert(record && dest && defaults);
    memcpy(dest, defaults, size);

    /* Make sure the initialization has been done. */
    config_init(file, line);

    /* Read the default record */
    retval = f_config_read_default_record((char *)record, (char *)dest,
        (int)size);
    if ( retval == ERROR )
        libassert(gerrno == (CFGERR_OPEN));
    return (dest);

} /* _c_config_read_defaults() */


/* ======================================================================= */
/**
 *  \brief Writes and commits \a size bytes from \a src to record \a record.
 *  \param record The record name to write to.
 *  \param src The memory area holding the data to write.
 *  \param size How much data to write.
 *  \param file The file this call was made from.
 *  \param line The line in the file where this call was made.
 *
 *  \return Nothing.
 */
/* ======================================================================= */
void _c_config_write(const char *record, const void *src, size_t size,
    const char *file, int line)
{
    int retval;

    libassert(record && src);

    /* Make sure the initialization has been done. */
    config_init(file, line);

    /* Write & commit the data */
    retval = f_config_write_record((char *)record, (char *)src, (int)size);
    libassert(retval != ERROR);
    retval = f_config_commit_record((char *)record);
    libassert(retval == 0);

} /* _c_config_write() */


/* ======================================================================= */
/**
 *  \brief Commits parameters to flash memory.
 *  \param file The file this call was made from.
 *  \param line The line in the file where this call was made.
 *  \return Nothing.
*/
/* ======================================================================= */
void _c_config_commit(const char *file, int line)
{
    sem_lock(file, line);
    /* Call the Cyclades script to save it to FLASH */
    /* This may take about 15 seconds. */
    system("saveconf \n");
    sem_unlock(file, line);
} /* _c_config_commit() */


/* ======================================================================= */
/**
 *  \brief Sets the debug level for the config library
 *  \param level the debug level (0-7), defined in gdebug.h, or
 *      (-1) indicating that we're calling it internally.
 *
 *  If the debug level hasn't been set yet and we call it internally, then
 *  go ahead and turn debug off, but as soon as a call is made to set the
 *  debug level, then don't ever change it internally.
 *
 *  \return nothing
*/
/* ======================================================================= */
void c_config_set_debug_level(int level)
{
    /* Not used here, so just pass it along to the config library. */
    f_config_set_debug_level(level);

} /* c_config_set_debug_level() */


/* ======================================================================= */
/**
 *  \brief Initializes the config library if it hasn't been done yet.
 *  \param file The file this call was made from.
 *  \param line The line in the file where this call was made.
 *  \return nothing
*/
/* ======================================================================= */
static void config_init(const char *file, int line)
{
    if ( c_config_initialized == 0 )
    {
        /* If sysman is running, should only need to call this */
        if ( f_config_init() != OK )
        {
            /* Assume no sysman (i.e. development workstation, manually
             * starting up processes) , so attempt default directories */
            libassert(f_config_initdir(G_CONFIG_PRIMARY_DIR, G_CONFIG_SHADOW_DIR,
                G_CONFIG_BACKUP_DIR, G_CONFIG_DEFAULT_DIR) == OK);
            libassert(f_config_init() == OK);
        }
        c_config_initialized = ~0;
    }
} /* config_init() */

            /* ======================================= */
            /* Firmware gconfig library ports to ESPAD */
            /* ======================================= */
#if defined(__GNU_LIBRARY__) && !defined(_SEM_SEMUN_UNDEFINED)
       /* union semun is defined by including <sys/sem.h> */
       #else
       /** According to X/OPEN we have to define it ourselves */
       union semun {
             int val;                  /**< value for SETVAL */
             struct semid_ds *buf;     /**< buffer for IPC_STAT, IPC_SET */
             unsigned short *array;    /**< array for GETALL, SETALL
                                            Linux specific part: */
             struct seminfo *__buf;    /**< buffer for IPC_INFO */
       };
#endif
/** Defines our debug level variable for info(), warn(), trace() & print. */
#define MODULE_DEBUG_LEVEL  f_config_dbglevel
/** Defines this 'module's' debug name. */
#define MODULE_NAME         "CCFG"

#include <sys/stat.h>
#include <fcntl.h>
#include <sys/ipc.h>
#include <sys/sem.h>
#include "debug.h"

/** The debug level for this 'module'. */
static int f_config_dbglevel = DBG_NONE;
/** Semaphore ID for locking read/writes. */
static int cfg_semid = -1;

static int make_directory_if_missing(char *path);
static int check_rec_size(const char *record, size_t size);

/* ======================================================================= */
/**
 *  \brief Initializes the config library if it hasn't been done yet.
 *      Creates a config and default directory.
 *  \param pdir Primary directory
 *  \param sdir Secondary directory - not used
 *  \param tdir Tertiary directory - not used
 *  \param ddir Default directory
 *  \return OK or ERROR.
*/
/* ======================================================================= */
static int f_config_initdir(char *pdir, char *sdir, char *tdir, char *ddir)
{
    int retval;

    gerrno = -1;    /* Force gerrno to something undefined */

    if ( (retval = make_directory_if_missing(pdir)) == OK )
        return (make_directory_if_missing(ddir));
    return (retval);
} /* f_config_initdir() */


/* ======================================================================= */
/**
 *  \brief Initializes the config library if it hasn't been done yet.
 *  \return OK or ERROR.
*/
/* ======================================================================= */
static int f_config_init(void)
{
    return (f_config_initdir(ESPAD_CONFIG_DIR, NULL, NULL,
        ESPAD_CONFIG_DEFAULT_DIR));
} /* f_config_init() */


/* ======================================================================= */
/**
 *  \brief Handles lower-level reads, similar to Altura gconfig library.
 *  \param recordName The extension-free name of the record to read.
 *  \param destPtr The memory area to read the record contents into.
 *  \param bufSize The number of bytes to read.
 *  \return The # of bytes read or ERROR with gerrno set appropriately.
 */
/* ======================================================================= */
static int f_config_read_record(char* recordName, char* destPtr, int bufSize)
{
    int fp;
    int r_size;
    char fname[256];

    gerrno = -1;    /* Force gerrno to something undefined */

    /* Perform some basic checking. */
    if ( check_rec_size(recordName, bufSize) == ERROR )
        return (ERROR);

    /* Open the file read/only. */
    snprintf(fname, sizeof(fname), "%s/%s.conf", ESPAD_CONFIG_DIR, recordName);
    fname[sizeof(fname) - 1] = '\0';
    if ( (fp = open(fname, O_RDONLY)) < 0 )
    {
        gerrno = CFGERR_OPEN;
        return (ERROR);
    }

    /* Read in the configuration. */
    r_size = read(fp, destPtr, bufSize);
    close(fp);
    if ( r_size < 0 )
    {
        gerrno = CFGERR_READ;
        return (ERROR);
    }
    return (r_size);
} /* f_config_read_record() */


/* ======================================================================= */
/**
 *  \brief Handles lower-level default reads, similar to Altura gconfig library.
 *  \param recordName The extension-free name of the default record to read.
 *  \param destPtr The memory area to read the record contents into.
 *  \param bufSize The number of bytes to read.
 *  \return The # of bytes read or ERROR with gerrno set appropriately.
 */
/* ======================================================================= */
static int f_config_read_default_record(char* recordName, char* destPtr, int bufSize)
{
    int fp;
    int r_size;
    char fname[256];

    gerrno = -1;    /* Force gerrno to something undefined */

    /* Perform some basic checking. */
    if ( check_rec_size(recordName, bufSize) == ERROR )
        return (ERROR);

    /* Open the file read/only. */
    snprintf(fname, sizeof(fname), "%s/%s.conf", ESPAD_CONFIG_DEFAULT_DIR,
        recordName);
    fname[sizeof(fname) - 1] = '\0';
    if ( (fp = open(fname, O_RDONLY)) < 0 )
    {
        gerrno = CFGERR_OPEN;
        return (ERROR);
    }

    /* Read in the configuration. */
    r_size = read(fp, destPtr, bufSize);
    close(fp);
    if ( r_size < 0 )
    {
        gerrno = CFGERR_READ;
        return (ERROR);
    }
    return (r_size);
} /* f_config_read_default_record() */


/* ======================================================================= */
/**
 *  \brief A stub for gconfig library.  Always returns ERROR.
 *  \param recordName Record name to roll back. (not used).
 *  \return ERROR, always with gerrno set to CFGERR_NOOLDCFG.
 */
/* ======================================================================= */
static int f_config_rollback_record(char *recordName)
{
    /* rollback never works since we don't save the previous record */
    gerrno = CFGERR_NOOLDCFG;
    return (ERROR);
} /* f_config_rollback_record() */


/* ======================================================================= */
/**
 *  \brief Handles lower-level writes, similar to Altura gconfig library.
 *  \param recordName The extension-free name of the record to write.
 *  \param srcPtr The memory area holding the record data to write.
 *  \param bufSize The number of bytes to write.
 *  \param file The file this call was made from.
 *  \param line The line in the file where this call was made.
 *  \return The # of bytes written or ERROR with gerrno set appropriately.
 */
/* ======================================================================= */
static int _f_config_write_record(char* recordName, char* srcPtr, int bufSize,
    const char *file, int line)
{
    int fp;
    int w_size;
    char fname[256];

    gerrno = -1;    /* Force gerrno to something undefined */

    /* Perform some basic checking. */
    if ( check_rec_size(recordName, bufSize) == ERROR )
        return (ERROR);

    /* Open the file for writing. */
    snprintf(fname, sizeof(fname), "%s/%s.conf", ESPAD_CONFIG_DIR, recordName);
    fname[sizeof(fname) - 1] = '\0';
    sem_lock(file, line);
    if ( (fp = open(fname, O_WRONLY|O_CREAT|O_TRUNC, S_IREAD|S_IWRITE)) < 0 )
    {
        sem_unlock(file, line);
        gerrno = CFGERR_OPEN;
        return (ERROR);
    }

    /* Write & commit (to RAM) the data */
    w_size = write(fp, srcPtr, bufSize);
    close(fp);
    sem_unlock(file, line);
    if ( w_size != bufSize )
    {
        gerrno = CFGERR_WRITE;
        return (ERROR);
    }
    return (w_size);
} /* _f_config_write_record() */


/* ======================================================================= */
/**
 *  \brief A stub for gconfig library.
 *  \param recordName Record name to roll back. (not used).
 *  \return OK, unless recordName is too long, then ERROR, with gerrno set to
 *      CFGERR_NAMELONG.
 */
/* ======================================================================= */
static int f_config_commit_record(char *recordName)
{
    gerrno = -1;    /* Force gerrno to something undefined */

    /* Perform some basic checking. */
    if ( check_rec_size(recordName, 0) == ERROR )
        return (ERROR);
    return (OK);

} /* f_config_commit_record() */


#if 0 /* Not used ... yet */
/* ======================================================================= */
/**
 *  \brief A stub for gconfig library.  Always returns ERROR.
 *  \note This could be used to issue something like,
 *      system("echo 0 > /proc/flash/script");
 *  \return ERROR, always with gerrno set to CFGERR_REMOVE.
 */
/* ======================================================================= */
static int f_config_erase(void)
{
    gerrno = CFGERR_REMOVE;
    return (ERROR);
} /* f_config_erase() */


/* ======================================================================= */
/**
 *  \brief A stub for gconfig library.  Always returns ERROR.
 *  \note This could be used to issue something like,
 *      system("rm -f /etc/gtech/config/recordName.conf");
 *  \return ERROR, always with gerrno set to CFGERR_REMOVE.
 */
/* ======================================================================= */
static int f_config_erase_record(char* recordName)
{
    /* Perform some basic checking. */
    if ( check_rec_size(recordName, 0) == ERROR )
        return (ERROR);
    gerrno = CFGERR_REMOVE;
    return (ERROR);
} /* f_config_erase_record() */


/* ======================================================================= */
/**
 *  \brief A stub for gconfig library.  Always returns ERROR.
 *  \param dir The directory to back up this file to.
 *  \note This could be used to issue something like,
 *      system("cp -f /etc/gtech/config/recordName.conf dir");
 *  \return ERROR, always with gerrno set to CFGERR_COPY.
 */
/* ======================================================================= */
static int f_config_backup(char *dir)
{
    gerrno = CFGERR_COPY;
    return (ERROR);
} /* f_config_backup() */
#endif /* 0 */

/* ======================================================================= */
/**
 *  \brief Sets the debug level for the config library
 *  \param level the debug level (0-7), defined in gdebug.h, or
 *      (-1) indicating that we're calling it internally.
 *
 *  If the debug level hasn't been set yet and we call it internally, then
 *  go ahead and turn debug off, but as soon as a call is made to set the
 *  debug level, then don't ever change it internally.
 *
 *  \return nothing
*/
/* ======================================================================= */
void f_config_set_debug_level(int level)
{
    static int f_config_debug_level_not_set = 1;

    if ( level >= 0 )
    {
        f_config_debug_level_not_set = 0;
        f_config_dbglevel = level & 0x07;
    }
    else if ( f_config_debug_level_not_set )
        f_config_dbglevel = DBG_NONE;
} /* f_config_set_debug_level() */

            /* ================================== */
            /* Gconfig library 'helper functions' */
            /* ================================== */

/* ======================================================================= */
/**
 *  \brief Converts an error number to a more useful human readable message.
 *  \param err_no The gerrno to convert to a string.
 *  \return A string representation of the gerrno error number, or "unknown".
*/
/* ======================================================================= */
static char *strgerror(int err_no)
{
    static char *strgerrs[27] = {
        "The library is not initialized",                   /* 0 */
        "The record name is too long",                      /* 1 */
        "CRC mismatch",                                     /* 2 */
        "Memory allocation failed",                         /* 3 */
        "Error creating config file",                       /* 4 */
        "Error opening config file",                        /* 5 */
        "Error closing config file",                        /* 6 */
        "Error reading from config file",                   /* 7 */
        "Error writing to config file",                     /* 8 */
        "Error synchronizing config file to disk",          /* 9 */
        "Error copying from source to destination",         /* 10 */
        "Error copying old config file to current",         /* 11 */
        "Error moving from source to destination",          /* 12 */
        "Error moving new config file to current",          /* 13 */
        "Error moving current config file to old",          /* 14 */
        "Error removing a config file",                     /* 15 */
        "Error removing the new config file",               /* 16 */
        "Error removing the current config file",           /* 17 */
        "Error removing the old config file",               /* 18 */
        "Error removing a config directory",                /* 19 */
        "No old config data during roll back",              /* 20 */
        "Error creating shared memory",                     /* 21 */
        "Error creating semaphore",                         /* 22 */
        "Error getting sem",                                /* 23 */
        "Error locking sem",                                /* 24 */
        "Error unlocking sem",                              /* 25 */
        "Error unlocking semaphore"                         /* 26 */
    };

    if ( (err_no < 0) || (err_no >= 27) )
        return ("Unknown");
    return (strgerrs[err_no]);

} /* strgerror() */


/* ======================================================================= */
/**
 *  \brief Test for existence of a directory and create it if missing
 *  \param path The path of a directory that must exist
 *  \return OK or ERROR if mkdir syscall fails (sets gerrno)
 *
 *  If a non-directory object exists with the name in 'path', it will be
 *  excised.  This function recursively creates any needed parent directories
 *  (if called with an absolute path, i.e., starting with '/').
*/
/* ======================================================================= */
static int make_directory_if_missing(char *path)
{
    struct stat sb;
    char mkdir_cmd[256];

    gerrno = -1;    /* Force gerrno to something undefined */

    /* First check if it's already there */
    if ( (stat(path, &sb) == 0) && S_ISDIR(sb.st_mode) )
    {
        info("Directory \"%s\" already exists.", path);
        return (OK);
    }
    /* In case there's something there, it's not a directory, so delete it. */
    unlink(path);

    /* Create the directories, recursively if necessary ... */
    snprintf(mkdir_cmd, sizeof(mkdir_cmd), "mkdir -p -m 755 %s", path);
    mkdir_cmd[sizeof(mkdir_cmd) - 1] = '\0';
    system(mkdir_cmd);

    /* ... now check the results  */
    if ( (stat(path, &sb) == 0) && S_ISDIR(sb.st_mode) )
    {
        info("Directory \"%s\" successfully created.", path);
        return (OK);
    }
    gerrno = CFGERR_CREAT;
    warn("make_directory_if_missing(%s): gerrno=%d (%s).", path,
        gerrno, strgerror(gerrno));
    return (ERROR);

} /* make_directory_if_missing() */


/* ======================================================================= */
/**
 *  \brief Performs basic sanity checking of record name and size.
 *  \param record The record name, limited to \a MAXREC_FILESIZE char's.
 *  \param size The size of the record, limited to \a MAXREC_RECSIZE bytes.
 *  \return OK or ERROR if either record or size too big with gerrno set
 */
/* ======================================================================= */
static int check_rec_size(const char *record, size_t size)
{
    gerrno = -1;    /* Force gerrno to something undefined */

    if ( strlen(record) > MAXREC_FILESIZE )
    {
        gerrno = CFGERR_NAMELONG;
        warn("check_rec_size(%s): gerrno=%d (%s).", record,
            gerrno, strgerror(gerrno));
        return (ERROR);
    }
    if ( size > MAXREC_RECSIZE )
    {
        gerrno = CFGERR_NOMEM;  /* silly, but this is what gconfig does */
        warn("check_rec_size(): size %d too big - max is %d, gerrno=%d (%s) ",
            size, MAXREC_RECSIZE, gerrno, strgerror(gerrno));
        return (ERROR);
    }
    return (OK);
} /* check_rec_size() */


/*************************************************************************
 *                             Semaphore                                 *
 *                                                                       *
 *  Allows only one process at a time access to config directories.      *
 *                                                                       *
 *************************************************************************/
/* ======================================================================= */
/**
 *  \brief lock the global semaphore in per-process variable 'cfg_semid'
 *  \param file The file this call was made from.
 *  \param line The line in the file where this call was made.
 *  \return Nothing.
*/
/* ======================================================================= */
static void sem_lock(const char *file, int line)
{
    struct sembuf sb[2] = {[0] {0, 0, 0}, [1] {0, 1, 0}};

    if ( cfg_semid == -1 )
        libassert((cfg_semid = semget (CFG_SEMKEY, 1, IPC_CREAT | 0666)) >= 0);

    libassert(semop(cfg_semid, sb, 2) >= 0);
} /* sem_lock() */


/* ======================================================================= */
/**
 *  \brief unlock the global semaphore in per-process variable 'cfg_semid'
 *  \param file The file this call was made from.
 *  \param line The line in the file where this call was made.
 *
 *  Once the semaphore is released, the process's previous sigmask restored.
 *
 *  \return Nothing.
*/
/* ======================================================================= */
static void sem_unlock(const char *file, int line)
{
    struct sembuf sb = {0, -1, IPC_NOWAIT};

    libassert(semop(cfg_semid, &sb, 1) >= 0);

} /* sem_unlock() */


#if 0
/********************
   test code
*********************/
int main(int argc, char *argv[])
{
    static const testcfg_struct read_struc;

    printf("Enter '1': Write Flash   '2': Read Flash  3: Write Defualts 4: "
        "Read Default %d\n", argc);

    if ( argc != 2 )
    {
        printf("no argc \n");
        exit(0);
    }

    if ( argv[1][0] == '1' )
    {
        /* Check directories, create Sem. */
        config_init();
        printf("call c_config_write() \n");
        c_config_write("TEST", &testcfg_defaults, sizeof(testcfg_struct));
        printf("c_config_write() done\n");
    }

    if ( argv[1][0] == '2' )
    {
        /* Check directories, create Sem. */
        config_init();
        printf("call c_config_read() \n");
        c_config_read("TEST", (char *) &read_struc, &testcfg_defaults,
            sizeof(testcfg_struct));
        printf("c_config_read() %s %s %s %s done\n",
            read_struc.test_str1, read_struc.test_str2,
            read_struc.test_str3, read_struc.test_str4);
    }

    if ( argv[1][0] == '3' )
    {
        /* Check directories, create Sem. */
        config_init();
        printf("This option is obsolete.  done.\n");
    }

    if ( argv[1][0] == '4' )
    {
        printf("call c_config_read_defaults() \n");
        c_config_read_defaults("TEST", (char *) &read_struc, &testcfg_defaults,
            sizeof(testcfg_struct));
        printf("c_config_read() %s %s %s %s done\n",
            read_struc.test_str1, read_struc.test_str2,
            read_struc.test_str3, read_struc.test_str4);
    }
    return (0);
} /* main() */

#endif /* 0 */


/*
 * End "$Id: c_config.c,v 1.5 2005/02/09 17:13:34 cmayncvs Exp $"
 */


