/* ===[ $RCSfile: fd.c,v $ ]=================================================

    This item is the property of GTECH Corporation, West Greenwich,
    Rhode Island, and contains confidential and trade secret information.
    It may not be transferred from the custody or control of GTECH except
    as authorized in writing by an officer of GTECH.  Neither this item
    nor the information it contains may be used, transferred, reproduced,
    published, or disclosed, in whole or in part, and directly or
    indirectly, except as expressly authorized by an officer of GTECH,
    pursuant to written agreement.

    Copyright (c) 2002-2005 GTECH Corporation.  All rights reserved.

   ======================================================================= */

/** \file

  $Id: fd.c,v 1.2 2005/02/04 23:05:08 cmayncvs Exp $

 \brief These functions are for managing lists of file descriptors.
 */
#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include "buf.h"
#include "list.h"
#include "fd.h"
#include "gassert.h"

/** Alist of file descriptors, functions and pointers. */
static list fd_list = 0;

/** List of file descriptors to remove after call all readable. */
static list rm_list = 0;

/** Block removing file descriptors. */
static int block_remove = 0;

/** Block removing fd's. */
#define BLOCK do{block_remove = 1;}while(0)
/** Unblock removing fd's and process rm_list. */
#define UNBLOCK do{block_remove = 0; do_remove (file, line);}while(0)

/**
 \brief Adds a file descriptor to the select list.
 \param fd File descriptor to add.
 \param proc Function to call when fd becomes readable.
 \param ptr Pointer to call proc with.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.

 An alist with the following form is constructed.
 ((fd (proc ptr)) (fd (proc ptr)) ...)
 */
void
_fd_add (int fd, void (*proc)(int, void*), void *ptr, const char *file, int line)
{
    libassert (proc);
    fd_list = _acons ((void *)fd, _cons (proc, _cons (ptr, 0, file, line), file, line), fd_list, file, line);
}

/**
 \brief Remove a file descriptor from the list.
 \param fd File descriptor to remove.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 */
static void
remove_fd (int fd, const char *file, int line)
{
    int eq (void *a, void *k)
    {
        return a == car (k);
    }
    void da (void *a)
    {
        destroy (_nth (1, (list)a, file, line));
        destroy ((list)a);
    }
    fd_list = removef ((void *)fd, fd_list, eq, da);
}

/**
 \brief Remove each file descriptor that is in the pending list from the select list.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 */
static void
do_remove (const char *file, int line)
{
    while (rm_list)
    {
        remove_fd ((int)car (rm_list), file, line);
        rm_list = cdrf (rm_list);
    }
}

/**
 \brief Removes a file descriptor from the select list.
 \param fd File descriptor to remove.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 \warning If \a ptr from fd_add points to malloc'd memory, you must free it.
 */
void
_fd_remove (int fd, const char *file, int line)
{
    rm_list = _cons ((void *)fd, rm_list, file, line);
    if (!block_remove)
        do_remove (file, line);
}

/**
 \brief Waits forever for a change in one or more file descriptors.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.

 For each file descriptor that becomes readable the supplied function is
 called with two arguments, the file descriptor and the point supplied to
 fd_add.
 */
void
_fd_select (const char *file, int line)
{
    _fd_select_with_timeout (0, file, line);
}

/**
 \brief Waits for a change in one or more file descriptors up to \a msec msecs.
 \param msec The number of milli-seconds to wait: 0 = wait forever.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.

 For each file descriptor that becomes readable the supplied function is
 called with two arguments, the file descriptor and the point supplied to
 fd_add.

 \return The return value of \a select().
 */
int
_fd_select_with_timeout (unsigned int msec, const char *file, int line)
{
    fd_set rfds;
    int retval;
    int max_fd = 0;
    struct timeval tv;

    void mkset (void *fdl)
    {
        FD_SET ((int)car ((list)fdl), &rfds);
        max_fd = MAX (max_fd, (int)car((list)fdl));
    }
    void call (void *fdl)
    {
        int fd = (int)car ((list)fdl);
        if (FD_ISSET (fd, &rfds))
            ((void(*)(int, void *))car (_nth (1, fdl, file, line))) (fd, _nth (1, _nth (1, fdl, file, line), file, line));
    }

    for (;;)
    {
        if (msec)
        {
            tv.tv_sec = msec / 1000;
            tv.tv_usec = (msec % 1000) * 1000;
        }
        FD_ZERO (&rfds);
        mapcar (mkset, fd_list);

        if (msec)
            retval = select (max_fd + 1, &rfds, 0, 0, &tv);
        else
            retval = select (max_fd + 1, &rfds, 0, 0, 0);

        libassert ((retval >= 0) || (retval == -1 && errno == EINTR));
        if (retval >= 0)
            break;
    }
    BLOCK;
    mapcar (call, fd_list);
    UNBLOCK;
    return retval;
}

/**
 \brief Print file descriptor statistics to a file.
 \param f File to print to.
 */
void
fd_print_stats (FILE *f)
{
    list t = fd_list;
    fprintf (f, "---File Descriptor List Statistics-----------------------------\n");
    fprintf (f, "   FD List -> (");
    while (t)
    {
        fprintf (f, "%d", (int)car (car (t)));
        if ((t = cdr (t)))
            fprintf (f, " ");
    }
    fprintf (f, ")\n\n");
}

#if 0

void
pl (list t)
{
    printf ("(");
    while (t)
    {
        printf ("(fd:%d (proc:%p ptr:%p))",
            (int)car (car (t)),
            car (nth(1, car (t))),
            nth (1, (nth(1, car (t)))));
        t = cdr (t);
    }
    printf (")\n");
}

void
test_proc (int fd, void *ptr)
{
    char t[100];
    int n;

    n = read (fd, t, 100);
    t[n] = '\0';
    printf ("Change on fd %d read %s with %s\n", fd, t, (char *)ptr);
}

void
test_proc_f (int fd, void *ptr)
{
    int i, n;
    char t[10];

    n = read (fd, t, 5);
    printf ("(read %d)\n", n);
    for (i = 0; i < n; i++)
        printf ("%c\n", t[i]);
}

int
main (void)
{
    int f, i;
    assert (f = open ("/tmp/test.fd", 0));
    pl (fd_list);
    fd_add (f, test_proc_f, 0);
    pl (fd_list);
    fd_add (0, test_proc, "TEST");
    pl (fd_list);
    for (i = 0;; i++)
    {
        if (i == 1)
        {
            fd_remove (f);
            pl (fd_list);
        }
        if (fd_select_with_timeout (5000) == 0)
            printf ("Select Timeout...\n");
        else
            list_print_stats (stdout);
    }

    return 0;
}

#endif

/*
 * End of $Id: fd.c,v 1.2 2005/02/04 23:05:08 cmayncvs Exp $
 */

